/*
//====================================================================
// MW_WSIRLE.c:
//
// This file contains the implementation of the
// WSI Run Length Encoding ADT as described in the
// MW_WSIRLE.h public interface file.
//
// Author: Damon M. Hill
// Copyright � 2005 WSI Corporation.  All rights reserved.
//====================================================================
*/

/*
//====================================================================
// Includes
//====================================================================
*/
#include "MW_WSIRLE.h"

#include "MW_OutputEncoder.h"

#ifdef HAS_MEMCPY
	#include <memory.h>
#endif

/*
//====================================================================
// Local private variables
//====================================================================
*/

static TByte_Unsigned MWWSIRLE_bool_runCompleted = 0;
static TByte_Unsigned MWWSIRLE_currentPixelHighNibbleValue = 0;
static TByte_Unsigned MWWSIRLE_currentRunPixelValue = 0;
static TFourByteInteger_Unsigned MWWSIRLE_currentRunLengthRemaining = 0;

/*
//====================================================================
//====================================================================
// Internal methods
//====================================================================
//====================================================================
*/

/*
//====================================================================
// MWWSIRLE_Internal_accumulateRun:
//
// Accumulates a run from the given input pixels.
//
// PARAMETERS:
//
//	numPixels:
//		This parameter should be set to the number of pixels
//		in the callers p_Pixels array.
//
//	p_Pixels:
//		This pointer to an array of pixels is used to
//		indicate the first of the numPixels in the pixel
//		array.
//
//	p_inout_numPixelsConsumed:
//		The counter referred to by this pointer is used to
//		track the number of input pixels in the callers array
//		which have already been consumed and should not be
//		encoded.  It is passed in as the value which reflects
//		the state of the pixels in the buffer, and
//		updated by the method before return.
//
// SIDE EFFECTS:
//	Changes the following state variables:
//		MWWSIRLE_bool_runCompleted
//		MWWSIRLE_currentRunPixelValue
//		MWWSIRLE_currentRunLengthRemaining
//
// RETURNS:
//		MWRes_Success
//
//====================================================================
*/

TMW_ResultCode MWWSIRLE_Internal_accumulateRun
(
	// Input pixel buffer parameters
	TFourByteInteger_Unsigned numPixels,
	TByte_Unsigned* p_Pixels,
	TFourByteInteger_Unsigned* p_inout_numPixelsConsumed
)
{
	/*
	//##############################################################
	// If there is a completed run that is not completely
	// expressed to output, we are done.
	//##############################################################
	*/
	if
	(
		(MWWSIRLE_bool_runCompleted != 0)
	)
	{
		return MWRes_Success;
	}

	/*
	//##############################################################
	// Accumulate run data from the input until a run
	// is completed or the input is exhausted
	//##############################################################
	*/
	while
	(
		(MWWSIRLE_bool_runCompleted == 0) &&
		(*p_inout_numPixelsConsumed < numPixels)
	)
	{
		if (MWWSIRLE_currentRunLengthRemaining > 0)
		{
			/*
			// Is the pixel a continuation of the currently
			// accumulating run?
			//
			// If so, then add to that run.
			// If not, the run is completed.
			*/
			if (p_Pixels[*p_inout_numPixelsConsumed] == MWWSIRLE_currentRunPixelValue)
			{
				// Add to the current run
				MWWSIRLE_currentRunLengthRemaining ++;
				*p_inout_numPixelsConsumed = (*p_inout_numPixelsConsumed) + 1;
			}
			else
			{
				/* Existing run is finished */
				MWWSIRLE_bool_runCompleted = 1;
				return MWRes_Success;
			}
		}
		else
		{
			/* We are starting a new run */
			MWWSIRLE_currentRunPixelValue = p_Pixels[*p_inout_numPixelsConsumed];
			*p_inout_numPixelsConsumed = (*p_inout_numPixelsConsumed) + 1;

			MWWSIRLE_currentRunLengthRemaining = 1;
		}

	} /* Loop until run is completed or all input is accumulated */

	/*
	//##############################################################
	// Success
	//##############################################################
	*/
	return MWRes_Success;

}

/*
//====================================================================
// MWWSIRLE_Internal_encodeRun:
//
// This encodes the internally encoded run into the output
// buffer if there is enough room. If there is not enough
// room, then MWRes_DataStructureFull is returned.
//
// PARAMETERS:
//
//	maxOutputBytes:
//		This indicates the maximum number of bytes that
//		can be stored in the output run code buffer
//		from the p_outputByteBuffer position forward.
//
//	p_outputByteBuffer:
//		This points to an array of bytes supplied by the
//		caller which is to be filled with run code
//		byte values based on the input pixels.
//
//	p_inout_numOutputBytesFilled:
//	This points to a counter of the number of
//		bytes in the caller's output byte buffer
//		which are already filled.  The method
//		will begin writing at the first byte after
//		the last one indicated by this counter. The
//		method will update the counter to refer to
//		any newly added bytes before this method
//		returns.
//
// SIDE EFFECTS:
//	Changes the following state variables:
//		MWWSIRLE_bool_runCompleted
//		MWWSIRLE_currentHighNibbleValue
//
// RETURNS:
//
//	MWRes_Success:
//		The run was encoded into the output buffer
//
//	MWRes_DataStructureFull:
//		There was not enough room in the output buffer
//
//	MWRes_BadCallingOrder:
//		There current run was not marked as completed.
//====================================================================
*/

TMW_ResultCode MWWSIRLE_Internal_encodeRun
(
	TFourByteInteger_Unsigned maxOutputBytes,
	TByte_Unsigned* p_outputByteBuffer,
	TFourByteInteger_Unsigned* p_inout_numOutputBytesFilled
)
{
	TFourByteInteger_Unsigned numBytesUsed;

	/*
	//##############################################################
	// Calculate some values
	//##############################################################
	*/

	numBytesUsed = *p_inout_numOutputBytesFilled;

	/*
	//##############################################################
	// The run may be longer than maximum allowed length, in which
	// case we use a loop to break it into run codes
	//##############################################################
	*/

	while (	MWWSIRLE_currentRunLengthRemaining > 0)
	{

		/*
		//##############################################################
		// High nibble change code needed?
		//##############################################################
		*/

		if
		(
			(MWWSIRLE_currentRunPixelValue & 0xF0) !=
			MWWSIRLE_currentPixelHighNibbleValue
		)
		{
			/* Is there enough room? */
			if (numBytesUsed >= maxOutputBytes)
			{
				return MWRes_DataStructureFull;
			}

			/* Write out code */
			p_outputByteBuffer[numBytesUsed] =
				0xF0 | ((MWWSIRLE_currentRunPixelValue >> 4) & 0x0F);

			/* Record new high nibble */
			MWWSIRLE_currentPixelHighNibbleValue = MWWSIRLE_currentRunPixelValue & 0xF0;

			/* One more byte used */
			*p_inout_numOutputBytesFilled = *p_inout_numOutputBytesFilled + 1;
			numBytesUsed = 	*p_inout_numOutputBytesFilled;

		}

		/*
		//##############################################################
		// Branch based on run-length
		//##############################################################
		*/
		if (MWWSIRLE_currentRunLengthRemaining > 256)
		{
			TFourByteInteger_Unsigned runLength;

			/* Max run length is 65536 */
			if (MWWSIRLE_currentRunLengthRemaining > 65536)
			{
				runLength = 65536;
			}
			else
			{
				runLength = MWWSIRLE_currentRunLengthRemaining;
			}

			/*
			// Code is 0xD<pixelValueLowNibble> <2byteRunLengthInLOHIOrder>
			// Is there enough room for 3 bytes?
			*/
			if (numBytesUsed >= maxOutputBytes-2)
			{
				return MWRes_DataStructureFull;
			}


			/* Code and low nibble of pixelValue */
			p_outputByteBuffer[numBytesUsed] =
				0xD0 | (MWWSIRLE_currentRunPixelValue & 0x0F);

			/* Low order byte of run length - 1 */
			p_outputByteBuffer[numBytesUsed +1] =
				(TByte_Unsigned) ((runLength-1) % 256);

			/* High order byte of run length - 1 */
			p_outputByteBuffer[numBytesUsed +2] =
				(TByte_Unsigned) ((runLength-1) / 256);

			/* Used 3 bytes */
			numBytesUsed += 3;

			/* Consumed some run length */
			MWWSIRLE_currentRunLengthRemaining -= runLength;

		}
		else if (MWWSIRLE_currentRunLengthRemaining > 13)
		{
			/*
			// Code is 0xE<pixelValueLowNibble>
			// Is there enough room for 2 bytes?
			*/
			if (numBytesUsed >= maxOutputBytes-1)
			{
				return MWRes_DataStructureFull;
			}

			/* Code and low nibble of pixelValue */
			p_outputByteBuffer[numBytesUsed] =
				0xE0 | (MWWSIRLE_currentRunPixelValue & 0x0F);

			/* 1 Byte run length - 1 */
			p_outputByteBuffer[numBytesUsed +1] =
				(TByte_Unsigned) (MWWSIRLE_currentRunLengthRemaining -1);

			numBytesUsed += 2;

			/* Consumed all of run length */
			MWWSIRLE_currentRunLengthRemaining = 0;

		}
		else if (MWWSIRLE_currentRunLengthRemaining > 0)
		{
			/*
			// Code is <runLength-1 left shift by 4><pixelValueLowNibble>
			// Is there enough room for 1 byte?
			*/
			if (numBytesUsed >= maxOutputBytes)
			{
				return MWRes_DataStructureFull;
			}

			/* 4 bit RunLength-1 in high nibble, low nibble of pixel value in low nibble */
			p_outputByteBuffer[numBytesUsed] =
				(((TByte_Unsigned) MWWSIRLE_currentRunLengthRemaining-1) << 4) |
				(MWWSIRLE_currentRunPixelValue & 0x0F);

			/* One byte used */
			numBytesUsed += 1;

			/* Consumed all of run length */
			MWWSIRLE_currentRunLengthRemaining = 0;
		}
		else
		{
			/* Can't encode run of length 0 */
			return MWRes_BadCallingOrder;
		}

		/*
		//##############################################################
		// Since we didn't run out of buffer space before writing out
		// a run code, increase the amount of output buffer space used
		//##############################################################
		*/
		*p_inout_numOutputBytesFilled = numBytesUsed;

		/* And high nibble is whatever it is for the pixel value of the run written */
		MWWSIRLE_currentPixelHighNibbleValue = MWWSIRLE_currentRunPixelValue & 0xF0;

		if (MWWSIRLE_currentRunLengthRemaining == 0)
		{
			/* All the run was output, so no completed run anymore. */
			MWWSIRLE_bool_runCompleted = 0;
		}

	} /* Loop around and see if there is any more run length remaining */

	/*
	//##############################################################
	// Success, all of run fit into the output
	//##############################################################
	*/
	return MWRes_Success;

}


//--------------------------------------------------------------------

TMW_ResultCode MWWSIRLE_Internal_decodeRun
(
	TFourByteInteger_Unsigned numBytes,
	TByte_Unsigned* p_bytes,
	TFourByteInteger_Unsigned* p_inout_numBytesConsumed
)
{
	TFourByteInteger_Unsigned numBytesLeft;
	TFourByteInteger_Unsigned numBytesConsumedThisCode;

	/*
	//##############################################################
	// Check how many bytes are left
	//##############################################################
	*/

	while (1)
	{
		/*
		//##############################################################
		// How many bytes are left?
		//##############################################################
		*/

		/* No bytes read yet this code */
		numBytesConsumedThisCode = 0;

		if (*p_inout_numBytesConsumed >= numBytes)
		{
			// No more input
			MWWSIRLE_currentRunLengthRemaining = 0;
			return MWRes_Success;
		}
		else
		{
			numBytesLeft = numBytes - (*p_inout_numBytesConsumed);
		}

		/*
		//##############################################################
		// Test byte for high nibble code
		//##############################################################
		*/

		if ((p_bytes[*p_inout_numBytesConsumed] & 0xF0) == 0xF0)
		{

			/* Low nibble gives new high nibble code */
			MWWSIRLE_currentPixelHighNibbleValue =
				(p_bytes[*p_inout_numBytesConsumed] & 0x0F) << 4;

			/* Consumed byte */
			*p_inout_numBytesConsumed = (*p_inout_numBytesConsumed) + 1;
		}
		else
		{
			if ((p_bytes[*p_inout_numBytesConsumed] & 0xF0) == 0xE0)
			{
				/* Enough bytes?*/
				if (numBytesLeft < 2)
				{
					return MWRes_InputBufferExhausted;
				}

				/* Low nibble is pixel value low nibble */
				MWWSIRLE_currentRunPixelValue =
					MWWSIRLE_currentPixelHighNibbleValue;

				MWWSIRLE_currentRunPixelValue |=
					(p_bytes[*p_inout_numBytesConsumed] & 0x0F);

				/* One byte run length-1 follows in LOHI order*/
				MWWSIRLE_currentRunLengthRemaining = p_bytes[*(p_inout_numBytesConsumed) + 1];
				MWWSIRLE_currentRunLengthRemaining += 1;

				/*Consumed 2 bytes */
				numBytesConsumedThisCode = 2;

			}
			else if ((p_bytes[*p_inout_numBytesConsumed] & 0xF0) == 0xD0)
			{

				/* Enough bytes?*/
				if (numBytesLeft < 3)
				{
					return MWRes_InputBufferExhausted;
				}

				/* Low nibble is pixel value low nibble */
				MWWSIRLE_currentRunPixelValue =
					MWWSIRLE_currentPixelHighNibbleValue;

				MWWSIRLE_currentRunPixelValue |=
					(p_bytes[*p_inout_numBytesConsumed] & 0x0F);

				/* Two byte run length-1 follows in LOHI order*/
				MWWSIRLE_currentRunLengthRemaining = p_bytes[*(p_inout_numBytesConsumed) + 1];
				MWWSIRLE_currentRunLengthRemaining +=
					(256 * (TFourByteInteger_Unsigned) p_bytes[*(p_inout_numBytesConsumed) + 2]);

				MWWSIRLE_currentRunLengthRemaining += 1;


				/* 3 bytes in all consumed */
				numBytesConsumedThisCode = 3;

			}
			else
			{
				/* High nibble is run length -1 */
				MWWSIRLE_currentRunLengthRemaining =
					((p_bytes[*p_inout_numBytesConsumed] & 0xF0) >> 4) + 1;

				/* Low nibble is pixel value low nibble */
				MWWSIRLE_currentRunPixelValue =
					MWWSIRLE_currentPixelHighNibbleValue;

				MWWSIRLE_currentRunPixelValue |=
					(p_bytes[*p_inout_numBytesConsumed] & 0x0F);

				/* Consumed 1 byte */
				numBytesConsumedThisCode = 1;
			}

			/*
			//##############################################################
			// Increment the number of bytes consumed and return, as we
			// have a run
			//##############################################################
			*/

			*p_inout_numBytesConsumed =
				(*p_inout_numBytesConsumed) + numBytesConsumedThisCode;

			return MWRes_Success;

		}


	} /* Loop around until run extracted */
}

/*
//====================================================================
//====================================================================
// Public methods
//====================================================================
//====================================================================
*/

TMW_ResultCode MWWSIRLE_beginRaster()
{
	/* Reset encoder state */
	MWWSIRLE_bool_runCompleted = 0;
	MWWSIRLE_currentPixelHighNibbleValue = 0;
	MWWSIRLE_currentRunPixelValue = 0;
	MWWSIRLE_currentRunLengthRemaining = 0;

	/* Success */
	return MWRes_Success;

}


/*--------------------------------------------------------------------*/

#define NUM_BYTES_IN_DECOMPRESSION_BUFFER 1024

/*--------------------------------------------------------------------*/

TMW_ResultCode MWWSIRLE_decompressProductBodyBlock
(
	TFourByteInteger_Unsigned numBytes,
	TByte_Unsigned* p_bytes,
	TFourByteInteger_Unsigned* p_out_numBytesConsumed,
	TByte_Unsigned bool_isLastBodyBlockInProduct
)
{
	TMW_ResultCode MWRes;
	TByte_Unsigned byteBuffer[NUM_BYTES_IN_DECOMPRESSION_BUFFER];
	TFourByteInteger_Unsigned numBytesInByteBuffer;
	TByte_Unsigned bool_containsEndOfProductBodyBlock;
	TByte_Unsigned bool_containsEndOfProduct;


#ifdef PARAM_TESTING
	/*
	//##############################################################
	// Test parameters
	//##############################################################
	*/
	if
	(
		(p_bytes == NULL) ||
		(p_out_numBytesConsumed == NULL)
	)
	{
		return MWRes_NullPointer;
	}
#endif

	/*
	//##############################################################
	// Init some variables
	//##############################################################
	*/
	bool_containsEndOfProduct = 0;
	bool_containsEndOfProductBodyBlock = 0;

	/*
	//##############################################################
	// Undo RLE into a buffer
	//##############################################################
	*/

	*p_out_numBytesConsumed = 0;
	MWWSIRLE_currentRunLengthRemaining = 0;

	while
	(
		(*p_out_numBytesConsumed < numBytes)  ||
		(MWWSIRLE_currentRunLengthRemaining != 0)
	)
	{
		/*
		//################################################
		// Get new run if needed
		//################################################
		*/

		if (MWWSIRLE_currentRunLengthRemaining == 0)
		{
			MWRes = MWWSIRLE_Internal_decodeRun
			(
				numBytes,
				p_bytes,
				p_out_numBytesConsumed
			);
			if (MWRes != MWRes_Success)
			{
				return MWRes;
			}
		}

		/*
		//################################################################
		// Output run into a byte buffer we can pass to output
		//################################################################
		*/
		numBytesInByteBuffer = 0;

		if (MWWSIRLE_currentRunLengthRemaining < NUM_BYTES_IN_DECOMPRESSION_BUFFER)
		{
#ifndef HAS_MEMCPY
			/*
			// Copy bytes
			*/
			for (numBytesInByteBuffer = 0; numBytesInByteBuffer < MWWSIRLE_currentRunLengthRemaining; numBytesInByteBuffer ++)
			{
				byteBuffer[numBytesInByteBuffer] = MWWSIRLE_currentRunPixelValue;
			}
#else
			/*
			// Set bytes
			*/
			memset
			(
				byteBuffer,
				MWWSIRLE_currentRunPixelValue,
				MWWSIRLE_currentRunLengthRemaining
			);
			numBytesInByteBuffer = MWWSIRLE_currentRunLengthRemaining;
#endif
			// All of run used up
			MWWSIRLE_currentRunLengthRemaining = 0;

		}
		else
		{
#ifndef HAS_MEMCPY
			/*
			// Copy bytes
			*/
			for (numBytesInByteBuffer = 0; numBytesInByteBuffer < NUM_BYTES_IN_DECOMPRESSION_BUFFER; numBytesInByteBuffer ++)
			{
				byteBuffer[numBytesInByteBuffer] = MWWSIRLE_currentRunPixelValue;
			}
#else
			/*
			// Set bytes
			*/
			memset
			(
				byteBuffer,
				MWWSIRLE_currentRunPixelValue,
				MWWSIRLE_currentRunLengthRemaining
			);
			numBytesInByteBuffer = MWWSIRLE_currentRunLengthRemaining;
#endif

			// Run length remaining removed
			MWWSIRLE_currentRunLengthRemaining -= NUM_BYTES_IN_DECOMPRESSION_BUFFER;

		}

		/*
		//##############################################################
		// Output the bytes
		//##############################################################
		*/

		if
		(
			(*p_out_numBytesConsumed == numBytes) &&
			(MWWSIRLE_currentRunLengthRemaining == 0)
		)
		{
			/*
			// Its the end of the product body block in this output if
			// all the input was consumed, and there is no run length
			// remaining to be expressed.
			*/
				bool_containsEndOfProductBodyBlock = 1;

			/*
			// Its also the end of the product if the product ends with
			// this product block.
			*/
			if (bool_isLastBodyBlockInProduct != 0)
			{
				bool_containsEndOfProduct = 1;
			}
		}

		/* Output bytes */
		MWRes = MWOutputEncoder_outputEncodedBytes
		(
			numBytesInByteBuffer,
			byteBuffer,
			bool_containsEndOfProductBodyBlock,
			bool_containsEndOfProduct
		);
		if (MWRes != MWRes_Success)
		{
			return MWRes;
		}

	} // Loop around to express more pixels

	/*
	//##############################################################
	// Done
	//##############################################################
	*/
	return MWRes_Success;

}

/*--------------------------------------------------------------------*/

TMW_ResultCode MWWSIRLE_decompressPartialProductBodyBlock
(
	TFourByteInteger_Unsigned numBytes,
	TByte_Unsigned* p_bytes,
	TFourByteInteger_Unsigned* p_out_numBytesConsumed,
	TByte_Unsigned bool_containsEndOfProductBodyBlock,
	TByte_Unsigned bool_isLastBodyBlockInProduct
)
{
	TMW_ResultCode MWRes;
	TByte_Unsigned byteBuffer[NUM_BYTES_IN_DECOMPRESSION_BUFFER];
	TFourByteInteger_Unsigned numBytesInByteBuffer;
	TByte_Unsigned bool_outputContainsEndOfProductBodyBlock;
	TByte_Unsigned bool_containsEndOfProduct;


#ifdef PARAM_TESTING
	/*
	//##############################################################
	// Test parameters
	//##############################################################
	*/
	if
	(
		(p_bytes == NULL) ||
		(p_out_numBytesConsumed == NULL)
	)
	{
		return MWRes_NullPointer;
	}
#endif

	/*
	//##############################################################
	// Init some variables
	//##############################################################
	*/
	bool_containsEndOfProduct = 0;
	bool_outputContainsEndOfProductBodyBlock = 0;

	/*
	//##############################################################
	// Undo RLE into a buffer
	//##############################################################
	*/

	*p_out_numBytesConsumed = 0;

	while
	(
		(*p_out_numBytesConsumed < numBytes)  ||
		(MWWSIRLE_currentRunLengthRemaining != 0)
	)
	{
		/*
		//################################################
		// Get new run if needed
		//################################################
		*/

		if (MWWSIRLE_currentRunLengthRemaining == 0)
		{
			MWRes = MWWSIRLE_Internal_decodeRun
			(
				numBytes,
				p_bytes,
				p_out_numBytesConsumed
			);
			if (MWRes != MWRes_Success)
			{
				return MWRes;
			}
		}

		/*
		//################################################################
		// Output run into a byte buffer we can pass to output
		//################################################################
		*/
		numBytesInByteBuffer = 0;

		if (MWWSIRLE_currentRunLengthRemaining < NUM_BYTES_IN_DECOMPRESSION_BUFFER)
		{
#ifndef HAS_MEMCPY
			/*
			// Copy bytes
			*/
			for (numBytesInByteBuffer = 0; numBytesInByteBuffer < MWWSIRLE_currentRunLengthRemaining; numBytesInByteBuffer ++)
			{
				byteBuffer[numBytesInByteBuffer] = MWWSIRLE_currentRunPixelValue;
			}
#else
			/*
			// Set bytes
			*/
			memset
			(
				byteBuffer,
				MWWSIRLE_currentRunPixelValue,
				MWWSIRLE_currentRunLengthRemaining
			);
			numBytesInByteBuffer = MWWSIRLE_currentRunLengthRemaining;
#endif

		}
		else
		{
#ifndef HAS_MEMCPY
			/*
			// Copy bytes
			*/
			for (numBytesInByteBuffer = 0; numBytesInByteBuffer < NUM_BYTES_IN_DECOMPRESSION_BUFFER; numBytesInByteBuffer ++)
			{
				byteBuffer[numBytesInByteBuffer] = MWWSIRLE_currentRunPixelValue;
			}
#else
			/*
			// Set bytes
			*/
			memset
			(
				byteBuffer,
				MWWSIRLE_currentRunPixelValue,
				MWWSIRLE_currentRunLengthRemaining
			);
			numBytesInByteBuffer = MWWSIRLE_currentRunLengthRemaining;
#endif

			// Run length remaining removed
			MWWSIRLE_currentRunLengthRemaining -= NUM_BYTES_IN_DECOMPRESSION_BUFFER;

		}

		/*
		//##############################################################
		// Output the bytes
		//##############################################################
		*/

		if
		(
			(*p_out_numBytesConsumed == numBytes) &&
			(MWWSIRLE_currentRunLengthRemaining == 0) &&
			(bool_containsEndOfProductBodyBlock != 0)
		)
		{
			/*
			// Its the end of the product body block in this output if
			// all the input was consumed, and there is no run length
			// remaining to be expressed.
			*/
			bool_outputContainsEndOfProductBodyBlock = 1;

			/*
			// Its also the end of the product if the product ends with
			// this product block.
			*/
			if (bool_isLastBodyBlockInProduct != 0)
			{
				bool_containsEndOfProduct = 1;
			}
		}

		/* Output bytes */
		MWRes = MWOutputEncoder_outputEncodedBytes
		(
			numBytesInByteBuffer,
			byteBuffer,
			bool_outputContainsEndOfProductBodyBlock,
			bool_containsEndOfProduct
		);
		if (MWRes != MWRes_Success)
		{
			return MWRes;
		}

	} // Loop around to express more pixels

	/*
	//##############################################################
	// Done
	//##############################################################
	*/
	return MWRes_Success;

}


/*--------------------------------------------------------------------*/

TMW_ResultCode MWWSIRLE_compressPixels
(
	/* Input pixel buffer parameters */
	TFourByteInteger_Unsigned numPixels,
	TByte_Unsigned* p_Pixels,
	TFourByteInteger_Unsigned* p_inout_numPixelsConsumed,
	/* Output run code parameters */
	TFourByteInteger_Unsigned maxOutputBytes,
	TByte_Unsigned* p_outputByteBuffer,
	TFourByteInteger_Unsigned* p_inout_numOutputBytesFilled
)
{
	TMW_ResultCode MWRes;

#ifdef PARAM_TESTING
	/*
	//##############################################################
	// Test parameters
	//##############################################################
	*/
	if
	(
		(p_Pixels == NULL) ||
		(p_inout_numPixelsConsumed == NULL) ||
		(p_outputByteBuffer == NULL) ||
		(p_inout_numOutputBytesFilled == NULL)
	)
	{
		return MWRes_NullPointer;
	}
#endif

	/*
	//##############################################################
	// Consume input while there is output remaining and
	// input remaining
	//##############################################################
	*/
	while
	(
		(*p_inout_numOutputBytesFilled < maxOutputBytes) &&
		(*p_inout_numPixelsConsumed < numPixels)
	)
	{

		/*
		//##############################################################
		// If we have a remaining run state to be encoded,
		// encode as much as will fit in the output.
		//##############################################################
		*/
		if (MWWSIRLE_bool_runCompleted != 0)
		{

			/*
			// We must continue to encode the remainder of this
			// run.
			*/
			MWRes = MWWSIRLE_Internal_encodeRun
			(
				maxOutputBytes,
				p_outputByteBuffer,
				p_inout_numOutputBytesFilled
			);
			if (MWRes != MWRes_Success)
			{
				/*
				// Either error, or the output buffer
				// filled before we could complete it all.
				*/
				return MWRes;
			}

		}

		/*
		//##############################################################
		// Accumulate any new data into any existing
		// run, if they share the same pixel value
		//##############################################################
		*/
		MWRes = MWWSIRLE_Internal_accumulateRun
		(
			numPixels,
			p_Pixels,
			p_inout_numPixelsConsumed
		);
		if (MWRes != MWRes_Success)
		{
			/* An error occurred */
			return MWRes;
		}


	} /* loop around to handle more input or output */

	/*
	//##############################################################
	// Done
	//##############################################################
	*/
	if (*p_inout_numPixelsConsumed == numPixels)
	{
		return MWRes_Success;
	}
	else if (*p_inout_numPixelsConsumed > numPixels)
	{
		return MWRes_CodeFault;
	}
	else
	{
		/*
		// Output buffer is full, must write some out before encoding
		// more runs
		*/
		return MWRes_DataStructureFull;
	}
}


/*--------------------------------------------------------------------*/

TMW_ResultCode MWWSIRLE_flushRun
(
	/* Output run code parameters */
	TFourByteInteger_Unsigned maxOutputBytes,
	TByte_Unsigned* p_outputByteBuffer,
	TFourByteInteger_Unsigned* p_inout_numOutputBytesFilled
)
{
	TMW_ResultCode MWRes;

#ifdef PARAM_TESTING
	/*
	//##############################################################
	// Test parameters
	//##############################################################
	*/
	if
	(
		(p_outputByteBuffer == NULL) ||
		(p_inout_numOutputBytesFilled == NULL)
	)
	{
		return MWRes_NullPointer;
	}
#endif

	/*
	//##############################################################
	// Write out any completed run information
	//##############################################################
	*/

	/* Run is completed if there is any accumulated run length */
	if
	(
		(MWWSIRLE_bool_runCompleted == 0) &&
		(MWWSIRLE_currentRunLengthRemaining > 0)
	)
	{
		MWWSIRLE_bool_runCompleted = 1;
	}

	/* Write out the run */
	if (MWWSIRLE_currentRunLengthRemaining > 0)
	{
		/*
		// We must continue to encode the remainder of this
		// run.
		*/
		MWRes = MWWSIRLE_Internal_encodeRun
		(
			maxOutputBytes,
			p_outputByteBuffer,
			p_inout_numOutputBytesFilled
		);
		if (MWRes != MWRes_Success)
		{
			/*
			// Either error, or the output buffer
			// filled before we could complete it all.
			*/
			return MWRes;
		}

	}

	/*
	//##############################################################
	// Done
	//##############################################################
	*/
	return MWRes_Success;

}

/*
//--------------------------------------------------------------------
*/

TMW_ResultCode MWWSIRLE_decompressPartialProductBodyBlockToRaster
(
	TFourByteInteger_Unsigned numBytes,
	TByte_Unsigned* p_bytes,
	TFourByteInteger_Unsigned* p_out_numBytesConsumed,
	TFourByteInteger_Unsigned numPixelsInDestination,
	TByte_Unsigned* p_destinationRaster,
	TFourByteInteger_Unsigned* p_inout_numPixelsWritten
)
{
	TMW_ResultCode MWRes;
	TFourByteInteger_Unsigned numBytesCopied;
	TFourByteInteger_Unsigned runLengthToCopy;


#ifdef PARAM_TESTING
	/*
	//##############################################################
	// Test parameters
	//##############################################################
	*/
	if
	(
		(p_bytes == NULL) ||
		(p_out_numBytesConsumed == NULL) ||
		(p_destinationRaster == NULL)
	)
	{
		return MWRes_NullPointer;
	}
#endif

	/*
	//##############################################################
	// Undo RLE codes into the raster
	//##############################################################
	*/

	*p_out_numBytesConsumed = 0;

	while
	(
		(*p_out_numBytesConsumed < numBytes)  ||
		(MWWSIRLE_currentRunLengthRemaining != 0)
	)
	{
		/*
		//################################################
		// Get new run if needed
		//################################################
		*/

		if (MWWSIRLE_currentRunLengthRemaining == 0)
		{
			MWRes = MWWSIRLE_Internal_decodeRun
			(
				numBytes,
				p_bytes,
				p_out_numBytesConsumed
			);
			if (MWRes != MWRes_Success)
			{
				return MWRes;
			}
		}

		/*
		//################################################################
		// Output run into the raster
		//################################################################
		*/

		if (MWWSIRLE_currentRunLengthRemaining > (numPixelsInDestination - (*p_inout_numPixelsWritten)) )
		{
			runLengthToCopy = numPixelsInDestination - (*p_inout_numPixelsWritten);
		}
		else
		{
			runLengthToCopy = MWWSIRLE_currentRunLengthRemaining;
		}


#ifndef HAS_MEMCPY
		/*
		// Copy bytes
		*/
		for (numBytesCopied = 0; numBytesCopied < runLengthToCopy; numBytesCopied ++)
		{
			p_destinationRaster[(*p_inout_numPixelsWritten) + numBytesCopied] = MWWSIRLE_currentRunPixelValue;
		}
#else
		/*
		// Set bytes
		*/
		memset
		(
			p_destinationRaster[(*p_inout_numPixelWritten)],
			MWWSIRLE_currentRunPixelValue,
			runLengthToCopy
		);
#endif

		/*
		//##############################################################
		// Update counters
		//##############################################################
		*/

		/* Update number of pixels written */
		*p_inout_numPixelsWritten = (*p_inout_numPixelsWritten) + runLengthToCopy;

		/* reduce run length by number written */
		MWWSIRLE_currentRunLengthRemaining -= runLengthToCopy;

		/*
		//##############################################################
		// Finished if all possible pixels written
		//##############################################################
		*/

		if ( (*p_inout_numPixelsWritten) >= numPixelsInDestination)
		{
			if (MWWSIRLE_currentRunLengthRemaining > 0)
			{
				return MWRes_DataStructureFull;
			}
			else
			{
				return MWRes_Success;
			}
		}

	} // Loop around to express more pixels

	/*
	//##############################################################
	// Done
	//##############################################################
	*/
	return MWRes_Success;

}

