/*
====================================================================
// MW_Transmogrifier.h:
//
// This header defines the public interface of an abstract data type
// that encapsulates the data and logic for transforming an input
// format product to an output format product.
//
// Author: Damon M. Hill
// Copyright � 2005 WSI Corporation.  All rights reserved.
//
====================================================================
*/
#ifndef MW_TRANSMOGRIFIER_HEADER
#define MW_TRANSMOGRIFIER_HEADER

/*
====================================================================
// Required includes
====================================================================
*/
#include "MW_DataFieldTypes.h"

/*
====================================================================
// WSIMW_OutputFunction:
//
// Format of the function which is passed to the transmogrifier
// which outputs framed bytes over the output pipe.
//
// The output bytes will be in the frame format specified in the
// receiver output format section of the WSI for Sirius receiver
// system specification.
//
// PARAMETERS:
//
// The first parameter holds the number of bytes to be output.
//
// The second parameter points to the contiguous array of bytes
// to be output.
//
// RETURNS:
//
// If the callback returns anything other than MWRes_Success,
// then the pipe output is considered to have been a failure and
// the return code is returned out of the output system to its
// callers. This effectively cancels processing on the current
// product.
//
====================================================================
*/
typedef TMW_ResultCode (TWSIMW_OutputFunction) (TFourByteInteger_Unsigned,
                                                TByte_Unsigned*);

/*
====================================================================
// MWTransmogrifier_setOutputCallback:
//
// This method is used to set the address of the callback function by
// which the dll directs output bytes to the output pipe of the
// system architecture.
//
// It must be called successfully at least once before calling
// MWTransmogrifier_ProcessProduct.
//
// PARAMETERS:
//
//	p_outputFunction:
//		Pass in the address of a function of the declared format
//		which accepts bytes to be written out over the output pipe.
//
//		The bytes passed to the function will be in the frame
//		format specified for output of WSI products from the
//		Sirius receiver.
//
// RETURNS:
//
//	MWRes_Success:
//		If the output function has been set.
//
//	MWRes_NullPointer:
//		If the function pointer was NULL
//
====================================================================
*/
TMW_ResultCode MWTransmogrifier_setOutputCallback
(
	TWSIMW_OutputFunction* p_outputFunction
);


/*
====================================================================
// MWTransmogrifier_processProduct:
//
// Given a product, performs the transmogrification that is specified
// within it.  The result is sent within this function call to the
// output channel.
//
// PARAMETERS:
//
//	numBytesInProduct:
//		The number of bytes in the product stream
//
//	p_bytesOfProduct:
//		Array holding the entire product stream
//
// RETURNS:
//
//	MWRes_Success:
//		On succesful transfer to output
//
//	MWRes_ProductNotExpectedSize
//		If the product was not the size expected from format
//		constraints.
//
//	MWRes_UnrecognizedTransmogrificationType:
//		If the source compression, destination compression pairing
//		was not one provided for by this code build.
//
====================================================================
*/
TMW_ResultCode MWTransmogrifier_processProduct
(
	TFourByteInteger_Unsigned numBytesInProduct,
	TByte_Unsigned* p_bytesOfProduct
);


/* End header wrapper	*/
#endif


