/*
//====================================================================
// MW_OutputEncoder.c
//
// Implmentation of the output encoder interface ADT.
//
// Author: Damon M. Hill
// Copyright � 2005 WSI Corporation.  All rights reserved.
//
//====================================================================
*/


/*
//====================================================================
// Required includes
//====================================================================
*/

#include "MW_OutputEncoder.h"
#include "MW_WSIRLE.h"
#include "MW_OutputDPUEncoder.h"

/*
//====================================================================
// Data
//====================================================================
*/

static TMW_CompressionType MWOutputEncoder_outputEncoder = None_MWCompressionType;

#define OUTPUT_BUFFER_LENGTH 1024

/*
//====================================================================
// Public methods
//====================================================================
*/

/*--------------------------------------------------------------------*/

TMW_ResultCode MWOutputEncoder_startNewProduct
(
	TTwoByteInteger_Unsigned productID
)
{

	// Tell DPU encoder to start product
	return MWOutputDPUEncoder_startNewProduct (productID);
}

//--------------------------------------------------------------------

TMW_ResultCode MWOutputEncoder_setOutputEncoder
(
	TMW_CompressionType outputFormat
)
{
	/* Rule out encoders not in this library */
	switch (outputFormat)
	{
	case DEFLATED_MWCompressionType:
	case WSI_RLE_DEFLATE_MWCompressionType:
	case WeatherHuffman_MWCompressionType:
		return MWRes_UnrecognizedTransmogrificationType;
		break;
	default:
		// Passthrough for others
		break;
	}

	/* set value*/
	MWOutputEncoder_outputEncoder = outputFormat;

	/* done*/
	return MWRes_Success;
}

/*--------------------------------------------------------------------*/

TMW_ResultCode MWOutputEncoder_startNewEncodingBlock()
{
	// Reset the chosen decoder
	if (MWOutputEncoder_outputEncoder == WSI_RLE_MWCompressionType)
	{
		return MWWSIRLE_beginRaster();
	}
	else
	{
		return MWRes_Success;
	}
}


/*--------------------------------------------------------------------*/

TMW_ResultCode MWOutputEncoder_outputUnencodedBytes
(
	TFourByteInteger_Unsigned numBytes,
	TByte_Unsigned* p_bytes,
	TByte_Unsigned bool_isEndOfProduct
)
{
	TMW_ResultCode MWRes;

	/*
	//##############################################################
	// Test the parameters
	//##############################################################
	*/
	if (p_bytes == NULL)
	{
		return MWRes_NullPointer;
	}

	/*
	//##############################################################
	// Just pass directly to DPU output
	//##############################################################
	*/

	MWRes = MWOutputDPUEncoder_outputBytes
	(
		numBytes,
		p_bytes,
		bool_isEndOfProduct
	);

	/*
	//##############################################################
	// Done
	//##############################################################
	*/

	return MWRes;



}

/*--------------------------------------------------------------------*/

TMW_ResultCode MWOutputEncoder_outputEncodedBytes
(
	TFourByteInteger_Unsigned numBytes,
	TByte_Unsigned* p_bytes,
	TByte_Unsigned bool_isEndOfEncodingBlock,
	TByte_Unsigned bool_isEndOfProduct
)
{
	TByte_Unsigned outputBuffer[OUTPUT_BUFFER_LENGTH];

	TMW_ResultCode MWRes=MWRes_OperationFailed;
	TFourByteInteger_Unsigned numBytesWrittenThisPass;
	TFourByteInteger_Unsigned numBytesWritten;
	TFourByteInteger_Unsigned numBytesInOutput;

	/*
	//##############################################################
	// Test the parameters
	//##############################################################
	*/
	if (p_bytes == NULL)
	{
		return MWRes_NullPointer;
	}

	if ((bool_isEndOfProduct != 0) && (bool_isEndOfEncodingBlock == 0))
	{
		// Can't end product without ending encoding
		return MWRes_BadParamValue;
	}

	/*
	//##############################################################
	// Route bytes to the decoder
	//##############################################################
	*/

	if (MWOutputEncoder_outputEncoder == None_MWCompressionType)
	{
		/* Just pass directly to DPU output */
		MWRes = MWOutputDPUEncoder_outputBytes
		(
			numBytes,
			p_bytes,
			bool_isEndOfProduct
		);
		return MWRes;
	}
	else if (MWOutputEncoder_outputEncoder == WSI_RLE_MWCompressionType)
	{
		/*
		##############################################
		Pass through the WSI RLE encoder
		##############################################
		*/

		numBytesWritten = 0;

		while (numBytesWritten < numBytes)
		{
			/* run input through RLE encoder */
			numBytesWrittenThisPass = 0;
			numBytesInOutput = 0;


			MWRes = MWWSIRLE_compressPixels
			(
				numBytes - numBytesWritten,
				p_bytes + numBytesWritten,
				&numBytesWrittenThisPass,
				OUTPUT_BUFFER_LENGTH,
				outputBuffer,
				&numBytesInOutput
			);
			if ((MWRes != MWRes_Success) && (MWRes != MWRes_DataStructureFull))
			{
				return MWRes;
			}

			numBytesWritten += numBytesWrittenThisPass;

			/* Output RLE results to the DPU encoder */
			MWRes = MWOutputDPUEncoder_outputBytes
			(
				numBytesInOutput,
				outputBuffer,
				0 // Need to flush buffer first
			);
			if (MWRes != MWRes_Success)
			{
				return MWRes;
			}

		} /* process more input until done */

		/*
		##############################################
		Flush the RLE Encoder if this is the end
		of the encoding block
		##############################################
		*/
		if (bool_isEndOfEncodingBlock != 0)
		{
			MWRes = MWRes_DataStructureFull;

			while (MWRes == MWRes_DataStructureFull)
			{
				numBytesInOutput = 0;

				/* flush RLE into output buffer */
				MWRes = MWWSIRLE_flushRun
				(
					OUTPUT_BUFFER_LENGTH,
					outputBuffer,
					&numBytesInOutput
				);

				/* write from output buffer to the DPU encoder*/
				MWRes = MWOutputDPUEncoder_outputBytes
				(
					numBytesInOutput,
					outputBuffer,
					0
				);

			}
		}

		/* Test for error*/
		if (MWRes != MWRes_Success)
		{
			return MWRes;
		}

		/*
		##############################################
		If this is the end of the product, make sure
		any current frame is ended.
		##############################################
		*/
		if (bool_isEndOfProduct != 0)
		{
			MWRes = MWOutputDPUEncoder_outputBytes
			(
				0,
				outputBuffer,
				1 // End the frame and mark it as the last in the product
			);
		}

		/* Done */
		return MWRes;

	}
	else
	{
		/* Um, we don't have a encoder for that. It should have never
			been accepted as a configuration option */

		return MWRes_CodeFault;
	}


}

