/******************************************************************************/
/*                Copyright (c) Sirius XM Satellite Radio, Inc.               */
/*                            All Rights Reserved                             */
/*      Licensed Materials - Property of Sirius XM Satellite Radio, Inc.      */
/******************************************************************************/
/***************************************************************************//**
*
* \file sxm_timers.c
* \author Eugene Lopatukhin
* \date 6/5/2014
*
* Implementation file for general purpose low resolution timers service.
*******************************************************************************/

#define DEBUG_TAG "timers"

#include <sxm_build.h>
#include <util/sxm_msg_queue_incl.h>
#include <util/sxm_time_internal.h>
#include <util/sxm_timers_incl.h>
#include <util/sxm_common_internal.h>
#include <util/sxm_noname_internal.h>

/** Internal Timers Logging macro definition */
#ifndef TM_LOG
#define TM_LOG(_f, ...) // sxm_file_log("%s: " _f "\n", __FUNCTION__, ##__VA_ARGS__)
#endif

/** Defines for supported timer flags/masks */
typedef enum {
    SXM_TIMER_NONE     = ((ushort)0x00), //!< Inactive rearming timer
    SXM_TIMER_ACTIVE   = ((ushort)0x01), //!< Active timer
    SXM_TIMER_PAUSED   = ((ushort)0x02), //!< Paused timer
    SXM_TIMER_ONE_SHOT = ((ushort)0x04)  //!< One-shot timer
} SXMTimerFlags;

/** Timers info structure */
typedef struct {
    uint timerHandle; //!< Unique timer handle
    uint timerInterval; //!< Timer interval in milliseconds
    SXMTimerCallback timerCallback; //!< Timer expiration callback
    void *timerCallbackArg; //!< Timer callback argument
    uint flags; //!< Timer flags (\ref SXMTimerFlags)
    /** Keeps state related data */
    struct {
        /** The time of the last shot of the timer (not-paused state) */
        SXMTimeSpec lastShot;
        /** Keeps number of milliseconds past from the last shot right before
         * the pause request. Valid during the paused state only.
         */
        uint pastTimeout;
    } sData;
} SXMTimerInfoStruct;

/** Timers service structure */
typedef struct {
    /** processing thread */
    pthread_t tid;
    /** Protection from multiple tread simultaneous access */
    pthread_mutex_t dataMutex;
    /** message queue for timers service thread */
    MSG_QUEUE_STRUCT queue;
    /** timer registration list */
    SXMList timerList;
    /** handle generator */
    uint lastTimerHandle;
    /** Keeps next time of the timers' evaluation; if no scheduled time the 
     * value equals to \ref gNullTimeSpec
     */
    SXMTimeSpec nextTick;
    /** Indicates that waking up event has been issued, but not processed yet */
    BOOL isWakingUp;
} SXMTimersService;

/** types of timers service messages */
typedef enum {
    /** Stops timers processing message */
    TIMERS_STOP,
    /** Forcibly wake up the thread to invalidate timers */
    TIMERS_WAKEUP
} SXMTimersServiceEventType;

enum {
    /** Timer service thread message queue size */
    TIMERS_MSG_QUEUE_SIZE = 4
};

/** Timer's thread name */
#define TIMERS_THREAD_NAME   	"TIMERS_THREAD"

/** Timer's thread associated queue */
#define TIMERS_QUEUE_NAME       "TIMERS_QUEUE"

/** timers service status */
static SXMStatus status = {SXM_SERVICE_STOPPED, SXM_SUBS_NONE};

#define IS_SERVICE_STOPPED() (status.service == SXM_SERVICE_STOPPED)
#define IS_SERVICE_RUNNING_OK() (status.service == SXM_SERVICE_OK)

/** Start and stop protection mutex */
static pthread_mutex_t apimutex = PTHREAD_MUTEX_INITIALIZER;

/** Timers service instance */
static SXMTimersService *service = NULL;

/** Message queue mutex */
static pthread_mutex_t queueMutex = PTHREAD_MUTEX_INITIALIZER;

/** The null time-spec value which is used as either null-like or infinite
 * timer value later in this code.
 */
static const SXMTimeSpec gNullTimeSpec = {0, 0L};

/** \name Private functions
 * @{
 */
static void *timers_thread(void *pData);
static SXMResultCode timers_allocate_service_resources(SXMTimersService **ppService);
static void timers_free_service_resources(SXMTimersService *pService);
static uint timers_handle_timer_tick(SXMTimersService *pService, BOOL bWokeUp);
static SXMTimerInfoStruct *timers_find(SXMTimersService *pService,
                                       uint timerHandle, SXMListEntry **ppEntry);
static int timers_compare_ms(const SXMTimeSpec *t1, const SXMTimeSpec *t2, uint *pDiff);
static void timers_adjust_past(SXMTimeSpec *t, uint msec);
static void timers_adjust_future(SXMTimeSpec *dst, const SXMTimeSpec *t, uint msec);
static void timers_time(SXMTimeSpec *t);
static int timers_start(void);
/** @} */

/** Private macros
 * @{
 */
#define DATA_LOCK(_s) pthread_mutex_lock(&(_s)->dataMutex)
#define DATA_UNLOCK(_s) pthread_mutex_unlock(&(_s)->dataMutex)
/** @} */

/***************************************************************************//**
* This function allocates some of the resource needed for service.
*
* \param[out] ppService service instance in case of success
* \return Corresponding error code
********************************************************************************/
static SXMResultCode timers_allocate_service_resources(SXMTimersService **ppService) {

    SXMResultCode rc = SXM_E_ERROR;
    SXMTimersService *pService;

    /* allocate service memory */
    pService = (SXMTimersService *)sxe_calloc(1, sizeof(SXMTimersService));
    if (pService == NULL) {
        rc = SXM_E_NOMEM;
    }
    else {
        pthread_mutexattr_t mtxAttr;

        /* initialize the generator */
        pService->lastTimerHandle = (uint)(size_t)pService;
        /* Put the timeout to infinite */
        pService->nextTick = gNullTimeSpec;
        /* Stable state */
        pService->isWakingUp = FALSE;
        /* make sure everything ok before proceeding */
        rc = sxm_queue_create(&pService->queue, TIMERS_MSG_QUEUE_SIZE, &queueMutex,
                              SXM_QUEUE_RESERVE_LEGACY, TIMERS_QUEUE_NAME, NULL);
        if (rc != SXM_E_OK) {
            sxe_free(pService);
            pService = NULL;
        }
        else {
            rc = sxm_list_create(&pService->timerList, SXM_LIST_PREALLOCATED);
            if (rc != SXM_E_OK) {
                sxm_queue_destroy(&pService->queue);
                sxe_free(pService);
                pService = NULL;
            }
            else {
                rc = SXM_E_ERROR;
                /* create the data protection mutex */
                if (pthread_mutexattr_init(&mtxAttr) == 0) {
                    pthread_mutexattr_settype(&mtxAttr, PTHREAD_MUTEX_RECURSIVE);
                    /* create protection mutex */
                    if (pthread_mutex_init(&pService->dataMutex, &mtxAttr) == 0) {
                        rc = SXM_E_OK;
                    }
                    /* free up attributes object */
                    pthread_mutexattr_destroy(&mtxAttr);
                }
                if (rc != SXM_E_OK) {
                    sxm_list_destroy(&pService->timerList);
                    sxm_queue_destroy(&pService->queue);
                    sxe_free(pService);
                    pService = NULL;
                }
            }
        }
    }

    if (rc == SXM_E_OK) {
        *ppService = pService;
    }

    return rc;
}

/***************************************************************************//**
* This function frees some of the resource needed for service.
*
* \param[in] pService service instance
********************************************************************************/
static void timers_free_service_resources(SXMTimersService *pService) {
    if (pService != NULL) {
        pthread_mutex_destroy(&pService->dataMutex);
        sxm_list_destroy(&pService->timerList);
        sxm_queue_destroy(&pService->queue);
        sxe_free(pService);
    }
}

/***************************************************************************//**
* This function starts timers service.
*
* \return SXM_E_OK on successful start error code otherwise otherwise
********************************************************************************/
static int timers_start(void) {

    SXMResultCode rc = SXM_E_OK;

    LOCK(apimutex);

    /* make sure not running before starting */
    if (IS_SERVICE_STOPPED()) {
        rc = timers_allocate_service_resources(&service);
        if (rc == SXM_E_OK) {
            SXMSem_t initCompletedSem;
            /* initialize signaling semaphore for thread initialization completion */
            if (sxm_sem_init(&initCompletedSem, 0) == SXM_E_OK) {
                /* processing thread creation */
                if (sxm_pthread_create(&service->tid, NULL, timers_thread, &initCompletedSem) == 0) {
                    /* init thread name */
                    sxe_thread_setname(service->tid, TIMERS_THREAD_NAME);
                    /* wait for thread initialization to complete */
                    rc = sxm_sem_wait(&initCompletedSem);
                    if (rc != SXM_E_OK) {
                        PLOG_UTL("Failed to wait semaphore (rc=%d), should not be a problem", rc);
                        rc = SXM_E_OK;
                    }
                }
                else {
                    timers_free_service_resources(service);
                    service = NULL;
                    rc = SXM_E_THREAD;
                }
                /* destroy init semaphore */
                sxm_sem_destroy(&initCompletedSem);
            }
            else {
                timers_free_service_resources(service);
                service = NULL;
                rc = SXM_E_RESOURCE;
            }
        }
        else {
            rc = SXM_E_NOMEM;
        }
    }
    else {
        /* service already running */
        rc = SXM_E_STATE;
    }

    UNLOCK(apimutex);

    return rc;
}

/***************************************************************************//**
* This function stops timers service.
*
* \retval SXM_E_OK Success
* \retval SXM_E_STATE Service already stopped
********************************************************************************/
int sxm_timers_stop(void) {

    SXMResultCode rc;

    LOCK(apimutex);

    /* make sure service is running */
    if (!IS_SERVICE_STOPPED()) {
        SXMListEntry *pListEntry;
        /* tell thread to kill itself */
        rc = sxm_queue_put(&service->queue, NULL, TIMERS_STOP, 0, MSG_PRIO_HIGH);
        if (rc == SXM_E_OK) {
            /* wait for thread to complete */
            pthread_join(service->tid, NULL);
            /* delete all timers */
            while ((pListEntry = sxm_list_first(&service->timerList)) != NULL) {
                SXMTimerInfoStruct *pTimer = (SXMTimerInfoStruct *)sxm_list_data(pListEntry);
                sxm_list_remove(&service->timerList, pListEntry);
                sxm_list_free(pTimer);
            }
            /* release resources */
            timers_free_service_resources(service);
            service = NULL;
        }
    }
    else {
        rc = SXM_E_STATE;
    }

    UNLOCK(apimutex);

    return rc;
}

/***************************************************************************//**
* High level processing thread for timers related messages
*
* \param[in] pData thread argument, in this case the initialization semaphore
*
* \return Thread execution result
********************************************************************************/
static void *timers_thread(void *pData) {
    uint waitTime = SXM_QUEUE_INFINITE_TIMEOUT;

    /* up and running */
    status.service = SXM_SERVICE_OK;

    /* notify service start routine that service thread initialization has completed */
    sxm_sem_post((SXMSem_t *)pData);

    while (!IS_SERVICE_STOPPED()) {
        SXMResultCode rc;
        SXMQueueMsg msg;
        TM_LOG(": waiting for the event during up-to %u msec(s)", waitTime);
        /* pend on message */
        rc = sxm_queue_get_ms(&service->queue, &msg, waitTime);
        if (rc == SXM_E_OK) {
            TM_LOG(": event %u received", msg.msgType);
            /* process message */
            switch (msg.msgType) {
                case TIMERS_STOP:
                    /* service no longer  running */
                    status.service = SXM_SERVICE_STOPPED;
                    break;
                case TIMERS_WAKEUP:
                    /* Something has changed among timers and we need to
                     * re-evaluate timers' timeouts.
                     */
                    waitTime = timers_handle_timer_tick(service, TRUE);
                    break;
                default:
                    break;
            }

            /* free message memory */
            sxm_queue_msg_release(&msg);
        }
        else if (rc == SXM_E_TIMEOUT) {
            TM_LOG(": process timeout");
            waitTime = timers_handle_timer_tick(service, FALSE);
        }
        else {
            TM_LOG(": unknown error code %d", rc);
        }
    }


    return NULL;
}

/***************************************************************************//**
* This function is used to create a timer.
*
* \param[in] timeInterval timer interval in seconds
* \param[out] pTimerHandle pointer to timer handle which gets populated when timer is created
* \param[in] timerCallbackFunction pointer to callback function called when timer fires.
* \param[in] timerCallbackArg callback argument passed to \p timerCallbackFunction during
*                             each call
*
* \retval TRUE on success
* \retval FALSE otherwise
********************************************************************************/
BOOL sxm_timers_create_timer(uint timeInterval, uint *pTimerHandle,
                             SXMTimerType timerType,
                             SXMTimerCallback timerCallbackFunction,
                             void *timerCallbackArg)
{
    return sxm_timers_create_timer_ms(timeInterval * SXM_MILLISEC_PER_SEC,
                                      pTimerHandle, timerType,
                                      timerCallbackFunction, timerCallbackArg);
}

/***************************************************************************//**
* This function is used to create a timer.
*
* \param[in] timeInterval timer interval in milliseconds
* \param[out] pTimerHandle pointer to timer handle populated when timer is created
* \param[in] timerCallbackFunction pointer to callback function which gets called when timer fires.
* \param[in] timerCallbackArg callback argument passed to \p timerCallbackFunction during
*                             each call
*
* \retval TRUE on success
* \retval FALSE otherwise
********************************************************************************/
BOOL sxm_timers_create_timer_ms(uint timeInterval, uint *pTimerHandle,
                                SXMTimerType timerType,
                                SXMTimerCallback timerCallbackFunction,
                                void *timerCallbackArg)
{
    BOOL rc = FALSE;

    // checking service status, starting if stopped
    if (IS_SERVICE_STOPPED()) {
        int startrc = timers_start();
        if ((startrc != SXM_E_OK) && (startrc != SXM_E_STATE)) {
            PLOG_UTL("Failed to start timers service (rc=%d)", startrc);
            return rc;
        }
    }

    if (IS_SERVICE_RUNNING_OK() && (timeInterval != 0U) && (pTimerHandle != NULL) &&
        (timerCallbackFunction != NULL) && (timerType < TIMER_TYPES_MAX))
    {
        SXMTimerInfoStruct *pTimer;
        pTimer = (SXMTimerInfoStruct *)sxm_list_allocate(sizeof(SXMTimerInfoStruct));
        if (pTimer != NULL) {

            /* Init timer info */
            pTimer->flags = (timerType == ONESHOT) ? SXM_TIMER_ONE_SHOT : SXM_TIMER_NONE;
            pTimer->timerInterval = timeInterval;
            pTimer->timerCallback = timerCallbackFunction;
            pTimer->timerCallbackArg = timerCallbackArg;
            pTimer->sData.lastShot = gNullTimeSpec;
            pTimer->sData.pastTimeout = 0U;

            DATA_LOCK(service);
            pTimer->timerHandle = service->lastTimerHandle++;
            /* add to the list */
            if (sxm_list_add(&service->timerList, pTimer, TRUE, NULL) == SXM_E_OK) {
                TM_LOG(": created new timer %u as %p, interval %u",
                    pTimer->timerHandle, pTimer, timeInterval);
                /* populate returned timer handle */
                *pTimerHandle = pTimer->timerHandle;
                rc = TRUE;
            }
            else {
                sxm_list_free(pTimer);
            }
            DATA_UNLOCK(service);
        }
    }

    return rc;
}

/***************************************************************************//**
* Searches for the timer by handle.
*
* \param[in] pService service instance
* \param[in] timerHandle pointer to timer handle which gets populated when timer is created
* \param[out] ppEntry corresponding list entry in case of success
*
* \return valid timer info structure or \p NULL in case of error
********************************************************************************/
static SXMTimerInfoStruct *timers_find(SXMTimersService *pService,
                                       uint timerHandle, SXMListEntry **ppEntry)
{
    SXMListEntry *pEntry;
    SXMTimerInfoStruct *pResult = NULL;

    pEntry = sxm_list_first(&pService->timerList);
    while (pEntry != NULL) {
        SXMTimerInfoStruct *pTimer = (SXMTimerInfoStruct *)sxm_list_data(pEntry);
        if ((pTimer != NULL) && (pTimer->timerHandle == timerHandle)) {
            if (ppEntry != NULL) {
                *ppEntry = pEntry;
            }
            pResult = pTimer;
            break;
        }
        pEntry = sxm_list_next(pEntry);
    }
    return pResult;
}

/***************************************************************************//**
* Provides current time in milliseconds.
* \param[out] t current time where the nanoseconds rounded milliseconds 
* \return valid time in milliseconds
********************************************************************************/
static void timers_time(SXMTimeSpec *t) {
    *t = gNullTimeSpec; /* initialize */
    sxm_clock_gettime(t);
    /* Do math round up to milliseconds */
    t->tv_nsec += (SXM_NANOSEC_PER_MILLISEC / 2);
    /* Ignore everything below 1 msec */
    t->tv_nsec /= SXM_NANOSEC_PER_MILLISEC;
    t->tv_nsec *= SXM_NANOSEC_PER_MILLISEC;

    return;
}

/***************************************************************************//**
* Compare two time specs and provide difference in milliseconds
*
* \param[in] t1 the first time
* \param[in] t2 the second time
* \param[out] pDelta absolute delta between times
*
* \relval >0 the first time is greater than the second one
* \retval =0 the first one is the same as the second one
* \retval <0 the first one is less than the second one;
*
*******************************************************************************/
static int timers_compare_ms(const SXMTimeSpec *t1, const SXMTimeSpec *t2, uint *pDelta)
{
    int rc = -1; /* negative by default */
    if (t1->tv_sec > t2->tv_sec) {
        rc = 1; /* The first one greater for sure */
    }
    else if (t1->tv_sec == t2->tv_sec) {
        if (t1->tv_nsec > t2->tv_nsec) {
            /* The first one greater due to nanoseconds while seconds the same */
            rc = 1;
        }
        else if (t1->tv_nsec == t2->tv_nsec) {
            /* Equal time specs */
            return 0;
        }
    }
    /* Compute abs delta */
    *pDelta = (rc > 0) ? (uint)sxm_time_get_absolute_time_diff(t1, t2) :
                         (uint)sxm_time_get_absolute_time_diff(t2, t1);
    return rc;
}

/***************************************************************************//**
* Adjust time spec to be in past by \p msec milliseconds
*
* \param[in,out] t the time to be adjusted
* \param[in] msec adjustment in milliseconds
*
*******************************************************************************/
static void timers_adjust_past(SXMTimeSpec *t, uint msec) {
    /* Count whole seconds */
    const time_t sec = (time_t)msec / SXM_MILLISEC_PER_SEC;
    /* It shall have at least the number of require seconds */
    if (t->tv_sec >= sec) {
        long nsec_rem = ((long)msec % SXM_MILLISEC_PER_SEC) * SXM_NANOSEC_PER_MILLISEC;
        t->tv_sec -= sec;
        /* can nanoseconds fulfill the request ? */
        if (t->tv_nsec >= nsec_rem) {
            t->tv_nsec -= nsec_rem;
        }
        /* Need to borrow one second */
        else if (t->tv_sec > 0) {
            --t->tv_sec; /* borrow one second */
            nsec_rem -= t->tv_nsec; /* reduce remaining nanoseconds */
            t->tv_nsec = SXM_NANOSEC_PER_SEC - nsec_rem;
        }
        else {
            /* All zeros in case if required offset if prior the start point */
            *t = gNullTimeSpec;
        }
    }
    else {
        /* All zeros in case if required offset if prior the start point */
        *t = gNullTimeSpec;
    }
    return;
}

/***************************************************************************//**
* Adjust time spec to be in future by \p msec milliseconds
*
* \param[out] dst time in future
* \param[in] t the time to be adjusted
* \param[in] msec adjustment in milliseconds
*
*******************************************************************************/
static void timers_adjust_future(SXMTimeSpec *dst, const SXMTimeSpec *t, uint msec) {
    dst->tv_sec = t->tv_sec + (time_t)(msec / SXM_MILLISEC_PER_SEC);
    dst->tv_nsec = t->tv_nsec + (long)((msec % SXM_MILLISEC_PER_SEC) * SXM_NANOSEC_PER_MILLISEC);
    if (dst->tv_nsec >= SXM_NANOSEC_PER_SEC) {
        dst->tv_sec += (dst->tv_nsec / SXM_NANOSEC_PER_SEC); /* add whole seconds */
        dst->tv_nsec %= SXM_NANOSEC_PER_SEC; /* keep leftover as nanoseconds */
    }
    return;
}

/***************************************************************************//**
* Tries to wake-up the thread in case if the passed timer requires this.
* 
* \param[in] pTimer timer instance which theoretically can cause the tic
*******************************************************************************/
static void timers_wakeup(const SXMTimerInfoStruct *pTimer) {
    /* Make sure the wake-up event is not in the queue */
    if (service->isWakingUp == FALSE) {
        BOOL bFireEvent = FALSE;
        /* Does the service only Null-like next scheduled tick? If so, the thread
         * should be woke up unconditionally base on this request
         */
        if ((gNullTimeSpec.tv_sec == service->nextTick.tv_sec) &&
            (gNullTimeSpec.tv_nsec == service->nextTick.tv_nsec))
        {
            TM_LOG(": the first event needed to wakeup from deep-sleep, timer %p",
                pTimer);
            bFireEvent = TRUE;
        }
        else {
            /* Check the next shot of the timer */
            SXMTimeSpec nextShot;
            uint delta;
            timers_adjust_future(&nextShot, &pTimer->sData.lastShot, pTimer->timerInterval);
            if (timers_compare_ms(&nextShot, &service->nextTick, &delta) < 0) {
                UNUSED_VAR(delta);
                TM_LOG(": need to re-evaluate timers, timer %p, expected timeout %u",
                    pTimer, delta);
                bFireEvent = TRUE;
            }
        }

        if (bFireEvent == TRUE) {
            /* Wake the thread */
            if (sxm_queue_put(&service->queue, NULL, TIMERS_WAKEUP, 0, MSG_PRIO_NORMAL) == SXM_E_OK) {
                TM_LOG(": event has been fired");
                /* Event in the queue */
                service->isWakingUp = TRUE;
            }
        }
    }
    return;
}

/***************************************************************************//**
* This function is used to delete a timer.
*
* \param[in] timerHandle the handle for the timer instance the caller wishes
*                        to delete.
*
* \retval TRUE on success
* \retval FALSE otherwise
********************************************************************************/
BOOL sxm_timers_delete_timer(uint timerHandle) {

    BOOL rc = FALSE;

    if (IS_SERVICE_RUNNING_OK()) {
        SXMListEntry *pEntry;
        SXMTimerInfoStruct *pTimer;

        DATA_LOCK(service);
        // There is no need to re-evaluate timeouts since if this is the closest
        // one no other timer can be closer that this one, if not - doesn't matter,
        // the closest one shall be expired first
        pTimer = timers_find(service, timerHandle, &pEntry);
        TM_LOG("removing timer %u as %p", timerHandle, pTimer);
        if (pTimer != NULL) {
            if (sxm_list_remove(&service->timerList, pEntry) == SXM_E_OK) {
                sxm_list_free(pTimer);
                rc = TRUE;
            }
        }
        DATA_UNLOCK(service);
    }

    return rc;
}

/***************************************************************************//**
* This function is used to start a timer.
*
* \param[in] timerHandle for the timer to be started
*
* \retval TRUE on success
* \retval FALSE otherwise
********************************************************************************/
BOOL sxm_timers_start_timer(uint timerHandle) {
    BOOL rc = FALSE;

    if (IS_SERVICE_RUNNING_OK()) {
        SXMTimerInfoStruct *pTimer;
        DATA_LOCK(service);
        pTimer = timers_find(service, timerHandle, NULL);
        TM_LOG("starting timer %u as %p", timerHandle, pTimer);
        if (pTimer != NULL) {
            /* Make timer active removing all flags except one-shot if it's set */
            pTimer->flags = SXM_TIMER_ACTIVE | (pTimer->flags & SXM_TIMER_ONE_SHOT);
            // Since now the next shot should be in timer's interval
            timers_time(&pTimer->sData.lastShot);
            // If the message is not in a process there is a need to wake up
            // the thread to re-evaluate timers if the currently un-pausing
            // timers requires this.
            timers_wakeup(pTimer);
            rc = TRUE;
        }
        DATA_UNLOCK(service);
    }

    return rc;
}

/***************************************************************************//**
* This function is used to stop a timer.
*
* \param[in] timerHandle handle for the timer to be stopped
*
* \retval TRUE on success
* \retval FALSE otherwise
********************************************************************************/
BOOL sxm_timers_stop_timer(uint timerHandle) {
    BOOL rc = FALSE;

    if (IS_SERVICE_RUNNING_OK()) {
        SXMTimerInfoStruct *pTimer;
        DATA_LOCK(service);
        /* There is no need to re-evaluate timeouts since if this is the closest
         * one no other timer can be closer that this one, if not - doesn't matter,
         * the closest one shall be expired first
         */
        pTimer = timers_find(service, timerHandle, NULL);
        TM_LOG("stopping timer %u as %p", timerHandle, pTimer);
        if (pTimer != NULL) {
            /* Inactivate the timer keeping one-shot sign if exists */
            pTimer->flags &= SXM_TIMER_ONE_SHOT;
            rc = TRUE;
        }
        DATA_UNLOCK(service);
    }

    return rc;
}

/***************************************************************************//**
* This function is used to re-start a timer.
*
* \param[in] timerHandle handle for the timer to be re-started
*
* \retval TRUE on success
* \retval FALSE otherwise
********************************************************************************/
BOOL sxm_timers_restart_timer(uint timerHandle) {
    return sxm_timers_start_timer(timerHandle);
}

/***************************************************************************//**
* This function is used to pause running timer.
*
* \param[in] timerHandle handle for the timer to be re-started
*
* \retval TRUE on success
* \retval FALSE otherwise
********************************************************************************/
BOOL sxm_timers_pause_timer(uint timerHandle) {
    BOOL rc = FALSE;

    if (IS_SERVICE_RUNNING_OK()) {
        SXMTimerInfoStruct *pTimer;
        DATA_LOCK(service);
        pTimer = timers_find(service, timerHandle, NULL);
        TM_LOG("pausing timer %u as %p", timerHandle, pTimer);
        // Pause the timer only if it exists and not paused yet.
        if ((pTimer != NULL) && ((pTimer->flags & SXM_TIMER_ACTIVE) == SXM_TIMER_ACTIVE))
        {
            // There is no need to re-evaluate timeouts since if this is the closest
            // one no other timer can be closer that this one, if not - doesn't matter,
            // the closest one shall be expired first
            if ((pTimer->flags & SXM_TIMER_PAUSED) == SXM_TIMER_NONE) {
                uint pastTimeout;
                SXMTimeSpec currTime;
                timers_time(&currTime);
                pTimer->flags |= SXM_TIMER_PAUSED;
                // Keep the difference between current time and the last shot
                // to let it be resumed and shot in remaining time after un-pause
                if (timers_compare_ms(&currTime, &pTimer->sData.lastShot, &pastTimeout) > 0)
                {
                    // To make sure the past time is not longer than the timer interval
                    // since even in this case the timer shot shall be done immediately
                    // once the expiration is detected
                    if (pastTimeout > pTimer->timerInterval) {
                        pTimer->sData.pastTimeout = pTimer->timerInterval;
                    }
                    else {
                        pTimer->sData.pastTimeout = pastTimeout;
                    }
                }
                else {
                    // Store the value into the timer. This place means that the
                    // last shot just happened or will happen in future, thus,
                    // there is no time which passed since the last shot.
                    pTimer->sData.pastTimeout = 0U;
                }
                TM_LOG(": timer %u paused after %u msec(s)",
                    timerHandle, pTimer->sData.pastTimeout);
            }
            rc = TRUE;
        }
        DATA_UNLOCK(service);
    }

    return rc;
}

/***************************************************************************//**
* This function is used to resume paused running timer.
*
* \param[in] timerHandle handle for the timer to be re-started
*
* \retval TRUE on success
* \retval FALSE otherwise
********************************************************************************/
BOOL sxm_timers_unpause_timer(uint timerHandle) {
    BOOL rc = FALSE;

    if (IS_SERVICE_RUNNING_OK()) {
        SXMTimerInfoStruct *pTimer;
        DATA_LOCK(service);
        pTimer = timers_find(service, timerHandle, NULL);
        TM_LOG("un-pausing timer %u as %p", timerHandle, pTimer);
        // Un-pause the timer only if it exists and in pause state.
        if ((pTimer != NULL) && ((pTimer->flags & SXM_TIMER_ACTIVE) == SXM_TIMER_ACTIVE))
        {
            if ((pTimer->flags & SXM_TIMER_PAUSED) == SXM_TIMER_PAUSED) {
                pTimer->flags &= (uint)(~SXM_TIMER_PAUSED);
                TM_LOG(": qualified for UNPAUSE");
                // Adjust the last shot time in accordance with save past time
                // since the latest shot which was upon pause.
                timers_time(&pTimer->sData.lastShot);
                timers_adjust_past(&pTimer->sData.lastShot, pTimer->sData.pastTimeout);
                // If the message is not in a process there is a need to wake up
                // the thread to re-evaluate timers if the currently un-pausing
                // timers requires this.
                timers_wakeup(pTimer);
            }
            else {
                TM_LOG(": Already UNPAUSE");
            }
            rc = TRUE;
        }
        DATA_UNLOCK(service);
    }

    return rc;
}

/***************************************************************************//**
* This function is used to check if timer is currently active.
*
* \param[in] timerHandle handle for the timer to be re-started
*
* \retval TRUE if timer is active
* \retval FALSE otherwise
********************************************************************************/
BOOL sxm_timers_is_active_timer(uint timerHandle) {

    BOOL rc = FALSE;

    if (IS_SERVICE_RUNNING_OK()) {
        SXMTimerInfoStruct *pTimer;
        DATA_LOCK(service);
        pTimer = timers_find(service, timerHandle, NULL);
        TM_LOG("state check for timer %u as %p", timerHandle, pTimer);
        if ((pTimer != NULL) &&
            ((pTimer->flags & SXM_TIMER_ACTIVE) == SXM_TIMER_ACTIVE))
        {
            rc = TRUE;
        }
        DATA_UNLOCK(service);
    }

    return rc;
}

/***************************************************************************//**
* This function is used get the time until next expiration in milliseconds
* 
* \note It's not recommended to call this function if there is no urgent need
*       to get this information in order to perform some critical operation outside
*       associated timer's callback.
*
* \param[in] timerHandle handle for the timer to be inquired
* \param[out] pValue placeholder for the function result in case of success
*
* \retval TRUE the time is provided
* \retval FALSE otherwise
********************************************************************************/
BOOL sxm_timers_get_time(uint timerHandle, uint *pValue) {

    BOOL rc = FALSE;
    if (pValue) {
        *pValue = 0U;
        if (IS_SERVICE_RUNNING_OK()) {
            SXMTimerInfoStruct *pTimer;
            DATA_LOCK(service);
            pTimer = timers_find(service, timerHandle, NULL);
            TM_LOG("getting time for timer %u as %p", timerHandle, pTimer);
            if ((pTimer != NULL) && ((pTimer->flags & SXM_TIMER_ACTIVE) == SXM_TIMER_ACTIVE))
            {
                if ((pTimer->flags & SXM_TIMER_PAUSED) == SXM_TIMER_PAUSED) {
                    TM_LOG(": provide time as for PAUSED");
                    *pValue = pTimer->timerInterval - pTimer->sData.pastTimeout;
                }
                else {
                    /* For the active timer we need to calculate remaining time
                     * based on last shot and the interval.
                     */
                    SXMTimeSpec currTime;
                    timers_time(&currTime);
                    TM_LOG(": provide time as for UNPAUSED");
                    if (timers_compare_ms(&currTime, &pTimer->sData.lastShot, pValue) < 0)
                    {
                        *pValue = 0U;
                    }
                }
                rc = TRUE;
            }
            DATA_UNLOCK(service);
        }
    }

    return rc;
}

/***************************************************************************//**
* This function is used set new interval for timer milliseconds. Timer must be
* stopped for this to work.
* 
*
* \param[in] timerHandle handle for the timer to be inquired
* \param[in] timeInterval interval in milliseconds
* \param[in] timerCallbackArg callback argument passed to \p timerCallbackFunction during
*                             each call
*
*
* \retval TRUE new interval set
* \retval FALSE otherwise
********************************************************************************/
BOOL sxm_timers_set_new_interval(uint timerHandle, uint timeInterval,  void *timerCallbackArg) {

    BOOL rc = FALSE;

    if (IS_SERVICE_RUNNING_OK()) {
        SXMTimerInfoStruct *pTimer;

        DATA_LOCK(service);
        pTimer = timers_find(service, timerHandle, NULL);
        /* make sure timer not running */
        if ((pTimer != NULL) && 
            ((pTimer->flags & (SXM_TIMER_PAUSED | SXM_TIMER_ACTIVE)) == SXM_TIMER_NONE)) {
            pTimer->timerInterval = timeInterval;
            pTimer->timerCallbackArg = timerCallbackArg;
            rc = TRUE;
        }
        DATA_UNLOCK(service);
    }
 
    return rc;
}

/***************************************************************************//**
* This function does several steps to process timers list:
*  -# Check all timers and find out the next closest expiration timer to set up
*     waiting time for the next timers processing procedure;
*  -# Do callbacks for expired at this moment timers;
*  -# One-shot expired timers will be marked as inactive, rearming ones in case
*     of expiration will be rescheduled to another shot in it's own period
*
* \param[in] pService service instance
* \param[in] bWokeUp \p TRUE if the call was done due to WAKEUP message, \p FALSE otherwise
* 
* \return next desired timeout
********************************************************************************/
static uint timers_handle_timer_tick(SXMTimersService *pService, BOOL bWokeUp) {
    /* Set the max possible wait time. The assumption that the INFINITE
     * timeout's value is greater than any other possible timer interval.
     * Thus, following logic of closest timer expiration interval
     * computation should decrease this number or keep it as-is if there is
     * no more active timers.
     */
    uint waitTime = SXM_QUEUE_INFINITE_TIMEOUT;
    
    if (IS_SERVICE_RUNNING_OK()) {
        SXMListEntry *pEntry;

        DATA_LOCK(pService);
        TM_LOG(": processing (woke-up is %d)", bWokeUp);
        /* Next tick is Null-like to make sure we can detect the case where no
         * next tick is scheduled
         */
        pService->nextTick = gNullTimeSpec;
        /* Look through all timers */
        pEntry = sxm_list_first(&pService->timerList);
        while (pEntry != NULL) {
            SXMTimerInfoStruct * const pTimer =
                (SXMTimerInfoStruct *)sxm_list_data(pEntry);

            /* Get the next timer if any */
            pEntry = sxm_list_next(pEntry);

            /* Process just fetched-up timer - take care about active only */
            if ((pTimer != NULL) &&
                ((pTimer->flags & (SXM_TIMER_ACTIVE | SXM_TIMER_PAUSED)) == SXM_TIMER_ACTIVE))
            {
                SXMTimeSpec currTime; 
                uint nextTimeout;
                SXMTimeSpec nextShotTime;

                TM_LOG(": evaluate the timer %p", pTimer);

                /* Calculate the next shot time for this timer*/
                timers_adjust_future(&nextShotTime, &pTimer->sData.lastShot, pTimer->timerInterval);
                /* Should it be shot right now? */
                timers_time(&currTime); /* Capture current time */
                if (timers_compare_ms(&nextShotTime, &currTime, &nextTimeout) <= 0) {
                    if ((pTimer->flags & SXM_TIMER_ONE_SHOT) == SXM_TIMER_ONE_SHOT) {
                        pTimer->flags &= (uint)(~SXM_TIMER_ACTIVE);
                        /* There is no the next shot for this timer so far */
                        nextTimeout = SXM_QUEUE_INFINITE_TIMEOUT;
                        TM_LOG(": stop single shot timer");
                    }
                    else {
                        /* This time is the last shot */
                        pTimer->sData.lastShot = currTime;
                        /* Next shot will be in timer's interval period */
                        nextTimeout = pTimer->timerInterval;
                        TM_LOG(": re-start rearming timer, interval %u", pTimer->timerInterval);
                    }
                    /* Fire callback */
                    TM_LOG(": do timer callback %p (%p)", pTimer->timerCallback, pTimer->timerCallbackArg);
                    pTimer->timerCallback(pTimer->timerCallbackArg);
                }
                /* Update next wait time */
                TM_LOG(": update next timeout %u vs %u", nextTimeout, waitTime);
                if (nextTimeout < waitTime) {
                    waitTime = nextTimeout;
                    pService->nextTick = nextShotTime;
                }
            }
            else {
                TM_LOG(": skip timer %p", pTimer);
            }
        }
        // If this function was called due to the wake-up event we need
        // to indicate they it has been processed and another one can be
        // fired via the queue if needed.
        if (bWokeUp == TRUE) {
            pService->isWakingUp = FALSE;
        }
        DATA_UNLOCK(pService);
    }
    return waitTime;
}
