/******************************************************************************/
/*                Copyright (c) Sirius XM Satellite Radio, Inc.               */
/*                            All Rights Reserved                             */
/*      Licensed Materials - Property of Sirius XM Satellite Radio, Inc.      */
/******************************************************************************/
/***************************************************************************//**
 *
 *  \file sxm_time_internal.h
 *  \author Alexander Pylchagin
 *  \date 9/12/2016
 *  \brief Time related primitives
 *
 ******************************************************************************/

#ifndef SXM_TIME_INTERNAL_H_
#define SXM_TIME_INTERNAL_H_

#include <time.h>
#include "sxm_build.h"
#include "sxm_typ.h"
#include "sxm_common.h"

#ifdef __cplusplus
extern "C"
{
#endif

/************************************************************************
 *                                                                      *
 *            TimeStamps                                                *
 *            ==========                                                *
 *                                                                      *
 *    Time in minutes since 2012 (epoch 15340 in unixtime)              *
 *                                                                      *
 ************************************************************************/
#define sxm_sxe_time_epoch_from_time(_t) (SXM_SXE_TIME_EPOCH0 + (uint)(_t) / MINUTES_IN_DAY)
#define sxm_sxe_time_epoch_to_time(_e) (((uint)(_e) - SXM_SXE_TIME_EPOCH0) * MINUTES_IN_DAY)

/** \name USEFUL TIME CONVERSION CONSTANTS
 * \@
 */
#define SXM_MILLISEC_PER_SEC     1000       //!< 1 second = 1,000 Milliseconds
#define SXM_MICROSEC_PER_SEC     1000000    //!< 1 second = 1,000,000 Microseconds
#define SXM_NANOSEC_PER_MICROSEC 1000       //!< 1 microsecond = 1,000 Nanoseconds
#define SXM_NANOSEC_PER_MILLISEC 1000000    //!< 1 millisecond = 1,000,000 Nanoseconds
#define SXM_NANOSEC_PER_SEC      1000000000 //!< 1 second = 1,000,000,000 Nanoseconds
/** @} */

/** Since some platforms like Windows don't have \p timespec structure defined
 * SXe declares own for internal needs
 */
typedef struct {
    time_t tv_sec;
    long tv_nsec;
} SXMTimeSpec;

extern SXESDK_INTERNAL_API void sxm_sxe_time_decode(uint, SXMTime *);

extern SXESDK_INTERNAL_API uint sxm_sxe_time_encode(SXMTime *);

extern SXESDK_INTERNAL_API uint sxm_sxe_time_encode_local(SXMTime *);

extern SXESDK_INTERNAL_API uint sxm_sxe_time_now(void);

extern SXESDK_INTERNAL_API uint sxm_sxe_time_now_utc(void);

extern SXESDK_INTERNAL_API void sxm_sxe_time_set_epoch_bias(ushort epochBias);

extern SXESDK_INTERNAL_API time_t sxm_time_get_absolute_time_diff(const SXMTimeSpec *,
                                                                  const SXMTimeSpec *);

extern SXESDK_INTERNAL_API void sxm_clock_gettime(SXMTimeSpec *);

#ifdef __cplusplus
}
#endif

#endif /* SXM_PLATFORM_INTERNAL_H_ */