/******************************************************************************/
/*                Copyright (c) Sirius XM Satellite Radio, Inc.               */
/*                            All Rights Reserved                             */
/*      Licensed Materials - Property of Sirius XM Satellite Radio, Inc.      */
/******************************************************************************/
/***************************************************************************//**
*
* \file sxm_tfile.c
* \author Leslie French
* \date 8/20/2013
* \brief Transaction files.
*
* A transactional file is a file in the filing system that can be updated using
* a series of transactions, and the previous state recovered if a transaction
* fails to complete.  The interface to the utility uses an opaque data pointer
* (\ref SXMTFile) which hides the details of the implementation. The internal
* structure and design of the tfile systems is described in \ref tfile_page
*
* \page tfile_page Transaction Files (tfiles)
*
* The Transaction File for most all services implementing this file are updated
* through the RFD feature described in section \ref rfd_page. The corresponding
* RFD service thread will not update the service Transaction File (tfile)
* (e.g. fueldb) unless the the service is active; sxm_<service_name>_start has
* been called. It is recommended to start all SXe services that are released into
* production so the corresponding T-Files are kept current with the latest broadcast
* updates.
*
* Transaction Files are used to implement writable databases where there is a
* guaranteed stable state and/or update restart point (barring catastrophic
* disk failures that render all or part of the disk file unreadable).
*
* Transaction Files are mostly use for updatable baseline files. Baseline files
* usually have a convenient internal structure, divided by states, markets or
* regions, which forms the basis for the 'transaction' approach to updates,
* along the lines of:
* ~~~~
* START a transaction -> UPDATE Alabama        -> COMMIT
* START a transaction -> UPDATE Alaska         -> COMMIT
* . . . . .
* START a transaction -> UPDATE DC             -> COMMIT
* START a transaction -> mark database updated -> COMMIT
* ~~~~
*
* At each COMMIT point, the database is in a usable state thought with some areas
* having new contents and some having old. If the system is powered down in the
* middle of a transaction it comes back up in the state 'as if' that transaction
* had never started, i.e. there is automatic rollback to the last COMMIT point.
*
* A single tfile comprises three areas:
*  -# The Transaction File Header Block
*  -# A pair of Root Blocks
*  -# The Data Store
*
* The Root Blocks and Data Store are considered as a sequence of fixed-sized
* data blocks. The Data Store is used to contain a number of variable-sized
* 'Objects', addressed as a block number and block count. The Root Blocks
* provide a catalog of 'Objects' indexed or referenced according to the
* addressing scheme of the baseline file.  As an example, the structure of
* the fuel baseline data will be used below.
*
* The Header Block is a fixed-size 8kbyte structure at the start of the
* transaction file. This is organized as 16 512-byte sectors:
* | Sectors | Contents                                              |
* |:-------:|:------------------------------------------------------|
* |   0     | The first transaction block                           |
* |   1     | The second transaction block                          |
* | 2 - 5   | The first allocation map                              |
* | 6 - 9   | The second allocation map                             |
* |10 - 15  | Unused on disk. Sectors 10 - 13 are used during transactions in memory|
*
* Each transaction block has the \ref TBlock structure.
*
* The Allocation Maps are \ref SXM_TFILE_ALLOC_MAP_BITLEN bits where each
* indicating that the corresponding block is free ('1') or in-use/unavailable ('0').
* The maximum size of a T-File is therefore SXM_TFILE_ALLOC_MAP_BITLEN blocks
* (128 Mbytes at 8 kbytes/block; 256 MBytes at 16 kbytes/block).  A block index
* (offset in the file) will fit into a 16-bit ushort.
*
* The structure of the Root Block depends on the nature of the file data.
* It is treated as a transparent data object by the T-File utility. For
* the Fuel/CanFuel baseline database, the Root Block is 8,192 bytes, containing
* 2,048 4-byte entries, one for each fuel region in the baseline. Each
* 4-byte entry comprises a 2-byte block offset, and a 2-byte block count.
*
* The 'object' to which each region in the fuel Root Block points has its
* own internal structure, indexing the individual fuel stations by FSUID,
* with records for all the fuel station data required for the fuel service.
*
* Putting this together, for a baseline file not in the middle of an update,
* we have the following structure (ignoring the second transaction block for
* the moment).
*
* \dot
* digraph tfile_example {
*     compound=true;
*     rankdir=source;
*     node [shape=box];
*     tb0node [label="Transaction Block 0"]
*     subgraph cluster_ab0node {
*         label="Allocation Block 0";
*         edge [style="invis"];
*         alloc0 [label="allocated"];
*         alloc1 [label="allocated"];
*         {rank=same; alloc0 -> alloc1;}
*     }
*     subgraph cluster_rb0 {
*         label="Root Block 0";
*         edge [style="invis"];
*         reg158_node [label="[158]: 47,3"];
*     }
*     reg158 [shape=record label="{Region 158|...|<fsuid6>FSUID 6|...|<rec6>record|...}"];
*     reg158_id6 [label="JOE'S GAS\n1432 East 5th St\nPEMVILLE\nNJ 08123"];
*
*     tb0node -> alloc0 [lhead=cluster_ab0node weight=1000];
*     tb0node -> reg158_node [lhead=cluster_rb0];
*     reg158_node -> alloc0 [ltail=cluster_rb0];
*     reg158_node -> reg158;
*     reg158 -> alloc1;
*     reg158:fsuid6 -> reg158:rec6;
*     reg158:rec6 -> reg158_id6 [dir=none];
* }
* \enddot
*
* To find the record for station #6 in region #158, the transaction block
* gives the pointer to the Root Block. Index 158 in the Root Block indicates
* that the region is in blocks 47, 48 and 49 of the file (all of which should
* be marked as 'in use' in the Allocation Map). Reading in the 24kbytes of data
* starting at block 47 in the file retrieves the region data. Index 6 in the
* header of the region object gives the offset for the fuel station record
* which contains the name, address, location and etc. of that station.
*
* The following steps are followed when a T-File is opened
*  -# The file is opened using the path function to resolve the file name.
*  -# The T-File header is read into memory.
*  -# The CRC of each transaction block is calculated
*  -# If the \ref SXMTFileBlock is OK, the CRC of the Allocation Block is calculated
*  -# If the Allocation Block is OK, the Root Block is read in to memory
*  -# The CRC of the Root Block is checked.
*
* If both Transaction Blocks pass all the tests, the one with the higher sequence
* number is considered Active. If neither Transaction Block passes all these tests,
* the file is considered corrupt. The corrupt state is reported up to the
* application through the SXM_E_BAD_XXX return values (where XXX indicates
* the specific T-File being reported). An application should be prepared to restore
* the file, or the filing system at this point. Although this is a fatal error
* for that service, other services may continue to run correctly.
*
* A newly-installed file has a sequence number of '1' in both transaction blocks,
* with correct Allocation Maps, Root Blocks and all CRCs.  During normal
* (read-only) operation, these data are never touched, and there is no reason
* to believe that the file will become spontaneously corrupt.
*
* Updating the file uses a shadow-allocation scheme, where all the operations are
* performed on the inactive block, and the act of writing that block to disk
* commits the transaction. In order to support allocations and de-allocations
* of data blocks without corrupting existing data, all allocations are taken from
* a Temporary Map (SXMTFileMap) which records only allocations and not deletions. This
* means that a block in use in the active map will never be re-allocated, even
* if it is freed in the shadow map.
*
* When the time comes to update the file, the following steps are done (assuming
* that TBlock0 is the active block, and SXMTFileBlock No.1 is the inactive (older) block):
* |         |                                                                          |
* |--------:|:-------------------------------------------------------------------------|
* |\b start | SXMTFileBlock No.1.sequence = SXMTFileBlock No.0.sequence + 1            |
* |         | Copy Allocation Map 0 to Allocation Map 1 and SXMTFileAllocMap           |
* |\b alloc | Find 'N' contiguous free ('1') bits in SXMTFileAllocMap                  |
* |         | Set these bits in use in both Map No.1 and SXMTFileAllocMap              |
* |\b free  | Clear the bits (set to '1') in Map No.1  (do not clear SXMTFileAllocMap!)|
* |\b root  | Change the index pointers in Root Block No.1                             |
* |\b write | Write newly-updated data blocks to disk (but not the transaction header) |
* |\b commit| Write Root No.1 to disk                                                  |
* |         | Write Allocation Map No.1 to disk                                        |
* |         | Update root CRC and alloc CRC in SXMTFileBlock No.1                      |
* |         | Calculate SXMTFileBlock No.1 CRC                                         |
* |         | Write SXMTFileBlock No.1                                                 |
* |         | SXMTFileBlock No.1 becomes the active block                              |
*
* Up until the COMMIT point, all the writes to the disk are to blocks that are marked as
* free in the active map, so no active data are changed. SXMTFileBlock No.1 on disk still
* has its old sequence number, so a restart at this point will still pick SXMTFileBlock No.0
* as the active block, and will see only data before the update started. At all points
* during the commit, the inactive blocks are being updated. Only the last write, 1 sector of
* SXMTFileBlock No.1 will cause the restart to see the new value.
*
* If for some reason, that last sector-write fails in the middle, the CRC for the block
* will be incorrect (since it is the last 4 bytes in the block), so the restart will
* not consider SXMTFileBlock No.1 as valid, and continue with SXMTFileBlock No.0 as
* the active block.
*
* The limitation of this scheme is that, between the start and the commit, all
* allocations are from blocks that are not in use in the active map. Even if blocks
* are freed during the update, they will not be re-used, so the file must contain
* sufficient free blocks to allow for the allocation of all blocks needed between
* commitment points. T-Files are usually built with around 25% extra capacity to handle
* updates, and the extra allocation is sized to ensure that any single transaction
* can be completed, or that a group of allocations can be completed. If, for some
* reason, a developer chooses to change (increase) the number of allocations between
* commit points, the additional space in the tfile should be increased accordingly.
*
* Once the commit has completed, any cached data blocks that might have been affected
* by the change must be reloaded (using the new index values from the newly-active
* block) since those cached block may now be marked 'free' in the new Allocation Map
* (and potentially re-used for completely different data in the next transaction).
*
* \section tfile_chain T-File Chain
*
* The T-File API works with \b chains. The chain is the set of logically connected
* blocks which all together are keeping solid piece of data related organized by the
* T-File owner. The blocks inside the set can be placed in any location within the
* T-File allowing better usage of T-File space and avoiding fragmentation.
*
* The generic chain is a single-direction linked list where each block has a knowledge
* of the next block id and it's own role in the chain: the first, the last or the middle
* block. Thus, the API protects caller from reading and writing data outside the chain
* and this is a owner's responsibility to control data integrity among chains.
*
* \section tfile_stream T-File Based Stream
*
* T-File Stream provides convenient and safe way of creating content to be put into
* the T-File. Each opration is controlled protecting caller from adding data  outside
* any type of memory. In most cases a service works with basic types like uint, ushort,
* byte, fix and nil-terminated strings, but the stream supports adding binary data as-is
* giving all responsibilities of data representation to the services.
*
*******************************************************************************/
#include "sxm_tfile.h"
#include "sxm_list.h"

#ifndef SDKFILES_STANDALONE_BUILD
#include <util/sxm_common_internal.h>
#include <util/sxm_noname_internal.h>
#else
#include "sdkfiles.h"
#endif

#include <stddef.h>

TYPE_SIZE_STATIC_ASSERT(SXMTFileAllocMap, 2048);
TYPE_SIZE_STATIC_LE_ASSERT(SXMTFileHeader, SXM_TFILE_HEADER_BYTELEN);

/** Internal T-File Logging macro definition */
#ifndef TF_LOG
#define TF_LOG(_f, ...) // sxm_file_log("%s: " _f, __FUNCTION__, ##__VA_ARGS__)
#endif

/** Defines all-bits are set mask */
#define TF_ALL_MASK (0xFFFFFFFFU)

/** Defines non-bits are set mask */
#define TF_NONE_MASK (0x00000000U)

/** Gives index of the other transaction 
 * \param[in] _tf_tr_idx current index
 */
#define TF_OTHER_IDX(_tf_tr_idx) (((_tf_tr_idx) ^ 1) & 0x1)

/** Gives index of the other transaction for the t-file instance
 * \param[in] _tf valid t-file instance
 */
#define TF_OTHER(_tf) TF_OTHER_IDX((_tf)->active)

/** Defines whether the block is used by the any active transaction. The block
 * considered as \b used if it either belongs to active stable transaction or
 * is allocated during current update
 * \param[in] _tf t-file instance
 * \param[in] _id block id
 */
#define TF_BLOCK_IS_USED(_tf, _id) \
    (!BITP((_tf)->hdr->map[(_tf)->active].bits, (_id)) || \
    (((_tf)->mask & SXM_TFILE_MASK_TRANSACTION) && !BITP((_tf)->hdr->temp.bits, (_id))))

/** \name Chained blocks primitives
 * Each blocks inside the chained t-file has a signature which is represented
 * by the first 4 bytes of each block. The signature has the following format
 * |Byte No.| Description |
 * |:------:|-------------|
 * |    3   | Set of flags 7bit - is the first, 6bit - is the last, 5bit-0bit reserved |
 * |  2-1   | Next block's id in the chain in inverted form (\~)|
 * |    0   | Checksum over previous 3 bytes as sum of their values mod 256|
 * @{
 */
/** Sign that this block is the first one in the chain */
#define TFILE_SIG_IS_BEGIN_MASK (1U << 31U)

/** Sign that this block is the last one in the chain */
#define TFILE_SIG_IS_END_MASK   (1U << 30U)

/** Calculates 1-byte checksum over signature.
 * The check sum is calculated over most significant 3 bytes
 * as 8-bit value
 * \param[in] sig signature to calculate
 * \return calculated value
 */
#define TFILE_SIG_CALC_CS(_sig) \
    (~((((_sig) >> 24U) & 0xFFU) + \
       (((_sig) >> 16U) & 0xFFU) + \
       (((_sig) >>  8U) & 0xFFU)) & 0xFFU)

/** Unpack next block from the signature */
#define TFILE_SIG_UNPACK_NEXTID(_sig) \
    ((const ushort)(((~(_sig)) >> 8U) & 0xFFFFU))

/** Unpack signature's checksum value as is */
#define TFILE_SIG_UNPACK_CS(_sig) ((_sig) & 0xFFU)

/** Defines signature section byte length */
#define TFILE_SIG_BYTELEN (4U)

/* Help checker to make sure the type which is used for signature
 * operations is the same size as the signature itself
 */
TYPE_SIZE_STATIC_ASSERT(uint, TFILE_SIG_BYTELEN);

/* Helper checker to make sure that reserved space inside the block
 * is enough to accommodate the signature
 */
TYPE_SIZE_STATIC_LE_ASSERT(TFILE_SIG_BYTELEN, SXM_TFILE_BLOCK_RESERVED);

/** Helper to create the chain to accommodate required number
 * of elements
 * \param[in] _len required number of items in the chain
 */
#define TFILE_NEW_CHAIN(_len) \
    ((SXMTFileChain*)sxe_calloc(1, \
        sizeof(SXMTFileChain) + (size_t)((_len) - 1) * sizeof(ushort)))
/** @} */

/** Stream data block with variable size definition */
typedef struct {
    size_t size; //!< \p data utilizations.
    byte data[1]; //!< Single block data
} SXMTStreamBlock; //!< Single stream block

/* T-File stream structure definitions */
struct  _sxm_tfile_stream {
    SXMTFile *tFile; //!< Associated t-file
    SXMList blocks; //!< List of blocks
    SXMList unused; //!< List of unused blocks
    SXMListEntry *entry; //!< The current working entry from \p blocks
    size_t size; //!< Overall stream utilization in bytes
    SXMTStreamBlock *tmp; //!< Intermediate block
    SXMTFileStreamState state; //!< Current state
};

/***************************************************************************//**
 * Writes a number of blocks to the file
 * \param[in] self a pointer to a valid transaction file structure
 * \param[in] block the index of the first block to be written
 * \param[in] count the number of (user) blocks to write
 * \param[in] src a pointer to the data to be written
 * \return -1 or the number of block written
 ********************************************************************************/
static int tf_write(SXMTFile *self, size_t block, size_t count, const void *src) {
    const size_t fsize = self->hdr->tb[self->active].fsize;
    int rc = -1;

    if ((block + count) <= fsize) {
        const uint bsize = self->hdr->tb[self->active].bsize;
        if (fseek(self->dbfile, (long)(bsize * block), SEEK_SET) == 0) {
            rc = (int)fwrite(src, bsize, count, self->dbfile);
            if (rc > 0) {
                /* Since here the file's content has new data, thus, mark it as modified
                 * regardless the result of further commands
                 */
                self->mask |= SXM_TFILE_MASK_MODIFIED;
            }
        }
        else {
            TF_LOG(": fseek(%u, %u) failed", (uint)block, (uint)count);
        }
    }
    else {
        TF_LOG(": Error: writing at %u + %u blocks exceeds %u blocks",
            (uint)block, (uint)count, (uint)fsize);
    }
    return rc;
}

/***************************************************************************//**
 * Reads a number of blocks from the file
 * 
 * \note If the \p dst is NULL the function does simple blocks check
 * 
 * \param[in] self a pointer to a valid transaction file structure
 * \param[in] block the index of the first block to be read
 * \param[in] count the number of (user) blocks to read
 * \param[in] dst a pointer to the area which will receive the data
 * \return the number of block read or \p -1 in case of error
 ********************************************************************************/
static int tf_read(SXMTFile *self, size_t block, size_t count, void *dst) {
    const uint bsize = self->hdr->tb[self->active].bsize;
    int rc = 0;
    size_t idx;
    /* Check that all blocks are used, thus, have reliable content */
    for (idx = block; idx < (block + count); ++idx) {
        if (!TF_BLOCK_IS_USED(self, idx)) {
            TF_LOG(": block %u is unused, reading %u block(s) from %u",
                (uint)idx, (uint)count, (uint)block);
            rc = -1;
            break;
        }
    }
    if (rc == 0) {
        if (dst != NULL) {
            if (fseek(self->dbfile, (long)(bsize * block), SEEK_SET) == 0) {
                rc = (int)fread(dst, bsize, count, self->dbfile);
            }
            else {
                TF_LOG(": fseek(%u) failed", (uint)(bsize * block));
                rc = -1;
            }
        }
        else {
            rc = (int)count;
        }
    }
    return rc;
}

/***************************************************************************//**
 * Reads the root block from the file
 *
 * \param[in] self a pointer to a valid transaction file structure
 * \param[in] tb transaction block
 *
 * \retval SXM_E_OK Success
 * \retval SXM_E_PIPE Failed I/O operations
 * \retval SXM_E_NOMEM Failed to allocate memory
 * \retval SXM_E_BAD_DB Corruption in t-file's root detected
 *
 ********************************************************************************/
static SXMResultCode tf_read_root(SXMTFile *self, SXMTFileBlock * const tb) {
    SXMResultCode rc;
    const uint size = tb->rootsize;

    if ((tb->rootptr + size) > tb->fsize) {
        rc = SXM_E_BAD_DB;
    }
    else {
        /* Allocate memory and read the root reading make sense only
         * if the root is set by t-file designer
         */
        rc = SXM_E_OK;
        if (size > 0U) {
            self->root = sxe_malloc(size * tb->bsize);
            if (self->root == NULL) {
                rc = SXM_E_NOMEM;
            }
            else if (tf_read(self, tb->rootptr, size, self->root) != (int)size) {
                TF_LOG(": failed to read root block");
                rc = SXM_E_PIPE;
            }
        }
        else {
            self->root = NULL;
        }
        /* Still ok? */
        if (rc == SXM_E_OK) {
            const uint crc = sxm_crc32_calculate(self->root, tb->bsize * size);
            if (crc != tb->rootcrc) {
                TF_LOG(": invalid crc %08x vs %08x", crc, tb->rootcrc);
                rc = SXM_E_BAD_DB;
            }
            else {
                rc = SXM_E_OK;
            }
        }
        if ((rc != SXM_E_OK) && (self->root != NULL)) {
            sxe_free(self->root);
            self->root = NULL;
        }
    }

    return rc;
}

/***************************************************************************//**
 * Writes the root block to the file
 *
 * \param[in] self a pointer to a valid transaction file structure
 * \param[in] tb transaction block
 * \retval SXM_E_OK Success
 * \retval SXM_E_PIPE I/O Error
 ********************************************************************************/
static SXMResultCode tf_write_root(SXMTFile *self, SXMTFileBlock * const tb) {
    SXMResultCode rc;
    int written;

    tb->rootcrc = sxm_crc32_calculate(self->root, tb->bsize * tb->rootsize);
    TF_LOG("(crc=%08x)", tb->rootcrc);

    written = tf_write(self, tb->rootptr, tb->rootsize, self->root);
    if (written != (int)tb->rootsize) {
        TF_LOG(": Error writing tfile root, retsult=%d (%u, %u)",
            written, tb->rootptr, tb->rootsize);
        rc = SXM_E_PIPE;
    }
    else {
        rc = SXM_E_OK;
    }
    return rc;
}

/***************************************************************************//**
 * Commits dedicated transaction
 * \param[in] self a pointer to a valid transaction file structure
 * \param[in] id transaction block index
 * \retval SXM_E_OK Success
 * \retval SXM_E_PIPE I/O Error
 * \retval SXM_E_BAD_DB Invalid transaction id, thus, engine doesn't support it
 ********************************************************************************/
static SXMResultCode tf_commit_tblock(SXMTFile *self, const byte id) {
    SXMResultCode  rc;
    int intRc;

    switch (0) { default: {
        SXMTFileHeader * const hdr = self->hdr;

        /* If such valid was asked thus the file is not
         * designed for that engine.
         */  
        if (id >= ARRAY_SIZE(hdr->tb)) {
            rc = SXM_E_BAD_DB;
            break;
        }

        rc = tf_write_root(self, &hdr->tb[id]);
        if (rc != SXM_E_OK) {
            TF_LOG(": failed to write root #%d", id);
            break;
        }
        rc = SXM_E_PIPE; /* Error from here */

        /* Save current timestamp */
        hdr->tb[id].committs = (uint)sxm_time_now_utc();

        hdr->tb[id].bbcrc =
            sxm_crc32_calculate(&hdr->map[id], sizeof(hdr->map[id]));
        hdr->tb[id].crc =
            sxm_crc32_calculate(&hdr->tb[id], offsetof(SXMTFileBlock, crc));

        intRc = fseek(self->dbfile, sizeof(hdr->tb) + ((id != 0) ? sizeof(hdr->map[0]) : 0), SEEK_SET);
        if (intRc != 0) {
            TF_LOG(": fseek() first failed, result=%d", intRc);
            break;
        }
        intRc = (int)fwrite(&hdr->map[id], sizeof(hdr->map[id]), 1, self->dbfile);
        if (intRc != 1) {
            TF_LOG(": fwrite() first failed, result=%d", intRc);
            break;
        }
        intRc = fseek(self->dbfile, ((id != 0) ? sizeof(hdr->tb[0]) : 0), SEEK_SET);
        if (intRc != 0) {
            TF_LOG(": fseek() second failed, result=%d", intRc);
            break;
        }
        intRc = (int)fwrite(&hdr->tb[id], sizeof(hdr->tb[id]), 1, self->dbfile);
        if (intRc != 1) {
            TF_LOG(": fwrite(%u) second failed, result=%d", (uint)sizeof(hdr->tb[id]), intRc);
            break;
        }
        rc = SXM_E_OK;
    }};

    if (rc == SXM_E_OK) {
        intRc = fflush(self->dbfile);
        if (intRc != 0) {
            TF_LOG(": fflush() failed, result=%d", intRc);
            rc = SXM_E_PIPE;
        }
    }
    return rc;
}

/***************************************************************************//**
 * Allocate continuous set of blocks
 * \param[in] self a pointer to a valid transaction file structure
 * \param[in] len number of blocks
 * \param[out] start the first allocated block in case of success
 * \param[out] found_len the length of the longest set of continuous found in
 *                       in case if the required amount is not available. The
 *                       \p NULL can be passed if caller is not interested in
 *                       less blocks than were asked as \p len. If this parameter
 *                       provided the caller in case of SXM_E_NOMEM shall check it
 *                       to see how many blocks the function has allocated.
 * \retval SXM_E_OK Block(s) allocated
 * \retval SXM_E_INVAL Invalid argument
 * \retval SXM_E_NOMEM No more free block to fulfill the request
 ********************************************************************************/
static SXMResultCode tf_alloc(SXMTFile *self, uint len, uint *start, uint *found_len)
{
    const uint fsize = self->hdr->tb[self->active].fsize;
    SXMResultCode rc = SXM_E_NOMEM;
    uint idx, idx_last = 0U, seq_len = 0U;
    uint idx_last_long = 0U, seq_len_long = 0U;
    SXMTFileHeader *hdr = self->hdr;
    /* just keep beginning of the mask */
    uint * const smask = &hdr->temp.bits[0];
    /* get the reference to the mask item with the very first available
     * block for the user data.
     */
    uint *mask = &hdr->temp.bits[self->dataptr / SXM_ARCH_INT_BITS];
    /* get the reference to the mask item which lays after the item
     * with the latest valid block index
     */
    uint * const emask = &hdr->temp.bits[((fsize - 1) / (sizeof(*mask) * SXM_ARCH_BITS_IN_BYTE)) + 1];

    /* Make sure the value is clean and ready to be used. */
    if (found_len != NULL) {
        *found_len = 0;
    }

    /* In order to speed up searching process at the first stage we
     * can isolate the first mask item where at least of unused block
     * exists.
     */
    while ((*mask == TF_NONE_MASK) && (mask < emask)) {
        ++mask; /* move to the next item due to lack of unused blocks */
    }
    /* Get the absolute index of the first bit of the found mask item */
    idx = (uint)(mask - smask) * SXM_ARCH_BITS_IN_BYTE * (uint)sizeof(*mask);
    /* Another step is to reduce number of comparison by isolating a byte
     * with the first available unused block. In worst case it could
     * reduce number of bit-wise checks from 31 down to 9 (2 + 7) 
     */
    if (idx < fsize) {
        uint tmp_mask = *mask;
        /* If there is no unused block in the LS 2 bytes
         * skip them
         */
        if ((tmp_mask & 0x0000FFFF) == 0U) {
            /* adjust the searching start point */
            idx += SXM_ARCH_INT_BITS / 2;
            tmp_mask >>= (SXM_ARCH_INT_BITS / 2);
        }
        /* If there is no unused block in the LS byte skip it */
        if ((tmp_mask & 0x00FF) == 0U) {
            /* adjust the searching start point */
            idx += SXM_ARCH_BITS_IN_BYTE;
            tmp_mask >>= SXM_ARCH_BITS_IN_BYTE;
        }
    }
    /* start looking for the unused block if the index still within
     * the file size
     */
    while (idx < fsize) {
        /* Do bit level check */
        if (BITP(smask, idx)) {
            if (seq_len == 0U) {
                /* This is a first found unused block, thus, save it */
                idx_last = idx;
            }
            ++seq_len; /* Keep counting */
            if (seq_len == len) {
                /* Found the sequence */
                rc = SXM_E_OK;
                break;
            }
        }
        /* Update something if we're in the middle of set of continuous blocks. */
        else if (seq_len != 0U) {
            /* Just in case keep tracking. */
            if (seq_len > seq_len_long) {
                seq_len_long = seq_len;
                idx_last_long = idx_last;
            }
            /* Oops, need to find next set of continuous '1' bits */
            seq_len = 0U;
        }
        ++idx;
    }
    
    /* - Does file have required amount of continuous blocks?
     * - Can the caller accept less blocks?
     */
    if ((rc != SXM_E_OK) && (found_len != NULL)) {
        /* Is the latest sequence the longest one? */
        if (seq_len > seq_len_long) {
            rc = SXM_E_NOENT;
        }
        /* Is there some sequence? */
        else if (seq_len_long > 0U) {
            seq_len = seq_len_long;
            idx_last = idx_last_long;
            rc = SXM_E_NOENT;
        }
        /* No one above means the file has ran out of free blocks and cannot
         * provide any blocks for the caller.
         */
    }

    /* Found some amount of unused blocks */
    if (rc != SXM_E_NOMEM) {
        uint * const other_mask = &hdr->map[TF_OTHER(self)].bits[0];
        *start = idx_last;
        for (idx = idx_last; idx < (idx_last + seq_len); ++idx) {
            BITC(smask, idx);
            BITC(other_mask, idx);
        }
        /* Here we can be only if the found_len is not NULL. Thus, if the files
         * has less blocks let application know how many and covert the return
         * code back to NOMEM case to keep logic the same
         */
        if (rc == SXM_E_NOENT) {
            *found_len = seq_len;
            rc = SXM_E_NOMEM;
        }
        self->mask |= SXM_TFILE_MASK_MODIFIED;
    }

    return rc;
}

#ifndef SXM_DEBUG_PRODUCTION
/***************************************************************************//**
 * Print out details from the active transaction block
 * \param[in] self a pointer to a valid transaction file structure
 ********************************************************************************/
static void tf_print_tb(const SXMTFile *self) {
    UNUSED_VAR(self); /* Kind of unused var in case of no-print required */

    TF_LOG("    Using Header Block %d", self->active);
    TF_LOG("        Sequence Number: %u", self->hdr->tb[self->active].seq);
    TF_LOG("             Block Size: %u", self->hdr->tb[self->active].bsize);
    TF_LOG("        User Block Size: %u", sxm_tfile_bsize(self));
    TF_LOG("              File Size: %u", self->hdr->tb[self->active].fsize);
    TF_LOG("            Map Pointer: %u", self->hdr->tb[self->active].bbptr);
    TF_LOG("                Map CRC: %08X", self->hdr->tb[self->active].bbcrc);
    TF_LOG("           Root Pointer: %u", self->hdr->tb[self->active].rootptr);
    TF_LOG("              Root Size: %u", self->hdr->tb[self->active].rootsize);
    TF_LOG("               Root CRC: %08X", self->hdr->tb[self->active].rootcrc);
    TF_LOG("              Timestamp: %08x", self->hdr->tb[self->active].committs);

    return;
}
#endif

/**
 * \name Opening and Reading
 * @{
 */

/***************************************************************************//**
 * Provides a pointer to the user data area in the transaction header
 * \param[in] self a pointer to a valid transaction file structure
 * \return a pointer to the current user data area or \p NULL in case of error
 ********************************************************************************/
uint *sxm_tfile_user(SXMTFile *self) {
    return (self != NULL) ? self->hdr->tb[self->active].user : NULL;
}

/***************************************************************************//**
 * Returns the size of each user-defined data block
 * \param[in] self a pointer to a valid transaction file structure
 * \return the user block size or \p 0 in case of error
 ********************************************************************************/
uint sxm_tfile_bsize(const SXMTFile *self) {
    uint ret;
    if (self != NULL) {
        ret = self->hdr->tb[self->active].bsize - SXM_TFILE_BLOCK_RESERVED;
    }
    else {
        ret = 0U;
    }
    return ret;
}

/***************************************************************************//**
 * Update version of the t-file's content.
 * 
 * \param[in] self a pointer to a valid transaction file structure
 * \param[in] ver new version to be set
 * \retval SXM_E_OK Success
 * \retval SXM_E_FAULT NULL Parameter
 * \retval SXM_E_STATE Wrong state
 ********************************************************************************/
int sxm_tfile_update_version(SXMTFile *self, uint ver) {
    SXMResultCode rc;
    if (self == NULL) {
        rc = SXM_E_FAULT;
    }
    else if ((self->mask & SXM_TFILE_MASK_TRANSACTION) != SXM_TFILE_MASK_TRANSACTION) {
        rc = SXM_E_STATE;
    }
    else {
        self->hdr->tb[TF_OTHER(self)].user[0] = ver;
        rc = SXM_E_OK;
    }

    return rc;
}

/***************************************************************************//**
 * Provides general statistics over the T-File.
 * 
 * \param[in] self a pointer to a valid transaction file structure
 * \param[out] st the placeholder to be filled by data in case of success
 * \retval SXM_E_OK Success
 * \retval SXM_E_FAULT NULL Parameter
 ********************************************************************************/
int sxm_tfile_stat(const SXMTFile *self, SXMTFileStat *st) {
    SXMResultCode rc;
    if ((self == NULL) || (st == NULL)) {
        rc = SXM_E_FAULT;
    }
    else {
        uint idx;

        memset(st, 0, sizeof(*st));

        st->tschema = self->hdr->tver;
        st->dschema = self->hdr->sver;
        st->dversion = self->hdr->tb[self->active].user[0];
        st->seq = self->hdr->tb[self->active].seq;
        st->bsize = self->hdr->tb[self->active].bsize;
        st->ubsize = sxm_tfile_bsize(self);
        st->fsize = self->hdr->tb[self->active].fsize;
        st->transaction = sxm_tfile_transaction_state(self);
        st->usize = 0U;

        /* Count number of unused blocks (based on current logic the map
         * has the same size, but we can simply count unused blocks regardless
         * real size of the file in blocks.
         */
        for (idx = 0; idx < ARRAY_SIZE(self->hdr->map[self->active].bits); ++idx) {
            uint x = self->hdr->map[self->active].bits[idx];
            x = x - ((x >> 1) & 0x55555555);
            x = (x & 0x33333333) + ((x >> 2) & 0x33333333);
            st->usize += (((x + (x >> 4)) & 0x0F0F0F0F) * 0x01010101) >> 24;
        }
        /* Simple math to find out used blocks count */
        st->usize = st->fsize - st->usize;
        rc = SXM_E_OK;
    }
    return rc;
}

/***************************************************************************//**
 * Returns a pointer to the current user-defined root block for the tfile
 * \param[in] self a pointer to a valid transaction file structure
 * \return a pointer to the root block of the transaction file or \p NULL in
 *         case of error
 ********************************************************************************/
void *sxm_tfile_root(SXMTFile *self) {
    return (self != NULL) ? self->root : NULL;
}

/***************************************************************************//**
 * Returns a current transactions count for the tfile
 * \param[in] self a pointer to a valid transaction file structure
 * \return a transactions count or \p 0 in case of error
 ********************************************************************************/
uint sxm_tfile_transactions(SXMTFile *self) {
    return (self != NULL) ? self->hdr->tb[self->active].seq : 0U;
}

/***************************************************************************//**
 * Closes the file and frees any associated memory
 * \param[in] self a pointer to a valid tfile structure
 ********************************************************************************/
void sxm_tfile_close(SXMTFile *self) {
    if (self != NULL) {
        if (self->dbfile != NULL) {
            sxm_file_close(self->dbfile);
        }
        if (self->root != NULL) {
            sxe_free(self->root);
        }
        if (self->root_origin != NULL) {
            sxe_free(self->root_origin);
        }
        sxe_free(self);
    }
    return;
}

/***************************************************************************//**
 * Creates a file in the filing system as a transaction file.
 *
 * In case of success the file will be created ans ready to be filled up
 * by the creator. All other APIs can be used to fill up t-file from any
 * sources the caller has. Please, be awarded that newly created via this function
 * doesn't support rollback feature and two different data sets since based on
 * design both transaction blocks have the same data, so, each commit operation
 * (\ref sxm_tfile_commit()) makes them identical. Thus, if the file owner wants
 * to have two separate data sets and have ability to rollback to the previous
 * one the file shall be reopened in corresponding mode via \ref sxm_tfile_open()
 * function.
 *
 * Another feature the function supports is creation of the dynamically growing
 * file. This functionality is turned on by passing SXM_TFILE_DYNAMIC_NO_OF_BLOCKS
 * as file size in blocks.
 * \warning The dynamic file supports only one active transaction. In other words,
 *          the file creator does one one sxm_tfile_start() followed by only one
 *          sxm_tfile_commit(). If this rule is broken by some reason the caller
 *          will not be able to do any write operations since the files will be
 *          truncated during sxm_tfile_commit() routine.  
 *
 * \note In case of failure no file is created, thus, the caller should
 *       not aware of removing it.
 *
 * \param[in] type the file location type
 * \param[in] module the name of the module requesting the file
 * \param[in] file the name of the file being requested
 * \param[in] format the internal data format unique identifier
 * \param[in] format_schema required schema version for the format.
 * \param[in] schema required T-File internal schema
 * \param[in] root_size root size on bytes
 * \param[in] blocks number of blocks, can be not more than 
 *                   SXM_TFILE_ALLOC_MAP_BITLEN or equals to
 *                   SXM_TFILE_DYNAMIC_NO_OF_BLOCKS for dynamic file.
 * \param[in] block_size block size in bytes
 * \param[in] pRc a pointer to an integer to hold the return code
 *
 * \return a pointer to the SXMTFile, or \c NULL if the file cannot be created
 *
 ********************************************************************************/
SXMTFile *sxm_tfile_create(char type, const char *module, const char *file,
                           const char *format, uint format_schema, uint schema,
                           uint root_size, uint blocks, uint block_size,
                           int *pRc)
{
    SXMTFile *ret = NULL;
    SXMResultCode rc = SXM_E_OK;

    switch (0) { default: {
        size_t ioRc;
        int intRc;
        uint idx;
        const uint hdrBlocks = (block_size > 0U) ? NUM_ALIGNED_BLOCKS(SXM_TFILE_HEADER_BYTELEN, block_size) : 0U;
        const uint hdrSize = hdrBlocks * block_size;
        const uint rootBlocks = (block_size > 0U) ? NUM_ALIGNED_BLOCKS(root_size, + block_size) : 0U;
        uint realBlocks; 

        /* Check input */
        if ((module == NULL) || (file == NULL) || (format == NULL) ||
            (blocks == 0U) || (block_size == 0U) || (pRc == NULL)) {
            rc = SXM_E_FAULT;
            break;
        }

        /* Check max supported file size */
        if ((blocks > SXM_TFILE_ALLOC_MAP_BITLEN) &&
            (blocks != SXM_TFILE_DYNAMIC_NO_OF_BLOCKS)) {
            TF_LOG(": unsupported number of blocks %u", blocks);
            rc = SXM_E_INVAL;
            break;
        }

        /* Check supported schema versions */
        if (schema != SXM_TFILE_VERSION) {
            rc = SXM_E_INVAL;
            TF_LOG(": unsupported T-File own schema %u", schema);
            break;
        }

        /* Create the object space for the header */
        ret = (SXMTFile *)sxe_calloc(1, sizeof(*ret) + hdrSize);
        if (ret == NULL) {
            rc = SXM_E_NOMEM;
            break;
        }
        /* Adjust header pointer */
        ret->hdr = (SXMTFileHeader *)(ret + 1);

        /* Open the file for creation */
        ret->dbfile = sxm_file_open(type, module, file, "wb");
        if (ret->dbfile == NULL) {
            rc = SXM_E_PIPE;
            break;
        }

        /* Init header */
        ret->hdr->magic = SXM_TFILE_MAGIC;
        ret->hdr->tver = schema;
        strncpy(ret->hdr->format, format, sizeof(ret->hdr->format) - 1);
        ret->hdr->sver = format_schema;
        ret->active = 0;
        ret->mask = SXM_TFILE_MASK_CREATION;
        if (blocks == SXM_TFILE_DYNAMIC_NO_OF_BLOCKS) {
            /* In case of dynamic file just set max supported number of blocks
             * by design and let caller do it's writes later to find out real
             * file size.
             */
            ret->mask |= SXM_TFILE_MASK_CREATION_DYN;
            realBlocks = SXM_TFILE_ALLOC_MAP_BITLEN;
        }
        else {
            realBlocks = blocks; 
        }
        ret->dataptr = hdrBlocks;

        /* Resize file (header) */
        if (rootBlocks > 0U) {
            ret->root = sxe_calloc(rootBlocks, block_size);
            if (ret->root == NULL) {
                rc = SXM_E_NOMEM;
                break;
            }
        }
        else {
            ret->root = NULL;
        }

        /* Init T-blocks */
        for (idx = 0; idx < ARRAY_SIZE(ret->hdr->tb); ++idx) {
            SXMTFileBlock *tb = &ret->hdr->tb[idx];
            tb->seq = 1;
            tb->bsize = block_size;
            tb->fsize = realBlocks;
            tb->rootptr = (rootBlocks > 0U) ? (hdrBlocks + idx * rootBlocks) : 0U;
            tb->rootsize = rootBlocks;
            ret->dataptr = MAX(ret->dataptr,  tb->rootptr + tb->rootsize);
#if !defined(SXM_DEBUG_PRODUCTION) || defined(SDKFILES_STANDALONE_BUILD)
            /* Store the CRC16 for user data section */
            ret->crcuser[idx] =
                sxm_crc16_calculate(&ret->hdr->tb[idx].user[0], sizeof(ret->hdr->tb[idx].user));
            /* Compute CRC32 for user root data section */
            tb->rootcrc = sxm_crc32_calculate(ret->root,
                ret->hdr->tb[idx].bsize * ret->hdr->tb[idx].rootsize);
#endif
        }

        /* Init maps */
        for (idx = (uint)ARRAY_SIZE(ret->hdr->tb) * rootBlocks + hdrBlocks;
             idx < realBlocks; ++idx) {
            BITS(ret->hdr->map[0].bits, idx);
        }
        for (idx = 1; idx < ARRAY_SIZE(ret->hdr->map); ++idx) {
            ret->hdr->map[idx] = ret->hdr->map[0];
        }

        /* Start constructing the file */
        intRc = fseek(ret->dbfile, 0, SEEK_SET);
        if (intRc != 0) {
            TF_LOG("fseek(0) has failed, result=%d", intRc);
            rc = SXM_E_PIPE;
            break;
        }

        /* Write header block(s) first */
        ioRc = fwrite(ret->hdr, hdrSize, 1, ret->dbfile);
        if (ioRc != 1) {
            rc = SXM_E_PIPE;
            break;
        }

        /* Enlarge file up to required number of blocks. On some systems the
         * fseek() can allow increasing file using internal file systems logic,
         * but this is not a portable way of doing this. In order to have
         * portability the straight forward way of writing blocks back
         * to back is used.
         */
        ioRc = rootBlocks;
        /* Write root sectors for all transaction blocks */
        for (idx = 0; (idx < ARRAY_SIZE(ret->hdr->tb)) && (ioRc == rootBlocks); ++idx) {
            ioRc = fwrite(ret->root, block_size, rootBlocks, ret->dbfile);
        }
        if (ioRc != rootBlocks) {
            TF_LOG(": failed to write root block data #%u", idx);
            rc = SXM_E_PIPE;
            break;
        }

        if ((ret->mask & SXM_TFILE_MASK_CREATION_DYN) != SXM_TFILE_MASK_CREATION_DYN) {
            /* In case if the root not required the blank memory block
             * is needed to make the size of the t-file as expected.
             */
            void *pattern = (ret->root != NULL) ? ret->root : sxe_calloc(1, block_size);
            if (pattern == NULL) {
                TF_LOG(": there is no memory for file size adjustment");
                rc = SXM_E_NOMEM;
                break;
            }
            ioRc = 1;
            /* Enlarge file in case of non-dynamic file up to requested files
             * size in blocks
             */
            for (idx = hdrBlocks + rootBlocks * (uint)ARRAY_SIZE(ret->hdr->tb);
                (idx < realBlocks) && (ioRc == 1); ++idx) {
                /* Using the data from the root block let's
                 * fill up all file making it as large as needed.
                 */
                ioRc = fwrite(pattern, block_size, 1, ret->dbfile);
            }
            /* Release the memory if it was allocated here */
            if (ret->root == NULL) {
                sxe_free(pattern);
            }
            if (ioRc != 1) {
                rc = SXM_E_PIPE;
                break;
            }
        }

        /* Commit both blocks */
        for (idx = 0; (idx < ARRAY_SIZE(ret->hdr->tb)) && (rc == SXM_E_OK); ++idx) {
            rc = tf_commit_tblock(ret, (byte)idx);
        }
    }};

    /* Clean up resources in case of error */
    if ((rc != SXM_E_OK) && (ret != NULL)) {
        sxm_tfile_close(ret);
        ret = NULL;

        /* Try to remove this file */
        (void) sxm_file_remove(type, module, file);
    }

    /* Populate result if requested */
    if (pRc != NULL) {
        *pRc = rc;
    }

    return ret;
}

/***************************************************************************//**
 * Opens a file in the filing system as a transaction file.
 *
 * \param[in] type the file location type
 * \param[in] module the name of the module requesting the file
 * \param[in] file the name of the file being requested
 * \param[in] mode the file open mode ('rb' or 'w+b' usually)
 * \param[in] format the internal data format unique identifier
 * \param[in] schema a pointer to an unsigned integer to hold opened
 *                   T-File schema.
 * \param[in] rc a pointer to an integer to hold the return code
 *
 * \return a pointer to the SXMTFile, or NULL if the file cannot be opened
 *
 ********************************************************************************/
SXMTFile *sxm_tfile_open(char type, const char *module, const char *file,
                         const char *mode, const char *format,
                         uint *schema, int *pRc)
{
    SXMTFile *ret = NULL;
    SXMResultCode rc;

    switch (0) { default: {
        uint valid = 0U;
        size_t idx, ioRc;
        int intRc;
        SXMResultCode localRc;

        /* Check input */
        if ((module == NULL) || (file == NULL) || (mode == NULL) || (format == NULL)) {
            rc = SXM_E_FAULT;
            break;
        }

        /* Initialize the return value if requested */
        if (schema != NULL) {
            *schema = (uint)-1;
        }

        ret = (SXMTFile*)sxe_calloc(1, sizeof(*ret) + SXM_TFILE_HEADER_BYTELEN);
        if (ret == NULL) {
            rc = SXM_E_NOMEM;
            break;
        }
        ret->hdr = (SXMTFileHeader *)(ret + 1);

        ret->dbfile = sxm_file_open(type, module, file, mode);
        if (ret->dbfile == NULL) {
            rc = SXM_E_NO_DB;
            break;
        }

        /* any error after here is 'file corrupt' */
        rc = SXM_E_BAD_DB;

        /* Read the header */
        ioRc = fread(ret->hdr, SXM_TFILE_HEADER_BYTELEN, 1, ret->dbfile);
        if (ioRc != 1) {
            TF_LOG("(bad header)");
            break;
        }

        /* Do the preliminary header check */
        if ((SXM_TFILE_MAGIC != ret->hdr->magic) ||
            (SXM_TFILE_VERSION != ret->hdr->tver))
        {
            TF_LOG("(bad file format) magic=%x, tver=%u", ret->hdr->magic, ret->hdr->tver);
            break;
        }
        intRc = strncmp(ret->hdr->format, format, sizeof(ret->hdr->format));
        if (intRc != 0) {
            TF_LOG("(bad data format) format=%s", ret->hdr->format);
            break;
        }

        /* Populate the real schema to the caller if requested */
        if (schema != NULL) {
            *schema = ret->hdr->sver;
        }

        for (idx = 0; idx < ARRAY_SIZE(ret->hdr->tb); ++idx) {
            const SXMTFileBlock *tb = &ret->hdr->tb[idx];
            const uint crc0 = sxm_crc32_calculate(tb, offsetof(SXMTFileBlock, crc));
            ret->dataptr = MAX(ret->dataptr, tb->rootptr + tb->rootsize);
            if  (crc0 == ret->hdr->tb[idx].crc) {
                uint crc1;
                TF_LOG("Header block #%u is valid (seq %u)", (uint)idx, tb->seq);
                crc1 = sxm_crc32_calculate(&ret->hdr->map[idx], sizeof(ret->hdr->map[idx]));
                if  (crc1 == tb->bbcrc) {
                    TF_LOG("Allocation Map %d is valid", (int)idx);
                    valid |= (1U << idx);
#if !defined(SXM_DEBUG_PRODUCTION) || defined(SDKFILES_STANDALONE_BUILD)
                    /* Store the CRC16 for user data section */
                    ret->crcuser[idx] =
                        sxm_crc16_calculate(&tb->user[0], sizeof(tb->user));
#endif
                }
            }
        }

        TF_LOG("(valid-start=%u)", valid);
        /* no valid header blocks, return NULL */
        if (valid == 0U) {
            break;
        }

        /* both can be used, i.e. more than one bit is set */
        if ((valid & (valid - 1U)) != 0U) {
            valid = (ret->hdr->tb[0].seq >= ret->hdr->tb[1].seq) ? 0U : 1U;
            /* try valid 'better' block first */
            localRc = tf_read_root(ret, &ret->hdr->tb[valid]);
            if (localRc == SXM_E_OK) {
                TF_LOG("(return-valid=%u)", valid);
                rc = SXM_E_OK;
            }
            else {
                TF_LOG("failed to load #%u (rc=%d), try another", valid, localRc);
                valid = TF_OTHER_IDX(valid); /* calculate the 'other' */
                localRc = tf_read_root(ret, &ret->hdr->tb[valid]);
                if (localRc == SXM_E_OK) {
                    TF_LOG("(return-valid=%u)", valid);
                    rc = SXM_E_OK;
                }
                else {
                    TF_LOG("failed to load root block for all transactions");
                }
            }
        }
        /* only one is valid at this point */
        else {
            valid = (valid == 1U) ? 0U /* block #0 */ : 1U /*  block #1 */;

            localRc = tf_read_root(ret, &ret->hdr->tb[valid]);
            if (localRc == SXM_E_OK) {
                TF_LOG("(return-valid=%u)", valid);
                rc = SXM_E_OK;
            }
            else {
                TF_LOG("failed to read the only valid root #%u (rc=%d)", valid, localRc);
            }
        }
        /* Leave this if nothing is ok */
        if (rc != SXM_E_OK) {
            break;
        }
        /* store the valid block which will be used */
        ret->active = (byte)valid;

        /* If by some reason the first user block is unknown at this point
         * the simple math can be used to find out the size of the header 
         * in blocks and assume that the very first user block lays right
         * after the header. In fact, such situation means that the file
         * has no root blocks for both transactions and this is how the file's
         * user designed it.
         * This calculation shall be done only here after the check above which
         * found the valid transaction block to be used as active one.
         */
        if (ret->dataptr == 0U) {
            const uint bsize = ret->hdr->tb[ret->active].bsize;
            ret->dataptr = NUM_ALIGNED_BLOCKS(SXM_TFILE_HEADER_BYTELEN, bsize);
        }

#ifndef SXM_DEBUG_PRODUCTION
        tf_print_tb(ret);
#endif
    }};

    if ((rc != SXM_E_OK) && (ret != NULL)) {
        sxm_tfile_close(ret);
        ret = NULL;
    }

    /* Populate to the app if needed */
    if (pRc != NULL) {
        *pRc = rc;
    }

    return ret;
}

/** @} */

/**
 * \name Transactions and Updates
 * @{
 */

/***************************************************************************//**
 * Starts a new update transaction. Transactions either
 * complete entirely, or are automatically rolled back at the next
 * system restart.
 * \param[in] self a pointer to a valid transaction file structure
 * \retval SXM_E_OK Success
 * \retval SXM_E_NOMEM There is no memory to start transaction
 * \retval SXM_E_BUSY Transaction in progress
 * \retval SXM_E_FAULT NULL Parameter
 ********************************************************************************/
int sxm_tfile_start(SXMTFile *self) {
    SXMResultCode rc;

    if (self == NULL) {
        rc = SXM_E_FAULT;
    }
    else if ((self->mask & SXM_TFILE_MASK_TRANSACTION) == SXM_TFILE_MASK_TRANSACTION) {
        rc = SXM_E_BUSY;
    }
    else {
        const byte other = TF_OTHER(self);
        SXMTFileHeader * const hdr = self->hdr;
        SXMTFileBlock * const block = &hdr->tb[self->active];
        const uint rootsize = block->rootsize * block->bsize;

        TF_LOG("(other=%d)", other);
        
        /* Allocate storage for the root block if needed. The lazy initialization
         * needed since the file can be opened for read-only purposes and
         * there is no need to keep the second copy of the root. In the same
         * time there is no need to do that if we're creating the file since
         * file is new and there should be no cancel actions in good design.
         */
        rc = SXM_E_OK;
        if (!(self->mask & SXM_TFILE_MASK_CREATION) && (self->root_origin == NULL)) {
            self->root_origin = sxe_malloc(rootsize + sizeof(block->user));
            if (self->root_origin == NULL) {
                rc = SXM_E_NOMEM;
            }
            else {
                self->user_origin = (uint*)(void*)((byte*)self->root_origin + rootsize);
            }
        }
        if (rc == SXM_E_OK) {
            /* Stash the root block and other-user data section as they're now */
            if (self->root_origin != NULL) {
                memcpy(self->root_origin, self->root, rootsize);
                memcpy(self->user_origin, hdr->tb[other].user,
                    sizeof(hdr->tb[other].user));
            }

            /* setup other transaction block and unified allocation map */
            hdr->tb[other].seq = block->seq + 1;
            memcpy(hdr->tb[other].user, hdr->tb[self->active].user,
                   sizeof(hdr->tb[other].user));
            hdr->map[other] = hdr->map[self->active];
            hdr->temp = hdr->map[self->active];
            self->mask |= SXM_TFILE_MASK_TRANSACTION;
        }
    }
    return rc;
}

/***************************************************************************//**
 * Cancels active transaction
 * \param[in] self a pointer to a valid transaction file structure
 *
 * \retval SXM_E_OK in case of success
 * \retval SXM_E_STATE if the T-File has no transaction
 * \retval SXM_E_FAULT if the passed argument is NULL
 *
 ********************************************************************************/
int sxm_tfile_cancel(SXMTFile *self) {
    SXMResultCode rc;

    if (self == NULL) {
        rc = SXM_E_FAULT;
    }
    else if ((self->mask & SXM_TFILE_MASK_TRANSACTION) != SXM_TFILE_MASK_TRANSACTION) {
        rc = SXM_E_STATE;
    }
    else {
        SXMTFileBlock * const block = &self->hdr->tb[self->active];
        SXMTFileBlock * const other = &self->hdr->tb[TF_OTHER(self)];
        /* Restore previous sequence number */
        other->seq = block->seq;
        /* Restore content of the root */
        if (self->root_origin != NULL) {
            memcpy(self->root, self->root_origin, block->rootsize * block->bsize);
            memcpy(other->user, self->user_origin, sizeof(other->user));
        }
        /* Make transaction inactive */
        self->mask &= ~(SXM_TFILE_MASK_TRANSACTION | SXM_TFILE_MASK_MODIFIED);
        rc = SXM_E_OK;
    }

    return rc;
}

/***************************************************************************//**
 * Commits active transaction
 * \param[in] self a pointer to a valid transaction file structure
 * \return SXM_E_OK or an error code
 ********************************************************************************/
int sxm_tfile_commit(SXMTFile *self) {
    SXMResultCode rc;

    if (self == NULL) {
        rc = SXM_E_FAULT;
    }
    else if ((self->mask & SXM_TFILE_MASK_TRANSACTION) != SXM_TFILE_MASK_TRANSACTION) {
        rc = SXM_E_STATE;
    }
    else {
#if !defined(SXM_DEBUG_PRODUCTION) || defined(SDKFILES_STANDALONE_BUILD)
        uint tb_idx, rootcrc;
#endif
        const byte other = TF_OTHER(self);
        TF_LOG("(other=%d)", (int)other);
#if !defined(SXM_DEBUG_PRODUCTION) || defined(SDKFILES_STANDALONE_BUILD)
        /* Check CRC16 for each of user data */
        for (tb_idx = 0; tb_idx < ARRAY_SIZE(self->hdr->tb); ++tb_idx) {
            const ushort crc =
                sxm_crc16_calculate(&self->hdr->tb[tb_idx].user[0],
                                    sizeof(self->hdr->tb[tb_idx].user[0]));
            if (crc != self->crcuser[tb_idx]) {
                TF_LOG(": user data for %d is changed", (int)tb_idx);
                self->mask |= SXM_TFILE_MASK_MODIFIED;
            }
        }
        /* Check CRC32 for the root */
        rootcrc = sxm_crc32_calculate(self->root,
            self->hdr->tb[self->active].bsize * self->hdr->tb[self->active].rootsize);
        if (rootcrc != self->hdr->tb[self->active].rootcrc) {
            TF_LOG(": root's user data for is changed");
            self->mask |= SXM_TFILE_MASK_MODIFIED;
        }
        /* Any changes? */
        if (!(self->mask & SXM_TFILE_MASK_MODIFIED)) {
            TF_LOG(": trying to commit file, which has no modifications");
        }
#endif
         /* This is a special case for files creation where the caller doesn't
          * know the file size and wished to make it dynamic. This is a common
          * case for read-only t-files where we don't need to take care about
          * spear space for further updates.
          */
        if ((self->mask & SXM_TFILE_MASK_CREATION_DYN) == SXM_TFILE_MASK_CREATION_DYN) {
            /* Need to adjust file size based on the allocation map */
            uint * const map = &self->hdr->map[other].bits[0];
            int item;
            /* Step 1: find the first items with at least one indication for
             *         used t-file block
             */
            for (item = ARRAY_SIZE(self->hdr->map[other].bits) - 1; item >= 0; --item) {
                /* All bits set means all blocks are unused */
                if (map[item] != TF_ALL_MASK) {
                    break;
                }
            }

            if (item >= 0) {
                int idx;
                /* Got it. */
                TF_LOG(": the first map item with used blocks is %d (%08x)", 
                    item, map[item]);
                /* Step 2: find out which bit is used the last in this item */
                for (idx = (sizeof(*map) * SXM_ARCH_BITS_IN_BYTE) - 1; idx >= 0; --idx) {
                    if (!ISBITSET(map[item], idx)) {
                        break;
                    }
                }
                if (idx >= 0) {
                    /* Calculate file size and set it to the t-block */
                    self->hdr->tb[other].fsize =
                        (uint)((uint)item * sizeof(*map) * SXM_ARCH_BITS_IN_BYTE) + (uint)idx
                        + 1U /* index to count */;
                    TF_LOG(": the first bit with used blocks is %d in %08x, fsize is %u", 
                        idx, map[item], self->hdr->tb[other].fsize);
                    /* Seal the file marking all blocks as used: */
                    memset(map, 0, sizeof(self->hdr->map[other].bits));
                }
            }
            /* Not longer such file */
            self->mask &= ~SXM_TFILE_MASK_CREATION_DYN;
        }
        rc = tf_commit_tblock(self, other);
        if (rc == SXM_E_OK) {
            /* In creation mode both transaction blocks shall
             * have the same information and reference the same
             * blocks, thus, update the active transaction
             * as well
             */
            if ((self->mask & SXM_TFILE_MASK_CREATION) == SXM_TFILE_MASK_CREATION) {
                const uint rootptr = self->hdr->tb[self->active].rootptr;
                TF_LOG("commiting the active transaction %d", self->active);
                self->hdr->tb[self->active] = self->hdr->tb[other];
                self->hdr->tb[self->active].rootptr = rootptr;
                self->hdr->map[self->active] = self->hdr->map[other];
                rc = tf_commit_tblock(self, self->active);
            }
            self->active = other;

            /* Make transaction inactive */
            self->mask &= ~(SXM_TFILE_MASK_TRANSACTION | SXM_TFILE_MASK_MODIFIED);
        }

    }
    return (int)rc;
}

/***************************************************************************//**
 * Returns a pointer to the user data area in the \b other transaction block.
 * This is the block that will become the current block once the transaction is
 * committed.
 * \param[in] self a pointer to a valid transaction file structure
 * \return a pointer to the user data in the inactive transaction block
 ********************************************************************************/
uint *sxm_tfile_other_user(SXMTFile *self) {
    uint *pRes;
    if (self == NULL) {
        pRes = NULL;
    }
    else {
        TF_LOG("(other=%d)", (int)TF_OTHER(self));
        pRes = &self->hdr->tb[TF_OTHER(self)].user[0];
    }
    return pRes;
}

/***************************************************************************//**
 * Provides T-File transaction state
 * \param[in] self a pointer to a valid transaction file structure
 *
 * \retval TRUE Transaction in progress
 * \retval FALSE Transaction file structure pointer is NULL or
 *               file not in transaction state
 ********************************************************************************/
BOOL sxm_tfile_transaction_state(const SXMTFile *self) {
    BOOL rc;
    if ((self != NULL) && ((self->mask & SXM_TFILE_MASK_TRANSACTION) == SXM_TFILE_MASK_TRANSACTION)) {
        rc = TRUE;
    }
    else {
        rc = FALSE;
    }
    return rc;
}

/* Chain structure definitions */ 
struct _sxm_tfile_chain {
    ushort count; //!< Number of items addressed by \p ids
    ushort ids[1]; //!< Keeps set of block id allocated together
};

/***************************************************************************//**
 * This function makes the block signature. In addition to packing the values
 * passed to the function the checksum is calculated to allow block's signature
 * verification.
 *
 * \param[in] next_id block next id in the chain
 * \param[in] is_begin sign of the chain start
 * \param[in] is_end sign of the chain end
 * \return completed signature
 ********************************************************************************/
static uint tf_sig_pack(const ushort next_id, uint is_begin, uint is_end) {
    const uint ret =
               ((is_begin ? TFILE_SIG_IS_BEGIN_MASK : 0) |
               (is_end ? TFILE_SIG_IS_END_MASK : (((uint)(~next_id) & 0xFFFFU) << 8U)));
    return ret | TFILE_SIG_CALC_CS(ret);
}

/***************************************************************************//**
 * This function verifies the signature to make sure it can be trusted and
 * used for further operations
 *
 * \param[in] self a pointer to a valid transaction file structure
 * \param[in] sig the signature for verification
 * \retval TRUE the signature is valid and can be used
 * \retval FALSE invalid signature detected
 ********************************************************************************/
static BOOL tf_sig_check(const SXMTFile *self, const uint sig) {
    BOOL rc = FALSE;
    if (TFILE_SIG_UNPACK_CS(sig) == TFILE_SIG_CALC_CS(sig)) {
        const ushort next_id = TFILE_SIG_UNPACK_NEXTID(sig);
        if (!(sig & TFILE_SIG_IS_END_MASK) &&
            (self->dataptr <= next_id) &&
            (next_id < self->hdr->tb[self->active].fsize)) {
            rc =TRUE;
        }
        else if ((sig & TFILE_SIG_IS_END_MASK) == TFILE_SIG_IS_END_MASK) {
            rc =TRUE;
        }
    }
    return rc;
}

/***************************************************************************//**
 * Reads a block with signature from the file
 * \warning this function shall be used only for reading blocks from t-file
 *          with support of chained blocks. There is non internal check on this
 *          thus please use carefully.
 * \param[in] self a pointer to a valid transaction file structure
 * \param[in] block the index of the first block to be read
 * \param[out] sig placeholder for the block's sig in case of success
 * \param[in,out] dst a pointer to the area which will receive the data
 * \return the number of block read or \p -1 in case of error
 ********************************************************************************/
static int tf_read_ex(SXMTFile *self, size_t block, uint *sig, void *dst) {
    const uint bsize = self->hdr->tb[self->active].bsize;
    int rc = -1, intRc;
    switch (0) { default: {
        if (block >= SXM_TFILE_ALLOC_MAP_BITLEN) {
            TF_LOG(": block #%u is out of allocation map", (uint)block);
            break;
        }
        /* Make sure the block is allocated */
        if (!TF_BLOCK_IS_USED(self, block)) {
            TF_LOG(": %u id is unused, i.e. unreliable", (uint)(bsize * block));
            break;
        }
        intRc = fseek(self->dbfile, (long)(bsize * block), SEEK_SET);
        if (intRc != 0) {
            TF_LOG(": fseek(%u) failed, result=%d", (uint)(bsize * block), intRc);
            break;
        }
        intRc = (int)fread(sig, SXM_TFILE_BLOCK_RESERVED, 1, self->dbfile);
        if (intRc != 1) {
            TF_LOG(": fread(%u) failed reading sig, result=%d", SXM_TFILE_BLOCK_RESERVED, intRc);
            break;
        }
        /* Validate signature checksum to ensure this is what was
         * written to the file.
         */
        if (tf_sig_check(self, *sig) == FALSE) {
            TF_LOG(": corrupted signature %02d vs %02d",
                TFILE_SIG_CALC_CS(*sig), TFILE_SIG_UNPACK_CS(*sig));
            break;
        }
        /* Read the block data only if the caller wished this. */
        if (dst != NULL) {
            /* Read remaining part of the block */
            intRc = (int)fread(dst, sxm_tfile_bsize_static(bsize), 1, self->dbfile);
            if (intRc != 1) {
                TF_LOG(": fread(%u) failed reading block content, result=%d",
                    sxm_tfile_bsize_static(bsize), intRc);
                break;
            }
        }
        rc = 1;
    }};
    return rc;
}

/***************************************************************************//**
 * Reads data from the file into an area into memory supplied by the application
 * it wishes. It does reading the file since the \p block up to the \p count
 * depending on the internal connections between them. Thus, the functions loads
 * blocks until the end block detected or the caller asks less.
 *
 * In some cases the caller may not need to keep the chain instance, thus, the 
 * caller may omit this parameter passing \p NULL - in this case the function
 * does only data reading.
 *
 * Another option is then \p dst is passed as \p NULL. In this cases the function
 * constructs the chains, but doesn't read the file content.
 *
 * \param[in] self a pointer to a valid transaction file structure
 * \param[in] block the starting (user) block number in the file
 * \param[in] count the number of  (user) block to read
 * \param[out] ppChain keeps chain's descriptor in case of success
 * \param[in,out] dst a pointer to the area which will receive the data
 *
 * \return count of blocks or \p -1 in case of error
 ********************************************************************************/
static int tf_read_chain(SXMTFile *self, uint block, uint count, SXMTFileChain **ppChain, void *dst) {
    int rc = 0;
    SXMTFileChain *pChain = NULL;
    /* Create the output chain only if the caller wants to know it. */
    if (ppChain != NULL) {
        pChain = TFILE_NEW_CHAIN(count);
        if (pChain == NULL) {
            TF_LOG(": Unable to allocate memory for chain handler");
            rc = -1;
        }
    }
    if (rc == 0) {
        byte *pDst = (byte*)dst;
        ushort next_block = (ushort)block;
        uint sig[NUM_ALIGNED_BLOCKS(SXM_TFILE_BLOCK_RESERVED, sizeof(uint))];
        uint idx;
        rc = 0;
        for (idx = 0; idx < count;) {
            TF_LOG(": loading #%u block", next_block);
            /* The theory saying that the seek operation over files
             * where only reading operations are performed causes
             * no additional low-level I/O calls in case if the seek
             * tries to move reading point to the same location where
             * it is right now.
             */
            if (tf_read_ex(self, next_block, &sig[0], pDst) != 1) {
                TF_LOG(": failed reading block %u", next_block);
                rc = -1;
                break;
            }
#ifndef SXM_DEBUG_PRODUCTION
            /* This is a simple warning which may indicate to the developer
             * that something is wrong with the block(s) the code is reading.
             */
            if ((next_block == block) && !(sig[0] & TFILE_SIG_IS_BEGIN_MASK)) {
                TF_LOG(": block %d is not the start of the chain", block);
            }
#endif
            /* Put the index into the chain if it exists */
            if (pChain != NULL) {
                pChain->ids[pChain->count++] = next_block;
            }
            ++idx; /* One more block is processed */
            if ((sig[0] & TFILE_SIG_IS_END_MASK) == TFILE_SIG_IS_END_MASK) {
                TF_LOG(": End of the chain detected in the file, found %u block(s)",
                    idx);
                break;
            }
            /* Keep going */
            next_block = (ushort)TFILE_SIG_UNPACK_NEXTID(sig[0]);
            if (pDst != NULL) {
                pDst += sxm_tfile_bsize_static(self->hdr->tb[self->active].bsize);
            }
        }

        /* If all went good just populate that number */
        if (rc != -1) {
            rc = (int)idx;
        }

        /* Check that all good, if not - remove the chain */
        if (rc < 0) {
            sxm_tfile_chain_free(pChain);
        }
        else if (ppChain != NULL) {
            *ppChain = pChain;
        }
    }

    return rc;
}

/***************************************************************************//**
 * Reverts back result of blocks allocation.
 * \warning this function must be used only for reverting back incorrectly
 *          or mistakenly allocated chains.
 *
 * \param[in] self a pointer to a valid transaction file structure
 * \param[in] chain instance of the chain to release
 ********************************************************************************/
static void tf_revert_chain(SXMTFile *self, const SXMTFileChain *chain) {
    uint idx;
    uint * const other_bits = &self->hdr->map[TF_OTHER(self)].bits[0];
    uint * const temp_bits = &self->hdr->temp.bits[0];

    for (idx = 0U; idx < chain->count; ++idx) {
        BITS(other_bits, chain->ids[idx]);
        BITS(temp_bits, chain->ids[idx]);
    }
    
    return;
}

/***************************************************************************//**
 * Writes a block with signature to the file
 * \warning this function shall be used only for writing blocks for t-file
 *          with support of chained blocks. There is no internal check on this,
 *          thus, please use carefully.
 * \param[in] self a pointer to a valid transaction file structure
 * \param[in] pChain the chain caller wishes to write
 * \param[in] id block sequential number
 * \param[in] src a pointer to the data to be written
 * \return the number of block written or \p -1 in case of error.
 ********************************************************************************/
static int tf_write_ex(SXMTFile *self, SXMTFileChain *pChain, ushort id, const void *src) {
    int rc = -1;

    switch (0) { default: {
        const uint fsize = self->hdr->tb[self->active].fsize;
        const uint bsize = self->hdr->tb[self->active].bsize;
        uint block,
             sig[NUM_ALIGNED_BLOCKS(SXM_TFILE_BLOCK_RESERVED, sizeof(uint))] = {0};
        ushort nextId;

        if (id >= pChain->count) {
            TF_LOG(": Error: writing at %d block which is out of chain", (int)id);
            break;
        }
        block = pChain->ids[id];
        if ((block + 1U) > fsize) {
            TF_LOG(": Error: writing at %d block exceeds %d blocks", (int)block, (int)fsize);
            break;
        }

        /* Calculate signature */
        if (id == (pChain->count - 1)) {
            nextId = 0U;
        }
        else {
            nextId = pChain->ids[id + 1];
        }
        
        sig[0] = tf_sig_pack(nextId /* next id */,
                             (id == 0U) /* first */,
                             (id == (pChain->count - 1U)) /* last */);

        /* By some reason regardless the position reported by ftell() and the
         * assumption that the position is correct after the fread() the following
         * fwrite() may not write data in correct position and even may not put any
         * data into the file. Based on this it looks like the call to fseek() is
         * required here.
         */
        rc = fseek(self->dbfile, (long)(bsize * block), SEEK_SET);
        if (rc != 0) {
            TF_LOG(": fseek(%u) failed (errno=%d)", bsize * block, errno);
            break;
        }

        /* Put signature */
        if (fwrite(&sig[0], SXM_TFILE_BLOCK_RESERVED, 1, self->dbfile) != 1) {
            TF_LOG(": fwrite() failed writing signature %08x", sig[0]);
            break;
        }

        /* Since here the file's content has new data, thus, mark it as modified
         * regardless the result of further commands
         */
        self->mask |= SXM_TFILE_MASK_MODIFIED;

        /* Put chunk of the buffer */
        if (fwrite(src, sxm_tfile_bsize_static(bsize), 1, self->dbfile) != 1) {
            TF_LOG(": fwrite(%u) failed writing block data",
                sxm_tfile_bsize_static(bsize));
            break;
        }
        rc = 1; /* We've written one block */
    }};

    return rc;
}

/***************************************************************************//**
 * Reads data from the file into an area into memory supplied by the application
 * it wishes. It does reading the file since the \p block up to the \p count
 * depending on the internal connections between them. Thus, the functions loads
 * blocks until the end block detected or the caller asks less.
 * 
 * In some cases the caller may not need to keep the chain instance, thus, the 
 * caller may omit this parameter passing \p NULL - in this case the function
 * does only data reading.
 * 
 * Another option is then \p dst is passed as \p NULL. In this cases the function
 * constructs the chains, but doesn't read the file content.
 * 
 * Both \p ppChain and \p dst must not be \p NULL. In this situation the function
 * call fails with corresponding error code.
 *
 * \param[in] self a pointer to a valid transaction file structure
 * \param[in] block the starting (user) block number in the file
 * \param[in] count the number of  (user) block to read
 * \param[out] ppChain keeps chain's descriptor in case of success
 * \param[in,out] dst a pointer to the area which will receive the data
 * 
 * \return count of blocks or \p -1 in case of error
 ********************************************************************************/
int sxm_tfile_read(SXMTFile *self, uint block, uint count, SXMTFileChain **ppChain, void *dst) {
    int rc;
    if ((self != NULL) && (count > 0U) && ((ppChain != NULL) || (dst != NULL)) &&
        (block >= self->dataptr) &&
        ((block + count) <= self->hdr->tb[self->active].fsize))
    {
        rc = tf_read_chain(self, block, count, ppChain, dst);
    }
    else {
        rc = -1;
    }

    return rc;
}

/***************************************************************************//**
 * Reads data from the file into an area, which it allocates in the heap.
 * This function does reading the file since the \p block up to the \p count
 * depending on the internal connections between them. Thus, the functions
 * loads blocks until the end block detected or the caller asks less.
 *
 * \param[in] self a pointer to a valid transaction file structure
 * \param[in] block the starting (user) block number in the file
 * \param[in] count the number of  (user) block to read
 * \param[out] ppChain keeps chain's descriptor in case of success
 * \param[out] pRc a pointer to an integer to hold the return code
 *
 * \return a pointer to memory allocated and filled with data, or \p NULL.
 *         The memory shall be released by \c sxe_free() call if not needed.
 ********************************************************************************/
void *sxm_tfile_alloc_and_read(SXMTFile *self, uint block, uint count,
                               SXMTFileChain **ppChain, int *pRc)
{
    SXMResultCode rc;
    void *pDst = NULL;
    if ((self == NULL) || (count == 0U)) {
        rc = SXM_E_FAULT;
    }
    else if ((block < self->dataptr) ||
             ((block + count) > self->hdr->tb[self->active].fsize)) {
        rc = SXM_E_INVAL;
    }
    else {
        pDst = sxe_malloc(count * self->hdr->tb[self->active].bsize);
        if (pDst == NULL) {
            TF_LOG(": Unable to allocate %d bytes",
                count * self->hdr->tb[self->active].bsize);
            rc = SXM_E_NOMEM;
        }
        else if (tf_read_chain(self, block, count, ppChain, pDst) <= 0) {
            TF_LOG(": failed to read %d block(s) since %d", count, block);
            sxe_free(pDst);
            pDst = NULL;
            rc = SXM_E_PIPE;
        }
        else {
            TF_LOG(": read and loaded %d block(s) since %d", count, block);
            rc = SXM_E_OK;
        }
    }

    /* Populate the result */
    if (pRc != NULL) {
        *pRc = rc;
    }

    return pDst;
}

/***************************************************************************//**
 * Writes a number of blocks to the file
 * \param[in] self a pointer to a valid transaction file structure
 * \param[in] pChain the chain of blocks to be written
 * \param[in] count the number of (user) blocks to write,
 *                  use (-1) to write whole chain.
 * \param[in] src a pointer to the data to be written
 * \return -1 or the number of blocks written
 ********************************************************************************/
int sxm_tfile_write(SXMTFile *self, SXMTFileChain *pChain, uint count, const void *src) {
    int rc;
    if ((self != NULL) && (pChain != NULL) && (src != NULL)) {
        /* Write blocks one by one following the chain */
        const uint bsize = self->hdr->tb[self->active].bsize - SXM_TFILE_BLOCK_RESERVED;
        const byte *pSrc = (const byte*)src;
        const ushort blocks = (ushort)MIN((ushort)count, pChain->count);
        ushort idx;
        rc = 0;
        /* Look through all blocks within the chain and write them
         * to the file
         */
        for (idx = 0U; idx < blocks; ++idx) {
            const int intRc = tf_write_ex(self, pChain, idx, pSrc);
            if (intRc <= 0) {
                TF_LOG(": failed writing the block %d with signature (%d)",
                    (int)pChain->ids[idx], intRc);
                rc = intRc;
                break;
            }
            ++rc; /* Keep counting the blocks */
            pSrc += bsize;
        }
    }
    else {
        rc = -1;
    }
    return rc;
}

/***************************************************************************//**
 * Allocates a set of user blocks from the free block pool.
 *
 * Blocks allocation process contains two steps:
 * -# Attempt to allocate contiguous set of blocks as implemented for the
 *    legacy t-file format;
 * -# In case of the regular allocation failed tries to allocate blocks by parts.
 *    Each allocation attempt tries to get as many blocks as remaining to fulfill
 *    the request. In worst case the allocation is done in one-by-one manner.
 *
 * \param[in] self a pointer to a valid transaction file structure
 * \param[in] len the number of (user) block to allocate
 * \param[out] ppChain a pointer the descriptor of the set of allocated blocks
 *
 * \retval SXM_E_OK Success
 * \retval SXM_E_NOMEM Failed to allocate memory or blocks within the t-file
 * \retval SXM_E_FAULT NULL Parameter
 * \retval SXM_E_STATE Wrong state
 *
 ********************************************************************************/
int sxm_tfile_alloc(SXMTFile *self, uint len, SXMTFileChain **ppChain) {
    SXMResultCode rc;
    if ((self == NULL) || (len == 0U) || (ppChain == NULL)) {
        rc = SXM_E_FAULT;
    }
    else  if ((self->mask & SXM_TFILE_MASK_TRANSACTION) != SXM_TFILE_MASK_TRANSACTION) {
        rc = SXM_E_STATE;
    }
    else {
        SXMTFileChain *ret = TFILE_NEW_CHAIN(len);
        if (ret == NULL) {
            rc = SXM_E_NOMEM;
        }
        else {
            uint start, found = 0U;
            /* Regardless of the t-file schema try to behave as
             * legacy t-file does - allocate contiguous set of blocks
             */
            rc = tf_alloc(self, len, &start, &found);
            if (rc == SXM_E_OK) {
                /* Fill up the chain */
                for (ret->count = 0; ret->count < len; ++ret->count ) {
                    ret->ids[ret->count] = (ushort)(start + ret->count);
                }
            }
            /* In case of failure let's try to allocate blocks by parts. */
            else if (found > 0U) {
                uint remainig = len;
                rc = SXM_E_OK;
                ret->count = 0;
                /* Utilize as many blocks as the t-file allows to. */
                do {
                    remainig -= found;
                    /* Fill up all found blocks */
                    for (; found; --found) {
                        ret->ids[ret->count++] = (ushort)start++;
                    }
                    if (remainig > 0U) {
                        rc = tf_alloc(self, remainig, &start, &found);
                        if (rc == SXM_E_OK) {
                            TF_LOG(": remainig %u block(s) found", remainig);
                            remainig = 0U;
                        }
                        else if ((rc != SXM_E_OK) && (found == 0U)) {
                            TF_LOG(": no more unused blocks");
                            break;
                        }
                    }
                } while (remainig > 0U);

                if (rc != SXM_E_OK) {
                    /* Clean up all blocks which can be allocated at this
                     * moment.
                     */
                    tf_revert_chain(self, ret);
                }
            }

            if (rc != SXM_E_OK) {
                sxe_free(ret);
            }
            else {
                *ppChain = ret;
            }
        }
    }
    return rc;
}

/***************************************************************************//**
 * Marks a set of blocks as free.
 * \note In case of success the chain instance becomes invalid.
 *
 * \param[in] self a pointer to a valid transaction file structure
 * \param[in] pChain the chain of blocks to free
 *
 * \retval SXM_E_OK Success
 * \retval SXM_E_FAULT NULL Parameter
 * \retval SXM_E_STATE Wrong state
 *
 ********************************************************************************/
int sxm_tfile_free(SXMTFile *self, SXMTFileChain *pChain) {
    SXMResultCode rc;

    if ((self == NULL) || (pChain == NULL)) {
        rc = SXM_E_FAULT;
    }
    else if ((self->mask & SXM_TFILE_MASK_TRANSACTION) != SXM_TFILE_MASK_TRANSACTION) {
        rc = SXM_E_STATE;
    }
    else {
        uint *bits = &self->hdr->map[TF_OTHER(self)].bits[0];

        /* only set the 'free' bit in new alloc map, not unified */
        while (pChain->count --> 0) {
            BITS(bits, pChain->ids[pChain->count]);
        }
        self->mask |= SXM_TFILE_MASK_MODIFIED;
        rc = SXM_E_OK;
    }
    return rc;
}

/***************************************************************************//**
 * Check the chain to make sure that all blocks referenced by the chain
 * are used inside the t-file
 *
 * \param[in] self a pointer to a valid transaction file structure
 * \param[in] pChain the chain of blocks
 *
 * \retval SXM_E_OK Success
 * \retval SXM_E_FAULT NULL Parameter
 * \retval SXM_E_CORRUPT The chain has incorrect block ids
 *
 ********************************************************************************/
int sxm_tfile_check(const SXMTFile *self, const SXMTFileChain *pChain) {
    int rc;
    if ((self == NULL) || (pChain == NULL)) {
        rc = SXM_E_FAULT;
    }
    else {
        const ushort *pidx = &pChain->ids[0],
                     *pend = &pChain->ids[pChain->count];
        rc = SXM_E_OK;
        for (; pidx < pend; ++pidx) {
            if ((*pidx >= SXM_TFILE_ALLOC_MAP_BITLEN) || !TF_BLOCK_IS_USED(self, *pidx)) {
                TF_LOG(": chain's item #%d is invalid, observed block id %u",
                    (int)(pend - pidx), *pidx);
                rc = SXM_E_CORRUPT;
                break;
            }
        }
    }
    return rc;
}

/***************************************************************************//**
 * Returns the first block's id within the chain
 * \param[in] pChain valid chain
 *******************************************************************************/
ushort sxm_tfile_chain_start(const SXMTFileChain *pChain) {
    ushort res;
    if (pChain != NULL) {
        res = pChain->ids[0];
    }
    else {
        res = (ushort)-1;
    }
    return res;
}

/***************************************************************************//**
 * Returns number of blocks addressed by the chain
 * \param[in] pChain valid chain
 *******************************************************************************/
ushort sxm_tfile_chain_count(const SXMTFileChain *pChain) {
    ushort res = 0U;
    if (pChain != NULL) {
        res = pChain->count;
    }
    else {
        res = 0U;
    }
    return res;
}

/***************************************************************************//**
 * Releases allocated chain w/o any I/O operation over it
 * \param[in] pChain valid chain
 *******************************************************************************/
void sxm_tfile_chain_free(SXMTFileChain *pChain) {
    if (pChain != NULL) {
        sxe_free((void*)pChain);
    }
    return;
}

/***************************************************************************//**
 * Adds new block to the stream depending on caller's wish
 *
 * \param[in] pStream valid stream instance
 * \param[in] bOnTop \p TRUE if the new block should be added prior the first one,
 *                   \p FALSE after the last one
 *
 * \retval SXM_E_OK Success
 * \retval SXM_E_FAULT NULL Parameter
 * \retval SXM_E_NOMEM Memory related failure
 *
 ********************************************************************************/
static SXMResultCode tsream_add_block(SXMTFileStream *pStream, BOOL bOnTop) {
    SXMResultCode rc;
    SXMListEntry *entry;
    SXMTStreamBlock *ret;

    /* Pick up the one from the unused list */
    entry = sxm_list_first(&pStream->unused);
    if (entry != NULL) {
        rc = (SXMResultCode) sxm_list_remove(&pStream->unused, entry);
        ret = (rc == SXM_E_OK) ? (SXMTStreamBlock *)sxm_list_data(entry) : NULL;
    }
    else {
        /* Allocate a new one if no one already allocated */
        ret = (SXMTStreamBlock*)
            sxm_list_allocate(sizeof(SXMTStreamBlock) + sxm_tfile_bsize(pStream->tFile) - sizeof(ret->data));
    }

    if (ret == NULL) {
        rc = SXM_E_NOMEM;
    }
    else {
        ret->size = 0; /* No data inside */
        rc = (SXMResultCode) sxm_list_add(&pStream->blocks, ret, bOnTop, NULL);
        if (rc == SXM_E_OK) {
            if ((bOnTop == FALSE) ||
                (sxm_list_size(&pStream->blocks) == 1 /* the very first block added */))
            {
                pStream->entry = sxm_list_last(&pStream->blocks);
            }
        }
        else {
            /* At this point no-one owns the object, remove it */
            sxm_list_free(ret);
        }
    }
    /* At this point only valid reason for error could be lack of heap
     * in order to make new t-block for the stream
     */
    if (rc != SXM_E_OK) {
        pStream->state = SXM_TFILE_STREAM_NOMEM;
    }
    return rc;
}

/***************************************************************************//**
 * Creates new stream attached to the T-File
 *
 * \param[in] tFile a pointer to a valid transaction file structure
 * \param[out] ppStream valid T-File stream in case of success
 *
 * \retval SXM_E_OK Success
 * \retval SXM_E_FAULT NULL Parameter
 * \retval SXM_E_NOMEM Heap allocation failed
 *
 ********************************************************************************/
int sxm_tstream_create(SXMTFile *tFile, SXMTFileStream **ppStream) {
    SXMResultCode rc;
    SXMTFileStream *ret = NULL;

    switch (0) { default: {
        /* Check input */
        if ((tFile == NULL) || (ppStream == NULL)) {
            rc = SXM_E_FAULT;
            break;
        }

        ret = (SXMTFileStream *)sxe_calloc(1, sizeof(SXMTFileStream));
        if (ret == NULL) {
            rc = SXM_E_NOMEM;
            break;
        }

        /* Init object */
        ret->tFile = tFile;
        ret->entry = NULL;
        ret->size = 0;
        ret->state = SXM_TFILE_STREAM_OK;
        rc = (SXMResultCode)sxm_list_create(&ret->blocks, SXM_LIST_PREALLOCATED);
        if (rc != SXM_E_OK) {
            break;
        }
        rc = (SXMResultCode)sxm_list_create(&ret->unused, SXM_LIST_PREALLOCATED);
        if (rc != SXM_E_OK) {
            break;
        }

        /* Allocate very first entry */
        rc = tsream_add_block(ret, FALSE);
    }}

    if (ret != NULL) {
        if (rc == SXM_E_OK) {
            *ppStream = ret;
        }
        else {
            sxm_tstream_destroy(ret);
        }
    }
    return rc;
}

/***************************************************************************//**
 * Write amount of bytes to the stream
 *
 * \param[in] pStream valid stream instance
 * \param[in] data the data to be written
 * \param[in] size size of the data addressed by \p data
 * \param[in] bOnTop \p TRUE if the new data should be added prior existing one,
 *                   \p FALSE after the last byte of the current one.
 *
 * \retval SXM_E_OK Success
 * \retval SXM_E_STATE Current state doesn't allow write operation
 * \retval SXM_E_FAULT NULL Parameter
 * \retval SXM_E_NOMEM Memory related failure
 *
 ********************************************************************************/
int sxm_tstream_write(SXMTFileStream *pStream, const void *data, size_t size, BOOL bOnTop) {
    SXMResultCode rc;
    if ((pStream == NULL) || (data == NULL)) {
        rc = SXM_E_FAULT;
    }
    else if (pStream->state != SXM_TFILE_STREAM_OK) {
        rc = SXM_E_STATE;
    }
    else if (size == 0) {
        rc = SXM_E_OK;
    }
    else {
        const size_t capacity = sxm_tfile_bsize(pStream->tFile);
        rc = SXM_E_OK;
        if (bOnTop == FALSE) {
            SXMTStreamBlock *curr = (SXMTStreamBlock *)sxm_list_data(pStream->entry);
            size_t to_write = capacity - curr->size;
            if (to_write >= size) {
                /* The block has plenty space to service the request
                 * thus, just write it
                 */
                memcpy(&curr->data[curr->size], data, size);
                pStream->size += size;
                curr->size += (uint)size;
            }
            else {
                /* More complicated case where current block has
                 * less space than required by the caller, so,
                 * following code will mess with blocks allocation
                 * to obtain enough space.
                 */
                const byte *pData = (const byte*)data;
                while (size > 0) {
                    if (to_write == 0) {
                        rc = tsream_add_block(pStream, FALSE);
                        if (rc != SXM_E_OK) {
                            TF_LOG(": failed to add block (%d)", rc);
                            break;
                        }
                        curr = (SXMTStreamBlock *)sxm_list_data(pStream->entry);
                        to_write = capacity - curr->size;
                    }
                    /* We're got space to serve the request, thus, do this */
                    if (to_write >= size) {
                        /* put only remaining part of the data */
                        memcpy(&curr->data[curr->size], pData, size);
                        pStream->size += size;
                        curr->size += size;
                        size = 0;
                    }
                    else {
                        /* put as many data as the block can accommodate and 
                         * let us more to the new iteration
                         */
                        memcpy(&curr->data[curr->size], pData, to_write);
                        pStream->size += to_write;
                        curr->size += to_write;
                        size -= to_write;
                        pData += to_write;
                        to_write = capacity - curr->size;
                    }
                }
            }
        }
        else {
            /* Count number of blocks for the data to be added on top regardless
             * how many spare bytes we've got in other places with one exception
             * for the completely empty stream
             */
            size_t count = NUM_ALIGNED_BLOCKS(size, capacity);
            if (count > 0) {
                /* As soon as by design the stream always has at least one block
                 * and it may happen that this only block is not used, for example,
                 * for just created and just cleaned stream. In this case we're
                 * able to use it to fulfill this request decreasing the number of
                 * newly needed blocks.
                 */
                if (pStream->size == 0) {
                    --count;
                }
                /* Add as many blocks as needed on the top if still needed */
                while (count-- > 0) {
                    rc = tsream_add_block(pStream, TRUE);
                    if (rc != SXM_E_OK) {
                        break;
                    }
                }
                /* if so far so good - write data sequentially from the first block
                 * to the last one. Even in case of empty stream the current block
                 * pointer, kept by the stream structure, remains addressing
                 * the original first block which will be the last used on once
                 * the process is over. Thus, all following regular write operations
                 * will not cause any problems with stream integrity.
                 */
                if (rc == SXM_E_OK) {
                    const byte *pData = (const byte*)data;
                    SXMListEntry *wp = sxm_list_first(&pStream->blocks);
                    while (size > 0) {
                        SXMTStreamBlock *block = (SXMTStreamBlock *)sxm_list_data(wp);
                        size_t to_write = capacity - block->size;
                        if (to_write > size) {
                            to_write = size;
                        }
                        memcpy(&block->data[block->size], pData, to_write);
                        block->size += to_write;
                        size -= to_write;
                        pData += to_write;
                        pStream->size += to_write;
                        wp = sxm_list_next(wp); /* Move to the next */
                    }
                }
            }
        }
    }
    return rc;
}

/***************************************************************************//**
 * Tells current position within the stream
 *
 * \param[in] pStream valid stream instance
 *
 * \return current write point
 *
 ********************************************************************************/
size_t sxm_tstream_tell(SXMTFileStream *pStream) {
    return (pStream != NULL) ? pStream->size : (size_t)0;
}

/***************************************************************************//**
 * The routine aligns the current buffer address forward to an 'N' byte boundary.
 * \note sets the buffer error state if no room
 *
 * \param[in] pStream valid stream instance
 * \param[in] bound bounding value 'N'
 *
 * \retval SXM_E_OK Success
 * \retval SXM_E_FAULT NULL Parameter
 * \retval SXM_E_NOMEM Memory related failure
 *
 ******************************************************************************/
int sxm_tstream_alignup(SXMTFileStream *pStream, size_t bound) {
    SXMResultCode rc;
    /* don't do anything if NULL pointers, zero bound, or buffer state not OK */
    if ((pStream == NULL) || (bound == 0)) {
        rc = SXM_E_FAULT;
    }
    else if ((pStream->size % bound) != 0) {
        /* Alignment needed */
        const size_t pattern = 0U;
        size_t remaining = bound - (pStream->size % bound);
        rc = SXM_E_OK;
        while ((rc == SXM_E_OK) && (remaining > 0)) {
            const size_t to_write = MIN(remaining, sizeof(pattern));
            rc = (SXMResultCode)sxm_tstream_putb(pStream, &pattern, to_write);
            remaining -= to_write;
        }
    }
    else {
        /* Already aligned */
        rc = SXM_E_OK;
    }
    return rc;
}

/***************************************************************************//**
 * Commits data to the t-file.
 *
 * \param[in] pStream valid stream instance
 * \param[out] start stream data very first block id
 * \param[out] count actual number of blocks utilized by the stream
 *
 * \retval SXM_E_OK Success
 * \retval SXM_E_STATE Current stream state doesn't allow to perform commit
 * \retval SXM_E_FAULT NULL Parameters
 * \retval SXM_E_PIPE I/O Failed
 * \retval SXM_E_ERROR Unexpected error(s)
 * \retval SXM_E_NOMEM Memory related error like no heap or no unused t-file blocks
 *                     to fulfill the request.
 *
 ********************************************************************************/
int sxm_tstream_commit(SXMTFileStream *pStream, uint *start, uint *count) {
    SXMResultCode rc;
    SXMTFileChain *pChain = NULL;

    switch (0) { default: {
        uint blockid = 0U;
        size_t tmp_used = 0; /* keeps intermediate storage utilization */
        /* keeps utilization of the block which is being processed */
        size_t block_used = 0;
        size_t capacity = 0; /* block capacity */
        /* keeps list entry associated with the block which is being processed */
        SXMListEntry *entry;
        SXMTStreamBlock *tmp = NULL;

        /* Check input */
        if ((pStream == NULL) || (start == NULL) || (count == NULL)) {
            rc = SXM_E_FAULT;
            break;
        }

        /* Both stream and the t-file shall have proper states to make commit */
        if (((pStream->state != SXM_TFILE_STREAM_OK) && (pStream->state != SXM_TFILE_STREAM_NOBLOCKS)) ||
            (sxm_tfile_transaction_state(pStream->tFile) == FALSE)) {
            rc = SXM_E_STATE;
            break;
        }

        capacity = sxm_tfile_bsize(pStream->tFile);
        *count = (uint)NUM_ALIGNED_BLOCKS(pStream->size, capacity);
        if (pStream->tmp == NULL) {
            pStream->tmp = (SXMTStreamBlock *)
                sxe_calloc(1, sizeof(SXMTStreamBlock) + capacity);
            if (pStream->tmp == NULL) {
                TF_LOG(": failed to allocate intermediate storage");
                pStream->state = SXM_TFILE_STREAM_NOMEM;
                rc = SXM_E_NOMEM;
                break;
            }
            else {
                pStream->tmp->size = capacity;
            }
        }

        /* Allocate blocks */
        rc = sxm_tfile_alloc(pStream->tFile, *count, &pChain);
        if (rc != SXM_E_OK) {
            TF_LOG(": failed to allocate %u blocks in tfile (%d)", *count, rc);
            pStream->state = SXM_TFILE_STREAM_NOBLOCKS;
            break;
        }
        blockid = 0U;
        entry = sxm_list_first(&pStream->blocks);

        /* The following routine writes data from the stream's blocks directly
         * to the t-file while there are no partly used ones in the middle of the
         * chain.
         * Once any such block is found the data will be written via intermediate
         * storage since data shall be aligned by blocks and shall contiguously
         * lay inside the stream's blocks within the file.
         */
        *start = sxm_tfile_chain_start(pChain);
        while (entry != NULL) {
            SXMTStreamBlock *block = (SXMTStreamBlock *)sxm_list_data(entry);
            size_t to_write = block->size - block_used;

            if (to_write == 0) {
                /* Get the next block since there is no
                 * data to be taken from the current one
                 */
                entry = sxm_list_next(entry);
                block = (entry != NULL) ? (SXMTStreamBlock *)sxm_list_data(entry) : NULL;
                block_used = 0;
            }
            else {
                const byte *data_to_write = NULL;

                /* Can we continue using the direct write? */
                if ((tmp == NULL) /* no intermediate block in use */ &&
                    (block->size != capacity) /* Current block is not fully utilized */ &&
                    (sxm_list_next(entry) != NULL) /* Current block is not the last one */) {
                    TF_LOG(": started using intermediate storage for block #%d", (int)blockid);
                    /* At this point the intermediate storage shall be used */
                    tmp = pStream->tmp;
                }

                /* Is the intermediate storage in use? */
                if (tmp != NULL) {
                    /* Use it to put as many data as possible from the current
                     * block
                     */
                    const size_t tmp_remaining = tmp->size - tmp_used;
                    if (tmp_remaining > 0) {
                        /* Write as many data as we can taking into account
                         * tmp buffer size and block's unused size
                         */
                        if (tmp_remaining < to_write) {
                            to_write = tmp_remaining;
                        }
                        memcpy(&tmp->data[tmp_used],
                               &block->data[block_used],
                               to_write);
                        block_used += to_write;
                        tmp_used += to_write;
                    }
                    /* At this point the data from the block or intermediate storage can be
                     * written to the t-file
                     */
                    else {
                        /* There is no magic - just tell code below to write data from the
                         * intermediate block to the files assuming that this storage is
                         * fully utilized
                         */
                        data_to_write = &tmp->data[0];
                    }
                }
                else {
                    data_to_write = &block->data[0];
                    /* It may happen that we're here for the last block inside the chain,
                     * thus, in opposite to the previous case the last block shall be written
                     * from this branch, but in order to be sure the block's leftover has
                     * predictable data we have to fill it up by some pattern, for instance,
                     * all zeros.
                     */
                    if (block->size < capacity) {
                        memset(&block->data[block->size], 0, capacity - block->size);
                    }
                    block_used = block->size; /* indicate that we're kind of used whole block */
                }

                /* Do the t-file write if data block is ready */
                if (data_to_write != NULL) {
                    if (tf_write_ex(pStream->tFile, pChain, (ushort)blockid, data_to_write) != 1) {
                        TF_LOG(": failed to write block #%d out of %d (t-file block #%d)",
                            blockid, (int)pChain->count, (int)pChain->ids[blockid]);
                        pStream->state = SXM_TFILE_STREAM_IO;
                        rc = SXM_E_PIPE;
                        break;
                    }
                    else {
                        /* Prepare for the next round */
                        tmp_used = 0;
                        ++blockid;
                    }
                }
            }
        }

        /* Check if there is a data to commit inside the tmp buffer which has
         * not been committed yet. This code is not working if at this moment
         * the commit has been performed directly.
         */
        if (((tmp != NULL) && (tmp_used > 0)) && (rc == SXM_E_OK)) {
            TF_LOG(": writing remaining data in %d byte(s)", (int)tmp_used);
            if (tmp_used < tmp->size) {
                /* In order to make sure the unused part of the 
                 * last block has predictable data, for instance, all
                 * zeros, it's safer to fill it up before write
                 */
                memset(&tmp->data[tmp_used], 0, tmp->size - tmp_used);
            }
            if (tf_write_ex(pStream->tFile, pChain, (ushort)blockid, tmp->data) != 1) {
                rc = SXM_E_PIPE;
                pStream->state = SXM_TFILE_STREAM_IO;
            }
        }
    }}
    sxm_tfile_chain_free(pChain);
    return rc;
}

/***************************************************************************//**
 * Clean up the stream for further usage.
 *
 * \param[in] pStream valid T-File stream
 *
 * \retval SXM_E_OK Success
 * \retval SXM_E_FAULT NULL Parameter
 *
 ********************************************************************************/
int sxm_tstream_clean(SXMTFileStream *pStream) {
    SXMResultCode rc;
    if (pStream == NULL) {
        rc = SXM_E_FAULT;
    }
    else {
        SXMListEntry *entry;
        rc = SXM_E_OK;
        /* Look through all used blocks and move them into
         * unused blocks list.
         */
        do {
            entry = sxm_list_first(&pStream->blocks);
            if (entry != NULL) {
                SXMTStreamBlock *block;
                rc = sxm_list_remove(&pStream->blocks, entry);
                if (rc != SXM_E_OK) {
                    TF_LOG(": failed to remove block (%d)", rc);
                    break;
                }
                block = (SXMTStreamBlock *)sxm_list_data(entry);
                rc = sxm_list_add(&pStream->unused, block, FALSE, NULL);
                if (rc != SXM_E_OK) {
                    TF_LOG(": failed to move block (%d)", rc);
                    sxm_list_free(block); /* Destroy the block */
                }
            }
        } while (entry != NULL);
        /* In accordance with initial statement the stream
         * should have at least one block
         */
        if (rc == SXM_E_OK) {
            rc = tsream_add_block(pStream, FALSE);
        }
        pStream->state = SXM_TFILE_STREAM_OK;
        pStream->size = 0;
    }
    return rc;
}

/***************************************************************************//**
 * Destroys stream instance including all associated resources.
 * \note This function does nothing to the t-file.
 *
 * \param[in] pStream valid T-File stream
 *
 * \retval SXM_E_OK Success
 * \retval SXM_E_FAULT NULL Parameter
 *
 ********************************************************************************/
int sxm_tstream_destroy(SXMTFileStream *pStream) {
    SXMResultCode rc;
    if (pStream == NULL) {
        rc = SXM_E_FAULT;
    }
    else {
        SXMListEntry *entry;
        /* Remove all blocks which currently used */
        do {
            entry = sxm_list_first(&pStream->blocks);
            if (entry != NULL) {
                sxm_list_remove(&pStream->blocks, entry);
                sxm_list_free(sxm_list_data(entry));
            }
        } while (entry != NULL);
        sxm_list_destroy(&pStream->blocks);

        /* Remove all blocks which currently not used aka reserved */
        do {
            entry = sxm_list_first(&pStream->unused);
            if (entry != NULL) {
                sxm_list_remove(&pStream->unused, entry);
                sxm_list_free(sxm_list_data(entry));
            }
        } while (entry != NULL);
        sxm_list_destroy(&pStream->unused);

        /* Release intermediate storage if any */
        if (pStream->tmp != NULL) {
            sxe_free(pStream->tmp);
        }
        sxe_free(pStream);
        rc = SXM_E_OK;
    }
    return rc;
}

/***************************************************************************//**
 * Provides current stream state.
 *
 * \param[in] pStream valid T-File stream
 *
 * \return Please, refer to SXMTFileStreamState for possible states
 *
 ********************************************************************************/
SXMTFileStreamState sxm_tstream_state(const SXMTFileStream *pStream) {
    return (pStream != NULL) ? pStream->state : SXM_TFILE_STREAM_UNKNOWN;
}

/** @} */
