/******************************************************************************/
/*                Copyright (c) Sirius XM Satellite Radio, Inc.               */
/*                            All Rights Reserved                             */
/*      Licensed Materials - Property of Sirius XM Satellite Radio, Inc.      */
/******************************************************************************/
/*******************************************************************************
 * \file sxm_tds_internal.h 
 * \author David Franklin
 * \date 10/30/2016
 *
 * Header file for TDS Data Protocol functions.
 ********************************************************************************/
#ifndef SXM_TDS_INTERNAL_H
#define SXM_TDS_INTERNAL_H

#include <sxm_build.h>
#include <util/sxm_bitbuff.h>

/* TDS Label types */
typedef enum {
    TDS_LABEL_CONSTANT,
    TDS_LABEL_VARIABLE,
} TDS_LABEL_TYPE;

/* TDS Data types */
typedef enum {
    /** Integer  */
    TDS_DATA_INT     = 0,
    /** ISO 8859-1 Coded, variable length
     * string terminated with a NULL
     * character 
     */
    TDS_DATA_STRING  = 1,
    /** Baudot Coded as per Section 7. The
     * field starts in LTR mode and is
     * terminated by an END symbol 
     */
    TDS_DATA_BAUDOT  = 2,
    /** Array of Integers */
    TDS_DATA_ARRAY   = 3,
    /** SCoded. The field
     * always ends with a NUL symbol
     * (0x00) 
     */
    TDS_DATA_SAMPA   = 4
} TDS_DATA_TYPE;

#define SXM_TDS_MAX_EXT_DATA_SIZE          (256)
#define TDS_DVAL_ARRAY_MAX_NUM             1020
#define TDS_MAX_CACHE_ROWS                 1024

typedef union{
    ushort num[TDS_DVAL_ARRAY_MAX_NUM]; 
    char text[TDS_MAX_CACHE_ROWS];
} TdsDval;

/* Table Definition Label defining fields */
typedef struct {
    byte valid;
    UINT8 lbid;
    UINT8 lbtype;
    UINT8 dtype;
    UINT16 lbsize;
    UINT8 lensize;
    UINT16 arraymax;
    TdsDval dval;
    UINT16 extcnt;
    char extdata[SXM_TDS_MAX_EXT_DATA_SIZE];
} TdsLabel;

extern int sxm_tds_read_label(SXMBitBuff *pkt, void *dstBuf, UINT32 dstItemSize, UINT32 dstBufSize, TdsLabel *table);
extern int sxm_tds_read_array(SXMBitBuff *pkt, void *dstBuf, UINT32 dstItemSize, UINT32 dstBufSize, TdsLabel *label, UINT32 *pArrayReadLen);
extern int sxm_tds_skip_label(SXMBitBuff *pkt, TdsLabel *label);

extern SXESDK_INTERNAL_API void sxm_tds_set_debug(uint dLevel);
extern SXESDK_INTERNAL_API uint sxm_tds_get_debug(void);

#endif
