/******************************************************************************/
/*                Copyright (c) Sirius XM Satellite Radio, Inc.               */
/*                            All Rights Reserved                             */
/*      Licensed Materials - Property of Sirius XM Satellite Radio, Inc.      */
/******************************************************************************/
/***************************************************************************//**
 *
 * \file sxm_sdkfiles.c
 * \author Eugene Mlodik, Sergey Kotov
 * \date 18/2/2016
 *
 * Implementation of the SDK file tool
 *
 ******************************************************************************/

#include <sxm_build.h>

#if (SXM_USE_SDKFILES)
#include <ctype.h>
#include <sys/stat.h>
#include <stdarg.h>
#include <util/sxm_common_internal.h>
#include <util/sxm_noname_internal.h>
#include "sxm_sdkfiles.h"
#include "sdkfiles.h"

/** Input files delimiter */
#define SXM_SDKFILES_INPUTS_DELIMITER   ';'

/** \name Compiler vs Code compatibility on data types level
 * @{
 */

TYPE_SIZE_STATIC_ASSERT(char, 1);
TYPE_SIZE_STATIC_ASSERT(byte, 1);
TYPE_SIZE_STATIC_ASSERT(ushort, 2);
TYPE_SIZE_STATIC_ASSERT(short, 2);
TYPE_SIZE_STATIC_ASSERT(fix, 4);
TYPE_SIZE_STATIC_ASSERT(int, 4);
TYPE_SIZE_STATIC_ASSERT(uint, 4);
TYPE_SIZE_STATIC_ASSERT(CUINT8, 1);
TYPE_SIZE_STATIC_ASSERT(UINT8, 1);
TYPE_SIZE_STATIC_ASSERT(INT8, 1);
TYPE_SIZE_STATIC_ASSERT(UINT16, 2);
TYPE_SIZE_STATIC_ASSERT(INT16, 2);
TYPE_SIZE_STATIC_ASSERT(CUINT16, 2);
TYPE_SIZE_STATIC_ASSERT(UINT32, 4);
TYPE_SIZE_STATIC_ASSERT(INT32, 4);

static SXMMakePathCallback_t gMakePathToRestore = NULL;
static char gMakePathBuffer[SDK_FILE_PATH_LEN_MAX + 1];
static SXMLogHandler_t sxm_sdkfiles_logit = NULL;
static BOOL crcInitialized = FALSE;

/** For 32-bit int value the structure has to have 8 bytes size
 * in order to fulfill SXe needs
 * \note There are some systems which has int in 64-bit in 64-bit build
 *       configuration. In this case the check shall fail. Please, contact
 *       to SXe development team if needed.
 */
typedef struct sdkfiles_alignment_check_struct
{
    char one_byte;
    int four_bytes;
} SDKFILES_ALIGNMENT_CHECK_STRUCT;

TYPE_SIZE_STATIC_ASSERT(SDKFILES_ALIGNMENT_CHECK_STRUCT, 8);

/** @} */

typedef struct
{
    const char *moduleName;
    /* This function parses file name to get version */
    SXM_FSINPUT_PATH_CALLBACK tfileVersion;
    /* This is build function to be used in all builders */
    int(*tfileBuildFunc)(const SXMFilesInput *pFilesInput, int version);
    SXMSdkfilesServiceType serviceType;
    const char *dbFileName;
    char dbFileType;

} SdkfilesServiceParams;

static const SdkfilesServiceParams lookupTable[] =
{
#if (SXM_USE_APOGEE_FULL == 1)
    {"apogee_ptrn", apogee_patterns_ver,    apogee_patterns_build,  SXM_TRAFFIC_PATTERN,    "patternsdb",   'W'},//TODO: Use internal definitions
#endif
    {"apogee_sramp",apogee_sxmramp_ver,     apogee_sxmramp_build,   SXM_TRAFFIC_SXMRAMP,    "rampsdb",      'W'},
    {"apogee",      locations_ver,          locations_build,        SXM_TRAFFIC,            "locations",    'R'},
    {"bcatimage",   bcatimage_ver,          bcatimage_build,        SXM_BCATIMAGE,          "bcatimagedb",  'W'},
    {"canfuel",     fuel_ver,               canfuel_build,          SXM_CANFUEL,            "canfueldb",    'W'},
    {"cclogo",      NULL,                   cclogo_build,           SXM_CCLOGO,             "cclogo",       ' '},
    {"ev",          ev_ver,                 ev_build,               SXM_EV,                 "evdb",         'W'},
    {"fuel",        fuel_ver,               fuel_build,             SXM_FUEL,               "fueldb",       'W'},
#if (SXM_USE_GMD)
    {"gmd",         NULL,                   gmd_build_tfile,        SXM_GMD,                "metadata",     'R'},
#endif
#if (SXM_USE_IGS)
    {"igs",         NULL,                   igs_build,              SXM_IGS,                "igs",          'R'},
#endif
    {"movies",      movies_ver,             movies_build,           SXM_MOVIES,             "moviesdb",     'W'},
    {"parking",     parking_ver,            parking_build,          SXM_PARKING,            "parkingdb",    'W'},
    {"phn",         NULL,                   phn_build,              SXM_PHONETICS,          "phn",          'W'},
    {"pix",         sxmpix_ver,             sxmpix_build,           SXM_PIX,                "pix",          'W'},
    {"safety",      NULL,                   safety_build,           SXM_SAFETY,             "safetydb",     'W'},
    {"sportslogo",  sportslogo_ver,         sportslogo_build,       SXM_SPORTSLOGO,         "sportslogodb", 'W'},
    {"states",      NULL,                   states_build,           SXM_STATES,             "states",       'R'},
    {"stocks",      NULL,                   stocks_build,           SXM_STOCKS,             "stocks",       'R'},
#if (!SXM_USE_GEN8)
    {"sxi",         NULL,                   sxi_build_tfile,        SXM_SXI,                "metadata",     'R'},
#endif
    {"wxalerts",    NULL,                   wxalerts_build,         SXM_WXALERTS,           "wxalertsdb",   'W'},
    {"wxtab",       wxtab_ver,              wxtab_build,            SXM_WXTAB,              "wxtabdb",      'W'}
};

static SXMResultCode error_logged;

/******************************************************************************
 *                                                                            *
 *            Internal Functions                                              *
 *            ==================                                              *
 *                                                                            *
 ******************************************************************************/

/***************************************************************************//**
 * The log routine formats a message, and presents it for printing through
 * the SXe SDK output routine (\ref sxm_logit)
 *
 * \param[in] fmt the format control (sprintf controls)
 * \param[in] ... any remaining arguments
 *
 ******************************************************************************/
static void sxm_sdkfile_log(char *fmt, ...)
{
    char buffer[SXM_PBUF_SIZE];
    va_list ap;

    if (sxm_sdkfiles_logit)
    {
        va_start(ap, fmt);
        vsnprintf(buffer, sizeof(buffer), fmt, ap);
        buffer[SXM_PBUF_SIZE-1] = '\0';
        sxm_sdkfiles_logit(buffer);
        va_end(ap);
    }
}

void fatal(const char *err, ...) {
    va_list ap;
    char errs[512];
    int res;

    va_start(ap, err);
    res = vsnprintf(errs, sizeof(errs), err, ap);
    va_end(ap);
    if (res < 0)
        strcpy(errs, "log creation failed");

    sxm_sdkfile_log("ERROR|%s\n\n", errs);
    error_logged = SXM_E_ERROR;

    return;
}

void non_fatal(const char *err, ...) {
    va_list ap;
    char errs[512];
    int res;

    va_start(ap, err);
    res = vsnprintf(errs, sizeof(errs), err, ap);
    va_end(ap);
    if (res < 0)
        strcpy(errs, "log creation failed");

    sxm_sdkfile_log("WARNING|%s\n\n", errs);

    return;
}

const char *get_file_name_pointer(const char *pFilePath) {

    const char *pFileName = strrchr(pFilePath, '/');

    if (NULL == pFileName) {
        pFileName = strrchr(pFilePath, '\\');
    }

    if (NULL != pFileName) {
        ++pFileName;
    }
    else {
        pFileName = pFilePath;
    }

    return pFileName;
}

int strncmp_lowercase(const char *pStr1, const char *pStr2, size_t len) {

    /* Case-independent comparison */
    while (tolower(*pStr1) == tolower(*pStr2)) {

        if (0 == len) {
            break;
        }

        if ('\0' == *pStr1) {
            len = 0;
            break;
        }

        len--;
        pStr1++;
        pStr2++;
    }

    return (int)len;
}

const char * strstr_lowercase(const char *pStr1, const char *pStr2) {

    size_t str1Len = strlen(pStr1);
    size_t str2Len = strlen(pStr2);

    while (str1Len >= str2Len) {
        if (0 == strncmp_lowercase(pStr1, pStr2, str2Len)) {
            return pStr1;
        }
        pStr1++;
        str1Len--;
    }

    return NULL;
}

static const char *get_next_input(const char *pInputPath,
                                  char *pOutBuf, size_t outBufSize)
{
    uint idx = 0;
    size_t filePathLen = 0;
    const char *pFilePathPtr = pInputPath;

    for (idx = 0;; idx++)
    {
        if (('\0' != pInputPath[idx]) &&
            (SXM_SDKFILES_INPUTS_DELIMITER != pInputPath[idx]))
        {
            filePathLen++;
        }
        else
        {
            if (filePathLen > 0)
            {
                if (filePathLen >= outBufSize)
                {
                    non_fatal("Insufficient  buffer size");
                    pFilePathPtr = NULL;
                    break;
                }

                /* Copy input path */
                strncpy(pOutBuf, pFilePathPtr, filePathLen);
                pOutBuf[filePathLen] = '\0';
            }

            if ('\0' == pInputPath[idx])
            {
                if (filePathLen > 0)
                {
                    /* Set pointer to the end */
                    pFilePathPtr = &pInputPath[idx];
                }
                else
                {
                    /* Nothing to return */
                    pFilePathPtr = NULL;
                }
                break;
            }
            else
            {
                /* Move pointer to the next input path */
                pFilePathPtr = &pInputPath[idx + 1];
                if (filePathLen > 0)
                {
                    break;
                }
            }
        }
    }

    return pFilePathPtr;
}

static int extract_files(const char *pBaselinePath,
                         const char *pTempPath,
                         const char *pFilteredFileExtension,
                         SXM_PKZIP_CALLBACK filteredFileCallback,
                         void *pCallbackData)
{
    int rc = SXM_E_OK;
    const char *pInputPtr = pBaselinePath;
    char input[SDK_FILE_PATH_LEN_MAX + 1] = "";
    FILE *pZipFile = NULL;

    while (SXM_E_OK == rc)
    {
        pInputPtr = get_next_input(pInputPtr, input,
                                   SDK_FILE_PATH_LEN_MAX + 1);
        if (NULL == pInputPtr)
        {
            break;
        }

        pZipFile = fopen(input, "rb");
        if (NULL == pZipFile)
        {
            rc = SXM_E_PIPE;
            break;
        }

        rc = sxm_deflate_pkzip_file(pZipFile, pTempPath,
                                    pFilteredFileExtension,
                                    filteredFileCallback,
                                    pCallbackData);
        fclose(pZipFile);
    }

    return rc;
}

typedef struct
{
    const char *pWorkingDir;
    char *pInputFile;
    size_t inputFileSize;

} GenericCallbackData;

static int generic_file_callback(const SXMPKZipFileDesc *pDesc, void *pUserData)
{
    const char *pFileName = get_file_name_pointer(pDesc->pFileName);
    GenericCallbackData *pCallbackData = (GenericCallbackData*)pUserData;

    /* For other services only one input file is provided.
       Use first list entry as input. */
    snprintf(pCallbackData->pInputFile, pCallbackData->inputFileSize,
             "%s/%s", pCallbackData->pWorkingDir, pFileName);
    pCallbackData->pInputFile[pCallbackData->inputFileSize - 1] = '\0';

    return SXM_E_OK;
}

static int get_input_path(const char *pBaselinePath,
                          const char *pWorkingDir,
                          SXMSdkfilesServiceType serviceType,
                          char *pInputFile,
                          size_t inputFileSize)
{
    int rc = SXM_E_OK;

    switch (serviceType)
    {
        case SXM_BCATIMAGE:
        case SXM_CANFUEL:
        case SXM_CCLOGO:
        case SXM_EV:
        case SXM_FUEL:
        case SXM_MOVIES:
        case SXM_PARKING:
        case SXM_PHONETICS:
        case SXM_PIX:
        case SXM_SAFETY:
        case SXM_SPORTSLOGO:
        case SXM_STATES:
        case SXM_STOCKS:
        case SXM_SXI:
        case SXM_WXALERTS:
        case SXM_WXTAB:
        {
            /* Input file is not required */
            pInputFile[0] = '\0';
        }
        break;

        case SXM_TRAFFIC:
        case SXM_TRAFFIC_PATTERN:
        case SXM_TRAFFIC_SXMRAMP:
        {
            /* Archive is extracted to working directory
               Use <working dir>\SXMData as input */
            snprintf(pInputFile, inputFileSize, "%s/SXMData", pWorkingDir);
            pInputFile[inputFileSize - 1] = '\0';
        }
        break;

        default:
        {
            /* For other services only one input file is provided */
            GenericCallbackData callbackData;

            callbackData.pWorkingDir = pWorkingDir;
            callbackData.pInputFile = pInputFile;
            callbackData.inputFileSize = inputFileSize;

            rc = extract_files(pBaselinePath, pWorkingDir,
                               NULL, generic_file_callback,
                               &callbackData);
        }
        break;
    }

    return rc;
}

fix fixread(char *inp) {
    uint i = 0, f = 0;
    uint dp = 1;
    int sign = 1;
    char *c = inp;

    /* sign */
    if  (*c == '-') {
    c++;
    sign = -1;
    }

    /* integer part */
    while (*c >= '0' && *c <= '9')
    i = i * 10 + (uint)(*c++ - '0');

    /* fraction */
    if (*c == '.') {
    c++;

    while (*c >= '0' && *c <= '9') {
        f = f * 10 + (uint)(*c++ - '0');
        dp *= 10;
        if (dp == 100000)
        break;
    }
    }

    return sign * (int)((i << 15) + ((f << 15) / dp));
}

byte *bput4(byte *dp, int i)
{
    *(int *)(void*)dp = i;
    return dp+4;
}

byte *bput2(byte *dp, short i)
{
    *(short *)(void*)dp = i;
    return dp+2;
}

byte *bput1(byte *dp, unsigned char i)
{
    *(unsigned char *)dp = i;
    return dp+1;
}

byte *bputs(byte *dp, String s)
{
    strcpy((char*)dp, s);
    return dp + strlen(s) + 1;
}

void salign(SXMTFileStream *stream, size_t size)
{
    sxm_tstream_alignup(stream, size);
}

void sputb(SXMTFileStream *stream, const void *b, size_t s)
{
    sxm_tstream_putb(stream, b, s);
}

void sputs(SXMTFileStream *stream, const char *s)
{
    sxm_tstream_puts(stream, s);
}

void sput4(SXMTFileStream *stream, int i)
{
    sxm_tstream_put(stream, i);
}

void sput2(SXMTFileStream *stream, short i)
{
    sxm_tstream_put(stream, i);
}

void sput1(SXMTFileStream *stream, unsigned char i)
{
    sxm_tstream_put(stream, i);
}

byte *bget1(byte *dp, unsigned char* pi)
{
    *pi = *(unsigned char*)dp;
    return dp + sizeof(unsigned char);
}

byte *bget2(byte *dp, ushort *pi)
{
    *pi = *(ushort*)(void*)dp;
    return dp + sizeof(ushort);
}

byte *bget4(byte *dp, int *pi)
{
    *pi = *(int*)(void*)dp;
    return dp + sizeof(int);
}

byte *bgets(byte *dp, String *ps)
{
    size_t len = strlen((const char*)dp);
    *ps = (String)dp;
    return dp + len + 1;
}

void dump(const byte *b, uint len)
{
    char text[40];
    uint i, j;

    text[32] = '\0';
    for (i = 0, j = 0; i < len; i++) {
    if  (j == 0)
        printf("%6u: ", i);
    if  (*b >= ' ' && *b < 0x7f)
        text[j] = (char)*b;
    else
        text[j] = '.';

    printf("%02x", *b++);
    j++;
    if  (j > 0 && (j % 4) == 0)
        printf(" ");
    if (j == 32) {
        printf("  '%s'\n", text);
        fflush(stdout);
        j = 0;
    }
    }

    if  (j > 0)
    while (j < 32) {
        printf("  ");
        if  ((j % 4) == 0)
        printf(" ");
        text[j] = '.';
        if (++j == 32) {
        printf("  '%s'\n", text);
        }
    }
}

static int fileSize(String path)
{
    struct stat ss;

    if (stat(path, &ss) == -1)
    {
        return -1;
    }

    return (int)ss.st_size;
}

byte *fileLoad(const char *path, uint *len)
{
    FILE *ft;
    int size;
    byte *ret;

    size = fileSize((String)path);
    if (size <= 0) {
        return NULL;
    }

    *len = (uint)size;

    ft = fopen(path, "rb");
    if (ft == NULL) {
        return NULL;
    }

    ret = (byte *)sxe_malloc(*len + 16);
    if (ret != NULL) {
        if (fread(ret, 1, *len, ft) != *len) {
            sxe_free(ret);
            ret = NULL;
        }
        else {
            ret[*len] = '\0';
        }
    }

    fclose(ft);
    return ret;
}

/************************************************************************
 *                                                                      *
 *            CRC-32 generator                                          *
 *            ================                                          *
 *                                                                      *
 *    IEEE 802.3 CRC as used in PNG                                     *
 *                                                                      *
 ************************************************************************/

static int initcrc = 0;
static uint crc[256];

static void buildCrC(void)
{
    uint c;
    uint n;
    uint k;

    for (n = 0; n < 256; n++) {
        c = n;
        for (k = 0; k < 8; k++) {
            if (c & 1) {
                c = 0xedb88320 ^ (c >> 1);
            }
            else {
                c =c >> 1;
            }
        }
        crc[n] = c;
    }

    initcrc = 1;
}

uint doCRC_part(uint c, const void *buf, uint len)
{
    uint i;
    const byte *pBuf = (const byte*)buf;

    if  (initcrc == 0)
        buildCrC();

    for (i = 0; i < len; i++)
        c = crc[(c ^ pBuf[i]) & 0xff] ^ (c >> 8);
    return c;
}

uint doCRC(const void *buf, uint len)
{
    return doCRC_part(0xffffffff, buf, len) ^ 0xffffffff;
}

void printColumn(uint colWidth,
                 const char *format,
                 ...)
{
    int rc;
    uint numPrinted = 0;
    uint numPaddingChars = 0;

    va_list args;
    va_start(args, format);

    /* Print arguments */
    numPrinted += (uint)printf(" ");
    rc = vprintf(format, args);
    if (rc > 0)
    {
        numPrinted += (uint)rc;
    }

    /* Calculate number of padding characters */
    if (colWidth > numPrinted)
    {
        numPaddingChars = colWidth - numPrinted;
    }

    /* Print padding characters and separator */
    printf("%*s", numPaddingChars, "|");

    va_end(args);

    return;
}

void printFooter(uint length)
{
    while (0 != length--)
    {
        putchar('-');
    }

    putchar('\n');
}

/** Initialized file generators
 * \param[in] makePath callback for real file path generations
 * \param[in] ctx the callback argument which will be passed to the \p makePath
 */
void sdkfiles_start(SXMMakePathCallback_t makePath)
{
    gMakePathToRestore = sxm_sdk_get_makepath();
    sxm_sdk_set_makepath(makePath);
}

/** De-initialized file generators */
void sdkfiles_stop(void)
{
    sxm_sdk_set_makepath(gMakePathToRestore);
}

const char *sdkfiles_get_path(char type, const char *module, const char *file) {

    return sxm_sdk_get_makepath()(gMakePathBuffer,
                                  SDK_FILE_PATH_LEN_MAX + 1,
                                  type, module, file);
}

int create_buffer_psv_parser(const SXMFilesInputRecord *pFileRecord,
                             char separator, uint lineSizeMax,
                             SxmCsvParser **ppRet)
{
    int rc;
    void *pData;
    size_t dataSize;

    rc = sxm_files_input_alloc_and_read(pFileRecord, &pData, &dataSize);
    if (SXM_E_OK != rc) {
        non_fatal("Input file read failed");
    }
    else {
        rc = sxm_csv_buffer_new(pData, dataSize,
                                lineSizeMax,
                                SXM_CSV_OPTION_DATA_OWNER,
                                separator, ppRet);
        if (SXM_E_OK != rc) {
            non_fatal("CSV parser creation failed");
            sxe_free(pData);
        }
    }

    return rc;
}

/******************************************************************************
 *                                                                            *
 *            Application Interface                                           *
 *            =====================                                           *
 *                                                                            *
 ******************************************************************************/

/***************************************************************************//**
 *
 * The routine creates a proprietary DB file format from the source baseline file.
 *
 * \param[in] version The version of the baseline, or (-1) to allow the sdkfiles
 *                    to derive the version number using different approaches
 *                    depending on the service to arrive as the baseline
 *                    version number
 * \param[in] serviceType A type of baseline to convert (SXM service)
 * \param[in] baslineType A type of file containing the baseline source
 *                        (compressed/uncompressed).
 * \param[in] pBaselinePath The path to the baseline.
 *                          Depending on serviceType and baselineType parameters
 *                          it could be either:
 *                          - directory path
 *                          - single file path
  *                         - list of files paths delimeted by ';' character
 * \param[in] pTempPath The path to the temporary folder with write permissions,
 *                      which sdkfiles is allowed to use while decompressing
 *                      baseline source in case original file is zipped.
 * \param[in] make_path An application callback function that is called when
 *                      application needs to compose DB output file name for
 *                      given file type, module and file.
 *
 * \return SXe error code
 * \retval SXM_E_OK Success
 * \retval SXM_E_INVAL Invalid parameter
 * \retval SXM_E_FAULT NULL parameter
 * \retval SXM_E_NOMEM Memory allocation failed
 * \retval SXM_E_PIPE File operation failed
 * \retval SXM_E_ERROR General error
 * \retval SXM_E_UNSUPPORTED Unsupported input file
 *
 ******************************************************************************/
int sxm_sdkfiles_create(int version,
                        SXMSdkfilesServiceType serviceType,
                        SXMSdkfilesBaselineType baselineType,
                        const char *pBaselinePath,
                        const char *pTempPath,
                        SXMMakePathCallback_t make_path)
{
    uint idx;
    char generatedInputFile[SDK_FILE_PATH_LEN_MAX + 1] = "";
    int rc = SXM_E_INVAL; /* Default return if no service match found */

    if ((SXM_FILE_PLAIN != baselineType) && (SXM_FILE_ZIP != baselineType))
    {
        return SXM_E_INVAL;
    }

    if ((NULL == pBaselinePath) || (NULL == pTempPath) || (NULL == make_path))
    {
        return SXM_E_FAULT;
    }

    /* Initialize CRC tables if not yet initialized */
    if (FALSE == crcInitialized)
    {
        sxm_crc16_initialize();
        sxm_crc32_initialize();
        crcInitialized = TRUE;
    }

    for (idx = 0; idx < ARRAY_SIZE(lookupTable); idx++)
    {
        if (serviceType == lookupTable[idx].serviceType)
        {
            BOOL zipInputFound = FALSE;
            /* Allocate and init files input object */
            SXMFilesInput *pFilesInput = NULL;
            int *pVersion = NULL;
            SXM_FSINPUT_PATH_CALLBACK pCallback = NULL;

            if (NULL == lookupTable[idx].tfileBuildFunc)
            {
                rc = SXM_E_INVAL;
                break;
            }

            if (NULL != strstr_lowercase(pBaselinePath, ".zip"))
            {
                zipInputFound = TRUE;
            }

            generatedInputFile[0] = '\0';

            if (SXM_FILE_ZIP == baselineType)
            {
                if (FALSE == zipInputFound)
                {
                    rc = SXM_E_INVAL;
                    break;
                }

                rc = get_input_path(pBaselinePath, pTempPath,
                                    lookupTable[idx].serviceType,
                                    &generatedInputFile[0],
                                    SDK_FILE_PATH_LEN_MAX + 1);
                if (SXM_E_OK != rc)
                {
                    break;
                }
            }
            else if (TRUE == zipInputFound)
            {
                rc = SXM_E_INVAL;
                break;
            }

            /* Set specified makepath function */
            sdkfiles_start(make_path);

            if (-1 == version) {
                pCallback = lookupTable[idx].tfileVersion;
                pVersion = &version;
            }

            rc = sxm_files_input_create(';', NULL, pBaselinePath, pCallback,
                pVersion, &pFilesInput);
            if (SXM_E_OK != rc) {
                non_fatal("Failed to process files input (rc=%d)", rc);
                break;
            }

            /* Call appropriate builder */
            rc = lookupTable[idx].tfileBuildFunc(pFilesInput, version);
            if (SXM_E_OK != rc) {
                rc = SXM_E_ERROR;
            }

            /* Cleanup */
            sxm_files_input_destroy(pFilesInput);
            sdkfiles_stop();
            break;
        }
    }

    return rc;
}

/***************************************************************************//**
 *
 * The routine returns a proprietary DB file version.
 *
 * \param[in] serviceType SXM service type
 * \param[in] make_path An application callback function that is called
 *                      when application needs to compose DB output file name
 *                      for given file type, module and file.
 *
 * \return DB version on success or -1 otherwise
 *
 ******************************************************************************/
int sxm_sdkfiles_getversion(SXMSdkfilesServiceType serviceType,
                            SXMMakePathCallback_t make_path)
{
    int rc;
    uint i;

    rc = sxm_sdk_set_makepath(make_path);
    if (SXM_E_OK != rc)
    {
        return -1;
    }

    for (i = 0; i < ARRAY_SIZE(lookupTable); i++)
    {
        SXMTFile *dbfile;
        uint schema;
        if (serviceType != lookupTable[i].serviceType)
        {
            continue;
        }

        dbfile = sxm_tfile_open(lookupTable[i].dbFileType,
                                lookupTable[i].moduleName,
                                lookupTable[i].dbFileName,
                                "rb",
                                lookupTable[i].moduleName,
                                &schema, &rc);
        if (dbfile)
        {
            SXMTFileStat st;
            rc = (sxm_tfile_stat(dbfile, &st) == SXM_E_OK) ? (int)st.dversion : -1;
            sxm_tfile_close(dbfile);
            return rc;
        }

        return rc;
    }

    return -1;
}

/***************************************************************************//**
 *
 * The routine allows the application to enable or disable debug printouts
 * of Sdkfiles utility functions.
 *
 * \param[in] printFunc Pointer to debug printout function.
 *                      All debug output is disabled if NULL is passed in.
 *
 ******************************************************************************/
void sxm_sdkfiles_set_debug(SXMLogHandler_t printFunc)
{
    sxm_sdkfiles_logit = printFunc;
}

#endif /* #if !defined(SXM_USE_SDKFILES) || (defined(SXM_USE_SDKFILES) && (SXM_USE_SDKFILES != 0)) */
