/******************************************************************************/
/*                Copyright (c) Sirius XM Satellite Radio, Inc.               */
/*                            All Rights Reserved                             */
/*      Licensed Materials - Property of Sirius XM Satellite Radio, Inc.      */
/******************************************************************************/
/***************************************************************************//**
 *
 * \file sxm_rfd.c
 * \date 8/20/2015
 * \author Leslie French
 *
 * \brief RFD decoder
 * \details The block size is transmitted in 14 bits, so the maximum
 * block size is 16k bytes. The maximum number of equations is 1024. At
 * 2bits/equation, the maximum size of the equation store is 64 units. Refer to
 * \ref rfd_page for more implementation details.
 *
 * \page rfd_page RFD
 *
 * The SXe SDK provides support for RFD decoding. The RFD decoder operates a
 * single service across all modules, and is configured to use a number of
 * fixed-size reconstruction 'slots', which can be shared across all the modules
 * requiring RFD support.  It is not necessary for each module to be aware of
 * the number of RFD files it may require, or hold file space for files not
 * currently being received.
 *
 * RFD collection runs in a single thread. Data blocks are passed from the main thread
 * into the RFD collection thread using a single packet buffer and a semaphore
 * to control access to the buffer.
 * RFD file assembling and processing happens in another single thread. 
 *
 * The Transaction File (tfile) described in section C.1 is updated through the
 * corresponding service RFD thread as long as the service (e.g. fuel) is active.
 * It is recommended to start all SXe services that are released into production so
 * the corresponding T-Files are kept current with the latest broadcast updates.
 * In this manner the tfile is current and operational when the customer decides to
 * subscribe to the particular service.
 *
 * The module is responsible for deciding if a particular RFD file needs to be collected.
 * To make that decision, it needs to examine the RFD metadata. A common metadata
 * format is used for all services that employ RFD for file transmission, and this
 * parsed into a common structure that can be used by all modules - \ref SXMRfdMetadata
 *
 ******************************************************************************/

#define DEBUG_TAG "rfd"

#include <util/sxm_common_internal.h>
#include <util/sxm_noname_internal.h>
#include <util/sxm_msg_queue_incl.h>
#include <util/sxm_list.h>

#ifdef _WIN32
    #define snprintf _snprintf
#endif

/** Internal logging function
 * This should be defined as an empty macro when not debugging RFD
 *
 * \param[in] _f log format string
 * \param[in] ... format arguments
 */
#define RFD_LOG(_f, ...) // sxm_file_log("%s" _f, __FUNCTION__, ##__VA_ARGS__)

#define RFD_LOG_CRITICAL PLOG_UPD

/** Number of slots in the slot store */
#define RFD_SLOTS        24

/** Fixed -- maximum block size */
#define RFD_MAX_BLOCK        (1 << 14)

/** Maximum size of filename in RFD */
#define RFD_FILE_NAME_SIZE_MAX (26U)

/** Size of temporary buffer for file reading - 4 KiB*/
#define RFD_FILE_BUFFER_SIZE (4096U)

/** Defines possible RFD Slot states */
typedef enum {
    RFD_SLOT_INITIAL = 0,
    RFD_SLOT_COLLECTED,
    RFD_SLOT_ASSEMBLED
} RFD_SLOT_STATE;

/** A 'slot' is stored on disk in the slot file */
typedef struct
{
    uint crc;               //!< CRC of the slot
    char module[16];        //!< module that owns the slot
    SXMRfdMetadata meta;    //!< metadata for that file or fragment
    RFD_SLOT_STATE state;   //!< state of the reassembly
    int frag;               //!< index into the SXMRfdMetadata fragment array
    int nblks;              //!< number of blocks in file
} RfdSlot;

/** EQstore is a sector on disk, one for each scrambled file block */
typedef struct
{
    uint eqcrc;             //!< CRC of this equation block
    uint bcrc;              //!< CRC of scrambled block
    uint bcrcs;             //!< CRC of solved block
    int solved;             //!< solved state
    uint fileid;            //!< id of file being solved
    uint eq[64];            //!< equation store
} EQStore;

/** This check is required to be sure that bytes array of size RFD_MAX_BLOCK can be defined in uints */
STATIC_ASSERT(!(RFD_MAX_BLOCK % sizeof(uint)),
    RFD_MAX_BLOCK_array_size_cannot_be_defined_in_uints);

/** RfdActive is memory for active re-assemblies */
typedef struct
{
    int permilCompete;      //!< what portion of the needed data blocks have been received 0 to 1000 (-1 if undetermined)
    FILE *eqs;              //!< the equation store
    FILE *data;             //!< the data store
    uint bsize;             //!< block size
    uint needed[32];        //!< bitfield of needed blocks
    union
    {
        SXMSector es;       //!< sector aligned
        EQStore eq;         //!< equation store
    } ie;
    uint ib[RFD_MAX_BLOCK / sizeof(uint)]; //!< block being solved
    union
    {
        SXMSector es;       //!< sector aligned
        EQStore eq;         //!< equation store
    } de;
    uint db[RFD_MAX_BLOCK / sizeof(uint)]; //!< block being used in solution
} RfdActive;

/** Update mutex */
static pthread_mutex_t mutex = PTHREAD_MUTEX_INITIALIZER;

/** Processing and collecting queues message types */
typedef enum {
    RFD_MSG_TYPE_COMPLETE = 0,
    RFD_MSG_TYPE_COLLECT,
    RFD_MSG_TYPE_PROCESS
} RFD_MSG_TYPE;

/** Defines RFD service active mask values */
enum {
    RFD_COLL_GOING = 0,           //!< Collecting thread is on-going
    RFD_PROC_GOING,               //!< Processing thread is on-going
    RFD_COLL_QUEUE_MUTEX_CREATED, //!< Collecting thread message queue mutex created
    RFD_PROC_QUEUE_MUTEX_CREATED  //!< Processing thread message queue mutex created
};

/** This check is required to correctly reserve space for a single slot in RFD slotsfile */
STATIC_ASSERT(sizeof(SXMSector) >= sizeof(RfdSlot),
    RfdSlot_is_more_than_one_SXMSector);

/** RfdMainService contain slots storage and RFD threads data */
typedef struct {
    pthread_t coll_pid;                 //!< Collecting thread handle
    pthread_t proc_pid;                 //!< Processing thread handle
    MSG_QUEUE_STRUCT proc_queue;        //!< Processing thread message queue
    pthread_mutex_t proc_queue_mutex;   //!< Processing thread message queue mutex
    MSG_QUEUE_STRUCT coll_queue;        //!< Collecting thread message queue
    pthread_mutex_t coll_queue_mutex;   //!< Collecting thread message queue mutex
    SXMList collected;                  //!< List of services ready for processing
    FILE *slotfile;                     //!< The slot store file on disk
    RfdActive *active[RFD_SLOTS];       //!< Dynamic storage

    union
    {
        SXMSector sect;
        RfdSlot slot;
    } slots[RFD_SLOTS];                 //!< The sector - aligned slot store

    UINT8 active_mask;                  //!< Active resources mask 
} RfdMainService;

/** A pointer to the RFD Main service structure */
static RfdMainService *service = NULL;

/************************************************************************
 *                                                                      *
 *            Private prototypes                                        *
 *            ======================                                    *
 *                                                                      *
 ************************************************************************/
static void rfd_uninit_internal(RfdMainService *s);
static int allClear(uint *map, int ml);
static int load_iblock(RfdActive *a, int ix);
static int save_iblock(RfdActive *a, int ix);
static int load_dblock(RfdActive *a, int ix);
static void clear_slot(uint is);
static void write_slot(int is);
static void clean_files(uint is);
static int create_files(uint is);
static int rfd_restore(int slot);
static uint keygen(uint *rblock, int width);
static int check_file_crc(int slot);
static int rfd_add_simple(int slot, SXMBitBuff *pkt, uint index);
static int rfd_add_gf2(int slot, SXMBitBuff *pkt, uint index, uint type);
static int rfd_add_gf4(int slot, SXMBitBuff *pkt, uint index, uint type);
static int process_block(SXMRfdService *s);
static int rfd_assembly_gf2(RfdSlot *t, int slot);
static int rfd_assembly_gf4(RfdSlot *t, int slot);
static void process_update(SXMRfdService *s);
static ptr rfd_process(ptr val);
static int rfd_check_and_start_processing(SXMRfdService *s);
static void collect_block(SXMRfdService *s);
static ptr rfd_collect(ptr val);
static void buildTables(void);
static int rfd_file_clean(SXMBitBuff *b);
static int rfd_gzip_buffer_clean(SXMBitBuff *b);

/************************************************************************
 *                                                                      *
 *            RFD Public API Functions                                  *
 *            ========================                                  *
 *                                                                      *
 ************************************************************************/

/***************************************************************************//**
 * This call is made automatically in the common LLI RFD handler before any
 * other RFD operations.
 *
 * The initializating code is only executed if the slot file is invalid (i.e.
 * only one the first call to sxm_rfd_init). Multiple threads are protected
 * through the mutex.
 *
 * If we can't open the slot file at all, give up with SXM_E_NOENT
 * Otherwise, only keep the valid slots; free up any files for
 * bad slots, in case there was data there.
 *
 * Then build the GF(4) multiplication tables
 *
 ********************************************************************************/
int sxm_rfd_init(void) {
    int rc = SXM_E_OK;

    RFD_LOG("()");

    LOCK(mutex);

    if  (service) {
        UNLOCK(mutex);
        RFD_LOG(": already initialized (%d)", SXM_E_OK);
        return SXM_E_OK;
    }

    // allocating memory for Rfd service structure
    if (!(service = (RfdMainService*)sxe_calloc(1, sizeof(*service)))) {
        RFD_LOG_CRITICAL("Error: unable to allocate memory for Rfd"
            " service structure");
        UNLOCK(mutex);
        return SXM_E_NOMEM;
    }

    do {
        uint i;

        service->slotfile = sxm_file_open_or_create('T', SXM_RFD_SERVICE_NAME, "slots", "r+b");
        if  (service->slotfile == NULL) {
            RFD_LOG_CRITICAL("Error: unable to open or create RFD slot file");
            rc = SXM_E_NOENT;
            break;
        }

        if ((rc = sxm_list_create(&service->collected, SXM_LIST_NONE)) != SXM_E_OK) {
            RFD_LOG_CRITICAL("Error: failed to create list");
            break;
        }

        if (!sxm_mutex_init(&service->coll_queue_mutex)) {
            BITSET(service->active_mask, RFD_COLL_QUEUE_MUTEX_CREATED);
        }
        else {
            RFD_LOG_CRITICAL("Error: unable to initialize RFD collection queue's mutex");
            rc = SXM_E_RESOURCE;
            break;
        }

        if ((rc = sxm_queue_create(&service->coll_queue, RFD_SLOTS,
                                   &service->coll_queue_mutex, SXM_QUEUE_RESERVE_LEGACY,
                                   "rfd_coll", NULL)) != SXM_E_OK) {
            RFD_LOG_CRITICAL("Error: unable to create RFD collection queue");
            break;
        }

        if (sxm_pthread_create(&service->coll_pid, NULL, rfd_collect, service) == 0) {
            BITSET(service->active_mask, RFD_COLL_GOING);
            sxe_thread_setname(service->coll_pid, "RFD_Collection");
        } else {
            rc = SXM_E_THREAD;
            break;
        }

        if (!sxm_mutex_init(&service->proc_queue_mutex)) {
            BITSET(service->active_mask, RFD_PROC_QUEUE_MUTEX_CREATED);
        }
        else {
            RFD_LOG_CRITICAL("Error: unable to initialize RFD processing queue's mutex");
            rc = SXM_E_RESOURCE;
            break;
        }

        if ((rc = sxm_queue_create(&service->proc_queue, RFD_SLOTS,
                                   &service->proc_queue_mutex, SXM_QUEUE_RESERVE_LEGACY,
                                   "rfd_proc", NULL)) != SXM_E_OK) {
            RFD_LOG_CRITICAL("Error: unable to create RFD processing queue");
            break;
        }

        if(sxm_pthread_create(&service->proc_pid, NULL, rfd_process, service) == 0) {
            BITSET(service->active_mask, RFD_PROC_GOING);
            sxe_thread_setname(service->proc_pid, "RFD_Processing");
        } else {
            rc = SXM_E_THREAD;
            break;
        }

        if (fread(&service->slots, sizeof(service->slots[0]), 
            RFD_SLOTS, service->slotfile) != RFD_SLOTS) {
            for (i = 0; i < RFD_SLOTS; i++) {
                clean_files(i);
                clear_slot(i);
            }
        } else
            for (i = 0; i < RFD_SLOTS; i++) {
                RfdSlot *s = (RfdSlot *)&(service->slots[i]);

                if  (s->module[0] != '\0' && 
                    sxm_crc32_calculate((byte*)s + 4, 508) != s->crc) {
                    RFD_LOG_CRITICAL("Error: RFD Slot %u corrupt", i);
                    clean_files(i);
                    clear_slot(i);
                }
            }

        buildTables();
    } while (FALSE);

    if (rc != SXM_E_OK) {
        rfd_uninit_internal(service);
        service = NULL;
    }

    UNLOCK(mutex);

    RFD_LOG("(rc=%d)", rc);

    return rc;
}


/***************************************************************************//**
 *
 * This function removes RFD cycle files
 *
 * \retval SXM_E_OK     Request accepted.
 * \retval SXM_E_NOENT  No files to delete
 *
 ********************************************************************************/
int sxm_rfd_cfile_clean(void) {
    int localRc, rc = SXM_E_NOENT;
    char name[32];
    uint i;

    localRc = sxm_file_remove('T', SXM_RFD_SERVICE_NAME, "slots");
    if (localRc != -1) {
        rc = SXM_E_OK;
    }

    for (i = 0; i < RFD_SLOTS; i++) {
        sprintf(name, "block%02u", i);
        localRc = sxm_file_remove('T', SXM_RFD_SERVICE_NAME, name);
        if (localRc != -1) {
            rc = SXM_E_OK;
        }
        sprintf(name, "eq%02u", i);
        localRc = sxm_file_remove('T', SXM_RFD_SERVICE_NAME, name);
        if (localRc != -1) {
            rc = SXM_E_OK;
        }
    }
    return rc;
}

/***************************************************************************//**
 * This function release common resources allocated for RFD processing.
 *
 ******************************************************************************/
void sxm_rfd_uninit(void)
{
    LOCK(mutex);

    if (service) {
        rfd_uninit_internal(service);
        service = NULL;
    }

    UNLOCK(mutex);
}

/***************************************************************************//**
 * Deletes the slot and its resources
 *
 * \param[in] is slot index
 *
 * \retval 0 Successfully removed
 *
 ********************************************************************************/
int sxm_rfd_remove(int is) {
    clean_files((uint)is);

    LOCK(mutex);
    clear_slot((uint)is);
    UNLOCK(mutex);

    return 0;
}

/***************************************************************************//**
 * Uncompresses and moved file into the service's location
 *
 * \param[in] lli LLI reference
 * \param[in] rfd RFD metadata
 * \param[in] type file type
 * \param[in] module the file owner module
 * \param[in] file the file name
 *
 * \retval SXM_E_OK Success
 * \retval SXM_E_ERROR Cannot open file
 * \retval SXM_E_CORRUPT Failed due to decompression issues caused by corrupted or
 *                       incorrectly compressed files.
 * \retval SXM_E_UNSUPPORTED Unsupported compression method/type 
 *
 ********************************************************************************/
int sxm_rfd_file_copy(ptr lli,
                      SXMRfdMetadata *rfd,
                      char type,
                      const char *module,
                      const char *file)
{
    FILE *pSrcFile = NULL;
    FILE *pDstFile = NULL;
    int rc = SXM_E_OK;
    int len;

    // Expand (or copy) the RFD file into the service's directory

    pSrcFile = sxm_lli_rfd_open(lli, rfd, 0);//TODOR: frag=?
    if (pSrcFile == NULL)
    {
        RFD_LOG_CRITICAL(": ERROR: Unable to open RFD file: %s", rfd->name);
        return SXM_E_ERROR;
    }

    pDstFile = sxm_file_open(type, module, file, "wb");
    if (pDstFile == NULL)
    {
        RFD_LOG_CRITICAL(": ERROR: Unable to open temporary file: %c/%s/%s",
            type, module, rfd->name);
        sxm_file_close(pSrcFile);
        return SXM_E_ERROR;
    }

    if (SXM_RFD_COMPRESSION_NONE == rfd->comp)
    {
        size_t bytesRead = 0, bytesWrite = rfd->size;
        byte readBuf[SXM_ARCH_FILE_SECTOR_SIZE];

        do
        {
            bytesRead = (bytesWrite > sizeof(readBuf)) ? 
                sizeof(readBuf) : bytesWrite;
            if (bytesRead == fread(readBuf, 1, bytesRead, pSrcFile))
            {
                if (1 != fwrite(readBuf, bytesRead, 1, pDstFile))
                {
                    RFD_LOG_CRITICAL(": ERROR: "
                        "Temporary RFD file write failed: %c/%s/%s",
                        type, module, rfd->name);
                    rc = SXM_E_ERROR;
                    break;
                }
                bytesWrite -= bytesRead;
            }
            else
            {
                RFD_LOG_CRITICAL(": ERROR: Cannot read RFD file.");
                rc = SXM_E_ERROR;
                break;
            }
        }
        while (bytesWrite > 0);
    }
    else if (SXM_RFD_COMPRESSION_GZIP == rfd->comp)
    {
        len = sxm_deflate_zipfile(pSrcFile, pDstFile);
        if (len <= 0)
        {
            RFD_LOG_CRITICAL(
                    "Failed to unzip RFD file to: %c/%s/%s",
                    type, module, rfd->name);
            rc = SXM_E_CORRUPT;
        }
    }
    else
    {
        RFD_LOG_CRITICAL(
            "Invalid compression type %u. Module: %s. File: %s.\n",
            rfd->comp, module, rfd->name);
        rc = SXM_E_UNSUPPORTED;
    }

    sxm_file_close(pDstFile);
    sxm_file_close(pSrcFile);

    return rc;
}

/***************************************************************************//**
 * Uncompresses data to buffer
 *
 * \param[in] lli LLI reference
 * \param[in] rfd RFD metadata
 * \param[out] out bitbuffer containing uncompressed data
 *
 * \retval SXM_E_OK Success
 * \retval SXM_E_NOMEM Cannot allocate memory for buffer
 * \retval SXM_E_PIPE Cannot open file
 * \retval SXM_E_UNSUPPORTED Unsupported compression method/type 
 *
 ********************************************************************************/
int sxm_rfd_file_buffer(ptr lli,
    SXMRfdMetadata *rfd,
    SXMBitBuff **out)
{
    FILE *src_file;
    int rc = SXM_E_OK;
    SXMBitFileBuff *fbuf;

    *out = NULL;
    
    if ((rfd->comp != SXM_RFD_COMPRESSION_GZIP) && (rfd->comp != SXM_RFD_COMPRESSION_NONE))
    {
        RFD_LOG_CRITICAL(": Invalid compression type %u, file: %s.\n",
            rfd->comp, rfd->name);
        return SXM_E_UNSUPPORTED;
    }

    src_file = sxm_lli_rfd_open(lli, rfd, 0);//TODOR: frag=?
    if (src_file == NULL)
    {
        RFD_LOG_CRITICAL(": ERROR: Unable to open RFD file: %s", rfd->name);
        return SXM_E_PIPE;
    }

    fbuf = (SXMBitFileBuff *)sxe_malloc(sizeof(SXMBitFileBuff));
    if (fbuf == NULL)
    {
        (void)sxm_file_close(src_file);
        return SXM_E_NOMEM;
    }
    else
    {
        rc = sxm_bitfilebuff_setup(fbuf, src_file, 0, 
            rfd->fragment[0].fragsize);
        if (rc != SXM_E_OK)
        {
            sxe_free(fbuf);
            (void)sxm_file_close(src_file);
        }
        else
        {
            fbuf->b.close = rfd_file_clean;
        }
    }

    if (rc == SXM_E_OK)
    {
        if (rfd->comp == SXM_RFD_COMPRESSION_GZIP)
        {
            *out = (SXMBitBuff *)sxe_malloc(sizeof(SXMBitZipBuff));
            if (*out == NULL)
            {
                rc = SXM_E_NOMEM;
                (void)sxm_bitbuff_close((SXMBitBuff *)fbuf);
                sxe_free(fbuf);
            }
            else
            {
                rc = sxm_bitzipbuff_setup((SXMBitZipBuff *)*out, (SXMBitBuff*)fbuf);
                if (rc != SXM_E_OK)
                {
                    (void)sxm_bitbuff_close((SXMBitBuff *)fbuf);
                    sxe_free(fbuf);
                    sxe_free(*out);
                    *out = NULL;
                }
                else
                {
                    (*out)->close = rfd_gzip_buffer_clean;
                }
            }
        }
        else /* case when file is actually uncompressed */
        {
            *out = (SXMBitBuff *)fbuf;
        }
    }

    return rc;
}

/***************************************************************************//**
 * This function removes service record from collected list
 *
 * \param[in] rfds RFD service pointer
 *
 ********************************************************************************/
void sxm_rfd_stop_processing(SXMRfdService *rfds)
{
    int rc = SXM_E_NOENT;
    SXMListEntry *entry, *nextentry;

    RFD_LOG(": looking for collected entries for service %s", rfds->service);

    LOCK(mutex);

    entry = sxm_list_first(&service->collected);
    while (entry) {
        nextentry = sxm_list_next(entry);
        if ((SXMRfdService*)(sxm_list_data(entry)) == rfds) {
            RFD_LOG(": found collected entry for service %s, removing",
                rfds->service);
            rc = sxm_list_remove(&service->collected, entry);
            if (rc != SXM_E_OK) {
                RFD_LOG_CRITICAL(": removal from list returned %d", rc);
            }
            break;
        }

        entry = nextentry;
    }

    if (rc == SXM_E_OK) {
        // if entry has been removed from collected list,
        // it means that we could post semaphore to avoid waiting at stop
        RFD_LOG(": removal for service %s completed, posting signal",
            rfds->service);
        sxm_sem_post(&rfds->semvar);
    }

    UNLOCK(mutex);
}


/***************************************************************************//**
 * This function is called to collect data block or initialize 
 * update process
 *
 * \param[in] rfds RFD service
 *
 * \retval SXM_E_OK         Operation successful
 * \retval SXM_E_RESOURCE   Queue was full, cannot place message
 *
 ********************************************************************************/
int sxm_rfd_collect(SXMRfdService *rfds)
{
    RFD_LOG(": collecting block from service %s", rfds->service);
    return sxm_queue_put(&service->coll_queue, &rfds, RFD_MSG_TYPE_COLLECT, 
        sizeof(rfds), MSG_PRIO_NORMAL);
}

/***************************************************************************//**
 *  RFD Service stop
 *
 *  This function deallocates resources used by specific data service in RFD
 *  main service.
 * 
 * \param[in] s RFD service
 *
 ********************************************************************************/
void sxm_rfd_service_stop(SXMRfdService *s)
{
    int i;

    // if thread is finished, lets correctly close all the files
    // and free the memory related to active items
    for (i = 0; i < SXM_RFD_MAX_COLLECTION; i++)
    {
        if (s->fid[i])
        {
            // extracting active slot structure
            RfdActive *a;

            if ((a = service->active[s->slot[i]]) != NULL)
            {
                if (a->eqs)
                {
                    sxm_file_close(a->eqs);
                    a->eqs = NULL;
                }

                if (a->data)
                {
                    sxm_file_close(a->data);
                    a->data = NULL;
                }

                sxe_free(a);
                service->active[s->slot[i]] = NULL;
            }
        }
    }

    RFD_LOG("(exit)");
}

/************************************************************************
 *                                                                      *
 *            Private Functions                                         *
 *            =================                                         *
 *                                                                      *
 ************************************************************************/
/***************************************************************************//**
 * Internal function for RFD file buffer cleanup
 *
 * \param[in] b SXMBitBuff structure pointer which contains SXMBitFileBuff 
 *              structure underneath. Buffer should be created by RFD service
 *
 * \return SXM_E_OK on success, other values if any error happened
 *
 *******************************************************************************/
static int rfd_file_clean(SXMBitBuff *b)
{
    int rc = SXM_E_OK;
    SXMBitFileBuff *fbuf = (SXMBitFileBuff *)b;

    if (fbuf == NULL)
    {
        return SXM_E_FAULT;
    }
    else if (sxm_file_close(fbuf->in) != SXM_E_OK)
    {
        rc = SXM_E_PIPE;
    }

    return rc;
}

/***************************************************************************//**
 * Internal function for RFD GZIP buffer cleanup
 *
 * \param[in] b SXMBitBuff structure pointer. Buffer should be created by 
 *              RFD service
 *
 * \return SXM_E_OK on success, other values if any error happened
 *
 *******************************************************************************/
static int rfd_gzip_buffer_clean(SXMBitBuff *b)
{
    int rc = SXM_E_OK;
    SXMBitZipBuff *zbuf = (SXMBitZipBuff *)b;

    if (zbuf == NULL)
    {
        return SXM_E_FAULT;
    }
    else if (sxm_bitbuff_close(zbuf->input) != SXM_E_OK)
    {
        rc = SXM_E_PIPE;
    }
    else
    {
        sxe_free(zbuf->input);
        (void)sxm_bitzipbuff_close(zbuf);
    }

    return rc;
}

/***************************************************************************//**
 * Internal function for RFD main service threads stop and deallocation.
 *
 * \param[in] s RFD main service pointer
 *
 *******************************************************************************/
static void rfd_uninit_internal(RfdMainService *s)
{
    int rc;

    sxm_list_destroy(&s->collected);

    if (ISBITSET(s->active_mask, RFD_PROC_GOING)) {
        rc = sxm_queue_put(&s->proc_queue, NULL,
            RFD_MSG_TYPE_COMPLETE, 0, MSG_PRIO_NORMAL);
        if (rc != SXM_E_OK) {
            RFD_LOG_CRITICAL(": ERROR - failed to post thread completion msg (%d)",
                rc);
        }
        pthread_join(s->proc_pid, NULL);
    }

    sxm_queue_destroy(&s->proc_queue);
    if (ISBITSET(s->active_mask, RFD_PROC_QUEUE_MUTEX_CREATED)) {
        sxm_mutex_destroy(&s->proc_queue_mutex);
    }

    if (ISBITSET(s->active_mask, RFD_COLL_GOING)) {
        rc = sxm_queue_put(&s->coll_queue, NULL,
            RFD_MSG_TYPE_COMPLETE, 0, MSG_PRIO_NORMAL);
        if (rc != SXM_E_OK) {
            RFD_LOG_CRITICAL(": ERROR - failed to post thread completion msg (%d)",
                rc);
        }
        pthread_join(s->coll_pid, NULL);
    }

    sxm_queue_destroy(&s->coll_queue);
    if (ISBITSET(s->active_mask, RFD_COLL_QUEUE_MUTEX_CREATED)) {
        sxm_mutex_destroy(&s->coll_queue_mutex);
    }

    if (s->slotfile) {
        sxm_file_close(s->slotfile);
    }

    sxe_free(s);
}

/***************************************************************************//**
 * Bitfield testing.
 *
 * \param[in] map bitmask
 * \param[in] ml number of items to check
 * \return Return 1 if no blocks are needed (map is all zeros) and 
 *         0 if one or more blocks are needed
 *******************************************************************************/
static int allClear(uint *map, int ml) {
    while (ml--)
    if  (*map++)
        return 0;

    return 1;
}

/***************************************************************************//**
 * Counts the number of bits that are set in a 32-bit value.
 *
 * This function uses a fairly fast algorithm for counting the 1's in a 32-bit
 * value.
 * 
 * \param[in] value the value who's bits should be checked
 *
 * \return the number of bits that are set in value 
 
 *******************************************************************************/
static int countOnes(UINT32 value) {
    value = value - ((value >> 1) & 0x55555555);
    value = (value & 0x33333333) + ((value >> 2) & 0x33333333);
    return (int)((((value + (value >> 4)) & 0x0F0F0F0F) * 0x01010101) >> 24);
}

/***************************************************************************//**
 * Counts the number of bits that are set several 32-bit values.
 *
 * This function uses a fairly fast algorithm for counting the 1's in a 32-bit
 * value.
 * 
 * \param[in] map pointer to the 1st 32-bit value in a multi-value bitmap
 * \param[in] ml number of 32-bit values in the bitmap
  *
 * \return the number of bits that are set in the bit map 
 
 *******************************************************************************/
static int countNeeded(uint *map, int ml) {
    int count = 0;
    while(ml--) {
        count += countOnes(*map);
        map++;
    }
    return count;
}

/************************************************************************
 *                                                                      *
 *            FILE I/O                                                  *
 *            ========                                                  *
 *                                                                      *
 ************************************************************************/

/***************************************************************************//**
 * Loads a block+equation into the 'incoming' area.
 *
 * \param[in] a rfd active
 * \param[in] ix 
 *
 * \retval SXM_E_OK operation successful
 * \retval SXM_E_PIPE loading failed
 ********************************************************************************/
static int load_iblock(RfdActive *a, int ix) {
    int rc = SXM_E_PIPE;

    do {
        if (0 != fseek(a->data, ix * (int)a->bsize, SEEK_SET)) {
            RFD_LOG(": fseek failed (%d)", errno);
            break;
        }

        if (1 != fread(a->ib, a->bsize, 1, a->data)) {
            RFD_LOG(": fread failed (feof=%d, ferror=%d)",
                feof(a->data), ferror( a->data));
            break;
        }

        if (0 != fseek(a->eqs, ix * (int)sizeof(SXMSector), SEEK_SET)) {
            RFD_LOG(": fseek failed (%d)", errno);
            break;
        }

        if (1 != fread(a->ie.es, sizeof(SXMSector), 1, a->eqs)) {
            RFD_LOG(": fread failed (feof=%d, ferror=%d)",
                feof(a->eqs), ferror(a->eqs));
            break;
        }

        rc = SXM_E_OK;
    }
    while (FALSE);

    return rc;
} 

/***************************************************************************//**
 * Writes a block+equation from the 'incoming' area.
 *
 * \param[in] a rfd active
 * \param[in] ix 
 *
 * \retval SXM_E_OK operation successful
 * \retval SXM_E_PIPE saving failed
 ********************************************************************************/
static int save_iblock(RfdActive *a, int ix) {
    int rc = SXM_E_PIPE;

    if  (a->ie.eq.solved)
    a->ie.eq.bcrcs = sxm_crc32_calculate(a->ib, a->bsize);
    else
    a->ie.eq.bcrc = sxm_crc32_calculate(a->ib, a->bsize);
    a->ie.eq.eqcrc = sxm_crc32_calculate((byte *)&a->ie.es + 4, 508);

    do {
        if (0 != fseek(a->data, ix * (int)a->bsize, SEEK_SET)) {
            RFD_LOG(": fseek failed (%d)", errno);
            break;
        }

        if (1 != fwrite(a->ib, a->bsize, 1, a->data)) {
            RFD_LOG(": fwrite failed (%d)", errno);
            break;
        }

        if (0 != fflush(a->data)) {
            RFD_LOG(": fflush failed (%d)", errno);
            break;
        }

        if (0 != fseek(a->eqs, ix * (int)sizeof(SXMSector), SEEK_SET)) {
            RFD_LOG(": fseek failed (%d)", errno);
            break;
        }

        if (1 != fwrite(a->ie.es, sizeof(SXMSector), 1, a->eqs)) {
            RFD_LOG(": fwrite failed (%d)", errno);
            break;
        }

        if (0 != fflush(a->eqs)) {
            RFD_LOG(": fflush failed (%d)", errno);
            break;
        }

        rc = SXM_E_OK;
    }
    while (FALSE);

    return rc;
} 

/***************************************************************************//**
 * Loads a block+equation into the 'data' area
 *
 * \param[in] a rfd active
 * \param[in] ix 
 *
 * \retval SXM_E_OK operation successful
 * \retval SXM_E_PIPE reading failed
 *
 ********************************************************************************/
static int load_dblock(RfdActive *a, int ix) {
    int rc = SXM_E_PIPE;

    do {
        if (0 != fseek(a->data, ix * (int)a->bsize, SEEK_SET)) {
            RFD_LOG(": fseek failed (%d)", errno);
            break;
        }

        if (1 != fread(a->db, a->bsize, 1, a->data)) {
            RFD_LOG(": fread failed (feof=%d, ferror=%d)",
                feof(a->data), ferror(a->data));
            break;
        }

        if (0 != fseek(a->eqs, ix * (int)sizeof(SXMSector), SEEK_SET)) {
            RFD_LOG(": fseek failed (%d)", errno);
            break;
        }

        if (1 != fread(a->de.es, sizeof(SXMSector), 1, a->eqs)) {
            RFD_LOG(": fread failed (feof=%d, ferror=%d)",
                feof(a->data), ferror(a->data));
            break;
        }

        rc = SXM_E_OK;
    }
    while (FALSE);

    return rc;
} 

/***************************************************************************//**
 *  Writes an empty slot
 *
 * \param[in] is 
 *
 ********************************************************************************/
static void clear_slot(uint is) {
    memset(&service->slots[is], 0, sizeof(SXMSector));
    if (fseek(service->slotfile, (long)(sizeof(SXMSector) * is), SEEK_SET) == 0) {
        if (fwrite(&service->slots[is], sizeof(SXMSector), 1,
            service->slotfile) == 1) {
            if (fflush(service->slotfile) != 0) {
                RFD_LOG(": fflush failed for slot %u (%d)", is, errno);
            }
        }
        else {
            RFD_LOG(": fwrite failed for slot %u (%d)", is, errno);
        }

    }
    else {
        RFD_LOG(": fseek failed for slot %u (%d)", is, errno);
    }

}

/***************************************************************************//**
 * This function updates slot crc and writes specified slot data and 
 *
 * \param[in] is 
 *
 ********************************************************************************/
static void write_slot(int is) {
    RfdSlot *s = (RfdSlot *)&service->slots[is];

    s->crc = sxm_crc32_calculate((byte *)&service->slots[is] + 4, 508);
    if (fseek(service->slotfile, (int)sizeof(SXMSector) * is, SEEK_SET) == 0) {
        if (fwrite(&service->slots[is], sizeof(SXMSector), 1,
            service->slotfile) == 1) {
            if (fflush(service->slotfile) != 0) {
                RFD_LOG(": fflush failed for slot %d (%d)", is, errno);
            }
        }
        else {
            RFD_LOG(": fwrite failed for slot %d (%d)", is, errno);
        }
    }
    else {
        RFD_LOG(": fseek failed for slot %d (%d)", is, errno);
    }
}

/***************************************************************************//**
 * This function removes the block and equation files
 *
 * \param[in] is 
 *
 ********************************************************************************/
static void clean_files(uint is) {
    char name[24];

    if (service->active[is]) {
        // Files must be closed before deletion
        if (service->active[is]->eqs != NULL)
            sxm_file_close(service->active[is]->eqs);

        if (service->active[is]->data != NULL)
            sxm_file_close(service->active[is]->data);

        sxe_free(service->active[is]);
        service->active[is] = NULL;
    }

    // Now delete files
    snprintf(name, sizeof(name), "eq%02u", is);
    sxm_file_remove('T', SXM_RFD_SERVICE_NAME, name);
    snprintf(name, sizeof(name), "block%02u", is);
    sxm_file_remove('T', SXM_RFD_SERVICE_NAME, name);
}

/***************************************************************************//**
 * This function allocates memory and creates the block and equation files
 * for specified slot
 *
 * \param[in] is Slot number
 *
 * \return  0 OK
 *         -1 Failed to create file
 *         -2 Failed to allocate block
 ********************************************************************************/
static int create_files(uint is) {
    char name[24];

    if (!(service->active[is] = (RfdActive *)sxe_calloc(1, sizeof(RfdActive))))
        return -2;

    snprintf(name, sizeof(name), "eq%02u", is);
    service->active[is]->eqs = sxm_file_open_or_create('T', SXM_RFD_SERVICE_NAME, name, "r+b");
    if (service->active[is]->eqs == NULL) {
        clean_files(is);
        clear_slot(is);
        return -1;
    }

    snprintf(name, sizeof(name), "block%02u", is);
    service->active[is]->data = sxm_file_open_or_create('T', SXM_RFD_SERVICE_NAME, name, "r+b");
    if (service->active[is]->data == NULL) {
        clean_files(is);
        clear_slot(is);
        return -1;
    }

    return 0;
}

/***************************************************************************//**
 * Opens the block. and equation stores and validates  the contents of
 * both files The data blocks are checked one by one, and marked as still
 * 'needed' if they were not saved correctly.
 *
 * \param[in] slot The slot number
 *
 * \retval SXM_E_OK Success
 * \retval SXM_E_PIPE File corrupt or access error; delete slot
 * \retval SXM_E_NOMEM Not enough memory to restore slot
 *
 ********************************************************************************/
static int rfd_restore(int slot) {
    RfdActive *a;
    RfdSlot *t = (RfdSlot *)&service->slots[slot];
    int result = SXM_E_OK;

    if (!(a = (RfdActive *)sxe_calloc(1, sizeof(*a))))
        return SXM_E_NOMEM;

    do {
        char name[26];
        uint crc;
        int i;

        snprintf(name, sizeof(name), "eq%02d", slot);
        a->eqs = sxm_file_open_or_create('T', SXM_RFD_SERVICE_NAME, name, "r+b");
        if (!a->eqs) {
            result = SXM_E_PIPE;
            break;
        }

        snprintf(name, sizeof(name), "block%02d", slot);
        a->data = sxm_file_open_or_create('T', SXM_RFD_SERVICE_NAME, name, "r+b");
        if (!a->data) {
            result = SXM_E_PIPE;
            break;
        }

        a->bsize = t->meta.fragment[t->frag].blksize;

        for (i = 0; i < t->nblks; i++) {
            if ((result = load_dblock(a, i)) != SXM_E_OK) {
                break;
            }

            crc = sxm_crc32_calculate((byte *)&a->de.es + 4, 508);
            if  (crc != a->de.eq.eqcrc) {
                BITS(a->needed, i);
                continue;
            }

            if  (a->de.eq.fileid != t->meta.fragment[t->frag].id) {
                BITS(a->needed, i);
                continue;
            }

            crc = sxm_crc32_calculate(a->db, a->bsize);
            if  (crc != a->de.eq.bcrc && crc != a->de.eq.bcrcs)
                BITS(a->needed, i);
        } 

    } while (FALSE);

    if (result == SXM_E_PIPE) {
        if (a->eqs) sxm_file_close (a->eqs);
        if (a->data) sxm_file_close (a->data);
        sxe_free(a);
        return SXM_E_PIPE;
    }

    //update percent (actually permil) complete
    a->permilCompete = 1000 * (t->nblks - countNeeded(a->needed, ARRAY_SIZE(a->needed))) / t->nblks;
    service->active[slot] = a;
    
    RFD_LOG(": %d.%01d%% received (%d needed of %d)", a->permilCompete/10, a->permilCompete % 10, countNeeded(a->needed, ARRAY_SIZE(a->needed)), t->nblks);

    return SXM_E_OK;
}

/***************************************************************************//**
 * \name GF(4)
 *
 * We could fix the mulby tables, but it's easy to calculate them once at startup
 * and it saves typos in lomg strings of symbols
 * @{
 ******************************************************************************/

static byte mulby2[256];
static byte mulby3[256];

static byte mul4[] = {0, 0, 0, 0, 0, 1, 2, 3, 0, 2, 3, 1, 0, 3, 1, 2};

static void buildTables(void) {
    uint i;

    for (i = 0; i < 256; i++) {
        mulby2[i] = (byte)
            ((mul4[8+((i >> 6) & 3)] << 6) |
             (mul4[8+((i >> 4) & 3)] << 4) |
             (mul4[8+((i >> 2) & 3)] << 2) |
             (mul4[8+((i >> 0) & 3)] << 0));

        mulby3[i] = (byte)
            ((mul4[12+((i >> 6) & 3)] << 6) |
             (mul4[12+((i >> 4) & 3)] << 4) |
             (mul4[12+((i >> 2) & 3)] << 2) |
             (mul4[12+((i >> 0) & 3)] << 0));
    }
}
/** @} */

/***************************************************************************//**
 * \name GF(4) equation fields
 *
 * Get and set the pos'th symbol in the GF(4) equations
 * @{
 ******************************************************************************/
static int gf4Get(uint *map, int pos) {
    return ((map[pos >> 4] >> ((pos & 15) << 1))) & 3;
}

static void gf4Set(uint *map, int pos, uint sym) {
    map[pos >> 4] |= (sym << ((pos & 15) << 1));
}
/** @} */

/***************************************************************************//**
 * Coefficients (same code as in RfdEncode).
 * This updates 'rblock' to index the next coefficient ready for the next call
 *
 * \param[in] rblock
 * \param[in] width
 *
 * \return
 *
 ********************************************************************************/
static uint keygen(uint *rblock, int width) {
    uint key = (*rblock)*1103515245;
    uint crc = sxm_crc32_calculate((byte *)&key, 4);

    *rblock = (*rblock + 1) & 0xfffff;
    if  (width == 1)
    return crc & 1;
    else
    return 1 + (crc % 3);
}

/***************************************************************************//**
 * Calculate the file CRC, then close the active files.
 * If the CRC is good mark the file complete.  This will trigger
 * the callback in the main thread code.
 *
 * \param[in] slot the slot number
 *
 * \retval SXM_E_OK CRC is correct
 * \retval SXM_E_ERROR file corrupt, delete slot
 * \retval SXM_E_PIPE cannot access block files
 *
 ********************************************************************************/
static int check_file_crc(int slot) {
    RfdActive *a = service->active[slot];
    RfdSlot *t = (RfdSlot *)(&service->slots[slot]);
    uint fcrc = (uint)-1;
    uint bl = a->bsize;
    int i = 0, ret = SXM_E_OK;
    uint left = t->meta.fragment[t->frag].fragsize;

    while (left > 0) {
        if ((ret = load_iblock(a, i)) != SXM_E_OK) {
            RFD_LOG(": cannot load block index %d.", i);
            break;
        }
        fcrc = sxm_crc32_part(fcrc, a->ib, (left > bl ? bl : left));
        left -= (left > bl ? bl : left);
        i++;
    }

    if  (a->eqs)
    {
        sxm_file_close(a->eqs);
        a->eqs = NULL;
    }
    if  (a->data)
    {
        sxm_file_close(a->data);
        a->data = NULL;
    }

    if (ret != SXM_E_OK) {
        // if read failed, just exit here
        return ret;
    }

    fcrc ^= 0xffffffff;

    if  (fcrc != t->meta.fragment[t->frag].crc) {
        //  bad news.  All we can do is scratch the whole fragment and start over
        RFD_LOG_CRITICAL("Error in Fragment. crc %08x over %d blocks.  From metadata %08x",
            fcrc, i, t->meta.fragment[t->frag].crc);
        return SXM_E_ERROR;
    }

    //  mark complete in slot store
    RFD_LOG("(slot %d) CRC OK; mark file OK", slot);

    t->state = RFD_SLOT_ASSEMBLED;
    LOCK(mutex);
    write_slot(slot);
    UNLOCK(mutex);

    return SXM_E_OK;
}

/***************************************************************************//**
 * Calculate the assembled file CRC
 *
 * \param[in] slot the slot number
 *
 * \retval SXM_E_OK CRC is correct
 * \retval SXM_E_ERROR file corrupt, delete slot
 * \retval SXM_E_PIPE cannot access file
 * \retval SXM_E_NOMEM cannot allocate buffer for reading
 *
 ********************************************************************************/
static int check_crc_assembled(int slot) {
    RfdSlot *t = &service->slots[slot].slot;
    uint fcrc = SXM_CRC32_INIT_VAL;
    int rc = SXM_E_OK;
    uint left = t->meta.fragment[t->frag].fragsize;
    uint loaded;
    char name[RFD_FILE_NAME_SIZE_MAX];
    FILE *blocks;
    byte *buffer;

    snprintf(name, sizeof(name), "block%02d", slot);
    blocks = sxm_file_open('T', SXM_RFD_SERVICE_NAME, name, "rb");
    if (blocks == NULL) {
        return SXM_E_PIPE;
    }

    buffer = (byte *)sxe_malloc(RFD_FILE_BUFFER_SIZE);
    if (buffer == NULL) {
        (void)sxm_file_close(blocks);
        return SXM_E_NOMEM;
    }

    while (left > 0) {
        loaded = (uint)fread(buffer, 1, MIN(left, RFD_FILE_BUFFER_SIZE), blocks);
        if (loaded == 0) {
            /* cannot read from file */
            rc = SXM_E_PIPE;
            break;
        }
        fcrc = sxm_crc32_part(fcrc, buffer, loaded);
        left -= loaded;
    }

    sxm_file_close(blocks);
    sxe_free(buffer);

    if (rc == SXM_E_OK) {
        fcrc ^= SXM_CRC32_INIT_VAL;
        if (fcrc != t->meta.fragment[t->frag].crc) {
            /* bad news.  All we can do is scratch the whole fragment and start over */
            RFD_LOG_CRITICAL(": error in fragment(file crc %08x, metadata crc %08x)",
                fcrc, t->meta.fragment[t->frag].crc);
            rc = SXM_E_ERROR;
        }
    }

    return rc;
}

/***************************************************************************//**
 * \name Equation Solvers
 *
 * The routines are called only when:
 *      - the slot is valid
 *      - the equation and data files are open
 *      - the equation store and block store have been checked
 *      - the 'needed' bits are set correctly
 * @{
 ********************************************************************************/
/***************************************************************************//**
 * Simple
 *
 * \param[in] slot the slot number
 * \param[in] pkt binary data
 * \param[in] index
 *
 * \retval SXM_E_OK complete
 * \retval SXM_E_NOENT incomplete
 * \retval SXM_E_PIPE file system error
 * \retval SXM_E_INVAL argument out of bounds
 *
 ********************************************************************************/
static int rfd_add_simple(int slot, SXMBitBuff *pkt, uint index) {
    RfdActive *a = service->active[slot];
    RfdSlot *t = (RfdSlot *)(&service->slots[slot]);

    RFD_LOG("(slot %d, index %u)", slot, index);

    if (index >= sizeof(a->needed) * SXM_ARCH_BITS_IN_BYTE) {
        return SXM_E_INVAL;
    }

    memcpy(a->ib, pkt->b, a->bsize);

    if  (BITP(a->needed, index)) {
        int rc;
        BITC(a->needed, index);
        if ((rc = save_iblock(a, (int)index)) != SXM_E_OK) {
            return rc;
        }
    }

    if  (allClear(a->needed, 32)) {
        // if all blocks collected, lets mark it ready for processing
        a->permilCompete = 1000;
        LOCK(mutex);
        ((RfdSlot*)(&service->slots[slot]))->state = RFD_SLOT_COLLECTED;
        write_slot(slot);
        UNLOCK(mutex);
        RFD_LOG("(File is ready for processing)");
        return SXM_E_OK;
    }

    //update percent (actually permil) complete
    a->permilCompete = 1000 * (t->nblks - countNeeded(a->needed, ARRAY_SIZE(a->needed))) / t->nblks;
    RFD_LOG(": %d.%01d%% received (%d needed of %d)", a->permilCompete/10, a->permilCompete % 10, countNeeded(a->needed, ARRAY_SIZE(a->needed)), t->nblks);
    return SXM_E_NOENT;
}

/***************************************************************************//**
 * FG2
 *
 * \param[in] slot the slot number
 * \param[in] pkt binary data
 * \param[in] index
 * \param[in] type
 *
 * \retval SXM_E_OK complete
 * \retval SXM_E_NOENT incomplete
 * \retval SXM_E_PIPE file system error
 * \retval SXM_E_INVAL Argument out of bounds
 *
 ********************************************************************************/
static int rfd_add_gf2(int slot, SXMBitBuff *pkt, uint index, uint type) {
    RfdActive *a = service->active[slot];
    RfdSlot *t = (RfdSlot *)(&service->slots[slot]);
    uint bl = a->bsize;
    int i, tbit;
    uint j;

    RFD_LOG("(slot %d, index %u, type %u)", slot, index, type);

    memset(a->ie.es, 0, 512);
    a->ie.eq.fileid = t->meta.fragment[t->frag].id;

    if(type == 0) {
        if (index < (SXM_ARCH_BITS_IN_BYTE * sizeof(a->ie.eq.eq))) {
            BITS(a->ie.eq.eq, index);
        }
        else {
            return SXM_E_INVAL;
        }
    }
    else
    for (i = 0; i < t->nblks; i++) 
        if (keygen(&index, 1) == 1)
        BITS(a->ie.eq.eq, i);
    
    memcpy(a->ib, pkt->b, a->bsize);

    for (tbit = t->nblks-1; tbit >= 0; tbit--) 
        if (BITP(a->ie.eq.eq, tbit)) {
            int rc;

            if (BITP(a->needed, tbit)) {
                BITC(a->needed, tbit);
                RFD_LOG("    saving block at %d\n", tbit);
                if ((rc = save_iblock(a, tbit)) != SXM_E_OK) {
                    return rc;
                }
                break;
            }
            else {
                uint *xb = a->db;
                uint *eb = (uint *)a->de.eq.eq;
                uint *e = (uint *)a->ie.eq.eq;
                uint *ib = a->ib;

                if ((rc = load_dblock(a, tbit)) != SXM_E_OK) {
                    return rc;
                }

                j = bl >> 2;
                while (j--)
                    *ib++ ^= *xb++;
                j = 32;
                while (j--) 
                    *e++ ^= *eb++;
            }
        }

    if  (allClear(a->needed, 32)) {
        a->permilCompete = 1000; 
        LOCK(mutex);
        t->state = RFD_SLOT_COLLECTED;
        write_slot(slot);
        UNLOCK(mutex);
        RFD_LOG("(File is complete)");
        return SXM_E_OK;
    }
    //update percent (actually permil) complete
    a->permilCompete = 1000 * (t->nblks - countNeeded(a->needed, ARRAY_SIZE(a->needed))) / t->nblks;
    RFD_LOG(": %d.%01d%% received (%d needed of %d)", a->permilCompete/10, a->permilCompete % 10, countNeeded(a->needed, ARRAY_SIZE(a->needed)), t->nblks);
    return SXM_E_NOENT;
}

/***************************************************************************//**
 * FG4
 *
 * \param[in] slot the slot number
 * \param[in] pkt binary data
 * \param[in] index
 * \param[in] type
 *
 * \retval SXM_E_OK complete
 * \retval SXM_E_NOENT incomplete
 * \retval SXM_E_PIPE file system error
 *
 ********************************************************************************/
static int rfd_add_gf4(int slot, SXMBitBuff *pkt, uint index, uint type) {
    RfdActive *a = service->active[slot];
    RfdSlot *t = (RfdSlot *)(&service->slots[slot]);
    int i, tbit, sym;
    uint bl = a->bsize, j;
    byte *tb;

    RFD_LOG("(slot %d, index %u, type %u)", slot, index, type);

    memset(a->ie.es, 0, 512);
    a->ie.eq.fileid = t->meta.fragment[t->frag].id;
    if  (type == 0)
    gf4Set(a->ie.eq.eq, (int)index, 1);
    else
    for (i = 0; i < t->nblks; i++) 
        gf4Set(a->ie.eq.eq, i, keygen(&index, 4));

    memcpy(a->ib, pkt->b, a->bsize);

    //  work out where to fit this block into the matrix

    for (tbit = t->nblks-1; tbit >= 0; tbit--) 
        if ((sym = gf4Get(a->ie.eq.eq, tbit)) != 0) {
            int rc;

            if (BITP(a->needed, tbit)) {
                byte *e = (byte *)a->ie.eq.eq;

                //  reduce this symbol to a '1' and insert it into the file
                tb = (byte *)a->ib;
                j = bl;
                if (sym == 2) {
                    while (j--) {
                        *tb = mulby3[*tb];
                        tb++;
                    }
                    j = 256;
                    while (j--) {
                        *e = mulby3[*e];
                        e++;
                    }
                }
                else if (sym == 3) {
                    while (j--) {
                        *tb = mulby2[*tb];
                        tb++;
                    }
                    j = 256;
                    while (j--) {
                        *e = mulby2[*e];
                        e++;
                    }
                }

                RFD_LOG("(saved in row %d)", tbit);
                BITC(a->needed, tbit);
                if ((rc = save_iblock(a, tbit)) != SXM_E_OK) {
                    return rc;
                }
                break;
            }
            else {
                byte *xb = (byte *)a->db;
                byte *t = (byte *)a->de.eq.eq;
                byte *e = (byte *)a->ie.eq.eq;
                byte *tb = (byte *)a->ib;

                if ((rc = load_dblock(a, tbit)) != SXM_E_OK) {
                    return rc;
                }

                j = bl;
                if  (sym == 1) {
                    while (j--)
                    *tb++ ^= *xb++;
                    j = 256;
                    while (j--)
                    *e++ ^= *t++;
                } else if (sym == 2) {
                    while (j--)
                    *tb++ ^= mulby2[*xb++];
                    j = 256;
                    while (j--)
                    *e++ ^=  mulby2[*t++];
                } else if (sym == 3){
                    while (j--)
                    *tb++ ^= mulby3[*xb++];
                    j = 256;
                    while (j--)
                    *e++ ^=  mulby3[*t++];
                }
            }
        }
    
    if  (allClear(a->needed, 32)) {
        a->permilCompete = 1000; 
        LOCK(mutex);
        t->state = RFD_SLOT_COLLECTED;
        write_slot(slot);
        UNLOCK(mutex);
        RFD_LOG("(File is complete)");
        return SXM_E_OK;
    }

    //update percent (actually permil) complete
    a->permilCompete = 1000 * (t->nblks - countNeeded(a->needed, ARRAY_SIZE(a->needed))) / t->nblks;
    RFD_LOG(": %d.%01d%% received (%d needed of %d)", a->permilCompete/10, a->permilCompete % 10, countNeeded(a->needed, ARRAY_SIZE(a->needed)), t->nblks);
    return SXM_E_NOENT;
}
/** @} */

/***************************************************************************//**
 * RFD Metadata Parser
 *
 * \param[in] pkt binary data
 * \param[out] out parsed RFD Metadata in case of success
 *
 * \retval 0 Success
 * \retval 1 Error
 *
 ********************************************************************************/
int sxm_rfd_parse(SXMBitBuff *pkt, SXMRfdMetadata *out) {
    int i;

    memset(out, 0, sizeof(SXMRfdMetadata));

    BAUDOT(out->name, BAUDOT_MODE_START_WITH_LETTERS,
        SXM_RFD_METADATA_FILE_NAME_LEN_MAX, SXM_RFD_METADATA_FILE_NAME_LEN_MAX);
    NEXT(out->size, 32);
    NEXT_TO(byte, out->carid, 4);
    NEXT_TO(byte, out->alg, 3);
    NEXT_TO(byte, out->comp, 3);
    NEXT_TO(ushort, out->lifetime, 14);
    NEXT_TO(byte, out->nfrags, 4);
    out->nfrags++;

    RFD_LOG(": %s size %d car %d alg %d comp %d life %d nf %d",
        out->name, out->size, out->carid, out->alg, out->comp,
        out->lifetime, out->nfrags);

    for  (i = 0; i < out->nfrags; i++) {
    NEXT_TO(ushort, out->fragment[i].id, 16);
    NEXT_TO(ushort, out->fragment[i].blksize, 14);
    NEXT(out->fragment[i].fragsize, 24);
    NEXT(out->fragment[i].crc, 32);
    }
    if  (FLAG()) {
    SKIP(7);
    NEXT_TO(ushort, out->dmi, 10);
    }
    return pkt->err;
}

/***************************************************************************//**
 * Provides access to the metadata for a slot.
 *
 * \param[in] is slot index
 *
 * \return a pointer to the the metadata for the requested slot
 *        The structure should be used read-only.
 *
 ********************************************************************************/
SXMRfdMetadata *sxm_rfd_extract_meta(int is) {
    RfdSlot *s = (RfdSlot *)(&service->slots[is]);
    return &s->meta;
} 

/***************************************************************************//**
 * Provides access to the permil complete for a slot.
 *
 * Indicates how many of the needed data blocks have been received for an RFD 
 * fragment. 
 * Keep in mind that there are error cases which allow the permil complete to
 * go backwards. It can even get up to 1000 and then drop to -1 or 0 without
 * actually receiving a complete and valid fragment.
 *
 * \param[in] is slot index in the the RFD services global slot store
 *
 * \return a value from 0 to 1000 if permil complete is known, 
           -1 if permil complete is unknown
 *
 ********************************************************************************/
int sxm_rfd_extract_permil_complete(int is) {
    RfdActive *a = service->active[is];
    if (a) {
        return a->permilCompete;
    } else {
        return -1; //slot doesn't exist, or hasn't been restored yet
    }
} 

/***************************************************************************//**
 * Routine rebuilds the service slot structure.
 * \note The service structure must have the service name set, at a minimum
 *
 * \param[in] svc a pointer to a RFD Service
 *
 * \return the number of slots matched
 *
 ********************************************************************************/
int sxm_rfd_find_all(SXMRfdService *svc) {
    int i, j = 0;

    for  (i = 0; i < RFD_SLOTS; i++) {
        RfdSlot *s = (RfdSlot *)(&service->slots[i]);

        if  (strequ(s->module, svc->service) && j < 4) {
            svc->slot[j] = i;
            // Reset state. Will be updated in the assembler thread.
            svc->state[j] = 0;
            svc->fid[j++] = s->meta.fragment[s->frag].id;
        }
    }

    return j;
}

/***************************************************************************//**
 * Find an empty slot in the slot store
 * 
 * - Create the equation and block files
 * - Write the slot data
 * - Write empty blocks to the equation and block files
 * - Then return the slot number
 * 
 * Back off any changes if the files cannot be written
 *
 * \param[in] rfd RFD service
 * \param[in] module module name
 * \param[in] frag index into the SXMRfdMetadata fragment array
 *
 * \return slot index or an -1 in case of error
 *
 ********************************************************************************/
int sxm_rfd_find_new(const SXMRfdMetadata *rfd, const char *module, int frag) {
    int rc;
    uint i, n, nblks;
    RfdSlot *s;
    RfdActive *a;

    nblks = (rfd->fragment[frag].fragsize + rfd->fragment[frag].blksize - 
        1) / rfd->fragment[frag].blksize;

    LOCK(mutex);

    for  (i = 0; i < RFD_SLOTS; i++) {
        s = (RfdSlot *)(&service->slots[i]);
        if  (s->module[0] != '\0')
            continue;

        // prepare the slot
        memset(s, 0, sizeof(service->slots[i]));
        strncpy(s->module, module, sizeof(s->module) - 1);
        memcpy(&s->meta, rfd, sizeof(SXMRfdMetadata));
        s->frag = frag;
        s->nblks = (int)nblks;
        
        RFD_LOG("(slot %u, module %s, file %s, %u blocks)", i, module, rfd->name, nblks);

        rc = create_files(i);
        //  create the equation and file store
        if  (rc != 0) {
            memset(s, 0, sizeof(RfdSlot));
            UNLOCK(mutex);
            return rc;
        }

        write_slot((int)i);
        //  unlock the mutex while we write the files
        UNLOCK(mutex);

        a = service->active[i];
        a->bsize = rfd->fragment[frag].blksize;

        if ((0 != fseek(a->eqs, 0, SEEK_SET)) || 
            (0 != fseek(a->data, 0, SEEK_SET))) {
            RFD_LOG(": fseek failed (%d)", errno);
            rc = -1;
        }
        else {
            for (n = 0; n < nblks; n++) {
                if (fwrite(a->ie.es, 512, 1, a->eqs) != 1) {
                    break;
                }
                if (fwrite(a->ib, a->bsize, 1, a->data) != 1) {
                    break;
                }
                BITS(a->needed, n);
            }
        }

        if ((rc != 0) || (n < nblks)) {
            clean_files(i);
            LOCK(mutex);
            clear_slot(i);
            UNLOCK(mutex);
            return -1;
        }

        fflush(a->eqs);
        fflush(a->data);

        return (int)i;
    }

    UNLOCK(mutex);

    return -1;
} 

/***************************************************************************//**
 * RFD BLOCKS COLLECTING
 * 
 * This code is executed in a single RFD collecting thread
 *
 * Here is a description what this function does:
 * Parse the header of the block to get the file-id and index
 * Find the slot for this service and file-id
 * Make sure the active structure is valid (files are open)
 * Add this block into the recovery (maybe completing the file)
 * If that didn't work, delete the slot and remove it from the
 * service list.
 *
 * \param[in] s RFD service
 *
 * \retval SXM_E_OK  file is complete
 * \retval SXM_E_NOENT  file is incomplete
 * \retval SXM_E_PIPE  file cannot be accessed
 * \retval SXM_E_UNSUPPORTED unknown coding algorithm
 *
 ********************************************************************************/
static int process_block(SXMRfdService *s)
{
    ushort id, type;
    SXMBitBuff b, *pkt;
    RfdSlot *t = NULL;
    ushort slot;
    int i, rc = SXM_E_OK;
    uint index = 0;
    uint ext;

    pkt = &b;

    sxm_bitbuff_setup(pkt, s->packet, (uint)s->pl);
    SKIP(4+s->carsize);
    NEXT_TO(ushort, id, 16);
    NEXT_TO(ushort, type, 3);

    if (type == 0)
        NEXT(index, 13);
    else if (type == 2 || type == 3)
        NEXT(index, 20);
    else
        return SXM_E_NOENT;       // don't handle full-XOR block (yet)

    ALIGN();

    if ((NEXT(ext, 8)) > 0)
        while (ext-- > 0)
            SKIP(8);
    
    for (slot = 0; slot < RFD_SLOTS; slot++) {
        t = (RfdSlot *)(&service->slots[slot]);
        if (strequ(t->module, s->service) && t->meta.fragment[t->frag].id == id)
            break;
    }
    
    if  (slot == RFD_SLOTS) {
        RFD_LOG("%s: packet for id %d type %d index %d (no slot)", s->service, id, type, index);
        return SXM_E_NOENT;
    }
    
    if (t->state != RFD_SLOT_INITIAL) {
        return SXM_E_OK;
    }

    /* restoring collecting structure if its not ready yet */
    if (service->active[slot] == NULL) {
        rc = rfd_restore(slot);
    }

    if (rc == SXM_E_NOMEM) {
        // lets report that file is incomplete
        // at next block message we could try to allocate memory again
        rc = SXM_E_NOENT;
    }
    
    if (rc == SXM_E_OK) {
        if (t->meta.alg == 0) {
            rc = rfd_add_simple(slot, pkt, index);
        }
        else if (t->meta.alg == 1) {
            rc = rfd_add_gf2(slot, pkt, index, type);
        }
        else if (t->meta.alg == 2) {
            rc = rfd_add_gf4(slot, pkt, index, type);
        }
        else {
            RFD_LOG(": unknown rfd coding algorithm (%u)",
                t->meta.alg);
            rc = SXM_E_UNSUPPORTED;
        }
    }

    if ((rc != SXM_E_OK) && (rc != SXM_E_NOENT)) {

        RFD_LOG_CRITICAL(": %s - error adding block (%d). Deleting slot %u.",
            s->service, rc, slot);
        for (i = 0; i < SXM_RFD_MAX_COLLECTION; i++) {
            if (s->fid[i] == id) {
                s->fid[i] = 0;
            }
        }

        (void)sxm_rfd_remove(slot);
    }

    return rc;
}

/***************************************************************************//**
 * This function assembles RFD file using GF(2)
 * 
 * \param[in] t     RFD slot pointer
 * \param[in] slot  RFD slot index
 *
 * \retval SXM_E_OK assembly is complete
 * \retval SXM_E_PIPE error during assembly
 *
 *******************************************************************************/
static int rfd_assembly_gf2(RfdSlot *t, int slot)
{
    int i, j;
    RfdActive *a = service->active[slot];
    uint bl = a->bsize;
    int rc = SXM_E_OK;

    for (i = 1; i < t->nblks; i++) {
        uint *es = a->ie.eq.eq;
        uint k;

        if ((rc = load_iblock(a, i)) != SXM_E_OK) {
            return rc;
        }


        for  (j = 0; j < i; j++)
            if  (BITP(es, j)) {
                uint *ib = a->ib;
                uint *xb = a->db;

                if ((rc = load_dblock(a, j)) != SXM_E_OK) {
                    return rc;
                }

                k = bl >> 2;
                while (k--)
                    *ib++ ^= *xb++;
            }

        a->ie.eq.solved = 1;
        if ((rc = (save_iblock(a, i))) != SXM_E_OK) {
            return rc;
        }
    }

    return rc;
}

/***************************************************************************//**
 * This function assembles RFD file using GF(4)
 * 
 * \param[in] t     RFD slot pointer
 * \param[in] slot  RFD slot index
 *
 * \retval SXM_E_OK assembly is complete
 * \retval SXM_E_PIPE error during assembly
 *
 *******************************************************************************/
static int rfd_assembly_gf4(RfdSlot *t, int slot)
{
    int i, j, sym;
    byte *xb, *xs;
    uint *xbi, *xsi;
    RfdActive *a = service->active[slot];
    uint bl = a->bsize;

    for (i = 1; i < t->nblks; i++) {
        uint *es = a->ie.eq.eq;
        int rc;
        uint k;

        if ((rc = load_iblock(a, i)) != SXM_E_OK) {
            return rc;
        }

        for (j = 0; j < i; j++) 
            if ((sym = gf4Get(es, j)) != 0) {
                if ((rc = load_dblock(a, j)) != SXM_E_OK) {
                    return rc;
                }

                xb = (byte *)a->db;
                xs = (byte *)a->ib;

                k = bl;
                if (sym == 1) {
                    xsi = a->ib;
                    xbi = a->db;
                    k >>= 2;
                    while (k--)
                        *xsi++ ^= *xbi++;
                } else if (sym == 2) 
                    while (k--)
                        *xs++ ^= mulby2[*xb++];
                else
                    while (k--)
                        *xs++ ^= mulby3[*xb++];
            }

        a->ie.eq.solved = 1;
        if ((rc = save_iblock(a, i)) != SXM_E_OK) {
            return rc;
        }
    }

    return SXM_E_OK;
}

/***************************************************************************//**
 * This function executes processing of RFD update.
 *
 * Here is detailed description of operations:
 * 1. Function checks if specified service RFD update is collected.
 * 2. For collected update, assembling and CRC check is executed.
 * 3. For assembled and CRC checked update service update function is called.
 * 
 * \param[in] s     RFD service pointer
 *
 *******************************************************************************/
static void process_update(SXMRfdService *s)
{
    int i, rc;

    for (i = 0; i < SXM_RFD_MAX_COLLECTION; i++)
    {
        // Check for active RFD collection
        if (s->fid[i] != 0)
        {
            RfdSlot *t = (RfdSlot *)&service->slots[s->slot[i]];
            BOOL verified = FALSE;

            // Check if update is collected and should be assembled
            if (t->state == RFD_SLOT_COLLECTED)
            {
                if (t->meta.alg == 1) {
                    rc = rfd_assembly_gf2(t, s->slot[i]);
                }
                else if (t->meta.alg == 2) {
                    rc = rfd_assembly_gf4(t, s->slot[i]);
                }
                else {
                    // nothing to do for other alg types
                    rc = SXM_E_OK;
                }

                RFD_LOG(": assembling completed (%d)", rc);

                if (rc == SXM_E_OK) {
                    rc = check_file_crc(s->slot[i]);
                    verified = TRUE;
                }

                if (rc != SXM_E_OK) {
                    RFD_LOG_CRITICAL(": assembling failed for %s, slot %d (%d)",
                        s->service, s->slot[i], rc);
                    s->fid[i] = 0;
                    sxm_rfd_remove(s->slot[i]);
                    break;
                }
            }

            if (t->state == RFD_SLOT_ASSEMBLED)
            {
                int status;

                /* calculating CRC for update file */
                if (verified == FALSE)
                {
                    rc = check_crc_assembled(s->slot[i]);
                    if (rc != SXM_E_OK) {
                        RFD_LOG_CRITICAL(": crc check failed for %s, slot %d (%d)",
                            s->service, s->slot[i], rc);
                        s->fid[i] = 0;
                        (void)sxm_rfd_remove(s->slot[i]);
                        break;
                    }
                }

                s->state[i] = 1;
                RFD_LOG("(Calling service collected routine)");
                status = s->collected(&t->meta);
                if (status == SXM_E_OK)
                {
                    // Service has completed processing.
                    // Delete appropriate RFD files and clear RFD slot.
                    RFD_LOG("(Removing RFD file)");
                    sxm_rfd_remove(s->slot[i]);
                    s->fid[i] = 0;
                }
                else
                {
                    RFD_LOG_CRITICAL("Error %d returned for RFD update '%s' service '%s'",
                                     status, t->meta.name, s->service);
                }

                break;
            }
        }
    }

    return;
}

/***************************************************************************//**
 * RFD processing thread loop function
 *
 * \param[in] val RFD Main service pointer
 *
 *******************************************************************************/
static ptr rfd_process(ptr val)
{
    int rc;
    SXMQueueMsg msg;
    RfdMainService *s = (RfdMainService *)val;

    RFD_LOG(": starting processing thread.");

    do {
        rc = sxm_queue_get(&s->proc_queue, &msg, SXM_QUEUE_INFINITE_TIMEOUT);
        if (rc == SXM_E_OK) {
            if (msg.msgType == RFD_MSG_TYPE_PROCESS) {
                SXMListEntry *entry;
                SXMRfdService *rfds;

                RFD_LOG(": received process message.");
                LOCK(mutex);
                entry = sxm_list_first(&s->collected);
                if (!entry) {
                    // looks like list is empty, lets wait for next msg
                    UNLOCK(mutex);
                }
                else {
                    rfds = (SXMRfdService *)sxm_list_data(entry);
                    rc = sxm_list_remove(&s->collected, entry);
                    if (rc != SXM_E_OK) {
                        RFD_LOG_CRITICAL(": ERROR - cannot remove entry from "
                            "collected (%d)", rc);
                    }
                    RFD_LOG(": service %s is ready for processing", rfds->service);
                    UNLOCK(mutex);

                    process_update(rfds);

                    RFD_LOG(": service %s processing completed, posting signal", 
                        rfds->service);
                    // post signal to show that collection/stop could be continued
                    sxm_sem_post(&rfds->semvar);
                }
            }

            /* free message memory */
            sxm_queue_msg_release(&msg);

        } else if (rc != SXM_E_TIMEOUT) {
            RFD_LOG_CRITICAL(": ERROR - cannot access RFD processing queue (rc=%d)", rc);
            break;
        }

    } while (msg.msgType != RFD_MSG_TYPE_COMPLETE);

    RFD_LOG(": finishing processing thread.");
    return NULL;
}

/***************************************************************************//**
 * This function checks if service RFD file collection is ready for
 * processing
 *
 * \param[in] s     RFD service pointer
 *
 *******************************************************************************/
static int rfd_check_and_start_processing(SXMRfdService *s)
{
    int i, rc = SXM_E_NOENT;

    for (i = 0; i < SXM_RFD_MAX_COLLECTION; i++)
    {
        // Check for active RFD collection
        if (s->fid[i] != 0)
        {
            RfdSlot *t = (RfdSlot *)&service->slots[s->slot[i]];

            // Check if update is ready to transfer to the processing
            // and apply update only once when it just became ready.
            // It can be the first service start or when RFD file collection completed.
            if ((t->state != RFD_SLOT_INITIAL) && (s->state[i] == 0))
            {
                RFD_LOG(": there is completed file for service '%s' file id %d "
                    "(state %d)", s->service, s->fid[i], t->state);
                /* restoring collecting structure if its not ready yet */
                if (service->active[s->slot[i]] == NULL) {
                    rc = rfd_restore(s->slot[i]);

                    if (rc == SXM_E_NOMEM) {
                        // lets report that file is incomplete
                        // at next block message we could try to allocate memory again
                        rc = SXM_E_NOENT;
                        break;
                    }
                    else if (rc != SXM_E_OK) {
                        RFD_LOG_CRITICAL(": '%s' - error restoring data (rc=%d). "
                            "Deleting slot %d.",
                            s->service, rc, s->slot[i]);
                        s->fid[i] = 0;
                        (void)sxm_rfd_remove(s->slot[i]);
                        break;
                    }
                }

                LOCK(mutex);
                rc = sxm_list_add(&service->collected, s, FALSE, NULL);
                UNLOCK(mutex);
                if (rc == SXM_E_OK) {
                    rc = sxm_queue_put(&service->proc_queue, NULL, 
                        RFD_MSG_TYPE_PROCESS, 0, MSG_PRIO_NORMAL);
                } else {
                    RFD_LOG_CRITICAL("Failed to add service %s to collected list",
                        s->service);
                }
                break;
            }
        }
    }

    return rc;
}

/***************************************************************************//**
 * Block collecting function
 *
 * \param[in] s     RFD service pointer
 *
 *******************************************************************************/
static void collect_block(SXMRfdService *s)
{
    int rc;

    RFD_LOG(": received message from service %s", s->service);

    rc = rfd_check_and_start_processing(s);
    if (rc == SXM_E_NOENT) {
        rc = process_block(s);
        // if collection is full, we could start processing
        if (rc == SXM_E_OK) {
            rc = rfd_check_and_start_processing(s);
        }
    }

    // in case if rfd_check_and_start_processing returned SXM_E_OK, nothing to
    // do here. Service handle is posted to RFD processing thread, and need
    // to keep semaphore blocked until the update processing completion.
    // Otherwise, need to post the semaphore and report error or print ordinary
    // log message

    if (rc == SXM_E_NOENT) {
        // if all blocks are not collected yet, lets post signal
        RFD_LOG(": posting signal to service %s", s->service);
        sxm_sem_post(&s->semvar);
    } else if (rc != SXM_E_OK) {
        RFD_LOG_CRITICAL(": ERROR - returned %d", rc);
        // error happened, need to post the signal to the service to unblock it
        sxm_sem_post(&s->semvar);
    }
}

/***************************************************************************//**
 * RFD Collection thread main loop
 * 
 * coll_queue is used to post new packets into the collection thread 
 *
 * If a slot contains a complete file, processing thread will run.
 * If there is a packet to collect, it is added to the appropriate RFD slot.
 *
 * \param[in] val RFD Main service
 *
 * \return thread function result
 *
 ********************************************************************************/
static ptr rfd_collect(ptr val)
{
    int rc;
    SXMQueueMsg msg;
    RfdMainService *s = (RfdMainService *)val;

    RFD_LOG(": starting collection thread.");

    do {
        rc = sxm_queue_get(&s->coll_queue, &msg, SXM_QUEUE_INFINITE_TIMEOUT);
        if (rc == SXM_E_OK) {
            if (msg.msgType == RFD_MSG_TYPE_COLLECT) {
                SXMRfdService** ppData =
                    (SXMRfdService**)sxm_queue_msg_data(&msg);
                if (ppData) {
                    collect_block(*ppData);
                }
                else {
                    RFD_LOG_CRITICAL(": ERROR - the COLLECT message must have a data");
                }
            }
            /* free message memory */
            sxm_queue_msg_release(&msg);
        }
        else if (rc != SXM_E_TIMEOUT) {
            RFD_LOG_CRITICAL(": ERROR - cannot access RFD collection queue (rc=%d)", rc);
            break;
        }

    } while (msg.msgType != RFD_MSG_TYPE_COMPLETE);

    RFD_LOG(": finishing collection thread.");
    return NULL;
}
