/******************************************************************************/
/*                Copyright (c) Sirius XM Satellite Radio, Inc.               */
/*                            All Rights Reserved                             */
/*      Licensed Materials - Property of Sirius XM Satellite Radio, Inc.      */
/******************************************************************************/
/***************************************************************************//**
 * 
 * \file sxm_platform.c
 * \author Tim O'Connell
 * \date 11/26/2013
 * \brief  Prevents name collisions on build environment level:
 *
 * \details Prevents name collisions  by adding a thin shim around certain
 * platform calls. Without these, we'd need to, for example, include Windows.h
 * in various headers / files, which would result in naming collisions with things
 * like SetServiceState, DMI, etc.
 * Part of this file is related to threads and sync operations.
 * Various flags related to these operations are used below:
 * HAVE_NO_PTHREAD_SETNAME - this flag should be defined for non-WIN32 builds
 * with pthreads library without pthread_setname_np function. SXe Cmake build
 * has automatic detection of the function existence and sets this flag if needed.
 *
 ************************************************************************/
#include <sxm_build.h>

#ifndef SDKFILES_STANDALONE_BUILD
#define _GNU_SOURCE
#include <pthread.h>
#include <semaphore.h>
#endif

#ifdef WIN32
    #include <time.h>
    #include <sys/timeb.h>
    #include <Windows.h>
#if SXM_DEBUG_MEMORY_TRACKING > 1
    #include <DbgHelp.h>
#endif
#else
    #if !defined(ANDROID) && defined(__linux__)
        #include <execinfo.h>
        #if SXM_DEBUG_MEMORY_TRACKING > 1
            #include <sys/types.h>
        #endif
    #endif
    #include <unistd.h>
    #include <stdlib.h>
#endif

#include <util/sxm_time_internal.h>
#include <util/sxm_platform_internal.h>

#if SXM_DEBUG_MEMORY_TRACKING > 0

#include <stdio.h>

/** Turn on MSVS embedded memory tracking mechanism */
#ifdef _MSC_VER
    #define _CRTDBG_MAP_ALLOC
    #include <stdlib.h>
    #include <crtdbg.h>
#endif

/** Defines number of simultaneously allocations under track */
#ifndef SXM_MEM_BLOCK_TRACK_MAX
#define SXM_MEM_BLOCK_TRACK_MAX (4096)
#endif
/** Block signature */
#define SXM_MEM_BLOCK_SIG ((size_t)0xFEEDF00D)
/** End-of-block footprint (guardian) */
#define SXM_MEM_BLOCK_FOOTPRINT ((size_t)0xDEADD00D)
/** Defines number of skipped function inside the trace */
#define SXM_MEM_TRACE_SKIP  (1)
/** Defines max captured trace depth */
#define SXM_MEM_TRACE_DEPTH (8)
#ifdef WIN32
/** Defines max function name */
#define SXM_MEM_TRACK_STACK_FUNC_NAME_MAX (256)
#endif

/** Calculates aligned size */
#define SXM_MEM_SIZE_ALIGN(_s) \
    ((((_s) + sizeof(size_t) - 1) / sizeof(size_t)) * sizeof(size_t))

/** Calculates and provides access to mem block footprint location
 * \param[in] _p mem block
 * \param[in] _s aligned user's block size
 */
#define SXM_MEM_FOOTPRINT_ADDR(_p, _s) \
    (*((size_t*)((unsigned char*)((SXE_MEM_BLOCK*)(_p) + 1) + (_s))))

/** Memory block descriptor */
typedef struct _sxe_mem_block {
    size_t sig; //!< Simple memory signature
    size_t seq; //!< Sequence number
    size_t size; //!< Block size
    size_t count; //!< Number if allocated blocks of same size
    size_t lives; //!< Number of reallocations (0 means no one)
    char tp; //!< c(alloc), m(alloc)
    const char *fl; //!< Source file
    size_t ln; //!< Line number
    size_t threadId; //!< The thread identifier
    volatile struct _sxe_mem_block **self; //!< Self location within the pool
    SXMTimeSpec ts; //!< Time spec
#if (SXM_DEBUG_MEMORY_TRACKING > 1)
    int nTraceLen; //!< Trace len/depth
#ifdef WIN32 
    PVOID aTrace[SXM_MEM_TRACE_DEPTH];
#elif !defined(ANDROID) && defined(__linux__)
    void *aTrace[SXM_MEM_TRACE_DEPTH + SXM_MEM_TRACE_SKIP];
#endif
#endif
} SXE_MEM_BLOCK;

static volatile size_t gMemMaxSize = 0;
static volatile size_t gMemMaxBlocks = 0;
static volatile size_t gMemTotalSize = 0;
static volatile size_t gMemTotalBlocks = 0;
static volatile size_t gMemNextSeq = 0;
static FILE *gMemFile = NULL;
#if (SXM_DEBUG_MEMORY_TRACKING > 1)
static BOOL gMemTrackerInitialized = FALSE;
#endif

#ifdef SDKFILES_STANDALONE_BUILD
/* In the stand-alone mode the sdkfiles shouldn't be multithreaded */
#define MEM_LOCK()
#define MEM_UNLOCK()
#else
static pthread_mutex_t gMemMutex = PTHREAD_MUTEX_INITIALIZER;
#define MEM_LOCK() pthread_mutex_lock(&gMemMutex)
#define MEM_UNLOCK() pthread_mutex_unlock(&gMemMutex)
#endif
static volatile SXE_MEM_BLOCK* gBlocks[SXM_MEM_BLOCK_TRACK_MAX];
static volatile size_t gBlocksLastEmptyFound = 0;

/***************************************************************************//**
 * Provides Thread ID
 * \param[in] pBlock the allocated memory block
 ******************************************************************************/
static size_t mem_tid(const volatile SXE_MEM_BLOCK *pBlock) {
    (void)(pBlock); /* unused var just in case */
#ifndef SDKFILES_STANDALONE_BUILD
#ifdef WIN32
    return (pBlock == NULL) ? (size_t)GetCurrentThreadId() : pBlock->threadId;
#else
    return (pBlock == NULL) ? (size_t)pthread_self() : pBlock->threadId;
#endif
#else
    return (size_t)0;
#endif
}

/***************************************************************************//**
 * Dump the info about the allocation in short form into the file
 * \note this function must be called under the lock (\ref gMemMutex)
 * \param[in] pBlock the allocated memory block
 ******************************************************************************/
static void mem_file_dump(const volatile SXE_MEM_BLOCK *pBlock, BOOL bIsFree) {
    if (gMemFile != NULL) {
        SXMTimeSpec ts = { 0, 0L };
#ifndef SDKFILES_STANDALONE_BUILD
        if (bIsFree == TRUE) {
            sxm_clock_gettime(&ts);
        }
#else
        (void)bIsFree;
#endif
        fprintf(gMemFile,
            "%u|%6u.%09u|%p|%6u.%09u|%p|%s|%d|%c|%u|%u|%u|%u\n",
            (unsigned int)pBlock->seq,
            (unsigned int)pBlock->ts.tv_sec, (unsigned int)pBlock->ts.tv_nsec, (void*)mem_tid(pBlock),
            (unsigned int)ts.tv_sec, (unsigned int)ts.tv_nsec, (void*)mem_tid(NULL),
            pBlock->fl, (int)pBlock->ln, pBlock->tp,
            (unsigned int)pBlock->lives,
            (unsigned int)pBlock->size, (unsigned int)pBlock->count, (unsigned int)(pBlock->size * pBlock->count)
        );
    }
    return;
}

#if (SXM_DEBUG_MEMORY_TRACKING > 1)
/***************************************************************************//**
 * Performers lazy initialization of the memory tracker
 * \param[in] pBlock the allocated memory block
 ******************************************************************************/
static void mem_track_init(void) {
    if (gMemTrackerInitialized == FALSE) {
#ifdef WIN32
        SymInitialize(GetCurrentProcess(), NULL, TRUE);
#endif
        gMemTrackerInitialized = TRUE;
    }
    return;
}

/***************************************************************************//**
 * Captures the trace at the place of the function call skipping the function
 * itself.
 * \param[in] pBlock the allocated memory block
 ******************************************************************************/
static void mem_trace_capture(volatile SXE_MEM_BLOCK *pBlock) {
#ifdef WIN32
    pBlock->nTraceLen =
        (int)CaptureStackBackTrace(SXM_MEM_TRACE_SKIP, SXM_MEM_TRACE_DEPTH,
                                   (PVOID*)&pBlock->aTrace[0], NULL);
#elif !defined(ANDROID) && defined(__linux__)
    pBlock->nTraceLen = backtrace((void**)&pBlock->aTrace[0],
                                  SXM_MEM_TRACE_SKIP + SXM_MEM_TRACE_DEPTH);
#endif
    return;
}

/***************************************************************************//**
 * Captures the trace at the place of the function call skipping the function
 * itself.
 * \param[in] pBlock the allocated memory block
 ******************************************************************************/
static void mem_trace_dump(FILE *pFile, const volatile SXE_MEM_BLOCK *pBlock) {
#ifdef WIN32
    HANDLE process = GetCurrentProcess();
    SYMBOL_INFO *symbol;
    symbol = (SYMBOL_INFO *)calloc(1, sizeof(*symbol) + SXM_MEM_TRACK_STACK_FUNC_NAME_MAX * sizeof(char));
    if (symbol != NULL) {
        int idx;
        symbol->SizeOfStruct = sizeof(*symbol);
        symbol->MaxNameLen = SXM_MEM_TRACK_STACK_FUNC_NAME_MAX;
        for (idx = 0; idx < pBlock->nTraceLen; ++idx) {
            (void)SymFromAddr(process, (DWORD64)pBlock->aTrace[idx], NULL, symbol);
            fprintf(pFile, "%2d : [%llu] %s\n", idx, symbol->Address, symbol->Name);
        }
        free(symbol);
    }
#elif !defined(ANDROID) && defined(__linux__)
    char **strings = backtrace_symbols((void* const *)&pBlock->aTrace[0], pBlock->nTraceLen);
    if (strings != NULL) {
        int idx;
        for (idx = SXM_MEM_TRACE_SKIP; idx < pBlock->nTraceLen; ++idx) {
            fprintf(pFile, "%2d : [%p] %s\n", idx, pBlock->aTrace[idx], strings[idx]);
        }
        free(strings);
    }
#endif
    return;
}

#else
#define mem_track_init()
#define mem_trace_capture(pBlock)
#define mem_trace_dump(pFilem, pBlock)
#endif

/***************************************************************************//**
 * Inserts the memory block into the pool.
 * \note This function must be called under lock
 *
 * \param[in] size number of bytes to allocate
 * \param[in] fl the source file where function is called
 * \param[in] ln the line where the function is called
 *******************************************************************************/
static void mem_pool_insert(volatile SXE_MEM_BLOCK *pBlock) {
    ++gMemTotalBlocks;
    gMemTotalSize += (pBlock->size * pBlock->count);
    if (gMemMaxBlocks < gMemTotalBlocks) {
        ++gMemMaxBlocks;
    }
    if (gMemMaxSize < gMemTotalSize) {
        gMemMaxSize = gMemTotalSize;
    }
    if (pBlock->self == NULL) {
        size_t idx, itr;
        for (itr = 0, idx = gBlocksLastEmptyFound;
            itr < sizeof(gBlocks) / sizeof(*gBlocks);
            ++itr, idx = ((idx + 1) % SXM_MEM_BLOCK_TRACK_MAX))
        {
            if (gBlocks[idx] == NULL) {
                gBlocks[idx] = pBlock;
                pBlock->self = &gBlocks[idx];
                gBlocksLastEmptyFound = idx;
                break;
            }
        }
    }
    else {
        *pBlock->self = pBlock;
    }
    return;
}

/***************************************************************************//**
 * SXe version of malloc with memory fingerprints
 *
 * \param[in] size number of bytes to allocate
 * \param[in] fl the source file where function is called
 * \param[in] ln the line where the function is called
 *******************************************************************************/
void* sxe_malloc_dbg(size_t size, const char* fl, size_t ln)
{
    const size_t tAligned = SXM_MEM_SIZE_ALIGN(size);
    mem_track_init();
    void *p = malloc(sizeof(SXE_MEM_BLOCK) + tAligned + sizeof(size_t));
    if (p != NULL) {
        volatile SXE_MEM_BLOCK *pBlock = (SXE_MEM_BLOCK *)p;
        pBlock->sig = SXM_MEM_BLOCK_SIG;
        pBlock->seq = gMemNextSeq++;
        pBlock->size = size;
        pBlock->count = 1;
        pBlock->lives = 0;
        pBlock->fl = fl;
        pBlock->ln = ln;
        pBlock->tp = 'm';
        pBlock->threadId = mem_tid(NULL);
        pBlock->self = NULL;
#ifndef SDKFILES_STANDALONE_BUILD
        sxm_clock_gettime((SXMTimeSpec *)&pBlock->ts);
#else
        pBlock->ts.tv_sec = 0;
        pBlock->ts.tv_nsec = 0L;
#endif
        mem_trace_capture(pBlock);
        SXM_MEM_FOOTPRINT_ADDR(p, tAligned) = SXM_MEM_BLOCK_FOOTPRINT;
        MEM_LOCK();
        mem_pool_insert(pBlock);
        MEM_UNLOCK();
    }
    return (p != NULL) ? ((SXE_MEM_BLOCK*)p + 1) : NULL;
}

/***************************************************************************//**
 * SXe version of calloc with memory fingerprints
 *
 * \param[in] count number of elements to allocate
 * \param[in] size size of each element
 * \param[in] fl the source file where function is called
 * \param[in] ln the line where the function is called
 *
 *******************************************************************************/
void* sxe_calloc_dbg(size_t count, size_t size, const char* fl, size_t ln)
{
    const size_t tAligned = SXM_MEM_SIZE_ALIGN(count * size);
    mem_track_init();
    void *p = calloc(1, sizeof(SXE_MEM_BLOCK) + tAligned + sizeof(size_t));
    if (p != NULL) {
        volatile SXE_MEM_BLOCK *pBlock = (SXE_MEM_BLOCK *)p;
        pBlock->sig = SXM_MEM_BLOCK_SIG;
        pBlock->seq = gMemNextSeq++;
        pBlock->size = size;
        pBlock->count = count;
        pBlock->fl = fl;
        pBlock->ln = ln;
        pBlock->tp = 'c';
        pBlock->threadId = mem_tid(NULL);
#ifndef SDKFILES_STANDALONE_BUILD
        sxm_clock_gettime((SXMTimeSpec *)&pBlock->ts);
#endif
        mem_trace_capture(pBlock);
        SXM_MEM_FOOTPRINT_ADDR(p, tAligned) = SXM_MEM_BLOCK_FOOTPRINT;
        MEM_LOCK();
        mem_pool_insert(pBlock);
        MEM_UNLOCK();
    }
    return (p != NULL) ? ((SXE_MEM_BLOCK*)p + 1) : NULL;
}

/***************************************************************************//**
 * SXe version of realloc with memory fingerprints
 *
 * \param[in] p Pointer to a memory block previously allocated with sxe_malloc, sxe_calloc or sxe_realloc.
 * \param[in] size New size for the memory block, in bytes.
 * \param[in] fl the source file where function is called
 * \param[in] ln the line where the function is called
 *
 *******************************************************************************/
void* sxe_realloc_dbg(void *p, size_t size, const char* fl, size_t ln)
{
    volatile SXE_MEM_BLOCK *pb = (p != NULL) ? ((SXE_MEM_BLOCK*)p - 1) : NULL;
    const size_t tAligned = SXM_MEM_SIZE_ALIGN(size);
    void *pOut = realloc((void*)pb, sizeof(SXE_MEM_BLOCK) + tAligned + sizeof(size_t));
    if (pOut != NULL) {
        volatile SXE_MEM_BLOCK *pBlock = (SXE_MEM_BLOCK *)pOut;
        /* put info about re-allocated block first */
        if (p != NULL) {
            MEM_LOCK();
            --gMemTotalBlocks;
            gMemTotalSize -= pBlock->count * pBlock->size;
            mem_file_dump(pBlock, TRUE);
            MEM_UNLOCK();
            ++pBlock->lives;
        }
        else {
            pBlock->lives = 0;
        }
        pBlock->sig = SXM_MEM_BLOCK_SIG;
        pBlock->seq = gMemNextSeq++;
        pBlock->size = size;
        pBlock->count = 1U;
        pBlock->fl = fl;
        pBlock->ln = ln;
        pBlock->tp = 'r';
        pBlock->threadId = mem_tid(NULL);
#ifndef SDKFILES_STANDALONE_BUILD
        sxm_clock_gettime((SXMTimeSpec *)&pBlock->ts);
#else
        pBlock->ts.tv_sec = 0;
        pBlock->ts.tv_nsec = 0L;
#endif
        mem_trace_capture(pBlock);
        SXM_MEM_FOOTPRINT_ADDR(pOut, tAligned) = SXM_MEM_BLOCK_FOOTPRINT;
        MEM_LOCK();
        mem_pool_insert(pBlock);
        MEM_UNLOCK();
    }
    return (pOut != NULL) ? ((SXE_MEM_BLOCK*)pOut + 1) : NULL;
}

/***************************************************************************//**
 * SXe version of free with memory fingerprints
 *
 *  \param[in] p address of the memory to be released
 *
 *******************************************************************************/
void sxe_free_dbg(void *p)
{
    if (p != NULL) {
        volatile SXE_MEM_BLOCK *pBlock = (SXE_MEM_BLOCK*)p - 1;
        if (pBlock->sig == SXM_MEM_BLOCK_SIG) {
            const size_t totalSize = pBlock->size * pBlock->count;
            p = (SXE_MEM_BLOCK*)p - 1; /* re-arrange the pointer */
            /* Check buffer overrun */
            if (SXM_MEM_FOOTPRINT_ADDR(pBlock, SXM_MEM_SIZE_ALIGN(totalSize)) != SXM_MEM_BLOCK_FOOTPRINT) {
                fprintf(stderr, "[MEM FAILURE] Memory overflow detected for %p [%s:%d]",
                    p, pBlock->fl, (unsigned int)pBlock->ln);
            }
            /* Clean up signatures */
            pBlock->sig = ~SXM_MEM_BLOCK_SIG;
            SXM_MEM_FOOTPRINT_ADDR(pBlock, SXM_MEM_SIZE_ALIGN(totalSize)) = ~SXM_MEM_BLOCK_FOOTPRINT;
            MEM_LOCK();
            mem_file_dump(pBlock, TRUE);

            --gMemTotalBlocks;
            gMemTotalSize -= totalSize;
            if (pBlock->self != NULL) {
                /* Looks like the block is inside the pool, check it */
                if ((*pBlock->self) == pBlock) {
                    /* The pool item is fine, clean the reference */
                    *pBlock->self = NULL;
                    /* Keep just released block as the next free one */
                    gBlocksLastEmptyFound = (size_t)(pBlock->self - &gBlocks[0]);
                }
                else {
                    fprintf(stderr, "[MEM FAILURE] Incorrect pool link for %p", p);
                }
            }
            MEM_UNLOCK();
        }
        else {
            fprintf(stderr, "[MEM FAILURE] Incorrect deallocation for %p", p);
        }
    }
    free(p);
    return;
}

/***************************************************************************//**
 * Provides memory statistics
 *
 * \param[out] totalBlocks total number of allocated blocks for now
 * \param[out] totalSize total number of allocated bytes for now
 * \param[out] maxBlocks max number of allocated blocks for now
 * \param[out] maxSize max number of allocated bytes for now
 *
 *******************************************************************************/
SXESDK_API void sxe_mem_stat(size_t *totalBlocks, size_t *totalSize,
                             size_t *maxBlocks, size_t *maxSize)
{
    size_t idx;
    MEM_LOCK();
    if (totalBlocks != NULL) {
        *totalBlocks = gMemTotalBlocks;
    }
    if (totalSize != NULL) {
        *totalSize = gMemTotalSize;
    }
    if (maxBlocks != NULL) {
        *maxBlocks = gMemMaxBlocks;
    }
    if (maxSize != NULL) {
        *maxSize = gMemMaxSize;
    }

    for (idx = 0; idx < (sizeof(gBlocks) / sizeof(*gBlocks)); ++idx) {
        if (gBlocks[idx] != NULL) {
            const volatile SXE_MEM_BLOCK *pBlk = gBlocks[idx];
            fprintf(stdout, "[%p] %6u: %65s:%06u, %c, %u x %u = %u\n",
                (void*)mem_tid(pBlk), (unsigned int)pBlk->seq, pBlk->fl,
                (unsigned int)pBlk->ln, pBlk->tp,
                (unsigned int)pBlk->size, (unsigned int)pBlk->count,
                (unsigned int)(pBlk->size * pBlk->count));
            mem_trace_dump(stdout, pBlk);
        }
    } 
    MEM_UNLOCK();
}

/***************************************************************************//**
 * Triggers file logging for memory allocation statistics
 *
 * \param[in] pFileName file path. Provide NULL to stop logging
 *
 *******************************************************************************/
SXESDK_API void sxe_mem_file(const char *pFileName) {
#if SXM_DEBUG_MEMORY_TRACKING
    MEM_LOCK();
    if ((gMemFile == NULL) && (pFileName != NULL)) {
        /* start file logging */
        gMemFile = fopen(pFileName, "wt");
        if (gMemFile == NULL) {
            fprintf(stderr, "failed to open file %s, errno=%d", pFileName, errno);
        }
        else {
            /* Put the file header for further log processing */
            fputs("ID|START|STID|END|ETID|FILE|LINE|TYPE|LIVES|SIZE|COUNT|TOTAL\n", gMemFile);
        }
    }
    else if ((gMemFile != NULL) && (pFileName == NULL)) {
        size_t idx;
        /* lets print out currently existing blocks, which are not released yet */
        for (idx = 0; idx < (sizeof(gBlocks) / sizeof(*gBlocks)); ++idx) {
            if (gBlocks[idx] != NULL) {
                const volatile SXE_MEM_BLOCK *pBlk = gBlocks[idx];
                mem_file_dump(pBlk, FALSE);
            }
        }

        /* stop file logging */
        fclose(gMemFile);
        gMemFile = NULL;
    }
    else {
        /* nothing to do here */
    }
    MEM_UNLOCK();
#endif
    return;
}
#endif

/***************************************************************************//**
 * Sleep for the specified number of seconds.
 *
 *  \param [in] sec number of seconds to sleep the caller's thread
 *
 *******************************************************************************/
SXESDK_API void sxe_sleep(uint sec)
{
#ifdef WIN32
    Sleep((DWORD)(sec * 1000U));
#else
    sleep(sec);
#endif
}

/***************************************************************************//**
 * Sleep for the specified number of nano-seconds.
 *
 *  \param [in] nsec number of nano-seconds to sleep the caller's thread
 *
 *******************************************************************************/
SXESDK_API void sxe_usleep(uint nsec)
{
#ifdef WIN32
    Sleep((DWORD)(nsec / 1000));
#else
    usleep((useconds_t)nsec);
#endif
}

#ifndef SDKFILES_STANDALONE_BUILD
/***************************************************************************//**
 * Sets name for the pthread.
 *
 * \param[in] thread valid pthread handle
 * \param[in] name null-terminated string to be used as a thread's name
 *
 *******************************************************************************/
SXESDK_API void sxe_thread_setname(pthread_t thread, const char *name)
{
#if defined(WIN32)
#if defined(_MSC_VER) && (_MSC_VER != 1200)
    #define WIN32_THREAD_NAME_LEN_MAX (16)
    #define WIN32_MS_VC_EXCEPTION (0x406D1388)

    #pragma pack(push, 8)
    typedef struct tagTHREADNAME_INFO
    {
        DWORD dwType; // must be 0x1000
        LPCSTR szName; // pointer to name (in user addr space)
        DWORD dwThreadID; // thread ID (-1=caller thread)
        DWORD dwFlags; // reserved for future use, must be zero
    } THREADNAME_INFO;
    #pragma pack(pop)

    char thrName[WIN32_THREAD_NAME_LEN_MAX + 1]; 
    THREADNAME_INFO info;

    memset(&thrName[0], 0, sizeof(thrName));
    strncpy(&thrName[0], name, WIN32_THREAD_NAME_LEN_MAX);

    info.dwType = 0x1000;
    info.szName = thrName;
    info.dwThreadID = pthread_getw32threadid_np(thread);
    info.dwFlags = 0;

    __try
    {
        RaiseException(WIN32_MS_VC_EXCEPTION, 0,
            sizeof(info)/sizeof(ULONG_PTR), (ULONG_PTR*)&info );
    }
    __except(EXCEPTION_EXECUTE_HANDLER)
    { }
#endif
#elif !(defined(HAVE_NO_PTHREAD_SETNAME))
    pthread_setname_np(thread, name);
#endif
}

/***************************************************************************//**
 * Creates new thread.
 *
 * \param[out] tid thread handle for newly created thread in case of success
 * \param[in] attr the set of thread attributes, can be NULL to use default ones
 * \param[in] start the thread routine
 * \param[in] arg thread's routine argument
 * 
 * \return 0 in case of success or error number in case of failure
 *
 *******************************************************************************/
SXESDK_API int sxm_pthread_create (pthread_t *tid,
    const pthread_attr_t *attr,
    void *(*start) (void *),
    void *arg)
{
    int ret;
    pthread_attr_t pt_attr, *ppt_attr;

    if (attr == NULL)
    {
        // Initialize thread creation attributes, if caller 
        // did not provide one
        ret = pthread_attr_init(&pt_attr);
#if SXM_STACK_SIZE_DEFAULT
        if (ret == 0)
        {
            // If defined stack size is not 0 set default stack size for new thread
            ret = pthread_attr_setstacksize(&pt_attr, SXM_STACK_SIZE_DEFAULT);
        }
#endif

        if (ret != 0)
        {
            // Error
            return ret;
        }

        // Use our attributes
        ppt_attr = &pt_attr;
    }
    else
    {
        // Use provided attributes
        ppt_attr = (pthread_attr_t*)attr;
    }
    
    // Create the thread
    ret = pthread_create(tid, ppt_attr, start, arg);

    if (attr == NULL)
    {
        // Destroy thread creation attributes
        pthread_attr_destroy(&pt_attr);
    }

    return ret;
}

/*******************************************************************************
 *
 * SEMAPHORES
 * 
 * This section provides porting layer for semaphores.
 * 
 * The #SXM_USE_SEM_MONOTONIC allows to use monotonic semaphore in all platform
 * where it's possible.
 * 
 *******************************************************************************/
#ifndef SXM_USE_SEM_MONOTONIC
/** This definition turns on pthread-cond based semaphore implementation
 * which tries to use MONOTONIC clock where it's possible. Unfortunately, some
 * ports may not have it and the REALTIME clock will be used instead.
 */
#define SXM_USE_SEM_MONOTONIC (1)
#endif

/** Platform specific definition for \ref SXMSem_t */
struct sxm_sem_t_ {
#if !SXM_USE_SEM_MONOTONIC
    sem_t sem; //!< POSIX semaphore
#else
    pthread_mutex_t mutex; //!< Controlling mutex
    pthread_condattr_t cond_attr; //!< Condition variable attributes
    pthread_cond_t cond; //!< Condition variable
    unsigned int value; //!< Current value
    unsigned int waiters; //!< Number of waiters
#endif
};

/***************************************************************************//**
* This function returns timeout.
*
* \param[out] pTO pointer to timeout structure which will get populated.
* \param[in] timeoutMillisec desired timeout in milliseconds
* 
* \return 0 in case of success, -1 in case of error
*
*******************************************************************************/
static int get_timeout_ms(struct timespec *pTO, uint timeoutMillisec)
{
    int result;
    struct timespec timeo;
#if defined(WIN32)
    // Windows PTHREAD implementation has no support of monotonic cond-wait
    // thus, the code should use real time clock
 
    struct _timeb currSysTime;
    //Uses _ftime_s() function because of pthread-win32 lib also used it
    //in realisation of sem_timedwait()/ptw32_relmillisecs() function
#if defined(__MINGW32__) || defined(__MINGW64__)
    _ftime( &currSysTime );
    result = (errno == 0) ? 0 : -1;
#else
    result = _ftime_s( &currSysTime );
#endif
    if (result == 0) {
        timeo.tv_sec = currSysTime.time + (timeoutMillisec / SXM_MILLISEC_PER_SEC);
        timeo.tv_nsec = (currSysTime.millitm + timeoutMillisec % SXM_MILLISEC_PER_SEC) * SXM_NANOSEC_PER_MILLISEC;
    }
    else {
        memset(&timeo, 0, sizeof(timeo));
    }

#else  /* ! WIN32 start */

#if SXM_USE_SEM_MONOTONIC
    result = clock_gettime(CLOCK_MONOTONIC, &timeo);
#else
    result = clock_gettime(CLOCK_REALTIME, &timeo);
#endif
    if (result == 0) {
        timeo.tv_sec += (time_t)(timeoutMillisec / SXM_MILLISEC_PER_SEC);
        timeo.tv_nsec += (long)((timeoutMillisec % SXM_MILLISEC_PER_SEC) * SXM_NANOSEC_PER_MILLISEC);
    }
    else {
        memset(&timeo, 0, sizeof(timeo));
    }

#endif /* ! WIN32 end */

    if (timeo.tv_nsec >= SXM_NANOSEC_PER_SEC) {
        timeo.tv_sec += timeo.tv_nsec / SXM_NANOSEC_PER_SEC;
        timeo.tv_nsec = timeo.tv_nsec % (long)SXM_NANOSEC_PER_SEC;
    }

    *pTO = timeo;
    return result;
}

/***************************************************************************//**
* This function returns current time for time difference calculations only
*
* \param[out] pTime pointer to timeout structure which will get populated.
*
*******************************************************************************/
void sxm_clock_gettime(SXMTimeSpec *pTime)
{
    if (pTime) {
#if defined(WIN32)
        BOOL bCollected = FALSE;
        LARGE_INTEGER liFrequency;
        if (QueryPerformanceFrequency(&liFrequency) == TRUE) {
            LARGE_INTEGER liCount;
            if ((liFrequency.QuadPart > 0) && QueryPerformanceCounter(&liCount) == TRUE) {
                pTime->tv_sec = liCount.QuadPart / liFrequency.QuadPart;
                // Calculate the nanoseconds
                pTime->tv_nsec =
                    (long)(((liCount.QuadPart % liFrequency.QuadPart) * SXM_MICROSEC_PER_SEC) / liFrequency.QuadPart) * SXM_NANOSEC_PER_MICROSEC;
                bCollected = TRUE;
            }
        }
        // If the high-performance timer is not working let's use the 
        // old fashion way via ticks counter
        if (bCollected == FALSE) {
#if defined(__MINGW32__) || defined(__MINGW64__)
            DWORD ticks;
            ticks = GetTickCount();
#else
            ULONGLONG ticks;
            ticks = GetTickCount64();
#endif
            pTime->tv_sec = ticks / SXM_MILLISEC_PER_SEC;
            pTime->tv_nsec = (ticks % SXM_MILLISEC_PER_SEC) * SXM_NANOSEC_PER_MILLISEC;
        }
#else
        struct timespec ts;
        int rc;
        rc = clock_gettime(CLOCK_MONOTONIC, &ts);
        if (rc == 0) {
            pTime->tv_sec = ts.tv_sec;
            pTime->tv_nsec = ts.tv_nsec;
        }
        else {
            pTime->tv_sec = 0;
            pTime->tv_nsec = 0;
        }
#endif
    }
 }

/***************************************************************************//**
 * Creates new semaphore.
 *
 * \param[out] pSem semaphore handle
 * \param[in] initValue semaphore init value
 * 
 * \retval SXM_E_OK Success
 * \retval SXM_E_FAULT NULL Parameter
 * \retval SXM_E_NOMEM failed to allocate memory for internal needs
 * \retval SXM_E_RESOURCE failed to initialize semaphore due to resources
 *
 *******************************************************************************/
int sxm_sem_init(SXMSem_t *pSem, unsigned int initValue)
{
    SXMResultCode rc = SXM_E_RESOURCE;
    if (pSem == NULL) {
        rc = SXM_E_FAULT;
    }
    else {
        struct sxm_sem_t_ *pRet =
#if SXM_DEBUG_MEMORY_TRACKING
            (struct sxm_sem_t_ *)sxe_calloc_dbg(1, sizeof(struct sxm_sem_t_), __FILE__, __LINE__);
#else
            (struct sxm_sem_t_ *)calloc(1, sizeof(struct sxm_sem_t_));
#endif
        if (pRet == NULL) {
            rc = SXM_E_NOMEM;
        }
        else {
#if !SXM_USE_SEM_MONOTONIC
            if (sem_init(&pRet->sem, 0, initValue) == 0) {
                rc = SXM_E_OK; 
            }
#else
            pthread_mutexattr_t tMutexAttr;
            int status;

            pRet->value = initValue;
            pRet->waiters = 0U;

            // Initialize the mutex attr
            pthread_mutexattr_init(&tMutexAttr);

            // Set the type explicitly to NORMAL since the DEFAULT behavior
            // is undefined.
            pthread_mutexattr_settype(&tMutexAttr, PTHREAD_MUTEX_NORMAL);
            // Initialize mutex used for condition variable.
            status = pthread_mutex_init(&pRet->mutex, &tMutexAttr);
            pthread_mutexattr_destroy(&tMutexAttr);

            if (status == 0) {
                pthread_condattr_init(&pRet->cond_attr);
#if !defined(WIN32) && !(defined(__ANDROID_API__) && __ANDROID_API__ < 21)
                // Set the clock attribute of a condition-variable attribute object.
                status = pthread_condattr_setclock(&pRet->cond_attr, CLOCK_MONOTONIC);
                if (status != 0) {
                    pthread_condattr_destroy(&pRet->cond_attr);
                    pthread_mutex_destroy(&pRet->mutex);
                }
                else
#endif
                {
                    // Initialize condition variable
                    status = pthread_cond_init(&pRet->cond, &pRet->cond_attr);
                    if (status == 0) {
                        rc = SXM_E_OK;
                    }
                    else {
                        pthread_condattr_destroy(&pRet->cond_attr);
                        pthread_mutex_destroy(&pRet->mutex);
                    }
                }
            }
#endif
            if (rc != SXM_E_OK) {
#if SXM_DEBUG_MEMORY_TRACKING
                sxe_free_dbg(pRet);
#else
                free(pRet);
#endif
            }
            else {
                *pSem = pRet;
            }
        }
    }
    return rc;
}

/***************************************************************************//**
 * Destroyers semaphore.
 * 
 * \warning The caller must be sure that the function is called while no other
 *          thread is pending on that semaphore. In case of calling it for the
 *          used semaphore the behavior is undefined and highly depends on the
 *          how the system handles this situation. This restriction came from the
 *          POXIS where destruction of semaphore causes undefined behavior.
 *
 * \param[in] pSem semaphore instance
 * 
 * \retval SXM_E_OK Success
 * \retval SXM_E_FAULT NULL argument
 * \retval SXM_E_ERROR General error has occurred during destruction
 *
 *******************************************************************************/
int sxm_sem_destroy(SXMSem_t *pSem)
{
    SXMResultCode rc = SXM_E_ERROR;
    if (pSem != NULL) {
        SXMSem_t sem = *pSem;
#if !SXM_USE_SEM_MONOTONIC
        if (sem_destroy(&sem->sem) == 0) {
            rc = SXM_E_OK;
        }
#else
        if (pthread_cond_destroy(&sem->cond) == 0) {
            // Destroy a condition-variable attribute object
            if (pthread_condattr_destroy(&sem->cond_attr) == 0) {
                if (pthread_mutex_destroy(&sem->mutex) == 0) {
                    rc = SXM_E_OK;
                }
            }
        }
#endif
#if SXM_DEBUG_MEMORY_TRACKING
        sxe_free_dbg(*pSem);
#else
        free(*pSem);
#endif
    }
    else {
        rc = SXM_E_FAULT;
    }
    return rc;
}

/***************************************************************************//**
* Increases semaphore counter.
*
* \param[in] pSem semaphore instance
* 
* \retval SXM_E_OK Semaphore counter increased
* \retval SXM_E_FAULT NULL argument
* \retval SXM_E_ERROR Failed to post
*
*******************************************************************************/
int sxm_sem_post(SXMSem_t *pSem)
{
    SXMResultCode rc = SXM_E_ERROR;
    if (pSem != NULL) {
        SXMSem_t sem = *pSem;
#if !SXM_USE_SEM_MONOTONIC
        if (sem_post(&sem->sem) == 0) {
            rc = SXM_E_OK;
        }
#else
        int status;
        // The calling thread must lock the mutex before modifying the data
        // it protects (the predicate). So as you would expect we must
        // take the mutex before we can modify the resources available.
        status = pthread_mutex_lock(&sem->mutex);
        if (status == 0) {
            // Increment resources available
            ++sem->value;
            // We need to signal any waiting threads. Don't rely on 0 resources
            // available condition to do signaling, as the scheduler could allow
            // threads calling post() to run several times before the waiting
            // threads are scheduled to run.
            if (sem->waiters > 0U) {
                // The pthread_cond_signal() function unblocks the highest
                // priority thread that's waiting on the condition variable.
                // If more than one thread at the highest priority is
                // waiting, pthread_cond_signal() unblocks the one that
                // has been waiting the longest.
                status = pthread_cond_signal(&sem->cond);
                if (status == 0) {
                    rc = SXM_E_OK;
                }
            }
            else {
                // No waiters, so, let's consider this as OK
                rc = SXM_E_OK;
            }
            pthread_mutex_unlock(&sem->mutex);
        }
#endif
    }
    else {
        rc = SXM_E_FAULT;
    }
    return rc;
}

/***************************************************************************//**
* Decreases semaphore counter if it's allowed or wait until it's available
* without any timeout.
*
* \param[in] pSem semaphore instance
* 
* \retval SXM_E_OK Semaphore acquired
* \retval SXM_E_FAULT NULL argument
* \retval SXM_E_ERROR Failed to wait by some internal reason
*
*******************************************************************************/
int sxm_sem_wait(SXMSem_t *pSem)
{
    SXMResultCode rc = SXM_E_ERROR;
    if (pSem != NULL) {
        SXMSem_t sem = *pSem;
#if !SXM_USE_SEM_MONOTONIC
        int ret;
        /* Acquire  semaphore. Please note, the ETIMEOUT can acquire in some
         * cases even on this type of waiter. For fair implementation
         * this will not be a problem.
         */
        do {
            ret = sem_wait(&sem->sem);
        } while ((ret == -1) && (errno == ETIMEDOUT));
        rc = (ret == 0) ? SXM_E_OK : SXM_E_ERROR;
#else
        int status;
        /* The calling thread must lock the mutex before modifying the data
         * it protects (the predicate). So as you would expect we must
         * take the mutex before we can modify the resources available.
         */
        status = pthread_mutex_lock(&sem->mutex);
        if (status == 0) {
            if (sem->value > 0U) {
                --sem->value;
                rc = SXM_E_OK;
            }
            else {
                /* Resource is not available, and we want to wait for it. */
                ++sem->waiters;
                /* Wait for resource forever... In accordance with recommendations
                 * the pthread_cond_wait() may exit has not been signaled, thus,
                 * the caller should check that the expected action has happened.
                 * Thus, in our case the expectation is that the value should
                 * be greater than 0, othwerwise, the wait() routine should be
                 * re-called.
                 */
                do {
                    status = pthread_cond_wait(&sem->cond, &sem->mutex);
                } while ((status == 0) && (sem->value == 0U));
                --sem->waiters;
                if (status == 0) {
                    if (sem->value > 0U) {
                        --sem->value;
                    }
                    rc = SXM_E_OK;
                }
            }
            pthread_mutex_unlock(&sem->mutex);
        }
#endif
    }
    else {
        rc = SXM_E_FAULT;
    }
    return rc;
}

/***************************************************************************//**
* Tries to decrease semaphore counter
*
* \param[in] pSem semaphore instance
* 
* \retval SXM_E_OK Semaphore acquired
* \retval SXM_E_FAULT NULL argument
* \retval SXM_E_TIMEOUT Indicates that the caller hasn't acquired the semaphore, try again
* \retval SXM_E_ERROR Failed to wait by some internal reason
*
*******************************************************************************/
int sxm_sem_trywait(SXMSem_t *pSem)
{
    SXMResultCode rc = SXM_E_ERROR;
    if (pSem != NULL) {
        int status;
        SXMSem_t sem = *pSem;
#if !SXM_USE_SEM_MONOTONIC
        status = sem_trywait(&sem->sem);
        if (status == 0) {
            rc = SXM_E_OK;
        }
        else if (errno == EAGAIN) {
            rc = SXM_E_TIMEOUT;
        }
#else
        // The calling thread must lock the mutex before modifying the data
        // it protects (the predicate). So as you would expect we must
        // take the mutex before we can modify the resources available.
        status = pthread_mutex_lock(&sem->mutex);
        if (status == 0) {
            if (sem->value > 0U) {
                --sem->value;
                rc = SXM_E_OK;
            }
            else {
                rc = SXM_E_TIMEOUT;
            }
            pthread_mutex_unlock(&sem->mutex);
        }
#endif
    }
    else {
        rc = SXM_E_FAULT;
    }
    return rc;
}

/***************************************************************************//**
* Decreases semaphore counter if it's allowed or wait until it's available
* during \p msec milliseconds
*
* \param[in] pSem semaphore instance
* \param[in] msec timeout in milliseconds
* 
* \retval SXM_E_OK Semaphore acquired
* \retval SXM_E_FAULT NULL argument
* \retval SXM_E_TIMEOUT Indicates that the caller hasn't acquired in time
* \retval SXM_E_ERROR Failed to wait by some internal reason
*
*******************************************************************************/
int sxm_sem_timedwait(SXMSem_t *pSem, unsigned int msec)
{
    SXMResultCode rc = SXM_E_ERROR;
    if (pSem == NULL) {
        rc = SXM_E_FAULT;
    }
    else {
        SXMSem_t sem = *pSem;
#if !SXM_USE_SEM_MONOTONIC
        struct timespec timeOut;
        int status;
        status = get_timeout_ms(&timeOut, msec);
        if (status == 0) {
            if (sem_timedwait(&sem->sem, &timeOut) == 0) {
                rc = SXM_E_OK;
            }
            else if (errno == ETIMEDOUT) {
                rc = SXM_E_TIMEOUT;
            }
        }
#else
        int status;
        /* The calling thread must lock the mutex before modifying the data
         * it protects (the predicate). So as you would expect we must
         * take the mutex before we can modify the resources available.
         */
        status = pthread_mutex_lock(&sem->mutex);
        if (status == 0) {
            if (sem->value > 0U) {
                --sem->value;
                rc = SXM_E_OK;
            }
            else {
                struct timespec timeout;
                status = get_timeout_ms(&timeout, msec);
                /* Resource is not available, and we want to wait for it. */
                ++sem->waiters;
                if (status == 0) {
                    do {
                        /* Wait for a specific timeout. Unlock mutex first...
                         * In accordance with recommendations the pthread_cond_timedwait()
                         * may exit has not been signaled, thus, the caller should
                         * check that the expected action has happened. Thus, in
                         * our case the expectation is that the value should be
                         * greater than 0, othwerwise, the wait() routine shold
                         * be re-called. No need to recalculate timeout since
                         * the timed wait works with absolute expiration moment.
                         */
#if (defined(__ANDROID_API__) && __ANDROID_API__ < 21)
                        status = pthread_cond_timedwait_monotonic_np(&sem->cond, &sem->mutex, &timeout);
#else
                        status = pthread_cond_timedwait(&sem->cond, &sem->mutex, &timeout);
#endif
                    } while ((status == 0) && (sem->value == 0U));
                }
                --sem->waiters;
                if ((status == 0) || (status == ETIMEDOUT)) {
                    if (sem->value > 0U) {
                        --sem->value;
                        rc = SXM_E_OK;
                    }
                    else {
                        rc = SXM_E_TIMEOUT;
                    }
                }
            }
            pthread_mutex_unlock(&sem->mutex);
        }
#endif
    }

    return rc;
}

#endif /* SDKFILES_STANDALONE_BUILD */
