/******************************************************************************/
/*                Copyright (c) Sirius XM Satellite Radio, Inc.               */
/*                            All Rights Reserved                             */
/*      Licensed Materials - Property of Sirius XM Satellite Radio, Inc.      */
/******************************************************************************/
/***************************************************************************//**
 * 
 * \file sxm_list.h
 * \author Alexander Pylchagin
 * \date 5/19/2014
 * \brief Internal Header file to Double Linked List implementation.
 *
 ******************************************************************************/

#ifndef SXM_LIST_H_
#define SXM_LIST_H_

#include "sxm_build.h"
#include "sxm_typ.h"
#include <stdlib.h>

#ifdef __cplusplus
extern "C" {
#endif

/** Linked list entry descriptor */
typedef struct _sxm_list_entry
{
    struct _sxm_list_entry *next; //!< Reference to the next entry
    struct _sxm_list_entry *prev; //!< Reference to the previous entry
    void *data; //!< Entry data
} SXMListEntry;

/** Linked List entity */
typedef struct
{
    SXMListEntry *head; //!< Reference to the list head
    SXMListEntry *tail; //!< Reference to the list tail
    size_t size; //!< Number of entries kept by the list
    uint flags; //!< Flags to define list behavior
} SXMList;

/** 
 * \name Creation flags
 * @{
 */
/** No special options (default) */
#define SXM_LIST_NONE         ((uint)0x0000)
/** The list entry is a part of the entries data */
#define SXM_LIST_PREALLOCATED ((uint)0x0001)
/** @} */

/**
 * \brief Defines list iteration callback
 * \param[in] list the reference to the list which is being iterated
 * \param[in] entry the reference to the entry which the callback is called for
 * \param[in] arg the caller argument which is passed during start of the iteration
 *                process.
 * \retval not-0 value to move to the next entry,
 * \retval 0 to interrupt iterations.
 */
typedef int (*SXMListCallack)(SXMList *list, SXMListEntry *entry, ptr arg);

/**
 * \brief Defines entries sorting callback
 * \param[in] data1 the first entry data
 * \param[in] data2 the second entry data
 * \param[in] arg the caller's argument which is passed via sort API
 * \return value > 0 if data1 appears to be greater than data2,
 *         value < 0 if data2 appears to be less than data2,
 *         value == 0 if data1 appears to be equal to data2
 */
typedef int (*SXMListSortCallack)(void *data1, void *data2, void *arg);

/**
 * \brief Get the size of the list in entries
 * \param[in] _list the valid SXMList reference
 * \return size of the list in entries
 */
#define sxm_list_size(_list) ((_list)->size)

/**
 * \brief Get the first entry of the list
 * \param[in] _list the valid SXMList reference
 * \return valid the first entry address or \c NULL if list is empty
 */
#define sxm_list_first(_list) ((_list)->head)

/**
 * \brief Get the last entry of the list
 * \param[in] _list the valid SXMList reference
 * \return valid the last entry address or \c NULL if list is empty
 */
#define sxm_list_last(_list) ((_list)->tail)

/**
 * \brief Get the entry's data reference
 * \param[in] _entry the valid SXMListEntry reference
 * \return Real entry data as \b void*
 */
#define sxm_list_data(_entry) ((_entry)->data)

/**
 * \brief Get the next entry
 * \param[in] _entry the valid SXMListEntry reference
 * \return valid reference to the next entry or
 *         \c NULL if the passed entry is the last one.
 */
#define sxm_list_next(_entry) ((_entry)->next)

/**
 * \brief Get the previous entry
 * \param[in] _entry the valid SXMListEntry reference
 * \return valid reference to the next entry or
 *         \c NULL if the passed entry is the first one.
 */
#define sxm_list_prev(_entry) ((_entry)->prev)


extern SXESDK_INTERNAL_API int sxm_list_create(SXMList *, uint);

extern SXESDK_INTERNAL_API void sxm_list_destroy(SXMList *);

extern SXESDK_INTERNAL_API int sxm_list_add(SXMList *, void *, BOOL,
                                            SXMListEntry **);

extern SXESDK_INTERNAL_API int sxm_list_remove(SXMList *, SXMListEntry *);

extern SXESDK_INTERNAL_API int sxm_list_iterate(SXMList *,
                                                SXMListCallack,
                                                void *);

extern SXESDK_INTERNAL_API int sxm_list_sort(SXMList *,
                                             SXMListSortCallack ,
                                             void *);

extern SXESDK_INTERNAL_API int sxm_list_remove_all(SXMList *);

extern SXESDK_INTERNAL_API void* sxm_list_allocate(size_t);

extern SXESDK_INTERNAL_API void sxm_list_free(void *);

extern SXESDK_INTERNAL_API SXMListEntry* sxm_list_entry(void *);

#ifdef __cplusplus
}
#endif

#endif /* SXM_LIST_H_ */
