/******************************************************************************/
/*                Copyright (c) Sirius XM Satellite Radio, Inc.               */
/*                            All Rights Reserved                             */
/*      Licensed Materials - Property of Sirius XM Satellite Radio, Inc.      */
/******************************************************************************/
/***************************************************************************//**
*
* \file sxm_iomanager.c
* \author Alexander Pylchagin
* \date 8/08/2017
* \brief Implementation of SXe I/O Management.
*
* \page ioman_page I/O Manager
*
* \section ioman_intro Introduction
* The I/O Manager component allows to balance I/O operations to persist
* operational data across registered components.
*
* \section ioman_design Design Overview
* This section explains some design and implementation aspects.
*
* \subsection ioman_design_const Constrain and Approaches
*
* \subsubsection ioman_design_app Application Interaction
* -# The application configures the manager via SXe config API (\ref sxm_sdk_set_config)
*    in order to set three parameters: cadence, quota and cycle quota.
* -# In order to simplify public interface the application working with set of
*    predefined profiles. Profiles can be timed-based or specific like power-down
*    and immediate. The Power-Down one changes nothing in the legacy operational
*    data persistence logic, i.e. the data will be save on stop if the application
*    doesn't remove save-on-stop flag.
* -# The configuration accepted by the manager while it's is initializing which
*    is happening only once; during the very first registration of the manager's
*    client.
* -# The application should have ability to pause and resume processing at any
*    moment. However, neither of ongoing I/O operation cannot be interrupted
*    which means if the service is granted to perform I/O operations it shall
*    finish them gracefully.
* -# The application should have ability to force saving for all registered
*    component regardless any quotas utilization at the moment.
*
* \subsubsection ioman_design_regist Registrants
* -# Each component provides own function which does I/O operations and returns
*    amount of data in sectors (\ref SXMSector) which was written to the file.
* -# The provided callback is being called from the I/O Manager thread, thus,
*    any thread safety issues should be covered by the component, which gives the
*    function.
* -# Since registered callbacks are being called from the I/O Manager thread
*    the only one component can do save operation in the same time. In order to
*    balance usage of persistence storage the round-robin model is used. In other
*    words, if the component does the save operation and consumed all cadence
*    quote the next by order component will be granted once the cadence gives 
*    quota.
* 
* \subsubsection ioman_design_intrs Internals
* -# The manager processes all request from it's own thread. The thread is the
*    only one who done state machine processing.
* -# Since the list of registrants can be pretty long the thread doesn't process
*    all of them back to back. Instead, once at least on registrant, which does
*    real I/O writes, is faced the manager send event to itself to continue
*    the process later starting from the registrant next to the just executed
*    (Round Robin).
* -# Each saving cycle grants only one attempt to each registrant in order to
*    prevent endless saving loops if the registrant's logic writes something
*    per each request.
* -# Forced-Saving has higher priority over the regular saving;
* -# The once the pause command is received no other command except resume can
*    wake the manager up. Even Forced-Save cannot wake up the manager since
*    the assumption is that if the application has decided to pause the manager
*    it definitely doesn't want to bother with I/O lags.
* -# In the case of Power-Down profile the service acquired a minimum of needed
*    resources in order to provide usage transparency between registrants and
*    the manager.
*
* \subsection ioman_design_state State Machine
* This section gives overview of the manager's states and transitions between them
* - \b Idle  - in this state the manager is waiting for external disturbance to start
*          processing
* - \b Saving - The process while the manager posts events to itself in order to
*          grand time for each component back-to-back. The process is stopped
*          once no one among registered component saves any data or quotas
*          have ran out.
* - \b Unlimited-Save - The process similar to the \b Saving, but in this mode the 
*          manager has no respect to any of quotas and grands I/O time to each
*          registrant until all of them reports no data written. If at the moment
*          of the Unlimited-Save request the manager is in Saving state the 
*          manager will return to it once the Unlimited once is finished.
* - \b Pause - The state where the manager does nothing waiting for the resume
*          command to start processing the registrants list in accordance with the
*          state the manager has prior the Pause command.
* 
********************************************************************************/
#define DEBUG_TAG "ioman"
#define DEBUG_VAR (service->debug_level)

#include <sxm_stdtrace.h>
#include <util/sxm_iomanager_internal.h>
#include <util/sxm_cfile.h>
#include <util/sxm_list.h>
#include <util/sxm_noname_internal.h>
#include <util/sxm_timers_incl.h>
#include <util/sxm_time_internal.h>
#include <util/sxm_msg_queue_incl.h>
#include <util/sxm_memory_internal.h>

/* Type forward declaration */
typedef struct _sxm_io_manager_struct SXMIOManager;

/* Power-Down profile */
const SXMIOManagerProfile SXM_IOMANAGER_POWER_DOWN_PROFILE =
    { &SXM_IOMANAGER_POWER_DOWN_PROFILE, 0U, 0U, 0U };

/* Quasi Immediate profile */
const SXMIOManagerProfile SXM_IOMANAGER_IMMEDIATE_PROFILE =
    { &SXM_IOMANAGER_IMMEDIATE_PROFILE, 30U, 0U, 0U };

/** \name Pre-defines timed profiles.
 * In those profiles the quota means delimiter for the max quota among all
 * registrants to get the per-cadence quota and the cycle quota mean multiplier
 * for the max quota among all registrants to get the cycle quota.
 * @{
 */
/* special key to detect internal timed profiles */
static const int IOMANAGER_PREDEF_TIMED_PROFILE_KEY = 0x10FAF10F;
/* Small profile */
const SXMIOManagerProfile SXM_IOMANAGER_SMALL_PROFILE =
    { &IOMANAGER_PREDEF_TIMED_PROFILE_KEY, 5U * SECONDS_IN_MINUTE, 4U, 20U };
/* Medium profile */
const SXMIOManagerProfile SXM_IOMANAGER_MEDIUM_PROFILE =
    { &IOMANAGER_PREDEF_TIMED_PROFILE_KEY, 5U * SECONDS_IN_MINUTE, 2U, 20U };
/* Large profile */
const SXMIOManagerProfile SXM_IOMANAGER_LARGE_PROFILE =
    { &IOMANAGER_PREDEF_TIMED_PROFILE_KEY, 5U * SECONDS_IN_MINUTE, 1U, 20U };
/** @} */

/** Supported events */
typedef enum _sxm_io_manager_event_enum {
    /** This event is sent by the timer once the required period expired.
     * Once event comes the quota utilization reset back to 0 indicating that
     * the manager can continue callback processing if needed
     */
    IOMANAGER_EVT_CADENCE,
    /** This event is sent by the timer once the cycle time has expired 
     * and each service can reset it's cycle quota
     */
    IOMANAGER_EVT_CYCLE_CADENCE,
    /** In order to distribute the processing in time and in order to let
     * service process some other events this event is used by the manager to
     * process only one callback, which has something to do, per event.
     */
    IOMANAGER_EVT_NEXT,
    /** Pauses the processing */
    IOMANAGER_EVT_PAUSE,
    /** Resumes the processing */
    IOMANAGER_EVT_RESUME,
    /** Forced saving process regardless any quotas */
    IOMANAGER_EVT_FORCE_SAVE,
    /** Stop event */
    IOMANAGER_EVT_STOP
} SXMIOManagerEvent;

/** Defines set of supported I/O Manager states */
typedef enum _sxm_io_manager_state_enum {
    /** The service is up and running, but waiting for the next command */
    IOMANAGER_STATE_IDLE,
    /** The service is up and running, but doing regular, caused by cadence
     * saving operations
     */
    IOMANAGER_STATE_SAVING,
    /** The service is up and running, and doing unlimited save procedure */
    IOMANAGER_STATE_SAVING_UNLIMITED,
    /** The service used to be running and now paused by the outside request */
    IOMANAGER_STATE_PAUSED,
    /** Service is stopping */
    IOMANAGER_STATE_STOPPING
} SXMIOManagerState;

/** Set of mask values to let info which parts of the service is initialized and
 * should be destroyed if necessary
 */
typedef enum _sxm_io_manager_items_mask_enum {
    IOMANAGER_NONE_MASK        = 0x0,  //!< None
    IOMANAGER_MUTEX_QUEUE_MASK = 0x1,  //!< Queue mutex is created
    IOMANAGER_MUTEX_LIST_MASK  = 0x2,  //!< List mutex is created
    IOMANAGER_LIST_MASK        = 0x4,  //!< List itself is created
    IOMANAGER_QUEUE_MASK       = 0x8,  //!< Queue itself is created
    IOMANAGER_TIMER_MASK       = 0x10, //!< Timer is set and running
    IOMANAGER_THREAD_MASK      = 0x20, //!< Thread is created
    IOMANAGER_ALL_MASK         = 0xFFFFFFFF
} SXMIOManegerItemsMask;

/** Useful constants */
enum {
    /** Defines message queue size */
    IOMANAGER_QUEUE_SIZE = 32,
    /** Defines cycle cadence in milliseconds */
    IOMANAGER_CYCLE_CADENCE = 4 * 60 * 60 * 1000 /* 4 hours */
};

/** Defines queue name */
#define IOMANAGER_QUEUE_NAME ("iom_queue")
/** Defines human readable thread name */
#define IOMANAGER_THREAD_NAME ("iom_thread")

/** Converts cadence to milliseconds
 * \param[in] _cadence the value in seconds
 */
#define IOMANAGER_CADENCE_TO_MSEC(_cadence) \
    ((_cadence) * SXM_MILLISEC_PER_SEC)

/** Function to update quotas if needed
 * \param[in] pService service instance
 */
typedef void (*SXMIOManagerUpdateQuotasFunc)(SXMIOManager *pService);

/** Function to switch timers is any
 * \param[in] pService service instance
 * \param[in] bActivate \p TRUE to active inactive timer, \p FALSE - vice versa
 * \retval TRUE The action has completed
 * \retval FALSE Some error occurred
 */
typedef BOOL (*SXMIOManagerTimerSwithFunc)(SXMIOManager *pService, BOOL bActivate);

/** Registration list entries */
typedef struct sxm_io_cfile_entry_struct {
    SXMIOManagerRegInfo sRegInfo; //!< Registration info
    size_t tQuotaCycleUsed; //!< Part of the cycle quota used
    size_t tRoundKey; //!< Keeps the latest round key the registrant was stimulated
    struct {
        /** Total number of written sectors */
        size_t tTotalWritten;
        /** Total number of attempts were given to let the registrant to save data */
        size_t tTotalAttemptGiven;
        /** Total number of used attempt, i.e. the attempt which does writes */
        size_t tTotalAttemptUsed;
    } stat; //!< Statistics
} SXMIOCFileEntry;

/** Defines the manager's instance */
struct _sxm_io_manager_struct {
    SXMIOManagerState eState; //!< Service state
    SXMIOManagerState eNextState; //!< The service next state where it's required
    SXMList mRegList; //!< List of registered callbacks
    pthread_mutex_t mRegListMtx; //!< List related mutex
    MSG_QUEUE_STRUCT msgQueue; //!< Message queue
    pthread_mutex_t msgQueueMtx; //!< Message queue associated mutex
    pthread_t tid; //!< Working thread
    uint debug_level; //!< Debug level
    uint timerHandle; //!< Timer handle
    SXMIOManagerProfile sProfile; //!< Current profile
    BOOL bRegListIsDirty; //!< Some changes have happened to the reg list
    size_t tQuotaTick; //!< The quota per tick aka cadence
    size_t tQuotaUsed; //!< Current quota utilization
    size_t tQuotaCycle; //!< The quota per cycle
    size_t tActiveRoundKey; //!< Current active saving round key
    SXMTimeSpec sQuotaCycleReset; //!< The moment when the cycle quotas were reset.
    SXMIOManagerUpdateQuotasFunc fUpdateMaxQuotas; //!< Function to update max quotas
    /** Function to handle the cadence in terms of quotas */
    SXMIOManagerUpdateQuotasFunc fUpdateQuotasPerCadence;
    /** Function to handle timers' state switching */
    SXMIOManagerTimerSwithFunc fTimersSwitch;
};

/** Service instance */
static SXMIOManager *service = NULL;

/** A mutex used to ensure serial execution of the Service commands. */
static pthread_mutex_t apimutex = PTHREAD_MUTEX_INITIALIZER;

/** Zero statistics to initialize the counters where it's needed */
static const SXMIOStat gZeroStat = { 0U };

/* Function forward declarations */
static void iom_timer(void *pArg);
static void* iom_thread(void *pArg);
static int iom_init(void);
static int iom_reg(SXMIOManager *pService, const SXMIOManagerRegInfo *pRegInfo);
static int iom_unreg(SXMIOManager *pService, const SXMIOManagerRegInfo *pRegInfo);
static void iom_update_quotas(SXMIOManager *pService);
static void iom_update_max_quotas(SXMIOManager *pService);
static void iom_update_cadence_quotas(SXMIOManager *pService);
static void iom_reset_quotas(SXMIOManager *pService);
static int iom_next(SXMIOManager *pService, BOOL bNewRound);
static BOOL iom_process_next(SXMIOManager *pService);
static BOOL iom_process_next_unlim(SXMIOManager *pService);
static BOOL iom_timers_switch(SXMIOManager *pService, BOOL bActivate);
static BOOL iom_timers_switch_no_timers(SXMIOManager *pService, BOOL bActivate);
static void iom_state(SXMIOManager *pService, SXMIOManagerState stTo, SXMIOManagerState stNext);
static int iom_uninit(SXMIOManager *pService, int mask);
static void iom_destroy(SXMIOManager *pService, int mask);
static int iom_send_event(SXMIOManager *pService, SXMIOManagerEvent evt);
static SXMIOCFileEntry* iom_find_entry(SXMIOManager *pService, const SXMIOManagerRegInfo *pRegInfo);

#ifndef SXM_DEBUG_PRODUCTION
static const char *iom_state_str(SXMIOManagerState st);
static const char *iom_evt_str(SXMIOManagerEvent evt);
#endif

/***************************************************************************//**
 * Timer's callback function.
 * \param[in] pArg the service instance
 ******************************************************************************/
static void iom_timer(void *pArg) {
    const int rc = iom_send_event((SXMIOManager*)pArg, IOMANAGER_EVT_CADENCE);
    if (rc != SXM_E_OK) {
        PLOG_UTL("Failed to send event from TIMER");
    }
    return;
}

/***************************************************************************//**
 * Thread function
 * \param[in] pArg the service instance
 * \return the same instance which was provided as the argument
 ******************************************************************************/
static void* iom_thread(void *pArg) {
    SXMIOManager *pService = (SXMIOManager *)pArg;
    SXMQueueMsg sMsg = { 0U, 0U, NULL };
    do {
        int rc;
        rc = sxm_queue_get(&pService->msgQueue, &sMsg, SXM_QUEUE_INFINITE_TIMEOUT);
        if (rc != SXM_E_OK) {
            if (rc != SXM_E_TIMEOUT) {
                /* exit immediately */
                PLOG_LLI("sxm_queue_get() failed (rc=%d), terminating", rc);
                break;
            }
            /* false signal, let's continue */
            DEBUG_UTL("False signal received from the queue");
        }
        else {
#ifndef SXM_DEBUG_PRODUCTION
            DEBUG_UTL("Message %s recieved, state %s",
                iom_evt_str((SXMIOManagerEvent)sMsg.msgType), iom_state_str(pService->eState));
#endif
            switch ((SXMIOManagerEvent)sMsg.msgType) {
                case IOMANAGER_EVT_CADENCE: {
                    if (pService->fUpdateQuotasPerCadence != NULL) {
                        pService->fUpdateQuotasPerCadence(pService);
                    }

                    iom_update_quotas(pService);

                    /* Continue processing cadence only in case of the proper state
                     */
                    switch (pService->eState) {
                        case IOMANAGER_STATE_IDLE: {
                            /* If the quota allows to do something, lets stimulate
                             * the processor
                             */
                            if (pService->fUpdateQuotasPerCadence != NULL) {
                                if (pService->tQuotaUsed < pService->tQuotaTick) {
                                    /* send event to start stimulating list processing */
                                    if (iom_next(pService, TRUE) == SXM_E_OK) {
                                        iom_state(pService, IOMANAGER_STATE_SAVING, pService->eState);
                                    }
                                }
                            }
                            else {
                                /* send event to start stimulating list processing */
                                if (iom_next(pService, TRUE) == SXM_E_OK) {
                                    iom_state(pService, IOMANAGER_STATE_SAVING_UNLIMITED, pService->eState);
                                }
                            }
                        }
                        break;
                        case IOMANAGER_STATE_SAVING_UNLIMITED: {
                            if (pService->fUpdateQuotasPerCadence != NULL) {
                                /* Since the service is saving the data in the forced
                                 * mode the service can try to re-do the same saving
                                 * operations after leaving the enforced-saving mode
                                 */
                                iom_state(pService, pService->eState, IOMANAGER_STATE_SAVING);
                            }
                        }
                        break;
                        default: {
                            DEBUG_UTL("There is no need CADENCE related action in state %s",
                                iom_state_str(pService->eState));
                        }
                        break;
                    }
                }
                break;
                case IOMANAGER_EVT_FORCE_SAVE: {
                    if (pService->eState == IOMANAGER_STATE_SAVING_UNLIMITED) {
                        DEBUG_UTL("The service already processing the force-save command");
                    }
                    else if (pService->eState == IOMANAGER_STATE_PAUSED) {
                        /* since the service is paused the application wanted
                         * to do not have writes during this time, thus,
                         * one the unpause is called the forced save should
                         * be started
                         */
                        iom_state(pService, pService->eState, IOMANAGER_STATE_SAVING_UNLIMITED);
                    }
                    else {
                        iom_state(pService, IOMANAGER_STATE_SAVING_UNLIMITED, pService->eState);
                        /* send event to stimulate the list processing */
                        (void) iom_next(pService, TRUE);
                    }
                }
                break;
                case IOMANAGER_EVT_PAUSE: {
                    DEBUG_UTL("Pausing the process");
                    switch (pService->eState) {
                        case IOMANAGER_STATE_IDLE:
                        case IOMANAGER_STATE_SAVING:
                        case IOMANAGER_STATE_SAVING_UNLIMITED: {
                            if (pService->fTimersSwitch(pService, FALSE) == TRUE) {
                                iom_state(pService, IOMANAGER_STATE_PAUSED, pService->eState);
                            }
                            else {
                                ERROR_UTL("Failed to pause the timer");
                            }
                        }
                        break;
                        default: {
                            DEBUG_UTL("There is no need in timer PAUSE, state %s",
                                iom_state_str(pService->eState));
                        }
                        break;
                    }
                }
                break;
                case IOMANAGER_EVT_RESUME: {
                    DEBUG_UTL("Resuming the process");
                    /* The service has paused processing routine, thus, all we
                     * need to do just un-pause it
                     */
                    if (pService->eState == IOMANAGER_STATE_PAUSED) {
                        if (pService->fTimersSwitch(pService, TRUE) == TRUE) {
                            iom_state(pService, pService->eNextState, IOMANAGER_STATE_IDLE);
                            if ((pService->eState == IOMANAGER_STATE_SAVING) ||
                                (pService->eState == IOMANAGER_STATE_SAVING_UNLIMITED))
                            {
                                /* let's resume the previous state via sending
                                 * event to start stimulating list processing.
                                 * Just in case make the new round to make sure
                                 * each of the registrants has a chance after
                                 * pause based on the possibly new quotas.
                                 */
                                (void) iom_next(pService, TRUE);
                            }
                        }
                        else {
                            ERROR_UTL("Failed to unpause the timer");
                        }
                    }
                    else {
                        DEBUG_UTL("There is no need in timer RESUME, state %s",
                            iom_state_str(pService->eState));
                    }
                }
                break;
                case IOMANAGER_EVT_NEXT: {
                    /* Process the event with respect to the current state */
                    switch (pService->eState) {
                        case IOMANAGER_STATE_SAVING_UNLIMITED: {
                            BOOL bContinue;
                            LOCK(pService->mRegListMtx);
                            bContinue = iom_process_next_unlim(pService);
                            UNLOCK(pService->mRegListMtx);
                            if (bContinue == FALSE) {
                                if (pService->eNextState == IOMANAGER_STATE_SAVING) {
                                    iom_state(pService, pService->eNextState, IOMANAGER_STATE_IDLE);
                                    bContinue = TRUE;
                                }
                                else {
                                    iom_state(pService, IOMANAGER_STATE_IDLE, IOMANAGER_STATE_IDLE);
                                }
                            }
                            if (bContinue == TRUE) {
                                DEBUG_UTL("Send NEXT event to continue in state %s",
                                    iom_state_str(pService->eState));
                                /* send even to continue */
                                (void) iom_next(pService, FALSE);
                            }
                        }
                        break;
                        case IOMANAGER_STATE_SAVING: {
                            BOOL bContinue = TRUE;
                            /* This check can be done w/o lock since even in the case
                            * we do missed the change inaccurate value will be used
                            * only once. During the very first start the save will happen
                            * only after registration where the value will be set
                            */
                            iom_update_quotas(pService);

                            if (pService->tQuotaUsed >= pService->tQuotaTick) {
                                DEBUG_UTL("Quota is out, waiting for the next 'cadence' tick");
                                bContinue = FALSE;
                            }
                            else {
                                LOCK(pService->mRegListMtx);
                                bContinue = iom_process_next(pService);
                                UNLOCK(pService->mRegListMtx);
                            }
                            if (bContinue == FALSE) {
                                /* move to idle state if there is nothing the service
                                 * can do at this moment.
                                 */
                                iom_state(pService, pService->eNextState, IOMANAGER_STATE_IDLE);
                            }
                            else {
                                DEBUG_UTL("Send NEXT event to continue");
                                /* send even to continue */
                                (void) iom_next(pService, FALSE);
                            }
                        }
                        break;
                        default: {
                            DEBUG_UTL("There is no actions in state %s",
                                iom_state_str(pService->eState));
                        }
                        break;
                    }
                }
                break;
                case IOMANAGER_EVT_STOP:
                    iom_state(pService, IOMANAGER_STATE_STOPPING, IOMANAGER_STATE_IDLE);
                    DEBUG_UTL("The service is about to stop")
                    break;
                default:
                    break;
            }
            /* Release message's resources */
            sxm_queue_msg_release(&sMsg);
        }
    } while (sMsg.msgType != IOMANAGER_EVT_STOP);
    return pArg;
}

/***************************************************************************//**
 * Starts the manager
 * \param[in] debugLevel the debug level
 * \param[in] mask levels of initialization mask
 ******************************************************************************/
static void iom_destroy(SXMIOManager *pService, int mask) {
    /* Remove the timer */
    if ((mask & IOMANAGER_TIMER_MASK) == IOMANAGER_TIMER_MASK) {
        (void) sxm_timers_delete_timer(pService->timerHandle);
    }
    /* Clean up the list */
    if ((mask & IOMANAGER_LIST_MASK) == IOMANAGER_LIST_MASK) {
        SXMListEntry *pEntry;
        while ((pEntry = sxm_list_first(&pService->mRegList)) != NULL) {
            (void) sxm_list_remove(&pService->mRegList, pEntry);
            (void) sxm_list_free(sxm_list_data(pEntry));
        }
        (void) sxm_list_destroy(&pService->mRegList);
    }
    /* destroy the queue */
    if ((mask & IOMANAGER_QUEUE_MASK) == IOMANAGER_QUEUE_MASK) {
        (void) sxm_queue_destroy(&pService->msgQueue);
    }
    /* remove mutexes */
    if ((mask & IOMANAGER_MUTEX_QUEUE_MASK) == IOMANAGER_MUTEX_QUEUE_MASK) {
        (void)pthread_mutex_destroy(&pService->msgQueueMtx);
    }
    if ((mask & IOMANAGER_MUTEX_LIST_MASK) == IOMANAGER_MUTEX_LIST_MASK) {
        (void)pthread_mutex_destroy(&pService->mRegListMtx);
    }
    /* remove the service itself */
    sxe_free(pService);
    return;
}

/***************************************************************************//**
 * Wraps up event sending
 * \param[in] pService the service instance
 * \param[in] evt the event to be sent
 * \return corresponding error code.
 ******************************************************************************/
static int iom_send_event(SXMIOManager *pService, SXMIOManagerEvent evt) {
    int rc = sxm_queue_put(&pService->msgQueue, NULL, evt, 0U, MSG_PRIO_NORMAL);

#ifndef SXM_DEBUG_PRODUCTION
    if (rc != SXM_E_OK) {
        PLOG_UTL("Unable to sent the %s event, rc=%d", iom_evt_str(evt), rc);
    }
#endif

    return rc;
}

/***************************************************************************//**
 * Finds the entry associated to the callback
 * \param[in] pService the service instance
 * \param[in] pRegInfo registration info to look for the entry
 * \return Valid entry if the callback is registered or \p NULL if not.
 ******************************************************************************/
static SXMIOCFileEntry* iom_find_entry(SXMIOManager *pService, const SXMIOManagerRegInfo *pRegInfo) {
    SXMListEntry *pEntry;
    SXMIOCFileEntry *pRes = NULL;

    DEBUG_UTL("pService: %p, pFunc: %p", pService, pRegInfo->pFunc);

    pEntry = sxm_list_first(&pService->mRegList);
    while (pEntry != NULL) {
        SXMIOCFileEntry *pData = (SXMIOCFileEntry *)sxm_list_data(pEntry);
        if (pData->sRegInfo.pFunc == pRegInfo->pFunc) {
            pRes = pData;
            break;
        }
        pEntry = sxm_list_next(pEntry);
    } 
    return pRes;
}

/***************************************************************************//**
 * Initialized the manager
 * \retval SXM_E_OK Success
 * \retval SXM_E_NOMEM Memory allocation issue
 * \retval SXM_E_THREAD Thread creation issue
 ******************************************************************************/
static int iom_init(void) {
    int rc;
    int mask = IOMANAGER_NONE_MASK;
    SXMIOManager *pService = NULL;

    switch (0) { default: {
        SXMSdkConfig sdkCfg;

        /* request the current profile which is set by the application */
        (void) sxm_sdk_get_config(&sdkCfg);

        /* Allocate memory */
        pService = (SXMIOManager *)sxe_calloc(1, sizeof(*pService));
        if (pService == NULL) {
            PLOG_UTL("No memory available");
            rc = SXM_E_NOMEM;
            break;
        }

        /* initialize the list */
        rc = sxm_list_create(&pService->mRegList, SXM_LIST_PREALLOCATED);
        if (rc != SXM_E_OK) {
            PLOG_UTL("failed to initialize the list");
            break;
        }
        mask |= IOMANAGER_LIST_MASK;

        rc = pthread_mutex_init(&pService->mRegListMtx, NULL);
        if (rc != 0) {
            PLOG_UTL("failed to initialize the list's mutex, rc=%d", rc);
            rc = SXM_E_THREAD;
            break;
        }
        mask |= IOMANAGER_MUTEX_LIST_MASK;

        /* store the profile */
        if ((sdkCfg.flags & SXM_SDK_CONFIG_IOMAN_PROFILE) == SXM_SDK_CONFIG_IOMAN_PROFILE) {
            pService->sProfile = sdkCfg.ioman_profile;
        }
        else {
            /* impossible case, but to make sure we're the default is set
                * use this branch
                */
            pService->sProfile = SXM_IOMANAGER_POWER_DOWN_PROFILE;
        }

        /* In case of the power-down profile there is no need to start
         * thread and create queue
         */
        if (pService->sProfile.pSig == &SXM_IOMANAGER_POWER_DOWN_PROFILE) {
            pService->fUpdateMaxQuotas = NULL;
            pService->fUpdateQuotasPerCadence = NULL;
            pService->fTimersSwitch = &iom_timers_switch_no_timers;
            pService->bRegListIsDirty = FALSE;
        }
        /* In case of immediate profile no quota management required */
        else if (pService->sProfile.pSig == &SXM_IOMANAGER_IMMEDIATE_PROFILE) {
            pService->fUpdateMaxQuotas = NULL;
            pService->fUpdateQuotasPerCadence = NULL;
            pService->fTimersSwitch = &iom_timers_switch;
            pService->bRegListIsDirty = FALSE;
        }
        else if (pService->sProfile.pSig == &IOMANAGER_PREDEF_TIMED_PROFILE_KEY) {
            pService->fUpdateMaxQuotas = &iom_update_max_quotas;
            pService->fUpdateQuotasPerCadence = &iom_update_cadence_quotas;
            pService->fTimersSwitch = &iom_timers_switch;
            pService->bRegListIsDirty = TRUE;
            sxm_clock_gettime(&pService->sQuotaCycleReset);
        }
        /* In case of custom profiles there is need to set the quotas as-is */
        else {
            pService->tQuotaTick =
                NUM_ALIGNED_BLOCKS(pService->sProfile.quota, SXM_ARCH_FILE_SECTOR_SIZE);
            pService->tQuotaCycle =
                NUM_ALIGNED_BLOCKS(pService->sProfile.cycle_quota, SXM_ARCH_FILE_SECTOR_SIZE);
            pService->fUpdateMaxQuotas = NULL;
            pService->fUpdateQuotasPerCadence = &iom_update_cadence_quotas;
            pService->fTimersSwitch = &iom_timers_switch;
            pService->bRegListIsDirty = TRUE;
            sxm_clock_gettime(&pService->sQuotaCycleReset);
        }

        /* initialize the mutexes */
        rc = pthread_mutex_init(&pService->msgQueueMtx, NULL);
        if (rc != 0) {
            PLOG_UTL("failed to initialize the queue's mutex, rc=%d", rc);
            rc = SXM_E_THREAD;
            break;
        }
        mask |= IOMANAGER_MUTEX_QUEUE_MASK;

        /* create the queue */
        rc = sxm_queue_create(&pService->msgQueue, IOMANAGER_QUEUE_SIZE,
                              &pService->msgQueueMtx, 0U,
                              IOMANAGER_QUEUE_NAME, &pService->debug_level);
        if (rc != SXM_E_OK) {
            PLOG_UTL("failed to create message queue, rc = %d", rc);
            break;
        }
        mask |= IOMANAGER_QUEUE_MASK;

        if (pService->sProfile.pSig != &SXM_IOMANAGER_POWER_DOWN_PROFILE) {
            /* set up the timers, but not activate it */
            pService->sProfile.cadence = IOMANAGER_CADENCE_TO_MSEC(pService->sProfile.cadence);
            if (sxm_timers_create_timer_ms((uint)pService->sProfile.cadence,
                                           &pService->timerHandle,
                                           REARMING, &iom_timer, pService) != TRUE)
            {
                PLOG_UTL("failed to create timer");
                rc = SXM_E_NOMEM;
                break;
            }
            mask |= IOMANAGER_TIMER_MASK;

            /* start timer since if this function is called there is at least
             * one candidate to be processed
             */
            if (sxm_timers_start_timer(pService->timerHandle) == FALSE) {
                PLOG_UTL("failed to start just created timer %u", pService->timerHandle);
                rc = SXM_E_ERROR;
                break;
            }
        }

        /* create the thread */
        rc = sxm_pthread_create(&pService->tid, NULL, &iom_thread, pService);
        if (rc != 0) {
            PLOG_UTL("failed to create thread, rc=%d", rc);
            rc = SXM_E_THREAD;
            break;
        }
        sxe_thread_setname(pService->tid, IOMANAGER_THREAD_NAME);
        mask |= IOMANAGER_THREAD_MASK;

        /* initialize remaining fields */
        pService->debug_level = 0U;
        pService->eState = IOMANAGER_STATE_IDLE;
        pService->eNextState = IOMANAGER_STATE_IDLE;
        pService->tActiveRoundKey = 0U;
        /* good to go */
        rc = SXM_E_OK;
    }}

    /* Rollback creation process in case of error */
    if (rc != SXM_E_OK) {
        if (pService != NULL) {
            iom_destroy(pService, mask);
            pService = NULL;
        }
        PLOG_UTL("Failed to init, rc=%s", sxm_sdk_format_result(rc));
    }
    service = pService; /* populate as-is */
    return rc;
}

/***************************************************************************//**
 * Registers the callback function
 * \param[in] pService service instance
 * \param[in] pRegInfo registration info
 * \retval SXM_E_OK Success
 * \retval SXM_E_FAULT NULL argument provided
 * \retval SXM_E_NOMEM There is no memory to fulfill the request
 ******************************************************************************/
static int iom_reg(SXMIOManager *pService, const SXMIOManagerRegInfo *pRegInfo) {
    SXMIOCFileEntry *pEntry;
    int rc;
    ENTER_UTL("pService: %p, pRegInfo: %p", pService, pRegInfo);

    pEntry = iom_find_entry(service, pRegInfo);
    if (pEntry != NULL) {
        DEBUG_UTL("The callback %s[%p] already registered", pRegInfo->psTag, pRegInfo->pFunc);
        rc = SXM_E_INVAL;
    }
    else {
        SXMIOCFileEntry *pNewEntry = (SXMIOCFileEntry*)sxm_list_allocate(sizeof(SXMIOCFileEntry));
        if (pNewEntry == NULL) {
            ERROR_UTL("No memory available for regitration, callback %s[%p]",
                pRegInfo->psTag, pRegInfo->pFunc);
            rc = SXM_E_NOMEM;
        }
        else {
            /* Converts bytes quota, requester by the caller, into sectors */
            pNewEntry->sRegInfo = *pRegInfo;
            pNewEntry->tQuotaCycleUsed = 0U;
            pNewEntry->tRoundKey = (size_t)(~pService->tActiveRoundKey);
            rc = sxm_list_add(&pService->mRegList, pNewEntry, FALSE, NULL);
            if (rc != SXM_E_OK) {
                ERROR_UTL("Failed to add new entry, rc=%s, callback %s[%p]",
                    sxm_sdk_format_result(rc), pRegInfo->psTag, pRegInfo->pFunc);
                sxm_list_free(pNewEntry);
            }
            else {
                DEBUG_UTL("[%s] new entry %p for callback %p has been added",
                    pNewEntry->sRegInfo.psTag, pNewEntry, pRegInfo->pFunc);
                pService->bRegListIsDirty = TRUE;
            }
        }
    }
    return rc;
}

/***************************************************************************//**
 * Unregisters the callback function
 * \param[in] pService service instance
 * \param[in,out] pRegInfo registration info for previously registered item.
 * \retval SXM_E_OK Success
 * \retval SXM_E_INVAL Invalid entry to search
 ******************************************************************************/
static int iom_unreg(SXMIOManager *pService, const SXMIOManagerRegInfo *pRegInfo)
{
    SXMIOCFileEntry *pEntry;
    int rc;
    ENTER_UTL("pService: %p, pRegInfo: %p", pService, pRegInfo);
    pEntry = iom_find_entry(service, pRegInfo);
    if (pEntry == NULL) {
        DEBUG_UTL("The callback %p is unknown", pRegInfo->pFunc);
        rc = SXM_E_INVAL;
    }
    else {
        SXMListEntry *pListEntry = sxm_list_entry(pEntry);
        rc = sxm_list_remove(&pService->mRegList, pListEntry);
        if (rc != SXM_E_OK) {
            ERROR_UTL("Failed to remove entry for callback %s[%p]",
                pRegInfo->psTag, pRegInfo->pFunc);
        }
        else {
            DEBUG_UTL("entry %p for callback %s[%p] has been remove, written %u, attempts %u of %u",
                pEntry, pEntry->sRegInfo.psTag, pEntry->sRegInfo.pFunc,
                (uint)pEntry->stat.tTotalWritten, (uint)pEntry->stat.tTotalAttemptUsed,
                (uint)pEntry->stat.tTotalAttemptGiven);
            sxm_list_free(sxm_list_data(pListEntry));
            pService->bRegListIsDirty = TRUE;
        }
    }
    return rc;
}


/***************************************************************************//**
 * Updates the quotas based on the list values in accordance with the pre-defined
 * timer based profiles and current state of the reg-list
 * \param[in] pService the service instance
 ******************************************************************************/
static void iom_update_quotas(SXMIOManager *pService) {
    /* This check can be done w/o lock since even in the case
     * we do missed the change inaccurate value will be used
     * only once. During the very first start the save will happen
     * only after registration where the value will be set
     */
    if ((pService->bRegListIsDirty == TRUE) && (pService->fUpdateMaxQuotas != NULL)) {
        DEBUG_UTL("Update quotas");
        LOCK(pService->mRegListMtx);
        pService->fUpdateMaxQuotas(pService);
        pService->bRegListIsDirty = FALSE;
        UNLOCK(pService->mRegListMtx);
    }
    return;
}

/***************************************************************************//**
 * Updates the quotas based on the list values in accordance with the pre-defined
 * timer based profiles.
 * \param[in] pService the service instance
 ******************************************************************************/
static void iom_update_max_quotas(SXMIOManager *pService) {
    SXMListEntry *pEntry = sxm_list_first(&pService->mRegList);
    size_t tQuotaMax = 0U;
    while (pEntry != NULL) {
        const size_t regQuota =
            ((SXMIOCFileEntry*)sxm_list_data(pEntry))->sRegInfo.tQuota;
        if (tQuotaMax < regQuota) {
            tQuotaMax = regQuota;
        }
        pEntry = sxm_list_next(pEntry); /* move to the next */
    }
    pService->tQuotaTick =
        NUM_ALIGNED_BLOCKS(tQuotaMax, SXM_ARCH_FILE_SECTOR_SIZE) / pService->sProfile.quota;
    pService->tQuotaCycle =
        NUM_ALIGNED_BLOCKS(tQuotaMax, SXM_ARCH_FILE_SECTOR_SIZE) * pService->sProfile.cycle_quota;

    return;
}

/***************************************************************************//**
 * Updates the quotas by handling the cadence
 * timer based profiles.
 * \param[in] pService the service instance
 ******************************************************************************/
static void iom_update_cadence_quotas(SXMIOManager *pService) {
    SXMTimeSpec sCurrTime;
    time_t tDiff;
    /* update the quota utilization */
    if (pService->tQuotaUsed > pService->tQuotaTick) {
        pService->tQuotaUsed -= pService->tQuotaTick;
    }
    else {
        pService->tQuotaUsed = 0U;
    }
    /* Check the time and in case of expiration reset cycle quotas */
    sxm_clock_gettime(&sCurrTime);
    tDiff = sxm_time_get_absolute_time_diff(&sCurrTime, &pService->sQuotaCycleReset);
    if (tDiff > IOMANAGER_CYCLE_CADENCE) {
        LOCK(pService->mRegListMtx);
        iom_reset_quotas(pService);
        UNLOCK(pService->mRegListMtx);
        pService->sQuotaCycleReset = sCurrTime;
    }
    return;
}

/***************************************************************************//**
 * Reset the cycle quota for all registrants
 * \param[in] pService the service instance
 ******************************************************************************/
static void iom_reset_quotas(SXMIOManager *pService) {
    SXMListEntry *pEntry = sxm_list_first(&pService->mRegList);
    DEBUG_UTL("pService: %p", pService);
    while (pEntry != NULL) {
        ((SXMIOCFileEntry*)sxm_list_data(pEntry))->tQuotaCycleUsed = 0U;
        pEntry = sxm_list_next(pEntry); /* move to the next */
    }
    return;
}

/***************************************************************************//**
 * Stimulates processing by sending NEXT event
 * \param[in] pService the service instance
 * \param[in] bNewRound indicates that the new round key required
 * \return Corresponding error code
 ******************************************************************************/
static int iom_next(SXMIOManager *pService, BOOL bNewRound) {
    const int rc = iom_send_event(pService, IOMANAGER_EVT_NEXT);
	if ((rc == SXM_E_OK) && (bNewRound == TRUE)) {
		pService->tActiveRoundKey++; /* new round */
		DEBUG_UTL("New round #%u started", (uint)pService->tActiveRoundKey);
	}
	return rc;
}

/***************************************************************************//**
 * The code looks through the list of registered entries and in case if one of
 * them has something to process another NEXT event will be sent to move forward.
 * If no one does write operation the process will be postponed until CADENCE
 * timer based profiles.
 * \param[in] pService the service instance
 * \retval FALSE There no more elements to process
 * \retval TRUE The function does saving and in a theory there is another
 *              registrant which can do saving
 ******************************************************************************/
static BOOL iom_process_next(SXMIOManager *pService) {
    size_t count = 0U;
    DEBUG_UTL("")
    while (count < sxm_list_size(&pService->mRegList)) {
        SXMListEntry *pEntry = sxm_list_first(&pService->mRegList);
        SXMIOCFileEntry *pCFileEntry = (SXMIOCFileEntry *)sxm_list_data(pEntry);
        /* Move the entry back to the list */
        (void) sxm_list_remove(&pService->mRegList, pEntry);
        (void) sxm_list_add(&pService->mRegList, pCFileEntry, FALSE, NULL);

        /* In order to avoid unlimited saving for the same registrant the
         * manager provides the round key which has to be different in comparison
         * to the currently used by the manager in order to let the registrant
         * proceed
         */
        if (pService->tActiveRoundKey != pCFileEntry->tRoundKey) {
            SXMIOStat sStat = gZeroStat;
            /* In case if cycle quota allocated for this registrant
             * has been ran out there no call to save the data
             * will be done until cycle quota reset
             */
            ++(pCFileEntry->stat.tTotalAttemptGiven);
            if (pCFileEntry->tQuotaCycleUsed < pService->tQuotaCycle) {
                int rc;
                /* make call to see what is happening */
                rc = pCFileEntry->sRegInfo.pFunc(&sStat, pCFileEntry->sRegInfo.pArg);
                DEBUG_UTL("Callback %s finished as %s, written %u sector(s)",
                    pCFileEntry->sRegInfo.psTag, sxm_sdk_format_result(rc),
                    (uint)sStat.sectorsWritten);
            }
            else {
                DEBUG_UTL("Callback %s has no more cycle quota, skip",
                    pCFileEntry->sRegInfo.psTag);
            }

            /* store the round key to indicate we're done on this round
             * for this particular registrant
             */
            pCFileEntry->tRoundKey = pService->tActiveRoundKey;

            /* check how many data was written */
            if (sStat.sectorsWritten > 0U) {
                pCFileEntry->stat.tTotalAttemptUsed++;
                pCFileEntry->stat.tTotalWritten += sStat.sectorsWritten;
                pService->tQuotaUsed += sStat.sectorsWritten;
                pCFileEntry->tQuotaCycleUsed += sStat.sectorsWritten;
                break;
            }

            DEBUG_UTL("There is no data has been written from %s, go to the next",
                pCFileEntry->sRegInfo.psTag);
        }
        else {
            DEBUG_UTL("%s has been granted at this round %u already, go to the next",
                pCFileEntry->sRegInfo.psTag, (uint)pCFileEntry->tRoundKey);
        }

        ++count;
    }
    return (count == sxm_list_size(&pService->mRegList)) ? FALSE : TRUE;
}

/***************************************************************************//**
 * The code looks through the list of registered entries and does saving regardless
 * any quota.
 * \param[in] pService the service instance
 * \retval FALSE There no more elements to process
 * \retval TRUE The function does saving and in a theory there is another
 *              registrant which can do saving
 ******************************************************************************/
static BOOL iom_process_next_unlim(SXMIOManager *pService) {
    SXMListEntry *pEntry = sxm_list_first(&pService->mRegList);
    while (pEntry != NULL) {
        int rc;
        SXMIOCFileEntry *pCFileEntry = (SXMIOCFileEntry *)sxm_list_data(pEntry);
        if (pCFileEntry->tRoundKey != pService->tActiveRoundKey) {
            SXMIOStat sStat = gZeroStat;
            /* make call to see what is happening */
            rc = pCFileEntry->sRegInfo.pFunc(&sStat, pCFileEntry->sRegInfo.pArg);
            DEBUG_UTL("Callback %s finished as %s, written %u sector(s)",
                pCFileEntry->sRegInfo.psTag, sxm_sdk_format_result(rc),
                (uint)sStat.sectorsWritten);    

            /* store the round key to indicate we're done on this round */
            pCFileEntry->tRoundKey = pService->tActiveRoundKey;

            pCFileEntry->stat.tTotalAttemptGiven++;
            if (sStat.sectorsWritten > 0U) {
                pCFileEntry->stat.tTotalWritten += sStat.sectorsWritten;
                pCFileEntry->stat.tTotalAttemptUsed++;
                break;
            }
            DEBUG_UTL("There is no data has been written from %s, go to the next entry",
                pCFileEntry->sRegInfo.psTag);
        }
        else {
            DEBUG_UTL("%s has been granted at this round %u already, go to the next",
                pCFileEntry->sRegInfo.psTag, (uint)pCFileEntry->tRoundKey);
        }

        /* move to the next */
        pEntry = sxm_list_next(pEntry);
    }
    return (pEntry == NULL) ? FALSE : TRUE;
}

/***************************************************************************//**
 * Trivial function to activate/deactivate timers
 * \param[in] pService the service instance
 * \param[in] bActivate desired timers' state
 * \retval TRUE OK
 * \retval FALSE Error
 ******************************************************************************/
static BOOL iom_timers_switch(SXMIOManager *pService, BOOL bActivate) {
    BOOL bRes;
    if (bActivate == TRUE) {
        bRes = sxm_timers_unpause_timer(pService->timerHandle);
    }
    else {
        bRes = sxm_timers_pause_timer(pService->timerHandle);
    }
    return bRes;
}

/***************************************************************************//**
 * Trivial function to emulate timers' activaion-deactivation for not-timers
 * profiles
 * \param[in] pService the service instance
 * \param[in] bActivate desired timers' state
 * \retval TRUE OK
 ******************************************************************************/
static BOOL iom_timers_switch_no_timers(SXMIOManager *pService, BOOL bActivate) {
    UNUSED_VAR(pService);
    UNUSED_VAR(bActivate);
    return TRUE;
}

/***************************************************************************//**
 * Trivial function to make states transition in one spot
 * \param[in] pService the service instance
 * \param[in] eTo new state
 * \param[in] stNext the next state which will be set once the manager
 *                   leaves the new one
 ******************************************************************************/
static void iom_state(SXMIOManager *pService, SXMIOManagerState eTo, SXMIOManagerState eNext) {
    DEBUG_UTL("%s -> %s, Next %s", iom_state_str(pService->eState),
        iom_state_str(eTo), iom_state_str(eNext));
    pService->eState = eTo;
    pService->eNextState = eNext;
    return;
}

/***************************************************************************//**
 * Uninitializes the service and all resources
 * \param[in] pService the service instance
 * \param[in] mask mask of what to release
 * \return Corresponding error code
 ******************************************************************************/
static int iom_uninit(SXMIOManager *pService, int mask) {
    int rc = SXM_E_OK;
    ENTER_UTL("pService: %p, mask: %08x", pService, mask);
    if (pService->sProfile.pSig == &SXM_IOMANAGER_POWER_DOWN_PROFILE) {
        /* power down profile has no timer */
        mask &= ~IOMANAGER_TIMER_MASK;
    }
    /* in case of the thread mask there is a sense in sending dedicated
     * message to that thread
     */
    if ((mask & IOMANAGER_THREAD_MASK) == IOMANAGER_THREAD_MASK) {
        rc = sxm_queue_put(&pService->msgQueue, NULL, IOMANAGER_EVT_STOP, 0U, MSG_PRIO_HIGH);
        if (rc == SXM_E_OK) {
            pthread_join(pService->tid, NULL);
        }
    }

    iom_destroy(pService, mask);
    
    return rc;
}

/***************************************************************************//**
 * This function can be used to aid in debugging by setting the desired debug
 * level for the service. This will enable Trace information.
 *
 * \param[in] debugLevel the debug level for the module
 *
 * \retval SXM_E_OK Success
 * \retval SXM_E_STATE Service has not been started
 ******************************************************************************/
int sxm_iomanager_set_debug(int debugLevel) {
    SXMResultCode rc;

    LOCK(apimutex);

    if (service == NULL) {
        rc = SXM_E_STATE;
    }
    else {
        service->debug_level = (uint)debugLevel;
        DEBUG_UTL("debugLevel: %d", debugLevel);
        rc = SXM_E_OK;
    }

    UNLOCK(apimutex);
    return rc;
}

/***************************************************************************//**
 * This function sends one of the predefined commands to the Manager to 
 * perform corresponding action. Refer to \ref SXMIOManagerCommand for full
 * set of supported commands.
 *
 * \param[in] cmd I/O manager command
 *
 * \retval SXM_E_OK Success
 * \retval SXM_E_INVAL Invalid argument
 * \retval SXM_E_STATE Service has not been started, or has not been configured yet
 ******************************************************************************/
int sxm_iomanager_command(SXMIOManagerCommand cmd) {
    int rc;
    LOCK(apimutex);
    if (service == NULL) {
        rc = SXM_E_STATE;
    }
    else {
        ENTER_UTL("cmd: %d", (int)cmd);
        switch (cmd) {
            case SXM_IOMANAGER_FORCE_SAVE: {
                rc = iom_send_event(service, IOMANAGER_EVT_FORCE_SAVE);
            }
            break;
            case SXM_IOMANAGER_PAUSE: {
                rc = iom_send_event(service, IOMANAGER_EVT_PAUSE);
            }
            break;
            case SXM_IOMANAGER_RESUME: {
                rc = iom_send_event(service, IOMANAGER_EVT_RESUME);
            }
            break;
            default: {
                ERROR_UTL("Unsupported command");
                rc = SXM_E_INVAL;
            }
            break;
        }
        LEAVE_UTL("rc=%s", sxm_sdk_format_result(rc));
    }
    UNLOCK(apimutex);
    return rc;
}

/***************************************************************************//**
 * Uninitializes the manager
 * \retval SXM_E_OK Success
 * \retval SXM_E_STATE The service is not running
 ******************************************************************************/
int sxm_iomanager_uninit(void) {
    int rc;

    LOCK(apimutex);
    if (service == NULL) {
        rc = SXM_E_STATE;
    }
    else {
        /** If the service still running it means that not all registrants 
         * does graceful un-registration procedure
         */
        PLOG_UTL("Not all registrants left the manager");
        rc = iom_uninit(service, IOMANAGER_ALL_MASK);
        if (rc == SXM_E_OK) {
            service = NULL;
        }
    }
    UNLOCK(apimutex);
    return rc;
}

/***************************************************************************//**
 * Registers the callback function. The process is performed from the caller
 * thread in order to ensure the caller that once the function returns registration
 * is completed.
 * \note If the manager is not running it will be initialized first.
 * \param[in] pRegInfo registration info
 * \retval SXM_E_OK Success
 * \retval SXM_E_FAULT NULL argument provided
 * \retval SXM_E_NOMEM There is no memory to fulfill the request
 ******************************************************************************/
int sxm_iomanager_reg(const SXMIOManagerRegInfo *pRegInfo) {
    int rc;
    LOCK(apimutex);
    if ((pRegInfo == NULL) || (pRegInfo->pFunc == NULL)) {
        rc = SXM_E_FAULT;
    }
    else {
        rc = (service == NULL) ? iom_init() : SXM_E_OK;
        if (rc == SXM_E_OK) {
            ENTER_UTL("pRegInfo: %p", pRegInfo);
            LOCK(service->mRegListMtx);
            rc = iom_reg(service, pRegInfo);
            UNLOCK(service->mRegListMtx);
            LEAVE_UTL("rc=%s", sxm_sdk_format_result(rc));
        }
    }

    UNLOCK(apimutex);
    return rc;
}

/***************************************************************************//**
 * Unregisters the callback function. he process is performed from the caller
 * thread in order to ensure the caller that once the function returns un-registration
 * is completed.
 * \note If the manager has this call to unregister the latest component the manager
 *       will be uninitialized during this call.
 * \param[in] pRegInfo registration info for previously registered item.
 * \retval SXM_E_OK Success
 * \retval SXM_E_STATE Service is not running
 * \retval SXM_E_FAULT NULL argument provided
 * \retval SXM_E_STATE the service is not running
 ******************************************************************************/
int sxm_iomanager_unreg(const SXMIOManagerRegInfo *pRegInfo) {
    int rc;
    LOCK(apimutex);

    if ((pRegInfo == NULL) || (pRegInfo->pFunc == NULL)) {
        rc = SXM_E_FAULT;
    }
    else if (service == NULL) {
        rc = SXM_E_STATE;
    }
    else {
        BOOL bIsEmpty;
        ENTER_UTL("pRegInfo: %p", pRegInfo);
        LOCK(service->mRegListMtx);
        rc = iom_unreg(service, pRegInfo);
        if (rc == SXM_E_OK) {
            bIsEmpty = (sxm_list_size(&service->mRegList) == 0U) ? TRUE : FALSE;
        }
        else {
            bIsEmpty = FALSE;
        }
        UNLOCK(service->mRegListMtx);
        LEAVE_UTL("rc=%s", sxm_sdk_format_result(rc));

        if (bIsEmpty == TRUE) {
            /* in case if the list is empty the manager can be destroyed */
            rc = iom_uninit(service, IOMANAGER_ALL_MASK);
            if (rc == SXM_E_OK) {
                service = NULL;
            }
        }
    }
    UNLOCK(apimutex);
    return rc;
}

#ifndef SXM_DEBUG_PRODUCTION
/***************************************************************************//**
 * Provides string representation for the manager's states
 * \param[in] st required state to be converted into the string
 * \return nil-terminated state's string representation
 ******************************************************************************/
static const char *iom_state_str(SXMIOManagerState st) {
    const char *pRes = "Unknown";
    switch (st) {
        case IOMANAGER_STATE_IDLE: pRes = "STATE_IDLE"; break;
        case IOMANAGER_STATE_SAVING: pRes = "STATE_SAVING"; break;
        case IOMANAGER_STATE_SAVING_UNLIMITED: pRes = "STATE_SAVING_UNLIMITED"; break;
        case IOMANAGER_STATE_PAUSED: pRes = "STATE_PAUSED"; break;
        case IOMANAGER_STATE_STOPPING: pRes = "STATE_STOPPING"; break;
    }
    return pRes;
}

/***************************************************************************//**
 * Provides string representation for the manager's events
 * \param[in] evt required state to be converted into the string
 * \return nil-terminated event's string representation
 ******************************************************************************/
static const char *iom_evt_str(SXMIOManagerEvent evt) {
    const char *pRes = "Unknown";
    switch (evt) {
        case IOMANAGER_EVT_CADENCE: pRes = "EVT_CADENCE"; break;
        case IOMANAGER_EVT_CYCLE_CADENCE: pRes = "EVT_CYCLE_CADENCE"; break;
        case IOMANAGER_EVT_NEXT: pRes = "EVT_NEXT"; break;
        case IOMANAGER_EVT_PAUSE: pRes = "EVT_PAUSE"; break;
        case IOMANAGER_EVT_RESUME: pRes = "EVT_RESUME"; break;
        case IOMANAGER_EVT_FORCE_SAVE: pRes = "EVT_FORCE_SAVE"; break;
        case IOMANAGER_EVT_STOP: pRes = "EVT_STOP"; break;
    }
    return pRes;
}
#endif
