/******************************************************************************/
/*                Copyright (c) Sirius XM Satellite Radio, Inc.               */
/*                            All Rights Reserved                             */
/*      Licensed Materials - Property of Sirius XM Satellite Radio, Inc.      */
/******************************************************************************/
/***************************************************************************//**
*
* \file sxm_grid.h
* \author Leslie French, Alexander Pylchagin
* \date 8/20/2013
*
* Public header file for GRID definitions.
*
* These are common routines for the services that have a baseline
* file (and corresponding transaction file) arranged in a 2x2
* grid.
*
*******************************************************************************/
#ifndef SXM_GRID_H
#define SXM_GRID_H

#include <sxm_build.h>
#include <sxm_sdk.h>

#ifndef SDKFILES_STANDALONE_BUILD
#include <util/sxm_common_internal.h>
#else
#include <sxm_typ.h>
#endif

/** Defines the Fuel MBR shrinking factor.
 * Reduce 10% from each side of MBR
 */
#if SXM_USE_FIX
#define SXM_GRID_DEFAULT_MBR_SHRINK_FACTOR ((SXMFLOAT)26214)
#else
#define SXM_GRID_DEFAULT_MBR_SHRINK_FACTOR ((SXMFLOAT)0.8)
#endif

/** Defines the Grid schema version */
#define SXM_GRID_TFILE_SCHEMA_VERSION (1U)

/** Packs the service schema and Grid own schema versions
 * into \c uint value
 * \param[in] _sver service schema version
 */
#define SXM_GRID_PACK_VERSION(_sver) \
    ((SXM_GRID_TFILE_SCHEMA_VERSION << 16) | (0xFFFF & (_sver)))

/** Extracts Grid own schema versions into \c uint value
 * \param[in] _v packed schema version from T-File
 */
#define SXM_GRID_EXTRACT_OWN_SCHEMA(_v) ((uint)((_v) >> 16))

/** Extracts service schema versions into \c uint value
 * \param[in] _v packed schema version from T-File
 */
#define SXM_GRID_EXTRACT_SERVICE_SCHEMA(_v) ((uint)((_v) & 0xFFFF))

/** Defines bit-size of the lat component inside the region id */
#define SXM_GRID_REGION_LAT_BITSIZE (5U)

/** Defines bit-size of the lon component inside the region id */
#define SXM_GRID_REGION_LON_BITSIZE (6U)

/** Pack lot and lat component in order to get corresponded region id
 * \param[in] _lon_comp lon component
 * \param[in] _lat_comp lat component
 */
#define SXM_GRID_REGION_PACK(_lon_comp, _lat_comp) \
    (((_lon_comp) << SXM_GRID_REGION_LAT_BITSIZE) | (_lat_comp))

/** Defines mix region id where offsets since -168.0/12.0 equal 0/0 */
#define SXM_GRID_REGION_ID_MIN (1)

/** Defines condition where no record exists in Grid Database for station ID */
#define SXM_GRID_DB_NO_RECORD (0xffffffff)

/** Defines max region id where offsets since -168.0/12.0 equal to 1 << 6/1 << 5 */
#define SXM_GRID_REGION_ID_MAX \
    ((1U << (SXM_GRID_REGION_LAT_BITSIZE + SXM_GRID_REGION_LON_BITSIZE)) - 1)

/** Defines helper macro to check validity of the region id
 * \param[in] _region region id to check
 */
#define SXM_GRID_REGION_IS_VALID(_region) \
    IS_IN_RANGE((_region), (int)SXM_GRID_REGION_ID_MIN, (int)SXM_GRID_REGION_ID_MAX)

/** Unpack lat component from region id
 * \param[in] _region region id
 */
#define SXM_GRID_REGION_LAT_UNPACK(_region) \
    (((1U << SXM_GRID_REGION_LAT_BITSIZE) - 1) & (_region))

/** Unpack lon component from region id
 * \param[in] _region region id
 */
#define SXM_GRID_REGION_LON_UNPACK(_region) \
    (((1U << SXM_GRID_REGION_LON_BITSIZE) - 1) & ((_region) >> SXM_GRID_REGION_LAT_BITSIZE))

/** Make unique id based on region and uid within the region
 * \param[in] _region region id
 * \param[in] _uid station id within the region
 */
#define SXM_GRID_ID_PACK(_region, _uid) ((uint)((_region) << 16) | (_uid))

/** Extract region id from unique id
 * \param[in] _id unique id
 */
#define SXM_GRID_UNPACK_REGION(_id) ((ushort)(((uint)(_id) >> 16) & 0xFFFF))

/** Extract UID from unique id
 * \param[in] _id unique id
 */
#define SXM_GRID_UNPACK_UID(_id) ((_id) & 0xFFFF)

/** Defines phone size in digits: \c 3 - area, \c 3 - exch, \c 4 - num*/
#define SXM_GRID_PHONE_DIGITLEN (10)

/** Defines phone size in bytes */
#define SXM_GRID_PHONE_BYTELEN ((SXM_GRID_PHONE_DIGITLEN * 4 /*bits*/) / 8)

/** Defines phone number format in accordance with the protocol spec */
#define SXM_GRID_PHONE_FORMAT ("(%03u)%03u-%04u")

/** Defines human readable phone len + nil */
#define SXM_GRID_PHONE_LENGTH   (14)

/** Defines number of regions to be simultaneously tracked by the grid */
#define SXM_GRID_NEEDED_REGIONS (4)

/** Defines invalid region index */
#define SXM_GRID_REGION_INVAL_IDX (-1)

/** Defines number of persistently stored AUs */
#define SXM_GRID_AUS_PER_REGION (16)

#define SXM_GRID_COMPLETE_AU_MASK(_x) (((_x) < SXM_GRID_AUS_PER_REGION)?((1U << (_x)) - 1):0xFFFF)

/** \name Defines POI extraction sources mask
 * @{
 */
#define SXM_POI_EXT_INACTIVE   (0) //!< Extraction is in active
#define SXM_POI_EXT_PPOI       (1) //!< PPOI extraction in progress
#define SXM_POI_EXT_EXTRACTION (2) //!< Regular extraction set in progress 
#define SXM_POI_EXT_GRID       (4) //!< Plain regions extraction
/** @} */

/** Cycle files name */
#define SXM_GRID_CFILE_NAME           "cycle"

/** \name SXMGridExtraction::flags valid masks
 * @{
 */
/** Bulk notify pending */
#define SXM_SDK_FLAG_NOTIFY         (1)
/** Currently in callback */
#define SXM_SDK_FLAG_INUSE          (2)
/** @} */

/** Defines the max au size supported by grid services */
#define SXM_GRID_AU_SIZE (5120U)

/** Defines the max au size supported by grid services */
#define SXM_GRID_DATA_EXPIRATION_TIMER (60U)

/** Contains a single incoming Access Unit */
typedef struct {
    /** Access Unit aligned by sector size */
    SXMSector data[(SXM_GRID_AU_SIZE + sizeof(SXMSector) - 1) / sizeof(SXMSector)];
} SXMGridAU;

/** Contains the location and signature of a saved Access Unit */
typedef struct {
    uint pktix;     //!< packet index
    byte *pkt;      //!< pointer to saved packet
    uint crc;       //!< packet CRC (signature)
    uint pl;        //!< packet length
    uint ts;        //!< time stamp this AU
} SXMGridSavedAU;

/** Represents all the information collection for a single grid square,
 * comprising the information from the baseline file plus any incoming
 * Access Units.
 */
typedef struct {
    ushort region;              //!< region number
    uint version;               //!< database version this region
    uint size;                  //!< number of bytes for this region
    uint *dbentry;              //!< index at start of region block
    void *base;                 //!< start of actual data in block
    SXMTFileChain *chain;       //!< Corresponded t-file chain instance
    SXMGridSavedAU safe[SXM_GRID_AUS_PER_REGION]; //!< saved AUs for this region
} SXMGridBaseLine;

/** Represents the state of an application-initiated extraction. */
typedef enum {
    SXM_GRID_EXT_STATE_IDLE = 0, //!< No active collection
    SXM_GRID_EXT_STATE_ACTIVE, //!< Collection is active
    SXM_GRID_EXT_STATE_EXTRACTING //!< Collection is being extracted
} SXMGridExtractionState;

/** Declare possible state combinations for Grid-based data services.
 * Current service state which is a combination of the following values:
 * - \p 0 - not started
 * - \p 1 - loaded
 * - \p 2 - timer
 * - \p 4 - request
 */
typedef enum {
    SXM_GRID_STATE_IDLE    = 0,
    SXM_GRID_STATE_LOADED  = 1,
    SXM_GRID_STATE_TIMER   = 2,
    SXM_GRID_STATE_REQUEST = 4,
    SXM_GRID_STATE_COMPLETE = (SXM_GRID_STATE_LOADED | SXM_GRID_STATE_REQUEST),
    SXM_GRID_STATE_READY   = (SXM_GRID_STATE_LOADED | SXM_GRID_STATE_TIMER),
    SXM_GRID_STATE_ALL     = (SXM_GRID_STATE_READY | SXM_GRID_STATE_REQUEST)
} SXMGridServiceStates;

/** RFD Update state
 * \warning the order of these values is important and shall not be
 *          changed without revising places where they are being used.
 */
typedef enum {
    SXM_GRID_RFD_STATE_IDLE = 0, //!< No active updates
    SXM_GRID_RFD_STATE_COLLECTING, //!< Update is being collected
    SXM_GRID_RFD_STATE_APPLYING //!< DB Update is in progress
} SXMGridRFDState;

#ifndef SDKFILES_STANDALONE_BUILD

/***************************************************************************//**
 *
 * Extraction notification callback
 *
 * The notification routine will be called whenever extraction data has changed
 *
 * \param[in] extHandle the handle for the extraction
 * \param[in] usercx the user context value from the original request
 *
 ******************************************************************************/
typedef void (*SXMGridExtCallback_t)(void *extHandle, void *usercx);

/** Represents the descriptor of an application-initiated extraction. */
typedef struct {
    pthread_mutex_t mutex;          //!< extraction lock
    uint flags;                     //!< see SXM_SDK_FLAG_ for values
    SXMGridExtractionState state;   //!< activity state of extraction
    SXMGridExtCallback_t notification; //!< user's notification
    void *usercx;                   //!< user's notification context
    SXMMBR mbr;                     //!< request MBR
    SXMMBR orig_mbr;                //!< Original collection mbr
    SXMPoint center;                //!< center point for distance sort
} SXMGridExtraction;

/** contains the location of an object in the extraction list, and a
 * distance metric from the sort point
 */
typedef struct {
    SXMPoint p;     //!< co-ordinates
    SXMFLOAT distance;     //!< distance from sort point in degrees-squared
} SXMGridLocation;

/** Defines service mask's mask */
#define SXM_GRID_PPOI_SERVICE_MASK (0xFF00)

/** Defines PPOI special mask */
#define SXM_GRID_PPOI_MASK (0x00FF)

/***************************************************************************//**
 * PPOI Extraction notification callback. The notification routine will be
 * called whenever PPOI extraction data has changed.
 * \param[in] usercx the user context value from the original request
 * \param[in] ppoi_user_flags bitwise-or of the user_flags fields of all the
 *                            PPOIs that changed
 ******************************************************************************/
typedef void (*SXMGridPPOIExtCallback_t)(void *usercx, byte ppoi_user_flags);

/** Declares helper type to defines qsort and qsearch callbacks */
typedef int (*SXMGridCompareCallaback_t)(const void*, const void*);

/** PPOI Handling structure */
typedef struct {
    pthread_mutex_t mutex;           //!< Lock extraction
    BOOL active;                     //!< Sign of the active ppoi feature
    BOOL dirty_flag;                 //!< true when ppoi station has changed
    SXMGridPPOIExtCallback_t notification;//!< PPOI change notification callback
    void *usercx;                    //!< User data passed to the notification callback
    byte user_flags;                 //!< Intermediate ppoi flags storage upon update
    uint mask[(SXM_PPOI_MAX + (SXM_ARCH_INT_BITS - 1)) / SXM_ARCH_INT_BITS];
    uint new_mask[(SXM_PPOI_MAX + (SXM_ARCH_INT_BITS - 1)) / SXM_ARCH_INT_BITS];

    uint item_size;          //!< Size of the single item in bytes
    uint capacity;           //!< Capacity of the fav storage
    uint *count;             //!< Current utilization of the fav storage
    ushort ppoispec_offset;  //!< Offset where the PPOISpec struct lays inside the item
    ushort data_offset;      //!< Offset where the fav's station data lays
    void *data;              //!< Fav data storage
    void *key;               //!< Fav item for searching needs
    SXMGridCompareCallaback_t index_compare_callback; //<! Index compare callback
    SXMGridCompareCallaback_t region_compare_callback; //<! Region compare callback
} SXMGridFavorites;

/** Declares single POI record extraction */
typedef int (*SXMGridPOIExtract_t)(void *service, void *ret,
                                   uint id, ushort flags, const void * rec);
/** POI Extraction Control Structure */
typedef struct {
    byte lat_idx_start; //!< LAT index of the first region in extraction
    byte lat_idx_end;   //!< LAT index of the last region in extraction
    byte lon_idx_start; //!< LON index of the last region in extraction
    byte lon_idx_width; //!< Width in regions of the area covered by MBR
    ushort region_index_extract;   //!< region iterator among covered by MBR
    ushort psuid_index_extract;    //!< PUID iterator within region
    SXMGridBaseLine grid_buff_poi; //!< Currently loaded region if it doesn't belong to needed ones
    SXMGridBaseLine* p_grid_base;  //!< Currently loaded region reference (either \p grid_buff_poi or from the grid)
    SXMMBR mbr;                    //!< POI extracted MBR
    ushort service_filter;         //!< Service's part of POI filter
    byte ppoi_filter;              //!< PPOI filter
    byte mask_src;                 //!< Extraction sources mask (working copy)
    byte mask_src_origin;          //!< Original extraction sources mask
    uint mask;                     //!< Variety of extraction masks
} SXMGridServicePoiExtract;

/** AU data */
typedef struct {
    uint new_group_au_flag;
    uint au_cnt;  // Received AUs. Each bit marks a processed AU.
    uint tot_au;
} SXMGridDataAU;

/** Declares RFD update qualification callback function */
typedef int (*SXMGridQualify_t)(void *, const SXMRfdMetadata *, uint *, uint *);

/** RFD processing control structure */
typedef struct {
    SXMGridRFDState state; //!< Update state
    uint version; //!< Currently collecting version
    /** Query the service to qualify the RFD update and in case of success
     * provide versions from the metadata record
     */
    SXMGridQualify_t qualify;
} SXMGridRFD;

/** Helper macro to provide grid service interface to grid APIs
 * \param[in] _s service structure for the particular
 */
#define SXM_2GRID(_s) ((_s) ? (&(_s)->grid) : NULL)

/** Declares the callback to be executed if the time shall be checked by
 * the service to invalidate expired data
 */
typedef void (*SXMGridCheckTimeStamp_t)(void*, const uint);
/** Declares replay callback */
typedef void (*SXMGridReplay_t)(void*);
/** Declares additional callback to be performed during reload-state routine */
typedef void (*SXMGridReloadState_t)(void*);
/** Declares extract callback function */
typedef int (*SXMGridExtract_t)(void *, void *, short, uint, int);
/** Declares data expiration callback */
typedef void(*SXMGridCheckDataExpiration_t)(void*);
/** Contains all the information required for the grid library to operate */
typedef struct {
    /** \name Data Service
     * @{
     */
    SXMDataService ds;          //!< Generic data service data
    ptr lli;                    //!< LLI handler
    pthread_mutex_t service_mutex; //!< Service level mutex
    void (*callback)(int, int); //!< Service level callback
    SXMTFile *dbfile;           //!< database (transaction) file
    ushort *dbroot;             //!< region index and count values
    uint *dbuser;               //!< user database parameters
    SXMGridServiceStates state; //!< Grid related service state
    uint sections_update;       //!< The sections of the cycle file that have changed.
    FILE *cycle;                //!< Cycle file
    SXMGridServicePoiExtract PoiExtract; //!< POI extraction descriptor
    SXMGridFavorites fav;       //!< Favorites handler
    SXMGridRFD rfd;             //!< RFD related data
    int db_version;             //!< Keeps current DB version

    /** Stations' related stuff */
    struct {
        uint min_id;            //!< Min station id
        uint item_size;         //!< Size of the single item in bytes
        uint capacity;          //!< Capacity of the service's storage in items
        uint *ids;              //!< Reference to service's storage for ids.
        byte *items;            //!< Reference to service's storage for meaningful items
        BOOL dirty_flag;        //!< true when station has changed
        BOOL au_flag;           //!< true when all AUs have been received
    } stations;

    SXMGridCheckDataExpiration_t data_exp_callback; // !<Timer callback which is used in to check the expiration of service dynamic data 
    SXMGridCheckTimeStamp_t check_timestamp; //!< Check timestamp routine called from service callback
    SXMDataServiceTimerCallback time_callback;//!< Timer callback which is used in case of missed time upon service subs notification.
    SXMGridReplay_t replay;     //!< AU replay routine
    SXMGridReloadState_t reload_state;//!< Optional callback to make data reload of the request
    SXMGridExtract_t extract;   //!< Specific extraction routine
    SXMGridPOIExtract_t poi_extract; //!< POI extraction routine
    /** @} */

    SXMGridExtraction ext;      //!< the single supported extraction

    SXMCFileRootBlock *root;    //!< cycle file root block
    ushort heap_sec;            //!< AU save area index
    SXMGridAU *heap;            //!< AU save area

    SXMGridBaseLine *bptr[SXM_GRID_NEEDED_REGIONS]; //!< set of loaded regions
    short *pheap;               //!< pointer to heap array of station indexes
    size_t heapsize;            //!< size of heap in bytes
    SXMGridLocation *ploc;      //!< pointer to array of station locations
    size_t locsize;             //!< size of locations in bytes
    int hc;                     //!< number of entries on heap
    int hstarts[SXM_GRID_NEEDED_REGIONS + 1]; //!< start markers for regions
    ushort sp;                    //!< count of stations (outside grid library)
    ushort needed[SXM_GRID_NEEDED_REGIONS]; //!< list of regions needed
    ushort centerix;              //!< index of region containing center point
} SXMGridService;

/** Defines function to merge one record */
typedef void (*SXMGridMergeRecord_t)(SXMGridService *, SXMTFileStream *, const char *);

/** Defines common part for service's updater */
typedef struct {
    SXMTFile *dbfile; //!< T-File structure
    ushort *root; //!< Quick accessor to t-file root
    SXMBitBuff *gzb; //!< Bit-buffer attached to uncompressed data
    uint maxs; //!< Max station inside the region
    SXMGridBaseLine base; //!< Currently loaded for update region
    SXMTFileStream *stream; //!< Currently used t-file stream
    uint *del; //!< Deleted items map
    uint *nw; //!< New/Updated items map
    uint *index_area; //!< Index area
    ushort region_size; //!< Region size on station including
    ushort map_size;    //!< Maps \p del and \p nw size in bytes
    /** Callback to performs copy operation over single record */
    SXMGridMergeRecord_t merge_record;
} SXMGridUpdate;

extern SXMResultCode sxm_grid_heap_alloc(SXMGridService *, size_t);

extern void sxm_grid_heap_free(SXMGridService *);

extern SXMResultCode sxm_grid_aucheck(SXMGridService *,
                                      uint,
                                      SXMPacket *,
                                      uint);

extern int sxm_grid_baseix(SXMGridService *,
                           ushort);

extern SXMResultCode sxm_grid_database_restart(SXMGridService *,
                                     const char *,
                                     const char *,
                                     uint);

extern SXMResultCode sxm_grid_database_get_version(SXMGridService *,
                                         uint *);

extern SXMResultCode sxm_grid_database_open(SXMGridService *,
                                  SXMTFile **,
                                  const char *,
                                  const char *,
                                  const char *,
                                  const char *,
                                  uint);

extern SXMResultCode sxm_grid_database_start(SXMGridService *,
                                   const char *,
                                   const char *,
                                   const char *,
                                   uint);

extern SXMResultCode sxm_grid_load_region(SXMGridService *,
                                SXMGridBaseLine *,
                                ushort,
                                BOOL);

extern void sxm_grid_decode_phone(SXMByteBuff *, char *, size_t);

extern void sxm_grid_encode_phone(byte *, ushort, ushort, ushort);

/** \name Service level functions
 * @{
 */
extern int sxm_grid_init(SXMGridService *, void**, size_t);

extern int sxm_grid_cycle_destroy(SXMGridService *, const void*);

extern void sxm_grid_rfd_handle_metadata(SXMGridService *, SXMBitBuff *, SXMPacket *);

extern void sxm_grid_notify_new_data(SXMGridService *);

/** @} */

/** \name Request manipulations
 * @{
 */
extern void* sxm_grid_get_poi_data(SXMGridService *, uint, int);

extern void sxm_grid_reload_state(SXMGridService *);

extern int sxm_grid_request(SXMGridService *, const SXMMBR *,
                            const SXMPoint *, SXMGridExtCallback_t,
                            void *, SXMGridExtraction **);

extern int sxm_grid_modify(SXMGridService *, SXMGridExtraction *,
                           const SXMMBR *, const SXMPoint *);

extern int sxm_grid_remove(SXMGridService *, SXMGridExtraction *);

extern void sxm_grid_delete_request(SXMGridService *);
/** @} */

/** \name Extractions
 * @{
 */
extern int sxm_grid_begin(SXMGridService *, SXMGridExtraction *, const SXMPoint *);

extern int sxm_grid_extract(SXMGridService *, SXMGridExtraction *,
                            void *, size_t);

extern int sxm_grid_end(SXMGridService *, SXMGridExtraction *);
/** @} */

/** \name POI/PPOI APIs
 * @{
 */
extern int sxm_grid_set_ppoi_list(SXMGridService *,
                                  const uint, const PPOISpec *,
                                  SXMGridPPOIExtCallback_t, void *);

extern void* sxm_grid_is_poi_from_favorite_list(SXMGridService *, uint);

extern int sxm_grid_is_favorite_region(SXMGridService *, int);

extern int sxm_grid_poi_read_location(SXMGridService *, SXMPoint *,
                                      const SXMGridBaseLine *, ushort);

extern int sxm_grid_poi_get_station(SXMGridService *, uint *,
                                    void **, const PPOISpec **,
                                    byte *);

extern void sxm_grid_poi_next(SXMGridService *, BOOL,
                              SXMGridServicePoiExtract *, uint, uint);

extern int sxm_grid_poi_begin(SXMGridService *, const SXMMBR *, ushort);

extern int sxm_grid_poi_extract_station(SXMGridService *, void *);

extern int sxm_grid_poi_end(SXMGridService*);
/** @} */

/** \name Grid service updater
 * @{
 */
extern int sxm_grid_update_init(SXMGridService *, SXMGridUpdate *, ushort, SXMRfdMetadata *);

extern int sxm_grid_update_uninit(SXMGridService *, SXMGridUpdate *);

extern int sxm_grid_merge_data(SXMGridService *, SXMGridUpdate *, uint, uint);

extern int sxm_grid_stream_commit(SXMGridService *, SXMGridUpdate *, uint);
/** @} */

/** \name Grid Utility Functions
 * @{
 */
extern int sxm_grid_get_mbr(SXMGridService *, SXMGridExtraction *, SXMMBR *);

extern int sxm_grid_point_closest(SXMGridService *, const SXMPoint *, SXMFLOAT *);

extern int sxm_grid_point_furthest(SXMGridService *, const SXMPoint *, SXMFLOAT *);

extern int sxm_grid_set_auto_save(SXMGridService *, BOOL);

extern int sxm_grid_save_data(SXMGridService *, int, int, byte, SXMIOStat *);

extern int sxm_grid_set_debug(SXMGridService *, int);

extern int sxm_grid_get_debug(SXMGridService *, int *);
/** @} */

/** \name Updater
 * {@
 */
extern void sxm_grid_rfd_sync(SXMGridService *);
/** @} */

/** \name Grid SDK functions
 * @{
 */
extern int sxm_grid_lli_start(SXMGridService *);

extern int sxm_grid_lli_stop(SXMGridService *);

extern int sxm_grid_lli_rfd_add(SXMGridService *, const void *, int);

extern FILE *sxm_grid_lli_rfd_open(SXMGridService *, SXMRfdMetadata *, int);
/** @} */

#endif /* SDKFILES_STANDALONE_BUILD */

/************************************************************************
 *                                                                      *
 *            State Grid-Based Services                                 *
 *            =========================                                 *
 *                                                                      *
 *    These are common routines for the services that have a baseline   *
 *    file (and corresponding transaction file) arranged by state.      *
 *    These services include:                                           *
 *        movies                                                        *
 *                                                                      *
 ************************************************************************/

/** Defines max number of states */
#define STATEGRID_MAX_STATE     (127)
/** Defines max number of AUs per state to be stored in c-file */
#define STATEGRID_MAX_AU_PER_STATE  (32)
/** Defines max number of states that can be cached */
#define STATEGRID_MAX_STATE_COUNT   (4)

/** Defines the State Grid schema version */
/// \brief Defines the State Grid schema version
#define STATEGRID_TFILE_SCHEMA_VERSION (1)

/** Packs the service schema and Grid own schema versions
 * into \c uint value
 * \param[in] _sver service schema version
 */
#define STATEGRID_PACK_VERSION(_sver) \
    ((STATEGRID_TFILE_SCHEMA_VERSION << 16) | (0xFFFF & (_sver)))

/** Extracts Grid own schema versions into \c uint value
 * \param[in] _v packed schema version from T-File
 */
#define STATEGRID_EXTRACT_OWN_SCHEMA(_v) \
    ((uint)((_v) >> 16))

/** Extracts service schema versions into \c uint value
 * \param[in] _v packed schema version from T-File
 */
#define STATEGRID_EXTRACT_SERVICE_SCHEMA(_v) \
    ((uint)((_v) & 0xFFFF))

/** Make unique id based on state and uid within the state
 * \param[in] _region
 * \param[in] _uid
 */
#define SXM_STATEGRID_ID_PACK(_region, _uid) \
    (((_region) << 16) | (_uid))

/** Extract state id from unique id
 * \param[in] _id unique id
 */
#define SXM_STATEGRID_UNPACK_STATE(_id) \
    (((uint)(_id) >> 16) & 0xFFFF)

/** Extract UID from unique id
 * \param[in] _id unique id
 */
#define SXM_STATEGRID_UNPACK_UID(_id) \
    ((_id) & 0xFFFF)

/** Represents all the information collection for a single stategrid state,
 * comprising the information from the baseline file plus any incoming
 * Access Units.
 */
typedef struct
{
    int region;                                         //!< State number (0 .. 127)
    uint *dbentry;                                      //!< index at start of region block
    byte *base;                                         //!< start of actual data in block
    SXMTFileChain *chain;                               //!< Corresponded t-file chain instance
    SXMGridSavedAU safe[STATEGRID_MAX_AU_PER_STATE];    //!< saved AUs for this region
} SXMStateGridBaseLine;

#ifndef SDKFILES_STANDALONE_BUILD

/** Contains all the information required for the stategrid library to operate */
typedef struct
{
    SXMGridExtraction ext;                                 //!< the single supported extraction

    SXMTFile *dbfile;                                      //!< database (transaction) file
    ushort *dbroot;                                        //!< region index and count values
    uint *dbuser;                                          //!< user database parameters

    SXMCFileRootBlock *root;                               //!< cycle file root block
    ushort heap_sec;                                       //!< AU save area index
    SXMGridAU *pAU;                                        //!< AU save area

    SXMStateGridBaseLine *bptr[STATEGRID_MAX_STATE_COUNT]; //!< up to 4 loaded states
    short *pheap;                                          //!< pointer to array of sorted site entries
    size_t heapsize;                                       //!< size of heap in bytes
    SXMGridLocation *ploc;                                 //!< pointer to array of nearby sites
    size_t locsize;                                        //!< size of locations in bytes
    int hc;                                                //!< number of entries on heap
    int hstarts[STATEGRID_MAX_STATE_COUNT+1];              //!< start markers for states
    uint sp;                                               //!< count of sites (outside grid library)

    uint st_req[4];                                        //!< List of required states for mbr (4x32 bits)
} SXMStateGridService;

extern int sxm_stategrid_heap_alloc(SXMStateGridService *, size_t);

extern void sxm_stategrid_heap_free(SXMStateGridService * );

extern void sxm_stategrid_heap_insert(SXMStateGridService *grid,
                                      short rn);

extern short sxm_stategrid_heap_extract(SXMStateGridService *grid);

extern void sxm_stategrid_add_location(SXMStateGridService *grid,
                                       SXMFLOAT lon,
                                       SXMFLOAT lat);

extern void sxm_stategrid_heap_rebuild(SXMStateGridService *grid,
                                       SXMPoint *center);

extern int sxm_stategrid_baseix(SXMStateGridService *grid,
                                int stateid);

extern int sxm_stategrid_aucheck(SXMStateGridService *grid,
                                 int hix,
                                 SXMPacket *fp,
                                 int aucnt);

extern int sxm_stategrid_get_slot(SXMStateGridService *grid);

extern int sxm_stategrid_reload(SXMStateGridService *grid);

extern int sxm_stategrid_state_has_data(SXMStateGridService *grid, int state);

extern void sxm_stategrid_invalidate(SXMStateGridService *grid);

extern int sxm_stategrid_total_count(SXMStateGridService *grid);

extern void sxm_stategrid_load_region(SXMStateGridService *grid,
                                      SXMStateGridBaseLine *bl,
                                      int region,
                                      BOOL bChainNeeded);

extern int sxm_stategrid_database_open(SXMTFile **ppTFile,
                                       const char *service,
                                       const char *dbfile,
                                       const char *mode,
                                       const char *format,
                                       uint *schema);

extern int sxm_stategrid_database_start(SXMStateGridService *grid,
                                        const char *service,
                                        const char *dbfile,
                                        const char *format,
                                        uint *schema);

extern int sxm_stategrid_database_restart(SXMStateGridService *grid,
                                          const char *service,
                                          const char *dbfile,
                                          uint *schema);

extern int sxm_stategrid_database_stop(void);

#endif /* SDKFILES_STANDALONE_BUILD */

#endif /* SXM_GRID_H */
