/******************************************************************************/
/*                Copyright (c) Sirius XM Satellite Radio, Inc.               */
/*                            All Rights Reserved                             */
/*      Licensed Materials - Property of Sirius XM Satellite Radio, Inc.      */
/******************************************************************************/
/***************************************************************************//**
 *
 * \file sxm_fixpoint.c
 * \author Leslie French
 * \date 8/20/2013
 * \brief FixPoint(1,16,15) arithmetic. Please, refer to \ref fixpoint_page for
 *        implementation details.
 *
 * \page fixpoint_page FixPoint arithmetic
 *
 * \section fixpoint_page_intro Introduction
 *
 * The fixpoint datatype avoids the need for floating-point arithmetic within
 * the SXe SDK. The fixpoint values range from -65536 to +65536 in steps of
 * 1/32768 (0.00003). The most common use of fixpoint values in the SXe SDK is
 * to hold latitude and longitude values, in degrees.  
 *
 * \section fixpoint_page_bindata Binary Data Conversion
 * 
 * For systems that do implement floating point values, the conversion to and
 * from fixpoint is easy:
 * ~~~{.c}
 * fix x = (int)((double)y * 32768.0);
 * double y = (fix)x / 32768.0;
 * ~~~
 *
 * Fixpoint values cannot be transformed to and from 32-bit float values without
 * loss of precision. Two macros provide conversion from integer values:
 * - \ref sxm_fix_int converts an integer into a fixpoint value, and
 * - \ref sxm_fix_real converts a integer + fractional part to a fixpoint:
 *
 * ~~~{.c}
 * fix x = sxm_fix_int(intpart);
 * fix x = sxm_fix_real(intpart, fracpart);
 * ~~~
 *
 * \note Applications should use the SXMFLOAT generic interface when passing
 *       parameters into the SXe SDK. The fix values are used internally and
 *       the external baseline files use fix for data storage.
 *
 ******************************************************************************/

#include <util/sxm_common_internal.h>
#include <util/sxm_noname_internal.h>
#if (SXM_USE_FIX == 0)
#include <math.h>
#endif

/** Cosine lookup table */
static fix ctable[] = {
    32768, 32763, 32748, 32723, 32688, 32643, 32588, 32524,
    32449, 32365, 32270, 32166, 32052, 31928, 31795, 31651,
    31499, 31336, 31164, 30983, 30792, 30592, 30382, 30163,
    29935, 29698, 29452, 29197, 28932, 28660, 28378, 28088,
    27789, 27482, 27166, 26842, 26510, 26170, 25822, 25466,
    25102, 24730, 24351, 23965, 23571, 23170, 22763, 22348,
    21926, 21498, 21063, 20622, 20174, 19720, 19261, 18795,
    18324, 17847, 17364, 16877, 16384, 15886, 15384, 14876,
    14365, 13848, 13328, 12803, 12275, 11743, 11207, 10668,
    10126,  9580,  9032,  8481,  7927,  7371,  6813,  6252,
     5690,  5126,  4560,  3993,  3425,  2856,  2286,  1715,
     1144,   572,     0
};

#if SXM_USE_FIX == 1
/** Length of 1 arc-degree of latitude in metric */
#define RADIUS_M  sxm_fix_real(111, 31884)
/** Length of 1 arc-degree of latitude in imperial */
#define RADIUS_I  sxm_fix_real(69, 17032)
#else
/** Length of 1 arc-degree of latitude in metric */
#define RADIUS_M  (111.31884)
/** Length of 1 arc-degree of latitude in imperial */
#define RADIUS_I  (69.17032)
#endif

#if !SXM_USE_FIX
/** Converts degrees into radians (only in float mode)
 *  (the fixpoint cosine function works in arc-degrees)
 */
#define RAD(_a) ((_a)*0.01745329)
#endif

/***************************************************************************//**
 * Splits \ref fix number into parts
 * 
 * \param[in] in value to split
 * \param[out] s value sign (1 - positive, -1 - negative)
 * \param[out] i integer part of the value always positive
 * \param[out] f fractional part of the value always positive
 *
 ********************************************************************************/
 static void split(fix in, int *s, int *i, int *f) {
    *s = 1;

    if  (in < 0) {
        *s = -1;
        in = -in;
    }
    *i = (in >> 15U);
    *f = (in & 0x7FFF);
}

/***************************************************************************//**
 * Quick accessor to SDK SXe measurement units
 * 
 * \return Current measurement unit
 ********************************************************************************/
static SXMSdkUnit fixpoint_unit(void) {
    SXMSdkConfig config;
    config.flags = SXM_SDK_CONFIG_UNIT;
    config.unit = SXM_SDK_UNIT_IMPERIAL; // Kind of default is function fails
    sxm_sdk_get_config(&config);
    return config.unit;
}

/***************************************************************************//**
 * Converts a integer + fractional part to a fixpoint.
 *
 * The sign of the result is the sign of int part (\c _i). The value of
 * frac part (\c _j)must be a positive integer in the range 0  99999. Note that
 * 0.5 is presented as sxm_fix_real(0, 50000), not sxm_fix_real(0,5) which is
 * actually only 0.00005
 *
 * \param[in] _i intr part
 * \param[in] _j frac part
 *
 * \return fixpoint value
 *
 ********************************************************************************/
SXESDK_API int sxm_fix_real(int _i, uint _j)
{
    return ((_i) < 0 ? -1 : 1) * ((((_i) < 0 ? -(_i) : (_i)) << 15U) +  (int)(((_j) << 15U)/100000));
}

/***************************************************************************//**
 * The fixpoint cosine of the input value.
 *
 *  note The input value for \a sin is in degrees (not radians) since this is 
 *       the most common input format for trigonometric calculations in 
 *       the SXe SDK.
 *
 * \param[in] in input in degrees, not radians
 *
 * \return calculated value
 *
 ********************************************************************************/
 SXESDK_API fix sxm_fix_cos(fix in) {
    int sign = 1;
    int i, f, s;

    if  (in < 0)
        in = -in;
    in = in % (360 << 15U);
    if  (in > (180 << 15U))
    in = (360 << 15U) - in;
    if  (in >= (90 << 15U)) {
        sign = -1;
        in = (180 << 15U) - in;
    }
    if  (in <= 0)
        return sign * (1 << 15U);

    split(in, &s, &i, &f);
    if  (i >= 90)
        return 0;
    return sign * (ctable[i] + ((ctable[i+1]-ctable[i])*f >> 15U));
}

/***************************************************************************//**
 * The fixpoint sine of the input value
 *
 * \note The input value for \a sin is in degrees (not radians) since this is 
 *       the most common input format for trigonometric calculations in 
 *       the SXe SDK.
 *
 * \param[in] in input in degrees
 *
 * \return calculated value
 *
 ********************************************************************************/
SXESDK_API fix sxm_fix_sin(fix in)
{
    return sxm_fix_cos(in - (90 << 15U));
}

/***************************************************************************//**
 * The fixpoint multiplication
 *
 * \param[in] a the first operant 
 * \param[in] b the second operant 
 *
 * \return computation result
 *
 ********************************************************************************/
 SXESDK_API fix sxm_fix_mul(fix a, fix b) {
    int s1, i1, f1, s2, i2, f2;

    split(a, &s1, &i1, &f1);
    split(b, &s2, &i2, &f2);

    return (s1*s2)*((i1*i2 << 15U) + (i1*f2 + i2*f1) + (f1*f2 >> 15U));
}

/***************************************************************************//**
 * Simulates floating point division for Fixed point systems.
 *
 * \param[in] a Numerator
 * \param[in] b Denominator
 *
 * \return fixed value containing the formatted result of the division or 
 *         0 - if Denominator is 0.
 *
 ********************************************************************************/
 SXESDK_API fix sxm_fix_div(fix a, fix b)
 {
     uint remainder = (a >= 0) ? (uint)a : (uint)(-a);
     uint divider =   (b >= 0) ? (uint)b : (uint)(-b);
     fix result;

     uint quotient = 0;
     uint bit = 1 << 15U;

     // This uses the basic binary restoring division algorithm.
     // It appears to be faster to do the whole division manually than
     // trying to compose a 64-bit divide out of 32-bit divisions on
     // platforms without hardware divide.

     if (b == 0)
         return 0;

     /* The algorithm requires D >= R */
     while (divider < remainder) {
         divider <<= 1U;
         bit <<= 1U;
     }

     if (divider & 0x80000000) {
         // Perform one step manually to avoid overflows later.
         // We know that divider's bottom bit is 0 here.
         if (remainder >= divider) {
             quotient |= bit;
             remainder -= divider;
         }
         divider >>= 1U;
         bit >>= 1U;
     }

     /* Main division loop */
     while (bit && remainder) {
         if (remainder >= divider) {
             quotient |= bit;
             remainder -= divider;
         }

         remainder <<= 1U;
         bit >>= 1U;
     }	 

     result = (fix)quotient;

     /* Figure out the sign of result */
     if (((uint)a ^ (uint)b) & 0x80000000U) {
         result = -result;
     }

     return result;
 }

/***************************************************************************//**
 * The routine reverses \ref sxm_fix_real to yield am integer part and a
 * fractional part in the range 099999.
 *
 * \note The last digit of frac part is only accurate to 0.00003
 * 
 * \param[in] in the fixpoint value to be converted.
 * \param[in] i the integer part of the fixpoint.
 * \param[in] f the fractional part of the fixpoint.
 *
 ********************************************************************************/
 SXESDK_API void sxm_fix_printable(fix in, int *i, int *f)
 {
    int s1, i1, f1;

    split(in, &s1, &i1, &f1);

    *i = s1 * i1;
    *f = (int)(((uint)f1 * 100000U) >> 15U);
    if ( (*i == 0) && (s1 == -1) )
        *f = -(*f);
}

/***************************************************************************//**
 * The routine converts a fixpoint value into a printable string.
 *
 * The value is printed to 5 decimal places, and the routine is suitable for
 * embedding directly in a call to printf (etc.), for example:
 * ~~~{.c}
 * char blon[16], blat[16];
 * fix lon, lat;
 * set lon and lat values
 * printf("Point value is [%s, %s]\n",
 *        sxm_fix_sprint(blon, lon), sxm_fix_sprint(blat, lat));
 * ~~~
 *
 * \param[in,out] b	a pointer to a buffer that is used for the string
 * \param[in] in the fixpoint value to be converted
 *
 * \return a pointer to the start of the character string (i.e. the value of \c b)
 *
 ********************************************************************************/
 SXESDK_API char *sxm_fix_sprint(char *b, fix in)
 {
    int i, f;

    sxm_fix_printable(in, &i, &f);

    if(f < 0)
    {
        sprintf(b, "-0.%05d", -f);
    }
    else
    {
        sprintf(b, "%d.%05d", i, f);
    }
    return b;
}

/***************************************************************************//**
 * Reads the fixpoint value from the null-terminates string.
 * 
 * The string representation will be converted into he nearest \ref fix value.
 * 
 * A character string in the form:
 * ~~~~
 * ['-'] {zero or more digits} ['.' {zero or more digits} ]
 * ~~~~
 *
 * The input pointer is updated to point past the last character consumed by
 * the routine. The reader automatically handles varying numbers of decimal
 * places, so "2.5" and "2.50" and "2.500000" will all generate the same value.
 * Digits beyond the precision of a fixpoint value will be discarded
 *
 * \param[in,out] inp pointer to a character pointer
 * 
 * \return the fixpoint value represented by the input
 *
 ********************************************************************************/
SXESDK_API fix sxm_fix_read(const char **inp)
{
    uint i = 0, f = 0;
    uint dp = 1;
    int sign = 1;
    const char *c = *inp;

    //  sign
    if  (*c == '-') {
        c++;
        sign = -1;
    }

    // integer part
    while (*c >= '0' && *c <= '9') 
        i = i * 10U + (uint)(*c++ - '0');

    //  fraction
    if (*c == '.') {
        c++;

        while (*c >= '0' && *c <= '9') {
            f = f * 10U + (uint)(*c++ - '0');
            dp *= 10U;
            if (dp == 100000U)
            break;
        }
    }

    //  discard extra-precision digits

    while  (*c >= '0' && *c <= '9')
        c++;

    *inp = c;
    return (fix)sign * (fix)((i << 15U) + ((f << 15U) / dp));
}

/***************************************************************************//**
 * Simulates floating point root calculation on fix numbers.
 *
 * The square root of x/32768 is root(2x/65536) or root(2x)/256
 * converting back to fix, that's 128*root(2x)/32768.
 *
 * So root(r) is isqrt(x << 1) << 7 where isqrt is the integer 
 * square root function
 *
 * \param[in] r Number to calculate root of.
 *
 * \return result in fix point, or 0 if r = 0.
 *
 ********************************************************************************/
static fix sxm_fix_iroot(fix r)
{
    uint root;      // holds isqrt(r)
    uint copy;

    //  approximate isqrt(r) by shifting r log2(r)/2 bits to the right
    root = (uint)r;
    copy = (uint)r >> 2U;
    while (copy > 0) {
        root >>= 1U;
        copy >>= 2U;
    }

    // do not divide by 0 !
    if (root != 0)
    {
        //  then use the iteration formula 4 times, returning the last iteration
        root = (root + (uint)r/root)/2U;
        root = (root + (uint)r/root)/2U;
        root = (root + (uint)r/root)/2U;
        root = (root + (uint)r/root)/2U;
    }
    else
    {
        root = 0;
    }
    
    return (fix)root;
}

/***************************************************************************//**
 * The fixpoint square root of the input value
 *
 * \param[in] r Number to calculate root of
 *
 * \return result in fix point, or 0 if r = 0.
 *
 ********************************************************************************/
SXESDK_API fix sxm_fix_root(fix r)
{
    if (r <= 0)
        return 0;

    //  if the number is small enough we can gain an extra 4 bits of precision
    if  (r > (8 << 15U))
        return sxm_fix_iroot(r << 1U) << 7U;
    else
        return sxm_fix_iroot(r << 9U) << 3U;
}

#if SXM_USE_FIX
/***************************************************************************//**
 * Calculates the 'squared degrees' value which is monotonic wrt distance
 * scaled by 128 to preserve accuracy at smaller distances.
 *
 * \param[in] lon0 longitude for the first point
 * \param[in] lat0 latitude for the first point
 * \param[in] lon0 longitude for the second point
 * \param[in] lat0 latitude for the second point.
 *
 * \return computation result as fixpoint value.
 *
 ********************************************************************************/
 static fix sxm_fix_distance2(fix lon0, fix lat0, fix lon1, fix lat1)
 {
    fix dlon = abs(lon0 - lon1);
    fix dlat = abs(lat0 - lat1);

    if  (dlon >= 32768 || dlat >= 32768) {
        fix dlon2 = sxm_fix_mul(dlon, dlon);
        fix cl = sxm_fix_mul(sxm_fix_cos(lat0), sxm_fix_cos(lat1));

        // multiply by 128
        return (sxm_fix_mul(dlat, dlat) + sxm_fix_mul(dlon2, cl)) << 7U;
    } else {
        fix dl2 = dlat*dlat;
        fix cor1 = (dlon * sxm_fix_cos(lat0)) >> 15U;
        fix cor2 = (dlon * sxm_fix_cos(lat1)) >> 15U;

        //multiply by 128, then divide by 32768 (shift 15-7 = 8)
        return (dl2 + cor1*cor2) >> 8U;
    }
}

/***************************************************************************//**
 * Calculates true distance in miles or km
 *
 * ~~~
 * distance = Rfix . sqrt ( dlat^2 + cos(lat0)*cos(lat1)*dlon^2 )
 * ~~~
 * where Rfix is radius * PI / 180.0 i fixpoint (about 111.32km)
 *             :  don't forget to undo the 128 scale factor introduced above
 * \param[in] distance2 a value previously returned by the sxm_fix_distance2 routine 
 *
 * \return computation result as fixpoint value.
 *
 ********************************************************************************/
 static fix sxm_fix_distance1(fix distance2)
 {
    const fix radius =
        (fixpoint_unit() == SXM_SDK_UNIT_METRIC)  ? RADIUS_M : RADIUS_I;

    if  (distance2 >= (32768*128))
        //  divide by 128 back to a true fixpoint
        return sxm_fix_mul(radius, sxm_fix_root(distance2 >> 7U));
    else 
        //  multiply by 256 to get 'squared-fixpoint' for integer root
        return sxm_fix_mul(radius, sxm_fix_iroot(distance2 << 8U));
}
#endif

/***************************************************************************//**
 * The sxm_fixmbr_inside routine tests if a point lies within an MBR (fixpoint
 * version)
 *
 * \param[in] mbr a pointer to the MBR data structure
 * \param[in] point	a pointer to the location to be tested
 *
 * \retval 1 if the point lies within the rectangle
 * \retval 0 otherwise
 *
 ********************************************************************************/
SXESDK_API int sxm_fixmbr_inside(const SXMFixMBR *mbr, const SXMFixPoint *point)
{
    return (point->lon > mbr->ll.lon) && (point->lon < mbr->ur.lon) &&
           (point->lat > mbr->ll.lat) && (point->lat < mbr->ur.lat);
}

/***************************************************************************//**
 * The tests if there is any overlap between two MBRs
 *
 * \param[in] self a pointer to one of the Map Regions
 * \param[in] other a pointer to the other Map Region
 *
 * \retval 1 if there is any overlap between the two MBRs
 * \retval 0 otherwise
 *
 ********************************************************************************/
SXESDK_API int sxm_fixmbr_intersects(const SXMFixMBR *self, const SXMFixMBR *other)
{
    return (other->ll.lon <= self->ur.lon) &&
           (other->ll.lat <= self->ur.lat) &&
           (other->ur.lon >= self->ll.lon) &&
           (other->ur.lat >= self->ll.lat);
}

/***************************************************************************//**
 * The routine calculates an MBR as a rectangle extending a fixed distance
 * from a center-point (fixpoint version)
 *
 * The MBR is initialized such that the boundary is at least miles from the
 * center point, in both latitude and longitude. The extent in degrees of
 * longitude will always exceed the extent in latitude because of the curvature
 * of the earth.
 *
 * \param[out] self a pointer to the MBR structure to be initialized
 * \param[in] center a pointer to the center point of the area
 * \param[in] miles	the distance from the center point 
 *
 ********************************************************************************/
 SXESDK_API void sxm_fixmbr_about(SXMFixMBR *self, const SXMFixPoint *center, fix miles)
 {
    fix dlat = sxm_fix_mul(miles, 474);
    fix dlon = sxm_fix_div(dlat, sxm_fix_cos(center->lat));

    self->ll.lon = center->lon - dlon;
    self->ll.lat = center->lat - dlat;
    self->ur.lon = center->lon + dlon;
    self->ur.lat = center->lat + dlat;
}

/***************************************************************************//**
 * The function calculates the center of the MBR based on the upper right and
 * lower left points of the MBR supplied (fixpoint version)
 *
 * \param[in] mbr Pointer to the populated SXMFixMBR structure 
 * \param[out] center Pointer to the populated SXMFixPoint structure containing
 *                    the center of the MBR
 *
 ********************************************************************************/
 SXESDK_API void sxm_fixmbr_center(const SXMFixMBR *mbr, SXMFixPoint *center)
{
    center->lat = sxm_fix_div(mbr->ll.lat + mbr->ur.lat, (fix)65536);
    center->lon = sxm_fix_div(mbr->ll.lon + mbr->ur.lon, (fix)65536);
}

/************************************************************************
 *                                                                      *
 *            SXMFLOAT routines                                         *
 *                                                                      *
 *    These compile to either fix or float operations                   *
 *                                                                      *
 ************************************************************************/

/** Service region is the valid region over which we are prepared
 *  to consider making collections and extractions.              
 */
static SXMMBR service_region =  {
#if SXM_USE_FIX
     {-(180 << 15U), (0 << 15U)}, {(0 << 15U), (90 << 15U)} 
#else
     {-180.0, 0.0}, {0.0, 90.0} 
#endif
 };

/***************************************************************************//**
 * The routine tests if a point lies within an MBR.
 *
 * \param[in] mbr a pointer to the MBR data structure
 * \param[in] point	a pointer to the location to be tested
 *
 * \retval 1 if the point lies within the rectangle
 * \retval 0 otherwise
 *
 ********************************************************************************/
SXESDK_API int sxm_mbr_inside(const SXMMBR *mbr, const SXMPoint *point)
{
    return ((point->lon > mbr->ll.lon) && (point->lon < mbr->ur.lon) &&
            (point->lat > mbr->ll.lat) && (point->lat < mbr->ur.lat)) ? 1 : 0;
}

/***************************************************************************//**
 * The routine tests if a one MBT lies within another one MBR.
 *
 * \param[in] mbr a pointer to the MBR data structure
 * \param[in] inner_mbr	a pointer to the MBR to be tested
 *
 * \retval 1 if the second (inner) MBR lies within the first one
 * \retval 0 otherwise 
 *
 ********************************************************************************/
SXESDK_API int sxm_mbr_inside_mbr(const SXMMBR *mbr, const SXMMBR *inner_mbr)
{
    return sxm_mbr_inside(mbr, &inner_mbr->ll) &&
           sxm_mbr_inside(mbr, &inner_mbr->ur);
}

/***************************************************************************//**
 * The sxm_fixmbr_intersects tests if there is any overlap between two MBRs.
 *
 * \param[in] self a pointer to one of the Map Regions
 * \param[in] other a pointer to the other Map Region
 *
 * \retval 1 if there is any overlap between the two MBRs
 * \retval 0 otherwise
 *
 ********************************************************************************/
SXESDK_API int sxm_mbr_intersects(const SXMMBR *self, const SXMMBR *other) {
    return (other->ll.lon <= self->ur.lon &&
            other->ll.lat <= self->ur.lat &&
            other->ur.lon >= self->ll.lon &&
            other->ur.lat >= self->ll.lat);
}

/***************************************************************************//**
 * The sxm_mbr_intersects_get tests if there is any overlap between two MBRs
 * and if it exists - calculate MBR which represents overlapped area.
 *
 * \param[in] self a pointer to one of the Map Regions
 * \param[in] other a pointer to the other Map Region
 * \param[out] other a pointer to the region which represents intersection area.
 *
 * \retval not-0 if there is any overlap between the two MBRs
 * \retval 0 otherwise
 *
 ********************************************************************************/
extern SXESDK_API int sxm_mbr_intersects_get(const SXMMBR *self,
                                             const SXMMBR *other,
                                             SXMMBR *inter)
{
    if (sxm_mbr_intersects(self, other))
    {
        inter->ll.lat = MAX(self->ll.lat, other->ll.lat);
        inter->ll.lon = MAX(self->ll.lon, other->ll.lon);
        inter->ur.lat = MIN(self->ur.lat, other->ur.lat);
        inter->ur.lon = MIN(self->ur.lon, other->ur.lon);
        return !0;
    }
    return 0;
}

/***************************************************************************//**
 * The routine calculates an MBR as a rectangle extending a fixed distance
 * from a center-point.
 *
 * The MBR is initialized such that the boundary is at least miles from the
 * center point, in both latitude and longitude. The extent in degrees of
 * longitude will always exceed the extent in latitude because of the curvature
 * of the earth.
 *
 * \param[out] self a pointer to the MBR structure to be initialized
 * \param[in] center a pointer to the center point of the area
 * \param[in] miles	the distance from the center point 
 *
 ********************************************************************************/
SXESDK_API void sxm_mbr_about(SXMMBR *self, const SXMPoint *center, SXMFLOAT miles)
{
    const SXMSdkUnit unit = fixpoint_unit();
    SXMFLOAT dlon, dlat;

#if SXM_USE_FIX
    if (unit == SXM_SDK_UNIT_METRIC)
        miles = sxm_fix_mul(miles, 20361);
    dlat = sxm_fix_mul(miles, 474);
    dlon = sxm_fix_div(dlat, sxm_fix_cos(center->lat));
#else
    if (unit == SXM_SDK_UNIT_METRIC)
        miles = miles*0.62137f;
    dlat = miles*0.014472286f;
    dlon = (float)(dlat/cos(RAD(center->lat)));
#endif

    self->ll.lon = center->lon - dlon;
    self->ll.lat = center->lat - dlat;
    self->ur.lon = center->lon + dlon;
    self->ur.lat = center->lat + dlat;
}

/***************************************************************************//**
 * The function calculates the center of the MBR based on the upper right and
 * lower left points of the MBR supplied
 *
 * \param[in] mbr Pointer to the populated SXMMBR structure 
 * \param[out] center Pointer to the populated SXMPoint structure containing
 *                    the center of the MBR
 *
 ********************************************************************************/
SXESDK_API void sxm_mbr_center(const SXMMBR *mbr, SXMPoint *center)
{
#if SXM_USE_FIX
    center->lat = sxm_fix_div(mbr->ll.lat + mbr->ur.lat, (fix)65536);
    center->lon = sxm_fix_div(mbr->ll.lon + mbr->ur.lon, (fix)65536);
#else
    center->lat = (mbr->ll.lat + mbr->ur.lat) / 2.0f;
    center->lon = (mbr->ll.lon + mbr->ur.lon) / 2.0f;
#endif
}

/***************************************************************************//**
 * The routine returns a value representing the \a distance between two points.
 * The only use for this value is to compare two distances (for example when
 * building the heap in the grid library).
 *
 * \param[in] one a pointer to one end of the line 
 * \param[in] two a pointer to the other end of the line
 *
 * \return a monotonic distance value 
 *
 ********************************************************************************/
SXESDK_API SXMFLOAT sxm_point_distance(const SXMPoint *one, const SXMPoint *two)
{
#if SXM_USE_FIX
    return sxm_fix_distance2(one->lon, one->lat, two->lon, two->lat);
#else
    SXMFLOAT dlon = one->lon - two->lon;
    SXMFLOAT dlat = one->lat - two->lat;

    return (SXMFLOAT)(dlat*dlat + cos(RAD(one->lat))*cos(RAD(two->lat))*dlon*dlon);
#endif
}

/***************************************************************************//**
 * The routine converts the distance value into an actual distance in either
 * miles or kilometers (depending on the global units setting).
 *
 * \param[in] distance2 a value previously returned by the
 *                      \ref sxm_point_distance routine
 *
 * \return the actual distance in measure units
 *
 ********************************************************************************/
SXESDK_API SXMFLOAT sxm_point_true_distance(SXMFLOAT distance2)
{
#if SXM_USE_FIX
    return sxm_fix_distance1(distance2);
#else
    return (SXMFLOAT)
        (((fixpoint_unit() == SXM_SDK_UNIT_METRIC) ? RADIUS_M : RADIUS_I) * sqrt(distance2));
#endif
}

/***************************************************************************//**
 * The  determines if the compared MBR is inside the MBR used by the service. 
 *
 * \param[in] self a pointer to the MBR to compared with
 *
 * \retval 1 if the \a self is inside the existing MBR,
 * \retval 0 otherwise
 *
 ********************************************************************************/
SXESDK_API int sxm_mbr_valid(const SXMMBR *self)
{
    return sxm_mbr_inside_mbr(&service_region, self);
}

/***************************************************************************//**
 * The  determines if the compared MBR is inside or equal to the MBR used 
 * by the service. 
 *
 * \param[in] self a pointer to the MBR to compared with
 *
 * \retval 1 if the \a self is inside or equal to the existing MBR,
 * \retval 0 otherwise
 *
 ********************************************************************************/
SXESDK_API int sxm_mbr_valid_eq(const SXMMBR *self)
{
    return sxm_mbr_inside_mbr(&service_region, self) || 
        !memcmp(&service_region, self, sizeof(SXMMBR));
}

/***************************************************************************//**
 * Reduces MBR's width and height shifting the furthest (from a center point)
 * corner
 *
 * \warning For scale factor less then 0 has undefined behavior.
 *
 * \param[in] mbr pointer to basic MBR
 * \param[in] factor scale factor ( > 1 increase, < 1 reduce)
 * \param[in] center point of interest
 * \param[out] reduced_mbr pointer to placeholder for operation result
 *
 ********************************************************************************/
SXESDK_API void sxm_mbr_reduce(const SXMMBR *mbr, SXMFLOAT factor,
                               const SXMPoint *center, SXMMBR *reduced_mbr)
{
    SXMFLOAT dfactor, dlon, dlat;

#if SXM_USE_FIX
    dfactor = (fix)32768 - factor;
    dlon = sxm_fix_mul(mbr->ur.lon - mbr->ll.lon, dfactor);
    dlat = sxm_fix_mul(mbr->ur.lat - mbr->ll.lat, dfactor);
#else
    dfactor = 1 - factor;
    dlon = (mbr->ur.lon - mbr->ll.lon) * dfactor;
    dlat = (mbr->ur.lat - mbr->ll.lat) * dfactor;
#endif

    // copying whole MBR data
    *reduced_mbr = *mbr;

    if ((mbr->ur.lat - center->lat) > (center->lat - mbr->ll.lat))
    {
        reduced_mbr->ur.lat -= dlat;
    }
    else
    {
        reduced_mbr->ll.lat += dlat;
    }

    if ((center->lon - mbr->ll.lon) > (mbr->ur.lon - center->lon))
    {
        reduced_mbr->ll.lon += dlon;
    }
    else
    {
        reduced_mbr->ur.lon -= dlon;
    }
}

/***************************************************************************//**
 * Scales MBR's width and height with unchanged the center point
 *
 * \warning For scale factor less then 0 has undefined behavior.
 *
 * \param[in] mbr pointer to basic MBR
 * \param[in] factor scale factor ( > 1 increase, < 1 reduce)
 * \param[out] scaled_mbr pointer to placeholder for operation result
 *
 ********************************************************************************/
SXESDK_API void sxm_mbr_scale(const SXMMBR *mbr, SXMFLOAT factor,
                              SXMMBR *scaled_mbr) 
{
    SXMFLOAT dlon, dlat, dfactor;

    if (factor < (SXMFLOAT)0)
    {
        factor = (SXMFLOAT)0;
    }

#if SXM_USE_FIX
    dfactor = (fix)16384 /* 0.5 */ - sxm_fix_div(factor, (fix)65536 /* 2.0 */);
    dlon = sxm_fix_mul(mbr->ur.lon - mbr->ll.lon, dfactor);
    dlat = sxm_fix_mul(mbr->ur.lat - mbr->ll.lat, dfactor);
#else
    dfactor = (SXMFLOAT)0.5 - (factor / 2);
    dlon = (mbr->ur.lon - mbr->ll.lon) * dfactor;
    dlat = (mbr->ur.lat - mbr->ll.lat) * dfactor;
#endif
    scaled_mbr->ll.lon = mbr->ll.lon + dlon;
    scaled_mbr->ur.lon = mbr->ur.lon - dlon;
    scaled_mbr->ll.lat = mbr->ll.lat + dlat;
    scaled_mbr->ur.lat = mbr->ur.lat - dlat;
}

/***************************************************************************//**
 * The determines if the supplied point is inside the current service MBR
 *
 * \param[in] self a pointer to the point to determine if the point is
 *                 inside the service MBR.
 *
 * \retval 1 if the point lies within the current service MBR
 * \retval 0 otherwise
 *
 ********************************************************************************/
SXESDK_API int sxm_point_valid(const SXMPoint *self)
{
    return sxm_mbr_inside(&service_region, self);
}

/***************************************************************************//**
 * The returns the distance to the closet edge of the mbr based on the \a point
 * provided. The actual distance is in either miles or kilometers (depending
 * on the global units setting).
 * 
 * \param[in] mbr a pointer to the mbr
 * \param[in] point a pointer to the point
 *
 * \return "-1" if point is not within the mbr,
 *         otherwise distance to the closest edge
 *
 ********************************************************************************/
SXESDK_API SXMFLOAT sxm_mbr_closest(const SXMMBR *mbr, const SXMPoint *point)
{
    SXMFLOAT d2, d3, d4, cc, mm;
    const SXMSdkUnit unit = fixpoint_unit();

#if SXM_USE_FIX
    cc = sxm_fix_cos(point->lat);
    if (!sxm_mbr_inside(mbr, point))
        return -sxm_fix_int(1);

    mm = sxm_fix_mul(point->lon - mbr->ll.lon, cc);
    d2 = mbr->ur.lat - point->lat;
    d3 = sxm_fix_mul(mbr->ur.lon - point->lon, cc);
    d4 = point->lat - mbr->ll.lat;

    if  (d2 < mm) mm = d2;
    if  (d3 < mm) mm = d3;
    if  (d4 < mm) mm = d4;

    return sxm_fix_mul(mm, ((unit == SXM_SDK_UNIT_METRIC) ? RADIUS_M : RADIUS_I));
#else
    cc = (float)cos(RAD(point->lat));
    if (!sxm_mbr_inside(mbr, point))
        return -1.0;

    mm = (point->lon - mbr->ll.lon)*cc;
    d2 = mbr->ur.lat - point->lat;
    d3 = (mbr->ur.lon - point->lon)*cc;
    d4 = point->lat - mbr->ll.lat;

    if  (d2 < mm) mm = d2;
    if  (d3 < mm) mm = d3;
    if  (d4 < mm) mm = d4;

    return mm * ((unit == SXM_SDK_UNIT_METRIC) ? 111.20177f : 69.09758f);
#endif
}

/***************************************************************************//**
 * The returns the distance to the furthest edge of the mbr based on the \a point
 * provided. The actual distance is in either miles or kilometers (depending on
 * the global units setting). 
 * 
 * \param[in] mbr a pointer to the mbr
 * \param[in] point a pointer to the point
 *
 * \return "-1" if point is not within the mbr,
 *         otherwise distance to the furthest edge
 *
 ********************************************************************************/
SXESDK_API SXMFLOAT sxm_mbr_furthest(const SXMMBR *mbr, const SXMPoint *point)
{
    SXMFLOAT d2, d3, d4, cc, mm;
    const SXMSdkUnit unit = fixpoint_unit();

#if SXM_USE_FIX
    cc = sxm_fix_cos(point->lat);
    if (!sxm_mbr_inside(mbr, point))
        return -sxm_fix_int(1);

    mm = sxm_fix_mul(point->lon - mbr->ll.lon, cc);
    d2 = mbr->ur.lat - point->lat;
    d3 = sxm_fix_mul(mbr->ur.lon - point->lon, cc);    
    d4 = point->lat - mbr->ll.lat;

    if  (d2 > mm) mm = d2;
    if  (d3 > mm) mm = d3;
    if  (d4 > mm) mm = d4;

    return sxm_fix_mul(mm, ((unit == SXM_SDK_UNIT_METRIC) ? RADIUS_M : RADIUS_I));
#else
    cc = (float)cos(RAD(point->lat));
    if (!sxm_mbr_inside(mbr, point))
        return -1.0;

    mm = (point->lon - mbr->ll.lon)*cc;
    d2 = mbr->ur.lat - point->lat;
    d3 = (mbr->ur.lon - point->lon)*cc;    
    d4 = point->lat - mbr->ll.lat;

    if  (d2 > mm) mm = d2;
    if  (d3 > mm) mm = d3;
    if  (d4 > mm) mm = d4;

    return mm * ((unit == SXM_SDK_UNIT_METRIC) ? 111.20177f : 69.09758f);
#endif
}

/************************************************************************
 *                                                                      *
 *            POLYGON  routines                                         *
 *                                                                      *
 ************************************************************************/
/** Polygon iteration callback
 * \param[in] p1 the first point of the line
 * \param[in] p2 the second point of the line
 * \param[in] arg callback argument which was passed along with iteration start
 * \retval 0 to stop iteration
 * \retval 1 to move to the next line if any
 */
typedef int (*SXMPolygonLineCallback)(const SXMPoint *p1, const SXMPoint *p2,
                                      ptr arg);

/** Ray tracing and Line clipping arguments */
typedef struct
{
    int hit; //!< Hits counter
    union
    {
        SXMMBR mbr; //!< MBR
        SXMPoint center; //!< Center point
    } geo; //!< GEO location data depending on iteration purposes
} SXMPolygonLineCallbackArg; //!< Polygon iteration callback arguments

/**
 * \name Clip locations
 * @{
 */
/** Point inside the MBR */
#define SXM_POLY_LINE_CLIP_INSIDE (0) /* 0000 */
/** Point lies to the left of the MBR */
#define SXM_POLY_LINE_CLIP_LEFT   (1) /* 0001 */
/** Point lies to the right of the MBR */
#define SXM_POLY_LINE_CLIP_RIGHT  (2) /* 0010 */
/** Point lies below the MBR */
#define SXM_POLY_LINE_CLIP_BOTTOM (4) /* 0100 */
/** Point lies above the MBR */
#define SXM_POLY_LINE_CLIP_TOP    (8) /* 1000 */
/** @} */

/**
 * \name Helper to wrap polymorphic math functions
 * @{
 */
#if SXM_USE_FIX
    #define SXM_POLY_ABS(_v) abs((_v))
    #define SXM_POLY_EPSILON  (1) /* MIN SXM fix value */
    #define SXM_POLY_DIV(_a, _b) sxm_fix_div((_a), (_b))
    #define SXM_POLY_MUL(_a, _b) sxm_fix_mul((_a), (_b))
#else
    #define SXM_POLY_ABS(_v) fabs((_v))
    #define SXM_POLY_EPSILON  (0.000015f) /* Reflects MIN SXM fix value */
    #define SXM_POLY_DIV(_a, _b) ((_a) / (_b))
    #define SXM_POLY_MUL(_a, _b) ((_a) * (_b))
#endif
/** @} */

#define SXM_POLY_SLOPE(_res, _a, _b, _is_line) \
    if (SXM_POLY_ABS(_b) < SXM_POLY_EPSILON) {     \
        (_is_line) = !0; (_res) = 0;           \
    } else {                                   \
        (_res) = SXM_POLY_DIV(_a, _b);         \
    }

/*************************************************************************//**
 * Helper function to do the callback call to each line belong to the polygon
 * described by the 'points'. The polygon can be either open or closed. For the
 * closed polygon the callback will be called one more time for the last and
 * the first points at the end. Thus, the last and the end point in the array
 * may not be the same.
 *
 * \param[in] points polygon points
 * \param[in] count number polygon points
 * \param[in] closed 1 if the closed polygon or 0 - if not
 * \param[in] callback callback function
 * \param[in] arg callback function argument
 *
 *****************************************************************************/
static void sxm_polygon_lines(const SXMPoint *points, uint count, int closed,
                              SXMPolygonLineCallback callback,
                              ptr arg)
{
    uint ptn_iter;
    int cont = 1;

    for (ptn_iter = 0; cont && (ptn_iter < (count - (closed ? 0 : 1))); ++ptn_iter)
    {
        cont = callback(&points[ptn_iter],  &points[(ptn_iter + 1) % count], arg);
    }
}

/*************************************************************************//**
*   This API is used to make a horizontal ray from a single point. After that
*   it computes the cross point and if it exist it returns the result. This
*   simple functionality can be very helpful if it is need to determine the
*   point's positions depending of the line.
*   It is very easy to determine if a single point is inside the polygon.
*   This API function should be looped through the lines of the polygon.
*   Depending of the number of the crossings we can clearly determine the
*   position of the point. If it is odd - the point is inside of the polygon.
*   Otherwise it is outside.
*
*****************************************************************************/
static int sxm_polygon_ray_tracing(const SXMPoint *p1, const SXMPoint *p2,
                                   ptr arg)
{
    SXMPolygonLineCallbackArg *rt_arg = (SXMPolygonLineCallbackArg *)arg;

    if (
        ((p1->lat <= rt_arg->geo.center.lat) &&
            (p2->lat > rt_arg->geo.center.lat)) ||
        ((p1->lat > rt_arg->geo.center.lat) &&
            (p2->lat <= rt_arg->geo.center.lat))
       )
    {
        SXMFLOAT result;
        result = SXM_POLY_DIV(rt_arg->geo.center.lat - p1->lat,
                              p2->lat - p1->lat);
        result = SXM_POLY_MUL(p2->lon - p1->lon, result) + p1->lon;
        if (rt_arg->geo.center.lon < result)
        {
            ++rt_arg->hit;
        }
    }

    // Continue in any case since based on the algorithms all lines shall
    // from the polygon shall be checked.
    return !0;
}

/*************************************************************************//**
 * Helper function to provide the location's mask of the point relatively
 * to the MBR.
 * 
 * \param[in] p a pointer to the point
 * \param[in] mbr a pointer to the mbr
 *
 * \return combination of
 *         \ref SXM_POLY_LINE_CLIP_LEFT,
 *         \ref SXM_POLY_LINE_CLIP_RIGHT,
 *         \ref SXM_POLY_LINE_CLIP_TOP,
 *         \ref SXM_POLY_LINE_CLIP_BOTTOM or
 *         \ref SXM_POLY_LINE_CLIP_INSIDE
 *
 * \sa sxm_polygon_line_clipping for more details.
 *
 *****************************************************************************/
static int sxm_polygon_line_clipping_code(const SXMPoint *p, const SXMMBR *mbr)
{
    int result = SXM_POLY_LINE_CLIP_INSIDE;
    if (p->lon < mbr->ll.lon)
    {
        result |= SXM_POLY_LINE_CLIP_LEFT;
    }
    else if (p->lon > mbr->ur.lon)
    {
        result |= SXM_POLY_LINE_CLIP_RIGHT;
    }
    if (p->lat < mbr->ll.lat)
    {
        result |= SXM_POLY_LINE_CLIP_BOTTOM;
    }
    else if (p->lat > mbr->ur.lat)
    {
        result |= SXM_POLY_LINE_CLIP_TOP;
    }
    return result;
}

/***************************************************************************//**
 *   This API is used to determine if the input line is crossing the square
 *   polygon. In case of crossing this function will provide the clipped line
 *   inside the square polygon. The algorithm is based on Cohen-Sutherland
 *   algorithm. It divides the space on 9 different areas. Each area has its own
 *   unique binary id. Please look to the table below for details.
 *
 *   |      |      |      |
 *   |:----:|:----:|:----:|
 *   | 1001 | 1000 | 1010 |
 *   | 0001 | 0000 | 0010 |
 *   | 0101 | 0100 | 0110 |
 *
 *   \c 0000 area is a square polygon which we check for crossings. Each time then a
 *   pair of coordinates is passed to the input the algorithm assigns a binary id
 *   to the points and computes its bitwise AND. If the result is > 0 it means
 *   that the line is situated on the nearby areas and there are no any
 *   crossings. If the result is = 0 the algorithm clips the line until the
 *   nearest boarder. Then it renews the id's, calculates the bitwise AND and
 *   repeats the clipping if necessary. This is done until both codes of line
 *   will be = 0 or until the bitwise AND of those codes will be > 0.
 *
 *   http://en.wikipedia.org/wiki/Cohen%E2%80%93Sutherland_algorithm
 *
 *   \param[in] ap1 the first line point
 *   \param[in] ap2 the second line point
 *   \param[in] arg the callback argument (\ref SXMPolygonLineCallbackArg)
 *
 *   \retval 1 to continue clipping calculation using next line
 *   \retval 0 otherwise
 *  
 ********************************************************************************/
static int sxm_polygon_line_clipping(const SXMPoint *ap1, const SXMPoint *ap2,
                                     ptr arg)
{
    SXMPolygonLineCallbackArg *lc_arg = (SXMPolygonLineCallbackArg*)arg;
    SXMPoint p1 = *ap1,
             p2 = *ap2;
    int code_a, code_b, code;
    int is_ver = 0, is_hor = 0;
    SXMFLOAT d_x, d_y, d_xy, d_yx, cur_x, cur_y;

    code_a = sxm_polygon_line_clipping_code(&p1, &lc_arg->geo.mbr);
    code_b = sxm_polygon_line_clipping_code(&p2, &lc_arg->geo.mbr);

    d_x = p2.lon - p1.lon;
    d_y = p2.lat - p1.lat;

    SXM_POLY_SLOPE(d_xy, d_x, d_y, is_hor);
    SXM_POLY_SLOPE(d_yx, d_y, d_x, is_ver);

    // Setting return value to TRUE by default
    lc_arg->hit = !0;
    while (code_a | code_b)
    {
        if (code_a & code_b)
        {
            // Both points are on the one side and
            // have no crossings with square area
            lc_arg->hit = 0;
            break;
        }

        // performing a check here to work only with point that
        // is not inside of the square area
        if (code_a)
        {
            code = code_a;
            cur_x = p1.lon;
            cur_y = p1.lat;
        }
        else
        {
            code = code_b;
            cur_x = p2.lon;
            cur_y = p2.lat;
        }

        //computing the crossing with asymptotes.
        if (code & SXM_POLY_LINE_CLIP_LEFT)
        {
            if (!is_hor)
            {
                cur_y += SXM_POLY_MUL(lc_arg->geo.mbr.ll.lon - cur_x, d_yx);
            }
            cur_x = lc_arg->geo.mbr.ll.lon;
        }
        else if (code & SXM_POLY_LINE_CLIP_RIGHT)
        {
            if (!is_hor)
            {
                cur_y += SXM_POLY_MUL(lc_arg->geo.mbr.ur.lon - cur_x, d_yx);
            }
            cur_x = lc_arg->geo.mbr.ur.lon;
        }
        else if (code & SXM_POLY_LINE_CLIP_BOTTOM)
        {
            // Bottom line
            if (!is_ver)
            {
                cur_x += SXM_POLY_MUL(lc_arg->geo.mbr.ll.lat - cur_y, d_xy);
            }
            cur_y = lc_arg->geo.mbr.ll.lat;
        }
        else if (code & SXM_POLY_LINE_CLIP_TOP)
        {
            // Top line
            if (!is_ver)
            {
                cur_x += SXM_POLY_MUL(lc_arg->geo.mbr.ur.lat - cur_y, d_xy);
            }
            cur_y = lc_arg->geo.mbr.ur.lat;
        }

        // renew codes
        if(code == code_a)
        {
            p1.lon = cur_x;
            p1.lat = cur_y;
            code_a = sxm_polygon_line_clipping_code(&p1, &lc_arg->geo.mbr);
        }
        else
        {
            p2.lon = cur_x;
            p2.lat = cur_y;
            code_b = sxm_polygon_line_clipping_code(&p2, &lc_arg->geo.mbr);
        }
    }
    return !lc_arg->hit;
}

/*************************************************************************//**
 * Determines the fact of intersection between MBR and polygon.
 *
 * Polygon and MBR intersection is detected in two steps:
 * -# Ray Tracing for the MBR's center point - this step is applicable
 *    for closed polygons only
 * -# Line Clipping
 *
 * \param[in] mbr a pointer to the MBR data structure
 * \param[in] points a pointer to structure containing the polygon's points
 * \param[in] count the number of \c points
 * \param[in] closed \b 1 if polygon is type closed, \b 0 otherwise
 *
 * \retval non-zero value if there is an intersection between the mbr and polygon
 * \retval 0 otherwise
 *
 *****************************************************************************/
SXESDK_API int sxm_mbr_intersects_polygon(const SXMMBR *mbr,
                               const SXMPoint *points,
                               uint count,
                               int closed)
{
    SXMPolygonLineCallbackArg arg;

    // The ray crossing applicable only for polygons.
    if (closed)
    {
        // Use the MBR's center point for ray tracing
        sxm_mbr_center(mbr, &arg.geo.center);
        arg.hit = 0;

        sxm_polygon_lines(points, count, closed,
                          sxm_polygon_ray_tracing, (ptr)&arg);
        if (arg.hit % 2)
        {
            // There is an intersection
            return !0;
        }
    }

    // Line clipping for all polygon types (close and opened) is applicable
    arg.geo.mbr = *mbr;
    arg.hit = 0;

    // Adjust the MBR for line clipping if it's too small, aka point.
    if ((SXM_POLY_ABS(mbr->ll.lat - mbr->ur.lat) < SXM_POLY_EPSILON) &&
        (SXM_POLY_ABS(mbr->ll.lon - mbr->ur.lon) < SXM_POLY_EPSILON))
    {
        SXMPoint center;
        sxm_mbr_center(&arg.geo.mbr, &center);
        sxm_mbr_about(&arg.geo.mbr, &center,
#if SXM_USE_FIX
            10 /* ~0.0003f * 32768.0f */
#else
            0.0003f
#endif
            );
    }

    sxm_polygon_lines(points, count, closed,
        sxm_polygon_line_clipping, (ptr)&arg);
    return arg.hit ? !0 : 0;
}
