/******************************************************************************/
/*                Copyright (c) Sirius XM Satellite Radio, Inc.               */
/*                            All Rights Reserved                             */
/*      Licensed Materials - Property of Sirius XM Satellite Radio, Inc.      */
/******************************************************************************/
/***************************************************************************//**
*
* \file sxm_files_input.h
* \author Sergey Kotov
* \date 3/30/2017
*
* Utility to work with a set of files from different sources.
* The input's file set can include files from the file system as well as files
* from ZIP archives. The utility provides methods to enumerate the files list,
* to extract separate file data into RAM buffer and to copy file content to
* the specified file system location.
*
*******************************************************************************/

#ifndef SXM_FILES_INPUT_H
#define SXM_FILES_INPUT_H

#include <stdio.h>
#include <sxm_build.h>
#include <sxm_typ.h>
#include <util/sxm_list.h>
#include <util/sxm_deflate.h>

#ifdef __cplusplus
extern "C"
{
#endif

/** Type of the source the file is originated from */
typedef enum {
    SXM_FILE_SOURCE_FS,
    SXM_FILE_SOURCE_ARCHIVE
} SXMFilesInputSource;

/** Properties of the files input record */
typedef struct
{
    /** Flag indicating that reported path is directory */
    BOOL isDirectory;
    /** Full path for regular file/directory or file name for file/directory from archive */
    char *pFileName;
    /** Type of the source the file is originated from */
    SXMFilesInputSource fileSourceType;
    /** Source archive file path for file from arhive
        or full file path (identical to pFileName) for regular file */
    const char *pSourceFileName;
    /** File size in bytes (0 for directory) */
    size_t fileSize;
} SXMFilesInputProps;

typedef struct _sxmFilesInputRecord SXMFilesInputRecord;

typedef struct _sxmFilesInput SXMFilesInput;

/***************************************************************************//**
 *
 * Callback to provide file name to caller
 *
 * \param[in] pFilePath NULL terminated file input path
 * \param[in] pUserData Pointer to the user data specified by the caller of
 *                      \ref sxm_files_input_create
 *
 * \return None
 *
 ******************************************************************************/
typedef void(*SXM_FSINPUT_PATH_CALLBACK)(const char *pFileInputPath,
                                         BOOL isArchive,
                                         void *pUserData);

/***************************************************************************//**
 *
 * Callback to enumerate files.
 *
 * \param[in] pProps pointer to \ref SXMFilesInputProps structure
 * \param[in] pRecord pointer to \ref SXMFileInputRecord
 * \param[in] pUserData Pointer to the user data specified by the caller of
 *                      \ref sxm_files_input_enumerate
 *
 * \return SXe error code
 *
 ******************************************************************************/
typedef int(*SXM_FSINPUT_CALLBACK)(const SXMFilesInputProps *pProps,
                                   const SXMFilesInputRecord *pRecord,
                                   void *pUserData);

SXESDK_INTERNAL_API int sxm_files_input_create(char delimiter,
                                               const char *pBaseDir,
                                               const char *pInputFiles,
                                               SXM_FSINPUT_PATH_CALLBACK pCallback,
                                               void *pUserData,
                                               SXMFilesInput **ppRet);

SXESDK_INTERNAL_API void sxm_files_input_destroy(SXMFilesInput *pFilesInput);

SXESDK_INTERNAL_API uint sxm_files_input_num_files(const SXMFilesInput *pFilesInput);

SXESDK_INTERNAL_API int sxm_files_input_enumerate(const SXMFilesInput *pFilesInput,
                                                  const char *pDetectExtension,
                                                  SXM_FSINPUT_CALLBACK pCallback,
                                                  void *pUserData);

SXESDK_INTERNAL_API int sxm_files_input_find(const SXMFilesInput *pFilesInput,
                                             const char *pFileName,
                                             const SXMFilesInputRecord **ppRecord);

SXESDK_INTERNAL_API int sxm_files_input_alloc_and_read(const SXMFilesInputRecord *pRecord,
                                                       void **ppOut, size_t *pDataSize);

SXESDK_INTERNAL_API int sxm_files_input_copy(const SXMFilesInputRecord *pRecord,
                                             FILE *pFileOut);

SXESDK_INTERNAL_API int sxm_files_input_find_alloc_and_read(const SXMFilesInput *pFilesInput,
                                                            const char *pFileName,
                                                            void **ppOut, size_t *pDataSize);

SXESDK_INTERNAL_API int sxm_files_input_find_and_copy(const SXMFilesInput *pInput,
                                                      const char *pFileName,
                                                      FILE *pFileOut);

#ifdef __cplusplus
}
#endif

#endif /* #ifndef SXM_FILES_INPUT_H */
