/******************************************************************************/
/*                Copyright (c) Sirius XM Satellite Radio, Inc.               */
/*                            All Rights Reserved                             */
/*      Licensed Materials - Property of Sirius XM Satellite Radio, Inc.      */
/******************************************************************************/
/***************************************************************************//**
*
* \file sxm_files_input.c
* \author Sergey Kotov
* \date 3/30/2017
*
* Utility to work with a set of files from different sources.
* The input's file set can include files from the file system as well as files
* from ZIP archives. The utility provides methods to enumerate the files list,
* to extract separate file data into RAM buffer and to copy file content to
* the specified file system location.
*
*******************************************************************************/

#define DEBUG_TAG   "fsinput"

#include <ctype.h>
#include <util/sxm_files_input.h>

#ifndef SDKFILES_STANDALONE_BUILD
#include <sxm_stdtrace.h>
#include <util/sxm_common_internal.h>

/** Debug macro redefinition */
#ifndef SXM_DEBUG_PRODUCTION
#undef DEBUG_VAR
static uint debugVar = 0; /* Disabled by default. -1 enables all logging */
#define DEBUG_VAR   debugVar

#endif /* #ifndef SXM_DEBUG_PRODUCTION */

#else /* #ifndef SDKFILES_STANDALONE_BUILD */

#include "sdkfiles.h"

#undef WARN_UTL
#define WARN_UTL non_fatal
#undef ERROR_UTL
#define ERROR_UTL non_fatal
#endif /* #ifndef SDKFILES_STANDALONE_BUILD ... #else ... */

/** Size of the temporary buffer used for file copy */
#define SXM_FILES_INPUT_TMP_BUF_SIZE    (1024) /* bytes */

/* Generic file record entity */
struct _sxmFilesInputRecord
{
    /** Properties of the files input record */
    SXMFilesInputProps props;
    /* Implementation specific functions */
    int(*read)(const SXMFilesInputRecord *pRecord, void *pBuf, size_t bufSize);
    int(*alloc_and_read)(const SXMFilesInputRecord *pRecord, void **ppBuf, size_t *pDataSize);
    int(*copy)(const SXMFilesInputRecord *pRecord, FILE *pFileOut);
    void(*release)(SXMFilesInputRecord *pRecord);
};

/* Archive-extracted file record entity */
typedef struct
{
    SXMFilesInputRecord generic; /* Must be first */
    FILE *pSourceFile;
    size_t fileOffset;
} SXMFilesInputArchiveRecord;

/* Opened file descriptor */
typedef struct
{
    FILE *pFile;
    char *pFilePath;
} SXMFilesInputSourceFileDesc;

/* Main object entity */
struct _sxmFilesInput
{
    /* List of OpenedFileDesc entries */
    SXMList openedFilesList;

    /* List of all SXMFilesInputRecord entries */
    SXMList recordsList;
};

typedef struct
{
    SXMList *pRecordsList;
    const SXMFilesInputSourceFileDesc *pSourceFileDesc;
} ZipFileCallbackData;

/*****************************************************************************
*            Internal Functions
*            ==================
******************************************************************************/

static int check_file_extension(const char *pFileName, const char *pExtension)
{
    /* Get file extension */
    const char *pFoundExtension = strrchr(pFileName, '.');
    if (NULL == pFoundExtension)
    {
        ERROR_UTL("No file extension found in '%s'", pFileName);
        return 0;
    }

    pFoundExtension++;

    /* Case-independent comparison */
    while (tolower(*pFoundExtension) == tolower(*pExtension))
    {
        if ('\0' == *pExtension)
        {
            return 1;
        }

        pExtension++;
        pFoundExtension++;
    }

    return 0;
}

static const char *get_next_input(const char *pInputPath,
                                  char delimiter,
                                  const char **ppInputPtr,
                                  size_t *pInputLen)
{
    uint idx = 0;
    size_t filePathLen = 0;
    const char *pFilePathPtr = pInputPath;

    for (idx = 0;; idx++)
    {
        if (('\0' != pInputPath[idx]) && (delimiter != pInputPath[idx]))
        {
            filePathLen++;
        }
        else
        {
            if (filePathLen > 0)
            {
                /* Save input pointer and length */
                *ppInputPtr = pFilePathPtr;
                *pInputLen = filePathLen;
            }

            if ('\0' == pInputPath[idx])
            {
                if (filePathLen > 0)
                {
                    /* Set pointer to the end */
                    pFilePathPtr = &pInputPath[idx];
                }
                else
                {
                    /* Nothing to return */
                    pFilePathPtr = NULL;
                    *ppInputPtr = NULL;
                    *pInputLen = 0;
                }
                break;
            }
            else
            {
                /* Move pointer to the next input path */
                pFilePathPtr = &pInputPath[idx + 1];
                if (filePathLen > 0)
                {
                    break;
                }
            }
        }
    }

    return pFilePathPtr;
}

/* Generic file record implementation */

static void generic_record_release(SXMFilesInputRecord *pRecord)
{
    sxe_free(pRecord->props.pFileName);

    sxe_free(pRecord);
}

static int prepare_input_file(const char *pFilePath,
                              FILE **ppFile, size_t *pFileSize)
{
    long fileSize = 0;

    *ppFile = fopen(pFilePath, "rb");
    if (NULL == *ppFile)
    {
        ERROR_UTL("File open failed: '%s'", pFilePath);
        return SXM_E_PIPE;
    }

    /* Get file size */
    if (0 != fseek(*ppFile, 0, SEEK_END))
    {
        ERROR_UTL("File seek failed: '%s'", pFilePath);
        fclose(*ppFile);
        return SXM_E_PIPE;
    }

    fileSize = ftell(*ppFile);
    if (fileSize < 0)
    {
        ERROR_UTL("ftell failed: '%s'", pFilePath);
        fclose(*ppFile);
        return SXM_E_PIPE;
    }

    if (0 != fseek(*ppFile, 0, SEEK_SET))
    {
        ERROR_UTL("File seek failed: '%s'", pFilePath);
        fclose(*ppFile);
        return SXM_E_PIPE;
    }

    *pFileSize = (size_t)fileSize;
    return SXM_E_OK;
}

static int fs_record_read(const SXMFilesInputRecord *pRecord,
                          void *pBuf, size_t bufSize)
{
    int rc;
    FILE *pFile;
    size_t fileSize;

    rc = prepare_input_file(pRecord->props.pFileName, &pFile, &fileSize);
    if (SXM_E_OK != rc)
    {
        return rc;
    }

    if (bufSize < fileSize)
    {
        ERROR_UTL("Insufficient buffer size (%u < %u)",
                  bufSize, fileSize);
        fclose(pFile);
        return SXM_E_RESOURCE;
    }

    if (fileSize != fread(pBuf, 1, fileSize, pFile))
    {
        ERROR_UTL("%u bytes reading failed: '%s'",
                  fileSize, pRecord->props.pFileName);
        return SXM_E_PIPE;
    }

    fclose(pFile);

    return SXM_E_OK;
}

static int fs_record_alloc_and_read(const SXMFilesInputRecord *pRecord,
                                    void **ppBuf, size_t *pDataSize)
{
    int rc;
    FILE *pFile;

    rc = prepare_input_file(pRecord->props.pFileName, &pFile, pDataSize);
    if (SXM_E_OK != rc)
    {
        return rc;
    }

    *ppBuf = sxe_malloc(*pDataSize);
    if (NULL == *ppBuf)
    {
        ERROR_UTL("Buffer allocation failed");
        fclose(pFile);
        return SXM_E_NOMEM;
    }

    if (*pDataSize != fread(*ppBuf, 1, *pDataSize, pFile))
    {
        ERROR_UTL("%u bytes reading failed: '%s'",
                  *pDataSize, pRecord->props.pFileName);

        sxe_free(*ppBuf);
        fclose(pFile);

        return SXM_E_PIPE;
    }

    fclose(pFile);

    return SXM_E_OK;
}

static int fs_record_copy(const SXMFilesInputRecord *pRecord, FILE *pFileOut)
{
    int rc = SXM_E_OK;
    FILE *pFileSource;
    size_t numBytesLeft;
    void *pReadBuf;
    size_t numBytesRead = 0;

    rc = prepare_input_file(pRecord->props.pFileName,
                            &pFileSource, &numBytesLeft);
    if (SXM_E_OK != rc)
    {
        return rc;
    }

    /* Allocate temporary buffer */
    pReadBuf = sxe_malloc(SXM_FILES_INPUT_TMP_BUF_SIZE);
    if (NULL == pReadBuf)
    {
        ERROR_UTL("Read buffer allocation failed");
        fclose(pFileSource);
        return SXM_E_NOMEM;
    }

    /* Copy file */
    while (0 != numBytesLeft)
    {
        numBytesRead = fread(pReadBuf, 1,
                             SXM_FILES_INPUT_TMP_BUF_SIZE,
                             pFileSource);
        if (0 == numBytesRead)
        {
            break;
        }

        if (numBytesRead != fwrite(pReadBuf, 1, numBytesRead, pFileOut))
        {
            ERROR_UTL("File writing failed");
            rc = SXM_E_PIPE;
            break;
        }

        numBytesLeft -= numBytesRead;
    }

    sxe_free(pReadBuf);

    fclose(pFileSource);

    if (0 != numBytesLeft)
    {
        ERROR_UTL("Not all data read: '%s'",
                  pRecord->props.pFileName);
        rc = SXM_E_PIPE;
    }

    return rc;
}

static int fs_record_create(char *pFilePath, size_t fileSize,
                            SXMFilesInputRecord **ppRecord)
{
    *ppRecord = (SXMFilesInputRecord*)sxe_malloc(sizeof(SXMFilesInputRecord));
    if (NULL == *ppRecord)
    {
        ERROR_UTL("FS record allocation failed");
        return SXM_E_NOMEM;
    }

    /* Populate record fields */
    (*ppRecord)->props.fileSize = fileSize;
    (*ppRecord)->props.fileSourceType = SXM_FILE_SOURCE_FS;
    (*ppRecord)->props.pFileName = pFilePath;
    (*ppRecord)->props.pSourceFileName = pFilePath;
    (*ppRecord)->props.isDirectory = FALSE;
    (*ppRecord)->read = fs_record_read;
    (*ppRecord)->alloc_and_read = fs_record_alloc_and_read;
    (*ppRecord)->copy = fs_record_copy;
    (*ppRecord)->release = generic_record_release;

    return SXM_E_OK;
}

/* Archive-extracted file record implementation */

static int archive_record_read(const SXMFilesInputRecord *pRecord,
                               void *pBuf, size_t bufSize)
{
    SXMFilesInputArchiveRecord *pArchRecord =
            (SXMFilesInputArchiveRecord*)pRecord;

    return sxm_deflate_pkzip_read(pArchRecord->pSourceFile,
                                  pArchRecord->fileOffset,
                                  pBuf, bufSize);
}

static int archive_record_alloc_and_read(const SXMFilesInputRecord *pRecord,
                                         void **ppBuf, size_t *pDataSize)
{
    SXMFilesInputArchiveRecord *pArchRecord =
            (SXMFilesInputArchiveRecord*)pRecord;

    return sxm_deflate_pkzip_alloc_and_read(pArchRecord->pSourceFile,
                                            pArchRecord->fileOffset,
                                            ppBuf, pDataSize);
}

static int archive_record_copy(const SXMFilesInputRecord *pRecord, FILE *pFileOut)
{
    SXMFilesInputArchiveRecord *pArchRecord =
            (SXMFilesInputArchiveRecord*)pRecord;

    return sxm_deflate_pkzip_copy(pArchRecord->pSourceFile,
                                  pArchRecord->fileOffset,
                                  pFileOut);
}

static int archive_record_create(FILE *pSourceFile,
                                 const char *pSourceFileName,
                                 const SXMPKZipFileDesc *pDesc,
                                 SXMFilesInputRecord **ppRecord)
{
    SXMFilesInputArchiveRecord *pArchiveRecord;
    char *pFileName;

    /* Allocate the record */
    pArchiveRecord = (SXMFilesInputArchiveRecord*)
                      sxe_malloc(sizeof(SXMFilesInputArchiveRecord));
    if (NULL == pArchiveRecord)
    {
        ERROR_UTL("ARCHIVE record allocation failed");
        return SXM_E_NOMEM;
    }

    /* Save file name */
    pFileName = (char*)sxe_malloc(strlen(pDesc->pFileName) + 1);
    if (NULL == pFileName)
    {
        ERROR_UTL("File name allocation failed");
        sxe_free(pArchiveRecord);
        return SXM_E_NOMEM;
    }

    strcpy(pFileName, pDesc->pFileName);

    /* Populate archive record fields */
    pArchiveRecord->pSourceFile = pSourceFile;
    pArchiveRecord->fileOffset = pDesc->fileOffset;

    /* Populate generic record fields */
    *ppRecord = &pArchiveRecord->generic;
    (*ppRecord)->props.fileSize = pDesc->uncompressedSize;
    (*ppRecord)->props.fileSourceType = SXM_FILE_SOURCE_ARCHIVE;
    (*ppRecord)->props.pFileName = pFileName;
    (*ppRecord)->props.pSourceFileName = pSourceFileName;
    (*ppRecord)->props.isDirectory = pDesc->isDirectory;
    (*ppRecord)->read = archive_record_read;
    (*ppRecord)->alloc_and_read = archive_record_alloc_and_read;
    (*ppRecord)->copy = archive_record_copy;
    (*ppRecord)->release = generic_record_release;

    return SXM_E_OK;
}

/* Main object implementation */

static int zip_files_callback(const SXMPKZipFileDesc *pDesc, void *pData)
{
    int rc;
    SXMFilesInputRecord *pRecord;
    ZipFileCallbackData *pCallbackData = (ZipFileCallbackData*)pData;

    /* Allocate list entry */
    rc = archive_record_create(pCallbackData->pSourceFileDesc->pFile,
                               pCallbackData->pSourceFileDesc->pFilePath,
                               pDesc, &pRecord);
    if (SXM_E_OK != rc)
    {
        return rc;
    }

    /* Add entry to the list */
    rc = sxm_list_add(pCallbackData->pRecordsList,
                      (ptr)pRecord, FALSE, NULL);
    if (SXM_E_OK != rc)
    {
        ERROR_UTL("Failed to add entry to the list (rc=%d)", rc);
        pRecord->release(pRecord);
        return rc;
    }

    return SXM_E_OK;
}

static int sxm_files_input_initialize(SXMFilesInput *pFilesInput,
                                      const char *pBaseDir,
                                      const char *pInputFiles,
                                      SXM_FSINPUT_PATH_CALLBACK pCallback,
                                      void *pUserData,
                                      char delimiter)
{
    const char *pInputPtr = NULL;
    size_t inputLen = 0;
    char *pSourceFilePath = NULL;
    FILE *pInputFile = NULL;
    size_t inputFileSize = 0;
    size_t baseDirLen = 0;
    int rc = SXM_E_OK;

    if (NULL != pBaseDir)
    {
        baseDirLen = strlen(pBaseDir) + 1; /* +1 to append '/' */
    }

    while (SXM_E_OK == rc)
    {
        /* Get next input pointer and size */
        pInputFiles = get_next_input(pInputFiles, delimiter,
                                     &pInputPtr, &inputLen);
        if (NULL == pInputPtr)
        {
            /* No more entries found */
            break;
        }

        /* Allocate and save file name */
        pSourceFilePath = (char*)sxe_calloc(1, baseDirLen + inputLen + 1);
        if (NULL == pSourceFilePath)
        {
            ERROR_UTL("Failed to allocate file name");
            rc = SXM_E_NOMEM;
            break;
        }

        if (0 != baseDirLen)
        {
            strncpy(pSourceFilePath, pBaseDir, baseDirLen - 1);
            pSourceFilePath[baseDirLen - 1] = '/';
        }
        strncat(pSourceFilePath, pInputPtr, inputLen);

        rc = prepare_input_file(pSourceFilePath, &pInputFile, &inputFileSize);
        if (SXM_E_OK != rc)
        {
            break;
        }

        /* Determine file source type */
        if (0 != check_file_extension(pSourceFilePath, "zip"))
        {
            SXMFilesInputSourceFileDesc *pSourceFileDesc;
            ZipFileCallbackData callbackData;

            if (NULL != pCallback)
            {
                pCallback(pSourceFilePath, TRUE, pUserData);
            }

            /* Allocate list entry */
            pSourceFileDesc = (SXMFilesInputSourceFileDesc*)
                               sxe_malloc(sizeof(SXMFilesInputSourceFileDesc));
            if (NULL == pSourceFileDesc)
            {
                ERROR_UTL("OpenedFileDesc allocation failed");
                rc = SXM_E_NOMEM;
                break;
            }

            pSourceFileDesc->pFile = pInputFile;
            pSourceFileDesc->pFilePath = pSourceFilePath;

            /* Add entry to the opened files list */
            rc = sxm_list_add(&pFilesInput->openedFilesList,
                              pSourceFileDesc, FALSE, NULL);
            if (SXM_E_OK != rc)
            {
                ERROR_UTL("Opened Files list adding failed (rc=%d)", rc);
                break;
            }

            /* Now the string buffer is owned by the list record */
            pSourceFilePath = NULL;
            /* Now opened file is owned by the list record */
            pInputFile = NULL;

            callbackData.pSourceFileDesc = pSourceFileDesc;
            callbackData.pRecordsList = &pFilesInput->recordsList;

            /* Enumerate zip file entries and add records
               for all files found in the archive */
            rc = sxm_deflate_pkzip_enumerate(pSourceFileDesc->pFile,
                                             zip_files_callback,
                                             &callbackData);
            if (SXM_E_OK != rc)
            {
                ERROR_UTL("'%s' ZIP file enumeration failed (rc=%d)",
                          pSourceFileDesc->pFilePath, rc);
                break;
            }
        }
        else
        {
            /* Add FS type record */
            SXMFilesInputRecord *pRecord;

            if (NULL != pCallback)
            {
                pCallback(pSourceFilePath, FALSE, pUserData);
            }

            /* Allocate list entry */
            rc = fs_record_create(pSourceFilePath, inputFileSize, &pRecord);
            if (SXM_E_OK != rc)
            {
                break;
            }

            /* Opened input file is not required yet, close it */
            fclose(pInputFile);
            pInputFile = NULL;

            /* Now the string is owned by the record */
            pSourceFilePath = NULL;

            /* Add entry to the list */
            rc = sxm_list_add(&pFilesInput->recordsList,
                              (ptr)pRecord, FALSE, NULL);
            if (SXM_E_OK != rc)
            {
                ERROR_UTL("Records list adding failed (rc=%d)", rc);
                pRecord->release(pRecord);
                break;
            }
        }
    }

    if (NULL != pInputFile)
    {
        fclose(pInputFile);
    }

    if (NULL != pSourceFilePath)
    {
        sxe_free(pSourceFilePath);
    }

    if (0 == sxm_list_size(&pFilesInput->recordsList))
    {
        ERROR_UTL("No inputs provided");
        rc = SXM_E_INVAL;
    }

    return rc;
}

/******************************************************************************
*            Module Interface
*            ================
******************************************************************************/

/***************************************************************************//**
*
* The routine creates files input object. It creates the list of files including
* all regular files and files stored inside ZIP archives.
*
* \param[in] delimiter Input file names delimiter in pInputFiles parameter
* \param[in] pBaseDir Path to the base directory common for all files
*                     listed in pInputFiles parameter. If this parameter is NULL,
*                     pInputFiles must include full files paths.
* \param[in] pInputPaths Semicolon-separated list of input files.
*                        If pBaseDir parameter is not NULL, this parameter must
*                        include relative files names or paths.
* \param[out] ppRet A pointer to pointer to \ref SXMFilesInput
*
* \return SXe error code
* \retval SXM_E_OK Success
* \retval SXM_E_INVAL Invalid parameter
* \retval SXM_E_FAULT NULL parameter
* \retval SXM_E_NOMEM Memory allocation failed
* \retval SXM_E_PIPE File operation failed
* \retval SXM_E_ERROR General error
* \retval SXM_E_UNSUPPORTED Unsupported input file
*
******************************************************************************/
int sxm_files_input_create(char delimiter,
                           const char *pBaseDir,
                           const char *pInputFiles,
                           SXM_FSINPUT_PATH_CALLBACK pCallback,
                           void *pUserData,
                           SXMFilesInput **ppRet)
{
    int rc = SXM_E_OK;

    switch (0) { default:
    {
        /* Check input parameters */
        if ((NULL == pInputFiles) || (NULL == ppRet))
        {
            rc = SXM_E_FAULT;
            break;
        }

        /* Allocate the instance */
        *ppRet = (SXMFilesInput*)sxe_calloc(1, sizeof(SXMFilesInput));
        if (NULL == *ppRet)
        {
            ERROR_UTL("Main object allocation failed");
            rc = SXM_E_NOMEM;
            break;
        }

        /* Initialize opened files list */
        rc = sxm_list_create(&(*ppRet)->openedFilesList, SXM_LIST_NONE);
        if (SXM_E_OK != rc)
        {
            ERROR_UTL("Opened files list init failed");
            break;
        }

        /* Initialize files records list */
        rc = sxm_list_create(&(*ppRet)->recordsList, SXM_LIST_NONE);
        if (SXM_E_OK != rc)
        {
            ERROR_UTL("Files records list init failed");
            break;
        }

        /* Process all inputs and populate the list of files records */
        rc = sxm_files_input_initialize(*ppRet, pBaseDir, pInputFiles,
            pCallback, pUserData, delimiter);

    }}; /* switch (0) { default: */

    if ((SXM_E_OK != rc) && (NULL != (*ppRet)))
    {
        sxm_files_input_destroy((*ppRet));
        *ppRet = NULL;
    }

    return rc;
}

void sxm_files_input_destroy(SXMFilesInput *pFilesInput)
{
    if (NULL != pFilesInput)
    {
        SXMListEntry *pEntry;

        /* Clear files records list */
        pEntry = sxm_list_first(&pFilesInput->recordsList);
        while (NULL != pEntry)
        {
            SXMFilesInputRecord *pRecord =
                (SXMFilesInputRecord*)sxm_list_data(pEntry);
            pRecord->release(pRecord);

            sxm_list_remove(&pFilesInput->recordsList, pEntry);

            pEntry = sxm_list_first(&pFilesInput->recordsList);
        }

        sxm_list_destroy(&pFilesInput->recordsList);

        /* Clear opened files list */
        pEntry = sxm_list_first(&pFilesInput->openedFilesList);
        while (NULL != pEntry)
        {
            SXMFilesInputSourceFileDesc *pDesc =
                (SXMFilesInputSourceFileDesc*)sxm_list_data(pEntry);

            fclose(pDesc->pFile);
            sxe_free(pDesc->pFilePath);
            sxe_free(pDesc);

            sxm_list_remove(&pFilesInput->openedFilesList, pEntry);

            pEntry = sxm_list_first(&pFilesInput->openedFilesList);
        }

        sxm_list_destroy(&pFilesInput->openedFilesList);

        /* Delete main object */
        sxe_free(pFilesInput);
    }

    return;
}

uint sxm_files_input_num_files(const SXMFilesInput *pFilesInput)
{
    if (NULL != pFilesInput)
    {
        return (uint)sxm_list_size(&pFilesInput->recordsList);
    }

    return 0;
}

int sxm_files_input_enumerate(const SXMFilesInput *pFilesInput,
                              const char *pDetectExtension,
                              SXM_FSINPUT_CALLBACK pCallback,
                              void *pUserData)
{
    int rc;
    SXMListEntry *pEntry;
    SXMFilesInputRecord *pRecord;

    if ((NULL == pFilesInput) || (NULL == pCallback))
    {
        return SXM_E_FAULT;
    }

    pEntry = sxm_list_first(&pFilesInput->recordsList);
    while (NULL != pEntry)
    {
        pRecord = (SXMFilesInputRecord*)sxm_list_data(pEntry);

        if ((NULL == pDetectExtension) ||
            ((FALSE == pRecord->props.isDirectory) &&
            (0 != check_file_extension(pRecord->props.pFileName,
                                       pDetectExtension))))
        {
            rc = pCallback(&pRecord->props, pRecord, pUserData);
            if (SXM_E_OK != rc)
            {
                return rc;
            }
        }

        pEntry = sxm_list_next(pEntry);
    }

    return SXM_E_OK;
}

int sxm_files_input_find(const SXMFilesInput *pFilesInput,
                         const char *pFileName,
                         const SXMFilesInputRecord **ppRecord)
{
    SXMListEntry *pEntry;

    if ((NULL == pFilesInput) || (NULL == pFileName) || (NULL == ppRecord))
    {
        return SXM_E_FAULT;
    }

    pEntry = sxm_list_first(&pFilesInput->recordsList);
    while (NULL != pEntry)
    {
        SXMFilesInputRecord *pRecord =
            (SXMFilesInputRecord*)sxm_list_data(pEntry);

        if (NULL != strstr(pRecord->props.pFileName, pFileName))
        {
            *ppRecord = pRecord;
            return SXM_E_OK;
        }

        pEntry = sxm_list_next(pEntry);
    }

    return SXM_E_NOENT;
}

int sxm_files_input_alloc_and_read(const SXMFilesInputRecord *pRecord,
                                   void **ppOut, size_t *pDataSize)
{
    if ((NULL == pRecord) || (NULL == ppOut))
    {
        return SXM_E_FAULT;
    }

    return pRecord->alloc_and_read(pRecord, ppOut, pDataSize);
}

int sxm_files_input_copy(const SXMFilesInputRecord *pRecord, FILE *pFileOut)
{
    if ((NULL == pRecord) || (NULL == pFileOut))
    {
        return SXM_E_FAULT;
    }

    return pRecord->copy(pRecord, pFileOut);
}

int sxm_files_input_find_alloc_and_read(const SXMFilesInput *pFilesInput,
                                        const char *pFileName,
                                        void **ppOut, size_t *pDataSize)
{
    int rc;
    const SXMFilesInputRecord *pRecord;

    if ((NULL == pFilesInput) || (NULL == pFileName) || (NULL == ppOut))
    {
        return SXM_E_FAULT;
    }

    /* Find record by name */
    rc = sxm_files_input_find(pFilesInput, pFileName, &pRecord);
    if (SXM_E_OK == rc)
    {
        rc = pRecord->alloc_and_read(pRecord, ppOut, pDataSize);
    }

    return rc;
}

int sxm_files_input_find_and_copy(const SXMFilesInput *pFilesInput,
                                  const char *pFileName,
                                  FILE *pFileOut)
{
    int rc;
    const SXMFilesInputRecord *pRecord;

    if ((NULL == pFilesInput) || (NULL == pFileName) || (NULL == pFileOut))
    {
        return SXM_E_FAULT;
    }

    /* Find record by name */
    rc = sxm_files_input_find(pFilesInput, pFileName, &pRecord);
    if (SXM_E_OK == rc)
    {
        rc = pRecord->copy(pRecord, pFileOut);
    }
    else
    {
        WARN_UTL("File '%s' not found", pFileName);
    }

    return rc;
}
