/******************************************************************************/
/*                Copyright (c) Sirius XM Satellite Radio, Inc.               */
/*                            All Rights Reserved                             */
/*      Licensed Materials - Property of Sirius XM Satellite Radio, Inc.      */
/******************************************************************************/
/***************************************************************************//**
*
* \file sxm_deflate.c
* \author Leslie French
* \date 8/20/2013
* \brief (G)ZIP deflate
*
* \details Files transmitted over the broadcast link may be compressed using the 'flate'
* algorithm, most commonly found in the 'zip' file format. The deflate routine
* expands these compressed files into their original contents. Since 'flate'
* is a bit-oriented compression scheme, the input is presented as a
* \ref SXMBitBuff structure.
*
*******************************************************************************/

#define DEBUG_TAG "deflate"

#include <util/sxm_deflate.h>

#ifndef SDKFILES_STANDALONE_BUILD
#include <sxm_stdtrace.h>
#include <util/sxm_common_internal.h>
#include <util/sxm_noname_internal.h>

/** Debug macro redefinition */
#ifndef SXM_DEBUG_PRODUCTION
#undef DEBUG_VAR
static uint debugVar = 0; /* Disabled by default. -1 enables all logging */
#define DEBUG_VAR   debugVar
#endif /* #ifndef SXM_DEBUG_PRODUCTION */

#else /* #ifndef SDKFILES_STANDALONE_BUILD */

#include "sdkfiles.h"

#undef DEBUG_UTL
#define DEBUG_UTL(_f, ...)
#undef ERROR_UTL
#define ERROR_UTL non_fatal

#define SKIP(_n)        sxm_bitbuff_skip(pkt, (uint)(_n))
#endif /* #ifndef SDKFILES_STANDALONE_BUILD ... #else ... */

#ifdef _WIN32
    #include <direct.h>
    #define os_mkdir(path) _mkdir(path)
    #define snprintf _snprintf
#else
    #include <sys/stat.h>
    #include <sys/types.h>
    #define os_mkdir(path) mkdir(path, 0777)
#endif

/**
 * \name Helpers
 *  The flate algorithm defines the bits starting at the 'other' end of the
 *  byte from the SXM usage, hence the 'rev' in these macros
 * @{
 */
#define EATR(_v, _l)    (0 != sxm_bitbuff_consume_rev(pkt, _v, _l))
#define NEXTR(_n)       sxm_bitbuff_read_rev(pkt, _n)
#define NEXT_BYTE()     sxm_bitbuff_read(pkt, 8)
/** @} */

/** Defines PKZIP header fields lengths */
#define SXM_PKZIP_SIGNATURE_BITLEN          32
#define SXM_PKZIP_VERSION_BITLEN            16
#define SXM_PKZIP_FLAGS_BITLEN              16
#define SXM_PKZIP_COMPRESSION_BITLEN        16
#define SXM_PKZIP_MOD_TIME_BITLEN           16
#define SXM_PKZIP_MOD_DATE_BITLEN           16
#define SXM_PKZIP_CRC_BITLEN                32
#define SXM_PKZIP_COMPRESSED_SIZE_BITLEN    32
#define SXM_PKZIP_UNCOMPRESSED_SIZE_BITLEN  32
#define SXM_PKZIP_FILE_NAME_LEN_BITLEN      16
#define SXM_PKZIP_EXTRA_FIELD_LEN_BITLEN    16

#define SXM_PKZIP_HEADER_BASE_LEN ((SXM_PKZIP_SIGNATURE_BITLEN + \
                                    SXM_PKZIP_VERSION_BITLEN + \
                                    SXM_PKZIP_FLAGS_BITLEN + \
                                    SXM_PKZIP_COMPRESSION_BITLEN + \
                                    SXM_PKZIP_MOD_TIME_BITLEN + \
                                    SXM_PKZIP_MOD_DATE_BITLEN + \
                                    SXM_PKZIP_CRC_BITLEN + \
                                    SXM_PKZIP_COMPRESSED_SIZE_BITLEN + \
                                    SXM_PKZIP_UNCOMPRESSED_SIZE_BITLEN + \
                                    SXM_PKZIP_FILE_NAME_LEN_BITLEN + \
                                    SXM_PKZIP_EXTRA_FIELD_LEN_BITLEN) / \
                                   SXM_ARCH_BITS_IN_BYTE)

/** Defines the size of the unpacking zip file intermediate buffer */
#define ZIP_FILE_BUFFER_SIZE        (65536)

#define PKZIP_LFH_SIGNATURE         (0x04034b50) /* Local File Header */
#define PKZIP_DATA_DESCRIPTOR_MASK  (0x08)
#define PKZIP_COMPRESSION_DEFLATE   (0x08)
#define PKZIP_COMPRESSION_EDEFLATE  (0x09)
#define PKZIP_NO_COMPRESSION        (0x00)

#define SXM_PKZIP_FILENAME_LEN_MAX  (129) /* Including '\0' */

typedef struct
{
    UINT32 signature;
    UINT16 version;
    UINT16 generalPurposeBitFlag;
    UINT16 compressionMethod;
    UINT16 time;
    UINT16 date;
    UINT32 crc;
    UINT32 compressedSize;
    UINT32 uncompressedSize;
    UINT16 filenameLength;
    UINT16 extraFieldLength;
    char fileName[SXM_PKZIP_FILENAME_LEN_MAX];
    UINT8 *extraField;
} PKZipLocalFileHeader;

/***************************************************************************//**
* Reverse the order of the 'len' lest-significant bits in a word
* 
* \param[in] in
* \param[in] len
*
* \return
*
*******************************************************************************/
static uint reverse(uint in, uint len) {
    uint res = 0;
    uint m1 = 1U;
    uint j;
    for (j = 0; j < len; j++) {
        uint r = (in & m1) >> j;
        m1 <<= 1;
        res |= (r << (len-j-1));
    }
    return res;
}

/***************************************************************************//**
* Build a Huffman table {mask,value} pairs with count values at each bit-length
* 
* \param[in] counts
* \param[in] maxc
* \param[out] masks
* \param[out] values
*
*******************************************************************************/
static void buildtable(uint counts[], uint maxc, uint masks[], uint values[]) {

    uint bl_counts[32];
    uint i;
    uint code = 0;
    uint next_code[32];

    memset(bl_counts, 0, 32 * 4);

    /* Count how many codes there are for each bit-length */

    for (i = 0; i < maxc; i++) {
        bl_counts[counts[i]]++;
    }

    /* Find the starting point of each bit-length */

    for (i = 1; i < 32; i++) {
        code = (code + bl_counts[i - 1]) << 1;
        next_code[i] = code;
    }

    /* Assign the values */

    for (i = 0; i < maxc; i++) {
        uint len = counts[i];
        if (len != 0) {
            uint mask = 0xffffffff >> (32-len);
            values[i] = reverse(next_code[len], len);
            masks[i] = mask;
            next_code[len]++;
        }
    }
}

/** The default Huffman table generator for deflate */
static uint hcindex[] =
    {16, 17, 18, 0, 8, 7, 9, 6, 10, 5, 11, 4, 12, 3, 13, 2, 14, 1, 15};

/***************************************************************************//**
* Deflate (expand) a single block into 'out'
* This works in two modes, depending on whether or not 'outf' is NULL
*
* If outf is non-NULL, then 'out' is a 64kbyte buffer which is used circularly, and written when full
* otherwise 'out' must be large enough to contain the whole of the data and is written sequentially
*
* \warning The 'back' value does cross block boundaries, so the preceding 32kbytes of data must
* be available at all times
* 
* \param[in] pkt
* \param[in] base
* \param[out] out
* \param[in] out_size
* \param[out] lastf
* \param[in] outf
* \param[out] zcrc
*
* \retval -1 in case of any error
* \retval 0 or any positive value in successful case
*
*******************************************************************************/
static int sxm_deflate_single(SXMBitBuff *pkt, byte *base,
                              byte *out, size_t out_size,
                              uint *lastf, FILE *outf, uint *zcrc) {
    uint last = 0;
    uint n;
    uint i;
    uint *litdist = NULL;
    uint *litvalues = NULL;
    uint *litmasks = NULL;
    uint hclengths[19];
    uint hcvalues[19];
    uint hcmasks[19];
    byte *ob = out;
    byte *obe = out + out_size;
    uint nlit = 0;
    uint ndist = 0;
    uint nbit;
    int wrapped = 0;

    *lastf = NEXTR(1); /* last-block flag */
    n = NEXTR(2);      /* block compression type */

    if (0 != pkt->err) {
        ERROR_UTL("Compression Type read failed");
        return -1;
    }

    DEBUG_UTL("last: %u, mode: %u", *lastf, n);

    if (n == 0) {
        uint l1;

        /* Align to a byte boundary then read length */
        sxm_bitbuff_align(pkt);
        l1 = NEXTR(16);
        NEXTR(16);

        while ((0 != l1--) && (ob < obe)) {
            *ob++ = (byte)NEXTR(8);

            if ((NULL != outf) && ((ob - base) == ZIP_FILE_BUFFER_SIZE)) {
                *zcrc = sxm_crc32_part(*zcrc, base, ZIP_FILE_BUFFER_SIZE);
                DEBUG_UTL("Writing 64k bytes");
                if (fwrite(base, ZIP_FILE_BUFFER_SIZE, 1, outf) != 1) {
                    ERROR_UTL("fwrite failed (errno=%d)", errno);
                    return -1;
                }
                wrapped = 1;
                ob = base;
            }
        }

        if (ob >= obe) {
            return (int)(ob - out);
        }
        else {
            *ob = '\0';
            if (NULL == outf) {
                return (int)(ob - out);
            }
            return (int)(ob - out) + (0 != wrapped ? ZIP_FILE_BUFFER_SIZE : 0);
        }
    }
    else if (n == 3) {
        ERROR_UTL("Unknown Compression mode 3");
        *lastf = 1;
        return 0;
    }

    for (i = 0; i < 19; i++) {
        hclengths[i] = 0;
    }

    if (n == 1) {
        int ix = 0;

        nlit = 288;
        ndist = 32;
        nbit = 5;

        /* Allocate the literals and distance array */

        litdist = (uint*)sxe_calloc(nlit + ndist + 4, sizeof(uint));
        if (litdist == NULL) {
            ERROR_UTL("Failed to allocate litdist");
            *lastf = 1;
            return 0;
        }

        litvalues = (uint*)sxe_calloc(nlit + ndist + 4, sizeof(uint));
        if (litvalues == NULL) {
            ERROR_UTL("Failed to allocate litvalues");
            sxe_free(litdist);
            *lastf = 1;
            return 0;
        }

        litmasks = (uint*)sxe_calloc(nlit + ndist + 4, sizeof(uint));
        if (litmasks == NULL) {
            ERROR_UTL("Failed to allocate litmasks");
            sxe_free(litdist);
            sxe_free(litvalues);
            *lastf = 1;
            return 0;
        }

        for (i = 0; i < 144; i++) {
            litdist[ix++] = 8;
        }
        for (i = 144; i < 256; i++) {
            litdist[ix++] = 9;
        }
        for (i = 256; i < 280; i++) {
            litdist[ix++] = 7;
        }
        for (i = 280; i < 288; i++) {
            litdist[ix++] = 8;
        }
        for (i = 0; i < 32; i++) {
            litdist[ix++] = 5;
        }
    } else {
        /* The offsets seem to have changed from the PKZIP specification */

        nlit = NEXTR(5) + 257;
        ndist = NEXTR(5) + 1;
        nbit = NEXTR(4) + 4;

        /* Allocate the literals and distance array */

        DEBUG_UTL("nlit: %u, ndist: %u, nbit: %u", nlit, ndist, nbit);

        litdist = (uint*)sxe_calloc(nlit + ndist + 4, sizeof(uint));
        if (litdist == NULL) {
            ERROR_UTL("Failed to allocate litdist");
            *lastf = 1;
            return 0;
        }

        litvalues = (uint*)sxe_calloc(nlit + ndist + 4, sizeof(uint));
        if (litvalues == NULL) {
            ERROR_UTL("Failed to allocate litvalues");
            sxe_free(litdist);
            *lastf = 1;
            return 0;
        }

        litmasks = (uint*)sxe_calloc(nlit + ndist + 4, sizeof(uint));
        if (litmasks == NULL) {
            ERROR_UTL("Failed to allocate litmasks");
            sxe_free(litdist);
            sxe_free(litvalues);
            *lastf = 1;
            return 0;
        }

        /* Compute the Huffman table used to encode the literals and distance array */

        /* Number of bits used to represent symbol hcindex[i] */
        for (i = 0; i < nbit; i++) {
            hclengths[hcindex[i]] = NEXTR(3);
        }

        /* Build a Huffman table from this data */

        DEBUG_UTL("Building tables");
        buildtable(hclengths, 19, hcmasks, hcvalues);

        /* Now extract all the bit lengths into an array */

        i = 0;

        while (i < (nlit + ndist)) {
            uint j;

            for (j = 0; j < 19; j++) {
                if ((0 != hclengths[j]) &&
                    EATR(hcvalues[j], hclengths[j])) {
                    break;
                }
            }

            if (j == 19) {
                ERROR_UTL("Error in NLIT decode");
                sxe_free(litdist);
                sxe_free(litvalues);
                sxe_free(litmasks);
                return -1;
            }

            if (j < 16) {
                litdist[i++] = j;
                last = j;
            } else if (j == 16) {
                uint rep = NEXTR(2) + 3U;
                while (0 != rep--) {
                    litdist[i++] = last;
                }
            } else if (j == 17) {
                uint rep = NEXTR(3) + 3U;
                while (0 != rep--) {
                    litdist[i++] = 0;
                }
                last = 0;
            } else {
                uint rep = NEXTR(7)+11;
                while (0 != rep--) {
                    litdist[i++] = 0;
                }
                last = 0;
            }
        }
    }

    /* Build the two Huffman table for literal/length codes and distance codes */
    buildtable(litdist, nlit, litmasks, litvalues);
    buildtable(litdist + nlit, ndist, litmasks+nlit, litvalues+nlit);

    /* Ready to (un)roll */

    DEBUG_UTL("Ready to unroll");

    for (;;) {
        uint j;

        if (0 != pkt->err) {
            ERROR_UTL("Packet error detected");
            sxe_free(litdist);
            sxe_free(litvalues);
            sxe_free(litmasks);
            return -1;
        }

        for (j = 0; j < nlit; j++) {
            if ((0 != litdist[j]) && EATR(litvalues[j], litdist[j])) {
                break;
            }
        }

        if (j == nlit) {
            ERROR_UTL("Error in Literal decode");
            sxe_free(litdist);
            sxe_free(litvalues);
            sxe_free(litmasks);
            return -1;
        }

        if (j < 256) {
            if (ob >= obe) {
                 break;
            }
            *ob++ = (byte)j;
            if ((NULL != outf) && ((ob - base) == ZIP_FILE_BUFFER_SIZE)) {
                *zcrc = sxm_crc32_part(*zcrc, base, ZIP_FILE_BUFFER_SIZE);
                DEBUG_UTL("Writing 64k bytes");
                if (fwrite(base, ZIP_FILE_BUFFER_SIZE, 1, outf) != 1) {
                    ERROR_UTL("fwrite failed (errno=%d)", errno);
                    sxe_free(litdist);
                    sxe_free(litvalues);
                    sxe_free(litmasks);
                    return -1;
                }
                wrapped = 1;
                ob = base;
            }
        } else if (j == 256) {
            break;
        } else {
            uint back;
            uint length = 0;
            uint k;

            if (j < 265) {
                length = j - 254;
            }
            else if (j < 269) {
                uint l0 = NEXTR(1);
                length = 11 + (j - 265) * 2 + l0;
            } else if (j < 273) {
                uint l0 = NEXTR(2);
                length = 19 + (j - 269) * 4 + l0;
            } else if (j < 277) {
                uint l0 = NEXTR(3);
                length = 35 + (j - 273) * 8 + l0;
            } else if (j < 281) {
                uint l0 = NEXTR(4);
                length = 67 + (j - 277) * 16+ l0;
            } else if (j < 285) {
                uint l0 = NEXTR(5);
                length = 131 + (j - 281) * 32 + l0;
            } else if (j == 285) {
                length = 258;
            }

            for (k = 0; k < ndist; k++) {
                if ((0 != litdist[nlit + k]) &&
                    EATR(litvalues[nlit + k], litdist[nlit + k])) {
                    break;
                }
            }

            if (k == ndist) {
                ERROR_UTL("Error in Literal decode");
                sxe_free(litdist);
                sxe_free(litvalues);
                sxe_free(litmasks);
                return -1;
            }

            if (k < 4) {
                back = k + 1;
            } else {
                uint bitk = (k-2)/2;
                uint l0 = NEXTR(bitk);
                uint bb = (k % 2)*(1U << bitk);
                uint bbase = (1U << (bitk + 1)) + 1;

                back = bbase + bb + l0;
            }

            if ((0 != back) && (0 != length)) {
                byte *p = ob - back;

                if (p < base) {
                    p += ZIP_FILE_BUFFER_SIZE;
                }
                while ((0 != length--) && (ob < obe)) {
                    *ob++ = *p++;
                     if (((ob - base) == ZIP_FILE_BUFFER_SIZE) && (NULL != outf)) {
                        *zcrc = sxm_crc32_part(*zcrc, base, ZIP_FILE_BUFFER_SIZE);
                        DEBUG_UTL("Writing 64k bytes");
                        if (fwrite(base, ZIP_FILE_BUFFER_SIZE, 1, outf) != 1) {
                            ERROR_UTL("fwrite failed (errno=%d)", errno);
                            sxe_free(litdist);
                            sxe_free(litvalues);
                            sxe_free(litmasks);
                            return -1;
                        }
                        wrapped = 1;
                        ob = base;
                    }
                    if (((p - base) == ZIP_FILE_BUFFER_SIZE) && (NULL != outf)) {
                        p = base;
                    }
                }
            } else {
                ERROR_UTL("No operation? back %u length %u", back, length);
            }
        }
    }

    sxe_free(litdist);
    sxe_free(litvalues);
    sxe_free(litmasks);

    DEBUG_UTL("Expanded to %u bytes", ob - out);
    if (ob < obe) {
        *ob = '\0';
    }

    if (NULL == outf) { /* never wraps */
        return (int)(ob - out);
    }

    return (int)(ob - out) + (0 != wrapped ? ZIP_FILE_BUFFER_SIZE : 0);
}

static int sxm_deflate_to_buffer(SXMBitBuff *pBitBuf,
                                 byte *pOutBuf,
                                 size_t outBufSize,
                                 uint *pComputedCrc)
{
    int rc;
    uint last;
    size_t uncompressedSize = 0;

    *pComputedCrc = SXM_CRC32_INIT_VAL;

    do
    {
        if (uncompressedSize >= outBufSize)
        {
            ERROR_UTL("Insufficient buffer size (%u >= %u)",
                      uncompressedSize, outBufSize);
            break;
        }

        rc = sxm_deflate_single(pBitBuf, pOutBuf,
                                pOutBuf + uncompressedSize,
                                outBufSize - uncompressedSize,
                                &last, NULL, pComputedCrc);
        if (rc < 0)
        {
            return rc;
        }

        uncompressedSize += (size_t)rc;

    } while (last == 0);

    sxm_bitbuff_align(pBitBuf);

    *pComputedCrc = sxm_crc32_part(*pComputedCrc, pOutBuf,
                                   (uint)uncompressedSize);
    *pComputedCrc ^= SXM_CRC32_INIT_VAL;

    return (int)uncompressedSize;
}

static int sxm_deflate_to_file(SXMBitBuff *pBitBuf,
                               byte *pTmpBuf,
                               FILE *pFileOut,
                               size_t *pDeflatedSize,
                               uint *pComputedCrc)
{
    int rc;
    uint last;

    *pDeflatedSize = 0;
    *pComputedCrc = SXM_CRC32_INIT_VAL;

    do
    {
        rc = sxm_deflate_single(pBitBuf, pTmpBuf,
                pTmpBuf + (*pDeflatedSize % ZIP_FILE_BUFFER_SIZE),
                ZIP_FILE_BUFFER_SIZE - (*pDeflatedSize % ZIP_FILE_BUFFER_SIZE),
                &last, pFileOut, pComputedCrc);
        if (rc < 0)
        {
            return SXM_E_ERROR;
        }

        *pDeflatedSize += (size_t)rc;

    } while (0 == last);

    if (0 != (*pDeflatedSize % ZIP_FILE_BUFFER_SIZE))
    {
        *pComputedCrc =
            sxm_crc32_part(*pComputedCrc, pTmpBuf,
                *pDeflatedSize % ZIP_FILE_BUFFER_SIZE);

        if (1 != fwrite(pTmpBuf,
                        *pDeflatedSize % ZIP_FILE_BUFFER_SIZE,
                        1, pFileOut))
        {
            ERROR_UTL("fwrite failed");
            return SXM_E_PIPE;
        }
    }

    sxm_bitbuff_align(pBitBuf);

    *pComputedCrc ^= SXM_CRC32_INIT_VAL;

    return SXM_E_OK;
}

static int sxm_deflate_copy_to_buffer(SXMBitBuff *pBitBuf,
                                      byte *pOutBuf,
                                      size_t dataSize,
                                      uint *pComputedCrc)
{
    if (dataSize != sxm_bitbuff_read_bytes(pBitBuf, dataSize, pOutBuf))
    {
        ERROR_UTL("Unable to read %u bytes", dataSize);
        return SXM_E_PIPE;
    }

    *pComputedCrc = sxm_crc32_calculate(pOutBuf, (uint)dataSize);

    return SXM_E_OK;
}

static int sxm_copy_to_file(SXMBitBuff *pBitBuf,
                            byte *pTmpBuf,
                            FILE *pFileOut,
                            size_t *pDataSize, /* in, out */
                            uint *pComputedCrc)
{
    size_t numBytesWritten = 0;
    size_t numBytesToWrite;

    *pComputedCrc = SXM_CRC32_INIT_VAL;

    while (numBytesWritten < (*pDataSize))
    {
        numBytesToWrite =
            MIN((*pDataSize - numBytesWritten), ZIP_FILE_BUFFER_SIZE);

        if (numBytesToWrite !=
            sxm_bitbuff_read_bytes(pBitBuf, numBytesToWrite, pTmpBuf))
        {
            ERROR_UTL("Unable to read %u data bytes", numBytesToWrite);
            return SXM_E_PIPE;
        }

        if (1 != fwrite(pTmpBuf, numBytesToWrite, 1, pFileOut))
        {
            ERROR_UTL("fwrite failed (errno=%d)", errno);
            return SXM_E_PIPE;
        }

        *pComputedCrc = sxm_crc32_part(*pComputedCrc,
                                       pTmpBuf,
                                       (uint)numBytesToWrite);

        numBytesWritten += numBytesToWrite;
    }

    *pDataSize = numBytesWritten;
    *pComputedCrc ^= SXM_CRC32_INIT_VAL;

    return SXM_E_OK;
}

/***************************************************************************//**
* Reads and verifies (G)ZIP (RFC 1952) header
*
* \param[in] pkt a Bit Buffer pointing to the start of the compressed data
*
* \return
*
*******************************************************************************/
static int sxm_deflate_verify_gzip_header(SXMBitBuff *pkt)
{
    uint t = NEXT_BYTE(); /* ID1 */
    uint f;

    t = (t << 8) | NEXT_BYTE(); /* ID2 */

    if (t != 0x1f8b)
    {
        ERROR_UTL("Not a GZIP file");
        return SXM_E_UNSUPPORTED;
    }

    if (NEXT_BYTE() != 8) /* CM */
    {
        ERROR_UTL("Not a FLATE compressed");
        return SXM_E_UNSUPPORTED;;
    }

    f = NEXT_BYTE(); /* FLG */

    SKIP(32 + 8 + 8); /* MTIME, XLF and OS */

    /* Skip any extra data in the file */

    if (f & 4) /* FLG.FEXTRA */
    {
        t = NEXT_BYTE();
        t = (t << 8) | NEXT_BYTE();

        SKIP(t * SXM_ARCH_BITS_IN_BYTE);
    }

    if (0 != (f & 8)) /* FLG.FNAME */
    {
        do
        {
            t = NEXT_BYTE();
        } while (0 != t);
    }

    if (0 != (f & 16)) /* FLG.FCOMMENT */
    {
        do
        {
            t = NEXT_BYTE();
        } while (0 != t);
    }

    if (0 != pkt->err)
    {
        ERROR_UTL("GZIP header reading failed");
        return SXM_E_ERROR;
    }

    return SXM_E_OK;
}

/***************************************************************************//**
* Reads and verifies PKZIP Local File Header
*
* \param[in] pkt a Bit Buffer pointing to the start of the compressed data
*
* \return
*
*******************************************************************************/
static int sxm_deflate_extract_pkzip_header(SXMBitBuff *pkt,
                                            PKZipLocalFileHeader *pHeader,
                                            size_t *pHeaderLen)
{
    pHeader->signature = NEXTR(SXM_PKZIP_SIGNATURE_BITLEN); /* Signature */

    if (PKZIP_LFH_SIGNATURE != pHeader->signature)
    {
        /* No more header info found */
        return SXM_E_NOENT;
    }

    pHeader->version = (UINT16)NEXTR(SXM_PKZIP_VERSION_BITLEN);
    pHeader->generalPurposeBitFlag = (UINT16)NEXTR(SXM_PKZIP_FLAGS_BITLEN);
    pHeader->compressionMethod = (UINT16)NEXTR(SXM_PKZIP_COMPRESSION_BITLEN);
    pHeader->time = (UINT16)NEXTR(SXM_PKZIP_MOD_TIME_BITLEN);
    pHeader->date = (UINT16)NEXTR(SXM_PKZIP_MOD_DATE_BITLEN);
    pHeader->crc = NEXTR(SXM_PKZIP_CRC_BITLEN);
    pHeader->compressedSize = NEXTR(SXM_PKZIP_COMPRESSED_SIZE_BITLEN);
    pHeader->uncompressedSize = NEXTR(SXM_PKZIP_UNCOMPRESSED_SIZE_BITLEN);
    pHeader->filenameLength = (UINT16)NEXTR(SXM_PKZIP_FILE_NAME_LEN_BITLEN);
    pHeader->extraFieldLength = (UINT16)NEXTR(SXM_PKZIP_EXTRA_FIELD_LEN_BITLEN);

    if (pHeader->filenameLength >= SXM_PKZIP_FILENAME_LEN_MAX)
    {
        ERROR_UTL("File name is too long (%u >= %u)",
                  pHeader->filenameLength,
                  SXM_PKZIP_FILENAME_LEN_MAX);
        return SXM_E_UNSUPPORTED;
    }

    /* Get file name */
    sxm_bitbuff_read_bytes(pkt, pHeader->filenameLength,
                           (byte*)&pHeader->fileName[0]);
    pHeader->fileName[pHeader->filenameLength] = '\0';

    SKIP(pHeader->extraFieldLength * SXM_ARCH_BITS_IN_BYTE);

    if (0 != pkt->err)
    {
        ERROR_UTL("Header reading failed");
        return SXM_E_ERROR;
    }

    if (NULL != pHeaderLen)
    {
        *pHeaderLen = (size_t)SXM_PKZIP_HEADER_BASE_LEN +
                      pHeader->filenameLength +
                      pHeader->extraFieldLength;
    }

    return SXM_E_OK;
}

static int sxm_deflate_pkzip_verify_data(SXMBitBuff *pkt,
                                         PKZipLocalFileHeader *pHeader,
                                         uint computedCrc,
                                         size_t actualUncompressedSize)
{
    if (PKZIP_DATA_DESCRIPTOR_MASK ==
        (pHeader->generalPurposeBitFlag & PKZIP_DATA_DESCRIPTOR_MASK))
    {
        /* Read CRC, compressed size and uncompressed size */
        pHeader->crc = NEXTR(32);
        pHeader->compressedSize = NEXTR(32);
        pHeader->uncompressedSize = NEXTR(32);
    }

    if ((computedCrc != pHeader->crc) ||
        (actualUncompressedSize != pHeader->uncompressedSize))
    {
        ERROR_UTL("PKZip verification failed. File '%s'. "
                  "Size %u (actual %u). Read CRC %08x (computed %08x)",
                  pHeader->fileName, pHeader->uncompressedSize,
                  actualUncompressedSize, pHeader->crc, computedCrc);
        return SXM_E_CORRUPT;
    }

    return SXM_E_OK;
}

static int sxm_deflate_pkzip_extract_to_buffer(SXMBitBuff *pBitBuf,
                                               PKZipLocalFileHeader *pHeader,
                                               byte *pOutbuf)
{
    int rc = SXM_E_OK;
    uint computedCrc = 0;
    size_t uncompressedDataSize = 0;

    if ((PKZIP_COMPRESSION_DEFLATE == pHeader->compressionMethod) ||
        (PKZIP_COMPRESSION_EDEFLATE == pHeader->compressionMethod))
    {
        rc = sxm_deflate_to_buffer(pBitBuf, pOutbuf,
                                   pHeader->uncompressedSize,
                                   &computedCrc);
        if (rc > 0)
        {
            uncompressedDataSize = (size_t)rc;
            rc = SXM_E_OK;
        }
        else
        {
            rc = SXM_E_ERROR;
        }
    }
    else if (PKZIP_NO_COMPRESSION == pHeader->compressionMethod)
    {
        rc = sxm_deflate_copy_to_buffer(pBitBuf, pOutbuf,
                                        pHeader->uncompressedSize,
                                        &computedCrc);

        uncompressedDataSize = pHeader->uncompressedSize;
    }
    else
    {
        ERROR_UTL("Unknown compression (%u) for '%s'",
                  pHeader->compressionMethod, pHeader->fileName);
        rc = SXM_E_UNSUPPORTED;
    }

    if (SXM_E_OK == rc)
    {
        rc = sxm_deflate_pkzip_verify_data(pBitBuf,
                                           pHeader,
                                           computedCrc,
                                           uncompressedDataSize);
    }

    return rc;
}

static int sxm_deflate_pkzip_extract_to_file(SXMBitBuff *pBitBuf,
                                             PKZipLocalFileHeader *pHeader,
                                             byte *pTmpbuf,
                                             FILE *pFileOut)
{
    int rc = SXM_E_OK;
    uint computedCrc = 0;
    size_t uncompressedDataSize = 0;

    if ((PKZIP_COMPRESSION_DEFLATE == pHeader->compressionMethod) ||
        (PKZIP_COMPRESSION_EDEFLATE == pHeader->compressionMethod))
    {
        rc = sxm_deflate_to_file(pBitBuf, pTmpbuf, pFileOut,
                                 &uncompressedDataSize,
                                 &computedCrc);
    }
    else if (PKZIP_NO_COMPRESSION == pHeader->compressionMethod)
    {
        uncompressedDataSize = pHeader->uncompressedSize;

        rc = sxm_copy_to_file(pBitBuf, pTmpbuf, pFileOut,
                              &uncompressedDataSize,
                              &computedCrc);
    }
    else
    {
        ERROR_UTL("Unknown compression (%u) for '%s'",
                  pHeader->compressionMethod,
                  pHeader->fileName);
        rc = SXM_E_UNSUPPORTED;
    }

    if (SXM_E_OK == rc)
    {
        rc = sxm_deflate_pkzip_verify_data(pBitBuf,
                                           pHeader,
                                           computedCrc,
                                           uncompressedDataSize);
    }

    return rc;
}

/***************************************************************************//**
* Deflates bit buffer
* 
* \note The various protocol elements that use flate compression will give
*       details of how to determine the appropriate size for the \c out buffer.
*
* \param[in] pkt a Bit Buffer pointing to the start of the compressed data
* \param[out] out a byte array sufficiently large to hold the uncompressed output
* \param[in] size the \p out buffer size in bytes
*
* \return length of expanded data, or -1 if error
*
*******************************************************************************/
int sxm_deflate(SXMBitBuff *pkt, byte *out, size_t out_size)
{
    uint crc;

    return sxm_deflate_to_buffer(pkt, out, out_size, &crc);
}

/***************************************************************************//**
* Deflates (G)ZIP bit buffer
*
* \param[in] pkt a Bit Buffer pointing to the start of the compressed data
* \param[out] out a byte array sufficiently large to hold the uncompressed output
* \param[in] out_size size of the \p out in bytes
*
* \return length of expanded data, or -1 if error
*
*******************************************************************************/
int sxm_deflate_zipbuffer(SXMBitBuff *pkt, byte *out, size_t out_size)
{
    if (SXM_E_OK == sxm_deflate_verify_gzip_header(pkt))
    {
        uint crc;

        return sxm_deflate_to_buffer(pkt, out, out_size, &crc);
    }

    return -1;
}

/***************************************************************************//**
* Deflates (G)ZIP file
*
* \param[in] in input file
* \param[in] out output file
*
* \return
*
*******************************************************************************/
int sxm_deflate_zipfile(FILE *in, FILE *out)
{
    int rc;
    SXMBitFileBuff fileBitBuf;
    SXMBitBuff *pkt = &fileBitBuf.b;
    byte *pTmpBuf;
    uint computedCrc;
    size_t actualUncompressedSize;
    uint readCrc = 0;
    size_t readUncompressedSize = 0;

    rc = sxm_bitfilebuff_setup(&fileBitBuf, in, 0, 0);
    if (SXM_E_OK != rc)
    {
        ERROR_UTL("sxm_bitfilebuff_setup failed (rc=%d)", rc);
        return -1;
    }

    rc = sxm_deflate_verify_gzip_header(pkt);
    if (SXM_E_OK != rc)
    {
        return -1;
    }

    pTmpBuf = (byte*)sxe_malloc(ZIP_FILE_BUFFER_SIZE);
    if (NULL == pTmpBuf)
    {
        ERROR_UTL("Failed to allocate output buffer");
        return -1;
    }

    rc = sxm_deflate_to_file(&fileBitBuf.b, pTmpBuf, out,
                             &actualUncompressedSize, &computedCrc);
    if (SXM_E_OK == rc)
    {
        /* The end of the file contains the uncompressed file CRC
           and uncompressed file length */
        readCrc = NEXTR(32);
        readUncompressedSize = NEXTR(32);

        if ((0 != pkt->err) ||
            (readCrc != computedCrc) ||
            (readUncompressedSize != actualUncompressedSize))
        {
            ERROR_UTL("GZip verification failed. Size %u (actual %u)."
                      " Read CRC %08x (Computed CRC %08x)",
                      readUncompressedSize, actualUncompressedSize,
                      readCrc, computedCrc);
            rc = -1;
        }
        else
        {
            rc = (int)actualUncompressedSize;
        }
    }

    sxe_free(pTmpBuf);

    return rc;
}

/***************************************************************************//**
 *
 * Deflates (PK)ZIP file content into the specified file system location.
 *
 * \param[in] pFile input file
 * \param[in] pOutDir output directory
 * \param[in] pDetectExtension expected deflated file extension
 * \param[in] fileCallback routine called on each file extraction
 * \param[in] pUserData pointer to a caller data passed to the fileCallback
 *
 * \return One of SXe error codes. If fileCallback return is not SXM_E_OK,
 *         sxm_deflate_pkzip_file will fail with provided error.
 * \retval SXM_E_OK     Success
 * \retval SXM_E_FAULT  One of required parameters is NULL
 * \retval SXM_E_NOMEM  Memory allocation failed
 * \retval SXM_E_PIPE   File operation failed
 * \retval SXM_E_ERROR  General error
 *
*******************************************************************************/
int sxm_deflate_pkzip_file(FILE *pFile,
                           const char *pOutDir,
                           const char *pDetectExtension,
                           SXM_PKZIP_CALLBACK fileCallback,
                           void *pUserData)
{
    SXMBitFileBuff fileBitbuf;
    byte *pTmpbuf;
    int rc = SXM_E_OK;
    PKZipLocalFileHeader header;
    SXMPKZipFileDesc fileDesc;
    char fullFileName[SXM_PKZIP_FILENAME_LEN_MAX];
    size_t headerSize = 0;
    size_t currentOffset = 0;

    if ((NULL == pFile) || (NULL == pOutDir))
    {
        return SXM_E_FAULT;
    }

    rc = sxm_bitfilebuff_setup(&fileBitbuf, pFile, 0, 0);
    if (SXM_E_OK != rc)
    {
        ERROR_UTL("bitfilebuff_setup failed (rc=%d)", rc);
        return rc;
    }

    /* Read the first header */
    fileDesc.fileOffset = 0;
    rc = sxm_deflate_extract_pkzip_header(&fileBitbuf.b, &header, &headerSize);
    if (SXM_E_OK != rc)
    {
        return rc;
    }

    pTmpbuf = (byte*)sxe_malloc(ZIP_FILE_BUFFER_SIZE);
    if (NULL == pTmpbuf)
    {
        ERROR_UTL("Failed to allocate output buffer");
        return SXM_E_NOMEM;
    }

    while (SXM_E_OK == rc)
    {
        /* Create full path to file been deflated */
        snprintf(&fullFileName[0], sizeof(fullFileName) - 1,
                 "%s/%s", pOutDir, &header.fileName[0]);

        if ((header.fileName[header.filenameLength - 1] == '\\') ||
            (header.fileName[header.filenameLength - 1] == '/'))
        {
            /* Create new directory */
            if (os_mkdir(&fullFileName[0]) != 0)
            {
                ERROR_UTL("mkdir failed (errno=%d) for '%s'",
                          errno, &fullFileName[0]);
                rc = SXM_E_PIPE;
                break;
            }

            if ((NULL != fileCallback) && (NULL == pDetectExtension))
            {
                fileDesc.isDirectory = TRUE;
                fileDesc.pFileName = &fullFileName[0];
                fileDesc.uncompressedSize = header.uncompressedSize;

                rc = fileCallback(&fileDesc, pUserData);
                if (SXM_E_OK != rc)
                {
                    ERROR_UTL("fileCallback error (rc=%d)", rc);
                    break;
                }
            }
        }
        else
        {
            FILE *pFileOut = fopen(&fullFileName[0], "wb");
            if (NULL == pFileOut)
            {
                ERROR_UTL("Failed to open file: %s", &fullFileName[0]);
                rc = SXM_E_PIPE;
                break;
            }

            rc = sxm_deflate_pkzip_extract_to_file(&fileBitbuf.b,
                                                   &header,
                                                   pTmpbuf,
                                                   pFileOut);

            fclose(pFileOut);

            if (SXM_E_OK != rc)
            {
                break;
            }

            if (NULL != fileCallback)
            {
                if ((NULL == pDetectExtension) ||
                    (NULL != strstr(&fullFileName[0], pDetectExtension)))
                {
                    fileDesc.isDirectory = FALSE;
                    fileDesc.pFileName = &fullFileName[0];
                    fileDesc.uncompressedSize = header.uncompressedSize;

                    rc = fileCallback(&fileDesc, pUserData);
                    if (SXM_E_OK != rc)
                    {
                        ERROR_UTL("fileCallback error (rc=%d)", rc);
                        break;
                    }
                }
            }
        }

        /* Update current position */
        currentOffset += headerSize + header.compressedSize;

        /* Save next header position */
        fileDesc.fileOffset = currentOffset;

        /* Read next header */
        rc = sxm_deflate_extract_pkzip_header(&fileBitbuf.b, &header, &headerSize);
    }

    if (SXM_E_NOENT == rc)
    {
        /* End of file */
        rc = SXM_E_OK;
    }

    sxe_free(pTmpbuf);

    return rc;
}

int sxm_deflate_pkzip_enumerate(FILE *pFile,
                                SXM_PKZIP_CALLBACK fileCallback,
                                void *pUserData)
{
    SXMBitFileBuff fileBitbuf;
    int rc = SXM_E_OK;
    PKZipLocalFileHeader header;
    SXMPKZipFileDesc fileDesc;
    size_t headerSize = 0;
    size_t currentOffset = 0;

    if ((NULL == pFile) || (NULL == fileCallback))
    {
        return SXM_E_FAULT;
    }

    rc = sxm_bitfilebuff_setup(&fileBitbuf, pFile, 0, 0);
    if (SXM_E_OK != rc)
    {
        ERROR_UTL("bitfilebuff_setup failed (rc=%d)", rc);
        return rc;
    }

    /* Read the first header */
    fileDesc.fileOffset = 0;
    rc = sxm_deflate_extract_pkzip_header(&fileBitbuf.b, &header, &headerSize);
    if (SXM_E_OK != rc)
    {
        return rc;
    }

    while (SXM_E_OK == rc)
    {
        if ((header.fileName[header.filenameLength - 1] == '/') ||
            (header.fileName[header.filenameLength - 1] == '\\'))
        {
            fileDesc.isDirectory = TRUE;
        }
        else
        {
            /* Skip file content if uncompressed size is available */
            if (PKZIP_DATA_DESCRIPTOR_MASK ==
                (header.generalPurposeBitFlag & PKZIP_DATA_DESCRIPTOR_MASK))
            {
                /* Uncompressed size is saved in the data descriptor section
                   after compessed file data */
                ERROR_UTL("Uncompressed size is unavailable");
                rc = SXM_E_UNSUPPORTED;
                break;
            }
            //TODO: More effective way would be to seek. Investigate.
            sxm_bitbuff_skip(&fileBitbuf.b,
                header.compressedSize * SXM_ARCH_BITS_IN_BYTE);

            fileDesc.isDirectory = FALSE;
        }

        fileDesc.pFileName = &header.fileName[0];
        fileDesc.uncompressedSize = header.uncompressedSize;

        rc = fileCallback(&fileDesc, pUserData);
        if (SXM_E_OK != rc)
        {
            ERROR_UTL("fileCallback error (rc=%d)", rc);
            break;
        }

        /* Update current position */
        currentOffset += headerSize + header.compressedSize;

        /* Save next header position */
        fileDesc.fileOffset = currentOffset;

        /* Read next header */
        rc = sxm_deflate_extract_pkzip_header(&fileBitbuf.b, &header, &headerSize);
    }

    if (SXM_E_NOENT == rc)
    {
        /* End of file */
        rc = SXM_E_OK;
    }

    return rc;
}

int sxm_deflate_pkzip_read(FILE *pFile,
                           size_t fileOffset,
                           void *pOutBuf,
                           size_t outBufSize)
{
    int rc = SXM_E_OK;
    SXMBitFileBuff fileBitbuf;
    PKZipLocalFileHeader header;

    if ((NULL == pFile) || (NULL == pOutBuf))
    {
        return SXM_E_FAULT;
    }

    rc = sxm_bitfilebuff_setup(&fileBitbuf, pFile, (uint)fileOffset, 0);
    if (SXM_E_OK != rc)
    {
        ERROR_UTL("bitfilebuff_setup failed (rc=%d)", rc);
        return rc;
    }

    rc = sxm_deflate_extract_pkzip_header(&fileBitbuf.b, &header, NULL);
    if (SXM_E_OK != rc)
    {
        return rc;
    }

    if (outBufSize < header.uncompressedSize)
    {
        ERROR_UTL("Insufficient buffer size (%u < %u)",
                  outBufSize, header.uncompressedSize);
        return SXM_E_RESOURCE;
    }

    rc = sxm_deflate_pkzip_extract_to_buffer(&fileBitbuf.b, &header, pOutBuf);

    return rc;
}

int sxm_deflate_pkzip_alloc_and_read(FILE *pFile,
                                     size_t fileOffset,
                                     void **ppOutBuf,
                                     size_t *pDataSize)
{
    int rc = SXM_E_OK;
    SXMBitFileBuff fileBitbuf;
    PKZipLocalFileHeader header;

    if ((NULL == pFile) || (NULL == ppOutBuf))
    {
        return SXM_E_FAULT;
    }

    rc = sxm_bitfilebuff_setup(&fileBitbuf, pFile, (uint)fileOffset, 0);
    if (SXM_E_OK != rc)
    {
        ERROR_UTL("bitfilebuff_setup failed (rc=%d)", rc);
        return rc;
    }

    rc = sxm_deflate_extract_pkzip_header(&fileBitbuf.b, &header, NULL);
    if (SXM_E_OK != rc)
    {
        return rc;
    }

    *ppOutBuf = sxe_malloc(header.uncompressedSize);
    if (NULL == *ppOutBuf)
    {
        ERROR_UTL("Buffer allocation failed");
        return SXM_E_NOMEM;
    }

    rc = sxm_deflate_pkzip_extract_to_buffer(&fileBitbuf.b,
                                             &header, *ppOutBuf);
    if (SXM_E_OK == rc)
    {
        if (NULL != pDataSize)
        {
            *pDataSize = header.uncompressedSize;
        }
    }
    else
    {
        sxe_free(*ppOutBuf);
        *ppOutBuf = NULL;
    }

    return rc;
}

int sxm_deflate_pkzip_copy(FILE *pFileSource,
                           size_t fileOffset,
                           FILE *pFileOut)
{
    SXMBitFileBuff fileBitbuf;
    PKZipLocalFileHeader header;
    byte *pTmpbuf;
    int rc = SXM_E_OK;

    if ((NULL == pFileSource) || (NULL == pFileOut))
    {
        return SXM_E_FAULT;
    }

    rc = sxm_bitfilebuff_setup(&fileBitbuf, pFileSource, (uint)fileOffset, 0);
    if (SXM_E_OK != rc)
    {
        ERROR_UTL("bitfilebuff_setup failed (rc=%d)", rc);
        return rc;
    }

    /* Read the header */
    rc = sxm_deflate_extract_pkzip_header(&fileBitbuf.b, &header, NULL);
    if (SXM_E_OK != rc)
    {
        return rc;
    }

    /* Allocate temporary buffer */
    pTmpbuf = (byte*)sxe_malloc(ZIP_FILE_BUFFER_SIZE);
    if (NULL == pTmpbuf)
    {
        ERROR_UTL("Failed to allocate output buffer");
        return SXM_E_NOMEM;
    }

    rc = sxm_deflate_pkzip_extract_to_file(&fileBitbuf.b,
                                           &header,
                                           pTmpbuf,
                                           pFileOut);

    sxe_free(pTmpbuf);

    return rc;
}
