/******************************************************************************/
/*                Copyright (c) Sirius XM Satellite Radio, Inc.               */
/*                            All Rights Reserved                             */
/*      Licensed Materials - Property of Sirius XM Satellite Radio, Inc.      */
/******************************************************************************/
/***************************************************************************//**
 *
 *  \file sxm_common_internal.h
 *  \author Leslie French
 *  \date 8/20/2013
 *  \brief Common Types and definitions for the SDK interface and
 *         the utility layer routines
 *
 ******************************************************************************/

#ifndef SXM_COMMON_INTERNAL_H
#define SXM_COMMON_INTERNAL_H
#include <pthread.h>

#include <sxm_build.h>
#include <sxm_typ.h>
#include <sxm_common.h>
#include <sxm_stdtrace.h>
#include <util/sxm_cfile.h>
#include <util/sxm_tfile.h>
#include <util/sxm_memory_internal.h>
#include <util/sxm_time_internal.h>
#include <util/sxm_platform_internal.h>
#include <util/sxm_bytebuff.h>
#include <util/sxm_bitbuff.h>
#include <util/sxm_deflate.h>
#if (SXM_USE_GEN8)
#include <sxi8/sxm_sxi8_internal.h>
#else
#include <sxi/sxm_sxi_internal.h>
#endif
/************************************************************************
 *                                                                      *
 *            Pthreads                                                  *
 *            ========                                                  *
 *                                                                      *
 *    All pthread mutexes must be initialized non-recursive.            *
 *    This is to ensure consistency across platforms.                   *
 *                                                                      *
 *    sxm_mutexattr_normal is a global variable that contains the       *
 *    correct initialization for a SDK mutex.                           *
 *                                                                      *
 ************************************************************************/
extern pthread_mutexattr_t sxm_mutexattr_normal;

/** Initialize mutex with default attributes
 * \param[in] _m the ptrhead mutex for initialization
 */
#define sxm_mutex_init(_m)    pthread_mutex_init(_m, &sxm_mutexattr_normal)

/** Destroys mutex
 * \param[in] _m the ptrhead mutex for initialization
 */
#define sxm_mutex_destroy(_m) pthread_mutex_destroy(_m)

/** Packed data */
typedef struct
{
    byte *packet;   //!< incoming Access Unit
    uint pl;        //!< packet length
    uint crc;       //!< packet CRC (once checked)
} SXMPacket;

/************************************************************************
 *                                                                      *
 *            RFD related defines                                       *
 *            ===================                                       *
 *                                                                      *
 ************************************************************************/

/**
 * Defines file name for temporary RFD file created by
 * sxm_rfd_file_copy() function
 */
#define SXM_RFD_TEMP_FILE_NAME  "update"
#define SXM_RFD_SERVICE_NAME    "rfd"

/************************************************************************
 *                                                                      *
 *            File Operations                                           *
 *            ===============                                           *
 *                                                                      *
 ************************************************************************/


extern SXESDK_INTERNAL_API int sxm_file_load_type(char,          //type
                                                  const char *,  // module
                                                  const char *,  // file
                                                  void *,        // buffer
                                                  size_t,        // max
                                                  size_t *);     // size

extern SXESDK_INTERNAL_API int sxm_file_load(const char *,  // module
                                             const char *,  // file
                                             void *,        // buffer
                                             size_t,        // max
                                             size_t *);     // size

extern SXESDK_INTERNAL_API int sxm_file_open_dev(const char *); // type

extern SXESDK_INTERNAL_API int sxm_file_save(const char *,      // module
                                             const char *,      // file
                                             const void *,      // buffer
                                             size_t );          // size

extern SXESDK_INTERNAL_API char *sxm_file_make_path(char,         // type
                                                    const char *, // module
                                                    const char *, // file
                                                    char *,       // out
                                                    size_t);      // size

extern SXESDK_INTERNAL_API int sxm_sdk_set_makepath(SXMMakePathCallback_t); // make_path_func

extern SXESDK_INTERNAL_API SXMMakePathCallback_t sxm_sdk_get_makepath(void);

/************************************************************************
 *                                                                      *
 *            CRCs                                                      *
 *            ====                                                      *
 *                                                                      *
 ************************************************************************/

/** Standard CRC-32 polynomial */
#define SXM_CRC32_POLYNOMIAL        (0xEDB88320U)
/** Standard CRC-32 initial value */
#define SXM_CRC32_INIT_VAL          (0xFFFFFFFFU)

extern SXESDK_INTERNAL_API ushort sxm_crc16_part(ushort crc,
                                                 const void *buf,
                                                 uint len);

extern SXESDK_INTERNAL_API ushort sxm_crc16_calculate(const void *buf,
                                                      uint len);

extern SXESDK_INTERNAL_API void sxm_crc16_initialize(void);

extern SXESDK_INTERNAL_API uint sxm_crc32_calculate(const void *buf,
                                                    uint len);

extern SXESDK_INTERNAL_API uint sxm_crc32_check(const void *buf,
                                                uint len,
                                                uint *crc);

extern SXESDK_INTERNAL_API void sxm_crc32_initialize(void);

extern SXESDK_INTERNAL_API uint sxm_crc32_part(uint crc,
                                               const void *buf,
                                               uint len);

/************************************************************************
 *                                                                      *
 *            RFD Decoding                                              *
 *            ============                                              *
 *                                                                      *
 ************************************************************************/

/** Defines max number of collection per service */
#define SXM_RFD_MAX_COLLECTION              (4)
/** Defines max package size in bytes */
#define SXM_RFD_PACKET_SIZE_MAX             (5120)  // bytes

/**
 * \name RFD Algorithms
 * @{
 */
#define SXM_RFD_ALGORITHM_DIRECT_BLOCK      (0)
#define SXM_RFD_ALGORITHM_GF2               (1)
#define SXM_RFD_ALGORITHM_GF4               (2)
/** @} */

/** \name Compression types
 * @{
 */
/** No data compression type */
#define SXM_RFD_COMPRESSION_NONE            (0)
/** GZIP data compression type */
#define SXM_RFD_COMPRESSION_GZIP            (1)
/** @} */

//  supports up to 4 simultaneous collections
//  current maximum AU size is 5,120 bytes  (5 SDTP packets)
//  for a 4,096 byte RFD blocksize
typedef struct
{
    char service[SXM_SERVICE_NAME_LEN_MAX];
    SXMSem_t semvar;
    int going;
    int (*collected)(SXMRfdMetadata *); //!< Pointer to the function called when an RFD fragment is completely processes (returns SXM_E_OK if fragment can be deleted)
    int slot[SXM_RFD_MAX_COLLECTION];   //!< An index into the RFD Services global slot store
    int fid[SXM_RFD_MAX_COLLECTION];    //!< Fragment ID of the fragment being received
    int state[SXM_RFD_MAX_COLLECTION];  //!< 1 if RFD fragment has been completely processed, 0 otherwise
    int carsize;
    byte packet[SXM_RFD_PACKET_SIZE_MAX];
    int pl;
} SXMRfdService;

extern SXESDK_INTERNAL_API SXMRfdMetadata *sxm_rfd_extract_meta(int);   // is

extern SXESDK_INTERNAL_API int sxm_rfd_extract_permil_complete(int);

extern SXESDK_INTERNAL_API int sxm_rfd_find_all(SXMRfdService *);   // service

extern SXESDK_INTERNAL_API int sxm_rfd_find_new(const SXMRfdMetadata *, // rfd
                                                const char *,           // module
                                                int);                   // frag

extern SXESDK_INTERNAL_API int sxm_rfd_init(void);

extern SXESDK_INTERNAL_API void sxm_rfd_uninit(void);

extern SXESDK_INTERNAL_API int sxm_rfd_parse(SXMBitBuff *,          // pkt
                                             SXMRfdMetadata *);     // out

extern SXESDK_INTERNAL_API int sxm_rfd_remove(int);                 // is

extern SXESDK_INTERNAL_API int sxm_rfd_collect(SXMRfdService *);    // rfds

extern SXESDK_INTERNAL_API void sxm_rfd_stop_processing(SXMRfdService *); // rfds

extern SXESDK_INTERNAL_API int sxm_rfd_file_copy(ptr,               // lli
                                                 SXMRfdMetadata *,  // rfd
                                                  char,             // type
                                                  const char *,     // module
                                                  const char *);    // file

extern SXESDK_INTERNAL_API int sxm_rfd_file_buffer(ptr,             // lli
                                                   SXMRfdMetadata *,// rfd
                                                   SXMBitBuff **);  // out

extern SXESDK_INTERNAL_API void sxm_rfd_service_stop(SXMRfdService *);  // rfds

extern SXESDK_INTERNAL_API int sxm_rfd_cfile_clean(void);

/************************************************************************
 *                                                                      *
 *            SDTP Re-assembly                                          *
 *            ================                                          *
 *                                                                      *
 ************************************************************************/

#define SXM_SDTP_MAX_SIZE                           1024

typedef struct
{
    byte *buffer;
    uint max;
    uint pl;
    uint seq;
    int state;
} SXMSdtp;

extern SXESDK_INTERNAL_API int sxm_sdtp_add(SXMSdtp *,      // this
                                            byte *);        // b

extern SXESDK_INTERNAL_API void sxm_sdtp_delete(SXMSdtp *); // this

extern SXESDK_INTERNAL_API SXMSdtp *sxm_sdtp_new(uint);     // maxau

/************************************************************************
 *                                                                      *
 *            LLI                                                       *
 *            ===                                                       *
 *                                                                      *
 ************************************************************************/

extern SXESDK_INTERNAL_API void sxm_lli_time_update(const SXMTime *);  // t

#endif /* SXM_COMMON_H */
