/******************************************************************************/
/*                Copyright (c) Sirius XM Satellite Radio, Inc.               */
/*                            All Rights Reserved                             */
/*      Licensed Materials - Property of Sirius XM Satellite Radio, Inc.      */
/******************************************************************************/
/***************************************************************************//**
 *
 * \file sxm_cfile.c
 * \author Leslie French
 * \date 8/20/2013
 * \brief Engine-cycle files. Please, refer to \ref cfile_page for implementation
 *        details.
 *
 * \page cfile_page C-File (Engine-Cycle Files)
 *
 * \section cfile_page_intro Introduction
 *
 * A cfile is used to preserve data over an engine cycle.  These are transient
 * data used to reduce the time needed to initialize and present the data display
 * to the customer at the next engine-on.  An example of cycle-file data is the
 * most-recently received fuel prices for the area around the vehicle.
 *
 * Cycle files have different properties from other file types used by the SXe SDK
 *  -# They will be written to more frequently than other files
 *  -# The write may not complete, if the power is removed too quickly
 *  -# All of the data stored in the cycle file can be recovered from the
 *     broadcast, given time
 *
 * Other properties of cycle data include:
 *  -# Stored data may be grouped into separate areas which can be updated and
 *     validated independently.
 *  -# Some stored data may be time-sensitive, depending on the service, so the age
 *     of the saved data are important.
 *  -# Much of the stored data may be received Access Units, or other objects of
 *     variable size and number.
 *
 * These different aspects are handled by a common disk-file format and a support
 * library.  A Cycle File (cfile) is a copy of an area of memory, that can be saved
 * (written) to disk and restored (read) from disk, though not necessarily at the
 * same memory address.  The cycle file is constructed from a number of 512-byte blocks,
 * corresponding to disk sectors.  Groups of contiguous blocks can be considered as
 * a separate section of the cfile, and read and written independently. Each section
 * is represented by a \ref SXMCFileSection structure.
 *
 * The first sector of every cycle file contains a common header (which always accommodates
 * integer number of sectors) - \ref SXMCFileRootBlock
 *
 * By way of example, the Fuel Service cycle file is 647 blocks (sectors) long,
 * divided as:
 * | Data Type     | Blocks |
 * |:--------------|:------:|
 * | Root          | 1      |
 * | BaseLine data | 4      |
 * | Texts         | 2      |
 * | AU store      | 640    |
 *
 * The AU (Access Unit) store is further structured as an array of 64 5,120-byte
 * AU locations. The first 64 bits of the map field in the root block are used to
 * record which of the AU locations are in use in the file.  The version field in
 * sections[1] (the texts section) holds the version of the received data,
 * extracted from the protocol, so subsequent price data can check that they are
 * using the correct text table.
 *
 * \section cfile_page_def Defining a Cycle File
 *
 * Each module within the SXe SDK that needs engine-cycle storage defines its own
 * format for the file.  For example, the SXi global metadata uses a cycle file.
 * Without considering the cycle-file structure, that service would like to see the
 * following:
 * ~~~{.c}
 * struct
 * {
 *     SXMLeague league[MAX_LEAGUE];
 *     SXMTeam team[MAX_TEAM];
 *     SXMTraffic traffic[MAX_TRAFFIC];
 *     SXMPreset preset[MAX_PRESET];
 * };
 * ~~~
 * where \c SXMLeague, \c SXMTeam, \c SXMTraffic and \c SXMPreset are defined
 * structures used to hold instances of those global metadata records, and the
 * \c MAX_ values are determined by the SXi protocol and EMI specification. Since
 * these can all be updated independently, it makes sense to create each of them
 * as a separate section within the file. Two macros are defined to help with this:
 * - The \ref CSIZE macro calculate the number of sectors required to hold an array of
 *   structures.
 * - The \ref CSECT macro defines a cfile section within a data structure definition.
 *   The \ref CSECT macro ensures that each section occupies an integral number of
 *   blocks in the file
 *
 * This gives the full definition of the cycle file structure as:
 * ~~~{.c}
 * typedef struct
 * {
 *    SXMCFileRootBlock g;
 *    CSECT(SXMLeague, league, MAX_LEAGUE) p0;
 *    CSECT(SXMTeam, team, MAX_TEAM) p1;
 *    CSECT(SXMTraffic, traffic, MAX_TRAFFIC) p2;
 *    CSECT(SXMPreset, preset, MAX_PRESET) p3;
 * } SXICycle;
 * ~~~
 *
 * The original data elements are then accessed as:
 * ~~~{.c}
 * cycle->p0.league
 * cycle->p1.team
 * cycle->p2.traffic
 * cycle->p3.preset
 * ~~~
 *
 * Continuing with the previous example, the four sections in the SXi cycle file are created
 * with the following four calls, in order:
 *
 * ~~~{.c}
 * sxm_cfile_add_section(root, SECTION_LEAGUE, CSIZE(SXMLeague, MAX_LEAGUE));
 * sxm_cfile_add_section(root, SECTION_TEAM, CSIZE(SXMTeam, MAX_TEAM));
 * sxm_cfile_add_section(root, SECTION_TRAFFIC, CSIZE(SXMTraffic, MAX_TRAFFIC));
 * sxm_cfile_add_section(root, SECTION_PRESET, CSIZE(SXMPreset, MAX_PRESET));
 * ~~~
 *
 *
 ******************************************************************************/

#include <stddef.h>
#include <errno.h>
#include <stdint.h>

#include <util/sxm_cfile.h>

#ifndef SDKFILES_STANDALONE_BUILD
#include <sxm_sdk.h>
#include <util/sxm_common_internal.h>
#include <util/sxm_noname_internal.h>
#else
#include "sdkfiles.h"
#endif

STATIC_ASSERT(!(sizeof(SXMCFileRootBlock) % sizeof(SXMSector)),
    SXMCFileRootBlock_must_be_divided_by_sizeof_SXMSector);

/** Defines root size in sectors */
#define SXM_CFILE_ROOT_SIZE_IN_SECTORS \
    ((sizeof(SXMCFileRootBlock) + sizeof(SXMSector) - 1) / sizeof(SXMSector))

/** Allows to check section ID validity */
#define SXM_CFILE_SECTION_ISVALID_ID(_root, _sno) \
    ((_sno) < (uint)(ARRAY_SIZE((_root)->sections)))

/** Internal T-File Logging macro definition */
#ifndef CF_LOG
#define CF_LOG(_f, ...)  /* sxm_file_log("%s: " _f, __FUNCTION__, ##__VA_ARGS__) */
#endif

/** \name Utility function(s)
 * @{
 */
/***************************************************************************//**
 *
 * This function returns local time
 * \return Value of local time in minutes since 00:00 Jan 1, 2012
 *         0 in case of an error
 *
 ******************************************************************************/
uint sxm_cfile_time_now(void)
{
    SXMTime t;
    int rc;
    memset(&t, 0, sizeof(t));
    rc = sxm_sxi_extract_time(&t);
    return (rc == SXM_E_NOENT) ? 0U : sxm_sxe_time_encode(&t);
}

/***************************************************************************//**
 * This function calculates section CRC32
 * \param[in] root c-file root
 * \param[in] sec section descriptor
 * \return CRC32 value
 ******************************************************************************/
static uint cfile_section_crc32(const SXMCFileRootBlock *root, const SXMCFileSection *sec)
{
    uint crc;
    if (sec->mask & SXM_CFILE_MASK_HEAP) {
        int idx;
        const uint item_len = sec->length / sec->count;
        crc = SXM_CRC32_INIT_VAL;
        for (idx = 0; idx < sec->count; ++idx) {
            /* Is block used? */
            if (!BITP(root->map, sec->heap_map_off + idx) /* is used? */) {
                crc = sxm_crc32_part(crc,
                        (const SXMSector*)root + sec->off + (uint)idx * item_len,
                        (uint)sizeof(SXMSector) * item_len);
                CF_LOG("[%s|root=%p] sec #%d, += elem #%d (bit #%d)",
                    root->format, root, (int)(sec - &root->sections[0]),
                    idx, (int)(sec->heap_map_off + idx));
            }
        }
        crc ^= SXM_CRC32_INIT_VAL;
    }
    else {
        crc = sxm_crc32_calculate((const SXMSector*)root + sec->off,
                                  (uint)sizeof(SXMSector) * sec->length);
    }
    return crc;
}

/***************************************************************************//**
 * The routine writes part of a cycle file to disk
 *
 * \param[in] self a pointer to an open cycle file
 * \param[in] offset the sector offset as which to start writing
 * \param[in] scount the number of sectors to write
 * \param[in] buffer a pointer to the data to be written
 *
 * \return the number of sectors written, or -1 on error
 *
 ******************************************************************************/
static int cfile_write(FILE *self, uint offset, uint scount, const void *buffer) {
    int rc = -1;
    switch (0) { default: {
        const uint absoffset = offset * (uint)sizeof(SXMSector);
        rc = (int)ftell(self);
        if (rc < 0) {
            rc = -1;
            CF_LOG("[self=%p]: failed to gather current position %d",
                self, errno);
            break;
        }
        /* Move to the new position if needed */
        if (rc != (int)absoffset) {
            rc = fseek(self, (long)absoffset, SEEK_SET);
            if (rc != 0) {
                rc = -1;
                CF_LOG("[self=%p]: fseek(%u) failed (errno=%d)",
                    self, absoffset, errno);
                break;
            }
            CF_LOG("[self=%p]: fseek()'ed to %d", self, (int)absoffset);
        }

        rc = (int)fwrite(buffer, sizeof(SXMSector), scount, self);
        if ((uint)rc != scount) {
            CF_LOG(":failed to write %d by %d block(s), errno=%d",
                (int)sizeof(SXMSector), (int)scount, errno);
        }
    }}

    return rc;
}

/***************************************************************************//**
 * This function calculates section CRC32
 * \param[in] self a pointer to an open cycle file
 * \param[in] root c-file root
 * \param[in] sec section descriptor
 * \param[out] pWBlocks writes statistics; can be \p NULL
 * \return number of written sectors or -1 in case of error
 ******************************************************************************/
static int cfile_section_write(FILE* self, const SXMCFileRootBlock *root,
                               SXMCFileSection *sec, size_t *pWBlocks)
{
    int rc, realRc;
    CF_LOG("[self=%p, %s|root=%p, section=%d]",
        self, root->format, root, (int)(sec - &root->sections[0]));

    /* Dynamic section can be written by parts only in case if it has been
     * written to the file, thus, file is big enough to accommodate whole
     * section. Thus, if section has never been written it shall be processed
     * as regular one.
     */
    if ((sec->mask & SXM_CFILE_MASK_HEAP) && (sec->ts > 0U)) {
        /* Write to the file only parts which has corresponding bit is set */
        uint idx;
        const uint item_len = sec->length / sec->count;
        realRc = 0; /* Assume all ok */
        for (idx = 0U; idx < sec->count; ++idx) {
            /* Is block used? */
            if (!BITP(root->map, sec->heap_map_off + idx) /* is used? */) {
                int partRc;
                partRc = cfile_write(self, sec->off + idx * item_len, item_len,
                    (const SXMSector*)root + sec->off + idx * item_len);
                if ((uint)partRc != item_len) {
                    realRc = -1;
                    break;
                }
                realRc += partRc;
                CF_LOG("[%s|root=%p] sec #%d, += elem #%d (bit #%d), rc=%d",
                    root->format, root, (int)(sec - &root->sections[0]),
                    idx, (int)(sec->heap_map_off + idx), partRc);
            }
        }
        /* Adjust return value since the caller expect whole sections
         * written, but in fact only a few or even no blocks were
         * put to the file.
         */
        rc = (realRc >= 0) ? sec->length : -1;
    }
    else {
        CF_LOG("[%s|root=%p] sec #%d, writing all",
            root->format, root, (int)(sec - &root->sections[0]));
        /* Write whole section to the file */
        realRc = cfile_write(self, sec->off, sec->length,
                         (const SXMSector*)root + sec->off);
        rc = (realRc != sec->length) ? -1 : realRc;
    }
    if (rc > 0) {
        const int intRc = fflush(self);
        if (intRc == 0) {
            sec->ts = sxm_cfile_time_now();
            sec->crc = cfile_section_crc32(root, sec);
            sec->mask &= (ushort)~SXM_CFILE_MASK_CORRUPTED;
            /* If requested add as many as was properly written during this call */
            if (pWBlocks != NULL) {
                *pWBlocks += (size_t)realRc;
            }
        }
        else {
            rc = intRc;
        }
    }
    CF_LOG("[%s|root=%p] rc=%d", root->format, root, rc);
    return rc;
}

/**************************************************************************//**
 * The routine will create a section in an empty file. The sections should
 * be created in the correct order, starting with ix = 0.
 *
 * \param[in] root a pointer to the root block for the cfile
 * \param[in] sno the section index
 * \param[in] len the number of sectors in this section
 *
 *****************************************************************************/
static void cfile_add_section(SXMCFileRootBlock *root, uint sno, uint len)
{
    SXMCFileSection * const sec = &root->sections[sno];

    memset(sec, 0, sizeof(*sec));
    if (sno == 0U) {
        /* Calculate offset taking into account the root block size
         * in sectors
         */
        sec->off = SXM_CFILE_ROOT_SIZE_IN_SECTORS;
    }
    else {
        /* Assuming that section are being added in order the previous
         * section is the last added one and its off can be used as-is
         */
        sec->off = (ushort)((sec - 1)->off + (sec - 1)->length);
    }
    sec->length = (ushort)len;
    sec->mask = SXM_CFILE_MASK_NONE;

    CF_LOG("[%s|root=%p]: added sec #%d (len=%d, off=%d)",
        root->format, root, (int)sno, (int)len, (int)sec->off);
}
/** @} */

/**
 * \name Sections' usage
 * @{
 */

/**************************************************************************//**
 * The routine returns the memory pointer of the start of a section within
 * the loaded file
 *
 * \param[in] root a pointer to the root block for the cfile
 * \param[in] sno the section number
 *
 * \return a pointer to the start of that section or \c NULL in case of error
 *
 *****************************************************************************/
void* sxm_cfile_base_section(SXMCFileRootBlock *root, uint sno) {
    if (root && SXM_CFILE_SECTION_ISVALID_ID(root, sno)) {
        return (SXMSector*)root + root->sections[sno].off;
    }
    return NULL;
}

/** @} */

/**
 * \name Usage of the resource map
 * @{
 */

/**************************************************************************//**
 * The routine returns a contiguous set of bits from the map, setting the
 * bits to in-use.
 *
 * \note There is no requirement that the returned value corresponds to
 * any position within the cycle file, though the most common use for the map
 * is to manage the heap of stored access units
 *
 * \param[in] root a pointer to the root block for the cfile
 * \param[in] sno the section number, -1 is supported to use map w/o section
 * \param[in] count the number of bits to reserve
 *
 * \retval -1 no free block of that length
 * \retval >=0 starting index in the map of the alllocated bits
 *
 *****************************************************************************/
int sxm_cfile_map_alloc(SXMCFileRootBlock *root, uint sno, uint count)
{
    int rc = -1;

    if (root && (count > 0U)) {
        uint idx, sidx, eidx;
        uint * const mask = &root->map[0];
        uint idx_last = 0U;
        uint seq_len = 0U;

        /* Choose proper set of bits to operate with */
        if (SXM_CFILE_SECTION_ISVALID_ID(root, sno) &&
            (root->sections[sno].mask & SXM_CFILE_MASK_HEAP)) {
            const SXMCFileSection *sec = &root->sections[sno];
            CF_LOG("[%s|root=%p]: dsec #%d (count=%d)",
                root->format, root, sno, count);
            sidx = sec->heap_map_off;
            eidx = sidx + sec->count;
        }
        else {
            CF_LOG("[%s|root=%p]: sec #%d (count=%d)",
                root->format, root, sno, count);
            sidx = 0;
            eidx = SXM_CFILE_MASK_ITEMS_COUNT;
        }

        /* Find available slot(s) by bit in root->map bit array */
        for (idx = sidx; idx < eidx; ++idx) {
            if (BITP(mask, idx)) {
                if (!seq_len) {
                    /* This is a first found unused block,
                     * thus, save it
                     */
                    idx_last = idx;
                }
                ++seq_len;
                if (seq_len == count) {
                    /* Found the sequence */
                    break;
                }
            }
            else {
                /* Oops, need to find next continuous block of '1' bits */
                seq_len = 0;
            }
        }

        /* Found required amount of unused blocks */
        if (seq_len == count) {
            /* Mark all blocks as used */
            for (idx = idx_last; idx < (idx_last + seq_len); ++idx) {
                BITC(mask, idx);
            }
            rc = (int)(idx_last - sidx);
            CF_LOG("[%s|root=%p]: alloc sec #%d [%u:%u bit(s)], rc=%d",
                root->format, root, sno, idx_last, idx_last + seq_len - 1, rc);
        }
        else {
            CF_LOG("[%s|root=%p]: failed for sec #%d, rc=%d",
                root->format, root, sno, rc);
        }
    }
    return rc;
}

/**************************************************************************//**
 * The routine returns a group of bits to the free pool.
 *
 * \note There is no requirement that the returned value corresponds to
 * any position within the cycle file, though the most common use for the map
 * is to manage the heap of stored access units
 *
 * \param[in] root a pointer to the root block for the cfile
 * \param[in] sno the section number, -1 is supported to use map w/o section
 * \param[in] start the starting bit number
 * \param[in] len the number of bits to mark free
 *
 * \retval SXM_E_OK Section has been added
 * \retval SXM_E_FAULT NULL parameter(s)
 * \retval SXM_E_INVAL Invalid argument
 *
 *****************************************************************************/
int sxm_cfile_map_free(SXMCFileRootBlock *root, uint sno, uint start, uint len) {
    SXMResultCode rc;
    if (!root) {
        rc = SXM_E_FAULT;
    }
    else if (len == 0U) {
        rc = SXM_E_INVAL;
    }
    else {
        uint idx, sidx, eidx;
        uint * const mask = &root->map[0];

        if (SXM_CFILE_SECTION_ISVALID_ID(root, sno) &&
            (root->sections[sno].mask & SXM_CFILE_MASK_HEAP)) {
            const SXMCFileSection *sec = &root->sections[sno];

            sidx = sec->heap_map_off + start;
            eidx = sidx + len;
            if (eidx > (uint)(sec->heap_map_off + sec->count)) {
                eidx = (uint)(sec->heap_map_off + sec->count);
            }
            CF_LOG("[%s|root=%p]: d-sec #%d (start=%d (sidx=%u), len=%d (eidx=%u))",
                root->format, root, sno, start, sidx, len, eidx);
        }
        /* Use map in legacy mode */
        else {
            sidx = start;
            eidx = sidx + len;
            if (eidx > SXM_CFILE_MASK_ITEMS_COUNT) {
                eidx = SXM_CFILE_MASK_ITEMS_COUNT;
            }
            CF_LOG("[%s|root=%p]: sec #%d (start=%d, len=%d (eidx=%u))",
                root->format, root, sno, start, len, eidx);
        }

        /* Update bits accordingly */
        for (idx = sidx; idx < eidx; ++idx) {
            BITS(mask, idx);
        }
        rc = SXM_E_OK;
    }
    return rc;
}

/** @} */

/**
 * \name C-File validation
 * @{
 */

/**************************************************************************//**
 * This routine can be used to check an individual section.
 *
 * \note If crc2 is good, then the whole file should be good.
 *
 * \param[in] root a pointer to the root block for the cfile
 * \param[in] sno the section index, previously initialized
 * \param[in] bDoDeepCheck allows to avoid doing CRC32 calculation again
 *                         over the section if this function is called
 *                         after doing overall file validation via
 *                         sxm_cfile_validate(). Use \p TRUE to enforce
 *                         deep check using CRC32 calculation.
 *
 * \retval 0 if the section-CRC is incorrect
 * \retval 1 if the section-CRC is correct
 *
 *****************************************************************************/
int sxm_cfile_check(SXMCFileRootBlock *root, uint sno, BOOL bDoDeepCheck) {
    int rc = 0;
    if (root && SXM_CFILE_SECTION_ISVALID_ID(root, sno)) {
        const SXMCFileSection *sec = &root->sections[sno];
        if (sec->length) {
            if (bDoDeepCheck == TRUE) {
                rc = (cfile_section_crc32(root, sec) == sec->crc);
                CF_LOG("[%s|root=%p]: checking crc32 sec #%d, rc=%d",
                    root->format, root, sno, rc);
            }
            else {
                CF_LOG("[%s|root=%p]: checking no-crc32 sec #%d, rc=%d",
                    root->format, root, sno, rc);
                rc = !(sec->mask & SXM_CFILE_MASK_CORRUPTED);
            }
        }
    }
    return rc;
}

/**************************************************************************//**
 * The function allows C-File user checking the data format via providing
 * required format and schema version
 *
 * \note If crc2 is good, then the whole file should be good.
 *
 * \param[in] root a pointer to the root block for the cfile
 * \param[in] format required C-File format
 * \param[in] schema required C-File content schema version
 *
 * \retval SXM_E_OK The C-File has desired format
 * \retval SXM_E_FAULT NULL parameters
 * \retval SXM_E_ERROR Error in cfile check
 *
 *****************************************************************************/
int sxm_cfile_check_format(SXMCFileRootBlock *root,
                           const char *format, uint *schema)
{
    int rc;
    if (!root || !format) {
        rc = SXM_E_FAULT;
    }
    else {
        rc = ((root->magic == SXM_CFILE_MAGIC) &&
                (root->cver == SXM_CFILE_VERSION) &&
                !strncmp(root->format, format, sizeof(root->format)))
                                ? SXM_E_OK : SXM_E_ERROR;
        if ((rc == SXM_E_OK) && schema) {
            *schema = root->sver;
        }
    }
    return rc;
}

/**************************************************************************//**
 * The sxm_cfile_validate routine does the basic checking of
 * the root block CRCs
 *
 * If root block is bad, the root block is corrupt, and the file should be
 * discarded and restarted.  If root block's CRC is good, but one of section's CRC,
 * is bad, then one or more of the sections did not save correctly.
 * The application can check each section individually, or simply discard
 * the whole file
 *
 * \param[in] root a pointer to the root block for the cfile
 * \param[in] blocks the number of blocks in the file
 *
 * \retval SXM_CFILE_INVALID Root block's CRC is bad
 * \retval SXM_CFILE_PARTIALY_VALID Root block's CRC is good, while one of sections has bad CRC
 * \retval SXM_CFILE_FULLY_VALID All CRCs are good, thus, file considered as valid
 *
 *****************************************************************************/
SXMCFileValidationResult sxm_cfile_validate(SXMCFileRootBlock *root)
{
    SXMCFileValidationResult rc;

    if (!root) {
        rc = SXM_CFILE_INVALID;
    }
    /* Do calculation over the root to make sure the root
     * contains trusted data
     */
    else if (sxm_crc32_calculate(root, offsetof(SXMCFileRootBlock, crc)) == root->crc) {
        SXMCFileSection *sec = &root->sections[0];
        const SXMCFileSection * const esec = &root->sections[ARRAY_SIZE(root->sections)];
        /* Do calculation over only valid blocks */
        rc = SXM_CFILE_FULLY_VALID; /* Assuming all is valid */
        CF_LOG("[%s|root=%p] validating file", root->format, root);
        for (; sec < esec; ++sec) {
            if (sec->length) {
                if (cfile_section_crc32(root, sec) == sec->crc) {
                    sec->mask &= (ushort)~SXM_CFILE_MASK_CORRUPTED;
                    CF_LOG("[%s|root=%p] good sec #%d",
                        root->format, root, (int)(sec - &root->sections[0]));
                }
                else {
                    sec->mask |= SXM_CFILE_MASK_CORRUPTED;
                    CF_LOG("[%s|root=%p] failed sno #%d crc check",
                        root->format, root, (int)(sec - &root->sections[0]));
                    rc = SXM_CFILE_PARTIALY_VALID; /* Not all sections are valid. */
                }
            }
            else {
                CF_LOG("[%s|root=%p] unused sec #%d",
                    root->format, root, (int)(sec - &root->sections[0]));
            }
        }
    }
    else {
        CF_LOG("[%s|root=%p]: failed root crc check", root->format, root);
        rc = SXM_CFILE_INVALID;
    }

    return rc;
}
/** @} */


/**
 * \name C-File Setup Routines
 * The first time a cfile is opened, it has no structure. The following routines
 * can be used to set up the appropriate values in the root and structure blocks
 * @{
 */
/**************************************************************************//**
 * The function allows specifying format of the file content.
 *
 * \param[in] root a pointer to the root block for the cfile
 * \param[in] format required C-File format
 * \param[in] sver required C-File content schema version
 *
 *****************************************************************************/
void sxm_cfile_set_format(SXMCFileRootBlock *root, const char *format, uint sver)
{
    if (root && format) {
        CF_LOG("[%s|root=%p, format=\'%s\', sver=%u", root->format, root, format, sver);
        root->magic = SXM_CFILE_MAGIC;
        root->cver = SXM_CFILE_VERSION;
        root->sver = sver;
        strncpy(root->format, format, sizeof(root->format) - 1);
        root->format[sizeof(root->format) - 1] = '\0';
    }
}

/**************************************************************************//**
 * The routine will create a section in an empty file. The sections should
 * be created in the correct order, starting with ix = 0.
 *
 * \param[in] root a pointer to the root block for the cfile
 * \param[in] sno the section index
 * \param[in] len the number of sectors in this section
 *
 * \retval SXM_E_OK Section has been added
 * \retval SXM_E_FAULT NULL parameter(s)
 * \retval SXM_E_INVAL Invalid argument
 *
 *****************************************************************************/
int sxm_cfile_add_section(SXMCFileRootBlock *root, uint sno, uint len)
{
    SXMResultCode rc;
    if (!root || !len) {
        rc = SXM_E_FAULT;
    }
    else if (!SXM_CFILE_SECTION_ISVALID_ID(root, sno)) {
        rc = SXM_E_INVAL;
    }
    else {
        cfile_add_section(root, sno, len);
        rc = SXM_E_OK;
    }
    return rc;
 }

/**************************************************************************//**
 * The routine will create a dynamic/heap section in an empty file. The sections
 * should be created in the correct order, starting with ix = 0.
 *
 * After creation the section has own set of bits inside the map-mask and the
 * owner is responcible for managing them via sxm_cfile_map_alloc() and
 * sxm_cfile_map_free() functions. In turn, C-File engine is responsible for
 * processing only blocks which are marked used.
 *
 * \sa CDSEC
 *
 * \param[in] root a pointer to the root block for the cfile
 * \param[in] sno the section index
 * \param[in] size single element size in bytes
 * \param[in] count the number of elements in this section
 *
 * \retval SXM_E_OK Section has been added
 * \retval SXM_E_FAULT NULL parameter(s)
 * \retval SXM_E_INVAL Invalid argument
 *
 *****************************************************************************/
int sxm_cfile_add_dsection(SXMCFileRootBlock *root, uint sno, uint size, uint count)
{
    SXMResultCode rc;
    if (!root || !size || !count) {
        rc = SXM_E_FAULT;
    }
    else if (!SXM_CFILE_SECTION_ISVALID_ID(root, sno)) {
        rc = SXM_E_INVAL;
    }
    else {
        SXMCFileSection *sec = &root->sections[sno];
        const SXMCFileSection *bsec = &root->sections[0];
        const SXMCFileSection * const esec = &root->sections[ARRAY_SIZE(root->sections)];
        ushort heap_map_off = 0U;
        ushort idx;

        /* Find the latest used bit in map */
        for (; bsec < esec; ++bsec) {
            if ((bsec != sec) && (bsec->mask & SXM_CFILE_MASK_HEAP) &&
                ((bsec->heap_map_off + bsec->count) > heap_map_off)) {
                heap_map_off = (ushort)(bsec->heap_map_off + bsec->count);
            }
        }

        if ((heap_map_off + count) <= (ARRAY_SIZE(root->map) * SXM_ARCH_INT_BITS)) {
            /* Add this new section as usual */
            cfile_add_section(root, sno,
                count * ((size + (uint)sizeof(SXMSector) - 1U) / (uint)sizeof(SXMSector)));
            sec->mask |= SXM_CFILE_MASK_HEAP;
            sec->heap_map_off = heap_map_off;
            sec->count = (ushort)count;
            /* Make sure bits which belong to the section are set */
            for (idx = sec->heap_map_off;
                  (idx < sec->heap_map_off + sec->count) && 
                  (idx < ARRAY_SIZE(root->map) * SXM_ARCH_INT_BITS); 
                 ++idx) {
                BITS(root->map, idx);
            }

            CF_LOG("[%s|root=%p]: added sec #%d as d/heap (size=%d, count=%d, heap_map_off=%d)",
                root->format, root, (int)sno, (int)size, (int)count, (int)sec->heap_map_off);
            rc = SXM_E_OK;
        }
        else {
            rc = SXM_E_INVAL;
        }

    }
    return rc;
}

/**************************************************************************//**
 * The routine will clear the data area for a section to zeros.
 *
 * \note This does not write the data to disk, it just clears the local memory.
 *
 * \param[in] root a pointer to the root block for the cfile
 * \param[in] sno the section index, previously initialized
 *
 * \retval SXM_E_OK Section has been added
 * \retval SXM_E_FAULT NULL parameter(s)
 * \retval SXM_E_INVAL Invalid argument
 *
 *****************************************************************************/
int sxm_cfile_clear_section(SXMCFileRootBlock *root, uint sno) {
    SXMResultCode rc;
    if (!root) {
        rc = SXM_E_FAULT;
    }
    else if (!SXM_CFILE_SECTION_ISVALID_ID(root, sno)) {
        rc = SXM_E_INVAL;
    }
    else {
        SXMCFileSection *sec = &root->sections[sno];
        CF_LOG("[%s|root=%p]: cleaning sec #%d", root->format, root, (int)sno);
        memset((SXMSector*)root + sec->off, 0, sec->length * sizeof(SXMSector));
        if (sec->mask & SXM_CFILE_MASK_HEAP) {
            /* If the section is dynamic need to clean up all related bits */
            rc = sxm_cfile_map_free(root, sno, 0, sec->count);
        }
        else {
            rc = SXM_E_OK;
        }
    }
    return rc;
}

/**************************************************************************//**
 * The routine initializes the map area in a root block to represent a number
 * of free blocks.
 *
 * The map routines assume that \c '1' in a bit position indicates a free block
 * and \c '0' is an in-use or not available block.
 *
 * \warning this function shall not be used in new code and shall be remove
 *          from exiting one via dynamic/heap sections usage.
 *
 * \param[in] root a pointer to the root block for the cfile
 * \param[in] sno the section number, -1 is supported to use map w/o section
 * \param[in] fq the number of free-queue elements, ignored for if \p sno is not -1
 *
 * \retval SXM_E_OK Successful initialization
 * \retval SXM_E_FAULT NULL parameter(s)
 * \retval SXM_E_INVAL Invalid argument
 *
 *****************************************************************************/
int sxm_cfile_map_init(SXMCFileRootBlock *root, uint sno, uint fq) {

    SXMResultCode rc = SXM_E_OK;

    if (root) {
        CF_LOG("[%s|root=%p]: int map for section %d, %d elements", root->format, root, sno, fq);
        if ((sno == (uint)-1) && (fq > 0) && (fq <= (SXM_ARCH_INT_BITS * ARRAY_SIZE(root->map)))) {
            const ushort full_items = (ushort)((fq / SXM_ARCH_INT_BITS) * SXM_ARCH_INT_SIZE);
            memset(&root->map[full_items / SXM_ARCH_INT_SIZE], 0, sizeof(root->map) - full_items);
            /* Fill full map items using memset */
            if (full_items > 0U) {
                memset(&root->map[0], 0xFF, full_items);
                fq -= (ushort)(full_items * SXM_ARCH_BITS_IN_BYTE);
            }
            /* Fill up remaining bits */
            if (fq > 0U) {
                root->map[full_items / SXM_ARCH_INT_SIZE] = (1U << fq) - 1U;
            }
        }
        else if (SXM_CFILE_SECTION_ISVALID_ID(root, sno) &&
            (root->sections[sno].mask & SXM_CFILE_MASK_HEAP)) {
            SXMCFileSection *sec = &root->sections[sno];
            uint count;
            uint items_to_set = 0U;
            uint reminder = sec->heap_map_off % SXM_ARCH_INT_BITS;
            uint offset = sec->heap_map_off / SXM_ARCH_INT_BITS;
            /* fill 1st map item */
            if (reminder > 0U) {
                items_to_set = MIN((uint)SXM_ARCH_INT_BITS - reminder, sec->count);
                root->map[offset++] |= (((1U << items_to_set) - 1U) << reminder);
            }
            count = (ushort)(sec->count - items_to_set);
            /* Fill full map items using memset */
            if (count > 0U) {
                items_to_set = (count / (uint)SXM_ARCH_INT_BITS) * (uint)SXM_ARCH_INT_SIZE;
                if (items_to_set > 0U) {
                    memset(&root->map[offset], 0xFF, items_to_set);
                    offset += items_to_set / (uint)SXM_ARCH_INT_SIZE;
                    count -= items_to_set * (uint)SXM_ARCH_BITS_IN_BYTE;
                }

                /* Fill up remaining bits */
                if (count > 0U) {
                    root->map[offset] |= ((1U << count) - 1U);
                }
            }
        }
        else {
            rc = SXM_E_INVAL;
        }
    }
    else {
        rc = SXM_E_FAULT;
    }

    return rc;
}
/** @} */

/**
 * \name Basic CFile I/O
 * @{
 */

/**************************************************************************//**
 * The routine reloads the data in the file into memory.
 *
 * At this point the data in the file have not been validated. If the file
 * did not exist, and was created on open, the return value is 0 (indicating
 * no blocks read), but the return pointer in ret will have been initialized
 * to point to the full sized area.
 *
 * \note The assumption that the \p size is always divided by sizeof(SXMSector).
 *
 * \param[in] self a pointer to an open cycle file
 * \param[in] size the number of byte expected to be loaded from the file.
 * \param[out] ret a pointer to an area pointer which is initialized from
 *                 the file data
 *
 * \return the number of sectors read from the file, or -1 if there was an error
 *
 *****************************************************************************/
int sxm_cfile_alloc_and_load(FILE *self, size_t size, ptr *ret) {
    int rc = -1;
    if (self && size && ret) {
        const size_t scount = (size + sizeof(SXMSector) - 1) / sizeof(SXMSector);
        CF_LOG("[self=%p]: loading %d byte(s) in %d block(s)",
            self, (int)size, (int)scount);
        *ret = (byte*)sxe_malloc(scount * sizeof(SXMSector));
        if (*ret) {
            if (fseek(self, 0, SEEK_SET) == 0) {
                rc = (int)fread(*ret, sizeof(SXMSector), scount, self);
                if (rc == (int)scount) {
                    CF_LOG("[self=%p]: read %d block(s) as requested",
                        self, (int)scount);
                }
                else {
                    CF_LOG("[self=%p]: failed to read %d block(s), rc=%d",
                        self, (int)scount, rc);
                }
            }
            else {
                rc = 0;
            }
        }
    }

    return rc;
}

/***************************************************************************//**
 * The  routine closes a cycle file, optionally committing the root block to disk.
 *
 * If \c root is non-NULL, the root block is first written to disk using
 * \ref sxm_cfile_commit. The file is then closed.
 *
 * \param[in] self a pointer to an open cycle file
 * \param[in] root a pointer to the root block for the file, or NULL
 *
 ******************************************************************************/
void sxm_cfile_close(FILE *self, SXMCFileRootBlock *root) {
    CF_LOG("[self=%p, %s|root=%p]", self, (root ? root->format : "(null)"), root);
    if (self != NULL) {
        if (root != NULL) {
            (void) sxm_cfile_commit(self, root, NULL);
        }
        sxm_file_close(self);
    }
}

/***************************************************************************//**
 * The routine writes the root block with the correct CRC and timestamp.
 *
 * For section-based cycle files, individual sections can be written to
 * disk, and the CRC values maintained in the root block.
 *
 * \param[in] self a pointer to an open cycle file
 * \param[in] root a pointer to the root block for the file, or NULL
* \param[out] pStat I/O statistics for this call; can be \p NULL
 *
 * \return 1 on success, 0 or -1 on error
 *
 ******************************************************************************/
int sxm_cfile_commit(FILE *self, SXMCFileRootBlock *root, SXMIOStat *pStat) {
    int rc = -1;
    if ((self != NULL) && (root != NULL)) {
        SXMCFileSection *sec = &root->sections[0];
        const SXMCFileSection * const esec = &root->sections[ARRAY_SIZE(root->sections)];
        size_t wBlocks = 0;

        CF_LOG("[self=%p, %s|root=%p] committing, crc=%08x",
            self, root->format, root, root->crc);

        /* Step 1: Calculate CRC over all sections */
        for (; sec < esec; ++sec) {
            if (sec->length > 0U) {
                const uint new_crc = cfile_section_crc32(root, sec);
                CF_LOG("[self=%p, root=%p] sec #%d has been committed (%08x -> %08x)",
                    self, root, (int)(sec - &root->sections[0]), sec->crc, new_crc);
                sec->crc = new_crc;
                sec->mask &= (ushort)~SXM_CFILE_MASK_CORRUPTED;
            }
            else {
                CF_LOG("[self=%p, root=%p] sec #%d has been skipped",
                    self, root, (int)(sec - &root->sections[0]));
            }
        }

        /* Step 2: Calculate CRC over the root block */
        root->ts = sxm_cfile_time_now();
        root->crc = sxm_crc32_calculate(root, offsetof(SXMCFileRootBlock, crc));
        CF_LOG("[self=%p, %s|root=%p]] committing root sector(s), crc=%08x",
            self, root->format, root, root->crc);

        /* Step 3: Write root to the file */
        rc = sxm_cfile_write(self, 0U, SXM_CFILE_ROOT_SIZE_IN_SECTORS, root);
        if (rc != SXM_CFILE_ROOT_SIZE_IN_SECTORS) {
            wBlocks = 0U;
            rc = -1;
            CF_LOG("[self=%p, %s|root=%p] failed to write root, errno=%d",
                self, root->format, root, errno);
        }
        else {
            wBlocks = SXM_CFILE_ROOT_SIZE_IN_SECTORS;
            CF_LOG("[self=%p, %s|root=%p] to root has been written",
                self, root->format, root);
        }
        /* populate the result */
        if (pStat != NULL) {
            pStat->sectorsWritten += wBlocks;
        }

    }
    return rc;
}

/***************************************************************************//**
 * The sxm_cfile_open routine opens a cycle file
 *
 * \note The file must reside in the ['T'](\ref sxe_file_page) portion of the
 *       filing-system namespace. If the file does not exist, the routine will
 *       attempt to create it, so it is not necessary to pre-populate the
 *       filing system with cycle files
 *
 * \param[in] module the name of the module requesting the file
 * \param[in] file the name of the file, relative to the module name
 *
 * \return pointer to the open file, or NULL in case of error
 *
 ******************************************************************************/
FILE *sxm_cfile_open(const char *module, const char *file) {
    FILE *ret = sxm_file_open_or_create('T', module, file, "r+b");
    CF_LOG(":opened file %s/%s as %p",
        (module ? module : "(null)"),  (file ? file : "(null)"), ret);
    return ret;
}

/***************************************************************************//**
 * The sxm_cfile_secure routine writes a section to disk, and updates the
 * section values in the root block
 *
 * \param[in] self a pointer to an open cycle file
 * \param[in] root a pointer to the root block for that file
 * \param[in] sno the section number to commit to disk
 * \param[out] pStat I/O statistics for this call; can be \p NULL
 *
 * \return the number of sectors written to disk, or -1 for error
 *
 ******************************************************************************/
int sxm_cfile_secure(FILE *self, SXMCFileRootBlock *root, uint sno, SXMIOStat *pStat) {
    int rc = -1;
    if (self && root && SXM_CFILE_SECTION_ISVALID_ID(root, sno)) {
        SXMCFileSection * sec = &root->sections[sno];
        const uint old_crc = sec->crc;
        size_t wBlocks = 0U;
        UNUSED_VAR(old_crc);
        rc = cfile_section_write(self, root, sec, &wBlocks);
        CF_LOG("[self=%p, %s|root=%p] sec #%u (%08x -> %08x), rc=%d, written=%u",
            self, root->format, root, sno, old_crc, sec->crc, rc, (uint)wBlocks);
        if (pStat != NULL) {
            pStat->sectorsWritten += wBlocks;
        }
    }
    return rc;
}

/***************************************************************************//**
 * The routine writes all or part of a cycle file to disk
 *
 * \param[in] self a pointer to an open cycle file
 * \param[in] offset the sector offset as which to start writing
 * \param[in] scount the number of sectors to write
 * \param[in] buffer a pointer to the data to be written
 *
 * \return the number of sectors written, or -1 on error
 *
 ******************************************************************************/
int sxm_cfile_write(FILE *self, uint offset, uint scount, const void *buffer) {
    int rc;

    if (self && buffer) {
        rc = cfile_write(self, offset, scount, buffer);
        if ((uint)rc == scount) {
            if (fflush(self) != 0) {
                rc = -1;
            }
        }
    }
    else {
        rc = -1;
    }
    return rc;
}

/***************************************************************************//**
 * The routine writes the whole of file, updating each of the defined
 * sections, and updating and writing the root block
 *
 * \param[in] self a pointer to an open cycle file
 * \param[in] root a pointer to the root block for that file
 * \param[out] pStat I/O statistics for this call; can be \p NULL
 *
 * \retval SXM_E_OK Whole file has been secured successfully
 * \retval SXM_E_FAULT NULL parameter(s)
 * \retval SXM_E_PIPE I/O Error detected
 *
 ******************************************************************************/
int sxm_cfile_secure_all(FILE *self, SXMCFileRootBlock *root, SXMIOStat *pStat) {
    int rc;

    if ((self != NULL) && (root != NULL)) {
        size_t wBlocks = 0U;
        CF_LOG("[self=%p, %s|root=%p, pStat=%p]", self, root->format, root, pStat);

        /* If the root has never been written the timestamp is 0,
         * thus, to reserve space in the file the root sector shall
         * be written here as-is
         */
        if (root->ts != 0U) {
            CF_LOG("[%s|root=%p] writing the root very first time", root->format, root);
            rc = cfile_write(self, 0, SXM_CFILE_ROOT_SIZE_IN_SECTORS, root);
            if (rc == SXM_CFILE_ROOT_SIZE_IN_SECTORS) {
                wBlocks = (size_t)SXM_CFILE_ROOT_SIZE_IN_SECTORS;
                rc = SXM_E_OK;
            }
            else {
                rc = SXM_E_PIPE;
            }
        }
        else {
            rc = SXM_E_OK;
        }

        /* Write all section one by one */
        if (rc == SXM_E_OK) {
            SXMCFileSection *sec = &root->sections[0];
            const SXMCFileSection * const esec = &root->sections[ARRAY_SIZE(root->sections)];
            for (; sec < esec; ++sec) {
                if (sec->length) {
                    const int intRc = cfile_section_write(self, root, sec, &wBlocks);
                    if (intRc <= 0) {
                        CF_LOG("[%s|root=%p]: failed to write sec #%d, rc=%d",
                            root->format, root, (int)(sec - &root->sections[0]), intRc);
                        rc = SXM_E_PIPE;
                        break;
                    }
                }
                else {
                    CF_LOG("[%s|root=%p] skip unused sec #%d",
                        root->format, root, (int)(sec - &root->sections[0]));
                }
            }
        }

        if (rc == SXM_E_OK) {
            /* Calculate CRC over the root block */
            root->ts = sxm_cfile_time_now();
            root->crc = sxm_crc32_calculate(root, offsetof(SXMCFileRootBlock, crc));

            CF_LOG("[%s|root=%p] updating and writing root at %u",
                root->format, root, root->ts);

            /* Write root to the file */
            rc = sxm_cfile_write(self, 0U, SXM_CFILE_ROOT_SIZE_IN_SECTORS, root);
            if (rc == SXM_CFILE_ROOT_SIZE_IN_SECTORS) {
                wBlocks += (size_t)SXM_CFILE_ROOT_SIZE_IN_SECTORS;
                rc = SXM_E_OK;
            }
            else {
                rc = SXM_E_PIPE;
            }
        }
        /* populate the result */
        if (pStat != NULL) {
            pStat->sectorsWritten += ((rc == SXM_E_OK) ? wBlocks : 0U);
        }
    }
    else {
        rc = SXM_E_FAULT;
    }
    return rc;
}

/***************************************************************************//**
 * The routine secures all the file, then closes the cycle file
 *
 * \param[in] self a pointer to an open cycle file
 * \param[in] root a pointer to the root block for that file
 *
 * \retval SXM_E_OK Whole file has been secured and closed successfully
 * \retval SXM_E_FAULT NULL parameter(s)
 * \retval SXM_E_PIPE I/O Error detected
 *
 ******************************************************************************/
int sxm_cfile_shutdown(FILE *self, SXMCFileRootBlock *root) {
    int rc;
    CF_LOG("[self=%p, %s|root=%p]",
        self, ((root != NULL) ? root->format : "(null)"), root);
    if (self != NULL) {
        rc = (root != NULL) ? sxm_cfile_secure_all(self, root, NULL) : SXM_E_OK;
        sxm_file_close(self);
    }
    else {
        rc = SXM_E_FAULT;
    }
    return rc;
}

/***************************************************************************//**
 * The routine sets a single value into the c-file's user area
 *
 * \param[in] root a pointer to the root block for that file
 * \param[in] offset the offset of the user's area \c uint item
 * \param[in] value new value to be set in case of success
 *
 * \retval SXM_E_OK Value is set
 * \retval SXM_E_FAULT NULL parameter
 * \retval SXM_E_INVAL Invalid offset
 *
 ******************************************************************************/
int sxm_cfile_uset(SXMCFileRootBlock *root, uint offset, uint value) {
    SXMResultCode rc;
    if (root == NULL) {
        rc = SXM_E_FAULT;
    }
    else if (offset >= ARRAY_SIZE(root->u)) {
        rc = SXM_E_INVAL;
    }
    else {
        root->u[offset] = value;
        rc = SXM_E_OK;
    }
    return rc;
}

/***************************************************************************//**
 * The routine gets a single value from the c-file's user area
 *
 * \param[in] root a pointer to the root block for that file
 * \param[in] offset the offset of the user's area \c uint item
 * \param[out] value existing value within the user area by \c offset
 *
 * \retval SXM_E_OK Value is set
 * \retval SXM_E_FAULT NULL parameter
 * \retval SXM_E_INVAL Invalid offset
 *
 ******************************************************************************/
int sxm_cfile_uget(const SXMCFileRootBlock *root, uint offset, uint *value) {
    SXMResultCode rc;
    if ((root == NULL) || (value == NULL)) {
        rc = SXM_E_FAULT;
    }
    else if (offset >= ARRAY_SIZE(root->u)) {
        rc = SXM_E_INVAL;
    }
    else {
        *value = root->u[offset];
        rc = SXM_E_OK;
    }
    return rc;
}

/** @} */
