/******************************************************************************/
/*                Copyright (c) Sirius XM Satellite Radio, Inc.               */
/*                            All Rights Reserved                             */
/*      Licensed Materials - Property of Sirius XM Satellite Radio, Inc.      */
/******************************************************************************/
/***************************************************************************//**
*
* \file sxm_bytebuff.h
* \author Leslie French
* \date 8/20/2013
*
* Header file for ByteStream Parsing.
* Useful for walking up and down SXi structures, extracting and injecting
* integers and strings.
*
*******************************************************************************/

#ifndef SXM_BYTEBUFF_H
#define SXM_BYTEBUFF_H

#include <sxm_build.h>
#include <sxm_typ.h>

#ifdef __cplusplus
extern "C"
{
#endif

/** The byte buffer states */
typedef enum
{
    SXM_BYTEBUFF_STATE_OK = 0, //!< Good and ready for further operations
    SXM_BYTEBUFF_STATE_OFLOW,  //!< Overflow, attempt to write past end of buffer
    SXM_BYTEBUFF_STATE_UFLOW,  //!< Underflow, attempt to read past end of buffer
    SXM_BYTEBUFF_STATE_UNKNOWN //!< Unknown state
} SXMByteBuffState;

/** A simple structure to track the current position with the data */
typedef struct
{
    byte *b;    //!< a pointer to the current position in the buffer
    byte *be;   //!< a pointer to the end of the buffer
    byte *bs;   //!< a pointer to the start of the buffer
    SXMByteBuffState state;    //!< current buffer state
} SXMByteBuff;

extern SXESDK_INTERNAL_API SXMByteBuff *sxm_bytebuff_init(SXMByteBuff *,// pBuff
                                                          void *,       // buff
                                                          int,          // size
                                                          int);         // skip

extern SXESDK_INTERNAL_API void sxm_bytebuff_seek(SXMByteBuff *,        // pBuff
                                                  uint);                // offset

extern SXESDK_INTERNAL_API byte sxm_bytebuff_b1(SXMByteBuff *);         // pBuff

extern SXESDK_INTERNAL_API void sxm_bytebuff_p1(SXMByteBuff *,          // pBuff
                                                           byte );      // value

extern SXESDK_INTERNAL_API ushort sxm_bytebuff_b2(SXMByteBuff *);       // pBuff

extern SXESDK_INTERNAL_API ushort sxm_bytebuff_g2(SXMByteBuff *);       // pBuff

extern SXESDK_INTERNAL_API void sxm_bytebuff_p2(SXMByteBuff *,          // pBuff
                                                          ushort);      // value

extern SXESDK_INTERNAL_API uint sxm_bytebuff_b4(SXMByteBuff *);         // pBuff

extern SXESDK_INTERNAL_API uint sxm_bytebuff_g4(SXMByteBuff *);         // pBuff

extern SXESDK_INTERNAL_API void sxm_bytebuff_p4(SXMByteBuff *,          // pBuff
                                                          uint);        // value

extern SXESDK_INTERNAL_API void sxm_bytebuff_string(SXMByteBuff *,      // PBuff
                                                    char *,             // pTo
                                                    uint);              // count

extern SXESDK_INTERNAL_API void sxm_bytebuff_strline(SXMByteBuff *,     // PBuff
                                                     char *,            // pTo
                                                     uint);             // count

extern SXESDK_INTERNAL_API void sxm_bytebuff_pstring(SXMByteBuff *,     // pBuf
                                                    const char *);      // pFrom

extern SXESDK_INTERNAL_API uint sxm_bytebuff_blob(SXMByteBuff *,        // pBuf
                                                  void *,               // pOut
                                                  uint);                // s

extern SXESDK_INTERNAL_API void sxm_bytebuff_pblob(SXMByteBuff *,       // pBuf
                                                   const void *,        // pIn
                                                   uint);               // s

extern SXESDK_INTERNAL_API void sxm_bytebuff_alignup(SXMByteBuff *,     // pBuf
                                                     uint);             // bound

extern SXESDK_INTERNAL_API void sxm_bytebuff_skip(SXMByteBuff *,        // pBuf
                                                  uint);                // skip

extern SXESDK_INTERNAL_API void sxm_bytebuff_string_skip(SXMByteBuff *);// pBuf

extern SXESDK_INTERNAL_API uint sxm_bytebuff_size(const SXMByteBuff*);  // pBuf

extern SXESDK_INTERNAL_API void* sxm_bytebuff_align(const SXMByteBuff*);// pBuf

extern SXESDK_INTERNAL_API uint sxm_bytebuff_tell(const SXMByteBuff*);  // pBuf

extern SXESDK_INTERNAL_API SXMByteBuffState sxm_bytebuff_state(const SXMByteBuff*); // pBuf

#ifdef __cplusplus
}
#endif

#endif /* #ifndef SXM_BYTEBUFF_H */
