/******************************************************************************/
/*                Copyright (c) Sirius XM Satellite Radio, Inc.               */
/*                            All Rights Reserved                             */
/*      Licensed Materials - Property of Sirius XM Satellite Radio, Inc.      */
/******************************************************************************/
/***************************************************************************//**
*
* \file sxm_bytebuff.c
* \author Leslie French
* \date 8/20/2013
*
* \brief Extracting byte-aligned data, for example in a SXi packet
*
* The bytebuff (byte buffer) routines are used to parse data structures that are
* byte-aligned, with big-endian integer values embedded in the data stream. 
* The various SXi structures follow this model. The routines use a simple
* structure to track the current position with the data is \ref SXMByteBuff. 
*
* The bytebuff (byte buffer) routines are also used to safely copy data to data
* buffers that are bounded by guarding against buffer overrun.
*
* The various SXi structures and buffers follow this model. The routines use a
* simple structure to track the current position with the data is \ref SXMByteBuff
* and set buffer overflow and underflow condition when a copy-in or copy-out
* function fails, respectively.
*
*******************************************************************************/

#include <stdio.h>
#include <string.h>
#include <util/sxm_bytebuff.h>

/** Helper macro definition to peek 2-bytes value in big-endian
 * \param[in] _b byte buffer
 */
#define BIGENDIAN2(_b) \
    ((_b[0] << 8 ) | _b[1])

/** Helper macro definition to peek 4-bytes value in big-endian
 * \param[in] _b byte buffer
 */
#define BIGENDIAN4(_b) \
    ((_b[0] << 24 ) | (_b[1] << 16) | (_b[2] << 8) | _b[3])

/***************************************************************************//**
 * The string routine reads a string terminated by termChar, incrementing the byte
 * buffer passed the NUL. The routine will not read past the end of the buffer.
 *
 * \param[in] pBuf a pointer to an initialized SXMByteBuff structure
 * \param[in] termChar string terminating character
 * \param[out] pTo a character array large enough to receive the string
 * \param[in] len length of destination string, including NUL character
 ******************************************************************************/
static void sxm_bytebuff_read_string(SXMByteBuff *pBuf, char termChar, char *pTo, uint len)
{
    if ((NULL != pBuf) && (pBuf->state == SXM_BYTEBUFF_STATE_OK))
    {
        if ((NULL != pTo) && (0 != len))
        {
            while ((pBuf->b < pBuf->be) && (termChar != *pBuf->b) && (0 != len--))
            {
                *pTo++ = (char)*pBuf->b++;
            }

            /* If still has room put the nil-terminator */
            if (0 != len)
            {
                *pTo = '\0';
            }
        }

        /* Need to take care about the string tail after output buffer has ended */
        while ((pBuf->b < pBuf->be) && (termChar != *pBuf->b++));
    }

    return;
}

/***************************************************************************//**
 * The init routine initializes an empty structure. The structure is not
 * allocated within the routine, an empty structure is passed in to the routine
 * which fills in the values and returns pointer to the original structure.
 *
 * \note Since the application is responsible for reserving the space for
 *       the ByteBuff, there is no delete function.
 * \note Buffer's state is set depending on \p size and \p skip in case of success
 *       initialization: SXM_BYTEBUFF_STATE_OK or SXM_BYTEBUFF_STATE_OFLOW
 *
 * \param[in] pBuf a pointer to an empty SXMByteBuff structure
 * \param[in] pBuffer the byte buffer to be processed
 * \param[in] size the length of the buffer, in bytes
 * \param[in] skip the number of initial byte of the buffer to skip
 *
 * \return the \p pBuf value passed in to the routine in case of success or
 *         \p NULL in case of error.
 ******************************************************************************/
SXMByteBuff *sxm_bytebuff_init(SXMByteBuff *pBuf, void *pBuffer, int size, int skip)
{
    if (pBuf && pBuffer && (size > 0) && (skip >= 0))
    {
        pBuf->bs = (byte*)pBuffer;
        if (skip > size)
        {
            pBuf->b = pBuf->bs + size;
            pBuf->state = SXM_BYTEBUFF_STATE_OFLOW;
        }
        else
        {
            pBuf->b = pBuf->bs + skip;
            pBuf->state = SXM_BYTEBUFF_STATE_OK;
        }
        pBuf->be = pBuf->bs + size;
        return pBuf;
    }

    return NULL;
}

/***************************************************************************//**
 * The seek routine sets current position by the offset
 * from the beginning of the buffer.
 *
 * \note Buffer's state is set depending on \p offset. in case of success
 *       If offset >= buffer size, the state is set to SXM_BYTEBUFF_STATE_OFLOW.
 *
 * \param[in] pBuf a pointer to a SXMByteBuff structure
 * \param[in] offset the offset from the beginning of the buffer
 *
 ******************************************************************************/
void sxm_bytebuff_seek(SXMByteBuff *pBuf, uint offset)
{
    if (NULL != pBuf)
    {
        pBuf->b = pBuf->bs + offset;

        if (pBuf->b > pBuf->be)
        {
            pBuf->b = pBuf->be;
            pBuf->state = SXM_BYTEBUFF_STATE_OFLOW;
        }
    }

    return;
}

/***************************************************************************//**
 * Copy data from the buffer.
 *
 * The function copies data up to the out buffer size or byte buffer remaining
 * data.
 * \note set buffer underflow state if at end of buffer
 *
 * \param[in] pBuf a pointer to an initialized SXMByteBuff structure
 * \param[in,out] pOut buffer for the read data
 * \param[in] len length of the \c pOut in bytes
 * \return number of copied bytes
 ******************************************************************************/
uint sxm_bytebuff_blob(SXMByteBuff *pBuf, void *pOut, uint len)
{
    // the number of bytes copied
    uint copied = 0;

    // if NULL pointers, or zero size, do nothing
    if (pBuf && pOut && len && (pBuf->state == SXM_BYTEBUFF_STATE_OK))
    {
        // if requested size exceeds remaining buffer size
        if (len > sxm_bytebuff_size(pBuf))
        {
            copied = sxm_bytebuff_size(pBuf);
            pBuf->state = SXM_BYTEBUFF_STATE_UFLOW;
        }
        else 
        {
            copied = len;
        }

        // the minimum of the requested size of remaining buffer size
        memcpy(pOut, pBuf->b, copied);
        pBuf->b += copied;
    }

    return copied;
}

/***************************************************************************//**
 * Put data to the buffer.
 *
 * The function puts data from the in buffer, incrementing the byte pointer
 * \note sets the buffer error state if no room
 *
 * \param[in] pBuf a pointer to an initialized SXMByteBuff structure
 * \param[in] pIn buffer for the data
 * \param[in] len size of the \c len in bytes
 ******************************************************************************/
void sxm_bytebuff_pblob(SXMByteBuff *pBuf, const void *pIn, uint len)
{
    // don't do anything if NULL pointers or buffer state not OK
    if (pBuf && pIn && (pBuf->state == SXM_BYTEBUFF_STATE_OK))
    {
        // now make sure it fits at least
        if ((pBuf->b + len) <= pBuf->be) 
        {
            memcpy(pBuf->b, pIn, len); 
            pBuf->b += len;
        }
        else
        {
            /* If ran out of room */
            pBuf->state = SXM_BYTEBUFF_STATE_OFLOW;
        }
    }
}

/***************************************************************************//**
 * The string routine reads a NUL-terminated string, incrementing the byte
 * buffer passed the NUL.  The routine will not read past the end of the buffer.
 *
 * \param[in] pBuf a pointer to an initialized SXMByteBuff structure
 * \param[in,out] pTo a character array large enough to receive the string.
 * \param[in] len length of destination string, including NUL character
 ******************************************************************************/
void sxm_bytebuff_string(SXMByteBuff *pBuf, char *pTo, uint len)
{
    sxm_bytebuff_read_string(pBuf, '\0', pTo, len);
    return;
}

/***************************************************************************//**
 * The string routine reads a newline-terminated string, incrementing the byte
 * buffer passed the NUL. The routine will not read past the end of the buffer.
 *
 * \param[in] pBuf a pointer to an initialized SXMByteBuff structure
 * \param[in,out] pTo a character array large enough to receive the string.
 * \param[in] len length of destination string, including NUL character
 ******************************************************************************/
void sxm_bytebuff_strline(SXMByteBuff *pBuf, char *pTo, uint len)
{
    sxm_bytebuff_read_string(pBuf, '\n', pTo, len);

    return;
}

/***************************************************************************//**
 * The string routine puts a NUL-terminated string, incrementing the byte
 * buffer passed the NUL. The routine will not write past the end of the buffer.
 * \note sets the buffer error state if no room
 *
 * \param[in] pBuf a pointer to an initialized SXMByteBuff structure
 * \param[in] pFrom a character array large enough to receive the string.
 ******************************************************************************/
void sxm_bytebuff_pstring(SXMByteBuff *pBuf, const char *pFrom)
{
    // don't do anything if NULL pointers, zero length, or bad buffer state
    if (pBuf && pFrom && (pBuf->state == SXM_BYTEBUFF_STATE_OK))
    {
        const char *pf = pFrom;
        // Copy data in symbol-by-symbol manner until end of the buffer
        // or end of the line
        while ((pBuf->b < pBuf->be) && (*pf != '\0'))
        {
            *pBuf->b++ = (byte)*pf++;
        }
        // Relying on the p1 function let's put string terminator
        // and if the buffer is out of space the state
        // will be set accordingly inside.
        sxm_bytebuff_p1(pBuf, (byte)'\0');
    }
}

/***************************************************************************//**
 * The routine returns a single byte value, incrementing the byte pointer
 * \note set buffer underflow state if at end of buffer
 *
 * \param[in] pBuf a pointer to an initialized SXMByteBuff structure
 *
 * \return byte value read from the buffer
 ******************************************************************************/
byte sxm_bytebuff_b1(SXMByteBuff *pBuf)
{
    byte data = 0;

    // don't do anything or bad buffer state
    if (pBuf && (pBuf->state == SXM_BYTEBUFF_STATE_OK))
    {
        // copy out the data if it is there
        if (pBuf->b < pBuf->be)
        {
            data = *pBuf->b++;
        }
        else
        {
            pBuf->state = SXM_BYTEBUFF_STATE_UFLOW;
        }
    }

    return data;
}

/***************************************************************************//**
 * The routine puts a byte value, incrementing the byte pointer
 * \note sets the buffer error state if no room
 *
 * \param[in] pBuf a pointer to an initialized SXMByteBuff structure
 * \param[in] value the data to be written
 ******************************************************************************/
void sxm_bytebuff_p1(SXMByteBuff *pBuf, byte value)
{
    sxm_bytebuff_pblob(pBuf, &value, sizeof(value));
}

/***************************************************************************//**
 * The routine returns a 16-bit unsigned short value stored in big-endian,
 * incrementing the byte pointer
 * \note set buffer underflow state if at end of buffer
 *
 * \param[in] pBuf a pointer to an initialized SXMByteBuff structure
 * \return 2-bytes unsigned value read from the buffer
 ******************************************************************************/
ushort sxm_bytebuff_b2(SXMByteBuff *pBuf)
{
    ushort data = 0;

    // don't do anything or bad buffer state
    if (pBuf && (pBuf->state == SXM_BYTEBUFF_STATE_OK))
    {
        // copy out the data if it is there
        if ((pBuf->b + sizeof(data)) <= pBuf->be)
        {
            data = (ushort)BIGENDIAN2(pBuf->b);
            pBuf->b += sizeof(data);
        }
        else
        {
            pBuf->state = SXM_BYTEBUFF_STATE_UFLOW;
        }
    }

    return data;
}

/***************************************************************************//**
 * The routine returns a 16-bit unsigned short value,
 * incrementing the byte pointer
 * \note set buffer underflow state if at end of buffer
 *
 * \param[in] pBuf a pointer to an initialized SXMByteBuff structure
 * \return 2-bytes unsigned value read from the buffer
 ******************************************************************************/
ushort sxm_bytebuff_g2(SXMByteBuff *pBuf)
{
    ushort data = 0;
    sxm_bytebuff_blob(pBuf, &data, sizeof(data));
    return data;
}

/***************************************************************************//**
 * This routine puts a 16-bit unsigned short value,
 * incrementing the byte pointer
 * \note sets the buffer error state if no room
 *
 * \param[in] pBuf a pointer to an initialized SXMByteBuff structure
 * \param[in] value the data to be written
 ******************************************************************************/
void sxm_bytebuff_p2(SXMByteBuff *pBuf, ushort value)
{
    sxm_bytebuff_pblob(pBuf, &value, sizeof(value));
}

/***************************************************************************//**
 * The routine returns a 32-bit unsigned integer value stored in big-endian,
 * incrementing the byte pointer
 * \note set buffer underflow state if at end of buffer
 *
 * \param[in] pBuf a pointer to an initialized SXMByteBuff structure
 * \return 4-bytes unsigned value read from the buffer
 ******************************************************************************/
uint sxm_bytebuff_b4(SXMByteBuff *pBuf)
{
    uint data = 0;

    // don't do anything or bad buffer state
    if (pBuf && (pBuf->state == SXM_BYTEBUFF_STATE_OK))
    {        
        // copy out the data if it is there
        if ((pBuf->b + sizeof(data)) <= pBuf->be)
        {
            data = (uint)BIGENDIAN4(pBuf->b);
            pBuf->b += sizeof(data);
        }
        else
        {
            pBuf->state = SXM_BYTEBUFF_STATE_UFLOW;
        }
    }

    return data;
}

/***************************************************************************//**
 * The routine returns a 32-bit unsigned integer value,
 * incrementing the byte pointer
 * \note set buffer underflow state if at end of buffer
 *
 * \param[in] pBuf a pointer to an initialized SXMByteBuff structure
 * \return 4-bytes unsigned value read from the buffer
 ******************************************************************************/
uint sxm_bytebuff_g4(SXMByteBuff *pBuf)
{
    uint data = 0;
    sxm_bytebuff_blob(pBuf, &data, sizeof(data));
    return data;
}

/***************************************************************************//**
 * This routine puts a 32-bit unsigned integer value,
 * incrementing the byte pointer
 * \note sets the buffer error state if no room
 *
 * \param[in] pBuf a pointer to an initialized SXMByteBuff structure
 * \param[in] value the data to be written
 ******************************************************************************/
void sxm_bytebuff_p4(SXMByteBuff *pBuf, uint value)
{
    sxm_bytebuff_pblob(pBuf, &value, sizeof(value));
}

/***************************************************************************//**
 * The routine aligns the current buffer address forward to an 'N' byte boundary.
 * \note sets the buffer error state if no room
 *
 * \param[in] pBuf a pointer to an initialized SXMByteBuff structure
 * \param[in] bound bounding value 'N'
 ******************************************************************************/
void sxm_bytebuff_alignup(SXMByteBuff *pBuf, uint bound)
{
    /* don't do anything if NULL pointers, zero bound, or buffer state not OK */
    if (pBuf && bound && (pBuf->state == SXM_BYTEBUFF_STATE_OK))
    {
        const uint offset = bound - (sxm_bytebuff_tell(pBuf) % bound);
        if (offset != bound)
        {
            /* now make sure it fits at least */
            if ((pBuf->b + offset) <= pBuf->be)
            {
                pBuf->b += offset;
            }
            else 
            {
                /* If ran out of room */
                pBuf->state = SXM_BYTEBUFF_STATE_OFLOW;
            }
        }
    }
}

/***************************************************************************//**
 * The routine advances the current buffer address 'N' bytes.
 * \note set buffer underflow state if at end of buffer
 *
 * \param[in] pBuf a pointer to an initialized SXMByteBuff structure
 * \param[in] skip number of bytes to advance
 ******************************************************************************/
void sxm_bytebuff_skip(SXMByteBuff *pBuf, uint skip)
{
    // don't do anything if NULL pointers, zero skip, or buffer state not OK
    if (pBuf && skip && (pBuf->state == SXM_BYTEBUFF_STATE_OK))
    {
        // now make sure skip size is valid
        if ((pBuf->b + skip) <= pBuf->be)
        {
            pBuf->b += skip;
        }
        // buffer can't be advanced requested amount
        else
        {
            /* If ran out of room */
            pBuf->state = SXM_BYTEBUFF_STATE_UFLOW;
        }
    }
}

/***************************************************************************//**
 * The routine advances the current buffer address nul-terminated string
 * including terminator. 
 * \note set buffer underflow state if at end of buffer
 *
 * \param[in] pBuf a pointer to an initialized SXMByteBuff structure
 *
 ******************************************************************************/
void sxm_bytebuff_string_skip(SXMByteBuff *pBuf)
{
    /* don't do anything if NULL pointers, zero count or bad state */
    if (pBuf && (pBuf->state == SXM_BYTEBUFF_STATE_OK))
    {
        char lastCheckedSymbol = '\0';
        /* Need to take care about the string tail after output buffer has ended */
        while ((pBuf->b < pBuf->be) && 
               ((lastCheckedSymbol = (char)*pBuf->b++) != '\0'))
        { }

        /* Have the loop left by eol condition? */
        if (lastCheckedSymbol != '\0')
        {
            /* If ran out of room */
            pBuf->state = SXM_BYTEBUFF_STATE_UFLOW;
        }
    }
}

/***************************************************************************//**
 * Return current size of the buffer in bytes
 * \param[in] pBuf a pointer to an initialized SXMByteBuff structure
 ******************************************************************************/
uint sxm_bytebuff_size(const SXMByteBuff *pBuf)
{
    return pBuf ? (uint)(pBuf->be - pBuf->b) : 0U;
}

/***************************************************************************//**
 * Returns the buffer start pointer
 * \param[in] pBuf a pointer to an initialized SXMByteBuff structure
 ******************************************************************************/
void* sxm_bytebuff_align(const SXMByteBuff *pBuf)
{
    return pBuf ? (void*)pBuf->b : NULL;
}

/***************************************************************************//**
 * Returns the current buffer offset aka position
 * \param[in] pBuf a pointer to an initialized SXMByteBuff structure
 ******************************************************************************/
uint sxm_bytebuff_tell(const SXMByteBuff *pBuf)
{
    return pBuf ? (uint)(pBuf->b - pBuf->bs) : 0U;
}

/***************************************************************************//**
 * Return current state of the buffer
 * \param[in] pBuf a pointer to an initialized SXMByteBuff structure
 * \return values from \ref SXMByteBuffState
 ******************************************************************************/
SXMByteBuffState sxm_bytebuff_state(const SXMByteBuff *pBuf)
{
    return pBuf ? pBuf->state : SXM_BYTEBUFF_STATE_UNKNOWN;
}
