/******************************************************************************/
/*                Copyright (c) Sirius XM Satellite Radio, Inc.               */
/*                            All Rights Reserved                             */
/*      Licensed Materials - Property of Sirius XM Satellite Radio, Inc.      */
/******************************************************************************/
/***************************************************************************//**
*
* \file sxm_bitbuff.c
* \author Leslie French
* \date 8/20/2013
*
* \brief BitField extractions
*
* The BitBuff (bitstream buffer) is the utility interface for parsing protocol
* units. It is also used internally by the deflate utility.
*
*******************************************************************************/

/** Debug macro redefinition */
#define DEBUG_TAG "bitbuff"

#include <util/sxm_bitbuff.h>

#ifndef SDKFILES_STANDALONE_BUILD
#include <sxm_stdtrace.h>
#include <util/sxm_common_internal.h>
#include <util/sxm_noname_internal.h>

#else /* #ifndef SDKFILES_STANDALONE_BUILD */
#include "sdkfiles.h"

#undef PLOG_UTL
#define PLOG_UTL non_fatal
#endif /* #ifndef SDKFILES_STANDALONE_BUILD ... #else ... */

#define BITBUFF_FILE_BLOCK_READ_SIZE (65536U)

/**
 * \brief The full BAUDOT table
 *
 * |   n   | n + 1 | n + 2 | n + 3 | n + 4 | n + 5 |
 * |:-----:|:-----:|:-----:|:-----:|:-----:|:-----:|
 * |   UC  |   LC  |  FIG  |  EUC  |  ELC  |  EFG  |
 *
 * \note Remapped to 8859-15
 */
char sxm_baudot_table[] = {
    '\0', '\0',  '\0', '\xc0', '\xe0',    '"',
     'E',  'e',   '3', '\xc1', '\xe1',    ';',
    '\0', '\0',   '+', '\xc2', '\xe2',    '<',
     'A',  'a',   '-', '\xc3', '\xe3',    '=',
     ' ',  ' ',   ' ', '\xc4', '\xe4',    '>',
     'S',  's',  '\'', '\xc5', '\xe5',    '[',
     'I',  'i',   '8', '\xc6', '\xe6',   '\\',
     'U',  'u',   '7', '\xc7', '\xe7',    ']',
    '\n', '\n','\x0d', '\xc8', '\xe8',    '^',
     'D',  'd',   '$', '\xc9', '\xe9',    '_',
     'R',  'r',   '4', '\xca', '\xea',    '`',
     'J',  'j',   '*', '\xcb', '\xeb',    '{',
     'N',  'n',   ',', '\xcc', '\xec',    '|',
     'F',  'f',   '!', '\xcd', '\xed',    '}',
     'C',  'c',   ':', '\xce', '\xee',    '~',
     'K',  'k',   '(', '\xcf', '\xef', '\xab',
     'T',  't',   '5', '\xbc', '\xbd', '\xbb',
     'Z',  'z',   '%', '\xd1', '\xf1', '\xa2',
     'L',  'l',   ')', '\xd2', '\xf2',   '\0',
     'W',  'w',   '2', '\xd3', '\xf3',   '\0',
     'H',  'h',   '#', '\xd4', '\xf4',   '\0',
     'Y',  'y',   '6', '\xd5', '\xf5',   '\0',
     'P',  'p',   '0', '\xd6', '\xf6',   '\0',
     'Q',  'q',   '1', '\xa1', '\xbf',   '\0',
     'O',  'o',   '9', '\xd8', '\xf8',   '\0',
     'B',  'b',   '?', '\xd9', '\xf9',   '\0',
     'G',  'g',   '&', '\xda', '\xfa',   '\0',
    '\0', '\0',  '\0', '\xdb', '\xfb',   '\0',
     'M',  'm',   '.', '\xdc', '\xfc',   '\0',
     'X',  'x',   '/', '\xdd', '\xfd',   '\0',
     'V',  'v',   '@', '\xa9', '\xae',   '\0',
    '\0', '\0',  '\0', '\xdf', '\xff',   '\0'
};

/** Predefined tables for fixed compressed blocks */
static const uint flitdist[] = { 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
    8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
    8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
    8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
    8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 9,
    9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
    9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
    9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
    9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
    7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 8, 8, 8, 8, 8, 8, 8, 8, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5,
    5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5 };

static const uint flitvalues[] = { 12, 140, 76, 204, 44, 172, 108, 236, 28, 156, 92, 220, 60, 188,
    124, 252, 2, 130, 66, 194, 34, 162, 98, 226, 18, 146, 82, 210, 50, 178, 114, 242, 10, 138, 74,
    202, 42, 170, 106, 234, 26, 154, 90, 218, 58, 186, 122, 250, 6, 134, 70, 198, 38, 166, 102,
    230, 22, 150, 86, 214, 54, 182, 118, 246, 14, 142, 78, 206, 46, 174, 110, 238, 30, 158, 94,
    222, 62, 190, 126, 254, 1, 129, 65, 193, 33, 161, 97, 225, 17, 145, 81, 209, 49, 177, 113,
    241, 9, 137, 73, 201, 41, 169, 105, 233, 25, 153, 89, 217, 57, 185, 121, 249, 5, 133, 69, 197,
    37, 165, 101, 229, 21, 149, 85, 213, 53, 181, 117, 245, 13, 141, 77, 205, 45, 173, 109, 237, 29,
    157, 93, 221, 61, 189, 125, 253, 19, 275, 147, 403, 83, 339, 211, 467, 51, 307, 179, 435, 115,
    371, 243, 499, 11, 267, 139, 395, 75, 331, 203, 459, 43, 299, 171, 427, 107, 363, 235, 491, 27,
    283, 155, 411, 91, 347, 219, 475, 59, 315, 187, 443, 123, 379, 251, 507, 7, 263, 135, 391, 71,
    327, 199, 455, 39, 295, 167, 423, 103, 359, 231, 487, 23, 279, 151, 407, 87, 343, 215, 471, 55,
    311, 183, 439, 119, 375, 247, 503, 15, 271, 143, 399, 79, 335, 207, 463, 47, 303, 175, 431, 111,
    367, 239, 495, 31, 287, 159, 415, 95, 351, 223, 479, 63, 319, 191, 447, 127, 383,255, 511, 0,
    64, 32, 96, 16, 80, 48, 112, 8, 72, 40, 104, 24, 88, 56, 120, 4, 68, 36, 100, 20, 84, 52, 116,
    3, 131, 67, 195, 35, 163, 99, 227, 0, 16, 8, 24, 4,20, 12, 28, 2, 18, 10, 26, 6, 22, 14, 30, 1,
    17, 9, 25, 5, 21, 13, 29, 3, 19, 11, 27, 7, 23, 15, 31, 0, 0, 0, 0 };

/** The default Huffman table generator for deflate */
static const uint hcindex[] = { 16, 17, 18, 0, 8, 7, 9, 6, 10, 5, 11,
    4, 12, 3, 13, 2, 14, 1, 15 };

/** Helper macroses for deflate */
#define EATR(_v, _l)    (0 != sxm_bitbuff_consume_rev(pkt, _v, _l))
#define NEXTR(_n)       sxm_bitbuff_read_rev(pkt, _n)
#define NEXT_BYTE()     sxm_bitbuff_read(pkt, 8)

/************************************************************************
*                                                                      *
*            Local prototypes                                          *
*            ============                                              *
*                                                                      *
************************************************************************/
static int bitbuff_seek_file(SXMBitBuff *b, size_t offset);
static int bitbuff_seek_buf(SXMBitBuff *b, size_t offset);
static size_t bitbuff_read_bytes_buf(SXMBitBuff *pBitBuf, size_t len, byte *out);
static size_t bitbuff_read_bytes_file(SXMBitBuff *pBitBuf, size_t len, byte *out);
static int bitbuff_seek_deflate(SXMBitBuff *b, size_t offset);
static size_t bitbuff_read_bytes_deflate(SXMBitBuff *b, size_t count, byte *out);
static void bitbuff_uninit_dyn_block(SXMBitZipBuff *zipbuff);
static int bitbuff_extract_comp_block(SXMBitZipBuff *zipbuff,
    const SXMDeflateCompBlock *block, uint max);
static int bitbuff_extract_dynamic_block(SXMBitZipBuff *zipbuff, uint max);
static int bitbuff_extract_fixed_block(SXMBitZipBuff *zipbuff, uint max);
static int bitbuff_extract_uncomp_block(SXMBitZipBuff *zipbuff, uint max);
static int bitbuff_read_gzip_header(SXMBitZipBuff *zipbuff);
static void bitbuff_deflate_build_table(const uint *counts, const uint maxc, uint *values);
static int bitbuff_start_dynamic(SXMBitZipBuff *zipbuff);
static uint reverse(uint in, uint len);
static int bitbuff_replenish_deflate(SXMBitBuff *b);

/************************************************************************
 *                                                                      *
 *            Replenishers                                              *
 *            ============                                              *
 *                                                                      *
 ************************************************************************/

/***************************************************************************//**
 * The file-based replenish function is used by the deflate utility to refill the
 * bit buffer from an external file:
 *
 * \param[in] b pointer to an initialized SXMBitBuff structure which hides
 *              SXMBitFileBuff underneath.
 * 
 * \return 0 if the buffer was reset, or 1 (and err set) if not
 *
 ******************************************************************************/
static int sxm_bitbuff_replenish_file(SXMBitBuff *b)
{
    SXMBitFileBuff *pBitFileBuf = (SXMBitFileBuff *)b;
    uint l;

    if (0 != pBitFileBuf->eof)
    {
        b->err = 1;
        return 1;
    }

    l = (uint)fread(pBitFileBuf->fb, 1, sizeof(pBitFileBuf->fb), pBitFileBuf->in);
    if (l < sizeof(pBitFileBuf->fb))
    {
        pBitFileBuf->eof = 1;
    }

    if ((pBitFileBuf->size > 0) && ((pBitFileBuf->pos + l) >= pBitFileBuf->size))
    {
        pBitFileBuf->eof = 1;
        l = pBitFileBuf->size - pBitFileBuf->pos;
        pBitFileBuf->pos += l;
    }

    pBitFileBuf->b.b = &pBitFileBuf->fb[0];
    pBitFileBuf->b.e = &pBitFileBuf->fb[0] + l;

    return 0;
}

/***************************************************************************//**
 * This function closes any hanldes assigned to the buffer and deallocates
 * any buffers
 *
 * \param[in] b pointer to an initialized SXMBitBuff structure
 * 
 * \retval SXM_E_OK Success
 *
 ******************************************************************************/
int sxm_bitbuff_close(SXMBitBuff *b)
{
    if (NULL != b->close)
    {
        return b->close(b);
    }

    return SXM_E_OK;
}

/************************************************************************
 *                                                                      *
 *            BitStream Reading                                         *
 *            =================                                         *
 *                                                                      *
 ************************************************************************/

/***************************************************************************//**
 * The align routine aligns the buffer pointer to the next byte boundary.
 * This is useful to skip over the variable-length PAD values that are used in
 * the bitstream protocols to align parts of the Access Unit to byte boundaries.
 *
 * \param[in] pBitBuf pointer to an initialized SXMBitBuff structure
 * 
 * \return byte pointer within buffer
 *
 ******************************************************************************/
byte *sxm_bitbuff_align(SXMBitBuff *pBitBuf)
{
    pBitBuf->valid = 0;

    return pBitBuf->b;
}

/***************************************************************************//**
 * The routine ensures that there are at least len bits available in the
 * seed value. This is a useful precursor before calling consume to collect
 * sufficient bits for the comparison (reserve reading version)
 *
 * \param[in] pBitBuf pointer to an initialized SXMBitBuff structure
 * \param[in] len the minimum number of bits to have in the \a seed
 *
 * \sa sxm_bitbuff_read_rev 
 * 
 ******************************************************************************/
static void sxm_bitbuff_ensure_r(SXMBitBuff *pBitBuf, uint len)
{
    while (pBitBuf->valid < len)
    {
        if (pBitBuf->b < pBitBuf->e)
        {
            pBitBuf->seed =
                (pBitBuf->seed & ((1U << pBitBuf->valid) - 1U)) |
                (((uint)*pBitBuf->b++) << pBitBuf->valid);
        }
        else if ((pBitBuf->replenish == NULL) ||
                 (pBitBuf->replenish(pBitBuf) != 0))
        {
            pBitBuf->err = 1U;
            pBitBuf->seed = (pBitBuf->seed & ((1U << pBitBuf->valid) - 1U));
        }
        else
        {
            pBitBuf->seed =
                (pBitBuf->seed & ((1U << pBitBuf->valid) - 1U)) |
                (((uint)*pBitBuf->b++) << pBitBuf->valid);
        }

        pBitBuf->valid += SXM_ARCH_BITS_IN_BYTE;
    }

    return;
}

/***************************************************************************//**
 * The routine ensures that there are at least len bits available in the
 * seed value. This is a useful precursor before calling consume to collect
 * sufficient bits for the comparison
 *
 * \param[in] pBitBuf pointer to an initialized SXMBitBuff structure
 * \param[in] len the minimum number of bits to have in the \a seed
 * 
 ******************************************************************************/
static void sxm_bitbuff_ensure(SXMBitBuff *pBitBuf, uint len)
{
    while (pBitBuf->valid < len)
    {
        if (pBitBuf->b < pBitBuf->e)
        {
            pBitBuf->seed =
                (pBitBuf->seed << SXM_ARCH_BITS_IN_BYTE) | *(pBitBuf->b)++;
        }
        else if ((pBitBuf->replenish == NULL) ||
                 (pBitBuf->replenish(pBitBuf) != 0))
        {
            pBitBuf->err = 1;
            pBitBuf->seed = pBitBuf->seed << SXM_ARCH_BITS_IN_BYTE;
        }
        else
        {
            pBitBuf->seed =
                (pBitBuf->seed << SXM_ARCH_BITS_IN_BYTE) | *(pBitBuf->b)++;
        }

        pBitBuf->valid += SXM_ARCH_BITS_IN_BYTE;
    }

    return;
}

/***************************************************************************//**
 * The read routine consumes and returns up to 32 bits from the input buffer.
 *
 * \param[in] pBitBuf pointer to an initialized SXMBitBuff structure
 * \param[in] len the number of bits to consume
 * 
 * \return the value extracted from the bit buffer
 *
 ******************************************************************************/
uint sxm_bitbuff_read(SXMBitBuff *pBitBuf, uint len)
{
    uint ret = 0;

    if (0 != pBitBuf->err)
    {
        return 0;
    }

    if (len > 16)
    {
        ret = sxm_bitbuff_read(pBitBuf, len-16) << 16;
        len = 16;
    }

    sxm_bitbuff_ensure(pBitBuf, len);
    pBitBuf->valid -= len;
    return ret | ((pBitBuf->seed >> pBitBuf->valid) & ((1U << len) - 1U));
}

/***************************************************************************//**
* The read routine consumes and returns bytes from buffer.
*
* \param[in] pBitBuf pointer to an initialized SXMBitBuff structure 
* \param[in] len the number of bytes to consume
* \param[out] out pointer to output buffer
*
* \return length of output data
*
******************************************************************************/
size_t sxm_bitbuff_read_bytes(SXMBitBuff *pBitBuf, size_t len, byte *out)
{
    /* Check for state and for alignment */
    if ((0 == pBitBuf->err) && (0 == pBitBuf->valid))
    {
        return pBitBuf->read_bytes(pBitBuf, len, out);
    }

    return 0;
}

/***************************************************************************//**
 * The skip routine steps over a number of bits. If the end of buffer reached
 * the *err* will be set to non-0 value indicating stream error.
 *
 * \param[in] pBitBuf pointer to an initialized SXMBitBuff structure
 * \param[in] len the number of bits to skip
 *
 ******************************************************************************/
void sxm_bitbuff_skip(SXMBitBuff *pBitBuf, uint len)
{
    if (0 == pBitBuf->err)
    {
        sxm_bitbuff_ensure(pBitBuf, len);
        pBitBuf->valid -= (len >= pBitBuf->valid) ? pBitBuf->valid : len;
    }

    return;
}

/***************************************************************************//**
 * The routine consumes and returns the next len bits from the buffer
 * in reversed (big-endian) order. Only the extraction position is reversed,
 * the value is not bit-swapped.
 *
 * \param[in] pBitBuf pointer to an initialized SXMBitBuff structure
 * \param[in] len number of bits to read
 * 
 * \return the value extracted from the bit buffer
 *
 ******************************************************************************/
uint sxm_bitbuff_read_rev(SXMBitBuff *pBitBuf, uint len)
{
    uint res;

    if (0 != pBitBuf->err)
    {
        return 0;
    }

    if (len > 16) /** \todo ??? */
    {
        uint ret1 = sxm_bitbuff_read_rev(pBitBuf, 16);
        uint ret2 = sxm_bitbuff_read_rev(pBitBuf, len-16) << 16;
        return ret2 | ret1;
    }

    sxm_bitbuff_ensure_r(pBitBuf, len);
    pBitBuf->valid -= len;
    res = pBitBuf->seed & ((1U << len) - 1U);
    pBitBuf->seed >>= len;

    return res;
}

/***************************************************************************//**
 * The consume routine steps over a number of bits if they match the supplied
 * value. This is useful when parsing Huffman-coded data to consume bits only
 * when they match a given symbol.
 *
 * \param[in] pBitBuf pointer to an initialized SXMBitBuff structure
 * \param[in] value the value that must match the current bitstream position
 * \param[in] len the number of bits of \c value that must match 
 * 
 * \retval 1 if a matching value was consumed
 * \retval 0 otherwise
 *
 ******************************************************************************/
int sxm_bitbuff_consume_rev(SXMBitBuff *pBitBuf, uint value, uint len)
{
    const uint mask = (1U << len) - 1U;

    sxm_bitbuff_ensure_r(pBitBuf, len);

    if ((pBitBuf->seed & mask) == value)
    {
        pBitBuf->valid -= len;
        pBitBuf->seed >>= len;
        return 1;
    }

    return 0;
}

/***************************************************************************//**
 * The setup routine initializes a BitBuff for subsequent value extractions.
 *
 * \param[in,out] pBitBuf pointer to a BitBuff to be initialized
 * \param[in] pBuff pointer to the start of the data to be processed
 * \param[in] unBufLen length of the data buffer, in bytes (not bits); 
 *
 ******************************************************************************/
void sxm_bitbuff_setup(SXMBitBuff *pBitBuf, byte *pBuff, uint unBufLen)
{
    pBitBuf->ob = pBitBuf->b = pBuff;
    pBitBuf->e = pBuff + unBufLen;
    pBitBuf->valid = 0;
    pBitBuf->seed = 0;
    pBitBuf->err = 0;
    pBitBuf->seek = bitbuff_seek_buf;
    pBitBuf->read_bytes = bitbuff_read_bytes_buf;
    pBitBuf->close = NULL;
    pBitBuf->replenish = NULL;
}

/***************************************************************************//**
 * The seek function
 *
 * \param[in] b pointer to an initialized SXMBitBuff structure
 * \param[in] offset number of bytes from the beginning to shift
 * 
 * \return 0 if seek operation successfull, or 1 if not
 *
 ******************************************************************************/
int sxm_bitbuff_seek(SXMBitBuff *pBitBuf, uint offset)
{
    pBitBuf->valid = 0;
    pBitBuf->err = (byte)pBitBuf->seek(pBitBuf, offset);

    return (int)pBitBuf->err;
}

/***************************************************************************//**
 * The setup routine initializes a BitBuff for subsequent value extractions
 * where the data is coming from the file.
 *
 * \param[in,out] pBitBuf pointer to a SXMBitFileBuff to be initialized
 * \param[in] pFile opened file with the source data
 * \param[in] offset offset within the file
 * \param[in] size size of the file (optional, can be 0)
 *
 * \retval SXM_E_OK Success
 * \retval SXM_E_PIPE Unable to perform seek operation
 *
 ******************************************************************************/
int sxm_bitfilebuff_setup(SXMBitFileBuff *pBitBuf, FILE *pFile, uint offset,
    uint size)
{
    if (0 != fseek(pFile, (long)offset, SEEK_SET))
    {
        return SXM_E_PIPE;
    }

    pBitBuf->in = pFile;
    pBitBuf->b.valid = 0;
    pBitBuf->b.seed = 0;
    pBitBuf->b.err = 0;
    pBitBuf->b.b = NULL;
    pBitBuf->b.e = NULL;
    pBitBuf->b.replenish = sxm_bitbuff_replenish_file;
    pBitBuf->b.seek = bitbuff_seek_file;
    pBitBuf->b.read_bytes = bitbuff_read_bytes_file;
    pBitBuf->b.close = (int(*)(SXMBitBuff*))sxm_bitfilebuff_close;
    pBitBuf->eof = 0;
    pBitBuf->size = size;
    pBitBuf->pos = 0;

    return SXM_E_OK;
}

/***************************************************************************//**
* Close a bit file (in the filing system) as a SXMBitFileBuff structure
*
* \param[in] b pointer to open SXMBitFileBuff
* 
* \retval SXM_E_OK Success
*
*******************************************************************************/
int sxm_bitfilebuff_close(SXMBitFileBuff *b)
{
    UNUSED_VAR(b);

    return SXM_E_OK;
}

/***************************************************************************//**
* Initializes SXMBitZipBuff structure to provide uncompressed bit stream
* using input pkt bit buffer
*
* \param[in] zipbuff preallocated SXMBitZipBuff structure
* \param[in] pkt bit buffer providing access to compressed data (file or RAM)
*
* \retval SXM_E_OK Success
* \retval SXM_E_PIPE Input bit buffer is in error state
*
*******************************************************************************/
int sxm_bitzipbuff_setup(SXMBitZipBuff *zipbuff, SXMBitBuff *pkt)
{
    if (0 != pkt->err)
    {
        return SXM_E_PIPE;
    }

    memset(zipbuff, 0, sizeof(*zipbuff));

    zipbuff->input = pkt;
    zipbuff->bufhead = zipbuff->ringbuf;
    zipbuff->buftail = zipbuff->ringbuf;
    zipbuff->bufend = zipbuff->ringbuf + sizeof(zipbuff->ringbuf) - 1;
    zipbuff->b.replenish = bitbuff_replenish_deflate;
    zipbuff->b.seek = bitbuff_seek_deflate;
    zipbuff->b.read_bytes = bitbuff_read_bytes_deflate;
    zipbuff->b.close = (int(*)(SXMBitBuff*))sxm_bitzipbuff_close;
    zipbuff->bfixed.nlit = 288;
    zipbuff->bfixed.ndist = 32;
    zipbuff->bfixed.nbit = 5;
    zipbuff->bfixed.litdist = (uint*)flitdist;
    zipbuff->bfixed.litvalues = (uint*)flitvalues;

    return SXM_E_OK;
}

/***************************************************************************//**
* Deinitializes SXMBitZipBuff structure
*
* \param[in] zipbuff preallocated SXMBitZipBuff structure
*
* \retval SXM_E_OK Success
*
*******************************************************************************/
int sxm_bitzipbuff_close(SXMBitZipBuff *zipbuff)
{
    bitbuff_uninit_dyn_block(zipbuff);
    memset(zipbuff, 0, sizeof(*zipbuff));

    return SXM_E_OK;
}

/************************************************************************
 *                                                                      *
 *            Baudot Decoding                                           *
 *            ===============                                           *
 *                                                                      *
 ************************************************************************/

/***************************************************************************//**
 * The baudot routine decodes a Baudot-encoded character string into the
 * supplied string buffer
 *
 * \param[in] pkt pointer to an initialized BitBuff structure
 * \param[in] dstBuf pointer to a character string to hold the decoded data
 * \param[in] mode the initial state of the decoder
 *                 (0=upper-letter; 1=digit, 2=lower-letter)
 * \param[in] numReadMax the maximum number of Baudot symbols to read
 * \param[in] dstBufSize the maximum number of characters to place into \c dstBuf,
 *                       including the NULL-terminator
 * \retval 0 if the string was correctly decoded
 * \retval non-zero otherwise.
 *
 ******************************************************************************/
int sxm_bitbuff_baudot(SXMBitBuff *pkt,
                       char *dstBuf,
                       uint mode,
                       int numReadMax,
                       int dstBufSize)
{
    uint readVal, ix;
    char out;
    uint shift = (mode &= 1U) >> 1U;

    while (((readVal = sxm_bitbuff_read(pkt, 5)) != 0)
           && (numReadMax-- > 0) && (dstBufSize > 1))
    {
        if (0 != pkt->err)
        {
            *dstBuf = '\0';
            return 1;
        }
        else if ((readVal == 2) && (mode == 0))
        {
            shift ^= 1;
        }
        else if (readVal == 0x1b)
        {
            /* escape */
            readVal = sxm_bitbuff_read(pkt, 5);
            if (mode == 1)
            {
                ix = 5;
            }
            else
            {
                ix = shift + 3;
            }

            if ((out = sxm_baudot_table[readVal * 6 + ix]) != '\0')
            {
                *dstBuf++ = out;
                dstBufSize--;
            }
        }
        else if (readVal == 0x1f)
        {
            /* ltr/fig mode */
            mode ^= 1;
        }
        else
        {
            /* Should be a valid lookup code */
            if (mode == 1)
            {
                ix = 2;
            }
            else
            {
                ix = shift;
            }

            if ((out = sxm_baudot_table[readVal * 6 + ix]) != '\0')
            {
                *dstBuf++ = out;
                dstBufSize--;
            }
        }
    }

    *dstBuf = '\0';

    /* If reading error occurred, just return error */
    if (0 != pkt->err)
    {
        return 1;
    }
    /* If numReadMax or dstBufSize exceeded, set error */
    else if (readVal != 0)
    {
        pkt->err = 2;
        return 1;
    }

    return 0;
}

/***************************************************************************//**
 * The baudot routine decodes a Baudot-encoded character string and skips it
 *
 * \param[in] pkt pointer to an initialized BitBuff structure
 * \param[in] mode the initial state of the decoder
 *                 (0=upper-letter; 1=digit, 2=lower-letter)
 *
 ******************************************************************************/
void sxm_bitbuff_skip_baudot(SXMBitBuff *pkt, uint mode)
{
    uint c,
    shift = (mode &= 1U) >> 1U;

    while ((c = sxm_bitbuff_read(pkt, 5)) != 0)
    {
        if (0 != pkt->err)
        {
            return;
        }
        else if ((c == 2) && (mode == 0U))
        {
            shift ^= 1;
        }
        else if (c == 0x1b)
        {
            /* escape */
            c = sxm_bitbuff_read(pkt, 5);
        }
        else if (c == 0x1f)
        {
            /* ltr/fig mode */
            mode ^= 1;
        }
    }

    return;
}

/************************************************************************
 *                                                                      *
 *            ASCII charstring                                          *
 *            ================                                          *
 *                                                                      *
 ************************************************************************/

/***************************************************************************//**
 * The cstring routine reads a C-style character string.
 *
 * \param[in] pkt pointer to an initialized BitBuff structure
 * \param[in] desc pointer to a character buffer to receive the string
 * \param[in] len maximum number of characters to read (including null)
 *
 ******************************************************************************/
void sxm_bitbuff_cstring(SXMBitBuff *pkt, char *desc, int len)
{
    char c = '\0';

    while (len-- > 0)
    {
        c = (char)sxm_bitbuff_read(pkt, SXM_ARCH_BITS_IN_BYTE);
        *desc++ = c;
        if ('\0' == c)
        {
            break;
        }
    }

    /* Check if sxm_bitbuff_read() returned '\0'
       which means the sufficient number of characters is read. */
    if ('\0' != c)
    {
        pkt->err = 2;
    }

    return;
}

/***************************************************************************//**
 * The iso routine reads a ISO 8859-1 coded text string.
 *
 * \param[in] pkt pointer to an initialized BitBuff structure
 * \param[in] desc pointer to a character buffer to receive the string
 * \param[in] len maximum number of characters to read (including null)
 *
 ******************************************************************************/
void sxm_bitbuff_iso(SXMBitBuff *pkt, char *desc, int len)
{
    char c = '\0';

    while (len-- > 0)
    {
        c = (char)sxm_bitbuff_read(pkt, SXM_ARCH_BITS_IN_BYTE);
        *desc++ = c;
        if ('\0' == c)
        {
            break;
        }
    }

    /* Check if sxm_bitbuff_iso() returned '\0'
       which means the sufficient number of characters is read. */
    if ('\0' != c)
    {
        pkt->err = 2;
    }

    return;
}

/***************************************************************************//**
 * The cstring routine skips a C-style character string.
 *
 * \param[in] pkt pointer to an initialized BitBuff structure
 *
 ******************************************************************************/
void sxm_bitbuff_skip_cstring(SXMBitBuff *pkt)
{
    char c;

    do
    {
        c = (char)sxm_bitbuff_read(pkt, 8);
    } while ('\0' != c);

    return;
}

/***************************************************************************//**
 * The iso routine skips a ISO 8859-1 coded text string.
 *
 * \param[in] pkt pointer to an initialized BitBuff structure
 * \param[in] len maximum number of characters to skip (including null)
 *
 ******************************************************************************/
void sxm_bitbuff_skip_iso(SXMBitBuff *pkt, uint len)
{
    char c;

    do
    {
        c = (char)sxm_bitbuff_read(pkt, SXM_ARCH_BITS_IN_BYTE);
    } while (('\0' != c) && (len-- > 0));

    return;
}

/************************************************************************
 *                                                                      *
 *            Debug Version of the parser                               *
 *            ===========================                               *
 *                                                                      *
 ************************************************************************/

/***************************************************************************//**
 * The align routine aligns the buffer pointer to the next byte boundary.
 * This is useful to skip over the variable-length PAD values that are used in
 * the bitstream protocols to align parts of the Access Unit to byte boundaries.
 * (debug version of the \ref sxm_bitbuff_align)
 *
 * \param[in] pBitBuf pointer to an initialized SXMBitBuff structure
 * \param[in] flag 1 enable debug functionality,
 *                 0 - behave like sxm_bitbuff_align
 * 
 * \return byte pointer within buffer
 *
 ******************************************************************************/
byte *sxm_bitbuff_align_debug(SXMBitBuff *pBitBuf, int flag)
{
    if (0 != flag)
    {
        PLOG_UTL("   ... Align");
    }

    return sxm_bitbuff_align(pBitBuf);
}

/***************************************************************************//**
 * The cstring routine reads a C-style character string.
 *
 * \param[in] pkt pointer to an initialized BitBuff structure
 * \param[in] var variable name
 * \param[in] desc pointer to a character buffer to receive the string
 * \param[in] len maximum number of characters to read (including null)
 * \param[in] flag 1 enable debug functionality,
 *                 0 - behave like sxm_bitbuff_cstring)
 *
 * \retval 0 if the string was correctly read
 * \retval non-0 otherwise
 *
 ******************************************************************************/
int sxm_bitbuff_cstring_debug(SXMBitBuff *pkt,
                              const char *var,
                              char *desc,
                              int len,
                              int flag)
{
    if (flag == 0)
    {
        sxm_bitbuff_cstring(pkt, desc, len);
        return pkt->err;
    }

    if (0 != pkt->err)
    {
        PLOG_UTL("   ..! %s [%d]: Error", var, len);
        return 1;
    }

    sxm_bitbuff_cstring(pkt, desc, len);

    if (0 != pkt->err)
    {
        PLOG_UTL("   ..! %s [%d]: Error", var, len);
        return 1;
    }

    PLOG_UTL("   ... %s [%d]: \"%s\"", var, len, desc);

    return 0;
}

/***************************************************************************//**
 * The read routine consumes and returns up to 32 bits from the input buffer.
 *
 * \param[in] pkt pointer to an initialized SXMBitBuff structure
 * \param[in] var variable name
 * \param[in] len the number of bits to consume
 * \param[in] flag 1 enable debug functionality,
 *                 0 - behave like sxm_bitbuff_read)
 * 
 * \return the value extracted from the bit buffer
 *
 ******************************************************************************/
uint sxm_bitbuff_read_debug(SXMBitBuff *pkt, const char *var, uint len, int flag)
{
    char bbuff[40];
    uint ret;
    int i, j = 0;

    if (flag == 0)
    {
        return sxm_bitbuff_read(pkt, len);
    }

    if (0 != pkt->err)
    {
        PLOG_UTL("   ..! %s [%d]: Error", var, len);
        return 0;
    }

    ret = sxm_bitbuff_read(pkt, len);

    for (i = (int)(len - 1); i >= 0; i--)
    {
        bbuff[j++] = ((ret >> i) & 1U) ? '1' : '0';
    }
    bbuff[j] = '\0';

    PLOG_UTL("   ... %s [%d]: '%s' (%d)", var, len, bbuff, ret);

    return ret;
}

/***************************************************************************//**
 * The read routine consumes and returns 1 bit from the input buffer.
 *
 * \param[in] pkt pointer to an initialized SXMBitBuff structure
 * \param[in] flag 1 enable debug functionality,
 *                 0 - behave like sxm_bitbuff_read for 1 bit reading)
 *                                                 
 * \return the value extracted from the bit buffer
 *
 ******************************************************************************/
uint sxm_bitbuff_flag_debug(SXMBitBuff *pkt, int flag)
{
    uint ret;

    if (flag == 0)
    {
        return sxm_bitbuff_read(pkt, 1);
    }

    if (0 != pkt->err)
    {
        PLOG_UTL("   ..! F+: Error");
        return 0;
    }

    ret = sxm_bitbuff_read(pkt, 1);

    PLOG_UTL("   ... F+: '%c'", '0'+ret);

    return ret;
}

/***************************************************************************//**
 * The baudot routine decodes a Baudot-encoded character string into the
 * supplied string buffer (debug version of the \ref sxm_bitbuff_baudot)
 *
 * \param[in] pkt pointer to an initialized BitBuff structure
 * \param[in] var variable name
 * \param[in] dstBuf pointer to a character string to hold the decoded data
 * \param[in] mode the initial state of the decoder
 *                 (0=upper-letter; 1=digit, 2=lower-letter)
 * \param[in] numReadMax the maximum number of Baudot symbols to read
 * \param[in] dstBufSize the maximum number of characters to place into \c dstBuf,
 *                       including the NULL-terminator
 * \param[in] flag 1 enable debug functionality,
 *                 0 - behave like sxm_bitbuff_baudot
 *
 * \return 0 if the string was correctly decoded, non-zero otherwise.
 *
 ******************************************************************************/
int sxm_bitbuff_baudot_debug(SXMBitBuff *pkt,
                             const char *var,
                             char *dstBuf,
                             uint mode,
                             int numReadMax,
                             int dstBufSize,
                             int flag)
{
    int ret;

    if (flag == 0)
    {
        return sxm_bitbuff_baudot(pkt, dstBuf, mode, numReadMax, dstBufSize);
    }

    if (0 != pkt->err)
    {
        PLOG_UTL("   ..! %s [%d/%d]: Error",
                 var, numReadMax, dstBufSize);
        return 1;
    }

    ret = sxm_bitbuff_baudot(pkt, dstBuf, mode, numReadMax, dstBufSize);

    if ((0 != pkt->err) || (ret != 0))
    {
        PLOG_UTL("   ..! %s [%d/%d]: Error",
                 var, numReadMax, dstBufSize);
        return 1;
    }

    PLOG_UTL("   ... %s [%d/%d]: \"%s\"",
             var, numReadMax, dstBufSize, dstBuf);

    return ret;
}

/***************************************************************************//**
 * The skip routine steps over a number of bits
 * (debug version of the \ref sxm_bitbuff_skip)
 *
 * \param[in] pBitBuf pointer to an initialized SXMBitBuff structure
 * \param[in] len the number of bits to skip
 * \param[in] flag 1 - enable debug functionality,
 *                 0 - behave like sxm_bitbuff_skip
 *
 ******************************************************************************/
void sxm_bitbuff_skip_debug(SXMBitBuff *pBitBuf, uint len, int flag)
{
    if (0 != flag)
    {
        PLOG_UTL("   ... Skip %u bytes %u bits",
                 len / SXM_ARCH_BITS_IN_BYTE,
                 len % SXM_ARCH_BITS_IN_BYTE);

        sxm_bitbuff_skip(pBitBuf, len);

        if (0 != pBitBuf->err)
        {
            PLOG_UTL("   ..! Skip: Error");
        }
    }
    else
    {
        sxm_bitbuff_skip(pBitBuf, len);
    }

    return;
}

/***************************************************************************//**
 * The cstring routine skips a C-style character string.
 * (debug version of the \ref sxm_bitbuff_skip_cstring)
 *
 * \param[in] pkt pointer to an initialized BitBuff structure
 * \param[in] flag 1 - enable debug functionality,
 *                 0 - behave like sxm_bitbuff_skip_cstring
 *
 ******************************************************************************/
void sxm_bitbuff_skip_cstring_debug(SXMBitBuff *pkt, int flag)
{
    if (0 != flag)
    {
        char c;

        PLOG_UTL("   ... Skipping string chars until NULL: ");

        do
        {
            c = (char)sxm_bitbuff_read(pkt, 8);
            PLOG_UTL("   ... %c", c);
        } while ('\0' != c);

        PLOG_UTL("   ... String completed.");

        if (0 != pkt->err)
        {
            PLOG_UTL("   ..! sxm_bitbuff_skip_cstring_debug: Error");
        }
    }
    else
    {
        sxm_bitbuff_skip_cstring(pkt);
    }

    return;
}

/***************************************************************************//**
 * The iso routine reads a ISO 8859-1 Coded text string.
 *
 * \param[in] pkt pointer to an initialized BitBuff structure
 * \param[in] var variable name
 * \param[in] desc pointer to a character buffer to receive the string
 * \param[in] len maximum number of characters to read (including null)
 * \param[in] flag 1 enable debug functionality,
 *                 0 - behave like sxm_bitbuff_iso
 *
 * \retval 0 if the string was correctly read
 * \retval non-0 otherwise
 *
 ******************************************************************************/
int sxm_bitbuff_iso_debug(SXMBitBuff *pkt,
                          const char *var,
                          char *desc,
                          int len,
                          int flag)
{
    if (flag == 0)
    {
        sxm_bitbuff_iso(pkt, desc, len);
        return pkt->err;
    }

    if (0 != pkt->err)
    {
        PLOG_UTL("   ..! %s [%d]: Error", var, len);
        return 1;
    }

    sxm_bitbuff_cstring(pkt, desc, len);

    if  (0 != pkt->err)
    {
        PLOG_UTL("   ..! %s [%d]: Error", var, len);
        return 1;
    }

    PLOG_UTL("   ... %s [%d]: \"%s\"", var, len, desc);

    return 0;
}

/************************************************************************
*                                                                      *
*            Local functions                                           *
*            =================                                         *
*                                                                      *
************************************************************************/
/***************************************************************************//**
* The read routine consumes and returns bytes from buffer. This implementation
* is for simple bitbuffer.
*
* \param[in] pBitBuf pointer to an initialized SXMBitBuff structure 
* \param[in] len the number of bytes to consume
* \param[out] out pointer to output buffer
*
* \return length of output data
*
******************************************************************************/
static size_t bitbuff_read_bytes_buf(SXMBitBuff *pBitBuf, size_t len, byte *out)
{
    if (len > 0)
    {
        uint bytesLeft = (uint)(pBitBuf->e - pBitBuf->b);

        if (bytesLeft < (uint)len)
        {
            len = (size_t)bytesLeft;
        }

        if (len > 0)
        {
            memcpy(out, pBitBuf->b, len);
            pBitBuf->b += len;
        }
        else
        {
            pBitBuf->err = 1;
        }
    }

    return len;
}

/***************************************************************************//**
* The read routine consumes and returns bytes from file buffer
*
* \param[in] pBitBuf pointer to an initialized SXMBitBuff structure which hides
 *              SXMBitFileBuff underneath
* \param[in] len the number of bytes to consume
* \param[out] out pointer to output buffer
*
* \return length of output data
*
******************************************************************************/
static size_t bitbuff_read_bytes_file(SXMBitBuff *pBitBuf, size_t len, byte *out)
{
    SXMBitFileBuff *fbuf = (SXMBitFileBuff *)pBitBuf;
    size_t ret = 0;
    uint left = (uint)len;

    if (fbuf->b.err)
    {
        return 0;
    }

    if (fbuf->b.valid)
    {
        return 0;
    }

    if (left > 0)
    {
        /* First read data from the buffer if it is not empty */
        uint to_copy = (uint)(fbuf->b.e - fbuf->b.b);
        if (to_copy > 0)
        {
            to_copy = (to_copy > left) ? left : to_copy;
            memcpy(out, fbuf->b.b, to_copy);
            left -= to_copy;
            fbuf->b.b += to_copy;
            ret = to_copy;
            out += ret;
        }

        /* checking if buffer is exhausted but number of bytes in buffer is
           not enough */
        if (left > 0)
        {
            uint to_read = left;
            uint fret, rvalue;

            if ((fbuf->size > 0) && ((fbuf->pos + to_read) >= fbuf->size))
            {
                to_read = fbuf->size - fbuf->pos;
                fbuf->pos += to_read;
            }
            
            do
            {
                /* reducing reading value to avoid Windows fread issue */
                rvalue = MIN(to_read, BITBUFF_FILE_BLOCK_READ_SIZE);
                fret = (uint)fread(out, sizeof(byte), rvalue, fbuf->in);
                to_read -= fret;
                ret += fret;
                out += fret;
            } while ((fret > 0) && (to_read > 0));

            if (ret < len)
            {
                // need to set err & eof flags to avoid other readings
                fbuf->b.err = 1;
                fbuf->eof = 1;
            }

        }
    }

    return ret;
}

/***************************************************************************//**
 * The seek function implementation for simple bitbuffer
 *
 * \param[in] b pointer to an initialized SXMBitBuff structure
 * \param[in] offset number of bytes from the beginning to shift
 * 
 * \return 0 if seek operation successfull, or 1 if not
 *
 ******************************************************************************/
static int bitbuff_seek_buf(SXMBitBuff *b, size_t offset)
{
    if ((uint)(b->e - b->ob) < (uint)offset)
    {
        return 1;
    }

    b->valid = 0;
    b->seed = 0;
    b->err = 0;
    b->b = b->ob + offset;

    return 0;
}

/***************************************************************************//**
 * The seek function implementation for bitfilebuffer
 *
 * \param[in] b pointer to an initialized SXMBitBuff structure which hides
 *              SXMBitFileBuff underneath
 * \param[in] offset number of bytes from the beginning to shift
 * 
 * \return 0 if seek operation successfull, or 1 if not
 *
 ******************************************************************************/
static int bitbuff_seek_file(SXMBitBuff *b, size_t offset)
{
    SXMBitFileBuff *pBitBuf = (SXMBitFileBuff *)b;
    if ((pBitBuf->size > 0) && ((uint)offset >= pBitBuf->size))
    {
        return 1;
    }

    if (0 != fseek(pBitBuf->in, (int)offset, SEEK_SET))
    {
        return 1;
    }

    pBitBuf->pos = (uint)offset;
    pBitBuf->b.err = 0;
    pBitBuf->b.b = pBitBuf->fb;
    pBitBuf->b.e = pBitBuf->fb;
    pBitBuf->eof = 0;

    return 0;
}

/***************************************************************************//**
* Extraction of uncompressed block
*
* \param[in, out] zipbuff pointer to open SXMBitZipBuff
* \param[in] max maximum size of data extract
* 
* \return 0 If any error happened or any positive value
*
*******************************************************************************/
static int bitbuff_extract_uncomp_block(SXMBitZipBuff *zipbuff, uint max)
{
    SXMBitBuff *pkt = zipbuff->input;
    size_t result = 0;
    uint to_copy, copied = 0;

    if ((zipbuff->state == SXM_BITZIP_BUFF_STATE_BLOCK_COMPLETED)  || 
        (zipbuff->state == SXM_BITZIP_BUFF_STATE_INITIAL))
    {
        sxm_bitbuff_align(pkt);
        zipbuff->rlen = NEXTR(16); /* reading len */
        NEXTR(16);
        if (zipbuff->rlen == 0)
        {
            zipbuff->state = SXM_BITZIP_BUFF_STATE_ERROR;
            return 0;
        }

        zipbuff->state = SXM_BITZIP_BUFF_STATE_UNCOMP;
    }

    /* extracting to out with block size */
    to_copy = (((uint)sizeof(zipbuff->ringbuf) - zipbuff->bufcount) > zipbuff->rlen) ?
        zipbuff->rlen : ((uint)sizeof(zipbuff->ringbuf) - zipbuff->bufcount);

    to_copy = (max > to_copy) ? to_copy : max;
        
    while (to_copy > 0)
    {
        uint chunk;

        /* calculating data size to copy */
        if ((zipbuff->bufhead > zipbuff->buftail) || (zipbuff->bufcount == 0))
        {
            chunk = (uint)(zipbuff->bufend - zipbuff->buftail + 1);
        }
        else
        {
            chunk = (uint)(zipbuff->bufhead - zipbuff->buftail);
        }

        chunk = (to_copy > chunk) ? chunk : to_copy;

        result = sxm_bitbuff_read_bytes(pkt, chunk, zipbuff->buftail);

        if ((result == 0) || (result != chunk))
        {
            /* error or input is exhausted, need to stop here */
            return 0;
        }

        copied += (uint)result;

        zipbuff->buftail += result;
        zipbuff->bufcount += (uint)result;
        if (zipbuff->buftail > zipbuff->bufend)
        {
            /* moving back to start if tail is passed the end */
            zipbuff->buftail = zipbuff->ringbuf;
        }

        to_copy -= (uint)result;
    }

    zipbuff->rlen -= copied;
    if (zipbuff->rlen == 0)
    {
        if (zipbuff->bfinal != 0)
        {
            zipbuff->state = SXM_BITZIP_BUFF_STATE_ALL_COMPLETED;
        }
        else
        {
            zipbuff->state = SXM_BITZIP_BUFF_STATE_BLOCK_COMPLETED;
        }
    }

    return (int)copied;
}

/***************************************************************************//**
* Reverse the order of the 'len' lest-significant bits in a word
* 
* \param[in] in
* \param[in] len
*
* \return reversed value
*
*******************************************************************************/
static uint reverse(uint in, uint len)
{
    uint res = 0;
    uint m1 = 1U;
    uint j;

    for  (j = 0; j < len; j++)
    {
        uint r = (in & m1) >> j;
        m1 <<= 1;
        res |= (r << (len - j - 1));
    }

    return res;
}

/***************************************************************************//**
* Build a Huffman table values with count values at each bit-length
* 
* \param[in] counts
* \param[in] maxc
* \param[out] values
*
*******************************************************************************/
static void bitbuff_deflate_build_table(const uint *counts, const uint maxc,
    uint *values)
{
    uint bl_counts[32] = {0};
    uint i;
    uint code = 0;
    uint next_code[32];

    /*  count how many codes there are for each bit-length */
    for (i = 0; i < maxc; i++)
    {
        bl_counts[counts[i]]++;
    }

    /*  find the starting point of each bit-length */
    for (i = 1; i < 32; i++)
    {
        code = (code + bl_counts[i - 1]) << 1;
        next_code[i] = code;
    }

    /*  assign the values */
    for (i = 0; i < maxc; i++)
    {
        uint len = counts[i];
        if (len != 0)
        {
            values[i] = reverse(next_code[len], len);
            next_code[len]++;
        }
    }

    return;
}

/***************************************************************************//**
* Cleans dynamic block and deallocates heap in the block
* 
* \param[in] zipbuff Main structure pointer
*
*******************************************************************************/
static void bitbuff_uninit_dyn_block(SXMBitZipBuff *zipbuff)
{
    if (zipbuff->bdynamic == NULL)
    {
        return;
    }

    if (zipbuff->bdynamic->litdist != NULL)
    {
        sxe_free(zipbuff->bdynamic->litdist);
    }

    if (zipbuff->bdynamic->litvalues != NULL)
    {
        sxe_free(zipbuff->bdynamic->litvalues);
    }

    sxe_free(zipbuff->bdynamic);
    zipbuff->bdynamic = NULL;

    return;
}

/***************************************************************************//**
* Preparing dynamic Huffman codes  
* 
* \param[in] zipbuffer initalized zipbuffer structure
*
* \return 0 if prepared successfully, -1 if any error happened
*
*******************************************************************************/
static int bitbuff_start_dynamic(SXMBitZipBuff *zipbuff)
{
    SXMBitBuff *pkt = zipbuff->input;

    if (zipbuff->bdynamic != NULL)
    {
        return 0;
    }

    switch (0) { default: {
        uint i, last = 0;
        uint hclengths[19] = {0};
        uint hcvalues[19];

        zipbuff->bdynamic = sxe_calloc(1, sizeof(*(zipbuff->bdynamic)));
        if (zipbuff->bdynamic == NULL)
        {
            break;
        }

        zipbuff->bdynamic->nlit = NEXTR(5) + 257;
        zipbuff->bdynamic->ndist = NEXTR(5) + 1;
        zipbuff->bdynamic->nbit = NEXTR(4) + 4;

        zipbuff->bdynamic->litdist = (uint*)sxe_calloc(
            zipbuff->bdynamic->nlit + zipbuff->bdynamic->ndist + 4, sizeof(uint));
        if (zipbuff->bdynamic->litdist == NULL)
        {
            break;
        }

        zipbuff->bdynamic->litvalues = (uint*)sxe_calloc(
            zipbuff->bdynamic->nlit + zipbuff->bdynamic->ndist + 4, sizeof(uint));
        if (zipbuff->bdynamic->litvalues == NULL)
        {
            break;
        }

        for (i = 0; i < zipbuff->bdynamic->nbit; i++)
        {
            hclengths[hcindex[i]] = NEXTR(3);
        }

        bitbuff_deflate_build_table(hclengths, 19, hcvalues);

        i = 0;

        while (i < (zipbuff->bdynamic->nlit + zipbuff->bdynamic->ndist))
        {
            uint j;
            for (j = 0; j < 19; j++)
            {
                if ((hclengths[j] != 0) && EATR(hcvalues[j], hclengths[j]))
                {
                    break;
                }
            }

            if (j == 19)
            {
                /* error in NLIT decode */
                break;
            }

            if (j < 16)
            {
                zipbuff->bdynamic->litdist[i++] = j;
                last = j;
            }
            else if (j == 16)
            {
                uint rep = NEXTR(2) + 3U;
                while (rep-- > 0)
                {
                    zipbuff->bdynamic->litdist[i++] = last;
                }
            }
            else if (j == 17)
            {
                uint rep = NEXTR(3) + 3U;
                while (rep-- > 0)
                {
                    zipbuff->bdynamic->litdist[i++] = 0;
                }
                last = 0;
            }
            else
            {
                uint rep = NEXTR(7) + 11;
                while (rep-- > 0)
                {
                    zipbuff->bdynamic->litdist[i++] = 0;
                }
                last = 0;
            }
        }

        bitbuff_deflate_build_table(zipbuff->bdynamic->litdist, zipbuff->bdynamic->nlit,
            zipbuff->bdynamic->litvalues);
        bitbuff_deflate_build_table(zipbuff->bdynamic->litdist + zipbuff->bdynamic->nlit,
            zipbuff->bdynamic->ndist, zipbuff->bdynamic->litvalues + 
            zipbuff->bdynamic->nlit);

        zipbuff->state = SXM_BITZIP_BUFF_STATE_DCOMP;

        return 0;
    }}

    bitbuff_uninit_dyn_block(zipbuff);
    zipbuff->state = SXM_BITZIP_BUFF_STATE_ERROR;

    return -1;
}

/***************************************************************************//**
*
* Extracting compressed block data
* 
* \param[in] zipbuffer initalized zipbuffer structure
* \param[in] block iniitalized block containing Huffman codes
* \param[in] max maximum size of data to extract
*
* \return -1 if error, 0 if max is less than minimum length of data to be
*         extracted or any positive value if extraction successfull
*
*******************************************************************************/
static int bitbuff_extract_comp_block(SXMBitZipBuff *zipbuff, 
    const SXMDeflateCompBlock *block, uint max)
{
    SXMBitBuff *pkt = zipbuff->input;
    uint result = 0;

    for (;;)
    {
        uint j;

        if (pkt->err)
        {
            PLOG_UTL("deflate: ***Packet Error Detected");
            zipbuff->state = SXM_BITZIP_BUFF_STATE_ERROR;
            return -1;
        }

        if (max < (result + zipbuff->rlen))
        {
            return (int)result;
        }

        if ((sizeof(zipbuff->ringbuf) - zipbuff->bufcount < zipbuff->rlen) ||
            (sizeof(zipbuff->ringbuf) - zipbuff->bufcount == 0))
        {
            /* buffer is full */
            return (int)result;
        }

        if (zipbuff->rlen != 0)
        {
            uint back, k;
            
            /* previous extract has unprocessed sequence */
            for (k = 0; k < block->ndist; k++)
            {
                if ((block->litdist[block->nlit + k] > 0) &&
                    EATR(block->litvalues[block->nlit + k],
                        block->litdist[block->nlit + k]))
                {
                    break;
                }
            }

            if (k == block->ndist)
            {
                PLOG_UTL("deflate. ***Error in Literal decode");
                return -1;
            }

            if (k < 4)
            {
                back = k + 1;
            }
            else
            {
                uint bitk = (k - 2) / 2;
                uint l0 = NEXTR(bitk);
                uint bb = (k % 2) * (1U << bitk);
                uint base = (1U << (bitk + 1)) + 1;

                back = base + bb + l0;
            }

            if (back > (SXM_BITZIP_BUFF_WINDOW_SIZE - 1))
            {
                PLOG_UTL("Out of window size.");
                return -1;
            }
            else
            {
                byte *p = zipbuff->buftail - back;

                /* rewinding "back" bytes */
                if (p < zipbuff->ringbuf)
                {
                    /* looks like back is crossed the ring separator*/
                    p += sizeof(zipbuff->ringbuf);
                }

                /* we are ready to fill buffer */
                result += zipbuff->rlen;

                while (zipbuff->rlen > 0)
                {
                    *(zipbuff->buftail)++ = *p++;
                    zipbuff->rlen--;
                    zipbuff->bufcount++;

                    if (zipbuff->buftail > zipbuff->bufend)
                    {
                        zipbuff->buftail = zipbuff->ringbuf;
                    }

                    if (p > zipbuff->bufend)
                    {
                        p = zipbuff->ringbuf;
                    }
                }
            }
        }

        for (j = 0; j < block->nlit; j++)
        {
            if ((block->litdist[j] > 0) &&
                EATR(block->litvalues[j], block->litdist[j]))
            {
                break;
            }
        }

        if (j == block->nlit)
        {
            PLOG_UTL("deflate. ***Error in Literal decode");
            zipbuff->state = SXM_BITZIP_BUFF_STATE_ERROR;
            return -1;
        }

        if (j < 256)
        {
            *(zipbuff->buftail)++ = (byte)j;

            if (zipbuff->buftail > zipbuff->bufend)
            {
                zipbuff->buftail = zipbuff->ringbuf;
            }

            result++;
            zipbuff->bufcount++;

            if (zipbuff->bufcount >= sizeof(zipbuff->ringbuf))
            {
                /* buffer is full */
                return (int)result;
            }
        }
        else if (j == 256)
        {
            /* finish here */
            zipbuff->state = SXM_BITZIP_BUFF_STATE_BLOCK_COMPLETED;
            return (int)result;
        }
        else
        {
            if (j < 265)
            {
                zipbuff->rlen = j - 254;
            }
            else if (j < 269)
            {
                uint l0 = NEXTR(1);
                zipbuff->rlen = 11 + (j - 265) * 2 + l0;
            }
            else if (j < 273)
            {
                uint l0 = NEXTR(2);
                zipbuff->rlen = 19 + (j - 269) * 4 + l0;
            }
            else if (j < 277)
            {
                uint l0 = NEXTR(3);
                zipbuff->rlen = 35 + (j - 273) * 8 + l0;
            }
            else if (j < 281)
            {
                uint l0 = NEXTR(4);
                zipbuff->rlen = 67 + (j - 277) * 16 + l0;
            }
            else if (j < 285)
            {
                uint l0 = NEXTR(5);
                zipbuff->rlen = 131 + (j - 281) * 32 + l0;
            }
            else if (j == 285)
            {
                zipbuff->rlen = 258;
            }
            else
            {
                PLOG_UTL("deflate. ***Error in length decode");
                zipbuff->state = SXM_BITZIP_BUFF_STATE_ERROR;
                return -1;
            }

            if ((sizeof(zipbuff->ringbuf) - zipbuff->bufcount) < 
                zipbuff->rlen)
            {
                /* not enough space to read sequence, exit here and wait for read */
                return (int)result;
            }
        }
    }
}

/***************************************************************************//**
*
* Start extraction of Huffman fixed codes block
* 
* \param[in] zipbuffer initalized zipbuffer structure
* \param[in] max maximum size of data to extract
*
* \return -1 if error, 0 if max is less than minimum length of data to be
*         extracted or any positive value if extraction successfull
*
*******************************************************************************/
static int bitbuff_extract_fixed_block(SXMBitZipBuff *zipbuff, uint max)
{
    int result = 0;

    if ((zipbuff->state == SXM_BITZIP_BUFF_STATE_INITIAL) ||
        (zipbuff->state == SXM_BITZIP_BUFF_STATE_BLOCK_COMPLETED))
    {
        zipbuff->state = SXM_BITZIP_BUFF_STATE_FCOMP;
    }

    result = bitbuff_extract_comp_block(zipbuff, &zipbuff->bfixed, max);

    return result;
}

/***************************************************************************//**
*
* Start extraction of Huffman dynamic codes block
* 
* \param[in] zipbuffer initalized zipbuffer structure
* \param[in] max maximum size of data to extract
*
* \return -1 if error, 0 if max is less than minimum length of data to be
*         extracted or any positive value if extraction successfull
*
*******************************************************************************/
static int bitbuff_extract_dynamic_block(SXMBitZipBuff *zipbuff, uint max)
{
    int result = 0;

    if ((zipbuff->state == SXM_BITZIP_BUFF_STATE_INITIAL) ||
        (zipbuff->state == SXM_BITZIP_BUFF_STATE_BLOCK_COMPLETED))
    {
        result = bitbuff_start_dynamic(zipbuff);
        if (result != 0)
        {
            return result;
        }
    }

    result = bitbuff_extract_comp_block(zipbuff, zipbuff->bdynamic, max);

    if ((zipbuff->state == SXM_BITZIP_BUFF_STATE_BLOCK_COMPLETED) || 
        (zipbuff->state == SXM_BITZIP_BUFF_STATE_ERROR))
    {
        /* block should be cleaned since it cannot be reused */
        bitbuff_uninit_dyn_block(zipbuff);
    }

    return result;
}

/***************************************************************************//**
* Read and verify gzip header
* 
* \param[in] zipbuffer initalized zipbuffer structure
*
* \return -1 if error, 0 if header contain expected data
*
*******************************************************************************/
static int bitbuff_read_gzip_header(SXMBitZipBuff *zipbuff)
{
    SXMBitBuff *pkt = zipbuff->input;

    uint t = NEXT_BYTE(); // ID1
    uint f;

    t = (t << 8) | NEXT_BYTE(); // ID2

    if (t != 0x1f8b)
    {
        PLOG_UTL("deflate. Not a GZIP file");
        return -1;
    }

    if (NEXT_BYTE() != 8) // CM
    {
        PLOG_UTL("deflate. File is not a FLATE compressed");
        return -1;
    }

    f = NEXT_BYTE(); // FLG

    sxm_bitbuff_skip(pkt, 32 + 8 + 8); // MTIME, XLF and OS

    /* Skip any extra data in the file */
    if (f & 4) // FLG.FEXTRA
    {
        t = NEXT_BYTE();
        t = (t << 8) | NEXT_BYTE();

        sxm_bitbuff_skip(pkt, t * SXM_ARCH_BITS_IN_BYTE);
    }

    if (f & 8) // FLG.FNAME
    {
        do
        {
            t = NEXT_BYTE();
        } while (t);
    }

    if (f & 16) // FLG.FCOMMENT
    {
        do
        {
            t = NEXT_BYTE();
        } while (t);
    }

    if (pkt->err)
    {
        return -1;
    }

    return 0;
}

/***************************************************************************//**
* Fill ring buffer with new uncompressed data
* 
* \param[in] b initalized zipbuffer structure
*
* \return 1 if error happened or data completed, 0 if buffer filled successfully
*
*******************************************************************************/
static int bitbuff_replenish_deflate(SXMBitBuff *b)
{
    SXMBitZipBuff *zipbuff = (SXMBitZipBuff *)b;
    SXMBitBuff *pkt = zipbuff->input;
    int result = 0;
    uint max;

    if ((zipbuff->b.b != zipbuff->bufhead) && (zipbuff->bufcount > 0))
    {
        uint delta = (uint)(zipbuff->b.b - zipbuff->bufhead);

        zipbuff->bufcount -= delta;
        zipbuff->pos += delta;

        zipbuff->bufhead = zipbuff->b.b;
        if (zipbuff->bufhead > zipbuff->bufend)
        {
            zipbuff->bufhead = zipbuff->ringbuf;
        }

        if (zipbuff->bufcount > 0)
        {
            /* need to process remaining chunk of data */
            zipbuff->b.b = zipbuff->bufhead;
            zipbuff->b.e = (zipbuff->buftail < zipbuff->bufhead) ?
                (zipbuff->bufend + 1) : zipbuff->buftail;
            return 0;
        }
    }

    if ((zipbuff->state == SXM_BITZIP_BUFF_STATE_ALL_COMPLETED) ||
        (zipbuff->state == SXM_BITZIP_BUFF_STATE_ERROR))
    {
        /* nothing to do */
        return 1;
    }

    /* replenish means we need to fill the buffer */
    /* that's why we fill it until block is exhausted */
    if (zipbuff->state == SXM_BITZIP_BUFF_STATE_INITIAL)
    {
        /* reading header */
        result = bitbuff_read_gzip_header(zipbuff);
        if (result != 0)
        {
            zipbuff->state = SXM_BITZIP_BUFF_STATE_ERROR;
            return 1;
        }

        /* initializing bit buffer pointer to head */
        zipbuff->state = SXM_BITZIP_BUFF_STATE_BLOCK_COMPLETED;
        zipbuff->b.b = zipbuff->bufhead;
    }

    max = (uint)(sizeof(zipbuff->ringbuf) - zipbuff->bufcount);

    if (zipbuff->state == SXM_BITZIP_BUFF_STATE_BLOCK_COMPLETED)
    {
        byte n;

        zipbuff->bfinal = (byte)NEXTR(1);     /* last block flag */
        n = (byte)NEXTR(2);                   /* block compression type */

        if (n == 0)
        {
            result = bitbuff_extract_uncomp_block(zipbuff, max);
        }
        else if (n == 1)
        {
            result = bitbuff_extract_fixed_block(zipbuff, max);
        }
        else if (n == 2)
        {
            result = bitbuff_extract_dynamic_block(zipbuff, max);
        }
        else
        {
            zipbuff->state = SXM_BITZIP_BUFF_STATE_ERROR;
            return 1;
        }
    }
    else if (zipbuff->state == SXM_BITZIP_BUFF_STATE_UNCOMP)
    {
        result = bitbuff_extract_uncomp_block(zipbuff, max);
    }
    else if (zipbuff->state == SXM_BITZIP_BUFF_STATE_DCOMP)
    {
        result = bitbuff_extract_comp_block(zipbuff, zipbuff->bdynamic, max);
        if ((zipbuff->state == SXM_BITZIP_BUFF_STATE_BLOCK_COMPLETED) ||
            (zipbuff->state == SXM_BITZIP_BUFF_STATE_ERROR))
        {
            /* block should be cleaned since it cannot be reused */
            bitbuff_uninit_dyn_block(zipbuff);
        }
    }
    else if (zipbuff->state == SXM_BITZIP_BUFF_STATE_FCOMP)
    {
        result = bitbuff_extract_comp_block(zipbuff, &zipbuff->bfixed, max);
    }
    else
    {
        PLOG_UTL("Unexpected state %d", (int)zipbuff->state);
    }

    if ((zipbuff->state == SXM_BITZIP_BUFF_STATE_BLOCK_COMPLETED) &&
        (zipbuff->bfinal != 0))
    {
        zipbuff->state = SXM_BITZIP_BUFF_STATE_ALL_COMPLETED;
    }

    if (result == 0)
    {
        /* nothing to do here, buffer is exhausted or in error state */
        return 1;
    }
    else
    {
        zipbuff->b.b = zipbuff->bufhead;
        zipbuff->b.e = (zipbuff->buftail <= zipbuff->bufhead) ?
            (zipbuff->bufend + 1) : zipbuff->buftail;
    }

    return 0;
}

static int bitbuff_seek_deflate(SXMBitBuff *b, size_t offset)
{
    SXMBitZipBuff *zipbuff = (SXMBitZipBuff *)b;

    if (zipbuff->state == SXM_BITZIP_BUFF_STATE_ERROR)
    {
        return 1;
    }

    if (zipbuff->pos == (uint)offset)
    {
        /* all is ok */
        return 0;
    }
    else if (zipbuff->pos > (uint)offset)
    {
        uint delta;
        /* need to move to file start */
        int rc = sxm_bitbuff_seek(zipbuff->input, 0);
        if (rc != 0)
        {
            zipbuff->state = SXM_BITZIP_BUFF_STATE_ERROR;
            zipbuff->b.err = 1;
            return 1;
        }

        zipbuff->state = SXM_BITZIP_BUFF_STATE_INITIAL;
        zipbuff->bufhead = zipbuff->ringbuf;
        zipbuff->buftail = zipbuff->ringbuf;
        zipbuff->bufcount = 0;
        zipbuff->pos = 0;
        zipbuff->rlen = 0;
        zipbuff->bfinal = 0;
        bitbuff_uninit_dyn_block(zipbuff);
        rc = 0;

        while (((uint)offset > (zipbuff->pos + zipbuff->bufcount)) && (rc == 0))
        {
            zipbuff->b.b = zipbuff->b.e;
            rc = bitbuff_replenish_deflate((SXMBitBuff *)zipbuff);
        }

        if (rc != 0)
        {
            /* offset is too large or error happened */
            zipbuff->state = SXM_BITZIP_BUFF_STATE_ERROR;
            zipbuff->b.err = 1;
            return 1;
        }

        delta = (uint)offset - zipbuff->pos;

        zipbuff->pos = (uint)offset;
        zipbuff->bufcount -= delta;
        zipbuff->bufhead += delta;
        if (zipbuff->bufhead > zipbuff->bufend)
        {
            zipbuff->bufhead -= sizeof(zipbuff->ringbuf);
        }

        zipbuff->b.b = zipbuff->bufhead;
    }
    else /* zipbuff->pos < offset case */
    {
        int rc = 0;
        uint delta;

        while (((uint)offset > (zipbuff->pos + zipbuff->bufcount)) && (rc == 0))
        {
            zipbuff->b.b = zipbuff->b.e;
            rc = bitbuff_replenish_deflate((SXMBitBuff *)zipbuff);
        }

        if (rc != 0)
        {
            /* offset is too large or error happened */
            zipbuff->state = SXM_BITZIP_BUFF_STATE_ERROR;
            zipbuff->b.err = 1;
            return 1;
        }

        delta = (uint)offset - zipbuff->pos;

        zipbuff->pos = (uint)offset;
        zipbuff->bufcount -= delta;
        zipbuff->bufhead += delta;
        if (zipbuff->bufhead > zipbuff->bufend)
        {
            zipbuff->bufhead -= sizeof(zipbuff->ringbuf);
        }

        zipbuff->b.b = zipbuff->bufhead;
    }

    return 0;
}

/***************************************************************************//**
* Fill provided buffer with uncompressed data
*
* \param[in] b initialized SXMBitZipBuff structure
* \param[in] count number of bytes to place to buffer, can be smaller or 
*                  larger than actual data size
* \param[in] out buffer to fill
*
* \return    number of bytes copied to buffer
*
*******************************************************************************/
static size_t bitbuff_read_bytes_deflate(SXMBitBuff *b, size_t count, byte *out)
{
    SXMBitZipBuff *zipbuff = (SXMBitZipBuff *)b;
    size_t result = 0;
    int rc = 0;

    if (zipbuff->state == SXM_BITZIP_BUFF_STATE_ERROR)
    {
        return 0;
    }

    /* before any actions, need to sync head and current bitbuff position */
    if ((zipbuff->b.b != zipbuff->bufhead) && (zipbuff->bufcount > 0))
    {
        uint delta = (uint)(zipbuff->b.b - zipbuff->bufhead);

        zipbuff->bufcount -= delta;
        zipbuff->pos += delta;

        zipbuff->bufhead = zipbuff->b.b;
        if (zipbuff->bufhead > zipbuff->bufend)
        {
            zipbuff->bufhead = zipbuff->ringbuf;
        }
    }

    while (((uint)count > 0) && (rc == 0))
    {
        if (zipbuff->bufcount > 0)
        {
            uint to_copy = (zipbuff->bufhead >= zipbuff->buftail) ? 
                (uint)(zipbuff->bufend - zipbuff->bufhead + 1) : zipbuff->bufcount;
            to_copy = ((uint)count > to_copy) ? to_copy : (uint)count;

            memcpy(out, zipbuff->bufhead, to_copy);
            out += to_copy;

            zipbuff->bufcount -= to_copy;
            zipbuff->bufhead += to_copy;
            zipbuff->pos += to_copy;
            if (zipbuff->bufhead > zipbuff->bufend)
            {
                zipbuff->bufhead = zipbuff->ringbuf;
            }

            zipbuff->b.b = zipbuff->bufhead;
            count -= to_copy;
            result += to_copy;
        
            if (count == 0)
            {
                /* completed, need to stop here */
                break;
            }
        }
        else
        {
            rc = bitbuff_replenish_deflate((SXMBitBuff*)zipbuff);
        }
    }

    return result;
}

