/************************************************************************
 *                                                                      *
 *            SXM_LLI                                                   *
 *            =======                                                   *
 *                                                                      *
 *                                 Copyright 2013 Sirius XM Radio, Inc. *
 *                                                 All Rights Reserved. *
 *               Licensed Materials - Property of Sirius XM Radio, Inc. *
 *                                                                      *
 *    Implementation of the common Low-level-interface API              *
 *    for components using the SDK directly                             *
 *                                                                      *
 ************************************************************************/
#if defined(SXM_USE_SMS)

#define DEBUG_TAG "sms_lli"
#define DEBUG_VAR ((shell) ? shell->debug_level : -1)

#include <sxm_build.h>
#include <stdio.h>
#include <sxm_sdk.h>
#include <util/sxm_noname_internal.h>
#include <sms_api.h>
#include <sxm_gz_defs.h>
#include <util/lli.h>
#include "sms_lli_internal.h"

static const char* LLI_SRH_DRIVER_NAME = "srh:";
static char sFilterText[8 * SXM_LLI_DATA_SERVICE_DMI_FILTER_SIZE + 1];

/* Forward declaration */
typedef struct LLIShell LLIShell;
typedef struct LLISmsService LLISmsService;
typedef struct LLICore LLICore;

struct LLISmsService
{
    DATASERVICE_MGR_OBJECT smsHandle;
    DATASERVICE_STATE_ENUM smsState;
    DATASERVICE_ERROR_CODE_ENUM smsError;
    LLIShell* shell;
    ushort shutDown;
};

/* SMS callbacks use the core service through a shell. We can free up service resources independent of the shell
 * This way the SMS generic data services behave more like the sdk lli data provider.  The sdk provider
 * is a synchronous stop.  An SMS generic data service is an asynchronous stop.
 */
struct LLIShell
{
    LLISmsService smsService[SXM_LLI_DATA_SERVICE_DSI_COUNT];
    pthread_mutex_t mutex;
    LLICore* core;
    pthread_t pid;
    int debug_level;
    byte shutDown;
    byte reportUp;
    byte newfilter;
    ushort waitBit;
};

struct LLICore
{
    SXMDataService *ds;
    SXMRfdService rfd;
    byte sdtpAuBuffer[5120];
    int manageGlobal;
};

pthread_mutex_t wait_mutex = PTHREAD_MUTEX_INITIALIZER;

#define WAIT_BIT_COUNT (256)
#define WAIT_BIT_UNUSED (0xffff)
uint wait_bits[UINTS_FOR_BITS(WAIT_BIT_COUNT)];

static void init_team_league_cache (void);

SXE_INLINE static void SetServiceStatus (SXMStatus *status, int value)
{
    status->service = value;
}

SXE_INLINE static void SetSubsStatus (SXMStatus *status, int value)
{
    status->substate = value;
}

SXE_INLINE static void reportUp (LLIShell* shell, int nature, int details)
{
    if (!shell->shutDown && shell->reportUp)
    {
        shell->core->ds->report (nature, details, shell->core->ds->dsuser);
    }
    else
    {
        DEBUG_LLI("Shell blocked report up. shutdown=%u reportup=%u",
                  shell->shutDown, shell->reportUp);
    }
}

SXE_INLINE static ushort getWaitBit (void)
{
    ushort i;

    LOCK(wait_mutex);
    for (i = 0; i < WAIT_BIT_COUNT; i++)
    {
        if (!BITP(wait_bits, i))
        {
            BITS(wait_bits, i);
            break;
        }
    }

    UNLOCK(wait_mutex);

    return i;
}

static void smsWaitStoppedCallback (LLIShell *shell, LLISmsService *smsService)
{
    DEBUG_LLI(": Waiting...");
    while (smsService->shutDown)
    {
        sxe_usleep(100000);
        DEBUG_LLI(": +100 msec");
    }

    return;
}

static void applyDmiFilter (LLIShell* shell)
{
    uint i, j;
    LLISmsService* smsService;
    SXMDataDsi *dsiData;
    SMSAPI_RETURN_CODE_ENUM smsError;
    DATASERVICE_DMI_CONFIG_STRUCT dmiConfig[SXI_SP_MAX_DMI_CNT];

    for (i = 0; i < SXM_LLI_DATA_SERVICE_DSI_COUNT; i++)
    {
        smsService = &shell->smsService[i];
        if (!smsService->smsHandle)
        {
            continue;
        }

        dsiData = &shell->core->ds->dsis[i];

        /* Note: dmi count will always be zero for services in unknown
         * dmi mode(dmic = 0 filter[0]=filter[1] = ... = -1).
         */
        for (j = 0; j < dsiData->dmi.dmic; j++)
        {
            dmiConfig[j].tDMI = (SXM_DMI) dsiData->dmi.dmis[j];

            if (BITP(dsiData->filter, j))
            {
                dmiConfig[j].bEnable = TRUE;
            }
            else
            {
                dmiConfig[j].bEnable = FALSE;
            }

            DEBUG_LLI("(%s) dsi=%d dmi=%d",
                    dmiConfig[j].bEnable ? "Enable" : "Disable",
                    dsiData->dsi,
                    dsiData->dmi.dmis[j]);
        }

        if (dsiData->dmi.dmic > 0)
        {
            smsError = DATA.eManageDataStream (smsService->smsHandle,
                               dmiConfig, j);
            if (smsError != SMSAPI_RETURN_CODE_SUCCESS)
            {
                DEBUG_LLI("applyDmiFilter() DATA.eManageDataStream() FAILURE: smsApiError=%d dsi=%d",
                        smsError, dsiData->dsi);
            }
        }
    }
}

static void evaluateServiceStatus (LLIShell* shell)
{
    LLISmsService* smsService;
    int i;
    int totalDsiCnt;
    int initCnt, stoppedCnt, errCnt, invalidCnt;

    /*
     *  SMS combined service status and subscription status in the same state space.
     * This function gets called only when SMS makes a callback indicating change
     * that specifies a service status related enum.
     *
     * We do not know which SMS service changed.  The current logic re-evaluates the
     * state of all dsi services to determine a new master state to report up.
     */
    initCnt = stoppedCnt = errCnt = invalidCnt = 0;

    for (i = 0, totalDsiCnt = 0; i < SXM_LLI_DATA_SERVICE_DSI_COUNT; i++)
    {
        if (shell->core->ds->dsis[i].dsi)
        {
            totalDsiCnt++;
            smsService = &shell->smsService[i];
            if (smsService->smsHandle)
            {
                switch (smsService->smsState)
                {
                    case DATASERVICE_STATE_INITIAL:
                    {
                        initCnt++;
                        break;
                    }
                    case DATASERVICE_STATE_STOPPED:
                    {
                        stoppedCnt++;
                        break;
                    }
                    case DATASERVICE_STATE_ERROR:
                    {
                        errCnt++;
                        break;
                    }
                    case DATASERVICE_STATE_INVALID:
                    {
                        invalidCnt++;
                        break;
                    }
                    default:
                        break;
                }
            }
        }
    }

    if (!totalDsiCnt)
    {
        return;
    }

    /* Only consider SMS service induced status change if the main service state is good */
    if (shell->core->ds->status->service != SXM_SERVICE_OK)
    {
        return;
    }

    /* Any service in error is unrecoverable and must be reported up */
    if (errCnt || invalidCnt)
    {
        reportUp (shell, SXM_CALLBACK_ERROR, SXM_E_PIPE);
    }
}

static void evaluateServiceSubscription (LLIShell* shell)
{
    LLISmsService* smsService;
    int i;
    int totalDsiCnt;
    int unsubCnt, subCnt, unavailableCnt, partialCnt;

    /*
     * SMS combined service status and subscription status in the same state space.
     * This function gets called only when SMS makes a callback indicating change
     * that specifies a subscription status related enum.
     *
     * We do not know which SMS service changed.  The current logic re-evaluates the
     * state of all dsi services to determine a new master state to report up.
    */

    unsubCnt = subCnt = unavailableCnt = partialCnt = 0;
    for (i = 0, totalDsiCnt = 0; i < SXM_LLI_DATA_SERVICE_DSI_COUNT; i++)
    {
        if (shell->core->ds->dsis[i].dsi)
        {
            totalDsiCnt++;
            smsService = &shell->smsService[i];
            if (smsService->smsHandle)
            {
                DEBUG_LLI("SMS state for DSI %d is %d",
                        shell->core->ds->dsis[i].dsi, smsService->smsState);

                switch (smsService->smsState)
                {
                    case DATASERVICE_STATE_UNSUBSCRIBED:
                    {
                        unsubCnt++;
                        shell->core->ds->dsis[i].substate = SXM_SUBS_NONE;
                        break;
                    }
                    case DATASERVICE_STATE_READY:
                    {
                        subCnt++;
                        shell->core->ds->dsis[i].substate = SXM_SUBS_FULL;
                        break;
                    }
                    case DATASERVICE_STATE_UNAVAILABLE:
                    {
                        unavailableCnt++;
                        shell->core->ds->dsis[i].substate = SXM_SUBS_UNKNOWN;
                        break;
                    }
                    case DATASERVICE_STATE_POI_UPDATES_ONLY:
                    {
                        partialCnt++;
                        shell->core->ds->dsis[i].substate = SXM_SUBS_PARTIAL;
                        break;
                    }
                    default:
                    {
                        shell->core->ds->dsis[i].substate = SXM_SUBS_UNKNOWN;
                        break;
                    }
                }
            }
        }
    }

    if (!totalDsiCnt)
    {
        return;
    }

    /* Only process subscription status if the main service state is good*/
    if (shell->core->ds->status->service != SXM_SERVICE_OK)
        return;

    /* SMS considers the following states as initializing completed.
     *  DATASERVICE_STATE_UNSUBSCRIBED,
     *  DATASERVICE_STATE_UNAVAILABLE,
     *  DATASERVICE_STATE_POI_UPDATES_ONLY,
     *  DATASERVICE_STATE_READY,
     *
     * When all dsi services reach one of these we can transition the master service state to ok.
    */

    if (unavailableCnt)
    {
        if (shell->core->ds->status->substate != SXM_SUBS_UNKNOWN)
        {
            SetSubsStatus (shell->core->ds->status, SXM_SUBS_UNKNOWN);
            reportUp (shell, SXM_CALLBACK_SUBS,
                      shell->core->ds->status->substate);
        }
    }
    else if (subCnt == totalDsiCnt)
    {
        if (shell->core->ds->status->substate != SXM_SUBS_FULL)
        {
            applyDmiFilter (shell);
            SetSubsStatus (shell->core->ds->status, SXM_SUBS_FULL);
            reportUp (shell, SXM_CALLBACK_SUBS,
                      shell->core->ds->status->substate);
        }
    }
    else if (subCnt || partialCnt)
    {
        if (shell->core->ds->status->substate != SXM_SUBS_PARTIAL)
        {
            applyDmiFilter (shell);
            SetSubsStatus (shell->core->ds->status, SXM_SUBS_PARTIAL);
            reportUp (shell, SXM_CALLBACK_SUBS,
                      shell->core->ds->status->substate);
        }
    }
    else
    {
        if (shell->core->ds->status->substate != SXM_SUBS_NONE)
        {
            SetSubsStatus (shell->core->ds->status, SXM_SUBS_NONE);
            reportUp (shell, SXM_CALLBACK_SUBS,
                      shell->core->ds->status->substate);
        }
    }

    /* If we had pending filter update issued before service's READY state we need to
     * update the filter now */
    if (shell->newfilter != 0)
    {
        shell->newfilter = 0;
        applyDmiFilter(shell);
    }
}

static void handleSmsStateChange (LLISmsService* smsService,
                      DATASERVICE_STATE_CHANGE_STRUCT* smsStateChange)
{
    LLIShell* shell;
    int i;

    if (!smsStateChange)
    {
        return;
    }

    smsService->smsState = smsStateChange->eCurrentState;
    shell = smsService->shell;

    LOCK(shell->mutex);
    if (shell->shutDown)
    {
        if (smsStateChange->eCurrentState == DATASERVICE_STATE_STOPPED)
        {
            DEBUG_LLI("smsService stopped. handle=%u", smsService->smsHandle);
            for (i = 0; i < SXM_LLI_DATA_SERVICE_DSI_COUNT; i++)
            {
                if (shell->smsService[i].shutDown
                        && (&shell->smsService[i] != smsService))
                {
                    DEBUG_LLI("waiting for smsService %u to stop. handle=%u",
                              i, shell->smsService[i]);
                    UNLOCK(shell->mutex);
                    smsService->shutDown = 0;
                    return;
                }
            }
            UNLOCK(shell->mutex);

            /* If we got here all sms services shut down.  Free the shell.  Only free if there is no one waiting */
            LOCK(wait_mutex);
            if (shell->waitBit != WAIT_BIT_UNUSED)
            {
                if (BITP(wait_bits, shell->waitBit))
                {
                    DEBUG_LLI(
                            " all sms services shut down. User waiting: defer cleanup. waitbit=%d",
                            shell->waitBit);
                    smsService->shutDown = 0;
                    UNLOCK(wait_mutex);
                    return;
                }
            }

            DEBUG_LLI("All sms services shut down. "
                      "No user waiting perform cleanup.");
            sxm_mutex_destroy (&shell->mutex);
            smsService->shutDown = 0;
            UNLOCK(wait_mutex);
            return;
        }
    }
    else
    {
        if ((smsStateChange->eCurrentState == DATASERVICE_STATE_INITIAL) ||
            (smsStateChange->eCurrentState == DATASERVICE_STATE_STOPPED) ||
            (smsStateChange->eCurrentState == DATASERVICE_STATE_ERROR) ||
            (smsStateChange->eCurrentState == DATASERVICE_STATE_INVALID))
        {
            evaluateServiceStatus (shell);
        }
        else
        {
            evaluateServiceSubscription (shell);
        }
    }
    UNLOCK(shell->mutex);
}

static void handleSmsNewData (LLIShell* shell, OSAL_BUFFER_HDL osalBuf)
{
    size_t size, rsize;

    if (!osalBuf)
    {
        return;
    }

    LOCK(shell->mutex);
    if (shell->shutDown)
    {
        OSAL.eBufferFree (osalBuf);
        DEBUG_LLI("shell blocked packet. shutdown=%u reportup=%u",
                  shell->shutDown, shell->reportUp);
    }
    else
    {
        size = OSAL.tBufferGetSize (osalBuf);
        if (size > (size_t) sizeof(shell->core->sdtpAuBuffer))
        {
            DEBUG_LLI("SMS data service sdtp au size invalid: %d", size);
        }
        else
        {
            rsize = OSAL.tBufferReadHead (osalBuf, shell->core->sdtpAuBuffer,
                                          size);
            if (rsize == size)
            {
                if (shell->core->manageGlobal)
                {
                    sms_lli_internal_populate_sports_metadata ();
                }
                shell->core->ds->complete (shell->core->sdtpAuBuffer, size, shell->core->ds->dsuser);
            }
            else
            {
                ERROR_LLI("OSAL.tBufferReadHead result size invalid: rcv=%d expected=%d", rsize, size);
            }
        }
        OSAL.eBufferFree (osalBuf);
    }

    UNLOCK(shell->mutex);
}

static void handleSmsTimer (LLIShell* shell)
{
    LOCK(shell->mutex);
    if (shell->shutDown)
    {
        DEBUG_LLI("shell blocked timer. shutdown=%u reportup=%u",
                  shell->shutDown, shell->reportUp);
    }
    else
    {
        if (shell->core->ds->timer)
        {
            if (shell->core->manageGlobal)
            {
                sms_lli_internal_populate_sports_metadata ();
            }
            shell->core->ds->timer(shell->core->ds->dsuser);
        }
    }
    UNLOCK(shell->mutex);
}

static void handleDBUpdateNotification(LLIShell* shell)
{
    LOCK(shell->mutex);
    if (shell->core != NULL)
    {
        shell->core->ds->report(SXM_CALLBACK_DB_UPDATED, SXM_E_OK, shell->core->ds->dsuser);
    }
    UNLOCK(shell->mutex);
}

static void genericDataServiceEventCallback (
    DATASERVICE_MGR_OBJECT handle,
    DATASERVICE_EVENT_MASK eventMask,
    void *eventArg,
    void *callbackArg)
{
    LLISmsService *smsService = (LLISmsService*) callbackArg;

    if (!smsService)
    {
        return;
    }

    switch (eventMask)
    {
        case DATASERVICE_EVENT_STATE:
        {
            DATASERVICE_STATE_CHANGE_STRUCT *serviceStatusUpdate = 
                (DATASERVICE_STATE_CHANGE_STRUCT*) eventArg;

            /* Store DSM thread's PID for further callback detection */
            smsService->shell->pid = pthread_self();

            handleSmsStateChange (smsService, serviceStatusUpdate);
            break;
        }
        case DATASERVICE_EVENT_NEW_DATA:
        {
            handleSmsNewData (smsService->shell, (OSAL_BUFFER_HDL) eventArg);
            break;
        }
        case DATASERVICE_EVENT_TIMEOUT:
        {
            handleSmsTimer (smsService->shell);
            break;
        }
        case DATASERVICE_EVENT_SERVICE:
        {
            /* SXE uses this event to do RFD notification */
            handleDBUpdateNotification(smsService->shell);
            break;
        }
    }
}

static char *sFilterString(LLIShell* shell, uint *filter)
{
    if (DEBUG_VAR == 0)
    {
        sFilterText[0] = '\0';
    }
    else
    {
        uint i;

        for (i = 0; i < SXM_LLI_DATA_SERVICE_DMI_FILTER_SIZE; ++i)
        {
            snprintf(&sFilterText[i * 8], 8, "%08X", filter[i]);
        }
        sFilterText[8 * SXM_LLI_DATA_SERVICE_DMI_FILTER_SIZE] = '\0';
    }
    return &sFilterText[0];
}

static int modifyFilter (LLIShell* shell, int dsi, int dmi, int enable)
{
    LLISmsService* smsService;
    DATASERVICE_DMI_CONFIG_STRUCT dmiConfig;
    SMSAPI_RETURN_CODE_ENUM smsError;
    SXMDataDsi *dsidata;
    int i, j;
    uint fpattern[SXM_LLI_DATA_SERVICE_DMI_FILTER_SIZE];

    if (!shell || shell->shutDown)
    {
        return SXM_E_FAULT;
    }

    BITSALL(fpattern);

    /* It is not clear when SMS will take dmi monitoring requests.
     *
     * DMI monitor update scheme:
     *
     *    Always try to update the SMS object.
     *    If the SMS object update fails fail the request and let the higher layer handle it.
     * The higher layer is pruning input data.
     */
    for (i = 0; i < SXM_LLI_DATA_SERVICE_DSI_COUNT; i++)
    {
        smsService = &shell->smsService[i];
        dsidata = &shell->core->ds->dsis[i];

        DEBUG_LLI("[input] Modifying DSI %d filter = %s", 
            dsidata->dsi, 
            sFilterString(shell, &dsidata->filter[0]));

        /* We were started in unknown dmi mode.  Stop all attempts to manipulate the service by dmi */
        if (dsidata->dmi.dmic == 0)
        {
            int eqf = memcmp(dsidata->filter, fpattern, sizeof(dsidata->filter));
            if (eqf == 0)
            {
                DEBUG_LLI(
                    "modifyFilter() attempt to manipulate dmi for service without fixed dmi. dsi=%d dmi=%d enable=%d\n",
                    dsi, dmi, enable);
                continue;
            }
        }

        if ((dsidata->dsi == dsi) && smsService->smsHandle)
        {
            BOOLEAN bUpdate = FALSE;

            for (j = 0; j < dsidata->dmi.dmic; j++)
            {
                if (dsidata->dmi.dmis[j] == dmi)
                {
                    int current = BITP(dsidata->filter, j);

                    if (enable && !current)
                    {
                        BITS(dsidata->filter, j);
                        bUpdate = TRUE;
                        dmiConfig.bEnable = TRUE;
                    }
                    else if (!enable && current)
                    {
                        BITC(dsidata->filter, j);
                        bUpdate = TRUE;
                        dmiConfig.bEnable = FALSE;
                    }
                    break;
                }
            }

            if (bUpdate == TRUE)
            {
                dmiConfig.tDMI = (SXM_DMI) dmi;
                smsError = DATA.eManageDataStream (smsService->smsHandle,
                                                    &dmiConfig, 1);
                DEBUG_LLI("DMI=%d enable=%d smsError=%d", dmi, enable, smsError);
                DEBUG_LLI("[output] Modified DSI %d filter = %s",
                    dsidata->dsi,
                    sFilterString(shell, &dsidata->filter[0]));

                if (smsError == SMSAPI_RETURN_CODE_SUCCESS)
                {
                    return SXM_E_OK;
                }
                else
                {
                    return SXM_E_PIPE;
                }
            }
        }
    }

    return SXM_E_NOENT;
}

SXMRfdService* sms_lli_internal_get_rfd (ptr handle)
{
    LLIShell* shell = (LLIShell*) handle;
    return &shell->core->rfd;
}

SXMDataService* sms_lli_internal_get_ds (ptr handle)
{
    LLIShell* shell = (LLIShell*) handle;
    return shell->core->ds;
}

/************************************************************************
 *                                                                      *
 *            LLI API                                                   *
 *            =======                                                   *
 *                                                                      *
 ************************************************************************/
int sxm_lli_start (SXMDataService *ds, ptr *handle)
{
    LLIShell* shell = NULL;
    LLICore* core = NULL;
    LLISmsService* smsService;
    int result = SXM_E_ERROR;
    int i, ok, bDeleteMutex = 0;
    BOOLEAN alldmi = FALSE;

    if (!ds)
    {
        ERROR_LLI("(ds=NULL) => SXM_E_FAULT");
        return SXM_E_FAULT;
    }

    do
    {
        uint fpattern[SXM_LLI_DATA_SERVICE_DMI_FILTER_SIZE];

        BITSALL(fpattern);

        /* Create a shell and core */
        shell = (LLIShell*) calloc (1, sizeof(LLIShell));
        core = (LLICore*) calloc (1, sizeof(LLICore));

        if (!core || !shell)
        {
            result = SXM_E_NOMEM;
            ERROR_LLI("Failed to allocate: shell=%x core=%x",
                      shell, core);
            break;
        }

        /* Init the shell */
        shell->waitBit = WAIT_BIT_UNUSED;
        shell->core = core;
        shell->debug_level = ds->debug_level; /* We have to save the debug level with the shell so printouts work when a shell is 'stranded' */
        ok = sxm_mutex_init(&shell->mutex);
        if (ok != EOK)
        {
            ERROR_LLI("Failed to init mutex. result=%d", ok);
            result = SXM_E_PIPE;
            break;
        }
        /* Need to destroy mutex in case of error */
        bDeleteMutex = 1;

        /* Init the core */
        core->ds = ds;

        ENTER_LLI("(ds=%u)", (uint )ds);

        /* Initialize the status of the service */
        SetServiceStatus (ds->status, SXM_SERVICE_STOPPED);
        SetSubsStatus (ds->status, SXM_SUBS_UNKNOWN);

        /* Timer callback is always passed in by the sxm_lli_timer() call.  Non null means at shutdown we
         * will have the SMS DATA object stop the timer.
         */
        ds->timer = 0;

        if (strcmp (ds->service, "sports") == 0)
        {
            core->manageGlobal = 1;
            init_team_league_cache ();
            /* Get the team and league data.  Do it before starting the SMS services.  They can make callbacks
             * immediately.  In normal operation SMS should have it.
             */
            sms_lli_internal_populate_sports_metadata ();
        }

        /* Start the underlying sms objects.  Each dsi gets a generic data service */
        for (i = 0, ok = 1; i < SXM_LLI_DATA_SERVICE_DSI_COUNT; i++)
        {
            smsService = &shell->smsService[i];
            smsService->shell = shell;

            /* dsi of 0, means not configured */
            if (!ds->dsis[i].dsi)
            {
                continue;
            }

            DEBUG_LLI(" SMS:DATA.hStart(); driver=%s dsi=%d",
                      LLI_SRH_DRIVER_NAME, ds->dsis[i].dsi);

            /* We need to accumulate the state of all dsi services into one state.
             * Start our record in the initial state.  Callback from SMS will get us from here.
             */
            smsService->smsState = DATASERVICE_STATE_INITIAL;
            smsService->smsError = DATASERVICE_ERROR_CODE_NONE;

            if (ds->dsis[i].dmi.dmic == 0)
            {
                int eqf = memcmp(ds->dsis[i].filter, fpattern, sizeof(ds->dsis[i].filter));
                if (eqf == 0)
                {
                    /* We are configured in the mode we do not know the dmi for a servcie */
                    alldmi = TRUE;
                    DEBUG_LLI("Full DMI set for DSI %d", ds->dsis[i].dsi);
                }
            }
            if( alldmi == FALSE )
            {
                DEBUG_LLI("Partial DMI set for DSI %d", ds->dsis[i].dsi);
            }

            smsService->smsHandle = DATA.hStart (
                    LLI_SRH_DRIVER_NAME,
                    (DSI) ds->dsis[i].dsi,
                    alldmi,
                    DATASERVICE_EVENT_STATE | DATASERVICE_EVENT_NEW_DATA
                            | DATASERVICE_EVENT_TIMEOUT | DATASERVICE_EVENT_SERVICE,
                    genericDataServiceEventCallback, smsService);

            if (smsService->smsHandle == DATASERVICE_MGR_INVALID_OBJECT)
            {
                ok = 0;
                result = SXM_E_PIPE;
                DEBUG_LLI(" Failed to start sms data service. dsi=%u",
                        (uint) ds->dsis[i].dsi);
                break;
            }
            else
            {
                DEBUG_LLI("Started data service DSI %d, DMI count %d, filter = %s", 
                    ds->dsis[i].dsi, ds->dsis[i].dmi.dmic, 
                    sFilterString(shell, &ds->dsis[i].filter[0]));

                /* If we started at least one service, we do not need to destroy
                 * mutex upon error. It will be utilized during services stopping
                 * and will be destroyed there */
                bDeleteMutex = 0;
            }
        }

        if (!ok)
        {
            break;
        }

        /* Success */
        *handle = (ptr) shell;
        /* We hold off reporting up until we are ready to leave.
         * The sms data service callbacks can fire before we get here.
         */
        shell->reportUp = 1;

        /* Service status is now OK */
        SetServiceStatus (ds->status, SXM_SERVICE_OK);

        LEAVE_LLI("(SXM_E_OK)");

        return SXM_E_OK;
    } while (0);

    /* Set the service state */
    SetServiceStatus (ds->status, SXM_SERVICE_STOPPED);
    SetSubsStatus (ds->status, SXM_SUBS_UNKNOWN);

    if (shell)
    {
        if (bDeleteMutex)
        {
            /* We came out before any SMS service had started.
             * Executing simple cleanup and leaving.
             */
            sxm_mutex_destroy(&shell->mutex);
        }
        else
        {
            /* If we started an SMS data service we must let its
             * shutdown take its course.
             */
            shell->shutDown = 1;
            shell->waitBit = getWaitBit ();

            for (i = 0; i < SXM_LLI_DATA_SERVICE_DSI_COUNT; i++)
            {
                smsService = &shell->smsService[i];
                if (smsService->smsHandle)
                {
                    smsService->shutDown = 1;
                    DATA.vStop (smsService->smsHandle);
                    /* Making service stop synchronous */
                    smsWaitStoppedCallback(shell, smsService);
                    smsService->smsHandle = NULL;
                }
            }
        }

        LEAVE_LLI("(result=%d)", (uint )result);
    }

    if (shell)
    {
        sxe_free (shell);
        shell = NULL;
    }

    if (core)
    {
        free (core);
        core = NULL;
    }

    return result;
}

int sxm_lli_stop (ptr handle)
{
    LLIShell *shell = (LLIShell*)handle;
    LLICore *core;
    int i;

    if (!shell)
    {
        ERROR_LLI("(handle=NULL) => SXM_E_FAULT");
        return SXM_E_FAULT;
    }

    ENTER_LLI("(handle=%u)", (uint )handle);

    if (sxm_lli_on_thread (handle))
    {
        LEAVE_LLI("(SXM_E_BUSY)");
        return SXM_E_BUSY;
    }

    /* Wait for all SMS services to get out of call ups and lock out all future
     * usage of the core.
     */
    LOCK(shell->mutex);

    /* Saving core pointer for future cleanup after shell is destroyed */
    core = shell->core;

    /* Start shutdown procedure */
    shell->shutDown = 1;
    shell->waitBit = getWaitBit ();
    /* Stop possible extra SMS usage managing team and league meta-data.  We are going down and do want to use the data */
    core->manageGlobal = 0;

    UNLOCK(shell->mutex);

    /* Set the state of the cores' shared ds object */
    SetServiceStatus (core->ds->status, SXM_SERVICE_STOPPED);
    SetSubsStatus (core->ds->status, SXM_SUBS_UNKNOWN);
    
    /* Shut down the rfd */
    sxm_lli_rfd_stop(handle);

    /* Stop the timer if one was started */
    if (core->ds->timer && shell->smsService[0].smsHandle)
    {
        DEBUG_LLI("SMS:DATA.vStopTimedEvent() smsHandle=%u",
                  (uint )shell->smsService[0].smsHandle);
        DATA.vStopTimedEvent (shell->smsService[0].smsHandle);
    }

    for (i = 0; i < SXM_LLI_DATA_SERVICE_DSI_COUNT; i++)
    {
        LLISmsService *smsService = &shell->smsService[i];
        if (smsService->smsHandle)
        {
            DEBUG_LLI("SMS:DATA.vStop() smsHandle=%u",
                      (uint )smsService->smsHandle);
            smsService->shutDown = 1;
            DATA.vStop (smsService->smsHandle);
            /* Making service stop synchronous */
            smsWaitStoppedCallback(shell, smsService);
            smsService->smsHandle = NULL;
        }
    }

    DEBUG_LLI("All smsServices stopped. freeing shell");
    LEAVE_LLI("(SXM_E_OK)");

    sxe_free (shell);
    shell = NULL;

    free (core);
    core = NULL;

    return SXM_E_OK;
}

int sxm_lli_remove (ptr handle, int dsi, int dmi)
{
    int rc;
    LLIShell* shell = (LLIShell*) handle;
    if (!shell)
    {
        ERROR_LLI("(handle=NULL) => SXM_E_FAULT");
        return SXM_E_FAULT;
    }
    ENTER_LLI("(handle=%x dsi=%d dmi=%d)", handle, dsi, dmi);
    rc = modifyFilter (shell, dsi, dmi, 0);
    LEAVE_LLI("(rc=%d)", rc);
    return rc;
}

int sxm_lli_request (ptr handle, int dsi, int dmi)
{
    int rc;
    LLIShell* shell = (LLIShell*) handle;
    if (!shell)
    {
        ERROR_LLI("(handle=NULL) => SXM_E_FAULT");
        return SXM_E_FAULT;
    }
    ENTER_LLI("(handle=%x dsi=%d dmi=%d)", handle, dsi, dmi);
    rc = modifyFilter (shell, dsi, dmi, 1);
    LEAVE_LLI("(rc=%d)", rc);
    return rc;
}

int sxm_lli_timer (ptr handle, SXMDataServiceTimerCallback callback, uint mins)
{
    LLISmsService* smsService;
    LLIShell* shell = (LLIShell*) handle;
    SMSAPI_RETURN_CODE_ENUM smsError;

    if (!shell)
    {
        ERROR_LLI("(handle=NULL) => SXM_E_FAULT");
        return SXM_E_FAULT;
    }

    ENTER_LLI("(handle=%u callback=%u mins=%d)", (uint )handle, (uint )callback,
              mins);

    if (shell->shutDown || !callback)
    {
        LEAVE_LLI("(SXM_E_FAULT) shutdown=%u callback=%x", shell->shutDown, callback);
        return SXM_E_FAULT;
    }

    /* We will always use the first entry since we use the structure from 0 -> n */
    smsService = &shell->smsService[0];
    if (!smsService->smsHandle)
    {
        LEAVE_LLI("(SXM_E_FAULT) smsHandle=%x", smsService->smsHandle);
        return SXM_E_FAULT;
    }

    shell->core->ds->timer = callback;
    smsError = DATA.eStartTimedEvent (smsService->smsHandle, TRUE, mins * 60);
    if (smsError == SMSAPI_RETURN_CODE_SUCCESS)
    {
        LEAVE_LLI("(SXM_E_OK)");
        return SXM_E_OK;
    }

    shell->core->ds->timer = 0;

    LEAVE_LLI("(SXM_E_PIPE)");
    return SXM_E_PIPE;
}

/*
 *  Thread
 */

int sxm_lli_on_thread (ptr handle)
{
    /* TODO: Need way in SMS to identify the calling thread is the callback thread */
    pthread_t selfPid = pthread_self();
    pthread_t lliPid = ((LLIShell *)handle)->pid;

#ifdef WIN32
    if (memcmp(&selfPid, &lliPid, sizeof(pthread_t)) == 0)
    {
        return 1;
    }
#else
    if (selfPid == lliPid)
    {
        return 1;
    }
#endif

    return 0;
}

/* Redefining DEBUG_VAR here so all further logs are printed out */
#undef DEBUG_VAR
#define DEBUG_VAR (-1)

/* sxi_sxm_global API for teams and leagues */
static int leaguever = -1;
static int lcount = 0;
static int teamver = -1;
static int tcount = 0;

/* sizes for these are defined in sxm_sdk.h */
static SXMLeague league[MAX_LEAGUE];
static SXMTeam team[MAX_TEAM];
static LEAGUE_MEMBERSHIP_STRUCT teammemb[MAX_TEAM];

static pthread_mutex_t update = PTHREAD_MUTEX_INITIALIZER;

int sxm_sxi_extract_time(SXMTime *ret)
{
    UN32 un32Sec;
    if (OSAL.eTimeGetLocal(&un32Sec) == OSAL_SUCCESS)
    {
        TIME_T tTime = un32Sec;
        struct tm sTime;
        OSAL.gmtime_r(&tTime, &sTime);
        ret->year = (ushort)(sTime.tm_year + 1900);
        ret->mon = (byte)(sTime.tm_mon + 1);
        ret->day = (byte)(sTime.tm_mday);
        ret->hour = (byte)(sTime.tm_hour);
        ret->min = (byte)(sTime.tm_min);
        ret->sec = (byte)(sTime.tm_sec);
        return SXM_E_OK;
    }
    return SXM_E_NOENT;
}

int sxm_sxi_global_league_version (int *ver)
{
    BOOLEAN ok, complete;

    ok = LEAGUE.bGetTableVersion (ver, &complete);
    if (ok == FALSE)
    {
        return SXM_E_NOENT;
    }
    if (complete == FALSE)
    {
        return SXM_E_NOENT;
    }
    return SXM_E_OK;
}

int sxm_sxi_global_league (int s, SXMLeague *out)
{
    int i;
    SXMLeague *t = league;

    for (i = 0; i <= lcount; i++, t++)
        if (t->id == s)
        {
            LOCK(update);
            memcpy (out, t, sizeof(SXMLeague));
            UNLOCK(update);
            return 0;
        }
    return SXM_E_NOENT;
}

int sxm_sxi_global_league_ix (int ix, SXMLeague *out)
{
    if (ix >= 0 && ix < lcount)
    {
        LOCK(update);
        memcpy (out, &league[ix], sizeof(SXMLeague));
        UNLOCK(update);
        return 0;
    }
    return SXM_E_NOENT;
}

int sxm_sxi_global_team_version (int *ver)
{
    BOOLEAN ok, complete;

    ok = TEAM.bGetTableVersion (ver, &complete);
    if (ok == FALSE)
    {
        return SXM_E_NOENT;
    }
    if (complete == FALSE)
    {
        return SXM_E_NOENT;
    }
    return SXM_E_OK;
}

int sxm_is_team_in_league(SXMTeam *t, int lid) {
    int rc = 0;
    int i;

    for (i = 0; i < t->cnt; i++) {
        if (t->sports[i] == lid) {
            rc = 1;
            break;
        }
    }
    return rc;
}

int sxm_sxi_global_team (int tn, int s, SXMTeam *out)
{
    int rc;
    int i;
    SXMTeam *t = team;

    /* Is out a valid pointer */
    if(out == NULL) {
        rc = SXM_E_FAULT;
    /* Is the collection of the sports team table completed ? */
    } else {
        rc = SXM_E_NOENT;
        /* YES - collection of the sports team table is completed */
        /* lets search for a given team */
        for  (i = 0; i <= tcount; i++, t++)

            /* Is there a match in the sports team identifier ? */
            /* Is there a match in the sports league identifier ? */
            if  (t->id == tn && sxm_is_team_in_league(t,s)) {

                /* found the sports team in a given sports league */
                LOCK(update);
                memcpy(out, t, sizeof(SXMTeam));
                UNLOCK(update);

                rc = SXM_E_OK;
                break;
            }
    }

    return rc;
}

int sxm_sxi_global_team_ix (int ix, SXMTeam *out)
{
    if (ix >= 0 && ix < tcount)
    {
        LOCK(update);
        memcpy (out, &team[ix], sizeof(SXMTeam));
        UNLOCK(update);
        return 0;
    }
    return SXM_E_NOENT;
}

int sxm_sxi_global_team_count (int* out)
{
    if (out)
    {
        *out = tcount;
        return SXM_E_OK;
    }
    return SXM_E_FAULT;
}

int sxm_sxi_global_table_begin (void)
{
    return SXM_E_OK;
}

int sxm_sxi_global_table_end (void)
{
    return SXM_E_OK;
}

static BOOLEAN leagueIterator (LEAGUE_OBJECT hLeague, void *arg)
{
    UN8 id;
    STRING_OBJECT hString;

    if (hLeague == NULL)
    {
        return FALSE;
    }
    if (lcount >= MAX_LEAGUE)
    {
        ERROR_LLI("Exceeded max league count. lcount=%d",
                      lcount);
        return FALSE;
    }

    id = LEAGUE.un8Id (hLeague);
    if (id == UN8_MAX)
    {
        ERROR_LLI("Invalid league id(%d)", id);
        return TRUE;
    }

    league[lcount].id = id;
    hString = LEAGUE.hName (hLeague);
    STRING.tCopyToCStr (hString, league[lcount].ln, sizeof(league[lcount].ln));
    hString = LEAGUE.hAbbreviation (hLeague);
    STRING.tCopyToCStr (hString, league[lcount].sn, sizeof(league[lcount].sn));
    lcount++;
    return TRUE;
}

SXE_INLINE static void populateLeagues (void)
{
    BOOLEAN bOk;

    lcount = 0;
    bOk = LEAGUE.bIterateContent (leagueIterator, NULL);
    if (!bOk)
    {
        ERROR_LLI("LEAGUE.bIterateContent failed");
    }
    sxm_sxi_global_league_version (&leaguever);
}

static BOOLEAN teamIterator (TEAM_OBJECT hTeam, void *arg)
{
    UN16 id;
    STRING_OBJECT hString;
    LEAGUE_MEMBERSHIP_STRUCT lm;
    SMSAPI_RETURN_CODE_ENUM rc;
    int i;
    SXMTeam* t;

    if (hTeam == NULL)
    {
        return FALSE;
    }
    if (tcount >= MAX_TEAM)
    {
        ERROR_LLI("Exceeded max team count. tcount=%d", tcount);
        return FALSE;
    }

    id = TEAM.un16Id (hTeam);
    if (id == UN16_MAX)
    {
        ERROR_LLI("Invalid team id(%d)", id);
        return TRUE;
    }

    rc = TEAM.eLeagueMembership (hTeam, &lm);
    if (rc != SMSAPI_RETURN_CODE_SUCCESS)
    {
        ERROR_LLI("TEAM.eLeagueMembership error. rc=%d", rc);
        return TRUE;
    }

    for (i = 0; i < tcount; i++)
    {
        if (team[i].id == id)
        {
            if (memcmp (&teammemb[i], &lm, sizeof(lm)) == 0)
            {
                team[i].sports[team[i].cnt++] = (char)(size_t)arg;
                return TRUE;
            }
        }
    }

    t = &team[i];

    hString = TEAM.hName (hTeam);
    STRING.tCopyToCStr (hString, t->ln, sizeof(t->ln));
    hString = TEAM.hAbbreviation (hTeam);
    STRING.tCopyToCStr (hString, t->sn, sizeof(t->sn));
    hString = TEAM.hNickname (hTeam);
    STRING.tCopyToCStr (hString, t->mn, sizeof(t->mn));
    t->id = id;
    t->sports[t->cnt++] = (char)(size_t)arg;
    if (i == tcount)
    {
        memcpy (&teammemb[i], &lm, sizeof(lm));
    }

    tcount++;
    return TRUE;
}

static BOOLEAN leagueTeamIterator (LEAGUE_OBJECT hLeague, void *arg)
{
    BOOLEAN bOk;
    UN8 id;

    if (hLeague == NULL)
    {
        return FALSE;
    }
    id = LEAGUE.un8Id (hLeague);
    if (id == UN8_MAX)
    {
        ERROR_LLI("Invalid league id(%d)", id);
        return TRUE;
    }

    bOk = TEAM.bIterateContent (hLeague, teamIterator, (void*)(size_t)id);
    if (!bOk)
    {
        ERROR_LLI("TEAM.bIterateContent failed");
    }

    return TRUE;
}

SXE_INLINE static void populateTeams (void)
{
    BOOLEAN bOk;

    tcount = 0;
    bOk = LEAGUE.bIterateContent (leagueTeamIterator, NULL);
    if (!bOk)
    {
        ERROR_LLI("LEAGUE.bIterateContent failed");
    }
    sxm_sxi_global_team_version (&teamver);
}

static void init_team_league_cache (void)
{
    teamver = -1;
    leaguever = -1;
    lcount = 0;
    tcount = 0;
}

static SXMResultCode remap_return_code(SMSAPI_RETURN_CODE_ENUM eReturn)
{
    SXMResultCode ret;

    switch (eReturn)
    {
    case SMSAPI_RETURN_CODE_SUCCESS:
    case SMSAPI_RETURN_CODE_NEW_SERVICE:
    case SMSAPI_RETURN_CODE_LOAD_SERVICE:
        ret = SXM_E_OK;
        break;

    case SMSAPI_RETURN_CODE_OP_NOT_SUPPORTED:
    case SMSAPI_RETURN_CODE_UNSUPPORTED_API:
    case SMSAPI_RETURN_CODE_API_NOT_ALLOWED:
        ret = SXM_E_UNSUPPORTED;
        break;

    case SMSAPI_RETURN_CODE_OUT_OF_RANGE_PARAMETER:
    case SMSAPI_RETURN_CODE_INVALID_INPUT:
    case SMSAPI_RETURN_CODE_INVALID_DECODER:
    case SMSAPI_RETURN_CODE_INVALID_OPTIONS:
    case SMSAPI_RETURN_CODE_CID_NOT_ALLOWED:
    case SMSAPI_RETURN_CODE_INVALID_CID_TYPE:
    case SMSAPI_RETURN_CODE_DUPLICATE_CONTENT:
    case SMSAPI_RETURN_CODE_INVALID_CHANNEL:
    case SMSAPI_RETURN_CODE_LOCKED_CHANNEL:
    case SMSAPI_RETURN_CODE_MATURE_CHANNEL:
    case SMSAPI_RETURN_CODE_CFG_VALUE_TOO_LONG:
    case SMSAPI_RETURN_CODE_CFG_NO_VALUE_SET:
    case SMSAPI_RETURN_CODE_MODULE_FWUPDATE_FILE_VERSION_INVALID:
        ret = SXM_E_INVAL;
        break;

    case SMSAPI_RETURN_CODE_NO_OBJECTS:
    case SMSAPI_RETURN_CODE_NOT_FOUND:
    case SMSAPI_RETURN_CODE_CONTENT_DOES_NOT_EXIST:
    case SMSAPI_RETURN_CODE_EMPTY_PRESET:
        ret = SXM_E_NOENT;
        break;

    case SMSAPI_RETURN_CODE_ALREADY_INITIALIZED:
    case SMSAPI_RETURN_CODE_NOT_INITIALIZED:
    case SMSAPI_RETURN_CODE_STILL_ACTIVE:
    case SMSAPI_RETURN_CODE_SERVICE_ALREADY_STARTED:
    case SMSAPI_RETURN_CODE_SERVICE_NOT_STARTED:
    case SMSAPI_RETURN_CODE_MODULE_FWUPDATE_ALREADY_IN_PROGRESS:
        ret = SXM_E_STATE;
        break;

    case SMSAPI_RETURN_CODE_OUT_OF_MEMORY:
    case SMSAPI_RETURN_CODE_UNABLE_TO_CREATE_CONTENT:
    case SMSAPI_RETURN_CODE_UNABLE_TO_ADD_CONTENT:
    case SMSAPI_RETURN_CODE_WRONG_SEEK_SERVICE:
        ret = SXM_E_NOMEM;
        break;

    case SMSAPI_RETURN_CODE_NOT_OWNER:
    case SMSAPI_RETURN_CODE_PARTIAL_TRANSACTION:
    case SMSAPI_RETURN_CODE_CFG_TAG_REMOVED:
    case SMSAPI_RETURN_CODE_CFG_MALFORMED_FILE:
    case SMSAPI_RETURN_CODE_CFG_FILE_BAD_CHECKSUM:
    case SMSAPI_RETURN_CODE_CFG_BASE64_ERROR:
    case SMSAPI_RETURN_CODE_CFG_NO_PARENT:
    case SMSAPI_RETURN_CODE_ERROR:
        ret = SXM_E_ERROR;
        break;

    case SMSAPI_RETURN_CODE_INSUFFICIENT_SIZE:
        ret = SXM_E_RESOURCE;
        break;

    default:
        ret = SXM_E_ERROR;
        break;
    }

    return ret;
}

void sms_lli_internal_populate_sports_metadata (void)
{
    int tv, lv;

    LOCK(update);
/*
    DEBUG_LLI("lver=%d tver=%d\n", leaguever, teamver);
 */
    sxm_sxi_global_team_version (&tv);
    sxm_sxi_global_league_version (&lv);

    if (lv != leaguever)
    {
        populateLeagues ();
    }
    if (tv != teamver)
    {
        populateTeams ();
    }
    UNLOCK(update);
}

int sxm_gz_extract_team_info(uint sportsTeamBroadcastId, SXMTeam *pTeamInfo)
{
    int rc;

    if (pTeamInfo != NULL) {
        rc = sxm_sxi_global_team(GET_TEAM_ID(sportsTeamBroadcastId), GET_LEAGUE_ID(sportsTeamBroadcastId), pTeamInfo);
    } else {
        rc = SXM_E_INVAL;
    }

    return rc;
}

int sxm_sxi_status (SXMStatus *ret)
{
    if (NULL == ret)
    {
        return -1;
    }
    else
    {
        ret->service = SXM_SERVICE_OK;
        ret->substate = SXM_SUBS_FULL;
        return 0;
    }
}

/*****************************************************************************//**
 * Set DB updated notification flag
 *
 * \param[in] handle a pointer the service's RFD data structure
 *
 ********************************************************************************/
void sxm_lli_sms_rfd_set_db_updated(ptr handle)
{
    LLIShell *shell = (LLIShell*)handle;
    SMSAPI_RETURN_CODE_ENUM eReturn;

    ENTER_API("(handle=%x)", handle);

    if (NULL != handle)
    {
        eReturn = DATA.eSendServiceEvent(shell->smsService[0].smsHandle);
        if (eReturn != SMSAPI_RETURN_CODE_SUCCESS)
        {
            DEBUG_LLI(": Cannot send SMS service event for RFD update (%d)",
                eReturn);
        }
    }
    LEAVE_API("()");
}

/***************************************************************************//**
 * This function modifies DMI list for a given service.
 *
 * First, the desired DSI is found in the list. Then if the DMI
 * has not already been set in the filter, it is set here. If DMI is set
 * already, no effect for given DMI.
 * If DMI has been set before and not present in provided list, it will be
 * disabled.
 *
 * Any DMIs not related to service will be silently ignored.
 *
 * \param[in] handle Pointer to the calling Service Data Structure.
 * \param[in] dsi The DSI which DMI list should be changed
 * \param[in] dmic The number of DMIs in the list
 * \param[in] dmis The list of DMIs to be enabled
 *
 * \retval SXM_E_OK DMI list updated successfully.
 * \retval SXM_E_FAULT NULL parameter.
 * \retval SXM_E_INVAL Parameter is wrong or out of range
 * \retval SXM_E_NOMEM Not enough RAM to complete operation.
 *
 ********************************************************************************/
int sxm_lli_update(ptr handle, int dsi, ushort dmic, ushort *dmis)
{
    LLIShell *shell = (LLIShell*)handle;
    DATASERVICE_MGR_OBJECT hService;
    DATASERVICE_STATE_ENUM eState;
    int rc = SXM_E_OK;
    uint i, cnt;
    SXMDataDsi *dsidata = NULL;
    uint newfilter[SXM_LLI_DATA_SERVICE_DMI_FILTER_SIZE];
    uint deltafilter[SXM_LLI_DATA_SERVICE_DMI_FILTER_SIZE];
    ushort deltaDMIc = 0;

    /* Check inputs */
    if ((handle == NULL) ||
        ((dmis == NULL) && (dmic > 0)))
    {
        return SXM_E_FAULT;
    }

    hService = shell->smsService[0].smsHandle;

    DEBUG_LLI("DMI count is %d", dmic);

    /* Search for DSI in service's structure*/
    for (i = 0;
        i < sizeof(shell->core->ds->dsis) / sizeof(shell->core->ds->dsis[0]);
        ++i) {
        if (shell->core->ds->dsis[i].dsi == dsi)
        {
            DEBUG_LLI("DSI %d is found", dsi);
            dsidata = &shell->core->ds->dsis[i];
            break;
        }
    }

    if (dsidata == NULL)
    {
        /* Cannot find DSI data. Does it belong to the service? */
        return SXM_E_INVAL;
    }

    DEBUG_LLI("[input] DMI filter is %s", sFilterString(shell, &dsidata->filter[0]));

    /* Configuring new DMI filter */
    memset( &newfilter[0], 0, sizeof(newfilter) );
    for (i = 0; i < dmic; ++i)
    {
        int j;

        for (j = 0; j < dsidata->dmi.dmic; ++j)
        {
            if (dsidata->dmi.dmis[j] == dmis[i])
            {
                DEBUG_LLI("Set bit %d in new filter", j);
                BITS(newfilter, j);
                break;
            }
        }
    }

    /* Compare filters, looking for differences only */
    for (i = 0; i < ARRAY_SIZE(deltafilter); ++i)
    {
        deltafilter[i] = newfilter[i] ^ dsidata->filter[i];
    }

    for (i = 0; i < sizeof(deltafilter) * SXM_ARCH_BITS_IN_BYTE; i++)
    {
        if (BITP(deltafilter, i))
        {
            deltaDMIc++;
        }
    }

    if (deltaDMIc > 0)
    {
        DATASERVICE_DMI_CONFIG_STRUCT *psDMIs = NULL;

        /* Create structures array */
        psDMIs = calloc(deltaDMIc, sizeof(DATASERVICE_DMI_CONFIG_STRUCT));
        if (psDMIs == NULL)
        {
            return SXM_E_NOMEM;
        }

        for (i = 0, cnt = 0; i < dsidata->dmi.dmic; i++)
        {
            if (BITP(deltafilter, i))
            {
                if (BITP(newfilter, i))
                {
                    DEBUG_LLI("Setting DMI %d", dsidata->dmi.dmis[i]);
                    psDMIs[cnt].bEnable = TRUE;
                    psDMIs[cnt++].tDMI = dsidata->dmi.dmis[i];
                }
                else
                {
                    DEBUG_LLI("Clearing DMI %d", dsidata->dmi.dmis[i]);
                    psDMIs[cnt].bEnable = FALSE;
                    psDMIs[cnt++].tDMI = dsidata->dmi.dmis[i];
                }
            }
        }

        DEBUG_LLI("[output] DMI filter is %s", 
            sFilterString(shell, &newfilter[0]));

        memcpy(&dsidata->filter[0], newfilter, sizeof(dsidata->filter));

        /* We can only update after the service is in READY state */
        eState = DATA.eState(hService);
        DEBUG_LLI("Dataservice state is %d for DSI %d", eState, dsi);

        if (DATASERVICE_STATE_READY == eState)
        {
            SMSAPI_RETURN_CODE_ENUM eReturnCode;

            /* Update the data stream configuration now */
            eReturnCode = DATA.eManageDataStream(hService, psDMIs, cnt);
            DEBUG_LLI("DATA.eManageDataStream for %u DMIs returned %s", cnt, OSAL.pacGetReturnCodeName(eReturnCode));
            rc = remap_return_code(eReturnCode);
        }
        else
        {
            /* Cannot update filter until we've got subscription, so setting the flag
             * and will apply new filter later on */
            DEBUG_LLI("Delayed filter set for DSI %d", dsi);
            shell->newfilter = 1;
        }

        free(psDMIs);
    }

    return rc;
}

#endif
