/******************************************************************************/
/*                Copyright (c) Sirius XM Satellite Radio, Inc.               */
/*                            All Rights Reserved                             */
/*      Licensed Materials - Property of Sirius XM Satellite Radio, Inc.      */
/******************************************************************************/
/***************************************************************************//**
*
* \file shared_lli.c
* \author(s) Leslie French
* \date 8/20/2013
* \brief Implementation of the common Low-level-interface API
*    for components using the SDK directly
*
* \details Core implementation of RFD.
* 
* The SMS and SDK lli store the structures needed for
* rfd differently.  These functions do all the work.
*
* The individual lli get the common structures then call these functions.
*
* \note his implementation supports 4 parallel collections per service
*
*******************************************************************************/

#define DEBUG_TAG "shared_lli"

#include <sxm_build.h>
#include <sxm_sdk.h>
#include <util/sxm_common_internal.h>
#include <util/lli.h>

/*****************************************************************************//**
 * Initializes a service's RFD data structure.
 * 
 * \param[in,out] rfds the service's RFD data structure to be initialized
 * \param[in] ds    the service's data service structure from which some
 *                  initialization data is read
 *
 * \retval SXM_E_OK         On success
 * \retval SXM_E_NOENT      If RFD failed to start (e.g. RFD slot file issue)
 * \retval SXM_E_ERROR      Failed to start an update 
 * \retval SXM_E_RESOURCE   Collection queue is full
 *
 ********************************************************************************/
int sxm_lli_core_rfd_init(SXMRfdService* rfds, const SXMDataService* ds) {
    int i, j, rc = SXM_E_OK;

    if (sxm_rfd_init() != SXM_E_OK)
    {
        return SXM_E_NOENT;
    }

    strcpy(rfds->service, ds->service);
    rfds->carsize = ds->carsize;
    rfds->collected = ds->collected;

    j = sxm_rfd_find_all(rfds);

    sxm_sem_init(&rfds->semvar, 1);
    rfds->going = 1;

    //  if there is a file marked as complete, start the RFD thread
    //  to (continue to) process it
    for  (i = 0; i < SXM_RFD_MAX_COLLECTION && i < j; i++)
    {
        if  (rfds->state[i] == 1) 
        {
            rc = sxm_sem_wait(&rfds->semvar);
            // lets notify rfd about completed file
            if (rc == SXM_E_OK)
                rc = sxm_rfd_collect(rfds);
            else
                rc = SXM_E_ERROR;
        }
    }

    return rc;
}

/*****************************************************************************//**
 * Opens RFD segment for reading providing file handle
 *
 * \param[in] rfds a pointer the service's RFD data structure
 * \param[in] meta a pointer to RFD file meta data.
 * \param[in] frag RFD file fragment.
 *
 * \return file handle to the RFD file or \c NULL in case of error
 *
 ********************************************************************************/
FILE *sxm_lli_core_rfd_open(SXMRfdService* rfds, SXMRfdMetadata* meta, int frag) {
    char name[32];
    int i;

    for (i = 0; i < SXM_RFD_MAX_COLLECTION; i++)
    {
        if (rfds->fid[i] == meta->fragment[frag].id)
        {
            sprintf(name, "block%02d", rfds->slot[i]);
            return sxm_cfile_open(SXM_RFD_SERVICE_NAME, name);
        }
    }
    return NULL;
}

/*****************************************************************************//**
 * Returns status of service's RFD slot
 * 
 * \param[in]  rfds   The service's RFD data structure
 * \param[in]  id     The service's slot number
 * \param[out] meta   Metadata structure pointer where slot metadata have to be
 *                    copied to. Could be NULL if metadata is not needed.
 *
 * \retval SXM_E_OK     If slot is busy
 * \retval SXM_E_NOENT  If slot is not used
 *
 ********************************************************************************/
int sxm_lli_core_rfd_status(SXMRfdService *rfds, int id, SXMRfdMetadata *meta)
{
    int rc = ((rfds->fid[id]) ? SXM_E_OK : SXM_E_NOENT);

    if ((rc == SXM_E_OK) && meta)
        *meta = *sxm_rfd_extract_meta(rfds->slot[id]);

    return rc;
}

/*****************************************************************************//**
 * Returns permil complete (like percent complete but from 0 to 1000) for an RFD
 * fragment.
 * 
 * Indicates how many of the needed blocks have been received for an RFD fragment. 
 * Keep in mind that there are error cases which allow the permil complete to
 * go backwards. It can even get up to 1000 and then drop to -1 or 0.
 *
 * A -1 value means that the permil complete is unknown. Data blocks may 
 * or may not have been received (e.g. on a previous power cycle) but we don't 
 * know yet how many.
 * 
 * \param[in]  rfds            The service's RFD data structure
 * \param[in]  slotIndex       The service's slot number [0 to SXM_RFD_MAX_COLLECTION-1]
 * \param[out] permilComplete  a value from 0 to 1000 if permil complete is known, 
                               -1 if permil complete is unknown
 *
 * \retval SXM_E_OK     Success, permilComplete now contains a value from -1 to 1000
 * \retval SXM_E_INVAL  If \p slotIndex is not valid
 * \retval SXM_E_FAULT  If \p permilComplete is NULL
 * \retval SXM_E_NOENT  If slot is not in use (empty slot)
 *
 ********************************************************************************/
int sxm_lli_core_rfd_permil_complete(SXMRfdService *rfds, uint slotIndex, int *permilComplete)
{
    
    int rc;
    
    if (slotIndex >= SXM_RFD_MAX_COLLECTION) {
        //slot index exceeds the slot array
        rc = SXM_E_INVAL;
    } else if(permilComplete == NULL) {
        rc = SXM_E_FAULT;
    } else if(rfds->fid[slotIndex] == 0) {
        //fragment ID of 0 is invalid, slot isn't in use
        rc = SXM_E_NOENT;
    } else {
        *permilComplete = sxm_rfd_extract_permil_complete(rfds->slot[slotIndex]);
        rc = SXM_E_OK;
    }

    return rc;
}

/*****************************************************************************//**
 * This function changes service's RFD slot state from busy to not used
 * 
 * \param[in]  rfds   the service's RFD data structure
 * \param[in]  id     the service's slot number
 *
 * \retval SXM_E_OK     on success
 *
 ********************************************************************************/
int sxm_lli_core_rfd_remove(SXMRfdService *rfds, int id)
{
    int rc = sxm_rfd_remove(rfds->slot[id]);

    if (rc == SXM_E_OK)
        rfds->fid[id] = 0;

    return rc;
}

/*****************************************************************************//**
 * Start RFD collection
 *
 * \param[in] rfds a pointer the service's RFD data structure
 * \param[in] ds a pointer to Data Service structure.
 * \param[in] meta a pointer to RFD file meta data.
 *
 * \retval SXM_E_OK Success
 * \retval SXM_E_NOENT if RFD failed to start (e.g. RFD slot file issue)
 * \retval SXM_E_NOMEM if not enough memory to start RFD processing
 *
 ********************************************************************************/
int sxm_lli_core_rfd_start(SXMRfdService* rfds, SXMDataService* ds, SXMRfdMetadata* meta)
{
    int i, f, slot;

    //  check each fragment in the file
    for  (f = 0; f < meta->nfrags; f++) {
        for  (i = 0; i < SXM_RFD_MAX_COLLECTION; i++) 
            if  (rfds->fid[i] == meta->fragment[f].id)
                break;

        if (i != SXM_RFD_MAX_COLLECTION) 
            continue;   //  already collecting this one

        slot = sxm_rfd_find_new(meta, ds->service, f);

        if  (slot == -1) {
            PLOG_UPD("%s: No free RFD slots", ds->service);
            return SXM_E_NOENT;
        }
        else if (slot == -2) {
            PLOG_UPD("%s: Not enough memory to start RFD processing",
                ds->service);
            return SXM_E_NOMEM;
        }
        
        for  (i = 0; i < SXM_RFD_MAX_COLLECTION; i++) 
            if  (rfds->fid[i] == 0) {
                rfds->state[i] = 0;
                rfds->slot[i] = slot;
                rfds->fid[i] = meta->fragment[f].id;
                break;
            }
        
        if (i == SXM_RFD_MAX_COLLECTION) {
            PLOG_UPD("%s: No free collection slots", ds->service);
            sxm_rfd_remove(slot);
            return SXM_E_NOENT;
        }
    }

    return SXM_E_OK;
}

/*****************************************************************************//**
 * This function adds data to an RFD.
 *
 * \param[in] rfds a pointer the service's RFD data structure
 * \param[in] packet Pointer to the data packet to add to the RFD.
 * \param[in] pl Starting place in the packet for the data to add.
 *
 * \retval SXM_E_OK         Sent packet for collection successfully 
 *                              or packet ignored
 * \retval SXM_E_ERROR      Cannot send packet for collection
 * \retval SXM_E_RESOURCE   Collection queue is full
 *   
 ********************************************************************************/
int sxm_lli_core_rfd_add(SXMRfdService *rfds, byte *packet, int pl)
{
    int rc;

    // trying to lock semaphore, if its locked packet is skipped
    rc = sxm_sem_trywait(&rfds->semvar);
    if (rc == SXM_E_OK) {
        memcpy(rfds->packet, packet, pl);
        rfds->pl = pl;
        rc = sxm_rfd_collect(rfds);
    } else if (rc == SXM_E_TIMEOUT)
        rc = SXM_E_OK;
    else
        rc = SXM_E_ERROR;

    return rc;
}

/*****************************************************************************//**
 * Deinitialize RFD service part
 *
 * \param[in] rfds a pointer the service's RFD data structure
 *
 ********************************************************************************/
void sxm_lli_core_rfd_stop(SXMRfdService *rfds)
{
    int rc = SXM_E_ERROR;

    if (rfds->going) {
        sxm_rfd_stop_processing(rfds);
        rc = sxm_sem_wait(&rfds->semvar);
    }

    if (rc == SXM_E_OK) {
        sxm_rfd_service_stop(rfds);
        rfds->going = 0;
        rc = sxm_sem_destroy(&rfds->semvar);
        if (rc != SXM_E_OK) {
            PLOG_UPD("%s: failed to destroy semaphore %d",
                     rfds->service, errno);
        }
    }
}
