/******************************************************************************/
/*                Copyright (c) Sirius XM Satellite Radio, Inc.               */
/*                            All Rights Reserved                             */
/*      Licensed Materials - Property of Sirius XM Satellite Radio, Inc.      */
/******************************************************************************/
/***************************************************************************//**
 * 
 * \file sxm_sports_internal.h
 * \author Leslie French, Sergey Kotov
 * \date 8/20/2013
 *
 * The internal header for Sports data service
 *
 ******************************************************************************/

#ifndef SXM_SPORTS_INTERNAL_H
#define SXM_SPORTS_INTERNAL_H

#include <sxm_sports.h>
#ifndef SDKFILES_STANDALONE_BUILD
#include <sxm_stdtrace.h>
#include <util/sxm_common_internal.h>
#include <util/sxm_noname_internal.h>
#endif

//#define DSPORT_ENABLE_PRINT 1

#ifdef SXM_DEBUG_PRODUCTION
#define DSPORT_PRINT_SPORTS()
#define DSPORT_PRINT_AFFILIATES()
#define DSPORT_PRINT_TABLEDEFS()
#define DSPORT_PRINT_TABLEDEF(_a, _b)
#define DSPORT_PRINT_ALL()
#else
#ifndef DSPORT_ENABLE_PRINT
#define DSPORT_PRINT_SPORTS()
#define DSPORT_PRINT_AFFILIATES()
#define DSPORT_PRINT_TABLEDEFS()
#define DSPORT_PRINT_TABLEDEF(_a, _b)
#define DSPORT_PRINT_ALL()
#else

//#define DSPORT_PRINT_SPORTS() print_sports()
#define DSPORT_PRINT_SPORTS()

//#define DSPORT_PRINT_AFFILIATES() print_affiliates()
#define DSPORT_PRINT_AFFILIATES()

//#define DSPORT_PRINT_TABLEDEFS() print_tabledefs()
#define DSPORT_PRINT_TABLEDEFS()

//#define DSPORT_PRINT_TABLEDEF(_a, _b) print_tabledef(_a, _b)
#define DSPORT_PRINT_TABLEDEF(_a, _b)

//#define DSPORT_PRINT_TABLES() print_tables()
#define DSPORT_PRINT_TABLES()

//#define DSPORT_PRINT_ALL()
#define DSPORT_PRINT_ALL()  print_sports(); 		\
                            print_affiliates(); 	\
                            print_tabledefs(); 		\
                            print_tables();

#endif
#endif

#define SPORTS_SERVICE_NAME "sports"
#define SPORTS_CFILE_NAME   "cycle"
#define SPORTS_CFILE_SCHEMA_VERSION (1)

#define SPORTS_EPOCH_MAKE(_time) (SXM_SXE_TIME_EPOCH0 + (((_time) + 120) / MINUTES_IN_DAY))
#define SPORTS_EPOCH_NOW() SPORTS_EPOCH_MAKE(sxm_sxe_time_now())

#define SPORTS_HAVE_CONFIG() (service->h->g.sections[SPORTS_SECTION_SPORTS].version != INVALID_VERSION)
#define SPORTS_HAVE_AFTABLE()(service->h->g.sections[SPORTS_SECTION_AFFS].version != INVALID_VERSION)

//
// All cycle file root block user values.
//

// none used yet

// Bits to manage extraction types.
#define EXTRACT_NESTED      (1u<<31)
#define EXTRACT_REQUIRE_REQ (0 | SXM_SPORTS_TREE | SXM_SPORTS_TABLE | SXM_SPORTS_TEAMS | SXM_SPORTS_TEAM_EVENTS | SXM_SPORTS_TDEF)

// sport, affiliate, and tabledef versions are 8 bit values.  id's are no bigger than 10 bits.  We will use this 16 bit value to indicate
// we do not have a version or id.
#define INVALID_VERSION (0xffff)
#define INVALID_ID      (0xffff)
// global refs are 7 bits.  We will use this 8 bit value to indicate we do not have one.
#define INVALID_GDREF   (0xff)
// used to identify stranded affiliates when parsing the affiliate description table.
#define INVALID_PARENT (0xfffe)
#define INVALID_HEAPIX (0xffff)

#define SEASON_STATUS_UNKNOWN (-1)

//Maximum reporting rows
#define MAX_TEAM_REPORT         255

// We use a mapping table to map ids based on bit stream protocol to our lesser counts.
#define MAX_SPORT_ID            255     // 8 bits
#define MAX_AFFILIATE_ID        1023    // 10 bits
#define MAX_TABLE_ID            1023    // 10 bits

// Counts are the number we support.
#define MAX_EXTRACTION_COUNT		32
#define MAX_SPORT_COUNT             32
#define MAX_AFFILIATE_COUNT         256
#define MAX_TABLE_COUNT             128
#define MAX_TABLES_PER_SPORT        16
#define MAX_AU_COUNT             	24
#define MAX_AU_ASSEMBLIES           4
#define MAX_CAROUSEL_DATA_SIZE		(15*1024)
#define MAX_HEAP_BLOCK_COUNT        (2048)
#define MAX_SPORTS_AU_SIZE          (5120)

// Count in symbols for various baudot strings in the protocol.
#define MAX_SPORT_NAME_CHAR         32
#define MAX_AFFILIATE_NAME_CHAR     64
#define MAX_TABLE_TITLE_CHAR        32
#define MAX_TEAM_NAME_CHAR			48
#define MAX_PARSE_CHAR				4096

#define MAX_SCHEDULE_INDEX 16
#define MAX_OTHERS_INDEX 23

// Can never have more than 4 affiliate levels in the hierarchy
#define MAX_AFFILIATE_NAME_LEVEL 3
#define MAX_LEAGUE_TREE_ENTRIES  64

#define MAX_TABLE_LABEL_COUNT 64
#define MAX_LABEL_TEXT_COUNT 32

// Class of Tables Supported
#define TCLASS_SUPPORTED (0 \
                            | (1<<TCLASS_H2H_EVENT) \
                            | (1<<TCLASS_RL_EVENT) \
                            | (1<<TCLASS_RL_SEASON_SUMMARY) \
                            | (1<<TCLASS_RL_NEWS) \
                            | (1<<TCLASS_RL_PARTICIPANT_STAT) \
                            | (1<<TCLASS_RL_PARTICIPANT_INDEX) \
                            | (1<<TCLASS_EVENT_DESC))

// Range of sport id that are fixed and published.
#define SPID_KNOWN_MIN SPID_FOOTBALL
#define SPID_KNOWN_MAX SPID_TENNIS
#define TIDX_KNOWN_MAX 17
#define TIDX_KNOWN_NEWS TIDX_KNOWN_MAX

#define NOTIFY_SCHEDULE 	(1<<0)
#define NOTIFY_OTHER     	(1<<1)


// Labels that are used for the fixed API.
#define LB_HOME_SPID 			0x0
#define LB_AWAY_SPID 			0x1
#define LB_EVENT_STATE			0x2
#define LB_EVENT_DIVISION		0x3
#define LB_EVENT_CLOCK 			0x4
#define LB_EVENT_START 			0x5
#define LB_HOME_SCORE_INT 		0x6
#define LB_HOME_SCORE_TXT 		0x9
#define LB_HOME_INFO_UINT 		0xa
#define LB_HOME_INFO_TXT 		0xd
#define LB_AWAY_SCORE_INT 		0xe
#define LB_AWAY_INFO_UINT       0x12
#define LB_HOME_CHAN	 		0x16
#define LB_AWAY_CHAN 			0x17
#define LB_NATIONAL_CHAN		0x18
#define LB_EXTRA_TEXT			0x19
#define LB_EXPLICIT_RANK		0x1a
#define LB_TEXT_BODY			0x1e
#define LB_STAT_UINT			0x1f
#define LB_RESULT				0x24
#define LB_TREF					0x25
#define LB_MAX					(LB_TREF + 1)

// requires special parsing like home and away
#define LB_P_PARTITION			0x1c

#define LB_DTYPE_INT				0x00
#define LB_DTYPE_TXT				0x01

#define TREF_IS_KNOWN			(1u<<31)

#define KEY_TYPE_INT 0
#define KEY_TYPE_TXT 1

#define ROWMAP_TEAMID_TXT_ONLY 0
#define ROWMAP_TEAMID_FULL     1

#define ROWMAP_TEAMID_SHORT    0
#define ROWMAP_TEAMID_MEDIUM   1
#define ROWMAP_TEAMID_LONG     2

#define GLOBAL_VERSION_STARTUP (-2)

#define SPORTS_TIMER_MINUTES (1)

#define SMALL_SYMCOUNT(_s) ((_s)*3)
#define BIG_SYMCOUNT(_b) ((_b)*2)

typedef struct {
    byte priority;
    byte lbtype;
    byte key;
    byte type;
    char* txt;
}TRowMapEntry;

typedef struct {
    TRowMapEntry col[SXM_SPORTS_KEY_MAX_COUNT];
    byte count;
    byte teamIdType;
    byte teamKeyBreakout[2][3];
}TRowMap;

/** LABEL DEFINITIONS */
typedef struct {
    byte lbid;
    byte dtype;
    byte lbtype;
    byte dmod;
    byte prio;
    byte havedmod;
    byte havetext;
    ushort lbsize;
    char text[MAX_LABEL_TEXT_COUNT];
}TCol;

typedef struct {
    ushort offset;
    ushort delta;
}SportsTrefmapEntry;

/** C-File section - TABLE DEFINITIONS */
typedef struct {
    ushort tclass;  // table class
    ushort ver;     // table version
    ushort lc;      // label count
    TCol l[MAX_TABLE_LABEL_COUNT];
}STable;

typedef struct {
    STable table[MAX_TABLE_COUNT];
    ushort map[MAX_TABLE_ID+1];
}STableSection;

/** C-File section - AFFILIATION */
typedef struct {
    ushort tabno;
    ushort tabver;
    ushort heapix;  // heap index
    ushort size;    // number of bytes
    ushort epoch;
    ushort season;
    ushort offset;
    ushort inuse;
    uint timeStamp;
    uint crc;
    SportsTrefmapEntry trefmap[MAX_AU_COUNT];
}Table;

typedef struct {
    char name[MAX_AFFILIATE_NAME_CHAR];
    ushort global;
    ushort afid;        /* affiliation ID */
    ushort spid;        /* sport ID */
    ushort ver;
    ushort parentid;
    ushort level;
    ushort rootid;
    Table schedule[MAX_SCHEDULE_INDEX+1];   /* schedule tables */
    Table others[MAX_OTHERS_INDEX+1];       /* others tables */
}Affiliation;

typedef struct {
    Affiliation aff[MAX_AFFILIATE_COUNT];
    ushort map[MAX_AFFILIATE_ID+1];
}AffiliationSection;

/** C-File section - SPORTS */
typedef struct {
    char name[MAX_SPORT_NAME_CHAR];
    ushort tables[MAX_TABLES_PER_SPORT];
    uint tno;
}Sport;

typedef struct {
    Sport sport[MAX_SPORT_COUNT];
    ushort map[MAX_SPORT_ID+1];
}SportSection;

/** C-File section - HEAP */
typedef struct {
    SXMSector heap[MAX_HEAP_BLOCK_COUNT];
}HeapSection;

//
//  definition of the sports cycle file
//

#define SPORT_BLOCKS		(sizeof(SportSave) >> 9)
#define ROOT                (&service->h->g)

/** C-File sections */
enum {
    SPORTS_SECTION_FIRST = 0,
    SPORTS_SECTION_SPORTS = SPORTS_SECTION_FIRST,
    SPORTS_SECTION_AFFS,
    SPORTS_SECTION_TABLES,
    SPORTS_SECTION_HEAP,
    SPORTS_SECTIONS_COUNT,
    SPORTS_SECTIONS_ALL = ((1 << SPORTS_SECTIONS_COUNT) - 1)
};

/** C-File structure */
typedef struct {
    SXMCFileRootBlock g;
    CSECT(SportSection, sports, 1) p0;
    CSECT(AffiliationSection, affs, 1) p1;
    CSECT(STableSection, tables, 1) p2;
    CSECT(HeapSection, heaps, 1) p3;
}SportSave;

// convenience macros for accessing table maps
#define SBLOCK      service->h->p0.sports[0].map
#define ABLOCK      service->h->p1.affs[0].map
#define TBLOCK      service->h->p2.tables[0].map

// convenience macros for accessing tables
#define SPORT       service->h->p0.sports[0].sport
#define AFF         service->h->p1.affs[0].aff
#define TABLE       service->h->p2.tables[0].table
#define HEAP        service->h->p3.heaps[0].heap

/* Following definitions are used in the internal implementation only */
#ifndef SDKFILES_BUILDER

//
// Data consumption
//

/** Carousel IDs */
enum {
    SPORTS_CAROUSEL_FIRST = 0,
    SPORTS_CAROUSEL_SPORTS = SPORTS_CAROUSEL_FIRST,
    SPORTS_CAROUSEL_AFFS,
    SPORTS_CAROUSEL_TABLES,
    SPORTS_CAROUSEL_DATA0,
    SPORTS_CAROUSEL_DATA1,
    SPORTS_CAROUSEL_RES0,
    SPORTS_CAROUSEL_RES1,
    SPORTS_CAROUSEL_COUNT,
    SPORTS_CAROUSEL_ALL = ((1 << SPORTS_CAROUSEL_COUNT) - 1)
};

typedef struct {
    char* league;
    char* rootAffiliate;
    ushort sportId;
}LeagueMapEntry;

/** table of well known affliations (leagues) */
static const LeagueMapEntry SportsLeagueMap[] = {
        {"mlb",				"MLB", 			SPID_BASEBALL},
        {"cbk",				"NCAA", 		SPID_BASKETBALL},
        {"wcbk",			"NCAA(Womens)", SPID_BASKETBALL},
        {"nba",				"NBA", 			SPID_BASKETBALL},
        {"wnba",			"WNBA", 		SPID_BASKETBALL},
        {"cfl",				"CFL", 			SPID_FOOTBALL},
        {"cfb",				"NCAA", 		SPID_FOOTBALL},
        {"nfl",				"NFL", 			SPID_FOOTBALL},
        {"nhl",				"NHL", 			SPID_ICEHOCKEY},
        {"indy",			"Indy", 		SPID_MOTORSPORT},
        {"nascar",			"NASCAR", 		SPID_MOTORSPORT},
        {"lpga",			"LPGA", 		SPID_GOLF},
        {"pga",				"PGA", 			SPID_GOLF},
        {"mls",				"MLS", 			SPID_SOCCER},
};

typedef struct {
    char* sport;
    ushort id;
}SportMapEntry;

/** table of well known Sports */
static const SportMapEntry SportsSportMap[] = {
        {"Multi", 		SPID_MULTI},
        {"Football", 	SPID_FOOTBALL},
        {"Baseball", 	SPID_BASEBALL},
        {"Basketball", 	SPID_BASKETBALL},
        {"Ice Hockey", 	SPID_ICEHOCKEY},
        {"Motorsport", 	SPID_MOTORSPORT},
        {"Golf", 		SPID_GOLF},
        {"Tennis", 		SPID_TENNIS},
};

#define TDEF_MAP_INVALID 0xff

typedef struct sports_extract_struct SportsExtract;
typedef struct sports_service_struct SportsService;
typedef struct sports_request_struct SportsRequest;

/** used during row data extraction */
typedef struct {
    SXMSportsRow* row;
    int tidx;
    int err;
} SportsRowHelper;

/** used during raw data extraction */
typedef struct {
    SXMSportsRaw* raw;
    uint tidx;
    int err;
} SportsRawHelper;

/** used to assemble Multi-AU Carousels */
typedef struct {
    byte* au[MAX_AU_COUNT];
    byte aubuf[MAX_CAROUSEL_DATA_SIZE];
    ushort auoff[MAX_AU_COUNT];
    ushort ausize[MAX_AU_COUNT];
    byte* auend;
    ushort auaid;
    ushort autot;
    ushort aucur;
    ushort startOff;
    SportsTrefmapEntry trefmap[MAX_AU_COUNT];

    byte* pktbuf;
    ushort pktlen;
}SportsAUAssemble;

typedef struct {
    ushort tidx;
    ushort homeTeamId;
    ushort visitTeamId;
    int startTime;
} TeamReport;

/** Sports Extraction structure */
struct sports_extract_struct {
    SportsExtract* link;
    SportsExtract* nested;
    SportsExtract* parent;
    SportsRequest* req;
    uint type;
    SXMBitBuff	bbuff;

    /* league tree extract. */
    short season;
    ushort err;				/* User must end extraction.  If it errors we set state and exit at a high level. */
    ushort lTreeCount;
    ushort lTree[MAX_LEAGUE_TREE_ENTRIES];

    /* team event extract specific fields. */
    SportsExtract* curExt;
    Affiliation* curAff;
    ushort curAfid;
    ushort curTidx;
    ushort rootAfid;
    ushort teamId;
    ushort isTeamSearch;
    ushort teamReportCnt;
    TeamReport TeamReport[MAX_TEAM_REPORT];

    Table* t;
    Table nestedTable;
    const TRowMap* knownMap;
     /* Map of tdef column to a known sport Row_Map_xxx structure. */
    byte tdefKnownMap[MAX_TABLE_LABEL_COUNT];
    /* lbsize for column. either from tdef or overridden with table instance. */
    ushort lbsize[MAX_TABLE_LABEL_COUNT];
    ushort tidx;
    ushort afid;
    ushort global;
    ushort entryno;
    ushort isRow;
    ushort epoch;
    ushort rowIdx;
    char buff[MAX_PARSE_CHAR];
    ptr dest;
    int (*parse)(SportsService*, SportsExtract*);
};

/** Sports Request structure */
struct sports_request_struct {
    char afname[MAX_AFFILIATE_NAME_CHAR];
    ushort afid;        /* affilation ID */
    ushort sportId;     /* sport ID */
    BOOL bsc;           /* broad-scope-change */
    BOOL addNotify;     /* new request notify */
    BOOL inuse;         /* call-back is active */
    /* Only clear when provisioning the request. */
    BOOL rdelete;
    BOOL deleteAtEnd; /* remove in call-back context */
    pthread_mutex_t mutex;  /* request lock */
    SportsExtract* ext; /* linked list of extractions */
    void (*callback)(ptr, int, int, ptr);  /* notification */
    ptr usercx;     /* user context */
} ;

#define SXM_SPORTS_NUM_REQUESTS_MAX (32)

/** Sports Service structure */
struct sports_service_struct{
    SXMDataService ds;          //!< service struct for the lli
    pthread_mutex_t mutex;      //!< service mutex
    void (*callback)(int, int); //!< service level callback
    ptr lli;                    //!< lli reference
    FILE *cycle;                //!< cycle file handle
    byte sections_update;       //!< cycle file save status
    SportSave *h;               //!< cycle file memory image
    SportsRequest request[SXM_SPORTS_NUM_REQUESTS_MAX]; //!< requests
    uint lastEpoch;             //!< epoch
    BOOL bsc;                   //!< broad scope change flag
    uint notify;                //!< notify flag
    int globalLeagueVer;        //!< global league table version
    int globalTeamVer;          //!< global team table version
    SportsExtract* freeExtRoot; //!< free extraction buffer pointer
    SportsExtract extBuffer[MAX_EXTRACTION_COUNT]; //!< the extraction buffers
    SportsAUAssemble aus[MAX_AU_ASSEMBLIES]; //!< the AU assemby bufffers
};

extern void sxm_sports_complete_au(byte *, int, ptr);

extern void sxm_sports_timer(void *);

extern int sxm_sports_protocol_start(SportsService *);

extern void sxm_sports_protocol_stop(void);

extern int sxm_sports_commit(SXMIOStat *);

//
// requests
//

extern int sxm_sports_request_add(SportsService* service,
                                  const char* pLeague,
                                  SPORTS_REQUEST_CALLBACK pNotification,
                                  ptr pUsercx,
                                  ptr* ppRequestHandle);

extern int sxm_sports_request_remove(SportsService*, ptr);

//
// extract
//

extern void sxm_sports_extract_init(SportsService *); // ims

extern void sxm_sports_extract_uninit(void);

extern int sxm_sports_extract_begin(	SportsService*,	//service
                                        ptr, 			//req
                                        int, 			//type
                                        int, 			//p1
                                        ptr*, 			//ext
                                        SXMSportsRow* 	//header
                                        );

extern int sxm_sports_extract_begin_nested(	SportsService*,	//service,
                                            ptr, 			//parentExt
                                            int,			//type
                                            int, 			//p1
                                            ptr*,			//ext
                                            SXMSportsRow* 	//header
                                            );

extern int sxm_sports_extract_extract_rxw(	SportsService*, //service
                                            ptr, 			// ext
                                            ptr, 			// out
                                            int 			// isRow
                                            );


extern int sxm_sports_extract_extract_end(	SportsService*,	//service
                                            ptr 			//ext
                                            );

//
//  LLI
//

extern int sxm_sports_lli_start(SportsService *);

extern int sxm_sports_lli_stop(void);

//
// Shared
//

extern Affiliation* sxm_sports_shared_find_root(SportsRequest*);

//
// Debug
//
#ifdef DSPORT_ENABLE_PRINT
extern void print_sports(void);
extern void print_affiliates(void);
extern void print_tabledefs(void);
extern void print_tables(void);
#endif

#endif /* #ifndef SDKFILES_STANDALONE_BUILD */

#endif /* #ifndef SXM_SPORTS_INTERNAL_H */