/******************************************************************************/
/*                Copyright (c) Sirius XM Satellite Radio, Inc.               */
/*                            All Rights Reserved                             */
/*      Licensed Materials - Property of Sirius XM Satellite Radio, Inc.      */
/******************************************************************************/
/***************************************************************************//**
 * 
 * \file sxm_sports_extract.c
 * \author Leslie French, Sergey Kotov
 * \date 8/20/2013
 *
 * Implementation of the extraction management for Sports
 *
 ******************************************************************************/

#include "sxm_build.h"

#define DEBUG_TAG "sports"

#if (SXM_USE_GEN8)
#include "sxm_gmd.h"
#endif

#include "sxm_sports_internal.h"

/************************************************************************
 *                                                                      *
 *            Private prototypes                                        *
 *            ======================                                    *
 *                                                                      *
 ************************************************************************/
static void row_init(SportsRowHelper *h, SXMSportsRow *row);

static void row_put_int(SportsRowHelper *h, int i, uint key);

static void row_put_str(SportsRowHelper *h, char *txt, int key);

static void raw_init(SportsRawHelper *h, SXMSportsRaw *raw,
    int tableref, int tableepoch);

static void raw_put_int(SportsRawHelper *h, int i, int lbid);

static void raw_put_str(SportsRawHelper *h, char *txt, int lbid);

static void raw_put_age(SportsRawHelper *h, uint ts);

static int write_sport(SportsService *service, SXMSportsRow *row,
    int id, char *name);

static int write_affilate(SportsService *service, SXMSportsRow *row,
    Affiliation *a, int season);

static int write_tdef(SportsService *service, SXMSportsRow *row, TCol *c);

#if(!SXM_USE_GEN8 || SXM_USE_GMD)
static int write_team(SportsService *service, SXMSportsRow *row, SXMTeam *tm);
#endif

static Affiliation *find_affiliate(SportsService *service, int rootid, int afid);

static int is_know_table(ushort tidx, ushort spid);

static int get_sport(SportsService *service, SportsExtract *e);

static int get_season_status(SportsService *service, Affiliation *a);

static int get_league(SportsService *service, SportsExtract *e);

static int get_league_tree(SportsService *service, SportsExtract *e);

static int skip_column(SportsService *service, SXMBitBuff *pkt, TCol *c,
    char *buff, uint lbsize);

static int get_row(SportsService *service, SportsExtract *e);

static int get_raw(SportsService *service, SportsExtract *e);

static int get_tdef(SportsService *service, SportsExtract *pXtract);

static Affiliation *get_next_aff(SportsService *service, SportsExtract *e);

static SportsExtract *get_next_table(SportsService *service, SportsExtract *e);

static int get_team_events(SportsService *service, SportsExtract *e);

static int get_team_list(SportsService *service, SportsExtract *e);

static int parse_row(SportsService *service, SportsExtract *e);

static BOOL request_handle_valid(SportsService *service, ptr req);

static BOOL extract_handle_valid(SportsService *service, ptr ext);

static BOOL request_has_affiliate(SportsService *service, ptr req);

static int begin_sports(SportsService *service, uint type, ptr *ext);

static int begin_leagues(SportsService *service, uint type, ptr *ext);

static void make_ltree(SportsService *service, ushort parent, ushort level,
    ushort *ltree, ushort *cnt);

static int begin_league_tree(SportsService *service, SportsRequest *req,
    uint type, ptr *ext);

static void make_know_table_map(STable *tdef, ushort spid, uint tidx,
    SportsExtract *e, uint isKnownTableReferenced);

static int begin_table(SportsService *service, SportsRequest *req, uint type,
    int p1, ptr *ext, SXMSportsRow *header);

static int begin_tdef(SportsService *service, SportsRequest *req, uint type,
    int p1, ptr *ext);

static int begin_nested_table(SportsService *service, SportsExtract *parentExt,
    uint type, int p1, ptr *ext, SXMSportsRow *header);

static int begin_nested_tdef(SportsService *service, SportsExtract *parentExt,
    uint type, int p1, ptr *ext);

static int begin_teams_events(SportsService *service, SportsRequest *req,
    uint type, int p1, ptr *ext);

static int begin_teams(SportsService *service, SportsRequest *req, uint type,
    int p1, ptr *ext);

static int locked_sports_extract_begin(SportsService *service, ptr req,
    int type, int p1, ptr* ext, SXMSportsRow* header);

static int locked_sports_extract_extract_rxw(SportsService *service,
    SportsExtract* e, ptr out, int isRow);

static int free_extract(SportsService *service, SportsExtract **r,
    SportsExtract *e);

static int locked_sports_extract_extract_end(SportsService *service,
    SportsExtract* e);


static const byte Label_Sign_Extend[LB_MAX] =
{
    /*0x00*/ 	0,0,0,0,0,0,0,1,  0,0,0,1,0,0,0,1,
    /*0x10*/ 	0,0,0,1,0,0,0,0,  0,0,0,0,0,0,0,0,
    /*0x20*/ 	1,0,0,0,0,0
};

/* Applies to football, basketball, hockey, soccer */
static const TRowMap RowMap_H2H_Sched =
{
    {
        {0, LB_HOME_SPID, 		SXM_SPORTS_H2H_HOME_ID,		KEY_TYPE_INT, NULL},
        {0, LB_AWAY_SPID, 		SXM_SPORTS_H2H_VISIT_ID,	KEY_TYPE_INT, NULL},
        {0, LB_EVENT_STATE, 	SXM_SPORTS_H2H_STATE,		KEY_TYPE_INT, NULL},
        {0, LB_EVENT_DIVISION, 	SXM_SPORTS_H2H_DIVISION,	KEY_TYPE_TXT, NULL},
        {0, LB_EVENT_CLOCK, 	SXM_SPORTS_H2H_CLOCK,		KEY_TYPE_INT, NULL},

        {0, LB_EVENT_START,		SXM_SPORTS_H2H_START_TIME,	KEY_TYPE_INT, NULL},
        {0, LB_HOME_SCORE_INT,  SXM_SPORTS_H2H_HOME_SCORE,	KEY_TYPE_INT, NULL},
        {0, LB_AWAY_SCORE_INT,  SXM_SPORTS_H2H_VISIT_SCORE, KEY_TYPE_INT, NULL},
        {0, LB_RESULT, 			SXM_SPORTS_H2H_RESULT,		KEY_TYPE_INT, NULL},
        {0, LB_HOME_CHAN, 		SXM_SPORTS_H2H_HOME_CHAN,	KEY_TYPE_INT, NULL},

        {0, LB_AWAY_CHAN, 		SXM_SPORTS_H2H_VISIT_CHAN,	KEY_TYPE_INT, NULL},
        {0, LB_NATIONAL_CHAN, 	SXM_SPORTS_H2H_NATION_CHAN,	KEY_TYPE_INT, NULL},

        {0, 0, 0, 0, NULL},
        {0, 0, 0, 0, NULL},
        {0, 0, 0, 0, NULL},

        {0, 0, 0, 0, NULL},
        {0, 0, 0, 0, NULL},
        {0, 0, 0, 0, NULL},
    },
    12,
    ROWMAP_TEAMID_FULL,
    {
        {SXM_SPORTS_H2H_HOME_ABBR, SXM_SPORTS_H2H_HOME_NICK, SXM_SPORTS_H2H_HOME_FULL},
        {SXM_SPORTS_H2H_VISIT_ABBR, SXM_SPORTS_H2H_VISIT_NICK, SXM_SPORTS_H2H_VISIT_FULL}
    }
};

/* Applies to baseball(same as all other h2h except no clock). */
static const TRowMap RowMap_Baseball_Sched =
{
    {
        {0, LB_HOME_SPID, 		SXM_SPORTS_H2H_HOME_ID,		KEY_TYPE_INT, NULL},
        {0, LB_AWAY_SPID, 		SXM_SPORTS_H2H_VISIT_ID,	KEY_TYPE_INT, NULL},
        {0, LB_EVENT_STATE, 	SXM_SPORTS_H2H_STATE,		KEY_TYPE_INT, NULL},
        {0, LB_EVENT_DIVISION, 	SXM_SPORTS_H2H_DIVISION,	KEY_TYPE_TXT, NULL},
        {0, LB_EVENT_START,		SXM_SPORTS_H2H_START_TIME,	KEY_TYPE_INT, NULL},

        {0, LB_HOME_SCORE_INT,  SXM_SPORTS_H2H_HOME_SCORE,	KEY_TYPE_INT, NULL},
        {0, LB_AWAY_SCORE_INT,  SXM_SPORTS_H2H_VISIT_SCORE, KEY_TYPE_INT, NULL},
        {0, LB_RESULT, 			SXM_SPORTS_H2H_RESULT,		KEY_TYPE_INT, NULL},
        {0, LB_HOME_CHAN, 		SXM_SPORTS_H2H_HOME_CHAN,	KEY_TYPE_INT, NULL},
        {0, LB_AWAY_CHAN, 		SXM_SPORTS_H2H_VISIT_CHAN,	KEY_TYPE_INT, NULL},

        {0, LB_NATIONAL_CHAN, 	SXM_SPORTS_H2H_NATION_CHAN,	KEY_TYPE_INT, NULL},
        {0, 0, 0, 0, NULL},
        {0, 0, 0, 0, NULL},
        {0, 0, 0, 0, NULL},
        {0, 0, 0, 0, NULL},

        {0, 0, 0, 0, NULL},
        {0, 0, 0, 0, NULL},
        {0, 0, 0, 0, NULL},
    },
    11,
    ROWMAP_TEAMID_FULL,
    {
        {SXM_SPORTS_H2H_HOME_ABBR, SXM_SPORTS_H2H_HOME_NICK, SXM_SPORTS_H2H_HOME_FULL},
        {SXM_SPORTS_H2H_VISIT_ABBR, SXM_SPORTS_H2H_VISIT_NICK, SXM_SPORTS_H2H_VISIT_FULL}
    }
};

/* News table entries. */
static const TRowMap TRowMap_News =
{
    {
        {0, LB_TEXT_BODY,		SXM_SPORTS_NEWS_ITEM,	KEY_TYPE_TXT, NULL},
        {0, 0, 0, 0, NULL},
        {0, 0, 0, 0, NULL},
        {0, 0, 0, 0, NULL},
        {0, 0, 0, 0, NULL},

        {0, 0, 0, 0, NULL},
        {0, 0, 0, 0, NULL},
        {0, 0, 0, 0, NULL},
        {0, 0, 0, 0, NULL},
        {0, 0, 0, 0, NULL},

        {0, 0, 0, 0, NULL},
        {0, 0, 0, 0, NULL},
        {0, 0, 0, 0, NULL},
        {0, 0, 0, 0, NULL},
        {0, 0, 0, 0, NULL},

        {0, 0, 0, 0, NULL},
        {0, 0, 0, 0, NULL},
        {0, 0, 0, 0, NULL},
    },
    1,
    ROWMAP_TEAMID_TXT_ONLY,
    {
        {0, 0, 0},
        {0, 0, 0}
    }
};

/* Applies to golf. */
static const TRowMap RowMap_Golf_Sched =
{
    {
        {0, LB_EXTRA_TEXT,		SXM_SPORTS_GOLF_TOURNEY,	KEY_TYPE_TXT, "Tourney"},
        {0, LB_EXTRA_TEXT,		SXM_SPORTS_GOLF_COURSE,		KEY_TYPE_TXT, "Course"},
        {0, LB_EVENT_STATE,		SXM_SPORTS_GOLF_STATE,		KEY_TYPE_INT, NULL},
        {0, LB_EVENT_START,		SXM_SPORTS_GOLF_START_TIME,	KEY_TYPE_INT, NULL},
        {0, LB_STAT_UINT,		SXM_SPORTS_GOLF_YARDS,		KEY_TYPE_INT, "Yds"},
        {0, LB_STAT_UINT,		SXM_SPORTS_GOLF_PURSE,		KEY_TYPE_INT, "Purse"},
        {0, LB_NATIONAL_CHAN, 	SXM_SPORTS_GOLF_NATION_CHAN,KEY_TYPE_INT, NULL},

        {0, LB_TREF,			SXM_SPORTS_GOLF_RANK,		KEY_TYPE_INT, NULL},
        {0, 0, 0, 0, NULL},
        {0, 0, 0, 0, NULL},

        {0, 0, 0, 0, NULL},
        {0, 0, 0, 0, NULL},
        {0, 0, 0, 0, NULL},
        {0, 0, 0, 0, NULL},
        {0, 0, 0, 0, NULL},

        {0, 0, 0, 0, NULL},
        {0, 0, 0, 0, NULL},
        {0, 0, 0, 0, NULL},
    },
    8,
    ROWMAP_TEAMID_TXT_ONLY,
    {
        {0, 0, 0},
        {0, 0, 0}
    }
};

static const TRowMap RowMap_Golf_Rank =
{
    {
        {0, LB_HOME_SPID,		SXM_SPORTS_GOLF_RANK_PLAYER,	KEY_TYPE_INT, NULL},
        {0, LB_HOME_SCORE_TXT,	SXM_SPORTS_GOLF_RANK_SCORE,		KEY_TYPE_TXT, NULL},
        {0, LB_EXPLICIT_RANK,	SXM_SPORTS_GOLF_RANK_RANK,		KEY_TYPE_TXT, NULL},
        {0, 0, 0, 0, NULL},
        {0, 0, 0, 0, NULL},

        {0, 0, 0, 0, NULL},
        {0, 0, 0, 0, NULL},
        {0, 0, 0, 0, NULL},
        {0, 0, 0, 0, NULL},
        {0, 0, 0, 0, NULL},

        {0, 0, 0, 0, NULL},
        {0, 0, 0, 0, NULL},
        {0, 0, 0, 0, NULL},
        {0, 0, 0, 0, NULL},
        {0, 0, 0, 0, NULL},

        {0, 0, 0, 0, NULL},
        {0, 0, 0, 0, NULL},
        {0, 0, 0, 0, NULL},
    },
    3,
    ROWMAP_TEAMID_TXT_ONLY,
    {
        {0, 0, 0},
        {0, 0, 0}
    }
};

/* Applies to motorsport */
static const TRowMap RowMap_MotorSport_Sched =
{
    {
        {0, LB_EXTRA_TEXT, 		SXM_SPORTS_AUTO_RACE,			KEY_TYPE_TXT,	"Race"},
        {0, LB_EXTRA_TEXT, 		SXM_SPORTS_AUTO_TRACK,			KEY_TYPE_TXT,	"Track"},
        {0, LB_EVENT_START,		SXM_SPORTS_AUTO_START_TIME,		KEY_TYPE_INT,	NULL},
        {0, LB_HOME_INFO_UINT, 	SXM_SPORTS_AUTO_LAPS,			KEY_TYPE_INT,	"Laps"},
        {0, LB_EVENT_STATE, 	SXM_SPORTS_AUTO_STATE,			KEY_TYPE_INT,	NULL},

        {0, LB_EXTRA_TEXT,		SXM_SPORTS_AUTO_WINNER,			KEY_TYPE_TXT,	"Winner"},
        {0, LB_NATIONAL_CHAN, 	SXM_SPORTS_AUTO_NATION_CHAN,	KEY_TYPE_INT,	NULL},
        {0, LB_TREF,			SXM_SPORTS_AUTO_RANK,			KEY_TYPE_INT,	NULL},
        {0, 0, 0, 0, NULL},
        {0, 0, 0, 0, NULL},

        {0, 0, 0, 0, NULL},
        {0, 0, 0, 0, NULL},
        {0, 0, 0, 0, NULL},
        {0, 0, 0, 0, NULL},
        {0, 0, 0, 0, NULL},

        {0, 0, 0, 0, NULL},
        {0, 0, 0, 0, NULL},
        {0, 0, 0, 0, NULL},
    },
    8,
    ROWMAP_TEAMID_TXT_ONLY,
    {
        {0, 0, 0},
        {0, 0, 0}
    }
};

static const TRowMap RowMap_MotorSport_Rank =
{
    {
        {0, LB_HOME_SPID,	 	SXM_SPORTS_AUTO_RANK_DRIVER,		KEY_TYPE_INT,	NULL},
        {0, LB_EXPLICIT_RANK,	SXM_SPORTS_AUTO_RANK_RANK,			KEY_TYPE_TXT,	NULL},
        {0, LB_HOME_INFO_UINT, 	SXM_SPORTS_AUTO_RANK_LAPS,			KEY_TYPE_INT,	"Laps"},
        {0, LB_HOME_INFO_TXT, 	SXM_SPORTS_AUTO_RANK_NUMBER,		KEY_TYPE_TXT,	"Number"},
        {0, LB_HOME_INFO_TXT,	SXM_SPORTS_AUTO_RANK_MAKE,			KEY_TYPE_TXT,	"Make"},

        {0, 0, 0, 0, NULL},
        {0, 0, 0, 0, NULL},
        {0, 0, 0, 0, NULL},
        {0, 0, 0, 0, NULL},
        {0, 0, 0, 0, NULL},

        {0, 0, 0, 0, NULL},
        {0, 0, 0, 0, NULL},
        {0, 0, 0, 0, NULL},
        {0, 0, 0, 0, NULL},
        {0, 0, 0, 0, NULL},

        {0, 0, 0, 0, NULL},
        {0, 0, 0, 0, NULL},
        {0, 0, 0, 0, NULL},
    },
    5,
    ROWMAP_TEAMID_TXT_ONLY,
    {
        {0, 0, 0},
        {0, 0, 0}
    }
};

static const TRowMap* Known_Schedule_Map[SPID_MAX] =
{
        NULL, 						//SPID_MULTI
        &RowMap_H2H_Sched,			//SPID_FOOTBALL
        &RowMap_Baseball_Sched,		//SPID_BASEBALL
        &RowMap_H2H_Sched,			//SPID_BASKETBALL
        &RowMap_H2H_Sched,			//SPID_ICEHOCKEY
        &RowMap_MotorSport_Sched,	//SPID_MOTORSPORT
        &RowMap_Golf_Sched,			//SPID_GOLF
        &RowMap_H2H_Sched,			//SPID_SOCCER
        NULL						//SPID_TENNIS
};

static SportsService *service = NULL;

/***************************************************************************//**
 *
 * This function initializes the file-scope service structure pointer
 *
 * \param[in] ims service structure pointer
 *
 * \return NONE
 *
 ******************************************************************************/
void sxm_sports_extract_init(SportsService *ims)
{
    service = ims;
}

/***************************************************************************//**
 *
 * This function uninitializes the file-scope service structure pointer
 *
 * \param[in] NONE
 *
 * \return NONE
 *
 ******************************************************************************/
void sxm_sports_extract_uninit(void)
{
    service = NULL;
}

/***************************************************************************//**
 *
 * This function initializes the SportsRow structure in preparation
 * for extraction - clears (zeros) all the elements in the row and
 * inits the row helper which tracks the current row index and error state.
 *
 * \param[in] h pointer to the SportsRowHelper structure
 * \param[in] row pointer to the SportsRow buffer
 *
 * \return NONE
 *
 ******************************************************************************/
static SXE_INLINE void row_init(SportsRowHelper* h, SXMSportsRow* row)
{
    h->row = row;
    h->tidx = 0;
    h->err = 0;
    memset(row, 0, sizeof(*row));
}

/***************************************************************************//**
 *
 * Private function to populate a single column of SXMSportsRow Object with
 * an integer value.
 *
 * \param[in] h pointer to the SportsRowHelper structure
 * \param[in] i index value
 * \param[in] key key identifier
 *
 * \return NONE
 *
 ******************************************************************************/
static SXE_INLINE void row_put_int(SportsRowHelper* h, int i, uint key)
{
    /* only put the data if error-free */
    if(!h->err)
    {
        h->row->present |= (1u<<key);   /* this key is present */
        h->row->value[key] = i;         /* integer key value */
    }
}

/***************************************************************************//**
 *
 * Private function to populate a single row of SXMSportsRow Object with a
 * string value.
 *
 * \param[in] h pointer to the SportsRowHelper structure
 * \param[in] txt pointer to key text string
 * \param[in] key key identifier
 *
 * \return NONE
 *
 ******************************************************************************/
static SXE_INLINE void row_put_str(SportsRowHelper* h, char* txt, int key){
    int len;

    /* only put the data if error-free and non-null string pointer */
    if(!h->err && txt)
    {
        len = (int)strlen(txt) + 1;

        /* prevent buffer overflow */
        if((h->tidx + len) > (int)ARRAY_SIZE(h->row->buffer))
        {
            h->err = 1;
            h->row->present = 0;
            h->row->highwater = 0;
            WARN_EXT("No room in buf. txt=%s key=%d len=%d bufoffset=%d", txt, key, len, h->tidx);
            return;
        }

        /* copy the string into the buffer */
        strcpy(&h->row->buffer[h->tidx], txt);
        h->row->value[key] = h->tidx;
        h->row->present |= (1u<<key);
        h->row->types |= (1u<<key);
        h->tidx += len;
        h->row->highwater = (uint)h->tidx;
    }
}

/***************************************************************************//**
 *
 * Private function to initialize the SportsRaw structure in preparation
 * for extraction - clears (zero) all the elements in the structure and
 * inits the raw structure helper with the table reference and epoch.
 *
 * \param[in] h pointer to the SportsRawHelper structure
 * \param[in] raw pointer to the SportsRaw buffer
 * \param[in] tableref  table reference
 * \param[in] tableepoch table epoch
 *
 * \return NONE
 *
 ******************************************************************************/
static SXE_INLINE void raw_init(SportsRawHelper* h, SXMSportsRaw* raw,
    int tableref, int tableepoch)
{
    h->raw = raw;
    h->tidx = 0;
    h->err = 0;
    memset(raw, 0, sizeof(*raw));
    raw->tableref = tableref;
    raw->tableepoch = tableepoch;
}

static SXE_INLINE void raw_put_int(SportsRawHelper* h, int i, int lbid)
{
    if(h->err)
    {
        return;
    }

    if(h->raw->count >= ARRAY_SIZE(h->raw->value))
    {
        WARN_EXT("No room for value. i=%s lbid=%d count=%d", i, lbid, h->raw->count);
        h->raw->count = 0;
        h->raw->highwater = 0;
        h->err = 2;
        return;
    }

    h->raw->value[h->raw->count] = i;
    h->raw->lbid[h->raw->count] = lbid;

    /* h->raw->types is defaulted to zero.  Only strings need to set a bit. */
    h->raw->count++;
}

static SXE_INLINE void raw_put_str(SportsRawHelper* h, char* txt, int lbid)
{
    int len;

    if(h->err || !txt)
    {
        return;
    }

    if(h->raw->count >= ARRAY_SIZE(h->raw->value))
    {
        WARN_EXT("No room for value. txt=%s lbid=%d count=%d", txt, lbid, h->raw->count);
        h->raw->count = 0;
        h->raw->highwater = 0;
        h->err = 2;
        return;
    }

    len = (int)strlen(txt) + 1;
    if((h->tidx + len) > (int)ARRAY_SIZE(h->raw->buffer))
    {
        h->err = 1;
        h->raw->count = 0;
        h->raw->highwater = 0;
        WARN_EXT("No room for txt. txt=%s lbid=%d len=%d bufoffset=%d", txt, lbid, len, h->tidx);
        return;
    }

    strcpy(&h->raw->buffer[h->tidx], txt);
    h->raw->value[h->raw->count] = h->tidx;
    h->raw->lbid[h->raw->count] = lbid;
    BITS(h->raw->types, h->raw->count);
    h->tidx += len;
    h->raw->highwater = h->tidx;
    h->raw->count++;
}

static SXE_INLINE void raw_put_age(SportsRawHelper* h, uint ts)
{
    uint n;
    if(!h->err)
    {
        n = sxm_sxe_time_now();
        h->raw->tableage = (n > ts) ? n - ts : 0;
    }
}

/***************************************************************************//**
 *
 * Private function to populate the Service Row Extraction object with the
 * specfied Sports row data.
 *
 * \param[in] service pointer to Sports Service structure
 * \param[in] row pointer to the Row Extraction object
 * \param[in] id the Sports ID
 * \param[in] name the Sport Name
 *
 * \retval h.err non-zero on error
 *
 ******************************************************************************/
static int write_sport(SportsService *service, SXMSportsRow* row, int id, char* name)
{
    SportsRowHelper h;

    /* initialize */
    row_init(&h, row);

    /* populate the row extraction object with the data */
    row_put_int(&h, id,   SXM_SPORTS_LIST_SPID);
    row_put_str(&h, name, SXM_SPORTS_LIST_SPNAME);

    return h.err;
}

static int write_affiliate(SportsService *service, SXMSportsRow* row, Affiliation* a, int season)
{

#if (SXM_USE_GEN8)
#if (SXM_USE_GMD)
    SportsRowHelper h;
    SXMLeague l;
    int rc;

    row_init(&h, row);
    row_put_int(&h, a->spid, SXM_SPORTS_LEAGUE_SPID);
    row_put_int(&h, a->afid, SXM_SPORTS_LEAGUE_AFID);
    if(season != SEASON_STATUS_UNKNOWN)
    {
        row_put_int(&h, season,	SXM_SPORTS_LEAGUE_SEASON);
    }
    row_put_str(&h, a->name,	SXM_SPORTS_LEAGUE_AFNAME);

    if(SBLOCK[a->spid] != INVALID_ID)
    {
        if (SBLOCK[a->spid] < ARRAY_SIZE(SPORT))
        {
            row_put_str(&h, SPORT[SBLOCK[a->spid]].name, SXM_SPORTS_LEAGUE_SPNAME);
        }
        else
        {
            ERROR_EXT("invalid SPORT index (a->spid=%d,SBLOCK[a->spid]=%d)",
                a->spid, SBLOCK[a->spid]);
        }
    }

    if(a->global != INVALID_GDREF)
    {
        row_put_int(&h, a->global, SXM_SPORTS_LEAGUE_GDREF);
        /*
         *Only leagues(root affiliates) return decoded meta-data for the league gdref.
         * Children return the gdref only it is required to look up team names.
         */
        if(a->level == 0)
        {
            rc = sxm_gmd_league_extract(a->global, &l);

            if(rc == SXM_E_OK)
            {
                row_put_str(&h, l.ln, SXM_SPORTS_LEAGUE_AFLNAME);
                row_put_str(&h, l.sn, SXM_SPORTS_LEAGUE_AFSNAME);
            }
            else
            {
                ERROR_EXT("Failed to get league metadata. lid=%d", a->global);
            }
        }
    }

    return h.err;
#else
    SportsRowHelper h;
	/* error */
	h.err = -1;
   return h.err;
#endif
#else 
    SportsRowHelper h;
    SXMLeague l;
    int rc;

    row_init(&h, row);
    row_put_int(&h, a->spid, SXM_SPORTS_LEAGUE_SPID);
    row_put_int(&h, a->afid, SXM_SPORTS_LEAGUE_AFID);
    if(season != SEASON_STATUS_UNKNOWN)
    {
        row_put_int(&h, season,	SXM_SPORTS_LEAGUE_SEASON);
    }
    row_put_str(&h, a->name,	SXM_SPORTS_LEAGUE_AFNAME);

    if(SBLOCK[a->spid] != INVALID_ID)
    {
        if (SBLOCK[a->spid] < ARRAY_SIZE(SPORT))
        {
            row_put_str(&h, SPORT[SBLOCK[a->spid]].name, SXM_SPORTS_LEAGUE_SPNAME);
        }
        else
        {
            ERROR_EXT("invalid SPORT index (a->spid=%d,SBLOCK[a->spid]=%d)",
                a->spid, SBLOCK[a->spid]);
        }
    }

    if(a->global != INVALID_GDREF)
    {
        row_put_int(&h, a->global, SXM_SPORTS_LEAGUE_GDREF);
        /*
         *Only leagues(root affiliates) return decoded meta-data for the league gdref.
         * Children return the gdref only it is required to look up team names.
         */
        if(a->level == 0)
        {
            rc = sxm_sxi_global_league(a->global, &l);
            if(rc == SXM_E_OK)
            {
                row_put_str(&h, l.ln, SXM_SPORTS_LEAGUE_AFLNAME);
                row_put_str(&h, l.sn, SXM_SPORTS_LEAGUE_AFSNAME);
            }
            else
            {
                ERROR_EXT("Failed to get league metadata. lid=%d", a->global);
            }
        }
    }

    return h.err;
#endif

}

static int write_tdef(SportsService* service, SXMSportsRow* row, TCol* c)
{
    SportsRowHelper h;

    row_init(&h, row);

    row_put_int(&h, c->lbid, SXM_SPORTS_TDEF_LBID);
    row_put_int(&h, c->lbtype, SXM_SPORTS_TDEF_LBTYPE);
    if(c->havedmod)
    {
    row_put_int(&h, c->dmod,	SXM_SPORTS_TDEF_DMOD);
    }
    row_put_int(&h, c->prio, SXM_SPORTS_TDEF_PRIORITY);
    if(c->havetext)
    {
    row_put_str(&h, c->text,	SXM_SPORTS_TDEF_LBTXT);
    }

    return h.err;
}

#if(!SXM_USE_GEN8 || SXM_USE_GMD)
static SXE_INLINE int write_team(SportsService *service, SXMSportsRow* row, SXMTeam* tm)
{
    SportsRowHelper h;

    row_init(&h, row);
    row_put_int(&h, tm->id, SXM_SPORTS_TEAM_TEAMID);
    row_put_str(&h, tm->sn, SXM_SPORTS_TEAM_ABBR);
    row_put_str(&h, tm->mn, SXM_SPORTS_TEAM_NICK);
    row_put_str(&h, tm->ln, SXM_SPORTS_TEAM_NAME);
    return h.err;
}
#endif

static SXE_INLINE Affiliation* find_affiliate(SportsService *service, int rootid, int afid)
{
    Affiliation* a;

    if (ABLOCK[afid] != INVALID_ID)
    {
        if (ABLOCK[afid] < ARRAY_SIZE(AFF))
        {
            a = &AFF[ABLOCK[afid]];
            if ((a->afid == afid) && (a->rootid == rootid))
            {
                return a;
            }
        }
        else
        {
            ERROR_EXT("invalid AFF index (afid=%d,ABLOCK[afid]=%d)",
                afid, ABLOCK[afid]);
        }
    }
    return NULL;
}

static SXE_INLINE int is_known_table(ushort tidx, ushort spid)
{
    return ((spid >= SPID_KNOWN_MIN) && (spid <= SPID_KNOWN_MAX) && (tidx <= TIDX_KNOWN_MAX));
}

/***************************************************************************//**
 *
 * Private function to extract the next row of Sports data from the Sports
 * Configuration database.
 *
 * \param[in] service pointer to Sports Service structure
 * \param[in] e pointer to the Sports Extraction structure
 *
 * \retval SXM_E_OK Success
 * \retval SXM_E_ERROR extraction error
 * \retval SXM_E_NOENT no sport
 *
 ******************************************************************************/
static int get_sport(SportsService *service, SportsExtract* e)
{
    int err = SXM_E_OK;

    /* the row service object to populate */
    SXMSportsRow* row = (SXMSportsRow*)e->dest;

    DEBUG_EXT("(rowIdx=%d)", e->rowIdx);

    /* extract the next sport name and ID from the database */
    for(;e->rowIdx <= MAX_SPORT_ID;)
    {
        /* SBLOCK is indexed by Sport ID */
        if(SBLOCK[e->rowIdx] != INVALID_ID)
        {
            if (SBLOCK[e->rowIdx] < ARRAY_SIZE(SPORT) )
            {
                /* write the row columns */
                err = write_sport(service, row,  e->rowIdx, SPORT[SBLOCK[e->rowIdx]].name);
                e->rowIdx++;
                return err ? SXM_E_ERROR : SXM_E_OK;
            }
            else
            {
                ERROR_EXT("invalid SPORT index (e->rowIdx=%d,SBLOCK[e->rowIdx]=%d)",
                    e->rowIdx, SBLOCK[e->rowIdx]);
            }
        }
        e->rowIdx++;
    }

    /* extractions have exhausted the database */
    return SXM_E_NOENT;
}

/***************************************************************************//**
 *
 * Private function to report the season status (in or out of season) for
 * an affiliation.
 *
 * \param[in] service pointer to Sports Service structure
 * \param[in] a pointer to the Sports affiliation descriptor
 *
 * \retval 0 out-of-season
 * \retval 1 in-season
 * \retval -1 indeterminate
 *
 * \note
 * The season status is determined by interrogating all known table data for
 * affiliations with root nodes common to the affiliation in question, and
 * then choosing the latest status amongst them.
 *
 ******************************************************************************/
static int get_season_status(SportsService *service, Affiliation* a)
{

    uint i, j;
    Affiliation* b;
    Table* t;
    int season = SEASON_STATUS_UNKNOWN;
    ushort epoch = 0;

    /* march through the affiliate database for the latest status */
    for(i=0; i<=MAX_AFFILIATE_ID; i++)
    {
        /* only consider valid entries of course */
        if(ABLOCK[i] != INVALID_ID)
        {
            if (ABLOCK[i] >= ARRAY_SIZE(AFF))
            {
                ERROR_EXT("invalid AFF index (i=%d,ABLOCK[i]=%d)",
                    i, ABLOCK[i]);
                continue;
            }
    
            b = &AFF[ABLOCK[i]];

            /* only consider affiliations with common roots (includes self) */
            if(b->rootid != a->rootid)
            {
                continue;
            }

            /* tables others than schedule tables */
            for(j=0;j<ARRAY_SIZE(b->others);j++)
            {

                /* the next 'others' table */
                t = &b->others[j];

                /* table exists if the size is non-zero */
                if(t->size){

                    /* select if more recent */
                    if(t->epoch > epoch)
                    {
                        epoch = t->epoch;
                        season = t->season;
                    }
                }
             }

            /* now the schedule tables */
            for(j=0;j<ARRAY_SIZE(b->schedule);j++)
            {

                /* the next 'schedule' table */
                t = &b->schedule[j];

                /* table exists if the size is non-zero */
                if(t->size)
                {

                    /* select if more recent */
                    if(t->epoch > epoch)
                    {
                        epoch = t->epoch;
                        season = t->season;
                    }
                }
            }
        }
    }

    return season;
}

/***************************************************************************//**
 *
 * Private function to extract the League data from the Sports Affliation
 * database.
 *
 * \param[in] service pointer to Sports Service structure
 * \param[in] e pointer to the Sports Extraction structure
 *
 * \retval SXM_E_OK Success
 * \retval SXM_E_ERROR extraction error
 * \retval SXM_E_NOENT no league
 *
 * \note Only the top-level leagues are extracted
 *
 ******************************************************************************/
static int get_league(SportsService *service, SportsExtract* e)
{
    int season, err;
    Affiliation* a;

    /* the row service object to populate */
    SXMSportsRow* row = (SXMSportsRow*)e->dest;

    DEBUG_EXT("(rowIdx=%d)", e->rowIdx);

    /* extract the next league name and season status from the database */
    for(;e->rowIdx <= MAX_AFFILIATE_ID;)
    {
        if(ABLOCK[e->rowIdx] != INVALID_ID)
        {
            if (ABLOCK[e->rowIdx] < ARRAY_SIZE(AFF))
            {
                a = &AFF[ABLOCK[e->rowIdx]];

                /* only root (top) levels */
                if(a->level == 0)
                {
                    season = get_season_status(service, a);
                    err = write_affiliate(service, row, a, season);
                    e->rowIdx++;
                    return err ? SXM_E_ERROR : SXM_E_OK;
                }
            }
            else
            {
                ERROR_EXT("invalid AFF index (e->rowIdx=%d,ABLOCK[e->rowIdx]=%d)",
                    e->rowIdx, ABLOCK[e->rowIdx]);
            }
        }
        e->rowIdx++;
    }

    /* extractions have exhausted the database */
    return SXM_E_NOENT;
}

/***************************************************************************//**
 *
 * Private function to extract the League Tree from the Sports Affliation
 * database.
 *
 * \param[in] service pointer to Sports Service structure
 * \param[in] e pointer to the Sports Extraction structure
 *
 * \retval SXM_E_OK Success
 * \retval SXM_E_ERROR extraction error
 * \retval SXM_E_NOENT no league
 *
 * \note
 * The tree was built by when the extraction was begun (sxm_sports_begin())
 * and is cached in the extraction descriptor. This function simply writes
 * out the next node in the tree.
 *
 ******************************************************************************/
static int get_league_tree(SportsService *service, SportsExtract* e)
{
    int err;
    SXMSportsRow* row = (SXMSportsRow*)e->dest;

    DEBUG_EXT("(rowIdx=%d, lTreeCount=%d)", e->rowIdx, e->lTreeCount);

    /* write out the next node in the league tree */
    if(e->rowIdx < e->lTreeCount)
    {
        err = write_affiliate(service, row, &AFF[ABLOCK[e->lTree[e->rowIdx]]], e->season);
        e->rowIdx++;
        return err ? SXM_E_ERROR : SXM_E_OK;
    }

    /* extractions have exhausted the database */
    return SXM_E_NOENT;
}

static int skip_column(SportsService *service, SXMBitBuff *pkt, TCol* c, char* buff, uint lbsize)
{
    int err = 0;

    if((c->lbtype == LB_HOME_SPID) || (c->lbtype == LB_AWAY_SPID) || (c->lbtype == LB_P_PARTITION))
    {
        if(FLAG())
        {
            BSKIP(0);
        }
        else
        {
            SKIP(10);
        }
    }
    else
    {
        if(c->dtype == LB_DTYPE_INT)
        {
            SKIP(lbsize);
        }
        else if(c->dtype == LB_DTYPE_TXT)
        {
            BSKIP(0);
        }
        else
        {
            err = 1;
        }
    }

    return (err || pkt->err);
}

/***************************************************************************//**
 *
 * Parse function used to extract the next 'row' service object.
 *
 * The routine will extract rows from only known tables.
 *
 * \param[in] service pointer to the service structure
 * \param[in] e extraction descriptor
 *
 * \return SXe error code
 * \retval SXM_E_OK Success
 * \retval SXM_E_NOENT no known table map or table defintion
 * \retval SXM_E_INVAL the time has rolled over between the 'begin' and 'extract'
 * \retval SXM_E_ERROR parse error
 * \retval get_row/get_raw return code
 *
 ******************************************************************************/
static int get_row(SportsService *service, SportsExtract* e)
{
    STable* tdef;
    SXMBitBuff *pkt;
    TCol* tdefCol;
    const TRowMapEntry* mapCol;
    char* sname;
    uint i, j, n, sign, nomatch, cnt, done, skip;
    SportsRowHelper h;
#if(!SXM_USE_GEN8 || SXM_USE_GMD)
    SXMTeam team;
#endif
    int rc;
    int getTeam[2];
    int result = SXM_E_OK;

    DEBUG_EXT("(Enter) rowIdx=%d entryno=%d isTeamSearch=%d",
              e->rowIdx, e->entryno, e->isTeamSearch);

    /* The table must be one of the known ones. */
    if(e->knownMap == NULL)
    {
        DEBUG_EXT("No known map");
        return SXM_E_NOENT;
    }

    /* We need the table def. */
    if(TBLOCK[e->t->tabno] == INVALID_ID)
    {
        DEBUG_EXT("No table def");
        return SXM_E_NOENT;
    }

    /* Guard against invalid index */
    if (TBLOCK[e->t->tabno] >= ARRAY_SIZE(TABLE))
    {
        DEBUG_EXT("invalid TABLE index (e->t->tabno=%d,TBLOCK[e->t->tabno]=%d)",
                e->t->tabno, TBLOCK[e->t->tabno]);
        return SXM_E_NOENT;
    }

    /* table defintion */
    tdef = &TABLE[TBLOCK[e->t->tabno]];

    /* label count */
    cnt = MIN(tdef->lc, MAX_TABLE_LABEL_COUNT);

    /* bit buffer */
    pkt = &e->bbuff;

    done = 0;
    while(!done)
    {
        /* Skip can only ever be set if it is a team search. */
        skip = nomatch = 0;
        getTeam[0] = getTeam[1] = -1;

        /* init the row extraction object */
        row_init(&h, (SXMSportsRow*)e->dest);

        /* for each label */
        for(i=0; i<cnt; i++)
        {
            /* End if we encountered an error on the previous column. */
            if(e->err)
            {
                done = 1;
                break;
            }

            /* No data for this column. */
            if(!FLAG())
            {
                continue;
            }

            /* the table defintion for this column */
            tdefCol = &tdef->l[i];

            /* Skip mode or not a known column. */
            if(skip || (e->tdefKnownMap[i] == TDEF_MAP_INVALID))
            {
                e->err = (ushort)skip_column(service, pkt, tdefCol, e->buff, e->lbsize[i]);
                continue;
            }

            /*
             * Known column.
             */
            mapCol = &(e->knownMap->col[e->tdefKnownMap[i]]);

            /* These get special parsing. */
            if((tdefCol->lbtype == LB_HOME_SPID) || (tdefCol->lbtype == LB_AWAY_SPID))
            {
                if(FLAG())
                {
                    nomatch |= (uint)(1u<<tdefCol->lbtype);

                    /* We saw the other id which did not match.  This one is a string so no match is possible. */
                    if(e->isTeamSearch && (nomatch == 0x03))
                    {
                        BSKIP(0);
                        skip = 1;
                    }
                    else
                    {
                        BAUDOT(e->buff, BAUDOT_MODE_START_WITH_LETTERS,
                            BIG_SYMCOUNT(MAX_TEAM_NAME_CHAR), MAX_TEAM_NAME_CHAR);

                        if(e->knownMap->teamIdType == ROWMAP_TEAMID_TXT_ONLY)
                        {
                            /* The columns key is used.  No need for the teamKeyBreakout[] mess or '^' format processing. */
                            row_put_str(&h, e->buff, mapCol->key);
                        }
                        else
                        {
                            DEBUG_EXT("%s", e->buff);
                            sname = strchr(e->buff, '^');
                            if(sname)
                            {
                                /* short^LongTeamName */
                                *sname = '\0';
                                sname++;
                                row_put_str(&h, e->buff, e->knownMap->teamKeyBreakout[mapCol->lbtype][ROWMAP_TEAMID_SHORT]);
                                row_put_str(&h, sname, e->knownMap->teamKeyBreakout[mapCol->lbtype][ROWMAP_TEAMID_LONG]);
                            }
                            else
                            {
                                /* No ^ long name only */
                                row_put_str(&h, e->buff, e->knownMap->teamKeyBreakout[mapCol->lbtype][ROWMAP_TEAMID_LONG]);
                            }
                        }
                    }
                }
                else
                {
                    NEXT(n, 10);

                    if(e->isTeamSearch && (n != e->teamId))
                    {
                        nomatch |= (uint)(1u<<tdefCol->lbtype);

                        /* We did not match either home or away.  Done with the row. */
                        if(nomatch == 0x03)
                        {
                            skip = 1;
                        }
                    }

                    if(!skip)
                    {
                        if(e->knownMap->teamIdType == ROWMAP_TEAMID_FULL)
                        {
                            row_put_int(&h, (int)n, mapCol->key);
                            /*
                             * We will store the need to look up team info.  This is to speed up team search by
                             * avoiding any team look up until we have found a match.
                             */
                            getTeam[mapCol->lbtype] = (int)n;
                        }
                        else
                        {
                            /* 
                             * ROWMAP_TEAMID_TXT_ONLY
                             * Must be text based for this known configuration.  Cannot make a team entry.
                             * Pass up a no team row.  The user has to sort it out.
                             */
                            DEBUG_EXT("Expect text team received int id tid=%d", n);
                        }
                    }
                }
            }
            else if(tdefCol->dtype == LB_DTYPE_INT)
            {
                NEXT(n, e->lbsize[i]);

                if(mapCol->lbtype == LB_EVENT_START)
                {
                    n = (uint)( (n*5u*60u) + (e->t->epoch*86400u + 2u*3600u) );
                    row_put_int(&h, (int)n, mapCol->key);
                }
                else if(mapCol->lbtype == LB_TREF)
                {
                    /* Special case for trefs.  We must mark they are from a known table
                     * in order to identify them for
                     * finding their known map when they are extracted.
                     * We need to remap to account for multi au re assemble to a single packet.
                     * Should never have sign extension either.
                     */
                    for(j=0; j<ARRAY_SIZE(e->t->trefmap); j++)
                    {
                        if(n <= e->t->trefmap[j].offset)
                        {
                            n -= e->t->trefmap[j].delta;
                            break;
                        }
                    }

                    n |= (uint)(TREF_IS_KNOWN);

                    row_put_int(&h, (int)n, mapCol->key);
                }
                else
                {
                    if(Label_Sign_Extend[tdefCol->lbtype])
                    {
                        sign = (uint)(n & (1u<<(e->lbsize[i]-1u)) );
                        n &= ~sign;
                        rc = (int)n;
                        if(n && sign)
                        {
                            rc *= -1;
                        }
                        row_put_int(&h, rc, mapCol->key);
                    }
                    else
                    {
                        row_put_int(&h, (int)n, mapCol->key);
                    }
                }
            }
            else if(tdefCol->dtype == LB_DTYPE_TXT)
            {
                BAUDOT(e->buff, BAUDOT_MODE_START_WITH_LETTERS,
                    SMALL_SYMCOUNT(e->lbsize[i]), e->lbsize[i]);
                row_put_str(&h, e->buff,	mapCol->key);
            }
            else
            {
                // Unable to handle column.  Bad news.
                e->err = 1;
                ERROR_EXT("Unhandled column: i=%d id=%d ver=%d lbtype=%d dtype=%d",
                          i, tdef->tclass, tdef->ver, tdefCol->lbtype, tdefCol->dtype);
            }

            if(h.err || pkt->err)
            {
                ERROR_EXT("Parse failed: h.err=%d pkt->err=%d", h.err, pkt->err);
                e->err = 1;
            }
        }

        e->rowIdx++;

        /* Keeper, non team search always exits here after one row is read. */
        if(!skip)
        {
            if(e->global != INVALID_GDREF)
            {
                for(j=0; j<2; j++)
                {
                    if(getTeam[j] == -1)
                    {
                        continue;
                    }
#if (SXM_USE_GEN8)
#if (SXM_USE_GMD)
                   rc = sxm_gmd_team_extract(getTeam[j], e->global, &team);

                    if (rc == SXM_E_OK)
                    {
                        row_put_str(&h, team.sn, e->knownMap->teamKeyBreakout[j][ROWMAP_TEAMID_SHORT]);
                        row_put_str(&h, team.mn, e->knownMap->teamKeyBreakout[j][ROWMAP_TEAMID_MEDIUM]);
                        row_put_str(&h, team.ln, e->knownMap->teamKeyBreakout[j][ROWMAP_TEAMID_LONG]);
                    }
                    else
                    {
                        ERROR_EXT("Failed to get team: rc=%d tid=%d lid=%d", rc, getTeam[j], e->global);
                    }

#else
                   rc = SXM_E_STATE;
#endif
#else
                   rc = sxm_sxi_global_team(getTeam[j], e->global, &team);

                   if (rc == SXM_E_OK)
                    {
                        row_put_str(&h, team.sn, e->knownMap->teamKeyBreakout[j][ROWMAP_TEAMID_SHORT]);
                        row_put_str(&h, team.mn, e->knownMap->teamKeyBreakout[j][ROWMAP_TEAMID_MEDIUM]);
                        row_put_str(&h, team.ln, e->knownMap->teamKeyBreakout[j][ROWMAP_TEAMID_LONG]);
                    }
                    else
                    {
                        ERROR_EXT("Failed to get team: rc=%d tid=%d lid=%d", rc, getTeam[j], e->global);
                    }
#endif
                }
            }

            /* Team search events need the index to identify the day and age of event. */
            if(e->isTeamSearch)
            {
                row_put_int(&h, e->curTidx, SXM_SPORTS_H2H_TABLE_INDEX);
                n = sxm_sxe_time_now();
                n = (n >= e->t->timeStamp) ? n - e->t->timeStamp : 0;
                row_put_int(&h, (int)n, SXM_SPORTS_H2H_TABLE_AGE);
            }
            break;
        }

        /*  Team search reached end of table data. */
        if(e->isTeamSearch && (e->rowIdx >= e->entryno))
        {
            result = SXM_E_NOENT;
            break;
        }
    };

    DEBUG_EXT("(Exit) rowIdx=%d", e->rowIdx);

    return (e->err) ? SXM_E_ERROR : result;
}

static int get_raw(SportsService *service, SportsExtract* e)
{
    STable* tdef;
    SXMBitBuff *pkt;
    TCol* tdefCol;
    uint i, n, k, nomatch, cnt, done, skip;
    SportsRawHelper h;
    int j;
    int result = SXM_E_OK;

    DEBUG_EXT("(Enter) rowIdx=%d entryno=%d isTeamSearch=%d", e->rowIdx, e->entryno, e->isTeamSearch);

    /* We need the table def. */
    if(TBLOCK[e->t->tabno] == INVALID_ID)
    {
        DEBUG_EXT("No table def");
        return SXM_E_NOENT;
    }

    /* Guard against invalid index */
    if (TBLOCK[e->t->tabno] >= ARRAY_SIZE(TABLE))
    {
        DEBUG_EXT("invalid TABLE index (e->t->tabno=%d,TBLOCK[e->t->tabno]=%d)",
                e->t->tabno, TBLOCK[e->t->tabno]);
        return SXM_E_NOENT;
    }

    tdef = &TABLE[TBLOCK[e->t->tabno]];
    cnt = MIN(tdef->lc, MAX_TABLE_LABEL_COUNT);
    pkt = &e->bbuff;

    done = 0;
    while(!done)
    {
        /* Skip can only ever be set if it is a team search. */
        skip = nomatch = 0;
        raw_init(&h, (SXMSportsRaw*)e->dest, SPORTS_TDEF_PARAM(e->t->tabno, tdef->ver), e->t->epoch);

        for(i=0; i<cnt; i++)
        {
            /* End if we encountered an error on the previous column. */
            if(e->err)
            {
                break;
            }

            /* No data for this column. */
            if(!FLAG())
            {
                continue;
            }

            tdefCol = &tdef->l[i];

            /* Not a label type in the protocol the time this code was written.
             * User has no context to its meaning and
             * we have no way to handle sign extension if it is signed.
             */
            if(skip || (tdefCol->lbtype >= LB_MAX))
            {
                e->err = (ushort)skip_column(service, pkt, tdefCol, e->buff, e->lbsize[i]);
                continue;
            }

            if (tdefCol->lbtype == LB_P_PARTITION)
            {
                if(FLAG())
                {
                    BAUDOT(e->buff, BAUDOT_MODE_START_WITH_LETTERS,
                        BIG_SYMCOUNT(MAX_TEAM_NAME_CHAR), MAX_TEAM_NAME_CHAR);
                    raw_put_str(&h, e->buff, tdefCol->lbid);
                }
                else
                {
                    NEXT(n, 10);
                    raw_put_int(&h, (int)n, tdefCol->lbid);
                }
            }
            else if((tdefCol->lbtype == LB_HOME_SPID) || (tdefCol->lbtype == LB_AWAY_SPID))
            {
                if(FLAG())
                {
                    nomatch |= (uint)(1u<<tdefCol->lbtype);

                    // We saw the other id which did not match.  This one is a string so no match is possible.
                    if(e->isTeamSearch && (nomatch == 0x03))
                    {
                        BSKIP(0);
                        skip = 1;
                    }
                    else
                    {
                        BAUDOT(e->buff, BAUDOT_MODE_START_WITH_LETTERS,
                            BIG_SYMCOUNT(MAX_TEAM_NAME_CHAR), MAX_TEAM_NAME_CHAR);
                        raw_put_str(&h, e->buff, tdefCol->lbid);
                    }
                }
                else
                {
                    NEXT(n, 10);

                    if(e->isTeamSearch && (n != e->teamId))
                    {
                        nomatch |= (uint)(1u<<tdefCol->lbtype);
                        // We did not match either home or away.  Done with the row.
                        if(nomatch == 0x03)
                        {
                            skip = 1;
                        }
                    }

                    if(!skip)
                    {
                        raw_put_int(&h, (int)n, tdefCol->lbid);
                    }
                }
            }
            else if(tdefCol->dtype == LB_DTYPE_INT)
            {
                NEXT(n, e->lbsize[i]);

                if(tdefCol->lbtype == LB_TREF)
                {
                    // Special case for trefs.  We need to remap to account for multi au re assemble to a single packet.
                    // Should never have sign extension either.
                    for(k=0; k<ARRAY_SIZE(e->t->trefmap); k++)
                    {
                        if(n <= e->t->trefmap[k].offset)
                        {
                            n -= e->t->trefmap[k].delta;
                            break;
                        }
                    }

                    raw_put_int(&h, (int)n, tdefCol->lbid);
                }
                else {
                    if(Label_Sign_Extend[tdefCol->lbtype])
                    {
                        k = (uint)(n & (1u<<(e->lbsize[i]-1u)));
                        n &= ~k;
                        j = (int)n;
                        if(n && k)
                        {
                            j *= -1;
                        }
                        raw_put_int(&h, j, tdefCol->lbid);
                    }
                    else 
                    {
                        raw_put_int(&h, (int)n, tdefCol->lbid);
                    }
                }
            }
            else if(tdefCol->dtype == LB_DTYPE_TXT)
            {
                BAUDOT(e->buff, BAUDOT_MODE_START_WITH_LETTERS,
                    SMALL_SYMCOUNT(e->lbsize[i]), e->lbsize[i]);
                raw_put_str(&h, e->buff, tdefCol->lbid);
            }
            else
            {
                /* Unable to handle column.  Bad news. */
                e->err = 1;
                ERROR_EXT("Unhandled column: i=%d id=%d ver=%d lbtype=%d dtype=%d",
                          i, tdef->tclass, tdef->ver, tdefCol->lbtype, tdefCol->dtype);
            }

            if(h.err || pkt->err)
            {
                e->err = 1;
                ERROR_EXT("Parse failed: h.err=%d pkt->err=%d", h.err, pkt->err);
            }
        }

        e->rowIdx++;

        /* Keeper, non team search always exits here after one row is read. */
        if(!skip)
        {
            /* Team search needs to output the age of the event. */
            if(e->isTeamSearch)
            {
                raw_put_age(&h, e->t->timeStamp);
            }
            break;
        }

        /*  Team search reached end of table data. */
        if(e->isTeamSearch && (e->rowIdx >= e->entryno))
        {
            result = SXM_E_NOENT;
            break;
        }
    };

    DEBUG_EXT("(Exit) rowIdx=%d", e->rowIdx);

    return (e->err) ? SXM_E_ERROR : result;
}

/**************************************************************************//**
 *
 * Parse function to extract a table definition
 *
 * \param[in] service SportsService structure pointer
 * \param[in] pXtract SportsExtract structure pointer
 *
 * \return SXe error code
 * \retval SXM_E_OK Success
 * \retval SXM_E_ERROR error processing request
 * \retval SXM_E_NOENT empty table definition
 *
 *****************************************************************************/
static int get_tdef(SportsService *service, SportsExtract* pXtract)
{
    int err;
    STable* tdef;
    SXMSportsRow* row = (SXMSportsRow*)pXtract->dest;

    DEBUG_EXT("(rowIdx=%d entryno=%d)", pXtract->rowIdx, pXtract->entryno);

    /* Recheck tdef existence.  Should not really need this.  Not too expensive. */
    if(TBLOCK[pXtract->tidx] == INVALID_ID)
    {
        DEBUG_EXT("No tdef: e->tidx=%d", pXtract->tidx);
        return SXM_E_ERROR;
    }

    /* Guard against invalid index */
    if (TBLOCK[pXtract->tidx] >= ARRAY_SIZE(TABLE))
    {
        DEBUG_EXT("invalid TABLE index (pXtract->tidx=%d,TBLOCK[pXtract->tidx]=%d)",
                pXtract->tidx, TBLOCK[pXtract->tidx]);
        return SXM_E_ERROR;
    }

    /* Bounds check label count again. */
    tdef = &TABLE[TBLOCK[pXtract->tidx]];
    if(tdef->lc != pXtract->entryno)
    {
        DEBUG_EXT("Label count mismatch: lc=%d entryno=%d", tdef->lc, pXtract->entryno);
        return SXM_E_ERROR;
    }

    /* now extract rows */
    for(;pXtract->rowIdx < pXtract->entryno; pXtract->rowIdx++)
    {
        /* Only return columns that were are known at the time the code is written.  Future
         * label types have no meaning.
         */
        if(tdef->l[pXtract->rowIdx].lbtype < LB_MAX)
        {
            err = write_tdef(service, row, &tdef->l[pXtract->rowIdx]);
            pXtract->rowIdx++;
            return (err) ? SXM_E_ERROR : SXM_E_OK;
        }
    }

    return SXM_E_NOENT;
}

static Affiliation* get_next_aff(SportsService *service, SportsExtract* e)
{
    Affiliation* a = NULL;

    DEBUG_EXT("(Enter) curAff=%p curAfid=%d", e->curAff, e->curAfid);

    if(e->curAff)
    {
        return &AFF[ABLOCK[e->curAfid]];
    }

    /* Lock onto the next affiliate. */
    e->curAfid++;
    for(; e->curAfid <= MAX_AFFILIATE_ID; e->curAfid++)
    {
        if(ABLOCK[e->curAfid] == INVALID_ID)
        {
            continue;
        }

        if (ABLOCK[e->curAfid] >= ARRAY_SIZE(AFF) )
        {
            DEBUG_EXT("invalid AFF index (e->curAfid=%d,ABLOCK[e->curAfid]=%d)",
                e->curAfid, ABLOCK[e->curAfid]);
            continue;
        }

        if(AFF[ABLOCK[e->curAfid]].rootid == e->rootAfid)
        {
            a = &AFF[ABLOCK[e->curAfid]];
            break;
        }
    }

    DEBUG_EXT("(Exit) a=%p curAfid=%d", a, e->curAfid);
    return a;
}

static SportsExtract* get_next_table(SportsService *service, SportsExtract* e){
    SportsExtract* tableExt = NULL;
    int rc;

    DEBUG_EXT("(Enter) curExt=%p curTidx=%d", e->curExt, e->curTidx);

    if(e->curExt && (e->curTidx <= MAX_SCHEDULE_INDEX))
    {
        return e->curExt;
    }

    // Lock onto the next table.
    e->curTidx++;
    for(; e->curTidx <= MAX_SCHEDULE_INDEX; e->curTidx++)
    {
        rc = locked_sports_extract_begin(service, e->req, SXM_SPORTS_TABLE, SPORTS_TABLE_PARAM(e->curAfid, e->curTidx), (ptr)&tableExt, NULL);
        DEBUG_EXT("(next try) curTidx=%d rc=%d", e->curTidx, rc);
        if(rc == SXM_E_OK)
        {
            /* The parse function only sees the table extract.  Not the team event extract.  Copy over what is needed in the table extract. */
            tableExt->isTeamSearch = 1;
            tableExt->teamId = e->teamId;
            tableExt->curTidx = e->curTidx;
            break;
        }
    }
    DEBUG_EXT("(Exit) tableExt=%p curTidx=%d", tableExt, e->curTidx);
    return tableExt;
}

static int get_team_events(SportsService *service, SportsExtract* e)
{
    int rc;
    uint curTeamReportCnt;
    SXMSportsRow *sportsRow;
    

    DEBUG_EXT("(Enter) curAff=%p curExt=%p curTidx=%d", e->curAff, e->curExt, e->curTidx);

    for(;;)
    {
        e->curAff = get_next_aff(service, e);
        if(!e->curAff)
        {
            DEBUG_EXT("(Exit) done: no more aff");
            return SXM_E_NOENT;
        }

        DEBUG_EXT("(e->curAff->afid=%d)", e->curAff->afid);
        e->curExt = get_next_table(service,  e);
        if(e->curExt)
        {

            for(;;)
            {
                e->curExt->isRow = e->isRow;
                rc = locked_sports_extract_extract_rxw(service, e->curExt, e->dest, e->isRow);
                if((e->curExt->rowIdx <= e->curExt->entryno) && (rc == SXM_E_OK))
                {
                    DEBUG_EXT("(Exit) found row: e->curAff->afid=%d e->curTidx=%d curExt->rowIdx=%d curExt->entryno=%d",
                            e->curAff->afid, e->curTidx, e->curExt->rowIdx, e->curExt->entryno);

                    /* setup the SXMSportsRow pointer */
                    sportsRow = (SXMSportsRow *) e->dest;

                    /* Was this game (home team vs visit team, date, start time) 
                    previously reported to the application ? */
                    for(curTeamReportCnt = 0; curTeamReportCnt < e->teamReportCnt; curTeamReportCnt++)
                    {
                        /* Was this row previously reported to the application ? */
                        if((e->TeamReport[curTeamReportCnt].homeTeamId == sportsRow->value[SXM_SPORTS_H2H_HOME_ID]) &&
                           (e->TeamReport[curTeamReportCnt].visitTeamId == sportsRow->value[SXM_SPORTS_H2H_VISIT_ID]) &&
                           (e->TeamReport[curTeamReportCnt].tidx == sportsRow->value[SXM_SPORTS_H2H_TABLE_INDEX]) &&
                           (e->TeamReport[curTeamReportCnt].startTime == sportsRow->value[SXM_SPORTS_H2H_START_TIME]))
                        {
                           /* this row was previously reported to the application, lets go for the next row */
                           break;
                        }
                    }

                    /* Was the row previously reported ? */
                    if(curTeamReportCnt >=  e->teamReportCnt)
                    {
                        /* NO -- this row was never previously reported to the application */
                        if(e->teamReportCnt < MAX_TEAM_REPORT)
                        {
                            /* lets add the new entry */
                            e->TeamReport[curTeamReportCnt].homeTeamId = (ushort)(sportsRow->value[SXM_SPORTS_H2H_HOME_ID]);
                            e->TeamReport[curTeamReportCnt].visitTeamId =(ushort)(sportsRow->value[SXM_SPORTS_H2H_VISIT_ID]);
                            e->TeamReport[curTeamReportCnt].tidx = (ushort)(sportsRow->value[SXM_SPORTS_H2H_TABLE_INDEX]);
                            e->TeamReport[curTeamReportCnt].startTime = sportsRow->value[SXM_SPORTS_H2H_START_TIME];
                            e->teamReportCnt++;

                           DEBUG_EXT("New Entry: table index %3i, visit team %3i, home team %3i, start time %i",
                                sportsRow->value[SXM_SPORTS_H2H_TABLE_INDEX], sportsRow->value[SXM_SPORTS_H2H_VISIT_ID],
                                sportsRow->value[SXM_SPORTS_H2H_HOME_ID], sportsRow->value[SXM_SPORTS_H2H_START_TIME]);
                       }
                       return rc;
                    }
                    else
                    {
                        /* YES -- this row was previously reported to the application 
                        therefore, lets skip reporting this row and get another one */
                        DEBUG_EXT("Dup Entry: table index %3i, visit team %3i, home team %3i, start time %i",
                            sportsRow->value[SXM_SPORTS_H2H_TABLE_INDEX], sportsRow->value[SXM_SPORTS_H2H_VISIT_ID], 
                            sportsRow->value[SXM_SPORTS_H2H_HOME_ID], sportsRow->value[SXM_SPORTS_H2H_START_TIME]);
                   }
            } else
                {
                    break;
                }
            }
            DEBUG_EXT("(Done with curExt) e->curAff->afid=%d e->curTidx=%d e->curExt->rowIdx=%d curExt->entryno=%d",
                    e->curAff->afid, e->curTidx, e->curExt->rowIdx, e->curExt->entryno);
            rc = locked_sports_extract_extract_end(service, e->curExt);
            e->curExt = NULL;
        }
        else
        {
            DEBUG_EXT("(No more tables for curAff) e->curAff->afid=%d", e->curAff->afid);
            e->curTidx = (ushort)-1;
            e->curAff = NULL;
        }
    }
}

static int get_team_list(SportsService *service, SportsExtract* e)
{

#if (SXM_USE_GEN8)
#if (SXM_USE_GMD)
    int rc = SXM_E_NOENT;
    SXMTeam tm;
    int team_count;

    DEBUG_EXT("(Enter) e->rowIdx=%d e->global=%d", e->rowIdx, e->global);
   if( (rc = sxm_gmd_team_count_extract(&team_count)) == SXM_E_OK)
    {
        for(;;)
        {
            if(e->rowIdx >= team_count)
            {
                rc = SXM_E_NOENT;
                break;
            }
            rc = sxm_gmd_team_ix_extract(e->rowIdx, &tm);

            e->rowIdx++;
            if(rc == SXM_E_OK)
            {

               if(sxm_gmd_team_in_league(&tm, e->global))
                {
                    rc = write_team(service, e->dest, &tm);
                    if(!rc)
                    {
                        rc = SXM_E_OK;
                        break;
                    }
                    ERROR_EXT("(failed to write team) rc=%d", rc);
                }
            }
            else
            {
                if (rc == SXM_E_INVAL) rc = SXM_E_NOENT;
                break;
            }
        }
    }
#else 
   int rc = SXM_E_NOENT;
#endif
#else
    int rc = SXM_E_NOENT;
    SXMTeam tm;
    int team_count;

    DEBUG_EXT("(Enter) e->rowIdx=%d e->global=%d", e->rowIdx, e->global);
    if( (rc = sxm_sxi_global_team_count(&team_count)) == SXM_E_OK)
    {
        for(;;)
        {
            if(e->rowIdx >= team_count)
            {
                rc = SXM_E_NOENT;
                break;
            }
            rc = sxm_sxi_global_team_ix(e->rowIdx, &tm);
            e->rowIdx++;
            if(rc == SXM_E_OK)
            {
                if(sxm_is_team_in_league(&tm, e->global))
                {
                    rc = write_team(service, e->dest, &tm);
                    if(!rc)
                    {
                        rc = SXM_E_OK;
                        break;
                    }
                    ERROR_EXT("(failed to write team) rc=%d", rc);
                }
            }
            else
            {
                if (rc == SXM_E_INVAL) rc = SXM_E_NOENT;
                break;
            }
        }
    }
#endif


    DEBUG_EXT("(Exit) e->rowIdx=%d e->global=%d", e->rowIdx, e->global);
    return rc;
}

/***************************************************************************//**
 *
 * Parse function used to extract the next service object.
 *
 * The routine will extract 'row' or 'raw' depending on the extraction context.
 *
 * \param[in] service pointer to the service structure
 * \param[in] e extraction descriptor
 *
 * \return SXe error code
 * \retval SXM_E_OK Success
 * \retval SXM_E_NOENT the table is exhausted, no rows remain
 * \retval SXM_E_INVAL the time has rolled over between the 'begin' and 'extract'
 * \retval get_row/get_raw return code
 *
 ******************************************************************************/
static int parse_row(SportsService *service, SportsExtract* e)
{
    int rc;

    /* check if there are any more rows to extract */
    if(e->rowIdx >= e->entryno)
    {
        return SXM_E_NOENT;
    }

    /* Catch epoch change. */
    if(e->epoch != SPORTS_EPOCH_NOW())
    {
        DEBUG_EXT("(epoch change since begin) e->epoch=%d epochNow=%d", e->epoch, SPORTS_EPOCH_NOW());
        return SXM_E_INVAL;
    }

    // extract either a 'row' object of a 'raw' object.
    if(e->isRow)
    {
        rc = get_row(service, e);
    }
    else
    {
        rc = get_raw(service, e);
    }

    return rc;
}

/**************************************************************************//**
 *
 * Private function to validate the request handle
 *
 * \param[in] service pointer to the service structure
 * \param[in] ext opaque pointer to request handle
 *
 * \return TRUE if handle is valid, FALSE otherwise
 * 
 * \note the handle must be for an active request
 *
 *****************************************************************************/
static BOOL request_handle_valid(SportsService *service, ptr req)
{
    uint i;

    /* Must be in our pool. */
    for(i=0; i<ARRAY_SIZE(service->request); i++)
    {
        if((SportsRequest*)req == &service->request[i])
        {
            if(service->request[i].sportId != INVALID_ID)
            {
                return TRUE;
            }
        }
    }
    return FALSE;
}

/**************************************************************************//**
 *
 * Private function to validate the extraction handle
 *
 * \param[in] service pointer to the service structure
 * \param[in] ext opaque pointer to extraction handle
 *
 * \return TRUE if handle is valid, FALSE otherwise
 * 
 * \note the handle must be for an active extraction
 *
 *****************************************************************************/
static BOOL extract_handle_valid(SportsService *service, ptr ext){
    uint i;
    SportsExtract* e;

    /* Must be in our pool. */
    for(i=0; i<ARRAY_SIZE(service->extBuffer); i++)
    {
        if((SportsExtract*)ext == &service->extBuffer[i])
        {
            /* Must not be free. */
            e = service->freeExtRoot;
            while(e)
            {
                if((SportsExtract*)ext == e)
                {
                    return FALSE;
                }
                e = e->link;
            }
            return TRUE;
        }
    }
    return FALSE;
}

/**************************************************************************//**
 *
 * Private function to validate if the request has affiliate
 *
 * \param[in] service pointer to the service structure
 * \param[in] req opaque pointer to request handle
 *
 * \return TRUE if affiliate found, FALSE otherwise
 * 
 * \note the handle must be for an active request.
 *
 *****************************************************************************/
static SXE_INLINE BOOL request_has_affiliate(SportsService *service, ptr req)
{
    SportsRequest* r = (SportsRequest*)req;
    Affiliation* a = sxm_sports_shared_find_root(r);
    if(a)
    {
        r->afid = a->afid;
        return TRUE;
    }

    return FALSE;
}

/**************************************************************************//**
 *
 * Private function to prepare the service for extraction of the 'live' sports
 * database for the SXM_SPORTS_LIST extraction type.
 *
 * The next free extraction descriptor is assigned, and then initialized
 * by assigning the extraction type and the extraction parse function which
 * will be used to perform the row extractions.
 *
 * \param[in] service pointer to the service structure
 * \param[in] type extraction type
 * \param[in] ext pointer to the extraction handle pointer
 *
 * \return SXe error code
 * \retval SXM_E_OK extraction initialized, *ext = extraction handle
 *
 * \note
 * If sucessfull (SXM_E_OK is returned), the 'ext' argument will contain
 * a pointer to the extraction resource which is subsequently used as an
 * argument to the sxm_sports_extract_row() API to perform the extraction,
 * and to the sxm_sports_end() API to terminate the extraction and free the
 * extraction resources.
 *
 * \note
 * The Sports List extraction operation does not require a previously issued
 * request.  It stands on its own. It is typically used in the context of the
 * Extended Request/Extract interface, to discover the list of sports currently
 * being supported in the broadcast, beyond any of the well-known sports which
 * are available for 'free' (pre-defined) in the Standard interface.
 *
 *****************************************************************************/
static int begin_sports(SportsService *service, uint type, ptr* ext)
{
    SportsExtract* e;

    /* assign the next free extraction descriptor */
    e = service->freeExtRoot;
    service->freeExtRoot = e->link;

    /* initialize it (clear it out) */
    memset(e, 0, sizeof(*e));

    /* set the type  */
    e->type = type;

    /* the parse function which will be called to perform the extraction */
    e->parse = get_sport;

    /* return the address of the extraction descriptor */
    *ext = (ptr)e;

    return SXM_E_OK;
}

/**************************************************************************//**
 *
 * Private function to prepare the service for extraction of the 'live' leagues
 * database for the SXM_SPORTS_LEAGUES extraction type.
 *
 * The next free extraction descriptor is assigned, and then initialized
 * by assigning the extraction type and the extraction parse function which
 * will be used to perform the row extractions.
 *
 * \param[in] service pointer to the service structure
 * \param[in] type extraction type
 * \param[in] ext pointer to the extraction handle pointer
 *
 * \return SXe error code
 * \retval SXM_E_OK extraction initialized, *ext = extraction handle
 *
 * \note
 * If sucessfull (SXM_E_OK is returned), the 'ext' argument will contain
 * a pointer to the extraction resource which is subsequently used as an
 * argument to the sxm_sports_extract_row() API to perform the extraction,
 * and to the sxm_sports_end() API to terminate the extraction and free the
 * extraction resources.
 *
 * \note
 * The Sports Leagues extraction operation does not require a previously issued
 * request.  It stands on its own. It is typically used in the context of the
 * Extended Request/Extract interface, to discover the list of leagues currently
 * being supported in the broadcast, beyond any of the well-known leagues which
 * are available for 'free' (pre-defined) in the Standard interface.
 *
 *****************************************************************************/
static int begin_leagues(SportsService *service, uint type, ptr* ext)
{
    SportsExtract* e;

    /* assign the next free extraction descriptor */
    e = service->freeExtRoot;
    service->freeExtRoot = e->link;

    /* initialize it (clear it out) */
    memset(e, 0, sizeof(*e));

    /* set the type  */
    e->type = type;

    /* the parse function which will be called to perform the extraction */
    e->parse = get_league;

    /* return the address of the extraction descriptor */
    *ext = (ptr)e;

    return SXM_E_OK;
}

/**************************************************************************//**
 *
 * Private function to construct the league tree from the top down.
 *
 * The affiliation database is parsed recursively to build the tree.
 *
 * \param[in] service pointer to the service structure
 * \param[in] parent affiliation ID of parent
 * \param[in] level tree level
 * \param[in] ltree league tree pointer
 * \param[in] current tree depth
 *
 * \return NONE
 *
 *****************************************************************************/
static void make_ltree(SportsService *service, ushort parent, ushort level,
    ushort* ltree, ushort* cnt)
{
    uint i;
    Affiliation* a;

    /* max level check */
    if(level >= MAX_AFFILIATE_NAME_LEVEL)
    {
        return;
    }

    /* walk the affiliation database */
    for(i=0; i<=MAX_AFFILIATE_ID; i++)
    {
        if(ABLOCK[i] == INVALID_ID)
        {
            continue;
        }

        /* Guard against out-of-range index */
        if (ABLOCK[i] >= ARRAY_SIZE(AFF) )
        {
            DEBUG_EXT("invalid AFF index (i=%d,ABLOCK[i]=%d)",  
                i, ABLOCK[i]);
            continue;
        }

        a = &AFF[ABLOCK[i]];
        if(a->parentid == parent)
        {
            ltree[*cnt] = a->afid;
            (*cnt)++;
            if(*cnt > MAX_LEAGUE_TREE_ENTRIES)
            {
                WARN_EXT("(exceeded max tree entries) cnt=%d max=%d", *cnt, MAX_LEAGUE_TREE_ENTRIES);
                return;
            }

            /* recurse */
            make_ltree(service, a->afid, a->level, ltree, cnt);
        }
    }
}

/**************************************************************************//**
 *
 * Private function to prepare the service for enumeration (extraction) of
 * all the affiliations below the top-level league for the SXM_SPORTS_TREE
 * extraction type.
 *
 * The next free extraction descriptor is assigned, and then initialized
 * by assigning the extraction type and the extraction parse function which
 * will be used to perform the row extractions.
 *
 * \param[in] service pointer to the service structure
 * \param[in] req the request handle
 * \param[in] type extraction type
 * \param[in] ext pointer to the extraction handle pointer
 *
 * \return SXe error code
 * \retval SXM_E_OK extraction initialized, *ext = extraction handle
 *
 * \note
 * If sucessfull (SXM_E_OK is returned), the 'ext' argument will contain
 * a pointer to the extraction resource which is subsequently used as an
 * argument to the sxm_sports_extract_row() API to perform the extraction,
 * and to the sxm_sports_end() API to terminate the extraction and free the
 * extraction resources.
 *
 * \note
 * The Sports Tree extraction operation requires a previously issued
 * request.  It may be used in the context of the Extended or Standard
 * Request/Extract interface, to enumerate all the affilations below the
 * top-level league.  The league in question can be one of the well-known
 * leagues available to the Standard Interface, or one of the leagues
 * 'discovered' by using the Extended Interface.
 *
 *****************************************************************************/
static int begin_league_tree(SportsService *service, SportsRequest* req,
    uint type, ptr* ext)
{
    SportsExtract* e;
    Affiliation* a;
    int rc = SXM_E_OK;

    DEBUG_EXT("(Enter) req=%p type=%d", req, type);

    do
    {
        /* this extraction must have a request */
        if(!req)
        {
            rc = SXM_E_FAULT;
            break;
        }

        /* BSC (Broad Scope Change) is active, return 'busy' */
        if(req->bsc == TRUE)
        {
            rc = SXM_E_BUSY;
            break;
        }

        /* root of league tree */
        a = sxm_sports_shared_find_root(req);
        if(!a)
        {
            rc = SXM_E_NOENT;
            break;
        }

        /* assign the next free extraction resource */
        e = service->freeExtRoot;
        service->freeExtRoot = e->link;

        /* init the extraction descriptor */
        memset(e, 0, sizeof(*e));
        e->type = type;
        e->parse = get_league_tree;

        /*
         * Pre parse the list of affiliates in correct order.
         * Extract walks the list.  Make the league the first entry.
         */
        e->lTreeCount = 1;
        e->lTree[0] = a->afid;
        e->season = (short)get_season_status(service, a);
        make_ltree(service, a->afid, a->level, e->lTree, &e->lTreeCount);

        e->req = req;
        e->link = req->ext;
        req->ext = e;
        *ext = (ptr)e;
    }while(0);

    DEBUG_EXT("(Exit) req=%p rc=%d", req, rc);
    return rc;
}

/**************************************************************************//**
 *
 * Private function to prepare the known table map used during row extraction.
 * Known tables are those tables whose table definitions and formats in the
 * broadcast are known and can be considered static tables by the application.
 * They can be extracted and interpreted independent of the dynamic table
 * defintions in the broadcast.
 *
 * This function verifies the known table defintion is compatible with the
 * current table defintion in the broadcast, and builds a map to each column
 * that is consistent with the actual table defintion.  This mapping is
 * necessary to prevent errors from occuring during parsing of the table
 * entries caused by mismatched fields between the known and actual tables.
 * 
 * \param[in] tdef pointer to the table definition
 * \param[in] spid the sport ID
 * \param[in] tidx the table Index
 * \param[in] e pointer to the extraction descriptor
 * \param[in] isKnownReferenced flags referenced tables
 *
 * \return NONE
 *
 *****************************************************************************/
static void make_known_table_map(STable* tdef, ushort spid, uint tidx,
    SportsExtract* e, uint isKnownReferenced)
{
    uint i, j, used, cnt;
    const TRowMap* rm = NULL;

    /* Make the known table - tdef mapping. */
    if(is_known_table((ushort)tidx, spid))
    {
        if(isKnownReferenced)
        {
            if(spid == SPID_GOLF)
            {
                rm = &RowMap_Golf_Rank;
            }
            else if (spid == SPID_MOTORSPORT)
            {
                rm = &RowMap_MotorSport_Rank;
            }
        }

        else if(tidx == TIDX_KNOWN_NEWS)
        {
            rm = &TRowMap_News;
        }
        else
        {
            if(spid < ARRAY_SIZE(Known_Schedule_Map) && (Known_Schedule_Map[spid]))
            {
                rm = Known_Schedule_Map[spid];
            }
        }
    }

    e->knownMap = rm;

    /*
     * Make map of tdefs that match known columns and fill in default lbsize for parsing.
     * Later we will overlay per instance lbsize overrides.
     *
     */
    cnt = MIN(tdef->lc, MAX_TABLE_LABEL_COUNT);
    if(rm)
    {
        for(i=0, used=0; i<cnt; i++)
        {
            e->tdefKnownMap[i] = TDEF_MAP_INVALID;
            e->lbsize[i] = tdef->l[i].lbsize;
            for(j=0; j<rm->count; j++)
            {

                if(used & (1u<<j))
                {
                    continue;
                }

                if((rm->col[j].priority == tdef->l[i].prio)
                    && (rm->col[j].lbtype == tdef->l[i].lbtype))
                {

                    if(rm->col[j].txt)
                    {
                        if(tdef->l[i].havetext)
                        {
                            if(strcmp(rm->col[j].txt, tdef->l[i].text) == 0)
                            {
                                // match
                                used |= 1u<<j;
                                e->tdefKnownMap[i] = (byte)j;
                            }
                        }
                    }
                    else
                    {
                        if(!tdef->l[i].havetext)
                        {
                            // match
                            used |= 1u<<j;
                            e->tdefKnownMap[i] = (byte)j;
                        }
                    }
                }
            }
        }
    }
    else
    {
        for(i=0; i<cnt; i++)
        {
            e->lbsize[i] = tdef->l[i].lbsize;
        }
    }
}

/**************************************************************************//**
 *
 * Private function to prepare the service for a sports table extraction.
 *
 * The next free extraction descriptor is assigned, and then initialized
 * by assigning the extraction type and the extraction parse function which
 * will be used to perform the row extractions.
 *
 * \param[in] service pointer to the service structure
 * \param[in] req the request handle
 * \param[in] type extraction type
 * \param[in] p1 table index value
 * \param[in] ext pointer to the extraction handle pointer
 * \param[in] header optional pointer to SportsRow header
 *
 * \return SXe error code
 * \retval SXM_E_OK extraction initialized, *ext = extraction handle
 *
 * \note
 * If sucessfull (SXM_E_OK is returned), the 'ext' argument will contain
 * a pointer to the extraction resource which is subsequently used as an
 * argument to the sxm_sports_extract_row() API to perform the extraction,
 * and to the sxm_sports_end() API to terminate the extraction and free the
 * extraction resources.
 *
 * \note
 * The Sports Table extraction operation requires a previously issued
 * request.  It may be used in the context of the Extended or Standard
 * Request/Extract interface, to enumerate extract the table below the
 * top-level league.  The league in question can be one of the well-known
 * leagues available to the Standard Interface, or one of the leagues
 * 'discovered' by using the Extended Interface.
 *
 * \note
 * If the 'header' pointer argument is NON-NULL, the SXM_SPORTS_TABLE
 * info is extracted to populate the header info.
 *
 *****************************************************************************/
static int begin_table(SportsService* service, SportsRequest* req, uint type, int p1,
    ptr* ext, SXMSportsRow* header)
{
    SportsExtract* e;
    Affiliation* a;
    Table* t = NULL;
    STable* tdef;
    uint afid, tidx, i, tepoch, epochNow, timeNow, err;
    SXMBitBuff *pkt;
    uint lbid, lbsize, haveTitle;
    SportsRowHelper h;

    /* NULL request pointer check */
    if(req == NULL)
    {
        return SXM_E_FAULT;
    }

    /* BSC (Broad Scope Change) is active, return 'busy' */
    if(req->bsc == TRUE)
    {
        return SXM_E_BUSY;
    }

    DEBUG_EXT("(Enter) req=%p req->afid=%d type=%d p1=%d header=%p", req, req->afid, type, p1, header);

    /* Find the root affiliate of the request. */
    a = sxm_sports_shared_find_root(req);
    if(a == NULL)
    {
        DEBUG_EXT("(%p) cannot find root aff", req);
        return SXM_E_NOENT;
    }

    /* Get the affiliate of the extract. */
    afid = SPORTS_TABLE_AF(p1);
    if (ABLOCK[afid] == INVALID_ID)
    {
        DEBUG_EXT("(%p) cannot find affiliate", req);
        return SXM_E_NOENT;
    }

    /* Guard against out-of-range index */
    if (ABLOCK[afid] >= ARRAY_SIZE(AFF))
    {
        DEBUG_EXT("invalid AFF index (afid=%d,ABLOCK[afid]=%d)",  
            afid, ABLOCK[afid]);
        return SXM_E_NOENT;
    }

    /* the affiliate descriptor */
    a = &AFF[ABLOCK[afid]];

    /* convert the current time to epoch */
    timeNow = sxm_sxe_time_now();
    epochNow = SPORTS_EPOCH_MAKE(timeNow);

    /* break out the index of the table being extracted */
    tidx = SPORTS_TABLE_IX(p1);

    /* Find the table. */
    if (tidx < ARRAY_SIZE(a->schedule))
    {
        tepoch = epochNow + (tidx - 8);
        for(i=0;i<ARRAY_SIZE(a->schedule); i++)
        {
            if(a->schedule[i].size && (a->schedule[i].epoch == tepoch))
            {
                t = &a->schedule[i];
                break;
            }
        }
    }
    else if (tidx < ARRAY_SIZE(a->others)+ ARRAY_SIZE(a->schedule))
    {
        if(a->others[tidx - ARRAY_SIZE(a->schedule)].size)
        {
            t = &a->others[tidx - ARRAY_SIZE(a->schedule)];
        }
    }
    else
    {
        /* couldn't find the table */
        t = NULL;
    }

    /* did we find the table ? */
    if(t == NULL)
    {
        DEBUG_EXT("(%p) cannot find table: afid=%d tidx=%d epochNow=%d", req, afid, tidx, epochNow);
        return SXM_E_NOENT;
    }

    /* We need the table def. */
    if(TBLOCK[t->tabno] == INVALID_ID)
    {
        DEBUG_EXT("%p) No tdef", req);
        return SXM_E_NOENT;
    }

    /* Guard against out-of-range index */
    if (TBLOCK[t->tabno] >= ARRAY_SIZE(TABLE))
    {
        DEBUG_EXT("invalid TABLE index (t->tabno=%d,TBLOCK[t->tabno]=%d)",  
            t->tabno, TBLOCK[t->tabno]);
        return SXM_E_NOENT;
    }

    /* The data instance and table def versions must match. */
    tdef = &TABLE[TBLOCK[t->tabno]];
    if(tdef->ver != t->tabver)
    {
        DEBUG_EXT("(%p) stored tdef - table version mismatch: tdef->ver=%d t->tabver=%d", req, tdef->ver, t->tabver);
        return SXM_E_NOENT;
    }

    e = service->freeExtRoot;
    service->freeExtRoot = e->link;
    memset(e, 0, sizeof(*e));

    /* make table map if this is a known table */
    make_known_table_map(tdef, a->spid, tidx, e, 0);

    /* Set up pointer to data. */
    pkt = &e->bbuff;
    sxm_bitbuff_setup(pkt, (byte *)&HEAP[t->heapix], t->size);
    SKIPBYTES(t->offset);
    SKIP(10 + 8);  /* + TABLEID + TABVER  already parsed when table was stored. */
    NEXT(haveTitle, 1);
    if(haveTitle)
    {
        BAUDOT(e->buff, BAUDOT_MODE_START_WITH_LETTERS,
            SMALL_SYMCOUNT(MAX_TABLE_TITLE_CHAR), MAX_TABLE_TITLE_CHAR);
    }
    while(FLAG())
    {
        /* 
         * Parse lbsize overrides.  We make per a table parse instance of lbsize
         * initialized from the tdef. Then we parse changes and overwrite.  In the
         * end we have an array of lbsizes to use for the table.
         *
         */
        NEXT(lbid, 10);
        NEXT(lbsize, 5)+1;
        err = MIN(tdef->lc, MAX_TABLE_LABEL_COUNT);
        for(i=0; i<err; i++)
        {
            if(tdef->l[i].lbid == lbid)
            {
                e->lbsize[i] = (ushort)lbsize;
            }
        }
    }
    NEXT_TO(ushort, e->entryno, 8);

    err = pkt->err;
    if(!err && header)
    {
        /* Write the table header info. */
        row_init(&h, header);
        row_put_int(&h, t->epoch, SXM_SPORTS_TABLE_EPOCH);
        row_put_int(&h, (timeNow >= t->timeStamp) ? timeNow - t->timeStamp : 0, SXM_SPORTS_TABLE_AGE);
        row_put_int(&h, (int)tdef->tclass, SXM_SPORTS_TABLE_CLASS);
        if(haveTitle)
        {
            row_put_str(&h, e->buff,	SXM_SPORTS_TABLE_TITLE);
        }
        row_put_int(&h, get_season_status(service, a), SXM_SPORTS_TABLE_SEASON);
        row_put_int(&h,a->afid,SXM_SPORTS_TABLE_AFID);
        row_put_str(&h,a->name,SXM_SPORTS_TABLE_AFNAME);
        err = (uint)h.err;
    }

    /* error check */
    if(err)
    {
        e->link = service->freeExtRoot;
        service->freeExtRoot = e;
        DEBUG_EXT("(%p) Parse/header err=%d pkt->err=%d", req, err, pkt->err);
        return SXM_E_NOENT;
    }

    e->epoch = (ushort)epochNow;
    e->type = type;
    e->parse = parse_row;   /* the parse function */
    e->tidx = (ushort)tidx; /* the table index */
    e->afid = (ushort)afid;
    e->global = a->global;
    e->req = req;
    e->t = t;

    t->inuse++;
    e->link = req->ext;
    req->ext = e;
    *ext = (ptr)e;

    DEBUG_EXT("(Exit) req=%p", req);
    return SXM_E_OK;
}

static int begin_tdef(SportsService* service, SportsRequest* req, uint type,
    int p1, ptr* ext)
{
    SportsExtract* e;
    STable* tdef;
    uint id, ver;

    /* BSC (Broad Scope Change) is active, return 'busy' */
    if(req->bsc == TRUE)
    {
        return SXM_E_BUSY;
    }

    DEBUG_EXT("(Enter) req=%p req->afid=%d type=%d p1=%d", req, req->afid, type, p1);

    /* We need the table def. */
    id = SPORTS_TDEF_ID(p1);
    if(TBLOCK[id] == INVALID_ID)
    {
        DEBUG_EXT("(%p) no tdef: id=%d", req, id);
        return SXM_E_NOENT;
    }

    /* Guard against out-of-range index */
    if (TBLOCK[id] >= ARRAY_SIZE(TABLE))
    {
        DEBUG_EXT("invalid TABLE index (id=%d,TBLOCK[id]=%d)",  
            id, TBLOCK[id]);
        return SXM_E_NOENT;
    }

    /* The table def versions must match. */
    tdef = &TABLE[TBLOCK[id]];
    ver = SPORTS_TDEF_VER(p1);
    if(tdef->ver != ver)
    {
        DEBUG_EXT("(%p) version mismatch: tdef->ver=%d paramVer", req, tdef->ver, ver);
        return SXM_E_NOENT;
    }

    e = service->freeExtRoot;
    service->freeExtRoot = e->link;
    memset(e, 0, sizeof(*e));

    e->entryno = tdef->lc;
    e->tidx = (ushort)id;			/* table id. */
    e->type = type;
    e->parse = get_tdef;
    e->req = req;

    e->link = req->ext;
    req->ext = e;
    *ext = (ptr)e;

    DEBUG_EXT("(Exit) req=%p", req);
    return SXM_E_OK;
}

static int begin_nested_table(SportsService* service, SportsExtract* parentExt,
    uint type, int p1, ptr* ext, SXMSportsRow* header)
{
    Affiliation* a;
    SportsExtract* e;
    STable* tdef;
    SXMBitBuff *pkt;
    uint i, tid, tver, lbid, lbsize, haveTitle, known, err;
    SXMBitBuff bbuff;
    SportsRowHelper h;

    DEBUG_EXT("(Enter) parentExt=%p parentExt->afid=%d type=%d p1=%d header=%p", parentExt, parentExt->afid, type, p1, header);

    if(!header)
    {
        return SXM_E_FAULT;
    }

    if(!parentExt->t)
    {
        DEBUG_EXT("(%p) parent extract has no table.", parentExt);
        return SXM_E_NOENT;
    }

    if(ABLOCK[parentExt->afid] == INVALID_ID)
    {
        DEBUG_EXT("(%p) parent extract aff not present: afid=%d", parentExt, parentExt->afid);
        return SXM_E_NOENT;
    }

    /* See if it is a known referenced table. */
    known = (uint)(p1 & TREF_IS_KNOWN);
    p1 &= (int)(~known);				/* The rest is the byte offset. */

    /* Set up a pointer to the data.  The parent holds the data. */
    pkt = &bbuff;	/* On stack.  Keeps error handling easy.  We will copy to new extract if it all works out. */
    sxm_bitbuff_setup(pkt, (byte *)&HEAP[parentExt->t->heapix], parentExt->t->size);

    SKIPBYTES(p1);  /* Bump to reference table. */
    NEXT(tid, 10);
    NEXT(tver, 8);

    if(pkt->err)
    {
        ERROR_EXT("(%p) packet err: %d", parentExt, pkt->err);
        return SXM_E_NOENT;
    }

    if(TBLOCK[tid] == INVALID_ID)
    {
        DEBUG_EXT("(%p) no tdef for nested table: tid=%d", parentExt, tid);
        return SXM_E_NOENT;
    }

    /* Guard against out-of-range index */
    if (TBLOCK[tid] >= ARRAY_SIZE(TABLE))
    {
        DEBUG_EXT("invalid TABLE index (tid=%d,TBLOCK[tid]=%d)",  
            tid, TBLOCK[tid]);
        return SXM_E_NOENT;
    }

    /* The data instance and tdef versions must match. */
    tdef = &TABLE[TBLOCK[tid]];
    if(tdef->ver != tver)
    {
        DEBUG_EXT("(%p) version mismatch: tdef->ver=%d nestedVer=%d", parentExt, tdef->ver, tver);
        return SXM_E_NOENT;
    }

    /* Get the new extraction. */
    e = service->freeExtRoot;
    service->freeExtRoot = e->link;
    memset(e, 0, sizeof(*e));

    a = &AFF[ABLOCK[parentExt->afid]];
    make_known_table_map(tdef, a->spid, parentExt->tidx, e, known);

    NEXT(haveTitle, 1);
    if(haveTitle)
    {
        BAUDOT(e->buff, BAUDOT_MODE_START_WITH_LETTERS,
            SMALL_SYMCOUNT(MAX_TABLE_TITLE_CHAR), MAX_TABLE_TITLE_CHAR);
    }

    /* Get the overrides. */
    while(FLAG())
    {
        /* Parse lbsize overrides.  We make per a table parse instance of lbsize initialized from the tdef.
         *Then we parse changes and overwrite.  In the end we have an array of lbsizes to use for the table.
         */
        NEXT(lbid, 10);
        NEXT(lbsize, 5)+1;
        err = MIN(tdef->lc, MAX_TABLE_LABEL_COUNT);
        for(i=0; i<err; i++)
        {
            if(tdef->l[i].lbid == lbid)
            {
                e->lbsize[i] = (ushort)lbsize;
            }
        }
    }
    NEXT_TO(ushort, e->entryno, 8);

    err = pkt->err;
    if(!err)
    {
        /* Write the table header info. */
        row_init(&h, header);
        row_put_int(&h, parentExt->t->epoch, SXM_SPORTS_TABLE_EPOCH);
        row_put_int(&h, (int)(parentExt->t->timeStamp), SXM_SPORTS_TABLE_AGE);
        row_put_int(&h, tdef->tclass, SXM_SPORTS_TABLE_CLASS);
        if(haveTitle)
        {
            row_put_str(&h, e->buff, SXM_SPORTS_TABLE_TITLE);
        }
        row_put_int(&h, get_season_status(service, a), SXM_SPORTS_TABLE_SEASON);
        err = (uint)h.err;
    }

    if(err)
    {
        e->link = service->freeExtRoot;
        service->freeExtRoot = e;
        ERROR_EXT("(%p) Parse/header err=%d pkt->err=%d", parentExt, err, pkt->err);
        return SXM_E_NOENT;
    }

    /* Set up the remainder of the extract */
    e->epoch = parentExt->epoch;
    e->type = type | EXTRACT_NESTED;
    e->parse = parse_row;
    e->tidx = parentExt->tidx;
    e->afid = parentExt->afid;
    e->global = parentExt->global;
    e->req = parentExt->req;
    memcpy(&e->bbuff, &bbuff, sizeof(bbuff));

    /* Keep the table we just parsed and make the extract table pointer reference it. */
    e->t = &e->nestedTable;
    e->t->offset = (ushort)p1;
    e->t->heapix = INVALID_HEAPIX;  		/* Not used for the nested table.  The parent already has the data in memory. */
    e->t->epoch = parentExt->t->epoch;
    e->t->timeStamp = parentExt->t->timeStamp;
    e->t->tabno = (ushort)tid;
    e->t->tabver = (ushort)tver;

    parentExt->t->inuse++;

    e->parent = parentExt;
    e->link = parentExt->nested;
    parentExt->nested = e;
    *ext = (ptr)e;

    DEBUG_EXT("(Exit) parentExt=%p", parentExt);
    return SXM_E_OK;
}

static int begin_nested_tdef(SportsService* service, SportsExtract* parentExt, uint type,
    int p1, ptr* ext)
{
    SportsExtract* e;
    STable* tdef;
    uint id, ver;

    DEBUG_EXT("(Enter) parentExt=%p parentExt->afid=%d type=%d p1=%d", parentExt, parentExt->afid, type, p1);

    /* We need the table def. */
    id = SPORTS_TDEF_ID(p1);
    if(TBLOCK[id] == INVALID_ID)
    {
        DEBUG_EXT("(0x%p) no tdef: id=%d", parentExt, id);
        return SXM_E_NOENT;
    }

    /* Guard against out-of-range index */
    if (TBLOCK[id] >= ARRAY_SIZE(TABLE))
    {
        DEBUG_EXT("invalid TABLE index (id=%d,TBLOCK[id]=%d)",  
            id, TBLOCK[id]);
        return SXM_E_NOENT;
    }

    // The table def versions must match.
    tdef = &TABLE[TBLOCK[id]];
    ver = SPORTS_TDEF_VER(p1);
    if(tdef->ver != ver)
    {
        DEBUG_EXT("(%p) version mismatch: tdef->ver=%d paramVer", parentExt, tdef->ver, ver);
        return SXM_E_NOENT;
    }

    e = service->freeExtRoot;
    service->freeExtRoot = e->link;
    memset(e, 0, sizeof(*e));

    e->entryno = tdef->lc;
    e->tidx = (ushort)id;			/* table id. */
    e->type = type | EXTRACT_NESTED;
    e->parse = get_tdef;
    e->req = parentExt->req;

    e->parent = parentExt;
    e->link = parentExt->nested;
    parentExt->nested = e;
    *ext = (ptr)e;

    DEBUG_EXT("(Exit) parentExt=%p", parentExt);
    return SXM_E_OK;
}

static int begin_team_events(SportsService* service, SportsRequest* req, uint type,
    int p1, ptr* ext)
{
    SportsExtract* e;
    Affiliation* a;

    if(!req)
    {
        return SXM_E_FAULT;
    }

    /* BSC (Broad Scope Change) is active, return 'busy' */
    if(req->bsc == TRUE)
    {
        return SXM_E_BUSY;
    }

    DEBUG_EXT("(Enter) req=%p req->afid=%d type=%d p1=%d", req, req->afid, type, p1);

    /* Check the root affiliate of the request. */
    a = sxm_sports_shared_find_root(req);
    if(!a)
    {
        DEBUG_EXT("(%p) cannot find root aff", req);
        return SXM_E_NOENT;
    }

    e = service->freeExtRoot;
    service->freeExtRoot = e->link;
    memset(e, 0, sizeof(*e));

    e->type = type;
    /* We will search all schedule tables of all affiliates with the desired root id. */
    e->curAfid = (ushort)-1;
    e->curTidx = (ushort)-1;
    e->rootAfid = a->afid;
    e->teamId = (ushort)p1;
    e->parse = get_team_events;
    e->req = req;

    e->link = req->ext;
    req->ext = e;
    *ext = (ptr)e;

    DEBUG_EXT("(Exit) req=%p", req);

    return SXM_E_OK;
}

static int begin_teams(SportsService* service, SportsRequest* req, uint type, int p1, ptr* ext)
{
    SportsExtract* e;
    Affiliation* a;

    if(!req)
    {
        return SXM_E_FAULT;
    }

    /* BSC (Broad Scope Change) is active, return 'busy' */
    if(req->bsc == TRUE)
    {
        return SXM_E_BUSY;
    }

    DEBUG_EXT("(Enter) req=%p req->afid=%d type=%d p1=%d", req, req->afid, type, p1);

    /* Check the root affiliate of the request. */
    a = sxm_sports_shared_find_root(req);
    if(!a)
    {
        DEBUG_EXT("(%p) cannot find root aff", req);
        return SXM_E_NOENT;
    }

    if(a->global == INVALID_GDREF)
    {
        DEBUG_EXT("(%p) no gdref", req);
        return SXM_E_NOENT;
    }

    e = service->freeExtRoot;
    service->freeExtRoot = e->link;
    memset(e, 0, sizeof(*e));

    e->type = type;
    e->global = a->global;
    e->parse = get_team_list;
    e->req = req;

    e->link = req->ext;
    req->ext = e;
    *ext = (ptr)e;

    DEBUG_EXT("(Exit) req=%p", req);

    return SXM_E_OK;
}

/***************************************************************************//**
 *
 * Helper function to begin sports extraction.  Obtains service mutex
 * before begin processing proceeds.
 *
 * \note called in application thread context with service mutex held
 *
 **************************************************************************/
static int locked_sports_extract_begin(SportsService *service, ptr req, int type,
    int p1, ptr* ext, SXMSportsRow* header)
{

    int rc = SXM_E_UNSUPPORTED;
    BOOL valid;

    DEBUG_EXT("(Enter) req=%p", req);

    do{
        /* We always require a config.  It also provides the list of sports. */
        if(!SPORTS_HAVE_CONFIG())
        {
            DEBUG_EXT("(%p) no config", req);
            rc = SXM_E_NOENT;
            break;
        }

        /* Need a free extraction resource. */
        if(service->freeExtRoot == NULL)
        {
            DEBUG_EXT("(%p) no free extracts", req);
            rc = SXM_E_NOMEM;
            break;
        }

        /* extract sports list (no request needed) */
        if(type == SXM_SPORTS_LIST)
        {
            rc = begin_sports(service, type, ext);
            break;
        }

        /* All these that follow require an affiliate table to work. */
        if(!SPORTS_HAVE_AFTABLE())
        {
            DEBUG_EXT("(%p) no aff table", req);
            rc = SXM_E_NOENT;
            break;
        }

        /* leagues list (no request needed) */
        if(type == SXM_SPORTS_LEAGUES)
        {
            rc = begin_leagues(service, type, ext);
            break;
        }

        /* Need a request to define league for all the rest (checking for NULL). */
        if (!req)
        {
            ERROR_EXT("NULL req handle");
            rc = SXM_E_FAULT;
            break;
        }

        /* Validate the request handle */
        valid = request_handle_valid(service, req);
        if(valid == FALSE)
        {
            ERROR_EXT("(%p) invalid req handle", req);
            rc = SXM_E_INVAL;
            break;
        }

        /* Request for known leagues can be made before the league is gathered.  Check
         * here and cache afid if it is collected.
         */
        valid = request_has_affiliate(service, req);
        if(valid == FALSE)
        {
            DEBUG_EXT("(%p) affiliate not present yet", req);
            rc = SXM_E_NOENT;
            break;
        }

        /* begin table extraction */
        if(type == SXM_SPORTS_TABLE)
        {
            rc = begin_table(service, (SportsRequest*)req, type, p1, ext, header);
            break;
        }

        /* Sports Tree extraction */
        if(type == SXM_SPORTS_TREE)
        {
            rc = begin_league_tree(service, (SportsRequest*)req, type, ext);
            break;
        }

        /* extracting table definition */
        if(type == SXM_SPORTS_TDEF)
        {
            rc = begin_tdef(service, (SportsRequest*)req, type, p1, ext);
            break;
        }

        /* teams */
        if(type == SXM_SPORTS_TEAMS)
        {
            rc = begin_teams(service, (SportsRequest*)req, type, p1, ext);
            break;
        }

        /* events */
        if(type == SXM_SPORTS_TEAM_EVENTS)
        {
            rc = begin_team_events(service, (SportsRequest*)req, type, p1, ext);
            break;
        }

    }while(0);

    DEBUG_EXT("(Exit) req=%p rc=%d", req, rc);
    return rc;
}

/***************************************************************************//**
 *
 * Helper function to begin sports extraction.  Obtains service mutex
 * before begin processing proceeds.
 *
 **************************************************************************/
int sxm_sports_extract_begin(SportsService *service, ptr req, int type, int p1,
    ptr* ext, SXMSportsRow* header)
{
    int rc;
    LOCK(service->mutex);
    rc = locked_sports_extract_begin(service, req, type, p1, ext, header);
    UNLOCK(service->mutex);
    return rc;
}

int sxm_sports_extract_begin_nested(SportsService *service, ptr parentExt, int type, int p1, ptr* ext, SXMSportsRow* header)
{
    SportsExtract* pe = (SportsExtract*)parentExt;
    int rc = SXM_E_UNSUPPORTED;
    BOOL valid;

    LOCK(service->mutex);

    DEBUG_EXT("(Enter) parentExt=%p", parentExt);

    do{
        /* Verify the parent extraction. */
        valid = extract_handle_valid(service, parentExt);
        if(valid == FALSE)
        {
            ERROR_EXT("(%p) invalid parentExt handle", parentExt);
            rc = SXM_E_INVAL;
            break;
        }

        /* No new nested extracts are allowed if the associated request received bsc. */
        if(pe->req->bsc == TRUE)
        {
            DEBUG_EXT("(%p) parentExt updating", parentExt);
            rc = SXM_E_BUSY;
            break;
        }

        /* We require a config and affiliates. */
        if(!SPORTS_HAVE_CONFIG() || !SPORTS_HAVE_AFTABLE())
        {
            DEBUG_EXT("(%p) no config or aff table", parentExt);
            rc = SXM_E_NOENT;
            break;
        }

        /* Need an extraction resource. */
        if(service->freeExtRoot == NULL)
        {
            DEBUG_EXT("(%p) no free extracts", parentExt);
            rc = SXM_E_NOMEM;
            break;
        }

        /* The parent extraction must be a table type. */
        if(!(pe->type & SXM_SPORTS_TABLE))
        {
            rc = SXM_E_NOENT;
            break;
        }

        if(type == SXM_SPORTS_TABLE)
        {
            /* A nested table extract cannot have a nested parent. */
            if(pe->type & EXTRACT_NESTED)
            {
                DEBUG_EXT("(%p) parent is nested: pe->type=%d", parentExt, pe->type);
                rc = SXM_E_NOENT;
            }
            else
            {
                rc = begin_nested_table(service, (SportsExtract*)parentExt, type, p1, ext, header);
            }
            break;
        }

        if(type == SXM_SPORTS_TDEF)
        {
            rc = begin_nested_tdef(service, (SportsExtract*)parentExt, type, p1, ext);
            break;
        }
    }while(0);

    DEBUG_EXT("(Exit) parentExt=%p rc=%d", parentExt, rc);
    UNLOCK(service->mutex);
    return rc;
}

static int locked_sports_extract_extract_rxw(SportsService *service, SportsExtract* e,
    ptr out, int isRow)
{
    int rc;
    BOOL valid;

    DEBUG_EXT("(Enter) ext=%p isRow=%d", e, isRow);

    do{
        valid = extract_handle_valid(service, e);
        if(valid == FALSE)
        {
            rc = SXM_E_INVAL;
            break;
        }

        if(e->type & EXTRACT_REQUIRE_REQ)
        {

            // BSC (Broad Scope Change) is active, return 'busy'
            if(e->req->bsc == TRUE)
            {
                rc = SXM_E_BUSY;
                break;
            }
        } else
        {

            /* BSC (Broad Scope Change) is active, return 'busy' */
            if(service->bsc == TRUE)
            {
                rc = SXM_E_BUSY;
                break;
            }
        }

        if(e->err)
        {
            rc = SXM_E_ERROR;
            break;
        }

        if(!SPORTS_HAVE_CONFIG())
        {
            rc = SXM_E_NOENT;
            break;
        }

        e->isRow = (ushort)isRow;
        e->dest = out;
        rc = e->parse(service, e);
    } while(0);

    DEBUG_EXT("(Exit) ext=%p rc=%d", e, rc);
    return rc;
}

int sxm_sports_extract_extract_rxw(SportsService *service, ptr ext, ptr out, int isRow)
{
    int rc;
    LOCK(service->mutex);
    rc = locked_sports_extract_extract_rxw(service, (SportsExtract *)ext, out, isRow);
    UNLOCK(service->mutex);
    return rc;
}

static int free_extract(SportsService *service, SportsExtract** r, SportsExtract* e)
{
    SportsExtract* free = NULL;
    SportsExtract* keep = NULL;
    SportsExtract* item = NULL;
    SportsExtract* next = *r;
    Table* t;

    while(next)
    {
        item = next;
        next = next->link;
        if(item == e)
        {
            free = e;
        } else
        {
            item->link = keep;
            keep = item;
        }
    };

    *r = keep;

    if(free)
    {
        t = (free->type & EXTRACT_NESTED) ? free->parent->t : free->t;
        if(t && t->inuse)
        {
            t->inuse--;
        }
        free->link = service->freeExtRoot;
        service->freeExtRoot = free;
        return 1;
    }
    return 0;
}

static int locked_sports_extract_extract_end(SportsService *service, SportsExtract* e)
{
    int rc = SXM_E_OK;
    int ok;
    BOOL valid;

    DEBUG_EXT("(Enter) ext=%p", e);
    do{
        valid = extract_handle_valid(service, e);
        if(valid == FALSE)
        {
            rc = SXM_E_INVAL;
            break;
        }

        /* Our internal table extract may be active.  Clean it up if it is around. */
        if(e->type & SXM_SPORTS_TEAM_EVENTS)
        {
            if(e->curExt)
            {
                ok = free_extract(service, &(e->req->ext), e);
                if(!ok)
                {
                    /* keep going to get the primary extract. */
                }
            }
        }

        if(e->type & EXTRACT_NESTED)
        {
            ok = free_extract(service, &(e->parent->nested), e);
            if(!ok)
            {
                ERROR_EXT("Failed to find and free nested ext: e=%p", e);
                rc = SXM_E_ERROR;
                break;
            }
        }
        else
        {
            if(e->nested)
            {
                rc = SXM_E_BUSY;
                break;
            }

            if(e->type & EXTRACT_REQUIRE_REQ)
            {
                ok = free_extract(service, &(e->req->ext), e);
                if(!ok)
                {
                    ERROR_EXT("Failed to find and free require request ext: e=%p", e);
                    rc = SXM_E_ERROR;
                    break;
                }
            }
            else
            {
                if(e->t && e->t->inuse)
                {
                    e->t->inuse--;
                }
                e->link = service->freeExtRoot;
                service->freeExtRoot = e;
            }
        }
    }while(0);

    DEBUG_EXT("(Exit) ext=%p rc=%d", e, rc);
    return rc;
}

int sxm_sports_extract_extract_end(SportsService *service, ptr ext)
{
    int rc;
    LOCK(service->mutex);
    rc = locked_sports_extract_extract_end(service, (SportsExtract *)ext);
    UNLOCK(service->mutex);
    return rc;
}