/******************************************************************************/
/*                Copyright (c) Sirius XM Satellite Radio, Inc.               */
/*                            All Rights Reserved                             */
/*      Licensed Materials - Property of Sirius XM Satellite Radio, Inc.      */
/******************************************************************************/
/***************************************************************************//**
 * 
 * \file sxm_sports.c
 * \author Leslie French, Sergey Kotov
 * \date 8/20/2013
 *
 * Implementation of the SXM SDK for Sports
 *
 ******************************************************************************/

#define DEBUG_TAG "sports"

#include "sxm_sports_internal.h"

/** Status indicates if the service is running,
 * and if the service is Subscribed.
 */
static SXMStatus status = {SXM_SERVICE_STOPPED, SXM_SUBS_UNKNOWN};

/** A pointer to the service structure for Sports */
static SportsService *service = NULL;

/** A mutex used to ensure serial execution of the Service commands */
static pthread_mutex_t apimutex = PTHREAD_MUTEX_INITIALIZER;

/** DSI for Sports service */
static const ushort sports_dsi = 530;

/******************************************************************************
 *                                                                            *
 *            Internal Functions                                              *
 *            ==================                                              *
 *                                                                            *
 ******************************************************************************/

static void ServiceFree(SportsService *pService);
static SportsService *ServiceCreate(void);
static void sxm_sports_report_up(int type, int param, ptr dsuser);
static BOOL sxm_sports_is_request_handle_valid(ptr requestHandle);
static void print_tabledef(STable *t, int tabid);
#ifdef DSPORT_ENABLE_PRINT
static const char *get_label_text(int id);
#endif
static void print_table(Table *t, int tref, int idx);
static int sports_save_data(SXMIOStat *pStat, void *pArg);

/***************************************************************************//**
 *
 * The function releases all allocated resources used by Sports service
 *
 * \param[in] pService a pointer to the service structure
 *
 ******************************************************************************/
static void ServiceFree(SportsService* pService)
{
    uint i;
    int result;

    /* guard against NULL pointer argument */
    if (pService != NULL)
    {
        LOCK(pService->mutex);
        for(i=0; i<ARRAY_SIZE(pService->request); i++)
        {
            result = pthread_mutex_destroy(&pService->request[i].mutex);
            if (result != SXM_E_OK)
            {
                ERROR_API("Destroy request mutex error. i=%d result=%d", i, result);
            }
#ifndef SXM_DEBUG_PRODUCTION
            if(pService->request[i].sportId != INVALID_ID)
            {
                ERROR_API("Stranded request. "
                    "i=%u sportId=%u afname=%s addNotify=%u bsc=%u "
                    "delete=%u deleteAtEnd=%u inuse=%u",
                    i,
                    pService->request[i].sportId,
                    pService->request[i].afname,
                    pService->request[i].addNotify,
                    pService->request[i].bsc,
                    pService->request[i].rdelete,
                    pService->request[i].deleteAtEnd,
                    pService->request[i].inuse);
            }
#endif
        }

        UNLOCK(pService->mutex);

#ifndef SXM_DEBUG_PRODUCTION
        {
            uint count = 0;
            SportsExtract* e = pService->freeExtRoot;
            while(e)
            {
                count++;
                e = e->link;
            }

            if(count != ARRAY_SIZE(pService->extBuffer))
            {
                ERROR_API("Stranded extracts. "
                    "free=%d total=%d", count, ARRAY_SIZE(pService->extBuffer));
            }
        }
#endif
        result = pthread_mutex_destroy(&pService->mutex);
        if (result != SXM_E_OK)
        {
            ERROR_API("Destroy config mutex error:"
                " result=%d", result);
        }

        /* Free the cycle file. */
        if(pService->h)
        {
            sxe_free(pService->h);
        }
        sxe_free(pService);
    }

    service = NULL;
    status.service = SXM_SERVICE_STOPPED;
    status.substate = SXM_SUBS_UNKNOWN;
}

/***************************************************************************//**
 *
 * The function allocates resources used for Sports service
 *
 * \return a pointer to the service structure
 * \retval valid pointer on success
 * \retval NULL in case of error
 *
 ******************************************************************************/
static SportsService* ServiceCreate(void)
{
    SportsService* ret;
    int result;
    uint i;

    /* allocate the service structure */
    ret = (SportsService*)sxe_calloc(1, sizeof(SportsService));
    if (!ret)
    {
        ERROR_API("Failed to allocate SportsService");
        return NULL;
    }

    ret->ds.status = &status;

    /* initialize the service mutex */
    result = sxm_mutex_init(&ret->mutex);
    if (result != 0)
    {
        ERROR_API("Failed to init config mutex. result=%d", result);
        sxe_free(ret);
        return NULL;
    }

    do
    {
        /* initialize the request structure */
        for(i = 0, result = 0; i < ARRAY_SIZE(ret->request); i++)
        {
            ret->request[i].sportId = INVALID_ID;
            result = sxm_mutex_init(&ret->request[i].mutex);
            if (result)
            {
                ERROR_API("Failed to init request mutex. i=%d result=%d", i, result);
                break;
            }
        }

        /* quit if mutex initialization failed */
        if(result)
        {
            break;
        }

        /* initialze the extraction buffers */
        ret->freeExtRoot = ret->extBuffer;
        for(i = 0; i < (ARRAY_SIZE(ret->extBuffer) - 1); i++)
        {
            ret->extBuffer[i].link = &ret->extBuffer[i + 1];
        }

        /* set BSC active, this will force a 'busy' state to be returned on extractions
           at service start until the BSC data are available */
        ret->bsc = TRUE;

        return ret;
    }
    while(0);

    /* there was an error, release the service structure */
    ServiceFree(ret);
    return NULL;
}

/***************************************************************************//**
 *
 * This function calls the defined callback function for the
 * function sxm_sports_start.
 * The module-level callback is described in the "Error Detection and Reporting"
 * section of the SX-9845-0340 Introduction Design Guide.
 *
 * \param[in] type      Item passed to the callback function.
 * \param[in] param     Item passed to the callback function.
 * \param[in] dsuser    Optional, not used here.
 *
 ******************************************************************************/
static void sxm_sports_report_up(int type, int param, ptr dsuser)
{
    CALLB_API("(type: %d, param: %d, dsuser: %p)", type, param, dsuser);
    service->callback(type, param);
}

/***************************************************************************//**
 *
 * Prints out CFile statistics for debug purposes.
 *
 * \param[in] rb CFile root block pointer
 *
 ********************************************************************************/
static void sxm_sports_cfile_stats(SXMCFileRootBlock* rb)
{
    uint i, run, free;
    ushort bin[33];
    const uint heap_block_count = rb->sections[SPORTS_SECTION_HEAP].length;

    memset(bin, 0, sizeof(bin));
    for(i=0, run=0, free=0; i<heap_block_count; i++)
    {
        if(BITP(rb->map, i))
        {
            free++;
            run++;
        }
        else
        {
            if(run)
            {
                if(run<=31) {bin[run]++;}
                else        {bin[32]++;}
            }
            run = 0;
        }
    }

    if(run)
    {
        if(run<=31) {bin[run]++;}
        else        {bin[32]++;}
    }

    DEBUG_API("SportsHeap");
    DEBUG_API("----------");
    DEBUG_API("Tot:    %d", heap_block_count);
    DEBUG_API("Free:   %d", free);
    DEBUG_API("Used:   %d", heap_block_count - free);
    DEBUG_API("%%used:  %d", ((heap_block_count - free) * 100) / heap_block_count );

    for(i=1;i<=31; i++)
    {
        DEBUG_API("%2d |Bin: %d   ", i, bin[i]);
    }
    DEBUG_API("%2d+|Bin: %d   ", i, bin[i]);
}

/***************************************************************************//**
 *
 * This function validates collection request handle
 *
 * \param[in] requestHandle the collection handle for the request
 *
 * \return boolean value indicating handle validity
 * \retval TRUE if handle is valid
 * \retval FALSE if handle is invalid
 *
 ******************************************************************************/
static BOOL sxm_sports_is_request_handle_valid(ptr requestHandle)
{
    /* Check if pointer lies in the requests handles array */
    if (requestHandle >= (ptr)&service->request[0] &&
        requestHandle <= (ptr)&service->request[SXM_SPORTS_NUM_REQUESTS_MAX - 1])
    {
        return TRUE;
    }

    return FALSE;
}

/******************************************************************************
 *                                                                            *
 *            Application Interface                                           *
 *            =====================                                           *
 *                                                                            *
 ******************************************************************************/

/***************************************************************************//**
 *
 * The start routine starts the Sports service.
 *
 * First it checks to see if the service has already started. If not then it
 * initializes information about the Sports service, then starts the Sports
 * protocol decoder and low level functions.
 *
 * \param[in] callback a routine that will receive Sports
 *                     module-level event indications
 * \param[in] debugLevel the debug level for the Sports module
 *
 * \return SXe error code
 * \retval SXM_E_OK Success
 * \retval SXM_E_STATE Already started
 * \retval SXM_E_NOMEM Memory allocation failed
 * \retval SXM_E_FAULT NULL callback parameter
 * \retval SXM_E_PIPE Radio error or failed to create cycle file
 * \retval SXM_E_THREAD Thread creation Failed
 * \retval SXM_E_ERROR General error
 *
 ******************************************************************************/
int sxm_sports_start(SPORTS_EVENT_CALLBACK callback, int debugLevel)
{
    int stoplli = 0;
    int rc = SXM_E_OK;
    SXMStatus sxe_status;

    /* serialize the Sports Service APIs */
    LOCK(apimutex);
    ENTER_START_API(debugLevel, "(callback: %p, debugLevel: %d)", callback, debugLevel);

    /* if the service is already started, return a 'state' error */
    if (status.service != SXM_SERVICE_STOPPED)
    {
        ERROR_API("Already started");
        LEAVE_START_API(debugLevel, "%s", sxm_sdk_format_result(SXM_E_STATE));
        UNLOCK(apimutex);
        return SXM_E_STATE;
    }

    /* if the SXM SDK is not started, return a 'state' error */
    sxm_sdk_status(&sxe_status);
    if (sxe_status.service != SXM_SERVICE_OK)
    {
        ERROR_API("SDK not initialized");
        LEAVE_START_API(debugLevel, "%s", sxm_sdk_format_result(SXM_E_STATE));
        UNLOCK(apimutex);
        return SXM_E_STATE;
    }
    else
    {
        sxm_sxi_status(&sxe_status);
        if (sxe_status.service != SXM_SERVICE_OK)
        {
            ERROR_API("Link not initialized");
            LEAVE_START_API(debugLevel, "%s", sxm_sdk_format_result(SXM_E_STATE));
            UNLOCK(apimutex);
            return SXM_E_STATE;
        }
    }

    /* a call back function must be specified */
    if (callback == NULL)
    {
        ERROR_API("No callback provided");
        LEAVE_START_API(debugLevel, "%s", sxm_sdk_format_result(SXM_E_FAULT));
        UNLOCK(apimutex);
        return SXM_E_FAULT;
    }

    /* create the service object */
    service = ServiceCreate();
    if (!service)
    {
        ERROR_API("Failed to create service");
        LEAVE_START_API(debugLevel, "%s", sxm_sdk_format_result(SXM_E_NOMEM));
        UNLOCK(apimutex);
        return SXM_E_NOMEM;
    }

    /* set the debug level */
    service->ds.debug_level = (int)debugLevel;

    DEBUG_API("sizeof(SportsService)=%d", sizeof(SportsService));
    DEBUG_API("sizeof(SportSave)=%d SPORT_BLOCKS=%d", sizeof(SportSave), SPORT_BLOCKS);

    /*  create the sports service object */
    service->callback = callback;
    service->ds.status = &status;
    service->ds.maxau = MAX_SPORTS_AU_SIZE;
    service->ds.report = &sxm_sports_report_up;
    service->ds.complete = &sxm_sports_complete_au;
    service->ds.iom.pFunc = &sports_save_data;
    service->ds.iom.tQuota = sizeof(SportSave);
    service->ds.dsis[0].dsi = sports_dsi;
    BITSALL(service->ds.dsis[0].filter);
    strcpy(service->ds.service, SPORTS_SERVICE_NAME);

    /* Initialize the Auto Save On Stop */
    service->ds.auto_save_on_stop = TRUE;
    service->sections_update = (byte)~SPORTS_SECTIONS_ALL;

    /* load and validate the persisted service data (cycle file) */
    rc = sxm_sports_protocol_start(service);
    if(rc == SXM_E_OK)
    {
        /* start the Low Level Interface (LLI) */
        rc = sxm_sports_lli_start(service);
        if(rc == SXM_E_OK)
        {
            /* initialize the timer */
            rc = sxm_lli_timer(service->lli, sxm_sports_timer, SPORTS_TIMER_MINUTES);
            if(rc == SXM_E_OK)
            {
                sxm_sports_cfile_stats(&service->h->g);
                sxm_sports_extract_init(service);
            }
            else
            {
                ERROR_API("Failed to start lli timer. rc=%d", rc);
                stoplli = sxm_sports_lli_stop();
                if (stoplli != SXM_E_OK)
                    ERROR_API("Stop LLI error. rc=%d", stoplli);
                ServiceFree(service);
                service = NULL;
            }
        }
        else
        {
            ERROR_API("Failed to start LLi. rc=%d", rc);
            ServiceFree(service);
            service = NULL;
        }
    }
    else
    {
        ERROR_API("Failed to start protocol. rc=%d", rc);
        ServiceFree(service);
        service = NULL;
    }

    LEAVE_START_API(debugLevel, "%s", sxm_sdk_format_result(rc));
    UNLOCK(apimutex);

    return rc;
}

/***************************************************************************//**
 *
 * This function removes the engine-cycle file.
 *
 * \retval SXM_E_OK     Request accepted.
 * \retval SXM_E_STATE  Service is active.
 * \retval SXM_E_NOENT  Cycle file does not exist
 *
 * \note  If the service is inactive, the cycle file is removed if it exists.
 *
 ********************************************************************************/
SXESDK_API int sxm_sports_cfile_clean(void) {

    int rc = SXM_E_OK;

    // trace
    ENTER_API("");

    // serialize this API
    LOCK(apimutex);

    // service must be inactive, otherwise return "state" error
    if (status.service != SXM_SERVICE_STOPPED) {
        rc = SXM_E_STATE;
    }
    else {
        // remove the cycle file
        rc = sxm_file_remove('T', SPORTS_SERVICE_NAME, SPORTS_CFILE_NAME);
        if (rc == -1) {
            rc = SXM_E_NOENT;
        }
        else {
            rc = SXM_E_OK;
        }
    }
    UNLOCK(apimutex);

    LEAVE_API("%s", sxm_sdk_format_result(rc));
    return rc;
}

/***************************************************************************//**
 *
 * This function passes the current state and subscription status of the service
 * to the OEM developer.
 *
 * First validate the provided pointer. If it is good then populate it with the
 * current status information. Possible values are shown in sxm_common.h
 *
 * \param[out] pStatus pointer to the status structure
 *
 * \return SXe error code
 * \retval SXM_E_OK the status structure SXMStatus
 *                  has been successfully updated
 * \retval SXM_E_FAULT NULL pStatus parameter
 *
 ******************************************************************************/
int sxm_sports_status(SXMStatus *pStatus)
{
    int rc = SXM_E_FAULT;

    if (pStatus != NULL)
    {
        /* fill the caller-supplied memory with the current status */
        memcpy(pStatus, &status, sizeof(SXMStatus));
        rc = SXM_E_OK;
    }

    return rc;
}

/***************************************************************************//**
 *
 * This function stops the Sports service and releases all allocated resources.
 *
 * \return SXe error code
 * \retval SXM_E_OK Success
 * \retval SXM_E_STATE Service has not been started
 * \retval SXM_E_BUSY Attempt to stop the service from its own callback
 *
 ******************************************************************************/
int sxm_sports_stop(void)
{
    int rc = SXM_E_OK;
    uint debugLevel = DEBUG_VAR;

    LOCK(apimutex);

    ENTER_API("");

    /* service can only be stopped in OK state */
    if (status.service != SXM_SERVICE_OK)
    {
        ERROR_API("Not running");
        rc = SXM_E_STATE;
    }
    else
    {
        status.service = SXM_SERVICE_STOPPING;
        UNLOCK(apimutex);
        rc = sxm_sports_lli_stop();
        LOCK(apimutex);
        if (rc == SXM_E_OK)
        {
            LOCK(service->mutex);
            sxm_sports_extract_uninit();
            sxm_sports_cfile_stats(&service->h->g);
            sxm_sports_protocol_stop();
            status.service = SXM_SERVICE_STOPPED;
            UNLOCK(service->mutex);

            ServiceFree(service);
            service = NULL;

            /* service is stopped */
            status.service = SXM_SERVICE_STOPPED;
            status.substate = SXM_SUBS_UNKNOWN;

        } else
        {
            status.service = SXM_SERVICE_OK;
        }
    }

    LEAVE_START_API(debugLevel, "%s", sxm_sdk_format_result(rc));
    UNLOCK(apimutex);

    return rc;
}

/***************************************************************************//**
 *
 * Creates a collection request.  
 *
 * \param[in] league league name
 * \param[in] notification callback on collection update
 * \param[in] usercx opaque user context for callback
 * \param[out] pRequestHandle a pointer to a variable
 *                            to hold the created request handle
 *
 * \return SXe error code
 * \retval SXM_E_OK Success
 * \retval SXM_E_STATE Service has not been started
 * \retval SXM_E_FAULT NULL parameter
 * \retval SXM_E_INVAL Invalid parameter (out of range)
 * \retval SXM_E_BUSY Maximum requests active
 *
 * \note
 * Multilple requests (up to SXM_SPORTS_NUM_REQUESTS_MAX) are allowed to
 * be outstanding at any particular point in time.  This is in contrast
 * to other SXe implemented data services which allow only a single request
 * at a time.
 *
 * \note
 * Requests made using the Standard Interface specify the string name
 * of a well know league, for example, "nfl".
 *
 * \par
 * Requests made using the Extended Interface specify the string
 * representation of the numeric AFID with a '#' character prefix. For
 * example, a request for Affilation ID 123 would take the form: "#123".
 *
 ******************************************************************************/
int sxm_sports_request(const char *pLeague,
                       SPORTS_REQUEST_CALLBACK pNotification,
                       ptr pUsercx,
                       ptr *ppRequestHandle)
{
    int rc;

    /* serialize the usage of the service APIs */
    LOCK(apimutex);

    ENTER_API("");

    /* if the service is not started, return 'state' error */
    if (status.service != SXM_SERVICE_OK)
    {
        rc = SXM_E_STATE;
    }

    /* if the pointer arguments are NULL, return 'fault' error */
    else if (!pLeague || !pNotification || !ppRequestHandle)
    {
        rc = SXM_E_FAULT;
    }
    else
    {
        /* add the request for the designated league */
        rc = sxm_sports_request_add(service,
                                    pLeague,
                                    pNotification,
                                    pUsercx,
                                    ppRequestHandle);
    }

    LEAVE_API("%s", sxm_sdk_format_result(rc));

    /* release the APIs lock */
    UNLOCK(apimutex);

    return rc;
}

/***************************************************************************//**
 *
 * Removes existing collection request
 *
 * The routine will remove the notifications for that league.
 * Data are still being collected within the SDK and
 * will continue to be available to any subsequent collection request.
 *
 * \param[in] requestHandle the request handle to be removed
 *
 * \return SXe error code
 * \retval SXM_E_OK Success
 * \retval SXM_E_STATE Service has not been started
 * \retval SXM_E_FAULT NULL requestHandle parameter
 * \retval SXM_E_INVAL Invalid request handle
 * \retval SXM_E_BUSY Attempt to remove request from the service callback
 *                    or request currently extracting
 *
 ******************************************************************************/
extern int sxm_sports_remove(ptr requestHandle)
{
    int rc;

    /* serialize the Sports Serviice APIs */
    LOCK(apimutex);

    ENTER_API("");

    /* service must be active */
    if (status.service != SXM_SERVICE_OK)
    {
        rc = SXM_E_STATE;
    }

    /* NULL request handle is an error */
    else if (!requestHandle)
    {
        rc = SXM_E_FAULT;
    }

    /* handle address is not NULL, make sure it is valid */
    else if (sxm_sports_is_request_handle_valid(requestHandle) == FALSE)
    {
        rc = SXM_E_INVAL;
    }

    /* sport ID must be valid */
    else if (((SportsRequest*)requestHandle)->sportId == INVALID_ID)
    {
        rc = SXM_E_INVAL;
    }
    else
    {
        /* try to remove this request */
        rc = sxm_sports_request_remove(service, requestHandle);
        if (rc == SXM_E_NOENT)
        {
            /* Request was not found, means handle was not valid */
            rc = SXM_E_INVAL;
        }
    }

    LEAVE_API("%s", sxm_sdk_format_result(rc));
    UNLOCK(apimutex);
    return rc;
}

/***************************************************************************//**
 *
 * Begins a Sports database extraction
 *
 * The Standard Interface comprises two stages.  The first is to enumerate
 * all the affiliations below the top-level league. The second stage is to
 * process the contents of any tables for those affiliations.
 * Both stages follow the usual begin, extract, end cycle.
 *
 * \param[in] requestHandle request handle returned from sxm_sports_request()
 * \param[in] extType a type of extraction being requested
 * \param[in] extIndex an index value determined by the type field
 * \param[out] pExtHandle points to a returned pointer to the extraction object
 * \param[out] pData a structure that is filled in with values
 *
 * \return SXe error code
 * \retval SXM_E_OK extraction has been started
 * \retval SXM_E_FAULT NULL pointer is provided
 * \retval SXM_E_INVAL invalid parameters provided or invalid
 *                     sports version.
 * \retval SXM_E_STATE service is not started
 * \retval SXM_E_BUSY broad scope data update is in progress
 * \retval SXM_E_NOMEM Memory not available to complete extractions
 * \retval SXM_NOENT no entry to process, e.g. no affiliate table or
 *                   config information to begin extraction.
 *
 **************************************************************************/
int sxm_sports_begin(ptr requestHandle,
                     int extType,
                     int extIndex,
                     ptr* pExtHandle,
                     SXMSportsRow *pData)
{
    int rc;

    LOCK(apimutex);
    ENTER_API("");

    /* service must be active */
    if (status.service != SXM_SERVICE_OK)
    {
        rc = SXM_E_STATE;
    }

    /* pointers must not be NULL */
    else if (!pExtHandle || !pData)
    {
        rc = SXM_E_FAULT;
    }
    else
    {
        /* begin the extraction */
        rc = sxm_sports_extract_begin(service,
                                      requestHandle,
                                      extType,
                                      extIndex,
                                      pExtHandle,
                                      pData);
    }

    LEAVE_API("%s", sxm_sdk_format_result(rc));
    UNLOCK(apimutex);
    return rc;
}

/***************************************************************************//**
 *
 * Begins a nested Sports extraction
 *
 * \param[in] parentExtHandle a handle to the parent extraction object
 * \param[in] type a type of extraction being requested
 * \param[in] index an index value determined by the type field
 * \param[out] pExtHandle a returned pointer to the extraction object
 * \param[out] pRowData a structure that is filled in with values
 *
 * \return SXe error code
 * \retval SXM_E_OK extraction has been started.
 * \retval SXM_E_BUSY Broad-scope data change in progress
 * \retval SXM_E_FAULT NULL pointer is provided
 * \retval SXM_E_INVAL invalid parameter encountered
 * \retval SXM_E_STATE service in wrong state
 * \retval SXM_E_NOMEM Memory not available to complete extractions
 * \retval SXM_E_NOENT no data available to extract
 *
 * \note
 * The handle is for a current extraction, not the original request.
 * The returned exthandle may then by used to retrieve the rows from
 * the referenced table.
 *
 **************************************************************************/
int sxm_sports_begin_nested(ptr parentExtHandle,
                            int type, int index,
                            ptr* pExtHandle,
                            SXMSportsRow* pRowData)
{
    int rc;

    LOCK(apimutex);
    ENTER_API("");

    if (status.service != SXM_SERVICE_OK)
    {
        rc = SXM_E_STATE;
    }
    else if (!parentExtHandle || !pExtHandle || !pRowData)
    {
        rc = SXM_E_FAULT;
    }
    else
    {
        rc = sxm_sports_extract_begin_nested(service,
                                             parentExtHandle,
                                             type, index,
                                             pExtHandle, pRowData);
    }

    LEAVE_API("%s", sxm_sdk_format_result(rc));
    UNLOCK(apimutex);
    return rc;
}

/***************************************************************************//**
 *
 * The routine returns the data for the next affiliation in the tree,
 * or row in the table.
 *
 * \param[in] extHandle a handle to the extraction request.
 * \param[out] pRowData a pointer to an SXM Sports Row object that will be
 *                   filled up with data.
 *
 * \return SXe error code
 * \retval SXM_E_OK data has been extracted.
 * \retval SXM_E_STATE service is not started.
 * \retval SXM_E_FAULT NULL parameter.
 * \retval SXM_E_NOENT No entry found.
 * \retval SXM_E_INVAL exthandle or data parameter is invalid
 * \retval SXM_E_BUSY Broad scope update in progress
 * \retval SXM_E_ERROR General error processing extraction
 *
 ********************************************************************************/
int sxm_sports_extract_row(ptr extHandle, SXMSportsRow* pRowData)
{
    int rc;

    LOCK(apimutex);
    ENTER_API("");

    if (status.service != SXM_SERVICE_OK)
    {
        rc = SXM_E_STATE;
    }
    else if (!extHandle || !pRowData)
    {
        rc = SXM_E_FAULT;
    }
    else
    {
        rc = sxm_sports_extract_extract_rxw(service, extHandle, (ptr)pRowData, 1);
    }

    LEAVE_API("%s", sxm_sdk_format_result(rc));
    UNLOCK(apimutex);
    return rc;
}

/***************************************************************************//**
 *
 * Extracts sports raw object
 *
 * \param[in] extHandle a handle to the extraction request.
 * \param[out] pRawData a pointer to an SXM Sports Raw object that will be
 *                      filled up with data.
 *
 * \return SXe error code
 * \retval SXM_E_OK data has been extracted.
 * \retval SXM_E_STATE service is not started.
 * \retval SXM_E_FAULT NULL parameter.
 * \retval SXM_E_NOENT No entry found.
 * \retval SXM_E_INVAL Invalid parameter
 * \retval SXM_E_BUSY Broad-scope data change in progress
 * \retval SXM_E_ERROR General service error
 *
 * \note Table notifications will be delivered in exactly the same way
 *       as in the Standard Interface.  However, the Row structure cannot
 *       be used to extract the data.  The sxm_sports_extract_raw routine
 *       will return data for a table along with the information to decode
 *       the values.
 *
 ********************************************************************************/
int sxm_sports_extract_raw(ptr extHandle, SXMSportsRaw *pRawData)
{
    int rc;

    LOCK(apimutex);
    ENTER_API("");

    if (status.service != SXM_SERVICE_OK)
    {
        rc = SXM_E_STATE;
    }
    else if (!extHandle || !pRawData)
    {
        rc = SXM_E_FAULT;
    }
    else
    {
        rc = sxm_sports_extract_extract_rxw(service, extHandle,
                                            (ptr)pRawData, 0);
    }

    LEAVE_API("%s", sxm_sdk_format_result(rc));
    UNLOCK(apimutex);
    return rc;
}

/***************************************************************************//**
 *
 * The routine terminates the extraction, and potentially unlocks
 * the data for update.
 *
 * \param[in] extHandle a handle to the extraction request.
 *
 * \return SXe error code
 * \retval SXM_E_OK Extraction ended successfully
 * \retval SXM_E_STATE Service is not started
 * \retval SXM_E_FAULT ext handle is NULL
 * \retval SXM_E_INVAL ext handle is invalid
 *
 * \note 
 * Each extraction, including nested extractions, must have a
 * corresponding call to sxm_sports_end with the correct exthandle.
 * All extractions for a request must be ended before any data for
 * that request will be updated.
 *
 ********************************************************************************/
int sxm_sports_end(ptr extHandle)
{
    int rc;

    LOCK(apimutex);
    ENTER_API("");

    if (status.service != SXM_SERVICE_OK)
    {
        rc = SXM_E_STATE;
    }
    else if (!extHandle)
    {
        rc = SXM_E_FAULT;
    }
    else
    {
        rc = sxm_sports_extract_extract_end(service, extHandle);
    }

    LEAVE_API("%s", sxm_sdk_format_result(rc));
    UNLOCK(apimutex);
    return rc;
}

/***************************************************************************//**
 *
 * Retrieves current version of sports tables configuration
 *
 * \param[out] pVersion a pointer to the returned DB version
 *
 * \return SXe error code
 * \retval SXM_E_OK DB version is retrieved and populated into pVer
 * \retval SXM_E_FAULT NULL parameter
 * \retval SXM_E_STATE Service in wrong state
 * \retval SXM_E_NOENT No version available
 *
 ************************************************************************/
int sxm_sports_get_version(uint *pVersion)
{
    int ret = SXM_E_OK;

    /* serialize the Sports Service APIs */
    LOCK(apimutex);

    /* trace */
    ENTER_API("(pVersion: %p)", pVersion);

    /* Return pointer shall not be NULL */
    if (!pVersion)
    {
        ret = SXM_E_FAULT;
        ERROR_API("NULL parameter");
    }
    /* service must active, otherwise return "state" error */
    else if ( status.service != SXM_SERVICE_OK )
    {
        ret = SXM_E_STATE;
        ERROR_API("Service not started");
    }
    else if (!service->h)
    {
        ret = SXM_E_NOENT;
        DEBUG_API("No version available");
    }
    else
    {
        *pVersion = service->h->g.sections[SPORTS_SECTION_SPORTS].version;

        if (*pVersion == INVALID_VERSION)
        {
            ret = SXM_E_NOENT;
            DEBUG_API("No version available");
        }
        else
        {
            DEBUG_API("version=%d", *pVersion);
        }
    }

    LEAVE_API("%s", sxm_sdk_format_result(ret));
    UNLOCK(apimutex);

    return ret;
}

/***************************************************************************//**
 *
 * This function can be used to aid in debugging
 * by setting the desired debug level for the service.
 * This will enable Trace information.
 *
 * \param[in] debugLevel the debug level for the module
 *
 * \return SXe error code
 * \retval SXM_E_OK Success
 * \retval SXM_E_STATE Service has not been started
 *
 ******************************************************************************/
int sxm_sports_set_debug(int debugLevel)
{
    int rc;

    if (SXM_SERVICE_OK != status.service)
    {
        rc = SXM_E_STATE;
    }
    else
    {
        service->ds.debug_level = (uint)debugLevel;
        DEBUG_API("debugLevel=%d", debugLevel);
        rc = SXM_E_OK;
    }

    return rc;
}

/***************************************************************************//**
 *
 * This function reports the debug level for the service.
 *
 * \param[out] pDebugLevel returned debug level for the module
 *
 * \return SXe error code
 * \retval SXM_E_OK Success
 * \retval SXM_E_STATE Service has not been started
 * \retval SXM_E_FAULT NULL pDebugLevel parameter
 *
 ******************************************************************************/
int sxm_sports_get_debug(int *pDebugLevel)
{
    int rc;

    ENTER_API("(pDebugLevel: %p)", pDebugLevel);

    if (SXM_SERVICE_OK != status.service)
    {
        rc = SXM_E_STATE;
    }
    else if (NULL == pDebugLevel)
    {
        rc = SXM_E_FAULT;
    }
    else
    {
        *pDebugLevel = (int)service->ds.debug_level;
        DEBUG_API("debugLevel: %d", *pDebugLevel);
        rc = SXM_E_OK;
    }

    LEAVE_API("%s", sxm_sdk_format_result(rc));
    return rc;
}

/***************************************************************************//**
 * The routine flushes any cached sports service data required to be persisted over
 * power-cycles to the persisted file system such as non-volatile memory (NVM). 
 *
 * \param[in,out] pStat the I/O statistics to update
 * \param[in] pArg the callback argument
 *
 * \retval  SXM_E_OK     Data has been written to file system successfully.
 * \retval  SXM_E_STATE  Service data not available
 * \retval  SXM_E_PIPE   Error writing service data to file system
 *
 ********************************************************************************/
static int sports_save_data(SXMIOStat *pStat, void *pArg)
{
    int rc;

    UNUSED_VAR(pArg);
    LOCK(apimutex);
    ENTER_LLI("pStat: %p, pArg: %p", pStat, pArg);

    /* check service state */
    if (status.service != SXM_SERVICE_OK)
    {
        ERROR_API("service unavailable");
        rc = SXM_E_STATE;
    }
    else
    {
        /* sync to prevent any changes during commit */
        LOCK(service->mutex);

        /* save to NVM any changed service data since last commit */
        rc = sxm_sports_commit(pStat);

        /* release the lock */
        UNLOCK(service->mutex);
    }

    LEAVE_LLI("%s", sxm_sdk_format_result(rc));
    UNLOCK(apimutex);
    return rc;
}

/***************************************************************************//**
 *
 * The sxm_sports_save_data routine flushes any cached sports service data
 * required to be persisted over power-cycles to the persisted file system 
 * such as non-volatile memory (NVM). 
 *
 * \retval  SXM_E_OK     Data has been written to file system successfully.
 * \retval  SXM_E_STATE  Service data not available
 * \retval  SXM_E_PIPE   Error writing service data to file system
 *
 ********************************************************************************/
int sxm_sports_save_data(void)
{
    return sports_save_data(NULL, NULL);
}

/*********************************************************************//**
 *
 * The sxm_sports_set_auto_save function allows the application to enable 
 * or disable the sxm_sports_stop routine from writing the cached sports service data
 * to the file system.
 *
 * \note if the application disables writing of sports cached data during the stop function,
 * then the application shall be responsible for persisting this data across service 
 * restarts and power cycling using the sxm_sports_save_data api call.
 *
 * \param[in]            auto_save_on_stop controls the autosave of cached sports
 *                       service data on call to sxm_sports_stop().
 *
 * \retval  SXM_E_OK     Request to save data on stop is successfull.
 * \retval  SXM_E_STATE  Service data not available
 *
************************************************************************/
int sxm_sports_set_auto_save(BOOL auto_save_on_stop)
{
    int ret = SXM_E_OK;

    /* serialize this API */
    LOCK(apimutex);

    /* trace */
    ENTER_API("");

    /* service must active, otherwise return "state" error */
    if ((status.service != SXM_SERVICE_OK))
    {
        ERROR_API("Service not started");
        ret = SXM_E_STATE;
    }
    else
    {
        service->ds.auto_save_on_stop = auto_save_on_stop;
    }

    LEAVE_API("%s", sxm_sdk_format_result(ret));
    UNLOCK(apimutex);
    return ret;
}

/************************************************************************
 *                                                                      *
 *            Debug Print                                               *
 *            =====================                                     *
 *                                                                      *
 ************************************************************************/
#ifdef DSPORT_ENABLE_PRINT

/**************************************************************************//**
 *
 * Function to print the current version of the SPORTS table, aka
 * service configuration table.
 *
 * \return NONE
 *
 *****************************************************************************/
void print_sports(void)
{
    uint i, j;
    char txt[256];
    Sport *s;

    /* make sure there is a SPORTS table to print */
    if (service->h->g.sections[SPORTS_SECTION_SPORTS].version == INVALID_VERSION)
    {
        sxm_file_log("No sport table: cfgver=%d",
                service->h->g.sections[SPORTS_SECTION_SPORTS].version);
        return;
    }

    sxm_file_log("Sport Table: cfgver=%d",
            service->h->g.sections[SPORTS_SECTION_SPORTS].version);
    sxm_file_log("---------------------------------");
    sxm_file_log("mapindex:mapvalue - name|id|tableids");
    for (i = 0; i <= MAX_SPORT_ID; i++)
    {
        if (SBLOCK[i] == INVALID_ID)
        {
            continue;
        }
        if (SBLOCK[i] >= ARRAY_SIZE(SPORT))
        {
            sxm_file_log("%3d:%3d - Invalid map entry", i, SBLOCK[i]);
            continue;
        }

        s = &SPORT[SBLOCK[i]];
        txt[0] = '\0';
        for (j = 0; j < s->tno; j++)
        {
            char junk[32];
            sprintf(junk, "%3d ", s->tables[j]);
            strcat(txt, junk);
        }

        sxm_file_log("%3d:%3d - %32s|%d|%s", i, SBLOCK[i], s->name, i, txt);
    }

}

void print_affiliates(void)
{
    int i;
    Affiliation *a;

    if (service->h->g.sections[SPORTS_SECTION_SPORTS].version == INVALID_VERSION)
    {
        sxm_file_log("No Affiliate table(no config): cfgver=%d",
                service->h->g.sections[SPORTS_SECTION_SPORTS].version);
        return;
    }

    if (service->h->g.sections[SPORTS_SECTION_AFFS].version == INVALID_VERSION)
    {
        sxm_file_log("No Affiliate table(no adver): cfgver=%d adver=%d",
                service->h->g.sections[SPORTS_SECTION_SPORTS].version,
                service->h->g.sections[SPORTS_SECTION_AFFS].version);
        return;
    }

    sxm_file_log("Affiliation Table: cfgver=%d adver=%d",
            service->h->g.sections[SPORTS_SECTION_SPORTS].version,
            service->h->g.sections[SPORTS_SECTION_AFFS].version);
    sxm_file_log("---------------------------------------------------------");
    sxm_file_log(
            "mapindex:mapvalue - afid|root|parent|level|gdref|spid|ver|name");
    for (i = 0; i <= MAX_AFFILIATE_ID; i++)
    {
        if (ABLOCK[i] == INVALID_ID)
        {
            continue;
        }
        if (ABLOCK[i] >= ARRAY_SIZE(AFF))
        {
            sxm_file_log("%3d:%3d - Invalid map entry", i, ABLOCK[i]);
            continue;
        }

        a = &AFF[ABLOCK[i]];
        sxm_file_log("%3d:%4d - %5d|%5d|%5d|%1d|%5d|%5d|%5d|\"%s\"", i,
                ABLOCK[i], a->afid, a->rootid, a->parentid, a->level,
                a->global, a->spid, a->ver, a->name);
    }
}

static void print_tabledef(STable *t, int tabid)
{
    int i;

    sxm_file_log("Table def: tabid=%d ver=%d type=%d", tabid, t->ver, t->tclass);
    sxm_file_log("-------------------------------------------------");
    sxm_file_log("col - lbid|dtype|lbtype|dmod|prio|lbsize|text");
    for (i = 0; i < t->lc; i++)
    {
        sxm_file_log("%3d - %3d|%3d|%3d|%3d|%3d|%3d|%s", i, t->l[i].lbid,
                t->l[i].dtype, t->l[i].lbtype, t->l[i].dmod, t->l[i].prio,
                t->l[i].lbsize, t->l[i].text);
    }
}

void print_tabledefs(void)
{
    int i;

    if (service->h->g.sections[SPORTS_SECTION_SPORTS].version == INVALID_VERSION)
    {
        sxm_file_log("No tables(no config): cfgver=%d",
                service->h->g.sections[SPORTS_SECTION_SPORTS].version);
        return;
    }

    sxm_file_log("Stored table defs: cfgver=%d",
            service->h->g.sections[SPORTS_SECTION_SPORTS].version);
    sxm_file_log("**********************************");
    for (i = 0; i <= MAX_TABLE_ID; i++)
    {
        if (TBLOCK[i] == INVALID_ID)
        {
            continue;
        }
        if (TBLOCK[i] >= ARRAY_SIZE(TABLE))
        {
            sxm_file_log("%3d:%3d - Invalid map entry", i, TBLOCK[i]);
            continue;
        }
        print_tabledef(&TABLE[TBLOCK[i]], i);
    }
}

/* Make some text descriptions for labels. */
#define LBTEXT_ACTION_EXECUTOR_ \
    RESULT_ACTION_(HomeID, 				0)\
    RESULT_ACTION_(VisitID,				1)\
    RESULT_ACTION_(State,				2)\
    RESULT_ACTION_(Division,			3)\
    RESULT_ACTION_(Clock,				4)\
    RESULT_ACTION_(StartTime,			5)\
    RESULT_ACTION_(HScoreUint,			6)\
    RESULT_ACTION_(HScoreInt,			7)\
    RESULT_ACTION_(HScoreTime,			8)\
    RESULT_ACTION_(HScoreText,			9)\
    RESULT_ACTION_(HTeamSupUint,		10)\
    RESULT_ACTION_(HTeamSupInt,			11)\
    RESULT_ACTION_(HTeamSupTime,		12)\
    RESULT_ACTION_(HTeamSupText,		13)\
    RESULT_ACTION_(VTeamScoreUint,		14)\
    RESULT_ACTION_(VTeamScoreInt,		15)\
    RESULT_ACTION_(VTeamScoreTime,		16)\
    RESULT_ACTION_(VTeamScoreText,		17)\
    RESULT_ACTION_(VTeamSupUint,		18)\
    RESULT_ACTION_(VTeamSupInt,			19)\
    RESULT_ACTION_(VTeamSupTime,		20)\
    RESULT_ACTION_(VTeamSupText,		21)\
    RESULT_ACTION_(HTeamComm,			22)\
    RESULT_ACTION_(VTeamComm,			23)\
    RESULT_ACTION_(NatComm,				24)\
    RESULT_ACTION_(ExtraInfo,			25)\
    RESULT_ACTION_(ExplicitRank,		26)\
    RESULT_ACTION_(TablePartition,		27)\
    RESULT_ACTION_(PartPartition,		28)\
    RESULT_ACTION_(TextHeadline,		29)\
    RESULT_ACTION_(TextBody,			30)\
    RESULT_ACTION_(StatUint,			31)\
    RESULT_ACTION_(StatInt,				32)\
    RESULT_ACTION_(StatTime,			33)\
    RESULT_ACTION_(StatText,			34)\
    RESULT_ACTION_(Possession,			35)\
    RESULT_ACTION_(Result,				36)\
    RESULT_ACTION_(TableRef,			37)\

enum label_txt {
#define RESULT_ACTION_(a,b) a = (b),
LBTEXT_ACTION_EXECUTOR_
#undef RESULT_ACTION_
};

static char ebuff[48];
/* The format of the unknown error is not thread safe. */
static const char *get_label_text(int id)
{
    // Use the result executor macro to create the cases and stringify the name.
    switch(id)
    {
    #define RESULT_ACTION_(a,b) case a: return #a "(" #b ")";
    LBTEXT_ACTION_EXECUTOR_
    #undef RESULT_ACTION_
    default:
    break;
    }
    sprintf(ebuff, "Unknown Label(%d)", id);
    return ebuff;
}

static void print_table(Table* t, int tref, int idx)
{
    STable* tdef;
    uint i, k, n;
    SXMBitBuff bbuff;
    SXMBitBuff *pkt;
    TCol* tdefCol;
    uint lbid, olbsize, haveTitle, rowCount, tid, tver;
    char title[MAX_TABLE_TITLE_CHAR];
    ushort lbsize[MAX_TABLE_LABEL_COUNT];
    char buff[MAX_PARSE_CHAR];
    int err;
    ushort ref[64];
    uint refCount;

    if (!t->size)
    {
        return;
    }

    tid = tver = 0;
    pkt = &bbuff;
    sxm_bitbuff_setup(pkt, (byte *)&HEAP[t->heapix], t->size);

    if(tref <= 0)
    {
        SKIPBYTES(t->offset);
        SKIP(10 + 8);  /* + TABLEID + TABVER  already parsed when table was stored. */
        tid = t->tabno;
        tver = t->tabver;

        sxm_file_log("********Table(%d)********", idx);
        sxm_file_log("tabno  :%d", t->tabno);
        sxm_file_log("tabver :%d", t->tabver);
        sxm_file_log("size   :%d", t->size);
        sxm_file_log("heapix :%d", t->heapix);
        sxm_file_log("epoch  :%d", t->epoch);
        sxm_file_log("season :%d", t->season);
        sxm_file_log("offset :%d",  t->offset);
        sxm_file_log("inuse  :%d", t->inuse);
        sxm_file_log("ts     :%d", t->timeStamp);
        sxm_file_log("crc    :%d", t->crc);
        sxm_file_log("-----------------------------");

        if(!t->size)
        {
            sxm_file_log("-Not Present");
            return;
        }
    }
    else
    {
        SKIPBYTES(tref);
        NEXT(tid, 10);
        NEXT(tver, 8);
        sxm_file_log("++++++++Ref Table(idx=%d tref=%d)++++++++", idx, tref);
        sxm_file_log("tabno  :%d", tid);
        sxm_file_log("tabver :%d", tver);
        sxm_file_log("-----------------------------");
        if(pkt->err){
            sxm_file_log("-**Packet err(%d)**", pkt->err);
            return;
        }
    }

    if(TBLOCK[tid] == INVALID_ID)
    {
        sxm_file_log("-No Tdef");
        return;
    }

    if (TBLOCK[tid] >= ARRAY_SIZE(TABLE))
    {
        sxm_file_log("invalid index (tid=%d,TBLOCK[tid]=%d)", tid, TABLE[tid]);
        return;
    }

    tdef = &TABLE[TBLOCK[tid]];
    if(tdef->ver != tver){
        sxm_file_log("-Not stored version: storedver=%d", tdef->ver);
        return;
    }

    NEXT(haveTitle, 1);

    if(haveTitle)
    {
        BAUDOT(title, BAUDOT_MODE_START_WITH_LETTERS,
            SMALL_SYMCOUNT(MAX_TABLE_TITLE_CHAR), MAX_TABLE_TITLE_CHAR);
        sxm_file_log("-Title    :%s", title);
    }

    for(i=0; i<tdef->lc; i++)
    {
        lbsize[i] = tdef->l[i].lbsize;
    }

    while(FLAG())
    {
        NEXT(lbid, 10);
        NEXT(olbsize, 5)+1;
        for(i=0; i<tdef->lc; i++)
        {
            if(tdef->l[i].lbid == lbid)
            {
                lbsize[i] = olbsize;
                sxm_file_log("-*lbsize: tdef=%d instance=%d", tdef->l[i].lbsize, olbsize);
            }
        }
    }

    NEXT(rowCount, 8);

    if(pkt->err)
    {
        sxm_file_log("-**Packet err(%d)**", pkt->err);
        return;
    }

    sxm_file_log("-RowCount :%d", rowCount);

    err = 0;
    while(!err && rowCount)
    {
        refCount = 0;
        sxm_file_log("^^^^^^^^^^Row(%d)^^^^^^^^^^", rowCount);
        for(i=0; i<tdef->lc; i++)
        {
            /* End if we encountered an error on the previous column. */
            if(err)
            {
                break;
            }

            /* No data for this column. */
            if(!FLAG())
            {
                continue;
            }

            tdefCol = &tdef->l[i];

            /* These get special parsing. */
            if(	(tdefCol->lbtype == LB_HOME_SPID) ||
                (tdefCol->lbtype == LB_AWAY_SPID) ||
                (tdefCol->lbtype == LB_P_PARTITION) )
            {
                if(FLAG())
                {
                    BAUDOT(buff, BAUDOT_MODE_START_WITH_LETTERS,
                        BIG_SYMCOUNT(MAX_TEAM_NAME_CHAR), MAX_TEAM_NAME_CHAR);
                    sxm_file_log("-%-24s: %s", get_label_text(tdefCol->lbtype), buff);
                }
                else
                {
                    NEXT(n, 10);
                    sxm_file_log("-%-24s: %d", get_label_text(tdefCol->lbtype), n);
                }
            }
            else if(tdefCol->dtype == LB_DTYPE_INT)
            {
                NEXT(n, lbsize[i]);
                sxm_file_log("-%-24s: %d", get_label_text(tdefCol->lbtype), n);
                if(tdefCol->lbtype == LB_TREF){
                    /* Special case for trefs.  We need to remap to account for multi au re assemble to a single packet.
                       Should never have sign extension either. */
                    for(k=0; k<ARRAY_SIZE(t->trefmap); k++)
                    {
                        if(n <= t->trefmap[k].offset)
                        {
                            n -= t->trefmap[k].delta;
                            break;
                        }
                    }
                    ref[refCount] = n;
                    refCount++;
                }
            }
            else if(tdefCol->dtype == LB_DTYPE_TXT)
            {
                BAUDOT(buff, BAUDOT_MODE_START_WITH_LETTERS,
                    SMALL_SYMCOUNT(lbsize[i]), lbsize[i]);
                sxm_file_log("-%-24s: %s", get_label_text(tdefCol->lbtype), buff);
            }
            else
            {
                /* Unable to handle column.  Bad news. */
                err = 1;
            }

            if(pkt->err)
            {
                err = 1;
            }
        }
        sxm_file_log("^^^^^^^^^^^^^^^^^^^^^^^^^^^^", rowCount);
        rowCount--;

        /* Dump ref tables. */
        if(!err)
        {
            for(i=0; i<refCount; i++)
            {
                sxm_file_log("i=%d ref[i]=%d", i, ref[i]);
                print_table(t, ref[i], idx);
            }
        }
        else
        {
            sxm_file_log("Error: err=%d pkt->err=%d ", err, pkt->err);
        }
    };

    if(tref <= 0)
    {
        sxm_file_log("*****************************");
    }
    else
    {
        sxm_file_log("+++++++++++++++++++++++++++++");
    }
}

void print_tables(void)
{
    uint i, j;
    Affiliation *a;

    if (service->h->g.sections[SPORTS_SECTION_SPORTS].version == INVALID_VERSION)
    {
        sxm_file_log("No tables(no config): cfgver=%d",
                service->h->g.sections[SPORTS_SECTION_SPORTS].version);
        return;
    }

    sxm_file_log("******************************************");
    sxm_file_log("Stored tables");
    sxm_file_log("******************************************");

    for (i = 0; i <= MAX_AFFILIATE_ID; i++)
    {
        if (ABLOCK[i] == INVALID_ID)
        {
            continue;
        }
        if (ABLOCK[i] >= ARRAY_SIZE(AFF))
        {
            sxm_file_log("invalid AFF index (i=%d,ABLOCK[i]=%d)", i, ABLOCK[i]);
            continue;
        }
        a = &AFF[ABLOCK[i]];
        sxm_file_log("*");
        sxm_file_log("****************%s(%d) ****************", a->name, a->afid);
        sxm_file_log("*");

        for (j = 0; j < ARRAY_SIZE(a->schedule); j++)
        {
            print_table(&a->schedule[j], 0, j);
        }
        for (j = 0; j < ARRAY_SIZE(a->others); j++)
        {
            print_table(&a->others[j], 0, j + ARRAY_SIZE(a->schedule));
        }
    }
}
#endif
