/************************************************************************
 *                                                                      *
 *            SXM_PARKING_INTERNAL.H                                    *
 *            ======================                                    *
 *                                                                      *
 *                                 Copyright 2013 Sirius XM Radio, Inc. *
 *                                                 All Rights Reserved. *
 *               Licensed Materials - Property of Sirius XM Radio, Inc. *
 *                                                                      *
 ************************************************************************/

#ifndef SXM_PARKING_INTERNAL_H
#define SXM_PARKING_INTERNAL_H

#include <sxm_parking.h>
#include <util/sxm_grid.h>
#ifndef SDKFILES_BUILDER
#include <util/sxm_common_internal.h>
#include <util/sxm_noname_internal.h>
#ifdef DEBUG_VAR
#undef  DEBUG_VAR
#endif
#define DEBUG_VAR ((service) ? service->grid.ds.debug_level : (uint)(-1))
#include <sxm_stdtrace.h>
#else
#include <sxm_build.h>
#include <sxm_typ.h>
#endif

/** Defines service name */
#define PARKING_SERVICE_NAME          "parking"
/** Database files name */
#define PARKING_SERVICE_DB_FILE_TYPE  'W'
#define PARKING_SERVICE_DB_NAME       PARKING_SERVICE_NAME"db"
/** Parking service Cycle file name */
#define PARKING_CFILE_NAME            SXM_GRID_CFILE_NAME


enum {
    /** Defines max number of collectible stations */
    PARKING_COLLECTION_STATIONS   = (2048),
    /** Defines max possible valid station id */
    PARKING_MAX_STATION_ID        = (8191),
    /** Defines mix possible valid station id */
    PARKING_MIN_STATION_ID        = (1),
    /** Defines number of station in the region */
    PARKING_MAX_STATIONS          = (PARKING_MAX_STATION_ID + 1),
    /** Defines number of supported regions in max */
    PARKING_MAX_REGIONS           = (SXM_GRID_REGION_ID_MAX + 1),
    /** The format of the cycle file */
    PARKING_CFILE_SCHEMA_VERSION  = 4,
    /** Defines supported Parking's T-File schema */
    PARKING_TFILE_SCHEMA_VERSION  = 4,
    /** Data valid time out */
    PARKING_MAX_DAYS              = 1,
    /** Data valid timeout in minutes */
    PARKING_MAX_MINUTES           = PARKING_MAX_DAYS * MINUTES_IN_DAY
};

/*************************************************/
/*                     RFD                       */
/*************************************************/

/** In-memory record */
typedef struct {
    ushort valid;
    ushort total;
    uint avail_time;
    byte full[5];
    byte status;
    ushort weekday_hour[24];
    ushort saturday_hour[24];
    ushort sunday_hour[24];
    ushort weekday_day;
    ushort saturday_day;
    ushort sunday_day;
    ushort ebird;
    ushort event;
    ushort other;
} ParkingRecord;

#ifndef SDKFILES_BUILDER

typedef struct {
    /** Main entrance */
    fix lon;
    fix lat;
    /** Offset from main entrance */
    fix dlon;
    fix dlat;
} ParkingLocation;

/** The only valid update file prefix which shall be
 * supported by the service
 */
#define PARKING_UPDATE_FNAME_PREFIX   'U'
/** The only supported length of the update file */
#define PARKING_UPDATE_FNAME_SIZE     (7)

/** the update structure */
typedef struct {
    SXMGridUpdate grid; //!< Grid related updater's data
    SXMParkingLocation fs;
    ParkingLocation location;
} ParkingUpdate;

/** RFD actions */
typedef enum {
    PARKING_UTYPE_DELETE = 0,
    PARKING_UTYPE_NEW,
    PARKING_UTYPE_MODIFY,
    PARKING_UTYPE_EOL
} ParkingUpdateTypes;
#endif /* #ifndef SDKFILES_BUILDER */

/** Single PPOI location storage */
typedef struct {
    ParkingRecord fav_station; //!< PPOI Dynamic data
    PPOISpec ppoi_spec; //!< Station specification (id and user flags)
} ParkingFavLocation;

/** C-File storage structure */
typedef struct {
    uint count; //!< Utilization of the \p locations
    ParkingFavLocation locations[SXM_PPOI_MAX]; //!< Storage
} ParkingFavPersData;

/** user-defined structure in the CFile root blocks */
enum {
    PARKING_SECTION_FIRST = 0,
    PARKING_BASE = PARKING_SECTION_FIRST,
    PARKING_HEAP,
    PARKING_PPOI,
    PARKING_SECTIONS_COUNT,
    PARKING_UPDATE_ALL = ((1 << PARKING_SECTIONS_COUNT) - 1),

    PARKING_FULLNESS_SAVE_SECTION_FIRST = 0,
    PARKING_FULLNESS_SAVE_HEAP = PARKING_FULLNESS_SAVE_SECTION_FIRST,
    PARKING_FULLNESS_SECTIONS_COUNT
};

/** C-File data structure */
typedef struct {
    SXMCFileRootBlock g;
    CSECT(SXMGridBaseLine, baseb, SXM_GRID_NEEDED_REGIONS) baseb;
    CDSECT(SXMGridAU, heap, SXM_GRID_AUS_PER_REGION * SXM_GRID_NEEDED_REGIONS) heap;
    CSECT(ParkingFavPersData, fav_loc, 1) fav_loc;
} ParkingSave;

/** Fullness data structure */
typedef struct {
    SXMCFileRootBlock g;
    CDSECT(SXMGridAU, heap, SXM_GRID_AUS_PER_REGION * SXM_GRID_NEEDED_REGIONS) heap;
} ParkingFullnessSave;

/*************************************************/
/*                  Database                     */
/*************************************************/

typedef struct {
    ushort region;
    SXMGridSavedAU safe[SXM_GRID_AUS_PER_REGION];  //!< saved AUs (for fullness data) for this region
} SXMParkingFullnesSavedAU;

#ifndef SDKFILES_BUILDER

/** Defines Carousel IDs */
typedef enum {
    PARKING_CARID_AVAL_DATA_0 = 0,
    PARKING_CARID_AVAL_DATA_1 = 1,
    PARKING_CARID_PRICE_AND_CAPACITY = 2,
    PARKING_CARID_RFD_DATA = 3,
    PARKING_CARID_RFD_METADATA = 4
} ParkingCarouselIDs;

/** Data service structure */
typedef struct {
    /** GRID descriptor */
    SXMGridService grid;
    /** Operational data storage */
    SXMParkingFullnesSavedAU fullness_au[SXM_GRID_NEEDED_REGIONS];

    /** Saved data over power cycles */
    ParkingSave *data;
    /** Operational fullness data accessor */
    ParkingFullnessSave *fullness_data;

    /** Updater */
    ParkingUpdate *upd;

    /** Operational stations */
    struct {
        /** Stations id's */
        uint ids[PARKING_COLLECTION_STATIONS];
        /** Corresponding data */
        ParkingRecord items[PARKING_COLLECTION_STATIONS];
    } stations;

    /** Used for change detection */
    ParkingRecord staged;

    /** Reference to the C-File data */
    ParkingFavPersData *fav_data;
    ParkingFavLocation key; //!< Searching key

    SXMGridDataAU price_au[SXM_GRID_NEEDED_REGIONS];
    SXMGridDataAU avail_au[SXM_GRID_NEEDED_REGIONS];
} ParkingService;

extern void sxm_parking_delete_request(ParkingService *);

extern void sxm_parking_complete_au(byte *, int, ptr);

extern int sxm_parking_database_start(ParkingService *);

extern void sxm_parking_database_stop(ParkingService *);

extern int sxm_parking_database_restart(ParkingService *);

extern int sxm_parking_protocol_start(ParkingService *);

extern int sxm_parking_protocol_stop(ParkingService *);

extern void sxm_parking_check_timestamp(ParkingService *);

/*************************************************/
/*             Database update                   */
/*************************************************/

extern void sxm_parking_update_init(ParkingService *);

extern void sxm_parking_update_uninit(ParkingService *);

extern int sxm_parking_update_start(SXMRfdMetadata *);

/*************************************************/
/*        User collection and extraction         */
/*************************************************/

extern int sxm_parking_user_extract(ParkingService *, SXMParkingLocation *);

extern char *sxm_parking_user_string_type(char *, int, int);

extern int sxm_parking_user_poi_extract(ParkingService *, SXMParkingLocation *);

#endif /* #ifndef SDKFILES_BUILDER */

#endif /* #ifndef SXM_PARKING_INTERNAL_H */
