/************************************************************************
 *                                                                      *
 *            SXM_PARKING.C                                             *
 *            =============                                             *
 *                                                                      *
 *                                 Copyright 2013 Sirius XM Radio, Inc. *
 *                                                 All Rights Reserved. *
 *               Licensed Materials - Property of Sirius XM Radio, Inc. *
 *                                                                      *
 *                           Implementation of the SXM SDK for Parking. *
 *                                                                      *
 ************************************************************************/
/*************************************************************************//**
 * \file sxm_parking.c
 * \author Leslie French
 * \date 5/20/2014
 *
 * Implementation of the SXM SDK for Parking. This includes all
 * API's needed to implement Parking using SXe.
 *
 ************************************************************************/

#define DEBUG_TAG "parking"

#include "sxm_parking_internal.h"

/** Status indicates if the service is running, and if
 *  the service is Subscribed.
 */
static SXMStatus status = {SXM_SERVICE_STOPPED, SXM_SUBS_UNKNOWN};

/** A mutex used to ensure serial execution of the Service commands. */
static pthread_mutex_t apimutex = PTHREAD_MUTEX_INITIALIZER;

/** A pointer to the service structure for Parking.*/
static ParkingService *service = NULL;

/* Forward declarations */
static INT32 parking_save_data(SXMIOStat *pStat, void *pArg);

/***************************************************************************//**
 * This function destroys the service object
 *
 ******************************************************************************/
static void parking_service_destroy(ParkingService **pService) {
    pthread_mutex_destroy(&(*pService)->grid.service_mutex);
    sxm_grid_heap_free(&(*pService)->grid);
    sxe_free((*pService));
    (*pService) = NULL;
}

/***************************************************************************//**
 * This function creates the service object
 *
 ******************************************************************************/
static int parking_service_create(ParkingService** ppService) {
    int rc;
    ParkingService *pService;

    pService = (ParkingService *)sxe_calloc(1, sizeof(ParkingService));
    if (pService == NULL) {
        rc = SXM_E_NOMEM;
    }
    else {
        rc = sxm_grid_heap_alloc(&pService->grid, ARRAY_SIZE(pService->stations.ids));
        if (rc != SXM_E_OK) {
            sxe_free(pService);
            pService = NULL;
        }
    }

    *ppService = pService;

    return rc;
}

/*****************************************************************************//**
 *
 * This function starts the SXM Parking service.
 *
 * First it checks to see if the service has already started. If not then it initializes 
 * information about the Parking service, then starts the Parking database, Parking protocol
 * decoder and low level functions. When successful, Parking data collection will start.
 *
 * \param[in] pCallback   function to receive event indications 
 * \param[in] debugLevel  debug logging level
 *
 * \retval SXM_E_OK       Start completed successfully.
 * \retval SXM_E_STATE    Service is already running.
 * \retval SXM_E_FAULT    NULL callback argument
 * \retval SXM_E_NOMEM    Memory could not be allocated for the service. 
 * \retval SXM_E_NO_DB    Failed because tfile does not exist.
 * \retval SXM_E_BAD_DB   Failed because tfile is corrupt.
 * \retval SXM_E_PIPE     Failed due to Low Level Interface Pipe error.
 * \retval SXM_E_THREAD   Failed due to error starting the pix thread.
 *
 ********************************************************************************/
int sxm_parking_start(void (*pCallback)(int, int), int debugLevel) {
    int rc = SXM_E_OK;
    SXMStatus sdk_status;
    SXMStatus link_status = {SXM_SERVICE_OK, SXM_SUBS_FULL};

    LOCK(apimutex);

    ENTER_START_API(debugLevel,
                    "(pCallback: %p, debugLevel: %d)",
                    pCallback, debugLevel);

    sxm_sdk_status(&sdk_status);
    sxm_sxi_status(&link_status);

    if (status.service != SXM_SERVICE_STOPPED) {
        ERROR_API("Already started");
        rc =  SXM_E_STATE;
    }
    else if (sdk_status.service != SXM_SERVICE_OK) {
        ERROR_API("SDK not started");
        rc = SXM_E_STATE;
    }
    else if (link_status.service != SXM_SERVICE_OK) {
        ERROR_API("SXI (link) not started");
        rc = SXM_E_STATE;
    }
    else if (!pCallback) {
        rc = SXM_E_FAULT;
    }
    else if ((rc = parking_service_create(&service)) != SXM_E_OK) {
        ERROR_API("Failed to create service structure");
    }
    else {
        service->grid.ds.debug_level = (uint)debugLevel;

        service->grid.ds.status = &status;
        service->grid.callback = pCallback;

        sxm_mutex_init(&service->grid.service_mutex);
        //  create the parking service object
        service->grid.ds.maxau = SXM_GRID_AU_SIZE;
        service->grid.ds.complete = &sxm_parking_complete_au;
        service->grid.ds.carsize = 3;
        service->grid.ds.collected = &sxm_parking_update_start;
        service->grid.ds.dsis[0].dsi = 640;
        service->grid.ds.iom.pFunc = &parking_save_data;
        service->grid.ds.iom.tQuota = sizeof(ParkingSave);
        BITSALL(service->grid.ds.dsis[0].filter);
        service->grid.ds.dsuser = service;
        strncpy(service->grid.ds.service, PARKING_SERVICE_NAME,
            sizeof(service->grid.ds.service) - 1);

        service->grid.ds.auto_save_on_stop = TRUE;
        service->grid.sections_update = (byte)~PARKING_UPDATE_ALL;

        //  if the return values is < 0 this is an error code
        //  else it is the current version of the file.
        rc = sxm_parking_database_start(service);
        if (rc != SXM_E_OK) {
            PLOG_API("Failed to start database (rc=%d)", rc);
            parking_service_destroy(&service);
        }
        else {
            sxm_parking_update_init(service);

            rc = sxm_parking_protocol_start(service);
            if (rc != SXM_E_OK) {
                ERROR_API("Failed to access cycle file",
                    &service->grid.ds.service[0]);
                sxm_parking_update_uninit(service);
                sxm_parking_database_stop(service);
                parking_service_destroy(&service);
            }
            else {
                rc = sxm_grid_lli_start(&service->grid);
                if (rc != SXM_E_OK) {
                    ERROR_API("Failed to start LLI.",
                        &service->grid.ds.service[0]);
                    sxm_parking_update_uninit(service);
                    sxm_parking_protocol_stop(service);
                    sxm_parking_database_stop(service);
                    parking_service_destroy(&service);

                    // service is stopped
                    status.service = SXM_SERVICE_STOPPED;
                    status.substate = SXM_SUBS_UNKNOWN;
                }
                else {
                    // sync RFD to set collection state
                    sxm_grid_rfd_sync(&service->grid);
                }
            }
        }
    }

    LEAVE_START_API(debugLevel, "%s", sxm_sdk_format_result(rc));
    UNLOCK(apimutex);
    return rc;
}

/***************************************************************************//**
 *
 * This function removes the engine-cycle file.
 *
 * \retval SXM_E_OK     Request accepted.
 * \retval SXM_E_STATE  Service is active.
 * \retval SXM_E_NOENT  Cycle file does not exist
 *
 * \note  If the service is inactive, the cycle file is removed if it exists.
 *
 ********************************************************************************/
SXESDK_API int sxm_parking_cfile_clean(void) {

    int rc = SXM_E_OK;

    // trace
    ENTER_API("");

    // serialize this API
    LOCK(apimutex);

    // service must be inactive, otherwise return "state" error
    if (status.service != SXM_SERVICE_STOPPED) {
        rc = SXM_E_STATE;
    }
    else {
        // remove the cycle file
        rc = sxm_file_remove('T', PARKING_SERVICE_NAME, PARKING_CFILE_NAME);
        if (rc == -1) {
            rc = SXM_E_NOENT;
        }
        else {
            rc = SXM_E_OK;
        }
    }
    UNLOCK(apimutex);

    LEAVE_API("%s", sxm_sdk_format_result(rc));
    return rc;
}

/***********************************************************************//**
 *
 * This function passes the current state and subscription status of the service 
 * to the OEM developer.
 *
 * First validate the provided pointer. If it is good then populate it with the
 * current status information. Possible values are shown in sxm_common.h
 *
 * \param[out] pStatus   pointer to status structure
 *
 * \retval SXM_E_OK      status query successful
 * \retval SXM_E_FAULT   null argument
 *
 ************************************************************************/
int sxm_parking_status(SXMStatus *pStatus) {
    SXMResultCode rc;

    if (pStatus) {
        *pStatus = status;
        rc = SXM_E_OK;
    }
    else {
        rc = SXM_E_FAULT;
    }
    return rc;
}

/***********************************************************************//**
 * This function stops the SXM Parking service.
 *
 * It will clean up memory, terminate any extraction currently in progress, 
 * and will fail if it is called from within a notification callback routine.
 * An unrecoverable SXe service event has occurred when calling the sxm_parking_stop 
 * function when either the return status code of SXM_E_STATE, SXM_E_FAULT or SXM_E_BUSY 
 * is returned with that status.service being SXM_SERVICE_OK. 
 * In this event, the application developer shall reset the SXe system so that the service(s) 
 * and the supporting framework can be reinitialized to a pristine condition. 
 *
 * \retval SXM_E_OK      service stopped successfully
 * \retval SXM_E_BUSY    Thread cannot stop itself.
 * \retval SXM_E_FAULT   NULL parameter (e.g. service structure is NULL).
 * \retval SXM_E_STATE   service is already stopped.
 *
 ************************************************************************/
int sxm_parking_stop(void) {
    int rc;
    uint debugLevel = DEBUG_VAR;

    LOCK(apimutex);

    ENTER_API("");

    if (status.service != SXM_SERVICE_OK) {
        rc = SXM_E_STATE;
    }
    else {

        status.service = SXM_SERVICE_STOPPING;
        UNLOCK(apimutex);
        rc = sxm_grid_lli_stop(&service->grid);
        LOCK(apimutex);

        if (rc != SXM_E_OK) {
            status.service = SXM_SERVICE_OK;
            ERROR_API("LLI could not be stopped");
        }
        else {
            rc = sxm_parking_protocol_stop(service);
            sxm_parking_update_uninit(service);
            sxm_parking_database_stop(service);
            if (service->grid.fav.active == TRUE) {
                sxm_mutex_destroy(&service->grid.fav.mutex);
            }
            sxm_grid_delete_request(&service->grid);
            parking_service_destroy(&service);

            // service is stopped
            status.service = SXM_SERVICE_STOPPED;
            status.substate = SXM_SUBS_UNKNOWN;
        }
    }

    LEAVE_START_API(debugLevel, "%s", sxm_sdk_format_result(rc));
    UNLOCK(apimutex);
    return rc;
}

/*****************************************************************************//**
 * This function Sets up an Parking request.
 *
 * \note   The Parking service supports only a single request at a time.  The SDK 
 * implementation limits the data returned from the request to at most four 2x2 degrees 
 * regions, no matter how large an area is specified in the request. 
 * Parking collection requests operate over a Map Bounding Rectangle (MBR).  The 
 * extraction requests return a list of Parking Locations sorted by distance 
 * from a Point.  The initial Point value may be specified on the collection request.
 * If no point is specified, extractions will proceed from the location point of the 
 * supplied MBR.
 * The Parking Service limits the number of locations in its list to no more than 
 * 2,000 locations.  This limit provides bounds on the memory and CPU used by the 
 * service.  If the requested area (mbr) contains more than 2,000 locations, the 
 * area is progressively reduced until it contains 2,000 or fewer locations with the 
 * location sort point at the center of the modified area (mbr).
 * As the application moves the extraction (location) point, it should check to see that 
 * the new point lies within the collection area (mbr), and update the area as the 
 * point approaches, or crosses, the boundary of the area.
 *
 * \param[in] pMbr        a pointer to the \ref SXMMBR structure in application 
 *                        memory defining the area over which ev data 
 *                        is to be collected
 * \param[in] pLocation   NULL, OR a pointer to the \ref SXMPoint structure in 
 *                        application memory defining the sorting point for 
 *                        parking data collection. If NULL is used then
 *                        the center point of the supplied SXMMBR is used.
 * \param[in] pNotify     function pointer of notification routine
 * \param[in] pUsercx     user context value supplied on the callback, 
 *                        e.g. text string indicating "parking"
 * \param[out] pHandle    pointer to returned request handle pointer
 *
 * \retval SXM_E_OK     %Request has been created.
 * \retval SXM_E_BUSY   Collection request is already active.
 * \retval SXM_E_FAULT  NULL parameter
 * \retval SXM_E_INVAL  Invalid parameter.
 * \retval SXM_E_STATE  service not started
 *
 ********************************************************************************/
int sxm_parking_request(const SXMMBR *pMbr, const SXMPoint *pLocation,
                        void (*pNotify)(ptr, ptr),
                        ptr pUsercx, ptr *pHandle) {
    int rc;
    LOCK(apimutex);
    ENTER_API("(pMbr: %p, pLocation: %p, pNotify: %p, pUsercx: %p, pHandle: %p)",
        pMbr, pLocation, pNotify, pUsercx, pHandle);

    rc = sxm_grid_request(SXM_2GRID(service),
                          pMbr, pLocation, pNotify, pUsercx,
                          (SXMGridExtraction **)pHandle);

    LEAVE_API("%s", sxm_sdk_format_result(rc));
    UNLOCK(apimutex);
    return rc;
}

/*****************************************************************************//**
 * This function modifies an existing Parking request.
 *
 * The modify routine will update the collection MBR and optionally the sort point.
 *
 * \note If a new location value is not supplied, the old value is retained 
 * (it does not reset to the center of the new MBR).  This is to support the most 
 * usual case of modifying a request, when a vehicle has moved sufficiently far from 
 * its previous position that a new collection is warranted, but the vehicle position 
 * is still close to the point of the last extraction.
 *
 * \param[out] pHandle    pointer to returned request handle pointer
 * \param[in] pMbr        a pointer to the \ref SXMMBR structure in application 
 *                        memory defining the area over which parking data 
 *                        is to be collected. If pMbr is not specified, 
 *                        the original MBR is used.
 * \param[in] pLocation   NULL, OR a pointer to the SXMPoint structure in 
 *                        application memory defining the sorting point for 
 *                        parking data collection. If NULL is used then
 *                        the center point of the supplied SXMMBR is used.
 *
 * \retval  SXM_E_OK       modify request successful
 * \retval  SXM_E_BUSY     collection request already active
 * \retval  SXM_E_INVAL    invalid parameter
 * \retval  SXM_E_FAULT    NULL pointer argument
 * \retval  SXM_E_STATE    service not started
 *
 *   sxm_ev_modify truth table
 *   |pHandle|   pMbr   | pLocation   |                      Description                               |
 *   |:-----:|:--------:|:-----------:|:--------------------------------------------------------------:|
 *   | NULL  |     X    |     X       | SXM_E_FAULT is returned, a valid "pHandle" must be provided.   |
 *   | Valid |   NULL   |    NULL     | SXM_E_FAULT is returned, must have one valid pMbr or pLocation |
 *   | Valid | New pMbr |    NULL     | Will use the "new" pMbr with the previous pLocation.           |
 *   | Valid |   NULL   |New pLocation| Will use the previous pMbr with the "new" pLocation.           |
 *
 ********************************************************************************/
int sxm_parking_modify(ptr pHandle, const SXMMBR *pMbr, const SXMPoint *pLocation) {
    int rc;
    LOCK(apimutex);
    ENTER_API("(pHandle: %p, pMbr: %p, pLocation: %p)", pHandle, pMbr, pLocation);

    rc = sxm_grid_modify(SXM_2GRID(service), (SXMGridExtraction *)pHandle,
                         pMbr, pLocation);

    LEAVE_API("%s", sxm_sdk_format_result(rc));
    UNLOCK(apimutex);
    return rc;
}
/*****************************************************************************//**
 *
 * This function removes an existing Parking request.
 *
 * The remove routine will remove a collection MBR, and all the data associated with it. 
 * Once the request has been removed, the handle should not be used on any subsequent calls.
 *
 * \param[in] pHandle     a handle to the extraction request.
 *
 * \retval SXM_E_OK       %Request removed.
 * \retval SXM_E_BUSY     %Request currently extracting.
 * \retval SXM_E_FAULT    NULL parameter 
 * \retval SXM_E_STATE    Service is not started
 * \retval SXM_E_INVAL    The request handle is invalid
 *
 ********************************************************************************/
int sxm_parking_remove(ptr pHandle) {
    int rc;
    LOCK(apimutex);
    ENTER_API("(pHandle: %p)", pHandle);

    rc = sxm_grid_remove(SXM_2GRID(service), (SXMGridExtraction*)pHandle);

    LEAVE_API("%s", sxm_sdk_format_result(rc));
    UNLOCK(apimutex);
    return rc;
}

/*****************************************************************************//**
 *
 * This function begins an Parking extraction and may also optionally changes 
 * the sorting center point.
 *
 * \param[in] pHandle     request handle
 * \param[in] pLocation   NULL, OR a pointer to the SXMPoint structure in 
 *                        application memory defining the sorting point for 
 *                        parking data collection.
 *
 * \retval  SXM_E_OK      extraction begun
 * \retval  SXM_E_BUSY    extraction already active
 * \retval  SXM_E_FAULT   NULL argument
 * \retval  SXM_E_INVAL   invalid parameter
 * \retval  SXM_E_STATE   service not started
 *
 ********************************************************************************/
int sxm_parking_begin(ptr pHandle, const SXMPoint *pLocation) {
    int rc;
    LOCK(apimutex);
    ENTER_API("(pHandle: %p, pLocation: %d)", pHandle, pLocation);

    rc = sxm_grid_begin(SXM_2GRID(service), (SXMGridExtraction*)pHandle, pLocation);

    LEAVE_API("%s", sxm_sdk_format_result(rc));
    UNLOCK(apimutex);
    return rc;
}
/*****************************************************************************//**
 *
 * This function extracts an individual parking locations in distance-sorted order 
 * from the current center point.
 *
 * \param[in] pHandle    a handle to the extraction request.
 * \param[out] pLocation a pointer to an SXM Parking Location that will be
 *                       filled up with parking data.
 *
 * \retval SXM_E_OK      data has been extracted.
 * \retval SXM_E_STATE   Service in wrong state.
 * \retval SXM_E_FAULT   NULL parameter. 
 * \retval SXM_E_NOENT   No entry found.
 * \retval SXM_E_INVAL   handle parameter is invalid
 * \retval SXM_E_PIPE    Failed to access location data within DB
 *
 ********************************************************************************/
int sxm_parking_extract_location(ptr pHandle, SXMParkingLocation *pLocation) {
    int rc;
    LOCK(apimutex);
    ENTER_API("(pHandle: %p, pLocation: %d)", pHandle, pLocation);

    rc = sxm_grid_extract(SXM_2GRID(service),
                          (SXMGridExtraction *)pHandle, pLocation, sizeof(*pLocation));

    LEAVE_API("%s", sxm_sdk_format_result(rc));
    UNLOCK(apimutex);
    return rc;
}

/*****************************************************************************//**
 * The extraction routine terminates the extraction, and 
 * unlocks the data for update.
 *
 * \param[in] pHandle request handle
 *
 * \retval  SXM_E_OK       extraction ended
 * \retval  SXM_E_FAULT    NULL argument
 * \retval  SXM_E_INVAL    invalid parameter
 * \retval  SXM_E_STATE    service or extraction not started
 *
 ********************************************************************************/
 int sxm_parking_end(ptr pHandle) {
     int rc;
     LOCK(apimutex);
     ENTER_API("(pHandle: %p", pHandle);

     rc = sxm_grid_end(SXM_2GRID(service), (SXMGridExtraction *)pHandle);

     LEAVE_API("%s", sxm_sdk_format_result(rc));
     UNLOCK(apimutex);
     return rc;
}

/************************************************************************
 *                                                                      *
 *            Support Routines                                          *
 *            ================                                          *
 *                                                                      *
 ************************************************************************/
/**********************************************************************//**
 *
 * This function gets current version of parking database.
 *
 * \note The version will only change once an RFD-based file update has been
 * completely and successfully applied.
 *
 * \param[out] pVer        a pointer to the returned DB version
 *          
 * \retval SXM_E_OK        DB version is retrieved and populated into pVer
 * \retval SXM_E_FAULT     NULL parameter
 * \retval SXM_E_STATE     Service in wrong state
 ************************************************************************/
int sxm_parking_get_version(uint *pVer) {
    int rc;
    LOCK(apimutex);
    ENTER_API("(pVer: %p)", pVer);

    rc = sxm_grid_database_get_version(SXM_2GRID(service), pVer);

    LEAVE_API("%s", sxm_sdk_format_result(rc));
    UNLOCK(apimutex);
    return rc;
}

/*****************************************************************************//**
 * This function returns distance from the sort point to the closest edge of the MBR.
 * 
 * The closest routine allow the application to adjust the collection MBR as described
 * in section "When to Modify?" of SX-9845-0340 Introduction Design Guide.
 *
 * \note        The returned value is in the current units (miles or kilometers) 
 *              as set by the global SDK "units" option.
 *
 * \param[in]   pSort   a pointer to the SXMPoint structure referencing the 
 *                      point from which to measure the distance.
 * \param[out]  pDist   a pointer to a float value that will contain the distance 
 *                      from the sort point (pSort) to the closest edge of
 *                      the MBR (SXMMBR).
 *
 * \retval  SXM_E_OK      Distance calculated successfully.
 * \retval  SXM_E_FAULT   NULL parameter.
 * \retval  SXM_E_INVAL   sort point lies outside MBR
 *
 ********************************************************************************/
int sxm_parking_point_closest(const SXMPoint *pSort, SXMFLOAT *pDist) {
    int rc;

    LOCK(apimutex);
    ENTER_API("(pSort: %p, pDist: %p)", pSort, pDist);

    rc = sxm_grid_point_closest(SXM_2GRID(service), pSort, pDist);

    LEAVE_API("%s", sxm_sdk_format_result(rc));
    UNLOCK(apimutex);
    return rc; 
}

/*****************************************************************************//**
 * This function returns distance from the sort point to the furthest edge of the MBR.
 *
 * The furthest routines allow the application to adjust the collection MBR as described
 * in section "When to Modify?" of SX-9845-0340 Introduction Design Guide
 *
 * \note The returned value is in the current units (miles or kilometers) 
 *               as set by the global SDK "units" option.
 *
 * \param[in]   pSort   a pointer to the SXMPoint structure referencing the 
 *                      point from which to measure the distance.
 * \param[out]  pDist   a pointer to a float value that will contain the distance 
 *                      from the sort point (pSort) to the closest edge of
 *                      the MBR (SXMMBR).
 *
 * \retval  SXM_E_OK      Distance calculated successfully.
 * \retval  SXM_E_FAULT   NULL parameter
 * \retval  SXM_E_INVAL   sort point lies outside MBR
 *
 ********************************************************************************/
int sxm_parking_point_furthest(const SXMPoint *pSort, SXMFLOAT *pDist) {
    int rc;

    LOCK(apimutex);
    ENTER_API("(pSort: %p, pDist: %p)", pSort, pDist);

    rc = sxm_grid_point_furthest(SXM_2GRID(service), pSort, pDist);

    LEAVE_API("%s", sxm_sdk_format_result(rc));
    UNLOCK(apimutex);
    return rc;
}

/*****************************************************************************//**
 *
 * This function returns actual request MBR.
 *
 * During request creation or modification the service can reduce collection
 * area based taking into account provided MBR and center point. As result
 * the actual area there data are being collected can be different than application
 * knows. This function allows application to get this real MBR and find out
 * actual collection area.
 *
 * \param[in] handle      the collection handle created previously.
 * \param[out] pMbr       the actual collection MBR.
 *
 * \retval SXM_E_OK       Success.
 * \retval SXM_E_STATE    Service is not running
 * \retval SXM_E_FAULT    NULL parameter(s).
 * \retval SXM_E_INVAL    Invalid request handle
 *
 ********************************************************************************/
int sxm_parking_get_mbr(ptr handle, SXMMBR *pMbr) {
    int rc;
    LOCK(apimutex);
    ENTER_API("(handle: %p, pMbr: %p)", handle, pMbr);

    rc = sxm_grid_get_mbr(SXM_2GRID(service), (SXMGridExtraction*)handle, pMbr);

    LEAVE_API("%s", sxm_sdk_format_result(rc));
    UNLOCK(apimutex);
    return rc;
}

/*****************************************************************************//**
 *
 * The debug interface allows the application to change the logging level on the 
 * parking module once it has been initialized.
 *
 * \param[in]           debugLevel  Level of debugging desired.
 *
 * \retval SXM_E_OK     set debug successful
 * \retval SXM_E_STATE  if service is not running
 *
 *
 ********************************************************************************/
int sxm_parking_set_debug(int debugLevel) {
    int rc;
    LOCK(apimutex);
    ENTER_API("(debugLevel: %d)", debugLevel);

    rc = sxm_grid_set_debug(SXM_2GRID(service), debugLevel);

    LEAVE_API("%s", sxm_sdk_format_result(rc));
    UNLOCK(apimutex);
    return rc;
}

/*****************************************************************************//**
 *
 * This function is used to get the debug level for the service.
 *
 * \param[out]          pDebugLevel returned debug level
 *
 * \retval SXM_E_OK     on success
 * \retval SXM_E_STATE  if service is not running
 * \retval SXM_E_FAULT  if pDebugLevel is NULL
 *
 ********************************************************************************/
int sxm_parking_get_debug(int *pDebugLevel) {
    int rc;
    LOCK(apimutex);
    ENTER_API("(pDebugLevel: %p)", pDebugLevel);

    rc = sxm_grid_get_debug(SXM_2GRID(service), pDebugLevel);

    LEAVE_API("%s", sxm_sdk_format_result(rc));
    UNLOCK(apimutex);
    return rc;
}

/*****************************************************************************//**
 *
 * This function is used to set or clear the Preferred Points of Interests (PPOIs)
 *
 * This function also sets a notification callback \p pNotify. The callback is
 * called when the dynamic data has changed for 1 or more PPOIs. The 1st argument 
 * to the callback is a regurgitation of the \p pUsercx, the second argument to the
 * callback is a bitwise-or of all the \ref PPOISpec::ppoi_user_flag values of all the PPOIs 
 * that changed.
 *
 * \note In order to clean up current PPOI list this function can be called with
 *       \p specCnt == 0: values of \p specs, \p pNotify and \p pUsercx will be ignored.
 *
 * \param[in]  specCnt number of elements in the \p specs.
 * \param[in]  specs array of points of interests, see \ref SXMParkingLocation::id field.
 * \param[in]  notification notification callback.
 * \param[in]  usercx user data passed to each callback call
 *
 * \retval SXM_E_OK     on success
 * \retval SXM_E_STATE  if service is not running
 * \retval SXM_E_FAULT  if \p ids is \p NULL while \p specCnt is greater than 0 
 * \retval SXM_E_INVAL  if \p specCnt is more than \ref SXM_PARKING_PPOI_MAX
 * \retval SXM_E_BUSY   POI or request extraction in progress.
 *
 ********************************************************************************/
int sxm_parking_set_ppoi_list(const uint specCnt, const PPOISpec *specs,
                              void (*notification)(ptr, byte), ptr usercx)
{
    int rc;
    LOCK(apimutex);
    ENTER_API("(specCnt: %u, specs: %p, notification: %p, usercx: %p)",
        specCnt, specs, notification, usercx);

    rc = sxm_grid_set_ppoi_list(SXM_2GRID(service),
        specCnt, specs, notification, usercx);

    LEAVE_API("%s", sxm_sdk_format_result(rc));
    UNLOCK(apimutex);
    return rc;
}

/*****************************************************************************//**
 *
 * This function begins an POI Parking extraction.
 *
 * \note Calls to sxm_parking_request, sxm_parking_modify and sxm_parking_set_ppoi_list
 *       will return SXM_E_BUSY until sxm_parking_poi_end is called.
 *
 * \param[in] mbr        POI extraction MBR.
 * \param[in] filterMask condition for extracted entry. The filter mask is defined
 *                       by SXM_LOCATION_IS_IN_PPOIS, SXM_LOCATION_IS_IN_COLLECTION, 
 *                       SXM_LOCATION_IS_IN_EXTRACTION and SXM_LOCATION_ALL. 
 *                       8 LSB are reserved for own application flags which will be used
 *                       for PPOI filtrations.
 *
 * \retval SXM_E_OK      POI extraction has been started.
 * \retval SXM_E_BUSY    POI extraction in progress.
 * \retval SXM_E_INVAL   invalid MBR provided
 * \retval SXM_E_NOENT   there is no station inside the MBR or for the mask.
 * \retval SXM_E_STATE   service is not started
 *
 ********************************************************************************/
int sxm_parking_poi_begin(const SXMMBR *mbr, ushort filterMask) {
    int rc;

    LOCK(apimutex);
    ENTER_API("(mbr: %p, filterMask: %x)", mbr, filterMask);

    rc = sxm_grid_poi_begin(SXM_2GRID(service), mbr, filterMask);

    LEAVE_API("%s", sxm_sdk_format_result(rc));
    UNLOCK(apimutex);
    return rc;
}

/*****************************************************************************//**
 *
 * This function terminates the POI extraction and allows calls to sxm_parking_request,
 * sxm_parking_modify and sxm_parking_set_ppoi_list to proceed without returning
 * SXM_E_BUSY.
 *
 * \retval SXM_E_OK Extraction ended successfully.
 * \retval SXM_E_STATE Service is not started or POI extraction not in progress
 *
 ********************************************************************************/
int sxm_parking_poi_end(void) {
    int rc;
    LOCK(apimutex);
    ENTER_API("");

    rc = sxm_grid_poi_end(SXM_2GRID(service));

    LEAVE_API("%s", sxm_sdk_format_result(rc));
    UNLOCK(apimutex);
    return rc;
}

/*****************************************************************************//**
 *
 * This function extracts the next individual parking location from the POI
 * extracted MBR in accordance with POI filter
 *
 * \param[out] pLocation a pointer to an SXM Parking Location that will be
 *                       filled up with parking data, see \ref SXMParkingLocation
 *
 * \retval SXM_E_OK      data has been extracted.
 * \retval SXM_E_STATE   Service in wrong state or POI extraction has not been activated
 * \retval SXM_E_FAULT   NULL parameter. 
 * \retval SXM_E_NOENT   There are no more entries. All entries have been extracted.
 * \retval SXM_E_INVAL   POI extraction is inactive
 *
 ********************************************************************************/
int sxm_parking_poi_extract_station(SXMParkingLocation *pLocation)
{
    int rc;
    LOCK(apimutex);
    ENTER_API("(pLocation: %p)", pLocation);

    rc = sxm_grid_poi_extract_station(SXM_2GRID(service), pLocation);

    LEAVE_API("%s", sxm_sdk_format_result(rc));
    UNLOCK(apimutex);
    return rc;
}

/***************************************************************************//**
 * The routine flushes any cached parking service data required to be persisted
 * over power-cycles to the persisted file system such as non-volatile memory (NVM). 
 *
 * \param[in,out] pStat the I/O statistics to update
 * \param[in] pArg the callback argument
 *
 * \retval  SXM_E_OK     Data has been written to file system successfully.
 * \retval  SXM_E_STATE  Service data not available
 * \retval  SXM_E_PIPE   Error writing service data to file system
 *
 ********************************************************************************/
static int parking_save_data(SXMIOStat *pStat, void *pArg) {
    int rc;
    LOCK(apimutex);
    ENTER_LLI("pStat: %p, pArg: %p", pStat, pArg);
    rc = sxm_grid_save_data((SXMGridService*)pArg,
                            PARKING_SECTION_FIRST, PARKING_SECTIONS_COUNT,
                            PARKING_UPDATE_ALL, pStat);

    LEAVE_LLI("%s", sxm_sdk_format_result(rc));
    UNLOCK(apimutex);
    return rc;
}

/***************************************************************************//**
 *
 * The sxm_parking_save_data routine flushes any cached parking service data
 * required to be persisted over power-cycles to the persisted file system 
 * such as non-volatile memory (NVM). 
 *
 * \retval  SXM_E_OK     Data has been written to file system successfully.
 * \retval  SXM_E_STATE  Service data not available
 * \retval  SXM_E_PIPE   Error writing service data to file system
 *
 ********************************************************************************/
int sxm_parking_save_data(void) {
    return parking_save_data(NULL, SXM_2GRID(service));
}

/*********************************************************************//**
 *
 * The sxm_parking_set_auto_save function allows the application to enable 
 * or disable the sxm_parking_stop routine from writing the cached parking
 * service data to the file system.
 *
 * \note if the application disables writing of parking cached data during
 * the stop function then the application shall be responsible for persisting
 * this data across service restarts and power cycling using 
 * the sxm_parking_save_data api call.
 *
 * \param[in]            auto_save_on_stop controls the autosave of cached parking
 *                       service data on call to sxm_parking_stop().
 *
 * \retval  SXM_E_OK     Request to save data on stop is successfully.
 * \retval  SXM_E_STATE  Service data not available
 *
************************************************************************/
int sxm_parking_set_auto_save(BOOL auto_save_on_stop)
{
    int ret;
    LOCK(apimutex);
    ENTER_API("(auto_save_on_stop = %d)", auto_save_on_stop);

    ret = sxm_grid_set_auto_save(SXM_2GRID(service), auto_save_on_stop);

    LEAVE_API("%s", sxm_sdk_format_result(ret));
    UNLOCK(apimutex);
    return ret;
}
