/******************************************************************************/
/*                Copyright (c) Sirius XM Satellite Radio, Inc.               */
/*                            All Rights Reserved                             */
/*      Licensed Materials - Property of Sirius XM Satellite Radio, Inc.      */
/******************************************************************************/
/*******************************************************************************
*
*   $Workfile:      sxm_sxi.h $
*   $Author(s):     Victor Abreu $
*   $Creation Date: 11/14/2013 $
*
*   DESCRIPTION
*
*       External Definition for the SXI Protocol.
*
*******************************************************************************/


#ifndef SXM_SXI_H
#define SXM_SXI_H

#include "sxm_build.h"

#if (!SXM_USE_GEN8) 
#include <sxm_common.h>

#if (SXM_USE_GMD)
#include "sxm_gmd.h"
#endif

#ifdef __cplusplus
extern "C" {
#endif

/****************************/
/* SXI INDICATION CODE      */
/****************************/
#define IC_NOMINAL              0x00

#define IC_ADD_UPDATE           0x01
#define IC_DELETE               0x02
#define IC_FAIL                 0x03
#define IC_NO_CHAN              0x04
#define IC_CHECK_ANTENNA        0x05
#define IC_NO_SIGNAL            0x06
#define IC_SUB_UPDATE           0x07
#define IC_CHAN_UNAVAIL         0x08
#define IC_CHAN_UNSUB           0x09
#define IC_CHAN_LOCKED          0x0A
#define IC_CHAN_MATURE          0x0B
#define IC_RESOURCE_LIMIT       0x1E
#define IC_NO_TRACKS            0x1F
#define IC_SCAN_NOMINAL         0x20
#define IC_CHAN_AUDIO_UNAVAIL   0x21
#define IC_RECORD_FAIL          0x22
#define IC_SCAN_ABORTED         0x23
#define IC_TUNEMIX_NOMINAL      0x24
#define IC_BULLETIN_NOMINAL     0x25
#define IC_BULLETIN_UNAVAIL     0x26
#define IC_FLASH_EVENT_NOMINAL  0x27
#define IC_FLASH_EVENT_UNAVAIL  0x28
#define IC_SEEK_END             0x29

/*******************************************/
/* I2S Parameters (USE IT ON YOUR OWN RISK)*/
/*******************************************/
#define SXM_SXI_I2S_FRAME_32BIT 0x20
#define SXM_SXI_I2S_FRAME_24BIT 0x10
#define SXM_SXI_I2S_FRAME_16BIT 0x00

/***********************************/
/* POWER MODES                     */
/***********************************/
#define SXI_PWR_MODE_DOWN           0
#define SXI_PWR_FACTORY_DEFAULT     1
#define SXI_PWR_ENTER_LOW_POWER     2
#define SXI_PWR_EXIT_LOW_POWER      3

/***********************************/
/* SXI PID                         */
/***********************************/
#define SXI_PID_TOGGLE_BIT_MASK         0x80000000u

/***********************************/
/* SXI EVENT                       */
/***********************************/
#define EV_SYS_INFO             0x00
#define EV_SYS_RESET            0x01
#define EV_SYS_ACTION           0x02

/***********************************/
/* CONFIGURABLE CHANNEL ATTRIBUTES */
/***********************************/
#define SXI_ATTRIBUTE_APPLY_SKIP        0
#define SXI_ATTRIBUTE_APPLY_LOCK        1
#define SXI_ATTRIBUTE_SCAN_EXCLUDE      2

/***********************************/
/* OVERRIDES                       */
/***********************************/
#define SXI_OVERRIDE_ALL_DISABLED                 0x00
#define SXI_OVERRIDE_PLAY_UNRESTRICTED_ENABLED    0x01
#define SXI_OVERRIDE_UNSUBSCRIBED_ENABLED         0x02
#define SXI_OVERRIDE_MATURE_ENABLED               0x04
#define SXI_OVERRIDE_LOCKED_ENABLED               0x08
#define SXI_OVERRIDE_SKIPPED_ENABLED              0x10
#define SXI_OVERRIDE_FREE_TO_AIR_ONLY_ENABLED     0x20

/***********************************/
/* CHANNEL ATTRIBUTES              */
/***********************************/
#define SXI_CHAN_ATTRIBUTE_NOT_AVAILABLE        0x01
#define SXI_CHAN_ATTRIBUTE_NOT_SUBSCRIBED       0x02
#define SXI_CHAN_ATTRIBUTE_MATURE               0x04
#define SXI_CHAN_ATTRIBUTE_LOCKED               0x08
#define SXI_CHAN_ATTRIBUTE_SKIPPED              0x10
#define SXI_CHAN_ATTRIBUTE_FREE_TO_AIR          0x20

/***********************************/
/* RECORD RESTRICTIONS             */
/***********************************/
#define SXI_RECORD_PERMITTED                    0x00
#define SXI_RECORD_CONTENT_RESTRICTED           0x01
#define SXI_RECORD_CHAN_RESTRICTED              0x02

/***********************************/
/* TIME ZONES                      */
/***********************************/
#define SXI_NEWFOUNDLAND_TIME_ZONE  0
#define SXI_ATLANTIC_TIME_ZONE      1
#define SXI_EASTERN_TIME_ZONE       2
#define SXI_CENTRAL_TIME_ZONE       3
#define SXI_MOUNTAIN_TIME_ZONE      4
#define SXI_PACIFIC_TIME_ZONE       5
#define SXI_ALASKA_TIME_ZONE        6
#define SXI_HAWAIIAN_TIME_ZONE      7
#define SXI_UTC_TIME_ZONE           255

/***********************************/
/* DST ADJUSTMENTS                 */
/***********************************/
#define SXI_DISABLE_DST_ADJUSTMENT  0
#define SXI_AUTO_DST_ADJUSTMENT     1

/***********************************/
/* AUDIO MUTE                      */
/***********************************/
#define SXI_UNMUTE                  0
#define SXI_MUTE_AUDIO              1
#define SXI_MUTE_AUDIO_AND_I2S      2

/***********************************/
/* AUDIO TONE                      */
/***********************************/
#define SXI_BASS_MIN_NUM            -12
#define SXI_BASS_MAX_NUM             12

#define SXI_TREBLE_MIN_NUM          -12
#define SXI_TREBLE_MAX_NUM           12

/***********************************/
/* AUDIO EQUALIZER                 */
/***********************************/
#define SXI_BAND_GAIN_MIN_NUM       -12
#define SXI_BAND_GAIN_MAX_NUM       12

/***********************************/
/* AUDIO VOLUME ADJUSTMENT         */
/***********************************/
#define SXI_VOLUMEADJ_MIN_NUM       SXI_SP_VOLADJ_MIN
#define SXI_VOLUMEADJ_MAX_NUM       SXI_SP_VOLADJ_MAX

/***********************************/
/* TONE GENERATION                 */
/***********************************/
#define SXI_TONEFREQCONTROL_MIN_NUM     0
#define SXI_TONEFREQCONTROL_MAX_NUM     255

#define SXI_AUDIO_TONE_ALERT_DISABLED   0x00
#define SXI_AUDIO_TONE_ALERT_SUB_NOTIFY 0x04
#define SXI_AUDIO_TONE_ALERT_1          0x08

#define SXI_AUDIO_TONE_BOTH_DISABLED    0x00
#define SXI_AUDIO_TONE_RIGHT_ENABLED    0x01
#define SXI_AUDIO_TONE_LEFT_ENABLED     0x02
#define SXI_AUDIO_TONE_BOTH_ENABLED     0x03
#define SXI_AUDIO_TONE_RIGHT_LEFT_MASK  0x03

#define SXI_TONEVOLUEADJ_MIN_NUM        -64
#define SXI_TONEVOLUEADJ_MAX_NUM        16

/***********************************/
/* IR RECORDING                    */
/***********************************/
#define SXI_IR_RECORDING_NOT_ACTIVE   0
#define SXI_IR_RECORDING_ACTIVE       1

/***********************************/
/* MONITOR CHANGE TYPES            */
/***********************************/
#define SXI_MONITOR_DISABLED             0
#define SXI_MONITOR_ENABLED              1
#define SXI_MONITOR_ALL_DISABLED         2

/***********************************/
/* STATUS MONITOR ITEMS            */
/***********************************/
#define SXI_MONITOR_SIGNAL_STATUS           0x00
#define SXI_MONITOR_ANTENNA_AIMING          0x03
#define SXI_MONITOR_DECODER_BITRATE         0x0B
#define SXI_MONITOR_SIGNAL                  0x0C
#define SXI_MONITOR_OVERLAY_SIGNAL          0x0D
#define SXI_MONITOR_MODULE_VERSION          0x0E
#define SXI_MONITOR_GPS_DATA                0x10
#define SXI_MONITOR_LINK_INFO               0x11
#define SXI_MONITOR_SCAN_AVAILABLE_ITEMS    0x12
#define SXI_MONITOR_AUDIO_PRESENCE          0x13

/***********************************/
/* STATUS VALUES                   */
/***********************************/
/* monitior item 0x00 - offset 0 */
#define SXI_SIGNAL_STATUS_0_BAR     0x00
#define SXI_SIGNAL_STATUS_1_BAR     0x01
#define SXI_SIGNAL_STATUS_2_BAR     0x02
#define SXI_SIGNAL_STATUS_3_BAR     0x03

/* monitior item 0x00 - offset 1 */
#define SXI_ANTENNA_OK              0x00
#define SXI_ANTENNA_DISCONNECTED    0x01
#define SXI_ANTENNA_SHORTED         0x02

/* monitor item 0x0C - offset 0 */
#define SXI_SIG_STRENGTH_NO_SIGNAL  0x00
#define SXI_SIG_STRENGTH_VERY_WEAK  0x01
#define SXI_SIG_STRENGTH_WEAK       0x02
#define SXI_SIG_STRENGTH_MARGINAL   0x03
#define SXI_SIG_STRENGTH_GOOD       0x04

/* monitor item 0x0C - offset 1 */
#define SXI_TUNER_STATUS_IF_AGC_ENG 0x20
#define SXI_TUNER_STATUS_RF_AGC_ENG 0x10
#define SXI_TUNER_STATUS_ANT_OVR    0x08
#define SXI_TUNER_STATUS_ANT_UNDR   0x04
#define SXI_TUNER_STATUS_ANT_DET    0x02
#define SXI_TUNER_STATUS_PLL_LOCK   0x01

/* monitor item 0x0C - offset 2 */
#define SXI_LOCK_STATUS_S1_TDM      0x20
#define SXI_LOCK_STATUS_S1_QPSK     0x10
#define SXI_LOCK_STATUS_S2_TDM      0x08
#define SXI_LOCK_STATUS_S2_QPSK     0x04
#define SXI_LOCK_STATUS_T_TDM       0x02
#define SXI_LOCK_STATUS_T_QPSK      0x01

/* monitor item 0x0D - offset 0 */
#define SXI_RCVR_STATE_PIPE_LCKA    0x80
#define SXI_RCVR_STATE_S1A_OTDM     0x40
#define SXI_RCVR_STATE_S2A_OTDM     0x20
#define SXI_RCVR_STATE_TA_OTDM      0x10
#define SXI_RCVR_STATE_PIPE_LCKB    0x08
#define SXI_RCVR_STATE_S1B_OTDM     0x04
#define SXI_RCVR_STATE_S2B_OTDM     0x02
#define SXI_RCVR_STATE_TB_OTDM      0x01

/* ENGINEERING MONITOR ITEMS */
/* monitor item 0x34 */
#define SXI_MONITOR_DECODER_STATUS_SIZE                 54

/* monitor item 0x35 */
#define SXI_MONITOR_SIGNAL_OFFSETS_SIZE                 28

/* monitor item 0x36 */
#define SXI_MONITOR_OVERLAY_PIPE_CONFIGURATION_SIZE     80

/* monitor item 0x37 */
#define SXI_MONITOR_DATA_LAYER_STATUS_SIZE             120

/* monitor item 0x39 */
#define SXI_MONITOR_MODULE_QUEUE_STATUS_SIZE            5

/* monitor item 0x3B */
#define SXI_MONITOR_AUDIO_DECODER_STATUS_SIZE           7

/* monitor item  0x3C */
#define SXI_MONITOR_UPC_VALUES_SIZE                     24

/* monitor item  0x3D */
#define SXI_MONITOR_EXTENDED_SIGNAL_QUALITY_SIZE        16

/* monitor item  0x3E */
#define SXI_MONITOR_OVERLAY_SIGNAL_QUALITY_SIZE         20

/***********************************/
/* SUBSCRIPTION STATUS             */
/***********************************/
#define SXI_NOT_SUBSCRIBED          0
#define SXI_SUBSCRIBED              1
#define SXI_SUSPEND_ALERT           2
#define SXI_SUSPENDED               3

/*******************************************/
/* USER CATEGORIES                         */
/*******************************************/
#define SXI_USER_CATID_64                   64
#define SXI_USER_CATID_65                   65
#define SXI_USER_CATID_66                   66

/***************************************/
/* MEMORY MONITOR DATA TYPES           */
/***************************************/
#define SXI_MEM_MON_8_BIT                       0
#define SXI_MEM_MON_16_BIT                      1
#define SXI_MEM_MON_32_BIT                      2

/*****************************************/
/*     SPORTS TEAM BIAS DEFINITIONS      */
/*****************************************/
#define SPORTS_BIAS_NOBIAS      0
#define SPORTS_BIAS_HOME        1
#define SPORTS_BIAS_AWAY        2
#define SPORTS_BIAS_ALTERNATE   3

/****************************/
/* FEATURED FAVORITES       */
/****************************/
#define SXI_FEATURED_FAVORITES_PURPOSE_SCANNING         0x02
#define SXI_FEATURED_FAVORITES_PURPOSE_CONVENIENCE      0x01

/****************************/
/* FLASH STATUSES           */
/****************************/
#define SXI_FLASH_EVENT_ALERT           0x00
#define SXI_FLASH_EVENT_EXPIRY          0x01
#define SXI_FLASH_EVENT_PLAYED          0x02

/*****************************************/
/* TMI - TRACK METADATA IDENTIFIERS      */
/*****************************************/
#define SXI_TMI_SONG_ID                     3
#define SXI_TMI_ARTIST_ID                   4
#define SXI_TMI_ALBUM_NAME                  7
#define SXI_TMI_CONTENT_INFO                8
#define SXI_TMI_SPORT_LEAGUE_BROADCAST_ID   10
#define SXI_TMI_SPORT_TEAM_BROADCAST_ID     11
#define SXI_TMI_TRAFFIC_CITY_ID             15
#define SXI_TMI_ITUNES_SONG_ID              16
#define SXI_TMI_SPORTS_BROADCAST_BIAS       280
#define SXI_TMI_SHOW_ID                     290

/***********************************/
/* METADATA FIELD SIZES            */
/***********************************/
/* Metadata field Max Sizes */
#define SXI_CHAN_NAME_SHORT_MAX             (8+1)
#define SXI_CHAN_NAME_MED_MAX               (10+1)
#define SXI_CHAN_NAME_LONG_MAX              (16+1)

#define SXI_CAT_NAME_SHORT_MAX              (8+1)
#define SXI_CAT_NAME_MED_MAX                (10+1)
#define SXI_CAT_NAME_LONG_MAX               (16+1)

#define SXI_ARTIST_NAME_BASIC_MAX           (16+1)
#define SXI_SONG_NAME_BASIC_MAX             (16+1)

#define SXI_ARTIST_NAME_EXTENDED_MAX        (36+1)
#define SXI_SONG_NAME_EXTENDED_MAX          (36+1)

#define SXI_CONTENT_INFO_CHAR_MAX           (36+1)

#define SXI_EXT_CH_METADATA_COUNT_MAX       (32)
#define SXI_EXT_TRACK_METADATA_COUNT_MAX    (32)

/* Ext Channel Metadata Max Sizes */
#define SXI_CHANNEL_SHORT_DESCRIPTION_MAX   (40)
#define SXI_CHANNEL_LONG_DESCRIPTION_MAX    (184)
#define SXI_RELATED_CHAN_LIST_MAX           (8)

/* Ext Track Metadata Max Sizes */
#define SXI_ALBUM_NAME_MAX                  (36 + 1)

/* Max broadcast id's */
#define SXI_SPORT_TEAM_BROADCAST_ID_MAX     (2)

/* featured favorites */
#define SXI_FEATURED_FAVORITES_BANK_TITLE_SHORT_MAX     (8+1)
#define SXI_FEATURED_FAVORITES_BANK_TITLE_LONG_MAX      (16+1)
#define SXI_FEATURED_FAVORITES_BANK_TITLE_VERBOSE_MAX   (36+1)
#define SXI_FEATURED_FAVORITES_BANK_DESCRIPTION_MAX     (180+1)
#define SXI_FEATURED_FAVORITES_BANK_ARRANGEMENT_MAX     34

/* itunes URL */
#define SXI_ITUNES_URL_MAX                              (128 + 1)

/* Global Metadata Max Sizes */
/* sports league */
#define SXI_LEAGUE_SHORT_NAME_MAX           (8+1)
#define SXI_LEAGUE_LONG_NAME_MAX            (36+1)
#define SXI_LEAGUE_TYPE_MAX                 (16+1)

/* sports team */
#define SXI_TEAM_ABBREVIATION_MAX            (6+1)
#define SXI_TEAM_NAME_MAX                   (24+1)
#define SXI_TEAM_NICKNAME_MAX               (16+1)

#define MAX_LEAGUE_PER_TEAM                 (8)
#define MAX_TIER_PER_TEAM                   (8)

/* traffic/weather */
#define SXI_TRAFFIC_WEATHER_ABBREVIATION_MAX    (3+1)
#define SXI_TRAFFIC_WEATHER_NAME_MAX            (24+1)

/* Buffered Content Indication */
#define SXI_BUFFERED_CONTENT_MAX_PIDS_PER_CHANNEL       250

/*******************************************/
/* IP Device Auth Command                  */
/*******************************************/
#define SXI_IP_AUTH_DEVICE_CHALLENGE_REQUEST_LEN (8)

/*************************************/
/* SXI AUTH STATE                    */
/*************************************/
#define SXI_AUTH_STATE_AUTHSTATEPELLET_LEN (22)

/*************************/
/* SXI Auth State        */
/*************************/
typedef struct {
    /** Signed authorization state. It is a 176 bit value (22 bytes). */
    byte AuthStatePellet[SXI_AUTH_STATE_AUTHSTATEPELLET_LEN];
} SXMAuthState;

/*************************************/
/* SXI IP DEVICE AUTH                */
/*************************************/
#define SXI_IP_DEVICE_AUTH_CHALLENGERESPONSE_LEN (22)

/*************************/
/* SXI IP Device Auth    */
/*************************/
typedef struct {
    /** Received ChallengeRequest[] with signature. It is a 176 bit value (22 bytes). */
    byte ChallengeResponse[SXI_IP_DEVICE_AUTH_CHALLENGERESPONSE_LEN];
} SXMIPDeviceAuth;

/*************************/
/* SXM GPS               */
/*************************/
typedef struct {
    SXMFLOAT xm_lat;                                 /* XM Latitude  */
    SXMFLOAT xm_lon;                                 /* XM Longitude */
    byte hours;                                      /* ExtUTCTimeHours */
    byte mins;                                       /* ExtUTCTimeMins */
    byte secs;                                       /* ExtUTCTimeSecs */
    SXMFLOAT ext_lat;                                /* Ext Latitude */
    SXMFLOAT ext_lon;                                /* Ext Longitude */
    int ext_alt;                                     /* Ext Altitude */
    byte ext_sats;                                   /* ExtSatsUsed:  Number of satellites used in the GPS solution */
    byte ext_quality;                                /* ExtQuality: A measure of signal quality from the external GPS */
}SXMGPS;

/*************************/
/* SXM LINK              */
/*************************/
typedef struct {
    uint ctimeout;                                   /* Link Layer Number of confirmation timeouts */
    uint data_drop;                                  /* Link Layer Number of data packets available but not sent to the Host */
    uint data_sent;                                  /* Link Layer Number of data packets transmitted */
    uint audio_drop;                                 /* Number of audio packets available but not sent to the Host */
    uint audio_sent;                                 /* Number of audio packets transmitted */
}SXMLink;

/*****************************************/
/*  CHANNEL CONTENT TYPE DEFINITIONS     */
/*****************************************/
#define CH_CONTENT_TYPE_MUSIC_BY_PID                (0)
#define CH_CONTENT_TYPE_MUSIC_BY_ARTIST_TITLE       (1)
#define CH_CONTENT_TYPE_TALK_BY_PID                 (2)
#define CH_CONTENT_TYPE_TALK_BY_ARTIST_TITLE        (3)
#define CH_CONTENT_TYPE_LIVE_BY_PID                 (4)
#define CH_CONTENT_TYPE_LIVE_BY_ARTIST_TITLE        (5)

/*****************************************/
/*  EXTENDED TRACK METADATA VALID BITS   */
/*****************************************/
#define SONG_ID_VALID_BIT               (0)
#define ARTIST_ID_VALID_BIT             (1)
#define ALBUM_NAME_VALID_BIT            (2)
#define SPORTS_LEAGUE_VALID_BIT         (3)
#define SPORTS_TEAM_VALID_BIT           (4)
#define SPORTS_TEAM_BIAS_VALID_BIT      (5)
#define TRAFFIC_WEATHER_VALID_BIT       (6)
#define ITUNES_ID_VALID_BIT             (7)
#define CONTENT_INFO_VALID_BIT          (8)
#define SHOW_ID_VALID_BIT               (9)

/*************************/
/* SXM PAD METADATA      */
/*************************/
typedef struct {
    uint valid;                                     /* Flags used to mark which fields are actually valid */
    uint songid;                                    /* Song Identifier */
    uint artistid;                                  /* Artist Identifier */
    char albumname[SXI_ALBUM_NAME_MAX];             /* Album Name */
    byte leaguebroadcastid;                         /* Sports League Identifier */
    uint teambroadcastid[SXI_SPORT_TEAM_BROADCAST_ID_MAX];  /* Sports Team Broadcast ID */
    byte teambroadcastbias;                         /* Sports Team Broadcast Bias */
    byte traffic;                                   /* Traffic/Weather Market Broadcast ID */
    uint itunes;                                    /* Apple iTunes Song ID */
    char content[SXI_CONTENT_INFO_CHAR_MAX];        /* Content Information */
    uint showid;                                    /* Show ID */
}SXMPadMeta;

/*************************/
/* SXM PAD DATA          */
/*************************/
typedef struct {
    uint pid;                                       /* Program Identifier */
    byte rr;                                        /* Recording restriction flags */
    char artistb[SXI_ARTIST_NAME_BASIC_MAX];        /* Artist Name, Basic format */
    char songb[SXI_SONG_NAME_BASIC_MAX];            /* Song Title, Basic format */
    char artiste[SXI_ARTIST_NAME_EXTENDED_MAX];     /* Artist Name, Extended format */
    char songe[SXI_SONG_NAME_EXTENDED_MAX];         /* Song Title, Extended format */
    SXMPadMeta m;                                   /* Track metadata */
}SXMPad;

/*************************/
/* SXM PID               */
/*************************/
typedef enum {
    SXM_PID_MUSIC   =  0,
    SXM_PID_SPORTS  =  1,
    SXM_PID_TRAFFIC =  2,
    SXM_PID_OTHER   =  3
} SXMPidType;

typedef struct {
    uint pid;                                       //!< compressed form */
    byte toggle;                                    //!< The toggle bit ensure creation of track/segment markers
                                                    // in legacy receivers at the appropriate times.
    SXMPidType type;                                //!< PID type (MUSIC, SPORTS, TRAFFIC, OTHER)
    union {
        /*************************/
        /* MUSIC PID             */
        /*************************/
        struct {
            uint songid;                            //!< song ID 
            byte subtype;                           //!<music genre
        } music;
         /*************************/
         /* SPORT PID             */
         /*************************/
        struct {
            ushort away;                            //!< SXI_TMI_SPORT_TEAM_BROADCAST_ID, away team
            ushort home;                            //!< SXI_TMI_SPORT_TEAM_BROADCAST_ID, home team
            byte league;                            //!< SXI_TMI_SPORT_LEAGUE_BROADCAST_ID
            byte bflag;                             //!< SXI_TMI_SPORTS_BROADCAST_BIAS
        } sports;
        /*************************/
        /* TRAFFIC PID           */
        /*************************/
        struct {
            byte id;                                //!< SXI_TMI_TRAFFIC_CITY_ID
        } traffic;
        /*************************/
        /* OTHER PID             */
        /*************************/
        struct {
            uint id;                                //!< other format value
            byte type;                              //!< other format type field
            byte subtype;                           //!< other format subtype field
        } other;
    } data;
} SXMPid;

/*************************/
/* SXM LA PAD DATA       */
/*************************/
typedef struct {
    uint pid;                                       /* Program Identifier */
    SXMPadMeta m;                                   /* Track metadata */
}SXMLAPad;

/*************************/
/* SXM SEEK              */
/*************************/
typedef struct {
    SXMPadMeta m;                                   /* Pad metadata for the seek event */
    ushort channel;                                 /* channel number */
    ushort sid;                                     /* SID */
    byte chattr;                                    /* Attributes of the channel(SmartFavorites, Mature,...,etc) */
    uint pid;                                       /* Program Identifier */
    byte id;                                        /* seek monitor ID */
    ushort tmi;                                     /* extended track metadata identifier */
    uint matched;                                   /* TMI matched value */
    BOOL is_end;                                    /* Seek Alert End flag */
}SXMSeek;

/*************************/
/* SXM SUB NOTIFICATION  */
/*************************/
#define SXI_SHORT_MESSAGE_SIZE          17
#define SXI_LONG_MESSAGE_SIZE           65
typedef struct {
    char shortmessage[SXI_SHORT_MESSAGE_SIZE];            /* Suspend Text */
    char shortphone[SXI_SHORT_MESSAGE_SIZE];              /* Suspend Phone Number */
    char longmessage[SXI_LONG_MESSAGE_SIZE];              /* Suspend Text */
    char longphone[SXI_LONG_MESSAGE_SIZE];                /* Suspend Phone Number */
}SXMSubscriptionNotification;

/*************************/
/* SXM SUBSCRIPTION      */
/*************************/
#define SXI_RADIOID_SIZE                9
#define SXI_REASON_TEXT_SIZE        (64+1)
#define SXI_PHONE_NUMBER_SIZE       (10+1)
typedef struct {
    byte code;                                      /* SXI Indication (IC_NOMINAL, IC_ERROR_?) */
    char id[SXI_RADIOID_SIZE];                      /* Radio Identification */
    byte status;                                    /* Subscription Status (Not Subscribed, Subscribed, ...) */
    byte reason;                                    /* Reason Code - codes correspond to text messages the Host can display */
    byte day;                                       /* Suspend Day */
    byte month;                                     /* Suspend Month */
    byte year;                                      /* Suspend Day */
    char text[SXI_REASON_TEXT_SIZE];                /* Suspend Text */
    char phone[SXI_PHONE_NUMBER_SIZE];              /* Suspend Phone Number */
    uint ivsm_id;                                   /* {SXi V3.5}: Module ID reported to the Host for management of In Vehicle Subscriber Messaging (IVSM). */
    byte extended_status;                           /* {SXi V3.5}: This field is used to communicate the details of the subscription status of the module. */
}SXMSubscription;

/*************************/
/* SXM ENG AUDIO DECODER */
/*************************/
typedef struct {
    byte eng[SXI_MONITOR_AUDIO_DECODER_STATUS_SIZE];        /* see Appendix I of SX-9845-0097 document */
}SXMEngAudioDecoder;

/*************************/
/* SXM ENG DATA LAYER    */
/*************************/
typedef struct {
    byte eng[SXI_MONITOR_DATA_LAYER_STATUS_SIZE];           /* see Appendix I of SX-9845-0097 document */
}SXMEngDataLayer;

/*************************/
/* SXM ENG DEBUG         */
/*************************/
#define SXI_DEBUG_MODULE_IND_TEXT_MAX_SIZE  126
typedef struct {
    ushort id;                                              /* Debug message identifier which can be used to reduce the
                                                               amount of text needed for the debug message by specifying a
                                                               pre-defined text string.*/
    char msg[SXI_DEBUG_MODULE_IND_TEXT_MAX_SIZE];           /* Debug message text string. */
}SXMEngDebug;

/*************************/
/* SXM ENG DECODER       */
/*************************/
typedef struct {
    byte eng[SXI_MONITOR_DECODER_STATUS_SIZE];              /* see Appendix I of SX-9845-0097 document */
}SXMEngDecoder;

/*************************/
/* SXM FACTORY           */
/*************************/
typedef struct {
    byte code;                                       /* SXI Indication code (Nominal, Invalid Image,...,Invalid CRC) */
    char id[SXI_RADIOID_SIZE];                       /* Radio Identification */
}SXMFactory;

/*************************/
/* SXM SXI PACKAGE       */
/*************************/
/* Option */
#define SXI_FA_OPTION_QUERY             0x00
#define SXI_FA_OPTION_SELECT            0x01
#define SXI_FA_OPTION_VALIDATE          0x02
#define SXI_FA_OPTION_REPORT            0x03
#define SXI_FA_OPTION_MAX_NUM           SXI_FA_OPTION_REPORT

/* PkgIndex */
#define SXI_FA_PKGINDEX_MAX_NUM         127

/*************************/
/* SXM ENG MEMORY        */
/*************************/
#define SXI_MEM_MON_IND_DATA_8BIT_MAX_SIZE      2048
#define SXI_MEM_MON_IND_DATA_16BIT_MAX_SIZE     1024
#define SXI_MEM_MON_IND_DATA_32BIT_MAX_SIZE     512
typedef struct {
    byte mid;                                               /* MemoryMonitorID - ID assigned to this memory monitor */
    byte type;                                              /* Number of data items read. */
    ushort count;                                           /* Type of data items read. Is it 8 bits, 16 bits, or 32 bits ? */
    union {
    byte b[SXI_MEM_MON_IND_DATA_8BIT_MAX_SIZE];             /* storage for 8 bit data type */
    ushort s[SXI_MEM_MON_IND_DATA_16BIT_MAX_SIZE];          /* storage for 16 bit data type */
    uint i[SXI_MEM_MON_IND_DATA_32BIT_MAX_SIZE];            /* storage for 32 bit data type */
    }data;
}SXMEngMemory;

/*************************/
/* SXM ENG OFFSET        */
/*************************/
typedef struct {
    byte eng[SXI_MONITOR_SIGNAL_OFFSETS_SIZE];              /*see Appendix I of SX-9845-0097 document */
}SXMEngOffset;

/*************************/
/* SXM ENG PIPE          */
/*************************/
typedef struct {
    byte eng[SXI_MONITOR_OVERLAY_PIPE_CONFIGURATION_SIZE];  /*see Appendix I of SX-9845-0097 document */
}SXMEngPipe;

/*************************/
/* SXM ENG QUALITY       */
/*************************/
typedef struct {
    byte eng[SXI_MONITOR_EXTENDED_SIGNAL_QUALITY_SIZE];     /*see Appendix I of SX-9845-0097 document */
    byte oeng[SXI_MONITOR_OVERLAY_SIGNAL_QUALITY_SIZE];     /*see Appendix I of SX-9845-0097 document */
}SXMEngQuality;

/*************************/
/* SXM ENG QUEUE         */
/*************************/
typedef struct {
    byte eng[SXI_MONITOR_MODULE_QUEUE_STATUS_SIZE];         /*see Appendix I of SX-9845-0097 document */
}SXMEngQueue;

/*************************/
/* SXM ENG UPC           */
/*************************/
typedef struct {
    byte eng[SXI_MONITOR_UPC_VALUES_SIZE];           /*see Appendix I of SX-9845-0097 document */
}SXMEngUPC;

/*************************/
/* SXI Statistics        */
/*************************/
typedef struct {
    uint cmd_count; //!< Number of issued commands
    uint cmd_retries_count; //!< Total number of command retries
    uint cmd_error_count; //!< Number of failed attempts to send command
    uint cmd_timeouts; //!< Number of reported command-send timeouts
    uint resp_count; //!< Number of received responses
    uint ind_count; //!< Total number of received indications
    uint ind_unknown_count; //!< Number of received unknown indications
    uint cfm_count; //!< Number of issued confirmations
    uint cfm_error_count; //!< Number of failed attempts to send confirmation
    uint queue_overflow; //!< Number of queue overflow situations
} SXMSXiStats;

/****************************/
/* SXM PRESET               */
/* (aka featured Favorites) */
/****************************/
#if (!SXM_USE_GMD)
typedef struct {
    byte id;                                                             /* SXI_GMI_FEATURED_FAVORITES_BANK_ID */
    byte order;                                                          /* SXI_GMI_FEATURED_FAVORITES_BANK_ORDER */
    byte seq;                                                            /* SXI_GMI_FEATURED_FAVORITES_BANK_SEQUENCE */
    byte purpose;                                                        /* SXI_GMI_FEATURED_FAVORITES_BANK_PURPOSE */
    char sn[SXI_FEATURED_FAVORITES_BANK_TITLE_SHORT_MAX];               /* SXI_GMI_FEATURED_FAVORITES_BANK_TITLE_SHORT */
    char mn[SXI_FEATURED_FAVORITES_BANK_TITLE_LONG_MAX];                /* SXI_GMI_FEATURED_FAVORITES_BANK_TITLE_LONG */
    char ln[SXI_FEATURED_FAVORITES_BANK_TITLE_VERBOSE_MAX];             /* SXI_GMI_FEATURED_FAVORITES_BANK_TITLE_VERBOSE */
    char en[SXI_FEATURED_FAVORITES_BANK_DESCRIPTION_MAX];               /* SXI_GMI_FEATURED_FAVORITES_BANK_DESCRIPTION */
    ushort bank1[SXI_FEATURED_FAVORITES_BANK_ARRANGEMENT_MAX];          /* SXI_GMI_FEATURED_FAVORITES_BANK_ARRANGEMENT1 */
    ushort bank2[SXI_FEATURED_FAVORITES_BANK_ARRANGEMENT_MAX];          /* SXI_GMI_FEATURED_FAVORITES_BANK_ARRANGEMENT2 */
    ushort bank3[SXI_FEATURED_FAVORITES_BANK_ARRANGEMENT_MAX];          /* SXI_GMI_FEATURED_FAVORITES_BANK_ARRANGEMENT3 */
    ushort bank4[SXI_FEATURED_FAVORITES_BANK_ARRANGEMENT_MAX];          /* SXI_GMI_FEATURED_FAVORITES_BANK_ARRANGEMENT4 */
    int bc1;                                                            /* SXI_GMI_FEATURED_FAVORITES_BANK_ARRANGEMENT1 */
    int bc2;                                                            /* SXI_GMI_FEATURED_FAVORITES_BANK_ARRANGEMENT2 */
    int bc3;                                                            /* SXI_GMI_FEATURED_FAVORITES_BANK_ARRANGEMENT3 */
    int bc4;                                                            /* SXI_GMI_FEATURED_FAVORITES_BANK_ARRANGEMENT4 */
}SXMPreset;
#endif

/****************************/
/* iTune URL                */
/****************************/
typedef struct {
    char    url[SXI_ITUNES_URL_MAX];                                    /* URL for the iTunes Song Tagging applications */
} SXMiTune;

/***************************/
/* Channel Update Callback */
/***************************/
typedef enum {
    SXM_SXI_CHANNEL_ADDED,
    SXM_SXI_CHANNEL_REMOVED,
    SXM_SXI_CHANNEL_UDPATED,
    SXM_SXI_CHANNEL_LINEUP,
    SXM_SXI_CHANNEL_UNKNOWN
} SXMSXiChannelAction;

/* parameter size */
#define SXI_REVISION_SIZE       3

/*************************/
/* SXM RADIO             */
/*************************/
typedef struct {
    byte type[SXI_REVISION_SIZE];       /* Identifiers indicate the type and hardware version information */
    byte hw_rev[SXI_REVISION_SIZE];     /* Module Product hardware revision information. */
    byte sw_rev[SXI_REVISION_SIZE];     /* Module Product software revision information */
    byte sxi_rev[SXI_REVISION_SIZE];    /* Module SXI Protocol software revision information */
    byte bb_rev[SXI_REVISION_SIZE];     /* Base Band chip set revision information */
    byte hdec_rev[SXI_REVISION_SIZE];   /* HDEC chip set revision information */
    byte rf_rev[SXI_REVISION_SIZE];     /* RF chip set revision information */
    byte spl_rev[SXI_REVISION_SIZE];    /* Module Serial Program Loader (SPL) software revision information */
    uint cap;                           /* 32-bit bitfield reporting the Module features and capabilities. */
    ushort blen;                        /* Instant Replay buffer capacity in seconds. */
    byte max_smf;                       /* Maximum number of Smart Favorites supported by Module */
    byte max_tmix;                      /* Maximum number of Tune Mix Channels supported by the Module */
    byte max_sptfl;                     /* Maximum number of Sports flash channels supported by the Module */
    byte max_twnow;                     /* Maximum number of Traffic/Weather Now Channels supported by the Module */
} SXMRadio;

#define SXM_SXI_CHANNEL_PACK(_action, _sid) (((int)(_action) << 16) | (_sid))
#define SXM_SXI_CHANNEL_UNPACK_ACTION(_param) (SXMSXiChannelAction)(((_param) >> 16) & 0xFFFF)
#define SXM_SXI_CHANNEL_UNPACK_SID(_param) ((_param) & 0xFFFF)


/*************************/
/* SXI SERVICE           */
/*************************/
extern DEPRECATED SXESDK_API int sxm_sxi_save_data(void);

extern DEPRECATED SXESDK_API int sxm_sxi_set_auto_save(BOOL);


/*************************/
/* SXI MESSAGES          */
/*************************/
extern SXESDK_API int sxm_sxi_package(int Option, int PkgIndex);

extern SXESDK_API int sxm_sxi_status_mon(int MonitorChangeType, int StatusListCnt, byte *StatusMonitorItem);

extern SXESDK_API int sxm_sxi_tonegen(int ToneFreqControl, int AudioToneOptions, int ToneVolumeAdj);

extern SXESDK_API int sxm_sxi_ip_device_auth(byte ChallengeRequest[SXI_IP_AUTH_DEVICE_CHALLENGE_REQUEST_LEN]);

extern SXESDK_API int sxm_sxi_auth_state(void);

extern SXESDK_API int sxm_sxi_ping(void);

extern SXESDK_API int sxm_sxi_debug(int DebugEnable, int CharacterSet, int KeepaliveDisable);

extern SXESDK_API int sxm_sxi_debug_reset(int ResetSelection);

extern SXESDK_API int sxm_sxi_debug_monitor(int MemoryMonitorID, int Region , int Address, int NumReads, int DataType, int ReadInterval);

extern SXESDK_API int sxm_sxi_debug_unmonitor(int MemoryMonitorID);

extern SXESDK_API int sxm_sxi_debug_write(int Region, int Address, int NumWrites, int DataType, ptr Data);

extern SXESDK_API int sxm_sxi_debug_tunnel(int NumMsgBytes, byte *Message);

extern SXESDK_API int sxm_sxi_debug_activate(int Delay, int RetryCnt, const char *Model_String);

extern SXESDK_API int sxm_sxi_i2s_set_frame_len(int len);


/*************************/
/* SXI EXTRACTIONS       */
/*************************/
extern SXESDK_API int sxm_sxi_extract_auth_state(SXMAuthState *ret);

extern SXESDK_API int sxm_sxi_extract_gps(SXMGPS *ret);

extern SXESDK_API int sxm_sxi_extract_link(SXMLink *ret);

extern SXESDK_API int sxm_sxi_extract_stat(SXMSXiStats *pRet);

extern SXESDK_API int sxm_sxi_extract_sub_notification(SXMSubscriptionNotification *ret, const SXMSubscription *p, SXMTime sxmtime, UINT8 *sn_state);

extern SXESDK_API int sxm_sxi_extract_ip_device_auth(SXMIPDeviceAuth *ret);

extern SXESDK_API int sxm_sxi_extract_eng_audio_decoder(SXMEngAudioDecoder *ret);

extern SXESDK_API int sxm_sxi_extract_eng_data_layer(SXMEngDataLayer *ret);

extern SXESDK_API int sxm_sxi_extract_eng_debug(SXMEngDebug *ret);

extern SXESDK_API int sxm_sxi_extract_eng_decoder(SXMEngDecoder *ret);

extern SXESDK_API int sxm_sxi_extract_eng_fa(SXMFactory *ret);

extern SXESDK_API int sxm_sxi_extract_eng_memory(SXMEngMemory *ret);

extern SXESDK_API int sxm_sxi_extract_eng_mfc(uint *ret);

extern SXESDK_API int sxm_sxi_extract_eng_offset(SXMEngOffset *ret);

extern SXESDK_API int sxm_sxi_extract_eng_pipe(SXMEngPipe *ret);

extern SXESDK_API int sxm_sxi_extract_eng_quality(SXMEngQuality *ret);

extern SXESDK_API int sxm_sxi_extract_eng_queue(SXMEngQueue *ret);

extern SXESDK_API int sxm_sxi_extract_eng_upc(SXMEngUPC *ret);

extern SXESDK_API int sxm_sxi_extract_radio(SXMRadio *ret);

/*************************/
/* SXI EXTENDED METADATA */
/*************************/
extern SXESDK_API int sxm_sxi_global_table_begin(void);

extern SXESDK_API int sxm_sxi_global_table_end(void);

extern SXESDK_API int sxm_sxi_global_league_version(int *out);

extern SXESDK_API int sxm_sxi_global_preset_count(int *out);

extern SXESDK_API int sxm_sxi_global_preset(int s, SXMPreset *out);

extern SXESDK_API int sxm_preset_begin(void);

extern SXESDK_API int sxm_preset_end(void);

extern SXESDK_API int sxm_sxi_global_preset_ix(int ix, SXMPreset *out);

extern SXESDK_API int sxm_sxi_global_preset_version(int *out);

extern SXESDK_API int sxm_sxi_global_team_version(int *out);

extern SXESDK_API int sxm_sxi_global_traffic_version(int *out);

extern SXESDK_API int sxm_sxi_global_itune_url(SXMiTune *out);

extern SXESDK_API int sxm_sxi_global_itune_url_version(int *out);


/*************************/
/* SXI LOOKUP            */
/*************************/
extern SXESDK_API ushort sxm_sxi_sid_to_chno(ushort sid);

extern SXESDK_API ushort sxm_sxi_chno_to_sid(ushort chno);

#ifdef __cplusplus
}
#endif

#endif /* !SXM_USE_GEN8 */


#endif /* SXM_SXI_H */

