/************************************************************************
 *                                                                      *
 *            SXM_STDTRACE.H                                            *
 *            ==============                                            *
 *                                                                      *
 *  Copyright 2013 Sirius XM Radio, Inc.                                *
 *  All Rights Reserved.                                                *
 *  Licensed Materials - Property of Sirius XM Radio, Inc.              *
 *                                                                      *
 *  The standard set of tracing macros                                  *
 *                                                                      *
 *  Debug Areas (functional layers):                                    *
 *      API - top-level API                                             *
 *      USR - user's own area of tracing                                *
 *      REQ - request processing                                        *
 *      EXT - extraction processing                                     *
 *      PKT - packet/protocol processing                                *
 *      UPD - RFD-style updates proceccing                              *
 *      DBF - database/baseline processing                              *
 *      LLI - low-level interface (LLI) processing                      *
 *      UTL - internal utilities                                        *
 *                                                                      *
 *  Debug Types (debug message type or reason):                         *
 *      CALL    - Entry to and exit from the SXe SDK routines           *
 *                as well as top-level calllbacks invocation.           *
 *      EVENT   - Indication of major data processing events            *
 *                such as completion of protocol data assembling        *
 *                or data availability/expiration.                      *
 *      DEBUG   - Details of the processing.                            *
 *      ERROR   - All errors including system call errors,              *
 *                logic errors, and protocol violation errors.          *
 *      WARNING - Warnings when module operates normally                *
 *                but not optimally. Example: unsupported or            *
 *                excess data detected.                                 *
 *      DUMP    - Contents of various data structures printed in hex.   *
 *      PLOG    - Special Debug Type for critical error messages which  *
 *                are always printed out in both debug and production   *
 *                build configurations regardless of debug settings.    *
 *                                                                      *
 ************************************************************************/

#ifndef SXM_STDTRACE_H
#define SXM_STDTRACE_H

#include <stddef.h>
#include <sxm_build.h>

#ifdef __cplusplus
extern "C" {
#endif

#define SXM_DBG_AREA_NUM_BITS  (16U)
#define SXM_DBG_TYPE_NUM_BITS  (16U)
#define SXM_DBG_AREA_BIT_MASK  ((1 << SXM_DBG_AREA_NUM_BITS) - 1)
#define SXM_DBG_TYPE_BIT_MASK  ((1 << SXM_DBG_AREA_NUM_BITS) - 1)

/** Debug Areas (functional layers).
  * Each value corresponds to a bit number (index)
  * in the most significant 16-bits debug mask.
  */
typedef enum
{
    /** Top-level API */
    SXM_DBG_AREA_API,

    /** Extra debug area which can be used
      * by the SXe SDK user as required
      */
    SXM_DBG_AREA_USR,

    /* Request processing */
    SXM_DBG_AREA_REQ,

    /** Extraction processing */
    SXM_DBG_AREA_EXT,

    /** Packet/protocol processing */
    SXM_DBG_AREA_PKT,

    /** RFD-style updates proceccing */
    SXM_DBG_AREA_UPD,

    /** Database/baseline processing */
    SXM_DBG_AREA_DBF,

    /** Low-level interface (LLI) processing */
    SXM_DBG_AREA_LLI,

    /** Internal utilities */
    SXM_DBG_AREA_UTL,

    /* NOTE: Following value must be the last in this enum */
    SXM_DBG_NUM_AREAS

} SxmDebugArea;

/** Debug Types (debug message types or reasons).
  * Each value corresponds to a bit number (index)
  * in the least significant 16-bits debug mask.
  */
typedef enum
{
    /** Entry to and exit from the SXe SDK routines
      * as well as top-level calllbacks invocation.
      */
    SXM_DBG_TYPE_CALL,

    /** Indication of major data processing events
      * such as completion of protocol data assembling
      * or data availability/expiration.
      */
    SXM_DBG_TYPE_EVENT,

    /** Details of the processing */
    SXM_DBG_TYPE_DEBUG,

    /** All errors including system call errors,
      * logic errors, and protocol violation errors.
      */
    SXM_DBG_TYPE_ERROR,

    /** Warnings when module operates normally but not optimally.
      * Example: unsupported or excess data detected.
      */
    SXM_DBG_TYPE_WARN,

    /** Contents of various data structures
      * printed in hexadecimal representation
      */
    SXM_DBG_TYPE_DUMP,

    /* NOTE: Following values must be the last in this enum */
    /** Production error log */
    SXM_DBG_TYPE_PLOG,
    SXM_DBG_NUM_TYPES

} SxmDebugType;

/** Macros to encode Debug Area and Debug Type */

#define SXM_DEBUG_MASK_VAL(_dbgArea, _dbgType) \
    (((1 << (_dbgArea)) << SXM_DBG_TYPE_NUM_BITS) | ( 1 << (_dbgType)))

#define SXM_DEBUG_ADD_AREA(_var, _dbgArea) \
    (_var |= ((1 << (_dbgArea)) << SXM_DBG_TYPE_NUM_BITS))

#define SXM_DEBUG_REMOVE_AREA(_var, _dbgArea) \
    (_var &= ~((1 << (_dbgArea)) << SXM_DBG_TYPE_NUM_BITS))

#define SXM_DEBUG_SET_AREA_FOR_ALL_TYPES(_var, _dbgArea) \
    (_var |= (((1 << (_dbgArea)) << SXM_DBG_TYPE_NUM_BITS) | ((1 << SXM_DBG_NUM_TYPES) - 1)))

#define SXM_DEBUG_ADD_TYPE(_var, _dbgType) \
    (_var |= (1 << (_dbgType)))

#define SXM_DEBUG_REMOVE_TYPE(_var, _dbgType) \
    (_var &= ~(1 << (_dbgType)))

#define SXM_DEBUG_SET_TYPE_FOR_ALL_AREAS(_var, _dbgType) \
    (_var |= ((((1 << SXM_DBG_NUM_AREAS) - 1) << SXM_DBG_TYPE_NUM_BITS) | (1 << (_dbgType))))

#define SXM_DEBUG_ADD(_var, _dbgArea, _dbgType) \
    (_var |= SXM_DEBUG_MASK_VAL((_dbgArea), (_dbgType)))

/** Macros to decode Debug Area and Debug Type */

#define SXM_DEBUG_IS_AREA_SET(_var, _dbgArea) \
    (0 != ((uint)(_var) &((1U << (_dbgArea)) << SXM_DBG_TYPE_NUM_BITS)))

#define SXM_DEBUG_IS_TYPE_SET(_var, _dbgType) \
    (0 != ((uint)(_var) & (1U << (_dbgType))))

#define SXM_DEBUG_IS_SET(_var, _dbgArea, _dbgType) \
    (SXM_DEBUG_IS_AREA_SET(_var, _dbgArea) && SXM_DEBUG_IS_TYPE_SET(_var, _dbgType))

/** Default definition of debug value used in debug macros */
#ifndef DEBUG_VAR
#define DEBUG_VAR ((service) ? service->ds.debug_level : (uint)(-1))
#endif

/** Default definition of debug tag used in debug macros */
#ifndef DEBUG_TAG
#define DEBUG_TAG ""
#endif

/** Debug prefixes printed in debug messages */
#define CALLB_PREFIX                    "[->"
#define CALLR_PREFIX                    "<->"
#define ENTER_PREFIX                    "->>"
#define LEAVE_PREFIX                    "<<-"
#define ERROR_PREFIX                    "---"
#define DEBUG_PREFIX                    "+++"
#define EVENT_PREFIX                    "^^^"
#define WARN_PREFIX                     "???"
#define DUMP_PREFIX                     "###"
#define PLOG_PREFIX                     "***"

/** Log functions prototypes */
extern SXESDK_INTERNAL_API void sxm_file_log(const char *fmt, ...);

extern SXESDK_INTERNAL_API void sxm_file_log_ext(SxmDebugArea debugArea,
                                                 SxmDebugType debugType,
                                                 const char * tag,
                                                 const char *prefix,
                                                 const char *function,
                                                 const char *fmt,
                                                 ...);

extern SXESDK_INTERNAL_API void sxm_file_dump(const void *data, size_t bc);

extern SXESDK_INTERNAL_API void sxm_file_dump_ext(SxmDebugArea debugArea,
                                                  SxmDebugType debugType,
                                                  const char * tag,
                                                  const char *prefix,
                                                  const char *function,
                                                  const void *data,
                                                  size_t bc);

/** Macros defined in DEBUG configuration only */
#ifdef SXM_DEBUG_PRODUCTION

#define SXMLOGF(_var, _area, _type, _tag, _prefix, _fmt, ...)
#define SXMDUMP(_var, _area, _type, _tag, _pd, _len)

#else /* #ifdef SXM_DEBUG_PRODUCTION */

#ifndef SXMLOGF
#define SXMLOGF(_var, _area, _type, _tag, _prefix, _fmt, ...) \
    if (SXM_DEBUG_IS_SET(_var, _area, _type)) { \
        sxm_file_log_ext(_area, _type, _tag, \
                         _prefix, __FUNCTION__, \
                         _fmt, ##__VA_ARGS__); }
#endif /* #ifndef SXMLOGF */

#define SXMDUMP(_var, _area, _type, _tag, _data, _len) \
    if (SXM_DEBUG_IS_SET(_var, _area, _type)) { \
        sxm_file_dump_ext(_area, _type, _tag, \
                          DUMP_PREFIX, __FUNCTION__, \
                          _data, (_len)); }

#endif /* #ifdef SXM_DEBUG_PRODUCTION ... #else ... */

/** Macro which always output the message regardless of debug type set */
#define SXMLOGF_PLOG(_dbgArea, _fmt, ...) \
    sxm_file_log_ext(_dbgArea, SXM_DBG_TYPE_PLOG, DEBUG_TAG, \
                     PLOG_PREFIX, __FUNCTION__, \
                     _fmt, ##__VA_ARGS__);

/** Basic debugging macros for debug types set */
#define SXMLOGF_CALLB(_dbgArea, _fmt, ...) \
    SXMLOGF(DEBUG_VAR, _dbgArea, SXM_DBG_TYPE_CALL,  DEBUG_TAG, CALLB_PREFIX, _fmt, ##__VA_ARGS__)
#define SXMLOGF_CALLR(_dbgArea, _fmt, ...) \
    SXMLOGF(DEBUG_VAR, _dbgArea, SXM_DBG_TYPE_CALL,  DEBUG_TAG, CALLR_PREFIX, _fmt, ##__VA_ARGS__)
#define SXMLOGF_ENTER(_dbgArea, _fmt, ...) \
    SXMLOGF(DEBUG_VAR, _dbgArea, SXM_DBG_TYPE_CALL,  DEBUG_TAG, ENTER_PREFIX, _fmt, ##__VA_ARGS__)
#define SXMLOGF_LEAVE(_dbgArea, _fmt, ...) \
    SXMLOGF(DEBUG_VAR, _dbgArea, SXM_DBG_TYPE_CALL,  DEBUG_TAG, LEAVE_PREFIX, _fmt, ##__VA_ARGS__)
#define SXMLOGF_EVENT(_dbgArea, _fmt, ...) \
    SXMLOGF(DEBUG_VAR, _dbgArea, SXM_DBG_TYPE_EVENT, DEBUG_TAG, EVENT_PREFIX, _fmt, ##__VA_ARGS__)
#define SXMLOGF_DEBUG(_dbgArea, _fmt, ...) \
    SXMLOGF(DEBUG_VAR, _dbgArea, SXM_DBG_TYPE_DEBUG, DEBUG_TAG, DEBUG_PREFIX, _fmt, ##__VA_ARGS__)
#define SXMLOGF_ERROR(_dbgArea, _fmt, ...) \
    SXMLOGF(DEBUG_VAR, _dbgArea, SXM_DBG_TYPE_ERROR, DEBUG_TAG, ERROR_PREFIX, _fmt, ##__VA_ARGS__)
#define SXMLOGF_WARN(_dbgArea, _fmt, ...) \
    SXMLOGF(DEBUG_VAR, _dbgArea, SXM_DBG_TYPE_WARN,  DEBUG_TAG, WARN_PREFIX,  _fmt, ##__VA_ARGS__)
#define SXMLOGF_DUMP(_dbgArea, _fmt, ...) \
    SXMLOGF(DEBUG_VAR, _dbgArea, SXM_DBG_TYPE_DUMP,  DEBUG_TAG, DUMP_PREFIX,  _fmt, ##__VA_ARGS__)
#define SXMLOGF_DUMPB(_dbgArea, _pd, _len) \
    SXMDUMP(DEBUG_VAR, _dbgArea, SXM_DBG_TYPE_DUMP,  DEBUG_TAG, _pd, _len)

/** Special API debugging macros to use in start functions
  * where object's debug mask is not initialized on function start.
  */
#define ENTER_START_API(_dbgVar, _fmt, ...) \
    SXMLOGF(_dbgVar, SXM_DBG_AREA_API, SXM_DBG_TYPE_CALL, DEBUG_TAG, ENTER_PREFIX, _fmt, ##__VA_ARGS__)
#define LEAVE_START_API(_dbgVar, _fmt, ...) \
    SXMLOGF(_dbgVar, SXM_DBG_AREA_API, SXM_DBG_TYPE_CALL, DEBUG_TAG, LEAVE_PREFIX, _fmt, ##__VA_ARGS__)

/** API Area debugging macros */
#define CALLB_API(_fmt, ...) SXMLOGF_CALLB(SXM_DBG_AREA_API, _fmt, ##__VA_ARGS__)
#define CALLR_API(_fmt, ...) SXMLOGF_CALLR(SXM_DBG_AREA_API, _fmt, ##__VA_ARGS__)
#define ENTER_API(_fmt, ...) SXMLOGF_ENTER(SXM_DBG_AREA_API, _fmt, ##__VA_ARGS__)
#define LEAVE_API(_fmt, ...) SXMLOGF_LEAVE(SXM_DBG_AREA_API, _fmt, ##__VA_ARGS__)
#define DEBUG_API(_fmt, ...) SXMLOGF_DEBUG(SXM_DBG_AREA_API, _fmt, ##__VA_ARGS__)
#define EVENT_API(_fmt, ...) SXMLOGF_EVENT(SXM_DBG_AREA_API, _fmt, ##__VA_ARGS__)
#define ERROR_API(_fmt, ...) SXMLOGF_ERROR(SXM_DBG_AREA_API, _fmt, ##__VA_ARGS__)
#define  WARN_API(_fmt, ...)  SXMLOGF_WARN(SXM_DBG_AREA_API, _fmt, ##__VA_ARGS__)
#define  PLOG_API(_fmt, ...)  SXMLOGF_PLOG(SXM_DBG_AREA_API, _fmt, ##__VA_ARGS__)
#define  DUMP_API(_fmt, ...)  SXMLOGF_DUMP(SXM_DBG_AREA_API, _fmt, ##__VA_ARGS__)
#define DUMPB_API(_pd, _len) SXMLOGF_DUMPB(SXM_DBG_AREA_API, _pd, _len)

/** USR Area debugging macros */
#define CALLB_USR(_fmt, ...) SXMLOGF_CALLB(SXM_DBG_AREA_USR, _fmt, ##__VA_ARGS__)
#define CALLR_USR(_fmt, ...) SXMLOGF_CALLR(SXM_DBG_AREA_USR, _fmt, ##__VA_ARGS__)
#define ENTER_USR(_fmt, ...) SXMLOGF_ENTER(SXM_DBG_AREA_USR, _fmt, ##__VA_ARGS__)
#define LEAVE_USR(_fmt, ...) SXMLOGF_LEAVE(SXM_DBG_AREA_USR, _fmt, ##__VA_ARGS__)
#define DEBUG_USR(_fmt, ...) SXMLOGF_DEBUG(SXM_DBG_AREA_USR, _fmt, ##__VA_ARGS__)
#define EVENT_USR(_fmt, ...) SXMLOGF_EVENT(SXM_DBG_AREA_USR, _fmt, ##__VA_ARGS__)
#define ERROR_USR(_fmt, ...) SXMLOGF_ERROR(SXM_DBG_AREA_USR, _fmt, ##__VA_ARGS__)
#define  WARN_USR(_fmt, ...)  SXMLOGF_WARN(SXM_DBG_AREA_USR, _fmt, ##__VA_ARGS__)
#define  PLOG_USR(_fmt, ...)  SXMLOGF_PLOG(SXM_DBG_AREA_USR, _fmt, ##__VA_ARGS__)
#define  DUMP_USR(_fmt, ...)  SXMLOGF_DUMP(SXM_DBG_AREA_USR, _fmt, ##__VA_ARGS__)
#define DUMPB_USR(_pd, _len) SXMLOGF_DUMPB(SXM_DBG_AREA_USR, _pd, _len)

/** REQ Area debugging macros */
#define CALLB_REQ(_fmt, ...) SXMLOGF_CALLB(SXM_DBG_AREA_REQ, _fmt, ##__VA_ARGS__)
#define CALLR_REQ(_fmt, ...) SXMLOGF_CALLR(SXM_DBG_AREA_REQ, _fmt, ##__VA_ARGS__)
#define ENTER_REQ(_fmt, ...) SXMLOGF_ENTER(SXM_DBG_AREA_REQ, _fmt, ##__VA_ARGS__)
#define LEAVE_REQ(_fmt, ...) SXMLOGF_LEAVE(SXM_DBG_AREA_REQ, _fmt, ##__VA_ARGS__)
#define DEBUG_REQ(_fmt, ...) SXMLOGF_DEBUG(SXM_DBG_AREA_REQ, _fmt, ##__VA_ARGS__)
#define EVENT_REQ(_fmt, ...) SXMLOGF_EVENT(SXM_DBG_AREA_REQ, _fmt, ##__VA_ARGS__)
#define ERROR_REQ(_fmt, ...) SXMLOGF_ERROR(SXM_DBG_AREA_REQ, _fmt, ##__VA_ARGS__)
#define  WARN_REQ(_fmt, ...)  SXMLOGF_WARN(SXM_DBG_AREA_REQ, _fmt, ##__VA_ARGS__)
#define  PLOG_REQ(_fmt, ...)  SXMLOGF_PLOG(SXM_DBG_AREA_REQ, _fmt, ##__VA_ARGS__)
#define  DUMP_REQ(_fmt, ...)  SXMLOGF_DUMP(SXM_DBG_AREA_REQ, _fmt, ##__VA_ARGS__)
#define DUMPB_REQ(_pd, _len) SXMLOGF_DUMPB(SXM_DBG_AREA_REQ, _pd, _len)

/** EXT Area debugging macros */
#define CALLB_EXT(_fmt, ...) SXMLOGF_CALLB(SXM_DBG_AREA_EXT, _fmt, ##__VA_ARGS__)
#define CALLR_EXT(_fmt, ...) SXMLOGF_CALLR(SXM_DBG_AREA_EXT, _fmt, ##__VA_ARGS__)
#define ENTER_EXT(_fmt, ...) SXMLOGF_ENTER(SXM_DBG_AREA_EXT, _fmt, ##__VA_ARGS__)
#define LEAVE_EXT(_fmt, ...) SXMLOGF_LEAVE(SXM_DBG_AREA_EXT, _fmt, ##__VA_ARGS__)
#define DEBUG_EXT(_fmt, ...) SXMLOGF_DEBUG(SXM_DBG_AREA_EXT, _fmt, ##__VA_ARGS__)
#define EVENT_EXT(_fmt, ...) SXMLOGF_EVENT(SXM_DBG_AREA_EXT, _fmt, ##__VA_ARGS__)
#define ERROR_EXT(_fmt, ...) SXMLOGF_ERROR(SXM_DBG_AREA_EXT, _fmt, ##__VA_ARGS__)
#define  WARN_EXT(_fmt, ...)  SXMLOGF_WARN(SXM_DBG_AREA_EXT, _fmt, ##__VA_ARGS__)
#define  PLOG_EXT(_fmt, ...)  SXMLOGF_PLOG(SXM_DBG_AREA_EXT, _fmt, ##__VA_ARGS__)
#define  DUMP_EXT(_fmt, ...)  SXMLOGF_DUMP(SXM_DBG_AREA_EXT, _fmt, ##__VA_ARGS__)
#define DUMPB_EXT(_pd, _len) SXMLOGF_DUMPB(SXM_DBG_AREA_EXT, _pd, _len)

/** PKT Area debugging macros */
#define CALLB_PKT(_fmt, ...) SXMLOGF_CALLB(SXM_DBG_AREA_PKT, _fmt, ##__VA_ARGS__)
#define CALLR_PKT(_fmt, ...) SXMLOGF_CALLR(SXM_DBG_AREA_PKT, _fmt, ##__VA_ARGS__)
#define ENTER_PKT(_fmt, ...) SXMLOGF_ENTER(SXM_DBG_AREA_PKT, _fmt, ##__VA_ARGS__)
#define LEAVE_PKT(_fmt, ...) SXMLOGF_LEAVE(SXM_DBG_AREA_PKT, _fmt, ##__VA_ARGS__)
#define DEBUG_PKT(_fmt, ...) SXMLOGF_DEBUG(SXM_DBG_AREA_PKT, _fmt, ##__VA_ARGS__)
#define EVENT_PKT(_fmt, ...) SXMLOGF_EVENT(SXM_DBG_AREA_PKT, _fmt, ##__VA_ARGS__)
#define ERROR_PKT(_fmt, ...) SXMLOGF_ERROR(SXM_DBG_AREA_PKT, _fmt, ##__VA_ARGS__)
#define  WARN_PKT(_fmt, ...)  SXMLOGF_WARN(SXM_DBG_AREA_PKT, _fmt, ##__VA_ARGS__)
#define  PLOG_PKT(_fmt, ...)  SXMLOGF_PLOG(SXM_DBG_AREA_PKT, _fmt, ##__VA_ARGS__)
#define  DUMP_PKT(_fmt, ...)  SXMLOGF_DUMP(SXM_DBG_AREA_PKT, _fmt, ##__VA_ARGS__)
#define DUMPB_PKT(_pd, _len) SXMLOGF_DUMPB(SXM_DBG_AREA_PKT, _pd, _len)

/** UPD Area debugging macros */
#define CALLB_UPD(_fmt, ...) SXMLOGF_CALLB(SXM_DBG_AREA_UPD, _fmt, ##__VA_ARGS__)
#define CALLR_UPD(_fmt, ...) SXMLOGF_CALLR(SXM_DBG_AREA_UPD, _fmt, ##__VA_ARGS__)
#define ENTER_UPD(_fmt, ...) SXMLOGF_ENTER(SXM_DBG_AREA_UPD, _fmt, ##__VA_ARGS__)
#define LEAVE_UPD(_fmt, ...) SXMLOGF_LEAVE(SXM_DBG_AREA_UPD, _fmt, ##__VA_ARGS__)
#define DEBUG_UPD(_fmt, ...) SXMLOGF_DEBUG(SXM_DBG_AREA_UPD, _fmt, ##__VA_ARGS__)
#define EVENT_UPD(_fmt, ...) SXMLOGF_EVENT(SXM_DBG_AREA_UPD, _fmt, ##__VA_ARGS__)
#define ERROR_UPD(_fmt, ...) SXMLOGF_ERROR(SXM_DBG_AREA_UPD, _fmt, ##__VA_ARGS__)
#define  WARN_UPD(_fmt, ...)  SXMLOGF_WARN(SXM_DBG_AREA_UPD, _fmt, ##__VA_ARGS__)
#define  PLOG_UPD(_fmt, ...)  SXMLOGF_PLOG(SXM_DBG_AREA_UPD, _fmt, ##__VA_ARGS__)
#define  DUMP_UPD(_fmt, ...)  SXMLOGF_DUMP(SXM_DBG_AREA_UPD, _fmt, ##__VA_ARGS__)
#define DUMPB_UPD(_pd, _len) SXMLOGF_DUMPB(SXM_DBG_AREA_UPD, _pd, _len)

/** UPD Area debugging macros */
#define CALLB_DBF(_fmt, ...) SXMLOGF_CALLB(SXM_DBG_AREA_DBF, _fmt, ##__VA_ARGS__)
#define CALLR_DBF(_fmt, ...) SXMLOGF_CALLR(SXM_DBG_AREA_DBF, _fmt, ##__VA_ARGS__)
#define ENTER_DBF(_fmt, ...) SXMLOGF_ENTER(SXM_DBG_AREA_DBF, _fmt, ##__VA_ARGS__)
#define LEAVE_DBF(_fmt, ...) SXMLOGF_LEAVE(SXM_DBG_AREA_DBF, _fmt, ##__VA_ARGS__)
#define DEBUG_DBF(_fmt, ...) SXMLOGF_DEBUG(SXM_DBG_AREA_DBF, _fmt, ##__VA_ARGS__)
#define EVENT_DBF(_fmt, ...) SXMLOGF_EVENT(SXM_DBG_AREA_DBF, _fmt, ##__VA_ARGS__)
#define ERROR_DBF(_fmt, ...) SXMLOGF_ERROR(SXM_DBG_AREA_DBF, _fmt, ##__VA_ARGS__)
#define  WARN_DBF(_fmt, ...)  SXMLOGF_WARN(SXM_DBG_AREA_DBF, _fmt, ##__VA_ARGS__)
#define  PLOG_DBF(_fmt, ...)  SXMLOGF_PLOG(SXM_DBG_AREA_DBF, _fmt, ##__VA_ARGS__)
#define  DUMP_DBF(_fmt, ...)  SXMLOGF_DUMP(SXM_DBG_AREA_DBF, _fmt, ##__VA_ARGS__)
#define DUMPB_DBF(_pd, _len) SXMLOGF_DUMPB(SXM_DBG_AREA_DBF, _pd, _len)

/** LLI Area debugging macros */
#define CALLB_LLI(_fmt, ...) SXMLOGF_CALLB(SXM_DBG_AREA_LLI, _fmt, ##__VA_ARGS__)
#define CALLR_LLI(_fmt, ...) SXMLOGF_CALLR(SXM_DBG_AREA_LLI, _fmt, ##__VA_ARGS__)
#define ENTER_LLI(_fmt, ...) SXMLOGF_ENTER(SXM_DBG_AREA_LLI, _fmt, ##__VA_ARGS__)
#define LEAVE_LLI(_fmt, ...) SXMLOGF_LEAVE(SXM_DBG_AREA_LLI, _fmt, ##__VA_ARGS__)
#define DEBUG_LLI(_fmt, ...) SXMLOGF_DEBUG(SXM_DBG_AREA_LLI, _fmt, ##__VA_ARGS__)
#define EVENT_LLI(_fmt, ...) SXMLOGF_EVENT(SXM_DBG_AREA_LLI, _fmt, ##__VA_ARGS__)
#define ERROR_LLI(_fmt, ...) SXMLOGF_ERROR(SXM_DBG_AREA_LLI, _fmt, ##__VA_ARGS__)
#define  WARN_LLI(_fmt, ...)  SXMLOGF_WARN(SXM_DBG_AREA_LLI, _fmt, ##__VA_ARGS__)
#define  PLOG_LLI(_fmt, ...)  SXMLOGF_PLOG(SXM_DBG_AREA_LLI, _fmt, ##__VA_ARGS__)
#define  DUMP_LLI(_fmt, ...)  SXMLOGF_DUMP(SXM_DBG_AREA_LLI, _fmt, ##__VA_ARGS__)
#define DUMPB_LLI(_pd, _len) SXMLOGF_DUMPB(SXM_DBG_AREA_LLI, _pd, _len)

/** UTL Area debugging macros */
#define CALLB_UTL(_fmt, ...) SXMLOGF_CALLB(SXM_DBG_AREA_UTL, _fmt, ##__VA_ARGS__)
#define CALLR_UTL(_fmt, ...) SXMLOGF_CALLR(SXM_DBG_AREA_UTL, _fmt, ##__VA_ARGS__)
#define ENTER_UTL(_fmt, ...) SXMLOGF_ENTER(SXM_DBG_AREA_UTL, _fmt, ##__VA_ARGS__)
#define LEAVE_UTL(_fmt, ...) SXMLOGF_LEAVE(SXM_DBG_AREA_UTL, _fmt, ##__VA_ARGS__)
#define DEBUG_UTL(_fmt, ...) SXMLOGF_DEBUG(SXM_DBG_AREA_UTL, _fmt, ##__VA_ARGS__)
#define EVENT_UTL(_fmt, ...) SXMLOGF_EVENT(SXM_DBG_AREA_UTL, _fmt, ##__VA_ARGS__)
#define ERROR_UTL(_fmt, ...) SXMLOGF_ERROR(SXM_DBG_AREA_UTL, _fmt, ##__VA_ARGS__)
#define  WARN_UTL(_fmt, ...)  SXMLOGF_WARN(SXM_DBG_AREA_UTL, _fmt, ##__VA_ARGS__)
#define  PLOG_UTL(_fmt, ...)  SXMLOGF_PLOG(SXM_DBG_AREA_UTL, _fmt, ##__VA_ARGS__)
#define  DUMP_UTL(_fmt, ...)  SXMLOGF_DUMP(SXM_DBG_AREA_UTL, _fmt, ##__VA_ARGS__)
#define DUMPB_UTL(_pd, _len) SXMLOGF_DUMPB(SXM_DBG_AREA_UTL, _pd, _len)

#ifdef __cplusplus
}
#endif

#endif /* #ifndef SXM_STDTRACE_H */
