/******************************************************************************/
/*                Copyright (c) Sirius XM Satellite Radio, Inc.               */
/*                            All Rights Reserved                             */
/*      Licensed Materials - Property of Sirius XM Satellite Radio, Inc.      */
/******************************************************************************/
/***************************************************************************//**
 * 
 * \file sxm_sports.h
 * \author Leslie French, Sergey Kotov
 * \date 8/20/2013
 *
 * Header for the production Sports service
 *
 ******************************************************************************/

#ifndef SXM_SPORTS_H
#define SXM_SPORTS_H

#include <sxm_common.h>

#ifdef __cplusplus
extern "C" {
#endif

/**
 * \defgroup sports_pubapi Sports Public API
 * \brief This section contains Sports Public API
 */
/**@{*/

/** \name Predefined Sports IDs
 * @{
 */
/** Multi Sport */
#define SPID_MULTI 				0
/** Football */
#define SPID_FOOTBALL			1
/** Baseball */
#define SPID_BASEBALL			2
/** Basketball */
#define SPID_BASKETBALL			3
/** Ice Hockey */
#define SPID_ICEHOCKEY			4
/** Motorsport */
#define SPID_MOTORSPORT			5
/** Golf */
#define SPID_GOLF				6
/** Soccer */
#define SPID_SOCCER				7
/** Tennis */
#define SPID_TENNIS				8
/** Max Sport IDs */
#define SPID_MAX				9
/** @} */

/** \name Table Classes
 * @{
 */
/** Head-To-Head, Sports Events */
#define TCLASS_H2H_EVENT 				0
/** Ranked List, Sports Eventst */
#define TCLASS_RL_EVENT 				1
/** Ranked List, Season Summary */
#define TCLASS_RL_SEASON_SUMMARY		2
/** Ranked List, News Items */
#define TCLASS_RL_NEWS				    3
/** Ranked List, Participant Statistics */
#define TCLASS_RL_PARTICIPANT_STAT		4
/** Ranked List, Participant Index */
#define TCLASS_RL_PARTICIPANT_INDEX		5
/** // Event Description */
#define TCLASS_EVENT_DESC				6
/** @} */

/******************************************************************************
 *                                                                            *
 *            The Sports Service Object                                       *
 *            =========================                                       *
 *                                                                            *
 ******************************************************************************/

/** \name League key (symbol) values
 * @{
 */
/** The Sport Identifier from the Sports Type Table */
#define SXM_SPORTS_LEAGUE_SPID          (0u)
/** The name of the Sport */
#define SXM_SPORTS_LEAGUE_SPNAME        (1u)
/** The Affiliation Identifier for the top-level league */
#define SXM_SPORTS_LEAGUE_AFID          (2u)
/** The name of the league */
#define SXM_SPORTS_LEAGUE_AFNAME        (3u)
/** The Global Data Reference ID for the League */
#define SXM_SPORTS_LEAGUE_GDREF         (4u)
/** In-season sign. '1' if the sport is in-season, else '0' */
#define SXM_SPORTS_LEAGUE_SEASON        (5u)
/** Affiliate Long Name */
#define SXM_SPORTS_LEAGUE_AFLNAME       (6u)
/** Affiliate Short Name */
#define SXM_SPORTS_LEAGUE_AFSNAME       (7u)
/** @} */

/** \name Tree key (symbol) values
 * @{
 */
/** The Sport Identifier from the Sports Type Table */
#define SXM_SPORTS_TREE_SPID            SXM_SPORTS_LEAGUE_SPID
/** The name of the Sport */
#define SXM_SPORTS_TREE_SPNAME          SXM_SPORTS_LEAGUE_SPNAME
/** The Affiliation Identifier for the top-level league */
#define SXM_SPORTS_TREE_AFID            SXM_SPORTS_LEAGUE_AFID
/** The name of the league */
#define SXM_SPORTS_TREE_AFNAME          SXM_SPORTS_LEAGUE_AFNAME
/** The Global Data Reference ID for the League */
#define SXM_SPORTS_TREE_GDREF           SXM_SPORTS_LEAGUE_GDREF
/** In-season sign. '1' if the league is in season; '0' otherwise */
#define SXM_SPORTS_TREE_SEASON          SXM_SPORTS_LEAGUE_SEASON
/** Affiliate Long Name */
#define SXM_SPORTS_TREE_AFLNAME         SXM_SPORTS_LEAGUE_AFLNAME
/** Affiliate Short Name */
#define SXM_SPORTS_TREE_AFSNAME         SXM_SPORTS_LEAGUE_AFSNAME
/** @} */

/** \name List key (symbol) values
 * @{
 */
/** The Sport Identifier from the Sports Type Table */
#define SXM_SPORTS_LIST_SPID            (0u)
/** The name of the Sport */
#define SXM_SPORTS_LIST_SPNAME          (1u)
/** @} */

/** \name Table key (symbol) values
 * @{
 */
/** The date for which the table contains data */
#define SXM_SPORTS_TABLE_EPOCH          (0u)
/** The title of the table */
#define SXM_SPORTS_TABLE_TITLE          (1u)
/** The time since when the table was last updated */
#define SXM_SPORTS_TABLE_AGE            (2u)
/** The class (type) of this table */
#define SXM_SPORTS_TABLE_CLASS          (3u)
/** The season flag of the sports */
#define SXM_SPORTS_TABLE_SEASON         (4u)
/* Affiliation ID */
#define SXM_SPORTS_TABLE_AFID           (5u)
/* Affiliation Name */
#define SXM_SPORTS_TABLE_AFNAME         (6u)

/** @} */

/** \name Table definition key (symbol) values
 * @{
 */
/** Label Identifier */
#define SXM_SPORTS_TDEF_LBID            (0u)
/** Label Type */
#define SXM_SPORTS_TDEF_LBTYPE          (1u)
/** Data Modifier */
#define SXM_SPORTS_TDEF_DMOD            (2u)
/** Display Priority */
#define SXM_SPORTS_TDEF_PRIORITY        (3u)
/** Label Text (column heading) */
#define SXM_SPORTS_TDEF_LBTXT           (4u)
/** @} */

/** \name Team definition key (symbol) values
 * @{
 */
/** Team Identifier used by Sports and Audio */
#define SXM_SPORTS_TEAM_TEAMID          (0u)
/** Abbreviated name of the team */
#define SXM_SPORTS_TEAM_ABBR            (1u)
/** Team Name */
#define SXM_SPORTS_TEAM_NAME            (2u)
/** Team Nickname */
#define SXM_SPORTS_TEAM_NICK            (3u)
/** @} */

/** \name Head-to-Head Sports Schedule key (symbol) values
 * @{
 */
/** Home Team Global Team Identifier */
#define SXM_SPORTS_H2H_HOME_ID          (0u)
/** Home Team Abbreviation */
#define SXM_SPORTS_H2H_HOME_ABBR        (1u)
/** Home Team Nickname */
#define SXM_SPORTS_H2H_HOME_NICK        (2u)
/** Home Team Fullname */
#define SXM_SPORTS_H2H_HOME_FULL        (3u)
/** Visiting Team Global Team Identifier */
#define SXM_SPORTS_H2H_VISIT_ID         (4u)
/** Visiting Team Abbreviation */
#define SXM_SPORTS_H2H_VISIT_ABBR       (5u)
/** Visiting Team Nickname */
#define SXM_SPORTS_H2H_VISIT_NICK       (6u)
/** Visiting Team Fullname */
#define SXM_SPORTS_H2H_VISIT_FULL       (7u)
/** Game Status */
#define SXM_SPORTS_H2H_STATE            (8u)
/** Current game division (Quarter etc.) */
#define SXM_SPORTS_H2H_DIVISION         (9u)
/** Game Start time */
#define SXM_SPORTS_H2H_START_TIME       (10u)
/** Home Team Score */
#define SXM_SPORTS_H2H_HOME_SCORE       (11u)
/** Visiting Team Score */
#define SXM_SPORTS_H2H_VISIT_SCORE      (12u)
/** Winning Team indicator */
#define SXM_SPORTS_H2H_RESULT           (13u)
/** Home Team Audio Coverage Channel */
#define SXM_SPORTS_H2H_HOME_CHAN        (14u)
/** Visiting Team Audio Coverage Channel */
#define SXM_SPORTS_H2H_VISIT_CHAN       (15u)
/** National (non-partisan) Coverage Channel */
#define SXM_SPORTS_H2H_NATION_CHAN      (16u)
/** Game Clock */
#define SXM_SPORTS_H2H_CLOCK            (17u)
/** Index of table containing entry */
#define SXM_SPORTS_H2H_TABLE_INDEX      (18u)
/** Age of table containing entry */
#define SXM_SPORTS_H2H_TABLE_AGE        (19u)
/** Extra info */

/** @} */

/** \name News key (symbol) values
 * @{
 */
/** News Items */
#define SXM_SPORTS_NEWS_ITEM            (0u)
/** @} */

/** \name Golf Schedules key (symbol) values
 * @{
 */
/** Tourney State */
#define SXM_SPORTS_GOLF_STATE           (0u)
/** TableRef for Ranking List */
#define SXM_SPORTS_GOLF_RANK            (1u)
/** Tourney Title */
#define SXM_SPORTS_GOLF_TOURNEY         (2u)
/** Course Name */
#define SXM_SPORTS_GOLF_COURSE          (3u)
/** Course Yardage */
#define SXM_SPORTS_GOLF_YARDS           (4u)
/** Purse (prize value) */
#define SXM_SPORTS_GOLF_PURSE           (5u)
/** National (non-partisan) Coverage Channel */
#define SXM_SPORTS_GOLF_NATION_CHAN     (6u)
/** Tourney Start Time */
#define SXM_SPORTS_GOLF_START_TIME      (7u)
/** @} */

/** \name Golf ranking key (symbol) values
 * @{
 */
/** Golfer Ranking */
#define SXM_SPORTS_GOLF_RANK_RANK       (0u)
/** Golfer Name */
#define SXM_SPORTS_GOLF_RANK_PLAYER     (1u)
/** Golfer Score */
#define SXM_SPORTS_GOLF_RANK_SCORE      (2u)
/** @} */

/** \name Auto racing key (symbol) values
 * @{
 */
/** Race State */
#define SXM_SPORTS_AUTO_STATE           (0u)
/** Race Start Time */
#define SXM_SPORTS_AUTO_START_TIME      (1u)
/** TableRef for Ranking List */
#define SXM_SPORTS_AUTO_RANK            (2u)
/** Race Title */
#define SXM_SPORTS_AUTO_RACE            (3u)
/** Track Name */
#define SXM_SPORTS_AUTO_TRACK           (4u)
/** Number of laps in complete race */
#define SXM_SPORTS_AUTO_LAPS            (5u)
/** Name of Winner */
#define SXM_SPORTS_AUTO_WINNER          (6u)
/** National (non-driver) Coverage Channel */
#define SXM_SPORTS_AUTO_NATION_CHAN     (7u)
/** @} */

/** \name Auto Racing Ranking key (symbol) values
 * @{
 */
/** Driver Ranking */
#define SXM_SPORTS_AUTO_RANK_RANK       (0u)
/** Driver Name */
#define SXM_SPORTS_AUTO_RANK_DRIVER     (1u)
/** Completed Laps */
#define SXM_SPORTS_AUTO_RANK_LAPS       (2u)
/** Car Number */
#define SXM_SPORTS_AUTO_RANK_NUMBER     (3u)
/** Car Manufacturer */
#define SXM_SPORTS_AUTO_RANK_MAKE       (4u)
/** @} */

/** Maximal key (symbol) zero-relative count */
#define SXM_SPORTS_KEY_MAX_COUNT        (SXM_SPORTS_H2H_TABLE_AGE + 1)

/** \name Extraction type
 * @{
 */
#define SXM_SPORTS_LIST                 (1u<<0)
#define SXM_SPORTS_LEAGUES              (1u<<1)
#define SXM_SPORTS_TREE                 (1u<<2)
#define SXM_SPORTS_TABLE                (1u<<3)
#define SXM_SPORTS_TEAMS                (1u<<4)
#define SXM_SPORTS_TEAM_EVENTS          (1u<<5)
#define SXM_SPORTS_TDEF                 (1u<<6)
#define SXM_SPORTS_UPDATED              (1u<<7)
/** @} */

/**
 * Combines Affiliation Id and Table Index to one packed value
 * \param[in] _a Affiliation Id
 * \param[in] _i Table Index
 * \return packed combination of Affiliation Id and Table Index
 */
#define SPORTS_TABLE_PARAM(_a, _i)      ((((_a) & 0x3ff) << 6) | ((_i) & 0x3f))
/**
 * Extracts Affiliation Id from the packed
 * combination of Affiliation Id and Table Index
 * \param[in] _param packed combination of Affiliation Id and Table Index
 * \return Unpacked Affiliation Id
 */
#define SPORTS_TABLE_AF(_param)         (((_param) >> 6) & 0x3ff)
/**
 * Extracts Table Index from the packed
 * combination of Affiliation Id and Table Index
 * \param[in] _param packed combination of Affiliation Id and Table Index
 * \return Unpacked Table Index
 */
#define SPORTS_TABLE_IX(_param)         ((_param) & 0x3f)

/**
 * Combines Table Id and Table Version to one packed value
 * \param[in] _id  Table Id
 * \param[in] _ver Table Version
 * \return Packed combination of Table Id and Table Version
 */
#define SPORTS_TDEF_PARAM(_id, _ver)    ((((_id) & 0x3ff) << 8) | ((_ver) & 0xff))
/**
 * Extracts Table Id from the packed combination of Table Id and Table Version
 * \param[in] _param Packed combination of Table Id and Table Version
 * \return Unpacked Table Id
 */
#define SPORTS_TDEF_ID(_param)          (((_param) >> 8) & 0x3ff)
/**
 * Extracts Table Version from the packed
 * combination of Table Id and Table Version
 * \param[in] _param Packed combination of Table Id and Table Version
 * \return Unpacked Table Version
 */
#define SPORTS_TDEF_VER(_param)         ((_param) & 0xff)

/** The Row Object used for standard service interface returned data */
typedef struct
{
    /** A bitfield of which values are present */
    uint present;

    /** A bitfield indicating the type of each value in the array */
    uint types;

    /** (when types bit is '0') an integer value for the index
     *  (when types bit is '1') an offset into the string buffer for the index
     */
    int value[32];

    /** The last character index used in the buffer */
    uint highwater;

    /** Buffer space used to hold string values */
    char buffer[1024];
} SXMSportsRow;

/** The Raw Object used for the extended interface returned data */
typedef struct
{
    /** A reference to the table definition for these data */
    int tableref;

    /** The epoch (day number) of the table */
    int tableepoch;

    /** The number of minutes since the table was received */
    uint tableage;

    /** The number of valid entries in the kvp array */
    uint count;

    /** A bitfield indicating the type of each value in the kvp array */
    uint types[2];

    /** (when types bit is '0') an integer value for the index
     *  (when types bit is '1') an offset into the string buffer for the index*/
    int value[64];

    /** The label identifier used to index into the table definition */
    int lbid[64];

    /** The last character index used in the buffer */
    uint highwater;

    /** Buffer space used to hold string values */
    char buffer[1024];
} SXMSportsRaw;

/***************************************************************************//**
 *
 * Service module-level notification callback
 *
 * The notification routine will be called whenever the service state
 * or subscription is changed
 *
 * \param[in] eventType module-level event type
 *                      (SXM_CALLBACK_ERROR or SXM_CALLBACK_SUBS)
 * \param[in] eventParam event-specific parameter value
 *
 * \note Refer to SX-9845-0340 Introduction Design Guide
 *       for Callbacks and Status Events values
 *
 ******************************************************************************/
typedef void (*SPORTS_EVENT_CALLBACK)(int eventType, int eventParam);

/***************************************************************************//**
 *
 * Request notification callback
 *
 * The notification routine will be called whenever data in the collection
 * has changed
 *
 * \param[in] requestHandle the handle for the original collection request
 * \param[in] type the type of information being updated
 * \param[in] param a value that is determined by the type field
 * \param[in] usercx the user context value from the original request
 *
 ******************************************************************************/
typedef void (*SPORTS_REQUEST_CALLBACK)(ptr requestHandle, int type, \
                                        int param, ptr usercx);

/******************************************************************************
 *                                                                            *
 *            Application Interface                                           *
 *            =====================                                           *
 *                                                                            *
 ******************************************************************************/

extern SXESDK_API int sxm_sports_start(SPORTS_EVENT_CALLBACK callback,
                                       int debugLevel);

extern SXESDK_API int sxm_sports_status(SXMStatus *pStatus);

extern SXESDK_API int sxm_sports_stop(void);

extern SXESDK_API int sxm_sports_request(const char *league,
                                         SPORTS_REQUEST_CALLBACK notification,
                                         ptr usercx,
                                         ptr *pRequestHandle);

extern SXESDK_API int sxm_sports_remove(ptr requestHandle);

extern SXESDK_API int sxm_sports_begin(ptr requestHandle,
                                       int extType,
                                       int extIndex,
                                       ptr *pExtHandle,
                                       SXMSportsRow *pData);

extern SXESDK_API int sxm_sports_begin_nested(ptr parentExtHandle,
                                              int type,
                                              int index,
                                              ptr *pExtHandle,
                                              SXMSportsRow *pData);

extern SXESDK_API int sxm_sports_extract_row(ptr extHandle,
                                             SXMSportsRow *pData);

extern SXESDK_API int sxm_sports_extract_raw(ptr extHandle,
                                             SXMSportsRaw *pRawData);

extern SXESDK_API int sxm_sports_end(ptr extHandle);

extern SXESDK_API int sxm_sports_set_debug(int debugLevel);

extern SXESDK_API int sxm_sports_get_debug(int *pDebugLevel);

extern SXESDK_API int sxm_sports_get_version(uint *pVersion);

extern SXESDK_API int sxm_sports_cfile_clean(void);

extern DEPRECATED SXESDK_API int sxm_sports_save_data(void);

extern DEPRECATED SXESDK_API int sxm_sports_set_auto_save(BOOL auto_save_on_stop);

/**
 * @}
 */

#ifdef __cplusplus
}
#endif

#endif
