/******************************************************************************/
/*                Copyright (c) Sirius XM Satellite Radio, Inc.               */
/*                            All Rights Reserved                             */
/*      Licensed Materials - Property of Sirius XM Satellite Radio, Inc.      */
/******************************************************************************//**
 * \file sxm_gz_defs.h
 * \author Eugene Lopatukhin
 * \date 11/20/2013
 *
 * Public header file for game zone service
 ******************************************************************************/

  /*********************************/
 /*  PREVENT REDUNDANT INCLUSION  */
/*********************************/
#ifndef SXM_GZ_DEFS_H
#define SXM_GZ_DEFS_H

#include <sxm_build.h>

#if (!SXM_USE_GEN8)

#include "sxm_typ.h"
#include "sxm_sxi.h"
#include "sxm_sdk.h"
#include "sxm_common.h"

#ifdef __cplusplus
extern "C" {
#endif

/**
 * \defgroup gamezone_pubapi (hla) Game Zone Public API
 * \brief This section defines the High Level Audio (hla) Game Zone Public API
 * 
 * A radio feature in which channels currently broadcasting live games involving any
 * of the user’s favorite teams are made available to the user in a list or virtual
 * category. GameZone is not directly related to SportsFlash, but its implementation
 * shares use of a Team/League Table provided by SiriusXM.
 * 
 * The Game Zone (GZ) service communicates with application through callbacks. When the
 * service is started callback function is supplied. This callback function is called
 * when communication with application level logic is required.
 */
/**@{*/

  /**************/
 /*  INCLUDES  */
/**************/

  /***************/
 /*  CONSTANTS  */
/***************/

/**
 * \name Encoding of callback parameter to specify league and list change type
 * @{
 */
#define GZ_CALLBACK_TYPE_MASK       0x000000ffu
#define GZ_CALLBACK_TYPE_SHIFT      0
#define GZ_LEAGUE_ID_MASK           0x0000ff00u
#define GZ_LEAGUE_ID_SHIFT          8

/* SportsTeamBroadcastId encoding */
#define SPORTS_LEAGUE_ID_MASK       0x007f0000u
#define SPORTS_TEAM_ID_MASK         0x000003ffu
#define SPORTS_LEAGUE_ID_SHIFT      16
#define SPORTS_TEAM_ID_SHIFT        0

#define MAX_LEAGUE_ID               (127)
/** @} */

/**************/
 /* TYPEDEFS */
/**************/

/**
 * \brief GZ service callback types
 * 
 * To get type and detail \ref GET_GZ_CALLBACK_TYPE(a) and \ref GET_GZ_LEAGUE_ID(a) macros
 * are provided by GZ service.
 */
typedef enum {
    GZ_ENTRY_ADDED,                     //!< New entry added to games list
    GZ_ENTRY_REMOVED,                   //!< %Entry removed from games list
    GZ_SPORTS_LEAGUE_INFO_UPDATE_OTA,   //!< OTA update took place for league list
    GZ_SPORTS_TEAM_INFO_UPDATE_OTA,     //!< OTA update took place for team list
    GZ_SERVICE_STOPPED                  //!< Game zone service stopped
} GZ_CALLBACK_TYPE;

/**
 * \brief %Entry for active game
 * 
 * Game data.
 */
typedef struct {
    UINT16 sid;
    UINT16 chanID;
    UINT8 sportsTeamBroadcastBias;
    uint sportsTeamBroadcastIds[SXI_SPORT_TEAM_BROADCAST_ID_MAX];
} GAME_INFO_STRUCT;

/**
 * \brief %Entry for league list
 */
typedef struct {
    uint activeGameCount;
    SXMLeague league;
} LEAGUE_INFO_STRUCT;

/***************/
 /*  VARIABLES  */
/***************/

  /************/
 /*  MACROS  */
/************/
/**
 * \name Callback parameter encoding macros
 */
/** @{ */
#define SET_GZ_CALLBACK_TYPE(a)     (((uint)a << GZ_CALLBACK_TYPE_SHIFT) & GZ_CALLBACK_TYPE_MASK)
#define SET_GZ_LEAGUE_ID(a)         (((uint)a << GZ_LEAGUE_ID_SHIFT) & GZ_LEAGUE_ID_MASK)
#define SET_GZ_CALLBACK_STATUS(t,l) ((SET_GZ_CALLBACK_TYPE(t) | SET_GZ_LEAGUE_ID(l)))
/** @} */

/**
 * \name Callback parameter decoding macros
 * 
 * They take in second Game Zone (GZ) callback parameter as argument.
 */
/** @{ */
#define GET_GZ_CALLBACK_TYPE(a) ((GZ_CALLBACK_TYPE)(((uint)a & GZ_CALLBACK_TYPE_MASK) >> GZ_CALLBACK_TYPE_SHIFT))  //!< decodes callback type, see #GZ_CALLBACK_TYPE
#define GET_GZ_LEAGUE_ID(a) ((UINT8)(((uint)a & GZ_LEAGUE_ID_MASK) >> GZ_LEAGUE_ID_SHIFT))                         //!< decodes league id
/** @} */


/************/
/**
 * \name Macros for decoding of Team and League IDs from Broadcast Team ID
 */
/** @{ */
#define GET_TEAM_ID(a)   ((UINT16)(((uint)a & SPORTS_TEAM_ID_MASK) >> SPORTS_TEAM_ID_SHIFT))    //!< decodes team id
#define GET_LEAGUE_ID(a) ((UINT8)(((uint)a & SPORTS_LEAGUE_ID_MASK) >> SPORTS_LEAGUE_ID_SHIFT)) //!< decodes league id
/** @} */

/************/
/**
* \name Encoding of Broadcast Team ID out of Team and League IDs
*/
/** @{ */
#define CREATE_BCAST_TEAM_ID(l,t)   ((((t) << SPORTS_TEAM_ID_SHIFT) & SPORTS_TEAM_ID_MASK) | \
                                     (((l) << SPORTS_LEAGUE_ID_SHIFT) & SPORTS_LEAGUE_ID_MASK))
/** @} */


  /****************/
 /*  PROTOTYPES  */
/****************/

/**************************************************************************************************/
/*			                 Game Zone Service High Level API                                     */
/**************************************************************************************************/
SXESDK_API void sxm_gz_application_callback_enable(BOOL enable);
SXESDK_API int sxm_gz_configure(UINT16 teamsCount, const UINT32 *teamsList);

SXESDK_API int sxm_gz_begin(void);
SXESDK_API int sxm_gz_end(void);
SXESDK_API int sxm_gz_extract_league_count(uint *pLeagueCount);
SXESDK_API int sxm_gz_extract_team_count(int *pCount,  int leagueId);
SXESDK_API int sxm_gz_extract_league_info(SXMLeague *pLeagueInfo, uint maxCount, uint *actCount);
SXESDK_API int sxm_gz_extract_team_info_for_league(SXMTeam *pTeamInfo, int leagueId, int maxCount, int *actCount); 

SXESDK_API int sxm_gz_extract_game_count(INT32 leagueId, uint *pGameCount);

SXESDK_API int sxm_gz_extract_league_list(LEAGUE_INFO_STRUCT *pInfo, uint maxCount, uint *actCount);

SXESDK_API int sxm_gz_extract_game_list(
        INT32 leagueId,
        GAME_INFO_STRUCT *pInfo,
        uint maxCount,
        uint *actCount);

SXESDK_API int sxm_gz_extract_team_info(uint sportsTeamBroadcastId,
        SXMTeam *pTeamInfo);

/**********************/
 /*  INLINE FUNCTIONS  */
/**********************/

/** @} */

#ifdef __cplusplus
}
#endif

#endif /* !SXM_USE_GEN8 */

#endif /* SXM_GZ_DEFS_H */
 
