/************************************************************************
 *                                                                      *
 *            Build and Analyse the various SDK files                   *
 *            =======================================                   *
 *                                                                      *
 *  Copyright 2015 Sirius XM Radio, Inc.                                *
 *  All Rights Reserved.                                                *
 *  Licensed Materials - Property of Sirius XM Radio, Inc.              *
 *                                                                      *
 *    Header for TFile Query Language (TQL) Processor                   *
 *                                                                      *
 ************************************************************************/

/**
 * \page tql_page T-File Query Language (TQL)
 *
 * \tableofcontents
 *
 * \section tql_intro Introduction
 * The TQL (T-File Query Language) implements the SQL-like language to
 * allow make simple filtering over the T-File content.
 *
 * \section tql_commands Commands
 *
 * Each command/sentence is defined like a sequence of symbols followed by 
 * semicolon (\b ;).
 * \note Currently only one command support in the same query.
 *
 * The there are two different command types:
 * -# Utility
 * -# Filtering/Selection
 *
 * \subsection tql_unility_cmd Utility queries
 * The first type provides set of command starting with period (\b .) followed
 * by set of words. The following table shows the set of supported utility 
 * commands:
 *
 * | Command name | Description |
 * |:----------:|:---------:|
 * | .schema; | Prints out the schema of the TQL data base |
 * | .help; | Prins out the help notice |
 *
 * \subsection tql_selection_cmd Selection queries
 * The second type allows selection of the certain amount of data from the T-File.
 * The command has the following view:
 *
 * \code{.sql}
 * select {*|column[,column2],...} from {table} [where {condition}]; 
 * \endcode
 * \par Where:
 * - <b>{*|column[,column2],...}</b> -
 *                               specified the set of columns which is required to show
 *                               as query result. It could be either set of table's
 *                               column separated by comma (\b ,) or start (\b *) to
 *                               show all table's columns.
 * - <b>{table}</b> - the name of the table to be enquired. Available tables for the query can
 *                    be obtained by <i>.schema;</i> command.
 * - <b>[where {condition}]</b> - the optional statement which can be used to specify
 *                                some conditions which shall be met by each row in order
 *                                to be shown as query result. Available columns for the query
 *                                condition can be obtained by <i>.schema;</i> command.
 * 
 * \section tql_condition Condition Statement
 * As it was described by \ref tql_selection_cmd the selection statement provides ability to
 * specify some condition to filter out output.
 *
 * Since the TQL is a SQL-like language it allows using of certain basic elements:
 * | Class | Members  | Description |
 * |:-----:|:--------:|------------:|
 * | Logical Operations| < > <= >= <> not and or like | Logical operations which compares to operands and give back boolean value|
 * | Arithmetical Operations | + - * \ % | Arithmetical operations which can performed over numerical data types|
 * | Functions | sin cos sqrt abs isnull | Set of embedded function to perform well-known operations |
 * | Variables | any table's column which | Allows referencing the row's column value to do the expression computation|
 *
 * \subsection tql_functions Functions
 * \par sin
 * Calculates argument sine. Available only if SXM_TQL_MATH_SUPPORT is set to 1.
 * \par con
 * Calculates argument cosine. Available only if SXM_TQL_MATH_SUPPORT is set to 1.
 * \par sqrt
 * Calculates square root for the argument. Available only if SXM_TQL_MATH_SUPPORT is set to 1.
 * \par abs
 * Calculates absolute value for the argument.
 * \par isnull
 * Returns boolean \c true if the passed string is empty or null string, or \c false if not.
 * \subsection tql_example Examples
 * This section shows what kind of queries the TQL supports.
 *
 * Let's introduce simple table and call it <b>test</b>:
 * | ID | NAME | LAT | LON |
 * |:--:|:----:|:---:|:---:|
 * |1|ALASKA|80.0|-170.0|
 * |2|FLORIDA|20.0|-40.0|
 * |3|MICHIGAN|40.0|-83.0|
 *
 * \par Example 0
 * \code{.sql}
 * .schema;
 * \endcode
 * Result:
 \verbatim
 TABLE test (id INT,desc STRING,lat REAL,lon REAL );
 \endverbatim
 *
 * \par Example 1
 * \code{.sql}
 * select * from test;
 * \endcode
 * Result:
 \verbatim
 |1|ALASKA|80.000000|-170.000000|
 |2|FLORIDA|30.000000|-40.000000|
 |3|MICHIRAN|40.000000|-83.000000|
 \endverbatim
 *
 * \par Example 2
 * \code{.sql}
 * select lat,lon from test;
 * \endcode
 * Result:
 \verbatim
 |80.000000|-170.000000|
 |30.000000|-40.000000|
 |40.000000|-83.000000|
 \endverbatim
 *
 * \par Example 3
 * \code{.sql}
 * select * from test where lat>30;
 * \endcode
 * Result:
 \verbatim
 |1|ALASKA|80.000000|-170.000000|
 |3|MICHIRAN|40.000000|-83.000000|
 \endverbatim
 *
 * \par Example 4
 * \code{.sql}
 * select * from test where lat>30 and lon>-170;
 * \endcode
 * Result:
 \verbatim
|3|MICHIRAN|40.000000|-83.000000|
 \endverbatim
 *
 * \par Example 5
 * \code{.sql}
 * select * from test where desc like 'RI';
 * \endcode
 * Result:
 \verbatim
|2|FLORIDA|30.000000|-40.000000|
 \endverbatim
 */

#ifndef SXM_TQL_PARSER_H
#define SXM_TQL_PARSER_H

#ifdef __cplusplus
extern "C"
{
#endif

/**
 * \brief SXM Tql Boolean Type
 */
typedef unsigned int  SXM_TQL_BOOL;

#define SXM_TQL_FALSE ((SXM_TQL_BOOL)(0))
#define SXM_TQL_TRUE  (!SXM_TQL_FALSE)

/** Defines max supported tables */
#define SXMTQL_TABLE_MAX_COUNT (24)
/** Defines max supported columns per table */
#define SXMTQL_TABLE_COLUMN_MAX_COUNT (32)

typedef enum
{
    SXM_TQL_TYPE_REAL = 'R',   //!< Float point value
    SXM_TQL_TYPE_INT  = 'I',   //!< Signed integer value
    SXM_TQL_TYPE_STR  = 'S',   //!< nil-terminated string
    SXM_TQL_TYPE_BOOL = 'B',   //!< Boolean 
    SXM_TQL_TYPE_BLOB = '*',   //!< Any other data types which are not covered
} SXMTqlType; //!< All supported data types

typedef struct
{
    int id; //!< Unique table id
    const char *name; //!< table name
} SXMTqlTable; //!< Table descriptor

typedef struct
{
    int id; //!< Unique id to address this row
    const char *name; //!< Unique column name in frame of the table
    SXMTqlType type; //!< Column data type
} SXMTqlTableColumn; //!< Single column description

typedef struct
{
    union
    {
        double dblV; //!< Double value
        int intV; //!< Integer value
        struct
        {
            const char *strV; //!< nil-terminated string
            char symbV[sizeof(char) + 1]; //!< Place holder for char type
        } str;
        SXM_TQL_BOOL boolV; //!< Boolean value
        struct
        {
            int size; //!< size of the single block
            int count; //!< number of data blocks within the blob
            void *data; //!< data address
        } blob; //!< Blob
    } value; //!< Value union
    SXMTqlType type; //!< Value data type to use proper field
} SXMTqlValue; //!< Represents the value 

/**
 * \brief TQL Statement Handle
 */
typedef struct _sxm_tql_statement_struct *SXMTqlStatement;

#define sxm_tql_val_set(_var, _field, _type, _v) \
    (_var)->type = SXM_TQL_TYPE_##_type;         \
    (_var)->value._field = (_v)                  \

#define sxm_tql_val_set_real(_var, _v) \
    sxm_tql_val_set((_var), dblV, REAL, (_v));
#define sxm_tql_val_set_int(_var, _v) \
    sxm_tql_val_set((_var), intV, INT, (_v));  
#define sxm_tql_val_set_str(_var, _v) \
    sxm_tql_val_set((_var), str.strV, STR, (_v));
#define sxm_tql_val_set_bool(_var, _v) \
    sxm_tql_val_set((_var), boolV, BOOL, (_v));

/**
 * \brief Probes the database to learn the tables supported
 * \param[in] tables preallocated memory
 * \param[in] capacity number of allocated items addressed by \c tables
 * \param[out] size place holder to let caller know how many tables filled up.
 *                  It must be less or equal to the \c capacity.
 * \param[in] arg the caller's own data address which is passed through the
 *                statement creation function.
 * \return SXM_E_OK if the database filled up the array or any other error code.
 */
 typedef int (*SXMTqlProbe)(SXMTqlTable *tables, int capacity, int *size,
                            const void *arg);

/**
 * \brief Probes the table to gather information about table columns
 * \param[in] table table id which is unique for the table.
 * \param[in] columns preallocated memory
 * \param[in] capacity number of allocated items addressed by \c columns
 * \param[out] size place holder to let caller know how many columns filled up.
 *                  It must be less or equal to the \c capacity.
 * \param[in] arg the caller's own data address which is passed through the
 *                statement creation function.
 * \return non-zero in success and 0 in case or error.
 */
typedef int (*SXMTqlTableProbe)(int table, SXMTqlTableColumn *columns,
                                int capacity, int *size, const void *arg);

/**
 * \brief Row data lookup function definition
 *
 * The caller shall implement this function to extract data from the service dependent row-data.
 * The TQL engine calls this function passing the name of the table and id for the field which
 * was previously asked via probe callback during statement creation.
 *
 * \param[in] table the table id
 * \param[in] id table's field unique identifier
 * \param[out] value the keeper for the row data for the corresponded column.
 * \param[in] row_data the address of the service dependent data to do the row extraction
 * \param[in] arg the caller's own data address which is passed through the
 *                statement creation function.
 */
typedef int (*SXMTqlRowLookup)(int table, int id, SXMTqlValue *value,
                               const void *row_data, const void *arg);

/**
 * \brief Output the value for the pair table-column
 *
 * The function is being called by the TQL to let implementation output
 * each particular value in accordance with the query.
 * \param[in] table the table id
 * \param[in] column table's column id
 * \param[in] value the value to be outputted
 * \param[in] arg the caller's own data address which is passed through the
 *                statement creation function.
 */ 
typedef int (*SXMTqlValueOutput)(int table, int column,
                                 const SXMTqlValue *value, const void *arg);

typedef struct
{
    SXMTqlProbe tprobe; //!< Table probes
    SXMTqlTableProbe cprobe; //!< Single table probe
    SXMTqlRowLookup row_lookup; //!< ROW lookup
    SXMTqlValueOutput value_output; //!< Outputs the value
} SXMTqlCallback; //!< TQL Callback structure

/**
 * \brief Creates the statement based on the query
 * \note In case of special commands like .schema; the statement will not
 *       be created since there is no statement to do evaluations,
 *       but the return code will be SXM_E_OK.
 * \param[out] statement the holder for the statement in case of success
 * \param[in] callback TQL callback interface
 * \param[in] query the TQL query to create the statement
 * \param[in] arg  the caller's own data address which will be passed to the callbacks
 * \return SXM_E_OK on case of success and some other code in case of error.
 */
int sxm_tql_statement(SXMTqlStatement *statement, const SXMTqlCallback *callback,
                      const char *query, const void *arg);

/**
 * \brief Evaluates the row data necessity based on the row data and statement
 * \param[in] statement valid TQL statement
 * \param[in] row_data the address of the raw which is known by the callback implementation
 * \return SXM_E_OK if row meets statement conditions;
 *         SXM_E_NOENT if the row has nothing to populate;
 *         Any other error code in case of failure
 */
int sxm_tql_evaluate(SXMTqlStatement statement, const void *row_data);

/** 
 * \brief Checks the table existence within the query.
 * \param[in] statement valid TQL statement
 * \param[in] name the table name to make a check
 * \return SXM_E_OK in case of the table with this id referenced by the query or
 *         any other code in case of failure.
 */
int sxm_tql_query_table(SXMTqlStatement statement, const char *name);

/**
 * \brief Destroys valid statement.
 * \note After this call the valid statement won't be valid and must not be used
 * \param[in] statement valid TQL statement
 * \return SXM_E_OK in case of success or some other code in case of failure.
 */
int sxm_tql_statement_destroy(SXMTqlStatement statement);

/** \name TQL Definition Helpers
 *  \brief Provides the simplest way of database definition,
 *  \note For complicated cases it is recommended to use interface directly
 * @{
 */
typedef struct
{
    SXMTqlTableColumn column; //!< TQL Column
    size_t field_offset; //!< Data field offset
    size_t field_byte_size; //!< Data field size for read data type
} SXMTqlTableColumnDef; //!< Column definition helper

typedef struct
{
    SXMTqlTable table; //!< TQL table
    SXMTqlTableColumnDef columns[SXMTQL_TABLE_COLUMN_MAX_COUNT + 1];   
} SXMTqlTableDef; //!< Table definition

/** Defines sample schema structure */
typedef const SXMTqlTableDef *(*SXMTqlShemeCallback)(void);

/**
 * \brief Declares the default tables definition name
 */
#define SXMTQL_SERVICE_TABLES_FUNC sxmtql_tables_get

/**
 * \brief Declares the default tables definition name
 */
#define SXMTQL_SERVICE_TABLES_FUNC_P sxmtql_tables

/**
 * \brief Declares the default tables definition name
 */
#define SXMTQL_SERVICE_TABLES &SXMTQL_SERVICE_TABLES_FUNC_P

/**
 * \brief Declares the default interface
 */
#define SXMTQL_DEFAULT_INTERFACE sxmtql_default_callback

/**
 * \brief Starts tables definition section
 */
#define SXMTQL_TABLES() \
    static const SXMTqlTableDef *SXMTQL_SERVICE_TABLES_FUNC(void) { \
        static SXMTqlTableDef tables[SXMTQL_TABLE_MAX_COUNT + 1]; \
        static int ready = 0; \
        unsigned int table = 0, column = 0; \
        if (ready) return &tables[0];

/**
 * \brief This macro helps to generate ID based on user provided one
 * 
 * Since by default the __COUNTER__ is used for ID generation and it may 
 * happen that application wants to use is to distinguish columns the macro
 * shall be used to let application define the ID and use it inside the TQL
 * callback interface.
 * \warning There is no control over the ID values which are used within single
 *          table. So, application must take care of it and use the macro instead of
 *          RAW values for unique identifiers.
 *
 * \param[in] _id user provided value as ID
 */
#define SXMTQL_MAKEID(_id) \
    ((_id) + 0xF0000 /* magic number */)

/**
 * \brief Single table definition
 * \param[in] _name table name
 */
#define SXMTQL_TABLE(_name) \
    if (table < (unsigned int)ARRAY_SIZE(tables)) { \
        const SXMTqlTable localTable = {__COUNTER__, #_name}; \
        tables[table].table = localTable; \
    }

/**
 * \brief Single table definition
 * \param[in] _id user predefined ID.
 * \param[in] _name table name
 */
#define SXMTQL_TABLE_ID(_id, _name)   \
    if (table < (unsigned int)ARRAY_SIZE(tables)) { \
        const SXMTqlTable localTable = {_id, #_name}; \
        tables[table].table = localTable; \
    }

/**
 * \brief Column definition with user pre-defined ID
 * \param[in] _id user predefined ID.
 * \param[in] _name column name
 * \param[in] tql_type column data type. \ref SXMTqlType
 * \param[in] _data_type field data keeping structure
 * \param[in] _data_type_field the field within _data_type
 */
#define SXMTQL_COLUMN_ID(_id, _name, tql_type, _data_type, _data_type_field) \
    if ((table < (unsigned int)ARRAY_SIZE(tables)) && (column < SXMTQL_TABLE_COLUMN_MAX_COUNT)) { \
        const SXMTqlTableColumnDef localColumn = \
                {{(_id), #_name, SXM_TQL_TYPE_##tql_type}, \
                    (size_t)&(((_data_type*)0)->_data_type_field), \
                    sizeof(((_data_type*)0)->_data_type_field)}; \
        tables[table].columns[column++] = localColumn; \
    }

/**
 * \brief Column definition
 * \param[in] _name column name
 * \param[in] tql_type column data type. \ref SXMTqlType
 * \param[in] _data_type field data keeping structure
 * \param[in] _data_type_field the field within _data_type
 */
#define SXMTQL_COLUMN(_name, tql_type, _data_type, _data_type_field)     \
    SXMTQL_COLUMN_ID(__COUNTER__, _name, tql_type, _data_type, _data_type_field)

/**
 * \brief Ends previously started table definition
 */
#define SXMTQL_TABLE_END() \
    if ((table < ARRAY_SIZE(tables)) && (column < SXMTQL_TABLE_COLUMN_MAX_COUNT)) { \
        const SXMTqlTableColumnDef localColumn = \
                {{-1, NULL, SXM_TQL_TYPE_BLOB}, (size_t)0U, (size_t)0U}; \
        tables[table].columns[column] = localColumn; \
        table++; column = 0; \
    }

/**
 * \brief End previously started tables definition
 */
#define SXMTQL_TABLES_END() \
        if (table < (unsigned int)ARRAY_SIZE(tables)) { \
            tables[table].table.id = -1;  \
        } \
        ready = !0; \
        return &tables[0]; \
    } \
    static SXMTqlShemeCallback SXMTQL_SERVICE_TABLES_FUNC_P = &SXMTQL_SERVICE_TABLES_FUNC;

/**
 * \brief Default callback implementation
 */
extern const SXMTqlCallback * const SXMTQL_DEFAULT_INTERFACE;

/**
 * @}
 */

#ifdef __cplusplus
}
#endif

#endif /* SXM_TQL_PARSER_H */
