/************************************************************************
 *                                                                      *
 *            Build and Analyse the states database                     *
 *            =====================================                     *
 *                                                                      *
 *                                 Copyright 2013 Sirius XM Radio, Inc. *
 *                                                 All Rights Reserved. *
 *               Licensed Materials - Property of Sirius XM Radio, Inc. *
 *                                                                      *
 *    Create and dump locations tfile                                   *
 *                                                                      *
 ************************************************************************/

#include "sdkfiles.h"

#define MAX_COUNTRY_ID 3
#define LONG_DATELINE_THRESHOLD   (300*32768)

#define STATES_SERVICE_NAME             "states"
#define STATES_SERVICE_DB_FILE_TYPE     'R'
#define STATES_SERVICE_DB_FILE_NAME     "states"

#define STATES_PSV_LINE_SIZE_MAX        128

// The individual country records are indexed through the user entries 
// in the transaction block:                                           
//                                                                       
// |                 |                                                     |
// |:----------------|:----------------------------------------------------|
// | user[0]         | version number                                      |
// | user[1]         | total number of state/province entries              |
// | user[2], user[3]| start of US entries and count of US entries         |
// | user[4], user[5]| start of CA entries and count of CA entries         |
// | user[6], user[7]| start of MX entries and count of MX entries         |
// | user[8], user[9]| start of PR entries and count of PR entries         |

//
// T-File
//
#define STATES_TFILE_SCHEMA_VERSION     (1)
#define STATES_TFILE_DEFAULT_VERSION    (1)
#define BLOCK_SIZE                      (4096)
#define ROOT_SIZE                       (BLOCK_SIZE * 2)

#define STATES_USER_ENTRIES_CNT         (1)

typedef struct {
    int mbr[4];
    ushort off;
    ushort bc;
    ushort count;
    ushort offn;
    ushort fips;
    byte country;
    byte start;
    char abbr[4];
}StateRec;

typedef struct {
    fix mbr[4];
    ushort fips;
    ushort offn;
}CountyRec;

typedef union {
    struct {
        StateRec states[128];
        char snames[2048];
    }d;
    struct {
        byte b[8192];
    }b;
}RootRec;

typedef struct {
    CountyRec cr[256];
    char cnames[4096];
}CRec;

//
// Stand Alone Section
//
#ifdef SDKFILES_STANDALONE_BUILD

typedef struct  
{
    const char *abbr, *sname, *county;
    float ll_lat, ll_lon, ur_lat, ur_lon; 
    ushort fips;
    byte country;
} LocationRow;

SXMTQL_TABLES()
    SXMTQL_TABLE(locations)
        SXMTQL_COLUMN(country, INT, LocationRow, country)
        SXMTQL_COLUMN(abbr, STR, LocationRow, abbr)
        SXMTQL_COLUMN(sname, STR, LocationRow, sname)
        SXMTQL_COLUMN(county, STR, LocationRow, county)
        SXMTQL_COLUMN(fips, INT, LocationRow, fips)
        SXMTQL_COLUMN(ll_lon, REAL, LocationRow, ll_lon)
        SXMTQL_COLUMN(ll_lat, REAL, LocationRow, ll_lat)
        SXMTQL_COLUMN(ur_lon, REAL, LocationRow, ur_lon)
        SXMTQL_COLUMN(ur_lat, REAL, LocationRow, ur_lat)
    SXMTQL_TABLE_END()
SXMTQL_TABLES_END()

#endif // #ifdef SDKFILES_STANDALONE_BUILD


/*******************************************************************************
 *
 * Function    :  emit
 *
 * Description :  Allocates new blocks for table and stores to database.
 *
 * Parameters  : 
 *          1  :  pStream = pointer to the data stream
 *          2  :  pSR = pointer to the State Record
 *          3  :  pCR = pointer to the County Record
 *          4  :  sizeCname = size of the County Names
 *
 * Returns     :  SXM_E_OK on success
 *
 * Note        :  
 *
 ********************************************************************************/
static int emit(SXMTFileStream *pStream, StateRec *pSR, CRec *pCR, size_t sizeCname) {
    int rc = SXM_E_OK;

    if  (pSR->count == 0) {
        pSR->off = 0;
        printf("    State %s.  off %3d bc %2d count %4d\n", 
               pSR->abbr, pSR->off, pSR->bc, pSR->count);
    } else {
        uint start, blockc;
        ushort ob = (ushort)(pSR->count*sizeof(CountyRec));
        int j;

        for  (j = 0; j < pSR->count; j++)
            pCR->cr[j].offn = (ushort)(pCR->cr[j].offn + ob);

        sxm_tstream_write(pStream, pCR->cr, ob, FALSE);
        sxm_tstream_write(pStream, pCR->cnames, sizeCname, FALSE);

        rc = sxm_tstream_commit(pStream, &start, &blockc);
        if (rc == SXM_E_OK) {
            pSR->off = (ushort)start;
            pSR->bc = (ushort)blockc;
            printf("    State %s.  off %3d bc %2d count %4d\n", 
               pSR->abbr, pSR->off, pSR->bc, pSR->count);
            rc = sxm_tstream_clean(pStream);
        }
    }
    return rc;
}

static int states_build_table(SXMTFile *dbfile, SxmCsvParser *pPsv) {

    int rc = SXM_E_OK;
    StateRec *s = NULL;
    CountyRec *c = NULL;
    CRec cr;
    char *cn = NULL;
    int ist, isn, ico, isc, ifips, imbr;
    int prev = 0, cp = 0, pcid = -1;
    uint pcount = 0, six = 0;
    char *sn;
    int dateline_issue = 0;
    SXMTFileStream *stream = NULL;
    RootRec *pRootBlock;
    uint *user;

    switch (0) { default: {

        /* Read fields indexes */

        ist = sxm_csv_index(pPsv, "STATE");
        if (ist < 0) {
            non_fatal("Column 'STATE' is not found");
            rc = SXM_E_INVAL;
            break;
        }

        ico = sxm_csv_index(pPsv, "COUNTRY");
        if (ico < 0) {
            non_fatal("Column 'COUNTRY' is not found");
            rc = SXM_E_INVAL;
            break;
        }

        isn = sxm_csv_index(pPsv, "SNAME");
        if (isn < 0) {
            non_fatal("Column 'SNAME' is not found");
            rc = SXM_E_INVAL;
            break;
        }

        isc = sxm_csv_index(pPsv, "COUNTY");
        if (isc < 0) {
            non_fatal("Column 'COUNTY' is not found");
            rc = SXM_E_INVAL;
            break;
        }

        ifips = sxm_csv_index(pPsv, "FIPS");
        if (ifips < 0) {
            non_fatal("Column 'FIPS' is not found");
            rc = SXM_E_INVAL;
            break;
        }

        imbr = sxm_csv_index(pPsv, "MBR");
        if (imbr < 0) {
            non_fatal("Column 'MBR' is not found");
            rc = SXM_E_INVAL;
            break;
        }

        // Get pointer to the root block
        pRootBlock = (RootRec *)sxm_tfile_root(dbfile);
        if (!pRootBlock) {
            non_fatal("Failed to get DB root");
            rc = SXM_E_ERROR;
            break;
        }

        // Put data to guide service where to find the logos data
        user = sxm_tfile_other_user(dbfile);
        if (!user) {
            non_fatal("Failed to get user data section");
            rc = SXM_E_ERROR;
            break;
        }

        // Create the stream for update
        rc = sxm_tstream_create(dbfile, &stream);
        if (rc != SXM_E_OK) {
            non_fatal("Failed to create tstream (rc=%d)", rc);
            break;
        }

        sn = pRootBlock->d.snames;

        while (SXM_E_OK == rc) {

            const char *state, *sname, *country, *county, *fips, *mbr;
            int cid;

            rc = sxm_csv_next_line(pPsv);
            if (SXM_E_OK != rc) {
                if (SXM_E_NOENT == rc) {
                    rc = SXM_E_OK;
                }
                break;
            }

            state = sxm_csv_str_val(pPsv, (uint)ist, NULL);
            sname = sxm_csv_str_val(pPsv, (uint)isn, NULL);
            country = sxm_csv_str_val(pPsv, (uint)ico, NULL);
            county = sxm_csv_str_val(pPsv, (uint)isc, NULL);
            fips = sxm_csv_str_val(pPsv, (uint)ifips, NULL);
            mbr = sxm_csv_str_val(pPsv, (uint)imbr, NULL);
            cid = atoi(country);

            // validate country ID is correct
            if (!IS_IN_RANGE(cid, 0, MAX_COUNTRY_ID)) {
                non_fatal("Invalid Country ID: %d, skipping. Line: %u",
                          cid, sxm_csv_current_line(pPsv));
                continue;
            }

            if (*sname != '-') {
                int tfips;

                if  (prev > 0) {
                    dateline_issue = 0;
                    rc = emit(stream, s, &cr, (size_t)(cn - cr.cnames));
                    if (rc != SXM_E_OK) 
                    {
                        non_fatal("Failed to save region %d (rc=%d)", cid, rc);
                        break;
                    }
                }

                if (SXM_STATES_INDEX_MAX <= six) {
                    non_fatal("Number of states is out of range (%u >= %u). Line: %u",
                              six, SXM_STATES_INDEX_MAX, sxm_csv_current_line(pPsv));
                    break;
                }

                tfips = atoi(fips);

                // FIPS value is ushort and cannot store value more than 65535
                if (65535 < tfips) {
                    non_fatal("Invalid FIPS value %d for state, skipping. Line: %u",
                              tfips, sxm_csv_current_line(pPsv));
                    continue;
                }

                if (cid != pcid) {
                    // Change of region
                    pcid = cid;
                    pcount = 0;

                    // Set the user section entry count for this region
                    user[2 * (pcid + 1)] = six;
                }

                pcount++;
                six++;

                // Set the user section entry count for this region
                user[2 * (pcid + 1) + 1] = pcount;

                s = &pRootBlock->d.states[prev];
                strncpy(s->abbr, state, sizeof(s->abbr));
                if (*mbr == '[') {
                    double d1, d2, d3, d4;
                    sscanf(mbr + 1, "%lf,%lf %lf,%lf", &d1, &d2, &d3, &d4);
                    s->mbr[0] = (int)(d1*32768.0);
                    s->mbr[1] = (int)(d2*32768.0);
                    s->mbr[2] = (int)(d3*32768.0);
                    s->mbr[3] = (int)(d4*32768.0);
                }
                else {
                    sscanf(mbr, "%d,%d,%d,%d",
                        &s->mbr[0], &s->mbr[1], &s->mbr[2], &s->mbr[3]);
                }

                //CR 2663:Alaska fips crosses international dateline
                if ((s->mbr[2] - s->mbr[0]) > LONG_DATELINE_THRESHOLD)
                {
                    dateline_issue = 1;
                }

                s->country = (byte)cid;
                s->start = (byte)six;
                s->fips = (ushort)tfips;
                s->offn = (ushort)(sn - pRootBlock->d.snames);
                strcpy(sn, sname);
                sn += strlen(sn) + 1;
                prev++;
                memset(&cr, 0, sizeof(CRec));
                cn = cr.cnames;
                cp = 0;
            }
            else {
                if (s) {
                    int cfips = atoi(fips);

                    // FIPS value is ushort and cannot store value more than 65535
                    if (65535 < cfips)
                    {
                        non_fatal("Invalid FIPS value %d for county, skipping. Line: %u",
                                  cfips, sxm_csv_current_line(pPsv));
                        continue;
                    }

                    c = &cr.cr[cp];
                    if  (*mbr == '[') {
                        double d1, d2, d3, d4;
                        sscanf(mbr+1, "%lf,%lf %lf,%lf", &d1, &d2, &d3, &d4);
                        c->mbr[0] = (int)(d1*32768.0);
                        c->mbr[1] = (int)(d2*32768.0);
                        c->mbr[2] = (int)(d3*32768.0);
                        c->mbr[3] = (int)(d4*32768.0);
                    }
                    else
                        sscanf(mbr, "%d,%d,%d,%d",
                            &c->mbr[0], &c->mbr[1], &c->mbr[2], &c->mbr[3]);

                    //CR 2663:Alaska fips crosses international dateline
                    if (dateline_issue)
                    {
                        if ((c->mbr[2] - c->mbr[0]) > LONG_DATELINE_THRESHOLD)
                        {
                            c->mbr[2] = c->mbr[0];
                            c->mbr[0] = (int)(-180.0*32768.0);
                        }

                        if (cp == 0)
                        {
                            s->mbr[0] = c->mbr[0];
                            s->mbr[2] = c->mbr[2];
                        }
                        else
                        {
                            if (s->mbr[0] > c->mbr[0])
                                s->mbr[0] = c->mbr[0];
                            if (s->mbr[2] < c->mbr[2])
                                s->mbr[2] = c->mbr[2];
                        }
                    }

                    c->fips = (ushort)cfips;
                    c->offn = (ushort)(cn - cr.cnames);
                    strcpy(cn, county);
                    cn += strlen(cn) + 1;
                    if (cn - cr.cnames > 4096) {
                        non_fatal("%s: Name overflow. Line: %u",
                                  s->abbr, sxm_csv_current_line(pPsv));
                        rc = SXM_E_ERROR;
                        break;
                    }
                    cp++;
                    s->count = (ushort)cp;
                }
            }
        }

        // Save the final entry
        if (s)
            rc = emit(stream, s, &cr, (size_t)(cn - cr.cnames));

        if (SXM_E_OK != rc) {
            break;
        }

        // Set the user total number of extries field
        user[STATES_USER_ENTRIES_CNT] = six;

    }} /* switch (0) { default: { */

    if (NULL != stream) {
        sxm_tstream_destroy(stream);
    }

    return rc;
}

static int states_files_input_callback(const SXMFilesInputProps *pProps,
                                       const SXMFilesInputRecord *pRecord,
                                       void *pUserData)
{
    int rc = SXM_E_OK;
    SxmCsvParser *pPsvParser = NULL;

    UNUSED_VAR(pProps);

    rc = create_buffer_psv_parser(pRecord, '|',
                                  STATES_PSV_LINE_SIZE_MAX,
                                  &pPsvParser);
    if (SXM_E_OK == rc) {

        rc = states_build_table((SXMTFile*)pUserData, pPsvParser);

        sxm_csv_delete(pPsvParser);
    }

    return rc;
}

/*******************************************************************************
 *
 * Function    :  states_build
 *
 * Description :  Creates a baseline states service database T-file.
 *
 * Parameters  : 
 *          1  :  pFilesInput = pointer to files input interface object
 *          2  :  version = version number to apply to file (-1 for auto).
 *
 * Returns     : SXe error code
 *
 * Note        :  if ver = -1, version number is auto-calculated from the data.
 *
 ********************************************************************************/
int states_build(const SXMFilesInput *pFilesInput, int version) {

    int rc = SXM_E_OK;
    SXMTFile *dbfile = NULL;

    switch (0) { default: {

        SXMTFileStat st;

        dbfile = sxm_tfile_create(STATES_SERVICE_DB_FILE_TYPE,
                                  STATES_SERVICE_NAME,
                                  STATES_SERVICE_DB_FILE_NAME,
                                  STATES_SERVICE_NAME,
                                  STATES_TFILE_SCHEMA_VERSION,
                                  SXM_TFILE_VERSION,
                                  ROOT_SIZE,
                                  (uint)SXM_TFILE_DYNAMIC_NO_OF_BLOCKS,
                                  BLOCK_SIZE, &rc);
        if (rc != SXM_E_OK) {
            non_fatal("Failed to create DB file (rc=%d)", rc);
            break;
        }

        rc = sxm_tfile_start(dbfile);
        if (rc != SXM_E_OK) {
            non_fatal("Failed to start DB transaction (rc=%d)", rc);
            break;
        }

        /* Enumarate files */
        rc = sxm_files_input_enumerate(pFilesInput, "psv",
                                       states_files_input_callback,
                                       dbfile);
        if (SXM_E_OK != rc) {
            non_fatal("Input files enumeration failed (rc=%d)", rc);
            break;
        }

        if (version == -1) {
            version = STATES_TFILE_DEFAULT_VERSION;
        }

        rc = sxm_tfile_update_version(dbfile, (uint)version);
        if (rc != SXM_E_OK) {
            non_fatal("Failed to update DB version (rc=%d)", rc);
            break;
        }

        rc = sxm_tfile_commit(dbfile);
        if (rc != SXM_E_OK) {
            non_fatal("Failed to commit DB file (rc=%d)", rc);
            break;
        }

        rc = sxm_tfile_stat(dbfile, &st);
        if (rc != SXM_E_OK) {
            non_fatal("Failed to gather stat from the t-file (rc=%d)", rc);
            break;
        }

        printf("Versions\n"
            "  Baseline Overall: %u\n"
            "  Schema          : %u\n"
            "Statistics: %u block(s) by %u byte(s), used %u block(s) [%u%%]\n",
            st.dversion, st.dschema,
            st.fsize, st.ubsize, st.usize, (st.usize * 100) / st.fsize);
    }}

    if (NULL != dbfile) {
        sxm_tfile_close(dbfile);
    }

    return rc;
}

#ifdef SDKFILES_STANDALONE_BUILD
/*******************************************************************************
 *
 * Function    :  states_check_tfile
 *
 * Description :  Processes a states service T-file and displays its contents.
 *
 * Parameters  : 
 *          1  :  in = filename to check (-t parameter).
 *          2  :  query = tql query
 *
 * Returns     :  Nothing
 *
 * Note        :  
 *
 ********************************************************************************/
void states_check_tfile(const char *query) {
    SXMTqlStatement tql = NULL;
    SXMTFile *dbfile = NULL;
    SXMTFileStat st;
    SXMTFileChain *chain = NULL;

    switch (0) { default: {
        int rc = SXM_E_OK;
        uint dbschema;
        const uint *user;
        RootRec *pRootRec;
        uint i;

        if (query)
        {
            if (SXM_E_OK != sxm_tql_statement(&tql, SXMTQL_DEFAULT_INTERFACE,
                                              query, SXMTQL_SERVICE_TABLES))
            {
                non_fatal("failed to create statement for TQL query %s", query);
                break;
            }
            else if (!tql)
            {
                break;
            }
        }

        dbfile = sxm_tfile_open(STATES_SERVICE_DB_FILE_TYPE,
                                STATES_SERVICE_NAME,
                                STATES_SERVICE_DB_FILE_NAME, "rb",
                                STATES_SERVICE_NAME, &dbschema, &rc);
        if (rc != SXM_E_OK) {
            non_fatal(":failed to open DB file(rc=%d)", rc);
            break;
        }

        sxm_tfile_stat(dbfile, &st);
        printf("\nVersions\n"
               "  Baseline Overall: %u\n"
               "  Schema          : %u\n"
               "Statistics: %u block(s) by %u byte(s), used %u block(s) [%u%%]\n\n\n",
            st.dversion, st.dschema,
            st.fsize, st.ubsize, st.usize,
            (st.usize * 100) / st.fsize);

        if (dbschema != STATES_TFILE_SCHEMA_VERSION)
        {
            non_fatal("Unsupported schema version %u\n", dbschema);
            break;
        }

        pRootRec = (RootRec *)sxm_tfile_root(dbfile);
        if (!pRootRec) {
            non_fatal("Failed to access memory for root!");
            break;
        }

        user = sxm_tfile_user(dbfile);
        if (!user) {
            non_fatal("Address of the user area is NULL");
            break;
        }

        if (!tql)
        {
            printf("States File has %d entries\n"
                   "    Country[0]: %4d %4d\n"
                   "    Country[1]: %4d %4d\n"
                   "    Country[2]: %4d %4d\n"
                   "    Country[3]: %4d %4d\n",
                   user[1],
                   user[2], user[3], user[4], user[5],
                   user[6], user[7], user[8], user[9]);
        }

        for  (i = 0; i < user[1]; i++)
        {
            const StateRec *s = &pRootRec->d.states[i];
            CountyRec *c;
            LocationRow row;
            int j;
                      
            memset(&row, 0, sizeof(row));
            row.country = s->country;
            row.sname = pRootRec->d.snames+s->offn;
            row.abbr = &s->abbr[0];
            row.fips = s->fips;
            row.ll_lon = fix2float(s->mbr[0]);
            row.ll_lat = fix2float(s->mbr[1]);
            row.ur_lon = fix2float(s->mbr[2]);
            row.ur_lat = fix2float(s->mbr[3]);

            if (!tql)
            {
                printf("[%2d]: id:%3d co:%d %s %24s fips %6d [%f, %f, %f, %f]\n",
                    i, s->start, row.country, row.abbr, row.sname, row.fips,
                    row.ll_lon, row.ll_lat, row.ur_lon, row.ur_lat);
                printf("    %3d counties at %4d in %2d blocks (Alloc OK)\n", s->count, s->off, s->bc);
            }
            else
            {
                rc = sxm_tql_evaluate(tql, &row);
                if (rc != SXM_E_OK)
                {
                    if (rc == SXM_E_NOENT)
                    {
                        rc = SXM_E_OK;
                    }
                    else
                    {
                        non_fatal("Failed to evaluate TQL statement (%d)", rc);
                        break;
                    }
                }
            }

            c = (CountyRec *)sxm_tfile_alloc_and_read(dbfile, s->off, s->bc, &chain, &rc);
            if (!c) {
                non_fatal(": failed to load county data %u block(s) from %u (rc=%d)\n", s->bc, s->off, rc);
                break;
            }
            else
            {
                for  (j = 0; j < s->count; j++)
                {
                    row.county = (const char*)c + c[j].offn;
                    row.fips = c[j].fips;
                    row.ll_lon = fix2float(c[j].mbr[0]);
                    row.ll_lat = fix2float(c[j].mbr[1]);
                    row.ur_lon = fix2float(c[j].mbr[2]);
                    row.ur_lat = fix2float(c[j].mbr[3]);
                    if (!tql)
                    {
                        printf("         [%3d]: %24s fips %6d [%f, %f, %f, %f]\n", j, row.county, row.fips,
                            row.ll_lon, row.ll_lat, row.ur_lon, row.ur_lat);
                    }
                    else
                    {
                        rc = sxm_tql_evaluate(tql, &row);
                        if (rc != SXM_E_OK)
                        {
                            if (rc == SXM_E_NOENT)
                            {
                                rc = SXM_E_OK;
                            }
                            else
                            {
                                non_fatal("Failed to evaluate TQL statement (%d)", rc);
                                break;
                            }
                        }
                    }
                }

                sxm_tfile_chain_free(chain);
                chain = NULL;

                sxe_free((void*)c);
                c = NULL;
            }
            if (!tql)
                printf("\n");

            if (SXM_E_OK != rc)
            {
                break;
            }
        }
    }}

    if (dbfile) 
    {
        sxm_tfile_close(dbfile);
    }

    if (tql)
    {
        sxm_tql_statement_destroy(tql);
    }
}
#endif // #ifdef SDKFILES_STANDALONE_BUILD
