/************************************************************************
 *                                                                      *
 *            Build and Analyse the locations database                  *
 *            ========================================                  *
 *                                                                      *
 *                                 Copyright 2013 Sirius XM Radio, Inc. *
 *                                                 All Rights Reserved. *
 *               Licensed Materials - Property of Sirius XM Radio, Inc. *
 *                                                                      *
 *    Create and dump locations tfile                                   *
 *                                                                      *
 ************************************************************************/

#include <ctype.h>
#include <sys/stat.h>
#include "sdkfiles.h"

#define SDKFILES_BUILDER
#include "apogee/sxm_apogee_internal.h"
#undef SDKFILES_BUILDER

#define SXM_LOCATION_DB_TFILE_DEFAULT_VERSION     (1)
#define SXM_LOCATION_DB_TFILE_ROOT_BLOCKS         (2)
#define SXM_LOCATION_DB_TFILE_ROOT_SIZE           (SXM_LOCATION_DB_TFILE_BLOCK_SIZE * SXM_LOCATION_DB_TFILE_ROOT_BLOCKS)

#define SXM_LOCATION_DB_TFILE_MAX_LINEAR_SXCOUNT  (0x3FFF)
#define SXM_LOCATION_DB_TFILE_MAX_RAMP_SXCOUNT    (1)

#define APOGEE_TABLE_DIR_PREFIX                 "table"
#define APOGEE_LOCATIONS_FILE_PREFIX            "locations"
#define APOGEE_TABLE_ID_LEN                     2
#define APOGEE_CSV_LINE_SIZE_MAX                256

//
// Stand Alone Section
//
#ifdef SDKFILES_STANDALONE_BUILD

// TQL tables definition
typedef struct  
{
    ushort id, offset, count;
    double ll_lon, ll_lat, ur_lon, ur_lat;
} TableRow;

typedef struct {
    ushort tableId, id, linear, lcount, ramp, rcount;
    double ll_lon, ll_lat, ur_lon, ur_lat;
}BSARow;

typedef struct {
    char *ltype;
    ushort tableId, bsaId, id, tmc1, tmc2, count, type;
    double ll_lon, ll_lat, ur_lon, ur_lat;
}LinearRow;

SXMTQL_TABLES()
    SXMTQL_TABLE(tables)
        SXMTQL_COLUMN(id, INT, TableRow, id)
        SXMTQL_COLUMN(offset, INT, TableRow, offset)
        SXMTQL_COLUMN(count, INT, TableRow, count)
        SXMTQL_COLUMN(ll_lon, REAL, TableRow, ll_lon)
        SXMTQL_COLUMN(ll_lat, REAL, TableRow, ll_lat)
        SXMTQL_COLUMN(ur_lon, REAL, TableRow, ur_lon)
        SXMTQL_COLUMN(ur_lat, REAL, TableRow, ur_lat)
    SXMTQL_TABLE_END()
    SXMTQL_TABLE(bsas)
        SXMTQL_COLUMN(tableid, INT, BSARow, tableId)
        SXMTQL_COLUMN(id, INT, BSARow, id)
        SXMTQL_COLUMN(linear, INT, BSARow, linear)
        SXMTQL_COLUMN(lcount, INT, BSARow, lcount)
        SXMTQL_COLUMN(ramp, INT, BSARow, ramp)
        SXMTQL_COLUMN(rcount, INT, BSARow, rcount)
        SXMTQL_COLUMN(ll_lon, REAL, BSARow, ll_lon)
        SXMTQL_COLUMN(ll_lat, REAL, BSARow, ll_lat)
        SXMTQL_COLUMN(ur_lon, REAL, BSARow, ur_lon)
        SXMTQL_COLUMN(ur_lat, REAL, BSARow, ur_lat)
    SXMTQL_TABLE_END()
    SXMTQL_TABLE(linears)
        SXMTQL_COLUMN(ltype, STR, LinearRow, ltype)
        SXMTQL_COLUMN(tableid, INT, LinearRow, tableId)
        SXMTQL_COLUMN(bsaid, INT, LinearRow, bsaId)
        SXMTQL_COLUMN(id, INT, LinearRow, id)
        SXMTQL_COLUMN(tmc1, INT, LinearRow, tmc1)
        SXMTQL_COLUMN(tmc2, INT, LinearRow, tmc2)
        SXMTQL_COLUMN(count, INT, LinearRow, count)
        SXMTQL_COLUMN(type, INT, LinearRow, type)
        SXMTQL_COLUMN(ll_lon, REAL, LinearRow, ll_lon)
        SXMTQL_COLUMN(ll_lat, REAL, LinearRow, ll_lat)
        SXMTQL_COLUMN(ur_lon, REAL, LinearRow, ur_lon)
        SXMTQL_COLUMN(ur_lat, REAL, LinearRow, ur_lat)
    SXMTQL_TABLE_END()
SXMTQL_TABLES_END()

// Query tables
#define QTYPE_NONE    (0)
#define QTYPE_TABLES  (1)
#define QTYPE_BSAS    (2)
#define QTYPE_LINEARS (3)

// Linear types
#define LTYPE_LINEAR (0)
#define LTYPE_RAMP   (1)
static const char *g_lTypes[2] = {"Linear", "Ramp"};

#endif // SDKFILES_STANDALONE_BUILD

typedef struct {
    SXMTFile *dbfile;
    Table *pRoot;
    SXMTFileStream *stream;
    BSA *pBsaBuffer;
    size_t bsaBufferSize;
    uint blockSize;
} LocationsSdkFilesInputData;

typedef struct {
    uint il_type;
    uint il_idx;
    uint il_llon;
    uint il_llat;
    uint il_ulon;
    uint il_ulat;
    uint il_tmc1;
    uint il_tmc2;
    uint il_cnt;
} Indexes;


static int prepare_db_file(LocationsSdkFilesInputData *pInputData) {
    int rc;

    switch (0) {
    default: {

        pInputData->dbfile = sxm_tfile_create(
            SXM_LOCATION_DB_TFILE_TYPE,
            SXM_LOCATION_DB_FOLDER_NAME,
            SXM_LOCATION_DB_FILE_NAME,
            SXM_LOCATION_DB_FILE_FORMAT,
            SXM_LOCATION_DB_TFILE_SCHEMA_VERSION,
            SXM_TFILE_VERSION,
            SXM_LOCATION_DB_TFILE_ROOT_SIZE,
            (uint)SXM_TFILE_DYNAMIC_NO_OF_BLOCKS,
            SXM_LOCATION_DB_TFILE_BLOCK_SIZE, &rc);
        if (rc != SXM_E_OK) {
            non_fatal("failed to create DB file (rc=%d)", rc);
            break;
        }

        rc = sxm_tfile_start(pInputData->dbfile);
        if (rc != SXM_E_OK) {
            non_fatal("failed to start transaction (rc=%d)", rc);
            break;
        }

        // Get pointer to the root block
        pInputData->pRoot = (Table *)sxm_tfile_root(pInputData->dbfile);
        if (pInputData->pRoot == NULL) {
            non_fatal("failed to get root");
            rc = SXM_E_ERROR;
            break;
        }

        // Create the stream for update
        rc = sxm_tstream_create(pInputData->dbfile, &pInputData->stream);
        if (rc != SXM_E_OK) {
            non_fatal("failed to create stream (rc=%d)", rc);
            break;
        }

        // Create buffer for BSA Indexes to linears
        pInputData->blockSize = sxm_tfile_bsize(pInputData->dbfile);
        pInputData->bsaBufferSize = NUM_ALIGNED_BLOCKS(
            SXM_APOGEE_MAX_TABLE_BSA_COUNT * sizeof(BSA),
            pInputData->blockSize) * pInputData->blockSize;
        pInputData->pBsaBuffer = (BSA*)sxe_malloc(pInputData->bsaBufferSize);
        if (pInputData->pBsaBuffer == NULL) {
            non_fatal("Failed to allocate %d bytes", pInputData->bsaBufferSize);
            rc = SXM_E_NOMEM;
            break;
        }
    }
    }

    if (rc != SXM_E_OK) {
        if (pInputData->pBsaBuffer != NULL) {
            sxe_free(pInputData->pBsaBuffer);
            pInputData->pBsaBuffer = NULL;
        }

        if (pInputData->stream != NULL) {
            (void)sxm_tstream_destroy(pInputData->stream);
            pInputData->stream = NULL;
        }

        if (pInputData->dbfile != NULL) {
            (void)sxm_tfile_cancel(pInputData->dbfile);
            sxm_tfile_close(pInputData->dbfile);
            pInputData->dbfile = NULL;
        }
    }

    return rc;
}

/*******************************************************************************
 *
 * Function    :  saveLinears
 *
 * Description :  Allocates new blocks for linears and stores to database.
 *
 * Parameters  : 
 *          1  :  pStream = pointer to the data stream
 *          2  :  pBSA = pointer to the BSA record for the linears
 *          3  :  isRamp = Set true for Linear, false for ramp data.
 *
 * Returns     :  SXM_E_OK on success
 *
 * Note        :  
 *
 ********************************************************************************/
static int saveLinears(SXMTFileStream *pStream, BSA *pBSA, BOOL isRamp) {
    int rc = SXM_E_OK;
    const size_t datasize = sxm_tstream_tell(pStream);
    uint start, blockc;

    if (pBSA && datasize) {
        rc = sxm_tstream_commit(pStream, &start, &blockc);
        if (rc == SXM_E_OK) {
            if (isRamp)
                pBSA->ramp = (ushort)start;
            else
                pBSA->linear = (ushort)start;

            rc = sxm_tstream_clean(pStream);
        }
    }
    return rc;
}

/*******************************************************************************
 *
 * Function    :  saveBsa
 *
 * Description :  Allocates new blocks for BSAs and stores to database.
 *
 * Parameters  : 
 *          1  :  pDb = pointer to the database
 *          2  :  pTable = pointer to the Table record for tbe BSAs
 *          3  :  pBSA = pointer to the BSA record
 *          4  :  blockCount = number of blocks needed for the data.
 *
 * Returns     :  SXM_E_OK on success
 *
 * Note        :  
 *
 ********************************************************************************/
static int saveBsa(SXMTFile *pDb, Table *pTable, BSA *pBSA, uint blockCount) {
    int rc;
    int ret;
    SXMTFileChain *pChain = NULL;

    rc = sxm_tfile_alloc(pDb, blockCount, &pChain);

    if (rc == SXM_E_OK) {
        pTable->offset = sxm_tfile_chain_start(pChain);
        ret = sxm_tfile_write(pDb, pChain, (uint)-1, pBSA);
        if (ret != (int)blockCount)
        {
            rc = SXM_E_PIPE;
        }
    }
    sxm_tfile_chain_free(pChain);

    return rc;
}

static int process_table_entry(SxmCsvParser *pCsvParser, const Indexes *pIdxs,
    Table *pTable) {
    int rc;

    switch (0) {
    default: {
        /* Read SXLLLON */
        rc = sxm_csv_get_fix_val(pCsvParser, pIdxs->il_llon,
            &pTable->ll_lon);
        if (SXM_E_OK != rc) {
            non_fatal("Failed to get SXLLLON. Line: %u",
                sxm_csv_current_line(pCsvParser));
            break;
        }

        /* Read SXLLLAT */
        rc = sxm_csv_get_fix_val(pCsvParser, pIdxs->il_llat,
            &pTable->ll_lat);
        if (SXM_E_OK != rc) {
            non_fatal("Failed to get SXLLLAT. Line: %u",
                sxm_csv_current_line(pCsvParser));
            break;
        }

        /* Read SXURLON */
        rc = sxm_csv_get_fix_val(pCsvParser, pIdxs->il_ulon,
            &pTable->ur_lon);
        if (SXM_E_OK != rc) {
            non_fatal("Failed to get SXURLON. Line: %u",
                sxm_csv_current_line(pCsvParser));
            break;
        }

        /* Read SXURLAT */
        rc = sxm_csv_get_fix_val(pCsvParser, pIdxs->il_ulat,
            &pTable->ur_lat);
        if (SXM_E_OK != rc) {
            non_fatal("Failed to get SXURLAT. Line: %u",
                sxm_csv_current_line(pCsvParser));
            break;
        }
    }
    }

    return rc;
}

static int process_bsa_entry(SxmCsvParser *pCsvParser, const Indexes *pIdxs,
    BSA *pBsaEntry) {
    int rc;

    switch (0) {
    default: {

        /* Read SXLLLON */
        rc = sxm_csv_get_fix_val(pCsvParser, pIdxs->il_llon,
            &pBsaEntry->ll_lon);
        if (SXM_E_OK != rc) {
            non_fatal("Failed to get SXLLLON. Line: %u",
                sxm_csv_current_line(pCsvParser));
            break;
        }

        /* Read SXLLLAT */
        rc = sxm_csv_get_fix_val(pCsvParser, pIdxs->il_llat,
            &pBsaEntry->ll_lat);
        if (SXM_E_OK != rc) {
            non_fatal("Failed to get SXLLLAT. Line: %u",
                sxm_csv_current_line(pCsvParser));
            break;
        }

        /* Read SXURLON */
        rc = sxm_csv_get_fix_val(pCsvParser, pIdxs->il_ulon,
            &pBsaEntry->ur_lon);
        if (SXM_E_OK != rc) {
            non_fatal("Failed to get SXURLON. Line: %u",
                sxm_csv_current_line(pCsvParser));
            break;
        }

        /* Read SXURLAT */
        rc = sxm_csv_get_fix_val(pCsvParser, pIdxs->il_ulat,
            &pBsaEntry->ur_lat);
        if (SXM_E_OK != rc) {
            non_fatal("Failed to get SXURLAT. Line: %u",
                sxm_csv_current_line(pCsvParser));
            break;
        }

        pBsaEntry->lcount = 0;
        pBsaEntry->rcount = 0;
    }
    }

    return rc;
}

static int get_indexes(SxmCsvParser *pCsvParser, Indexes *pIdxs) {
    int rc = SXM_E_INVAL;

    switch (0) {
    default: {
        int tmp;

        tmp = sxm_csv_index(pCsvParser, "SXMTYPE");
        if (tmp < 0) {
            fatal("Column 'SXMTYPE' is not found");
            break;
        }
        pIdxs->il_type = (uint)tmp;

        tmp = sxm_csv_index(pCsvParser, "SXMINDEX");
        if (tmp < 0) {
            fatal("Column 'SXMINDEX' is not found");
            break;
        }
        pIdxs->il_idx = (uint)tmp;

        tmp = sxm_csv_index(pCsvParser, "SXLLLON");
        if (tmp < 0) {
            fatal("Column 'SXLLLON' is not found");
            break;
        }
        pIdxs->il_llon = (uint)tmp;

        tmp = sxm_csv_index(pCsvParser, "SXLLLAT");
        if (tmp < 0) {
            fatal("Column 'SXLLLAT' is not found");
            break;
        }
        pIdxs->il_llat = (uint)tmp;

        tmp = sxm_csv_index(pCsvParser, "SXURLON");
        if (tmp < 0) {
            fatal("Column 'SXURLON' is not found");
            break;
        }
        pIdxs->il_ulon = (uint)tmp;

        tmp = sxm_csv_index(pCsvParser, "SXURLAT");
        if (tmp < 0) {
            fatal("Column 'SXURLAT' is not found");
            break;
        }
        pIdxs->il_ulat = (uint)tmp;

        tmp = sxm_csv_index(pCsvParser, "SXSTART");
        if (tmp < 0) {
            fatal("Column 'SXSTART' is not found");
            break;
        }
        pIdxs->il_tmc1 = (uint)tmp;

        tmp = sxm_csv_index(pCsvParser, "SXEND");
        if (tmp < 0) {
            fatal("Column 'SXEND' is not found");
            break;
        }
        pIdxs->il_tmc2 = (uint)tmp;

        tmp = sxm_csv_index(pCsvParser, "SXCOUNT");
        if (tmp < 0) {
            fatal("Column 'SXCOUNT' is not found");
            break;
        }
        pIdxs->il_cnt = (uint)tmp;

        rc = SXM_E_OK;
    }
    }

    return rc;
}

static int process_lr_entry(SxmCsvParser *pCsvParser, const Indexes *pIdxs,
    Linear *pLinear, uint maxcount) {
    int rc;

    switch (0) {
    default: {
        const char *tmc1, *tmc2, *cnt;
        memset(pLinear, 0, sizeof(*pLinear));

        /* Read SXLLLON */
        rc = sxm_csv_get_fix_val(pCsvParser, pIdxs->il_llon,
            &pLinear->ll_lon);
        if (SXM_E_OK != rc) {
            non_fatal("Failed to get SXLLLON. Line: %u",
                sxm_csv_current_line(pCsvParser));
            break;
        }

        /* Read SXLLLAT */
        rc = sxm_csv_get_fix_val(pCsvParser, pIdxs->il_llat,
            &pLinear->ll_lat);
        if (SXM_E_OK != rc) {
            non_fatal("Failed to get SXLLLAT. Line: %u",
                sxm_csv_current_line(pCsvParser));
            break;
        }

        /* Read SXURLON */
        rc = sxm_csv_get_fix_val(pCsvParser, pIdxs->il_ulon,
            &pLinear->ur_lon);
        if (SXM_E_OK != rc) {
            non_fatal("Failed to get SXURLON. Line: %u",
                sxm_csv_current_line(pCsvParser));
            break;
        }

        /* Read SXURLAT */
        rc = sxm_csv_get_fix_val(pCsvParser, pIdxs->il_ulat,
            &pLinear->ur_lat);
        if (SXM_E_OK != rc) {
            non_fatal("Failed to get SXURLAT. Line: %u",
                sxm_csv_current_line(pCsvParser));
            break;
        }

        /* Read SXSTART */
        tmc1 = sxm_csv_str_val(pCsvParser, pIdxs->il_tmc1, &rc);
        if (rc != SXM_E_OK) {
            non_fatal("Failed to get SXSTART. Line: %u",
                sxm_csv_current_line(pCsvParser));
            break;
        }
        else {
            pLinear->tmc1 = (ushort)atoi(tmc1);
        }


        /* Read SXEND */
        tmc2 = sxm_csv_str_val(pCsvParser, pIdxs->il_tmc2, &rc);
        if (rc != SXM_E_OK) {
            non_fatal("Failed to get SXEND. Line: %u",
                sxm_csv_current_line(pCsvParser));
            break;
        }
        else {
            pLinear->tmc2 = (ushort)atoi(tmc2);
        }

        /* Read SXCOUNT */
        cnt = sxm_csv_str_val(pCsvParser, pIdxs->il_cnt, &rc);
        if (rc != SXM_E_OK) {
            non_fatal("Failed to get SXCOUNT. Line: %u",
                sxm_csv_current_line(pCsvParser));
            break;
        }
        else {
            pLinear->count = (ushort)atoi(cnt);
            if (pLinear->count > maxcount) {
                non_fatal("Invalid SXCOUNT %u at line %u, "
                    "ignoring SXSTART::SXEND (%u::%u)\n",
                    pLinear->count, sxm_csv_current_line(pCsvParser),
                    pLinear->tmc1, pLinear->tmc2);
            }

        }
    }
    }

    return rc;
}

static void print_table_stats(Table *pTable, uint bc_bsa, BSA *pBSA) {
    uint i;
    printf("\tNumber of BSAs: %u (%u block(s) at %u)\n",
        pTable->count, bc_bsa, pTable->offset);
    for (i = 0; i < pTable->count; i++) {
        printf("\t\t%6u: Linear %4u @ %4u.   Ramp %4u @ %4u.\n", i,
            pBSA[i].lcount, pBSA[i].linear, pBSA[i].rcount, pBSA[i].ramp);
    }
}

static int doMarket(const SXMFilesInputProps *pProps,
    const SXMFilesInputRecord *pRecord, LocationsSdkFilesInputData *pInputData,
    Table *pTable) {

    int rc = SXM_E_OK;
    SxmCsvParser *pCsvParser = NULL;
    Linear newLinear;
    BSA *pBsaEntry = NULL;
    BOOL parsingRamps = FALSE;

    switch (0) { default: {
        Indexes idxs;
        int bsa = -1;

        rc = create_buffer_psv_parser(pRecord, ',',
            APOGEE_CSV_LINE_SIZE_MAX,
            &pCsvParser);
        if (rc != SXM_E_OK) {
            fatal("Failed to create CSV parser: %s", pProps->pFileName);
            break;
        }

        rc = get_indexes(pCsvParser, &idxs);
        if (rc != SXM_E_OK) {
            fatal("Failed to get columns indexes: %s", pProps->pFileName);
            break;
        }

        // Clear the BSA table
        memset(pInputData->pBsaBuffer, 0, pInputData->bsaBufferSize);
        while (SXM_E_OK == rc) {
            const char *tt;

            rc = sxm_csv_next_line(pCsvParser);
            if (SXM_E_OK != rc) {
                if (SXM_E_NOENT == rc) {
                    rc = SXM_E_OK;
                }
                break;
            }

            tt = sxm_csv_str_val(pCsvParser, idxs.il_type, &rc);
            if (rc != SXM_E_OK) {
                non_fatal("Failed to get SXMTYPE. Line: %u",
                    sxm_csv_current_line(pCsvParser));
                rc = SXM_E_OK;
            }
            else if (*tt == 'T') {
                rc = process_table_entry(pCsvParser, &idxs, pTable);
                if (rc != SXM_E_OK) {
                    fatal("Failed to get table information");
                    break;
                }
            } else if (*tt == 'B') {
                int csv_rc;
                const char *pBsa = sxm_csv_str_val(pCsvParser, idxs.il_idx, &csv_rc);

                if (pBsaEntry != NULL) {
                    // Store previous BSA Linears/Ramps
                    rc = saveLinears(pInputData->stream, pBsaEntry, parsingRamps);
                    if (rc != SXM_E_OK) {
                        fatal("Failed to store BSA %d", bsa);
                        break;
                    }
                    pBsaEntry = NULL;
                }

                if (SXM_E_OK != csv_rc) {
                    non_fatal("Failed to get SXMINDEX. Line: %u",
                        sxm_csv_current_line(pCsvParser));
                    rc = SXM_E_OK;
                }

                bsa = atoi(pBsa);
                if ((bsa > SXM_APOGEE_MAX_TABLE_BSA_COUNT) || (bsa < 0)) {
                    non_fatal("Invalid BSA index (%d). Line: %u", bsa,
                        sxm_csv_current_line(pCsvParser));
                    /* moving to next entry */
                }
                else {
                    // Setup new BSA
                    pBsaEntry = &pInputData->pBsaBuffer[bsa];
                    rc = process_bsa_entry(pCsvParser, &idxs, pBsaEntry);
                    if (rc != SXM_E_OK) {
                        non_fatal("Failed to process bsa entry %d", bsa);
                        pBsaEntry = NULL;
                    }
                    else {
                        parsingRamps = FALSE;
                        pTable->count++;
                    }
                }
            }
            else if ((*tt == 'L' || *tt == 'l') && (pBsaEntry != NULL)) {
                rc = process_lr_entry(pCsvParser, &idxs, &newLinear, 
                    SXM_LOCATION_DB_TFILE_MAX_LINEAR_SXCOUNT);
                if (SXM_E_OK == rc) {
                    rc = sxm_tstream_write(pInputData->stream, &newLinear, 
                        sizeof(Linear), FALSE);
                    if (SXM_E_OK != rc) {
                        fatal("Cannot store linear information (rc=%d)", rc);
                        break;
                    }

                    pBsaEntry->lcount++;
                }
                else {
                    non_fatal("Failed to read linear information.");
                    rc = SXM_E_OK;
                }
            } else if ((*tt == 'R') && (pBsaEntry != NULL)) {
                if (FALSE == parsingRamps) {
                    // Store previous BSA Linears
                    rc = saveLinears(pInputData->stream, pBsaEntry, parsingRamps);
                    if (SXM_E_OK != rc) {
                        fatal("Cannot store BSA entry (rc=%d)", rc);
                        break;
                    }
                    parsingRamps = TRUE;
                }

                rc = process_lr_entry(pCsvParser, &idxs, &newLinear, SXM_LOCATION_DB_TFILE_MAX_RAMP_SXCOUNT);
                if (SXM_E_OK == rc) {
                    rc = sxm_tstream_write(pInputData->stream, &newLinear, sizeof(Linear), FALSE);
                    if (SXM_E_OK != rc) {
                        fatal("Cannot store ramp information (rc=%d)", rc);
                        break;
                    }

                    pBsaEntry->rcount++;
                }
                else {
                    non_fatal("Failed to read ramp information.");
                    rc = SXM_E_OK;
                }
            }
            else if (*tt == 'C') {
                ;
            }
            else {
                non_fatal("Unknown record type %s", tt);
            }
        }

        if (rc == SXM_E_OK) {
            uint bc_bsa;

            if (pBsaEntry != NULL) {
                // Store previous BSA Linears/Ramps
                rc = saveLinears(pInputData->stream, pBsaEntry, parsingRamps);
                if (rc != SXM_E_OK) {
                    fatal("Cannot store BSA %d linears and ramps information"
                        " (rc=%d)", bsa, rc);
                    break;
                }
            }

            // Write out BSAs blocks
            bc_bsa = (uint)NUM_ALIGNED_BLOCKS(pTable->count * sizeof(BSA), 
                pInputData->blockSize);
            rc = saveBsa(pInputData->dbfile, pTable, pInputData->pBsaBuffer,
                bc_bsa);
            if (rc != SXM_E_OK) {
                fatal("Cannot store BSAs %d blocks (rc=%d)", bc_bsa, rc);
                break;
            }

            print_table_stats(pTable, bc_bsa, pInputData->pBsaBuffer);
        }
    }}

    if (NULL != pCsvParser) {
        sxm_csv_delete(pCsvParser);
    }

    return rc;
}

static int parse_locations_file_name(const char *pFileName, byte *pTableId) {

    const char *pParsingPointer = get_file_name_pointer(pFileName);
    char acBuffer[APOGEE_TABLE_ID_LEN + 1];
    size_t bufLen = 0;

    /* Set pointer to the directory name */
    bufLen = strlen(APOGEE_TABLE_DIR_PREFIX);
    pParsingPointer =
        get_file_name_pointer(pFileName) - 1 - APOGEE_TABLE_ID_LEN - bufLen;

    /* Verify directory name */
    if (0 != strncmp_lowercase(pParsingPointer,
        APOGEE_TABLE_DIR_PREFIX,
        bufLen))
    {
        return SXM_E_INVAL;
    }
    pParsingPointer += bufLen;

    /* Get Table Id from directory name */
    bufLen = 0;
    while (bufLen < APOGEE_TABLE_ID_LEN) {
        if (0 == isdigit(*pParsingPointer)) {
            return SXM_E_INVAL;
        }
        acBuffer[bufLen++] = *(pParsingPointer++);
    }
    acBuffer[APOGEE_TABLE_ID_LEN] = '\0';
    *pTableId = (byte)atoi(acBuffer);

    if (*pTableId > SXM_APOGEE_MAX_TABLE_COUNT) {
        return SXM_E_INVAL;
    }

    pParsingPointer++;

    /* Verify Locations file name */
    bufLen = strlen(APOGEE_LOCATIONS_FILE_PREFIX);
    if (0 != strncmp_lowercase(pParsingPointer,
        APOGEE_LOCATIONS_FILE_PREFIX,
        bufLen))
    {
        return SXM_E_INVAL;
    }
    pParsingPointer += bufLen;

    /* Compare Table Id */
    bufLen = 0;
    while (bufLen < APOGEE_TABLE_ID_LEN) {
        if (0 == isdigit(*pParsingPointer)) {
            return SXM_E_INVAL;
        }
        acBuffer[bufLen++] = *(pParsingPointer++);
    }
    acBuffer[APOGEE_TABLE_ID_LEN] = '\0';

    if (*pTableId != (byte)atoi(acBuffer)) {
        return SXM_E_INVAL;
    }

    return SXM_E_OK;
}

static int locations_files_input_callback(const SXMFilesInputProps *pProps,
    const SXMFilesInputRecord *pRecord, void *pUserData) {

    int rc;
    LocationsSdkFilesInputData *pInputData = (LocationsSdkFilesInputData*)pUserData;
    byte tableId;

    /* Verify file name */
    rc = parse_locations_file_name(pProps->pFileName, &tableId);
    if (SXM_E_OK != rc) {
        non_fatal("Invalid file name: %s", pProps->pFileName);
        /* continue processing of other files */
        rc = SXM_E_OK;
    }
    else {
        printf("Table %u found:\n", tableId);
        rc = doMarket(pProps, pRecord, pInputData, &pInputData->pRoot[tableId]);
    }

    return rc;
}

/*******************************************************************************
*
* Function    :  locations_ver
*
* Description :  Parses input file names to get version
*
* Parameters  :
*          1  :  pSource = pointer to file name with path
*          2  :  isArchive = TRUE if passed file name is archive
*          3  :  pUserData = pointer to version variable
*
* Returns     :  Nothing
*
********************************************************************************/
void locations_ver(const char *pSource, BOOL isArchive, void *pUserData) {

    const char *start;

    /* checking if this is zip file */
    if (TRUE != isArchive) {
        return;
    }

    start = strrchr(pSource, 'v');
    if (NULL != start) {
        ++start;
        if (('\0' != *start) && (isdigit((unsigned char)(*start)) != 0)) {
            *((int*)pUserData) = atoi(start);
        }
    }

    return;
}

/*******************************************************************************
 *
 * Function    :  locations_build
 *
 * Description :  Creates a baseline locations database T-file.
 *
 * Parameters  :
 *          1  :  pFilesInput = Input files iterator (-i parameter).
 *          2  :  version = version number to apply to file (-1 for default).
 *
 * Returns     :  Nothing
 *
 * Note        :  This database is readonly.
 *
 ********************************************************************************/
int locations_build(const SXMFilesInput *pFilesInput, int version) {
    int rc = SXM_E_OK;
    LocationsSdkFilesInputData filesInputData;

    memset(&filesInputData, 0, sizeof(filesInputData));

    switch (0) {
    default: {
        SXMTFileStat st;

        rc = prepare_db_file(&filesInputData);
        if (rc != SXM_E_OK) {
            fatal("Failed to prepare db file (rc=%d)", rc);
            break;
        }

        printf("\n---------- Locations processing started --------------\n");

        /* Enumarate csv files */
        rc = sxm_files_input_enumerate(pFilesInput, "csv",
            locations_files_input_callback,
            &filesInputData);
        if (SXM_E_OK != rc) {
            non_fatal("Input files enumeration failed (rc=%d)", rc);
        }

        if (-1 == version) {
            /* version value is not forced */
        }

        // Set the DB version
        rc = sxm_tfile_update_version(filesInputData.dbfile, (uint)version);
        if (rc != SXM_E_OK) {
            fatal(": failed to update version (rc=%d)", rc);
            break;
        }

        rc = sxm_tfile_commit(filesInputData.dbfile);
        if (rc != SXM_E_OK) {
            fatal(": failed to commit file (rc=%d)", rc);
            break;
        }
        rc = sxm_tfile_stat(filesInputData.dbfile, &st);
        if (rc != SXM_E_OK) {
            non_fatal(": failed to gather stat from the t-file (rc=%d)", rc);
            break;
        }

        printf("Versions\n"
            "  Baseline Overall: %u\n"
            "  Schema          : %u\n"
            "Statistics: %u block(s) by %u byte(s), used %u block(s) [%u%%]\n",
            st.dversion, st.dschema,
            st.fsize, st.ubsize, st.usize, (st.usize * 100) / st.fsize);
    }
    }

    if (filesInputData.stream != NULL) {
        sxm_tstream_destroy(filesInputData.stream);
    }

    if (filesInputData.dbfile != NULL) {
        sxm_tfile_close(filesInputData.dbfile);
    }

    if (filesInputData.pBsaBuffer != NULL) {
        sxe_free(filesInputData.pBsaBuffer);
    }

    printf("------ Locations processing completed --------\n");

    return rc;
}

#ifdef SDKFILES_STANDALONE_BUILD
static void tql_tableConvert(ushort id, Table *src, TableRow *row)
{
    row->id = id;
    row->count = src->count;
    row->offset = src->offset;
    row->ll_lat = fix2float(src->ll_lat);
    row->ll_lon = fix2float(src->ll_lon);
    row->ur_lat = fix2float(src->ur_lat);
    row->ur_lon = fix2float(src->ur_lon);
}

static void tql_bsaConvert(ushort tableId, ushort id, BSA *src, BSARow *row)
{
    row->id = id;
    row->tableId = tableId;
    row->lcount = src->lcount;
    row->linear = src->linear;
    row->rcount = src->rcount;
    row->ramp = src->ramp;
    row->ll_lat = fix2float(src->ll_lat);
    row->ll_lon = fix2float(src->ll_lon);
    row->ur_lat = fix2float(src->ur_lat);
    row->ur_lon = fix2float(src->ur_lon);
}

static void tql_linearConvert(byte ltype, ushort tableId, ushort bsaId, ushort id, Linear *src, LinearRow *row)
{
    row->ltype = (char*)&(g_lTypes[ltype])[0];
    row->id = id;
    row->tableId = tableId;
    row->bsaId = bsaId;
    row->tmc1 = src->tmc1;
    row->tmc2 = src->tmc2;
    row->count = src->count;
    row->type = src->type;
    row->ll_lat = fix2float(src->ll_lat);
    row->ll_lon = fix2float(src->ll_lon);
    row->ur_lat = fix2float(src->ur_lat);
    row->ur_lon = fix2float(src->ur_lon);
}

/*******************************************************************************
 *
 * Function    :  locations_check_tfile
 *
 * Description :  Processes a locations service T-file and displays its contents.
 *
 * Parameters  : 
 *          1  :  in = filename to check (-t parameter).
 *          2  :  query = tql query
 *
 * Returns     :  Nothing
 *
 * Note        :  
 *
 ********************************************************************************/
void locations_check_tfile(const char *query) {
    SXMTqlStatement tql = NULL;
    SXMTFile *dbfile = NULL;
    SXMTFileStat st;

    switch (0) { default: {
        int rc = SXM_E_OK;
        uint dbschema;
        Table *pTables;
        BSA *pBSA;
        Linear *pLinear;
        ushort i, j, k;
        ushort qtype = QTYPE_NONE;
        uint blockSize;

        if (query) {
            if (SXM_E_OK != sxm_tql_statement(&tql, SXMTQL_DEFAULT_INTERFACE,
                                              query, SXMTQL_SERVICE_TABLES)) {
                non_fatal("failed to create statement for TQL query %s", query);
                break;
            }
            else if (!tql) {
                break;
            }

            if (SXM_E_OK == sxm_tql_query_table(tql, "tables")) {
                qtype = QTYPE_TABLES;
            }
            else if (SXM_E_OK == sxm_tql_query_table(tql, "bsas")) {
                qtype = QTYPE_BSAS;
            }
            else if (SXM_E_OK == sxm_tql_query_table(tql, "linears")) {
                qtype = QTYPE_LINEARS;
            }
            else {
                break;
            }
        }

        dbfile = sxm_tfile_open(SXM_LOCATION_DB_TFILE_TYPE,
                                APOGEE_SERVICE_NAME,
                                SXM_LOCATION_DB_FILE_NAME, "rb",
                                SXM_LOCATION_DB_FILE_FORMAT,
                                &dbschema, &rc);
        if (rc != SXM_E_OK) {
            non_fatal("Failed to open DB file (rc=%d)", rc);
            break;
        }

        sxm_tfile_stat(dbfile, &st);
        printf("\nVersions\n"
               "  Baseline Overall: %u\n"
               "  Schema          : %u\n"
               "Statistics: %u block(s) by %u byte(s), used %u block(s) [%u%%]\n\n\n",
            st.dversion, st.dschema,
            st.fsize, st.ubsize, st.usize,
            (st.usize * 100) / st.fsize);

        if (dbschema != SXM_LOCATION_DB_TFILE_SCHEMA_VERSION) {
            non_fatal("Unsupported schema version %u\n", dbschema);
            break;
        }

        pTables = (Table *)sxm_tfile_root(dbfile);
        blockSize = sxm_tfile_bsize(dbfile);
        if (!pTables) {
            non_fatal("Failed to access memory for root!");
            break;
        }

        //    dump_alloc_map();
        for  (i = 0; i < 36; i++) {
            ushort bsac = pTables[i].count;
            uint bc = (uint)NUM_ALIGNED_BLOCKS(bsac * sizeof(BSA), blockSize);

            if  (bsac == 0)
                continue;

            if (!tql) {
                printf("Table %2u: (%u, %u) [%f %f %f %f]\n", i, pTables[i].offset, bc,
                    fix2float(pTables[i].ll_lon), fix2float(pTables[i].ll_lat), 
                    fix2float(pTables[i].ur_lon), fix2float(pTables[i].ur_lat));
            }
            else {
                if (qtype == QTYPE_TABLES) {
                    TableRow row;

                    tql_tableConvert(i, &pTables[i], &row);
                    rc = sxm_tql_evaluate(tql, &row);
                    if (rc != SXM_E_OK) {
                        if (rc == SXM_E_NOENT) {
                            rc = SXM_E_OK;
                        }
                        else {
                            non_fatal("Failed to evaluate TQL statement (%d)", rc);
                            break;
                        }
                    }
                    continue;
                }
            }

            pBSA = (BSA*)sxm_tfile_alloc_and_read(dbfile, pTables[i].offset, bc, NULL, &rc);
            if (!pBSA) {
                non_fatal(": failed to load %u block(s) from %u (rc=%d)\n",
                   bc, pTables[i].offset, rc);
                break;
            }
            else {
                for  (j = 0; j < bsac; j++) {
                    if (!tql) {
                        printf("    BSA %2u:  [%f %f %f %f] %u:%u %u:%u\n", j,
                            fix2float(pBSA[j].ll_lon), fix2float(pBSA[j].ll_lat), 
                            fix2float(pBSA[j].ur_lon), fix2float(pBSA[j].ur_lat),
                            pBSA[j].linear, pBSA[j].lcount, pBSA[j].ramp, pBSA[j].rcount);
                    }
                    else {
                        if (qtype == QTYPE_BSAS) {
                            BSARow row;

                            tql_bsaConvert(i, j, &pBSA[j], &row);
                            rc = sxm_tql_evaluate(tql, &row);
                            if (rc != SXM_E_OK) {
                                if (rc == SXM_E_NOENT) {
                                    rc = SXM_E_OK;
                                }
                                else {
                                    non_fatal("Failed to evaluate TQL statement (%d)", rc);
                                    break;
                                }
                            }
                            continue;
                        }
                    }

                    if  (pBSA[j].lcount > 0) {
                        bc = (uint)NUM_ALIGNED_BLOCKS(pBSA[j].lcount * sizeof(Linear), blockSize);

                        pLinear = (Linear*)sxm_tfile_alloc_and_read(dbfile, pBSA[j].linear, bc, NULL, &rc);
                        if (!pLinear) {
                            non_fatal(": failed to load %u block(s) from %u (rc=%d)\n",
                               bc, pBSA[j].linear, rc);
                            break;
                        }
                        else {
                            for  (k = 0; k < pBSA[j].lcount; k++) {
                                if (!tql) {
                                    printf("        Linear %4u: [%f %f %f %f] %u..%u %u:%u\n", k,
                                        fix2float(pLinear[k].ll_lon), fix2float(pLinear[k].ll_lat), 
                                        fix2float(pLinear[k].ur_lon), fix2float(pLinear[k].ur_lat),
                                        pLinear[k].tmc1, pLinear[k].tmc2, pLinear[k].count, pLinear[k].type);
                                }
                                else {
                                    if (qtype == QTYPE_LINEARS) {
                                        LinearRow row;

                                        tql_linearConvert(LTYPE_LINEAR, i, j, k, &pLinear[k], &row);
                                        rc = sxm_tql_evaluate(tql, &row);
                                        if (rc != SXM_E_OK) {
                                            if (rc == SXM_E_NOENT) {
                                                rc = SXM_E_OK;
                                            }
                                            else {
                                                non_fatal("Failed to evaluate TQL statement (%d)", rc);
                                                break;
                                            }
                                        }
                                        continue;
                                    }
                                }
                            }

                            sxe_free(pLinear);
                        }
                    }

                    if  (pBSA[j].rcount > 0) {
                        bc = (uint)NUM_ALIGNED_BLOCKS(pBSA[j].rcount * sizeof(Linear), blockSize);

                        pLinear = (Linear*)sxm_tfile_alloc_and_read(dbfile, pBSA[j].ramp, bc, NULL, &rc);
                        if (!pLinear) {
                            non_fatal(": failed to load %u block(s) from %u (rc=%d)\n",
                               bc, pBSA[j].ramp, rc);
                            break;
                        }
                        else {
                            for  (k = 0; k < pBSA[j].rcount; k++) {
                                if (!tql) {
                                    printf("        Ramp %4u: [%f %f %f %f] %u..%u %u:%u\n", k,
                                        fix2float(pLinear[k].ll_lon), fix2float(pLinear[k].ll_lat), 
                                        fix2float(pLinear[k].ur_lon), fix2float(pLinear[k].ur_lat),
                                        pLinear[k].tmc1, pLinear[k].tmc2, pLinear[k].count, pLinear[k].type);
                                }
                                else {
                                    if (qtype == QTYPE_LINEARS) {
                                        LinearRow row;

                                        tql_linearConvert(LTYPE_RAMP, i, j, k, &pLinear[k], &row);
                                        rc = sxm_tql_evaluate(tql, &row);
                                        if (rc != SXM_E_OK) {
                                            if (rc == SXM_E_NOENT) {
                                                rc = SXM_E_OK;
                                            }
                                            else {
                                                non_fatal("Failed to evaluate TQL statement (%d)", rc);
                                                break;
                                            }
                                        }
                                        continue;
                                    }
                                }
                            }

                            sxe_free(pLinear);
                        }
                    }
                }

                sxe_free(pBSA);      
            }
        }
    }}


    if (dbfile) {
        sxm_tfile_close(dbfile);
    }

    if (tql) {
        sxm_tql_statement_destroy(tql);
    }
}
#endif // SDKFILES_STANDALONE_BUILD
