/************************************************************************
 *                                                                      *
 *            Build and Analise the Grid-based data service             *
 *            ==================================================        *
 *                                                                      *
 *                                 Copyright 2013 Sirius XM Radio, Inc. *
 *                                                 All Rights Reserved. *
 *               Licensed Materials - Property of Sirius XM Radio, Inc. *
 *                                                                      *
 ************************************************************************/
#include <ctype.h>
#include "sdkfiles.h"

/** Helper macro to extract single digit from the string
 * and convert it into 4-bit value
 */
#define GRID_GET_DIGIT(_var1, _var2, _tar) \
    if (isdigit(_var1) && isdigit(_var2)) \
        (_tar) = (byte)((((_var1) - '0') << 4) | ((_var2) - '0')); else break

/** Length of AMEN1 and AMEN2 baseline fields (8 characters) */
#define BASELINE_AMENITY_FIELD_LENGTH   (8)

int grid_check_location(ushort region, fix lat, fix lon) {
    int rc;
    SXMFixMBR mbr;
    // Convert region to MBR
    mbr.ll.lon = float2fix(-168.0) + float2fix(SXM_GRID_REGION_LON_UNPACK(region) << 1);
    mbr.ll.lat = float2fix(12.0) + float2fix(SXM_GRID_REGION_LAT_UNPACK(region) << 1);
    mbr.ur.lon = mbr.ll.lon + float2fix(2.0);
    mbr.ur.lat = mbr.ll.lat + float2fix(2.0);
    // Adjust the region by min possible value for fix to avoid failures on
    // region border due to float-to-fix conversion precision.
    --mbr.ll.lon;
    --mbr.ll.lat;
    ++mbr.ur.lon;
    ++mbr.ur.lat;
    // Do the check
    rc = (lon >= mbr.ll.lon) && (lon <= mbr.ur.lon) &&
         (lat >= mbr.ll.lat) && (lat <= mbr.ur.lat);
    return rc;
}

int grid_saveregion(SXMTFile *dbfile, SXMTFileStream *stream, int region,
                    ushort *ib, uint *index_area) {
    uint start, blockc;
    int rc;
    const size_t datasize = sxm_tstream_tell(stream);

    // Write the index area
    sxm_tstream_write(stream, index_area, (index_area[0] + 1) * sizeof(uint), TRUE);
    rc = sxm_tstream_commit(stream, &start, &blockc);
    if (rc == SXM_E_OK) {
        const uint bsize = sxm_tfile_bsize(dbfile);
        const uint bsize_res = bsize - ((uint)datasize % bsize);
        ib[2 * region] = (ushort)start;
        ib[2 * region + 1] = (ushort)blockc;
        printf("Save region %4d (%3d blocks).  Maxrecord %5d  DataSize %8u  Reserved %8u (%3u%%)\n",
            region, blockc, index_area[0], (uint)datasize, bsize_res,
            (bsize_res * 100) / bsize);
    }
    sxm_tstream_clean(stream);
    return rc;
}

const void* grid_encode_phone(void *buff, size_t s, const char *phone) {
    byte * const sphone = (byte*)buff;
    BOOL isOk = FALSE;

    // Clean up storage
    memset(buff, 0, s);

    switch (0) { default: {
        if (s < SXM_GRID_PHONE_BYTELEN) {
            non_fatal("Encoded phone buffer too small (%u < %u)",
                      s, SXM_GRID_PHONE_BYTELEN);
            break;
        }
        // Check for the empty phone number
        if (phone[0] == '\0') {
            // Keep the buffer untouched since it has all zeros which means
            // "no-phone" for the service logic.
            isOk = TRUE;
            break;
        }
        // Check non-digit symbols in known positions
        if ((phone[0] != '(') || (phone[4] != ')') || (phone[8] != '-')) {
            break;
        }
        GRID_GET_DIGIT(phone[ 1], phone[ 2], sphone[0]);
        GRID_GET_DIGIT(phone[ 3], phone[ 5], sphone[1]);
        GRID_GET_DIGIT(phone[ 6], phone[ 7], sphone[2]);
        GRID_GET_DIGIT(phone[ 9], phone[10], sphone[3]);
        GRID_GET_DIGIT(phone[11], phone[12], sphone[4]);
        isOk = TRUE;
    }}

    if (isOk != TRUE) {
        non_fatal("Unable to parse string '%s' "
                  "as phone number in format: %s",
                  phone, SXM_GRID_PHONE_FORMAT);
        memset(buff, 0, s);
    }

    return buff;
}

const char *grid_decode_phone(char *phone, size_t s, const void *buff) {
    const byte *b = (const byte *)buff;
    const uint area = ((uint)(b[0] >> 4) * 100) + ((b[0] & 0xF) * 10U) + (b[1] >> 4);
    const uint exch = ((b[1] & 0xF) * 100U) + ((uint)(b[2] >> 4) * 10) + (b[2] & 0xF);
    const uint num  = ((uint)(b[3] >> 4) * 1000) + ((uint)(b[3] & 0xF) * 100) + ((uint)(b[4] >> 4) * 10) + (b[4] & 0xF);

    // If all items are zeros consider this as empty phone number
    if (!area && !exch && !num) {
        phone[0] = '\0';
    }
    else {
        snprintf(phone, s, SXM_GRID_PHONE_FORMAT, area, exch, num);
    }
    return phone;
}

ushort grid_encode_amen(const char *amen) {
    ushort ret = 0;
    uint i, j;

    if (strlen(amen) != BASELINE_AMENITY_FIELD_LENGTH) {
        non_fatal("AMEN field (%s) size is invalid (%u != %u)",
            amen, strlen(amen), BASELINE_AMENITY_FIELD_LENGTH);
        return 0;
    }

    for (i = 0, j = 14; i < BASELINE_AMENITY_FIELD_LENGTH; i++, j -= 2) {
        if (amen[i] == 'A') {
            ret = (ushort)(ret | (1 << j));
        }
        else if  (amen[i] == 'B') {
            ret = (ushort)(ret |(2 << j));
        }
        else if  (amen[i] == 'N') {
            ret = (ushort)(ret |(3 << j));
        }
        else if (amen[i] != 'U') {
            non_fatal("Unexpected AMEN value: %c from %s",
                      amen[i], amen);
        }
    }

    return ret;
}

void grid_cfile_sec_stat(const SXMCFileRootBlock *root, int sno) {
    printf(" SEC #%d [CRC %08x, TS %08u, VER %4u, OFF %4u, LEN %4u, CNT %4u, MSK %4x, MAPOFF %4d]\n",
        sno,
        root->sections[sno].crc,
        root->sections[sno].ts,
        (uint)root->sections[sno].version,
        (uint)root->sections[sno].off,
        (uint)root->sections[sno].length,
        (uint)root->sections[sno].count,
        (uint)root->sections[sno].mask,
        (uint)root->sections[sno].heap_map_off);
}