/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 * PRIVATE HEADER
 *
 ******************************************************************************/

  /*********************************/
 /** PREVENT REDUNDANT INCLUSION **/
/*********************************/
#ifndef _SXILL_H_
#define _SXILL_H_

  /**************/
 /** INCLUDES **/
/**************/

#include <stdio.h>
#include <stddef.h>

#include "standard.h"
#include "osal.h"

#include "sti_api.h"
#include "sxill.h"
#ifndef SXILL_USE_LEGACY_RING_BUFFER
#include "sxill_ringbuffer.h"
#endif

// Object Debugging:
// If DEBUG_OBJECT isn't defined
// define it locally
#ifndef DEBUG_OBJECT
#define DEBUG_OBJECT 0
#endif

// If logging hasn't been configured already
// then only enable when we're in debug
#ifndef SMS_LOGGING
#if SMS_DEBUG == 1 && OSAL_LOG == 1
#define SMS_LOGGING 1
#else
#define SMS_LOGGING 0
#endif
#endif

#if SMS_LOGGING == 1
#define SXILL_TXRX_STATS_OUTPUT
#define SXILL_RX_STATS_OUTPUT
#endif

// Include the debug definitions
// header, which depends on how
// DEBUG_OBJECT is defined
#include "sms_debug_definitions.h"
#include "sti_debug_definitions.h"

  /***************/
 /** CONSTANTS **/
/***************/

// Read timeout
#define SXILL_RTO_MSEC (1000)
// No rx-limit is 3 RTO periods, before reporting a problem
#define SXILL_NO_RX_LIMIT   (3)

// minimum macro
#define MIN(X,Y) ((X) < (Y) ? (X) : (Y))

// Payload is only lower 5 bits
#define SXILL_PAYLOAD_TYPE_MASK (0x1F)

// Endianess translation macros between MODULE and HOST
#if defined OSAL_CPU_BIG_ENDIAN

#define SXILL_LEN_MODULE_TO_HOST(x) (x)
#define SXILL_CHECK_VALUE_MODULE_TO_HOST(x) (x)

#elif defined OSAL_CPU_LITTLE_ENDIAN

#define SXILL_LEN_MODULE_TO_HOST(x) SWAPUN16(x)
#define SXILL_CHECK_VALUE_MODULE_TO_HOST(x) SWAPUN16(x)

#else
#error Must define either OSAL_CPU_BIG_ENDIAN or OSAL_CPU_LITTLE_ENDIAN
#endif

#ifdef SXILL_USE_LEGACY_RING_BUFFER
// Ring Buffer Macros

/* Ingress */
#define NUM_INGRESS_BYTES_AVAILABLE \
    tNumAvailableInRingBuffer(psRingBuffer)

#define MOVE_INGRESS(x) \
    vAdvanceRingBuffer(psRingBuffer, offsetof(SXILL_RING_BUFFER_STRUCT, pun8In), (x));

/* Egress */
#define NUM_EGRESS_BYTES_AVAILABLE \
    tNumToEndOfEgressBytes(psRingBuffer, offsetof(SXILL_RING_BUFFER_STRUCT, pun8Out))

#define MOVE_EGRESS(x)  \
    vAdvanceRingBuffer(psRingBuffer, offsetof(SXILL_RING_BUFFER_STRUCT, pun8Out), (x));

/* In-Process */
#define NUM_IN_PROCESS_BYTES_AVAILABLE \
    tNumToEndOfEgressBytes(psRingBuffer, offsetof(SXILL_RING_BUFFER_STRUCT, pun8InProcess))

#define MOVE_IN_PROCESS(x)  \
    vAdvanceRingBuffer(psRingBuffer, offsetof(SXILL_RING_BUFFER_STRUCT, pun8InProcess), (x));

/* Reset */
#define RESET_RING_BUFFER(psRingBuffer) \
    psRingBuffer->pun8InProcess = psRingBuffer->pun8Out
#endif

// Framing Codes
//
// See SX-9845-0098, Section 2.1
#define SXILL_SYNC_BYTE0  (0xDE)
#define SXILL_SYNC_BYTE1  (0xC6)
#ifndef SXILL_USE_LEGACY_RING_BUFFER
#define SXILL_SYNC_BYTE_NONE  (0)
#define SXILL_SYNC_BYTE0_MASK (SXILL_SYNC_BYTE0 << 8)
#define SXILL_SYNC_BYTE1_MASK (SXILL_SYNC_BYTE1)
#endif

// Pre-computed check value for the 2 SYNC bytes.
// Why would we keep computing this?
#define SXILL_INITIAL_CHECK_VALUE   ((SXILL_CHECK_VALUE)(0x85A5))

// SXILL Frame Elements and Offset from beginning of the SXILL frame
#define SXILL_SEQ_OFFSET    ( 0 )
#define SXILL_PAYLOAD_TYPE_OFFSET   ( SXILL_SEQ_OFFSET + sizeof(SXILL_SEQ) )
#define SXILL_LEN_OFFSET    ( SXILL_PAYLOAD_TYPE_OFFSET + sizeof(SXILL_PAYLOAD_TYPE) )
#define SXILL_LEN_BYTE0		( SXILL_LEN_OFFSET )
#define SXILL_LEN_BYTE1		( SXILL_LEN_OFFSET + 1 )
#define SXILL_SUM_OFFSET(x) ( SXILL_LEN_OFFSET + sizeof(SXILL_LEN) + (x) )

// SXILL Maximum Payload Length (See SX-9845-0098, Section 2.1.5)
// also be sure to check the table SXILL_PAYLOAD_TABLE_STRUCT below.
#define SXILL_MAX_PAYLOAD_LEN (1400)

// SXILL Frame Macros
#define SXILL_FRAME_HDR_LEN \
    (sizeof(SXILL_SEQ) + sizeof(SXILL_PAYLOAD_TYPE) + sizeof(SXILL_LEN))
#define SXILL_FRAME_CHECK_VALUE_LEN(x) \
    (sizeof(SXILL_SYNC) + SXILL_FRAME_HDR_LEN + (x))
#define SXILL_FRAME_LEN(x) \
    (SXILL_FRAME_HDR_LEN + (x) + sizeof(SXILL_CHECK_VALUE))
#define SXILL_FRAME_LEN_MAX \
    (SXILL_FRAME_LEN(SXILL_MAX_PAYLOAD_LEN))

// TxRx Event Queue Size
// The is the maximum number of SXiLL events (tx and rx)
// which can be queued for both Rx and Tx simultaneously.
// Bosch ID#20: queue size increased from 32 to 128
#define SXILL_TXRX_EVENTS (128)

/*

 *****************************************************************************
 SXiLL Tx and Rx Buffer Specification

*/

// Tx Payload Buffer (used by CMD and CFM).

// This is the average size of a tx payload we want to
// be able to support.
#define SXILL_TX_PAYLOAD_UNIT_SIZE (32)

// This is the number of simultaneous tx units we can support.
// We generally don't need too many separate tx payload units
// because we send only one command at a time.
// Keep in mind this is the block pool where SXi CMD requests
// come from, as well as SXi CFM's are build. CFMs are not
// that critical since we allocate, populate, send and free
// the CFM payload all in one context. When the IND is received.
// CFMs are also very small.
#define SXILL_TX_PAYLOAD_UNITS (SXILL_TXRX_EVENTS / 2)

// Rx Payload Buffer (used by RESP and IND)

// Rx Frame buffer services ALL connections to a single device
// Choose a block size of one fourth of a max frame.
#define SXILL_RX_FRAME_RATIO     (4)
#define SXILL_RX_FRAME_COMPUTED_BLOCK_SIZE \
    ((SXILL_FRAME_LEN_MAX+1) / SXILL_RX_FRAME_RATIO + 1)

// Chose a number of blocks suitable to hold the maximum number
// of frames we might queue. We know we cannot queue more payloads
// than the number of txrx events available. But incomming IND
// messages can be quite large. This effectively allocates the
// maximum SXiLL frame size from the maximum number of events we may
// have available. Since this is the case, it makes NO sense
// to make the num of blocks any bigger than this.
#define SXILL_RX_FRAME_NUM_BLOCKS (SXILL_TXRX_EVENTS * SXILL_RX_FRAME_RATIO)

/*
 *****************************************************************************
*/

#ifdef SXILL_RX_STATS_OUTPUT

#define SXILL_RX_STATS_LOG_NAME "SXiLL:Rx"
#define SXILL_RX_STATS_LOG_SIZE ((256 * 1024) - 1)  // 256k
#define SXILL_RX_STATS_POLLRATE_IN_SECONDS 1
#define SXILL_RX_STATS_HEADER \
    "Day\tMonth\tDate\tTime\tYear\tFrames\tCantPost\tNoMoreBuffers\tInvalidLength\tCkSumFailures" \
    "\tBufOverflow\tIgnoredFrames\tOutOfSync\tKeepAlives\tErrors\tReadExceedsReuqest" \
    "\tNumUnprocessedBytesIsNotZero\tWroteNothingToBuffer" \
    "\tLLBytes\tPayloadBytes\tOverflow" \
    "\tun32IngressOverflow\tun32EgressUnderflow\tun32InProcessUnderflow\tun32UnprocessedOverflow\tun32IllegalAdvancement" \
    "\n"

#define SXILL_RX_STATS_ROW "%s\t%u\t%u\t%u\t%u\t%u\t%u\t%u\t%u\t%u\t%u\t%u\t%u\t%u\t%u\t%u\t%u\t%u\t%u\t%u\t%u\t%u\n"

#define SXILL_RX_REQ_LOG_NAME "SXiLL:ReadReq"
#define SXILL_RX_REQ_LOG_SIZE ((256 * 1024) - 1)  // 256k
#define SXILL_RX_REQ_HEADER \
    "BytesRequested\tBytesRead\n"

#define SXILL_RX_REQ_ROW "%u\t%u\n"

#endif

#ifdef SXILL_TXRX_STATS_OUTPUT

#define SXILL_TXRX_STATS_LOG_NAME "SXiLL:TxRx"
#define SXILL_TXRX_STATS_LOG_SIZE ((256 * 1024) - 1)  // 256k
#define SXILL_TXRX_STATS_POLLRATE_IN_SECONDS 1
#define SXILL_TXRX_STATS_HEADER \
    "Day\tMonth\tDate\tTime\tYear\tun32MsgFrames\tun32DroppedFrames\tun32UnknownEvent\tun32MsgBytes" \
    "\tun32MsgFrames\tun32DeviceErrors\tun32SendErrors\tun32Nothing" \
    "\tun32LinkLayer\tun32Payload\tun32Overflow" \
    "\tun32Errors\n"

#define SXILL_TXRX_STATS_ROW "%s\t%u\t%u\t%u\t%u\t%u\t%u\t%u\t%u\t%u\t%u\t%u\t%u\n"

#endif

  /**************/
 /** TYPEDEFS **/
/**************/

/*

 SXI LL Field Types

 Good idea to never use native types in protocols as they may change!
 So instead we define our own types for each of the fields.

 SX-9845-0098, Section 2.1 currently defines each of these fields
 */

// ********************* //
typedef UN16 SXILL_SYNC; // 16-bits
typedef UN8 SXILL_SEQ; // 8-bits
//typedef UN8 SXILL_PAYLOAD_TYPE; // 8-bits
typedef UN16 SXILL_LEN; // 16-bits
typedef UN16 SXILL_CHECK_VALUE; // 16-bits

// ********************* //

// Verify that the above SXILL_CHECK_VALUE is precisely 2 bytes
// Otherwise the check value function wont work.
// If they are not, the you will probably get a compiler error which says...
// "size of array 'msg' is negative", which means you better know what your
// doing. YOU MUST FIX THIS IF IT IS NOT!
COMPILE_TIME_ASSERT(sizeof(SXILL_CHECK_VALUE) == sizeof(UN16),
    SXILL_CHECK_VALUE_must_be_2_bytes);

/*
 Other SXI LL Field Types
 */
typedef UN16 SXILL_DSI; // 16-bits

/*
 * SXiLL Frame Header
 */
__PACKED__
struct sxill_frame_hdr_struct
{
    union
    {
        struct
        {
            SXILL_SEQ tSequenceNumber;
            SXILL_PAYLOAD_TYPE tPayloadType;
            SXILL_LEN tNumPayloadBytes;
        } sHeader;

        struct
        {
            UN8 aun8Bytes[4];
        } sData;

    } uFormat;

};
__UNPACKED__

typedef struct sxill_frame_hdr_struct SXILL_FRAME_HDR_STRUCT;

// Verify that the above SXILL_FRAME_HDR_STRUCT is precisely 4 bytes
// this is important to verify that the compiler is in fact packing
// this structure properly and the MACROs are working as intended.
// If they are not, the you will probably get a compiler error which says...
// "size of array 'msg' is negative", which means this structure is
// not packed by the compiler properly. YOU MUST FIX THIS IF IT IS NOT!
COMPILE_TIME_ASSERT(sizeof(SXILL_FRAME_HDR_STRUCT) == 4,
    SXILL_FRAME_HDR_STRUCT_must_be_4_bytes);

/*
 * SXiLL Check Value
 */
__PACKED__
struct sxill_check_value_struct
{
    union
    {
        struct
        {
            SXILL_CHECK_VALUE tCheckValue;
        } sValue;

        struct
        {
            UN8 aun8Bytes[sizeof(SXILL_CHECK_VALUE)];
        } sData;

    } uFormat;

};
__UNPACKED__

typedef struct sxill_check_value_struct SXILL_CHECK_VALUE_STRUCT;

// Verify that the above SXILL_CHECK_VALUE_STRUCT is precisely 2 bytes
// this is important to verify that the compiler is in fact packing
// this structure properly and the MACROs are working as intended.
// If they are not, the you will probably get a compiler error which says...
// "size of array 'msg' is negative", which means this structure is
// not packed by the compiler properly. YOU MUST FIX THIS IF IT IS NOT!
COMPILE_TIME_ASSERT(sizeof(SXILL_CHECK_VALUE_STRUCT) == 2,
    SXILL_CHECK_VALUE_STRUCT_must_be_2_bytes);

/*
 SXILL TxRx (Transceiver Handler) Structures
 */

// SXILL Byte Level Statistics
typedef struct sxill_byte_statistics_struct
{
    UN32 un32LinkLayer;
    UN32 un32Payload;
    UN32 un32Overflow;

} SXILL_BYTE_STATISTICS_STRUCT;

#ifdef SXILL_TXRX_STATS_OUTPUT

// SXILL TxRx Statistics
typedef struct sxill_txrx_statistics_struct
{
    // Receiver related statistics
    struct
    {
        UN32 un32MsgBytes; // Number of MSG bytes received
        UN32 un32MsgFrames; // Number of MSG frames received
        UN32 un32DroppedFrames; // Number of frames dropped due to no header or connection
        UN32 un32UnknownEvent; // Number of events received which were unknown
    } sRx;

    // Transmitter related statistics
    struct
    {
        UN32 un32MsgFrames; // Number of MSG frames sent
        UN32 un32Nothing; // Attempt to transmit nothing
        UN32 un32UnableToFree;
        UN32 un32SendErrors;
        UN32 un32NoFrame;
        UN32 un32DeviceErrors;
        UN32 un32TxHandlerErrors;

        // Byte-level TX statistics
        SXILL_BYTE_STATISTICS_STRUCT sBytes;
    } sTx;

    // General Errors (bad!)
    UN32 un32Errors;

} SXILL_TXRX_STATISTICS_STRUCT;

#endif

/*
 * This is a definition of a simple ring-buffer
 * The slight difference here to more traditional ring buffers
 * is that this one has two out pointers. One to track
 * the out position (consumed) and another to track the
 * in-progress position. This helps us efficiently "re-wind"
 * in case of re-sync.
 */
typedef struct sxill_ring_buffer_struct
{
#ifndef SXILL_USE_LEGACY_RING_BUFFER
    SXILL_RING_BUFFER_DATA_STRUCT sMaster;
    SXILL_RING_BUFFER_DATA_STRUCT sProgress;

#else
    // Ring buffer bounding pointers
    UN8 const *pun8Start;
    UN8 const *pun8End;

    // Handy indexing pointers
    UN8 *pun8In;
    UN8 const *pun8Out;
    UN8 const *pun8InProcess;

    // Number of bytes in ring buffer
    size_t tCount;

    // Number of bytes remaining in ring buffer
    size_t tRemaining;

    // Number of bytes left unprocessed
    size_t tUnprocessed;

#endif

    // Ring buffer space
    UN8 aun8Data[SXILL_FRAME_LEN_MAX * 4]; // Sans SYNC, 4 frames

#ifdef SXILL_RX_STATS_OUTPUT

    // Ring buffer errors
    UN32 un32IngressOverflow;
    UN32 un32EgressUnderflow;
    UN32 un32InProcessUnderflow;
    UN32 un32UnprocessedOverflow;
    UN32 un32IllegalAdvancement;

#endif

} SXILL_RING_BUFFER_STRUCT;

// SXILL TxRx Control Structure
// Used to store variables and handles which represent a specific installation
// of the SXILL protocol, but not specific to any one connection. Thus there
// is only one instance of this structure per installation of the SXILL protocol
// mapped to a single physical device.
typedef struct sxill_txrx_control_struct
{
    // I/O Device (TxRx)
    FILE *psDevice;

    // Frame buffer. We need space for a maximum of SYNC plus frame
    UN8 aun8TxFrameBuffer[sizeof(SXILL_SYNC) + SXILL_FRAME_LEN_MAX];
    size_t tTxFrameBufferLen;

    // Last Tx Sequence Number
    SXILL_SEQ tTxSeq;

    // The TxRx task shutsdown once the Rx task is shutdown
    // so since we will receieve things from the Rx block pool
    // which may still be in our queue, we need to remove
    // the rx tasks resources when we shutdown.
    struct sxill_rx_frame_control_struct const *psRxCtrl;

#ifdef SXILL_TXRX_STATS_OUTPUT

    // TxRx Statistics
    SXILL_TXRX_STATISTICS_STRUCT sStatistics;
    OSAL_OBJECT_HDL hLog;
    UN32 un32LastWrite;

#endif

} SXILL_TXRX_CONTROL_STRUCT;

/*
 SXILL Rx (Receive Handler) Structures
 */

#ifdef SXILL_RX_STATS_OUTPUT

typedef enum sxill_ring_buffer_reason_code_enum
{
    SXILL_RING_BUFFER_UNPROCESSED_NOT_ZERO,
    SXILL_RING_BUFFER_LINEAR_SEGMENT_IS_ZERO,
    SXILL_RING_BUFFER_INGRESS_OVERFLOW,
    SXILL_RING_BUFFER_EGRESS_UNDERFLOW,
    SXILL_RING_BUFFER_INPROCESS_UNDERFLOW,
    SXILL_RING_BUFFER_UNPROCESSED_OVERFLOW,
    SXILL_RING_BUFFER_ILLEGAL_ADVANCEMENT

} SXILL_RING_BUFFER_REASON_CODE_ENUM;

// I-Loop stages
typedef enum sxill_iloop_stage_enum
{
    SXILL_ILOOP_STAGE_OK,
    SXILL_ILOOP_DETECTED_AND_AVERTED,
    SXILL_ILOOP_MESSAGES_PROCESSED_OK

} SXILL_ILOOP_STAGE_ENUM;

// SXILL Rx Statistics
typedef struct sxill_rx_statistics_struct
{
    UN32 un32MsgFrames; // Number of received MSG frames

    UN32 un32CannotPostEvent;
    UN32 un32NoMoreBuffers;
    UN32 un32InvalidLength;
    UN32 un32ChecksumFailures;
    UN32 un32BufferOverflow;
    UN32 un32IgnoredFrames;
    UN32 un32OutOfSync;
    UN32 un32KeepAlives;
    UN32 un32Errors;
    UN32 un32ReadExceedsRequest; // Number of bytes read from driver
                                // exceeds the requested amount
    UN32 un32NumUnprocessedBytesIsNotZero;
    UN32 un32WroteNothingToBuffer;
    SXILL_ILOOP_STAGE_ENUM eILoopStage;
    UN32 un32ILoopAversions;

    // Byte-level RX statistics
    SXILL_BYTE_STATISTICS_STRUCT sRxBytes;

} SXILL_RX_STATISTICS_STRUCT;

#endif

/*
 * Defines set of sync state machine states
 */
typedef enum sxill_sync_state_enum
{
#ifdef SXILL_USE_LEGACY_RING_BUFFER
    SXILL_SYNC_STATE_SEARCHING,
    SXILL_SYNC_STATE_DETECTED
#else
    SXILL_SYNC_STATE_RESET,
    SXILL_SYNC_STATE_SEARCHING,
    SXILL_SYNC_STATE_HEADER,
    SXILL_SYNC_STATE_PAYLOAD,
    SXILL_SYNC_STATE_CHECK_VALUE,
    SXILL_SYNC_STATE_READY,
    SXILL_SYNC_STATE_RESET_AFTER_READY
#endif
} SXILL_SYNC_STATE_ENUM;

#ifdef SXILL_USE_LEGACY_RING_BUFFER
/*
 * These are sub-states of the sync state
 */
typedef enum sxill_sync_detected_state_enum
{
    SXILL_SYNC_STATE_HEADER,
    SXILL_SYNC_STATE_PAYLOAD,
    SXILL_SYNC_STATE_CHECK_VALUE

} SXILL_SYNC_DETECTED_STATE_ENUM;
#endif

/*
 * This is the SXiLL sync state machine structure
 * used to track progress while syncing received
 * SXiLL messages.
 */
typedef struct sxill_sync_state_machine_context_struct
{
    SXILL_SYNC_STATE_ENUM eSyncState;
#ifdef SXILL_USE_LEGACY_RING_BUFFER
    SXILL_SYNC_DETECTED_STATE_ENUM eSyncDetectedState;

    // Tracking pointers for collecting SYNC bytes
    UN8 const *pun8CurrentSyncByte;
    UN8 const *pun8LastSyncByte;

    // Mark for current start of frame
    UN8 const *pun8StartOfFrame;

    // Tracking pointers for collecting HEADER bytes
    SXILL_FRAME_HDR_STRUCT sFrameHeader;
    UN8 *pun8CurrentHeaderByte;
    UN8 const *pun8LastHeaderByte;

    // Counters for number of payload bytes in the current frame
    SXILL_LEN tNumPayloadBytes;
    size_t tNumPayloadBytesProcessed;

    // Tracking pointers for collecting CHECK VALUE bytes
    SXILL_CHECK_VALUE tComputedCheckValue;
    SXILL_CHECK_VALUE_STRUCT sCheckValue;
    UN8 *pun8CurrentCheckValueByte;
    UN8 const *pun8LastCheckValueByte;
#else
    // Tracking Sync Sequence
    SXILL_SYNC tSync;

    // Tracking pointers for collecting HEADER bytes
    SXILL_FRAME_HDR_STRUCT sFrameHeader;
    size_t tFrameHeaderSize;

    // Payload size in host endian
    SXILL_LEN tNumPayloadBytes;

    // Tracking pointers for collecting CHECK VALUE bytes
    SXILL_CHECK_VALUE tComputedCheckValue;
    SXILL_CHECK_VALUE_STRUCT sCheckValue;
    size_t tCheckValueSize;
#endif

} SXILL_SYNC_STATE_MACHINE_CONTEXT_STRUCT;

typedef struct sxill_rx_frame_control_struct
{
    // I/O Device (from TxRx)
    FILE *psDevice;

    // Rx frame ring buffer
    SXILL_RING_BUFFER_STRUCT sRingBuffer;

    // State machine context
    SXILL_SYNC_STATE_MACHINE_CONTEXT_STRUCT sContext;

    // Protocol handle
    STI_PROTOCOL_HDL hProtocol;

    // No rx activity counter
    UN32 un32NoRxCount;

#ifdef SXILL_RX_STATS_OUTPUT

    // Rx Statistics
    SXILL_RX_STATISTICS_STRUCT sStatistics;
    OSAL_OBJECT_HDL hLog;
    OSAL_OBJECT_HDL hReqLog;
    UN32 un32LastWrite;

#endif

} SXILL_RX_FRAME_CONTROL_STRUCT;

/*
 * This structure helps us validate
 * min/max payload sizes an also which payloads to handle.
 * See SX-9845-0098, Section 2.1.3 and 2.1.5
 */
typedef struct sxill_payload_table_struct
{
    // Byte definitions for payloads of this type.
    UN16 un16NumPayloadBytesMin;
    UN16 un16NumPayloadBytesMax;

    // Flag to indicate we either process or do not precess these payloads
    BOOLEAN bProcess;

} SXILL_PAYLOAD_TABLE_STRUCT;

/****************/
/** PROTOTYPES **/
/****************/

// Protocol Connection Processing Prototypes
static STI_PROTOCOL_CONNECTION_INIT_HANDLER(SXILL_pvConnectionInit);
static STI_PROTOCOL_CONNECTION_UNINIT_HANDLER(SXILL_vConnectionUninit);
static N16 SXILL_n16ConnectionCompare(void *pvArg1, void *pvArg2);

// TxRx Prototypes
static STI_PROTOCOL_TXRX_INIT_HANDLER(SXILL_bTxRxInit);
static STI_PROTOCOL_UNINIT_HANDLER(SXILL_vTxRxUnInit);
static STI_PROTOCOL_TXRX_HANDLER(SXILL_n32TxRxHdlr);

// RxFrame Prototypes
static STI_PROTOCOL_RX_INIT_HANDLER(SXILL_bRxFrameInit);
static STI_PROTOCOL_UNINIT_HANDLER(SXILL_vRxFrameUnInit);
static STI_PROTOCOL_RX_HANDLER(SXILL_n32RxFrameHdlr);

// Local (helper) functions

static UN16 un16CalculateCheckValue(UN16 un16CheckValue, void const *pvBuffer,
    UN32 un32Length);

static size_t tWritePayload(
    UN8 *pun8Dst,
    size_t tSize,
    SXILL_PAYLOAD_TYPE tPayloadType,
    OSAL_BUFFER_HDL hSrcBuffer
        );

static BOOLEAN bSendFrame(
    SXILL_TXRX_CONTROL_STRUCT *psTxRxCtrl,
    UN8 *pun8Src,
    size_t tSrcLength
        );

static BOOLEAN bExtractFrameHdr(
    OSAL_BUFFER_HDL hRxFrame,
    SXILL_FRAME_HDR_STRUCT *psFrameHdr
        );

static void vProcessMessage(SXILL_RX_FRAME_CONTROL_STRUCT *psRxFrameCtrl);

static void vProcessLinkLayerMessages(
    SXILL_RX_FRAME_CONTROL_STRUCT *psRxFrameCtrl,
    OSAL_BUFFER_HDL hRxFrame,
    BOOLEAN *pbPostEvent
        );

static void vResetSyncStateMachine(
    SXILL_SYNC_STATE_MACHINE_CONTEXT_STRUCT *psContext);

static void vSyncStateMachine(SXILL_RX_FRAME_CONTROL_STRUCT *psRxFrameCtrl);

#ifdef SXILL_USE_LEGACY_RING_BUFFER
__INLINE__ static void vAdvanceRingBuffer(
    SXILL_RING_BUFFER_STRUCT *psRingBuffer, size_t tPtrOffset, size_t *ptOffset);

__INLINE__ static size_t tNumToEndOfEgressBytes(
    SXILL_RING_BUFFER_STRUCT *psRingBuffer, size_t tMemberOffset);

__INLINE__ static size_t tNumAvailableInRingBuffer(
    SXILL_RING_BUFFER_STRUCT *psRingBuffer);
#endif

static void vInitRingBuffer(SXILL_RING_BUFFER_STRUCT *psRingBuffer);

static void vHandleRx(
    STI_PROTOCOL_HDL hProtocol,
    SXILL_TXRX_CONTROL_STRUCT *psCtrl,
    STI_PROTOCOL_RX_PAYLOAD_STRUCT *psRx
        );

static void vHandleProtocolSpecific(
    STI_PROTOCOL_HDL hProtocol,
    SXILL_TXRX_CONTROL_STRUCT *psCtrl,
    STI_PROTOCOL_SPECIFIC_STRUCT const *psProtocol
        );

static void vHandleOOB(
    STI_PROTOCOL_HDL hProtocol,
    SXILL_TXRX_CONTROL_STRUCT *psCtrl,
    STI_PROTOCOL_OOB_STRUCT const *psOOB
        );

#if DEBUG_OBJECT != 0
static void vDumpFrame(OSAL_BUFFER_HDL hBuffer);
#endif

#ifdef SXILL_RX_STATS_OUTPUT
static void vWriteRxStats(SXILL_RX_FRAME_CONTROL_STRUCT *psRxFrameCtrl);
static void vWriteRxReq(OSAL_OBJECT_HDL hLog, size_t tNumToRead,
    size_t tNumRead);
static void vWriteRingBufferState(
    long lLineNum,
    SXILL_RING_BUFFER_REASON_CODE_ENUM eReasonCode,
    const SXILL_RING_BUFFER_STRUCT *psRingBuffer,
    size_t tArg
        );
#endif

#ifdef SXILL_TXRX_STATS_OUTPUT
static void vWriteTxRxStats(SXILL_TXRX_CONTROL_STRUCT *psTxRxCtrl);
#endif

/***************/
/** VARIABLES **/
/***************/
#ifdef SXILL_USE_LEGACY_RING_BUFFER
const UN8 gaun8SyncBytes[sizeof(SXILL_SYNC)] =
{ SXILL_SYNC_BYTE0, SXILL_SYNC_BYTE1 };
#endif

// See SX-9845-0098, Section 2.1.3 and 2.1.5
const SXILL_PAYLOAD_TABLE_STRUCT gasPayloadTable[32] =
{
    { 2, 32, TRUE }, // Link
    { 2, 1024, TRUE }, // Control
    { 2, 1042, TRUE }, // Data
    { 2, 1088, TRUE }, // Audio
    { 0, 65535, FALSE }, // Undefined - 4
    { 0, 65535, FALSE }, // Undefined - 5
    { 0, 65535, FALSE }, // Undefined - 6
    { 0, 65535, FALSE }, // Undefined - 7
    { 0, 65535, FALSE }, // Undefined - 8
    { 0, 65535, FALSE }, // Undefined - 9
    { 0, 65535, FALSE }, // Undefined - 10
    { 0, 65535, FALSE }, // Undefined - 11
    { 0, 65535, FALSE }, // Undefined - 12
    { 0, 65535, FALSE }, // Undefined - 13
    { 0, 65535, FALSE }, // Undefined - 14
    { 0, 65535, FALSE }, // Undefined - 15
    { 0, 65535, FALSE }, // Undefined - 16
    { 0, 65535, FALSE }, // Undefined - 17
    { 0, 65535, FALSE }, // Undefined - 18
    { 0, 65535, FALSE }, // Undefined - 19
    { 0, 65535, FALSE }, // Undefined - 20
    { 0, 65535, FALSE }, // Undefined - 21
    { 0, 65535, FALSE }, // Undefined - 22
    { 0, 65535, FALSE }, // Undefined - 23
    { 0, 65535, FALSE }, // Undefined - 24
    { 0, 65535, FALSE }, // Undefined - 25
    { 0, 65535, FALSE }, // Undefined - 26
    { 0, 65535, FALSE }, // Undefined - 27
    { 0, 65535, FALSE }, // Undefined - 28
    { 0, 65535, FALSE }, // Undefined - 29
    { 0, 65535, FALSE }, // Undefined - 30
    { 1, 1024, FALSE } // Debug
};

const size_t GtSxillRxFrameBlockSize = SXILL_RX_FRAME_COMPUTED_BLOCK_SIZE;

const STI_PROTOCOL_INTERFACE GsSXILL =
{
    "SXILL-UART",

    // Connection Handlers
    { SXILL_pvConnectionInit, SXILL_vConnectionUninit,
        SXILL_n16ConnectionCompare },

    // TxRx Task Handlers
    { 8192, // Bytes of stack
        SXILL_bTxRxInit, SXILL_vTxRxUnInit, SXILL_n32TxRxHdlr },

    // RxFrame Task Handlers
    { 8192, // Bytes of stack
        SXILL_bRxFrameInit, SXILL_vRxFrameUnInit, SXILL_n32RxFrameHdlr }
};

/**********************/
/** INLINE FUNCTIONS **/
/**********************/

#endif	// _SXILL_H_
