/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains various APIs and functions which implement the STI
 *  protocol handler. A protocol handler typically consists of 2 tasks;
 *  one strictly for receiving data and another transceiver task responsible
 *  for processing responses and transmitting as needed. Although it is not
 *  required that a protocol driver support both tasks.
 *
 ******************************************************************************/

#include <string.h>
#include <time.h>

#include "standard.h"
#include "osal.h"

#include "sti_api.h"
#include "sti_framework.h"
#include "sti_sync.h"

#include "sti_protocol.h"
#include "_sti_protocol.h"

/*****************************************************************************
*
*   STIP_hInstallProtocol
*
*   This API is called internally by the STI Framework to install an
*   instance of a protocol driver.
*
*   Inputs:
*       psIntf - A valid protocol interface.
*		ePriority - the priority level desired for this protocol
*       psProtocolEntry - A pointer representing the protocol entry maintained
*           by the STI framework for this protocol.
*       vRxCallback - A pointer to a callback function which is invoked
*           when a transport layer payload is received over this connection.
*           If this pointer is NULL no responses will be processed. If
*           something other than NULL is provided upon receipt of a payload
*           the callback function is invoked with the STI_HDL over which it
*           was received and, and the payload data. See Section 8.7 for a
*           description of this function's prototype requirements.
*       pvRxCallbackArg - A pointer to void which represents a data type
*           known to the caller and must be cast in the callback function.
*
*   Outputs:
*       A valid STI_PROTOCOL_HDL on success. Otherwise returns
*       STI_INVALID_PROTOCOL_HDL.
*
*****************************************************************************/
STI_PROTOCOL_HDL STIP_hInstallProtocol (
    const STI_PROTOCOL_ENTRY_STRUCT *psProtocolEntry,
    STI_PROTOCOL_CALLBACK vCallback,
    void *pvCallbackArg
        )
{
    STI_PROTOCOL_HDL hProtocol = STI_PROTOCOL_INVALID_HDL;
    STI_PROTOCOL_STRUCT *psProtocol; // STI_PROTOCOL_HDL
    char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];

    // Check inputs
    if(psProtocolEntry == NULL) // must have an entry
    {
        return STI_PROTOCOL_INVALID_HDL;
    }

    // Check more inputs
    if( (psProtocolEntry->psProtocolIntf == NULL) || // Must have an interface
            (psProtocolEntry->psProtocolIntf->pacName == NULL) || // Must have a name
                (psProtocolEntry->pvDevice == NULL) // Must have an I/O Device
                    )
    {
        return STI_PROTOCOL_INVALID_HDL;
    }

    // Install an instance of the Protocol Handler
    // on this Physical Interface...

    // Construct a unique name for this protocol object
    snprintf(&acName[0], sizeof(acName), "%s:Obj",
        psProtocolEntry->psProtocolIntf->pacName);

    // Allocate a task control structure (zero initialized)
    psProtocol = (STI_PROTOCOL_STRUCT *)
        OSAL.pvMemoryAllocate(
            &acName[0],
            sizeof(STI_PROTOCOL_STRUCT), // STI_PROTOCOL_HDL
            TRUE );
    if(psProtocol == NULL)
    {
        return STI_PROTOCOL_INVALID_HDL;
    }

    do
    {
        STI_PROTOCOL_TXRX_TASK_STRUCT *psTxRxCtrl;
        STI_PROTOCOL_RX_TASK_STRUCT *psRxCtrl;
        const STI_PROTOCOL_INTERFACE *psIntf;

        // Record back-trace pointer into the connection list
        // for this protocol
        psProtocol->psProtocolEntry = psProtocolEntry;

        // Initialize timeout (maximum)
        psProtocol->n32TimeoutMsec =
            STI_PROTOCOL_REPORTING_TIMEOUT_SEC * 1000;

        // Initialize active state
        psProtocol->bActive = FALSE;

        // Local pointers to simplify look of the code
        psRxCtrl = &psProtocol->sRxCtrl;
        psTxRxCtrl = &psProtocol->sTxRxCtrl;
        psIntf = psProtocolEntry->psProtocolIntf;

        // For both tasks, if a handler has been provided,
        // set up that task's settings
        if(psIntf->sTxRx._n32Hdlr !=
            (STI_PROTOCOL_TXRX_HANDLER_PROTOTYPE)NULL)
        {
            // Populate task info
            psTxRxCtrl->un32StackSize =
                psIntf->sTxRx.un32StackSize;
            psTxRxCtrl->un32Options =
                STI_PROTOCOL_TASK_OPTIONS;
            psTxRxCtrl->sIntf =
                psIntf->sTxRx;
            psTxRxCtrl->psProtocol =
                psProtocol;
            psTxRxCtrl->pvDevice =
                psProtocolEntry->pvDevice;
            psTxRxCtrl->hEventQueue =
                OSAL_INVALID_OBJECT_HDL;
            psTxRxCtrl->un32EventQueueSize =
                0;
            psTxRxCtrl->pvProtocolData = NULL;
            psTxRxCtrl->eTaskPriority = OSAL_TASK_PRIORITY_HIGH;
            psTxRxCtrl->bShutdown = FALSE;
            psTxRxCtrl->hTxBlockPool =
                OSAL_INVALID_OBJECT_HDL;
            psTxRxCtrl->hRxBlockPool =
                OSAL_INVALID_OBJECT_HDL;
            psTxRxCtrl->bTaskInstalled = FALSE;

            // Construct a unique name for this task and keep it.
            snprintf(&psTxRxCtrl->acName[0],
                    sizeof(psTxRxCtrl->acName),
                    "%s:TxRx",
                    psIntf->pacName
                        );

            // Construct an appropriate unique name for this sync
            snprintf(&acName[0],
                        sizeof(acName),
                        "%s:Sync",
                        &psTxRxCtrl->acName[0]
                            );

            // Create TxRx Sync
            psTxRxCtrl->hSync = STIS_hCreate(acName);
            if(psTxRxCtrl->hSync == STI_SYNC_INVALID_HDL)
            {
                // Error!
                printf("STI Error! Cannot create txrx sync (%s).\n", acName);
                break;
            }
        }

        if(psIntf->sRx._n32Hdlr !=
                    (STI_PROTOCOL_RX_HANDLER_PROTOTYPE)NULL)
        {
            // Populate task info
            psRxCtrl->un32StackSize =
                psIntf->sRx.un32StackSize;
            psRxCtrl->un32Options =
                STI_PROTOCOL_TASK_OPTIONS;
            psRxCtrl->sIntf =
                psIntf->sRx;
            psRxCtrl->pvDevice =
                psProtocolEntry->pvDevice;
            psRxCtrl->psProtocol =
                psProtocol;
            psRxCtrl->eTaskPriority = OSAL_TASK_PRIORITY_HIGH;
            psRxCtrl->bTaskInstalled = FALSE;

            // Construct a unique name for this task
            snprintf(&psRxCtrl->acName[0],
                    sizeof(psRxCtrl->acName),
                    "%s:Rx",
                    psIntf->pacName
                        );

            // Construct an appropriate unique name for this sync
            snprintf(&acName[0],
                        sizeof(acName),
                        "%s:Sync",
                        &psRxCtrl->acName[0]
                            );

            // Create Rx Sync
            psRxCtrl->hSync = STIS_hCreate(acName);
            if(psRxCtrl->hSync == STI_SYNC_INVALID_HDL)
            {
                // Error!
                printf("STI Error! Cannot create rx sync (%s).\n", acName);
                break;
            }
        }

        // Start the TxRx task if a handler was provided
        if(psIntf->sTxRx._n32Hdlr !=
            (STI_PROTOCOL_TXRX_HANDLER_PROTOTYPE)NULL)
        {
            // Install and wait for completion
            psTxRxCtrl->bTaskInstalled = bInstallTxRxTask(psTxRxCtrl);
            if (psTxRxCtrl->bTaskInstalled == FALSE)
            {
                printf("STI Error! Cannot install TxRx Task(%s).\n",
                    &psTxRxCtrl->acName[0]);
                break;
            }
        }
        // No TxRx task, so start the Rx task instead
        else if (psIntf->sRx._n32Hdlr !=
                (STI_PROTOCOL_RX_HANDLER_PROTOTYPE)NULL)
        {
            // Only install the Rx Task here if there wasn't a
            // TxRx task specified.  If a TxRx task was specified, then
            // the TxRx task will install the Rx task once the TxRx task
            // is up and running

            // Install and wait for completion
            psRxCtrl->bTaskInstalled = bInstallRxTask(psRxCtrl);
            if (psRxCtrl->bTaskInstalled == FALSE)
            {
                printf("STI Error! Cannot install Rx Task (%s).\n",
                    &psRxCtrl->acName[0]);
                break;
            }
        }

        // *******************************************************************
        // SYNCHRONIZATION POINT - INITIALIZE
        // This is a synchronization point where the caller of
        // STI_hInstallProtocol() is held off until the TxRx and
        // Rx threads have been created, initialized and running.
        // *******************************************************************

        // Everything checks-out
        hProtocol = (STI_PROTOCOL_HDL)psProtocol;

    } while (FALSE);

    // Check if everything passed
    if(hProtocol == STI_PROTOCOL_INVALID_HDL)
    {
        // Something failed, clean-up this mess.
        STIP_bUninstallProtocol((STI_PROTOCOL_HDL)psProtocol);
    }
    else
    {
        // All is a go!
        psProtocol->bActive = TRUE;
    }

    return hProtocol;
}

/*****************************************************************************
*
*   STIP_bUninstallProtocol
*
*   Uninstalls a previously installed protocol.
*
*   Inputs:
*       hProtocol - The handle of the protocol to uninstall.
*
*   Outputs:
*       Returns TRUE on success, otherwise FALSE on error.
*
*****************************************************************************/
BOOLEAN STIP_bUninstallProtocol (
    STI_PROTOCOL_HDL hProtocol
        )
{
    STI_PROTOCOL_STRUCT *psProtocol =
        (STI_PROTOCOL_STRUCT *)hProtocol; // STI_PROTOCOL_HDL
    OSAL_RETURN_CODE_ENUM eReturnCode;
    BOOLEAN bFinished, bDestroyed;

    // Check input
    if(hProtocol == STI_PROTOCOL_INVALID_HDL)
    {
        return FALSE;
    }

    // Request kill of Rx Task, if it was created and no TxRx task
    // was created.
    if((psProtocol->sRxCtrl.hTask != OSAL_INVALID_OBJECT_HDL) &&
        (psProtocol->sTxRxCtrl.hTask == OSAL_INVALID_OBJECT_HDL))
    {
        // Signal Rx-Task
        eReturnCode = OSAL.eTaskDelete(psProtocol->sRxCtrl.hTask);
        if(eReturnCode == OSAL_SUCCESS)
        {
            psProtocol->sRxCtrl.hTask = OSAL_INVALID_OBJECT_HDL;
        }

        // Wait for RxTask
        bFinished = STIS_bWait(psProtocol->sRxCtrl.hSync, NULL);
        if(bFinished == TRUE)
        {
            // Destroy the sync-object
            bDestroyed = STIS_bDestroy(psProtocol->sRxCtrl.hSync);
            if(bDestroyed == TRUE)
            {
                // Initialize handle
                psProtocol->sRxCtrl.hSync = STI_SYNC_INVALID_HDL;
            }
        }
    }
    // Request kill of TxRx Task, if it was created.
    else if(psProtocol->sTxRxCtrl.hTask != OSAL_INVALID_OBJECT_HDL)
    {
        eReturnCode = OSAL.eTaskDelete(psProtocol->sTxRxCtrl.hTask);
        if(eReturnCode == OSAL_SUCCESS)
        {
            psProtocol->sTxRxCtrl.hTask = OSAL_INVALID_OBJECT_HDL;
        }

        // Wait for TxRxTask (which also waits for the Rx task)
        bFinished = STIS_bWait(psProtocol->sTxRxCtrl.hSync, NULL);
        if(bFinished == TRUE)
        {
            // Destroy the sync-object
            bDestroyed = STIS_bDestroy(psProtocol->sTxRxCtrl.hSync);
            if(bDestroyed == TRUE)
            {
                // Initialize handle
                psProtocol->sTxRxCtrl.hSync = STI_SYNC_INVALID_HDL;
            }
        }
    }
    else // Nothing installed
    {
        // Error!
        printf("STI Error! Nothing installed (%s).\n",
            psProtocol->psProtocolEntry->psProtocolIntf->pacName);

        // Such case is possible under error conditions, where
        // protocol initialization has failed for some reason. 
        // In this section, both protocol tasks have exited, or 
        // going to exit. So, make sure, that they didn't 
        // leave their stuff.

        if (psProtocol->sRxCtrl.hSync != STI_SYNC_INVALID_HDL)
        {
            // Destroy the sync-object
            bDestroyed = STIS_bDestroy(psProtocol->sRxCtrl.hSync);
            if(bDestroyed == TRUE)
            {
                // Initialize handle
                psProtocol->sRxCtrl.hSync = STI_SYNC_INVALID_HDL;
            }
        }

        if (psProtocol->sTxRxCtrl.hSync != STI_SYNC_INVALID_HDL)
        {
            // Destroy the sync-object
            bDestroyed = STIS_bDestroy(psProtocol->sTxRxCtrl.hSync);
            if(bDestroyed == TRUE)
            {
                // Initialize handle
                psProtocol->sTxRxCtrl.hSync = STI_SYNC_INVALID_HDL;
            }
        }
    }

    // Release protocol structure --- done after threads die!
    OSAL.vMemoryFree(psProtocol);

    return TRUE;
}

/*****************************************************************************
*
*   STIP_psGetProtocolEntry
*
*****************************************************************************/
const STI_PROTOCOL_ENTRY_STRUCT *STIP_psGetProtocolEntry(
    STI_PROTOCOL_HDL hProtocol
        )
{
    const STI_PROTOCOL_ENTRY_STRUCT *psProtocolEntry = NULL;

    // Check input protocol handle
    if(hProtocol != STI_PROTOCOL_INVALID_HDL)
    {
        STI_PROTOCOL_STRUCT *psProtocol =
            (STI_PROTOCOL_STRUCT *)hProtocol; // STI_PROTOCOL_HDL
        psProtocolEntry = psProtocol->psProtocolEntry;
    }

    return psProtocolEntry;
}

/*****************************************************************************
*
*   STIP_bGetProtocolData
*
*   This API is used by Protocol Drivers to query the protocol data
*   pointer they set when the protocol was installed.
*
*   Inputs:
*   	hProtocol - A valid STI protocol handle for which to retrieve the
*           protocol data for.
*       ppvProtocolData - A pointer populated with the protocol's data
*
*   Outputs:
*   	Returns TRUE on success, otherwise FALSE on error.
*
*****************************************************************************/
BOOLEAN STIP_bGetProtocolData(
    STI_PROTOCOL_HDL hProtocol,
    void **ppvProtocolData
        )
{
    BOOLEAN bSuccess = FALSE;
    STI_PROTOCOL_STRUCT *psProtocol =
        (STI_PROTOCOL_STRUCT *)hProtocol; // STI_PROTOCOL_HDL

    // Check input protocol handle
    if((hProtocol != STI_PROTOCOL_INVALID_HDL) &&
        (ppvProtocolData != NULL))
    {
        *ppvProtocolData = psProtocol->sTxRxCtrl.pvProtocolData;
        bSuccess = TRUE;
    }

    return bSuccess;
}

/*****************************************************************************
*
*   STIP_hAllocateTxPayload
*
*   This API is used to allocate an application tx payload buffer from
*   the specified STI connection. The payload buffer allocated is an
*   OSAL Dynamic Buffer which may be populated by the application using
*   OSAL Dynamic Buffer APIs (such as OSAL.tBufferWriteHead() and
*   OSAL.tBufferWriteTail()). Once allocated this buffer belongs to and
*   is associated with this connection.
*
*   Inputs:
*       hProtocol - A valid STI protocol handle from which the
*           allocated buffer will be obtained.
*
*   Outputs:
*       A valid OSAL_BUFFER_HDL is returned on success, otherwise
*       OSAL_INVALID_BUFFER_HDL is returned.
*
*****************************************************************************/
OSAL_BUFFER_HDL STIP_hAllocateTxPayload ( STI_PROTOCOL_HDL hProtocol )
{
    OSAL_BUFFER_HDL hPayload = OSAL_INVALID_BUFFER_HDL;

    // Check inputs
    if(hProtocol != STI_PROTOCOL_INVALID_HDL)
    {
        STI_PROTOCOL_STRUCT *psProtocol =
            (STI_PROTOCOL_STRUCT *)hProtocol; // STI_PROTOCOL_HDL

        // Now, allocate something from the payload block pool...
        hPayload = OSAL.hBufferAllocate(
            psProtocol->sTxRxCtrl.hTxBlockPool,
            FALSE, // No read pending
            FALSE, // No write pending
            OSAL_BUFFER_ALLOCATE_OPTION_NONE);
        if(hPayload == OSAL_INVALID_BUFFER_HDL)
        {
            printf("STI Error! Cannot allocate tx payload (%s).\n",
                psProtocol->sTxRxCtrl.acName);
        }
    }

    return hPayload;
}

/*****************************************************************************
*
*   STIP_hAllocateRxPayload
*
*   This API is used to allocate an application rx payload buffer from
*   the specified STI connection. The payload buffer allocated is an
*   OSAL Dynamic Buffer which may be populated by the application using
*   OSAL Dynamic Buffer APIs (such as OSAL.tBufferWriteHead() and
*   OSAL.tBufferWriteTail()). Once allocated this buffer belongs to and
*   is associated with this connection.
*
*   Inputs:
*       hProtocol - A valid STI protocol handle from which the
*           allocated buffer will be obtained.
*
*   Outputs:
*       A valid OSAL_BUFFER_HDL is returned on success, otherwise
*       OSAL_INVALID_BUFFER_HDL is returned.
*
*****************************************************************************/
OSAL_BUFFER_HDL STIP_hAllocateRxPayload ( STI_PROTOCOL_HDL hProtocol )
{
    OSAL_BUFFER_HDL hPayload = OSAL_INVALID_BUFFER_HDL;

    // Check inputs
    if(hProtocol != STI_PROTOCOL_INVALID_HDL)
    {
        STI_PROTOCOL_STRUCT *psProtocol =
            (STI_PROTOCOL_STRUCT *)hProtocol; // STI_PROTOCOL_HDL

        // Now, allocate something from the payload block pool...
        hPayload = OSAL.hBufferAllocate(
            psProtocol->sTxRxCtrl.hRxBlockPool,
            FALSE, // No read pending
            FALSE, // No write pending
            OSAL_BUFFER_ALLOCATE_OPTION_NONE);
        if(hPayload == OSAL_INVALID_BUFFER_HDL)
        {
            printf("STI Error! Cannot allocate rx payload (%s).\n",
                psProtocol->sTxRxCtrl.acName);
        }
    }

    return hPayload;
}

/*****************************************************************************
*
*   STIP_eSendPayload
*
*   This API is used to send or transmit a previously allocated and populated
*   payload via the associated connection. Once successfully sent, the payload
*   is queued by the connection and is the responsibility of the STI Protocol
*   Driver to eventually free the payload once it has been transmitted or
*   transmission failure occurs.
*
*   Inputs:
*       hSTI - The connection to send this payload over
*       hProtocol - A valid STI protocol handle for which the allocated buffer
*           will be transmitted over.
*       pvConnectionArgument - A void * type which is Protocol Driver specific
*           and must be cast to the appropriate type during event processing.
*           Protocol Drivers may use this as a way to communicate event data to
*           the Transceiver Handler.
*       bBlock - TRUE to perform a blocking operation, otherwise FALSE.
*       hPayload - A valid STI payload buffer handle that is to queued
*           for transmission.
*       un8Retries - The number of additional attempts to tx this payload.
*       n32Timeout - The number of msec to wait for a response before a
*           retry is attempted.
*
*   Outputs:
*       TRUE on success. Otherwise a value of FALSE is returned on error.
*
*****************************************************************************/
STI_PROTOCOL_RESULT_CODE_ENUM STIP_eSendPayload (
    STI_HDL hSTI,
    STI_PROTOCOL_HDL hProtocol,
    BOOLEAN bBlock,
    OSAL_BUFFER_HDL hPayload,
    UN8 un8Retries,
    N32 n32Timeout
        )
{
    STI_PROTOCOL_STRUCT *psProtocol =
        (STI_PROTOCOL_STRUCT *)hProtocol; // STI_PROTOCOL_HDL
    STI_PROTOCOL_EVENT_STRUCT *psMsg;
    BOOLEAN bSuccess;

    // Check inputs...
    if( (hProtocol == STI_PROTOCOL_INVALID_HDL) ||
        (hPayload == OSAL_INVALID_BUFFER_HDL) ||
        // Any re-try without some actual timeout is not permitted
        // nor does it make sense.
        ((un8Retries > 0) && (n32Timeout == OSAL_OBJ_TIMEOUT_NONE))
            )
    {
        return STI_RESULT_CODE_BAD_INPUT;
    }

    // Check connection state
    if(psProtocol->bActive == FALSE)
    {
        // Forget about it...
        return STI_RESULT_CODE_ERROR;
    }

    // Allocate a message
    psMsg = psAllocateMessage(psProtocol, bBlock, FALSE);
    if(psMsg == NULL)
    {
        return STI_RESULT_CODE_ERROR;
    }

    // Populate message
    psMsg->eType = STI_PROTOCOL_EVENT_TX_PAYLOAD;
    psMsg->sMsg.sTx.hSTI = hSTI;
    psMsg->sMsg.sTx.hPayload = hPayload;
    psMsg->sMsg.sTx.un8Retries = un8Retries;
    psMsg->sMsg.sTx.n32Timeout = n32Timeout;

    bSuccess = bPostMessage(psMsg);
    if(bSuccess == FALSE)
    {
        return STI_RESULT_CODE_ERROR;
    }

    return STI_RESULT_CODE_OK;
}

/*****************************************************************************
*
*   STIP_bReceivePayload
*
*   This API is used by Protocol Drivers to post events to the Transceiver
*   Handler. The Transceiver handler only runs when events are posted to it,
*   otherwise it is idle. Events posted are Protocol Driver specific.
*
*   Inputs:
*   	hProtocol - A valid STI protocol handle for which post this event.
*   		A protocol handle represents a protocol instance or installation
*   		which can service one or more connections. Therefore if the event
*   		is connection specific, sufficient information must be provided
*   		in the event data being posted.
*       bBlock - TRUE to perform a blocking operation, otherwise FALSE.
*       hPayload - A valid STI payload buffer handle that is to queued
*           for reception.
*
*   Outputs:
*   	TRUE on success (event posted). Otherwise a value of FALSE is returned
*   	when the event could not be posted.
*
*****************************************************************************/
BOOLEAN STIP_bReceivePayload (
    STI_PROTOCOL_HDL hProtocol,
    BOOLEAN bBlock,
    OSAL_BUFFER_HDL hPayload
        )
{
    STI_PROTOCOL_STRUCT *psProtocol =
        (STI_PROTOCOL_STRUCT *)hProtocol; // STI_PROTOCOL_HDL
    STI_PROTOCOL_EVENT_STRUCT *psMsg;

    // Check inputs...
    if( (hProtocol == STI_PROTOCOL_INVALID_HDL) ||
        (hPayload == OSAL_INVALID_BUFFER_HDL) )
    {
        return FALSE;
    }

    // Allocate a message
    psMsg = psAllocateMessage(psProtocol, bBlock, FALSE);
    if(psMsg == NULL)
    {
        return FALSE;
    }

    // Populate message
    psMsg->eType = STI_PROTOCOL_EVENT_RX_PAYLOAD;
    psMsg->sMsg.sRx.hPayload = hPayload;

    return bPostMessage(psMsg);
}

/*****************************************************************************
*
*   STIP_bPostOOB
*
*   This API is used by Protocol Drivers to post OOB events to the Transceiver
*   Handler. The Transceiver handler only runs when events are posted to it,
*   otherwise it is idle.
*
*   Inputs:
*   	hProtocol - A valid STI protocol handle for which post this event.
*   		A protocol handle represents a protocol instance or installation
*   		which can service one or more connections. Therefore if the event
*   		is connection specific, sufficient information must be provided
*   		in the event data being posted.
*       pvProtocolConnectionData -
*       bBlock - TRUE to perform a blocking operation, otherwise FALSE.
*   	pvEventData - A void * type which is Protocol Driver specific and must
*   		be cast to the appropriate type during event processing. Protocol
*   		Drivers may use this as a way to communicate event data to the
*   		Transceiver Handler.
*
*   Outputs:
*   	TRUE on success (event posted). Otherwise a value of FALSE is returned
*   	when the event could not be posted.
*
*****************************************************************************/
BOOLEAN STIP_bPostOOB (
    STI_PROTOCOL_HDL hProtocol,
    void *pvProtocolConnectionData,
    BOOLEAN bBlock,
    STI_OOB_TYPE_ENUM eType,
    void *pvEventData
        )
{
    STI_PROTOCOL_STRUCT *psProtocol =
        (STI_PROTOCOL_STRUCT *)hProtocol; // STI_PROTOCOL_HDL
    STI_PROTOCOL_EVENT_STRUCT *psMsg;

    if(hProtocol == STI_PROTOCOL_INVALID_HDL)
    {
        return FALSE;
    }

    // Allocate a message
    psMsg = psAllocateMessage(psProtocol, bBlock, TRUE);
    if(psMsg == NULL)
    {
        return FALSE;
    }

    // Populate message
    psMsg->eType = STI_PROTOCOL_EVENT_OOB;
    psMsg->sMsg.sOOB.eType = eType;
    psMsg->sMsg.sOOB.pvProtocolConnectionData = pvProtocolConnectionData;
    psMsg->sMsg.sOOB.pvData = pvEventData;

    return bPostMessage(psMsg);
}

/*****************************************************************************
*
*   STIP_bPostTimeout
*
*   This API is used by Protocol Drivers to post timeout events to the TxRx
*   Handler. The Transceiver handler only runs when events are posted to it,
*   otherwise it is idle.
*
*   Inputs:
*   	hProtocol - A valid STI protocol handle for which post this event.
*       hSTI - A valid STI handle to post this timeout on.
*       bBlock - TRUE to perform a blocking operation, otherwise FALSE.
*
*   Outputs:
*   	TRUE on success (event posted). Otherwise a value of FALSE is returned
*   	when the event could not be posted.
*
*****************************************************************************/
BOOLEAN STIP_bPostTimeout (
    STI_PROTOCOL_HDL hProtocol,
    STI_HDL hSTI,
    BOOLEAN bBlock
        )
{
    BOOLEAN bPosted = FALSE;
    STI_PROTOCOL_EVENT_STRUCT *psMsg;
    STI_PROTOCOL_STRUCT *psProtocol =
        (STI_PROTOCOL_STRUCT *)hProtocol; // STI_PROTOCOL_HDL

    // Allocate a message
    psMsg = psAllocateMessage(psProtocol, bBlock, TRUE);
    if(psMsg != NULL)
    {
        // Populate message
        psMsg->eType = STI_PROTOCOL_EVENT_TIMEOUT;
        psMsg->sMsg.sTimeout.hSTI = hSTI;

        // Post message on event queue
        bPosted = bPostMessage(psMsg);
    }

    return bPosted;
}

/*****************************************************************************
*
*   STIP_bSetError
*
*   This API is used by Protocol Drivers to indicate a protocol connection
*   is in error.
*
*   Inputs:
*   	hProtocol - A valid STI protocol handle which is in error.
*
*   Outputs:
*   	TRUE on success (handled). Otherwise a value of FALSE is returned
*   	when it could not be processed.
*
*****************************************************************************/
BOOLEAN STIP_bSetError (
    STI_PROTOCOL_HDL hProtocol
        )
{
    BOOLEAN bPosted = FALSE;
    STI_PROTOCOL_EVENT_STRUCT *psMsg;
    STI_PROTOCOL_STRUCT *psProtocol =
        (STI_PROTOCOL_STRUCT *)hProtocol; // STI_PROTOCOL_HDL

    // Allocate a message
    psMsg = psAllocateMessage(psProtocol, TRUE, TRUE);
    if(psMsg != NULL)
    {
        // Populate message
        psMsg->eType = STI_PROTOCOL_EVENT_ERROR;

        // Post message on event queue
        bPosted = bPostMessage(psMsg);
    }

    return bPosted;
}

// Local (helper) functions

/*******************************************************************************
*
*   n32TxRxTask
*
*   This is the Transceiver task used in the STI Framework
*
*   Inputs:
*       pvArg - A pointer to the transceiver task control structure
*
*   Outputs:
*       A value of OSAL_TASK_REPORT_NO_ERROR if no error , otherwise some
*       value other than OSAL_TASK_REPORT_NO_ERROR on error.
*
*****************************************************************************/
static N32 n32TxRxTask ( void *pvArg )
{
    BOOLEAN bSuccess;
    N32 n32TaskErr = !OSAL_TASK_REPORT_NO_ERROR;
    STI_PROTOCOL_TXRX_TASK_STRUCT *psTask =
        (STI_PROTOCOL_TXRX_TASK_STRUCT *)pvArg;

    do
    {
        UN32 un32TimeSec = 0, un32LastReportTimeSec = 0;

        // Initialize this task.
        psTask->bTaskInstalled = bTxRxTaskInitialize(psTask);
        if(psTask->bTaskInstalled == FALSE)
        {
            // STI sync signal will occur at the 
            // end of the task.

            // Error!
            break;
        }

        // Release installer
        STIS_bSignal(psTask->hSync, NULL);

        // Forever
        for(;;)
        {
            OSAL_RETURN_CODE_ENUM eReturnCode;
            STI_PROTOCOL_EVENT_STRUCT *psEvent;
            UN32 un32EventSize;

            // Get a message from the Event Queue
            eReturnCode =
                OSAL.eQueueGet (
                    psTask->hEventQueue,
                    ((void **)&psEvent), &un32EventSize,
                    psTask->psProtocol->n32TimeoutMsec);

            if(eReturnCode == OSAL_SUCCESS)
            {
                // Process event
                switch((STI_PROTOCOL_EVENT_ENUM)psEvent->eType)
                {
                    case STI_PROTOCOL_EVENT_SHUTDOWN:
                    {
                        BOOLEAN bDestroyed;

                        // The RX Task has stopped, so we can now shutdown

                        // Destroy the sync-object
                        bDestroyed = STIS_bDestroy(
                            psTask->psProtocol->sRxCtrl.hSync);
                        if(bDestroyed == TRUE)
                        {
                            // Initialize handle
                            psTask->psProtocol->sRxCtrl.hSync =
                                STI_SYNC_INVALID_HDL;

                            // Signal TxRx-Thread it is ok to shutdown
                            psTask->bShutdown = TRUE;
                        }
                    }
                    break;

                    case STI_PROTOCOL_EVENT_RX_SHUTDOWN:
                    {
                        // Instruct the Rx Task to stop if it exists
                        if(psTask->psProtocol->sRxCtrl.hTask !=
                            OSAL_INVALID_OBJECT_HDL)
                        {
                            // Rx task exists, so once it is shutdown, we will
                            // be informed we can shutdown.
                            eReturnCode =
                                OSAL.eTaskDelete(
                                    psTask->psProtocol->sRxCtrl.hTask);
                            if(eReturnCode == OSAL_SUCCESS)
                            {
                                psTask->psProtocol->sRxCtrl.hTask =
                                    OSAL_INVALID_OBJECT_HDL;
                            }
                        }
                        else
                        {
                            // No Rx task so, just shut ourselves down.

                            // Signal TxRx-Thread it is ok to shutdown
                            psTask->bShutdown = TRUE;
                        }
                    }
                    break;

                    case STI_PROTOCOL_EVENT_ERROR:
                        psTask->psProtocol->bActive = FALSE;
                    break;

                    default:
                    {
                        // Call Task Handler (interface must exist or task
                        // would never have been created).
                        n32TaskErr =
                            psTask->sIntf._n32Hdlr(
                                psTask->psProtocol,
                                psTask->pvProtocolData,
                                psEvent
                                    );
                    }
                    break;
                }

                // Free message
                OSAL.eMessageFree(psEvent);
            }

            // Track report time and report if needed
            OSAL.vTimeUp( &un32TimeSec, (UN16 *)NULL );

            if ( ( ( un32TimeSec - un32LastReportTimeSec )
               >= STI_PROTOCOL_REPORTING_TIMEOUT_SEC ) ||
                (eReturnCode == OSAL_TIMEOUT) )
            {
                // The timeout interval has passed since the
                // last time we reported to the task monitor, so it's
                // time to report again...
                eReturnCode = OSAL.eTaskReport( n32TaskErr );
                if (eReturnCode == OSAL_SUCCESS)
                {
                    // record time so we can check for the next report time
                    un32LastReportTimeSec = un32TimeSec;
                }
                else
                {
                    // Error!
                    printf("%s is unable to check in (%s)\n",
                           &psTask->acName[0],
                           OSAL.pacGetReturnCodeName(eReturnCode));
                }
            }

            // Check if this task generated an error, or
            // if this task is to shutdown
            if((n32TaskErr != OSAL_TASK_REPORT_NO_ERROR) ||
                (TRUE == psTask->bShutdown))
            {
                // Exit task
                break;
            }

        } // Forever

    } while(FALSE);

    // Clean up this task if it was initialized.
    // If it doesn't, it should be un-initialized (rolled-back)
    // during initializtion.
    if (psTask->bTaskInstalled == TRUE)
    {
        // Clean-up this task
        vTxRxTaskUninitialize(psTask);
    }

    // Release uninstaller
    bSuccess = STIS_bSignal(psTask->hSync, NULL);
    if(bSuccess != TRUE)
    {
        // Error!
        n32TaskErr = !OSAL_TASK_REPORT_NO_ERROR;
    }

    // Indicate task has exited
    return n32TaskErr;
}

/*****************************************************************************
*
*   bTxRxTaskInitialize
*
*   This function is used to initialize the Transceiver Task.
*
*   Inputs:
*       psTask - A pointer to the task control structure
*
*   Outputs:
*       TRUE if task is initialized successfully
*	    FALSE if task does not successfully initialize
*
*****************************************************************************/
static BOOLEAN bTxRxTaskInitialize(
    STI_PROTOCOL_TXRX_TASK_STRUCT *psTask
        )
{
    BOOLEAN bRetVal = FALSE;

    // Check input
    if(psTask == NULL)
    {
        return FALSE;
    }

    do
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;

        // Register Task with OSAL
        eReturnCode = OSAL.eTaskRegister(
                       STI_PROTOCOL_REPORTING_INTERVAL_SEC,
                       vSleepHandler,
                       vWakeupHandler,
                       vShutdownTxRxHandler,
                       psTask );
        if(eReturnCode != OSAL_SUCCESS)
        {
            // Error!
            printf("STI Error! Unable to register TxRx.\n");
            vTxRxTaskUninitialize(psTask);
            break;
        }

        // Call protocol initialization (if defined)
        if( psTask->sIntf._bInit !=
           (STI_PROTOCOL_TXRX_INIT_HANDLER_PROTOTYPE)NULL )
        {
            BOOLEAN bSuccess;

            bSuccess = psTask->sIntf._bInit(
                &psTask->acName[0],
                psTask->psProtocol,
                psTask->pvDevice,
                &psTask->un32EventQueueSize,
                &psTask->hTxBlockPool,
                &psTask->hRxBlockPool,
                &psTask->pvProtocolData
                    );
            if(bSuccess == FALSE)
            {
                // Error!
                printf("STI Error! Unable to initialize TxRx.\n");
                vTxRxTaskUninitialize(psTask);
                break;
            }
        }

        // Create required queues (if needed)...

        // Create Event queue?
        if(psTask->un32EventQueueSize > 0)
        {
            char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];

            // Construct an appropriate unique name
            snprintf(&acName[0],
                     sizeof(acName),
                     "%s:EventQ",
                     &psTask->acName[0]
                         );

            // Create Event Queue
            eReturnCode =
                OSAL.eQueueCreate(
                    &psTask->hEventQueue,
                    &acName[0],
                    psTask->un32EventQueueSize,
                    sizeof(STI_PROTOCOL_EVENT_STRUCT),
                    OSAL_QUEUE_OPTION_FIXED_SIZE
                        );

            if(eReturnCode != OSAL_SUCCESS)
            {
                // Error!
                printf("STI Error! Unable to create event queue %s.\n",
                       OSAL.pacGetReturnCodeName(eReturnCode));
                vTxRxTaskUninitialize(psTask);
                break;
            }
        }

        // If there is a Rx Handler for this protocol, lets start
        // that task.
        if (psTask->psProtocol->sRxCtrl.sIntf._n32Hdlr !=
            (STI_PROTOCOL_RX_HANDLER_PROTOTYPE)NULL)
        {
            BOOLEAN bRxTaskInstalled;

            // Use TxRx task's protocol data for the Rx task
            psTask->psProtocol->sRxCtrl.pvProtocolData = psTask->pvProtocolData;

            // Install the task
            bRxTaskInstalled = bInstallRxTask(
                &psTask->psProtocol->sRxCtrl);

            if (bRxTaskInstalled == FALSE)
            {
                // Error!
                printf("STI Error! Unable to start Rx task.\n");
                vTxRxTaskUninitialize(psTask);
                break;
            }
        }

        // Everything initialized
        bRetVal = TRUE;

    } while(FALSE);

    return bRetVal;
}

/*****************************************************************************
*
*   vTxRxTaskUninitialize
*
*   This function will un-initialize any components that
*   have been created at start up.
*
*   Inputs:
*       psTask - A pointer to the task control structure
*
*   Outputs:
*       None
*
*****************************************************************************/
static void vTxRxTaskUninitialize(
    STI_PROTOCOL_TXRX_TASK_STRUCT *psTask
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Check input
    if(psTask == NULL)
    {
        return;
    }

    // Call protocol un-initialization (if defined)
    if( psTask->sIntf._vUninit != (STI_PROTOCOL_UNINIT_HANDLER_PROTOTYPE)NULL )
    {
        psTask->sIntf._vUninit(psTask->psProtocol, psTask->pvProtocolData);
    }

    // Invalidate protocol data
    psTask->pvProtocolData = NULL;

    // If the Queue hasn't been destroyed yet, destroy it now
    if(psTask->hEventQueue != OSAL_INVALID_OBJECT_HDL)
    {
        eReturnCode = OSAL.eQueueDelete(psTask->hEventQueue);
        if(eReturnCode == OSAL_SUCCESS)
        {
            // Invalidate queue handle
            psTask->hEventQueue = OSAL_INVALID_OBJECT_HDL;
        }
    }

    // Destroy associated block pool if one exists
    if(psTask->hTxBlockPool != OSAL_INVALID_OBJECT_HDL)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;

        eReturnCode =
            OSAL.eBlockPoolDelete(psTask->hTxBlockPool);
        if(eReturnCode == OSAL_SUCCESS)
        {
            // Deleting a block pool also effectively deletes any
            // buffers and/or outstanding blocks allocated from it
            psTask->hTxBlockPool = OSAL_INVALID_OBJECT_HDL;
        }
    }

    // Destroy associated block pool if one exists
    if(psTask->hRxBlockPool != OSAL_INVALID_OBJECT_HDL)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;

        eReturnCode =
            OSAL.eBlockPoolDelete(psTask->hRxBlockPool);
        if(eReturnCode == OSAL_SUCCESS)
        {
            // Deleting a block pool also effectively deletes any
            // buffers and/or outstanding blocks allocated from it
            psTask->hRxBlockPool = OSAL_INVALID_OBJECT_HDL;
        }
    }

    // Unregister Task with OSAL
    OSAL.eTaskUnregister();

    // Invalidate task handle, to indicate task no longer exists
    psTask->hTask = OSAL_INVALID_OBJECT_HDL;

    return;
}

/*******************************************************************************
*
*   n32RxTask
*
*   This is the Receiver task used in the STI Framework
*
*   Inputs:
*       pvArg - A pointer to the Receiver task control structure
*
*   Outputs:
*       A value of OSAL_TASK_REPORT_NO_ERROR if no error , otherwise some
*       value other than OSAL_TASK_REPORT_NO_ERROR on error.
*
*****************************************************************************/
static N32 n32RxTask ( void *pvArg )
{
    N32 n32TaskErr = !OSAL_TASK_REPORT_NO_ERROR;
    STI_PROTOCOL_RX_TASK_STRUCT *psTask =
        (STI_PROTOCOL_RX_TASK_STRUCT *)pvArg;

    do
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;
        UN32 un32TimeSec = 0, un32LastReportTimeSec = 0;

        // Initialize this task!
        psTask->bTaskInstalled = bRxTaskInitialize(psTask);

        // Release installer unconditionally
        STIS_bSignal(psTask->hSync, NULL);

        if(psTask->bTaskInstalled == FALSE)
        {
            // Error!
            break;
        }

        // Forever
        for(;;)
        {
            n32TaskErr =
                psTask->sIntf._n32Hdlr(
                    psTask->psProtocol,
                    psTask->pvProtocolData);

            // Track report time and report if needed
            OSAL.vTimeUp( &un32TimeSec, (UN16 *)NULL );

            // Determine how much time has elapsed since the last check-in.
            // If the report timeout has elapsed then check-in with the
            // task monitor.
            if ( ( un32TimeSec - un32LastReportTimeSec )
               >= STI_PROTOCOL_REPORTING_TIMEOUT_SEC )
            {
                // The timeout interval has passed since the
                // last time we reported to the task monitor, so it's
                // time to report again...
                eReturnCode = OSAL.eTaskReport( n32TaskErr );
                if (eReturnCode == OSAL_SUCCESS)
                {
                    // record time so we can check for the next report time
                    un32LastReportTimeSec = un32TimeSec;
                }
                else
                {
                    // Error!
                    printf("%s is unable to check in (%s)\n",
                           &psTask->acName[0],
                           OSAL.pacGetReturnCodeName(eReturnCode));
                }
            }

            // Acquire control mutex, then release
            eReturnCode = OSAL.eSemTake(
                psTask->hControlSemaphore, OSAL_OBJ_TIMEOUT_NONE);
            if(eReturnCode == OSAL_SUCCESS)
            {
                // Give it back
                eReturnCode = OSAL.eSemGive(psTask->hControlSemaphore);
                if (eReturnCode != OSAL_SUCCESS)
                {
                    printf("STI error: unable to release semaphore! (%s)\n",
                           OSAL.pacGetReturnCodeName(eReturnCode));
                }
            }
            else
            {
                // Exit task
                printf("Shutdown!");
                break;
            }

        } // Forever

    } while (FALSE);

    // Clean up this task if it was initialized.
    // If it doesn't, it should be un-initialized (rolled-back)
    // during initializtion.
    if (psTask->bTaskInstalled == TRUE)
    {
        // Clean-up this task
        vRxTaskUninitialize(psTask);
    }

    // Indicate task has exited
    return n32TaskErr;
}

/*****************************************************************************
*
*   bRxTaskInitialize
*
*   This function is used to initialize the Receiver Task.
*
*   Inputs:
*       psTask - A pointer to the task control structure
*
*   Outputs:
*       TRUE if task is initialized successfully
*	    FALSE if task does not successfully initialize
*
*****************************************************************************/
static BOOLEAN bRxTaskInitialize(
    STI_PROTOCOL_RX_TASK_STRUCT *psTask
        )
{
    BOOLEAN bRetVal = FALSE;

    // Check input
    if(psTask == NULL)
    {
        return FALSE;
    }

    do
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;
        char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];

        // Construct a unique name for this sem
        snprintf(&acName[0], sizeof(acName), "%s:Control",
                    &psTask->acName[0]);

        // Create control semaphore (available)
        eReturnCode = OSAL.eSemCreate(
            &psTask->hControlSemaphore,
            &acName[0], 1, 1, OSAL_SEM_OPTION_NONE);
        if(eReturnCode != OSAL_SUCCESS)
        {
            // Error!
            printf("STI Error! Cannot create control semaphore.\n");
            vRxTaskUninitialize(psTask);
            break;
        }

        // Register Task with OSAL
        eReturnCode = OSAL.eTaskRegister(
                       STI_PROTOCOL_REPORTING_INTERVAL_SEC,
                       vSleepHandler,
                       vWakeupHandler,
                       vShutdownRxHandler,
                       psTask );
        if(eReturnCode != OSAL_SUCCESS)
        {
            // Error!
            printf("STI Error! Unable to register Rx.\n");
            vRxTaskUninitialize(psTask);
            break;
        }

        // Call protocol initialization (if defined)
        if( psTask->sIntf._bInit !=
           (STI_PROTOCOL_RX_INIT_HANDLER_PROTOTYPE)NULL )
        {
            BOOLEAN bSuccess;

            bSuccess = psTask->sIntf._bInit(
                &psTask->acName[0],
                psTask->psProtocol,
                psTask->pvDevice,
                psTask->psProtocol->n32TimeoutMsec,
                &psTask->pvProtocolData
                    );
            if(bSuccess == FALSE)
            {
                // Rx Frame initialization failed, which means 
                // Rx protocol data was not created and it currently
                // point to the protocol data of the TxRx task 
                // (used for initial Rx task initialization).
                // Must invalidate this to left it for TxRx task (who 
                // is the owner).

                // Invalidate protocol data.
                psTask->pvProtocolData = NULL;

                // Error!
                printf("STI Error! Unable to initialize Rx.\n");
                vRxTaskUninitialize(psTask);
                break;
            }
        }

        // Everything initialized
        bRetVal = TRUE;

    } while(FALSE);

    return bRetVal;
}

/*****************************************************************************
*
*   vRxTaskUninitialize
*
*   This function will un-initialize any components that
*   have been created at start up.
*
*   Inputs:
*       psTask - A pointer to the task control structure
*
*   Outputs:
*       None
*
*****************************************************************************/
static void vRxTaskUninitialize(
    STI_PROTOCOL_RX_TASK_STRUCT *psTask
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Check input
    if(psTask == NULL)
    {
        return;
    }

    // Call protocol (RxTask) un-initialization (if defined)
    if( psTask->sIntf._vUninit !=
       (STI_PROTOCOL_UNINIT_HANDLER_PROTOTYPE)NULL )
    {
        psTask->sIntf._vUninit(
            psTask->psProtocol,
            psTask->pvProtocolData);
    }

    // Invalidate protocol data
    psTask->pvProtocolData = NULL;

    // Unregister Task with OSAL
    OSAL.eTaskUnregister();

    // Delete the semaphore (no longer needed) - force post
    if(psTask->hControlSemaphore != OSAL_INVALID_OBJECT_HDL)
    {
        eReturnCode = OSAL.eSemDelete(psTask->hControlSemaphore);
        if(eReturnCode == OSAL_SUCCESS)
        {
            psTask->hControlSemaphore = OSAL_INVALID_OBJECT_HDL;
        }
    }

    // Invalidate task handle, to indicate task no longer exists
    psTask->hTask = OSAL_INVALID_OBJECT_HDL;

    // Tell the TxRx task that we are now done
    bPostEvent(psTask->psProtocol,
        STI_PROTOCOL_EVENT_SHUTDOWN, NULL);

    return;
}

/*****************************************************************************
*
*   vSleepHandler
*
*   This function handles putting a task to sleep
*
*   Inputs:
*       pvArg - A pointer to the main protocol driver control structure.
*
*   Outputs:
*       None.
*
*****************************************************************************/
static void vSleepHandler( void *pvArg )
{
    // Nothing to do
    //
    // If we ever decide to do something, will need to split this
    // into separate TxRx and Rx handlers

    return;
}

/*****************************************************************************
*
*   vWakeupHandler
*
*   This function handles waking a task
*
*   Inputs:
*       pvArg - A pointer to the main protocol driver control structure.
*
*   Outputs:
*       None.
*
*****************************************************************************/
static void vWakeupHandler( void *pvArg )
{
    // Nothing to do
    //
    // If we ever decide to do something, will need to split this
    // into separate TxRx and Rx handlers

    return;
}

/*****************************************************************************
*
*   vShutdownTxRxHandler
*
*   This function handles shutting the TxRx task down
*
*   Inputs:
*       pvArg - A pointer to the main protocol driver control structure.
*
*   Outputs:
*       None.
*
*****************************************************************************/
static void vShutdownTxRxHandler( void *pvArg )
{
    STI_PROTOCOL_TXRX_TASK_STRUCT *psTxRxCtrl =
        (STI_PROTOCOL_TXRX_TASK_STRUCT *)pvArg;

    // Tell the TxRx task to shutdown the Rx task.
    bPostEvent(psTxRxCtrl->psProtocol,
        STI_PROTOCOL_EVENT_RX_SHUTDOWN, NULL);

    return;
}

/*****************************************************************************
*
*   vShutdownRxHandler
*
*   This function handles shutting the Rx task down
*
*   Inputs:
*       pvArg - A pointer to the main protocol driver control structure.
*
*   Outputs:
*       None.
*
*****************************************************************************/
static void vShutdownRxHandler( void *pvArg )
{
    STI_PROTOCOL_RX_TASK_STRUCT *psRxCtrl =
        (STI_PROTOCOL_RX_TASK_STRUCT *)pvArg;

    // Take control mutex to force shutdown
    OSAL.eSemTake(
        psRxCtrl->hControlSemaphore, OSAL_OBJ_TIMEOUT_INFINITE);

    return;
}

/*****************************************************************************
*
*   bInstallTxRxTask
*
*   This function is used to install the protocol TxRx Task.
*
*   Inputs:
*       psTxRxCtrl - pointer to one of the task structures
*
*   Outputs:
*       BOOLEAN
*
*****************************************************************************/
static BOOLEAN bInstallTxRxTask (
    STI_PROTOCOL_TXRX_TASK_STRUCT *psTxRxCtrl
        )
{
    BOOLEAN bSuccess = FALSE;

    do
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;

        // Create the task
        eReturnCode = OSAL.eTaskCreate(
                &psTxRxCtrl->hTask,
                &psTxRxCtrl->acName[0],
                n32TxRxTask,
                (void*)psTxRxCtrl,
                psTxRxCtrl->un32StackSize,
                psTxRxCtrl->eTaskPriority,
                psTxRxCtrl->un32Options);
        if(eReturnCode != OSAL_SUCCESS)
        {
            // Error!
            printf("STI Error! Cannot create TxRx Task.\n");
            break;
        }

        // Wait for TxRxTask
        bSuccess = STIS_bWait(psTxRxCtrl->hSync, NULL);

        if ((bSuccess == TRUE) &&
            (psTxRxCtrl->bTaskInstalled == FALSE))
        {
            // Make sure, if the task was not installed,
            // the task handle is set to INVALID
            psTxRxCtrl->hTask = OSAL_INVALID_OBJECT_HDL;
            bSuccess = FALSE;
        }

    } while (FALSE);

    return bSuccess;
}

/*****************************************************************************
*
*   bInstallRxTask
*
*   This function is used to install the protocol Rx Task.
*
*   Inputs:
*       psRxCtrl - pointer to one of the task structures
*
*   Outputs:
*       BOOLEAN
*
*****************************************************************************/
static BOOLEAN bInstallRxTask ( STI_PROTOCOL_RX_TASK_STRUCT *psRxCtrl )
{
    BOOLEAN bSuccess = FALSE;

    do
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;

        // Create the task
        eReturnCode = OSAL.eTaskCreate(
                &psRxCtrl->hTask,
                &psRxCtrl->acName[0],
                n32RxTask,
                (void*)psRxCtrl,
                psRxCtrl->un32StackSize,
                psRxCtrl->eTaskPriority,
                psRxCtrl->un32Options);
        if(eReturnCode != OSAL_SUCCESS)
        {
            // Error!
            printf("STI Error! Cannot create Rx Task.\n");
            break;
        }

        // Wait for RxTask
        bSuccess = STIS_bWait(psRxCtrl->hSync, NULL);

        if ((bSuccess == TRUE) &&
            (psRxCtrl->bTaskInstalled == FALSE))
        {
            // Make sure, if the task was not installed,
            // the task handle is set to INVALID
            psRxCtrl->hTask = OSAL_INVALID_OBJECT_HDL;
            bSuccess = FALSE;
        }

    } while (FALSE);

    return bSuccess;
}

/*****************************************************************************
*
*   bPostEvent
*
*   This API is used by Protocol Drivers to post events to the Transceiver
*   Handler. The Transceiver handler only runs when events are posted to it,
*   otherwise it is idle. Events posted are Protocol Driver specific.
*
*   Inputs:
*   	hProtocol - A valid STI protocol handle for which post this event.
*   		A protocol handle represents a protocol instance or installation
*   		which can service one or more connections. Therefore if the event
*   		is connection specific, sufficient information must be provided
*   		in the event data being posted.
*   	un32EventType - A UN32 describing a Protocol Driver specific event
*   		type. This can be used by protocol drivers as a code or key into
*   		event types.
*   	pvEventData - A void * type which is Protocol Driver specific and must
*   		be cast to the appropriate type during event processing. Protocol
*   		Drivers may use this as a way to communicate event data to the
*   		Transceiver Handler.
*
*   Outputs:
*   	TRUE on success (event posted). Otherwise a value of FALSE is returned
*   	when the event could not be posted.
*
*****************************************************************************/
static BOOLEAN bPostEvent (
    STI_PROTOCOL_HDL hProtocol,
    UN32 un32EventType,
    void *pvEventData
        )
{
    STI_PROTOCOL_STRUCT *psCtrl =
        (STI_PROTOCOL_STRUCT *)hProtocol; // STI_PROTOCOL_HDL
    STI_PROTOCOL_EVENT_STRUCT *psMsg;
    BOOLEAN bBlock = FALSE, bUrgent = FALSE;

    if(hProtocol == STI_PROTOCOL_INVALID_HDL)
    {
        return FALSE;
    }

    // Is this the last RX event?
    if(un32EventType == (UN32)STI_PROTOCOL_EVENT_SHUTDOWN)
    {
        // Yes, so make sure the TxRx task shutsdown without
        // processing any further events, and also make sure
        // this event actually gets on the queue.
        bBlock = TRUE;
        bUrgent = TRUE;
    }

    // Allocate a message
    psMsg = psAllocateMessage(psCtrl, bBlock, bUrgent);
    if(psMsg == NULL)
    {
        return FALSE;
    }

    // Populate message
    psMsg->eType = (STI_PROTOCOL_EVENT_ENUM)un32EventType;
    psMsg->sMsg.sProto.pvData = pvEventData;

    // Post message on event queue
    return bPostMessage(psMsg);
}

/*****************************************************************************
*
*   psAllocateMessage
*
*****************************************************************************/
static STI_PROTOCOL_EVENT_STRUCT *psAllocateMessage(
    STI_PROTOCOL_STRUCT const *psProtocol,
    BOOLEAN bBlock,
    BOOLEAN bUrgent
         )
{
    STI_PROTOCOL_EVENT_STRUCT *psMsg = NULL;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    UN32 un32QueueFlags;

    // Configure Queue Flags
    if (bBlock == FALSE)
    {
        // We can drop transmissions when the tx message
        // queue is full
        un32QueueFlags = OSAL_QUEUE_FLAG_NONBLOCK;
    }
    else
    {
        // Reliable protocols need STI to not
        // drop transmission requeusts due to
        // full Tx message queues
        un32QueueFlags = OSAL_QUEUE_FLAG_BLOCK;
    }

    // Configure Queue Flags
    if (bUrgent == TRUE)
    {
        // This needs to go to the front of the queue
        un32QueueFlags |= OSAL_QUEUE_FLAG_URGENT;
    }

    // Allocate a message from the TxRx Event Queue
    eReturnCode =
        OSAL.eMessageAllocate (
            psProtocol->sTxRxCtrl.hEventQueue,
            ((void **)&psMsg),
            un32QueueFlags );
    if(eReturnCode != OSAL_SUCCESS)
    {
        // Error!
        psMsg = NULL;
    }

    return psMsg;
}

/*****************************************************************************
*
*   bPostMessage
*
*****************************************************************************/
static BOOLEAN bPostMessage(
    STI_PROTOCOL_EVENT_STRUCT const *psMsg
         )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Post message on event queue
    eReturnCode = OSAL.eQueuePut(
        psMsg,
        sizeof(STI_PROTOCOL_EVENT_STRUCT));
    if(eReturnCode != OSAL_SUCCESS)
    {
        // Free message
        OSAL.eMessageFree(psMsg);
        return FALSE;
    }

    return TRUE;
}
