/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 * PRIVATE HEADER
 *
 ******************************************************************************/

  /*********************************/
 /** PREVENT REDUNDANT INCLUSION **/
/*********************************/
#ifndef _STI_PROTOCOL_H_
#define _STI_PROTOCOL_H_

  /**************/
 /** INCLUDES **/
/**************/

#include <stdio.h>

#include "standard.h"
#include "osal.h"

#include "sti_api.h"

  /***************/
 /** CONSTANTS **/
/***************/

// Task options go here. Same for both tasks
#define STI_PROTOCOL_TASK_OPTIONS				(OSAL_TASK_OPTION_NONE)

// Check-in must occur at least once every 30 seconds
#define STI_PROTOCOL_REPORTING_INTERVAL_SEC     (30) // seconds

// Definition for the maximum duration to allow a task to run without
// checking in with the task monitor. This duration must be less
// than 1/2 the check-in interval. The 'less-than' part is important because
// assume the interval is 10 seconds, and "tick" time is 10 msec.
// After 4990 msec a task run's but since 5000 msec have not elapsed, the task
// does not check-in. Instead it pends for another 5000 msec, making a total of
// 9990 msec before it checks in again. Due to the acuracy of the internal
// clock ticker which is +/- 10 msec the task may just miss the next
// check-in interval and cause the task monitor to fail. Thus the rule
// of thumb here should be the task should check-in no less than 2 times per
// check-in interval (not once, which is what happens when 1/2 the interval
// is used. So we simply use a 3rd which in integer math will always round down.
#define STI_PROTOCOL_REPORTING_TIMEOUT_SEC      \
    ( STI_PROTOCOL_REPORTING_INTERVAL_SEC / 3 )

  /**************/
 /** TYPEDEFS **/
/**************/

/*
    Generic Protocol Task Structures
*/

typedef struct sti_protocol_txrx_task_struct
{
    // Protocol structure (STI_PROTOCOL_HDL)
    struct sti_protocol_struct *psProtocol;

    // Task Name
    char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];

    // Task Handle
    OSAL_OBJECT_HDL hTask;

    // Task Stack Size
    UN32 un32StackSize;

    // Task Options
    UN32 un32Options;

    // Protocol Interface...
    STI_PROTOCOL_TXRX_HANDLER_STRUCT sIntf;

    // Device I/O Handle
    void *pvDevice;

    // Protocol Event Queue Handles and Sizes
    OSAL_OBJECT_HDL hEventQueue;
    UN32 un32EventQueueSize;

    // Protocol Specific Data (TxRx)
    void *pvProtocolData;

    // Task priority set during protocol installation
    OSAL_TASK_PRIORITY_ENUM eTaskPriority;

    // Shutdown flag
    BOOLEAN bShutdown;

    // Tx Payload (Packet) Buffer Handle
    OSAL_OBJECT_HDL hTxBlockPool;

    // Rx Payload (Packet) Buffer Handle
    OSAL_OBJECT_HDL hRxBlockPool;

    // Sync object
    STI_SYNC_HDL hSync;

    // Installation flag
    BOOLEAN bTaskInstalled;

} STI_PROTOCOL_TXRX_TASK_STRUCT;

typedef struct sti_protocol_rx_task_struct
{
    // Protocol structure (STI_PROTOCOL_HDL)
    struct sti_protocol_struct *psProtocol;

    // Task Name
    char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];

    // Task Handle
    OSAL_OBJECT_HDL hTask;

    // Task Stack Size
    UN32 un32StackSize;

    // Task Options
    UN32 un32Options;

    // Protocol Interface...
    STI_PROTOCOL_RX_HANDLER_STRUCT sIntf;

    // Device I/O Handle
    void *pvDevice;

    // Protocol Specific Data (Rx)
    void *pvProtocolData;

    // Task priority set during protocol installation
    OSAL_TASK_PRIORITY_ENUM eTaskPriority;

    // Semaphore used to sync installation, and coordinate
    // shutdown.
    OSAL_OBJECT_HDL hControlSemaphore;

    // Sync object
    STI_SYNC_HDL hSync;

    // Installation flag
    BOOLEAN bTaskInstalled;

} STI_PROTOCOL_RX_TASK_STRUCT;

/*
    Protocol Control Structure

    One instance of this structure exists for each installation
    of a protocol. This means one instance for each protocol over a
    physical interface.

*/
typedef struct sti_protocol_struct
{
    // Back-trace protocol entry pointer into STI-Framework
    const STI_PROTOCOL_ENTRY_STRUCT *psProtocolEntry;

    // Timeout
    N32 n32TimeoutMsec;

    // Active state
    BOOLEAN bActive;

    // Task Control Structure (TxRx)
    STI_PROTOCOL_TXRX_TASK_STRUCT sTxRxCtrl;

    // Task Structure (Rx)
    STI_PROTOCOL_RX_TASK_STRUCT sRxCtrl;

} STI_PROTOCOL_STRUCT; // (STI_PROTOCOL_HDL)

  /************/
 /** MACROS **/
/************/

  /****************/
 /** PROTOTYPES **/
/****************/

static N32 n32TxRxTask ( void *pvArg );
static BOOLEAN bTxRxTaskInitialize( STI_PROTOCOL_TXRX_TASK_STRUCT *psTask );
static void vTxRxTaskUninitialize( STI_PROTOCOL_TXRX_TASK_STRUCT *psTask );
static N32 n32RxTask ( void *pvArg );
static BOOLEAN bRxTaskInitialize( STI_PROTOCOL_RX_TASK_STRUCT *psTask );
static void vRxTaskUninitialize( STI_PROTOCOL_RX_TASK_STRUCT *psTask );
static void vSleepHandler( void *pvArg );
static void vWakeupHandler( void *pvArg );
static void vShutdownTxRxHandler( void *pvArg );
static void vShutdownRxHandler( void *pvArg );

static BOOLEAN bInstallTxRxTask ( STI_PROTOCOL_TXRX_TASK_STRUCT *psTxRxCtrl );
static BOOLEAN bInstallRxTask ( STI_PROTOCOL_RX_TASK_STRUCT *psRxCtrl );

static BOOLEAN bPostEvent (
    STI_PROTOCOL_HDL hProtocol,
    UN32 un32EventType,
    void *pvEventData
        );

static STI_PROTOCOL_EVENT_STRUCT *psAllocateMessage(
    STI_PROTOCOL_STRUCT const *psCtrl,
    BOOLEAN bBlock,
    BOOLEAN bUrgent
         );

static BOOLEAN bPostMessage(
    STI_PROTOCOL_EVENT_STRUCT const *psMsg
         );

  /***************/
 /** VARIABLES **/
/***************/

  /**********************/
 /** INLINE FUNCTIONS **/
/**********************/

#endif	// _STI_PROTOCOL_H_
