/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/

#ifndef STI_API_H_
#define STI_API_H_

#include "standard.h"
#include "osal.h"

  /**********************/
 /** GLOBAL CONSTANTS **/
/**********************/

  /*********************/
 /** GLOBAL TYPEDEFS **/
/*********************/

/*
    Satellite Transport Interface (STI) handle definition

    An object of STI_HDL is returned to the caller of the API STI_hConnect()
    and must be supplied by the caller to each of the STI_xxx() APIs which
    take an STI handle. The caller should check for handle validity by
    comparing it to STI_INVALID_HDL.
*/

typedef void * STI_HDL;
#define STI_INVALID_HDL ((STI_HDL)NULL)

/*
    STI API Result Codes from STI API calls
*/

typedef enum sti_protocol_result_code_enum
{
    STI_RESULT_CODE_OK, // all is well
    STI_RESULT_CODE_ERROR, // something went wrong
    STI_RESULT_CODE_TIMEOUT, // timeout
    STI_RESULT_CODE_BAD_INPUT // you did something wrong

} STI_PROTOCOL_RESULT_CODE_ENUM;


/*
    STI Tx msg/event
*/
typedef struct sti_protocol_tx_payload_struct
{
    // STI connection this is being sent over
    STI_HDL hSTI;

    // Message to tx
    OSAL_BUFFER_HDL hPayload;

    // Number of retry attempts allowed
    UN8 un8Retries;

    // Timeout between attempts
    N32 n32Timeout;

} STI_PROTOCOL_TX_PAYLOAD_STRUCT;

/*
    STI Rx msg/event
*/
typedef struct sti_protocol_rx_payload_struct
{
    // Message to rx
    OSAL_BUFFER_HDL hPayload;

} STI_PROTOCOL_RX_PAYLOAD_STRUCT;

/*
    Difficult to say what types of OOB responses
    may be sent, but perhaps we just want to
    indicate something about our link, or statistics, etc.
    We can add new OOBs to the framework if needed at
    a later time. For now we just communicate link errors.
*/
typedef enum sti_oob_type_enum
{
    STI_OOB_TYPE_LINK_ERROR,
    STI_OOB_TYPE_PROTOCOL_SPECIFIC

} STI_OOB_TYPE_ENUM;

/*
    STI OOB msg/event
*/

typedef struct sti_protocol_oob_struct
{
    STI_OOB_TYPE_ENUM eType;
    void *pvProtocolConnectionData; // optional
    void *pvData; // TBD by protocol specific driver

} STI_PROTOCOL_OOB_STRUCT;

/*
    STI Protocol specific msg/event.
    Not really used right now, but provision is made for
    the future.
*/

typedef struct sti_protocol_specific_struct
{
    // Your free to define this how you like
    // in your STI-protocol implementation.
    void *pvData;

} STI_PROTOCOL_SPECIFIC_STRUCT;

/*
    STI Timeout msg/event
*/

typedef struct sti_protocol_timeout_struct
{
    // STI connection this is being sent over
    STI_HDL hSTI;

} STI_PROTOCOL_TIMEOUT_STRUCT;

/*
    STI Error msg/event
*/

typedef struct sti_protocol_error_struct
{
    // Nothing needed, placeholder
    UN32 un32Dummy;

} STI_PROTOCOL_ERROR_STRUCT;

/*
    An STI callback may be of different types.
    These are the various callback which can happen
    as a result of STI API actions. These are used
    by STI-protocols to signal back to the STI
    framework when things have been handled.
*/
typedef enum sti_protocol_callback_msg_type_enum
{
    STI_PROTOCOL_CALLBACK_MSG_TYPE_RX_PAYLOAD, // Recvd a payload
    STI_PROTOCOL_CALLBACK_MSG_TYPE_TX_PAYLOAD, // Tx'd a payload
    STI_PROTOCOL_CALLBACK_MSG_TYPE_TX_TIMEOUT, // a timeout happened
    STI_PROTOCOL_CALLBACK_MSG_TYPE_RX_OOB // OOB msg published

} STI_PROTOCOL_CALLBACK_MSG_TYPE_ENUM;

/*
    The STI callback struct provided to the protocol
    provided callback function which contains the information
    relative to the protocol callback type being invoked.
*/

typedef struct sti_protocol_callback_msg_struct
{
    STI_PROTOCOL_CALLBACK_MSG_TYPE_ENUM eType;

    // Various response types
    union
    {
        STI_PROTOCOL_TX_PAYLOAD_STRUCT *psTx;
        STI_PROTOCOL_RX_PAYLOAD_STRUCT *psRx;
        STI_PROTOCOL_OOB_STRUCT *psOOB;
        STI_PROTOCOL_SPECIFIC_STRUCT *psProto;

    } sMsg;

    // Caller provided argument
    void *pvArg;

} STI_PROTOCOL_CALLBACK_MSG_STRUCT;

/*
    Satellite Protocol Handle definition

    The caller should check for handle validity by
    comparing it to STI_PROTOCOL_INVALID_HDL.
*/

typedef void * STI_PROTOCOL_HDL;
#define STI_PROTOCOL_INVALID_HDL ((STI_PROTOCOL_HDL)NULL)

/*
    STI transmit/receive payload callback definition.
    This is a callback which is called whenever callbacks are
    trigger over the connection for which it is provided to.
*/

typedef BOOLEAN (*STI_PROTOCOL_CALLBACK)(
    STI_HDL hSTI,
    STI_PROTOCOL_HDL hProtocol,
    STI_PROTOCOL_CALLBACK_MSG_STRUCT const *psCallback
        );

/*
    These are STI-protocol events which are to be handled
    by the STI framework and passed on to the STI-protocol
    implementations.
*/
typedef enum sti_protocol_event_enum
{
    STI_PROTOCOL_EVENT_RX_PAYLOAD,
    STI_PROTOCOL_EVENT_TX_PAYLOAD,
    STI_PROTOCOL_EVENT_OOB,
    STI_PROTOCOL_EVENT_TIMEOUT,
    STI_PROTOCOL_EVENT_ERROR,
    STI_PROTOCOL_EVENT_PROTOCOL_SPECIFIC,
    STI_PROTOCOL_EVENT_SHUTDOWN,
    STI_PROTOCOL_EVENT_RX_SHUTDOWN

} STI_PROTOCOL_EVENT_ENUM;

/*
    STI Protocol Event Structure

    This is an event structure used when posting to the protocol's
    queue. As usual it is a big union of all the possible event data.
*/

typedef struct sti_protocol_event_struct
{
    // Event Type
    STI_PROTOCOL_EVENT_ENUM eType;

    union
    {
        STI_PROTOCOL_TX_PAYLOAD_STRUCT sTx;
        STI_PROTOCOL_RX_PAYLOAD_STRUCT sRx;
        STI_PROTOCOL_OOB_STRUCT sOOB;
        STI_PROTOCOL_SPECIFIC_STRUCT sProto;
        STI_PROTOCOL_TIMEOUT_STRUCT sTimeout;
        STI_PROTOCOL_ERROR_STRUCT sError;

    } sMsg;

} STI_PROTOCOL_EVENT_STRUCT;


/*
    Generic Protocol Handler Prototypes

    These are function prototypes which must be used to provide an
    appropriate interface to various protocol handler implementations.
*/

// STI Protocol Connection Init Handler
#define STI_PROTOCOL_CONNECTION_INIT_HANDLER(x)     \
    void * (x)                                      \
    (                                               \
        const char *pacName,                        \
        STI_PROTOCOL_HDL hProtocol,                 \
        STI_HDL hSTI,                               \
        va_list *ptArgList                          \
    )
typedef STI_PROTOCOL_CONNECTION_INIT_HANDLER \
    (*STI_PROTOCOL_CONNECTION_INIT_HANDLER_PROTOTYPE);

// STI Protocol Connection Uninit Handler
#define STI_PROTOCOL_CONNECTION_UNINIT_HANDLER(x)   \
    void (x)                                        \
    (                                               \
        void *pvProtocolConnectionData              \
    )
typedef STI_PROTOCOL_CONNECTION_UNINIT_HANDLER \
    (*STI_PROTOCOL_CONNECTION_UNINIT_HANDLER_PROTOTYPE);

// STI Protocol UnInit Handler
#define STI_PROTOCOL_UNINIT_HANDLER(x)              \
    void (x)                                        \
    (                                               \
        STI_PROTOCOL_HDL hProtocol,                 \
        void *pvProtocolData                        \
    )
typedef STI_PROTOCOL_UNINIT_HANDLER \
    (*STI_PROTOCOL_UNINIT_HANDLER_PROTOTYPE);

// STI Protocol Handlers

#define STI_PROTOCOL_TXRX_INIT_HANDLER(x)           \
    BOOLEAN (x)                                     \
    (                                               \
        const char *pacTaskName,                    \
        STI_PROTOCOL_HDL hProtocol,                 \
        void *pvDevice,                             \
        UN32 *pun32EventQueueSize,                  \
        OSAL_OBJECT_HDL *phTxBlockPool,             \
        OSAL_OBJECT_HDL *phRxBlockPool,             \
        void **ppvProtocolData                      \
    )
typedef STI_PROTOCOL_TXRX_INIT_HANDLER \
    (*STI_PROTOCOL_TXRX_INIT_HANDLER_PROTOTYPE);

#define STI_PROTOCOL_TXRX_HANDLER(x)                \
    N32 (x)                                         \
    (                                               \
        STI_PROTOCOL_HDL hProtocol,                 \
        void *pvProtocolData,                       \
        STI_PROTOCOL_EVENT_STRUCT *psEvent          \
    )
typedef STI_PROTOCOL_TXRX_HANDLER \
    (*STI_PROTOCOL_TXRX_HANDLER_PROTOTYPE);

#define STI_PROTOCOL_RX_INIT_HANDLER(x)             \
    BOOLEAN (x)                                     \
    (                                               \
        const char *pacTaskName,                    \
        STI_PROTOCOL_HDL hProtocol,                 \
        void *pvDevice,                             \
        N32 n32TimeoutMsec,                         \
        void **ppvProtocolData                      \
    )
typedef STI_PROTOCOL_RX_INIT_HANDLER \
    (*STI_PROTOCOL_RX_INIT_HANDLER_PROTOTYPE);

#define STI_PROTOCOL_RX_HANDLER(x)                  \
    N32 (x)                                         \
    (                                               \
        STI_PROTOCOL_HDL hProtocol,                 \
        void *pvProtocolData                        \
    )
typedef STI_PROTOCOL_RX_HANDLER \
    (*STI_PROTOCOL_RX_HANDLER_PROTOTYPE);

/*
    Protocol 'Connection' Handler Interface

    Protocol interface implementations must either implement the function
    or provide a NULL pointer for the function.
*/

typedef struct sti_protocol_connection_handler_struct
{
    // Initialize connection
    STI_PROTOCOL_CONNECTION_INIT_HANDLER_PROTOTYPE _pvInit;

    // Uninitialize connection
    STI_PROTOCOL_CONNECTION_UNINIT_HANDLER_PROTOTYPE _vUninit;

    // Function to call when a connection needs to be compared
    OSAL_LL_COMPARE_HANDLER _n16Compare;

} STI_PROTOCOL_CONNECTION_HANDLER_STRUCT;

/*
    Protocol 'Transceiver' Handler Interface

    Protocol interface implementations must either implement the function
    or provide a NULL pointer for the function.
*/

typedef struct sti_protocol_txrx_handler_struct
{
    // Stack size required (bytes)
    UN32 un32StackSize;

    // Function to Initialize the protocol
    STI_PROTOCOL_TXRX_INIT_HANDLER_PROTOTYPE _bInit;

    // Function to Uninitialize the protocol
    STI_PROTOCOL_UNINIT_HANDLER_PROTOTYPE _vUninit;

    // Function to run the protocol(TxRx Hdlr)
    STI_PROTOCOL_TXRX_HANDLER_PROTOTYPE _n32Hdlr;

} STI_PROTOCOL_TXRX_HANDLER_STRUCT;

/*
    Protocol 'Receiver' Handler Interface

    Protocol interface implementations must either implement the function
    or provide a NULL pointer for the function.
*/

typedef struct sti_protocol_rx_handler_struct
{
    // Stack size required (bytes)
    UN32 un32StackSize;

    // Function to Initialize the protocol
    STI_PROTOCOL_RX_INIT_HANDLER_PROTOTYPE _bInit;

    // Function to Uninitialize the protocol
    STI_PROTOCOL_UNINIT_HANDLER_PROTOTYPE _vUninit;

    // Function to run the protocol(Rx Hdlr)
    STI_PROTOCOL_RX_HANDLER_PROTOTYPE _n32Hdlr;

} STI_PROTOCOL_RX_HANDLER_STRUCT;

/*
    Generic Protocol Interface (must be provided by each protocol)

    This combines all the previous handler interface structures.
*/

typedef struct sti_protocol_intf
{
    // Protocol Name
    const char *pacName;

    // Protocol Connection Interface
    STI_PROTOCOL_CONNECTION_HANDLER_STRUCT sConnection;

    // Transceiver Interface
    STI_PROTOCOL_TXRX_HANDLER_STRUCT sTxRx;

    // Receiver Interface
    STI_PROTOCOL_RX_HANDLER_STRUCT sRx;

} STI_PROTOCOL_INTERFACE;

  /*****************************/
 /** GLOBAL STRUCT TEMPLATES **/
/*****************************/

  /****************************/
 /** GLOBAL UNION TEMPLATES **/
/****************************/

  /**********************/
 /** GLOBAL VARIABLES **/
/**********************/

  /*******************/
 /** GLOBAL MACROS **/
/*******************/

  /***********************/
 /** GLOBAL PROTOTYPES **/
/***********************/

/*
    Called by system (init, uninit)
*/

BOOLEAN STI_bInit( void );

void STI_vUninit( void );

/*
    Called by Applications (connect, disconnect)
*/

STI_HDL STI_hConnect (
    const char *pacName,
    STI_PROTOCOL_CALLBACK vCallback,
    void *pvCallbackArg,
    void *pvConnectionArgument,
    const STI_PROTOCOL_INTERFACE *psProtocolIntf,
    void *pvDevice,
    ...
        );

BOOLEAN STI_bDisconnect ( STI_HDL hSTI );

OSAL_BUFFER_HDL STI_hAllocatePayload ( STI_HDL hSTI );

BOOLEAN STI_bFreePayload ( OSAL_BUFFER_HDL hPayload );

/*
    Called by protocol specific drivers
*/

BOOLEAN STIP_bGetProtocolData(
    STI_PROTOCOL_HDL hProtocol,
    void **ppvProtocolData
        );

OSAL_BUFFER_HDL STIP_hAllocateRxPayload (
    STI_PROTOCOL_HDL hProtocol
        );

BOOLEAN STIP_bReceivePayload (
    STI_PROTOCOL_HDL hProtocol,
    BOOLEAN bBlock,
    OSAL_BUFFER_HDL hPayload
        );

BOOLEAN STIP_bPostOOB (
    STI_PROTOCOL_HDL hProtocol,
    void *pvProtocolConnectionData,
    BOOLEAN bBlock,
    STI_OOB_TYPE_ENUM eType,
    void *pvEventData
        );

BOOLEAN STI_bReceivePayload (
    STI_PROTOCOL_HDL hProtocol,
    void *pvProtocolConnectionData,
    STI_PROTOCOL_RX_PAYLOAD_STRUCT *psRx
        );

BOOLEAN STI_bTransmitPayload (
    STI_HDL hSTI,
    STI_PROTOCOL_TX_PAYLOAD_STRUCT *psTx
        );

BOOLEAN STI_bSendTimeout (
    STI_HDL hSTI
        );

BOOLEAN STI_bReceiveTimeout (
    STI_HDL hSTI
        );

BOOLEAN STI_bSendOOB (
    STI_HDL hSTI,
    STI_OOB_TYPE_ENUM eOOBType,
    void *pvEventData
        );

BOOLEAN STI_bReceiveOOB (
    STI_PROTOCOL_HDL hProtocol,
    void *pvProtocolConnectionData,
    STI_OOB_TYPE_ENUM eOOBType,
    void *pvData
        );

void *STI_pvGetConnectionArgument ( STI_HDL hSTI );

BOOLEAN STI_bSetError (
    STI_HDL hSTI
        );

STI_PROTOCOL_RESULT_CODE_ENUM STI_eSendPayload (
    STI_HDL hSTI,
    BOOLEAN bBlock,
    OSAL_BUFFER_HDL hPayload,
    UN8 un8Retries,
    N32 n32Timeout
        );

#endif /* STI_API_H */
