/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio Inc.               */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio Inc.      */
/*                           Proprietary & Confidential                       */
/******************************************************************************/
/*******************************************************************************
*
*
*
*
*
*******************************************************************************/

#include <string.h>
#include <stdio.h>

#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "sms_version.h"
#include "sms_obj.h"

#include <MW_Transmogrifier.h>
#include <MWT_Output_Files.h>
#include <MWT_SiriusReceiverFrameDecoder.h>
#ifndef NO_SUPPORT_PNG
#include <MWT_WSIRLEToPNG.h>
#endif
#include <MWT_WSIRLEToBMP.h>
#include <MWT_WSIRLEToRAW.h>

#include "wsi_platform_port.h"
#include "_wsi_platform_port.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   hInit
*
*   This function will create all the basics needed for this WSI decoder to
*   operate.
*
*****************************************************************************/
static WSI_DECODER_OBJECT hInit (
    SMS_OBJECT hParent,
    IMAGE_FORMAT_ENUM eFormat
            )
{
    WSI_DECODER_OBJECT_STRUCT *psObj = NULL;
    WSI_DECODER_OBJECT hResult = WSI_DECODER_INVALID_OBJECT;
    TMW_ResultCode MWRes;

    do
    {
        // Create the object
        psObj = (WSI_DECODER_OBJECT_STRUCT*)SMSO_hCreate(
                    WSI_DECODER_OBJECT_NAME": Object",
                    sizeof(WSI_DECODER_OBJECT_STRUCT),
                    SMS_INVALID_OBJECT,
                    FALSE);

        if (psObj == NULL)
        {
            break;
        }

        // Initialize
        psObj->pWsiInputData = NULL;
        psObj->un32WsiInputDataSize = 0;
        psObj->un8PlaneNum = 2;

        // Choose proper raster convert function based on the requested format
        switch (eFormat)
        {
            case IMAGE_FORMAT_PNG:
            {
#ifndef NO_SUPPORT_PNG
                psObj->tRasterConvertFunc =
                    MWTWSIRLEToPNG_convertRasterProductFile;
#endif
            }
            break;
            case IMAGE_FORMAT_BMP:
            {
                psObj->tRasterConvertFunc =
                    MWTWSIRLEToBitmap_convertRasterProductFile;
            }
            break;
            case IMAGE_FORMAT_RAW:
            {
                psObj->tRasterConvertFunc =
                    MWTWSIRLEToRAW_convertRasterProductFile;
            }
            break;
            case IMAGE_FORMAT_JPEG:
            case IMAGE_INVALID_FORMAT:
            {
                psObj->tRasterConvertFunc = NULL;
            }
            break;
        }

        if (psObj->tRasterConvertFunc == NULL)
        {
            break;
        }

        // Register decoder callback function
        MWRes = MWTransmogrifier_setOutputCallback(&MWOutputFiles_outputBytes);
        if (MWRes != MWRes_Success)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WSI_DECODER_OBJECT_NAME
                    ": failed to set library output callback to file "
#if SMS_DEBUG == 1
                    "output function, error %s\n",
                    pacGetReturnCodeName(MWRes)
#else
                    "output function, error %d\n", (int)MWRes
#endif
                    );

            break;
        }

        // Populate the object to the caller
        hResult = (WSI_DECODER_OBJECT) psObj;

    } while (FALSE);

    if ((hResult == WSI_DECODER_INVALID_OBJECT) && (psObj != NULL))
    {
        // Error!
        vUninit((WSI_DECODER_OBJECT)psObj);
    }

    return hResult;
}

/*****************************************************************************
*
*   bDecodePayload
*
*  Decodes and decompresses AGW access unit and saves resulting file to
*  the required file on the file system for further process
*
*  Inputs:
*
*    hPayload - AGW Access Unit for decoding
*    paOutputFile - path to file for decompressed and deframed data
*
*  Outputs:
*    TRUE - if decoding complete successfully and file can be processed
*    FALSE - in case of error
*
*****************************************************************************/
static BOOLEAN bDecodePayload(
    WSI_DECODER_OBJECT hWsiDecoder,
    OSAL_BUFFER_HDL hPayload,
    const char *paOutputFile
        )
{
    BOOLEAN bSuccess = FALSE;
    WSI_DECODER_OBJECT_STRUCT *psObj =
        (WSI_DECODER_OBJECT_STRUCT *)hWsiDecoder;
    TMW_ResultCode MWRes;
    char *paRxOutFilePath = psObj->aBuffer;

#if SMS_DEBUG==1
    char aDeframedLogFilePath[WSI_DECODER_SHARED_BUFFER_SIZE] = { 0 };
#endif

    size_t tPayloadSize = OSAL.tBufferGetSize(hPayload);

    if ((psObj->pWsiInputData != NULL) &&
        (tPayloadSize > psObj->un32WsiInputDataSize))
    {
        // Release old buffer if it doesn't fit for new data
        SMSO_vDestroy((SMS_OBJECT)psObj->pWsiInputData);
        psObj->pWsiInputData = NULL;
        psObj->un32WsiInputDataSize = 0;
    }

    if (psObj->pWsiInputData == NULL)
    {
        // Allocate new buffer if it absent
        psObj->pWsiInputData = (char*)SMSO_hCreate(
                WSI_DECODER_OBJECT_NAME": RAW input buffer",
                tPayloadSize,
                (SMS_OBJECT)hWsiDecoder,
                FALSE);
        psObj->un32WsiInputDataSize = tPayloadSize;
    }

    do
    {
        if (psObj->pWsiInputData == NULL)
        {
            break;
        }

        // Read all payload data into the raw buffer
        if (OSAL.tBufferReadHead(hPayload, psObj->pWsiInputData, tPayloadSize) != tPayloadSize)
        {
            puts(WSI_DECODER_OBJECT_NAME
                 ": Unable to re-copy payload data to separate buffer\n");
            break;
        }

        // Create the name for the rxout file path
        snprintf(paRxOutFilePath, WSI_DECODER_SHARED_BUFFER_SIZE,
                 "%s%s", paOutputFile, WSI_DECODER_RXOUT_EXT);

        //======================================================================
        // DECODING PRODUCT ACCESS UNIT
        //======================================================================

        // Set output file path in output module
        MWRes = MWOutputFiles_setFilePath(paRxOutFilePath);
        if (MWRes != MWRes_Success)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WSI_DECODER_OBJECT_NAME"Failed to create receiver "
#if SMS_DEBUG == 1
                    "decompression output file '%s' (%s)",
                    paRxOutFilePath, pacGetReturnCodeName(MWRes));
#else
                    "decompression output file '%s' (%d)",
                    paRxOutFilePath, (int)MWRes);
#endif
            break;
        }

        // Pass product buffer into library to perform testing if input
        // is in broadcast format and decoder it
        MWRes = MWTransmogrifier_processProduct(
                    (TFourByteInteger_Unsigned)tPayloadSize,
                    (TByte_Unsigned*)psObj->pWsiInputData
                        );

        // Done with output file
        MWOutputFiles_endFile();

        // Handle result of processing
        if (MWRes != MWRes_Success)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WSI_DECODER_OBJECT_NAME
#if SMS_DEBUG == 1
                    ": decompression to file '%s' (%s)",
                    paRxOutFilePath, pacGetReturnCodeName(MWRes));
#else
                    ": decompression to file '%s' (%d)",
                    paRxOutFilePath, (int)MWRes);
#endif
            break;
        }
        else
        {
            printf(WSI_DECODER_OBJECT_NAME
                    ": decompression completed and saved to %s.\n",
                    paRxOutFilePath);
        }

        //======================================================================
        // DEFRAMING DATA PRODUCT UNIT (DPU)
        //======================================================================
#if SMS_DEBUG==1
        // Set up log file path
        snprintf(aDeframedLogFilePath, WSI_DECODER_SHARED_BUFFER_SIZE,
                 "%s%s",
                 paOutputFile, WSI_DECODER_RXOUT_DEFRAMED_LOG_EXT
                     );
#endif
        MWTSiriusFrameDecoder_reset();
        MWRes = MWTSiriusFrameDecoder_deframeProductFile(
            paRxOutFilePath,      /* Input decoded file with DPUs */
            paOutputFile,         /* Output deframed data file path */
#if SMS_DEBUG==1
            TRUE,                 /* Print out header info to log file */
            aDeframedLogFilePath  /* Output log file path */
#else
            FALSE,                /* Print out header info to log file */
            NULL                  /* Output log file path */
#endif
        );

        if (MWRes == MWRes_Success)
        {
            printf (WSI_DECODER_OBJECT_NAME
                    ": deframed file save to '%s'\n",
                    paOutputFile);
        }
        else
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WSI_DECODER_OBJECT_NAME
#if SMS_DEBUG == 1
                    ": failed to deframe rxout file to '%s' (%s)",
                    paOutputFile, pacGetReturnCodeName(MWRes));
#else
                    ": failed to deframe rxout file to '%s' (%d)",
                    paOutputFile, (int)MWRes);
#endif
            break;
        }

        bSuccess = TRUE;
    } while (FALSE);

    // Try to remove rxout file since it is not longer required
    remove(paRxOutFilePath);
    if (bSuccess == FALSE)
    {
        // Try to remove file in case of any error since
        // the WSI decoider
        remove(paOutputFile);
    }

    return bSuccess;
}

/*****************************************************************************
*
*   bGenerateRaster
*
*  Generates raster file based on the file with the deframed product raster
*  data in. If the function cannot generate raster no one file will be created.
*
*  Inputs:
*
*    hPayload - AGW Access Unit for decoding
*    paDeframedFile - path to the file with input data
*    paOutputFile - path to file for raster file
*
*  Outputs:
*    TRUE - if decoding complete successfully and file can be processed
*    FALSE - in case of error
*
*****************************************************************************/
static BOOLEAN bGenerateRaster(
    WSI_DECODER_OBJECT hWsiDecoder,
    const char *paDeframedFile,
    const char *paOutputFile
        )
{
    BOOLEAN bSuccess = FALSE;

    if (hWsiDecoder != WSI_DECODER_INVALID_OBJECT)
    {
        WSI_DECODER_OBJECT_STRUCT *psObj =
            (WSI_DECODER_OBJECT_STRUCT *)hWsiDecoder;
        TMW_ResultCode MWRes;

        // Do conversion
        MWRes = psObj->tRasterConvertFunc(
                        paDeframedFile,
                        paOutputFile,
                        WSI_RLE_MWCompressionType,
                        psObj->un8PlaneNum);

        if (MWRes != MWRes_Success)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WSI_DECODER_OBJECT_NAME
#if SMS_DEBUG == 1
                    ": error converting file '%s' to raster (%s [%0x])\n",
                    paDeframedFile, pacGetReturnCodeName(MWRes), MWRes);
#else
                    ": error converting file '%s' to raster (%0d)\n",
                    paDeframedFile, (int)MWRes);
#endif

            // Remove raster file since in vase of error the WSI decoder might
            // create it as empty.
            remove(paOutputFile);

            // Bad news, everyone...
            bSuccess = FALSE;
        }
        else
        {
            printf(WSI_DECODER_OBJECT_NAME": file '%s' converted into raster '%s'\n",
                    paDeframedFile, paOutputFile);

            bSuccess = TRUE;
        }
    }

    return bSuccess;
}

/*****************************************************************************
*
*   bSetPlane
*
*   This function sets Plane number which will be decoded
*   Applicable for WINDS product type only
*
*****************************************************************************/
static BOOLEAN bSetPlane(
    WSI_DECODER_OBJECT hWsiDecoder,
    UN8 un8Plane
        )
{
    BOOLEAN bResult = FALSE;

    if (hWsiDecoder != WSI_DECODER_INVALID_OBJECT)
    {
        WSI_DECODER_OBJECT_STRUCT *psObj =
            (WSI_DECODER_OBJECT_STRUCT *)hWsiDecoder;

        if (psObj->un8PlaneNum <= 2)
        {
            psObj->un8PlaneNum = un8Plane;
            bResult = TRUE;
        }
    }

    return bResult;
}

/*****************************************************************************
*
*   vUninit
*
*   This function uninitializes the WSI decoder
*
*****************************************************************************/
static void vUninit (
       WSI_DECODER_OBJECT hWsiDecoder
            )
{
    WSI_DECODER_OBJECT_STRUCT *psObj =
        (WSI_DECODER_OBJECT_STRUCT *)hWsiDecoder;

    if (hWsiDecoder != WSI_DECODER_INVALID_OBJECT)
    {
        MWTransmogrifier_setOutputCallback(NULL);

        if (psObj->pWsiInputData != NULL)
        {
            SMSO_vDestroy((SMS_OBJECT)psObj->pWsiInputData);
            psObj->pWsiInputData = NULL;
            psObj->un32WsiInputDataSize = 0;
        }

        SMSO_vDestroy((SMS_OBJECT)hWsiDecoder);
    }

    return;
}

/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

#if SMS_DEBUG == 1
/*******************************************************************************
*
*   pacGetReturnCodeName
*
*******************************************************************************/
static const char *pacGetReturnCodeName (
    TMW_ResultCode eResultCode
        )
{
    const char *pacReturnCodeName = "Unknown";
    UN16 un16Index;
    UN16 un16MapSize;

    un16MapSize = sizeof(GsResultCodeMap) / sizeof(GsResultCodeMap[0]);

    for (un16Index = 0; un16Index < un16MapSize; ++un16Index)
    {
        if (GsResultCodeMap[un16Index].eResultCode == eResultCode)
        {
            pacReturnCodeName = GsResultCodeMap[un16Index].pacText;
            break;
        }
    }

    return pacReturnCodeName;
}
#endif
