/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:ALERT_MGR implementation for the
 *  Simple Module Services (SMS)
 *
 ******************************************************************************/
#include <stdio.h>
#include <string.h>
#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "sms_obj.h"
#include "sql_interface_obj.h"
#include "dataservice_mgr_impl.h"
#include "string_obj.h"
#include "location_obj.h"
#include "locid_obj.h"
#include "rfd_interface_obj.h"
#include "db_util.h"
#include "baudot.h"
#include "dsrl_target_obj.h"
#include "dsrl_entry_obj.h"
#include "ws_alert_msg_obj.h"
#include "ws_alerts_location_obj.h"
#include "ws_alerts_db_constants.h"
#include "ws_alerts_mgr_obj.h"
#include "_ws_alerts_mgr_obj.h"
#include "shape_obj.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
 *                           PUBLIC FUNCTIONS
 *****************************************************************************/

/*****************************************************************************
 *
 *   hStart
 *
 *   This function will create all the basics needed for this service to
 *   operate.  However, all initial processing to actually get this service
 *   running is done at a later time.
 *
 *****************************************************************************/
static WS_ALERTS_SERVICE_OBJECT hStart(
    const char *pacSRHDriverName,
    DATASERVICE_EVENT_MASK tEventRequestMask,
    DATASERVICE_EVENT_CALLBACK vEventCallback,
    void *pvAppEventCallbackArg,
    DATASERVICE_OPTIONS_STRUCT const *psOptions
        )
{
    WS_ALERTS_MGR_OBJECT_STRUCT *psObj =
        ( WS_ALERTS_MGR_OBJECT_STRUCT * )WS_ALERTS_SERVICE_INVALID_OBJECT;
    DATASERVICE_CREATE_STRUCT sCreate;
    BOOLEAN bOk = FALSE;
    BOOLEAN bSuccess = FALSE;
    DATASERVICE_OPTION_VALUES_STRUCT sOptionValues;

    // Process options
    bOk = DATASERVICE_IMPL_bProcessOptions(
              WS_ALERTS_SUPPORTED_OPTIONS, psOptions, &sOptionValues);
    if( bOk == FALSE )
    {
        // Bad options!
        return WS_ALERTS_SERVICE_INVALID_OBJECT;
    }

    // Populate our data service creation structure
    DATASERVICE_IMPL_vInitCreateStruct( &sCreate );
    sCreate.pacSRHDriverName = pacSRHDriverName;
    sCreate.pacServiceObjectName = WS_ALERTS_MGR_OBJECT_NAME;
    sCreate.tServiceObjectSize = sizeof(WS_ALERTS_MGR_OBJECT_STRUCT);
    sCreate.tDataID = (DATASERVICE_ID)GsWSAlertsIntf.tDSI;

    // Get the interface to tell us how much memory
    // this service needs at a minimum for the given
    // startup options
    sCreate.tSuggestedOTABufferByteSize =
        GsWSAlertsIntf.tMinimumOTABufferByteSize(sOptionValues.bUpdateRefDB);

    // Configure the data service's static event attributes
    sCreate.vEventCallback = vEventHandler;
    sCreate.tEventRequestMask = (
                                    DATASERVICE_EVENT_ALL |
                                    DATASERVICE_INTERNAL_EVENT_DSRL );

    // Ask the data service manager controller to
    // create our manager object and do everything
    // necessary to create the underlying objects required
    // in order to support this service
    psObj = ( WS_ALERTS_MGR_OBJECT_STRUCT * )
            DATASERVICE_IMPL_hCreateNewService( &sCreate );

    if( psObj == NULL )
    {
        // Can't create the service, fail out!
        // Free options memory
        DATASERVICE_IMPL_vFreeOptions(&sOptionValues);

        return WS_ALERTS_SERVICE_INVALID_OBJECT;
    }

    // Update the path and the flag for the reference db
    psObj->pacRefDatabaseDirPath = sOptionValues.pcRefDBPath;
    psObj->bRefDBUpdatesEnabled = sOptionValues.bUpdateRefDB;

    // Initialize manager parameters
    psObj->pacCurRefDatabaseFilePath = NULL;
    psObj->pacNextRefDatabaseFilePath = NULL;
    psObj->hInterface = WS_ALERTS_INTERFACE_INVALID_OBJECT;

    // Initialize asynchronous update configuration
    SMSU_vInitialize(
        &psObj->sEvent,
        psObj,
        DATASERVICE_EVENT_NONE,
        tEventRequestMask,
        ( SMSAPI_OBJECT_EVENT_CALLBACK )vEventCallback,
        pvAppEventCallbackArg );

    bSuccess = bInitAppFacingObject( psObj );

    if( bSuccess == FALSE )
    {
        // Error!
        vUninitObject( psObj, TRUE );
        DATASERVICE_IMPL_vDestroy((DATASERVICE_IMPL_HDL)psObj);
        return WS_ALERTS_SERVICE_INVALID_OBJECT;
    }

    // The service may now start
    bOk = DATASERVICE_IMPL_bStart((DATASERVICE_IMPL_HDL)psObj);

    if( bOk == FALSE )
    {
        // Error!
        vUninitObject( psObj, TRUE );
        DATASERVICE_IMPL_vDestroy((DATASERVICE_IMPL_HDL)psObj);
        return WS_ALERTS_SERVICE_INVALID_OBJECT;
    }

    return ( WS_ALERTS_SERVICE_OBJECT )psObj;
}

/*****************************************************************************
*
*   eGetReferenceDataVersion
*
*****************************************************************************/
DATASERVICE_ERROR_CODE_ENUM eGetReferenceDataVersion (
    const char *pcContainingDirectoryPath,
    DATASERVICE_REF_DATA_VER *ptCurrentRefDataVer,
    DATASERVICE_REF_DATA_VER *ptNextRefDataVer
        )
{
    DATASERVICE_ERROR_CODE_ENUM eReturnCode = DATASERVICE_ERROR_UNKNOWN;
    BOOLEAN bOk;
    char *pacDatabaseFilePathA = NULL,
         *pacDatabaseFilePathB = NULL;

    if (ptCurrentRefDataVer == NULL)
    {
        return DATASERVICE_ERROR_CODE_GENERAL;
    }

    do
    {
        bOk = DB_UTIL_bCreateFilePath(
            pcContainingDirectoryPath,
            WS_ALERTS_DATABASE_FOLDER,
            WS_ALERTS_REF_DATABASE_FILENAMEA,
            &pacDatabaseFilePathA);

        if (bOk != TRUE)
        {
            break;
        }

        bOk = DB_UTIL_bCreateFilePath(
            pcContainingDirectoryPath,
            WS_ALERTS_DATABASE_FOLDER,
            WS_ALERTS_REF_DATABASE_FILENAMEB,
            &pacDatabaseFilePathB);

        if (bOk != TRUE)
        {
            break;
        }

        // Check W/S alerts ref databases
        eReturnCode =
            DB_UTIL_eCheckReferenceBanks (
                &pacDatabaseFilePathA[0],
                &pacDatabaseFilePathB[0],
                n32ExtractDataVersion, NULL,
                GsWSAlertsIntf.tMaxVersionBitlen,
                ptCurrentRefDataVer, ptNextRefDataVer );

    } while (FALSE);

    // Remove database path resources
    if (pacDatabaseFilePathA != NULL)
    {
        SMSO_vDestroy(
            (SMS_OBJECT)pacDatabaseFilePathA);
    }

    if (pacDatabaseFilePathB != NULL)
    {
        SMSO_vDestroy(
            (SMS_OBJECT)pacDatabaseFilePathB);
    }

    return eReturnCode;
}


/*****************************************************************************
 *                           FRIEND FUNCTIONS
 *****************************************************************************/

/*****************************************************************************
*                            PRIVATE FUNCTIONS
*****************************************************************************/

/*****************************************************************************
 *
 *   psGetAppFacingObject
 *
 *****************************************************************************/
static WS_ALERTS_APP_OBJECT_STRUCT *psGetAppFacingObject(
    WS_ALERTS_SERVICE_OBJECT hWSAlertsService
        )
{
    WS_ALERTS_MGR_OBJECT_STRUCT *psObj =
        ( WS_ALERTS_MGR_OBJECT_STRUCT * )hWSAlertsService;
    BOOLEAN bValid, bLocked;

    do
    {
        bValid = DATASERVICE_IMPL_bValid((DATASERVICE_IMPL_HDL)hWSAlertsService );

        if( bValid == FALSE )
        {
            break;
        }

        bLocked = SMSO_bLock( ( SMS_OBJECT )psObj->psAppObj,
            OSAL_OBJ_TIMEOUT_INFINITE);

        if( bLocked == FALSE )
        {
            break;
        }

        return psObj->psAppObj;
    }
    while( FALSE );

    return NULL;
}

/*****************************************************************************
 *
 *   vReleaseAppFacingObject
 *
 *****************************************************************************/
static void vReleaseAppFacingObject(
    WS_ALERTS_APP_OBJECT_STRUCT *psAppObj
        )
{
    if( psAppObj != NULL )
    {
        SMSO_vUnlock( ( SMS_OBJECT )psAppObj );
    }

    return;
}

/*****************************************************************************
 *
 *   vEntryRelease
 *
 *****************************************************************************/
static void vEntryRelease(
    DSRL_ENTRY_OBJECT hDSRLEntry
        )
{
    if (hDSRLEntry != DSRL_ENTRY_INVALID_OBJECT)
    {
        WS_ALERTS_DSRL_ENTRY_DESC_STRUCT *psEntryDesc;
        WS_ALERT_MSG_OBJECT hMsg = WS_ALERT_MSG_INVALID_OBJECT;

        psEntryDesc = (WS_ALERTS_DSRL_ENTRY_DESC_STRUCT*)
            DSRL_ENTRY_pvServiceData(hDSRLEntry);

        if (psEntryDesc != NULL)
        {
            // Remove entry's list of DSRLs
            if (psEntryDesc->hDSRLList != OSAL_INVALID_OBJECT_HDL)
            {
                OSAL_RETURN_CODE_ENUM eReturnCode;

                eReturnCode = OSAL.eLinkedListRemoveAll( psEntryDesc->hDSRLList,
                    NULL );
                if (eReturnCode != OSAL_SUCCESS)
                {
                    SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                        WS_ALERTS_MGR_OBJECT_NAME": cannot cleanup linked list (%s)",
                        OSAL.pacGetReturnCodeName(eReturnCode) );
                }

                eReturnCode = OSAL.eLinkedListDelete( psEntryDesc->hDSRLList );
                if (eReturnCode != OSAL_SUCCESS)
                {
                    SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                        WS_ALERTS_MGR_OBJECT_NAME": cannot delete linked list (%s)",
                        OSAL.pacGetReturnCodeName(eReturnCode) );
                }

                psEntryDesc->hDSRLList = OSAL_INVALID_OBJECT_HDL;
            }
        }

        // Remove WS Alerts message
        hMsg = DSRL_ENTRY.hWSAlertMsg(hDSRLEntry);
        WS_ALERT_MSG_vDestroy(hMsg);
    }

    return;
}

/*****************************************************************************
 *
 *   vLocationDescRelease
 *
 *****************************************************************************/
static void vLocationDescRelease(
    WS_ALERTS_LOCATION_OBJECT hObj
        )
{
    if( hObj != WS_ALERTS_LOCATION_INVALID_OBJECT )
    {
        WS_ALERTS_LOCATION_vDestroy(hObj);
    }

    return;
}

/*****************************************************************************
 *
 *   vReleaseMsgLocDesc
 *
 *****************************************************************************/
static void vReleaseMsgLocDesc(
    WS_ALERTS_MSG_LOC_STRUCT *psMsgLocDesc
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

    if( psMsgLocDesc != NULL )
    {
        if( psMsgLocDesc->hLocationsList != OSAL_INVALID_OBJECT_HDL )
        {
            eReturnCode = OSAL.eLinkedListRemoveAll( psMsgLocDesc->hLocationsList, NULL );
            if( eReturnCode != OSAL_SUCCESS )
            {
                SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                    WS_ALERTS_MGR_OBJECT_NAME": cannot cleanup linked list (%s)",
                    OSAL.pacGetReturnCodeName( eReturnCode )
                );
            }

            eReturnCode = OSAL.eLinkedListDelete( psMsgLocDesc->hLocationsList );
            if( eReturnCode != OSAL_SUCCESS )
            {
                SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                    WS_ALERTS_MGR_OBJECT_NAME": cannot delete linked list (%s)",
                    OSAL.pacGetReturnCodeName( eReturnCode )
                );
            }
        }

        SMSO_vDestroy( ( SMS_OBJECT )psMsgLocDesc );
    }

    return;
}

/*****************************************************************************
 *
 *   bSelectLocationsByRTree
 *
 *   Selects from the Data Base and returns list of locations which
 *   describing rectangle has intersection with specified DSRL location
 *
 *****************************************************************************/
static BOOLEAN bSelectLocationsByRTree(
    WS_ALERTS_APP_OBJECT_STRUCT *psObj,
    LOCATION_OBJECT hLocation
        )
{
    BOOLEAN bResult = FALSE;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

    do
    {
        WS_ALERTS_LOCATION_OBJECT hWSAlertsLocation =
            WS_ALERTS_LOCATION_INVALID_OBJECT;
        OSAL_FIXED_OBJECT hTopRightLat,
            hTopRightLon,
            hBottomLeftLat,
            hBottomLeftLon,
            hCenterLat,
            hCenterLon,
            hOffset;
        OSAL_FIXED_OBJECT_DATA atFixedData[OSAL_FIXED_OBJECT_SIZE * 4];
        OSAL_FIXED_OBJECT_DATA atOffsetFixedData[OSAL_FIXED_OBJECT_SIZE * 1];
        UN8 un8NumFixed = 0;    // Keep track of the fixed objects
        WS_ALERTS_LOCATION_BOX_STRUCT sLocationBox;
        WS_ALERTS_DB_QUERY_OBJECTS_STRUCT sResult;
        WS_ALERTS_LOCATION_IDS_ITERATOR_STRUCT sIterator;
        BOOLEAN bZeroRadius = FALSE;
        SMSAPI_RETURN_CODE_ENUM eReturn;

        OSAL.bMemSet( &sResult, 0, sizeof( sResult ) );

        // value used in DSRL target transformation
        // if its radius is 0.
        hOffset = OSAL_FIXED.hCreateInMemory(
            WS_ALERTS_LOCATION_OFFSET_FIXED_VALUE,
            LOCATION_BINPOINT,
            atOffsetFixedData );

        // Check the lat/lon provided
        hCenterLat = LOCATION.hLat( hLocation );
        hCenterLon = LOCATION.hLon( hLocation );

        // Is the target location populated?
        if( ( hCenterLat == OSAL_FIXED_INVALID_OBJECT ) ||
            ( hCenterLon == OSAL_FIXED_INVALID_OBJECT ) )
        {
            // No -- but that's okay...just hold off for now
            bResult = TRUE;
            break;
        }

        // Check if the location is an area or just single point
        eReturn = LOCATION.eIsPoint(hLocation, &bZeroRadius);
        if(eReturn != SMSAPI_RETURN_CODE_SUCCESS)
        {
            // Location is invalid... Not sure how can it happen at this point
            // but anyway aborting.
            break;
        }

        // Create the objects.  If this fails, LOCATION_bTopRight
        // will result in an error
        hTopRightLat = OSAL_FIXED.hCreateInMemory( 0, 0,
            &atFixedData[OSAL_FIXED_OBJECT_SIZE * ( un8NumFixed++ )] );
        hTopRightLon = OSAL_FIXED.hCreateInMemory( 0, 0,
            &atFixedData[OSAL_FIXED_OBJECT_SIZE * ( un8NumFixed++ )] );

        if( bZeroRadius == FALSE )
        {
            // Get the top right lat, lon
            bResult = LOCATION_bTopRight( hLocation,
                hTopRightLat, hTopRightLon );

            if( bResult == FALSE )
            {
                break;
            }
        }
        else
        {
            // Transform DSRL target into small rectangle if its radius is 0
            // for correct work of algorithms checking areas intersection.
            eReturnCode = OSAL_FIXED.eAdd( hCenterLat, hOffset, hTopRightLat );
            if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WS_ALERTS_MGR_OBJECT_NAME
                    ": failed to calculate TopRightLat (%s)",
                    OSAL.pacGetReturnCodeName(eReturnCode));

                bResult = FALSE;
                break;
            }

            eReturnCode = OSAL_FIXED.eAdd( hCenterLon, hOffset, hTopRightLon );
            if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WS_ALERTS_MGR_OBJECT_NAME
                    ": failed to calculate hTopRightLon (%s)",
                    OSAL.pacGetReturnCodeName(eReturnCode));

                bResult = FALSE;
                break;
            }
        }

        // Create the objects.  If this fails, LOCATION_bBottomLeft
        // will result in an error
        hBottomLeftLat = OSAL_FIXED.hCreateInMemory( 0, 0,
            &atFixedData[OSAL_FIXED_OBJECT_SIZE * ( un8NumFixed++ )] );
        hBottomLeftLon = OSAL_FIXED.hCreateInMemory( 0, 0,
            &atFixedData[OSAL_FIXED_OBJECT_SIZE * ( un8NumFixed++ )] );

        if( bZeroRadius == FALSE )
        {
            // Get the bottom left lat, lon
            bResult = LOCATION_bBottomLeft( hLocation,
                hBottomLeftLat, hBottomLeftLon );

            if( bResult == FALSE )
            {
                break;
            }
        }
        else
        {
            eReturnCode = OSAL_FIXED.eSubtract( hCenterLat, hOffset, hBottomLeftLat );
            if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WS_ALERTS_MGR_OBJECT_NAME
                    ": failed to calculate hBottomLeftLat (%s)",
                    OSAL.pacGetReturnCodeName(eReturnCode));

                bResult = FALSE;
                break;
            }

            eReturnCode = OSAL_FIXED.eSubtract( hCenterLon, hOffset, hBottomLeftLon );
            if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WS_ALERTS_MGR_OBJECT_NAME
                    ": failed to calculate hBottomLeftLon (%s)",
                    OSAL.pacGetReturnCodeName(eReturnCode));

                bResult = FALSE;
                break;
            }
        }

        sLocationBox.n32FixedTopRightLat =
            OSAL_FIXED.n32ScaledValue( hTopRightLat, LOCATION_BINPOINT );
        sLocationBox.n32FixedTopRightLon =
            OSAL_FIXED.n32ScaledValue( hTopRightLon, LOCATION_BINPOINT );
        sLocationBox.n32FixedBottomLeftLat =
            OSAL_FIXED.n32ScaledValue( hBottomLeftLat, LOCATION_BINPOINT );
        sLocationBox.n32FixedhBottomLeftLon =
            OSAL_FIXED.n32ScaledValue( hBottomLeftLon, LOCATION_BINPOINT );

        // set SQL request parameters

        psObj->asBindParams[WS_ALERTS_RTREE_STMT_BOTTOM_LAT_PARAM].eType =
            SQL_BIND_TYPE_UN32;
        psObj->asBindParams[WS_ALERTS_RTREE_STMT_BOTTOM_LAT_PARAM].pvData =
            (void*)(size_t)sLocationBox.n32FixedBottomLeftLat;

        psObj->asBindParams[WS_ALERTS_RTREE_STMT_TOP_LAT_PARAM].eType =
            SQL_BIND_TYPE_UN32;
        psObj->asBindParams[WS_ALERTS_RTREE_STMT_TOP_LAT_PARAM].pvData =
            (void*)(size_t)sLocationBox.n32FixedTopRightLat;

        psObj->asBindParams[WS_ALERTS_RTREE_STMT_LEFT_LON_PARAM].eType =
            SQL_BIND_TYPE_UN32;
        psObj->asBindParams[WS_ALERTS_RTREE_STMT_LEFT_LON_PARAM].pvData =
            (void*)(size_t)sLocationBox.n32FixedhBottomLeftLon;

        psObj->asBindParams[WS_ALERTS_RTREE_STMT_RIGHT_LON_PARAM].eType =
            SQL_BIND_TYPE_UN32;
        psObj->asBindParams[WS_ALERTS_RTREE_STMT_RIGHT_LON_PARAM].pvData =
            (void*)(size_t)sLocationBox.n32FixedTopRightLon;

        // request from Western Hemisphere
        bResult = SQL_INTERFACE.bExecutePreparedStatement(
            psObj->hSQLRefConnection,
            psObj->hSelectIdsFromWestRTreeStmt,
            (SQL_QUERY_RESULT_HANDLER)bDBProcessSelectLocIds,
            psObj->hLocIdList,
            WS_ALERTS_RTREE_STMT_PARAMS_COUNT,
            psObj->asBindParams);

        if( bResult == FALSE )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": SQL request failed" );
            break;
        }

        // request from Eastern Hemisphere
        bResult = SQL_INTERFACE.bExecutePreparedStatement(
            psObj->hSQLRefConnection,
            psObj->hSelectIdsFromEastRTreeStmt,
            (SQL_QUERY_RESULT_HANDLER)bDBProcessSelectLocIds,
            psObj->hLocIdList,
            WS_ALERTS_RTREE_STMT_PARAMS_COUNT,
            psObj->asBindParams);

        if( bResult == FALSE )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": SQL request failed" );
            break;
        }

        sIterator.psObj = psObj;
        sIterator.hWSAlertsLocation = hWSAlertsLocation;

        //iterate list of locations IDs to get locations info from DB
        eReturnCode = OSAL.eLinkedListIterate(
            psObj->hLocIdList,
            (OSAL_LL_ITERATOR_HANDLER)bIterateLocationIds,
            &sIterator);
        if ((eReturnCode != OSAL_SUCCESS) && (eReturnCode != OSAL_NO_OBJECTS))
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": Failed to iterate location ids"
                "list (%s)", OSAL.pacGetReturnCodeName(eReturnCode));
            break;
        }

        // request selecting polygons sent with alerts
        sResult.hObjectsList = psObj->hLocationList;
        sResult.psAppObj = psObj;

        bResult = SQL_INTERFACE.bExecutePreparedStatement(
            psObj->hSQLPersistConnection,
            psObj->hSelectPolygonsStmt,
            (SQL_QUERY_RESULT_HANDLER)bDBProcessSelectPolygons,
            &sResult,
            WS_ALERTS_RTREE_STMT_PARAMS_COUNT,
            psObj->asBindParams);

        if( bResult == FALSE )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": SQL request failed" );
            break;
        }

    } while (FALSE);

    if( psObj->hLocIdList != OSAL_INVALID_OBJECT_HDL )
    {
        eReturnCode = OSAL.eLinkedListRemoveAll( psObj->hLocIdList, NULL );

        if( eReturnCode != OSAL_SUCCESS )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": failed to cleanup"
                " Location IDs list (%s)", OSAL.pacGetReturnCodeName( eReturnCode ) );
        }
    }

    return bResult;
}

/*****************************************************************************
 *
 *   bIterateLocationIds
 *
 *****************************************************************************/
static BOOLEAN bIterateLocationIds (
    LOC_ID tLocId,
    WS_ALERTS_LOCATION_IDS_ITERATOR_STRUCT *psIterator
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

    // Getting location from either pool or DB
    psIterator->hWSAlertsLocation = hGetLocationById(psIterator->psObj, tLocId);
    // and adding it into Locations list
    eReturnCode = OSAL.eLinkedListAdd(psIterator->psObj->hLocationList,
        OSAL_INVALID_LINKED_LIST_ENTRY_PTR,
        psIterator->hWSAlertsLocation );

    if( eReturnCode != OSAL_SUCCESS )
    {
        SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
            WS_ALERTS_MGR_OBJECT_NAME": failed to add"
            " to Location IDs list (%s)", OSAL.pacGetReturnCodeName( eReturnCode ) );
    }

    return TRUE;
}

/*****************************************************************************
 *
 *   bRemoveExtraLocations
 *
 *  Removes location objects from the list
 *  which are not overlapping target area actually.
 *
 *****************************************************************************/
static BOOLEAN bRemoveExtraLocations(
    WS_ALERTS_APP_OBJECT_STRUCT *psObj,
    LOCATION_OBJECT hLocation
        )
{
    BOOLEAN bResult = TRUE;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    SMSAPI_RETURN_CODE_ENUM eSmsReturnCode = SMSAPI_RETURN_CODE_ERROR;
    WS_ALERTS_LOCATION_OBJECT hWSAlertsLocation =
        WS_ALERTS_LOCATION_INVALID_OBJECT;
    SHAPE_OBJECT hIteratedShape = SHAPE_INVALID_OBJECT;
    WS_ALERT_MSG_LOCATION_CROSS_STRUCT sIterateArg;
    OSAL_LINKED_LIST_ENTRY hLocationEntry =
        OSAL_INVALID_LINKED_LIST_ENTRY;
    OSAL_LINKED_LIST_ENTRY hLocationEntryToDelete =
        OSAL_INVALID_LINKED_LIST_ENTRY;

    sIterateArg.hCurrentLocation = hLocation;
    hLocationEntry = OSAL.hLinkedListFirst(psObj->hLocationList,
        (void **)&hWSAlertsLocation);

    while ( hLocationEntry != OSAL_INVALID_LINKED_LIST_ENTRY)
    {
        STRING_OBJECT hText =
            WS_ALERTS_LOCATION.hDescription(hWSAlertsLocation);

        sIterateArg.bCross = FALSE;
        sIterateArg.un8NumOfRayCrossings = 0; //Initial state
        hIteratedShape = WS_ALERTS_LOCATION.hShape(hWSAlertsLocation);

        // Iterate the shape object and find out how many
        // crossings are there.
        eSmsReturnCode = SHAPE_eIterateLines(
            hIteratedShape,
            bShapeIteratorCallbackPoint,
            (void *)&sIterateArg);

        if(eSmsReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": Error in iterating shapes" );
            bResult = FALSE;
            break;
        }

        // Checking if number of crossings is odd
        if( (sIterateArg.un8NumOfRayCrossings % 2) != 0 )
        {
            //the location is inside of the polygon.
            sIterateArg.bCross = TRUE;
        }

        if ( sIterateArg.bCross == FALSE )
        {
            // The location is not inside the polygon but it is still
            // inside rtree area so we should perform an extra check for
            // an area around the current location
            eSmsReturnCode = SHAPE_eIterateLines(
                hIteratedShape,
                bShapeIteratorCallbackSquare,
                (void *)&sIterateArg);

            if( eSmsReturnCode != SMSAPI_RETURN_CODE_SUCCESS )
            {
                SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                    WS_ALERTS_MGR_OBJECT_NAME": Error in iterating lines" );
                bResult = FALSE;
                break;
            }

            if ( sIterateArg.bCross == FALSE )
            {
                // Location is not inside the polygon so we must delete
                // it in order to perform a filtration
                hLocationEntryToDelete = hLocationEntry;
            }
        }

        hLocationEntry = OSAL.hLinkedListNext(hLocationEntry,
            (void **)&hWSAlertsLocation);

        if ( hLocationEntryToDelete != OSAL_INVALID_LINKED_LIST_ENTRY )
        {
            printf(WS_ALERTS_MGR_OBJECT_NAME": Removing entry %s (%d)\n",
                STRING.pacCStr(hText), sIterateArg.un8NumOfRayCrossings);

            eReturnCode = OSAL.eLinkedListRemove(hLocationEntryToDelete);

            if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WS_ALERTS_MGR_OBJECT_NAME": failed to remove location"
                    " from list (%s)",
                    OSAL.pacGetReturnCodeName(eReturnCode) );
                bResult = FALSE;
                break;
            }

            hLocationEntryToDelete = OSAL_INVALID_LINKED_LIST_ENTRY;
        }
        else
        {
            printf(WS_ALERTS_MGR_OBJECT_NAME": Keeping entry %s (%d)\n",
                STRING.pacCStr(hText), sIterateArg.un8NumOfRayCrossings);
        }
    }

    return bResult;
}

/*****************************************************************************
 *
 *   bGetAlertMsgsList
 *
 *   This function returns a list of alerts actual for specified target area
 *
 *   The algorithm is as follows:
 *   1. Get IDs of locations from DB that have intersection with the given area
 *      On this step we search intersection between describing rectangle of location
 *      and DSRL rectangle using Rtree feature of SQLite
 *   2. Create objects for locations with selected Ids
 *   3. Filter locations that are near but actualy do not have intersection
 *      with target area
 *   4. Get from DB IDs of messages using locations’ IDs
 *   5. Create Message objects and place them into the DSRL.
 *
 *****************************************************************************/
static BOOLEAN bGetAlertMsgsList(
    DSRL_TARGET_OBJECT hTarget,
    WS_ALERTS_APP_OBJECT_STRUCT *psAppObj,
    DSRL_OBJECT hDSRL
        )
{
    BOOLEAN bResult = FALSE;
    BOOLEAN bSuccess = FALSE;
    BOOLEAN bIsPolygon = FALSE;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    WS_ALERTS_LOCATION_OBJECT hWSAlertsLocation =
        WS_ALERTS_LOCATION_INVALID_OBJECT;
    OSAL_LINKED_LIST_ENTRY hLocationEntry =
        OSAL_INVALID_LINKED_LIST_ENTRY;
    WS_ALERTS_MSG_LOC_CONT_STRUCT sMsgLocationCont;
    WS_ALERTS_MESSAGE_IDS_ITERATOR_STRUCT sIterator;
    WS_ALERTS_DSRL_DESC_STRUCT *psDSRLDesc = NULL;
    LOC_ID tLocId = 0;
    WS_ALERT_MSG_ID tMsgId = 0;

    do
    {
        WS_ALERTS_DB_QUERY_OBJECTS_STRUCT sResult;
        LOCATION_OBJECT hLocation = LOCATION_INVALID_OBJECT;

        OSAL.bMemSet( &sResult, 0, sizeof( sResult ) );
        OSAL.bMemSet( &sIterator, 0, sizeof( sIterator ) );

        hLocation = DSRL_TARGET.hLocation(hTarget);

        if (hLocation == LOCATION_INVALID_OBJECT)
        {
            break;
        }

        // Phase 1: Using R-Tree in DB to select locations and polygons
        // which are suitable for our target location
        bSuccess = bSelectLocationsByRTree(
            psAppObj,
            hLocation);

        if ( bSuccess == FALSE )
        {
            break;
        }

        // Phase 2: Math filtration is done here
        // using Cohen-Sutherland algorithm
        bSuccess = bRemoveExtraLocations(
            psAppObj,
            hLocation);

        if ( bSuccess == FALSE )
        {
            break;
        }

        OSAL.bMemSet( &sMsgLocationCont, 0, sizeof( sMsgLocationCont ) );
        sMsgLocationCont.hMsgIdList = psAppObj->hMsgIdList;

        // iterate locations to get related alerts
        hLocationEntry =
            OSAL.hLinkedListFirst( psAppObj->hLocationList, ( void ** )&hWSAlertsLocation );

        while( hLocationEntry != OSAL_INVALID_LINKED_LIST_ENTRY )
        {
            sMsgLocationCont.hCurrentLocation = hWSAlertsLocation;
            bSuccess = WS_ALERTS_LOCATION_bIsPolygon( hWSAlertsLocation, &bIsPolygon );

            if( bSuccess == FALSE )
            {
                SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                    WS_ALERTS_MGR_OBJECT_NAME": check for polygon failed");
                break;
            }

            if ( bIsPolygon == TRUE )
            {
                tMsgId = WS_ALERTS_LOCATION_tMsgId( hWSAlertsLocation );
                bSuccess = bCreateMsgIdList( &sMsgLocationCont, tMsgId );

                if ( bSuccess == FALSE )
                {
                    SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                        WS_ALERTS_MGR_OBJECT_NAME": cannot create MsgIdList for polygon");
                    break;
                }
            }
            else
            {
                tLocId = WS_ALERTS_LOCATION_tLocId( hWSAlertsLocation );
                printf (WS_ALERTS_MGR_OBJECT_NAME": tLocId = %d\n", tLocId);

                // Specify a search box within the target area

                psAppObj->asBindParams[WS_ALERTS_SELECT_MSGS_STMT_LOCID_PARAM].eType =
                    SQL_BIND_TYPE_UN32;
                psAppObj->asBindParams[WS_ALERTS_SELECT_MSGS_STMT_LOCID_PARAM].pvData =
                    (void*)(size_t)tLocId;

                bSuccess = SQL_INTERFACE.bExecutePreparedStatement(
                    psAppObj->hSQLPersistConnection,
                    psAppObj->hSelectAlertMsgsByLocStmt,
                    (SQL_QUERY_RESULT_HANDLER)bDBProcessSelectMsgIds,
                    &sMsgLocationCont,
                    WS_ALERTS_SELECT_MSGS_STMT_PARAMS_COUNT,
                    psAppObj->asBindParams);

                if( bSuccess == FALSE )
                {
                    SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                        WS_ALERTS_MGR_OBJECT_NAME": SQL request failed");
                    break;
                }
            }

            hLocationEntry =
                OSAL.hLinkedListNext( hLocationEntry, ( void ** )&hWSAlertsLocation );
        }

        if ( bSuccess == FALSE )
        {
            break;
        }

        eReturnCode = OSAL.eTimeGet(&sIterator.un32CurUTCsec);
        if (eReturnCode == OSAL_SUCCESS)
        {
            printf(WS_ALERTS_MGR_OBJECT_NAME
                " Current time is %d seconds UTC.\n", sIterator.un32CurUTCsec);
        }
        else
        {
            puts(WS_ALERTS_MGR_OBJECT_NAME
                " Failed to get current time.");
            break;
        }

        psDSRLDesc = 
            (WS_ALERTS_DSRL_DESC_STRUCT*)DSRL_pvServiceData(hDSRL);
        if (psDSRLDesc == NULL)
        {
            break;
        }

        sIterator.psObj = psAppObj;
        sIterator.hDSRL = hDSRL;
        sIterator.un32NearestEndTime = WS_ALERTS_CLEAN_UP_TIMER_UTC_SEC_MAX;
        sIterator.bResult = FALSE;

        // Iterate the messageID list to create message
        // objects and place them into DSRL entries
        eReturnCode = OSAL.eLinkedListIterate(
            psAppObj->hMsgIdList,
            (OSAL_LL_ITERATOR_HANDLER)bIterateMsgIds,
            &sIterator);

        if ( (eReturnCode != OSAL_SUCCESS) &&
             (eReturnCode != OSAL_NO_OBJECTS) )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": failed to iterate list");
            break;
        }

        // do we have message with expiration time?
        if ( sIterator.un32NearestEndTime < psAppObj->un32TimerUTCSec )
        {
            // save expiration time to start clean up timer later
            psAppObj->un32TimerUTCSec = sIterator.un32NearestEndTime;
        }
        bResult = TRUE;

    } while( FALSE );

    if( psAppObj->hLocationList != OSAL_INVALID_OBJECT_HDL )
    {
        eReturnCode = OSAL.eLinkedListRemoveAll( psAppObj->hLocationList, NULL );

        if( eReturnCode != OSAL_SUCCESS )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": failed to cleanup"
                " Locations list (%s)", OSAL.pacGetReturnCodeName( eReturnCode ) );
        }
    }

    if( psAppObj->hMsgIdList != OSAL_INVALID_OBJECT_HDL )
    {
        eReturnCode = OSAL.eLinkedListRemoveAll( psAppObj->hMsgIdList,
            (OSAL_LL_RELEASE_HANDLER)vReleaseMsgLocDesc );

        if( eReturnCode != OSAL_SUCCESS )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": failed to cleanup"
                " Message IDs list (%s)", OSAL.pacGetReturnCodeName( eReturnCode ) );
        }
    }

    return bResult;
}

/*******************************************************************************
 *
 *   hCreateMessage
 *
 *******************************************************************************/
static WS_ALERT_MSG_OBJECT hCreateMessage(
    WS_ALERTS_APP_OBJECT_STRUCT *psAppObj,
    OSAL_OBJECT_HDL *phLocationsList,
    WS_ALERT_MSG_ID tMsgID,
    UN32 un32CurrentTime
        )
{
    WS_ALERTS_MSG_ROW_STRUCT sAlertMsgRow;
    OSAL_OBJECT_HDL hMsgTypeList = OSAL_INVALID_OBJECT_HDL;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    WS_ALERT_MSG_OBJECT hResult = WS_ALERT_MSG_INVALID_OBJECT;

    OSAL.bMemSet(&sAlertMsgRow, 0, sizeof(sAlertMsgRow));

    do 
    {
        int iReturn;
        BOOLEAN bSuccess;
        WS_ALERTS_DSRL_ENTRY_DESC_STRUCT *psEntryDesc = NULL;

        // creating SQL request using message ID and time
        iReturn = snprintf(psAppObj->acBuffer, sizeof(psAppObj->acBuffer),
            WS_ALERTS_SELECT_ALERT_MSGS_BY_MSGID, tMsgID, un32CurrentTime);
        if (iReturn <= 0)
        {
            // failed to create the request
            break;
        }

        printf(WS_ALERTS_MGR_OBJECT_NAME": SQL request %s will be performed.\n",
            psAppObj->acBuffer);

        bSuccess = SQL_INTERFACE.bQuery(
            psAppObj->hSQLPersistConnection,
            psAppObj->acBuffer,
            (SQL_QUERY_RESULT_HANDLER)bDBProcessSelectMessages,
            &sAlertMsgRow);
        if(bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": SQL request failed" );
            break;
        }

        if(sAlertMsgRow.bPopulated == FALSE)
        {
            // we didn't receive the data from the query
            break;
        }

        // create the list of types related to the message
        eReturnCode = OSAL.eLinkedListCreate(&hMsgTypeList,
            WS_ALERTS_MGR_OBJECT_NAME":hMsgTypeList",
            NULL,
            OSAL_LL_OPTION_NONE);
        if( eReturnCode != OSAL_SUCCESS )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": cannot create locations "
                "list (%s)", OSAL.pacGetReturnCodeName(eReturnCode));
            break;
        }

        // Adding types of this message to the list
        bSuccess = bAddTypesToMsg(psAppObj, hMsgTypeList, tMsgID);
        if( bSuccess == FALSE )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": Failed to add msg "
                "types to list");
            break;
        }

        //Need to add the rest of the locations that are not in current area
        bSuccess = bAddRestLocationsToMsg(psAppObj, *phLocationsList, tMsgID);
        if( bSuccess == FALSE )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": Failed to add locations to msg");
            break;
        }

        // Create the message object
        hResult = WS_ALERT_MSG_hCreate(
            (SMS_OBJECT)psAppObj,
            sizeof(WS_ALERTS_DSRL_ENTRY_DESC_STRUCT),
            &sAlertMsgRow,
            *phLocationsList,
            hMsgTypeList);
        if(hResult == WS_ALERT_MSG_INVALID_OBJECT)
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": failed to create "
                "message object");
            break;
        }

        // now MSG object is responsible for lists
        hMsgTypeList = OSAL_INVALID_OBJECT_HDL;
        *phLocationsList = OSAL_INVALID_OBJECT_HDL;

        psEntryDesc = 
            (WS_ALERTS_DSRL_ENTRY_DESC_STRUCT *)DSRL_ENTRY_pvServiceData(
            (DSRL_ENTRY_OBJECT)hResult);
        if (psEntryDesc == NULL)
        {
            // service data doesn't exist, let's stop here
            break;
        }

        eReturnCode = OSAL.eLinkedListAdd(psAppObj->hEntriesList, 
            &psEntryDesc->hPoolEntry, hResult);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME 
                ": failed to add new entry to pool (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode));
            break;
        }

        return hResult;
    } while (FALSE);

    // cleanup section

    if (sAlertMsgRow.hMsgText != STRING_INVALID_OBJECT)
    {
        STRING_vDestroy(sAlertMsgRow.hMsgText);
    }

    if (hMsgTypeList != OSAL_INVALID_OBJECT_HDL)
    {
        eReturnCode = OSAL.eLinkedListRemoveAll(hMsgTypeList, 
            (OSAL_LL_RELEASE_HANDLER)STRING_vDestroy);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME 
                ": failed to cleanup msg type list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode));
        }

        eReturnCode = OSAL.eLinkedListDelete(hMsgTypeList);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME 
                ": failed to delete msg type list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode));
        }
    }

    if (hResult != WS_ALERT_MSG_INVALID_OBJECT)
    {
        WS_ALERT_MSG_vDestroy(hResult);
    }

    return WS_ALERT_MSG_INVALID_OBJECT;
}

/*******************************************************************************
 *
 *   bIterateMsgIds
 *
 *******************************************************************************/
static BOOLEAN bIterateMsgIds(
    WS_ALERTS_MSG_LOC_STRUCT *psMsgLocDesc,
    WS_ALERTS_MESSAGE_IDS_ITERATOR_STRUCT *psIterator
        )
{
    BOOLEAN bContinue = FALSE;

    if (psIterator == NULL)
    {
        return FALSE;
    }

    do
    {
        WS_ALERT_MSG_OBJECT hMsg;
        TIME_T tEndTime;
        BOOLEAN bSuccess = FALSE;

        if (psMsgLocDesc == NULL)
        {
            // input parameters are wrong
            break;
        }

        // check if we already have this message in the pool
        hMsg = hSearchEntry(psIterator->psObj, psMsgLocDesc->tMsgId);
        if(hMsg != WS_ALERT_MSG_INVALID_OBJECT)
        {
            // Get message end time
            tEndTime = WS_ALERT_MSG.tEndTime(hMsg);

            // Check if message is expired
            if ((tEndTime != 0) &&
                (tEndTime <= (TIME_T)psIterator->un32CurUTCsec))
            {
                // message expired. OK, then we do not want to add this message
                // to DSRL
                hMsg = WS_ALERT_MSG_INVALID_OBJECT;
                bContinue = TRUE;
                break;
            }
        }
        else
        {
            // need to create the message from DB
            hMsg = hCreateMessage(psIterator->psObj,
                &psMsgLocDesc->hLocationsList, psMsgLocDesc->tMsgId,
                psIterator->un32CurUTCsec);
            if (hMsg == WS_ALERT_MSG_INVALID_OBJECT)
            {
                // something went wrong during the creation
                bContinue = TRUE;
                break;
            }

            // Get message end time
            tEndTime = WS_ALERT_MSG.tEndTime(hMsg);
        }

        // now adding the message to our DSRL
        bSuccess = bEntryAddToDSRL((DSRL_ENTRY_OBJECT)hMsg, psIterator->hDSRL,
            psIterator->psObj->un32TransactionID);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": failed to add entry");
            break;
        }

        // Calculate next DSRL clean up time
        if ((tEndTime != 0) && 
            (tEndTime < (TIME_T)psIterator->un32NearestEndTime))
        {
            psIterator->un32NearestEndTime = (UN32)tEndTime;
        }

        bContinue = TRUE;
    } while (FALSE);

    psIterator->bResult = bContinue;

    return bContinue;
}

/*******************************************************************************
 *
 *   bDBProcessSelectLocIds
 *
 *******************************************************************************/
static BOOLEAN bDBProcessSelectLocIds(
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    OSAL_OBJECT_HDL hLocIdList
        )
{
    LOC_ID tLocId = 0;
    WS_ALERTS_LOCATIONS_ROW_STRUCT sAlertLocationRow;
    BOOLEAN bContinue = TRUE;
    OSAL.bMemSet( &sAlertLocationRow, 0, sizeof( sAlertLocationRow ) );

    do
    {
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

        // Verify input
        if( hLocIdList == OSAL_INVALID_OBJECT_HDL )
        {
            bContinue = FALSE;
            break;
        }

        tLocId = (LOC_ID)psColumn[WS_ALERTS_LOCATIONS_FIELD_LOCID].uData.sUN32.un32Data;
        eReturnCode = OSAL.eLinkedListAdd(
            hLocIdList,
            OSAL_INVALID_LINKED_LIST_ENTRY_PTR,
            ( void * )tLocId );

        if( eReturnCode != OSAL_SUCCESS )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": failed to add location id to a list (%s)",
                OSAL.pacGetReturnCodeName( eReturnCode ) );
            break;
        }
    }
    while( FALSE );

    return bContinue;
}

/*******************************************************************************
 *
 *   bDBProcessSelectLocations
 *
 *******************************************************************************/
static BOOLEAN bDBProcessSelectLocations(
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    WS_ALERTS_DB_RESULT_STRUCT *psResult
        )
{
    BOOLEAN bOk = TRUE;
    size_t tLength = 0;
    WS_ALERTS_DB_ALERT_LOCATION_FIELDS_ENUM eCurrentField = 
        WS_ALERTS_DB_LOCATION_TABINDEX;
    N32 *pn32Lon = NULL;
    N32 *pn32Lat = NULL;
    size_t tIndex = 0;

    // If there is at least one data row and the correct
    // number of columns (just make sure we have enough),
    // then we have good results
    if( (n32NumberOfColumns != WS_ALERTS_DB_LOCATION_MAX_FIELDS) ||
        (psColumn[WS_ALERTS_DB_LOCATION_LON].uData.sBLOB.tSize !=
         psColumn[WS_ALERTS_DB_LOCATION_LAT].uData.sBLOB.tSize ) )
    {
        return FALSE;
    }

    if (psResult->bResult == TRUE)
    {
        printf(WS_ALERTS_MGR_OBJECT_NAME": bDBProcessSelectLocations: location duplicate \n");
    }

    psResult->bResult = TRUE;

    do
    {
        switch( eCurrentField )
        {
            case WS_ALERTS_DB_LOCATION_LTEXT:
            {
                const char *pacLocationDesc;

                pacLocationDesc = STRING.pacCStr(psResult->uDbRow.sAlertLocationRow.hInfo);

                tLength = strlen( ( const char * )
                    psColumn[WS_ALERTS_DB_LOCATION_LTEXT].uData.sCString.pcData );

                psResult->uDbRow.sAlertLocationRow.hInfo = STRING.hCreate(
                    ( const char * )psColumn[WS_ALERTS_DB_LOCATION_LTEXT].uData.sCString.pcData,
                    tLength);

                printf(WS_ALERTS_MGR_OBJECT_NAME": Matching Location %s\n",
                    pacLocationDesc);
            }
            break;
            case WS_ALERTS_DB_LOCATION_EXTREF:
            {
                tLength = strlen( ( const char * )
                    psColumn[WS_ALERTS_DB_LOCATION_EXTREF].uData.sCString.pcData );

                psResult->uDbRow.sAlertLocationRow.hExtRef = STRING.hCreate(
                    ( const char * )psColumn[WS_ALERTS_DB_LOCATION_EXTREF].uData.sCString.pcData,
                    tLength);
            }
            break;
            case WS_ALERTS_DB_LOCATION_EXTTYPE:
            {
                tLength = strlen( ( const char * )
                    psColumn[WS_ALERTS_DB_LOCATION_EXTTYPE].uData.sCString.pcData );

                psResult->uDbRow.sAlertLocationRow.hExtType = STRING.hCreate(
                    ( const char * )psColumn[WS_ALERTS_DB_LOCATION_EXTTYPE].uData.sCString.pcData,
                    tLength);
            }
            break;
            case WS_ALERTS_DB_LOCATION_LON:
            {
                pn32Lon = ( N32 * ) SMSO_hCreate(
                    WS_ALERTS_MGR_OBJECT_NAME":LonObj",
                    psColumn[WS_ALERTS_DB_LOCATION_LON].uData.sBLOB.tSize,
                    (SMS_OBJECT)psResult->psAppObj, FALSE );

                if( pn32Lon == NULL )
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        WS_ALERTS_MGR_OBJECT_NAME": Failed to allocate memory" );
                    bOk = FALSE;
                    break;
                }

                // Copy the points data
                bOk = OSAL.bMemCpy(
                    ( void * )&pn32Lon[0],
                    psColumn[WS_ALERTS_DB_LOCATION_LON].uData.sBLOB.pvData,
                    psColumn[WS_ALERTS_DB_LOCATION_LON].uData.sBLOB.tSize );
            }
            break;
            case WS_ALERTS_DB_LOCATION_LAT:
             {
                 pn32Lat = ( N32 * ) SMSO_hCreate(
                     WS_ALERTS_MGR_OBJECT_NAME":LatObj",
                     psColumn[WS_ALERTS_DB_LOCATION_LAT].uData.sBLOB.tSize,
                     (SMS_OBJECT)psResult->psAppObj, FALSE );

                 if( pn32Lat == NULL )
                 {
                     SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                         WS_ALERTS_MGR_OBJECT_NAME": Failed to allocate memory" );
                     bOk = FALSE;
                     break;
                 }

                 // Copy the points data
                 bOk = OSAL.bMemCpy(
                     ( void * )&pn32Lat[0],
                     psColumn[WS_ALERTS_DB_LOCATION_LAT].uData.sBLOB.pvData,
                     psColumn[WS_ALERTS_DB_LOCATION_LAT].uData.sBLOB.tSize );
             }
             break;
            //unnecessary fields, skip them
            case WS_ALERTS_DB_LOCATION_LCODE:
            case WS_ALERTS_DB_LOCATION_TABINDEX:
            default:
                break;
        }

        if( bOk == FALSE )
        {
            break;
        }
    }
    while( ++eCurrentField < WS_ALERTS_DB_LOCATION_MAX_FIELDS );

    if( bOk == TRUE )
    {
        psResult->uDbRow.sAlertLocationRow.pn32LonLat = ( N32 * ) SMSO_hCreate(
            WS_ALERTS_MGR_OBJECT_NAME":LonLatObj",
            psColumn[WS_ALERTS_DB_LOCATION_LON].uData.sBLOB.tSize +
            psColumn[WS_ALERTS_DB_LOCATION_LAT].uData.sBLOB.tSize,
            (SMS_OBJECT)psResult->psAppObj, FALSE );

        if( psResult->uDbRow.sAlertLocationRow.pn32LonLat != NULL )
        {
            psResult->uDbRow.sAlertLocationRow.tNumVertices =
                ( size_t )( psColumn[WS_ALERTS_DB_LOCATION_LON].uData.sBLOB.tSize
                / WS_ALERTS_DB_LON_BYTE_SIZE );

            for ( tIndex = 0; tIndex < psResult->uDbRow.sAlertLocationRow.tNumVertices; tIndex++)
            {
                //place lon & lat into the even/odd positions of a single array.
                psResult->uDbRow.sAlertLocationRow.pn32LonLat[ 2 * tIndex ] = pn32Lon[ tIndex ];
                psResult->uDbRow.sAlertLocationRow.pn32LonLat[ 2 * tIndex + 1 ] = pn32Lat[ tIndex ];
            }

        }
        else
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": Failed to allocate memory");
        }
    }
    else
    {
        psResult->uDbRow.sAlertLocationRow.pn32LonLat = NULL;
    }

    if( pn32Lon != NULL )
    {
        SMSO_vDestroy( ( SMS_OBJECT )pn32Lon );
    }

    if( pn32Lat != NULL )
    {
        SMSO_vDestroy( ( SMS_OBJECT )pn32Lat );
    }

    return TRUE;
}

/*******************************************************************************
 *
 *   bDBProcessSelectPolygons
 *
 *******************************************************************************/
static BOOLEAN bDBProcessSelectPolygons(
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    WS_ALERTS_DB_QUERY_OBJECTS_STRUCT *psResult
        )
{
    WS_ALERTS_LOCATIONS_ROW_STRUCT sAlertLocationRow;
    WS_ALERTS_LOCATION_OBJECT hWSAlertsLocation =
        WS_ALERTS_LOCATION_INVALID_OBJECT;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    OSAL_LINKED_LIST_ENTRY hLocationEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    BOOLEAN bOk = TRUE, bContinue = TRUE;
    WS_ALERTS_DB_POLYGON_ENUM eCurrentField = WS_ALERTS_POLYGON_POLYGON_ID;

    OSAL.bMemSet( &sAlertLocationRow, 0, sizeof( sAlertLocationRow ) );

    // If there is at least one data row and the correct
    // number of columns (just make sure we have enough),
    // then we have good results
    if( n32NumberOfColumns != WS_ALERTS_DB_POLYGON_MAX_FIELDS )
    {
        return FALSE;
    }

    sAlertLocationRow.bResult = TRUE;

    do
    {
        switch( eCurrentField )
        {

            case WS_ALERTS_POLYGON_POLYGON_ID:
            {
                sAlertLocationRow.tLocId = (LOC_ID)WS_ALERTS_CONSTRUCT_LOCATION_ID_FROM_POLYGON_ID(
                    psColumn[WS_ALERTS_POLYGON_POLYGON_ID].uData.sUN32.un32Data);
                sAlertLocationRow.bIsPolygon = TRUE;

                WS_ALERTS_LOCATION_vSetDummyId(
                    psResult->psAppObj->hDummyLoc, sAlertLocationRow.tLocId);

                eReturnCode = OSAL.eLinkedListSearch(
                    psResult->psAppObj->hLocationPool,
                    &hLocationEntry,
                    psResult->psAppObj->hDummyLoc);

                if( eReturnCode == OSAL_SUCCESS )
                {
                    hWSAlertsLocation =
                            ( WS_ALERTS_LOCATION_OBJECT )OSAL.pvLinkedListThis( hLocationEntry );
                    bContinue = FALSE;
                }
            }
            break;
            case WS_ALERTS_POLYGON_MSG_ID:
            {
                sAlertLocationRow.tMsgId = (WS_ALERT_MSG_ID)
                    psColumn[WS_ALERTS_POLYGON_MSG_ID].uData.sUN32.un32Data;
            }
            break;
            case WS_ALERTS_POLYGON_LONLAT:
            {
                sAlertLocationRow.pn32LonLat =
                    (N32 *) SMSO_hCreate(
                    WS_ALERTS_MGR_OBJECT_NAME":LonLatObj",
                    psColumn[WS_ALERTS_POLYGON_LONLAT].uData.sBLOB.tSize,
                    SMS_INVALID_OBJECT, FALSE );
                // Copy the points data
                bOk = OSAL.bMemCpy(
                    (void *)&sAlertLocationRow.pn32LonLat[0],
                    psColumn[WS_ALERTS_POLYGON_LONLAT].uData.sBLOB.pvData,
                    psColumn[WS_ALERTS_POLYGON_LONLAT].uData.sBLOB.tSize );

                if( bOk == FALSE )
                {
                    SMSO_vDestroy( (SMS_OBJECT)sAlertLocationRow.pn32LonLat );
                    sAlertLocationRow.pn32LonLat = NULL;
                }
                else
                {
                    // each vertex is a pair of 4 byte lon & lat
                    sAlertLocationRow.tNumVertices =
                        (size_t)( psColumn[WS_ALERTS_POLYGON_LONLAT].uData.sBLOB.tSize
                            / 8 );
                }
            }
            break;
            //unnecessary fields, skip them
            case WS_ALERTS_POLYGON_MIN_LON:
            case WS_ALERTS_POLYGON_MAX_LON:
            case WS_ALERTS_POLYGON_MIN_LAT:
            case WS_ALERTS_POLYGON_MAX_LAT:
            default:
                break;
        }

        if( bContinue == FALSE )
        {
            break;
        }
    } while( ++eCurrentField < WS_ALERTS_POLYGON_MAX_FIELDS );

    if( hWSAlertsLocation == WS_ALERTS_LOCATION_INVALID_OBJECT )
    {
        hWSAlertsLocation =
            WS_ALERTS_LOCATION_hCreate(
                (SMS_OBJECT)psResult->psAppObj,
                &sAlertLocationRow );

        if( hWSAlertsLocation == WS_ALERTS_LOCATION_INVALID_OBJECT )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": failed to create wsalert location object");
        }
        else
        {
            eReturnCode = OSAL.eLinkedListAdd(
                psResult->psAppObj->hLocationPool,
                OSAL_INVALID_LINKED_LIST_ENTRY_PTR,
                (void *)hWSAlertsLocation );

            if( eReturnCode != OSAL_SUCCESS )
            {
                SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                    WS_ALERTS_MGR_OBJECT_NAME": failed to put location into the pool(%s)",
                    OSAL.pacGetReturnCodeName( eReturnCode ) );
                WS_ALERTS_LOCATION_vDestroy(hWSAlertsLocation);
            }

            eReturnCode = OSAL.eLinkedListAdd(
                psResult->hObjectsList,
                OSAL_INVALID_LINKED_LIST_ENTRY_PTR,
                (void *)hWSAlertsLocation );

            if( eReturnCode != OSAL_SUCCESS )
            {
                SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                    WS_ALERTS_MGR_OBJECT_NAME": failed to add location to list (%s)",
                    OSAL.pacGetReturnCodeName( eReturnCode ) );
            }
        }
    }
    else
    {
        eReturnCode = OSAL.eLinkedListAdd(
            psResult->hObjectsList,
            OSAL_INVALID_LINKED_LIST_ENTRY_PTR,
            (void *)hWSAlertsLocation );

        if( eReturnCode != OSAL_SUCCESS )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": failed to add location to list (%s)",
                OSAL.pacGetReturnCodeName( eReturnCode ) );
        }
    }

    vRemoveAlertLocationRow(&sAlertLocationRow);
    return TRUE;
}

/*******************************************************************************
 *
 *   bDBProcessSelectMessages
 *
 *******************************************************************************/
static BOOLEAN bDBProcessSelectMessages(
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    WS_ALERTS_MSG_ROW_STRUCT *psResult
        )
{
    size_t tLength = 0;
    WS_ALERTS_MESSAGE_FIELDS_ENUM eCurrentField = 
        WS_ALERTS_MESSAGE_MSG_ID;

    // If there is at least one data row and the correct
    // number of columns (just make sure we have enough),
    // then we have good results
    if( n32NumberOfColumns != WS_ALERTS_MESSAGE_MAX_FIELDS )
    {
        return FALSE;
    }

    psResult->bPopulated = TRUE; //This will be hardcoded here unless one day we
    //will decide not to add the rest of locations in the Message to the list.

    do
    {
        switch( eCurrentField )
        {
            case WS_ALERTS_MESSAGE_MSG_ID:
            {
                psResult->tMsgId = (WS_ALERT_MSG_ID)
                    psColumn[WS_ALERTS_MESSAGE_MSG_ID].uData.sUN32.un32Data;
            }
            break;
            case WS_ALERTS_MESSAGE_STATE_ID:
            {
                psResult->un8StateId =
                    (UN8)psColumn[WS_ALERTS_MESSAGE_STATE_ID].uData.sUN32.un32Data;
            }
            break;
            case WS_ALERTS_MESSAGE_LANG:
            {
                psResult->eMsgLang = (SMS_LANGUAGE_ENUM)
                    psColumn[WS_ALERTS_MESSAGE_LANG].uData.sUN32.un32Data;
            }
            break;
            case WS_ALERTS_MESSAGE_TEXT:
            {
                tLength =
                    strlen( ( const char * )
                        psColumn[WS_ALERTS_MESSAGE_TEXT].uData.sCString.pcData
                          );
                psResult->hMsgText =
                    STRING.hCreate(
                        ( const char * )psColumn[WS_ALERTS_MESSAGE_TEXT].uData.sCString.pcData,
                        tLength);
            }
            break;
            case WS_ALERTS_MESSAGE_START_TIME:
            {
                psResult->tEventStartTime =
                    psColumn[WS_ALERTS_MESSAGE_START_TIME].uData.sUN32.un32Data;
            }
            break;
            case WS_ALERTS_MESSAGE_END_TIME:
            {
                psResult->tEventEndTime =
                    psColumn[WS_ALERTS_MESSAGE_END_TIME].uData.sUN32.un32Data;
            }
            break;
            case WS_ALERTS_MESSAGE_PRIORITY:
            {
                psResult->tPriority =
                    psColumn[WS_ALERTS_MESSAGE_PRIORITY].uData.sUN32.un32Data;
            }
            break;
            case WS_ALERTS_MESSAGE_POLYGON:
            {
                psResult->bPolygon =
                    (BOOLEAN)psColumn[WS_ALERTS_MESSAGE_POLYGON].uData.sUN32.un32Data;
            }
            break;
            //no need to store version and rate, skip them
            case WS_ALERTS_MESSAGE_VERSION:
            case WS_ALERTS_MESSAGE_RATE:
            default:
                break;
        }
    }
    while( ++eCurrentField < WS_ALERTS_MESSAGE_MAX_FIELDS );

    return TRUE;
}

/*******************************************************************************
 *
 *   bDBProcessSelectMsgIds
 *
 *******************************************************************************/
static BOOLEAN bDBProcessSelectMsgIds(
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    WS_ALERTS_MSG_LOC_CONT_STRUCT *psMsgLocCont
        )
{
    WS_ALERT_MSG_ID tMsgId;
    BOOLEAN bOk = FALSE;

    tMsgId = (WS_ALERT_MSG_ID)psColumn[WS_ALERTS_DB_MESSAGE_LOCATIONS_MSG_ID].uData.sUN32.un32Data;
    bOk = bCreateMsgIdList(psMsgLocCont, tMsgId);

    if ( bOk != TRUE )
    {
        printf(WS_ALERTS_MGR_OBJECT_NAME": bDBProcessSelectMsgIds: cannot create MsgIdList\n");
    }

    return TRUE;
}

/*******************************************************************************
 *
 *   bCreateMsgIdsList
 *
 *   creates object storing message Id and its locations and places this
 *   object to a list
 *
 *******************************************************************************/
static BOOLEAN bCreateMsgIdList(
    WS_ALERTS_MSG_LOC_CONT_STRUCT *psMsgLocCont,
    WS_ALERT_MSG_ID tMsgId
        )
{
    OSAL_LINKED_LIST_ENTRY hTempEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    WS_ALERTS_MSG_LOC_STRUCT *psMsgLocDesc = NULL;
    BOOLEAN bSuccess = FALSE;

    do
    {
        // Dummy struct to be used in search
        WS_ALERTS_MSG_LOC_STRUCT sDummyMsgLoc;
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
        OSAL_RETURN_CODE_ENUM eOsalReturnCode = OSAL_ERROR;

        // Verify input
        if( (psMsgLocCont == NULL) ||
            (psMsgLocCont->hMsgIdList == OSAL_INVALID_OBJECT_HDL) )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME
                ": invalid input" );
            break;
        }

        sDummyMsgLoc.tMsgId = tMsgId;

        eOsalReturnCode = OSAL.eLinkedListSearch(
            psMsgLocCont->hMsgIdList,
            &hTempEntry,
            ( void * )&sDummyMsgLoc );

        if( ( eOsalReturnCode == OSAL_OBJECT_NOT_FOUND ) ||
            ( eOsalReturnCode == OSAL_NO_OBJECTS ) )
        {
            psMsgLocDesc = ( WS_ALERTS_MSG_LOC_STRUCT * )
                SMSO_hCreate(
                    WS_ALERTS_MGR_OBJECT_NAME,
                    sizeof(WS_ALERTS_MSG_LOC_STRUCT),
                    SMS_INVALID_OBJECT, FALSE);

            if( psMsgLocDesc == NULL )
            {
                SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                    WS_ALERTS_MGR_OBJECT_NAME
                    ": failed to create message to location desc" );
                break;
            }

            eReturnCode = OSAL.eLinkedListCreate(
                &psMsgLocDesc->hLocationsList,
                WS_ALERTS_MGR_OBJECT_NAME":Message_hLocationList",
                NULL,
                OSAL_LL_OPTION_NONE );

            if( eReturnCode != OSAL_SUCCESS )
            {
                SMSAPI_DEBUG_vPrintErrorFull(
                    gpacThisFile, __LINE__,
                    WS_ALERTS_MGR_OBJECT_NAME": cannot create location list (%s)",
                    OSAL.pacGetReturnCodeName( eReturnCode ) );
                break;
            }

            psMsgLocDesc->tMsgId = tMsgId;

            eReturnCode = OSAL.eLinkedListAdd( psMsgLocCont->hMsgIdList,
                OSAL_INVALID_LINKED_LIST_ENTRY_PTR,
                psMsgLocDesc );

            if( eReturnCode != OSAL_SUCCESS )
            {
                SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                    WS_ALERTS_MGR_OBJECT_NAME": add to msg list failed"
                    " (%s)", OSAL.pacGetReturnCodeName( eReturnCode ) );
                break;
            }

            bSuccess = TRUE;
        }
        else if( eOsalReturnCode == OSAL_SUCCESS )
        {
            // The message is already present in list so there is nothing
            // to do here anymore.
            bSuccess = TRUE;
        }
        else
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": search in sub list failed"
                " (%s)", OSAL.pacGetReturnCodeName( eOsalReturnCode ) );
            break;
        }
    }
    while( FALSE );

    if ( bSuccess == FALSE )
    {
        vReleaseMsgLocDesc( psMsgLocDesc );
    }

    return bSuccess;
}

/*****************************************************************************
 *
 *   n16CompareLocationsByMsgId
 *
 *****************************************************************************/
static N16 n16CompareLocationsByMsgId(
    WS_ALERTS_MSG_LOC_STRUCT *psMsgLoc1,
    WS_ALERTS_MSG_LOC_STRUCT *psMsgLoc2
        )
{
    N16 n16Result = N16_MIN;

    if ((psMsgLoc1 != NULL) && 
        (psMsgLoc2 != NULL))
    {
        n16Result = COMPARE(psMsgLoc1->tMsgId, psMsgLoc2->tMsgId);
    }

    return n16Result;
}

/*****************************************************************************
 *
 *   vRemoveAlertLocationRow
 *
 *****************************************************************************/
static void vRemoveAlertLocationRow(
    WS_ALERTS_LOCATIONS_ROW_STRUCT *psAlertLocationRow
        )
{
    //Remove sAlertLocationRow
    if ( psAlertLocationRow != NULL )
    {
        if (psAlertLocationRow->hInfo != STRING_INVALID_OBJECT)
        {
            STRING_vDestroy(psAlertLocationRow->hInfo);
            psAlertLocationRow->hInfo = STRING_INVALID_OBJECT;
        }

        if (psAlertLocationRow->hExtRef != STRING_INVALID_OBJECT)
        {
            STRING_vDestroy(psAlertLocationRow->hExtRef);
            psAlertLocationRow->hExtRef = STRING_INVALID_OBJECT;
        }

        if (psAlertLocationRow->hExtType != STRING_INVALID_OBJECT)
        {
            STRING_vDestroy(psAlertLocationRow->hExtType);
            psAlertLocationRow->hExtType = STRING_INVALID_OBJECT;
        }

        if( psAlertLocationRow->pn32LonLat != (N32 *)SMS_INVALID_OBJECT )
        {
            SMSO_vDestroy( ( SMS_OBJECT )psAlertLocationRow->pn32LonLat );
            psAlertLocationRow->pn32LonLat = NULL;
        }
    }

    return;
}

/*****************************************************************************
 *
 *  n16CompareMsgByID
 *
 *****************************************************************************/
static N16 n16CompareMsgByID(
    WS_ALERT_MSG_OBJECT hMsg1,
    WS_ALERT_MSG_OBJECT hMsg2
        )
{
    N16 n16Result = N16_MIN;

    if ((hMsg1 != WS_ALERT_MSG_INVALID_OBJECT) && 
        (hMsg2 != WS_ALERT_MSG_INVALID_OBJECT))
    {
        WS_ALERT_MSG_ID tMsgID1, tMsgID2;

        tMsgID1 = WS_ALERTS_MSG_tMsgID(hMsg1);
        tMsgID2 = WS_ALERTS_MSG_tMsgID(hMsg2);
        n16Result = COMPARE(tMsgID1, tMsgID2);
    }

    return n16Result;
}

/*****************************************************************************
 *
 *  n16CompareEntriesByLocID
 *
 *****************************************************************************/
static N16 n16CompareEntriesByLocID(
    WS_ALERTS_LOCATION_OBJECT hWsAlertsLocation1,
    WS_ALERTS_LOCATION_OBJECT hWsAlertsLocation2
        )
{
    LOC_ID tLocId1;
    LOC_ID tLocId2;

    tLocId1 = WS_ALERTS_LOCATION_tLocId(hWsAlertsLocation1);
    tLocId2 = WS_ALERTS_LOCATION_tLocId(hWsAlertsLocation2);

    return COMPARE(tLocId1, tLocId2);
}

/*****************************************************************************
 *
 *   n16DSRLSortByPriority
 *
 *****************************************************************************/
static N16 n16DSRLSortByPriority (
    DSRL_OBJECT hDSRL,
    DSRL_ENTRY_OBJECT hEntry1,
    DSRL_ENTRY_OBJECT hEntry2,
    void *pvSortArg
        )
{
    WS_ALERT_MSG_OBJECT hWSAlertsMsg1 =
        (WS_ALERT_MSG_OBJECT)DSRL_ENTRY.hWSAlertMsg(hEntry1);
    WS_ALERT_MSG_OBJECT hWSAlertsMsg2 =
        (WS_ALERT_MSG_OBJECT)DSRL_ENTRY.hWSAlertMsg(hEntry2);
    UN16 un16Priority1 = WS_ALERT_MSG.un16Priority(hWSAlertsMsg1);
    UN16 un16Priority2 = WS_ALERT_MSG.un16Priority(hWSAlertsMsg2);

    // Priorities can hold values from 0 to 15, so conversion
    // UN16->N16 is considered safe.
    N16 n16Return = ((N16)un16Priority2) - ((N16)un16Priority1);

    return n16Return;
}

/*****************************************************************************
 *
 *   bHandleCreateList
 *
 *****************************************************************************/
static BOOLEAN bHandleCreateList(
    WS_ALERTS_MGR_OBJECT_STRUCT *psObj,
    DSRL_ARG_STRUCT *psDSRLArg
        )
{
    BOOLEAN bResult = FALSE;
    WS_ALERTS_APP_OBJECT_STRUCT *psAppObj = NULL;

    psAppObj = psGetAppFacingObject((WS_ALERTS_SERVICE_OBJECT)psObj);
    if (psAppObj != NULL)
    {
        bResult = bCreateList( psAppObj, psDSRLArg );

        // start timer to remove expired DSRL entries
        if ( psAppObj->un32TimerUTCSec != WS_ALERTS_CLEAN_UP_TIMER_UTC_SEC_MAX )
        {
            vStartCleanUpTimer(psObj, psAppObj->un32TimerUTCSec);
            psAppObj->un32TimerUTCSec = WS_ALERTS_CLEAN_UP_TIMER_UTC_SEC_MAX;
        }

        vReleaseAppFacingObject( psAppObj );
    }

    return bResult;
}

/*****************************************************************************
 *
 *   bHandleModifyList
 *
 *****************************************************************************/
static BOOLEAN bHandleModifyList(
    WS_ALERTS_MGR_OBJECT_STRUCT *psObj,
    DSRL_ARG_STRUCT *psDSRLArg
        )
{
    BOOLEAN bResult = FALSE;
    WS_ALERTS_APP_OBJECT_STRUCT *psAppObj = NULL;

    psAppObj = psGetAppFacingObject((WS_ALERTS_SERVICE_OBJECT)psObj);
    if (psAppObj != NULL)
    {
        bResult = bModifyList( psAppObj, psDSRLArg );

        // start timer to remove expired DSRL entries
        if ( psAppObj->un32TimerUTCSec != WS_ALERTS_CLEAN_UP_TIMER_UTC_SEC_MAX )
        {
            vStartCleanUpTimer(psObj, psAppObj->un32TimerUTCSec);
            psAppObj->un32TimerUTCSec = WS_ALERTS_CLEAN_UP_TIMER_UTC_SEC_MAX;
        }

        vReleaseAppFacingObject( psAppObj );
    }

    return bResult;
}

/*****************************************************************************
 *
 *   bHandleRefreshList
 *
 *****************************************************************************/
static BOOLEAN bHandleRefreshList(
    WS_ALERTS_MGR_OBJECT_STRUCT *psObj,
    DSRL_ARG_STRUCT *psDSRLArg
        )
{
    BOOLEAN bResult = FALSE;
    WS_ALERTS_APP_OBJECT_STRUCT *psAppObj = NULL;

    psAppObj = psGetAppFacingObject((WS_ALERTS_SERVICE_OBJECT)psObj);
    if (psAppObj != NULL)
    {
        bResult = bRefreshList( psAppObj, psDSRLArg );

        // start timer to remove expired DSRL entries
        if ( psAppObj->un32TimerUTCSec != WS_ALERTS_CLEAN_UP_TIMER_UTC_SEC_MAX )
        {
            vStartCleanUpTimer(psObj, psAppObj->un32TimerUTCSec);
            psAppObj->un32TimerUTCSec = WS_ALERTS_CLEAN_UP_TIMER_UTC_SEC_MAX;
        }

        vReleaseAppFacingObject( psAppObj );
    }

    return bResult;
}

/*****************************************************************************
 *
 *   vHandleDeleteList
 *
 *****************************************************************************/
static void vHandleDeleteList(
    WS_ALERTS_MGR_OBJECT_STRUCT *psObj,
    DSRL_ARG_STRUCT *psDSRLArg
        )
{
    WS_ALERTS_APP_OBJECT_STRUCT *psAppObj = NULL;

    psAppObj = psGetAppFacingObject((WS_ALERTS_SERVICE_OBJECT)psObj);
    if (psAppObj != NULL)
    {
        bDeleteList( psAppObj, psDSRLArg );
        vReleaseAppFacingObject( psAppObj );
    }

    return;
}

/*****************************************************************************
 *
 *   vEventHandler
 *
 *   This function runs in the context of an SMS resource which has been
 *   assigned to this service.
 *
 *****************************************************************************/
static void vEventHandler(
    DATASERVICE_MGR_OBJECT hDataService,
    DATASERVICE_EVENT_MASK tCurrentEvent,
    void *pvEventArg,
    void *pvEventCallbackArg
        )
{
    WS_ALERTS_MGR_OBJECT_STRUCT *psObj;
    BOOLEAN bValid = FALSE, bStopEvent = FALSE;
    SMSAPI_EVENT_MASK tEventMask = DATASERVICE_EVENT_NONE;

    // Get our alerts handle from the callback argument
    psObj = ( WS_ALERTS_MGR_OBJECT_STRUCT * )pvEventCallbackArg;
    // Is this object valid?
    bValid = DATASERVICE_IMPL_bValid( ( DATASERVICE_IMPL_HDL )psObj );

    // Only handle events for valid objects...
    if( bValid == FALSE )
    {
        return;
    }

    switch( tCurrentEvent )
    {
            // Handle Alerts Service events here...
            // State has changed
        case DATASERVICE_EVENT_STATE:
        {
            BOOLEAN bStateChanged;
            DATASERVICE_STATE_CHANGE_STRUCT const *psStateChange =
                ( DATASERVICE_STATE_CHANGE_STRUCT const *)pvEventArg;

            // Process the state transition
            bStateChanged = DATASERVICE_IMPL_bStateFSM(
                (DATASERVICE_IMPL_HDL)psObj,
                psStateChange,
                &GsWSAlertsStateHandlers,
                (void *)psObj);

            if (bStateChanged == TRUE)
            {
                // The state has been updated
                tEventMask |= DATASERVICE_EVENT_STATE;
                // Is the service stopped now?
                if (psStateChange->eCurrentState ==
                        DATASERVICE_STATE_STOPPED)
                {
                    bStopEvent = TRUE;
                }
            }
        }
        break;
        // This service has a message to process
        case DATASERVICE_EVENT_NEW_DATA:
        {
            OSAL_BUFFER_HDL hPayload = ( OSAL_BUFFER_HDL )pvEventArg;
            BOOLEAN bOk;

            // Ensure the payload handle is valid
            // If it isn't, there's a problem with
            // SMS
            if( hPayload == OSAL_INVALID_BUFFER_HDL )
            {
                // Set the error
                vSetError( psObj,
                           DATASERVICE_ERROR_CODE_GENERAL );
            }
            // Only process this message if we are not stopping
            else
            {
                printf(WS_ALERTS_MGR_OBJECT_NAME": Payload Received (%u)\n",
                       OSAL.tBufferGetSize(hPayload));
                // Process this message now
                bOk =
                    GsWSAlertsIntf.bProcessMessage( psObj->hInterface, &hPayload );

                if( bOk == TRUE )
                {
                    puts(WS_ALERTS_MGR_OBJECT_NAME": Message Payload Processed Ok");
                }
                else
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        WS_ALERTS_MGR_OBJECT_NAME": Failed to process message");
                    DATASERVICE_IMPL_vLog(
                        WS_ALERTS_MGR_OBJECT_NAME": Failed to process message\n");
                }
                // We're all done with this payload
                DATASERVICE_IMPL_bFreeDataPayload( hPayload );
            }
        }
        break;
        // We need to do some work with our DSRLs
        case DATASERVICE_INTERNAL_EVENT_DSRL:
        {
            BOOLEAN bSuccess = TRUE;
            DSRL_ARG_STRUCT *psDSRLArg =
                ( DSRL_ARG_STRUCT * )pvEventArg;

            if( psDSRLArg == NULL )
            {
                SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                    WS_ALERTS_MGR_OBJECT_NAME": DSRL_EVENT with no targets" );
                break;
            }

            switch( psDSRLArg->eAction )
            {
                case DSRL_ACTION_ADD:
                {
                    // Create a new list
                    bSuccess = bHandleCreateList( psObj, psDSRLArg );
                }
                break;
                case DSRL_ACTION_MODIFY:
                {
                    bSuccess = bHandleModifyList( psObj, psDSRLArg );
                }
                break;
                case DSRL_ACTION_REFRESH:
                {
                    bSuccess = bHandleRefreshList( psObj, psDSRLArg );
                }
                break;
                case DSRL_ACTION_REMOVE:
                {
                    // Handle the deletion
                    vHandleDeleteList( psObj, psDSRLArg );
                    bSuccess = TRUE;
                }
                break;
                case DSRL_ACTION_INVALID:
                default:
                {
                    bSuccess = FALSE;
                }
                break;
            }

            if( bSuccess == FALSE )
            {
                // If an error occurred, indicate a state change
                // to the application
                tEventMask |= DATASERVICE_EVENT_STATE;
                // Indicate an error
                vSetError( psObj, DATASERVICE_ERROR_CODE_GENERAL );
            }
        }
        break;
        case DATASERVICE_EVENT_TIMEOUT:
        {
            puts(WS_ALERTS_MGR_OBJECT_NAME" There are expired msgs in DSRL. Removing them");
            // Clean up DSRL
            vRemoveExpiredDSRLEntries( psObj );
        }
        break;
        default:
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": Got unknown event (%u)",
                tCurrentEvent );
        }
        break;
    }

    if (bStopEvent == TRUE)
    {
        // Uninitialize the object, but leave
        // enough to be useful to the callback
        vUninitObject(psObj, FALSE);
    }

    // Update event mask with any relevant events which have occurred
    SMSU_tUpdate(&psObj->sEvent, tEventMask);

    // Notify of any change via any registered callback which may be present
    SMSU_bNotify(&psObj->sEvent);

    if (bStopEvent == TRUE)
    {
        // Filter out all further weather manager updates
        SMSU_tFilter(&psObj->sEvent, DATASERVICE_EVENT_ALL);

        vUninitObject(psObj, TRUE);
    }

    return;
}

/*****************************************************************************
 *
 *   bHandleServiceReady
 *
 *   This function is called when SMS is ready for the service to startup.
 *   At this time, the service has a context in which to operate (SMS
 *   assigned a resource to us). When this call is made, this service will
 *   perform all of its time-intensive startup procedures.
 *
 *****************************************************************************/
static BOOLEAN bHandleServiceReady(
    WS_ALERTS_MGR_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bOk = TRUE, bServiceUpdated = FALSE, bResult = FALSE;
    DATASERVICE_ERROR_CODE_ENUM eErrorCode =
        DATASERVICE_ERROR_CODE_NONE;
    WS_ALERTS_APP_OBJECT_STRUCT *psAppObj = NULL;
    char *pacDatabaseFilePath = NULL, *pacDatabaseFilePathB = NULL;
    char *pacPersistentFilePath = NULL;

    do
    {
        psAppObj = psGetAppFacingObject( ( WS_ALERTS_SERVICE_OBJECT )psObj );

        if( psAppObj == NULL )
        {
            // Error, not sure why this would happen
            eErrorCode = DATASERVICE_ERROR_CODE_GENERAL;
            break;
        }

        // Register for a timed event
        psObj->hCheckDbEvent =
            DATASERVICE_IMPL_hRegisterTimedEvent(
                (DATASERVICE_IMPL_HDL)psObj, (void *)NULL);

        if(psObj->hCheckDbEvent == DATASERVICE_TIMED_EVENT_INVALID_HDL)
        {
            // Error!
            eErrorCode = DATASERVICE_ERROR_CODE_GENERAL;
            break;
        }

        // Construct the paths needed by the databases
        // (based on information gathered from the config manager)
        bOk = DB_UTIL_bCreateFilePath(
            psObj->pacRefDatabaseDirPath,
            WS_ALERTS_DATABASE_FOLDER,
            WS_ALERTS_REF_DATABASE_FILENAMEA,
            &pacDatabaseFilePath );

        if( bOk == FALSE )
        {
            // Error!  Couldn't build database path
            eErrorCode = DATASERVICE_ERROR_CODE_GENERAL;
            break;
        }

        bOk = DB_UTIL_bCreateFilePath(
            psObj->pacRefDatabaseDirPath,
            WS_ALERTS_DATABASE_FOLDER,
            WS_ALERTS_REF_DATABASE_FILENAMEB,
            &pacDatabaseFilePathB );

        if( bOk == FALSE )
        {
            // Error!  Couldn't build database path
            eErrorCode = DATASERVICE_ERROR_CODE_GENERAL;
            break;
        }

        // Connect to the ref database bank, but
        // never perform an integrity check
        psAppObj->hSQLRefConnection =
            DB_UTIL_hConnectToReferenceBank(
                &pacDatabaseFilePath[0],
                &pacDatabaseFilePathB[0],
                &psObj->pacCurRefDatabaseFilePath,
                n32ExtractDataVersion, NULL,
                TRUE, &bServiceUpdated,
                GsWSAlertsIntf.tMaxVersionBitlen,
                ( DB_UTIL_CHECK_VERSION_HANDLER )bVerifyDBSchemaVersion,
                ( void * )( size_t )WS_ALERTS_DATABASE_FILE_VERSION,
                &eErrorCode,
                SQL_INTERFACE_OPTIONS_READONLY |
                SQL_INTERFACE_OPTIONS_SKIP_CORRUPTION_CHECK );

        if( psObj->pacCurRefDatabaseFilePath == pacDatabaseFilePath )
        {
            psObj->pacNextRefDatabaseFilePath = pacDatabaseFilePathB;
        }
        else if( psObj->pacCurRefDatabaseFilePath == pacDatabaseFilePathB )
        {
            psObj->pacNextRefDatabaseFilePath = pacDatabaseFilePath;
        }
        else
        {
            // Error! Reference database path is not assigned
            break;
        }

        // If the connection failed, indicate this
        if( psAppObj->hSQLRefConnection == SQL_INTERFACE_INVALID_OBJECT )
        {
            break;
        }

        psAppObj->hSelectIdsFromWestRTreeStmt =
            SQL_INTERFACE.hCreatePreparedStatement(
                psAppObj->hSQLRefConnection,
                WS_ALERTS_SELECT_LOCATIONS_IDS_FROM_WEST_RTREE_TABLE);

        if (psAppObj->hSelectIdsFromWestRTreeStmt ==
            SQL_PREPARED_STATEMENT_INVALID_HANDLE)
        {
            break;
        }

        psAppObj->hSelectIdsFromEastRTreeStmt =
            SQL_INTERFACE.hCreatePreparedStatement(
                psAppObj->hSQLRefConnection,
                WS_ALERTS_SELECT_LOCATIONS_IDS_FROM_EAST_RTREE_TABLE);

        if (psAppObj->hSelectIdsFromEastRTreeStmt ==
            SQL_PREPARED_STATEMENT_INVALID_HANDLE)
        {
            break;
        }

#if WS_ALERTS_CREATE_DB_IN_MEMORY == 1

        // Open a new memory database
        psAppObj->hSQLPersistConnection = 
            SQL_INTERFACE.hConnect(
                SQL_MEMORY_DATABASE_NAME,
                SQL_INTERFACE_OPTIONS_CREATE_IF_NOT_FOUND |
                SQL_INTERFACE_OPTIONS_SKIP_CORRUPTION_CHECK,
                &eErrorCode );

        if( psAppObj->hSQLPersistConnection == SQL_INTERFACE_INVALID_OBJECT )
        {
            break;
        }

        bOk = bCreatePersistDBTables(psAppObj->hSQLPersistConnection, NULL);
        if (bOk == FALSE)
        {
            eErrorCode = DATASERVICE_ERROR_CODE_DATABASE_ACCESS_FAILURE;
            break;
        }

#else
        // Build the path to the persistent storage DB
        bOk = DB_UTIL_bCreateFilePath(
            NULL, // We don't know the base path
            WS_ALERTS_DATABASE_FOLDER,
            WS_ALERTS_PERSIST_DATABASE_FILENAME,
            &pacPersistentFilePath );

        if( bOk == FALSE )
        {
            // Error!  Couldn't build database path
            eErrorCode = DATASERVICE_ERROR_CODE_GENERAL;
            break;
        }

        if( bServiceUpdated == TRUE )
        {
            // We just updated our reference DB -- so
            // now we trash the persistent DB
            remove( &pacPersistentFilePath[0] );
        }

        // Connect to the persitent storage database
        psAppObj->hSQLPersistConnection =
            DB_UTIL_hConnectToPersistent(
                &pacPersistentFilePath[0],
                bCreatePersistDBTables,
                ( void * )NULL,
                ( DB_UTIL_CHECK_VERSION_HANDLER )bVerifyDBSchemaVersion,
                ( void * )( size_t )WS_ALERTS_PERSIST_DATABASE_FILE_VERSION,
                &eErrorCode );

        if( psAppObj->hSQLPersistConnection == SQL_INTERFACE_INVALID_OBJECT )
        {
            break;
        }

        // clean up persistent DB
        vInitDbCleanUp(psAppObj);
#endif

        psAppObj->hSelectAlertMsgsByLocStmt =
            SQL_INTERFACE.hCreatePreparedStatement(
                psAppObj->hSQLPersistConnection,
                WS_ALERTS_SELECT_ALERT_MSGS_BY_LOCATION);

        if (psAppObj->hSelectAlertMsgsByLocStmt ==
            SQL_PREPARED_STATEMENT_INVALID_HANDLE)
        {
            break;
        }

        psAppObj->hSelectPolygonsStmt =
            SQL_INTERFACE.hCreatePreparedStatement(
                psAppObj->hSQLPersistConnection,
                WS_ALERTS_SELECT_POLYGONS);

        if (psAppObj->hSelectPolygonsStmt ==
            SQL_PREPARED_STATEMENT_INVALID_HANDLE)
        {
            break;
        }

        psObj->un16DBContentVersion =
            (UN16)n32ExtractDataVersion(psAppObj->hSQLRefConnection, NULL);

        // Start the interface
        psObj->hInterface = GsWSAlertsIntf.hInit(
            ( WS_ALERTS_SERVICE_OBJECT )psObj,
            DATASERVICE_IMPL_hSMSObj((DATASERVICE_IMPL_HDL)psObj),
            psAppObj->hSQLRefConnection,
            (RFD_UPDATE_VERSION)psObj->un16DBContentVersion,
            psObj->bRefDBUpdatesEnabled);

        if( psObj->hInterface == WS_ALERTS_INTERFACE_INVALID_OBJECT )
        {
            eErrorCode = DATASERVICE_ERROR_CODE_GENERAL;
            break;
        }

        bOk = bLoadLUTVersions( psAppObj );

        if( bOk == FALSE )
        {
            // Error!
            eErrorCode = DATASERVICE_ERROR_CODE_GENERAL;
            break;
        }

        bResult = TRUE;

    } while( FALSE );

    vReleaseAppFacingObject( psAppObj );

    if (pacPersistentFilePath != NULL)
    {
        SMSO_vDestroy( (SMS_OBJECT)pacPersistentFilePath );
    }

    if ( bResult == FALSE )
    {
        // Set the error we experienced
        vSetError( psObj, eErrorCode );

        if (pacDatabaseFilePath != NULL)
        {
            SMSO_vDestroy((SMS_OBJECT)pacDatabaseFilePath);
        }

        if (pacDatabaseFilePathB != NULL)
        {
            SMSO_vDestroy((SMS_OBJECT)pacDatabaseFilePathB);
        }

        psObj->pacCurRefDatabaseFilePath = NULL;
        psObj->pacNextRefDatabaseFilePath = NULL;
    }

    return bResult;
}

/*****************************************************************************
*
*   bHandleServiceError
*
*****************************************************************************/
static BOOLEAN bHandleServiceError (
    WS_ALERTS_MGR_OBJECT_STRUCT *psObj
        )
{
    // Stop any clean up timer we have now 'cause
    // who knows what's going on up there
    vStopCleanUpTimer(psObj);

    return TRUE;
}

/*****************************************************************************
 *
 *   bCreatePersistDBTables
 *
 *****************************************************************************/
static BOOLEAN bCreatePersistDBTables(
    SQL_INTERFACE_OBJECT hSQLConnection,
    void *pvArg
        )
{
    char acBuffer[WS_ALERTS_MAX_SQL_STRING_LENGTH];
    BOOLEAN bSuccess = FALSE, bTransactionStarted = FALSE;
    int iReturn = 0;

    do
    {
        bTransactionStarted = SQL_INTERFACE.bStartTransaction( hSQLConnection );

        if( bTransactionStarted == FALSE )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME
                ": failed to start transaction to recreate persistent storage" );
            break;
        }

        bSuccess = SQL_INTERFACE.bExecuteCommand( hSQLConnection,
            WS_ALERTS_CREATE_MESSAGES_TABLE );

        if( bSuccess == FALSE )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME
                ": failed to create wsalerts messages table" );
            break;
        }

        bSuccess = SQL_INTERFACE.bExecuteCommand( hSQLConnection,
            WS_ALERTS_CREATE_END_TIME_INDEX_MSG_TABLE );

        if( bSuccess == FALSE )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME
                ": failed to create index on endTime field in messages table" );
            break;
        }

        bSuccess = SQL_INTERFACE.bExecuteCommand( hSQLConnection,
            WS_ALERTS_CREATE_MESSAGE_LOCATIONS_TABLE );

        if( bSuccess == FALSE )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME
                ": failed to create wsalerts message_locations table" );
            break;
        }

        bSuccess = SQL_INTERFACE.bExecuteCommand( hSQLConnection,
            WS_ALERTS_CREATE_MSG_LOC_ID_INDEX_LOC_TABLE );

        if( bSuccess == FALSE )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME
                ": failed to create index on LocationID field "
                "in message_locations table" );
            break;
        }

        bSuccess = SQL_INTERFACE.bExecuteCommand( hSQLConnection,
            WS_ALERTS_CREATE_MESSAGE_TYPES_TABLE );

        if( bSuccess == FALSE )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME
                ": failed to create wsalerts message_types table" );
            break;
        }

        bSuccess = SQL_INTERFACE.bExecuteCommand( hSQLConnection,
            WS_ALERTS_CREATE_POLYGONS_TABLE );

        if( bSuccess == FALSE )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME
                ": failed to create wsalerts polygons table" );
            break;
        }

        bSuccess = SQL_INTERFACE.bExecuteCommand( hSQLConnection,
            WS_ALERTS_CREATE_DB_VERSION_TABLE );

        if( bSuccess == FALSE )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME
                ": failed to create wsalerts db_version table" );
            break;
        }

        // Build the version string
        iReturn = snprintf( &acBuffer[0], sizeof(acBuffer),
            WS_ALERTS_INSERT_VERSION_ROW,
            WS_ALERTS_PERSIST_DATABASE_FILE_VERSION );

        if (iReturn <= 0)
        {
            bSuccess = FALSE;
            break;
        }

        bSuccess = SQL_INTERFACE.bExecuteCommand(hSQLConnection, &acBuffer[0]);

        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME
                ": failed to set db version in persistant storage");
            break;
        }
    }
    while( FALSE );

    if( bTransactionStarted == TRUE )
    {
        BOOLEAN bTransactionEnded;
        bTransactionEnded = SQL_INTERFACE.bEndTransaction( hSQLConnection );

        if( bTransactionEnded == FALSE )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME
                ": failed to end transaction" );
            bSuccess = FALSE;
        }
    }

    return bSuccess;
}

#if WS_ALERTS_CREATE_DB_IN_MEMORY == 0

/*****************************************************************************
 *
 *   vInitDbCleanUp
 *
 *****************************************************************************/
static void vInitDbCleanUp(
    WS_ALERTS_APP_OBJECT_STRUCT *psAppObj
        )
{
    UN32 un32CurUTCsec = 0;
    OSAL_RETURN_CODE_ENUM eReturnCode;

    eReturnCode = OSAL.eTimeGet(&un32CurUTCsec);
    if (eReturnCode == OSAL_SUCCESS)
    {
        printf(WS_ALERTS_MGR_OBJECT_NAME
            " Current time is %d seconds UTC.\n", un32CurUTCsec);

        // remove expired messages
        vRemoveExpiredMsgs(psAppObj, un32CurUTCsec);
    }

    return;
}

/*******************************************************************************
 *
 *   vRemoveExpiredMsgs
 *
 *   Removes messages with end time in the past
 *
 *******************************************************************************/
static void vRemoveExpiredMsgs(
    WS_ALERTS_APP_OBJECT_STRUCT *psAppObj,
    UN32 un32CurrentTime
        )
{
    BOOLEAN bSuccess = FALSE;
    int iReturn = 0;

    DATASERVICE_IMPL_vLog(WS_ALERTS_MGR_OBJECT_NAME": Remove all expired messages\n");

    // Start a new transaction
    bSuccess = SQL_INTERFACE.bStartTransaction(psAppObj->hSQLPersistConnection);
    if (bSuccess == FALSE)
    {
        return;
    }

    do
    {
        // prepare command to remove locations the messages are actual for
        iReturn = snprintf( psAppObj->acBuffer, sizeof( psAppObj->acBuffer ),
            WS_ALERTS_DELETE_EXPIRED_LOCATIONS,
            un32CurrentTime );

        if (iReturn <= 0)
        {
            break;
        }

        printf(WS_ALERTS_MGR_OBJECT_NAME": SQL request %s will be performed.\n",
            psAppObj->acBuffer);

        bSuccess = SQL_INTERFACE.bExecuteCommand(
            psAppObj->hSQLPersistConnection, &psAppObj->acBuffer[0]);

        if( bSuccess == FALSE )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": SQL request failed");
            break;
        }

        // prepare command to remove types of messages
        iReturn = snprintf( psAppObj->acBuffer, sizeof( psAppObj->acBuffer ),
            WS_ALERTS_DELETE_EXPIRED_MSG_TYPES,
            un32CurrentTime );

        if (iReturn <= 0)
        {
            break;
        }

        printf(WS_ALERTS_MGR_OBJECT_NAME": SQL request %s will be performed.\n",
            psAppObj->acBuffer);

        bSuccess = SQL_INTERFACE.bExecuteCommand(
            psAppObj->hSQLPersistConnection, &psAppObj->acBuffer[0]);

        if( bSuccess == FALSE )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": SQL request failed");
            break;
        }

        // prepare command to remove locations described by polygons
        iReturn = snprintf( psAppObj->acBuffer, sizeof( psAppObj->acBuffer ),
            WS_ALERTS_DELETE_EXPIRED_POLYGONS,
            un32CurrentTime );

        if (iReturn <= 0)
        {
            break;
        }

        printf(WS_ALERTS_MGR_OBJECT_NAME": SQL request %s will be performed.\n",
            psAppObj->acBuffer);

        bSuccess = SQL_INTERFACE.bExecuteCommand(
            psAppObj->hSQLPersistConnection, &psAppObj->acBuffer[0]);

        if( bSuccess == FALSE )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": SQL request failed");
            break;
        }

        // at last prepare command removing messages itself
        iReturn = snprintf( psAppObj->acBuffer, sizeof( psAppObj->acBuffer ),
            WS_ALERTS_DELETE_EXPIRED_MESSAGES,
            un32CurrentTime );

        if (iReturn <= 0)
        {
            break;
        }

        printf(WS_ALERTS_MGR_OBJECT_NAME": SQL request %s will be performed.\n",
            psAppObj->acBuffer);

        bSuccess = SQL_INTERFACE.bExecuteCommand(
            psAppObj->hSQLPersistConnection, &psAppObj->acBuffer[0]);

        if( bSuccess == FALSE )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": SQL request failed");
        }

    } while ( FALSE );

    SQL_INTERFACE.bEndTransaction(psAppObj->hSQLPersistConnection);

    return;
}

#endif

/*****************************************************************************
 *
 *   bVerifyDBSchemaVersion
 *
 *****************************************************************************/
static BOOLEAN bVerifyDBSchemaVersion(
    SQL_INTERFACE_OBJECT hSQLConnection,
    UN32 un32SchemaVer
        )
{
    WS_ALERTS_DB_QUERY_RESULT_STRUCT sQueryResult;
    BOOLEAN bOk, bVersionMatched = FALSE;

    if (hSQLConnection == SQL_INTERFACE_INVALID_OBJECT)
    {
        return FALSE;
    }

    // Perform the SQL query and process the result
    // (it will provide us with a data row)
    bOk = SQL_INTERFACE.bQuery(
        hSQLConnection,
        WS_ALERTS_SELECT_DB_VERSION,
        bProcessSelectDBVersionResult,
        &sQueryResult);

    if ((bOk == TRUE) &&
        (sQueryResult.bResultantRows == TRUE))
    {
        WS_ALERTS_VERSION_ROW_STRUCT *psVersionRow =
            &sQueryResult.uDbRow.sVersion;

        if (psVersionRow->un32Version == un32SchemaVer)
        {
            // Version match!
            bVersionMatched = TRUE;
        }
    }

    return bVersionMatched;
}

/*******************************************************************************
*
*   bProcessSelectDBVersionResult
*
*   This function is provided to the SQL_INTERFACE_OBJECT in order to process
*   a "select all" query made on the database version relation.  It populates a
*   pointer to a WS_ALERTS_VERSION_ROW_STRUCT with the information found
*   in the database.
*
*******************************************************************************/
static BOOLEAN bProcessSelectDBVersionResult (
    SQL_QUERY_COLUMN_STRUCT *psColumns,
    N32 n32NumberOfColumns,
    void *pvArg
        )
{
    DB_VERSION_FIELDS_ENUM eCurrentField =
        (DB_VERSION_FIELDS_ENUM)0;
    WS_ALERTS_DB_QUERY_RESULT_STRUCT *psResult =
        (WS_ALERTS_DB_QUERY_RESULT_STRUCT *)pvArg;
    WS_ALERTS_VERSION_ROW_STRUCT *psVersionRow;
    SQL_QUERY_COLUMN_STRUCT *psCurrentCol;

    // Verify input
    if (psResult == NULL)
    {
        return FALSE;
    }

    // If there are the correct
    // number of columns, then we have good results
    if (n32NumberOfColumns == DB_VERSION_MAX_FIELDS)
    {
        psResult->bResultantRows = TRUE;
    }
    else
    {
        psResult->bResultantRows = FALSE;
        return FALSE;
    }

    // Get a more convenient pointer
    psVersionRow = &psResult->uDbRow.sVersion;
    // Clear the version structure
    OSAL.bMemSet(psVersionRow, 0, sizeof(WS_ALERTS_VERSION_ROW_STRUCT));

    // We have just performed an SQL query in order to
    // determine the version of the database.
    // Process the results (should be only one row)
    do
    {
        // Grab the current column data
        psCurrentCol = &psColumns[(UN8)eCurrentField];

        // Decode the current field based on it's type
        switch (eCurrentField)
        {
            case DB_VERSION_FIELD_DB_VER:
            {
                psVersionRow->un32Version =
                    (UN8)psCurrentCol->uData.sUN32.un32Data;
            }
            break;
            default: // Shouldn't happen
                break;
        }
    }
    while ( ++eCurrentField < DB_VERSION_MAX_FIELDS);

    if (eCurrentField == DB_VERSION_MAX_FIELDS)
    {
        return TRUE;
    }

    return FALSE;
}

/*****************************************************************************
 *
 *   vStartCleanUpTimer
 *
 *****************************************************************************/
static void vStartCleanUpTimer(
    WS_ALERTS_MGR_OBJECT_STRUCT *psObj,
    UN32 un32NewUTCsec
        )
{
    BOOLEAN bOk;
    UN32 un32TimerValue = 0, un32CurUTCsec = 0;
    OSAL_RETURN_CODE_ENUM eReturnCode;

    eReturnCode = OSAL.eTimeGet(&un32CurUTCsec);
    if (eReturnCode == OSAL_SUCCESS)
    {
        printf(WS_ALERTS_MGR_OBJECT_NAME
            " Current time is %d seconds UTC.\n", un32CurUTCsec);

        if ( un32NewUTCsec > un32CurUTCsec )
        {
            un32TimerValue = un32NewUTCsec - un32CurUTCsec;

            // set timer to the nearest expiration date
            bOk = DATASERVICE_IMPL_bSetTimedEvent(
                psObj->hCheckDbEvent, TRUE, FALSE, un32TimerValue);

            if (bOk == TRUE)
            {
                puts(WS_ALERTS_MGR_OBJECT_NAME" Starting clean up timer");
            }
        }
    }

    return;
}

/*****************************************************************************
 *
 *   vStopCleanUpTimer
 *
 *****************************************************************************/
static void vStopCleanUpTimer (
    WS_ALERTS_MGR_OBJECT_STRUCT *psObj
        )
{
    puts(WS_ALERTS_MGR_OBJECT_NAME" Stopping cancel timer");

    // Stop the cancel event from occurring
    DATASERVICE_IMPL_bStopTimedEvent(psObj->hCheckDbEvent);

    return;
}

/*****************************************************************************
 *
 *   bLoadLUTVersions
 *
 *****************************************************************************/
static BOOLEAN bLoadLUTVersions(
    WS_ALERTS_APP_OBJECT_STRUCT *psAppObj
        )
{
    WS_ALERTS_DB_QUERY_OBJECTS_STRUCT sResult;
    int iReturn = 0;
    BOOLEAN bSuccess = FALSE;
    BOOLEAN bResult = FALSE;

    do
    {
        sResult.psAppObj = psAppObj;
        sResult.hObjectsList = psAppObj->hLUTVersionsList;
        sResult.bResult = FALSE;

        iReturn = snprintf( psAppObj->acBuffer,
            sizeof( psAppObj->acBuffer ),
            WS_ALERTS_SELECT_LUT_VERSION );

        printf(WS_ALERTS_MGR_OBJECT_NAME": SQL query: %s\n",
            psAppObj->acBuffer);

        if (iReturn <= 0)
        {
            break;
        }

        bSuccess = SQL_INTERFACE.bQuery( psAppObj->hSQLRefConnection,
            psAppObj->acBuffer,
            (SQL_QUERY_RESULT_HANDLER)bProcessSelectTableVersion,
            &sResult);

        if ( (bSuccess != TRUE) ||
             (sResult.bResult != TRUE) )
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": failed to read table version" );
            bResult = FALSE;
            break;
        }

        bResult = TRUE;
    }
    while( FALSE );

    return bResult;
}

/*******************************************************************************
 *
 *   bProcessSelectTableVersion
 *
 *******************************************************************************/
static BOOLEAN bProcessSelectTableVersion (
    SQL_QUERY_COLUMN_STRUCT *psColumns,
    N32 n32NumberOfColumns,
    WS_ALERTS_DB_QUERY_OBJECTS_STRUCT *psResult
        )
{
    WS_ALERTS_LUT_VERSION_STRUCT *psLutVersion = NULL;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    UN32 un32Class = UN32_MAX;
    UN32 un32Type = UN32_MAX;
    UN32 un32Lang = UN32_MAX;
    int iReturn = 0;

    do
    {
        psResult->bResult = FALSE;

        iReturn = sscanf((char *)psColumns[DATA_VERSION_TYPE].uData.sCString.pcData,
            "C%3uT%3uL%3u", &un32Class, &un32Type, &un32Lang);

        if ( (iReturn != 3) ||
             (un32Class > UN8_MAX) ||
             (un32Type > UN8_MAX) ||
             (un32Lang > UN8_MAX) )
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": invalid table name");
            break;
        }

        psLutVersion = ( WS_ALERTS_LUT_VERSION_STRUCT * ) SMSO_hCreate(
            WS_ALERTS_MGR_OBJECT_NAME":LUTVersionObj",
            sizeof(WS_ALERTS_LUT_VERSION_STRUCT),
            (SMS_OBJECT)psResult->psAppObj, FALSE );

        if( psLutVersion == NULL )
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": cannot allocate memory for"
                " LUTVersionObj");
            break;
        }

        psLutVersion->un32LutId = WS_ALERTS_GET_LUT_ID( (UN8)un32Class, (UN8)un32Type, (UN8)un32Lang);

        psLutVersion->un8Version = (UN8)psColumns[DATA_VERSION_VERSION].uData.sUN32.un32Data;

        eReturnCode = OSAL.eLinkedListAdd( psResult->hObjectsList,
            OSAL_INVALID_LINKED_LIST_ENTRY_PTR, ( void * )psLutVersion );

        if( eReturnCode != OSAL_SUCCESS )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": cannot add entry to list (%s)",
                OSAL.pacGetReturnCodeName( eReturnCode ) );
            SMSO_vDestroy( ( SMS_OBJECT )psLutVersion );
        }

        psResult->bResult = TRUE;

    }
    while( FALSE );

    return psResult->bResult;
}

/*****************************************************************************
 *
 *   bInitAppFacingObject
 *
 *****************************************************************************/
static BOOLEAN bInitAppFacingObject(
    WS_ALERTS_MGR_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bResult = FALSE;

    do
    {
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
        DATASERVICE_DSRL_CONFIG_STRUCT sDSRLConfig;

        // Begin DSRL config
        DATASERVICE_IMPL_vInitializeDSRLConfig(&sDSRLConfig);

        // Set service type
        sDSRLConfig.eServiceType = DATASERVICE_TYPE_WS_ALERTS;

        // Set the parent object size
        sDSRLConfig.tParentObjectSize = sizeof(WS_ALERTS_APP_OBJECT_STRUCT);

        // Set the WS Alerts DSLR descriptor size
        sDSRLConfig.tServiceDataSize = sizeof(WS_ALERTS_DSRL_DESC_STRUCT);

        // Configure the device feature
        sDSRLConfig.bDeviceEnabled = TRUE;
        sDSRLConfig.un32DeviceNotifyDistance = WS_ALERTS_DEVICE_DISTANCE_THRESHOLD;
        sDSRLConfig.eDeviceNotifyUnits = DISTANCE_UNIT_TYPE_MILES;

        // Create it
        psObj->psAppObj = ( WS_ALERTS_APP_OBJECT_STRUCT * )
            DATASERVICE_IMPL_hConfigureDSRL(
                (DATASERVICE_IMPL_HDL)psObj, &sDSRLConfig);

        if( psObj->psAppObj == NULL )
        {
            break;
        }

        eReturnCode = OSAL.eLinkedListCreate(
            &( psObj->psAppObj->hDSRLList ),
            WS_ALERTS_MGR_OBJECT_NAME":DSRLList",
            NULL,
            OSAL_LL_OPTION_NONE );

        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": Cannot create DSRLList object.");
            break;
        }

        eReturnCode = OSAL.eLinkedListCreate(
            &( psObj->psAppObj->hEntriesList ),
            WS_ALERTS_MGR_OBJECT_NAME":EntriesList",
            (OSAL_LL_COMPARE_HANDLER)n16CompareMsgByID,
            OSAL_LL_OPTION_NONE );

        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": Cannot create EntriesList" );
            break;
        }

        eReturnCode = OSAL.eLinkedListCreate(
            &( psObj->psAppObj->hRemovedEntriesList ),
            WS_ALERTS_MGR_OBJECT_NAME":RemovedEntList",
            NULL,
            OSAL_LL_OPTION_UNIQUE );

        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": Cannot create RemovedEntList" );
            break;
        }

        eReturnCode = OSAL.eLinkedListCreate(
            &( psObj->psAppObj->hLocationPool ),
            WS_ALERTS_MGR_OBJECT_NAME":LocationPool",
            (OSAL_LL_COMPARE_HANDLER)n16CompareEntriesByLocID,
            OSAL_LL_OPTION_RBTREE );

        if( eReturnCode != OSAL_SUCCESS )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": Cannot create LocationPool object." );
            break;
        }

        eReturnCode = OSAL.eLinkedListCreate(
            &( psObj->psAppObj->hLUTVersionsList ),
            WS_ALERTS_MGR_OBJECT_NAME":LUTVersionsList",
            (OSAL_LL_COMPARE_HANDLER) n16CompareByLutId,
            OSAL_LL_OPTION_NONE );

        if( eReturnCode != OSAL_SUCCESS )
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": Cannot create Lut Versions list.");
            break;
        }

        eReturnCode = OSAL.eLinkedListCreate(
            &( psObj->psAppObj->hLocIdList ),
            WS_ALERTS_MGR_OBJECT_NAME":hLocIdList",
            NULL,
            OSAL_LL_OPTION_NONE);
        if( eReturnCode != OSAL_SUCCESS )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": cannot create location IDs list (%s)",
                OSAL.pacGetReturnCodeName( eReturnCode ) );
            break;
        }

        eReturnCode = OSAL.eLinkedListCreate(
            &( psObj->psAppObj->hLocationList ),
            WS_ALERTS_MGR_OBJECT_NAME":LocationList",
            NULL,
            OSAL_LL_OPTION_LINEAR );
        if( eReturnCode != OSAL_SUCCESS )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": cannot create locations list (%s)",
                OSAL.pacGetReturnCodeName( eReturnCode ) );
            break;
        }

        eReturnCode = OSAL.eLinkedListCreate(
            &( psObj->psAppObj->hMsgIdList ),
            WS_ALERTS_MGR_OBJECT_NAME":MsgIdList",
            ( OSAL_LL_COMPARE_HANDLER )n16CompareLocationsByMsgId,
            OSAL_LL_OPTION_NONE );
        if( eReturnCode != OSAL_SUCCESS )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": cannot create MsgIdList list (%s)",
                OSAL.pacGetReturnCodeName( eReturnCode ) );
            break;
        }

        psObj->psAppObj->hDummyMsg = WS_ALERT_MSG_hCreateDummy(
            (SMS_OBJECT)psObj->psAppObj);
        if (psObj->psAppObj->hDummyMsg == WS_ALERT_MSG_INVALID_OBJECT)
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": cannot create dummy message");
            break;
        }

        psObj->psAppObj->hDummyLoc = WS_ALERTS_LOCATION_hCreateDummy(
            (SMS_OBJECT)psObj->psAppObj);
        if (psObj->psAppObj->hDummyLoc == WS_ALERTS_LOCATION_INVALID_OBJECT)
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": cannot create dummy location");
            break;
        }

        psObj->psAppObj->eCurrentLanguage = LOCALIZATION.eGetLanguage();
        psObj->psAppObj->un32TimerUTCSec = WS_ALERTS_CLEAN_UP_TIMER_UTC_SEC_MAX;

        bResult = TRUE;
    } while( FALSE );

    if (NULL != psObj->psAppObj)
    {
        // Relinquish ownership
        SMSO_vUnlock( ( SMS_OBJECT )psObj->psAppObj );
    }

    return bResult;
}

/*****************************************************************************
 *
 *   vSetError
 *
 *****************************************************************************/
static void vSetError(
    WS_ALERTS_MGR_OBJECT_STRUCT *psObj,
    DATASERVICE_ERROR_CODE_ENUM eErrorCode
        )
{
    SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
        WS_ALERTS_MGR_OBJECT_NAME": vSetError (%d)", eErrorCode
                                );
    // Tell the DSM about it
    DATASERVICE_IMPL_vError((DATASERVICE_IMPL_HDL)psObj,eErrorCode );
    return;
}

/*****************************************************************************
 *
 *   n32ExtractDataVersion
 *
 *****************************************************************************/
static N32 n32ExtractDataVersion(
    SQL_INTERFACE_OBJECT hConnection,
    void *pvArg
        )
{
    N32 n32Result = DB_UTIL_DB_UNDER_CONSTRUCTION_VER;
    N16 n16ContentVer = 0;
    BOOLEAN bLoaded;

    if (hConnection == SQL_INTERFACE_INVALID_OBJECT)
    {
        return DB_UTIL_DB_UNDER_CONSTRUCTION_VER;
    }

    // Get the Content Version (table!)
    bLoaded = bLoadDBContentVersion(
        hConnection, &n16ContentVer);

    if (bLoaded == TRUE)
    {
        n32Result = (N32)n16ContentVer;
    }

    return n32Result;
}

/*****************************************************************************
*
*   bLoadDBContentVersion
*
*****************************************************************************/
static BOOLEAN bLoadDBContentVersion (
    SQL_INTERFACE_OBJECT hSQLConnection,
    N16 *pn16DBVer
        )
{
    WS_ALERTS_DB_QUERY_RESULT_STRUCT sQueryResult;
    BOOLEAN bOk;

    if ( (hSQLConnection == SQL_INTERFACE_INVALID_OBJECT) ||
         (pn16DBVer == NULL) )
    {
        return FALSE;
    }

    sQueryResult.bResultantRows = FALSE;
    // Perform the SQL query and process the result
    // (it will provide us with a data row)
    bOk = SQL_INTERFACE.bQuery(
        hSQLConnection,
        WS_ALERTS_SELECT_UPDATE_NUMBER,
        (SQL_QUERY_RESULT_HANDLER)bProcessSelectUpdateVersionResult,
        &sQueryResult);

    if ( (bOk == TRUE) &&
         (sQueryResult.bResultantRows == TRUE) )
    {
        *pn16DBVer = (N16)sQueryResult.uDbRow.sVersion.un32Version;
    }

    return sQueryResult.bResultantRows;
}

/*******************************************************************************
*
*   bProcessSelectUpdateVersionResult
*
*   This function is provided to the SQL_INTERFACE_OBJECT in order to process
*   a "select all" query made on the location update database version relation.
*   It populates a pointer to an UPDATE_VERSION_ROW_STRUCT with the information
*   found in the database.
*
*******************************************************************************/
static BOOLEAN bProcessSelectUpdateVersionResult (
    SQL_QUERY_COLUMN_STRUCT *psColumns,
    N32 n32NumberOfColumns,
    WS_ALERTS_DB_QUERY_RESULT_STRUCT *psResult
        )
{
    UPDATE_VERSION_FIELDS_ENUM eCurrentField =
        (UPDATE_VERSION_FIELDS_ENUM)0;

    // Verify input
    if (psResult == NULL)
    {
        return FALSE;
    }

    // If there are the correct
    // number of columns, then we have good results
    if (n32NumberOfColumns == UPDATE_VERSION_MAX_FIELDS)
    {
        psResult->bResultantRows = TRUE;
    }
    else
    {
        psResult->bResultantRows = FALSE;
        return FALSE;
    }

    // We have just performed an SQL query in order to
    // determine the update version in the database.
    // Process the results (should be only one row)
    do
    {
        // Decode the current field based on it's type
        switch (eCurrentField)
        {
            case UPDATE_VERSION:
            {
                psResult->uDbRow.sVersion.un32Version =
                    (N16)psColumns[eCurrentField].uData.sUN32.un32Data;
            }
            break;
            default: // Shouldn't happen
                break;
        }
    }
    while (++eCurrentField < UPDATE_VERSION_MAX_FIELDS);

    return FALSE;
}


/*****************************************************************************
 *
 *   vUninitAppFacingObject
 *
 *****************************************************************************/
static void vUninitAppFacingObject(
    WS_ALERTS_MGR_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bLocked;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    WS_ALERTS_APP_OBJECT_STRUCT *psAppObj = psObj->psAppObj;

    // Lock the app object
    bLocked = SMSO_bLock((SMS_OBJECT)psAppObj, OSAL_OBJ_TIMEOUT_INFINITE);

    if (bLocked == FALSE)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            WS_ALERTS_MGR_OBJECT_NAME": Unable to lock app object");
        return;
    }

    // Disconnect from the reference database
    if( psAppObj->hSQLRefConnection != SQL_INTERFACE_INVALID_OBJECT )
    {
        // Destroy related prepared statements
        SQL_INTERFACE.vDestroyPreparedStatement(
            psAppObj->hSQLRefConnection,
            psAppObj->hSelectIdsFromWestRTreeStmt);

        psAppObj->hSelectIdsFromWestRTreeStmt =
            SQL_PREPARED_STATEMENT_INVALID_HANDLE;

        SQL_INTERFACE.vDestroyPreparedStatement(
            psAppObj->hSQLRefConnection,
            psAppObj->hSelectIdsFromEastRTreeStmt);

        psAppObj->hSelectIdsFromEastRTreeStmt =
            SQL_PREPARED_STATEMENT_INVALID_HANDLE;

        SQL_INTERFACE.vDisconnect( psAppObj->hSQLRefConnection );
        psAppObj->hSQLRefConnection = SQL_INTERFACE_INVALID_OBJECT;
    }

    if( psAppObj->pacRefDatabaseFilePath != ( char * )SMS_INVALID_OBJECT )
    {
        // Remove database path resources
        SMSO_vDestroy( ( SMS_OBJECT )
            &psAppObj->pacRefDatabaseFilePath[0] );
        psAppObj->pacRefDatabaseFilePath = ( char * )SMS_INVALID_OBJECT;
    }

    // Disconnect from the persistent database
    if( psAppObj->hSQLPersistConnection != SQL_INTERFACE_INVALID_OBJECT )
    {
        // Destroy related prepared statements
        SQL_INTERFACE.vDestroyPreparedStatement(
            psAppObj->hSQLPersistConnection,
            psAppObj->hSelectAlertMsgsByLocStmt);

        psAppObj->hSelectAlertMsgsByLocStmt =
            SQL_PREPARED_STATEMENT_INVALID_HANDLE;

        SQL_INTERFACE.vDestroyPreparedStatement(
            psAppObj->hSQLPersistConnection,
            psAppObj->hSelectPolygonsStmt);

        psAppObj->hSelectPolygonsStmt =
            SQL_PREPARED_STATEMENT_INVALID_HANDLE;

        SQL_INTERFACE.vDisconnect( psAppObj->hSQLPersistConnection );
        psAppObj->hSQLPersistConnection = SQL_INTERFACE_INVALID_OBJECT;
    }

    if( psAppObj->pacPersistDatabaseFilePath != ( char * )SMS_INVALID_OBJECT )
    {
        // Remove database path resources
        SMSO_vDestroy( ( SMS_OBJECT )
            &psAppObj->pacPersistDatabaseFilePath[0] );
        psAppObj->pacPersistDatabaseFilePath = ( char * )SMS_INVALID_OBJECT;
    }

    printf(WS_ALERTS_MGR_OBJECT_NAME": Destroying DSRL nodes.\n");

    if (psAppObj->hDSRLList != OSAL_INVALID_OBJECT_HDL)
    {
        eReturnCode = OSAL.eLinkedListRemoveAll(psAppObj->hDSRLList,
            (OSAL_LL_RELEASE_HANDLER)vDSRLRelease
            );
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": failed to remove DSRL items (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                );
        }

        eReturnCode = OSAL.eLinkedListDelete(psAppObj->hDSRLList);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": failed to delete DSRLs list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                );
        }

        psAppObj->hDSRLList = OSAL_INVALID_OBJECT_HDL;
    }

    if (psAppObj->hEntriesList != OSAL_INVALID_OBJECT_HDL)
    {
        eReturnCode = OSAL.eLinkedListRemoveAll(psAppObj->hEntriesList,
            (OSAL_LL_RELEASE_HANDLER)vEntryRelease
            );
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": failed to remove entry items (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                );
        }

        eReturnCode = OSAL.eLinkedListDelete(psAppObj->hEntriesList);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": failed to delete entry list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                );
        }

        psAppObj->hEntriesList = OSAL_INVALID_OBJECT_HDL;
    }

    if (psAppObj->hRemovedEntriesList != OSAL_INVALID_OBJECT_HDL)
    {
        eReturnCode =
            OSAL.eLinkedListRemoveAll(psAppObj->hRemovedEntriesList, NULL);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": failed to clean removed entries"
                " list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                );
        }

        eReturnCode =
            OSAL.eLinkedListDelete(psAppObj->hRemovedEntriesList);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": failed to delete removed"
                " entries list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                );
        }

        psAppObj->hRemovedEntriesList = OSAL_INVALID_OBJECT_HDL;
    }

    if( psAppObj->hLocationPool != OSAL_INVALID_OBJECT_HDL )
    {
        eReturnCode = OSAL.eLinkedListRemoveAll(
            psAppObj->hLocationPool,
            ( OSAL_LL_RELEASE_HANDLER )vLocationDescRelease);
        if (eReturnCode == OSAL_SUCCESS)
        {
            OSAL.eLinkedListDelete( psAppObj->hLocationPool );
            psAppObj->hLocationPool = OSAL_INVALID_OBJECT_HDL;
        }
    }

    if( psAppObj->hLUTVersionsList != OSAL_INVALID_OBJECT_HDL )
    {
        eReturnCode = OSAL.eLinkedListRemoveAll(
            psAppObj->hLUTVersionsList,
            (OSAL_LL_RELEASE_HANDLER) SMSO_vDestroy);
        if (eReturnCode == OSAL_SUCCESS)
        {
            OSAL.eLinkedListDelete( psAppObj->hLUTVersionsList );
            psAppObj->hLUTVersionsList = OSAL_INVALID_OBJECT_HDL;
        }
    }

    if( psAppObj->hLocIdList != OSAL_INVALID_OBJECT_HDL )
    {
        eReturnCode = OSAL.eLinkedListRemoveAll( psAppObj->hLocIdList, NULL );

        if( eReturnCode != OSAL_SUCCESS )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": failed to cleanup"
                " Location IDs list (%s)", OSAL.pacGetReturnCodeName( eReturnCode ) );
        }

        eReturnCode = OSAL.eLinkedListDelete( psAppObj->hLocIdList );

        if( eReturnCode != OSAL_SUCCESS )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": failed to delete"
                " Location IDs list (%s)", OSAL.pacGetReturnCodeName( eReturnCode ) );
        }

        psAppObj->hLocIdList = OSAL_INVALID_OBJECT_HDL;
    }

    if( psAppObj->hLocationList != OSAL_INVALID_OBJECT_HDL )
    {
        eReturnCode = OSAL.eLinkedListRemoveAll( psAppObj->hLocationList, NULL );

        if( eReturnCode != OSAL_SUCCESS )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": failed to cleanup"
                " Locations list (%s)", OSAL.pacGetReturnCodeName( eReturnCode ) );
        }

        eReturnCode = OSAL.eLinkedListDelete( psAppObj->hLocationList );

        if( eReturnCode != OSAL_SUCCESS )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": failed to delete"
                " Locations list (%s)", OSAL.pacGetReturnCodeName( eReturnCode ) );
        }

        psAppObj->hLocationList = OSAL_INVALID_OBJECT_HDL;
    }

    if( psAppObj->hMsgIdList != OSAL_INVALID_OBJECT_HDL )
    {
        eReturnCode = OSAL.eLinkedListRemoveAll( psAppObj->hMsgIdList,
            (OSAL_LL_RELEASE_HANDLER)vReleaseMsgLocDesc );

        if( eReturnCode != OSAL_SUCCESS )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": failed to cleanup"
                " Message IDs list (%s)", OSAL.pacGetReturnCodeName( eReturnCode ) );
        }

        eReturnCode = OSAL.eLinkedListDelete( psAppObj->hMsgIdList );

        if( eReturnCode != OSAL_SUCCESS )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": failed to delete"
                " Message IDs list (%s)", OSAL.pacGetReturnCodeName( eReturnCode ) );
        }

        psAppObj->hMsgIdList = OSAL_INVALID_OBJECT_HDL;
    }

    if (psAppObj->hDummyMsg != WS_ALERT_MSG_INVALID_OBJECT)
    {
        WS_ALERT_MSG_vDestroy(psAppObj->hDummyMsg);
        psAppObj->hDummyMsg = WS_ALERT_MSG_INVALID_OBJECT;
    }

    if (psAppObj->hDummyLoc != WS_ALERTS_LOCATION_INVALID_OBJECT)
    {
        WS_ALERTS_LOCATION_vDestroy(psAppObj->hDummyLoc);
        psAppObj->hDummyLoc = WS_ALERTS_LOCATION_INVALID_OBJECT;
    }

    // Destroy the app object
    DATASERVICE_IMPL_vDestroyDSRLParent( (DATASERVICE_IMPL_HDL)psObj, ( SMS_OBJECT )psAppObj );
    psObj->psAppObj = NULL;

    return;
}

/*****************************************************************************
 *
 *   vUninitObject
 *
 *****************************************************************************/
static void vUninitObject(
    WS_ALERTS_MGR_OBJECT_STRUCT *psObj,
    BOOLEAN bFullDelete
        )
{
    if (psObj->psAppObj != NULL)
    {
        // Releasing App interface object
        vUninitAppFacingObject(psObj);
    }

    // Releasing plugin object
    if( psObj->hInterface != WS_ALERTS_INTERFACE_INVALID_OBJECT )
    {
        GsWSAlertsIntf.vUnInit( psObj->hInterface );
        psObj->hInterface = WS_ALERTS_INTERFACE_INVALID_OBJECT;
    }

    // Destroy persistent storage pathnames
    if( psObj->pacRefDatabaseDirPath != NULL )
    {
        SMSO_vDestroy( (SMS_OBJECT)psObj->pacRefDatabaseDirPath);
        psObj->pacRefDatabaseDirPath = NULL;
    }

    if( psObj->pacCurRefDatabaseFilePath != NULL )
    {
        SMSO_vDestroy( (SMS_OBJECT)psObj->pacCurRefDatabaseFilePath);
        psObj->pacCurRefDatabaseFilePath = NULL;
    }

    if( psObj->pacNextRefDatabaseFilePath != NULL )
    {
        SMSO_vDestroy( (SMS_OBJECT)psObj->pacNextRefDatabaseFilePath);
        psObj->pacNextRefDatabaseFilePath = NULL;
    }

    // Clear the timed event handle, we're done with it
    psObj->hCheckDbEvent = DATASERVICE_TIMED_EVENT_INVALID_HDL;

    if (bFullDelete == TRUE)
    {
        // Destroy the SMS Update Event object
        SMSU_vDestroy(&psObj->sEvent);
    }

    return;
}

/*****************************************************************************
 *
 *   bConstructMessages
 *
 *   This function is part of the manager interface API and is utilized by
 *   the interface to decode incoming alert messages.
 *
 *   This API must only be called when already in the
 *   context of the alerts manager.
 *
 *   Inputs:
 *       hAlertsService - The alerts service object handle provided to
 *           the interface.
 *       psMsgSetDesc - The package of encoded messages.
 *       bRemoveMsgsFromDB - TRUE if message set is the first in the series
 *          so old messages should be removed from the DB.
 *
 *   Output: BOOLEAN TRUE if messages decoded successfully, FALSE on error or unwanted
 *       data
 *
 *****************************************************************************/
static BOOLEAN bConstructMessages(
    WS_ALERTS_SERVICE_OBJECT hAlertsService,
    WS_ALERTS_MSG_SET_DESC_STRUCT *psMsgSetDesc,
    BOOLEAN bRemoveMsgsFromDB
        )
{
    WS_ALERTS_APP_OBJECT_STRUCT *psAppObj = NULL;
    WS_ALERTS_MSG_ELEMENTS_ITERATOR_STRUCT sIterator;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    BOOLEAN bResult = FALSE;

    WS_ALERTS_MGR_OBJECT_STRUCT *psObj =
                    ( WS_ALERTS_MGR_OBJECT_STRUCT * )hAlertsService;

    if ( (hAlertsService == WS_ALERTS_SERVICE_INVALID_OBJECT) ||
         (psMsgSetDesc == NULL) )
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            WS_ALERTS_MGR_OBJECT_NAME": invalid input");
        return FALSE;
    }

    psAppObj = psGetAppFacingObject( ( WS_ALERTS_SERVICE_OBJECT )psObj );
    if ( psAppObj == NULL )
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            WS_ALERTS_MGR_OBJECT_NAME": invalid App object");
        return FALSE;
    }

    do
    {
        if ( bRemoveMsgsFromDB == TRUE )
        {
            vRemoveOldMsgs(psAppObj, psMsgSetDesc);
        }

        //sIterator.sMsgDesc is initialized inside the iterate function
        sIterator.psObj = psObj;
        sIterator.psAppObj = psAppObj;
        sIterator.psMsgSetDesc = psMsgSetDesc;
        sIterator.bCreateNewMsg = TRUE;
        sIterator.bResult = FALSE;
        sIterator.bFastForward = FALSE;

        eReturnCode = OSAL.eLinkedListIterate(
            psMsgSetDesc->hMsgElementsList,
            ( OSAL_LL_ITERATOR_HANDLER )bIterateMsgElements, &sIterator );
        if( (eReturnCode != OSAL_SUCCESS) ||
            (sIterator.bResult != TRUE) )
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": cannot iterate Message Elements ist"
                    );
            break;
        }

        bResult = TRUE;

    } while (FALSE);

    if (bResult == FALSE)
    {
        DATASERVICE_IMPL_vLog(WS_ALERTS_MGR_OBJECT_NAME": failed to process AU\n");
    }

    vReleaseAppFacingObject( psAppObj );

    return bResult;
}

/*****************************************************************************
 *
 *   bIterateMsgElements
 *
 *****************************************************************************/
static BOOLEAN bIterateMsgElements(
    WS_ALERTS_MGR_MESSAGE_ELEMENT_STRUCT *psMsgElem,
    WS_ALERTS_MSG_ELEMENTS_ITERATOR_STRUCT *psIterator
        )
{
    WS_ALERTS_DB_QUERY_RESULT_STRUCT sArg;
    BOOLEAN bSuccess = FALSE;
    BOOLEAN bResult = FALSE;

    const char *pacSpace = " ";
    const char *pacCommaSpace = ", ";
    const char *pacPeriodSpace = ". ";
    const char *pacBreak = "\n";

    do
    {
        // pass by elements which cannot be decoded
        // by current version of LUT
        if( psIterator->bFastForward == TRUE )
        {
            bSuccess = bIsMsgDelimiter(psMsgElem);

            if( bSuccess == TRUE )
            {
                psIterator->bFastForward = FALSE;
            }
            bResult = TRUE;
            break;

        }

        OSAL.bMemSet( &sArg, 0, sizeof( sArg ) );

        if( psIterator->bCreateNewMsg == TRUE )
        {
            //init msg desc
            bInitMsgDesc( &psIterator->sMsgDesc );
            psIterator->bCreateNewMsg = FALSE;
        }

        if ( psMsgElem->bIsRegularElement == TRUE )
        {
            bSuccess = bCheckVerMin(
                psIterator->psAppObj,
                psMsgElem->uType.sLutDesc.un32DecodeTableId,
                psMsgElem->uType.sLutDesc.un8MinVersion);
            if ( bSuccess == FALSE )
            {
                DATASERVICE_IMPL_vLog(WS_ALERTS_MGR_OBJECT_NAME": LUT version check failed. \n");

                psIterator->bFastForward = TRUE;

                vDestroyMsgDesc( &psIterator->sMsgDesc );
                psIterator->bCreateNewMsg = TRUE;

                bResult = TRUE;
                break;
            }
        }

        //Message delimiter handling
        bSuccess = bIsMsgDelimiter( psMsgElem );
        if( bSuccess == TRUE )
        {
            psIterator->sMsgDesc.eMsgLang = psIterator->psMsgSetDesc->eLanguage;
            psIterator->sMsgDesc.un8StateId = psIterator->psMsgSetDesc->un8StateId;
            psIterator->sMsgDesc.tVersion = psIterator->psMsgSetDesc->un8Version;
            psIterator->sMsgDesc.bIsHighRate = psIterator->psMsgSetDesc->bIsHighRate;

            // Start a new transaction
            bSuccess = SQL_INTERFACE.bStartTransaction(
                psIterator->psAppObj->hSQLPersistConnection);
            if (bSuccess == TRUE)
            {
                bSuccess = bStoreMsgInDB( psIterator->psAppObj, &psIterator->sMsgDesc,
                                          psMsgElem->uData.un16MsgSignature);
                if( bSuccess == FALSE )
                {
                    // Error! DB failure
                    DATASERVICE_IMPL_vLog(WS_ALERTS_MGR_OBJECT_NAME": failed to store message to the DB\n");
                    printf(WS_ALERTS_MGR_OBJECT_NAME": failed to store message to the DB\n");
                    vSetError( psIterator->psObj, DATASERVICE_ERROR_CODE_DATABASE_ACCESS_FAILURE );
                }

                SQL_INTERFACE.bEndTransaction(psIterator->psAppObj->hSQLPersistConnection);
            }

            vDestroyMsgDesc( &psIterator->sMsgDesc );
            vDestroySArg( &sArg );

            psIterator->bCreateNewMsg = TRUE;

            //Message delimiter processed
            bResult = TRUE;
            break;
        }

        bSuccess = bProcessMessage(
            psIterator->psAppObj,
            psIterator->psMsgSetDesc,
            psMsgElem,
            &sArg,
            &psIterator->sMsgDesc );
        if (bSuccess == FALSE)
        {
            printf(WS_ALERTS_MGR_OBJECT_NAME": Error processing message\n");
            DATASERVICE_IMPL_vLog(WS_ALERTS_MGR_OBJECT_NAME": Error processing message\n");
            break;
        }

        if( psMsgElem->sTextFormat.bIsCapital == TRUE )
        {
            STRING_bCapitalize( sArg.hText );
        }

        switch( psMsgElem->sTextFormat.eTail )
        {
            case WS_ALERTS_MSG_ELEM_TAIL_SPACE:
            {
                STRING.tAppendCStr( sArg.hText, pacSpace );
                break;
            }
            case WS_ALERTS_MSG_ELEM_TAIL_COMMA_SPACE:
            {
                STRING.tAppendCStr( sArg.hText, pacCommaSpace );
                break;
            }
            case WS_ALERTS_MSG_ELEM_TAIL_PERIOD_SPACE:
            {
                STRING.tAppendCStr( sArg.hText, pacPeriodSpace );
                break;
            }
            case WS_ALERTS_MSG_ELEM_TAIL_NO_SUFFIX:
            default:
                break;

        }

        if( psMsgElem->sTextFormat.bIsBrake == TRUE )
        {
            STRING.tAppendCStr( sArg.hText, pacBreak );
        }

        STRING.tAppendString( sArg.hText, psIterator->sMsgDesc.hMsgText );

        vDestroySArg( &sArg );

        bResult = TRUE;

    } while (FALSE);

    psIterator->bResult = bResult;

    return bResult;
}

/*******************************************************************************
*
*   bIsMsgDelimiter
*
*   returns TRUE if the message element is the final element
*   of the current message.
*
*******************************************************************************/
static BOOLEAN bIsMsgDelimiter(
    WS_ALERTS_MGR_MESSAGE_ELEMENT_STRUCT *psMsgElem
        )
{
    if( ( psMsgElem->bIsRegularElement == FALSE ) &&
        ( psMsgElem->uType.eType == WS_ALERTS_MSG_ELEM_TYPE_MSG_DELIMITER) )
    {
        return TRUE;
    }
    else
    {
        return FALSE;
    }
}

/*******************************************************************************
*
*   bCheckVerMin
*
*******************************************************************************/
static BOOLEAN bCheckVerMin(
    WS_ALERTS_APP_OBJECT_STRUCT *psAppObj,
    UN32 un32LutId,
    UN8 un8MinVersion
        )
{
    BOOLEAN bResult = FALSE;
    WS_ALERTS_LUT_VERSION_STRUCT *psLutVersion = NULL;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode = OSAL_ERROR;
    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

    do
    {
        WS_ALERTS_LUT_VERSION_STRUCT sSearchCriteria;

        sSearchCriteria.un32LutId = un32LutId;

        eOsalReturnCode = OSAL.eLinkedListSearch(
            psAppObj->hLUTVersionsList,
            &hEntry,
            &sSearchCriteria);

        if ((eOsalReturnCode == OSAL_SUCCESS) &&
            (hEntry != OSAL_INVALID_LINKED_LIST_ENTRY))
        {
            psLutVersion =
                (WS_ALERTS_LUT_VERSION_STRUCT*)OSAL.pvLinkedListThis(hEntry);

            if ( psLutVersion == NULL )
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WS_ALERTS_MGR_OBJECT_NAME
                    ": failed to get Lut version from LL entry"
                );
                break;
            }

            if ( psLutVersion->un8Version >= un8MinVersion )
            {
                bResult = TRUE;
            }
        }
        else
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME
                ": failed to get table version" );
        }

    }while ( FALSE );

    return bResult;
}

/*****************************************************************************
 *
 *   bInitMsgDesc
 *
 *****************************************************************************/
static BOOLEAN bInitMsgDesc(
    WS_ALERT_MSG_DESC_STRUCT *psMsgDesc
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    BOOLEAN bResult = FALSE;

    do
    {
        bResult = OSAL.bMemSet( psMsgDesc, 0, sizeof( *psMsgDesc ) );
        if (bResult == FALSE)
        {
            break;
        }

        psMsgDesc->hMsgText = STRING.hCreate( "", 0 );
        if (psMsgDesc->hMsgText == STRING_INVALID_OBJECT)
        {
            break;
        }

        eReturnCode = OSAL.eLinkedListCreate( &psMsgDesc->hMsgTypesList,
            WS_ALERTS_MGR_OBJECT_NAME":MsgTypesList",
            NULL,
            OSAL_LL_OPTION_NONE );

        if( eReturnCode != OSAL_SUCCESS )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": cannot create linked list (%s)",
                OSAL.pacGetReturnCodeName( eReturnCode ) );
            break;
        }

        eReturnCode = OSAL.eLinkedListCreate( &psMsgDesc->hMsgLocationsIDs,
            WS_ALERTS_MGR_OBJECT_NAME":MsgLocationsList",
            NULL,
            OSAL_LL_OPTION_NONE );

        if( eReturnCode != OSAL_SUCCESS )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": cannot create linked list (%s)",
                OSAL.pacGetReturnCodeName( eReturnCode ) );
            break;
        }

        eReturnCode = OSAL.eLinkedListCreate( &psMsgDesc->hPolygons,
            WS_ALERTS_MGR_OBJECT_NAME":PolygonsList",
            NULL,
            OSAL_LL_OPTION_NONE );

        if( eReturnCode != OSAL_SUCCESS )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": cannot create linked list (%s)",
                OSAL.pacGetReturnCodeName( eReturnCode ) );
            break;
        }

        bResult = TRUE;

    } while (FALSE);

    return bResult;
}

/*****************************************************************************
 *
 *   vDestroyMsgTypesEntry
 *
 *****************************************************************************/
static void vDestroyMsgTypesEntry(
    WS_ALERTS_MSG_TYPES_ENTRY_STRUCT *psMsgTypesEntry
        )
{
    if( psMsgTypesEntry != NULL )
    {
        STRING_vDestroy( psMsgTypesEntry->hText );
        STRING.vDestroy( psMsgTypesEntry->hMsgType );
        OSAL.vLinkedListMemoryFree( psMsgTypesEntry );
    }
}

/*****************************************************************************
 *
 *   vDestroyMsgDesc
 *
 *****************************************************************************/
static void vDestroyMsgDesc(
    WS_ALERT_MSG_DESC_STRUCT *psMsgDesc
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

    if( psMsgDesc->hMsgText != STRING_INVALID_OBJECT )
    {
        STRING_vDestroy( psMsgDesc->hMsgText );
        psMsgDesc->hMsgText = STRING_INVALID_OBJECT;
    }

    if( psMsgDesc->hMsgTypesList != OSAL_INVALID_OBJECT_HDL )
    {
        eReturnCode = OSAL.eLinkedListRemoveAll(
            psMsgDesc->hMsgTypesList,
            (OSAL_LL_RELEASE_HANDLER)vDestroyMsgTypesEntry);

        if( eReturnCode == OSAL_SUCCESS )
        {
            eReturnCode = OSAL.eLinkedListDelete( psMsgDesc->hMsgTypesList );
            if( eReturnCode == OSAL_SUCCESS )
            {
                psMsgDesc->hMsgTypesList = OSAL_INVALID_OBJECT_HDL;
            }
            else
            {
                SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                    WS_ALERTS_MGR_OBJECT_NAME": cannot delete linked list (%s)",
                    OSAL.pacGetReturnCodeName( eReturnCode ) );
            }
        }
        else
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": cannot cleanup linked list (%s)",
                OSAL.pacGetReturnCodeName( eReturnCode ) );
        }
    }

    if( psMsgDesc->hMsgLocationsIDs != OSAL_INVALID_OBJECT_HDL )
    {
        eReturnCode = OSAL.eLinkedListRemoveAll(
            psMsgDesc->hMsgLocationsIDs, NULL );

        if( eReturnCode != OSAL_SUCCESS )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": cannot cleanup linked list (%s)",
                OSAL.pacGetReturnCodeName( eReturnCode ) );
        }

        eReturnCode = OSAL.eLinkedListDelete( psMsgDesc->hMsgLocationsIDs );

        if( eReturnCode != OSAL_SUCCESS )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": cannot delete linked list (%s)",
                OSAL.pacGetReturnCodeName( eReturnCode ) );
        }

        psMsgDesc->hMsgLocationsIDs = OSAL_INVALID_OBJECT_HDL;
    }

    if( psMsgDesc->hPolygons != OSAL_INVALID_OBJECT_HDL )
    {
        eReturnCode = OSAL.eLinkedListRemoveAll(
            psMsgDesc->hPolygons,
            ( OSAL_LL_RELEASE_HANDLER )NULL );

        if( eReturnCode != OSAL_SUCCESS )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": cannot cleanup linked list (%s)",
                OSAL.pacGetReturnCodeName( eReturnCode ) );
        }

        eReturnCode = OSAL.eLinkedListDelete( psMsgDesc->hPolygons );

        if( eReturnCode != OSAL_SUCCESS )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": cannot delete linked list (%s)",
                OSAL.pacGetReturnCodeName( eReturnCode ) );
        }

        psMsgDesc->hPolygons = OSAL_INVALID_OBJECT_HDL;
    }

    return;
}

/*****************************************************************************
 *
 *   vDestroySArg
 *
 *****************************************************************************/
static void vDestroySArg(
    WS_ALERTS_DB_QUERY_RESULT_STRUCT *psArg
        )
{
    if( psArg->hText != STRING_INVALID_OBJECT )
    {
        STRING_vDestroy( psArg->hText );
        psArg->hText = STRING_INVALID_OBJECT;
    }

    if( psArg->hExtType != STRING_INVALID_OBJECT )
    {
        STRING_vDestroy( psArg->hExtType );
        psArg->hExtType = STRING_INVALID_OBJECT;
    }

    if( psArg->hExtRef != STRING_INVALID_OBJECT )
    {
        STRING_vDestroy( psArg->hExtRef );
        psArg->hExtRef = STRING_INVALID_OBJECT;
    }

    return;
}

/*****************************************************************************
 *
 *   hGetDateTimeString
 *
 *****************************************************************************/
static STRING_OBJECT hGetDateTimeString(
    TIME_T tSecs
        )
{
    char acBuf[OSAL_ASCBUFSIZE];
    char *pcOutput;
    STRING_OBJECT hResult = STRING_INVALID_OBJECT;

    pcOutput = OSAL.ctime_r(&tSecs, acBuf);
    if (pcOutput != acBuf)
    {
        SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
            WS_ALERTS_MGR_OBJECT_NAME": cannot make date and time string" );
    }
    else
    {
        size_t tLength = 0;

        tLength = strlen(acBuf);
        hResult = STRING.hCreate(acBuf, tLength);
    }

    return hResult;
}

/*****************************************************************************
 *
 *   bProcessMessage
 *
 *****************************************************************************/
static BOOLEAN bProcessMessage(
    WS_ALERTS_APP_OBJECT_STRUCT *psAppObj,
    WS_ALERTS_MSG_SET_DESC_STRUCT *psMsgSetDesc,
    WS_ALERTS_MGR_MESSAGE_ELEMENT_STRUCT *psMsgElem,
    WS_ALERTS_DB_QUERY_RESULT_STRUCT *psArg,
    WS_ALERT_MSG_DESC_STRUCT *psMsgDesc
        )
{
    BOOLEAN bSuccess = FALSE;
    UN32 un32LocationKey = 0;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    UN8 un8LUTClassId = 0;
    UN8  un8LUTTableId = 0;
    const char *pacTableName = NULL;
    int iReturn = 0;

    do
    {
        if( psMsgElem->bIsRegularElement ==TRUE )
        {
            un8LUTClassId = WS_ALERTS_GET_CLASS_ID_FROM_LUT_ID(
                psMsgElem->uType.sLutDesc.un32DecodeTableId);

            pacTableName = STRING.pacCStr(psMsgElem->uType.sLutDesc.hDecodeTableName);
            if(pacTableName == NULL)
            {
                break;
            }

            switch( un8LUTClassId )
            {
                case WS_ALERTS_LUT_CLASS_ALERT_TYPE:
                {
                    WS_ALERTS_MSG_TYPES_ENTRY_STRUCT *psMsgTypesEntry = NULL;
                    char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];

                    iReturn = snprintf( &psAppObj->acBuffer[0],
                        sizeof( psAppObj->acBuffer ),
                        WS_ALERTS_SELECT_ALERT_TYPE_ROW,
                        pacTableName,
                        psMsgElem->uType.sLutDesc.un16EntryId);

                    if (iReturn <= 0)
                    {
                        break;
                    }

                    bSuccess = SQL_INTERFACE.bQuery(
                        psAppObj->hSQLRefConnection, &psAppObj->acBuffer[0],
                        ( SQL_QUERY_RESULT_HANDLER )bProcessAlertTypeElement,
                        ( void * )psArg );

                    if( bSuccess != TRUE )
                    {
                        printf(WS_ALERTS_MGR_OBJECT_NAME": bProcessMessage: Unsuccessfull DB query");
                        break;
                    }

                    // Create message type entry
                    snprintf( &acName[0],
                        OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL,
                        WS_ALERTS_MGR_OBJECT_NAME"MsgTypeEntry:%u",
                        psMsgElem->uType.sLutDesc.un16EntryId );

                    psMsgTypesEntry = (WS_ALERTS_MSG_TYPES_ENTRY_STRUCT *)
                        OSAL.pvLinkedListMemoryAllocate(
                            &acName[0],
                            sizeof(WS_ALERTS_MSG_TYPES_ENTRY_STRUCT),
                            FALSE );

                    if( psMsgTypesEntry != NULL )
                    {
                        psMsgTypesEntry->hMsgType = psArg->hExtRef;
                        psMsgTypesEntry->hText = STRING.hDuplicate(psArg->hText);

                        eReturnCode = OSAL.eLinkedListAdd( psMsgDesc->hMsgTypesList,
                            OSAL_INVALID_LINKED_LIST_ENTRY_PTR, ( void * )psMsgTypesEntry );

                        if( eReturnCode != OSAL_SUCCESS )
                        {
                            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                                WS_ALERTS_MGR_OBJECT_NAME": cannot add entry to list (%s)",
                                OSAL.pacGetReturnCodeName( eReturnCode ) );

                            STRING.vDestroy(psMsgTypesEntry->hText);
                            OSAL.vLinkedListMemoryFree( psMsgTypesEntry );
                            break;
                        }
                    }

                    //Detach ExtRef STRING OBJECT. psMsgDesc is responsible for it.
                    psArg->hExtRef = STRING_INVALID_OBJECT;

                    if ( psMsgDesc->tPriority < psArg->un8Priority )
                    {
                        psMsgDesc->tPriority = psArg->un8Priority;
                    }
                    break;
                }
                case WS_ALERTS_LUT_CLASS_ALERT_LOCATION:
                {
                    iReturn = snprintf( &psAppObj->acBuffer[0],
                        sizeof( psAppObj->acBuffer ),
                        WS_ALERTS_SELECT_ALERT_LOCATION_ROW,
                        pacTableName,
                        psMsgElem->uType.sLutDesc.un16EntryId,
                        psMsgElem->uType.sLutDesc.un16EntryId2);

                    if (iReturn <= 0)
                    {
                        break;
                    }

                    bSuccess = SQL_INTERFACE.bQuery(
                        psAppObj->hSQLRefConnection, &psAppObj->acBuffer[0],
                        ( SQL_QUERY_RESULT_HANDLER )bProcessAlertLocationElement,
                        ( void * )psArg );

                    if( bSuccess != TRUE )
                    {
                        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                            WS_ALERTS_MGR_OBJECT_NAME": SQL request failed");
                    }

                    un8LUTTableId = WS_ALERTS_GET_TAB_ID_FROM_LUT_ID(
                        psMsgElem->uType.sLutDesc.un32DecodeTableId);

                    un32LocationKey = WS_ALERTS_GENERATE_LOCATION_ID( un8LUTTableId,
                        psMsgElem->uType.sLutDesc.un16EntryId,
                        psMsgElem->uType.sLutDesc.un16EntryId2);

                    eReturnCode = OSAL.eLinkedListAdd( psMsgDesc->hMsgLocationsIDs,
                        OSAL_INVALID_LINKED_LIST_ENTRY_PTR, ( void * )un32LocationKey );

                    if( eReturnCode != OSAL_SUCCESS )
                    {
                        SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                            WS_ALERTS_MGR_OBJECT_NAME": cannot add entry to list (%s)",
                            OSAL.pacGetReturnCodeName( eReturnCode )
                            );
                        break;
                    }

                    break;
                }
                case WS_ALERTS_LUT_CLASS_COMMON_PHRASE_CODE:
                {
                    iReturn = snprintf( &psAppObj->acBuffer[0],
                        sizeof( psAppObj->acBuffer ),
                        WS_ALERTS_SELECT_ALERT_COMMON_PHRASE_ROW,
                        pacTableName,
                        psMsgElem->uType.sLutDesc.un16EntryId);

                    if (iReturn <= 0)
                    {
                        break;
                    }

                    bSuccess =
                        SQL_INTERFACE.bQuery(
                            psAppObj->hSQLRefConnection, &psAppObj->acBuffer[0],
                            ( SQL_QUERY_RESULT_HANDLER )bProcessAlertCommonPhraseElement,
                            ( void * )psArg );

                    if( bSuccess != TRUE )
                    {
                        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                            WS_ALERTS_MGR_OBJECT_NAME": SQL request failed");
                    }

                    break;
                }
            }
        }
        else
        {
            switch( psMsgElem->uType.eType )
            {
                case WS_ALERTS_MSG_ELEM_TYPE_MSG_DELIMITER:
                {
                    break;
                }
                case WS_ALERTS_MSG_ELEM_TYPE_START_TIME_ELEM:
                {
                    psArg->hText = hGetDateTimeString(psMsgElem->uData.tEventTime);
                    psMsgDesc->tEventStartTime = psMsgElem->uData.tEventTime;
                    break;
                }
                case WS_ALERTS_MSG_ELEM_TYPE_END_TIME_ELEM:
                {
                    psArg->hText = hGetDateTimeString(psMsgElem->uData.tEventTime);
                    psMsgDesc->tEventEndTime = psMsgElem->uData.tEventTime;
                    break;
                }
                case WS_ALERTS_MSG_ELEM_TYPE_GENERIC_TIME_ELEM:
                {
                    psArg->hText = hGetDateTimeString(psMsgElem->uData.tEventTime);
                    break;
                }
                case WS_ALERTS_MSG_ELEM_TYPE_POLYGON_ELEM:
                {
                    eReturnCode = OSAL.eLinkedListAdd(
                        psMsgDesc->hPolygons, OSAL_INVALID_LINKED_LIST_ENTRY_PTR,
                        ( void * )&psMsgElem->uData.sPolygonDesc);

                    if( eReturnCode != OSAL_SUCCESS )
                    {
                        SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                            WS_ALERTS_MGR_OBJECT_NAME": cannot add entry to list (%s)",
                            OSAL.pacGetReturnCodeName(eReturnCode) );
                    }
                    else
                    {
                        psMsgDesc->bPolygon = TRUE;
                    }
                    break;
                }
                case WS_ALERTS_MSG_ELEM_TYPE_TEXT_ELEM:
                {
                    psArg->hText = STRING.hDuplicate(psMsgElem->uData.hText);
                    break;
                }
                case WS_ALERTS_MSG_ELEM_TYPE_PHONE_NUMBER_ELEM:
                {
                    psArg->hText = STRING.hDuplicate(psMsgElem->uData.hText);
                    break;
                }
                case WS_ALERTS_MSG_ELEM_TYPE_SYSTEM_PRIORITY_ELEM:
                {
                    psArg->un8Priority = psMsgElem->uData.un8Priority;
                    break;
                }
                case WS_ALERTS_MSG_ELEM_TYPE_PHONEME_ELEM:
                case WS_ALERTS_MSG_ELEM_TYPE_LARGE_CONTAINER_ELEM:
                case WS_ALERTS_MSG_ELEM_TYPE_SMALL_CONTAINER_ELEM:
                default:
                    break;
            }
        }
    }
    while (FALSE);

    return TRUE;
}

/*****************************************************************************
 *
 *   bProcessAlertLocationElement
 *
 *****************************************************************************/
static BOOLEAN bProcessAlertLocationElement(
    SQL_QUERY_COLUMN_STRUCT *psColumns,
    N32 n32NumberOfColumns,
    void *pvArg
        )
{
    WS_ALERTS_DB_QUERY_RESULT_STRUCT *psArg =
        ( WS_ALERTS_DB_QUERY_RESULT_STRUCT * )pvArg;
    BOOLEAN bOk = FALSE;
    size_t tLength = 0;

    tLength = strlen(
        ( const char * ) psColumns[WS_ALERTS_DB_LOCATION_LTEXT].uData.sCString.pcData
              );
    psArg->hText = STRING.hCreate(
        ( const char* )psColumns[WS_ALERTS_DB_LOCATION_LTEXT].uData.sCString.pcData,
        tLength);

    return bOk;
}

/*****************************************************************************
 *
 *   bProcessAlertTypeElement
 *
 *****************************************************************************/
static BOOLEAN bProcessAlertTypeElement(
    SQL_QUERY_COLUMN_STRUCT *psColumns,
    N32 n32NumberOfColumns,
    void *pvArg
        )
{
    WS_ALERTS_DB_QUERY_RESULT_STRUCT *psArg =
        ( WS_ALERTS_DB_QUERY_RESULT_STRUCT * )pvArg;
    size_t tLength = 0;

    tLength =
        strlen( ( const char * )
            psColumns[WS_ALERTS_DB_ALERT_TYPE_EXTREF].uData.sCString.pcData );
    psArg->hExtRef =
        STRING.hCreate(
            ( const char * )psColumns[WS_ALERTS_DB_ALERT_TYPE_EXTREF].uData.sCString.pcData,
            tLength );
    psArg->un8Priority =
        ( UN8 )
        psColumns[WS_ALERTS_DB_ALERT_TYPE_APRIORITY].uData.sUN32.un32Data;

    tLength =
        strlen( ( const char * )
            psColumns[WS_ALERTS_DB_ALERT_TYPE_ATEXT].uData.sCString.pcData );
    psArg->hText =
        STRING.hCreate(
            ( const char * )psColumns[WS_ALERTS_DB_ALERT_TYPE_ATEXT].uData.sCString.pcData,
            tLength);

    return (TRUE);
}

/*****************************************************************************
 *
 *   bProcessAlertCommonPhraseElement
 *
 *****************************************************************************/
static BOOLEAN bProcessAlertCommonPhraseElement(
    SQL_QUERY_COLUMN_STRUCT *psColumns,
    N32 n32NumberOfColumns,
    void *pvArg
        )
{
    WS_ALERTS_DB_QUERY_RESULT_STRUCT *psArg =
        ( WS_ALERTS_DB_QUERY_RESULT_STRUCT * )pvArg;
    size_t tLength = 0;
    tLength =
        strlen( ( const char * )
                psColumns[WS_ALERTS_DB_ALERT_TYPE_CPTEXT].uData.sCString.pcData );
    psArg->hText =
        STRING.hCreate(
            ( const char * )psColumns[WS_ALERTS_DB_ALERT_TYPE_CPTEXT].uData.sCString.pcData,
            tLength);

    return (TRUE);
}

/*****************************************************************************
 *
 *   bStoreMsgInDB
 *
 *****************************************************************************/
static BOOLEAN bStoreMsgInDB(
    WS_ALERTS_APP_OBJECT_STRUCT *psAppObj,
    WS_ALERT_MSG_DESC_STRUCT *psMsgDesc,
    UN16 un16MsgSig
        )
{
    BOOLEAN bOk = FALSE;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

    do
    {
        psAppObj->tCurrentMsgId = psMsgDesc->tId = (psMsgDesc->un8StateId << 16) | un16MsgSig;

        // Insert the new row into the messages table
        bOk = SQL_INTERFACE.bExecutePreparedCommand(
            psAppObj->hSQLPersistConnection,
            WS_ALERTS_INSERT_MSG_ROW,
            ( WS_ALERTS_DB_MESSAGE_MAX_FIELDS ),
            ( PREPARED_QUERY_COLUMN_CALLBACK )bPrepareMessageColumn,
            ( void * )psMsgDesc );

        if( bOk == TRUE )
        {
            eReturnCode = OSAL.eLinkedListIterate(
                psMsgDesc->hMsgLocationsIDs,
                ( OSAL_LL_ITERATOR_HANDLER )bIterateLocationsToFill, psAppObj );
            if( (eReturnCode != OSAL_SUCCESS) &&
                (eReturnCode != OSAL_NO_OBJECTS) )
            {
                bOk = FALSE;
                break;
            }

            eReturnCode = OSAL.eLinkedListIterate(
                psMsgDesc->hMsgTypesList,
                ( OSAL_LL_ITERATOR_HANDLER )bIterateMsgTypesToFill, psAppObj );
            if( eReturnCode != OSAL_SUCCESS )
            {
                bOk = FALSE;
                break;
            }

            if( psMsgDesc->bPolygon == TRUE )
            {
                eReturnCode = OSAL.eLinkedListIterate(
                    psMsgDesc->hPolygons,
                    ( OSAL_LL_ITERATOR_HANDLER )bIteratePolygonsToFill, psAppObj );
                if( eReturnCode != OSAL_SUCCESS )
                {
                    bOk = FALSE;
                    break;
                }
            }
        }
    }
    while (FALSE);

    return bOk;
}

/*****************************************************************************
 *
 *   bIterateLocationsToFill
 *
 *****************************************************************************/
static BOOLEAN bIterateLocationsToFill(
    LOC_ID tLocationId,
    WS_ALERTS_APP_OBJECT_STRUCT *psObj
        )
{
    WS_ALERTS_LOCATION_KEY_STRUCT sLocationKey;
    BOOLEAN bOk = FALSE;
    sLocationKey.tMsgId = psObj->tCurrentMsgId;
    sLocationKey.tLocationId = tLocationId;
    // Insert the new row into the message-locations table
    bOk = SQL_INTERFACE.bExecutePreparedCommand(
        psObj->hSQLPersistConnection,
        WS_ALERTS_INSERT_MSG_LOCATION_ROW,
        WS_ALERTS_DB_MESSAGE_LOCATIONS_MAX_FIELDS,
        ( PREPARED_QUERY_COLUMN_CALLBACK )bPrepareLocationsColumn,
        ( void * )&sLocationKey );

    if( bOk == FALSE )
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            WS_ALERTS_MGR_OBJECT_NAME": SQL request failed");
    }

    return bOk;
}

/*****************************************************************************
 *
 *   bIterateMsgTypesToFill
 *
 *****************************************************************************/
static BOOLEAN bIterateMsgTypesToFill(
    WS_ALERTS_MSG_TYPES_ENTRY_STRUCT *psMsgTypesEntry,
    WS_ALERTS_APP_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bOk = FALSE;
    WS_ALERTS_DB_RESULT_STRUCT sResult;

    sResult.bResult = FALSE;
    sResult.psAppObj = psObj;
    sResult.uDbRow.sMsgTypesRow.hMsgType = psMsgTypesEntry->hMsgType;
    sResult.uDbRow.sMsgTypesRow.hText = psMsgTypesEntry->hText;
    // Insert the new row into the message-locations table
    bOk = SQL_INTERFACE.bExecutePreparedCommand(
        psObj->hSQLPersistConnection,
        WS_ALERTS_INSERT_MSG_TYPE_ROW,
        WS_ALERTS_DB_MESSAGE_TYPES_MAX_FIELDS,
        ( PREPARED_QUERY_COLUMN_CALLBACK )bPrepareMsgTypesColumn,
        ( void * )&sResult );

    if( bOk == FALSE )
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            WS_ALERTS_MGR_OBJECT_NAME": SQL request failed");
    }

    return bOk;
}

/*****************************************************************************
 *
 *   bIteratePolygonsToFill
 *
 *****************************************************************************/
static BOOLEAN bIteratePolygonsToFill(
    WS_ALERTS_POLYGON_DESC_STRUCT *psPolygon,
    WS_ALERTS_APP_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bOk = FALSE;
    psPolygon->tMsgId = psObj->tCurrentMsgId;
    // Insert the new row into the message-locations table
    bOk = SQL_INTERFACE.bExecutePreparedCommand(
        psObj->hSQLPersistConnection,
        WS_ALERTS_INSERT_MSG_POLYGONS_ROW,
        WS_ALERTS_DB_POLYGON_MAX_FIELDS - 1,
        ( PREPARED_QUERY_COLUMN_CALLBACK )bPreparePolygonColumn,
        ( void * )psPolygon );

    if( bOk == FALSE )
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            WS_ALERTS_MGR_OBJECT_NAME": SQL request failed");
    }

    return bOk;
}

/*****************************************************************************
 *
 *   bPrepareMessageColumn
 *
 *****************************************************************************/
static BOOLEAN bPrepareMessageColumn(
    SQL_COLUMN_INDEX tIndex,
    SQL_BIND_TYPE_ENUM *peType,
    size_t *ptDataSize,
    void **ppvData,
    WS_ALERT_MSG_DESC_STRUCT *psMsgDesc
        )
{
    BOOLEAN bSuccess = TRUE;
    WS_ALERTS_DB_MESSAGE_FIELDS_ENUM eCurrentField =
        ( WS_ALERTS_DB_MESSAGE_FIELDS_ENUM )tIndex;

    switch( eCurrentField )
    {
        case WS_ALERTS_DB_MESSAGE_MESSAGE_ID:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = ( void * )( size_t )psMsgDesc->tId;
        }
        break;
        case WS_ALERTS_DB_MESSAGE_STATE_ID:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = ( void * )( size_t )psMsgDesc->un8StateId;
        }
        break;
        case WS_ALERTS_DB_MESSAGE_LANG:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = ( void * )( size_t )psMsgDesc->eMsgLang;
        }
        break;
        case WS_ALERTS_DB_MESSAGE_TEXT:
        {
            *peType = SQL_BIND_TYPE_STRING_OBJECT;
            *ppvData = ( void * )( size_t )psMsgDesc->hMsgText;
        }
        break;
        case WS_ALERTS_DB_MESSAGE_START_TIME:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = ( void * )( size_t )psMsgDesc->tEventStartTime;
        }
        break;
        case WS_ALERTS_DB_MESSAGE_END_TIME:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = ( void * )( size_t )psMsgDesc->tEventEndTime;
        }
        break;
        case WS_ALERTS_DB_MESSAGE_PRIORITY:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = ( void * )( size_t )psMsgDesc->tPriority;
        }
        break;
        case WS_ALERTS_DB_MESSAGE_POLYGON:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = ( void * )( size_t )psMsgDesc->bPolygon;
        }
        break;
        case WS_ALERTS_DB_MESSAGE_VERSION:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = ( void * )( size_t )psMsgDesc->tVersion;
        }
        break;
        case WS_ALERTS_DB_MESSAGE_RATE:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = ( void * )( size_t )psMsgDesc->bIsHighRate;
        }
        break;
        case WS_ALERTS_DB_MESSAGE_MAX_FIELDS:
        default:
            bSuccess = FALSE;
            break;
    }

    return bSuccess;
}

/*****************************************************************************
 *
 *   bPrepareLocationsColumn
 *
 *****************************************************************************/
static BOOLEAN bPrepareLocationsColumn(
    SQL_COLUMN_INDEX tIndex,
    SQL_BIND_TYPE_ENUM *peType,
    size_t *ptDataSize,
    void **ppvData,
    WS_ALERTS_LOCATION_KEY_STRUCT *psLocationKey
        )
{
    BOOLEAN bSuccess = TRUE;
    WS_ALERTS_DB_MESSAGE_LOCATIONS_FIELDS_ENUM eCurrentField =
        ( WS_ALERTS_DB_MESSAGE_LOCATIONS_FIELDS_ENUM )tIndex;

    switch( eCurrentField )
    {
        case WS_ALERTS_DB_MESSAGE_LOCATIONS_MSG_ID:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = ( void * )( size_t )psLocationKey->tMsgId;
        }
        break;
        case WS_ALERTS_DB_MESSAGE_LOCATIONS_LOCATION_ID:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = ( void * )( size_t )psLocationKey->tLocationId;
        }
        break;
        case WS_ALERTS_DB_MESSAGE_LOCATIONS_MAX_FIELDS:
        default:
            bSuccess = FALSE;
            break;
    }

    return bSuccess;
}

/*****************************************************************************
 *
 *   bPrepareMsgTypesColumn
 *
 *****************************************************************************/
static BOOLEAN bPrepareMsgTypesColumn(
    SQL_COLUMN_INDEX tIndex,
    SQL_BIND_TYPE_ENUM *peType,
    size_t *ptDataSize,
    void **ppvData,
    WS_ALERTS_DB_RESULT_STRUCT *psResult
        )
{
    BOOLEAN bSuccess = TRUE;
    WS_ALERTS_DB_MESSAGE_TYPES_FIELDS_ENUM eCurrentField =
        ( WS_ALERTS_DB_MESSAGE_TYPES_FIELDS_ENUM )tIndex;

    switch( eCurrentField )
    {
        case WS_ALERTS_DB_MESSAGE_TYPES_MSG_ID:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = ( void * )( size_t )psResult->psAppObj->tCurrentMsgId;
        }
        break;
        case WS_ALERTS_DB_MESSAGE_TYPES_TYPE_TEXT:
        {
            *peType = SQL_BIND_TYPE_STRING_OBJECT;
            *ppvData = psResult->uDbRow.sMsgTypesRow.hMsgType;
        }
        break;
        case WS_ALERTS_DB_MESSAGE_TYPES_ATEXT:
        {
            *peType = SQL_BIND_TYPE_STRING_OBJECT;
            *ppvData = psResult->uDbRow.sMsgTypesRow.hText;
        }
        break;
        case WS_ALERTS_DB_MESSAGE_TYPES_MAX_FIELDS:
        default:
            bSuccess = FALSE;
            break;
    }

    return bSuccess;
}

/*****************************************************************************
 *
 *   bPreparePolygonColumn
 *
 *****************************************************************************/
static BOOLEAN bPreparePolygonColumn(
    SQL_COLUMN_INDEX tIndex,
    SQL_BIND_TYPE_ENUM *peType,
    size_t *ptDataSize,
    void **ppvData,
    WS_ALERTS_POLYGON_DESC_STRUCT *psPolygon
        )
{
    BOOLEAN bSuccess = TRUE;
    WS_ALERTS_DB_POLYGON_FIELDS_ENUM eCurrentField =
        ( WS_ALERTS_DB_POLYGON_FIELDS_ENUM )tIndex;

    switch( eCurrentField )
    {
        case WS_ALERTS_DB_POLYGON_MSG_ID:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = ( void * )(UN32)psPolygon->tMsgId;
        }
        break;
        case WS_ALERTS_DB_POLYGON_MIN_LON:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = ( void * )(UN32)psPolygon->n32MinLon;
        }
        break;
        case WS_ALERTS_DB_POLYGON_MAX_LON:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = ( void * )(UN32)psPolygon->n32MaxLon;
        }
        break;
        case WS_ALERTS_DB_POLYGON_MIN_LAT:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = ( void * )(UN32)psPolygon->n32MinLat;
        }
        break;
        case WS_ALERTS_DB_POLYGON_MAX_LAT:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = ( void * )(UN32)psPolygon->n32MaxLat;
        }
        break;
        case WS_ALERTS_DB_POLYGON_LONLAT:
        {
            *peType = SQL_BIND_TYPE_BLOB;
            *ppvData = ( void * )&psPolygon->pn32PolygonLonLat[0];
            *ptDataSize = psPolygon->tLonLatArraySize * sizeof(N32);
        }
        break;
        case WS_ALERTS_DB_POLYGON_MAX_FIELDS:
        default:
            bSuccess = FALSE;
            break;
    }

    return bSuccess;
}

/*****************************************************************************
 *
 *   vStateUpdateCompleted
 *
 *  This function is used as callback called when all AUs for specific state
 *  are processed. Instead of refreshing DSRLs upon every AU processing, we
 *  do this only after all State AUs are parsed
 *
 *****************************************************************************/
static void vStateUpdateCompleted(
    WS_ALERTS_SERVICE_OBJECT hAlertsService
        )
{
    WS_ALERTS_APP_OBJECT_STRUCT *psAppObj = NULL;

    WS_ALERTS_MGR_OBJECT_STRUCT *psObj =
        ( WS_ALERTS_MGR_OBJECT_STRUCT * )hAlertsService;

    psAppObj = psGetAppFacingObject( ( WS_ALERTS_SERVICE_OBJECT )psObj );
    if (psAppObj != NULL)
    {
        BOOLEAN bSuccess = FALSE;

        bSuccess = bRefreshDSRLs(psAppObj);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": DSRLs refresh failed"
                    );
        }

        // start timer to remove expired DSRL entries
        if ( psAppObj->un32TimerUTCSec != WS_ALERTS_CLEAN_UP_TIMER_UTC_SEC_MAX )
        {
            vStartCleanUpTimer(psObj, psAppObj->un32TimerUTCSec);
            psAppObj->un32TimerUTCSec = WS_ALERTS_CLEAN_UP_TIMER_UTC_SEC_MAX;
        }

        vReleaseAppFacingObject( psAppObj );

    }

    DATASERVICE_IMPL_vLog(WS_ALERTS_MGR_OBJECT_NAME": Completed processing AU data for state\n");

    return;
}

/*******************************************************************************
 *
 *   vRemoveOldMsgs
 *
 *   Removes obsolete messages for particular state since
 *   new data received for this state
 *
 *******************************************************************************/
static void vRemoveOldMsgs(
    WS_ALERTS_APP_OBJECT_STRUCT *psObj,
    WS_ALERTS_MSG_SET_DESC_STRUCT *psMsgSetDesc
        )
{
    BOOLEAN bSuccess = FALSE;
    int iReturn = 0;

    DATASERVICE_IMPL_vLog(WS_ALERTS_MGR_OBJECT_NAME": Remove all old messages\n");

    // Start a new transaction
    bSuccess = SQL_INTERFACE.bStartTransaction(psObj->hSQLPersistConnection);
    if (bSuccess == FALSE)
    {
        return;
    }

    do
    {
        // Specify a search box within the target area
        iReturn = snprintf( psObj->acBuffer, sizeof( psObj->acBuffer ),
            WS_ALERTS_DELETE_OBSOLETE_LOCATIONS,
            psMsgSetDesc->un8StateId, psMsgSetDesc->eLanguage,
            psMsgSetDesc->bIsHighRate );

        if (iReturn <= 0)
        {
            break;
        }

        printf(WS_ALERTS_MGR_OBJECT_NAME": SQL request %s will be performed.\n",
            psObj->acBuffer);

        bSuccess = SQL_INTERFACE.bExecuteCommand(
            psObj->hSQLPersistConnection, &psObj->acBuffer[0]);

        if( bSuccess == FALSE )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": SQL request failed");
            break;
        }

        // Specify a search box within the target area
        iReturn = snprintf( psObj->acBuffer, sizeof( psObj->acBuffer ),
            WS_ALERTS_DELETE_OBSOLETE_MSG_TYPES,
            psMsgSetDesc->un8StateId, psMsgSetDesc->eLanguage,
            psMsgSetDesc->bIsHighRate);

        if (iReturn <= 0)
        {
            break;
        }

        printf(WS_ALERTS_MGR_OBJECT_NAME": SQL request %s will be performed.\n",
            psObj->acBuffer);

        bSuccess = SQL_INTERFACE.bExecuteCommand(
            psObj->hSQLPersistConnection, &psObj->acBuffer[0]);

        if( bSuccess == FALSE )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": SQL request failed");
            break;
        }

        // Specify a search box within the target area
        iReturn = snprintf( psObj->acBuffer, sizeof( psObj->acBuffer ),
            WS_ALERTS_DELETE_OBSOLETE_POLYGONS,
            psMsgSetDesc->un8StateId, psMsgSetDesc->eLanguage,
            psMsgSetDesc->bIsHighRate);

        if (iReturn <= 0)
        {
            break;
        }

        printf(WS_ALERTS_MGR_OBJECT_NAME": SQL request %s will be performed.\n",
            psObj->acBuffer);

        bSuccess = SQL_INTERFACE.bExecuteCommand(
            psObj->hSQLPersistConnection, &psObj->acBuffer[0]);

        if( bSuccess == FALSE )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": SQL request failed");
            break;
        }

        iReturn = snprintf( psObj->acBuffer, sizeof( psObj->acBuffer ),
            WS_ALERTS_DELETE_OBSOLETE_MESSAGES,
            psMsgSetDesc->un8StateId, psMsgSetDesc->eLanguage,
            psMsgSetDesc->bIsHighRate);

        if (iReturn <= 0)
        {
            break;
        }

        printf(WS_ALERTS_MGR_OBJECT_NAME": SQL request %s will be performed.\n",
            psObj->acBuffer);

        bSuccess = SQL_INTERFACE.bExecuteCommand(
            psObj->hSQLPersistConnection, &psObj->acBuffer[0]);

        if( bSuccess == FALSE )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": SQL request failed");
        }

    } while ( FALSE );

    SQL_INTERFACE.bEndTransaction(psObj->hSQLPersistConnection);

    return;
}

/*****************************************************************************
 *
 *   bDBUpdateBegin
 *
 *   This function is part of the manager interface API and is utilized by
 *   the interface to indicate a database update has been received.
 *
 *   This API must only be called when already in the
 *   context of the alerts manager.
 *
 *   Inputs:
 *       hAlertsService - The alerts service object handle provided to
 *           the interface.
 *       un8UpdateFromVersion - The version of the database that
 *           this update is based upon
 *
 *   Output: BOOLEAN TRUE if the update is needed, FALSE on error or unwanted
 *       data
 *
 *****************************************************************************/
static BOOLEAN bDBUpdateBegin(
    SQL_INTERFACE_OBJECT hConnection,
    char *pacBuffer,
    size_t tBufferSize
        )
{
    BOOLEAN bResult = FALSE;
    int iReturn = 0;

    do
    {
        // Generate the SQL command to update the version row
        // in the database
        iReturn = snprintf(pacBuffer, tBufferSize,
            WS_ALERTS_SET_UPDATE_NUMBER, DB_UTIL_DB_UNDER_CONSTRUCTION_VER );

        if (iReturn <= 0)
        {
            break;
        }

        // update in database
        bResult = SQL_INTERFACE.bExecuteCommand(hConnection, pacBuffer);
        if (bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": Unable to set update number" );
        }

    } while (FALSE);

    return bResult;
}

/*****************************************************************************
 *
 *   bDBUpdateEnd
 *
 *   This function is part of the manager interface API and is utilized by
 *   the interface to indicate a database update has been completed and
 *   a report the new database version number.
 *
 *   This API must only be called when already in the
 *   context of the alerts manager.
 *
 *   Inputs:
 *       hConnection - SQL interface, to manipulate the opened DataBase.
 *       *pacBuffer - pointer to buffer string to store SQL commands
 *       tBufferSize - buffer size.
 *       tUpdateVersion - RFD update version number.
 *
 *   Output: BOOLEAN TRUE on success, FALSE on error
 *
 *****************************************************************************/
static BOOLEAN bDBUpdateEnd(
    SQL_INTERFACE_OBJECT hConnection,
    char *pacBuffer,
    size_t tBufferSize,
    RFD_UPDATE_VERSION tUpdateVersion
        )
{
    BOOLEAN bResult = FALSE;

    do
    {
        N32 n32Result;

        // While the DB_UTIL function will print its own error, we also print
        // one here to make it easier to trace the DB failure back to a
        // particular service.
        bResult = DB_UTIL_bUpdateTimestamp( hConnection, pacBuffer, tBufferSize );
        if ( bResult == FALSE )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                    WS_ALERTS_MGR_OBJECT_NAME": Unable to set timestamp.");
        }

        n32Result = snprintf( pacBuffer, tBufferSize, WS_ALERTS_SET_UPDATE_NUMBER,
            tUpdateVersion );
        if ( n32Result <= 0 )
        {
            bResult = FALSE;
            break;
        }

        // Updating the baseline version comes last, as this effectively stamps
        // the DB as "ready."
        bResult = SQL_INTERFACE.bExecuteCommand( hConnection, pacBuffer );
        if ( bResult == FALSE )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": Failed to update DB data version" );
            break;
        }

    } while ( FALSE );

    return bResult;
}

/*****************************************************************************
 *
 *   bDeleteRow
 *
 *****************************************************************************/
static BOOLEAN bDeleteRow(
    SQL_INTERFACE_OBJECT hConnection,
    char *pacBuffer,
    size_t tBufferSize,
    WS_ALERTS_LUT_UPDATE_DESC_STRUCT *psLutUpdateDesc
        )
{
    OSAL_LINKED_LIST_ENTRY hLabelEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    WS_ALERTS_LABEL_DESC_STRUCT *psLabelDescKey1 = NULL;
    WS_ALERTS_LABEL_DESC_STRUCT *psLabelDescKey2 = NULL;
    BOOLEAN bResult = FALSE;
    int iReturn = 0;
    const char *pacUpdateKey1 = NULL;
    const char *pacUpdateKey2 = NULL;
    const char *pacLutName = NULL;

    do
    {
        hLabelEntry = OSAL.hLinkedListFirst(psLutUpdateDesc->hLabelsList,
            ( void ** )&psLabelDescKey1);

        if( (hLabelEntry == OSAL_INVALID_LINKED_LIST_ENTRY) ||
            (psLabelDescKey1 == NULL) )
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": Invalid key"
                    );
            break;
        }

        pacLutName = STRING.pacCStr(psLutUpdateDesc->hLutName);
        pacUpdateKey1 = STRING.pacCStr(psLabelDescKey1->hLbText);

        if ( psLutUpdateDesc->un8ClassId == WS_ALERTS_LUT_LOCATION_CLASS_ID )
        {
            LOC_ID tLocId;

            hLabelEntry = OSAL.hLinkedListNext(hLabelEntry,
                ( void ** )&psLabelDescKey2);

            if( (hLabelEntry == OSAL_INVALID_LINKED_LIST_ENTRY) ||
                (psLabelDescKey2 == NULL) )
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WS_ALERTS_MGR_OBJECT_NAME": Invalid second key"
                        );
                break;
            }

            pacUpdateKey2 = STRING.pacCStr(psLabelDescKey2->hLbText);

            tLocId = WS_ALERTS_GENERATE_LOCATION_ID(
                psLutUpdateDesc->un8TableId,
                psLabelDescKey1->uCurLbValue.un16Value,
                psLabelDescKey2->uCurLbValue.un16Value);

            iReturn = snprintf(pacBuffer, tBufferSize,
                WS_ALERTS_DELETE_RTREE_ROW_BY_ID,
                WS_ALERTS_LOCATIONS_EAST_RTREE_TABLE_NAME,
                tLocId );

            if (iReturn <= 0)
            {
                return FALSE;
            }

            printf(WS_ALERTS_MGR_OBJECT_NAME": SQL request %s will be performed.\n",
                pacBuffer);

            bResult = SQL_INTERFACE.bExecuteCommand(hConnection, pacBuffer);
            if (bResult == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                    WS_ALERTS_MGR_OBJECT_NAME": Failed to delete location rtree" );
                return FALSE;
            }

            iReturn = snprintf(pacBuffer, tBufferSize,
                WS_ALERTS_DELETE_RTREE_ROW_BY_ID,
                WS_ALERTS_LOCATIONS_WEST_RTREE_TABLE_NAME,
                tLocId );

            if (iReturn <= 0)
            {
                return FALSE;
            }

            printf(WS_ALERTS_MGR_OBJECT_NAME": SQL request %s will be performed.\n",
                pacBuffer);

            bResult = SQL_INTERFACE.bExecuteCommand(hConnection, pacBuffer);
            if (bResult == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WS_ALERTS_MGR_OBJECT_NAME": Failed to delete location rtree" );
                return FALSE;
            }

            iReturn = snprintf(pacBuffer,
                tBufferSize,
                WS_ALERTS_DELETE_LOCATION_ROW,
                pacLutName,
                pacUpdateKey1,
                psLabelDescKey1->uCurLbValue.un16Value,
                pacUpdateKey2,
                psLabelDescKey2->uCurLbValue.un16Value );
        }
        else
        {
            iReturn = snprintf(pacBuffer,
                tBufferSize,
                WS_ALERTS_DELETE_ROW,
                pacLutName,
                pacUpdateKey1,
                psLabelDescKey1->uCurLbValue.un16Value );
        }

        if (iReturn <= 0)
        {
            return FALSE;
        }

        printf(WS_ALERTS_MGR_OBJECT_NAME": SQL request %s will be performed.\n",
            pacBuffer);

       bResult = SQL_INTERFACE.bExecuteCommand(hConnection, pacBuffer);
       if (bResult == FALSE)
       {
           SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
               WS_ALERTS_MGR_OBJECT_NAME": Failed to update DB data version" );
       }
    } while ( FALSE );

    return bResult;
}

/*****************************************************************************
 *
 *   bUpdateRow
 *
 *****************************************************************************/
static BOOLEAN bUpdateRow(
    SQL_INTERFACE_OBJECT hConnection,
    char *pacBuffer,
    size_t tBufferSize,
    WS_ALERTS_LUT_UPDATE_DESC_STRUCT *psLutUpdateDesc
        )
{
    int iReturn = 0;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode;
    OSAL_LINKED_LIST_ENTRY hLabelEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    STRING_OBJECT hResString = STRING_INVALID_OBJECT;
    BOOLEAN bSuccess = TRUE;
    UN32 un32Size;
    WS_ALERTS_LABEL_DESC_STRUCT *psLabelDesc = NULL;
    WS_ALERTS_LABEL_DESC_STRUCT *psLabelDescKey2 = NULL;
    const char *pacLutName = NULL;
    const char *pacResString = NULL;
    const char *pacLbText = NULL;
    const char *pacLbText2 = NULL;

    do
    {
        hResString = STRING.hCreate("",0);
        if (hResString == STRING_INVALID_OBJECT)
        {
            bSuccess = FALSE;
            break;
        }

        hLabelEntry = OSAL.hLinkedListFirst(
            psLutUpdateDesc->hLabelsList, (void **)&psLabelDesc);

        while( hLabelEntry != OSAL_INVALID_LINKED_LIST_ENTRY )
        {
            // Construct SQL request string
            if( psLabelDesc->bPresence != FALSE )
            {
                STRING.tAppendString(psLabelDesc->hLbText, hResString);
                STRING.tAppendCStr(hResString, " = ?");
            }

            hLabelEntry = OSAL.hLinkedListNext(hLabelEntry, (void **)&psLabelDesc);

            if( hLabelEntry != OSAL_INVALID_LINKED_LIST_ENTRY )
            {
                if( psLabelDesc->bPresence != FALSE )
                {
                    STRING.tAppendCStr(hResString, ", ");
                }
            }
        }

        hLabelEntry = OSAL.hLinkedListFirst(
            psLutUpdateDesc->hLabelsList, (void **)&psLabelDesc );

        pacLutName = STRING.pacCStr(psLutUpdateDesc->hLutName);
        pacResString = STRING.pacCStr(hResString);
        pacLbText = STRING.pacCStr(psLabelDesc->hLbText);

        //prepare SQL command
        if(psLutUpdateDesc->un8ClassId != WS_ALERTS_LUT_LOCATION_CLASS_ID)
        {
            iReturn = snprintf(pacBuffer,
                tBufferSize,
                WS_ALERTS_UPDATE_LUT_ROW,
                pacLutName,
                pacResString,
                pacLbText,
                psLabelDesc->uCurLbValue.un16Value );
        }
        else
        {
            hLabelEntry = OSAL.hLinkedListNext( hLabelEntry, (void **)&psLabelDescKey2 );

            pacLbText2 = STRING.pacCStr(psLabelDescKey2->hLbText);

            iReturn = snprintf(pacBuffer,
                tBufferSize,
                WS_ALERTS_UPDATE_LUT_ROW_DOUBLE_KEY,
                pacLutName,
                pacResString,
                pacLbText,
                psLabelDesc->uCurLbValue.un16Value,
                pacLbText2,
                psLabelDescKey2->uCurLbValue.un16Value);
        }

        if (iReturn <= 0)
        {
            bSuccess = FALSE;
            break;
        }

        printf(WS_ALERTS_MGR_OBJECT_NAME": SQL request %s will be performed.\n",
            pacBuffer);

        // Execute prepared statement
        eOsalReturnCode = OSAL.eLinkedListItems(psLutUpdateDesc->hLabelsList, &un32Size);
        if( eOsalReturnCode != OSAL_SUCCESS ) 
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, WS_ALERTS_MGR_OBJECT_NAME
                ": failed to get number of items for the labels list");
            bSuccess = FALSE;
            break;
        }
        hLabelEntry = OSAL.hLinkedListFirst(psLutUpdateDesc->hLabelsList, NULL);
        if( hLabelEntry == OSAL_INVALID_LINKED_LIST_ENTRY)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, WS_ALERTS_MGR_OBJECT_NAME
                ": failed to get the first entry from the labels list");
            bSuccess = FALSE;
            break;
        }
        bSuccess = SQL_INTERFACE.bExecutePreparedCommand(
            hConnection,
            pacBuffer,
            un32Size,
            ( PREPARED_QUERY_COLUMN_CALLBACK )bPrepareColumnForUpdate,
            ( void * )&hLabelEntry );

        if( bSuccess != TRUE )
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": Failed to update the row");
            break;
        }

        if ( psLutUpdateDesc->un8ClassId == WS_ALERTS_LUT_LOCATION_CLASS_ID )
        {
            bSuccess = bUpdateRtreeTables(
                hConnection,
                pacBuffer,
                tBufferSize,
                psLutUpdateDesc,
                psLabelDesc->uCurLbValue.un16Value,
                psLabelDescKey2->uCurLbValue.un16Value );

            if ( bSuccess == FALSE )
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, WS_ALERTS_MGR_OBJECT_NAME
                    ": failed to update rtree tables");
            }
        }

    }while ( FALSE );

    if (hResString != STRING_INVALID_OBJECT)
    {
        STRING.vDestroy(hResString);
    }

    return bSuccess;
}

/*****************************************************************************
 *
 *   bInsertRow
 *
 *****************************************************************************/
static BOOLEAN bInsertRow(
    SQL_INTERFACE_OBJECT hConnection,
    char *pacBuffer,
    size_t tBufferSize,
    WS_ALERTS_LUT_UPDATE_DESC_STRUCT *psLutUpdateDesc
        )
{
    int iReturn = 0;
    OSAL_LINKED_LIST_ENTRY hLabelEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    // repeating "?, " NumOfLabels times plus null symbol
    size_t tStringIndex = 0;
    BOOLEAN bFirstElement = TRUE;
    size_t tNumOfElements = 0;
    BOOLEAN bSuccess = FALSE;
    STRING_OBJECT hUpdateQuery = STRING_INVALID_OBJECT;
    WS_ALERTS_LABEL_DESC_STRUCT *psLabelDesc = NULL;
    WS_ALERTS_LABEL_DESC_STRUCT *psLabelDescKey2 = NULL;
    char *pacTempString = NULL;
    UN16 un16TempStringSize = 0;
    const char *pacUpdateString = NULL;
    const char *pacLutName = NULL;

    do
    {
        un16TempStringSize = psLutUpdateDesc->un8NumOfLabels*3 + 1;

        pacTempString = ( char * ) SMSO_hCreate(
            WS_ALERTS_MGR_OBJECT_NAME":TempString",
            un16TempStringSize,
            SMS_INVALID_OBJECT, FALSE );

        if( pacTempString == NULL )
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": Failed to allocate memory");
            bSuccess = FALSE;
            break;
        }

        hUpdateQuery = STRING.hCreate(
            WS_ALERTS_INSERT_LUT_ROW,
            strlen( WS_ALERTS_INSERT_LUT_ROW ) );

        if ( hUpdateQuery == STRING_INVALID_OBJECT )
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": can't create update string" );
            bSuccess = FALSE;
            break;
        }

        hLabelEntry = OSAL.hLinkedListFirst(
            psLutUpdateDesc->hLabelsList, (void **)&psLabelDesc );

        while( hLabelEntry != OSAL_INVALID_LINKED_LIST_ENTRY )
        {
            if( ( bFirstElement != TRUE ) && ( psLabelDesc->bPresence == TRUE ) )
            {
                STRING.tAppendCStr(hUpdateQuery, WS_ALERTS_STRING_ELEMENT_COMMA_SPACE);
                pacTempString[ tStringIndex++ ] = WS_ALERTS_STRING_ELEMENT_COMMA;
                pacTempString[ tStringIndex++ ] = WS_ALERTS_STRING_ELEMENT_SPACE;
            }

            if( psLabelDesc->bPresence == TRUE )
            {
                STRING.tAppendString(psLabelDesc->hLbText, hUpdateQuery);
                pacTempString[ tStringIndex++ ] = WS_ALERTS_STRING_ELEMENT_QUESTION_MARK;
                tNumOfElements++;
            }
            bFirstElement = FALSE;
            hLabelEntry = OSAL.hLinkedListNext( hLabelEntry, (void **)&psLabelDesc );
        }

        pacTempString[ tStringIndex++ ] = WS_ALERTS_STRING_ELEMENT_TERMINATOR;

        STRING.tAppendCStr( hUpdateQuery, WS_ALERTS_STRING_ELEMENT_BRACKETS_VALUES );
        STRING.tAppendCStr( hUpdateQuery, pacTempString );
        STRING.tAppendCStr( hUpdateQuery, WS_ALERTS_STRING_ELEMENT_BRACKET_SEMICOLON );

        hLabelEntry = OSAL.hLinkedListFirst(
            psLutUpdateDesc->hLabelsList, (void **)&psLabelDesc );

        pacUpdateString = STRING.pacCStr(hUpdateQuery);
        pacLutName = STRING.pacCStr(psLutUpdateDesc->hLutName);

        //prepare SQL command
        iReturn = snprintf(pacBuffer,
            tBufferSize,
            pacUpdateString,
            pacLutName );

        if (iReturn <= 0)
        {
            break;
        }

        printf(WS_ALERTS_MGR_OBJECT_NAME": SQL request %s will be performed.\n",
            pacBuffer );

        // Insert the new row into the LUT
        bSuccess = SQL_INTERFACE.bExecutePreparedCommand(
            hConnection,
            pacBuffer,
            tNumOfElements,
            ( PREPARED_QUERY_COLUMN_CALLBACK )bPrepareUpdateColumn,
            ( void * )&hLabelEntry );

        if( TRUE != bSuccess)
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": Failed to update the row" );
        }

        if ( psLutUpdateDesc->un8ClassId == WS_ALERTS_LUT_LOCATION_CLASS_ID )
        {
            hLabelEntry = OSAL.hLinkedListFirst(
                psLutUpdateDesc->hLabelsList, (void **)&psLabelDesc );
            hLabelEntry = OSAL.hLinkedListNext( hLabelEntry, (void **)&psLabelDescKey2 );

            bSuccess = bUpdateRtreeTables(
                hConnection,
                pacBuffer,
                tBufferSize,
                psLutUpdateDesc,
                psLabelDesc->uCurLbValue.un16Value,
                psLabelDescKey2->uCurLbValue.un16Value );

            if ( bSuccess == FALSE )
            {
                break;
            }
        }

    }while ( FALSE );

    if ( pacTempString != NULL )
    {
        SMSO_vDestroy( ( SMS_OBJECT )pacTempString );
    }

    if ( hUpdateQuery != STRING_INVALID_OBJECT )
    {
        STRING_vDestroy(hUpdateQuery);
    }

    return bSuccess;
}

/*******************************************************************************
 *
 *   bDBProcessSelectLocationRtree
 *
 *******************************************************************************/
static BOOLEAN bDBProcessSelectLocationRtree(
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    WS_ALERTS_RTREE_DESC_STRUCT *psRtreeDesc
        )
{
    BOOLEAN bOk = TRUE;
    N32 *pn32LatLon = NULL;
    size_t tIndex = 0;
    size_t tSize = 0;
    size_t tNumVertices = 0;

    if (psRtreeDesc == NULL)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            WS_ALERTS_MGR_OBJECT_NAME": Invalid input");
        return FALSE;
    }

    if( (n32NumberOfColumns != WS_ALERTS_DB_LOCATION_MAX_FIELDS) ||
        ( psColumn[WS_ALERTS_DB_LOCATION_LON].uData.sBLOB.tSize !=
          psColumn[WS_ALERTS_DB_LOCATION_LAT].uData.sBLOB.tSize ) )
    {
        return FALSE;
    }

    tSize = psColumn[WS_ALERTS_DB_LOCATION_LON].uData.sBLOB.tSize;
    tNumVertices = (size_t)( tSize / WS_ALERTS_DB_LON_BYTE_SIZE );

    do
    {
        // Create an object to store Lon and Lat arrays where the first
        // tNumVertices elements are LAT and the second set of tNumVertices
        // for LON.
        pn32LatLon = ( N32 * ) SMSO_hCreate(
            WS_ALERTS_MGR_OBJECT_NAME":LatLonObj", 2 * tSize,
            SMS_INVALID_OBJECT, FALSE );

        if( pn32LatLon == NULL )
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": Failed to allocate memory");
            bOk = FALSE;
            break;
        }

        // Copy the points data
        bOk = OSAL.bMemCpy(
            ( void * )&pn32LatLon[0],
            psColumn[WS_ALERTS_DB_LOCATION_LAT].uData.sBLOB.pvData,
            tSize );

        if( bOk == FALSE )
        {
            break;
        }

         // Copy the points data
         bOk = OSAL.bMemCpy(
             ( void * )&pn32LatLon[tNumVertices],
             psColumn[WS_ALERTS_DB_LOCATION_LON].uData.sBLOB.pvData,
             tSize );

        if( bOk == FALSE )
        {
            break;
        }
    }
    while (FALSE);

    if( bOk == TRUE )
    {
        psRtreeDesc->n32LatMin = pn32LatLon[0];
        psRtreeDesc->n32LatMax = pn32LatLon[0];
        psRtreeDesc->n32LonMin = pn32LatLon[tNumVertices];
        psRtreeDesc->n32LonMax = pn32LatLon[tNumVertices];

        for ( tIndex = 1; tIndex < tNumVertices; tIndex++)
        {
            if ( psRtreeDesc->n32LatMin > pn32LatLon[tIndex] )
            {
                psRtreeDesc->n32LatMin = pn32LatLon[tIndex];
            }

            if ( psRtreeDesc->n32LatMax < pn32LatLon[tIndex] )
            {
                psRtreeDesc->n32LatMax = pn32LatLon[tIndex];
            }

            if ( psRtreeDesc->n32LonMin > pn32LatLon[tNumVertices + tIndex] )
            {
                psRtreeDesc->n32LonMin = pn32LatLon[tNumVertices + tIndex];
            }

            if ( psRtreeDesc->n32LonMax < pn32LatLon[tNumVertices + tIndex] )
            {
                psRtreeDesc->n32LonMax = pn32LatLon[tNumVertices + tIndex];
            }
        }

        psRtreeDesc->bResult = TRUE;
    }

    if( pn32LatLon != NULL )
    {
        SMSO_vDestroy( ( SMS_OBJECT )pn32LatLon );
    }

    return TRUE;
}

/*******************************************************************************
 *
 *   bUpdateRtreeTables
 *
 *   Updates rtree tables.
 *
 *   Inputs:
 *       hConnection - valid connection to the REF DB
 *       pacBuffer - buffer to use for SQL queries
 *       tBufferSize - buffer's size
 *       psLutUpdateDesc - current update data
 *       un16Tabindex, un16Lcode - key of location to update
 *
 *   Output:
 *       TRUE on success, FALSE on error
 *
 *******************************************************************************/
static BOOLEAN bUpdateRtreeTables(
    SQL_INTERFACE_OBJECT hConnection,
    char *pacBuffer,
    size_t tBufferSize,
    WS_ALERTS_LUT_UPDATE_DESC_STRUCT *psLutUpdateDesc,
    UN16 un16Tabindex,
    UN16 un16Lcode
        )
{
    WS_ALERTS_RTREE_DESC_STRUCT sRtreeDesc;
    BOOLEAN bSuccess = FALSE;
    N32 n32LatMin, n32LatMax, n32LonMin, n32LonMax;
    LOC_ID tLocId;
    int iReturn = 0;
    const char *pacLutName = NULL;

    do
    {
        OSAL.bMemSet(&sRtreeDesc, 0, sizeof(sRtreeDesc));

        //construct location ID
        tLocId = WS_ALERTS_GENERATE_LOCATION_ID(
            psLutUpdateDesc->un8TableId,
            un16Tabindex,
            un16Lcode);

        //remove old rtree entries
        vRemoveRtreeEntryById(
            hConnection,
            pacBuffer,
            tBufferSize,
            tLocId);

        // get updated location and calculate min/max lon and lat
        // to form rtree entry
        pacLutName = STRING.pacCStr(psLutUpdateDesc->hLutName);

        iReturn = snprintf(pacBuffer, tBufferSize,
            WS_ALERTS_SELECT_LOCATION,
            pacLutName,
            un16Tabindex,
            un16Lcode
                );
        if (iReturn <= 0)
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": Failed to create SQL request" );
            break;
        }
        printf(WS_ALERTS_MGR_OBJECT_NAME": SQL request %s will be performed.\n",
            pacBuffer );

        bSuccess = SQL_INTERFACE.bQuery(
            hConnection, pacBuffer,
            (SQL_QUERY_RESULT_HANDLER)bDBProcessSelectLocationRtree,
            &sRtreeDesc );
        if ( (bSuccess == FALSE) ||
             (sRtreeDesc.bResult == FALSE) )
        {
            printf(WS_ALERTS_MGR_OBJECT_NAME": can't get complete location from DB\n");
            break;
        }

        n32LatMin = sRtreeDesc.n32LatMin;
        n32LatMax = sRtreeDesc.n32LatMax;
        n32LonMin = sRtreeDesc.n32LonMin;
        n32LonMax = sRtreeDesc.n32LonMax;

        //create rectangle if we have one point or line
        if ( n32LonMin == n32LonMax )
        {
            if ( (n32LonMax == WS_ALERTS_RTREE_MAX_LON) ||
                 (n32LonMin == WS_ALERTS_RTREE_MIN_LON) )
            {
                // Area is constrained to 180 meridian. Since max longitude
                // is equal to min, we are setting min to -180 degrees and
                // max to 180 for proper calculations down the stream
                n32LonMax = WS_ALERTS_RTREE_MAX_LON;
                n32LonMin = WS_ALERTS_RTREE_MIN_LON;
            }
            else
            {
                n32LonMin = n32LonMin - WS_ALERTS_RTREE_RANGE;
                n32LonMax = n32LonMax + WS_ALERTS_RTREE_RANGE;
            }
        }

        if ( n32LatMin == n32LatMax )
        {
            n32LatMin = n32LatMin - WS_ALERTS_RTREE_RANGE;
            n32LatMax = n32LatMax + WS_ALERTS_RTREE_RANGE;
        }

        if ( (n32LonMax > 0) && (n32LonMin < 0) )
        {
            if ( n32LonMax == WS_ALERTS_RTREE_MAX_LON )
            {
                n32LonMax = n32LonMax - WS_ALERTS_RTREE_RANGE;
            }

            if ( n32LonMin == WS_ALERTS_RTREE_MIN_LON )
            {
                n32LonMin = n32LonMin + WS_ALERTS_RTREE_RANGE;
            }

            iReturn = snprintf(pacBuffer, tBufferSize,
                WS_ALERTS_INSERT_EASTERN_LOCATION_RTREE,
                tLocId,
                n32LatMin, n32LatMax,
                n32LonMax, WS_ALERTS_RTREE_MAX_LON);

            if (iReturn <= 0)
            {
                bSuccess = FALSE;
                break;
            }

            printf(WS_ALERTS_MGR_OBJECT_NAME": SQL string %s\n", pacBuffer);

            bSuccess = SQL_INTERFACE.bExecuteCommand(hConnection, pacBuffer);
            if (bSuccess == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WS_ALERTS_MGR_OBJECT_NAME": Failed to insert location rtree");
                break;
            }

            iReturn = snprintf(pacBuffer, tBufferSize,
                WS_ALERTS_INSERT_WESTERN_LOCATION_RTREE,
                tLocId,
                n32LatMin, n32LatMax, WS_ALERTS_RTREE_MIN_LON, n32LonMin);

            if (iReturn <= 0)
            {
                bSuccess = FALSE;
                break;
            }

            printf(WS_ALERTS_MGR_OBJECT_NAME": SQL string %s\n", pacBuffer);

            bSuccess = SQL_INTERFACE.bExecuteCommand(hConnection, pacBuffer);
            if (bSuccess == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WS_ALERTS_MGR_OBJECT_NAME": Failed to insert location rtree" );
                break;
            }

        }
        else if ( (n32LonMax > 0) && (n32LonMin > 0) )
        {
            iReturn = snprintf(pacBuffer, tBufferSize,
                WS_ALERTS_INSERT_EASTERN_LOCATION_RTREE,
                tLocId,
                n32LatMin, n32LatMax, n32LonMin, n32LonMax );

            if (iReturn <= 0)
            {
                bSuccess = FALSE;
                break;
            }

            printf(WS_ALERTS_MGR_OBJECT_NAME": SQL string %s\n", pacBuffer);

            bSuccess = SQL_INTERFACE.bExecuteCommand(hConnection, pacBuffer);
            if (bSuccess == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WS_ALERTS_MGR_OBJECT_NAME": Failed to insert location rtree");
                break;
            }

        }
        else if ( (n32LonMax < 0) && (n32LonMin < 0) )
        {
            iReturn = snprintf(pacBuffer, tBufferSize,
                WS_ALERTS_INSERT_WESTERN_LOCATION_RTREE,
                tLocId,
                n32LatMin, n32LatMax, n32LonMin, n32LonMax );

            if (iReturn <= 0)
            {
                bSuccess = FALSE;
                break;
            }

            printf(WS_ALERTS_MGR_OBJECT_NAME": SQL string %s\n", pacBuffer);

            bSuccess = SQL_INTERFACE.bExecuteCommand(hConnection, pacBuffer);
            if (bSuccess == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WS_ALERTS_MGR_OBJECT_NAME": Failed to insert location rtree");
                break;
            }
        }

        bSuccess = TRUE;

    }
    while (FALSE);

    return bSuccess;
}

/*******************************************************************************
 *
 *   vRemoveRtreeEntryById
 *
 *******************************************************************************/
static void vRemoveRtreeEntryById(
    SQL_INTERFACE_OBJECT hConnection,
    char *pacBuffer,
    size_t tBufferSize,
    LOC_ID tId
        )
{
    BOOLEAN bSuccess = FALSE;
    int iReturn = 0;

    do
    {
        iReturn = snprintf(pacBuffer, tBufferSize,
            WS_ALERTS_DELETE_RTREE_ROW_BY_ID,
            WS_ALERTS_LOCATIONS_EAST_RTREE_TABLE_NAME,
            tId );

        if (iReturn <= 0)
        {
            break;
        }

        printf(WS_ALERTS_MGR_OBJECT_NAME": SQL string %s\n", pacBuffer);

        bSuccess = SQL_INTERFACE.bExecuteCommand(hConnection, pacBuffer);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": Failed to delete location rtree");
            break;
        }

        iReturn = snprintf(pacBuffer, tBufferSize,
            WS_ALERTS_DELETE_RTREE_ROW_BY_ID,
            WS_ALERTS_LOCATIONS_WEST_RTREE_TABLE_NAME,
            tId);

        if (iReturn <= 0)
        {
            break;
        }

        printf(WS_ALERTS_MGR_OBJECT_NAME": SQL string %s\n", pacBuffer);

        bSuccess = SQL_INTERFACE.bExecuteCommand(hConnection, pacBuffer);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": Failed to delete location rtree");
            break;
        }
    }
    while (FALSE);

    return;
}

/*******************************************************************************
 *
 *   n16CompareByLutId
 *
 *******************************************************************************/
static N16 n16CompareByLutId(
    WS_ALERTS_LUT_VERSION_STRUCT* psLUTVersion1,
    WS_ALERTS_LUT_VERSION_STRUCT* psLUTVersion2
        )
{
    N16 n16Result = N16_MIN;

    if ((psLUTVersion1 != NULL) && (psLUTVersion2 != NULL))
    {
        n16Result = COMPARE(psLUTVersion1->un32LutId, psLUTVersion2->un32LutId);
    }

    return n16Result;
}

/*******************************************************************************
 *
 *   bPrepareUpdateColumn
 *
 *******************************************************************************/
static BOOLEAN bPrepareUpdateColumn(
    SQL_COLUMN_INDEX tIndex,
    SQL_BIND_TYPE_ENUM *peType,
    size_t *ptDataSize,
    void **ppvData,
    OSAL_LINKED_LIST_ENTRY *phLabelEntry
        )
{
    BOOLEAN bSuccess = FALSE;
    WS_ALERTS_LABEL_DESC_STRUCT *psLabelDesc = NULL;

    do
    {
        if ( phLabelEntry == NULL )
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME
                ": invalid phLabelEntry" );
            break;
        }

        while( *phLabelEntry != OSAL_INVALID_LINKED_LIST_ENTRY )
        {
            psLabelDesc =
                (WS_ALERTS_LABEL_DESC_STRUCT*)OSAL.pvLinkedListThis(*phLabelEntry);

            if ( psLabelDesc == NULL )
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WS_ALERTS_MGR_OBJECT_NAME
                    ": failed to get label desc from LL entry");
                bSuccess = FALSE;
                break;
            }

            if ( psLabelDesc->bPresence == TRUE )
            {
                bSuccess = TRUE;
                break;
            }

            *phLabelEntry = OSAL.hLinkedListNext(*phLabelEntry, NULL);
        }

        if( bSuccess== FALSE )
        {
            break;
        }

        switch( psLabelDesc->eDataType )
        {
            case WS_ALERTS_PARTITION_DTYPE_INT:
            {
                *peType = SQL_BIND_TYPE_UN32;
                *ppvData = (void *)(size_t)psLabelDesc->uCurLbValue.un16Value;
            }
            break;
            case WS_ALERTS_PARTITION_DTYPE_STRING:
            {
                *peType = SQL_BIND_TYPE_C_STRING;
                *ppvData = (void *)psLabelDesc->uCurLbValue.pcValue;
            }
            break;
            case WS_ALERTS_PARTITION_DTYPE_BAUDOT_STRING:
            {
                *peType = SQL_BIND_TYPE_STRING_OBJECT;
                *ppvData = psLabelDesc->uCurLbValue.hValue;
            }
            break;
            case WS_ALERTS_PARTITION_DTYPE_ARRAY_OF_INT:
            {
                *peType = SQL_BIND_TYPE_BLOB;
                *ppvData = ( void * )&psLabelDesc->uCurLbValue.pn32Data[0];
                *ptDataSize = psLabelDesc->tCurrentArrayNum * sizeof(N32);
            }
            break;
            default:
                break;

        }
        *phLabelEntry = OSAL.hLinkedListNext(*phLabelEntry, NULL);
    }
    while (FALSE);

    return bSuccess;
}

/*******************************************************************************
 *
 *   bGetCurrentDbVersion
 *
 *******************************************************************************/
static BOOLEAN bGetCurrentDbVersion(
    SQL_INTERFACE_OBJECT hConnection,
    char *pacBuffer,
    size_t tBufferSize,
    STRING_OBJECT hLutName,
    N32 *pn32DBVer
        )
{
    WS_ALERTS_DB_QUERY_RESULT_STRUCT sQueryResult;
    char acLUTNameBuffer[WS_ALERTS_LUT_NAME_BUFFER_LEN];
    BOOLEAN bSuccess = FALSE;
    int iReturn;

    if ( (hConnection == SQL_INTERFACE_INVALID_OBJECT) ||
         (pn32DBVer == NULL) )
    {
        return FALSE;
    }

    do
    {
        const char *pacLutname;
        sQueryResult.bResultantRows = FALSE;
        *pn32DBVer = N32_MIN;

        pacLutname = STRING.pacCStr(hLutName);

        iReturn = snprintf( acLUTNameBuffer, sizeof(acLUTNameBuffer),
            WS_ALERTS_LUT_NAME_STRING, pacLutname );

        if (iReturn <= 0)
        {
            break;
        }

        iReturn = snprintf(pacBuffer, tBufferSize,
            WS_ALERTS_SELECT_VERSION_FROM_LUT, &acLUTNameBuffer[0]);

        if (iReturn <= 0)
        {
            break;
        }

        // Perform the SQL query and process the result
        // (it will provide us with a data row)
        bSuccess = SQL_INTERFACE.bQuery(
            hConnection,
            pacBuffer,
            (SQL_QUERY_RESULT_HANDLER)bProcessSelectUpdateVersionResult,
            &sQueryResult);

        if ( (bSuccess == TRUE) &&
             (sQueryResult.bResultantRows == TRUE) )
        {
            *pn32DBVer = (N32)sQueryResult.uDbRow.sVersion.un32Version;
        }

    }while (FALSE);

    return bSuccess;
}

/*******************************************************************************
 *
 *   bCheckLutTable
 *
 *******************************************************************************/
static BOOLEAN bCheckLutTable(
    SQL_INTERFACE_OBJECT hConnection,
    char *pacBuffer,
    size_t tBufferSize,
    WS_ALERTS_LUT_UPDATE_DESC_STRUCT *psLutUpdateDesc
        )
{
    BOOLEAN bSuccess = FALSE;

    //check if provided table is present in DB
    bSuccess = bLUTTableExist( hConnection, pacBuffer, tBufferSize, psLutUpdateDesc );

    if ( bSuccess == FALSE )
    {
        //need to create new table
        bSuccess = bCreateLUTTable( hConnection, pacBuffer, tBufferSize, psLutUpdateDesc );

        if ( bSuccess == FALSE )
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": failed to create new LUT table" );
            return bSuccess;
        }
    }
    else
    {
        bSuccess = bCheckLabelsList( hConnection, pacBuffer, tBufferSize, psLutUpdateDesc);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": failed to check labels from updates");
            return bSuccess;
        }

        bSuccess = bChangeLUTVersion( hConnection, pacBuffer, tBufferSize, psLutUpdateDesc);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": failed to change LUT version");
            return bSuccess;
        }
    }

    return bSuccess;
}

/*******************************************************************************
*
*   bLUTTableExist
*
*   This function is need to check the table with name like CxTxLx on existance in DB.
*   Name is based on Class, Tab and Lang versions (x).
*
*   return TRUE - if table exist, FALSE - if table not exist
*******************************************************************************/
static BOOLEAN bLUTTableExist(
    SQL_INTERFACE_OBJECT hConnection,
    char *pacBuffer,
    size_t tBufferSize,
    WS_ALERTS_LUT_UPDATE_DESC_STRUCT *psLutUpdateDesc
        )
{
    int iReturn = 0;
    BOOLEAN bSuccess = FALSE;
    BOOLEAN bTableExist = FALSE;

    do
    {
        const char *pacLutname;

        pacLutname = STRING.pacCStr(psLutUpdateDesc->hLutName);

        //check if provided table is present in DB
        iReturn = snprintf(pacBuffer, tBufferSize,
                  WS_ALERTS_SELECT_LUT_DESC2,
                  pacLutname );
        if (iReturn <= 0)
        {
            break;
        }

        bSuccess = SQL_INTERFACE.bQuery( hConnection,
            pacBuffer,
            (SQL_QUERY_RESULT_HANDLER)bProcessCheckTableOnExistance,
            &bTableExist);

        if ( bSuccess != TRUE )
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": failed to check table existance" );
            // in case of error - return TRUE, to prevent table creation.
            bTableExist = TRUE;
            break;
        }

    }while ( FALSE );

    return bTableExist;
}

/*******************************************************************************
*
*   bProcessCheckTableOnExistance
*
*******************************************************************************/
static BOOLEAN bProcessCheckTableOnExistance (
    SQL_QUERY_COLUMN_STRUCT *psColumns,
    N32 n32NumberOfColumns,
    BOOLEAN *pbTableExist
        )
{
    if ( n32NumberOfColumns > 0 )
    {
        //table exist
        *pbTableExist = TRUE;
    }
    else
    {
        //if no matches found, CB function will not be called, but in case of API changes:
        *pbTableExist = FALSE;
    }

    return TRUE;
}

/*******************************************************************************
*
*   bCreateLUTTable
*
*******************************************************************************/
static BOOLEAN bCreateLUTTable(
    SQL_INTERFACE_OBJECT hConnection,
    char *pacBuffer,
    size_t tBufferSize,
    WS_ALERTS_LUT_UPDATE_DESC_STRUCT *psLutUpdateDesc
        )
{
    size_t tIndex = 0, tFieldCount = 0;
    WS_ALERTS_LABEL_DESC_STRUCT *psLabelDescArr = NULL;
    WS_ALERTS_LABEL_DESC_STRUCT *psEntry = NULL;
    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    BOOLEAN bRetValue = FALSE;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    BOOLEAN bSuccess = FALSE;
    int iRetVal = 0;
    char *pacCurBuffer = pacBuffer;
    const char *pacLutname = NULL;
    size_t tRemainingSize = tBufferSize;

    do
    {
        if ( (psLutUpdateDesc == NULL) || (psLutUpdateDesc->hLabelsList == OSAL_INVALID_OBJECT_HDL) )
        {
            break; //invalid parameters
        }

        pacLutname = STRING.pacCStr(psLutUpdateDesc->hLutName);

        //create SMS object: array of WS_ALERTS_LABEL_DESC_STRUCT, with size of OSAL.eLinkedListItems()

        eReturnCode = OSAL.eLinkedListItems( psLutUpdateDesc->hLabelsList, &tFieldCount );

        if ( eReturnCode != OSAL_SUCCESS )
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": failed to get field count of new LUT table" );
            break;
        }

        if ( tFieldCount == 0 )
        {
            break; //table description empty
        }

        psLabelDescArr = (WS_ALERTS_LABEL_DESC_STRUCT*)
            SMSO_hCreate( WS_ALERTS_MGR_OBJECT_NAME":LabelDescArr",
                sizeof(WS_ALERTS_LABEL_DESC_STRUCT) * tFieldCount,
                SMS_INVALID_OBJECT, FALSE );

        if (psLabelDescArr == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": cannot create psLabelDescArr object" );
            break;
        }

        //iterate linked lilst psPartDesc->hLabelsList and fill the array
        hEntry = OSAL.hLinkedListFirst( psLutUpdateDesc->hLabelsList, (void **)&psEntry );
        //list is linear
        while( hEntry != OSAL_INVALID_LINKED_LIST_ENTRY )
        {
            psLabelDescArr[tIndex].un8LabelId = psEntry->un8LabelId;
            psLabelDescArr[tIndex].hLbText = psEntry->hLbText;
            psLabelDescArr[tIndex].eDataType = psEntry->eDataType;
            psLabelDescArr[tIndex].uCurLbValue = psEntry->uCurLbValue;
            psLabelDescArr[tIndex].tCurrentArrayNum = psEntry->tCurrentArrayNum;
            psLabelDescArr[tIndex].uCurLbValue.pn32Data = psEntry->uCurLbValue.pn32Data;

            tIndex++;
            hEntry = OSAL.hLinkedListNext( hEntry, (void **)&psEntry );
        }

        //create new table
        iRetVal = snprintf(pacCurBuffer, tRemainingSize,
            WS_ALERTS_CREATE_LUT_TABLE_BEGIN, pacLutname );
        if ( iRetVal <= 0 )
        {
            break;
        }

        // Move string buffer pointer and recalculate remaining size
        tRemainingSize -= iRetVal;
        pacCurBuffer += iRetVal;

        //add columns
        for ( tIndex = 0; tIndex < tFieldCount; ++tIndex )
        {
            const char *pacColumnType = "";
            const char *pacColumnName = "";

            switch (psLabelDescArr[tIndex].eDataType)
            {
                case WS_ALERTS_PARTITION_DTYPE_INT:
                {
                    pacColumnType = WS_ALERTS_TYPE_INT;
                    break;
                }
                case WS_ALERTS_PARTITION_DTYPE_STRING:
                case WS_ALERTS_PARTITION_DTYPE_BAUDOT_STRING:
                {
                    pacColumnType = WS_ALERTS_TYPE_TEXT;
                    break;
                }
                case WS_ALERTS_PARTITION_DTYPE_ARRAY_OF_INT:
                {
                    pacColumnType = WS_ALERTS_TYPE_BLOB;
                    break;
                }
                default:
                    break; //data not valid
            }

            pacColumnName = STRING.pacCStr(psLabelDescArr[tIndex].hLbText);

            iRetVal = snprintf(pacCurBuffer, tRemainingSize,
                               WS_ALERTS_CREATE_LUT_TABLE_JOIN, pacColumnName, pacColumnType );
            if ( iRetVal <= 0 )
            {
                break;
            }

            // Move string buffer pointer and recalculate remaining size
            tRemainingSize -= iRetVal;
            pacCurBuffer += iRetVal;

            //if it is not last pair of elements, we should place ',' after it
            if ( tIndex != (tFieldCount - 1) )
            {
                if (tRemainingSize > 0)
                {
                    strcat(pacCurBuffer, ",");
                    --tRemainingSize;
                    ++pacCurBuffer;
                }
                else
                {
                    break;
                }
            }
        }

        if(tRemainingSize > 3)
        {
            strcpy(pacCurBuffer, " );" );
            // move buffer pointer after copy
            pacCurBuffer += 3;
            tRemainingSize -= 3;
        }
        else
        {
            break;
        }

        printf(WS_ALERTS_MGR_OBJECT_NAME": SQL command \n%s\nwill be executed",
            pacBuffer);

        bSuccess = SQL_INTERFACE.bExecuteCommand(hConnection, pacBuffer);
        if ( bSuccess == FALSE )
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": Unable to execute sql command: WS_ALERTS_CREATE_LUT_TABLE" );
            break;
        }

        //update data_version table
        iRetVal = snprintf(pacBuffer, tBufferSize,
            WS_ALERTS_INSERT_LUT_DATA_VERSION,
            pacLutname,
            psLutUpdateDesc->un16UpdateVersion);
        if ( iRetVal <= 0 )
        {
            break;
        }

        bSuccess = SQL_INTERFACE.bExecuteCommand(hConnection, pacBuffer);
        if ( bSuccess == FALSE )
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": Unable to execute sql command" );
            break;
        }

        //update scheme for newly created LUT

        for ( tIndex = 0; tIndex < tFieldCount; ++tIndex )
        {
            const char *pacColumnType = "";
            const char *pacColumnName = "";

            switch (psLabelDescArr[tIndex].eDataType)
            {
                case WS_ALERTS_PARTITION_DTYPE_INT:
                {
                    pacColumnType = WS_ALERTS_TYPE_INT;
                    break;
                }
                case WS_ALERTS_PARTITION_DTYPE_STRING:
                case WS_ALERTS_PARTITION_DTYPE_BAUDOT_STRING:
                {
                    pacColumnType = WS_ALERTS_TYPE_TEXT;
                    break;
                }
                case WS_ALERTS_PARTITION_DTYPE_ARRAY_OF_INT:
                {
                    pacColumnType = WS_ALERTS_TYPE_BLOB;
                    break;
                }
                default:
                    break; //data not valid
            }

            //check if we have a valid type value
            if( strcmp(pacColumnType, "") == 0 )
            {
                bSuccess = FALSE;
                break;
            }

            pacColumnName = STRING.pacCStr(psLabelDescArr[tIndex].hLbText);

            //update LUT_desc table
            iRetVal = snprintf(pacBuffer, tBufferSize,
                WS_ALERTS_INSERT_ROW_LUT_DESC,
                pacLutname,
                psLabelDescArr[tIndex].un8LabelId,
                pacColumnName,
                pacColumnType
                );

            if ( iRetVal <= 0 )
            {
                bSuccess = FALSE;
                break;
            }

            bSuccess = SQL_INTERFACE.bExecuteCommand(hConnection, pacBuffer);
            if ( bSuccess == FALSE )
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WS_ALERTS_MGR_OBJECT_NAME": Unable to execute sql command" );
                break;
            }
        }

        if(bSuccess != FALSE)
        {
            bRetValue = TRUE;
        }

    }while(FALSE);

    //destroy sms object
    if ( psLabelDescArr != NULL )
    {
        SMSO_vDestroy( ( SMS_OBJECT )psLabelDescArr );
    }

    return bRetValue;
}

/*******************************************************************************
*
*       bCheckLabelsList
*
*******************************************************************************/
static BOOLEAN bCheckLabelsList(
    SQL_INTERFACE_OBJECT hConnection,
    char *pacBuffer,
    size_t tBufferSize,
    WS_ALERTS_LUT_UPDATE_DESC_STRUCT *psLutUpdateDesc
        )
{
    OSAL_OBJECT_HDL hLUTDescList = OSAL_INVALID_OBJECT_HDL;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    int iReturn = 0;
    BOOLEAN bSuccess = FALSE;
    char acLUTNameBuffer[WS_ALERTS_LUT_NAME_BUFFER_LEN];
    WS_ALERTS_LABELS_ITERATOR_STRUCT sIterator;
    const char *pacLutname;

    do
    {
        pacLutname = STRING.pacCStr(psLutUpdateDesc->hLutName);

        eReturnCode = OSAL.eLinkedListCreate( &hLUTDescList,
            WS_ALERTS_MGR_OBJECT_NAME":hLUTDescList",
            NULL,
            OSAL_LL_OPTION_NONE );
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": failed to create list hLUTDescList");
            bSuccess = FALSE;
            break;
        }

        //performing SQL query here to fill up a LUT description list
        iReturn = snprintf(acLUTNameBuffer, sizeof(acLUTNameBuffer),
                WS_ALERTS_LUT_NAME_STRING,
                pacLutname);

        if (iReturn <= 0)
        {
            bSuccess = FALSE;
            break;
        }

        iReturn = snprintf(pacBuffer, tBufferSize,
            WS_ALERTS_SELECT_LUT_DESC, &acLUTNameBuffer[0]);

        if (iReturn <= 0)
        {
            bSuccess = FALSE;
            break;
        }

        printf(WS_ALERTS_MGR_OBJECT_NAME": bCheckLabelsList: SQL query will be performed: %s\n",
            pacBuffer);

        // Perform the SQL query and process the result
        bSuccess = SQL_INTERFACE.bQuery(
            hConnection,
            pacBuffer,
            (SQL_QUERY_RESULT_HANDLER)bProcessSelectLutDescription,
            hLUTDescList);

        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": Unable to fill up hLUTDescList" );
            break;
        }

        // Populate the iterator struct with the caller's
        // arguments
        sIterator.hConnection = hConnection;
        sIterator.pacBuffer = pacBuffer;
        sIterator.tBufferSize = tBufferSize;
        sIterator.hLUTDescList = hLUTDescList;
        sIterator.bResult = FALSE;

        // Iterate the label list and check if we have
        // any updates in LUT's columns
        eReturnCode = OSAL.eLinkedListIterate(
            psLutUpdateDesc->hLabelsList,
            (OSAL_LL_ITERATOR_HANDLER)bIterateLabelDescs,
            &sIterator);
        if ((eReturnCode == OSAL_SUCCESS) &&
            (sIterator.bResult == TRUE))
        {
            bSuccess = TRUE;
        }

    }
    while(FALSE);

    if( hLUTDescList != OSAL_INVALID_OBJECT_HDL )
    {
        eReturnCode = OSAL.eLinkedListRemoveAll( hLUTDescList,
            (OSAL_LL_RELEASE_HANDLER)vReleaseDescriptionRow );

        if( eReturnCode != OSAL_SUCCESS )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": failed to cleanup"
                " list (%s)", OSAL.pacGetReturnCodeName( eReturnCode ) );
        }

        eReturnCode = OSAL.eLinkedListDelete( hLUTDescList );

        if( eReturnCode != OSAL_SUCCESS )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": failed to delete"
                " list (%s)", OSAL.pacGetReturnCodeName( eReturnCode ) );
        }

        hLUTDescList = OSAL_INVALID_OBJECT_HDL;
    }

    return bSuccess;
}

/*****************************************************************************
*
*   bIterateLabelDescs
*
*****************************************************************************/
static BOOLEAN bIterateLabelDescs (
    WS_ALERTS_LABEL_DESC_STRUCT *psLabelDesc,
    void *pvArg
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    int iReturn = 0;
    N16 n16Compare;
    char acLUTNameBuffer[WS_ALERTS_LUT_NAME_BUFFER_LEN];
    const char *pacTypeBuffer = NULL;
    WS_ALERTS_LUT_DESC_ROW *psDescRow = NULL;
    OSAL_LINKED_LIST_ENTRY hDescEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    WS_ALERTS_LABEL_DESC_STRUCT *psDescToUpd = NULL;
    BOOLEAN bSuccess = TRUE;

    WS_ALERTS_LABELS_ITERATOR_STRUCT *psIterator =
        (WS_ALERTS_LABELS_ITERATOR_STRUCT *)pvArg;

    if ((psLabelDesc == NULL) || (psIterator == NULL))
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            WS_ALERTS_MGR_OBJECT_NAME": bIterateLabelDescs() NULL in list or argument\n");
        return FALSE;
    }

    do
    {
        switch(psLabelDesc->eDataType)
        {
            case WS_ALERTS_PARTITION_DTYPE_INT:
            {
                pacTypeBuffer = WS_ALERTS_TYPE_INT;
            }
            break;

            case WS_ALERTS_PARTITION_DTYPE_STRING:
            case WS_ALERTS_PARTITION_DTYPE_BAUDOT_STRING:
            {
                pacTypeBuffer = WS_ALERTS_TYPE_TEXT;
            }
            break;

            case WS_ALERTS_PARTITION_DTYPE_ARRAY_OF_INT:
            {
                pacTypeBuffer = WS_ALERTS_TYPE_BLOB;
            }
            break;

            default:
                SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                    WS_ALERTS_MGR_OBJECT_NAME": unknown type in the update label. Type id: %d",
                    psLabelDesc->eDataType );
                bSuccess = FALSE;
            break;
        }

        if (bSuccess == FALSE)
        {
            break;
        }

        eReturnCode = OSAL.eLinkedListLinearSearch(
            psIterator->hLUTDescList,
            &hDescEntry,
            (OSAL_LL_COMPARE_HANDLER) n16LUTDescComparator,
            (void*)(size_t) psLabelDesc->un8LabelId);

        if ( (eReturnCode == OSAL_SUCCESS) &&
             (hDescEntry != OSAL_INVALID_LINKED_LIST_ENTRY) )
        {
            // We have found an appropriate record. Check the values to make sure that we have no differences
            // between the data in broadcast and DB.
            psDescRow = (WS_ALERTS_LUT_DESC_ROW *) OSAL.pvLinkedListThis(hDescEntry);

            STRING_bToUpper(psDescRow->hFieldName);
            STRING_bToUpper(psLabelDesc->hLbText);

            n16Compare = STRING.n16Compare(psDescRow->hFieldName, psLabelDesc->hLbText, TRUE);
            if(n16Compare != 0)
            {
                SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                    WS_ALERTS_MGR_OBJECT_NAME": The label name differs from name in the scheme!" );
                bSuccess = FALSE;
                break;
            }

            n16Compare = STRING.n16CompareCStr(pacTypeBuffer,0,psDescRow->hFieldType);
            if(n16Compare != 0)
            {
                SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                    WS_ALERTS_MGR_OBJECT_NAME": The label type differs from the type in the scheme!" );
                bSuccess = FALSE;
                break;
            }
        }
        else if ( eReturnCode == OSAL_OBJECT_NOT_FOUND )
        {
            psDescToUpd = psLabelDesc;
        }
        else
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": Error in searching hLUTDescList. Error: (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode) );
            bSuccess = FALSE;
            break;
        }

        if(psDescToUpd != NULL)
        {
            char acLbTextBuffer[WS_ALERTS_SHARED_BUFFER_LEN];
            const char *pacLabelText;

            pacLabelText = STRING.pacCStr(psDescToUpd->hLbText);

            iReturn = snprintf(acLbTextBuffer, sizeof(acLbTextBuffer),"%s",
                pacLabelText);

            if (iReturn <= 0)
            {
                bSuccess = FALSE;
                break;
            }

            // We have reached the end of the LUT description list and haven't found
            // an appropriate LUT description record. This means that we have
            // a new column in update so we must update LUT description and LUT
            iReturn = strcmp(pacTypeBuffer, WS_ALERTS_TYPE_TEXT);

            if (iReturn != 0) //Not Equal
            {
                SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                    WS_ALERTS_MGR_OBJECT_NAME": The type of the new column is not text! Type of the new column: %s",
                    pacTypeBuffer );
                bSuccess = FALSE;
                break;
            }

            // Generate the SQL command to update the LUT description table
            iReturn = snprintf(psIterator->pacBuffer,
                psIterator->tBufferSize,
                WS_ALERTS_INSERT_ROW_LUT_DESC,
                &acLUTNameBuffer[0],
                psDescToUpd->un8LabelId,
                &acLbTextBuffer[0],
                pacTypeBuffer);

            if (iReturn <= 0)
            {
                bSuccess = FALSE;
                break;
            }

            // update in database
            bSuccess = SQL_INTERFACE.bExecuteCommand(psIterator->hConnection, psIterator->pacBuffer);
            if (bSuccess == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WS_ALERTS_MGR_OBJECT_NAME": Unable to update database");
                break;
            }

            // Generate the SQL command to update the LUT
            iReturn = snprintf(psIterator->pacBuffer,
                psIterator->tBufferSize,
                WS_ALERTS_CREATE_LUT_COLUMN,
                &acLUTNameBuffer[0],
                &acLbTextBuffer[0],
                pacTypeBuffer);

            if (iReturn <= 0)
            {
                bSuccess = FALSE;
                break;
            }

            // update in database
            bSuccess = SQL_INTERFACE.bExecuteCommand(
                psIterator->hConnection, psIterator->pacBuffer);
            if (bSuccess == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WS_ALERTS_MGR_OBJECT_NAME": Unable to update database");
                break;
            }

        }
    }
    while (FALSE);

    psIterator->bResult = bSuccess;

    return bSuccess;
}

/*******************************************************************************
*
*   n16LUTDescComparator
*
*******************************************************************************/

static N16 n16LUTDescComparator (
    void *pvObj1,
    void *pvObj2
        )
{
    WS_ALERTS_LUT_DESC_ROW *psDescRow = (WS_ALERTS_LUT_DESC_ROW *)pvObj1;
    UN8 un8LabelId = (UN8)(size_t)pvObj2;

    return ( psDescRow->un8LabelId == un8LabelId ) ? 0 : -1;
}

/*******************************************************************************
*
*   bProcessSelectLutDescription
*
*******************************************************************************/
static BOOLEAN bProcessSelectLutDescription (
    SQL_QUERY_COLUMN_STRUCT *psColumns,
    N32 n32NumberOfColumns,
    OSAL_OBJECT_HDL psResult
        )
{
    WS_ALERTS_LUT_DESC_ROW *psDescRow = NULL;
    size_t tLength = 0;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    BOOLEAN bSuccess = FALSE;

    psDescRow = (WS_ALERTS_LUT_DESC_ROW*) SMSO_hCreate(
        WS_ALERTS_MGR_OBJECT_NAME":DescRow",
        sizeof(*psDescRow), SMS_INVALID_OBJECT, FALSE);
    if (psDescRow == NULL)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            WS_ALERTS_MGR_OBJECT_NAME": Failed to allocate memory");
        return FALSE;
    }

    do
    {
        tLength = strlen( (const char * )psColumns[WS_ALERTS_LUT_DESC_FIELD_TABLE_NAME].uData.sCString.pcData );
        psDescRow->hTableName = STRING.hCreate(
            (const char * )psColumns[WS_ALERTS_LUT_DESC_FIELD_TABLE_NAME].uData.sCString.pcData,
            tLength);

        psDescRow->un8LabelId = psColumns[WS_ALERTS_LUT_DESC_FIELD_LABID].uData.sUN32.un32Data & 255;

        tLength = strlen( (const char * )psColumns[WS_ALERTS_LUT_DESC_FIELD_NAME].uData.sCString.pcData );
        psDescRow->hFieldName = STRING.hCreate(
            (const char * )psColumns[WS_ALERTS_LUT_DESC_FIELD_NAME].uData.sCString.pcData,
            tLength);

        tLength = strlen( (const char * )psColumns[WS_ALERTS_LUT_DESC_FIELD_TYPE].uData.sCString.pcData );
        psDescRow->hFieldType = STRING.hCreate(
            (const char * )psColumns[WS_ALERTS_LUT_DESC_FIELD_TYPE].uData.sCString.pcData,
            tLength);

        eReturnCode = OSAL.eLinkedListAdd(psResult,
            OSAL_INVALID_LINKED_LIST_ENTRY_PTR, (void *)psDescRow);

        if( eReturnCode != OSAL_SUCCESS )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": cannot add entry to list (%s)",
                OSAL.pacGetReturnCodeName( eReturnCode ) );
            break;
        }

        bSuccess =TRUE;

    }
    while(FALSE);

    if ( bSuccess == FALSE )
    {
        if ( psDescRow->hTableName != STRING_INVALID_OBJECT )
        {
            STRING_vDestroy(psDescRow->hTableName);
            psDescRow->hTableName = STRING_INVALID_OBJECT;
        }

        if ( psDescRow->hFieldName != STRING_INVALID_OBJECT )
        {
            STRING_vDestroy(psDescRow->hFieldName);
            psDescRow->hFieldName = STRING_INVALID_OBJECT;
        }

        if ( psDescRow->hFieldType != STRING_INVALID_OBJECT )
        {
            STRING_vDestroy(psDescRow->hFieldType);
            psDescRow->hFieldType = STRING_INVALID_OBJECT;
        }
    }

    return bSuccess;
}


/*******************************************************************************
*
*       vReleaseDescriptionRow
*
*******************************************************************************/
static void vReleaseDescriptionRow(
    WS_ALERTS_LUT_DESC_ROW *psLUTDesc
        )
{
    if (psLUTDesc->hFieldName != STRING_INVALID_OBJECT)
    {
        STRING_vDestroy(psLUTDesc->hFieldName);
        psLUTDesc->hFieldName = STRING_INVALID_OBJECT;
    }
    if (psLUTDesc->hFieldType != STRING_INVALID_OBJECT)
    {
        STRING_vDestroy(psLUTDesc->hFieldType);
        psLUTDesc->hFieldType = STRING_INVALID_OBJECT;
    }
    if (psLUTDesc->hTableName != STRING_INVALID_OBJECT)
    {
        STRING_vDestroy(psLUTDesc->hTableName);
        psLUTDesc->hTableName = STRING_INVALID_OBJECT;
    }

    SMSO_vDestroy((SMS_OBJECT)psLUTDesc);

    return;
}

/*****************************************************************************
 *
 *   bShapeIteratorCallbackPoint
 *
 *****************************************************************************/
static BOOLEAN bShapeIteratorCallbackPoint(
    OSAL_FIXED_OBJECT hLatBegin,
    OSAL_FIXED_OBJECT hLonBegin,
    OSAL_FIXED_OBJECT hLatEnd,
    OSAL_FIXED_OBJECT hLonEnd,
    void *pvArg
        )
{
    WS_ALERT_MSG_LOCATION_CROSS_STRUCT *sIterativeLocationCross =
        (WS_ALERT_MSG_LOCATION_CROSS_STRUCT *) pvArg;
    BOOLEAN bResult;

    bResult = LOCATION_bRayCross (
        hLonBegin,
        hLatBegin,
        hLonEnd,
        hLatEnd,
        sIterativeLocationCross->hCurrentLocation);

    if( bResult == TRUE )
    {
        sIterativeLocationCross->un8NumOfRayCrossings++;
    }

    return TRUE; //We should iterate all the lines in order to find a crossing.
}

/*****************************************************************************
 *
 *   bShapeIteratorCallbackSquare
 *
 *****************************************************************************/
static BOOLEAN bShapeIteratorCallbackSquare(
    OSAL_FIXED_OBJECT hLatBegin,
    OSAL_FIXED_OBJECT hLonBegin,
    OSAL_FIXED_OBJECT hLatEnd,
    OSAL_FIXED_OBJECT hLonEnd,
    void *pvArg
        )
{
    WS_ALERT_MSG_LOCATION_CROSS_STRUCT *sIterativeLocationCross =
        (WS_ALERT_MSG_LOCATION_CROSS_STRUCT *) pvArg;
    BOOLEAN bReturn = TRUE;
    sIterativeLocationCross->bCross =
        LOCATION_bLineClip (
            hLonBegin,
            hLatBegin,
            hLonEnd,
            hLatEnd,
            sIterativeLocationCross->hCurrentLocation);

    if( sIterativeLocationCross->bCross == TRUE )
    {
        // If we have found the crossing, there is no need to
        // iterate other lines.
        bReturn = FALSE;
    }

    return bReturn;
}

/*******************************************************************************
*
*        bRefDBBank
*
*******************************************************************************/
static BOOLEAN bRefDBBank (
    WS_ALERTS_SERVICE_OBJECT hWsAlertsService,
    STRING_OBJECT *phInUseDB,
    STRING_OBJECT *phNextDB
        )
{
    BOOLEAN bLocked, bSuccess = FALSE;

    if ((phInUseDB == NULL) || (phNextDB == NULL))
    {
        return FALSE;
    }

    // Initialize inputs
    *phInUseDB = *phNextDB = STRING_INVALID_OBJECT;

    bLocked = DATASERVICE_IMPL_bLock((DATASERVICE_IMPL_HDL)hWsAlertsService);
    if (bLocked == TRUE)
    {
        WS_ALERTS_MGR_OBJECT_STRUCT *psObj =
            (WS_ALERTS_MGR_OBJECT_STRUCT *)hWsAlertsService;

        do
        {
            // Provide the caller with the path info
            *phInUseDB = STRING.hCreate(
                psObj->pacCurRefDatabaseFilePath,
                strlen(psObj->pacCurRefDatabaseFilePath));

            if (*phInUseDB == STRING_INVALID_OBJECT)
            {
                // Error! Stop here
                break;
            }

            *phNextDB = STRING.hCreate(
                psObj->pacNextRefDatabaseFilePath,
                strlen(psObj->pacNextRefDatabaseFilePath));

            if (*phNextDB == STRING_INVALID_OBJECT)
            {
                // Clear the other string now
                STRING_vDestroy(*phInUseDB);
                *phInUseDB = STRING_INVALID_OBJECT;

                break;
            }

            // All went well
            bSuccess = TRUE;
        } while (FALSE);

        DATASERVICE_IMPL_vUnlock((DATASERVICE_IMPL_HDL)hWsAlertsService);
    }

    return bSuccess;
}

/*****************************************************************************
*
*   vTooOldDBVer
*
*****************************************************************************/
static void vTooOldDBVer (
    WS_ALERTS_SERVICE_OBJECT hWSAlertsService
        )
{
    if (hWSAlertsService != WS_ALERTS_SERVICE_INVALID_OBJECT)
    {
        vSetError((WS_ALERTS_MGR_OBJECT_STRUCT*)hWSAlertsService,
            DATASERVICE_ERROR_CODE_DATABASE_TOO_OLD);
    }

    return;
}

/*******************************************************************************
 *
 *   bAddTypesToMsg
 *
 *******************************************************************************/
static BOOLEAN bAddTypesToMsg(
    WS_ALERTS_APP_OBJECT_STRUCT *psObj,
    OSAL_OBJECT_HDL hMsgTypeList,
    WS_ALERT_MSG_ID tMsgId
        )
{
    WS_ALERTS_DB_QUERY_OBJECTS_STRUCT sResult;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    BOOLEAN bResult = FALSE;
    int iReturn = 0;
    UN32 un32Items = 0;

    sResult.psAppObj = psObj;
    sResult.hObjectsList = hMsgTypeList;
    sResult.bResult = FALSE;

    do
    {
        iReturn = snprintf( psObj->acBuffer, sizeof( psObj->acBuffer ),
            WS_ALERTS_SELECT_MSG_TYPES_BY_MSG_ID,
            tMsgId );
        if (iReturn <= 0)
        {
            break;
        }

        printf(WS_ALERTS_MGR_OBJECT_NAME": SQL request %s will be performed.\n",
            psObj->acBuffer);

        bResult = SQL_INTERFACE.bQuery(
            psObj->hSQLPersistConnection, psObj->acBuffer,
            ( SQL_QUERY_RESULT_HANDLER )bDBProcessSelectMsgTypes,
            &sResult );

        if( bResult == FALSE )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": SQL request failed" );
            break;
        }

        eReturnCode = OSAL.eLinkedListItems( hMsgTypeList, &un32Items );

        if( eReturnCode != OSAL_SUCCESS )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": failed to get number"
                " items in list (%s)",
                OSAL.pacGetReturnCodeName( eReturnCode ) );
            bResult = FALSE;
            break;
        }
        else
        {
            if( un32Items == 0 )
            {
                SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                    WS_ALERTS_MGR_OBJECT_NAME": list of types is empty" );
                bResult = FALSE;
                break;
            }
        }

    } while (FALSE);

    return bResult;
}

/*******************************************************************************
 *
 *   bAddRestLocationsToMsg
 *
 *******************************************************************************/
static BOOLEAN bAddRestLocationsToMsg(
    WS_ALERTS_APP_OBJECT_STRUCT *psObj,
    OSAL_OBJECT_HDL hLocationsList,
    WS_ALERT_MSG_ID tMsgId
        )
{
    WS_ALERTS_DB_QUERY_OBJECTS_STRUCT sResult;
    BOOLEAN bResult = FALSE;
    int iReturn = 0;

    sResult.psAppObj = psObj;
    sResult.hObjectsList = hLocationsList;
    sResult.bResult = FALSE;

    iReturn = snprintf( psObj->acBuffer, sizeof( psObj->acBuffer ),
        WS_ALERTS_SELECT_LOCATIONS_BY_MSG_ID,
        tMsgId );
    if (iReturn > 0)
    {
        printf(WS_ALERTS_MGR_OBJECT_NAME": SQL request %s will be performed.\n",
            psObj->acBuffer);

        bResult = SQL_INTERFACE.bQuery(
            psObj->hSQLPersistConnection, psObj->acBuffer,
            ( SQL_QUERY_RESULT_HANDLER )bDBProcessSelectLocIdsFromMsgId,
            &sResult );

        if( bResult == FALSE )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": SQL request failed" );
        }
    }

    iReturn = snprintf( psObj->acBuffer, sizeof( psObj->acBuffer ),
        WS_ALERTS_SELECT_POLYGONS_BY_MSG_ID,
        tMsgId );
    if (iReturn > 0)
    {
        printf(WS_ALERTS_MGR_OBJECT_NAME": SQL request %s will be performed.\n",
            psObj->acBuffer);

        bResult = SQL_INTERFACE.bQuery(
            psObj->hSQLPersistConnection, psObj->acBuffer,
            ( SQL_QUERY_RESULT_HANDLER )bDBProcessSelectPolygons,
            &sResult );

        if( bResult == FALSE )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": SQL request failed" );
        }
    }

    return bResult;
}

/*******************************************************************************
 *
 *   bDBProcessSelectLocIds
 *
 *******************************************************************************/
static BOOLEAN bDBProcessSelectLocIdsFromMsgId(
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    WS_ALERTS_DB_QUERY_OBJECTS_STRUCT *psResult
        )
{
    if( psColumn[0].eType == SQL_COLUMN_TYPE_INTEGER )
    {
        WS_ALERTS_LOCATION_OBJECT hWsAlertsLocation = WS_ALERTS_LOCATION_INVALID_OBJECT;
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

        hWsAlertsLocation = hGetLocationById(
            psResult->psAppObj, psColumn[0].uData.sUN32.un32Data);

        if (hWsAlertsLocation == WS_ALERTS_LOCATION_INVALID_OBJECT)
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": Invalid location object");
            return FALSE;
        }

        eReturnCode = OSAL.eLinkedListAdd(
            psResult->hObjectsList, OSAL_INVALID_LINKED_LIST_ENTRY_PTR,
            (void *)hWsAlertsLocation);
        if ( eReturnCode != OSAL_SUCCESS )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": Cannot add location into the list");
        }
    }
    else
    {
        //it's not supposed to return smth different than integer
        SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
            WS_ALERTS_MGR_OBJECT_NAME": SQL request failed! Bad type in the DB.");
    }

    return TRUE;
}

/*******************************************************************************
 *
 *   bDBProcessSelectMsgTypes
 *
 *******************************************************************************/
static BOOLEAN bDBProcessSelectMsgTypes(
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    WS_ALERTS_DB_QUERY_OBJECTS_STRUCT *psResult
        )
{
    size_t tLength = 0;
    WS_ALERTS_MSG_TYPES_ENTRY_STRUCT *psMsgTypesEntry = NULL;
    char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_SUCCESS;

    // Create message types entry
    snprintf( &acName[0],
        OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL,
        WS_ALERTS_MGR_OBJECT_NAME"MsgTypeEntry:%u",
        psResult->psAppObj->tCurrentMsgId );

    psMsgTypesEntry = (WS_ALERTS_MSG_TYPES_ENTRY_STRUCT *)
        OSAL.pvLinkedListMemoryAllocate(
            &acName[0],
            sizeof(WS_ALERTS_MSG_TYPES_ENTRY_STRUCT),
            FALSE );

    if( psMsgTypesEntry == NULL )
    {
        SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
           WS_ALERTS_MGR_OBJECT_NAME": Can't allocate MsgTypeEntry object." );
        return FALSE;
    }

    do
    {
        if( psColumn[0].eType == SQL_COLUMN_TYPE_C_STRING )
        {
            psMsgTypesEntry->hMsgType = STRING.hCreate(
                ( const char * )psColumn[0].uData.sCString.pcData, tLength );

            if( psMsgTypesEntry->hMsgType == STRING_INVALID_OBJECT )
            {
                SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                    WS_ALERTS_MGR_OBJECT_NAME": Can't create string object" );
                eReturnCode = OSAL_ERROR;
                break;
            }
        }
        else
        {
            //it's not supposed to return smth different than integer
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": SQL request failed! Bad type in the DB." );
           eReturnCode = OSAL_ERROR;
           break;
        }

        if( psColumn[1].eType == SQL_COLUMN_TYPE_C_STRING )
        {
            psMsgTypesEntry->hText = STRING.hCreate(
                ( const char * )psColumn[1].uData.sCString.pcData, tLength );

            if( psMsgTypesEntry->hText == STRING_INVALID_OBJECT )
            {
                SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                    WS_ALERTS_MGR_OBJECT_NAME": Can't create string object" );
                eReturnCode = OSAL_ERROR;
                break;
            }
        }
        else
        {
            //it's not supposed to return smth different than integer
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": SQL request failed! Bad type in the DB." );
            eReturnCode = OSAL_ERROR;
            break;
        }
    } while (FALSE);

    if( eReturnCode == OSAL_SUCCESS )
    {
        eReturnCode = OSAL.eLinkedListAdd(
           psResult->hObjectsList, OSAL_INVALID_LINKED_LIST_ENTRY_PTR,
               (void *)psMsgTypesEntry );

        if ( eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": Can't add message type into the list");
            STRING.vDestroy( psMsgTypesEntry->hText );
            STRING.vDestroy( psMsgTypesEntry->hMsgType );
            OSAL.vLinkedListMemoryFree( psMsgTypesEntry );
        }
    }
    else
    {
        STRING.vDestroy( psMsgTypesEntry->hText );
        STRING.vDestroy( psMsgTypesEntry->hMsgType );
        OSAL.vLinkedListMemoryFree( psMsgTypesEntry );
    }
    return TRUE;
}

/*******************************************************************************
 *
 *   hGetLocationById
 *
 *******************************************************************************/
static WS_ALERTS_LOCATION_OBJECT hGetLocationById (
    WS_ALERTS_APP_OBJECT_STRUCT *psObj,
    LOC_ID tLocationId
        )
{
    WS_ALERTS_LOCATION_OBJECT hResult = WS_ALERTS_LOCATION_INVALID_OBJECT;

    do
    {
        OSAL_LINKED_LIST_ENTRY hEntry =
            OSAL_INVALID_LINKED_LIST_ENTRY;
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

        WS_ALERTS_LOCATION_vSetDummyId(
            psObj->hDummyLoc, tLocationId);

        //Do we have such location in our pool?
        eReturnCode = OSAL.eLinkedListSearch(
            psObj->hLocationPool,
            &hEntry,
            psObj->hDummyLoc
                );
        if (eReturnCode == OSAL_SUCCESS)
        {
            hResult =
                (WS_ALERTS_LOCATION_OBJECT)OSAL.pvLinkedListThis(hEntry);
            break;
        }
        else if (eReturnCode == OSAL_OBJECT_NOT_FOUND)
        {
            int iReturn = 0;
            BOOLEAN bSuccess = FALSE;
            WS_ALERTS_DB_RESULT_STRUCT sResult;
            WS_ALERTS_LOCATIONS_ROW_STRUCT *psAlertLocationRow = NULL;

            OSAL.bMemSet(&sResult, 0, sizeof(sResult));

            sResult.psAppObj = psObj;
            psAlertLocationRow = &sResult.uDbRow.sAlertLocationRow;

            iReturn = snprintf(psObj->acBuffer, sizeof(psObj->acBuffer),
                WS_ALERTS_SELECT_LOCATION_BY_ID,
                WS_ALERTS_GET_TYPE_FROM_LOCATION_ID(tLocationId),
                psObj->eCurrentLanguage,
                WS_ALERTS_GET_TABINDEX_FROM_LOCATION_ID(tLocationId),
                WS_ALERTS_GET_LCODE_FROM_LOCATION_ID(tLocationId) );
            if (iReturn <= 0)
            {
                SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                    WS_ALERTS_MGR_OBJECT_NAME": Failed to create SQL request" );
                break;
            }
            printf(WS_ALERTS_MGR_OBJECT_NAME": SQL request %s will be performed.\n",
                psObj->acBuffer);

            bSuccess = SQL_INTERFACE.bQuery(
                psObj->hSQLRefConnection, psObj->acBuffer,
                (SQL_QUERY_RESULT_HANDLER)bDBProcessSelectLocations,
                &sResult
                    );
            if(bSuccess == FALSE )
            {
                SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                    WS_ALERTS_MGR_OBJECT_NAME": SQL request failed" );
                vRemoveAlertLocationRow(psAlertLocationRow);
                break;
            }

            psAlertLocationRow->tLocId = tLocationId;

            hResult = WS_ALERTS_LOCATION_hCreate(
                (SMS_OBJECT)psObj,
                psAlertLocationRow );

            vRemoveAlertLocationRow(psAlertLocationRow);
            if (hResult == WS_ALERTS_LOCATION_INVALID_OBJECT)
            {
                SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                    WS_ALERTS_MGR_OBJECT_NAME": Cannot create location %d",
                    tLocationId );
                break;
            }

            eReturnCode = OSAL.eLinkedListAdd(
                psObj->hLocationPool,
                OSAL_INVALID_LINKED_LIST_ENTRY_PTR,
                (void *)hResult );

            if ( eReturnCode != OSAL_SUCCESS )
            {
                WS_ALERTS_LOCATION_vDestroy(hResult);
                hResult = WS_ALERTS_LOCATION_INVALID_OBJECT;

                SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                    WS_ALERTS_MGR_OBJECT_NAME": Failed to add location into pool (%s)",
                    OSAL.pacGetReturnCodeName(eReturnCode) );
            }
        }
        else
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": Locations pool search failed (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode) );
        }
    } while (FALSE);

    return hResult;
}

/*******************************************************************************
*
*       bChangeLUTVersion
*
*******************************************************************************/
static BOOLEAN bChangeLUTVersion(
    SQL_INTERFACE_OBJECT hConnection,
    char *pacBuffer,
    size_t tBufferSize,
    WS_ALERTS_LUT_UPDATE_DESC_STRUCT *psLutUpdateDesc
        )
{
    BOOLEAN bResult = FALSE;
    int iReturn = 0;
    char acLUTNameBuffer[WS_ALERTS_MAX_LUT_NAME_LENGTH + 1];
    const char *pacLutName;

    do
    {
        pacLutName = STRING.pacCStr(psLutUpdateDesc->hLutName);

        iReturn = snprintf(acLUTNameBuffer, sizeof(acLUTNameBuffer),
            WS_ALERTS_LUT_NAME_STRING,
            pacLutName);

        if (iReturn <= 0)
        {
            break;
        }

        iReturn = snprintf(pacBuffer, tBufferSize,
            WS_ALERTS_INSERT_OR_REPLACE,
            &acLUTNameBuffer[0], psLutUpdateDesc->un16UpdateVersion);

        if (iReturn <= 0)
        {
            break;
        }

        printf(WS_ALERTS_MGR_OBJECT_NAME": SQL query will be performed: %s\n", pacBuffer);

        // Perform the SQL query and process the result
        bResult = SQL_INTERFACE.bExecuteCommand(hConnection, pacBuffer);

        if (bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": Unable to fill up hLUTDescList" );
            break;
        }

    } while (FALSE);

    return bResult;
}

/*******************************************************************************
*
*       bCheckRowExist
*
*******************************************************************************/
static BOOLEAN bCheckRowExist(
    SQL_INTERFACE_OBJECT hConnection,
    char *pacBuffer,
    size_t tBufferSize,
    WS_ALERTS_LUT_UPDATE_DESC_STRUCT *psLutUpdateDesc,
    BOOLEAN *pbExist
        )
{
    WS_ALERTS_LABEL_DESC_STRUCT *psLabelDescKey1 = NULL;
    WS_ALERTS_LABEL_DESC_STRUCT *psLabelDescKey2 = NULL;
    OSAL_LINKED_LIST_ENTRY hLabelEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    BOOLEAN bSuccess = FALSE;
    const char *pacLutName = NULL;
    const char *pacLbText = NULL;
    const char *pacLbText2 = NULL;
    int iReturn = 0;

    do
    {
        if ( (hConnection == SQL_INTERFACE_INVALID_OBJECT) ||
             (psLutUpdateDesc == NULL) ||
             (pacBuffer == NULL) ||
             (pbExist == NULL) )
        {
            break;
        }

        hLabelEntry = OSAL.hLinkedListFirst(psLutUpdateDesc->hLabelsList, ( void ** )&psLabelDescKey1);
        if( psLutUpdateDesc->un8ClassId == WS_ALERTS_LUT_LOCATION_CLASS_ID )
        {
            hLabelEntry = OSAL.hLinkedListNext(hLabelEntry, ( void ** )&psLabelDescKey2);
        }

        if (psLabelDescKey1 == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": We should have at least one key to perform an SQL query!");
            break;
        }

        pacLutName = STRING.pacCStr(psLutUpdateDesc->hLutName);
        pacLbText = STRING.pacCStr(psLabelDescKey1->hLbText);

        if ( psLutUpdateDesc->un8ClassId == WS_ALERTS_LUT_LOCATION_CLASS_ID )
        {
            if (psLabelDescKey2 == NULL)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WS_ALERTS_MGR_OBJECT_NAME": We should have two keys to update Location");
                break;
            }

            pacLbText2 = STRING.pacCStr(psLabelDescKey2->hLbText);

            iReturn = snprintf(pacBuffer,
                tBufferSize,
                WS_ALERTS_SELECT_FROM_DOUBLE_KEY,
                pacLutName,
                pacLbText,
                psLabelDescKey1->uCurLbValue.un16Value,
                pacLbText2,
                psLabelDescKey2->uCurLbValue.un16Value);
        }
        else
        {
            iReturn = snprintf(pacBuffer,
                tBufferSize,
                WS_ALERTS_SELECT_FROM_KEY,
                pacLutName,
                pacLbText,
                psLabelDescKey1->uCurLbValue.un16Value);
        }

        if (iReturn <= 0)
        {
            break;
        }

        printf(WS_ALERTS_MGR_OBJECT_NAME": SQL query will be performed: %s\n", pacBuffer);

        *pbExist = FALSE;

        bSuccess = SQL_INTERFACE.bQuery(
            hConnection,
            pacBuffer,
            ( SQL_QUERY_RESULT_HANDLER )bDBProcessSelectRowFromKey,
            pbExist );
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": Failed to search the row");
            break;
        }
    }
    while (FALSE);

    printf(WS_ALERTS_MGR_OBJECT_NAME": Result of row search: %d.\n",
        *pbExist);

    return bSuccess;
}

/*******************************************************************************
 *
 *   bDBProcessSelectRowFromKey
 *
 *******************************************************************************/
static BOOLEAN bDBProcessSelectRowFromKey(
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    BOOLEAN *pbResult
        )
{
    *pbResult = TRUE;
    return FALSE;
}

/*******************************************************************************
 *
 *   bPrepareColumnForUpdate
 *
 *******************************************************************************/
static BOOLEAN bPrepareColumnForUpdate(
    SQL_COLUMN_INDEX tIndex,
    SQL_BIND_TYPE_ENUM *peType,
    size_t *ptDataSize,
    void **ppvData,
    OSAL_LINKED_LIST_ENTRY *phLabelEntry
        )
{
    WS_ALERTS_LABEL_DESC_STRUCT *psLabelDesc = NULL;
    BOOLEAN bSuccess = FALSE;

    if ( phLabelEntry == NULL )
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            WS_ALERTS_MGR_OBJECT_NAME
            ": bad phLabelEntry");
        return FALSE;
    }

    if ( *phLabelEntry == OSAL_INVALID_LINKED_LIST_ENTRY )
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            WS_ALERTS_MGR_OBJECT_NAME
            ": invalid entry phLabelEntry" );
        return FALSE;
    }

    psLabelDesc = (WS_ALERTS_LABEL_DESC_STRUCT*)OSAL.pvLinkedListThis(*phLabelEntry);

    do
    {
        if (psLabelDesc->bPresence == TRUE)
        {
            bSuccess = TRUE;
            switch (psLabelDesc->eDataType)
            {
                case WS_ALERTS_PARTITION_DTYPE_INT:
                {
                    *peType = SQL_BIND_TYPE_UN32;
                    *ppvData = (void*)(size_t)psLabelDesc->uCurLbValue.un16Value;
                }
                break;
                case WS_ALERTS_PARTITION_DTYPE_STRING:
                {
                    *peType = SQL_BIND_TYPE_C_STRING;
                    *ppvData = psLabelDesc->uCurLbValue.pcValue; 
                }
                break;
                case WS_ALERTS_PARTITION_DTYPE_BAUDOT_STRING:
                {
                    *peType = SQL_BIND_TYPE_STRING_OBJECT;
                    *ppvData = psLabelDesc->uCurLbValue.hValue;
                }
                break;
                case WS_ALERTS_PARTITION_DTYPE_ARRAY_OF_INT:
                {
                    *peType = SQL_BIND_TYPE_BLOB;
                    *ppvData = ( void * )&psLabelDesc->uCurLbValue.pn32Data[0];
                    *ptDataSize = psLabelDesc->tCurrentArrayNum * sizeof(N32);
                }
                break;
                default:
                {
                    bSuccess = FALSE;
                }
                break;
            }
        }

        *phLabelEntry = OSAL.hLinkedListNext(*phLabelEntry, (void **)&psLabelDesc);
    }
    while (( *phLabelEntry != OSAL_INVALID_LINKED_LIST_ENTRY ) && ( bSuccess == FALSE ));

    if ( bSuccess != TRUE )
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            WS_ALERTS_MGR_OBJECT_NAME": Failed to find the array");
    }

    return bSuccess;
}

// DSRL functions
/*******************************************************************************
*
*        bCreateList
*
*******************************************************************************/
static BOOLEAN bCreateList (
    WS_ALERTS_APP_OBJECT_STRUCT *psAppObj,
    DSRL_ARG_STRUCT *psDSRLArg
        )
{
    BOOLEAN bResult = FALSE;

    do
    {
        BOOLEAN bSuccess = FALSE;

        printf(WS_ALERTS_MGR_OBJECT_NAME": Creating DSRL\n");

        if ((psDSRLArg == NULL) || (psAppObj == NULL))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": Invalid parameter"
                    );
            break;
        }

        bSuccess = bDSRLDescInit(psDSRLArg, psAppObj);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": Failed to init DSRL desc for"
                " DSRL %p",
                psDSRLArg->hDSRL
                    );
            break;
        }

        // Set the finalizer
        bSuccess = DSRL_bSetFinalizeFunction(psDSRLArg->hDSRL,
            (DSRL_FINALIZE_FUNCTION)vDSRLFinalizeEntry, psAppObj
                );
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": failed to set finalizer for DSRL %p",
                psDSRLArg->hDSRL
                    );
            break;
        }

        bSuccess = bDSRLAddTargets(psDSRLArg);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": failed to add targets to DSRL %p",
                psDSRLArg->hDSRL
                    );
            break;
        }

        bSuccess = bDSRLUpdateContent(psDSRLArg->hDSRL, psAppObj);
        if (bSuccess == FALSE)
        {
            break;
        }

        vChangeDSRLState(psDSRLArg->hDSRL, DSRL_STATE_READY);

        bSuccess = bRemoveUnusedEntries(psAppObj);
        if (bSuccess != TRUE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": failed to remove unused entries"
                    );
            break;
        }

        // Looks good
        bResult = TRUE;
    } while (FALSE);

    return bResult;
}

/*******************************************************************************
*
*        hSearchEntry
*
*******************************************************************************/
static WS_ALERT_MSG_OBJECT hSearchEntry(
    WS_ALERTS_APP_OBJECT_STRUCT *psAppObj,
    WS_ALERT_MSG_ID tMsgID
        )
{
    WS_ALERT_MSG_OBJECT hResult = WS_ALERT_MSG_INVALID_OBJECT;
    BOOLEAN bSuccess;

    // update msg id in the dummy message to the requested msg id
    bSuccess = WS_ALERT_MSG_bSetMsgID(psAppObj->hDummyMsg, tMsgID);
    if (bSuccess == TRUE)
    {
        OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
        OSAL_RETURN_CODE_ENUM eReturnCode;

        eReturnCode = OSAL.eLinkedListSearch(psAppObj->hEntriesList, &hEntry,
            psAppObj->hDummyMsg);
        if (eReturnCode == OSAL_SUCCESS)
        {
            hResult = (WS_ALERT_MSG_OBJECT)OSAL.pvLinkedListThis(hEntry);
        }
        else if (eReturnCode != OSAL_OBJECT_NOT_FOUND)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": failed to search in list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode));
        }
    }

    return hResult;
}

/*******************************************************************************
*
*        bModifyList
*
*******************************************************************************/
static BOOLEAN bModifyList(
    WS_ALERTS_APP_OBJECT_STRUCT *psAppObj,
    DSRL_ARG_STRUCT *psDSRLArg
        )
{
    BOOLEAN bResult = FALSE;

    if ((psDSRLArg == NULL) ||
        (psDSRLArg->hDSRL == DSRL_INVALID_OBJECT))
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            WS_ALERTS_MGR_OBJECT_NAME": Invalid parameter"
                );
        return FALSE;
    }

    do
    {
        BOOLEAN bSuccess = FALSE;

        // This operation tells us explicitly if
        // we should do this or not
        if (psDSRLArg->uAction.sModify.bForceDSRLStateChange == TRUE)
        {
            vChangeDSRLState(
                psDSRLArg->hDSRL,
                DSRL_STATE_UPDATING
                    );
        }

        printf(WS_ALERTS_MGR_OBJECT_NAME" :DSRL %p modify: modify type is %d\n",
            psDSRLArg->hDSRL, psDSRLArg->uAction.sModify.eModifyType
                );

        // Modify the DSRL as the user specified
        switch (psDSRLArg->uAction.sModify.eModifyType)
        {
            case DSRL_MODIFY_OPERATION_ADD:
            {
                bSuccess = bDSRLAddTargets(psDSRLArg);
            }
            break;

            case DSRL_MODIFY_OPERATION_REPLACE:
            {
                bSuccess = bDSRLReplaceTargets(psDSRLArg);
            }
            break;

            case DSRL_MODIFY_OPERATION_REMOVE:
            {
                bSuccess = bDSRLRemoveTargets(psDSRLArg);
            }
            break;

            case DSRL_MODIFY_OPERATION_REMOVEALL:
            {
                bSuccess = bDSRLRemoveAllTargets(psDSRLArg);
            }
            break;

            case DSRL_MODIFY_OPERATION_INVALID:
            default:
                break;
        }

        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": DSRL modify targets operation failed"
                    );
            break;
        }

        bSuccess = bDSRLUpdateContent(psDSRLArg->hDSRL, psAppObj);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": DSRL modify entries operation failed"
                    );
            break;
        }

        vChangeDSRLState(psDSRLArg->hDSRL, DSRL_STATE_READY);

        bSuccess = bRemoveUnusedEntries(psAppObj);
        if (bSuccess != TRUE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": failed to remove unused entries"
                    );
            break;
        }

        bResult = TRUE;
    } while (FALSE);

    if (bResult == FALSE)
    {
        vChangeDSRLState(psDSRLArg->hDSRL, DSRL_STATE_ERROR);
    }

    return bResult;
}

/*******************************************************************************
*
*        bDeleteList
*
*******************************************************************************/
static BOOLEAN bDeleteList(
    WS_ALERTS_APP_OBJECT_STRUCT *psAppObj,
    DSRL_ARG_STRUCT *psDSRLArg
        )
{
    BOOLEAN bResult = FALSE;

    do
    {
        WS_ALERTS_DSRL_DESC_STRUCT *psDSRLDesc = NULL;
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
        BOOLEAN bSuccess = FALSE;

        if ((psDSRLArg == NULL) ||
            (psDSRLArg->hDSRL == DSRL_INVALID_OBJECT))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": Invalid parameter"
                    );
            break;
        }

        printf(WS_ALERTS_MGR_OBJECT_NAME": Removing DSRL %p\n",
            psDSRLArg->hDSRL
                );

        // Remove / free all entries in the DSRL
        DSRL_vRemoveAllEntries(psDSRLArg->hDSRL);

        psDSRLDesc = (WS_ALERTS_DSRL_DESC_STRUCT*)DSRL_pvServiceData(
            psDSRLArg->hDSRL
                );
        if (psDSRLDesc == NULL)
        {
            break;
        }

        eReturnCode = OSAL.eLinkedListRemove(psDSRLDesc->hDSRLListEntry);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": cannot remove DSRL from pool (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
        }

        vDSRLRelease(psDSRLArg->hDSRL);
        psDSRLArg->hDSRL = DSRL_INVALID_OBJECT;

        bSuccess = bRemoveUnusedEntries(psAppObj);
        if (bSuccess != TRUE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": failed to remove unused entries"
                    );
            break;
        }

        bResult = TRUE;
    } while (FALSE);

    return bResult;
}

/*******************************************************************************
*
*        bRefreshDSRLs
*
*******************************************************************************/
static BOOLEAN bRefreshDSRLs(
    WS_ALERTS_APP_OBJECT_STRUCT *psAppObj
        )
{
    BOOLEAN bResult = FALSE;

    if (psAppObj != NULL)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

        printf(WS_ALERTS_MGR_OBJECT_NAME": Refreshing DSRLs \n");

        eReturnCode = OSAL.eLinkedListIterate(psAppObj->hDSRLList,
            (OSAL_LL_ITERATOR_HANDLER)bDSRLRefresh, psAppObj
                );
        if ((eReturnCode != OSAL_SUCCESS) &&
            (eReturnCode != OSAL_NO_OBJECTS))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": failed to refresh DSRLs list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
        }
        else
        {
            bResult = bRemoveUnusedEntries(psAppObj);
            if (bResult == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WS_ALERTS_MGR_OBJECT_NAME": failed to remove unused entries"
                        );
            }
        }
    }

    return bResult;
}

/*******************************************************************************
*
*        bRefreshList
*
*******************************************************************************/
static BOOLEAN bRefreshList(
    WS_ALERTS_APP_OBJECT_STRUCT *psAppObj,
    DSRL_ARG_STRUCT *psDSRLArg
        )
{
    BOOLEAN bResult = FALSE;

    if ((psDSRLArg == NULL) ||
        (psDSRLArg->hDSRL == DSRL_INVALID_OBJECT))
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            WS_ALERTS_MGR_OBJECT_NAME": Invalid parameter"
                );
        return FALSE;
    }

    do
    {
        BOOLEAN bSuccess = FALSE;

        bSuccess = bDSRLUpdateContent(psDSRLArg->hDSRL, psAppObj);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": failed to update DSRL content"
                    );
            break;
        }

        vChangeDSRLState(psDSRLArg->hDSRL, DSRL_STATE_READY);

        bSuccess = bRemoveUnusedEntries(psAppObj);
        if (bSuccess != TRUE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": failed to remove unused entries"
                    );
            break;
        }

        bResult = TRUE;
    } while (FALSE);

    if (bResult == FALSE)
    {
        vChangeDSRLState(psDSRLArg->hDSRL, DSRL_STATE_ERROR);
    }

    return bResult;
}

/*******************************************************************************
*
*        bDSRLRefresh
*
*******************************************************************************/
static BOOLEAN bDSRLRefresh(
    DSRL_OBJECT hDSRL,
    WS_ALERTS_APP_OBJECT_STRUCT *psAppObj
        )
{
    BOOLEAN bContinue = FALSE;

    if ((hDSRL != DSRL_INVALID_OBJECT) && (psAppObj != NULL))
    {
        bContinue = bDSRLUpdateContent(hDSRL, psAppObj);
        if (bContinue == FALSE)
        {
            vChangeDSRLState(hDSRL, DSRL_STATE_ERROR);
        }
        else
        {
            vChangeDSRLState(hDSRL, DSRL_STATE_READY);
        }
    }

    return bContinue;
}

/*****************************************************************************
*
*   n16FindTargetEntry
*
*****************************************************************************/
static N16 n16FindTargetEntry(
    DSRL_TARGET_OBJECT hTarget1,
    DSRL_TARGET_OBJECT hTarget2
        )
{
    N16 n16Result = N16_MIN;

    if ((hTarget1 != DSRL_TARGET_INVALID_OBJECT) &&
        (hTarget2 != DSRL_TARGET_INVALID_OBJECT))
    {
        BOOLEAN bResult = FALSE;

        bResult = DSRL_TARGET_bCompare(hTarget1, hTarget2);

        n16Result = (bResult == TRUE) ? 0 : 1;
    }

    return n16Result;
}

/*****************************************************************************
*
*   bDSRLRemoveAllTargets
*
*****************************************************************************/
static BOOLEAN bDSRLRemoveAllTargets(
    DSRL_ARG_STRUCT *psDSRLArg
        )
{
    BOOLEAN bResult = TRUE;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    WS_ALERTS_DSRL_DESC_STRUCT *psDSRLDesc = NULL;

    psDSRLDesc = (WS_ALERTS_DSRL_DESC_STRUCT*)DSRL_pvServiceData(psDSRLArg->hDSRL);
    if (psDSRLDesc == NULL)
    {
        bResult = FALSE;
    }
    else
    {
        eReturnCode = OSAL.eLinkedListRemoveAll(psDSRLDesc->hTargetList,
            (OSAL_LL_RELEASE_HANDLER)DSRL_TARGET_vDestroyByType
                );
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": failed to remove DSRL targets"
                " (%s)", OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            bResult = FALSE;
        }
    }

    return bResult;
}

/*****************************************************************************
*
*   vDSRLRemoveTarget
*
*****************************************************************************/
static void vDSRLRemoveTarget(
    DSRL_OBJECT hDSRL,
    DSRL_TARGET_OBJECT hTargetObj
        )
{
    do
    {
        OSAL_LINKED_LIST_ENTRY hTargetEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
        WS_ALERTS_DSRL_DESC_STRUCT *psDSRLDesc = NULL;
        DSRL_TARGET_OBJECT hTargetToRemove = DSRL_TARGET_INVALID_OBJECT;

        if (hTargetObj == DSRL_TARGET_INVALID_OBJECT)
        {
            break;
        }

        psDSRLDesc = (WS_ALERTS_DSRL_DESC_STRUCT*)DSRL_pvServiceData(hDSRL);
        if (psDSRLDesc == NULL)
        {
            break;
        }

        eReturnCode = OSAL.eLinkedListLinearSearch(
            psDSRLDesc->hTargetList,
            &hTargetEntry,
            (OSAL_LL_COMPARE_HANDLER)n16FindTargetEntry,
            hTargetObj
                );

        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": failed to find target entry (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }

        hTargetToRemove =
            (DSRL_TARGET_OBJECT)OSAL.pvLinkedListThis(hTargetEntry);
        if (hTargetToRemove == DSRL_TARGET_INVALID_OBJECT)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": failed to get target handle"
                    );
            break;
        }

        eReturnCode = OSAL.eLinkedListRemove(hTargetEntry);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": failed to remove target entry"
                " from list of DSRL targets (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
        }

        DSRL_TARGET_vDestroyByType(hTargetToRemove);
    } while (FALSE);

    return;
}

/*****************************************************************************
*
*   bDSRLRemoveTargets
*
*****************************************************************************/
static BOOLEAN bDSRLRemoveTargets(
    DSRL_ARG_STRUCT *psDSRLArg
        )
{
    size_t tDSRLTargetIndex;

    for (tDSRLTargetIndex = 0;
         tDSRLTargetIndex < psDSRLArg->tNumTargets;
         tDSRLTargetIndex++)
    {
        vDSRLRemoveTarget(
            psDSRLArg->hDSRL,
            psDSRLArg->ahTargetList[tDSRLTargetIndex]
                );
    }

    return TRUE;
}

/*****************************************************************************
*
*   bDSRLReplaceTargets
*
*****************************************************************************/
static BOOLEAN bDSRLReplaceTargets(
    DSRL_ARG_STRUCT *psDSRLArg
        )
{
    BOOLEAN bResult = FALSE;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    WS_ALERTS_DSRL_DESC_STRUCT *psDSRLDesc = NULL;

    psDSRLDesc = (WS_ALERTS_DSRL_DESC_STRUCT*)DSRL_pvServiceData(psDSRLArg->hDSRL);
    if (psDSRLDesc == NULL)
    {
        return FALSE;
    }

    eReturnCode = OSAL.eLinkedListRemoveAll(psDSRLDesc->hTargetList,
        (OSAL_LL_RELEASE_HANDLER)DSRL_TARGET_vDestroyByType
            );
    if (eReturnCode !=  OSAL_SUCCESS)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            WS_ALERTS_MGR_OBJECT_NAME": failed to release target desc"
            " items in DSRL desc (%s)",
            OSAL.pacGetReturnCodeName(eReturnCode)
                );
    }
    else
    {
        //Add new targets
        bResult = bDSRLAddTargets(psDSRLArg);
    }

    return bResult;
}

/*****************************************************************************
*
*   vDestroyUnusedEntry
*
*****************************************************************************/
static void vDestroyUnusedEntry(
    DSRL_ENTRY_OBJECT hDSRLEntry
        )
{
    do
    {
        WS_ALERTS_DSRL_ENTRY_DESC_STRUCT *psEntryDesc = NULL;
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
        UN32 un32Items = 0;

        if (hDSRLEntry == DSRL_ENTRY_INVALID_OBJECT)
        {
            // list corruption
            break;
        }

        psEntryDesc = (WS_ALERTS_DSRL_ENTRY_DESC_STRUCT*)
            DSRL_ENTRY_pvServiceData(hDSRLEntry);
        if (psEntryDesc == NULL)
        {
            break;
        }

        // in some cases DSRL list could even be not created yet
        // in this case we suggest that no assigned DSRLs exist
        if (psEntryDesc->hDSRLList != OSAL_INVALID_OBJECT_HDL)
        {
            eReturnCode = OSAL.eLinkedListItems(psEntryDesc->hDSRLList,
                &un32Items
                    );
            if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WS_ALERTS_MGR_OBJECT_NAME": failed to get number of DSRL"
                    " desc items in list (%s)",
                    OSAL.pacGetReturnCodeName(eReturnCode)
                        );
                break;
            }
        }
        else
        {
            un32Items = 0;
        }

        if (un32Items != 0)
        {
            break;
        }

        eReturnCode = OSAL.eLinkedListRemove(psEntryDesc->hPoolEntry);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": failed to remove DSRL"
                " entry from pool (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
        }

        printf(WS_ALERTS_MGR_OBJECT_NAME": Releasing entry %p\n",
            hDSRLEntry
                );

        vEntryRelease(hDSRLEntry);

    } while (FALSE);

    return;
}

/*****************************************************************************
*
*   bRemoveUnusedEntries
*
*****************************************************************************/
static BOOLEAN bRemoveUnusedEntries(
    WS_ALERTS_APP_OBJECT_STRUCT *psAppObj
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

    eReturnCode = OSAL.eLinkedListRemoveAll(psAppObj->hRemovedEntriesList,
        (OSAL_LL_RELEASE_HANDLER)vDestroyUnusedEntry
            );
    if ((eReturnCode != OSAL_SUCCESS) && (eReturnCode != OSAL_NO_OBJECTS))
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            WS_ALERTS_MGR_OBJECT_NAME": failed to remove unused DSRL"
            " entries (%s)",
            OSAL.pacGetReturnCodeName(eReturnCode)
                );
        return FALSE;
    }

    return TRUE;
}

/*****************************************************************************
*
*   vRemoveExpiredDSRLEntries
*
*****************************************************************************/
static void vRemoveExpiredDSRLEntries(
    WS_ALERTS_MGR_OBJECT_STRUCT *psObj
        )
{
    WS_ALERTS_APP_OBJECT_STRUCT *psAppObj;
    WS_ALERTS_EXPIRED_MSGS_ITERATOR_STRUCT sIterator;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

    psAppObj = psGetAppFacingObject( ( WS_ALERTS_SERVICE_OBJECT )psObj );

    if( psAppObj == NULL )
    {
        return;
    }

    sIterator.un32CurUTCsec = 0;
    sIterator.un32NextExpirUTCsec = WS_ALERTS_CLEAN_UP_TIMER_UTC_SEC_MAX;

    eReturnCode = OSAL.eTimeGet(&sIterator.un32CurUTCsec);
    if (eReturnCode == OSAL_SUCCESS)
    {
        printf(WS_ALERTS_MGR_OBJECT_NAME
            " Current time is %d seconds UTC.\n", sIterator.un32CurUTCsec);

        //Iterate the DSRL Entry Pool to remove expired entries
        eReturnCode = OSAL.eLinkedListIterate(
            psAppObj->hEntriesList,
            (OSAL_LL_ITERATOR_HANDLER)bDSRLRemoveExpiredEntry,
            &sIterator);
        if ((eReturnCode != OSAL_SUCCESS) &&
            (eReturnCode != OSAL_NO_OBJECTS))
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": Failed to iterate DSRL Entries pool");
        }

        vChangeAllDSRLsState( psAppObj, DSRL_STATE_READY);
    }

    vReleaseAppFacingObject( psAppObj );

    // start next clean up timer
    if ( sIterator.un32NextExpirUTCsec != WS_ALERTS_CLEAN_UP_TIMER_UTC_SEC_MAX )
    {
        vStartCleanUpTimer(psObj, sIterator.un32NextExpirUTCsec);
    }

    return;
}

/*****************************************************************************
 *
 *   bDSRLRemoveExpiredEntry
 *
 *****************************************************************************/
static BOOLEAN bDSRLRemoveExpiredEntry(
    DSRL_ENTRY_OBJECT hDSRLEntry,
    WS_ALERTS_EXPIRED_MSGS_ITERATOR_STRUCT *psIterator
        )
{
    WS_ALERT_MSG_OBJECT hMsg;
    TIME_T tEventEndTime;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

    if ( psIterator == NULL )
    {
        return FALSE;
    }

    do
    {
        hMsg = DSRL_ENTRY.hWSAlertMsg( hDSRLEntry );

        if( hMsg == WS_ALERT_MSG_INVALID_OBJECT )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": Invalid message in DSRL entry");
            break;
        }

        // Get message end time
        tEventEndTime =
            WS_ALERT_MSG.tEndTime(hMsg);

        // Is the message expirable
        if ( tEventEndTime == 0 )
        {
            // this message never expires, skip it
            break;
        }

        // Is the message still actual?
        if (tEventEndTime <= (TIME_T)psIterator->un32CurUTCsec)
        {
            WS_ALERTS_DSRL_ENTRY_DESC_STRUCT *psEntryDesc;

            psEntryDesc = (WS_ALERTS_DSRL_ENTRY_DESC_STRUCT*)
                DSRL_ENTRY_pvServiceData(hDSRLEntry);

            if( psEntryDesc == NULL )
            {
                SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                    WS_ALERTS_MGR_OBJECT_NAME": Invalid DSRL Entry");
                break;
            }

            // remove entry from containing it DSRLs
            eReturnCode = OSAL.eLinkedListIterate(
                psEntryDesc->hDSRLList,
                (OSAL_LL_ITERATOR_HANDLER)bDSRLRemoveEntry,
                hDSRLEntry);
            if ( (eReturnCode != OSAL_SUCCESS) &&
                 (eReturnCode != OSAL_NO_OBJECTS) )
            {
                SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                    WS_ALERTS_MGR_OBJECT_NAME": Failed to iterate Entry DSRLs list");
                break;
            }
        }
        else
        {
            // calculate next expiration time
            if ( tEventEndTime < (TIME_T)psIterator->un32NextExpirUTCsec )
            {
                psIterator->un32NextExpirUTCsec = (UN32)tEventEndTime;
            }
        }

    } while(FALSE);

    return TRUE;
}

/*****************************************************************************
*
*   bTargetEntriesGenerator
*
*****************************************************************************/
static BOOLEAN bTargetEntriesGenerator(
    DSRL_TARGET_OBJECT hTarget,
    WS_ALERTS_DSRL_ENTRIES_LIST_ITERATOR_STRUCT *psIterator
        )
{
    BOOLEAN bContinue = TRUE;

    if ((hTarget == DSRL_TARGET_INVALID_OBJECT) || 
        (psIterator == NULL))
    {
        return FALSE;
    }

    bContinue = bGetAlertMsgsList(hTarget, 
        psIterator->psAppObj, psIterator->hDSRL);

    printf(WS_ALERTS_MGR_OBJECT_NAME": bGetEntriesList called with result %u\n",
        bContinue
            );

    return bContinue;
}

/*****************************************************************************
*
*   n16CompareHandles
*
*****************************************************************************/
static N16 n16CompareHandles(
    void *pvArg1,
    void *pvArg2
        )
{
    return COMPARE(pvArg1, pvArg2);
}

/*****************************************************************************
*
*   bDSRLDescInit
*
*****************************************************************************/
static BOOLEAN bDSRLDescInit(
    DSRL_ARG_STRUCT *psDSRLArg,
    WS_ALERTS_APP_OBJECT_STRUCT *psAppObj
        )
{
    WS_ALERTS_DSRL_DESC_STRUCT *psDSRLDesc = (WS_ALERTS_DSRL_DESC_STRUCT*)NULL;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    BOOLEAN bResult = FALSE;

    do
    {
        BOOLEAN bSuccess = FALSE;

        if (psDSRLArg->hDSRL == DSRL_INVALID_OBJECT)
        {
            break;
        }

        bSuccess = DSRL_bSetFinalizeFunction(psDSRLArg->hDSRL,
            (DSRL_FINALIZE_FUNCTION)vDSRLFinalizeEntry, psAppObj
                );
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": Failed to set finalizer for DSRL %p",
                psDSRLArg->hDSRL
                    );
            break;
        }

        // Set default sorting by priority
        bSuccess = DSRL_bSetDefaultSortFunction(psDSRLArg->hDSRL,
            n16DSRLSortByPriority, psAppObj
                );

        // If bSuccess is FALSE it just means that DSRL already has
        // sorting function assigned by the application.
        // It is OK because we are just making sure 'default' sort is
        // applied
        if (bSuccess == FALSE)
        {
            printf( WS_ALERTS_MGR_OBJECT_NAME": Default sorting is not applied to DSRL %p\n",
                psDSRLArg->hDSRL
                    );
        }

        // get the DSRL desc pointer from DSRL
        psDSRLDesc = (WS_ALERTS_DSRL_DESC_STRUCT*)DSRL_pvServiceData(
            psDSRLArg->hDSRL
                );
        if (psDSRLDesc == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": cannot get DSRL desc from DSRL"
                    );
            break;
        }

        eReturnCode = OSAL.eLinkedListCreate(
            &psDSRLDesc->hTargetList,
            WS_ALERTS_MGR_OBJECT_NAME":DSRLDesc:TgtList",
            NULL,
            OSAL_LL_OPTION_NONE
                );
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": cannot create targets list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }

        eReturnCode = OSAL.eLinkedListCreate(
            &psDSRLDesc->hEntriesList,
            WS_ALERTS_MGR_OBJECT_NAME":DSRLDesc:EntriesList",
            n16CompareHandles,
            OSAL_LL_OPTION_UNIQUE
                );
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": cannot create entry list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }

        // Add this to our list of tracked DSRL lists of targets for this service
        eReturnCode =
            OSAL.eLinkedListAdd(psAppObj->hDSRLList,
            &(psDSRLDesc->hDSRLListEntry),
            psDSRLArg->hDSRL);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": cannot add DSRL to pool (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }

        bResult = TRUE;
    } while (FALSE);

    if ((bResult == FALSE) && (psDSRLDesc != NULL))
    {
        if (psDSRLDesc->hDSRLListEntry != OSAL_INVALID_LINKED_LIST_ENTRY)
        {
            eReturnCode = OSAL.eLinkedListRemove(psDSRLDesc->hDSRLListEntry);
            if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WS_ALERTS_MGR_OBJECT_NAME": cannot remove DSRL from pool (%s)",
                    OSAL.pacGetReturnCodeName(eReturnCode)
                        );
            }
            psDSRLDesc->hDSRLListEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
        }
        vDSRLRelease(psDSRLArg->hDSRL);
    }

    return bResult;
}

/*****************************************************************************
*
*   bEntryRemoveDSRLLink
*
*****************************************************************************/
BOOLEAN bEntryRemoveDSRLLink(
    WS_ALERTS_APP_OBJECT_STRUCT *psAppObj,
    DSRL_ENTRY_OBJECT hDSRLEntry,
    DSRL_OBJECT hDSRL
        )
{
    BOOLEAN bResult = FALSE;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    WS_ALERTS_DSRL_ENTRY_DESC_STRUCT *psEntryDesc = NULL;

    psEntryDesc = (WS_ALERTS_DSRL_ENTRY_DESC_STRUCT*)
        DSRL_ENTRY_pvServiceData(hDSRLEntry);

    if ((psEntryDesc != NULL) &&
        (psEntryDesc->hDSRLList != OSAL_INVALID_OBJECT_HDL))
    {
        eReturnCode = OSAL.eLinkedListSearch(psEntryDesc->hDSRLList,
            &hEntry, hDSRL
                );
        if (eReturnCode == OSAL_SUCCESS)
        {
            eReturnCode = OSAL.eLinkedListRemove(hEntry);
            if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WS_ALERTS_MGR_OBJECT_NAME": failed to remove DSRL desc"
                    " from list (%s)",
                    OSAL.pacGetReturnCodeName(eReturnCode)
                        );
            }

            // adding entry to list of removed entries
            eReturnCode = OSAL.eLinkedListAdd(
                psAppObj->hRemovedEntriesList, NULL, hDSRLEntry
                    );
            if ((eReturnCode != OSAL_SUCCESS) &&
                (eReturnCode != OSAL_ERROR_LIST_ITEM_NOT_UNIQUE))
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WS_ALERTS_MGR_OBJECT_NAME": failed to add DSRL entry to list"
                    " of removed entries (%s)",
                    OSAL.pacGetReturnCodeName(eReturnCode)
                        );
            }
        }
        else
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": failed to find DSRL desc"
                " in list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
        }

        bResult = TRUE;
    }

    return bResult;
}

/*****************************************************************************
*
*   bDSRLRemoveEntryLink
*
*****************************************************************************/
static BOOLEAN bDSRLRemoveEntryLink(
    DSRL_OBJECT hDSRL,
    DSRL_ENTRY_OBJECT hDSRLEntry
        )
{
    BOOLEAN bResult = FALSE;

    if ((hDSRL != DSRL_INVALID_OBJECT) &&
        (hDSRLEntry != DSRL_ENTRY_INVALID_OBJECT))
    {
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
        OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
        WS_ALERTS_DSRL_DESC_STRUCT *psDSRLDesc = NULL;

        psDSRLDesc = (WS_ALERTS_DSRL_DESC_STRUCT*)DSRL_pvServiceData(hDSRL);
        if (psDSRLDesc != NULL)
        {
            eReturnCode = OSAL.eLinkedListSearch(psDSRLDesc->hEntriesList,
                &hEntry, hDSRLEntry
                    );
            if (eReturnCode == OSAL_SUCCESS)
            {
                eReturnCode = OSAL.eLinkedListRemove(hEntry);
                if (eReturnCode != OSAL_SUCCESS)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        WS_ALERTS_MGR_OBJECT_NAME": failed to remove entry"
                        " from list (%s)",
                        OSAL.pacGetReturnCodeName(eReturnCode)
                            );
                }
            }
            else
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WS_ALERTS_MGR_OBJECT_NAME": failed to find entry"
                    " in list (%s)",
                    OSAL.pacGetReturnCodeName(eReturnCode)
                        );
            }
        }

        bResult = TRUE;
    }

    return bResult;
}

/*****************************************************************************
*
*   vDSRLFinalizeEntry
*
*****************************************************************************/
static void vDSRLFinalizeEntry(
    DSRL_OBJECT hDSRL,
    DSRL_ENTRY_OBJECT hDSRLEntry,
    WS_ALERTS_APP_OBJECT_STRUCT *psAppObj
        )
{
    if ((hDSRL != DSRL_INVALID_OBJECT) &&
        (hDSRLEntry != DSRL_ENTRY_INVALID_OBJECT))
    {
        BOOLEAN bSuccess = FALSE;

        bSuccess = bEntryRemoveDSRLLink(psAppObj, hDSRLEntry, hDSRL);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": failed to remove entry"
                " to DSRL link"
                    );
        }

        bSuccess = bDSRLRemoveEntryLink(hDSRL, hDSRLEntry);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": failed to remove DSRL"
                " to entry link"
                    );
        }

    }
    else
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            WS_ALERTS_MGR_OBJECT_NAME": failed to finalize entry: hDSRL %p,"
            " hDSRLEntry %p, callback arg %p", hDSRL, hDSRLEntry, psAppObj
                );
    }

    return;
}

/*****************************************************************************
*
*   bDSRLAddTarget
*
*****************************************************************************/
static BOOLEAN bDSRLAddTarget(
    DSRL_OBJECT hDSRL,
    DSRL_TARGET_OBJECT hTargetObj
        )
{
    BOOLEAN bResult = FALSE;

    do
    {
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
        WS_ALERTS_DSRL_DESC_STRUCT *psDSRLDesc = NULL;

        if (hTargetObj == DSRL_TARGET_INVALID_OBJECT)
        {
            break;
        }

        psDSRLDesc = (WS_ALERTS_DSRL_DESC_STRUCT*)DSRL_pvServiceData(hDSRL);

        if (psDSRLDesc == NULL)
        {
            break;
        }
        // Add this target to our DSRL description
        eReturnCode = OSAL.eLinkedListAdd(
            psDSRLDesc->hTargetList,
            OSAL_INVALID_LINKED_LIST_ENTRY_PTR,
            hTargetObj
                );
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": failed to add target to list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }

        bResult = TRUE;
    } while (FALSE);

    return bResult;
}

/*****************************************************************************
*
*   bDSRLAddTargets
*
*****************************************************************************/
static BOOLEAN bDSRLAddTargets(
    DSRL_ARG_STRUCT *psDSRLArg
        )
{
    size_t tDSRLTargetIndex;
    BOOLEAN bResult = TRUE;

    for (tDSRLTargetIndex = 0;
         tDSRLTargetIndex < psDSRLArg->tNumTargets;
         tDSRLTargetIndex++)
    {
        bResult = bDSRLAddTarget(psDSRLArg->hDSRL,
            psDSRLArg->ahTargetList[tDSRLTargetIndex]
                );
        if (bResult == FALSE)
        {
            break;
        }
        else
        {
            // if target has been added successfully
            // need to clean it in array to avoid target destroy
            // in general DATASERVICE event processor
            psDSRLArg->ahTargetList[tDSRLTargetIndex] =
                DSRL_TARGET_INVALID_OBJECT;
        }
    }

    return bResult;
}

/*****************************************************************************
*
*   vChangeAllDSRLsState
*
*****************************************************************************/
static void vChangeAllDSRLsState(
    WS_ALERTS_APP_OBJECT_STRUCT *psAppObj,
    DSRL_STATE_ENUM eState
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

    eReturnCode = OSAL.eLinkedListIterate(psAppObj->hDSRLList,
        (OSAL_LL_ITERATOR_HANDLER)bDSRLChangeState , (void *)(size_t)eState
            );
    if ((eReturnCode != OSAL_SUCCESS) && (eReturnCode != OSAL_NO_OBJECTS))
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            WS_ALERTS_MGR_OBJECT_NAME": cannot iterate linked list (%s)",
            OSAL.pacGetReturnCodeName(eReturnCode)
                );
    }

    return;
}

/*****************************************************************************
*
*   bDSRLChangeState
*
* Just a wrapper to use it in iterator
*
*****************************************************************************/
static BOOLEAN bDSRLChangeState (
    DSRL_OBJECT hDSRL,
    DSRL_STATE_ENUM eState
        )
{
    vChangeDSRLState(hDSRL, eState);

    return TRUE;
}

/*****************************************************************************
*
*   vChangeDSRLState
*
*****************************************************************************/
static void vChangeDSRLState(
    DSRL_OBJECT hDSRL,
    DSRL_STATE_ENUM eState
        )
{
    DSRL_STATE_ENUM eCurState = DSRL_STATE_ERROR;

    if (hDSRL != DSRL_INVALID_OBJECT)
    {
        eCurState = DSRL.eState(hDSRL);
        if (eCurState != DSRL_STATE_ERROR)
        {
            DSRL_vSetState(hDSRL, eState);
        }
    }

    return;
}

/*****************************************************************************
*
*   bEntryAddToDSRL
*
*****************************************************************************/
static BOOLEAN bEntryAddToDSRL(
    DSRL_ENTRY_OBJECT hDSRLEntry,
    DSRL_OBJECT hDSRL,
    UN32 un32TransactionID
        )
{
    BOOLEAN bContinue = FALSE, bAdded = FALSE;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

    do
    {
        DSRL_ADD_REPLACE_RESULT_ENUM eDSRLResult = DSRL_ADD_REPLACE_ERROR;
        WS_ALERTS_DSRL_ENTRY_DESC_STRUCT *psEntryDesc = NULL;
        WS_ALERTS_DSRL_DESC_STRUCT *psDSRLDesc = NULL;
        OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

        if ((hDSRL == DSRL_INVALID_OBJECT) ||
            (hDSRLEntry == DSRL_ENTRY_INVALID_OBJECT))
        {
            break;
        }

        psEntryDesc = 
            (WS_ALERTS_DSRL_ENTRY_DESC_STRUCT *)DSRL_ENTRY_pvServiceData(
            hDSRLEntry);
        if (psEntryDesc == NULL)
        {
            break;
        }

        // updating the transaction ID to make the entry actual
        psEntryDesc->un32TransactionID = un32TransactionID;

        psDSRLDesc = (WS_ALERTS_DSRL_DESC_STRUCT *)DSRL_pvServiceData(hDSRL);
        if (psDSRLDesc == NULL)
        {
            break;
        }

        // adding entry to list of entries in DSRL
        eReturnCode = OSAL.eLinkedListAdd(psDSRLDesc->hEntriesList,
            &hEntry, hDSRLEntry);
        if (eReturnCode == OSAL_ERROR_LIST_ITEM_NOT_UNIQUE)
        {
            // nothing to do here, item in list already
            bAdded = TRUE;
            bContinue = TRUE;
            break;
        }
        else if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": cannot add entry"
                " to list (%s)", OSAL.pacGetReturnCodeName(eReturnCode));
            break;
        }

        // now changing DSRL state to UPDATING
        vChangeDSRLState(hDSRL, DSRL_STATE_UPDATING);

        eDSRLResult = DSRL_eAddEntry(hDSRL, hDSRLEntry);
        if (eDSRLResult != DSRL_ADD_REPLACE_OK)
        {
            printf(WS_ALERTS_MGR_OBJECT_NAME": entry cannot be "
                "added for some reason.\n");

            // removing the entry from DSRL desc entries list
            eReturnCode = OSAL.eLinkedListRemove(hEntry);
            if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WS_ALERTS_MGR_OBJECT_NAME": cannot remove entry"
                    " from list (%s)", OSAL.pacGetReturnCodeName(eReturnCode));
                break;
            }

            bContinue = TRUE;
            break;
        }
        bAdded = TRUE;

        if (psEntryDesc->hDSRLList == OSAL_INVALID_OBJECT_HDL)
        {
            // in case if DSRLs list in EntryDesc struct is not created
            // lets create it
            eReturnCode = OSAL.eLinkedListCreate(&(psEntryDesc->hDSRLList),
                WS_ALERTS_MGR_OBJECT_NAME":DSRLEntry:DSRLList",
                n16CompareHandles, OSAL_LL_OPTION_NONE);
            if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WS_ALERTS_MGR_OBJECT_NAME": cannot create link list (%s)",
                    OSAL.pacGetReturnCodeName(eReturnCode));
                break;
            }
        }

        eReturnCode = OSAL.eLinkedListAdd(psEntryDesc->hDSRLList,
            OSAL_INVALID_LINKED_LIST_ENTRY_PTR, hDSRL);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": cannot add entry"
                " to list (%s)", OSAL.pacGetReturnCodeName(eReturnCode));
            break;
        }

        bContinue = TRUE;
    } while (FALSE);

    if (((bContinue == FALSE) && (bAdded == TRUE)) || 
        ((bContinue == TRUE) && (bAdded == FALSE)))
    {
        //lets remove entry from DSRL to avoid any issues
        DSRL_vRemoveEntry(hDSRL, hDSRLEntry);
    }

    return bContinue;
}

/*****************************************************************************
 *
 *   bDSRLRemoveEntry
 *
 *****************************************************************************/
static BOOLEAN bDSRLRemoveEntry(
    DSRL_OBJECT hDSRL,
    DSRL_ENTRY_OBJECT hDSRLEntry
        )
{
    BOOLEAN bResult = FALSE;

    if (hDSRLEntry != DSRL_ENTRY_INVALID_OBJECT)
    {
        vChangeDSRLState( hDSRL, DSRL_STATE_UPDATING );

        DSRL_vRemoveEntry( hDSRL, hDSRLEntry );
        bResult = TRUE;
    }

    return bResult;
}

/*****************************************************************************
*
*   bEntryRemoveFromDSRL
*
*****************************************************************************/
static BOOLEAN bEntryRemoveFromDSRL(
    DSRL_ENTRY_OBJECT hDSRLEntry,
    WS_ALERTS_DSRL_ENTRIES_LIST_ITERATOR_STRUCT *psIterator
        )
{
    BOOLEAN bContinue = FALSE;

    if ((hDSRLEntry != DSRL_ENTRY_INVALID_OBJECT) &&
        (psIterator != NULL))
    {
        WS_ALERTS_DSRL_ENTRY_DESC_STRUCT *psEntryDesc;

        psEntryDesc = 
            (WS_ALERTS_DSRL_ENTRY_DESC_STRUCT *)DSRL_ENTRY_pvServiceData(
            hDSRLEntry);
        if ((psEntryDesc != NULL) && (psEntryDesc->un32TransactionID != 
            psIterator->psAppObj->un32TransactionID))
        {
            // since we are going to change DSRL content,
            // need to set state to updating
            vChangeDSRLState(psIterator->hDSRL, DSRL_STATE_UPDATING);

            DSRL_vRemoveEntry(psIterator->hDSRL, hDSRLEntry);
        }

        bContinue = TRUE;
    }

    return bContinue;
}

/*****************************************************************************
*
*   bDSRLRemoveEntries
*
*****************************************************************************/
static BOOLEAN bDSRLRemoveEntries(
    WS_ALERTS_APP_OBJECT_STRUCT *psAppObj,
    DSRL_OBJECT hDSRL
        )
{
    BOOLEAN bResult = FALSE;

    do
    {
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
        WS_ALERTS_DSRL_ENTRIES_LIST_ITERATOR_STRUCT sIterator;
        WS_ALERTS_DSRL_DESC_STRUCT *psDSRLDesc;

        psDSRLDesc = 
            (WS_ALERTS_DSRL_DESC_STRUCT*)DSRL_pvServiceData(hDSRL);
        if (psDSRLDesc == NULL)
        {
            break;
        }

        sIterator.hDSRL = hDSRL;
        sIterator.psAppObj = psAppObj;

        eReturnCode = OSAL.eLinkedListIterate(psDSRLDesc->hEntriesList,
            (OSAL_LL_ITERATOR_HANDLER)bEntryRemoveFromDSRL, &sIterator
                );
        if (eReturnCode == OSAL_NO_OBJECTS)
        {
            puts(WS_ALERTS_MGR_OBJECT_NAME": list is empty");
        }
        else if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": cannot iterate"
                " list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }

        bResult = TRUE;
    } while(FALSE);

    return bResult;
}

/*****************************************************************************
*
*   bDSRLUpdateContent
*
*****************************************************************************/
static BOOLEAN bDSRLUpdateContent(
    DSRL_OBJECT hDSRL,
    WS_ALERTS_APP_OBJECT_STRUCT *psAppObj
        )
{
    BOOLEAN bResult = FALSE;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

    do
    {
        BOOLEAN bSuccess = FALSE;
        WS_ALERTS_DSRL_DESC_STRUCT *psDSRLDesc = NULL;
        WS_ALERTS_DSRL_ENTRIES_LIST_ITERATOR_STRUCT sIterator;

        psDSRLDesc = (WS_ALERTS_DSRL_DESC_STRUCT*)DSRL_pvServiceData(hDSRL);
        if (psDSRLDesc == NULL)
        {
            break;
        }

        // Now incrementing the transaction ID to use it for newly
        // generated DSRL entries
        // we suppose that transaction ID will uniquely identify these entries
        psAppObj->un32TransactionID++;

        sIterator.psAppObj = psAppObj;
        sIterator.hDSRL = hDSRL;

        eReturnCode = OSAL.eLinkedListIterate(psDSRLDesc->hTargetList,
            (OSAL_LL_ITERATOR_HANDLER)bTargetEntriesGenerator, &sIterator
                );
        if ((eReturnCode != OSAL_SUCCESS) && 
            (eReturnCode != OSAL_NO_OBJECTS))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": Error iterating targets list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }

        // removing non-actual entries from DSRL
        // based on transaction ID
        bSuccess = bDSRLRemoveEntries(psAppObj, hDSRL);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS_MGR_OBJECT_NAME": cannot remove entries from DSRL");
            break;
        }

        bResult = TRUE;
    } while (FALSE);

    return bResult;
}

/*****************************************************************************
*
*   vDSRLRelease
*
*****************************************************************************/
static void vDSRLRelease(
    DSRL_OBJECT hDSRL
        )
{
    if (hDSRL != DSRL_INVALID_OBJECT)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
        WS_ALERTS_DSRL_DESC_STRUCT *psDSRLDesc = NULL;

        printf(WS_ALERTS_MGR_OBJECT_NAME": Releasing DSRL %p\n", hDSRL);

        psDSRLDesc = (WS_ALERTS_DSRL_DESC_STRUCT*)DSRL_pvServiceData(hDSRL);
        if (psDSRLDesc != NULL)
        {
            if (psDSRLDesc->hTargetList != OSAL_INVALID_OBJECT_HDL)
            {
                eReturnCode = OSAL.eLinkedListRemoveAll(
                    psDSRLDesc->hTargetList,
                    (OSAL_LL_RELEASE_HANDLER)DSRL_TARGET_vDestroyByType
                        );
                if (eReturnCode != OSAL_SUCCESS)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        WS_ALERTS_MGR_OBJECT_NAME": failed to cleanup"
                        " list (%s)", OSAL.pacGetReturnCodeName(eReturnCode)
                            );
                }

                eReturnCode = OSAL.eLinkedListDelete(
                    psDSRLDesc->hTargetList
                        );
                if (eReturnCode != OSAL_SUCCESS)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        WS_ALERTS_MGR_OBJECT_NAME": failed to delete"
                        " list (%s)", OSAL.pacGetReturnCodeName(eReturnCode)
                            );
                }
                psDSRLDesc->hTargetList = OSAL_INVALID_OBJECT_HDL;
            }

            if (psDSRLDesc->hEntriesList != OSAL_INVALID_OBJECT_HDL)
            {
                eReturnCode = OSAL.eLinkedListRemoveAll(
                    psDSRLDesc->hEntriesList, NULL
                        );
                if (eReturnCode != OSAL_SUCCESS)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        WS_ALERTS_MGR_OBJECT_NAME": failed to cleanup"
                        " list (%s)", OSAL.pacGetReturnCodeName(eReturnCode)
                            );
                }

                eReturnCode = OSAL.eLinkedListDelete(psDSRLDesc->hEntriesList);
                if (eReturnCode != OSAL_SUCCESS)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        WS_ALERTS_MGR_OBJECT_NAME": failed to delete"
                        " list (%s)", OSAL.pacGetReturnCodeName(eReturnCode)
                            );
                }
                psDSRLDesc->hEntriesList = OSAL_INVALID_OBJECT_HDL;
            }
        }

        DSRL_vDestroy(hDSRL);
    }

    return;
}
