/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:WS_ALERTS_LOCATION implementation for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/
#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "sms_obj.h"
#include "_ws_alerts_location_obj.h"
#include "ws_alerts_location_obj.h"
#include "location_obj.h"
#include "shape_obj.h"
#include "string_obj.h"

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   hShape
*
*****************************************************************************/
static SHAPE_OBJECT hShape (
    WS_ALERTS_LOCATION_OBJECT hWsAlertsLocation
        )
{
    BOOLEAN bOwner = FALSE;
    bOwner = SMSO_bOwner((SMS_OBJECT)hWsAlertsLocation);

    if(bOwner == TRUE)
    {
        WS_ALERTS_LOCATION_OBJECT_STRUCT *psObj =
            (WS_ALERTS_LOCATION_OBJECT_STRUCT *)hWsAlertsLocation;
        return psObj->hShape;
    }

    return SHAPE_INVALID_OBJECT;
}

/*****************************************************************************
*
*   hLocationExtType
*
*****************************************************************************/
static STRING_OBJECT hLocationExtType (
    WS_ALERTS_LOCATION_OBJECT hWsAlertsLocation
        )
{
    BOOLEAN bOwner = FALSE;
    bOwner = SMSO_bOwner((SMS_OBJECT)hWsAlertsLocation);

    if(bOwner == TRUE)
    {
        WS_ALERTS_LOCATION_OBJECT_STRUCT *psObj =
            (WS_ALERTS_LOCATION_OBJECT_STRUCT *)hWsAlertsLocation;
        return psObj->hExtType;
    }

    return STRING_INVALID_OBJECT;
}

/*****************************************************************************
*
*   hDescription
*
*****************************************************************************/
static STRING_OBJECT hDescription (
    WS_ALERTS_LOCATION_OBJECT hWsAlertsLocation
        )
{
    BOOLEAN bOwner = FALSE;
    bOwner = SMSO_bOwner((SMS_OBJECT)hWsAlertsLocation);

    if(bOwner == TRUE)
    {
        WS_ALERTS_LOCATION_OBJECT_STRUCT *psObj =
            (WS_ALERTS_LOCATION_OBJECT_STRUCT *)hWsAlertsLocation;
        return psObj->hInfo;
    }

    return STRING_INVALID_OBJECT;
}

/*****************************************************************************
*
*   n32FPrintf
*
*****************************************************************************/
static N32 n32FPrintf (
    WS_ALERTS_LOCATION_OBJECT hWsAlertsLocation,
    FILE *psFile,
    SMSAPI_OUTPUT_OPTION_ENUM eOutputOption
        )
{
    BOOLEAN bOwner = FALSE;
    N32 n32Return = 0, n32Temp = 0;
    WS_ALERTS_LOCATION_OBJECT_STRUCT *psObj = NULL;

    bOwner = SMSO_bOwner((SMS_OBJECT)hWsAlertsLocation);
    if ((bOwner == FALSE) || (psFile == NULL) ||
        ((eOutputOption != SMS_OUTPUT_OPTION_TERSE) &&
         (eOutputOption != SMS_OUTPUT_OPTION_VERBOSE) &&
         (eOutputOption != SMS_OUTPUT_OPTION_GROSS)))
    {
        return EOF;
    }

    psObj = (WS_ALERTS_LOCATION_OBJECT_STRUCT *) hWsAlertsLocation;
    n32Return += fprintf(psFile, "  WEATHER/SECURITY ALERT LOCATION:\n\tObject : %p\n",
        psObj);
    n32Return += fprintf(psFile, "\tID : %u\n", psObj->tLocId);
    n32Return += fprintf(psFile, "\tType : %s (%s)\n",
        STRING.pacCStr(psObj->hExtType), STRING.pacCStr(psObj->hExtRef));
    n32Return += fprintf(psFile, "\tInfo : %s\n",
        STRING.pacCStr(psObj->hInfo));

    switch (eOutputOption)
    {
        case SMS_OUTPUT_OPTION_TERSE:
        {
            if ( psObj->bIsPolygon == TRUE )
            {
                n32Return += fprintf(psFile, "\tShape : ");
                n32Temp = SHAPE.n32FPrintf( psObj->hShape, psFile, eOutputOption );

                if (n32Temp > 0)
                {
                    n32Return += n32Temp;
                }
            }
        }
        break;

        case SMS_OUTPUT_OPTION_VERBOSE:
        {
            n32Return += fprintf(psFile, "\tShape : ");
        }
        // fall through down to GROSS printout

        case SMS_OUTPUT_OPTION_GROSS:
        {
            n32Temp = SHAPE.n32FPrintf( psObj->hShape, psFile, eOutputOption );

            if (n32Temp > 0)
            {
                n32Return += n32Temp;
            }
        }
        break;

    }

    return n32Return;
}

/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   WS_ALERTS_LOCATION_hCreate
*
*****************************************************************************/
WS_ALERTS_LOCATION_OBJECT WS_ALERTS_LOCATION_hCreate(
    SMS_OBJECT hParent,
    WS_ALERTS_LOCATIONS_ROW_STRUCT *psAlertLocationRow
        )
{
    WS_ALERTS_LOCATION_OBJECT hResult = WS_ALERTS_LOCATION_INVALID_OBJECT;
    WS_ALERTS_LOCATION_OBJECT_STRUCT *psObj = NULL;

    do
    {
        BOOLEAN bSuccess = TRUE;
        size_t tIndex = 0;
        UN16 un16Points;

        // Implemented algorithm is based on assumption that processed polygon
        // is always a closed shape (the ending point is equal to the beginning
        // one. If it is not the case, we need to add an extra point to close
        // the shape.
        N32 n32LastPointLat = 0;
        N32 n32LastPointLon = 0;
        BOOLEAN bLastPointToAdd = FALSE;

        if (psAlertLocationRow == NULL)
        {
            break;
        }

        // Initial value is equal to current number of vertices
        un16Points = (UN16)psAlertLocationRow->tNumVertices;

        // Create an instance of the WS_ALERTS_LOCATION_OBJECT_STRUCT object
        psObj = (WS_ALERTS_LOCATION_OBJECT_STRUCT *)
            SMSO_hCreate(WS_ALERTS_LOCATION_OBJECT_NAME":Data",
            sizeof(WS_ALERTS_LOCATION_OBJECT_STRUCT), hParent, FALSE);
        if(psObj == NULL)
        {
            // Error!
            break;
        }

        if (psAlertLocationRow->bIsPolygon == TRUE)
        {
            psObj->bIsPolygon = TRUE;
            psObj->tMsgId = psAlertLocationRow->tMsgId;
            psObj->tLocId = psAlertLocationRow->tLocId;
            psObj->hInfo = STRING.hCreate(WS_ALERTS_POLYGON_TEXT, sizeof(WS_ALERTS_POLYGON_TEXT));
            if (psObj->hInfo == STRING_INVALID_OBJECT)
            {
                // Error! no memory?
                break;
            }
        }
        else
        {
            psObj->tLocId = psAlertLocationRow->tLocId;
            psObj->hInfo = STRING.hDuplicate(psAlertLocationRow->hInfo);

            if (psObj->hInfo == STRING_INVALID_OBJECT)
            {
                break;
            }

            psObj->hExtRef = STRING.hDuplicate(psAlertLocationRow->hExtRef);

            if (psObj->hExtRef == STRING_INVALID_OBJECT)
            {
                break;
            }

            psObj->hExtType = STRING.hDuplicate(psAlertLocationRow->hExtType);

            if (psObj->hExtType == STRING_INVALID_OBJECT)
            {
                break;
            }
        }

        psObj->hShape = SHAPE_hCreate((SMS_OBJECT)hParent);
        if (psObj->hShape == SHAPE_INVALID_OBJECT)
        {
            break;
        }

        // getting Lat and Lon of the last point
        n32LastPointLon = psAlertLocationRow->pn32LonLat[(un16Points - 1) * 2];
        n32LastPointLat = psAlertLocationRow->pn32LonLat[(un16Points - 1) * 2 + 1];

        // verifying if polygon is closed
        if ((n32LastPointLon != psAlertLocationRow->pn32LonLat[0]) ||
            (n32LastPointLat != psAlertLocationRow->pn32LonLat[1]))
        {
            // increment the number of points since one more point will be added equal to the first one
            un16Points++;
            bLastPointToAdd = TRUE;
        }

        bSuccess = SHAPE_bPreparePoints(
            psObj->hShape,
            un16Points,
            SHAPE_TYPE_CLOSED );

        if (TRUE != bSuccess)
        {
            break;
        }

        for( tIndex = 0;
             (tIndex < psAlertLocationRow->tNumVertices) &&
             (bSuccess == TRUE);
             tIndex++ )
        {
            bSuccess = SHAPE_bSetPoint(
                psObj->hShape,
                (UN16)tIndex,
                psAlertLocationRow->pn32LonLat[tIndex * 2],
                psAlertLocationRow->pn32LonLat[tIndex * 2 +1],
                LOCATION_BINPOINT );
        }

        // adding last point equal to the first one to close polygon
        if (TRUE == bSuccess && TRUE == bLastPointToAdd)
        {
            bSuccess = SHAPE_bSetPoint(
                psObj->hShape,
                (UN16)psAlertLocationRow->tNumVertices,
                n32LastPointLat,
                n32LastPointLon,
                LOCATION_BINPOINT );
        }

        if (TRUE != bSuccess)
        {
            break;
        }

        hResult = (WS_ALERTS_LOCATION_OBJECT)psObj;
    } while (FALSE);

    if (hResult == WS_ALERTS_LOCATION_INVALID_OBJECT &&
        psObj != NULL)
    {
        vDestroyObject(psObj);
    }

    return hResult;
}

/*****************************************************************************
*
*   WS_ALERTS_LOCATION_tLocId
*
*****************************************************************************/
LOC_ID WS_ALERTS_LOCATION_tLocId(
    WS_ALERTS_LOCATION_OBJECT hWsAlertsLocation
        )
{
    LOC_ID tLocId = LOC_INVALID_ID;

    if (WS_ALERTS_LOCATION_INVALID_OBJECT != hWsAlertsLocation)
    {
        WS_ALERTS_LOCATION_OBJECT_STRUCT *psObj =
            (WS_ALERTS_LOCATION_OBJECT_STRUCT *)hWsAlertsLocation;

        tLocId = psObj->tLocId;
    }

    return tLocId;
}

/*****************************************************************************
*
*   WS_ALERTS_LOCATION_bIsPolygon
*
*****************************************************************************/
BOOLEAN WS_ALERTS_LOCATION_bIsPolygon(
    WS_ALERTS_LOCATION_OBJECT hWsAlertsLocation,
    BOOLEAN *pbIsPolygon
        )
{
    BOOLEAN bOwner = FALSE;

    if ( pbIsPolygon == NULL )
    {
        return FALSE;
    }

    bOwner = SMSO_bOwner((SMS_OBJECT)hWsAlertsLocation);

    if(bOwner == TRUE)
    {
        WS_ALERTS_LOCATION_OBJECT_STRUCT *psObj =
            (WS_ALERTS_LOCATION_OBJECT_STRUCT *)hWsAlertsLocation;
        *pbIsPolygon = psObj->bIsPolygon;
        return TRUE;
    }

    return FALSE;
}

/*****************************************************************************
*
*   WS_ALERTS_LOCATION_tMsgId
*
*****************************************************************************/
WS_ALERT_MSG_ID WS_ALERTS_LOCATION_tMsgId(
    WS_ALERTS_LOCATION_OBJECT hWsAlertsLocation
        )
{
    WS_ALERT_MSG_ID tMsgId = WS_ALERT_MSG_INVALID_ID;
    BOOLEAN bOwner = FALSE;
    bOwner = SMSO_bOwner((SMS_OBJECT)hWsAlertsLocation);

    if(bOwner == TRUE)
    {
        WS_ALERTS_LOCATION_OBJECT_STRUCT *psObj =
            (WS_ALERTS_LOCATION_OBJECT_STRUCT *)hWsAlertsLocation;
        tMsgId = psObj->tMsgId;
    }

    return tMsgId;
}

/*****************************************************************************
*
*   WS_ALERTS_LOCATION_vDestroy
*
*****************************************************************************/
void WS_ALERTS_LOCATION_vDestroy (
    WS_ALERTS_LOCATION_OBJECT hWSAlertsLoc
        )
{
    BOOLEAN bOwner = FALSE;
    printf(WS_ALERTS_LOCATION_OBJECT_NAME": attempting to destroy object 0x%p\n",
        hWSAlertsLoc);
    bOwner = SMSO_bOwner((SMS_OBJECT)hWSAlertsLoc);

    if(bOwner == TRUE)
    {
        WS_ALERTS_LOCATION_OBJECT_STRUCT *psObj =
            (WS_ALERTS_LOCATION_OBJECT_STRUCT *)hWSAlertsLoc;
        vDestroyObject(psObj);
    }

    return;
}

/*****************************************************************************
*
*   WS_ALERTS_LOCATION_hCreateDummy
*
*****************************************************************************/
WS_ALERTS_LOCATION_OBJECT WS_ALERTS_LOCATION_hCreateDummy(
    SMS_OBJECT hParent
        )
{
    WS_ALERTS_LOCATION_OBJECT_STRUCT *psObj;

    // Create a dummy instance of the WS_ALERTS_LOCATION_OBJECT_STRUCT object
    psObj = (WS_ALERTS_LOCATION_OBJECT_STRUCT *)
        SMSO_hCreate(WS_ALERTS_LOCATION_OBJECT_NAME":DummyData",
        sizeof(WS_ALERTS_LOCATION_OBJECT_STRUCT), hParent, FALSE);

    if (NULL != psObj)
    {
        psObj->tLocId = LOC_INVALID_ID;
    }

    return (WS_ALERTS_LOCATION_OBJECT)psObj;
}

/*****************************************************************************
*
*   WS_ALERTS_LOCATION_vSetDummyId
*
*****************************************************************************/
void WS_ALERTS_LOCATION_vSetDummyId(
    WS_ALERTS_LOCATION_OBJECT hWSAlertsLoc,
    LOC_ID tLocId
        )
{
    if (WS_ALERTS_LOCATION_INVALID_OBJECT != hWSAlertsLoc)
    {
        WS_ALERTS_LOCATION_OBJECT_STRUCT *psObj =
            (WS_ALERTS_LOCATION_OBJECT_STRUCT *)hWSAlertsLoc;

        psObj->tLocId = tLocId;
    }

    return;
}

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   vDestroyObject
*
*****************************************************************************/
static void vDestroyObject(
    WS_ALERTS_LOCATION_OBJECT_STRUCT *psObj
        )
{
    if (psObj != NULL)
    {
        if (psObj->hInfo != STRING_INVALID_OBJECT)
        {
            STRING_vDestroy(psObj->hInfo);
            psObj->hInfo = STRING_INVALID_OBJECT;
        }

        if (psObj->hExtRef != STRING_INVALID_OBJECT)
        {
            STRING_vDestroy(psObj->hExtRef);
            psObj->hExtRef = STRING_INVALID_OBJECT;
        }

        if (psObj->hExtType != STRING_INVALID_OBJECT)
        {
            STRING_vDestroy(psObj->hExtType);
            psObj->hExtType = STRING_INVALID_OBJECT;
        }

        if (psObj->hShape != SHAPE_INVALID_OBJECT)
        {
            SHAPE_vDestroy( psObj->hShape );
            psObj->hShape = SHAPE_INVALID_OBJECT;
        }

        // Free object instance itself
        SMSO_vDestroy((SMS_OBJECT)psObj);
    }

    return;
}
