/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 * This file defines the constants used when utilizing the alerts
 * database file
 *
 ******************************************************************************/

  /*********************************/
 /** PREVENT REDUNDANT INCLUSION **/
/*********************************/
#ifndef WS_ALERTS_DB_CONSTANTS_H_
#define WS_ALERTS_DB_CONSTANTS_H_

  /**************/
 /** INCLUDES **/
/**************/

#include "sms_api.h"
#include "location_obj.h"

  /**********************/
 /** GLOBAL CONSTANTS **/
/**********************/

/* Database File Name */
#define WS_ALERTS_DATABASE_FOLDER "ws_alerts"
#define WS_ALERTS_REF_DATABASE_FILENAMEA "ws_alerts_ref.db"
#define WS_ALERTS_REF_DATABASE_FILENAMEB "ws_alerts_ref.b.db"
#define WS_ALERTS_PERSIST_DATABASE_FILENAME "ws_alerts.db"

/* Database Version */
#define WS_ALERTS_DATABASE_FILE_VERSION (5)
#define WS_ALERTS_PERSIST_DATABASE_FILE_VERSION (6)

#define WS_ALERTS_LOCATIONS_WEST_RTREE_TABLE_NAME "loc_west_rtree"
#define WS_ALERTS_LOCATIONS_EAST_RTREE_TABLE_NAME "loc_east_rtree"

#define WS_ALERTS_DATABASE_VERSION_TABLE_NAME "db_version"
#define WS_ALERTS_UPDATE_VERSION_TABLE_NAME "data_version"
#define WS_ALERTS_LUT_DESCRIPTION_TABLE_NAME "LUT_desc"

#define WS_ALERTS_LUT_NAME "C%dT%dL%d"
#define WS_ALERTS_LUT_NAME_STRING "%s"

#define WS_ALERTS_MESSAGES_TABLE_NAME "messages"
#define WS_ALERTS_MESSAGE_LOCATIONS_TABLE_NAME "message_locations"
#define WS_ALERTS_MESSAGE_TYPES_TABLE_NAME "message_types"
#define WS_ALERTS_POLYGONS_TABLE_NAME "polygons"

#define WS_ALERTS_POLYGON_TEXT "free polygon"

// Constant used to size buffers used to perform
// DB queries.  Size is based on largest string
// size (all defines above specify field lengths)
#define WS_ALERTS_MAX_SQL_STRING_LENGTH (2000)
#define WS_ALERTS_MAX_LUT_NAME_LENGTH (12)

#define WS_ALERTS_RTREE_RANGE      ((N32)62)    // 0.000015 * 2^22

#define WS_ALERTS_RTREE_MAX_LON    ((N32)0x2D000000)    // 180 * 2^22
#define WS_ALERTS_RTREE_MIN_LON    (-(N32)0x2D000000)   // -180 * 2^22


#define WS_ALERTS_DB_LON_BYTE_SIZE 4

// Insert rtree for ws alerts location
#define WS_ALERTS_INSERT_EASTERN_LOCATION_RTREE \
    "insert or replace into "WS_ALERTS_LOCATIONS_EAST_RTREE_TABLE_NAME \
    " values(%u, %i, %i, %i, %i);"

#define WS_ALERTS_INSERT_WESTERN_LOCATION_RTREE \
    "insert or replace into "WS_ALERTS_LOCATIONS_WEST_RTREE_TABLE_NAME \
    " values(%u, %i, %i, %i, %i);"

#define WS_ALERTS_SELECT_LOCATION \
    "select tabindex, lcode, ltext, extref, exttype, lon, lat " \
    "from %s where " \
    "tabindex = %d and " \
    "lcode = %d;"

#define WS_ALERTS_DELETE_RTREE_ROW_BY_ID \
    "delete from %s where id = %d;"

// Select all from version
#define WS_ALERTS_SELECT_DB_VERSION \
    "select version from "WS_ALERTS_DATABASE_VERSION_TABLE_NAME";"

// Select ws_alerts db update number
#define WS_ALERTS_SELECT_UPDATE_NUMBER \
    "select version from "WS_ALERTS_UPDATE_VERSION_TABLE_NAME" where type='update_number';"

// Select ws_alerts db LUT version
#define WS_ALERTS_SELECT_VERSION_FROM_LUT \
    "select version from "WS_ALERTS_UPDATE_VERSION_TABLE_NAME" where type='%s';"

// Select ws_alerts LUT description
#define WS_ALERTS_SELECT_LUT_DESC \
    "select tableName, labid, fieldName, fieldType from " \
    WS_ALERTS_LUT_DESCRIPTION_TABLE_NAME" where tableName='%s';"

// Select ws_alerts db LUT description
#define WS_ALERTS_SELECT_LUT_DESC2 \
    "select type, version from "WS_ALERTS_UPDATE_VERSION_TABLE_NAME" where type='%s';"

// Select ws_alerts db LUT version
#define WS_ALERTS_SELECT_LUT_VERSION \
    "select type, version from "WS_ALERTS_UPDATE_VERSION_TABLE_NAME" where type LIKE \'C%%T%%L%%\';"

// Create ws_alerts LUT
#define WS_ALERTS_CREATE_LUT_TABLE_BEGIN \
    "create table %s ( "

// Select ws_alerts db LUT version: join strings
#define WS_ALERTS_CREATE_LUT_TABLE_JOIN \
    "%s %s"

// Select ws_alerts db LUT version
#define WS_ALERTS_CREATE_LUT_TABLE_END \
    "%s );"

// update data_version table with new LUT table
#define WS_ALERTS_INSERT_LUT_DATA_VERSION \
    "insert into "WS_ALERTS_UPDATE_VERSION_TABLE_NAME" values ( '%s', %d );"

// Insert into LUT description table new row
#define WS_ALERTS_INSERT_ROW_LUT_DESC \
    "insert into "WS_ALERTS_LUT_DESCRIPTION_TABLE_NAME" values ('%s', %u, '%s', '%s');"

// Make a new column in the specific LUT
#define WS_ALERTS_CREATE_LUT_COLUMN \
    "alter table %s add column %s %s;"

// Update name field for LUT description table
#define WS_ALERTS_UPDATE_LUT_DESC_NAME_FIELD \
    "update "WS_ALERTS_LUT_DESCRIPTION_TABLE_NAME" set fieldName='%s' where tableName='%s' and labid=%u;"

// Update type field for LUT description table
#define WS_ALERTS_UPDATE_LUT_DESC_TYPE_FIELD \
    "update "WS_ALERTS_LUT_DESCRIPTION_TABLE_NAME" set fieldType='%s' where tableName='%s' and labid=%u;"

// Rename LUT Table
#define WS_ALERTS_RENAME_LUT \
    "ALTER TABLE %s RENAME TO %s;"

// Create new LUT Table
#define WS_ALERTS_CREATE_LUT \
    "CREATE TABLE %s (%s);"

// Move data from one table to another
#define WS_ALERTS_MOVE_LUT_DATA \
    "INSERT INTO %s(%s) SELECT %s FROM %s;"

// Drop table
#define WS_ALERTS_DROP_TABLE \
    "DROP TABLE %s;"

// Change update number field
#define WS_ALERTS_SET_UPDATE_NUMBER \
    "update "WS_ALERTS_UPDATE_VERSION_TABLE_NAME" set version=%d where type='update_number';"

#define WS_ALERTS_INSERT_VERSION_ROW \
    "INSERT INTO "WS_ALERTS_DATABASE_VERSION_TABLE_NAME" VALUES(%u);"

#define WS_ALERTS_CREATE_MESSAGES_TABLE "create table "WS_ALERTS_MESSAGES_TABLE_NAME" (  " \
    "messageID integer primary key,  " \
    "stateID int, " \
    "language tinyint, " \
    "msgText text, " \
    "startTime int, " \
    "endTime int, " \
    "priority tinyint, " \
    "polygon tinyint, " \
    "version int, " \
    "rate tinyint " \
    "); "

#define WS_ALERTS_CREATE_END_TIME_INDEX_MSG_TABLE  "create index " \
    "endTime_idx on "WS_ALERTS_MESSAGES_TABLE_NAME" (endTime);"

#define WS_ALERTS_CREATE_MESSAGE_LOCATIONS_TABLE  "create table " \
    WS_ALERTS_MESSAGE_LOCATIONS_TABLE_NAME" (  " \
    "messageID int,  " \
    "LocationID int " \
    "); "

#define WS_ALERTS_CREATE_MSG_LOC_ID_INDEX_LOC_TABLE  "create index " \
    "msg_locid_idx on "WS_ALERTS_MESSAGE_LOCATIONS_TABLE_NAME" (LocationID asc);"

#define WS_ALERTS_CREATE_MESSAGE_TYPES_TABLE  "create table " \
    WS_ALERTS_MESSAGE_TYPES_TABLE_NAME" (  " \
    "messageID int,  " \
    "msgType text, " \
    "aText text, " \
    "primary key (messageID, msgType) " \
    "); "

#define WS_ALERTS_CREATE_POLYGONS_TABLE  "create table " \
    WS_ALERTS_POLYGONS_TABLE_NAME" (  " \
    "polygonID integer primary key autoincrement,  " \
    "messageID int,  " \
    "minLon int,  " \
    "maxLon int,  " \
    "minLat int,  " \
    "maxLat int,  " \
    "lonLat blob " \
    "); "

#define WS_ALERTS_CREATE_DB_VERSION_TABLE  "create table " \
    WS_ALERTS_DATABASE_VERSION_TABLE_NAME" (  " \
    "version int  " \
    "); "

// Insert a new text row
#define WS_ALERTS_INSERT_MSG_ROW \
    "insert or replace into "WS_ALERTS_MESSAGES_TABLE_NAME \
    " (messageID, stateID, language, msgText, startTime, endTime, "\
    "priority, polygon, version, rate) " \
    "values(?, ?, ?, ?, ?, ?, ?, ?, ?, ?);"

// Insert a new msg_location row
#define WS_ALERTS_INSERT_MSG_LOCATION_ROW \
    "insert into "WS_ALERTS_MESSAGE_LOCATIONS_TABLE_NAME \
    "(messageID, LocationID)" \
    "values(?, ?);"

// Insert a new msg_type row
#define WS_ALERTS_INSERT_MSG_TYPE_ROW \
    "insert or replace into "WS_ALERTS_MESSAGE_TYPES_TABLE_NAME \
    "(messageID, msgType, aText)" \
    "values(?, ?, ?);"

// Insert a new polygon row
#define WS_ALERTS_INSERT_MSG_POLYGONS_ROW \
    "insert into "WS_ALERTS_POLYGONS_TABLE_NAME \
    " (messageID, minLon, maxLon, minLat, maxLat, lonLat) " \
    "values(?, ?, ?, ?, ?, ?);"

//RFD commands
#define WS_ALERTS_INSERT_LUT_ROW \
    "insert into %s ("

#define WS_ALERTS_UPDATE_LUT_ROW \
    "update %s set %s where %s = %d;"

#define WS_ALERTS_UPDATE_LUT_ROW_DOUBLE_KEY \
    "update %s set %s where %s = %d and %s = %d;"

#define LOCATIONS_RTREE_TABLE_NAME "locations_rtree"

// RTree prepared statements parameter indexes
#define WS_ALERTS_RTREE_STMT_BOTTOM_LAT_PARAM   0
#define WS_ALERTS_RTREE_STMT_TOP_LAT_PARAM      1
#define WS_ALERTS_RTREE_STMT_LEFT_LON_PARAM     2
#define WS_ALERTS_RTREE_STMT_RIGHT_LON_PARAM    3
#define WS_ALERTS_RTREE_STMT_PARAMS_COUNT       4

#define WS_ALERTS_SELECT_LOCATIONS_IDS_FROM_WEST_RTREE_TABLE \
    "select id from "WS_ALERTS_LOCATIONS_WEST_RTREE_TABLE_NAME" where " \
    "maxLat >= ?1 and minLat <= ?2 and maxLon >= ?3 and minLon <= ?4;"

#define WS_ALERTS_SELECT_LOCATIONS_IDS_FROM_EAST_RTREE_TABLE \
    "select id from "WS_ALERTS_LOCATIONS_EAST_RTREE_TABLE_NAME" where " \
    "maxLat >= ?1 and minLat <= ?2 and maxLon >= ?3 and minLon <= ?4;"

#define WS_ALERTS_SELECT_POLYGONS \
    "select polygonID, messageID, minLon, maxLon, minLat, maxLat, lonLat from " \
    WS_ALERTS_POLYGONS_TABLE_NAME" where " \
    "maxLat >= ?1 and minLat <= ?2 and maxLon >= ?3 and minLon <= ?4;"

// Select location from LUT by id
#define WS_ALERTS_SELECT_LOCATION_BY_ID \
    "select tabindex, lcode, ltext, extref, exttype, lon, lat " \
    "from C1T%dL%d where " \
    "tabindex = %d and " \
    "lcode = %d;"

// Select alert msg record by msg id
#define WS_ALERTS_SELECT_ALERT_MSGS_BY_MSGID \
    "select messageID, stateID, language, msgText, startTime, endTime, priority, " \
    "polygon, version, rate from messages where " \
    "messageID = %d and (endTime = 0 or endTime > %u);"

// Select alert msg record by location
#define WS_ALERTS_SELECT_MSGS_STMT_LOCID_PARAM    0
#define WS_ALERTS_SELECT_MSGS_STMT_PARAMS_COUNT   1

#define WS_ALERTS_SELECT_ALERT_MSGS_BY_LOCATION \
    "select messageID from "WS_ALERTS_MESSAGE_LOCATIONS_TABLE_NAME" where " \
    "LocationID = ? order by messageID;"

#define WS_ALERTS_SELECT_POLYGONS_BY_MSG_ID \
    "select polygonID, messageID, minLon, maxLon, minLat, maxLat, lonLat from " \
    WS_ALERTS_POLYGONS_TABLE_NAME" where messageID = %d;"

// Select alert type row
#define WS_ALERTS_SELECT_ALERT_TYPE_ROW \
    "select acode, extref, apriority, atext from %s where acode = %d;"

// Select alert type row
#define WS_ALERTS_SELECT_ALERT_LOCATION_ROW \
    "select tabindex, lcode, ltext, extref, exttype, lon, lat " \
    "from %s where tabindex = %d AND lcode = %d;"

// Select alert common phrase row
#define WS_ALERTS_SELECT_ALERT_COMMON_PHRASE_ROW \
    "select cpcode, cptext from %s where cpcode = %d;"

#define WS_ALERTS_DELETE_OBSOLETE_LOCATIONS \
    "delete from "WS_ALERTS_MESSAGE_LOCATIONS_TABLE_NAME \
    " where messageID in " \
    "(select messageID from "WS_ALERTS_MESSAGES_TABLE_NAME \
    " where stateID = %d and language = %d and rate = %d);"

#define WS_ALERTS_DELETE_OBSOLETE_MSG_TYPES \
    "delete from "WS_ALERTS_MESSAGE_TYPES_TABLE_NAME \
    " where messageID in " \
    "(select messageID from "WS_ALERTS_MESSAGES_TABLE_NAME \
    " where stateID = %d and language = %d and rate = %d);"

#define WS_ALERTS_DELETE_OBSOLETE_POLYGONS \
    "delete from "WS_ALERTS_POLYGONS_TABLE_NAME \
    " where messageID in " \
    "(select messageID from "WS_ALERTS_MESSAGES_TABLE_NAME \
    " where stateID = %d and language = %d and rate = %d);"

#define WS_ALERTS_DELETE_OBSOLETE_MESSAGES \
    "delete from "WS_ALERTS_MESSAGES_TABLE_NAME \
    " where stateID = %d and language = %d and rate = %d;"

#define WS_ALERTS_DELETE_EXPIRED_LOCATIONS \
    "delete from "WS_ALERTS_MESSAGE_LOCATIONS_TABLE_NAME \
    " where messageID in " \
    "(select messageID from "WS_ALERTS_MESSAGES_TABLE_NAME \
    " where endTime > 0 and endTime <= %u);"

#define WS_ALERTS_DELETE_EXPIRED_MSG_TYPES \
    "delete from "WS_ALERTS_MESSAGE_TYPES_TABLE_NAME \
    " where messageID in " \
    "(select messageID from "WS_ALERTS_MESSAGES_TABLE_NAME \
    " where endTime > 0 and endTime <= %u);"

#define WS_ALERTS_DELETE_EXPIRED_POLYGONS \
    "delete from "WS_ALERTS_POLYGONS_TABLE_NAME \
    " where messageID in " \
    "(select messageID from "WS_ALERTS_MESSAGES_TABLE_NAME \
    " where endTime > 0 and endTime <= %u);"

#define WS_ALERTS_DELETE_EXPIRED_MESSAGES \
    "delete from "WS_ALERTS_MESSAGES_TABLE_NAME \
    " where endTime > 0 and endTime <= %u;"

//RFD Update commands
#define WS_ALERTS_DELETE_ROW \
    "delete from %s where %s = %d;"

#define WS_ALERTS_DELETE_LOCATION_ROW \
    "delete from %s where %s = %d and %s = %d;"

// Select location by msg id
#define WS_ALERTS_SELECT_LOCATIONS_BY_MSG_ID \
    "select LocationID from "WS_ALERTS_MESSAGE_LOCATIONS_TABLE_NAME" where messageID = %d;"

// Select msg types by msg id
#define WS_ALERTS_SELECT_MSG_TYPES_BY_MSG_ID \
    "select msgType, aText from "WS_ALERTS_MESSAGE_TYPES_TABLE_NAME" where messageID = %d;"

#define WS_ALERTS_SELECT_MSG_TYPE_ROW \
    "select acode, extref, apriority, atext from C%dT%dL%d where acode = %d;"

#define WS_ALERTS_SELECT_LOCATION_ROW \
    "select tabindex, lcode, ltext, extref, exttype, lon, lat from C%dT%dL%d where " \
    "tabindex = %d and lcode = %d;"

#define WS_ALERTS_SELECT_COMMON_PHRASE_ROW \
    "select cpcode, cptext from C%dT%dL%d where cpcode = %d;"

#define WS_ALERTS_INSERT_OR_REPLACE \
    "insert or replace into "WS_ALERTS_UPDATE_VERSION_TABLE_NAME" (type, version) values ('%s', %d);"

#define WS_ALERTS_SELECT_FROM_KEY \
    "SELECT * FROM %s WHERE %s = %d;"

#define WS_ALERTS_SELECT_FROM_DOUBLE_KEY \
    "SELECT * FROM %s WHERE %s = %d AND %s = %d;"

//State or Marine Zone indicator
#define WS_ALERT_STATE_FLAG (0)
#define WS_ALERT_ZONE_FLAG  (1)

// The ID of the state
typedef UN8 WS_ALERT_STATE_ID;
#define WS_ALERT_NOT_ASSIGNED_STATE_ID ((WS_ALERT_STATE_ID)0)
#define WS_ALERT_STATE_ID_MAX     (96)
#define WS_ALERT_STATE_ID_MIN     (1)

// Enumeration specifying all the available fields
// in the db_version table
typedef enum db_version_fields_enum
{
    DB_VERSION_FIELD_DB_VER = 0,
    DB_VERSION_MAX_FIELDS
} DB_VERSION_FIELDS_ENUM;

// Enumeration specifying all the available fields
// in the data_version table
typedef enum data_version_fields_enum
{
    DATA_VERSION_TYPE = 0,
    DATA_VERSION_VERSION,
    DATA_VERSION_MAX_FIELDS
} DATA_VERSION_FIELDS_ENUM;

typedef enum update_version_fields_enum
{
    UPDATE_VERSION = 0,
    UPDATE_VERSION_MAX_FIELDS
} UPDATE_VERSION_FIELDS_ENUM;

typedef enum ws_alerts_db_alert_location_fields_enum
{
    WS_ALERTS_DB_LOCATION_TABINDEX = 0,
    WS_ALERTS_DB_LOCATION_LCODE,
    WS_ALERTS_DB_LOCATION_LTEXT,
    WS_ALERTS_DB_LOCATION_EXTREF,
    WS_ALERTS_DB_LOCATION_EXTTYPE,
    WS_ALERTS_DB_LOCATION_LON,
    WS_ALERTS_DB_LOCATION_LAT,
    WS_ALERTS_DB_LOCATION_MAX_FIELDS

} WS_ALERTS_DB_ALERT_LOCATION_FIELDS_ENUM;

typedef enum ws_alerts_db_alert_type_fields_enum
{
    WS_ALERTS_DB_ALERT_TYPE_ACODE = 0,
    WS_ALERTS_DB_ALERT_TYPE_EXTREF,
    WS_ALERTS_DB_ALERT_TYPE_APRIORITY,
    WS_ALERTS_DB_ALERT_TYPE_ATEXT

} WS_ALERTS_DB_ALERT_TYPE_FIELDS_ENUM;

typedef enum ws_alerts_db_alert_phrase_fields_enum
{
    WS_ALERTS_DB_ALERT_TYPE_CPCODE = 0,
    WS_ALERTS_DB_ALERT_TYPE_CPTEXT

} WS_ALERTS_DB_ALERT_PHRASE_FIELDS_ENUM;

// Enumeration specifying all the available fields
// in the message table
typedef enum ws_alerts_db_message_fields_enum
{
    WS_ALERTS_DB_MESSAGE_MESSAGE_ID = 0,
    WS_ALERTS_DB_MESSAGE_STATE_ID,
    WS_ALERTS_DB_MESSAGE_LANG,
    WS_ALERTS_DB_MESSAGE_TEXT,
    WS_ALERTS_DB_MESSAGE_START_TIME,
    WS_ALERTS_DB_MESSAGE_END_TIME,
    WS_ALERTS_DB_MESSAGE_PRIORITY,
    WS_ALERTS_DB_MESSAGE_POLYGON,
    WS_ALERTS_DB_MESSAGE_VERSION,
    WS_ALERTS_DB_MESSAGE_RATE,
    WS_ALERTS_DB_MESSAGE_MAX_FIELDS

} WS_ALERTS_DB_MESSAGE_FIELDS_ENUM;

typedef enum ws_alerts_message_fields_enum
{
    WS_ALERTS_MESSAGE_MSG_ID = 0,
    WS_ALERTS_MESSAGE_STATE_ID,
    WS_ALERTS_MESSAGE_LANG,
    WS_ALERTS_MESSAGE_TEXT,
    WS_ALERTS_MESSAGE_START_TIME,
    WS_ALERTS_MESSAGE_END_TIME,
    WS_ALERTS_MESSAGE_PRIORITY,
    WS_ALERTS_MESSAGE_POLYGON,
    WS_ALERTS_MESSAGE_VERSION,
    WS_ALERTS_MESSAGE_RATE,
    WS_ALERTS_MESSAGE_MAX_FIELDS

} WS_ALERTS_MESSAGE_FIELDS_ENUM;

// Enumeration specifying all the available fields
// in the message table
typedef enum ws_alerts_db_message_locations_fields_enum
{
    WS_ALERTS_DB_MESSAGE_LOCATIONS_MSG_ID = 0,
    WS_ALERTS_DB_MESSAGE_LOCATIONS_LOCATION_ID,
    WS_ALERTS_DB_MESSAGE_LOCATIONS_MAX_FIELDS

} WS_ALERTS_DB_MESSAGE_LOCATIONS_FIELDS_ENUM;

// Enumeration specifying all the available fields
// in the message_types table
typedef enum ws_alerts_db_message_types_fields_enum
{
    WS_ALERTS_DB_MESSAGE_TYPES_MSG_ID = 0,
    WS_ALERTS_DB_MESSAGE_TYPES_TYPE_TEXT,
    WS_ALERTS_DB_MESSAGE_TYPES_ATEXT,
    WS_ALERTS_DB_MESSAGE_TYPES_MAX_FIELDS

} WS_ALERTS_DB_MESSAGE_TYPES_FIELDS_ENUM;

// Enumeration specifying all the available fields
// in the polygon table
typedef enum ws_alerts_db_polygon_fields_enum
{
    WS_ALERTS_DB_POLYGON_MSG_ID = 0,
    WS_ALERTS_DB_POLYGON_MIN_LON,
    WS_ALERTS_DB_POLYGON_MAX_LON,
    WS_ALERTS_DB_POLYGON_MIN_LAT,
    WS_ALERTS_DB_POLYGON_MAX_LAT,
    WS_ALERTS_DB_POLYGON_LONLAT,
    WS_ALERTS_DB_POLYGON_POLYGON_ID,
    WS_ALERTS_DB_POLYGON_MAX_FIELDS

} WS_ALERTS_DB_POLYGON_FIELDS_ENUM;

// Enumeration specifying all the available fields
// in the polygon table
typedef enum ws_alerts_db_polygon_enum
{
    WS_ALERTS_POLYGON_POLYGON_ID = 0,
    WS_ALERTS_POLYGON_MSG_ID ,
    WS_ALERTS_POLYGON_MIN_LON,
    WS_ALERTS_POLYGON_MAX_LON,
    WS_ALERTS_POLYGON_MIN_LAT,
    WS_ALERTS_POLYGON_MAX_LAT,
    WS_ALERTS_POLYGON_LONLAT,
    WS_ALERTS_POLYGON_MAX_FIELDS

} WS_ALERTS_DB_POLYGON_ENUM;

// Enumeration specifying all the available fields
// in the ws_alerts locations table C?T?L?
typedef enum ws_alerts_locations_fields_enum
{
    WS_ALERTS_LOCATIONS_FIELD_LOCID = 0,
    WS_ALERTS_ALERT_LOCATIONS_FIELD_INFO,
    WS_ALERTS_ALERT_LOCATIONS_FIELD_EXREF,
    WS_ALERTS_ALERT_LOCATIONS_FIELD_EXTTYPE,
    WS_ALERTS_ALERT_LOCATIONS_FIELD_LONLAT,
    WS_ALERTS_ALERT_LOCATIONS_MAX_FIELDS
} WS_ALERTS_ALERT_LOCATIONS_FIELDS_ENUM;

// Enumeration specifying all the available fields
// in the LUT description table
typedef enum ws_alerts_lut_description_enum
{
    WS_ALERTS_LUT_DESC_FIELD_TABLE_NAME = 0,
    WS_ALERTS_LUT_DESC_FIELD_LABID,
    WS_ALERTS_LUT_DESC_FIELD_NAME,
    WS_ALERTS_LUT_DESC_FIELD_TYPE
} WS_ALERTS_ALERT_LUT_DESCRIPTION_ENUM;

// Structure which specifies all attributes
// of a single alert location row that is
// available within the wsalerts database
// These are the raw values that are retrieved from the database
typedef struct ws_alerts_locations_row_struct
{
    LOC_ID tLocId;           // LocID (Type + Tabindex + Lcode)
    BOOLEAN bIsPolygon;
    WS_ALERT_MSG_ID tMsgId;
    STRING_OBJECT hInfo;
    STRING_OBJECT hExtRef;
    STRING_OBJECT hExtType;
    size_t tNumVertices;
    N32 *pn32LonLat;
    BOOLEAN bResult;
} WS_ALERTS_LOCATIONS_ROW_STRUCT;

// Structure which specifies all attributes
// of a single alert message row that is
// available within the wsalerts database
// These are the raw values that are retrieved from the database
typedef struct ws_alerts_msg_row_struct
{
    WS_ALERT_MSG_ID tMsgId;
    UN8 un8StateId;
    SMS_LANGUAGE_ENUM eMsgLang;
    STRING_OBJECT hMsgText;
    TIME_T tEventStartTime;
    TIME_T tEventEndTime;
    size_t tPriority;
    BOOLEAN bPolygon;
    //Populated flag
    BOOLEAN bPopulated;
} WS_ALERTS_MSG_ROW_STRUCT;

// Structure which specifies all attributes
// of a single database version row that is
// available within the ws_alerts database
// These are the raw values that are retrieved from the database
typedef struct ws_alerts_version_row_struct
{
    UN8 type;
    UN32 un32Version;           // Database schema version
} WS_ALERTS_VERSION_ROW_STRUCT;

typedef struct update_version_row_struct
{
    N16 n16Version;
} UPDATE_VERSION_ROW_STRUCT;

typedef struct ws_alerts_msg_types_row_struct
{
    STRING_OBJECT hMsgType;
    STRING_OBJECT hText;
} WS_ALERTS_MSG_TYPES_ROW_STRUCT;

  /****************************/
 /** GLOBAL UNION TEMPLATES **/
/****************************/

// Union to group all the row definitions
// in a convenient manner
typedef union ws_alerts_db_row_union
{
    WS_ALERTS_VERSION_ROW_STRUCT sVersion;
    WS_ALERTS_LOCATIONS_ROW_STRUCT sAlertLocationRow;
    WS_ALERTS_MSG_TYPES_ROW_STRUCT sMsgTypesRow;
} WS_ALERTS_DB_ROW_UNION;

typedef struct ws_alerts_lut_desc_row
{
    STRING_OBJECT hTableName;
    UN8 un8LabelId;
    STRING_OBJECT hFieldName;
    STRING_OBJECT hFieldType;
} WS_ALERTS_LUT_DESC_ROW;

#endif /* WS_ALERTS_DB_CONSTANTS_H_ */
