/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:ALERT_MSG implementation for the
 *  Simple Module Services (SMS)
 *
 ******************************************************************************/

#include "sms_api.h"
#include "sms_obj.h"

#include "_ws_alert_msg_obj.h"
#include "ws_alert_msg_obj.h"
#include "ws_alerts_location_obj.h"

#include "sms_api_debug.h"

static const char *gpacThisFile = __FILE__;

/*****************************************************************************
 *                             PUBLIC FUNCTIONS
 *****************************************************************************/

/*****************************************************************************
 *
 *   hMsgText
 *
 *****************************************************************************/
static STRING_OBJECT hMsgText (
    WS_ALERT_MSG_OBJECT hAlertMsg
        )
{
    STRING_OBJECT hResult = STRING_INVALID_OBJECT;
    BOOLEAN bOwner = FALSE;

    // Verify ownership of this object
    bOwner =
        DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hAlertMsg);
    if (bOwner == TRUE)
    {
        WS_ALERT_MSG_OBJECT_STRUCT *psObj =
            (WS_ALERT_MSG_OBJECT_STRUCT*) hAlertMsg;

        hResult = psObj->hMsgText;
    }

    return hResult;
}

/*****************************************************************************
 *
 *   tStartTime
 *
 *****************************************************************************/
static TIME_T tStartTime (
    WS_ALERT_MSG_OBJECT hAlertMsg
        )
{
    TIME_T tResult = 0;
    BOOLEAN bOwner = FALSE;

    // Verify ownership of this object
    bOwner =
        DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hAlertMsg);
    if (bOwner == TRUE)
    {
        WS_ALERT_MSG_OBJECT_STRUCT *psObj =
            (WS_ALERT_MSG_OBJECT_STRUCT*) hAlertMsg;

        tResult = psObj->tEventStartTime;
    }

    return tResult;
}

/*****************************************************************************
 *
 *   tEndTime
 *
 *****************************************************************************/
static TIME_T tEndTime (
    WS_ALERT_MSG_OBJECT hAlertMsg
        )
{
    TIME_T tResult = 0;
    BOOLEAN bOwner = FALSE;

    // Verify ownership of this object
    bOwner =
        DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hAlertMsg);
    if (bOwner == TRUE)
    {
        WS_ALERT_MSG_OBJECT_STRUCT *psObj =
            (WS_ALERT_MSG_OBJECT_STRUCT*) hAlertMsg;

        tResult = psObj->tEventEndTime;
    }

    return tResult;
}

/*****************************************************************************
 *
 *   eMsgLang
 *
 *****************************************************************************/
static SMS_LANGUAGE_ENUM eMsgLang (
    WS_ALERT_MSG_OBJECT hAlertMsg
        )
{
    SMS_LANGUAGE_ENUM eResult = SMS_INVALID_LANGUAGE;
    BOOLEAN bOwner = FALSE;

    // Verify ownership of this object
    bOwner =
        DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hAlertMsg);
    if (bOwner == TRUE)
    {
        WS_ALERT_MSG_OBJECT_STRUCT *psObj =
            (WS_ALERT_MSG_OBJECT_STRUCT*) hAlertMsg;

        eResult = psObj->eMsgLang;
    }

    return eResult;
}

/*****************************************************************************
 *
 *   un16Priority
 *
 *****************************************************************************/
static UN16 un16Priority (
    WS_ALERT_MSG_OBJECT hAlertMsg
        )
{
    UN16 un16Result = 0;
    BOOLEAN bOwner = FALSE;

    // Verify ownership of this object
    bOwner =
        DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hAlertMsg);
    if (bOwner == TRUE)
    {
        WS_ALERT_MSG_OBJECT_STRUCT *psObj =
            (WS_ALERT_MSG_OBJECT_STRUCT*) hAlertMsg;

        un16Result = (UN16)psObj->tPriority;
    }

    return un16Result;
}

/*****************************************************************************
 *
 *   tID
 *
 *****************************************************************************/
static WS_ALERT_MSG_ID tID (
    WS_ALERT_MSG_OBJECT hAlertMsg
        )
{
    WS_ALERT_MSG_ID tMsgId = WS_ALERT_MSG_INVALID_ID;
    BOOLEAN bOwner = FALSE;

    // Verify ownership of this object
    bOwner =
        DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hAlertMsg);
    if (bOwner == TRUE)
    {
        WS_ALERT_MSG_OBJECT_STRUCT *psObj =
            (WS_ALERT_MSG_OBJECT_STRUCT*) hAlertMsg;

        tMsgId = psObj->tMsgId;
    }

    return tMsgId;
}

/*****************************************************************************
 *
 *   bIsMarineZone
 *
 *****************************************************************************/
static BOOLEAN bIsMarineZone (
    WS_ALERT_MSG_OBJECT hAlertMsg
        )
{
    BOOLEAN bResult = FALSE, bOwner;

    // Verify ownership of this object
    bOwner =
        DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hAlertMsg);
    if (bOwner == TRUE)
    {
        WS_ALERT_MSG_OBJECT_STRUCT *psObj =
            (WS_ALERT_MSG_OBJECT_STRUCT*) hAlertMsg;

        bResult = WS_ALERTS_IS_MARINE_ZONE(psObj->un8StateId);
    }

    return bResult;
}

/*****************************************************************************
 *
 *   hAlertText
 *
 *****************************************************************************/
static STRING_OBJECT hAlertText (
    WS_ALERT_MSG_OBJECT hAlertMsg,
    STRING_OBJECT hAlertType
        )
{
    STRING_OBJECT hResult = STRING_INVALID_OBJECT;
    BOOLEAN bOwner = FALSE;
    OSAL_LINKED_LIST_ENTRY hFirst, hEntry;
    WS_ALERTS_MSG_TYPES_ENTRY_STRUCT *psEntry = NULL;

    // Verify ownership of this object
    bOwner =
        DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hAlertMsg);
    if (bOwner == TRUE)
    {
        WS_ALERT_MSG_OBJECT_STRUCT *psObj =
            (WS_ALERT_MSG_OBJECT_STRUCT*) hAlertMsg;

        // Grab the first entry, ensure it's valid
        hFirst = hEntry = OSAL.hLinkedListFirst(
        psObj->hMsgTypesList, (void **)&psEntry);
        if ((hEntry == OSAL_INVALID_LINKED_LIST_ENTRY) ||
            (psEntry == NULL))
        {
            // List is empty
            return hResult;
        }

        while ((hEntry != OSAL_INVALID_LINKED_LIST_ENTRY) &&
            (psEntry != NULL))
        {
            if (STRING.n16CaseCompare(hAlertType, psEntry->hMsgType) == 0)
            {
                hResult = psEntry->hText;
                break;
            }

            psEntry = NULL;
            hEntry = OSAL.hLinkedListNext(hEntry, (void **)&psEntry);
            if (hEntry == hFirst)
            {
                // We've wrapped around
                hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
                psEntry = NULL;
            }
        }
    }

    return hResult;
}

/*****************************************************************************
 *
 *   eIterateLocations
 *
 *   This API is used to iterate locations in a valid WS_ALERTS_MSG_OBJECT.
 *
 *****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eIterateLocations (
    WS_ALERT_MSG_OBJECT hAlertMsg,
    WS_ALERT_MSG_LOCATIONS_ITERATOR_CALLBACK bCallback,
    void *pvCallbackArg
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode = OSAL_ERROR;
    BOOLEAN bOwner = FALSE;
    WS_ALERT_MSG_OBJECT_STRUCT *psObj = (WS_ALERT_MSG_OBJECT_STRUCT*) hAlertMsg;

    do
    {
        // Check inputs
        if ((hAlertMsg == WS_ALERT_MSG_INVALID_OBJECT) || (bCallback == NULL))
        {
            eReturnCode = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }

        bOwner = DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hAlertMsg);
        if (bOwner != TRUE)
        {
            eReturnCode = SMSAPI_RETURN_CODE_NOT_OWNER;
            break;
        }

        eOsalReturnCode = OSAL.eLinkedListIterate(
            psObj->hLocationsList,
            (OSAL_LL_ITERATOR_HANDLER)bCallback,
            pvCallbackArg);
        if ( (eOsalReturnCode != OSAL_SUCCESS) &&
             (eOsalReturnCode != OSAL_NO_OBJECTS) )
        {
            break;
        }

        eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eReturnCode;
}

/*****************************************************************************
 *
 *   bMsgTypesCallback
 *
 *****************************************************************************/
static BOOLEAN bMsgTypesCallback(
    WS_ALERTS_MSG_TYPES_ENTRY_STRUCT *psMsgTypesEntry,
    WS_ALERT_MSG_OBJECT_STRUCT *psObj
        )
{
    return psObj->bCallback(psMsgTypesEntry->hMsgType, psObj->pvCallbackArg);
}

/*****************************************************************************
 *
 *   eIterateAlertTypes
 *
 *   This API is used to iterate alert types in a valid WS_ALERTS_MSG_OBJECT.
 *
 *****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eIterateAlertTypes (
    WS_ALERT_MSG_OBJECT hAlertMsg,
    WS_ALERT_MSG_TYPES_ITERATOR_CALLBACK bCallback,
    void *pvCallbackArg
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode = OSAL_ERROR;
    BOOLEAN bOwner = FALSE;
    WS_ALERT_MSG_OBJECT_STRUCT *psObj = (WS_ALERT_MSG_OBJECT_STRUCT*) hAlertMsg;

    do
    {
        // Check inputs
        if ((hAlertMsg == WS_ALERT_MSG_INVALID_OBJECT) || (bCallback == NULL))
        {
            eReturnCode = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }

        bOwner = DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hAlertMsg);
        if (bOwner != TRUE)
        {
            eReturnCode = SMSAPI_RETURN_CODE_NOT_OWNER;
            break;
        }

        psObj->bCallback = bCallback;
        psObj->pvCallbackArg = pvCallbackArg;

        eOsalReturnCode = OSAL.eLinkedListIterate(
            psObj->hMsgTypesList,
            (OSAL_LL_ITERATOR_HANDLER)bMsgTypesCallback, psObj);
        if ( (eOsalReturnCode != OSAL_SUCCESS) &&
             (eOsalReturnCode != OSAL_NO_OBJECTS) )
        {
            break;
        }

        eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eReturnCode;
}

/*****************************************************************************
 *
 *   n32FPrintf
 *
 *****************************************************************************/
static N32 n32FPrintf (
    WS_ALERT_MSG_OBJECT hAlertMsg,
    FILE *psFile,
    SMSAPI_OUTPUT_OPTION_ENUM eOutputOption
        )
{
    N32 n32Return = EOF;
    BOOLEAN bOwner = FALSE;
    WS_ALERT_MSG_PRINT_ITERATOR_STRUCT sIterator;

    // Determine if the handle is valid
    bOwner = DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hAlertMsg);

    if ((bOwner == TRUE) && (psFile != NULL) &&
        ((eOutputOption == SMS_OUTPUT_OPTION_TERSE) ||
         (eOutputOption == SMS_OUTPUT_OPTION_VERBOSE) ||
         (eOutputOption == SMS_OUTPUT_OPTION_GROSS)))
    {
        WS_ALERT_MSG_OBJECT_STRUCT *psObj =
            (WS_ALERT_MSG_OBJECT_STRUCT*)hAlertMsg;
        char acBuffer[WS_ALERTS_MSG_SHORT_BUFFER_SIZE];

        n32Return = fprintf(psFile, "WEATHER/SECURITY ALERT:\n");
        if (eOutputOption != SMS_OUTPUT_OPTION_TERSE)
        {
            n32Return += fprintf(psFile, "\thAlertMsg = %p\n",
                psObj);
        }
        n32Return += fprintf(psFile, "-----------------------------------------\n");
        n32Return += fputs(STRING.pacCStr(psObj->hMsgText), psFile);
        n32Return += fprintf(psFile, "\n-----------------------------------------\n");

        if (eOutputOption != SMS_OUTPUT_OPTION_TERSE)
        {
            n32Return += fprintf(psFile, "\tMessage ID = %u\n",
                psObj->tMsgId);
            n32Return += fprintf(psFile, "\tState ID = %u\n",
                psObj->un8StateId);

            switch (psObj->eMsgLang)
            {
                case SMS_LANGUAGE_ENGLISH:
                {
                    n32Return += fprintf(psFile, "\tLanguage = English\n");
                }
                break;
                case SMS_LANGUAGE_SPANISH:
                {
                    n32Return += fprintf(psFile, "\tLanguage = Spanish\n");
                }
                break;
                case SMS_LANGUAGE_FRENCH:
                {
                    n32Return += fprintf(psFile, "\tLanguage = French\n");
                }
                break;
                case SMS_INVALID_LANGUAGE:
                default:
                {
                    n32Return += fprintf(psFile, "\tLanguage = %u (UNKNOWN)\n",
                        psObj->eMsgLang);
                }
                break;
            }

            if(psObj->hMsgTypesList != OSAL_INVALID_OBJECT_HDL)
            {
                // Configure iterator
                sIterator.psFile = psFile;
                sIterator.n32NumBytesWritten = 0;
                sIterator.eOutputOption = eOutputOption;

                n32Return += fprintf(psFile, "\tMessage type(s): ");
                OSAL.eLinkedListIterate(
                    psObj->hMsgTypesList,
                    (OSAL_LL_ITERATOR_HANDLER)bPrintTypeIterator,
                    &sIterator);

                if(sIterator.n32NumBytesWritten > 0)
                {
                    n32Return += sIterator.n32NumBytesWritten;
                }
                n32Return += fprintf(psFile, "\n");
            }

            n32Return += fprintf(psFile, "\tPriority = %u\n",
                psObj->tPriority);
            if(psObj->tEventStartTime > 0)
            {
                OSAL.ctime_r(&psObj->tEventStartTime, acBuffer);
            }
            else
            {
                // Buffer is definitely longer than 4 bytes,
                // so no worries while using strcpy()
                strcpy(acBuffer, "N/A");
            }
            n32Return += fprintf(psFile, "\tStart Time: %s\n",
                acBuffer);
            if(psObj->tEventEndTime > 0)
            {
                OSAL.ctime_r(&psObj->tEventEndTime, acBuffer);
            }
            else
            {
                // Buffer is definitely longer than 4 bytes,
                // so no worries while using strcpy()
                strcpy(acBuffer, "N/A");
            }
            n32Return += fprintf(psFile, "\tEnd Time: %s\n",
                acBuffer);

            if (eOutputOption == SMS_OUTPUT_OPTION_GROSS)
            {
                if(psObj->hLocationsList != OSAL_INVALID_OBJECT_HDL)
                {
                    // Configure iterator
                    sIterator.psFile = psFile;
                    sIterator.n32NumBytesWritten = 0;
                    sIterator.eOutputOption = eOutputOption;

                    n32Return += fprintf(psFile, "\tAreas:\n");
                    OSAL.eLinkedListIterate(
                        psObj->hLocationsList,
                        (OSAL_LL_ITERATOR_HANDLER)bPrintLocationIterator,
                        &sIterator);

                    if(sIterator.n32NumBytesWritten > 0)
                    {
                        n32Return += sIterator.n32NumBytesWritten;
                    }
                }
            }
        }
    }

    return n32Return;
}

/*****************************************************************************
 *                              FRIEND FUNCTIONS
 *****************************************************************************/
/*****************************************************************************
 *
 *   WS_ALERT_MSG_hCreate
 *
 *****************************************************************************/
WS_ALERT_MSG_OBJECT WS_ALERT_MSG_hCreate(
    SMS_OBJECT hParent,
    size_t tDSRLEntryDescSize,
    WS_ALERTS_MSG_ROW_STRUCT *psMsgRow,
    OSAL_OBJECT_HDL hLocList,
    OSAL_OBJECT_HDL hTypeList
        )
{
    WS_ALERT_MSG_OBJECT_STRUCT *psMsgObj =
        (WS_ALERT_MSG_OBJECT_STRUCT*)WS_ALERT_MSG_INVALID_OBJECT;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    BOOLEAN bOk = FALSE;
    UN32 un32Items = 0;

    do
    {
        // Check inputs
        if ( (hParent == SMS_INVALID_OBJECT) ||
             (hLocList == OSAL_INVALID_OBJECT_HDL) ||
             (hTypeList == OSAL_INVALID_OBJECT_HDL) )
        {
            break;
        }

        // Create an instance of the Alert Msg object
        psMsgObj = (WS_ALERT_MSG_OBJECT_STRUCT*)
            DSRL_ENTRY_hCreate(
                WS_ALERT_MSG_OBJECT_NAME,
                DSRL_ENTRY_TYPE_WS_ALERTS_MSG,
                sizeof(*psMsgObj),
                tDSRLEntryDescSize,
                hParent,
                FALSE );
        if (psMsgObj == (WS_ALERT_MSG_OBJECT_STRUCT*)DSRL_ENTRY_INVALID_OBJECT)
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERT_MSG_OBJECT_NAME": failed to create DSRL_ENTRY" );
            break;
        }

        psMsgObj->tMsgId = psMsgRow->tMsgId;
        psMsgObj->un8StateId = psMsgRow->un8StateId;
        psMsgObj->eMsgLang = psMsgRow->eMsgLang;

        psMsgObj->hMsgText = psMsgRow->hMsgText;
        psMsgRow->hMsgText = STRING_INVALID_OBJECT;
        if (psMsgObj->hMsgText == STRING_INVALID_OBJECT)
        {
            break;
        }

        psMsgObj->tEventStartTime = psMsgRow->tEventStartTime;
        psMsgObj->tEventEndTime = psMsgRow->tEventEndTime;
        psMsgObj->tPriority = psMsgRow->tPriority;

        psMsgObj->hMsgTypesList = hTypeList;

        eReturnCode = OSAL.eLinkedListItems(hLocList, &un32Items);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERT_MSG_OBJECT_NAME": failed to get number of"
                " items in list (%s)", OSAL.pacGetReturnCodeName(eReturnCode) );
            break;
        }

        psMsgObj->un8NumLocations = (UN8)un32Items;
        psMsgObj->hLocationsList = hLocList;

        bOk = TRUE;
    } while (FALSE);

    if ((bOk == FALSE) &&
        (psMsgObj != (WS_ALERT_MSG_OBJECT_STRUCT*)WS_ALERT_MSG_INVALID_OBJECT))
    {
        WS_ALERT_MSG_vDestroy((WS_ALERT_MSG_OBJECT)psMsgObj);
        psMsgObj = (WS_ALERT_MSG_OBJECT_STRUCT*)WS_ALERT_MSG_INVALID_OBJECT;
    }

    return (WS_ALERT_MSG_OBJECT)psMsgObj;
}

/*****************************************************************************
 *
 *   WS_ALERT_MSG_hCreateDummy
 *
 *****************************************************************************/
WS_ALERT_MSG_OBJECT WS_ALERT_MSG_hCreateDummy(
    SMS_OBJECT hParent
        )
{
    WS_ALERT_MSG_OBJECT_STRUCT *psResult = NULL;

    if (hParent != SMS_INVALID_OBJECT)
    {
        // Create an instance of the Alert Msg object
        psResult = (WS_ALERT_MSG_OBJECT_STRUCT *)
            DSRL_ENTRY_hCreate(
            WS_ALERT_MSG_OBJECT_NAME,
            DSRL_ENTRY_TYPE_WS_ALERTS_MSG,
            sizeof(*psResult),
            0,
            hParent,
            FALSE);
    }

    return (WS_ALERT_MSG_OBJECT)psResult;
}

/*****************************************************************************
 *
 *   WS_ALERT_MSG_bSetMsgID
 *
 *****************************************************************************/
BOOLEAN WS_ALERT_MSG_bSetMsgID(
    WS_ALERT_MSG_OBJECT hAlertMsg,
    WS_ALERT_MSG_ID tMsgID
        )
{
    BOOLEAN bOwner = FALSE;

    // Verify ownership of this object
    bOwner = DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hAlertMsg);
    if(bOwner == TRUE)
    {
        WS_ALERT_MSG_OBJECT_STRUCT *psObj =
            (WS_ALERT_MSG_OBJECT_STRUCT *)hAlertMsg;

        psObj->tMsgId = tMsgID;
    }

    return bOwner;
}

/*****************************************************************************
 *
 *   WS_ALERT_MSG_vDestroy
 *
 *****************************************************************************/
void WS_ALERT_MSG_vDestroy(
   WS_ALERT_MSG_OBJECT hAlertMsg
        )
{
    BOOLEAN bOwner = FALSE;

    printf(WS_ALERT_MSG_OBJECT_NAME": attempting to destroy object 0x%p\n",
        hAlertMsg
            );

    // Verify ownership of this object
    bOwner = DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hAlertMsg);
    if(bOwner == TRUE)
    {
        WS_ALERT_MSG_OBJECT_STRUCT *psObj =
            (WS_ALERT_MSG_OBJECT_STRUCT*)hAlertMsg;
        vDestroyObject(psObj);
    }

    return;
}

/*****************************************************************************
*
*   WS_ALERTS_MSG_tMsgID
*
*****************************************************************************/
WS_ALERT_MSG_ID WS_ALERTS_MSG_tMsgID (
    WS_ALERT_MSG_OBJECT hMsg
        )
{
    BOOLEAN bOwner = FALSE;
    WS_ALERT_MSG_ID tMsgId = WS_ALERT_MSG_INVALID_ID;

    bOwner = DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hMsg);
    if(bOwner == TRUE)
    {
        WS_ALERT_MSG_OBJECT_STRUCT *psObj =
            (WS_ALERT_MSG_OBJECT_STRUCT*)hMsg;

        tMsgId = psObj->tMsgId;
    }

    return tMsgId;
}

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/
/*****************************************************************************
*
*   vDestroyMsgTypesEntry
*
*****************************************************************************/
static void vDestroyMsgTypesEntry(
    WS_ALERTS_MSG_TYPES_ENTRY_STRUCT *psMsgTypesEntry
        )
{
    if (psMsgTypesEntry != NULL)
    {
        STRING.vDestroy(psMsgTypesEntry->hText);
        STRING.vDestroy(psMsgTypesEntry->hMsgType);
        OSAL.vLinkedListMemoryFree(psMsgTypesEntry);
    }
}

/*****************************************************************************
*
*   vDestroyObject
*
*****************************************************************************/
static void vDestroyObject(
    WS_ALERT_MSG_OBJECT_STRUCT *psObj
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

    if (psObj != NULL)
    {
        if (psObj->hMsgText != STRING_INVALID_OBJECT)
        {
            STRING_vDestroy(psObj->hMsgText);
            psObj->hMsgText = STRING_INVALID_OBJECT;
        }

        if( psObj->hMsgTypesList != OSAL_INVALID_OBJECT_HDL )
        {
            eReturnCode = OSAL.eLinkedListRemoveAll(
                psObj->hMsgTypesList,
                (OSAL_LL_RELEASE_HANDLER)vDestroyMsgTypesEntry );

            if( eReturnCode != OSAL_SUCCESS )
            {
                SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                    WS_ALERT_MSG_OBJECT_NAME": cannot cleanup linked list (%s)",
                    OSAL.pacGetReturnCodeName( eReturnCode ) );
            }

            eReturnCode = OSAL.eLinkedListDelete( psObj->hMsgTypesList );

            if( eReturnCode != OSAL_SUCCESS )
            {
                SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                    WS_ALERT_MSG_OBJECT_NAME": cannot delete linked list (%s)",
                    OSAL.pacGetReturnCodeName( eReturnCode ) );
            }

            psObj->hMsgTypesList = OSAL_INVALID_OBJECT_HDL;
        }

        if( psObj->hLocationsList != OSAL_INVALID_OBJECT_HDL )
        {
            eReturnCode = OSAL.eLinkedListRemoveAll(
                psObj->hLocationsList,
                (OSAL_LL_RELEASE_HANDLER)NULL);

            if( eReturnCode != OSAL_SUCCESS )
            {
                SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                    WS_ALERT_MSG_OBJECT_NAME": cannot cleanup linked list (%s)",
                    OSAL.pacGetReturnCodeName( eReturnCode ) );
            }

            eReturnCode = OSAL.eLinkedListDelete( psObj->hLocationsList );

            if( eReturnCode != OSAL_SUCCESS )
            {
                SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                    WS_ALERT_MSG_OBJECT_NAME": cannot delete linked list (%s)",
                    OSAL.pacGetReturnCodeName( eReturnCode ) );
            }

            psObj->hLocationsList = OSAL_INVALID_OBJECT_HDL;
        }

        // Free object instance itself
        DSRL_ENTRY_vDestroy((DSRL_ENTRY_OBJECT)psObj);
    }

    return;
}

/*****************************************************************************
*
*   bPrintTypeIterator
*
*****************************************************************************/
static BOOLEAN bPrintTypeIterator (
    void *pvData,
    WS_ALERT_MSG_PRINT_ITERATOR_STRUCT *psIterator
        )
{
    if( (pvData != NULL) &&
        (psIterator != NULL) &&
        (psIterator->psFile != NULL) )
    {
        const char *pacMsgType, *pacMsgText;
        N32 n32Temp = 0;
        WS_ALERTS_MSG_TYPES_ENTRY_STRUCT *psMsgTypesEntry =
            (WS_ALERTS_MSG_TYPES_ENTRY_STRUCT *)pvData;

        if ( (psIterator->eOutputOption == SMS_OUTPUT_OPTION_TERSE) ||
             (psIterator->eOutputOption == SMS_OUTPUT_OPTION_VERBOSE) )
        {
            pacMsgType = STRING.pacCStr(psMsgTypesEntry->hMsgType);
            pacMsgText = STRING.pacCStr(psMsgTypesEntry->hText);
            n32Temp = fprintf(psIterator->psFile, "Type:%s Text:%s",
                pacMsgType, pacMsgText);
        }
        else if ( psIterator->eOutputOption == SMS_OUTPUT_OPTION_GROSS )
        {
            n32Temp = STRING.n32FPrintf(
                psMsgTypesEntry->hMsgType, psIterator->psFile);

            n32Temp += STRING.n32FPrintf(
                psMsgTypesEntry->hText, psIterator->psFile);
        }

        if (n32Temp > 0)
        {
            psIterator->n32NumBytesWritten += n32Temp;
        }
    }

    return TRUE;
}

/*****************************************************************************
*
*   bPrintLocationIterator
*
*****************************************************************************/
static BOOLEAN bPrintLocationIterator (
    void *pvData,
    WS_ALERT_MSG_PRINT_ITERATOR_STRUCT *psIterator
        )
{
    if( (pvData != NULL) &&
        (psIterator != NULL) &&
        (psIterator->psFile != NULL) )
    {
        N32 n32Temp;
        WS_ALERTS_LOCATION_OBJECT hWSALertsLocation =
            (WS_ALERTS_LOCATION_OBJECT)pvData;

        n32Temp = WS_ALERTS_LOCATION.n32FPrintf(
            hWSALertsLocation,
            psIterator->psFile,
            psIterator->eOutputOption);

        if (n32Temp > 0)
        {
            psIterator->n32NumBytesWritten += n32Temp;
        }
    }

    return TRUE;
}
