/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:WEATHER_MSG implementation for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/
#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "sms_obj.h"
#include "_weather_msg_obj.h"
#include "weather_msg_obj.h"
#include "forecast_obj.h"
#include "dsrl_entry_obj.h"
#include "ski_conditions_obj.h"
#include "string_obj.h"

#include "sms_api_debug.h"

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   eGetForecast
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eGetForecast (
    WEATHER_MSG_OBJECT hWeatherMsg,
    WEATHER_FORECAST_TYPE_ENUM eForecastType,
    FORECAST_OBJECT *phForecast
        )
{
    BOOLEAN bOwner;
    SMSAPI_RETURN_CODE_ENUM eResult = SMSAPI_RETURN_CODE_ERROR;
    WEATHER_MSG_OBJECT_STRUCT *psObj = (WEATHER_MSG_OBJECT_STRUCT *)hWeatherMsg;

    do
    {
        if ((phForecast == NULL) ||
            (eForecastType >= WEATHER_FORECAST_TYPE_MAX))
        {
            eResult = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }
        bOwner = DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hWeatherMsg);
        if (bOwner == FALSE)
        {
            eResult = SMSAPI_RETURN_CODE_NOT_OWNER;
            break;
        }
        if (psObj->eType == WEATHER_MSG_TYPE_FORECAST)
        {
            if(psObj->uData.ahForecasts[eForecastType] != FORECAST_INVALID_OBJECT)
            {
                *phForecast = psObj->uData.ahForecasts[eForecastType];
                eResult = SMSAPI_RETURN_CODE_SUCCESS;
            }
            else
            {
                eResult = SMSAPI_RETURN_CODE_NOT_FOUND;
            }
        }
    } while (FALSE);

    return eResult;
}


/*****************************************************************************
*
*   eGetSkiConditions
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eGetSkiConditions (
    WEATHER_MSG_OBJECT hWeatherMsg,
    SKI_CONDITIONS_OBJECT *phSkiCond
        )
{
    BOOLEAN bOwner;
    SMSAPI_RETURN_CODE_ENUM eResult = SMSAPI_RETURN_CODE_ERROR;
    WEATHER_MSG_OBJECT_STRUCT *psObj = (WEATHER_MSG_OBJECT_STRUCT *)hWeatherMsg;

    do
    {
        if (phSkiCond == NULL)
        {
            eResult = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }
        bOwner = DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hWeatherMsg);
        if (bOwner == FALSE)
        {
            eResult = SMSAPI_RETURN_CODE_NOT_OWNER;
            break;
        }
        if (psObj->eType == WEATHER_MSG_TYPE_SKI_REPORT)
        {
            if(psObj->uData.hSkiCond != SKI_CONDITIONS_INVALID_OBJECT)
            {
                *phSkiCond = psObj->uData.hSkiCond;
                eResult = SMSAPI_RETURN_CODE_SUCCESS;
            }
            else
            {
                eResult = SMSAPI_RETURN_CODE_NOT_FOUND;
            }
        }
    } while (FALSE);

    return eResult;
}


/*****************************************************************************
*
*   hLocation
*
*****************************************************************************/
static LOCATION_OBJECT hLocation (
    WEATHER_MSG_OBJECT hWeatherMsg
        )
{
    BOOLEAN bOwner;
    LOCATION_OBJECT hResult = LOCATION_INVALID_OBJECT;

    bOwner = DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hWeatherMsg);
    if (bOwner == TRUE)
    {
        WEATHER_MSG_OBJECT_STRUCT *psObj = (WEATHER_MSG_OBJECT_STRUCT *)hWeatherMsg;
        hResult = psObj->hLocation;
    }

    return hResult;
}


/*****************************************************************************
*
*   hICAO
*
*****************************************************************************/
static STRING_OBJECT hICAO (
    WEATHER_MSG_OBJECT hWeatherMsg
        )
{
    BOOLEAN bOwner;
    STRING_OBJECT hResult = STRING_INVALID_OBJECT;

    bOwner = DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hWeatherMsg);
    if (bOwner == TRUE)
    {
        WEATHER_MSG_OBJECT_STRUCT *psObj = (WEATHER_MSG_OBJECT_STRUCT *)hWeatherMsg;
        hResult = psObj->hICAO;
    }

    return hResult;
}


/*****************************************************************************
*
*   eType
*
*****************************************************************************/
static WEATHER_MSG_OBJECT_TYPE_ENUM eType (
    WEATHER_MSG_OBJECT hWeatherMsg
        )
{
    BOOLEAN bOwner;
    WEATHER_MSG_OBJECT_TYPE_ENUM eMsgType = WEATHER_MSG_TYPE_INVALID;

    bOwner = DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hWeatherMsg);
    if (bOwner == TRUE)
    {
        WEATHER_MSG_OBJECT_STRUCT *psObj =
            (WEATHER_MSG_OBJECT_STRUCT *)hWeatherMsg;

        // Extract the type for the caller
        eMsgType = psObj->eType;
    }

    return eMsgType;
}


/*****************************************************************************
*
*   n32FPrintf
*
*****************************************************************************/
static N32 n32FPrintf (
    WEATHER_MSG_OBJECT hWeatherMsg,
    FILE *psFile,
    SMSAPI_OUTPUT_OPTION_ENUM eOutputOption
        )
{
    N32 n32Return = 0, n32Temp;
    BOOLEAN bOwner;
    WEATHER_MSG_OBJECT_STRUCT *psObj =
        (WEATHER_MSG_OBJECT_STRUCT *)hWeatherMsg;
    STRING_OBJECT hString;
    LOCID_OBJECT hLocID;
    LOC_ID tThisID;

    // Determine if the handle is valid
    bOwner = DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hWeatherMsg);
    if ((bOwner == FALSE) || (psFile == NULL))
    {
        return EOF;
    }

    // Print Weather Msg information header
    n32Return += fprintf(psFile, "Weather Report (ICAO Name: ");
    n32Temp = STRING.n32FWrite(psObj->hICAO, psFile);
    if (n32Temp > 0)
    {
        n32Return += n32Temp;
    }
    hLocID = LOCATION.hLocID(psObj->hLocation);
    tThisID = LOCID.tID(hLocID);
    n32Return += fprintf(psFile, ")\n\t(LOCID: %d)",tThisID);
    n32Return += fprintf(psFile, "\n\tType: %s",
        (psObj->eType == WEATHER_MSG_TYPE_FORECAST)?"Forecast":"Ski Report");
    n32Return += fprintf(psFile, "\n\tDescription: ");
    hString = LOCATION.hDescription(psObj->hLocation);
    n32Temp = STRING.n32FWrite(hString, psFile);
    if (n32Temp > 0)
    {
        n32Return += n32Temp;
    }
    n32Return += fprintf(psFile, "\n\tState: ");
    hString = LOCATION.hState(psObj->hLocation);
    n32Temp = STRING.n32FWrite(hString, psFile);
    if (n32Temp > 0)
    {
        n32Return += n32Temp;
    }
    n32Return += fprintf(psFile, "\n");

    if (psObj->eType == WEATHER_MSG_TYPE_FORECAST)
    {
        WEATHER_FORECAST_TYPE_ENUM eIndex;
        for(eIndex = WEATHER_FORECAST_TYPE_CURRENT;
            eIndex < WEATHER_FORECAST_TYPE_MAX;
            ++eIndex)
        {
            if (psObj->uData.ahForecasts[eIndex] != FORECAST_INVALID_OBJECT)
            {
                // Since the FORECAST object has no idea about its type let's
                // print out it here before calling FORECAST print function.
                n32Temp = fprintf(psFile, "\nType: %s\n",
                                  pacGetForecastTypeName(eIndex));
                if (n32Temp > 0)
                {
                    n32Return += n32Temp;
                }
                n32Temp = FORECAST.n32FPrintf(psObj->uData.ahForecasts[eIndex],
                                              psFile, eOutputOption);
                if (n32Temp > 0)
                {
                    n32Return += n32Temp;
                }
            }
        }
    }
    else if (psObj->eType == WEATHER_MSG_TYPE_SKI_REPORT)
    {
        if (psObj->uData.hSkiCond != SKI_CONDITIONS_INVALID_OBJECT)
        {
            n32Temp = SKI_CONDITIONS.n32FPrintf(psObj->uData.hSkiCond, psFile, eOutputOption);
            if (n32Temp > 0)
            {
                n32Return += n32Temp;
            }
        }
    }

    return n32Return;
}


/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   WEATHER_MSG_hCreate
*
*****************************************************************************/
WEATHER_MSG_OBJECT WEATHER_MSG_hCreate(
    SMS_OBJECT hParent,
    size_t tDSRLEntryDescSize,
    LOCATION_OBJECT hLocation,
    STRING_OBJECT hICAO,
    WEATHER_MSG_OBJECT_TYPE_ENUM eType
            )
{
    WEATHER_MSG_OBJECT_STRUCT *psObj =
        (WEATHER_MSG_OBJECT_STRUCT *)WEATHER_MSG_INVALID_OBJECT;
    UN8 i = 0;

    // Create an instance of the WEATHER_MSG object
    psObj = (WEATHER_MSG_OBJECT_STRUCT *)
        DSRL_ENTRY_hCreate(
            WEATHER_MSG_OBJECT_NAME,
            DSRL_ENTRY_TYPE_WEATHER_MSG,
            sizeof(WEATHER_MSG_OBJECT_STRUCT),
            tDSRLEntryDescSize,
            hParent,
            TRUE);

    if(psObj == NULL)
    {
        // Error!
        return WEATHER_MSG_INVALID_OBJECT;
    }

    psObj->eType = eType;
    psObj->hLocation = LOCATION.hDuplicate(hLocation);
    psObj->hICAO = STRING.hDuplicate(hICAO);

    if (eType == WEATHER_MSG_TYPE_FORECAST)
    {
        // Initialize object per inputs
        for (i = 0; i < WEATHER_FORECAST_TYPE_MAX; i++)
        {
            psObj->uData.ahForecasts[i] = FORECAST_INVALID_OBJECT;
        }
    }
    else if (eType == WEATHER_MSG_TYPE_SKI_REPORT)
    {
        psObj->uData.hSkiCond = SKI_CONDITIONS_INVALID_OBJECT;
    }

    return (WEATHER_MSG_OBJECT)psObj;
}

/*****************************************************************************
*
*   WEATHER_MSG_hCreateDummy
*
*****************************************************************************/
WEATHER_MSG_OBJECT WEATHER_MSG_hCreateDummy(
    SMS_OBJECT hParent,
    LOCATION_OBJECT hLocation
        )
{
    WEATHER_MSG_OBJECT_STRUCT *psObj =
        (WEATHER_MSG_OBJECT_STRUCT *)WEATHER_MSG_INVALID_OBJECT;

    // Create an instance of the WEATHER_MSG object
    psObj = (WEATHER_MSG_OBJECT_STRUCT *)
        DSRL_ENTRY_hCreate(
        WEATHER_MSG_OBJECT_NAME,
        DSRL_ENTRY_TYPE_WEATHER_MSG,
        sizeof(WEATHER_MSG_OBJECT_STRUCT),
        0,
        hParent,
        TRUE);

    if(psObj == NULL)
    {
        // Error!
        return WEATHER_MSG_INVALID_OBJECT;
    }

    psObj->hLocation = hLocation;

    return (WEATHER_MSG_OBJECT)psObj;
}

/*****************************************************************************
*
*   WEATHER_MSG_vDestroy
*
*****************************************************************************/
void WEATHER_MSG_vDestroy (
    WEATHER_MSG_OBJECT hWeatherMsg
        )
{
    BOOLEAN bOwner;

    bOwner = DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hWeatherMsg);
    if(bOwner == TRUE)
    {
        WEATHER_MSG_OBJECT_STRUCT *psObj =
            (WEATHER_MSG_OBJECT_STRUCT *)hWeatherMsg;
        vDestroyObject(psObj);
    }

    return;
}

/*****************************************************************************
*
*   WEATHER_MSG_bSetForecast
*
*****************************************************************************/
BOOLEAN WEATHER_MSG_bSetForecast (
    WEATHER_MSG_OBJECT hWeatherMsg,
    WEATHER_FORECAST_TYPE_ENUM eForecastType,
    FORECAST_OBJECT hForecast
        )
{
    BOOLEAN bResult = FALSE, bOwner;

    bOwner = DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hWeatherMsg);
    if(bOwner == TRUE)
    {
        WEATHER_MSG_OBJECT_STRUCT *psObj = (WEATHER_MSG_OBJECT_STRUCT *)hWeatherMsg;
        if (psObj->eType == WEATHER_MSG_TYPE_FORECAST)
        {
            if (psObj->uData.ahForecasts[eForecastType] != FORECAST_INVALID_OBJECT)
            {
                FORECAST_vDestroy(psObj->uData.ahForecasts[eForecastType]);
            }

            psObj->uData.ahForecasts[eForecastType] = hForecast;

            bResult = TRUE;
        }
    }

    return bResult;
}


/*****************************************************************************
*
*   WEATHER_MSG_bSetSkiCond
*
*****************************************************************************/
BOOLEAN WEATHER_MSG_bSetSkiCond (
    WEATHER_MSG_OBJECT hWeatherMsg,
    SKI_CONDITIONS_OBJECT hSkiCond
        )
{
    BOOLEAN bResult = FALSE, bOwner;

    bOwner = DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hWeatherMsg);
    if(bOwner == TRUE)
    {
        WEATHER_MSG_OBJECT_STRUCT *psObj = (WEATHER_MSG_OBJECT_STRUCT *)hWeatherMsg;
        if (psObj->eType == WEATHER_MSG_TYPE_SKI_REPORT)
        {
            if (psObj->uData.hSkiCond != SKI_CONDITIONS_INVALID_OBJECT)
            {
                SKI_CONDITIONS_vDestroy(psObj->uData.hSkiCond);
            }

            psObj->uData.hSkiCond = hSkiCond;
            bResult = TRUE;
        }
    }

    return bResult;
}


/*****************************************************************************
*
*   WEATHER_MSG_bSetLocation
*
*****************************************************************************/
BOOLEAN WEATHER_MSG_bSetLocation (
    WEATHER_MSG_OBJECT hWeatherMsg,
    LOCATION_OBJECT hLocation,
    STRING_OBJECT hICAO
        )
{
    BOOLEAN bOwner = FALSE;
    BOOLEAN bResult = FALSE;

    bOwner = DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hWeatherMsg);
    if(bOwner == TRUE)
    {
        WEATHER_MSG_OBJECT_STRUCT *psObj = (WEATHER_MSG_OBJECT_STRUCT *)hWeatherMsg;
        if (psObj->hLocation != LOCATION_INVALID_OBJECT)
        {
            LOCATION.vDestroy(psObj->hLocation);
        }
        psObj->hLocation = LOCATION.hDuplicate(hLocation);

        if (psObj->hICAO != STRING_INVALID_OBJECT)
        {
            STRING_vDestroy(psObj->hICAO);
        }
        psObj->hICAO = STRING.hDuplicate(hICAO);

        bResult = TRUE;
    }

    return bResult;
}


/*****************************************************************************
*
*   WEATHER_MSG_tGetLOCID
*
*****************************************************************************/
LOC_ID WEATHER_MSG_tGetLOCID(
    WEATHER_MSG_OBJECT hWeatherMsg
        )
{
    LOC_ID tResult = LOC_INVALID_ID;

    do
    {
        WEATHER_MSG_OBJECT_STRUCT *psObj =
            (WEATHER_MSG_OBJECT_STRUCT*)hWeatherMsg;
        BOOLEAN bOwner = FALSE;
        LOCID_OBJECT hLOCID = LOCID_INVALID_OBJECT;

        if (hWeatherMsg == WEATHER_MSG_INVALID_OBJECT)
        {
            break;
        }

        bOwner = DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hWeatherMsg);
        if(bOwner == FALSE)
        {
            break;
        }

        if (psObj->hLocation == LOCATION_INVALID_OBJECT)
        {
            break;
        }

        hLOCID = LOCATION.hLocID(psObj->hLocation);
        if (hLOCID == LOCID_INVALID_OBJECT)
        {
            break;
        }

        tResult = LOCID.tID(hLOCID);
    } while (FALSE);

    return tResult;
}

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   vDestroy
*
*****************************************************************************/
static void vDestroyObject(WEATHER_MSG_OBJECT_STRUCT *psObj)
{
    UN8 i = 0;

    if (psObj->hLocation != LOCATION_INVALID_OBJECT)
    {
        LOCATION.vDestroy(psObj->hLocation);
        psObj->hLocation = LOCATION_INVALID_OBJECT;
    }

    if (psObj->hICAO != STRING_INVALID_OBJECT)
    {
        STRING_vDestroy(psObj->hICAO);
        psObj->hICAO = STRING_INVALID_OBJECT;
    }

    if (psObj->eType == WEATHER_MSG_TYPE_FORECAST)
    {
        for (i = 0; i < WEATHER_FORECAST_TYPE_MAX; i++)
        {
            if (psObj->uData.ahForecasts[i] != FORECAST_INVALID_OBJECT)
            {
                FORECAST_vDestroy(psObj->uData.ahForecasts[i]);
                psObj->uData.ahForecasts[i] = FORECAST_INVALID_OBJECT;
            }
        }
    }
    else if (psObj->eType == WEATHER_MSG_TYPE_SKI_REPORT)
    {
        if (psObj->uData.hSkiCond != SKI_CONDITIONS_INVALID_OBJECT)
        {
            SKI_CONDITIONS_vDestroy(psObj->uData.hSkiCond);
            psObj->uData.hSkiCond = SKI_CONDITIONS_INVALID_OBJECT;
        }
    }

    // Free object instance itself
    DSRL_ENTRY_vDestroy((DSRL_ENTRY_OBJECT)psObj);

    return;
}

/*****************************************************************************
*
*   pacGetForecastTypeName
*
*****************************************************************************/
static const char *pacGetForecastTypeName(
    WEATHER_FORECAST_TYPE_ENUM eType
        )
{
    const char *pacResult = "UNKNOWN";
    switch (eType)
    {
        case WEATHER_FORECAST_TYPE_CURRENT:
        {
            pacResult = MACRO_TO_STRING(WEATHER_FORECAST_TYPE_CURRENT);
        }
        break;

        case WEATHER_FORECAST_TYPE_0_3_HOUR:
        {
            pacResult = MACRO_TO_STRING(WEATHER_FORECAST_TYPE_0_3_HOUR);
        }
        break;

        case WEATHER_FORECAST_TYPE_3_6_HOUR:
        {
            pacResult = MACRO_TO_STRING(WEATHER_FORECAST_TYPE_3_6_HOUR);
        }
        break;

        case WEATHER_FORECAST_TYPE_SUNDAY:
        {
            pacResult = MACRO_TO_STRING(WEATHER_FORECAST_TYPE_SUNDAY);
        }
        break;

        case WEATHER_FORECAST_TYPE_MONDAY:
        {
            pacResult = MACRO_TO_STRING(WEATHER_FORECAST_TYPE_MONDAY);
        }
        break;

        case WEATHER_FORECAST_TYPE_TUESDAY:
        {
            pacResult = MACRO_TO_STRING(WEATHER_FORECAST_TYPE_TUESDAY);
        }
        break;

        case WEATHER_FORECAST_TYPE_WEDNESDAY:
        {
            pacResult = MACRO_TO_STRING(WEATHER_FORECAST_TYPE_WEDNESDAY);
        }
        break;

        case WEATHER_FORECAST_TYPE_THURSDAY:
        {
            pacResult = MACRO_TO_STRING(WEATHER_FORECAST_TYPE_THURSDAY);
        }
        break;

        case WEATHER_FORECAST_TYPE_FRIDAY:
        {
            pacResult = MACRO_TO_STRING(WEATHER_FORECAST_TYPE_FRIDAY);
        }
        break;

        case WEATHER_FORECAST_TYPE_SATURDAY:
        {
            pacResult = MACRO_TO_STRING(WEATHER_FORECAST_TYPE_SATURDAY);
        }
        break;

        case WEATHER_FORECAST_TYPE_MAX:
        {
            pacResult = MACRO_TO_STRING(WEATHER_FORECAST_TYPE_SATURDAY);
        }
        break;
    }

    return pacResult;
}
