/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 * This is the Tabular Weather Interface header file.
 *
 ******************************************************************************/

#ifndef WEATHER_INTERFACE_H_
#define WEATHER_INTERFACE_H_

#include "standard.h"
#include "osal.h"
#include "sms_api.h"
#include "weather_db_constants.h"
#include "sql_interface_obj.h"
#include "location_obj.h"

  /**********************/
 /** GLOBAL CONSTANTS **/
/**********************/

//Timeout value for weather message/hash item
#define WEATHER_MSG_TIMEOUT (60 * 30) // 30 minutes in seconds

//Timer period to check if item is expired
#define WEATHER_SERVICE_TIMER_PERIOD (60) // One minute

  /*********************/
 /** GLOBAL TYPEDEFS **/
/*********************/

// Interface handle
typedef void * WEATHER_INTERFACE_OBJECT;
#define WEATHER_INTERFACE_INVALID_OBJECT \
            (WEATHER_INTERFACE_OBJECT)NULL

typedef enum weather_flag_enum {
    WEATHER_FLAG_FALSE = FALSE,
    WEATHER_FLAG_TRUE = TRUE,
    WEATHER_FLAG_NOT_AVAILABLE
} WEATHER_FLAG_ENUM;

//definitions for unparsed data types
typedef UN8 WIND_UNPARSED_DATA;
typedef UN8 WEATHER_EVENT_UNPARSED_DATA;
typedef UN8 UV_INFO_UNPARSED_DATA;
typedef UN8 NEW_SNOW_UNPARSED_DATA;
typedef UN8 TRAILS_UNPARSED_DATA;
typedef UN8 PRECIP_PROB_UNPARSED_DATA;
typedef UN8 PRECIP_UNPARSED_DATA;
typedef UN8 HUMIDITY_UNPARSED_DATA;
typedef UN8 POLLEN_UNPARSED_DATA;

typedef void * WEATHER_HASH_OBJECT;
#define WEATHER_HASH_INVALID_OBJECT ((WEATHER_HASH_OBJECT)0)

  /*****************************/
 /** GLOBAL STRUCT TEMPLATES **/
/*****************************/

typedef struct wind_data_struct {
    UN8 un8Speed;
    UN16 un16Direction;
} WIND_DATA_STRUCT;

typedef struct weather_event_data_struct {
    UN8 un8EventCode;
    STRING_OBJECT hDescription;
    UN8 un8IconCode;
} WEATHER_EVENT_DATA_STRUCT;

typedef struct uv_info_data_struct {
    UN8 un8Low;
    UN8 un8High;
    UV_RISK_ENUM eRisk;
} UV_INFO_DATA_STRUCT;

typedef struct new_snow_data_struct {
    UN8 un8Low;
    UN8 un8High;
} NEW_SNOW_DATA_STRUCT;

typedef struct trails_data_struct {
    UN8 un8Low;
    UN8 un8High;
} TRAILS_DATA_STRUCT;

typedef struct precip_data_struct {
    PRECIPITATION_TYPE_ENUM eType;
    OSAL_FIXED_OBJECT hLow;
    OSAL_FIXED_OBJECT hHigh;
} PRECIP_DATA_STRUCT;

typedef struct humidity_data_struct {
    UN8 un8Low;
    UN8 un8High;
} HUMIDITY_DATA_STRUCT;

typedef struct pollen_data_struct {
    AIR_POLLEN_LEVEL_ENUM eLevel;
    UN8 un8Value;
} POLLEN_DATA_STRUCT;

typedef struct forecast_data_struct {
    WEATHER_EVENT_UNPARSED_DATA tEvent;
    N16 n16Temp;
    N16 n16Tmax;
    N16 n16Tmin;
    UN8 un8Pop;
    PRECIP_UNPARSED_DATA tPrecip;
    WIND_UNPARSED_DATA tData;
    HUMIDITY_UNPARSED_DATA tHumid;
    FORECAST_CLOUD_COVER_ENUM eCloudCover;
    UV_INFO_UNPARSED_DATA tUV;
    AIR_QUALITY_ENUM eAirQuality;
    POLLEN_UNPARSED_DATA tPollen;
    UN16 un16Flags;
    WEATHER_HASH_OBJECT hHashEntry;
} FORECAST_DATA_STRUCT;

typedef struct ski_data_struct
{
    WEATHER_FLAG_ENUM eOpStat;
    WEATHER_EVENT_UNPARSED_DATA tEvent;
    N16 n16Temp;
    SKI_WIND_CONDITION_ENUM eWind;
    SKI_SNOW_CONDITIONS_ENUM eSnow;
    UN8 un8BMin;
    UN8 un8BMax;
    NEW_SNOW_UNPARSED_DATA tNew;
    UN8 un8Lift;
    TRAILS_UNPARSED_DATA tTrail;
    WEATHER_FLAG_ENUM eMake;
    WEATHER_FLAG_ENUM eGroom;
    WEATHER_FLAG_ENUM eNight;
    WEATHER_FLAG_ENUM eBoard;
    UN16 un16Flags;
    WEATHER_HASH_OBJECT hHashEntry;
} SKI_DATA_STRUCT;

typedef enum forecast_valid_fields_enum
{
    FORECAST_AVERAGE_TEMP_FLAG = 0,
    FORECAST_HIGH_TEMP_FLAG,
    FORECAST_LOW_TEMP_FLAG,
    FORECAST_PRECIP_CHANCE_FLAG,
    FORECAST_PRECIP_AMOUNT_FLAG,
    FORECAST_WIND_DIRECTION_FLAG,
    FORECAST_WIND_SPEED_FLAG,
    FORECAST_HUMIDITY_FLAG,
    FORECAST_CLOUD_COVER_FLAG,
    FORECAST_UV_INDEX_FLAG,
    FORECAST_AIR_QUALITY_FLAG,
    FORECAST_POLLEN_COUNT_FLAG
} FORECAST_VALID_FIELDS_ENUM;

typedef enum ski_flags_enum {
    SKI_EVENT_CODE_FLAG,
    SKI_CURRENT_TEMP_FLAG,
    SKI_WIND_COND_FLAG,
    SKI_SNOW_COND_FLAG,
    SKI_MIN_BASE_DEPTH_FLAG,
    SKI_MAX_BASE_DEPTH_FLAG,
    SKI_NEW_SNOW_FLAG,
    SKI_NUM_OF_LIFTS_FLAG,
    SKI_NUM_OF_TRAILS_FLAG
} SKI_FLAGS_ENUM;

// The weather interface APIs
typedef struct weather_plugin_interface_struct
{
    const DSI tDSI;

    const size_t tMaxVersionBitlen;

    size_t (*tMinimumOTABufferByteSize) (
        BOOLEAN bDBUpdatesEnabled
            );

    WEATHER_INTERFACE_OBJECT (*hInit) (
        WEATHER_SERVICE_OBJECT hWeatherService,
        SMS_OBJECT hParent,
        BOOLEAN bDBUpdatesEnabled,
        BOOLEAN bClearPersistent,
        UN8 un8CurDBVersion
            );

    void (*vUnInit) (
        WEATHER_INTERFACE_OBJECT hInterface
            );

    BOOLEAN (*bProcessMessage) (
        WEATHER_INTERFACE_OBJECT hInterface,
        OSAL_BUFFER_HDL *phPayload
            );

    BOOLEAN (*bEnableSkiReports) (
        WEATHER_INTERFACE_OBJECT hInterface
            );

    BOOLEAN (*bDisableSkiReports) (
        WEATHER_INTERFACE_OBJECT hInterface
            );

    BOOLEAN (*bProcessWindData) (
        WIND_UNPARSED_DATA tData,
        WIND_DATA_STRUCT *psData
            );

    BOOLEAN (*bProcessWeatherEventData) (
        WEATHER_EVENT_UNPARSED_DATA tData,
        WEATHER_EVENT_DATA_STRUCT *psData
            );

    BOOLEAN (*bProcessUvInfoData) (
        UV_INFO_UNPARSED_DATA tData,
        UV_INFO_DATA_STRUCT *psData
            );

    BOOLEAN (*bProcessNewSnowData) (
        NEW_SNOW_UNPARSED_DATA tData,
        NEW_SNOW_DATA_STRUCT *psData
            );

    BOOLEAN (*bProcessTrailsData) (
        TRAILS_UNPARSED_DATA tData,
        TRAILS_DATA_STRUCT *psData
            );

    BOOLEAN (*bProcessPrecipitationData) (
        PRECIP_UNPARSED_DATA tData,
        PRECIP_DATA_STRUCT *psData
            );

    BOOLEAN (*bProcessHumidityData) (
        HUMIDITY_UNPARSED_DATA tData,
        HUMIDITY_DATA_STRUCT *psData
            );

    BOOLEAN (*bProcessPollenData) (
        POLLEN_UNPARSED_DATA tData,
        POLLEN_DATA_STRUCT *psData
            );

    BOOLEAN (*bIsTimerNeeded) (
        WEATHER_INTERFACE_OBJECT hInterface
            );

    BOOLEAN (*bProcessTimerCall) (
        WEATHER_INTERFACE_OBJECT hInterface
            );

    BOOLEAN (*bLoadForecastData) (
        WEATHER_INTERFACE_OBJECT hInterface,
        SMS_OBJECT hParent,
        OSAL_OBJECT_HDL hWeatherMsgsList
            );

    BOOLEAN (*bLoadSkiData) (
        WEATHER_INTERFACE_OBJECT hInterface,
        SMS_OBJECT hParent,
        OSAL_OBJECT_HDL hWeatherMsgsList
            );

} WEATHER_PLUGIN_INTERFACE_STRUCT;

typedef struct weather_mgr_interface_struct
{
    BOOLEAN (*bRefDBBank) (
        WEATHER_SERVICE_OBJECT hWeatherService,
        STRING_OBJECT *phInUseDB,
        STRING_OBJECT *phNextDB
            );

    BOOLEAN (*bDBUpdateBegin) (
       SQL_INTERFACE_OBJECT hConnection,
       char *pacBuffer,
       size_t tBufferSize
            );

    BOOLEAN (*bDBUpdateEnd) (
       SQL_INTERFACE_OBJECT hConnection,
       char *pacBuffer,
       size_t tBufferSize,
       UN8 un8DBVer
            );

    BOOLEAN (*bStartForecastProcessing) (
        WEATHER_SERVICE_OBJECT hWeatherService
            );

    BOOLEAN (*bProcessForecastReport) (
        WEATHER_SERVICE_OBJECT hWeatherService,
        LOC_ID tID,
        WEATHER_FORECAST_TYPE_ENUM eType,
        FORECAST_DATA_STRUCT *psForecastData
            );

    BOOLEAN (*bEndForecastProcessing) (
        WEATHER_SERVICE_OBJECT hWeatherService
            );

    BOOLEAN (*bStartSkiProcessing) (
        WEATHER_SERVICE_OBJECT hWeatherService
            );

    BOOLEAN (*bProcessSkiReport) (
        WEATHER_SERVICE_OBJECT hWeatherService,
        LOC_ID tID,
        SKI_DATA_STRUCT *psData
            );

    BOOLEAN (*bEndSkiProcessing) (
        WEATHER_SERVICE_OBJECT hWeatherService
            );

    BOOLEAN (*bAddLocation) (
        SQL_INTERFACE_OBJECT hConnection,
        char *pacBuffer,
        size_t tBufferSize,
        WEATHER_LOCATION_ROW_STRUCT *psLocationData
            );

    BOOLEAN (*bModifyLocation) (
        SQL_INTERFACE_OBJECT hConnection,
        char *pacBuffer,
        size_t tBufferSize,
        WEATHER_LOCATION_ROW_STRUCT *psLocationData
            );

    BOOLEAN (*bDeleteLocation) (
        SQL_INTERFACE_OBJECT hConnection,
        char *pacBuffer,
        size_t tBufferSize,
        WEATHER_LOCATION_ROW_STRUCT *psLocationData
            );

    BOOLEAN (*bForecastHashRemoved) (
        WEATHER_SERVICE_OBJECT hWeatherService,
        WEATHER_HASH_OBJECT hHashEntry
            );

    BOOLEAN (*bForecastHashRepeated) (
        WEATHER_SERVICE_OBJECT hWeatherService,
        WEATHER_HASH_OBJECT hHashEntry
            );

    BOOLEAN (*bSkiHashRemoved) (
        WEATHER_SERVICE_OBJECT hWeatherService,
        WEATHER_HASH_OBJECT hHashEntry
            );

    BOOLEAN (*bSkiHashRepeated) (
        WEATHER_SERVICE_OBJECT hWeatherService,
        WEATHER_HASH_OBJECT hHashEntry
            );

} WEATHER_MGR_INTERFACE_STRUCT;

/****************************/
/** GLOBAL UNION TEMPLATES **/
/****************************/

/**********************/
/** GLOBAL VARIABLES **/
/**********************/

// Interface into the Weather plugins
extern const WEATHER_PLUGIN_INTERFACE_STRUCT GsWeatherIntf;

// Interface into the Weather Manager
extern const WEATHER_MGR_INTERFACE_STRUCT GsWeatherMgr;

/*******************/
/** GLOBAL MACROS **/
/*******************/

/***********************/
/** GLOBAL PROTOTYPES **/
/***********************/

/* Object Friend Prototypes */

#endif /* WEATHER_INTERFACE_H_ */
