/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:UVINFO implementation for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/
#include <string.h>

#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "sms_obj.h"
#include "string_obj.h"
#include "_uvinfo_obj.h"
#include "uvinfo_obj.h"

#include "sms_api_debug.h"

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   eIndexRange
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eIndexRange (
    UVINFO_OBJECT hUVInfo,
    UN8 *pun8UVIndexRangeLow,
    UN8 *pun8UVIndexRangeHigh
        )
{
    BOOLEAN bLocked = FALSE;
    SMSAPI_RETURN_CODE_ENUM eResult = SMSAPI_RETURN_CODE_ERROR;
    UVINFO_OBJECT_STRUCT *psObj = (UVINFO_OBJECT_STRUCT *) hUVInfo;

    do
    {
        if ((pun8UVIndexRangeHigh == NULL) || (pun8UVIndexRangeLow == NULL))
        {
            eResult = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }

        bLocked = SMSO_bLock((SMS_OBJECT)hUVInfo, OSAL_OBJ_TIMEOUT_INFINITE);
        if (bLocked == FALSE)
        {
            break;
        }

        if (psObj->bAvailable == FALSE)
        {
            eResult = SMSAPI_RETURN_CODE_NOT_FOUND;
            break;
        }
        eResult = SMSAPI_RETURN_CODE_SUCCESS;
        *pun8UVIndexRangeHigh = psObj->sData.un8High;
        *pun8UVIndexRangeLow = psObj->sData.un8Low;
    } while (FALSE);

    if (bLocked == TRUE)
    {
        SMSO_vUnlock((SMS_OBJECT)hUVInfo);
    }

    return eResult;
}


/*****************************************************************************
*
*   eRiskRating
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eRiskRating (
    UVINFO_OBJECT hUVInfo,
    UV_RISK_ENUM *peUVRisk
        )
{
    BOOLEAN bLocked = FALSE;
    SMSAPI_RETURN_CODE_ENUM eResult = SMSAPI_RETURN_CODE_ERROR;
    UVINFO_OBJECT_STRUCT *psObj = (UVINFO_OBJECT_STRUCT *) hUVInfo;

    do {
        if (peUVRisk == NULL)
        {
            eResult = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }

        bLocked = SMSO_bLock((SMS_OBJECT)hUVInfo, OSAL_OBJ_TIMEOUT_INFINITE);
        if (bLocked == FALSE)
        {
            break;
        }

        if (psObj->bAvailable == FALSE)
        {
            eResult = SMSAPI_RETURN_CODE_NOT_FOUND;
            break;
        }
        eResult = SMSAPI_RETURN_CODE_SUCCESS;
        *peUVRisk = psObj->sData.eRisk;
    } while (FALSE);

    if (bLocked == TRUE)
    {
        SMSO_vUnlock((SMS_OBJECT)hUVInfo);
    }

    return eResult;
}


/*****************************************************************************
*
*   n32FPrintf
*
*****************************************************************************/
static N32 n32FPrintf (
    UVINFO_OBJECT hUVInfo,
    FILE *psFile
        )
{
    N32 n32Return = 0;
    UVINFO_OBJECT_STRUCT *psObj = (UVINFO_OBJECT_STRUCT *)hUVInfo;
    BOOLEAN bOwner;

    // Determine if we own the object
    bOwner = SMSO_bOwner((SMS_OBJECT)hUVInfo);
    if ((bOwner == FALSE) || (psFile == NULL))
    {
        return EOF;
    }

    // Print UVINFO information header
    n32Return += fprintf(psFile, "\nUVInfo (Object: 0x%X)\n",
                            psObj);

    if (psObj->bAvailable == TRUE)
       {
        const char *pacUVRisk;

        pacUVRisk = pacRisk(psObj->sData.eRisk);
        n32Return += fprintf(psFile, "\tUV Index: %d - %d\n \tUV Risk Rating: %s\n",
               psObj->sData.un8Low, psObj->sData.un8High, pacUVRisk);
    }

    return n32Return;
}


/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   UVINFO_hCreate
*
*****************************************************************************/
UVINFO_OBJECT UVINFO_hCreate(
        SMS_OBJECT hParent,
        BOOLEAN bAvailable,
        UV_INFO_UNPARSED_DATA tData
            )
{
    UVINFO_OBJECT_STRUCT *psObj =
        (UVINFO_OBJECT_STRUCT *)UVINFO_INVALID_OBJECT;

    // Create an instance of the UVINFO object
    psObj = (UVINFO_OBJECT_STRUCT *)
        SMSO_hCreate(
            UVINFO_OBJECT_NAME,
            sizeof(UVINFO_OBJECT_STRUCT),
            hParent,
            FALSE
                );

    if(psObj == NULL)
    {
        // Error!
        return UVINFO_INVALID_OBJECT;
    }

    psObj->bAvailable = bAvailable;

    // Initialize object per inputs
    if (bAvailable == TRUE)
    {
        psObj->bAvailable = GsWeatherIntf.bProcessUvInfoData(tData, &psObj->sData);
    }


    return (UVINFO_OBJECT)psObj;
}

/*****************************************************************************
*
*   UVINFO_vDestroy
*
*****************************************************************************/
void UVINFO_vDestroy (
    UVINFO_OBJECT hUVInfo
        )
{
    BOOLEAN bOwner;

    bOwner = SMSO_bOwner((SMS_OBJECT)hUVInfo);
    if(bOwner == TRUE)
    {
        // Free object instance
        SMSO_vDestroy((SMS_OBJECT)hUVInfo);
    }
    return;
}


/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   pacRisk
*
*****************************************************************************/
static const char *pacRisk (
    UV_RISK_ENUM eRisk
        )
{
    const char *pacUVRisk = "Unknown";

    switch (eRisk)
    {
        case UV_RISK_LOW:
        {
            pacUVRisk = "Low";
        }
        break;

        case UV_RISK_MODERATE:
        {
            pacUVRisk = "Moderate";
        }
        break;

        case UV_RISK_HIGH:
        {
            pacUVRisk = "High";
        }
        break;

        case UV_RISK_VERYHIGH:
        {
            pacUVRisk = "Very High";
        }
        break;

        case UV_RISK_EXTREME:
        {
            pacUVRisk = "Extreme";
        }
        break;

    }

    return pacUVRisk;
}
