/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:TW_SEEK_CONTENT implementation for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/

#include "standard.h"
#include "osal.h"

#include "sms_version.h"
#include "sms_api.h"

#include "decoder_obj.h"
#include "string_obj.h"

#include "seek.h"
#include "tw_seek.h"
#include "tw_seek_content.h"
#include "_tw_seek_content.h"

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/
/*****************************************************************************
*
*   hReportId
*
* This API is used to retrieve the Traffic/Weather Seek Content's Report Id.
*
* Inputs:
*
*   hSeekContent    A handle to a valid SEEK_CONTENT_OBJECT for which the
*                   caller wishes to query the Report Id
*
* Outputs:
*
*   A valid CID_OBJECT or CID_INVALID_OBJECT.
*
*****************************************************************************/
static CID_OBJECT hReportId (
    SEEK_CONTENT_OBJECT hSeekContent
        )
{
    SEEK_SERVICE_ENUM eService;
    CID_OBJECT hId = CID_INVALID_OBJECT;

    // verify the content is associated with the Traffic/Weather Seek Service
    // and that the caller is the owner of that service
    eService = SEEK_CONTENT.eService(hSeekContent);
    if (eService == SEEK_SERVICE_TRAFFIC_WEATHER)
    {
        // get the id. A SEEK_CONTENT_OBJECT is really a CAL_CONTENT_OBJECT
        // so we can just call the appropriate CAL_CONTENT API
        hId = CAL_CONTENT.hCID((CAL_CONTENT_OBJECT)hSeekContent);
    }
    return hId;
}

/*****************************************************************************
*
*   hMarketNameText
*
* This API is used to retrieve the Traffic/Weather Seek Content's MarketName.
*
* Inputs:
*
*   hSeekContent    A handle to a valid SEEK_CONTENT_OBJECT for which the
*                   caller wishes to query the Registered Market Name
*
* Outputs:
*
*   A valid STRING_OBJECT or STRING_INVALID_OBJECT.
*
*****************************************************************************/
static STRING_OBJECT hMarketName (
    SEEK_CONTENT_OBJECT hSeekContent
        )
{
    SEEK_SERVICE_ENUM eService;
    STRING_OBJECT hMarketNameText = STRING_INVALID_OBJECT;

    // verify the content is associated with the Traffic/Weather Seek Service
    // and that the caller is the owner of that service
    eService = SEEK_CONTENT.eService(hSeekContent);
    if (eService == SEEK_SERVICE_TRAFFIC_WEATHER)
    {
        // get the text. A SEEK_CONTENT_OBJECT is really a CAL_CONTENT_OBJECT
        // so we can just call the appropriate CAL_CONTENT API
        hMarketNameText = CAL_CONTENT.hArtistText((CAL_CONTENT_OBJECT)hSeekContent);
    }
    return hMarketNameText;
}

/*****************************************************************************
*
*   hMarketIdText
*
* This API is used to retrieve the Traffic/Weather Seek Content's
*   Market Id Text.
*
*   hSeekContent    A handle to a valid SEEK_CONTENT_OBJECT for which the
*                   caller wishes to query the Registered Market Id Text
*
* Outputs:
*
*   A valid STRING_OBJECT or STRING_INVALID_OBJECT.
*
*****************************************************************************/
static STRING_OBJECT hMarketId (
    SEEK_CONTENT_OBJECT hSeekContent
        )
{
    SEEK_SERVICE_ENUM eService;
    STRING_OBJECT hMarketIdText = STRING_INVALID_OBJECT;

    // verify the content is associated with the Traffic/Weather Seek Service
    // and that the caller is the owner of that service
    eService = SEEK_CONTENT.eService(hSeekContent);
    if (eService == SEEK_SERVICE_TRAFFIC_WEATHER)
    {
        // get the text. A SEEK_CONTENT_OBJECT is really a CAL_CONTENT_OBJECT
        // so we can just call the appropriate CAL_CONTENT API
        hMarketIdText = CAL_CONTENT.hTitleText((CAL_CONTENT_OBJECT)hSeekContent);
    }
    return hMarketIdText;
}


/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   TW_SEEK_CONTENT_vSetServiceSpecificInfo
*
* This function is used to populate the TW_SEEK_CONTENT_REGISTERED_ITEM_STRUCT
*
*****************************************************************************/
void TW_SEEK_CONTENT_vSetServiceSpecificInfo(
    void *pvItem,
    CID_OBJECT hMarketCID,
    SEEK_SERVICE_OBJECT hSeekService
        )
{
    SEEK_SERVICE_ENUM eService;

    eService = SEEK_eService(hSeekService);
    if (eService == SEEK_SERVICE_TRAFFIC_WEATHER)
    {
        TW_SEEK_CONTENT_REGISTERED_ITEM_STRUCT *psTWSpecificInfo;
        SEEK_CONTENT_REGISTERED_ITEM_STRUCT *psGenericItem;

        psGenericItem = (SEEK_CONTENT_REGISTERED_ITEM_STRUCT *)pvItem;

        psTWSpecificInfo = (TW_SEEK_CONTENT_REGISTERED_ITEM_STRUCT *)
                               &psGenericItem->uServiceSpecific;

        // populate the service specific portion
        psTWSpecificInfo->hMarketCID =
            CID.hDuplicate(hMarketCID);
    }

    return;
}


/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   n32FPrintf
*
*    This is used to print out to a file human readable service specific
*    information about the items registered to the seek service
*
*****************************************************************************/
static N32 n32FPrintf (
    void *pvServiceSpecificData,
    FILE *psFile,
    SMSAPI_OUTPUT_OPTION_ENUM eOutputOption
        )
{
    N32 n32NumWritten = 0;

    // check inputs
    if ( (pvServiceSpecificData == NULL) || (psFile == NULL) )
    {
        return EOF;
    }

    switch(eOutputOption)
    {
        case SMS_OUTPUT_OPTION_TERSE:
        case SMS_OUTPUT_OPTION_VERBOSE:
        case SMS_OUTPUT_OPTION_GROSS:
        default:
        {
            // nothing to be printed out
        }
        break;
    }

    return n32NumWritten;
}

/*****************************************************************************
*
*   vUnInit
*
*****************************************************************************/
static void vUnInit(
    void *pvServiceSpecific
        )
{
    // check input
    if (pvServiceSpecific != NULL)
    {
        SEEK_CONTENT_UNION *puServiceSpecific =
            (SEEK_CONTENT_UNION *)pvServiceSpecific;

        TW_SEEK_CONTENT_REGISTERED_ITEM_STRUCT *psRegItem =
            (TW_SEEK_CONTENT_REGISTERED_ITEM_STRUCT*)puServiceSpecific;

        // destroy the CIDs
        CID.vDestroy(psRegItem->hMarketCID);

        // uninitialize
        psRegItem->hMarketCID = CID_INVALID_OBJECT;
    }

    return;
}

/*****************************************************************************
*
*   bSearchForCurrent
*
*****************************************************************************/
static BOOLEAN bSearchForCurrentAfterEnable (
    void *pvServiceSpecific
        )
{
    // we want to search for current
    return TRUE;
}
